function coazpmap(featv) ;
% COAZPMAP, Power MAP of Zernike feature vectors
%      Corneal Topography Analysis System
%   Creates a gray level map with intensities representing powers
%           of Zernike coefficients (in the usual array)
%      Intended to be called by coaram.m
% Input:
%       featv - Zernike feature vector (in standard order)
% Output:
%     Only graphics, in current Figure

%    Copyright (c) J. S. Marron, N. Locantore, 1998
%    Department of Statistics, University of North Carolina




%  Set up global variables
%
%  First some general globals
global INFILESTR NMAX IBASIS ;
%  INFILESTR - string with input filename
%  NMAX - maximum number of Zernike columns
          %    this gets reloaded when a feature vec is reloaded from file
%  IBASIS - 1 for (triangular) Zernike basis, 2 for Radial version

%  Next the paths for file control
global INDATPATH ;
%  INDATPATH - path for input Keratron files

%  Next the globals for handles to figure windows and plots
global RAMHAND PMAPHANDPOS;
%  RAMHAND - handle for Figure Window with Display, Decomposition Menu
%  PMAPHANDPOS - handle to position figure window



%  Calculate matrix of gray scales
%
ngray = 256 ;
mout = zeros(NMAX+2,NMAX+2) ;
          %  initialize everything to black
          %  note pcolor deletes last row and column

vpower = [] ;
vphase = [] ;
i = 1 ;     %  index of full feature vector
for n = 0:NMAX ;   %  loop through Zernike columns (Jacobi orders)
                   %      to construct vector of powers
  if n/2 == round(n/2) ;    %  for n even
    mstart = 0 ;   %  start at 0-th Fourier frequency
  else ;                    %  for n odd
    mstart = 1 ;   %  start at 1-st Fourier frequency
  end ;
  mend = n ;   %  end at n-th Fourier frequency

  for m = mstart:2:mend ;    %  Loop through Fourier frequencies
    if m == 0 ;
      power = featv(i)^2 ;
      phase = NaN ;
      i = i + 1 ;
    else ;
      power = featv(i)^2 + featv(i+1)^2 ;
      phase = atan2(featv(i),featv(i+1)) ;
          %  atan2 is 4 quadrant arctan
      i = i + 2 ;
    end ;
    vpower = [vpower; power] ;
    vphase = [vphase; phase] ;
  end ;
end ;

vlpower = log10(vpower + 10^(-12)) ;
          %  essentially truncate below at -12
maxpow = max(vlpower) ;
minpow = min(vlpower) ;
vipow = (vlpower - minpow) / (maxpow - minpow) ;
          %  map to [0,1]

i = 0 ;    %  Index when coefficients grouped into a vector
for n = 0:NMAX ;   %  loop through Zernike columns (Jacobi orders)
  if n/2 == round(n/2) ;    %  for n even
    mstart = 0 ;   %  start at 0-th Fourier frequency
  else ;                    %  for n odd
    mstart = 1 ;   %  start at 1-st Fourier frequency
  end ;
  mend = n ;   %  end at n-th Fourier frequency

  for m = mstart:2:mend ;    %  Loop through Fourier frequencies
    i = i + 1 ;
    mout(NMAX - m + 1,n + 1) = vipow(i) ;
  end ;
end ;




%  Create New Window for This plot
%
pmaphand = figure ;
set(pmaphand, 'Position', PMAPHANDPOS);
set(pmaphand, 'NumberTitle', 'off');
set(pmaphand, 'Name', 'Power Display');
%  Generate map
%
pcolor(mout) ;
  shading flat ;
    titstr = ['log10(Power) of Coeffs, ' INFILESTR] ;
  title(titstr) ;
  caxis([0,1.2]) ;
          %  this is not 1, because somehow the biggest powers
          %  turn up as black not white
          %       (this is a kludge)
  colormap(gray(256)) ;
  set(gca,'XTickMode','manual') ;
  set(gca,'YTickMode','manual') ;
  set(gca,'XTickLabelMode','manual') ;
  set(gca,'YTickLabelMode','manual') ;

%% Below was all commented out when I got this.


%  Now overlay phase indicators


% hold on ;
%   i = 0 ;    %  Index when coefficients grouped into a vector
%   for n = 0:NMAX ;   %  loop through Zernike columns (Jacobi orders)
%     if n/2 == round(n/2) ;    %  for n even
%       mstart = 0 ;   %  start at 0-th Fourier frequency
%     else ;                    %  for n odd
%       mstart = 1 ;   %  start at 1-st Fourier frequency
%     end ;
%     mend = n ;   %  end at n-th Fourier frequency
%   
%     for m = mstart:2:mend ;    %  Loop through Fourier frequencies
%       i = i + 1 ;
% 
%       plot(vx,vy,'k') ;
% 
%     end ;
%   end ;
% hold off ;




