function vout = coamdf(indata,iendtype) ;
% COAMDF Missing Data Flag
%      Corneal Topography Analysis System
%   Creates a vector giving index of "last non-missing value"
%   for each radial in "spider web" of data matrix "indata"
%         Note: typically call this twice, once for "rdata",
%         and once for "cdata" or "hdata", depending on context,
%         then take minimum
%      Intended to be called by coazrec.m
% Inputs:
%       indata - input data matrix, assumed to be 256 x 26 matrix
%                   (rows index angles, columns index radii)
%                   so the first two elements in Keratron files
%                   have already been removed,
%                   and so has an initial column of all zeros
%       iendtype - 1 (or not given) return index before first non-missing
%                  2  return really last non-missing
% Output:
%       vout - 256x1 vector, which for each radial, gives index of
%                  last non-missing index along that radial,
%                  hence an integer from 1,...,26 (26 = no missings)
%                  (gives empty output when input size is wrong)

%    Copyright (c) J. S. Marron, N. Locantore 1998
%    Department of Statistics, University of North Carolina


if nargin == 1 ;    %  then iendtype not input, so use default
  iet = 1 ;
else ;              %  inendtype input, so use that
  iet = iendtype    
end ;


%  Check size of input matrix
%
insize = size(indata) ;
if  (insize(1) == 256)  &  (insize(2) == 26) ;    
          %  then sizes are right, so proceed

  vout = 26 * ones(256,1) ;
          %  Initialize output vector

  mflag = (indata == 0) ;
          %  ones in locations with a missing value
  vflag = sum(mflag')' ;
          %  row sum of above version
  vflag = (vflag > 0) ;
          %  one in rows where there is a missing
  mflag = mflag(vflag,:) ;
          %  pluck off only rows with a missing

  if iet == 1 ;        %  get index before first non-missing
    [temp,svout] = max(mflag') ;
          %  first transpose, so radials are columns with nearest at top
          %  second find index of first one
    svout = svout' - 1 ;
          %  turn into column vector, and get index of 0 before first 1
  elseif iet == 2 ;    %  get really last non-missing
    [temp,svout] = min(rot90(mflag)) ;
          %  first rotate 90 degrees counterclockwise, so radials
          %    are columns with farthest at top
          %  second find index of first 0, coming down column
    svout = 27 - svout' ;
          %  turn into column vector, and adjust to right end
  end ;

  vout(vflag) = svout ;
          %  replace values where there are some missings with index
          %  of last non-missing

else ;    %  then have wrong sizes, so write error message, 
          %  and output an empty vector

  disp('   ') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp('!!!   Error from coamdf:  input matrix is wrong size') ;
  disp('!!!                returning empty missing data flag') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

  vout = [] ;

end ;




