/*
AUTOREG

Purpose:   Estimates regression models with p-th order autoregressive 
           errors.

Format:    coefs = autoreg(dataset,depvar,indvars,cflag,order,tol,
                   initonly,prinit,prinout,timeit,noprint,&vardef);

Input:     dataset---string containing complete name of data set

           depvar---string or 1x1 character vector containing name of
           dependent variable

           indvars---Kx1; character vector containing names of independent
           variables

           cflag---1x1; if 1, constant will be used in model; else not.

           order---order of the autoregressive process; must be greater than
           0 and less than the number of observations.

             tol---1x1; convergence tolerance.

        initonly---1x1; if 1, only initial estimates will be computed.

          prinit---1x1; if 1, estimates will be printed on every iteration.
                   Pressing any key while the program is running will toggle
                   this printing on and off.

         prinout---1x1; if 1, initial and final estimates printed.

          timeit---1x1; if 1, computation time and number of iterations 
                   printed.

         noprint---1x1; if 1, all printing suppressed.

         &vardef---pointer to a proc that defines new variables, if any

         The following globals are used:

             dta---this contains all the data from the data set

           chekdata, chekvars, makevars, getvars, timstr, datestr, endmess
                ---these are procs

Output:   coefs---Kx1; vector of estimated regression coefficients (the other
                  estimated parameters could be returned also, if desired;
                  just change the RETP statement).

           If specified, output is printed as well.

Remarks:   See AUTOREG.DOC for further documentation.
  
Type:      Proc with global references.
----------------------------------------------------------------------------*/
proc autoreg(dataset,depvar,indvars,cflag,order,tol,initonly,prinit,
             prinout,timeit,noprint, &vardef);

  local vardef:proc;

  local kb, kp, tobs, e, nmp, nmk, np, ybar, yy, xx, xy, beta0, tstart,
        uhat, rv, tv, df, sigsq0, see, acov, varu, acor, dmt, dmtt, phi0,
        gm, rho, qpi, qp, cholqpi, converge, iter, param0, ystar, xstar,
        ytstar, xtstar, ysy, xsy, xsx, beta1, sigsq1, gma, phi1, mtlnl1,
        dmtlnl, param1, db, cxx, vcb, stderrb, tb, pvb, vcphi, stderrph,
        tphi, pvphi, vcsigsq, rsq, stderr, i, vp, gq, mtlnl0, tsumsq,
        detqpi, sigsq0t, klnl, yesprint, omat, mask, fmt, lg,
        datanms, dat, x, y, tm, dt, errflag, f1;

tm = timestr; dt = datestr; cls;
"====================== OUTPUT FROM AUTOREG ===========================";
"                 TIME: " tm "   DATE: " dt;
"======================================================================";
?;

   depvar = "" $+ depvar; /* convert to string, if it is not */

  /* use OLSDA if order == 0 */

   if order == 0;
     "ERROR: The specified order must be greater than 0."; end;
   endif;

/* *************************** GET VARIABLES ****************************** */
  /* ---------------- check on existance of data set */
   dataset = chekdata(dataset); /* if changed, this is placed in memory */

  /* ---------------- check on existance of variables */
   errflag = 0;
   datanms = getname(dataset);
   errflag = chekvars(depvar,datanms,"the dependent variable");
   errflag = chekvars(indvars,datanms,"independent variables");
if errflag; end; endif;

  /* open the dataset */
    open f1 = ^dataset;

  /* read the data set */

    dta = readr(f1,rowsf(f1));  /* read in all rows -- will fail if
                                   dataset is too big */

       call makevars(dta,datanms,datanms); /* create all vars in data set */
       call vardef;        /* compute new vars -- must be global */

      x = getvars(indvars,datanms);
      y = getvars(depvar,datanms);

  /* pack data to get rid of any missing values -- this will automatically
     take care of lags if lags were used */

   dat = packr(x~y);
    y = submat(dat, 0, cols(dat) );
    x = submat(dat, 0, seqa(1,1,cols(dat)-1) );

    f1 = close(f1);  /* close the data set */
/* **************** END OF VARIABLES SECTION ********************************* 
   This whole section could be removed, and the proc definition changed to:

  proc autoreg(x,y,depvar,indvars,cflag,order,tol,initonly,prinit,
              prinout,timeit,noprint);

*************************************************************************** */
tstart = hsec;

    if cflag;
      x = ones(rows(x),1)~x;
      indvars = "CONSTANT"|indvars;      /* add this for printing */
    endif;

   kp = order;
   kb = cols(x); /* number of independent variables */
 tobs = rows(x); /* number of observations */
 klnl = tobs*( ln(2*pi) + 1 );  /* constant to be added to log likelihood */

 yesprint = not noprint;

/* define symetric array of numbers */
   vp = seqa(1,1,kp+1);
   np = seqa(1,1,kp);
  nmp = (ones(kp+1,1).*vp') + vp - 1; /* used in initial approx to phi */
  nmk = nmp[1:kp,1:kp];             /* symmetric array of numbers */

clear dmt; 
@ ============================ Initial Estimates =========================== @

@ Mean and Variance of dependent variable @

   tobs = rows(y);                  @ Number of observations. @
   ybar = meanc(y);

@ Step 1: Initial OLS estimates @

     yy = y'y;
     xx = moment(x,0);
     xy = x'y;

  beta0 = solpd(xy,xx);

@ Step 2: Compute residuals and related statistics @

   uhat = y - x*beta0;

     rv = uhat'uhat;
     tv = yy - tobs*ybar*ybar;
     df = tobs - kb;
    see = sqrt(rv/df);

@ Step 3: Compute autocovariances and autocorrelations of residuals. @
          /* this is the same as: autocov(uhat,0,kp) */

   acov = rev( conv(uhat, rev(uhat), tobs-kp, tobs) ) / tobs;
   varu = acov[1,1];
   if varu == 0;
"ERROR: No residual variance in the specified model."; end;
   endif;

   acor = acov/varu;

@ Step 4: Compute D-matrix and obtain first approximations to
          autoregresive parameters @

   gosub sdmat; /* returns dmt, uses uhat, kp */

   dmtt = tobs*dmt./(tobs+1-nmp);
   phi0 = dmtt[2:kp+1,1]/dmtt[2:kp+1,2:kp+1];  @ Initial Estimates of Phi @

@ Step 5: Compute logl for initial estimates. @
trap 1;

   @ Generate true qp matrix @
   gm = phi0.*ones(kp,1)';
  rho = ( phi0/(eye(kp) - shiftr(gm',-np,0)' - shiftr(gm',np,0)') );
   qp = submat( rotater( (rev(rho)| 1 | rho )', np-1 ), 0, np+kp )
        / (1-phi0'rho);

        qpi = invpd(qp);
     detqpi = detl;
    if scalerr(qpi);
"WARNING: The model is not stationary at the initial estimates.
Therefore the estimates cannot be computed."; end;
   else;
      sigsq0 =
          (dmt[1,1]-2*phi0'dmt[2:kp+1,1]+phi0'dmt[2:kp+1,2:kp+1]*phi0)/tobs;

     mtlnl0  = klnl + tobs*( ln(sigsq0) ) - ln( detqpi );
   endif;

     cholqpi = chol(qpi);

@ Step 6: Compute Variances, etc, for initial estimates. @

      gosub compstat(xx,rv/df); /* compute as though just ols */

@ Step 7: Print out initial parameter estimates @
if prinout and yesprint;
"------------------- INITIAL ESTIMATES (Based on OLS) --------------------";
?;     gosub printres;
endif;

   if initonly == 1; retp(beta0); endif;

@ ============================ ML ESTIMATES ================================ @
    converge = 0;  iter=1;
    param0 = beta0|sigsq0|phi0;  /* initial paramter vector */
if prinit and yesprint;
?;?;
" ------------------------ COMPUTING ML ITERATIONS ------------------------ ";
?;
endif;
    do until converge;    /* see below for convergence test */

     @ 1. Given phi0, qpi, and cholqpi, transform y and x and estimate beta @

         ytstar = conv(1|-phi0,y,kp+1,tobs);   @ Transform last T-P obs. @
         xtstar = conv(1|-phi0,x,kp+1,tobs);

          ystar = (cholqpi*y[1:kp,.])|ytstar;  @ Transform first P obs
                                                 and concatenate @
          xstar = (cholqpi*x[1:kp,.])|xtstar;

            ysy = ystar'ystar;
            xsx = moment(xstar,0);
            xsy = xstar'ystar;

          beta1 = solpd(xsy,xsx);               @ Regression @
                   clear ystar, xstar;

    @ 2. New Sigsq (variance of white-noise error).  @

             rv = ysy - beta1'xsy;

         sigsq1 = rv/tobs;

    @ 3. New Untransformed Residuals.  @

           uhat = y - x*beta1;

    @ 4. DMAT -- this is used to compute phi  @

           gosub sdmat; /* returns dmt, uses uhat, kp */

    @ 5. New Estimates of phi. @

           gma = -qp*(phi0.*np); @ Partials of det(invpd(qp)) @

          phi1 = (gma*sigsq1 + dmt[2:kp+1,1]) / dmt[2:kp+1,2:kp+1];

    @ 6. Compute new qpi (inverse(qp)), qp, and cholqpi @

   @ Generate true qp matrix @
   gm = phi1.*ones(kp,1)';
  rho = ( phi1/(eye(kp) - shiftr(gm',-np,0)' - shiftr(gm',np,0)') );
   qp = submat( rotater( (rev(rho)| 1 | rho )', np-1 ), 0, np+kp )
        / (1-phi1'rho);

        qpi = invpd(qp);
    if scalerr(qpi); format 1,0;
"WARNING: The model is not stationary at the estimates for iteration " iter ".
Therefore the estimates cannot be computed."; format 10,6; end;
   else;
     mtlnl1  = klnl + tobs*( ln(sigsq1) ) - ln( detl );
   endif;

     cholqpi = chol(qpi);

    @ 7. Print Results for this iteration. @

    if prinit and yesprint;

  format 1,0;
" --------------------------- ITERATION " iter "---------------------------";
         format 10,6;
"        Beta: ";; beta1';
"         Phi: ";; phi1';
"       Sigsq: ";; sigsq1;
"     -2*lnL = " mtlnl1;;
 dmtlnl=100*(mtlnl1-mtlnl0)/mtlnl0; "    %Change in -2*lnL = ";; /re dmtlnl;
?;
     endif;

  @ 9. Computation of change in parameter values -- db. @

          param1 = beta1|sigsq1|phi1;
           db = (param1 - param0) ./ param1;        @ Proportionate change
                                                      parameters. @
           if abs(db) < tol;                        @ convergence test @
                   converge = 1;
           endif;

  @ 10.  Reset Starting Values for next iteration.   @

          beta0 = beta1;  sigsq0 = sigsq1; phi0 = phi1;
         mtlnl0 = mtlnl1; param0 = param1;

   iter = iter+1;

   if key; prinit = not prinit; endif; /* switch printing on if key pressed */

   endo;

@ ---------------------- END OF ML ITERATION LOOP ------------------- @
if timeit and yesprint;  format 4,2;
?;  " Total Time for Computation and Printing: ";;
    (hsec-tstart)/100 "(seconds)"; format 4,0;
    "                    Number of Iterations: "  iter-1;  format 10,6;
?;
endif;

      @  Compute Estimates of Asymptotic Variance-Covariance Matrices @
               gosub compstat(xsx,sigsq1);

               acor=1|rho;
               varu=sigsq0/(1-rho'phi0);   @ 0-th order autocovariance @
               acov=acor*varu;
               see=sqrt(varu);   @ ML estimate of std. error of estimate @

      @ Print Results @

if prinout and yesprint;
"========================= MAXIMUM LIKELIHOOD RESULTS ======================";
?;          gosub printres;
endif;

format 16,8;
call endmess;
retp(beta1);

/*------------------------- SUBROUTINES FOLLOW -----------------------------*/
@ COMPSTAT:  Subroutine to compute variances, etc, @

goto endcmpst;
compstat: pop sigsq0t; pop cxx;   @ pop X'INV(S)*X @

         vcb = sigsq0t * invpd(cxx);       @ For betas @
     stderrb = sqrt(diag(vcb));
          tb = beta0 ./ stderrb;
         pvb = 2 * cdftc(abs(tb),tobs-kb);         @ Prob Values @

       vcphi = qpi / tobs;              @ For phi  @
    stderrph = sqrt(diag(vcphi));
        tphi = phi0 ./ stderrph;
       pvphi = 2*cdftc(abs(tphi),tobs-kp);     @ Prob Values @

     vcsigsq = sigsq0t*sigsq0t*2/tobs;   @ For sigsq @

      @  R-squared  @

         rsq = 1-(rv/tv);

clear cxx;
return;
endcmpst: ;

@ SDMAT -- computes matrix of symmetric products and crossproducts -- see
          Box & Jenkins, page 276-279.
@
sdmat: /* uses uhat, kp; returns dmt */
      dmt = zeros(kp+1,kp+1);
      i = 0;
      do until i > kp;

       dmt[i+1:kp+1,i+1] =
    rev( conv(trimr(uhat,i,0), rev(trimr(uhat,0,i)), tobs-kp-i, tobs-2*i ) );

      i = i + 1;
      endo;

    dmt = ((dmt.*(.not(eye(kp+1)))+dmt'));
return;

@  PRINTRES: This subroutine prints the results. @

goto endprnou;
printres:
if indvars $== ""; indvars = 0 $+ "X" $+
                    ftocv(seqa(1,1,rows(beta0)),floor(log(rows(beta0)))+1,0);
endif;
if depvar $/= "";
 "                 ____________________________________";
 "                    Dependent variable:  " $depvar;
 "                 ------------------------------------";?;
endif;
     format 7,0;
     "                Number of Observations: ";; tobs;
     format 10,3;
     "                             R-squared:";; rsq;
     "            Standard Error of Estimate:";; see;
     " Variance of White Noise Error (sigsq):";; sigsq0t;
     "                     Variance of sigsq:";; vcsigsq;
     "                    -2*log(likelihood):";; mtlnl0;
?;
"                COEFFICIENTS OF INDEPENDENT VARIABLES (beta)";
     print;
     "Var             Coef           Std. Error     t-Ratio       P-Value";
     "-------------------------------------------------------------------";
omat=indvars~beta0~stderrb~tb~pvb;
mask=0~1~1~1~1;
let fmt[5,3]="-*.*s" 10 8
             "*.*lf " 14 6
             "*.*lf " 14 6
             "*.*lf " 14 6
             "*.*lf " 11 3;

call printfm(omat,mask,fmt);

?;
"                      AUTOREGRESSIVE PARAMETERS (Phi)";
     print;
     "Lag         Phi            Std. Error     T-Ratio        P-Value";
     "----------------------------------------------------------------";
lg = seqa(1,1,rows(phi0));
omat=lg~phi0~stderrph~tphi~pvphi;
mask=1~1~1~1~1;
let fmt[5,3]="-*.*lf" 6 0
             "*.*lf " 14 6
             "*.*lf " 14 6
             "*.*lf " 14 6
             "*.*lf " 11 3;

call printfm(omat,mask,fmt);

?;
"                   AUTOCORRELATIONS AND AUTOCOVARIANCES";
     print;
"            Lag           Autocovariances      Autocorrelations";
"            ----------------------------------------------------";
omat=(0|lg)~acov~acor;
mask=1~1~1;
let fmt[3,3]="*.*lf " 13 0
             "*.*lf " 20 6
             "*.*lf " 20 6;
call printfm(omat,mask,fmt);
return;

endprnou: ;
endp;
