(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    360894,      16891]*)
(*NotebookOutlinePosition[    361549,      16914]*)
(*  CellTagsIndexPosition[    361505,      16910]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Fisher Linear Discriminant Direction, Full (40 dim \
subspace), Normals vs. Schizos, for Major Axis Rescaled, Full Population, on \
raw Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {0.99999999577465, 
          0.00000000042254, \(-0.00000000140845\), \(-0.15265874436620\), 
          0.05051707605634, \(-0.06959121563380\), \(-0.23566859253521\), 
          0.01841770549296, 0.02504751112676, 
          0.03545001084507, \(-0.00452289971831\), \(-0.08314208704225\), 
          0.00545232028169, \(-0.07516635943662\), \(-0.03967742253521\), \
\(-0.01263550084507\), 0.02432813154930, 
          0.01841314887324, \(-0.00793547971831\), \(-0.04315542014085\), \
\(-0.01110062661972\), \(-0.03424806577465\), \(-0.00997004577465\), \
\(-0.01400951478873\), 0.02235066478873, \(-0.00055634633803\), 
          0.00031299535211, \(-0.00816849971831\), \(-0.00305151535211\), 
          0.00108292000000, \(-0.00114177281690\), \(-0.00645633408451\), 
          0.00787526211268, \(-0.00631870070423\), 0.00468007873239, 
          0.00181971619718, 0.00642344676056, 0.00725774338028, 
          0.00033030056338, \(-0.00271897239437\), \(-0.00346696915493\), 
          0.00029988901408, 0.00166032788732, 0.00120431774648, 
          0.00509558619718, 
          0.00156389971831, \(-0.00014830422535\), \(-0.00034171577465\), \
\(-0.00357382661972\), 0.00491082211268, 0.00050473633803, 0.00110795901408, 
          0.00000952042254, \(-0.00171832591549\), \(-0.00025740760563\), \
\(-0.00044140380282\), \(-0.00035070605634\), 0.00281390267606, 
          0.00016366535211, 
          0.00014455901408, \(-0.00195638549296\), \(-0.00106211915493\), \
\(-0.00056562169014\), \(-0.00060771816901\), 
          0.00051153267606, \(-0.00022239112676\), 
          0.00035875732394, \(-0.00069439746479\), \(-0.00055557535211\), 
          0.00008571070423, \(-0.00018296197183\), \(-0.00020264000000\), 
          0.00002939464789, \(-0.00077437957746\), \(-0.00005848859155\), \
\(-0.00063389633803\), 0.00049846464789, 0.00028082816901, 
          0.00040762183099, \(-0.00026705014085\)};\)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(axis\  = \ {0.00000491000033, \(-0.00000026580137\), \
\(-0.00000528620113\), 
          0.04584156794971, \(-0.16897240749752\), \(-0.09752517400355\), \
\(-0.02037067699823\), \(-0.02903799700939\), 0.10191628000031, 
          0.33096625206516, 0.02349969186659, \(-0.04333836300926\), 
          0.12099591811490, 0.04439915672275, 0.05932408842232, 
          0.01638175128019, \(-0.16344316865461\), \(-0.07589896711497\), \
\(-0.14719615810992\), \(-0.02168031332263\), 0.05761105009255, 
          0.04243190689640, \(-0.08224217910477\), \(-0.13424966238168\), 
          0.07902471321615, 0.04117182039094, 
          0.21877833239323, \(-0.14567316306495\), 0.05901485197812, 
          0.01457203954265, \(-0.04220779180112\), 0.07022328157180, 
          0.07053250766516, 0.03130482296718, \(-0.04253992185254\), 
          0.23063697680764, \(-0.13750826859381\), 
          0.01826463814544, \(-0.10843260855029\), 0.04495681998072, 
          0.10329045729439, \(-0.19074924288135\), 
          0.00556969369942, \(-0.10765671018051\), 
          0.01534429846113, \(-0.07288319740934\), 
          0.15919243931690, \(-0.24743330948860\), \(-0.01639854747746\), \
\(-0.19851984867529\), 0.06577299075030, 
          0.20612526478687, \(-0.05303692666888\), \(-0.14030058435502\), 
          0.00353852276515, 0.18604131645783, \(-0.02044945408049\), 
          0.04295329841792, 
          0.02582979341321, \(-0.07408809211037\), \(-0.11959083289004\), \
\(-0.11600297052587\), \(-0.12302085661116\), \(-0.05544196525135\), \
\(-0.00299301615127\), \(-0.22257924101469\), 0.13143145702067, 
          0.05283092696347, 0.02257671691462, 0.10634478935962, 
          0.12442860368955, \(-0.14547361258032\), 
          0.01999382535797, \(-0.06406367248037\), 0.16622889930196, 
          0.15452488076552, 0.08241740480521, \(-0.05615229727949\), 
          0.03726818795823, 0.12521064055150};\)\)], "Input"],

Cell[BoxData[
    \(General::"spell1" \(\(:\)\(\ \)\) 
      "Possible spelling error: new symbol name \"\!\(axis\)\" is similar to \
existing symbol \"\!\(Axis\)\"."\)], "Message"]
}, Open  ]],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the mean", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87469 .15276 m
.87085 .14472 L
.86557 .13669 L
.86248 .13304 L
.85886 .12948 L
.85155 .12405 L
.84696 .12157 L
.84166 .11939 L
.83914 .11857 L
.8364 .11782 L
.83398 .11728 L
.83135 .11679 L
.82876 .11642 L
.82637 .11617 L
.82522 .11608 L
.82395 .116 L
.82275 .11594 L
.82164 .11591 L
.82096 .11589 L
.82022 .11589 L
.81888 .11589 L
.81758 .11592 L
.81638 .11596 L
.81507 .11603 L
.81365 .11614 L
.81078 .11642 L
.80821 .11677 L
.80545 .11724 L
.80053 .11836 L
.79591 .11977 L
.79183 .12136 L
.78344 .12596 L
.77973 .12882 L
.77608 .13239 L
.7733 .13582 L
.77075 .13977 L
.76712 .14756 L
.76553 .15225 L
.76427 .1568 L
.76187 .16752 L
.75963 .17802 L
.75703 .18767 L
.75395 .19588 L
.74961 .20414 L
.74429 .21153 L
.73735 .21892 L
.72085 .23195 L
.7029 .24315 L
.68604 .25187 L
.66784 .25917 L
Mistroke
.64959 .26466 L
.6113 .27329 L
.59148 .27675 L
.57003 .27957 L
.54761 .2817 L
.5373 .2825 L
.52647 .28324 L
.51651 .28379 L
.51089 .28402 L
.50821 .2841 L
.50567 .28417 L
.50337 .28422 L
.50207 .28424 L
.50087 .28425 L
.49986 .28426 L
.49875 .28427 L
.49759 .28427 L
.4965 .28427 L
.4953 .28427 L
.49404 .28426 L
.49296 .28425 L
.49178 .28424 L
.48914 .28419 L
.48669 .28413 L
.48148 .28394 L
.47583 .28366 L
.46556 .28297 L
.44644 .28127 L
.42676 .27917 L
.41653 .2779 L
.40567 .27629 L
.38576 .27236 L
.36712 .26747 L
.34812 .26156 L
.32838 .25451 L
.31172 .24732 L
.30247 .24245 L
.29407 .23723 L
.28614 .23127 L
.27985 .22531 L
.27716 .22212 L
.2747 .21855 L
.27284 .21491 L
.2721 .21291 L
.27183 .212 L
.27159 .21104 L
.27142 .21017 L
.27135 .20972 L
.27128 .20922 L
.27119 .20836 L
Mistroke
.27115 .20755 L
.27114 .20706 L
.27115 .20654 L
.27116 .20604 L
.27119 .20558 L
.27128 .2047 L
.27135 .20422 L
.27142 .20378 L
.2718 .20209 L
.27208 .20115 L
.27238 .20028 L
.27388 .19702 L
.27567 .19413 L
.27807 .191 L
.28396 .18477 L
.29146 .17782 L
.29813 .17136 L
.30282 .16569 L
.30379 .16417 L
.30464 .16258 L
.30528 .16111 L
.3055 .16047 L
.30571 .15978 L
.30586 .15916 L
.30598 .15851 L
.30607 .15782 L
.30612 .15719 L
.30613 .15684 L
.30612 .15646 L
.30608 .15578 L
.30599 .15512 L
.30588 .15453 L
.30573 .15397 L
.30556 .15345 L
.30507 .1523 L
.30446 .15121 L
.30364 .15008 L
.30256 .14889 L
.30118 .14765 L
.29815 .14557 L
.29463 .14378 L
.29093 .14233 L
.28651 .14099 L
.28136 .13981 L
.27624 .13895 L
.27374 .13862 L
.27138 .13836 L
.2669 .13798 L
.2645 .13783 L
.26193 .13771 L
Mistroke
.26064 .13767 L
.25923 .13763 L
.2579 .13761 L
.25667 .13759 L
.25557 .13758 L
.25435 .13758 L
.25308 .13759 L
.2519 .1376 L
.25059 .13762 L
.24921 .13765 L
.24672 .13772 L
.24378 .13784 L
.24112 .13797 L
.23104 .13868 L
.22143 .13962 L
.21244 .14073 L
.20271 .14222 L
.19245 .14422 L
.18377 .14638 L
.1744 .14934 L
.16488 .15325 L
.15671 .15762 L
.14813 .16366 L
.14112 .1704 L
.13568 .17764 L
.13344 .18158 L
.13139 .18614 L
.13001 .19016 L
.12942 .19232 L
.1289 .19462 L
.12853 .19667 L
.12837 .19776 L
.12822 .19893 L
.1281 .20002 L
.12801 .20101 L
.12795 .20201 L
.1279 .20295 L
.12786 .20405 L
.12785 .20523 L
.12785 .20634 L
.12788 .20736 L
.12793 .20856 L
.12797 .20924 L
.12802 .20987 L
.12824 .21225 L
.12858 .21476 L
.12897 .21703 L
.13014 .22211 L
.13143 .22643 L
.13312 .23103 L
Mistroke
.13686 .23913 L
.14167 .24731 L
.14776 .25566 L
.16213 .27039 L
.17754 .28157 L
.19582 .2913 L
.23439 .30614 L
.27183 .31857 L
.30862 .32969 L
.34781 .34004 L
.36761 .34435 L
.38663 .34755 L
.40511 .34975 L
.41555 .35066 L
.42514 .35131 L
.43453 .35182 L
.44459 .35225 L
.45039 .35244 L
.45568 .35259 L
.45823 .35264 L
.46101 .3527 L
.46364 .35274 L
.46605 .35277 L
.46852 .3528 L
.46988 .35281 L
.47117 .35281 L
.47236 .35282 L
.47346 .35282 L
.47467 .35282 L
.47596 .35282 L
.47725 .35282 L
.47798 .35281 L
.47864 .35281 L
.47987 .3528 L
.48116 .35279 L
.4839 .35276 L
.48691 .35271 L
.48981 .35264 L
.49244 .35257 L
.49836 .35237 L
.50384 .35213 L
.50901 .35184 L
.51914 .35113 L
.52981 .35012 L
.54876 .34764 L
.55834 .346 L
.56857 .34393 L
.58681 .33943 L
.62658 .32693 L
.66472 .31512 L
Mistroke
.70257 .30611 L
.74317 .29823 L
.78177 .28951 L
.80221 .28349 L
.81131 .28023 L
.82068 .27629 L
.82907 .27208 L
.83789 .26671 L
.85206 .25526 L
.85895 .2479 L
.86452 .24067 L
.87008 .23177 L
.87425 .22322 L
.87726 .21514 L
.87848 .21095 L
.87956 .20638 L
.8804 .20176 L
.8807 .19968 L
.88096 .19738 L
.88119 .19482 L
.88128 .19342 L
.88135 .1921 L
.8814 .19087 L
.88143 .18974 L
.88144 .18911 L
.88144 .18843 L
.88144 .1872 L
.88143 .18617 L
.8814 .18523 L
.88136 .18415 L
.88132 .18311 L
.8812 .18127 L
.88104 .17926 L
.88081 .17704 L
.88051 .17468 L
.87982 .17041 L
.87879 .16557 L
.87758 .16104 L
.87469 .15276 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Col603Oo001lo`040?oo00So1@04o`80C?l400Ko100co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03?l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4Ol00`3oo`0^o`00OOl200_o0P05o`030?oo04Oo00<0ool04_l00`3oo`0]o`00TOl00`3oo`16o`03
0?oo01Co00<0ool0;?l004So3P0ko`030?oo04Go00<0ool05_l00`3oo`0[o`00@Ol700ko100go`03
0?oo04Go00<0ool05_l00`3oo`0[o`00??l501Wo0P0eo`030?oo04Co00<0ool06?l00`3oo`0Zo`00
>Ol3023o00<0ool0<_l00`3oo`14o`030?oo01So00<0ool0:_l003Oo0P0So`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Go0P0Vo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`03
0?oo02Ko00<0ool0<Ol00`3oo`13o`030?oo01_o00<0ool0:?l003?o00<0ool09_l203?o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`030?oo01co00<0ool0
:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07Ol00`3oo`0Wo`00<?l00`3oo`0Wo`03
0?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`030?oo02Ko00<0ool0
8?l0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Ko00<0ool0
8Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool09ol002ko00<0ool09Ol201ko1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0To`04
0?oo00So00@0ool01Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`80
2_l200Ko00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00
;_l00`3oo`0So`030?oo03[o00<0ool0?ol00`3oo`0Oo`030?oo02So0007o`D00_l200Co0P0Ho`03
0?oo00So1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01Oo0P08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol201Wo00<0ool0
2?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806ol00`3oo`06o`80
2_l302;o00<0ool08_l200[o0`07o`806_l200[o00<0ool06?l200[o0`02o`00;ol00`3oo`0Vo`03
0?oo03Ko00<0ool0=ol00`3oo`0Uo`030?oo02[o000`o`030?oo02Ko00<0ool0=Ol00`3oo`0eo`80
9ol00`3oo`0[o`00<?l00`3oo`0Wo`80=Ol00`3oo`0co`80:Ol00`3oo`0[o`00<Ol00`3oo`0Xo`80
<ol00`3oo`0bo`030?oo02So00<0ool0;?l0037o00<0ool0:_l2037o00<0ool0<?l202[o00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol302[o00<0ool0
:?l302[o00<0ool00P0Xo`030?oo00Oo000do`030?oo033o0`0Wo`030?oo02Ko0P0_o`030?oo03;o
000eo`80<ol202Go00<0ool08_l4033o00<0ool0<ol003Oo00<0ool0<_l4027o00<0ool07Ol503?o
00<0ool0=?l003So00<0ool0=Ol501co00<0ool06?l503Ko0P0go`00>Ol203[o1`0Eo`030?oo01?o
1@0io`80>Ol003_o00<0ool0?_l900co00<0ool02?l;03_o0`0ko`00??l204Oo5`14o`80?_l003ko
0P1Ao`030?oo04Wo0`10o`00@?l204oo00<0ool0AOl404?o0012o`80COl00`3oo`10o`D0Aol004Co
0`1:o`030?oo03_o1@1<o`00Aol204So00<0ool0=Ol6057o0019o`<0AOl00`3oo`0`o`D0Eol004co
0`0^o`801?l200?o1005o`030?oo02_o1@1Lo`00Col402Wo00@0ool02?l00`3oo`06o`030?oo02Oo
101Qo`00Dol302Ko00@0ool02Ol00`3oo`05o`809?l406Go001Fo`<08ol0103oo`0:o`030?oo00Co
00<0ool08?l306Wo001Io`@07ol0103oo`08o`040?oo00Go00<0ool07Ol306co001Mo`<07Ol200[o
0P06o`030?oo01_o0P1_o`00H?l302ko00<0ool06?l3077o001So`@0:_l00`3oo`0Fo`80M?l006Oo
100Vo`030?oo01;o101fo`00Jol5027o00<0ool03_l407[o001`o`D07?l00`3oo`07o`L0O_l007Go
9P25o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87302 .1376 m
.87981 .12804 L
.88256 .125 L
.88337 .12424 L
.88391 .12379 L
.884 .12373 L
.88406 .12369 L
.88409 .1237 L
.88403 .12378 L
.88386 .12395 L
.88189 .12549 L
.87817 .1278 L
.87297 .13041 L
.86489 .13351 L
.8607 .13476 L
.85588 .13594 L
.85031 .137 L
.84771 .13738 L
.84485 .13772 L
.84306 .1379 L
.8414 .13804 L
.83983 .13814 L
.83809 .13824 L
.83652 .13829 L
.83505 .13833 L
.83348 .13835 L
.8318 .13834 L
.8303 .13831 L
.82871 .13826 L
.82583 .13811 L
.82424 .138 L
.82274 .13787 L
.81935 .13751 L
.81374 .1367 L
.80783 .13554 L
.79731 .13269 L
.78865 .12953 L
.77448 .12261 L
.76992 .11991 L
.76695 .11815 L
.76635 .11783 L
.76582 .11758 L
.76558 .11749 L
.76535 .11742 L
.76502 .11737 L
.7648 .11743 L
.76464 .11758 L
.76457 .11782 L
.76456 .11813 L
.76458 .11835 L
Mistroke
.76462 .11862 L
.76475 .11918 L
.76525 .12085 L
.76674 .12524 L
.76857 .13098 L
.77055 .1383 L
.77149 .14251 L
.77236 .14732 L
.77268 .14946 L
.77298 .15177 L
.7732 .15379 L
.77338 .15593 L
.77345 .15696 L
.77351 .15809 L
.77358 .16007 L
.7736 .16117 L
.77361 .16235 L
.77359 .16344 L
.77357 .16444 L
.77348 .16622 L
.77334 .16813 L
.77324 .16907 L
.77313 .17006 L
.77288 .17179 L
.77258 .17342 L
.7722 .17512 L
.77138 .17799 L
.77041 .1805 L
.76938 .18256 L
.76809 .18457 L
.76681 .18614 L
.76521 .18763 L
.76361 .18875 L
.76208 .18955 L
.76063 .19011 L
.75902 .19056 L
.7582 .19073 L
.75729 .19088 L
.75365 .1912 L
.75272 .19125 L
.75174 .19131 L
.75084 .19137 L
.75003 .19144 L
.74931 .19153 L
.74855 .19165 L
.74784 .1918 L
.7472 .19197 L
.7458 .19248 L
.74513 .19282 L
.7444 .19327 L
Mistroke
.74314 .19427 L
.74205 .19544 L
.74082 .19709 L
.7397 .19898 L
.73551 .20859 L
.73011 .22237 L
.72636 .23036 L
.72216 .2374 L
.71748 .24319 L
.71457 .24587 L
.71173 .24789 L
.70922 .24923 L
.70774 .24984 L
.70634 .25029 L
.70484 .25066 L
.70345 .2509 L
.70181 .25105 L
.70025 .25107 L
.69887 .251 L
.69734 .25081 L
.69573 .25051 L
.69421 .25011 L
.69148 .24918 L
.68852 .24784 L
.68181 .24366 L
.67584 .23877 L
.66491 .22768 L
.65972 .22202 L
.65546 .21772 L
.65333 .21595 L
.65229 .21524 L
.65175 .21493 L
.65126 .21469 L
.65044 .21438 L
.65002 .21428 L
.64959 .21422 L
.64885 .21424 L
.64822 .2144 L
.64756 .21474 L
.647 .21521 L
.64641 .21591 L
.64583 .21684 L
.64532 .21793 L
.64487 .21913 L
.6441 .22181 L
.64339 .22523 L
.6428 .22897 L
.64165 .23925 L
.63953 .26565 L
.6384 .27864 L
Mistroke
.63709 .29091 L
.63519 .3044 L
.63302 .3155 L
.6317 .3207 L
.63035 .32513 L
.62858 .32978 L
.62681 .33341 L
.6248 .33654 L
.62274 .33885 L
.62155 .33984 L
.62041 .34059 L
.61911 .34121 L
.61837 .34148 L
.61768 .34166 L
.61649 .34185 L
.6152 .3419 L
.61407 .34182 L
.6128 .3416 L
.6114 .34121 L
.60987 .34064 L
.607 .33918 L
.60103 .33493 L
.59524 .32972 L
.56426 .29702 L
.55036 .28436 L
.53589 .27419 L
.52357 .26808 L
.51358 .26447 L
.496 .25804 L
.4878 .25435 L
.4829 .25245 L
.48033 .25164 L
.47787 .25101 L
.47549 .25052 L
.47313 .25017 L
.47178 .25003 L
.47052 .24993 L
.46909 .24986 L
.46755 .24984 L
.46607 .24985 L
.46443 .24993 L
.46286 .25004 L
.46139 .25019 L
.45963 .25041 L
.45768 .25073 L
.45401 .25147 L
.44568 .25385 L
.43746 .25698 L
.42185 .26451 L
.40606 .27334 L
Mistroke
.39869 .27742 L
.39289 .28024 L
.39062 .28111 L
.3896 .28144 L
.38871 .28168 L
.38775 .28187 L
.38727 .28194 L
.38679 .28199 L
.38604 .282 L
.38529 .28193 L
.38491 .28186 L
.38454 .28175 L
.38391 .2815 L
.38359 .28131 L
.38331 .28112 L
.38273 .28058 L
.38228 .28001 L
.38193 .27942 L
.38157 .27863 L
.38128 .27781 L
.38088 .27615 L
.38072 .27523 L
.38056 .27415 L
.38035 .27208 L
.38018 .26999 L
.38001 .26737 L
.3798 .26443 L
.37955 .26184 L
.37918 .25893 L
.37862 .25585 L
.3779 .25301 L
.37686 .25004 L
.37539 .24704 L
.37374 .24458 L
.37161 .24227 L
.36947 .24054 L
.36813 .23968 L
.36676 .23894 L
.36535 .2383 L
.36373 .23769 L
.3623 .23725 L
.36065 .23684 L
.35876 .23648 L
.35769 .23632 L
.35666 .2362 L
.35476 .23604 L
.35263 .23595 L
.35058 .23594 L
.34825 .23602 L
.34598 .23618 L
Mistroke
.34381 .23639 L
.33955 .23696 L
.33482 .23779 L
.32262 .24069 L
.31091 .24417 L
.2851 .253 L
.27477 .25655 L
.26531 .25945 L
.26144 .26043 L
.25973 .26079 L
.25825 .26105 L
.25702 .26122 L
.2558 .26134 L
.2547 .26139 L
.2536 .26136 L
.25269 .26126 L
.25186 .26106 L
.25125 .26083 L
.25069 .2605 L
.25021 .26007 L
.25 .25979 L
.24984 .25952 L
.24963 .25892 L
.24953 .25828 L
.24954 .25762 L
.24964 .25685 L
.24981 .25615 L
.25007 .25532 L
.25095 .2533 L
.25209 .2512 L
.25992 .24032 L
.27081 .22739 L
.28026 .21609 L
.28414 .21094 L
.28718 .20613 L
.28827 .20399 L
.28905 .20211 L
.28957 .2005 L
.28978 .1997 L
.28996 .19884 L
.29009 .19804 L
.29019 .1973 L
.29025 .19654 L
.29028 .19611 L
.2903 .19572 L
.29031 .19503 L
.29031 .19427 L
.2903 .19356 L
.29027 .19289 L
.29013 .18995 L
Mistroke
.29009 .18907 L
.29006 .18821 L
.29005 .18742 L
.29005 .18654 L
.29007 .18568 L
.2901 .18488 L
.29016 .18393 L
.29023 .18302 L
.29043 .18134 L
.29075 .17944 L
.29122 .17726 L
.29189 .17475 L
.29373 .16941 L
.29831 .15941 L
.30332 .14998 L
.30528 .14607 L
.30593 .14457 L
.30647 .14308 L
.30667 .14235 L
.3068 .14171 L
.30686 .1411 L
.30686 .14078 L
.30684 .14051 L
.3068 .14024 L
.30674 .14001 L
.30665 .13977 L
.30653 .13954 L
.30637 .13931 L
.30619 .13912 L
.30601 .13897 L
.30578 .13882 L
.30534 .13861 L
.30476 .13844 L
.30444 .13837 L
.30406 .13832 L
.30331 .13827 L
.30247 .13827 L
.30159 .13832 L
.30073 .1384 L
.29971 .13852 L
.28703 .14105 L
.279 .14274 L
.27096 .14422 L
.26334 .14531 L
.25949 .14571 L
.25731 .14589 L
.2553 .14602 L
.25313 .14612 L
.25111 .14618 L
.24932 .14619 L
Mistroke
.24739 .14616 L
.24534 .14609 L
.24432 .14603 L
.24321 .14595 L
.24129 .14578 L
.23953 .14557 L
.23634 .14505 L
.23306 .14432 L
.22989 .14337 L
.22721 .14233 L
.22302 .14016 L
.22104 .13881 L
.21945 .13751 L
.21656 .13449 L
.21441 .1313 L
.2131 .12869 L
.21225 .12682 L
.21189 .12616 L
.21172 .12592 L
.21156 .12573 L
.21137 .12558 L
.21126 .12552 L
.21115 .12548 L
.21094 .12546 L
.21073 .12549 L
.2106 .12554 L
.21045 .12561 L
.21017 .12579 L
.20944 .12646 L
.20766 .12859 L
.19175 .14841 L
.18036 .16047 L
.16678 .17274 L
.1542 .18208 L
.14388 .1881 L
.13902 .19032 L
.13448 .19198 L
.13084 .19298 L
.128 .19358 L
.12706 .19378 L
.12636 .19397 L
.12605 .19408 L
.12591 .19415 L
.12577 .19422 L
.12566 .19429 L
.12557 .19436 L
.12543 .19452 L
.12533 .19469 L
.12527 .19487 L
.12525 .1951 L
Mistroke
.12526 .19523 L
.12527 .19537 L
.12533 .19565 L
.12542 .19597 L
.12569 .19666 L
.13214 .20913 L
.13795 .22014 L
.14569 .23341 L
.15389 .24472 L
.15876 .25002 L
.16437 .25492 L
.16982 .25857 L
.17503 .26122 L
.18018 .26315 L
.18571 .26466 L
.19775 .26676 L
.20355 .26769 L
.20604 .26816 L
.20873 .26877 L
.21324 .27011 L
.21561 .27104 L
.21777 .27206 L
.22206 .27471 L
.22628 .27842 L
.23036 .28344 L
.23378 .28916 L
.2365 .295 L
.23913 .30195 L
.24415 .3185 L
.24839 .33221 L
.25067 .33765 L
.25207 .34019 L
.25346 .34216 L
.25476 .34357 L
.25629 .34477 L
.2571 .34524 L
.25801 .34562 L
.25889 .34589 L
.25973 .34604 L
.26073 .34612 L
.26125 .34612 L
.26183 .34609 L
.26289 .34594 L
.26391 .34572 L
.26516 .34535 L
.26634 .3449 L
.26913 .34355 L
.27429 .34027 L
.28019 .33577 L
.29272 .32541 L
Mistroke
.29897 .32049 L
.30458 .31656 L
.30934 .31377 L
.31174 .31264 L
.3142 .31171 L
.31544 .31136 L
.31653 .31111 L
.3176 .31095 L
.31858 .31086 L
.31964 .31086 L
.32058 .31093 L
.32157 .31111 L
.32257 .3114 L
.32345 .31176 L
.32423 .31217 L
.32502 .31271 L
.32583 .31339 L
.32708 .31476 L
.32825 .3165 L
.32936 .31877 L
.33034 .32153 L
.3311 .32442 L
.33166 .32723 L
.33218 .33058 L
.33256 .33378 L
.33322 .3411 L
.33379 .34723 L
.33419 .3504 L
.33465 .35306 L
.33518 .35532 L
.33554 .35648 L
.33591 .35748 L
.3367 .35909 L
.33721 .35986 L
.33771 .36047 L
.33823 .36098 L
.33881 .36143 L
.3395 .36184 L
.34019 .36213 L
.34088 .36232 L
.34155 .36243 L
.34234 .36247 L
.34323 .36244 L
.34376 .36238 L
.34433 .36229 L
.34542 .36205 L
.34665 .36168 L
.34808 .36114 L
.35368 .35833 L
.36629 .3499 L
Mistroke
.38092 .33912 L
.39565 .32896 L
.40098 .32587 L
.40597 .32353 L
.4073 .32305 L
.40849 .32269 L
.4095 .32245 L
.41053 .32226 L
.41146 .32217 L
.41238 .32215 L
.41312 .32221 L
.41388 .32234 L
.4146 .32255 L
.41499 .32271 L
.41533 .32287 L
.41592 .32323 L
.4165 .32368 L
.41703 .32419 L
.41748 .32474 L
.41824 .32593 L
.419 .32758 L
.41959 .32931 L
.42071 .33384 L
.42166 .33826 L
.42277 .34268 L
.42354 .34516 L
.42454 .34785 L
.42684 .35245 L
.4284 .35481 L
.43032 .35721 L
.43454 .36119 L
.43934 .36442 L
.44563 .36743 L
.44961 .36885 L
.4535 .36998 L
.45826 .37108 L
.46302 .37192 L
.46757 .37252 L
.47192 .37294 L
.47446 .37312 L
.47679 .37324 L
.47809 .37329 L
.4795 .37334 L
.48206 .37339 L
.4845 .3734 L
.48673 .37338 L
.4893 .37331 L
.49173 .37322 L
.49396 .3731 L
.49602 .37297 L
Mistroke
.50063 .37258 L
.50517 .37206 L
.51006 .37136 L
.51943 .36953 L
.52771 .36724 L
.53563 .36427 L
.54214 .36103 L
.54735 .35772 L
.56451 .34185 L
.57266 .33431 L
.58117 .32852 L
.5859 .32618 L
.59137 .32411 L
.59397 .32333 L
.59678 .32262 L
.60191 .32164 L
.60473 .32125 L
.60629 .32108 L
.60777 .32094 L
.60914 .32083 L
.61062 .32074 L
.61197 .32068 L
.61322 .32063 L
.61399 .32061 L
.61483 .32059 L
.61633 .32057 L
.61778 .32057 L
.6191 .32059 L
.62052 .32061 L
.62131 .32063 L
.62206 .32065 L
.6251 .32076 L
.63047 .32103 L
.63318 .32116 L
.63597 .32129 L
.63711 .32133 L
.63833 .32136 L
.63902 .32138 L
.63965 .32139 L
.64086 .3214 L
.64189 .32141 L
.643 .3214 L
.64414 .32137 L
.6452 .32134 L
.64624 .32129 L
.64737 .32123 L
.6494 .32107 L
.65139 .32088 L
.65323 .32066 L
.66224 .31922 L
Mistroke
.67188 .31753 L
.67779 .31664 L
.6838 .3159 L
.68735 .31555 L
.69133 .31523 L
.69503 .315 L
.69851 .31484 L
.70181 .31473 L
.70369 .31469 L
.70547 .31466 L
.70733 .31463 L
.70903 .31462 L
.71089 .31461 L
.71183 .31461 L
.71284 .31461 L
.7147 .31462 L
.71641 .31463 L
.72019 .31465 L
.72187 .31466 L
.72343 .31467 L
.72502 .31467 L
.72586 .31467 L
.72675 .31466 L
.72825 .31465 L
.72985 .31462 L
.73132 .31459 L
.73266 .31454 L
.73394 .31448 L
.73508 .31441 L
.73753 .31421 L
.73886 .31407 L
.74021 .31388 L
.74249 .31348 L
.74474 .31292 L
.74697 .31214 L
.7487 .31134 L
.75036 .31033 L
.75293 .30813 L
.76046 .29565 L
.76281 .29194 L
.76593 .28827 L
.76927 .2855 L
.7715 .28411 L
.77381 .28293 L
.77868 .28112 L
.78142 .28038 L
.78464 .27968 L
.78757 .27918 L
.79087 .27872 L
.79722 .27803 L
Mistroke
.80531 .27733 L
.8144 .27648 L
.82243 .27548 L
.82705 .27473 L
.83138 .27388 L
.83986 .27175 L
.84876 .26869 L
.85805 .26438 L
.86597 .25966 L
.87932 .24893 L
.88877 .23838 L
.89216 .23353 L
.89442 .22952 L
.89528 .22759 L
.89562 .22666 L
.89588 .22575 L
.89604 .22504 L
.8961 .22466 L
.89614 .22432 L
.89617 .22375 L
.89613 .2232 L
.89604 .22269 L
.89589 .22226 L
.89566 .22184 L
.89553 .22165 L
.89535 .22146 L
.89502 .22116 L
.89461 .22089 L
.89416 .22068 L
.8937 .22051 L
.89321 .22038 L
.89272 .22028 L
.89214 .2202 L
.89148 .22013 L
.89074 .22009 L
.88999 .22006 L
.88929 .22005 L
.8885 .22005 L
.88758 .22005 L
.8867 .22006 L
.88566 .22007 L
.88463 .22007 L
.88366 .22006 L
.88274 .22003 L
.88171 .21999 L
.8806 .21992 L
.87948 .21982 L
.87825 .21967 L
.87708 .21948 L
.87601 .21926 L
Mistroke
.87352 .21859 L
.87229 .21815 L
.87095 .21757 L
.86851 .21623 L
.86641 .21467 L
.86417 .21246 L
.86227 .20995 L
.86073 .20725 L
.85953 .20444 L
.85847 .20103 L
.85805 .19918 L
.85767 .19705 L
.85744 .19536 L
.85727 .19375 L
.85715 .19193 L
.85708 .18994 L
.85708 .188 L
.85714 .18613 L
.85723 .18443 L
.85738 .18255 L
.85756 .18081 L
.8578 .17888 L
.85845 .17493 L
.86007 .16776 L
.86265 .15941 L
.86605 .15087 L
.87302 .1376 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0@ol00`3oo`16o`00O?l0103oo`08o`D01?l204Co0`16o`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04?o00@0ool07?l00`3oo`0Vo`00@ol203Oo00@0ool02Ol00`3o0005o`03
0?oo04?o00@0oooo0P0Ho`80:Ol004?o0`0go`802ol200Go00<0ool0A?l00`3oo`02o`805Ol00`3o
000Yo`00@_l0103oo`1;o`030?oo04Co00<0ool01?l300ko10000ol0o`0Yo`00@Ol00`3oo`02o`03
0?oo017o100co`030?oo04Co00<0ool01ol>00Co00<0ool0:Ol0047o00<0ool00ol200go1003o`03
0?oo037o00<0ool0AOl00`3oo`0Ho`030?oo02Wo0010o`030?oo00Ko3@07o`030?oo037o00<0ool0
AOl00`3oo`0Go`030?oo02[o000oo`030?oo01[o00<0ool0<_l00`3oo`15o`030?oo01Oo00<0ool0
:_l003ko00<0ool06ol00`3oo`0bo`030?oo04Go00<0ool05_l00`3oo`0[o`00??l201go00<0ool0
<ol00`3oo`15o`030?oo01Ko00<0ool0:ol003_o00<0ool07Ol00`3oo`0co`030?oo04Go00<0ool0
5Ol00`3oo`0/o`00>_l00`3oo`0No`030?oo03?o00<0ool0AOl00`3oo`0Eo`030?oo02co000io`03
0?oo01ko00<0ool0=?l00`3oo`15o`030?oo01Co00<0ool0;Ol003So00<0ool07ol00`3oo`0do`03
0?oo04Go00<0ool05?l00`3oo`0]o`00=ol00`3oo`0Oo`030?oo027o0P04o`800ol400Go00<0ool0
AOl00`3oo`0Do`030?oo02go000fo`030?oo023o00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0
AOl00`3oo`0Do`030?oo02go000eo`030?oo027o00<0ool08?l0103oo`09o`030?oo00Go0P15o`03
0?oo01Go00<0ool0;Ol003?o0P0To`030?oo01[o1002o`040?oo00[o00<0ool01?l00`3oo`13o`80
5ol00`3oo`0]o`00<?l302Ko00<0ool08?l0103oo`08o`040?oo00Go00<0ool0?ol501Oo00<0ool0
;_l002go0`0Yo`030?oo027o0P0:o`801_l00`3oo`0mo`807Ol00`3oo`0]o`00;Ol00`3oo`0Yo`03
0?oo03Go00<0ool0??l201ko00<0ool0;Ol002ko00<0ool09ol203Oo00<0ool0??l00`3oo`0Mo`03
0?oo02go000^o`030?oo02Oo00<0ool0=_l00`3oo`0ko`030?oo01ko00<0ool0;Ol000Oo1@02o`80
1?l201Wo00<0ool01ol501[o00<0ool01Ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0;o`03
0?oo01Go1@04o`807?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
6Ol00`3oo`05o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool02_l00`3oo`0Go`03
0?oo00Go00<0ool07?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05ol00`3oo`08o`03
0?oo01So00<0ool01_l0103oo`0;o`030?oo023o00<0ool08Ol00`3o000300Wo00<0ool01ol00`3o
o`0Ho`030?oo00Ko0P0Lo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201[o00<0ool0
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00H0o`00o`07o`802_l00`3o
o`0Ho`030?oo00So1`0Eo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01[o00<0ool01ol00`3o
o`0Go`030?oo00Oo00@0ool02Ol00`3oo`0Ro`030?oo027o00@0o`000_l00`3oo`04o`030?oo00Wo
00<0ool06?l00`3oo`0>o`030?oo01?o00<0ool02?l00`3oo`02o`002?l200[o0`0Jo`030?oo00Ko
0P0Ho`030?oo00Wo0P0:o`<08_l00`3oo`0Ro`801?l00`3oo`03o`<02?l00`3oo`0Ho`804?l00`3o
o`0Bo`802_l300;o000bo`030?oo01ko00<0ool0>ol00`3oo`0Ro`030?oo00Co00<0ool03Ol00`3o
o`0Yo`030?oo02Co000bo`030?oo01go00<0ool04_l502Go00<0ool08_l00`3oo`05o`030?oo00_o
00<0ool0:_l00`3oo`0To`00<ol00`3oo`0Ko`030?oo013o0`05o`<08_l00`3oo`0Ro`030?oo00Ko
0P0;o`030?oo02Wo00<0ool09Ol003?o00<0ool06_l00`3oo`0:o`L02_l00`3oo`0Po`030?oo02;o
00<0ool02?l200Oo0P0[o`030?oo02Ko000do`030?oo01Wo00<0ool02?l201;o00<0ool07ol00`3o
o`0Ro`030?oo00[o00<0ool00ol202_o00<0ool09ol000coo`0<00Wo000<o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol200So00<0ool01_l01@3oool000?o00<0ool00_l200;o
00<0ool01_l00`3oo`06o`040?oo00Go00<0ool01_l00`3oo`0200Co00@0oooo0`02o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l00`3o0006o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo02Wo0P0Do`030?oo00?o0`0=o`030?oo00Wo00<0ool03Ol200go0`0To`030?oo00Co
00<0ool0:Ol00`3oo`07o`030?oo023o00<0ool01ol003[o0P0Bo`H07?l00`3oo`0;o`804Ol202?o
00<0ool0>?l202go000lo`D0<?l00`3oo`08o`804ol00`3o0002023o00<0ool0=_l202oo0011o`@0
;?l00`3oo`06o`805Ol00`3oo`02o`<07Ol00`3oo`0co`<0<Ol004Go00<0ool0:Ol00`3oo`04o`80
5ol00`3oo`05o`806_l00`3oo`0`o`@0=?l004Ko0P0Yo`030?oo00;o0P0Io`030?oo00Oo0P0Ho`03
0?oo02Oo2@0ho`00B?l00`3oo`0Vo`D06ol00`3oo`09o`030?oo01Go00<0ool08_l5047o0019o`03
0?oo02Ko00<0ool07?l00`3oo`0:o`030?oo01Co00<0ool08?l204Ko001:o`030?oo04Co00<0ool0
2ol00`3oo`0Co`030?oo01oo00<0ool0A_l004[o00<0ool0A?l00`3oo`0<o`030?oo01;o00<0ool0
7_l00`3oo`17o`00B_l00`3oo`14o`030?oo00go00<0ool04Ol00`3oo`0No`030?oo04Oo001;o`03
0?oo04?o00<0ool03_l00`3oo`0?o`030?oo01ko00<0ool0B?l004_o00<0ool0@ol00`3oo`0?o`03
0?oo00ko00<0ool07_l00`3oo`18o`00Bol00`3oo`13o`030?oo013o00<0ool03Ol00`3oo`0Mo`03
0?oo04Wo001<o`030?oo013o100Jo`801?l200?o1005o`030?oo017o00<0ool03?l00`3oo`0Lo`03
0?oo04[o001<o`030?oo00oo00@0oooo0P0Ho`040?oo00So00<0ool01_l00`3oo`0Bo`030?oo00_o
00<0ool02olA04go001<o`030?oo00go0P05o`030?oo01Ko00@0ool02Ol00`3oo`05o`805?l00`3o
o`0:o`030?oo00Co1`1No`00COl00`3oo`0:o`801ol00`3oo`0Bo`<000Go0?oo000:o`030?oo00Co
00<0ool05?l00`3o000@06Go001=o`030?oo00Wo00<0ool02?l00`3oo`0?o`800_l300;o00<0ool0
1_l0103oo`05o`030?oo01?o0`09o`030?oo06[o001>o`030?oo00Oo00<0ool02Ol00`3oo`0>o`03
0?oo00?o00@0o`002_l200Ko00<0ool04Ol200?o00<0ool01_l00`3oo`1Zo`00C_l00`3oo`06o`03
0?oo00[o00<0ool03?l200Ko00<0ool04ol00`3oo`0@o`030?oo00?o00<0ool01Ol00`3oo`1[o`00
C_l00`3oo`04o`803Ol00`3oo`0;o`030?oo00Oo00<0ool04_l00`3oo`0?o`030?oo00Go0P05o`03
0?oo06_o001?o`030?oo00;o00<0ool03Ol00`3oo`0:o`030?oo00So00<0ool04_l00`3oo`0>o`03
0?oo00So1@1^o`00Col20003o`00013o00<0ool02?l200_o00<0ool04_l00`3oo`0=o`030?oo07co
001@o`804_l00`3oo`07o`030?oo00_o00<0ool04_l00`3oo`0<o`030?oo07go001To`801ol00`3o
o`0=o`030?oo017o00<0ool02ol00`3oo`1no`00IOl00`3oo`03o`804?l00`3oo`0Ao`030?oo00[o
00<0ool0Ool006Go00@0oooo0P0Co`030?oo013o00<0ool02Ol00`3oo`20o`00IOl401Ko0P0@o`03
0?oo00Go1023o`00POl200ko00<0ool00ol208Oo0023o`D02Ol608Wo001mo`801?l200?o2P2>o`00
O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`03
0?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o
003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87323 .1395 m
.87869 .13012 L
.88077 .12693 L
.88132 .12604 L
.8815 .12568 L
.88162 .1254 L
.88165 .12529 L
.88165 .12519 L
.88158 .12508 L
.88143 .12505 L
.88119 .12509 L
.88101 .12514 L
.88082 .12521 L
.88037 .12539 L
.87901 .12599 L
.87029 .12961 L
.86329 .13184 L
.85882 .13296 L
.85427 .1339 L
.84974 .13462 L
.84726 .13493 L
.84459 .13521 L
.8423 .13539 L
.83974 .13555 L
.83738 .13563 L
.83516 .13567 L
.83368 .13568 L
.8323 .13566 L
.83078 .13563 L
.82916 .13557 L
.82644 .13542 L
.82489 .13531 L
.82343 .13518 L
.81829 .1346 L
.81283 .13374 L
.80698 .13254 L
.80108 .13103 L
.79142 .12789 L
.77807 .1222 L
.77299 .11976 L
.77109 .1189 L
.77038 .11861 L
.76968 .11837 L
.7691 .1182 L
.76863 .11811 L
.7682 .11808 L
.76781 .11813 L
.76764 .11819 L
.76747 .11828 L
.76723 .11851 L
Mistroke
.76706 .11883 L
.76696 .11921 L
.76691 .11967 L
.76691 .12017 L
.76694 .12068 L
.76701 .12129 L
.76713 .12204 L
.76728 .12291 L
.76765 .12468 L
.77001 .13527 L
.77071 .13901 L
.77133 .14273 L
.77181 .14623 L
.77203 .14817 L
.77221 .15001 L
.77235 .15174 L
.77246 .15333 L
.77254 .15508 L
.7726 .15696 L
.77262 .15844 L
.77261 .16004 L
.77257 .16169 L
.7725 .16321 L
.77238 .16498 L
.77223 .16662 L
.77184 .16957 L
.77156 .17116 L
.77127 .17257 L
.77048 .17558 L
.76957 .1782 L
.76862 .18037 L
.76748 .18244 L
.7661 .18441 L
.76474 .18595 L
.7634 .18716 L
.76063 .18899 L
.75904 .18973 L
.75754 .19028 L
.75413 .1912 L
.75097 .19196 L
.74926 .19247 L
.74773 .19307 L
.74637 .19378 L
.74518 .19458 L
.74386 .19572 L
.74265 .19704 L
.74031 .20043 L
.73779 .20527 L
.73291 .2164 L
.72722 .22831 L
Mistroke
.72341 .23468 L
.71944 .23991 L
.7145 .2447 L
.71154 .24676 L
.71001 .24761 L
.7086 .24827 L
.70719 .24882 L
.70565 .2493 L
.70479 .24951 L
.70399 .24967 L
.7024 .24991 L
.70102 .25001 L
.69948 .25003 L
.69787 .24993 L
.69635 .24973 L
.69548 .24958 L
.69453 .24938 L
.6928 .24892 L
.68955 .24775 L
.68575 .24593 L
.68227 .24385 L
.67045 .2345 L
.66488 .22934 L
.65965 .22462 L
.65763 .22301 L
.65563 .22162 L
.65454 .22101 L
.6536 .22057 L
.65269 .22027 L
.65187 .2201 L
.65115 .22005 L
.65042 .22012 L
.64983 .22027 L
.6492 .22054 L
.64855 .22095 L
.64789 .22155 L
.64729 .22225 L
.64677 .223 L
.64573 .22501 L
.64472 .22771 L
.64384 .23078 L
.6431 .23398 L
.64144 .24341 L
.63997 .25396 L
.63716 .27534 L
.63539 .28699 L
.63316 .29894 L
.63045 .30999 L
.62746 .31876 L
.62549 .32308 L
Mistroke
.62346 .32652 L
.62152 .32906 L
.62037 .33025 L
.61927 .33121 L
.6182 .33196 L
.6172 .33254 L
.61606 .33306 L
.6148 .33346 L
.61356 .33369 L
.61285 .33377 L
.61218 .3338 L
.61093 .33375 L
.60956 .33356 L
.60827 .33327 L
.60684 .33282 L
.60418 .33169 L
.59896 .32853 L
.57172 .30356 L
.55639 .28962 L
.54809 .2832 L
.54048 .27818 L
.5336 .27439 L
.52655 .27124 L
.51545 .26749 L
.49771 .26215 L
.48905 .25881 L
.48456 .25716 L
.48016 .25581 L
.47782 .25522 L
.47514 .25469 L
.47379 .25448 L
.47231 .25428 L
.47148 .25419 L
.47071 .25412 L
.46919 .25402 L
.46835 .25398 L
.46746 .25395 L
.46582 .25394 L
.4649 .25396 L
.46402 .25399 L
.46305 .25403 L
.46199 .2541 L
.46005 .25428 L
.45818 .25451 L
.45455 .25511 L
.45074 .25593 L
.44706 .25691 L
.43998 .25921 L
.42377 .26611 L
.4106 .27269 L
Mistroke
.39836 .2786 L
.39575 .27965 L
.3932 .28049 L
.39206 .2808 L
.39104 .28101 L
.39018 .28115 L
.38929 .28123 L
.3884 .28124 L
.38793 .28122 L
.3875 .28117 L
.38706 .28109 L
.38668 .281 L
.38628 .28088 L
.38587 .28072 L
.38523 .2804 L
.38469 .28003 L
.38414 .27955 L
.3836 .27895 L
.38277 .27766 L
.38236 .27681 L
.38202 .27597 L
.38146 .27424 L
.38095 .27221 L
.37923 .26346 L
.37861 .26073 L
.37789 .25812 L
.37609 .25335 L
.37477 .25083 L
.3733 .24864 L
.37141 .24643 L
.36903 .24434 L
.36686 .24291 L
.36425 .24159 L
.36151 .24059 L
.35875 .23987 L
.35708 .23956 L
.35545 .23932 L
.3536 .23913 L
.35153 .239 L
.35046 .23897 L
.34928 .23896 L
.34708 .23899 L
.34486 .2391 L
.34276 .23926 L
.34037 .23951 L
.33805 .23981 L
.33355 .24052 L
.32294 .24277 L
.30306 .24832 L
.28261 .25459 L
Mistroke
.27313 .25717 L
.2691 .25806 L
.26748 .25836 L
.2658 .25862 L
.26428 .2588 L
.26296 .25891 L
.26167 .25895 L
.261 .25895 L
.26042 .25892 L
.25981 .25887 L
.25928 .2588 L
.25873 .25871 L
.25819 .25858 L
.25737 .25831 L
.25696 .25812 L
.2566 .25792 L
.25606 .25752 L
.25559 .25704 L
.25527 .25657 L
.25501 .256 L
.25482 .25532 L
.25473 .25453 L
.25475 .25371 L
.25485 .25291 L
.25502 .25204 L
.2553 .25107 L
.25598 .2492 L
.25838 .24445 L
.26486 .23462 L
.27345 .22308 L
.28063 .2132 L
.28327 .20913 L
.28547 .20523 L
.28694 .20197 L
.28788 .19915 L
.28826 .19765 L
.28856 .19607 L
.28876 .19464 L
.2889 .19334 L
.28917 .19004 L
.28931 .1885 L
.28949 .18678 L
.28973 .18498 L
.29001 .18327 L
.29099 .17906 L
.29251 .17428 L
.29445 .16945 L
.303 .15233 L
.30492 .14834 L
.30571 .14639 L
Mistroke
.30602 .14551 L
.30625 .14474 L
.30641 .14406 L
.30652 .14338 L
.30656 .14302 L
.30657 .14269 L
.30655 .14212 L
.30647 .14163 L
.30633 .14123 L
.30613 .14082 L
.30584 .14044 L
.30548 .14012 L
.30508 .13986 L
.30455 .13962 L
.30396 .13944 L
.30335 .13931 L
.30273 .13923 L
.30198 .13917 L
.30111 .13915 L
.30017 .13917 L
.29906 .13924 L
.29794 .13934 L
.29685 .13946 L
.29082 .14037 L
.28388 .14159 L
.27553 .14299 L
.27126 .14363 L
.26714 .14415 L
.26336 .14455 L
.2613 .14473 L
.25937 .14487 L
.25749 .14498 L
.25579 .14506 L
.25391 .14511 L
.25192 .14514 L
.25028 .14513 L
.24851 .14509 L
.24687 .14501 L
.24537 .14492 L
.24351 .14477 L
.24181 .14459 L
.23873 .14415 L
.23561 .14353 L
.23246 .14271 L
.2297 .1418 L
.22694 .14066 L
.22225 .13803 L
.21888 .13538 L
.21648 .13294 L
.21458 .13064 L
Mistroke
.21376 .12961 L
.21298 .1287 L
.21234 .12809 L
.21198 .12783 L
.21165 .12764 L
.21134 .12752 L
.21105 .12745 L
.21072 .12743 L
.21036 .12747 L
.21001 .12757 L
.20968 .1277 L
.20928 .12791 L
.20888 .12816 L
.20717 .12957 L
.20162 .1355 L
.18404 .1543 L
.17108 .16628 L
.15859 .17617 L
.13729 .1888 L
.13008 .19174 L
.12888 .19222 L
.12782 .19271 L
.12703 .19316 L
.12647 .1936 L
.12623 .19384 L
.12605 .19408 L
.12589 .19436 L
.12583 .19451 L
.12577 .19468 L
.1257 .19498 L
.12566 .19533 L
.12566 .19569 L
.12569 .19603 L
.12572 .19625 L
.12576 .1965 L
.12587 .19696 L
.12622 .19815 L
.12728 .20084 L
.1347 .21636 L
.14028 .22689 L
.14754 .23864 L
.15556 .24876 L
.16384 .2563 L
.16868 .25954 L
.17407 .26232 L
.17931 .26436 L
.18415 .2658 L
.19626 .26842 L
.20146 .26954 L
.207 .27107 L
Mistroke
.21191 .27295 L
.21626 .27525 L
.22054 .27836 L
.22476 .28253 L
.22892 .28808 L
.23244 .2942 L
.2381 .30724 L
.24289 .3203 L
.24548 .32686 L
.24802 .33222 L
.25036 .33598 L
.25176 .33772 L
.25312 .33906 L
.2546 .34019 L
.25547 .3407 L
.25632 .34109 L
.25723 .34142 L
.25809 .34165 L
.25912 .34181 L
.26011 .34188 L
.26106 .34187 L
.26211 .34177 L
.26313 .34161 L
.26408 .34139 L
.2665 .34062 L
.26892 .33959 L
.27404 .33677 L
.28004 .33281 L
.29133 .32475 L
.30198 .31789 L
.30758 .31511 L
.30905 .31453 L
.31056 .31402 L
.31184 .31365 L
.31317 .31335 L
.31434 .31315 L
.31538 .31304 L
.3164 .31299 L
.31734 .31301 L
.31839 .3131 L
.31933 .31327 L
.32033 .31353 L
.32134 .3139 L
.32229 .31435 L
.32312 .31485 L
.32485 .31624 L
.32622 .31777 L
.32751 .31965 L
.32875 .32202 L
.32976 .32447 L
Mistroke
.33159 .33069 L
.33403 .34296 L
.33459 .34575 L
.33524 .34854 L
.33591 .35093 L
.33662 .35296 L
.33734 .35459 L
.33821 .35613 L
.33876 .3569 L
.33932 .35756 L
.34048 .3586 L
.34114 .35903 L
.34188 .35942 L
.34255 .35968 L
.34334 .35989 L
.34423 .36004 L
.34475 .36008 L
.34524 .3601 L
.34619 .36007 L
.34723 .35996 L
.3483 .35977 L
.34953 .35947 L
.35189 .3587 L
.35679 .35651 L
.36961 .34891 L
.3946 .33289 L
.40016 .32988 L
.40554 .32749 L
.40698 .32698 L
.4083 .32657 L
.40952 .32625 L
.4106 .32603 L
.41158 .32588 L
.41257 .32579 L
.41311 .32578 L
.41361 .32578 L
.4145 .32584 L
.41533 .32596 L
.41578 .32606 L
.41618 .32617 L
.41697 .32646 L
.41775 .32684 L
.41843 .32727 L
.41903 .32774 L
.42007 .32877 L
.42115 .33024 L
.42204 .3318 L
.42383 .33597 L
.42676 .34368 L
.42894 .34822 L
Mistroke
.43153 .35228 L
.43527 .35655 L
.43754 .35854 L
.44033 .36057 L
.44305 .36222 L
.44624 .36382 L
.45269 .3663 L
.45652 .3674 L
.46081 .36839 L
.46465 .3691 L
.46904 .36972 L
.47392 .37022 L
.47671 .37042 L
.47805 .3705 L
.47929 .37056 L
.48171 .37065 L
.48299 .37068 L
.48436 .3707 L
.48566 .37071 L
.4869 .37071 L
.4892 .37069 L
.49137 .37064 L
.49375 .37055 L
.49623 .37042 L
.49852 .37027 L
.5026 .36992 L
.50688 .36945 L
.51179 .36877 L
.51627 .36799 L
.5242 .36623 L
.53092 .36426 L
.53759 .36173 L
.54376 .35872 L
.54889 .35561 L
.5646 .34262 L
.57324 .33543 L
.5779 .33225 L
.58254 .32962 L
.58736 .32739 L
.59197 .3257 L
.59728 .32419 L
.60005 .32357 L
.60311 .32301 L
.60588 .32259 L
.60885 .32224 L
.61198 .32196 L
.61337 .32187 L
.61487 .32178 L
.61636 .32171 L
.61772 .32166 L
Mistroke
.61902 .32162 L
.62038 .32159 L
.62158 .32158 L
.62271 .32157 L
.62393 .32156 L
.62523 .32156 L
.62664 .32157 L
.62793 .32157 L
.6308 .3216 L
.63148 .3216 L
.63221 .32161 L
.63353 .32161 L
.63431 .32161 L
.63502 .32161 L
.63567 .32161 L
.63636 .32161 L
.63749 .32159 L
.6387 .32157 L
.63983 .32155 L
.64086 .32151 L
.64204 .32146 L
.64332 .3214 L
.64564 .32124 L
.64794 .32103 L
.65043 .32075 L
.65498 .32009 L
.66397 .31845 L
.67481 .31649 L
.68118 .31555 L
.68486 .3151 L
.68845 .31472 L
.6919 .31442 L
.69572 .31414 L
.69893 .31396 L
.7025 .3138 L
.70587 .31368 L
.70944 .3136 L
.71271 .31354 L
.71573 .3135 L
.71916 .31345 L
.72098 .31342 L
.72266 .3134 L
.72555 .31333 L
.72704 .31328 L
.72859 .31322 L
.73023 .31315 L
.73196 .31304 L
.73344 .31293 L
.73496 .31279 L
.73769 .31246 L
Mistroke
.74004 .31206 L
.74227 .31157 L
.74423 .31099 L
.74751 .30965 L
.74915 .30874 L
.75054 .30779 L
.75334 .30532 L
.75733 .30029 L
.76156 .29435 L
.76393 .29153 L
.76682 .28873 L
.7706 .28598 L
.77471 .28384 L
.77914 .28218 L
.78378 .28093 L
.78942 .27985 L
.79603 .27893 L
.81125 .27727 L
.82043 .27602 L
.82905 .2744 L
.83657 .27249 L
.8447 .26974 L
.85286 .26611 L
.86009 .26206 L
.87331 .25211 L
.88307 .24194 L
.88903 .23362 L
.89016 .23163 L
.89111 .22968 L
.89172 .22816 L
.89197 .22741 L
.89217 .22666 L
.89226 .22626 L
.89232 .22591 L
.8924 .22526 L
.89242 .22493 L
.89241 .22459 L
.89239 .22428 L
.89236 .22401 L
.89226 .22355 L
.89209 .22309 L
.89185 .22266 L
.89157 .2223 L
.8912 .22194 L
.89074 .22161 L
.89029 .22136 L
.88974 .22111 L
.88909 .22089 L
.88833 .22067 L
.88678 .22035 L
Mistroke
.88323 .21978 L
.88129 .21943 L
.87942 .21901 L
.87768 .21852 L
.87581 .21787 L
.87393 .21703 L
.8719 .21588 L
.87002 .21452 L
.86808 .21273 L
.86637 .21072 L
.86493 .20857 L
.86368 .20621 L
.86265 .20375 L
.86166 .20064 L
.86096 .19757 L
.86067 .19583 L
.86042 .19387 L
.86032 .19285 L
.86023 .19174 L
.86012 .1897 L
.86008 .1876 L
.86011 .18557 L
.86019 .18372 L
.86032 .18167 L
.86054 .1794 L
.86085 .17687 L
.86172 .17167 L
.86279 .16683 L
.864 .16231 L
.86852 .14952 L
.87323 .1395 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P14o`<0A_l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`14o`040?l004Co0013o`80=ol0103oo`09o`030?l000Go00<0ool0A?l0103oool201Oo
0P0Yo`00@ol303Oo0P0;o`801Ol00`3oo`14o`030?oo00?o100Ao`<0:_l004;o00@0ool0Bol00`3o
o`14o`030?oo00Oo1P04o`L00_l00`3oo`0Xo`00@Ol00`3oo`02o`80BOl00`3oo`15o`030?oo00_o
1P07o`030?oo02Wo0010o`030?oo00Go0P0=o`L0<ol00`3oo`15o`030?oo01So00<0ool0:Ol003oo
00<0ool02?l=00Ko00<0ool0<Ol00`3oo`15o`030?oo01So00<0ool0:Ol003ko00<0ool07?l00`3o
o`0ao`030?oo04Go00<0ool05ol00`3oo`0Zo`00?Ol00`3oo`0Lo`030?oo03;o00<0ool0AOl00`3o
o`0Go`030?oo02[o000lo`030?oo01go00<0ool0<_l00`3oo`15o`030?oo01Ko00<0ool0:ol003_o
00<0ool07Ol00`3oo`0co`030?oo04Go00<0ool05_l00`3oo`0[o`00>_l00`3oo`0No`030?oo03?o
00<0ool0AOl00`3oo`0Eo`030?oo02co000io`030?oo01ko00<0ool0=?l00`3oo`15o`030?oo01Go
00<0ool0;?l003Oo0P0Qo`030?oo03Co00<0ool0AOl00`3oo`0Eo`030?oo02co000fo`030?oo023o
00<0ool08Ol200Co0P03o`@01Ol00`3oo`14o`030?oo01Ko00<0ool0;?l003Co0P0So`030?oo023o
00@0ool02?l00`3oo`06o`030?oo04Co00<0ool05Ol00`3oo`0]o`00<ol00`3oo`0So`030?oo023o
00@0ool02Ol00`3oo`05o`80AOl00`3oo`0Eo`030?oo02go000ao`809_l00`3oo`0Jo`@00_l0103o
o`0:o`030?oo00Co00<0ool0@ol00`3oo`0Fo`030?oo02go000`o`030?oo02Go00<0ool08Ol0103o
o`08o`040?oo00Go00<0ool0@?l301Wo00<0ool0;Ol002go0`0Xo`030?oo02;o0P0:o`801_l00`3o
o`0no`807?l00`3oo`0]o`00;Ol00`3oo`0Xo`030?oo03Ko00<0ool0?Ol00`3oo`0Lo`80;_l002go
00<0ool0:?l00`3oo`0fo`030?oo03co00<0ool07_l00`3oo`0/o`00;_l00`3oo`0Vo`030?oo03Oo
00<0ool0??l00`3oo`0No`030?oo02co0007o`D00_l200Co0P0Ho`030?oo00So1@0Io`030?oo00Ko
0P04o`801?l202?o00<0ool08_l200Co0P04o`802ol00`3oo`0Eo`D01_l00`3oo`0Io`D00_l200Co
0P03o`002Ol00`3oo`07o`040?oo01So00<0ool02Ol00`3oo`0Ho`030?oo00Ko00@0ool02?l0103o
o`0Ro`030?oo027o00@0ool02?l0103oo`0:o`030?oo01Oo00<0ool01_l00`3oo`0Ko`030?oo00Oo
00@0ool00_l000Wo00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00_o
00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool01ol00`3oo`0Ho`030?oo00Oo0P0Ko`030?oo00[o
0@01o`7o00000ol0000200Co00<0ool02?l201[o00<0ool01004o`030?oo01Oo00@0oooo1002o`04
0?oo00Wo0P0So`030?oo027o00<0ool00`07o`802_l00`3oo`0Ho`030?oo00Wo100Go`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01Wo00<0ool02?l00`3oo`0Fo`030?oo00So00@0ool02Ol00`3o
o`0Ro`030?oo027o00D0o`00o`0200Ko00<0ool02?l00`3oo`0Io`030?oo00go00<0ool05?l00`3o
o`08o`030?oo00;o0008o`802_l301[o00<0ool01_l201Oo00<0ool02_l200[o0`0Ro`030?oo02;o
0P03o`030?oo00Co0`08o`030?oo01So0P0?o`030?oo01?o0P0:o`<00_l003;o00<0ool07_l00`3o
o`0ko`030?oo02?o00<0ool00_l200ko00<0ool0:Ol202Ko000bo`030?oo01go00<0ool0??l00`3o
o`0So`030?oo00Co00<0ool02ol00`3oo`0Yo`030?oo02Go000co`030?oo01_o00<0ool03ol:02Co
00<0ool08_l00`3oo`06o`802_l00`3oo`0Yo`030?oo02Ko000do`030?oo01[o00<0ool03?l300Wo
0`0Ro`030?oo02;o00<0ool02?l00`3oo`06o`030?oo02Wo00<0ool09ol003Co00<0ool06Ol00`3o
o`0;o`803_l00`3oo`0Po`030?oo02;o00<0ool02Ol200Co0`0Zo`030?oo02So000<ool03009o`00
3?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Co00<0o`002?l00`3oo`06o`03
0?oo00;o00@0ool00ol40003o`3o00Oo00<0ool01_l0103oo`05o`030?oo00Ko00<0ool00_l800Go
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o0006o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`<01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`07o`003?l00`3oo`0Xo`805_l700ko00<0ool02Ol00`3oo`0=o`@01ol50003o`3o02?o00<0ool0
1?l00`3oo`0Yo`030?oo00Go0P0So`030?oo00Oo000ho`<0=Ol00`3oo`0;o`804?l202Co00<0ool0
=_l202oo000ko`<0<_l00`3oo`09o`804ol3027o00<0ool0=Ol2037o000no`@0;_l00`3oo`07o`80
5Ol00`3oo`0301ko00<0ool0<ol203?o0012o`<0;?l00`3oo`05o`030?oo01Go00<0ool00ol401[o
00<0ool0;ol403Go0015o`030?oo02Wo00<0ool00ol201So00<0ool01ol00`3oo`0Go`030?oo02Wo
1P0io`00A_l00`3oo`0Yo`D06_l00`3oo`08o`030?oo01Ko00<0ool09?l503oo0017o`030?oo04Oo
00<0ool02Ol00`3oo`0Eo`030?oo027o0`14o`00B?l00`3oo`16o`030?oo00[o0P0Eo`030?oo01oo
0P17o`00BOl00`3oo`15o`030?oo00co00<0ool04_l00`3oo`0Oo`030?oo04Ko0019o`030?oo04Go
00<0ool03Ol2017o00<0ool07ol00`3oo`17o`00B_l00`3oo`14o`030?oo00oo00<0ool03_l00`3o
o`0No`030?oo04So001:o`030?oo04Co00<0ool04?l00`3oo`0=o`030?oo01ko00<0ool0B?l004_o
00<0ool0@ol00`3oo`0Ao`030?oo00_o00<0ool07Ol204_o001;o`030?oo02oo0P04o`800ol400Go
00<0ool04_l00`3oo`0:o`030?oo01_o0`1<o`00C?l00`3oo`0>o`D06_l0103oo`08o`030?oo00Ko
00<0ool04ol00`3oo`08o`030?oo00_o4@1?o`00C?l00`3oo`0=o`030?oo00?o00<0ool05ol0103o
o`09o`030?oo00Go0P0Eo`030?oo00Oo00<0ool01_l5063o001<o`030?oo00_o0P07o`030?oo01Ko
00@0ool02_l00`3oo`04o`030?oo01Go00<0ool03`1Uo`00COl00`3oo`09o`030?oo00Oo00<0ool0
4_l500;o00<0ool01_l0103oo`05o`030?oo01Go0`05o`030?oo06co001=o`030?oo00Oo0P0;o`03
0?oo00ko0`03o`@02_l200Ko00<0ool04_l300;o00<0ool00ol00`3oo`1/o`00COl00`3oo`05o`80
3Ol00`3oo`0=o`030?oo00Go00<0ool04_l00`3oo`0Ao`030?oo00Co1@1_o`00C_l00`3oo`03o`03
0?oo00go00<0ool02ol200So00<0ool04_l00`3oo`0@o`030?oo07Wo001?o`D04?l00`3oo`0:o`03
0?oo00Wo00<0ool04Ol00`3oo`0?o`030?oo07[o001To`802Ol200co00<0ool04Ol00`3oo`0>o`03
0?oo07_o001Uo`030?oo00Go0P0?o`030?oo013o00<0ool03?l207ko001Uo`030?oo00?o0P0Ao`03
0?oo013o00<0ool02ol00`3oo`1no`00IOl200?o00<0ool04_l00`3oo`0?o`030?oo00Wo0P21o`00
I_l401Ko0P0?o`030?oo00So00<0ool0POl0087o0P0>o`030?oo00Co1024o`00Pol400[o00@0oooo
0`28o`00Qol>08_o001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87344 .14139 m
.87757 .13221 L
.87841 .1303 L
.87898 .12885 L
.87915 .12832 L
.87927 .12783 L
.87934 .12737 L
.87933 .12701 L
.87925 .12671 L
.8791 .1265 L
.87887 .12633 L
.8787 .12627 L
.87852 .12623 L
.87808 .1262 L
.87779 .12621 L
.8775 .12624 L
.87689 .12633 L
.87613 .12649 L
.86761 .12882 L
.86041 .13052 L
.8557 .13139 L
.851 .13208 L
.84665 .13254 L
.84409 .13274 L
.84165 .13289 L
.84031 .13294 L
.83903 .13298 L
.83661 .13302 L
.83525 .13303 L
.83398 .13301 L
.83258 .13299 L
.83109 .13294 L
.82868 .13283 L
.82646 .13269 L
.82403 .13248 L
.82147 .13222 L
.81719 .13167 L
.81257 .13091 L
.80261 .12871 L
.79418 .12625 L
.78032 .12122 L
.77783 .12029 L
.77557 .11951 L
.77392 .11903 L
.7731 .11886 L
.77241 .11875 L
.77182 .11872 L
.77131 .11874 L
.7709 .11881 L
.77051 .11893 L
Mistroke
.77013 .11914 L
.7698 .11943 L
.76953 .11978 L
.76934 .12016 L
.76916 .12068 L
.76905 .12124 L
.76897 .12193 L
.76894 .12231 L
.76893 .12275 L
.76894 .12355 L
.76898 .12449 L
.76905 .12544 L
.76914 .12637 L
.77042 .13648 L
.7711 .14268 L
.77126 .14446 L
.77141 .1464 L
.77152 .14824 L
.77161 .14996 L
.77167 .15184 L
.77169 .15284 L
.77171 .1539 L
.77171 .15501 L
.7717 .15606 L
.77168 .157 L
.77165 .158 L
.77158 .15983 L
.77152 .16075 L
.77145 .16174 L
.77112 .16507 L
.77089 .16676 L
.77065 .16827 L
.76999 .1715 L
.76923 .1743 L
.7684 .17675 L
.76646 .18093 L
.76521 .18294 L
.76394 .18461 L
.76119 .18731 L
.75956 .18851 L
.75804 .18943 L
.75273 .19193 L
.74992 .19324 L
.74739 .19475 L
.74473 .197 L
.7424 .19974 L
.7379 .20723 L
.73241 .21837 L
.72887 .22515 L
.7251 .23144 L
Mistroke
.72097 .23707 L
.71677 .24154 L
.71437 .24357 L
.71164 .24545 L
.71009 .24632 L
.70862 .24703 L
.70721 .24761 L
.70565 .24813 L
.7042 .2485 L
.70285 .24876 L
.70155 .24893 L
.70017 .24904 L
.69891 .24906 L
.69773 .24902 L
.69643 .24891 L
.69503 .24873 L
.69353 .24845 L
.6919 .24805 L
.68892 .24708 L
.68338 .24456 L
.67249 .23744 L
.66221 .22975 L
.6597 .22813 L
.6584 .22741 L
.65723 .22685 L
.65619 .22643 L
.65527 .22615 L
.65476 .22603 L
.65429 .22595 L
.6538 .2259 L
.65327 .22587 L
.65242 .22592 L
.65195 .226 L
.65152 .22611 L
.65107 .22627 L
.65067 .22644 L
.6498 .22694 L
.64909 .2275 L
.64834 .22825 L
.64705 .22998 L
.64572 .23245 L
.64455 .23529 L
.64252 .24187 L
.64046 .25058 L
.63827 .26143 L
.63397 .28261 L
.62901 .30127 L
.62601 .30928 L
.62239 .3163 L
.62012 .31949 L
Mistroke
.61784 .3219 L
.61567 .32357 L
.6144 .3243 L
.6132 .32483 L
.612 .32523 L
.61068 .32552 L
.60941 .32566 L
.60822 .32569 L
.60681 .3256 L
.60599 .32548 L
.60522 .32534 L
.6038 .32499 L
.60226 .32448 L
.5988 .32297 L
.59542 .32105 L
.58883 .31644 L
.56342 .29501 L
.5487 .2841 L
.54173 .27988 L
.53559 .27675 L
.52373 .27216 L
.51776 .27046 L
.51248 .26917 L
.49319 .26402 L
.48456 .26121 L
.47946 .25981 L
.47681 .25922 L
.47428 .25877 L
.47202 .25845 L
.47084 .25832 L
.46955 .25821 L
.46823 .25812 L
.46698 .25807 L
.46547 .25804 L
.46407 .25804 L
.4625 .25809 L
.46104 .25816 L
.45939 .25829 L
.45762 .25847 L
.45458 .25888 L
.45114 .25949 L
.44359 .26134 L
.4362 .2637 L
.4223 .26922 L
.40829 .2754 L
.40162 .2781 L
.39868 .27911 L
.39619 .27981 L
.39501 .28008 L
.39396 .28027 L
Mistroke
.39292 .28041 L
.39239 .28046 L
.39183 .28049 L
.39129 .2805 L
.39073 .28049 L
.39021 .28046 L
.38974 .28041 L
.3889 .28026 L
.38849 .28015 L
.38806 .28002 L
.38733 .27974 L
.3867 .27942 L
.38606 .27901 L
.38541 .2785 L
.3844 .27746 L
.38346 .27616 L
.38259 .27459 L
.38186 .273 L
.38037 .26893 L
.3774 .26058 L
.37516 .25595 L
.37378 .25373 L
.37201 .25142 L
.37028 .2496 L
.36818 .24781 L
.36596 .24633 L
.36369 .24513 L
.36055 .24389 L
.35876 .24336 L
.35676 .24288 L
.35494 .24255 L
.35317 .2423 L
.35212 .24219 L
.35114 .24211 L
.35005 .24203 L
.34887 .24198 L
.34695 .24194 L
.34482 .24196 L
.34276 .24204 L
.34085 .24216 L
.33854 .24236 L
.33599 .24264 L
.33121 .24332 L
.29288 .25232 L
.28359 .25457 L
.27897 .25551 L
.27686 .25588 L
.27499 .25616 L
.2732 .25638 L
.27168 .25652 L
Mistroke
.27084 .25657 L
.27008 .2566 L
.26931 .25662 L
.2685 .25662 L
.26725 .25656 L
.26666 .2565 L
.26603 .25642 L
.265 .25624 L
.26412 .256 L
.26324 .25568 L
.26252 .25532 L
.26182 .25486 L
.2612 .2543 L
.26075 .25374 L
.2604 .25317 L
.26011 .25249 L
.25989 .2517 L
.25978 .25098 L
.25974 .25015 L
.25977 .24923 L
.25987 .24832 L
.26003 .24743 L
.26023 .2466 L
.26082 .24462 L
.26286 .23976 L
.26589 .23405 L
.27242 .22343 L
.28782 .1913 L
.2893 .18479 L
.29052 .18072 L
.29199 .17668 L
.30016 .15968 L
.3041 .15182 L
.30492 .14991 L
.30562 .14802 L
.30587 .14717 L
.30607 .14641 L
.3062 .14575 L
.30629 .14509 L
.30632 .14448 L
.30631 .14395 L
.30624 .14341 L
.3061 .14287 L
.306 .14262 L
.30588 .14236 L
.3056 .14192 L
.30526 .14153 L
.30488 .14122 L
.30448 .14096 L
.30405 .14074 L
Mistroke
.30354 .14053 L
.30292 .14035 L
.3022 .14019 L
.30145 .14008 L
.30073 .14001 L
.29988 .13996 L
.29937 .13994 L
.29882 .13994 L
.29777 .13995 L
.29656 .14001 L
.29584 .14005 L
.29515 .1401 L
.2926 .14034 L
.28961 .14068 L
.27708 .14235 L
.27311 .14283 L
.26868 .1433 L
.26474 .14365 L
.26257 .1438 L
.26054 .14392 L
.25874 .144 L
.25681 .14406 L
.25577 .14408 L
.25468 .14409 L
.25276 .14408 L
.25102 .14405 L
.24943 .14399 L
.24771 .1439 L
.2459 .14377 L
.24423 .14362 L
.24242 .14341 L
.23893 .14288 L
.23589 .14225 L
.23322 .14156 L
.22786 .13967 L
.22527 .13846 L
.22314 .1373 L
.21696 .13289 L
.21477 .13114 L
.21384 .13049 L
.21303 .13001 L
.21263 .12982 L
.2122 .12966 L
.21177 .12953 L
.21131 .12944 L
.21089 .1294 L
.21045 .12941 L
.21001 .12946 L
.20961 .12954 L
.2091 .12969 L
Mistroke
.20861 .12988 L
.20743 .13049 L
.20498 .13222 L
.18828 .14806 L
.14308 .18343 L
.13509 .18779 L
.13179 .18955 L
.12941 .19095 L
.12789 .19209 L
.12725 .19274 L
.12679 .19336 L
.12657 .19373 L
.12641 .19408 L
.12627 .19448 L
.12616 .19493 L
.12609 .19534 L
.12606 .19574 L
.12605 .1962 L
.12607 .19672 L
.12613 .19729 L
.12622 .19784 L
.12634 .19849 L
.12652 .19922 L
.12743 .20213 L
.13025 .20914 L
.14103 .23078 L
.14838 .24222 L
.15275 .24776 L
.15771 .253 L
.16235 .25699 L
.16771 .26066 L
.17279 .26337 L
.17766 .2654 L
.19762 .27106 L
.20291 .27271 L
.2083 .2749 L
.21268 .27728 L
.21723 .28054 L
.2216 .2847 L
.2259 .29003 L
.23273 .30152 L
.23874 .31423 L
.2441 .3251 L
.24695 .32964 L
.25021 .33349 L
.25221 .33516 L
.25324 .33583 L
.25421 .33635 L
.2551 .33675 L
.25611 .3371 L
Mistroke
.25714 .33737 L
.25772 .33749 L
.25827 .33757 L
.2593 .33766 L
.25989 .33768 L
.26045 .33768 L
.26149 .33762 L
.26262 .33749 L
.26365 .33732 L
.26464 .3371 L
.26695 .33643 L
.27202 .33434 L
.27787 .33127 L
.28981 .32422 L
.30097 .31836 L
.30375 .31722 L
.3063 .31635 L
.30767 .31596 L
.30913 .31561 L
.31043 .31537 L
.3116 .31522 L
.31286 .31512 L
.31402 .3151 L
.31466 .31512 L
.31524 .31516 L
.3159 .31523 L
.31652 .31532 L
.3171 .31543 L
.31771 .31557 L
.31877 .31589 L
.31999 .31637 L
.32104 .3169 L
.32268 .31798 L
.32434 .31944 L
.32592 .32124 L
.32725 .32316 L
.32935 .32707 L
.33124 .33177 L
.33484 .34293 L
.33646 .34744 L
.33748 .34974 L
.33853 .35169 L
.33962 .3533 L
.34092 .35478 L
.34165 .35544 L
.34246 .35605 L
.34327 .35653 L
.34405 .3569 L
.34495 .35724 L
.34546 .35739 L
Mistroke
.34596 .3575 L
.34689 .35765 L
.34743 .3577 L
.34793 .35773 L
.3485 .35774 L
.34904 .35773 L
.34965 .3577 L
.35031 .35763 L
.35103 .35754 L
.35173 .35742 L
.35307 .35713 L
.3555 .35644 L
.35825 .35545 L
.36358 .35308 L
.37699 .34578 L
.38973 .33855 L
.39636 .33508 L
.40212 .33246 L
.40662 .33081 L
.40894 .33015 L
.41093 .32973 L
.41201 .32956 L
.41297 .32946 L
.41398 .3294 L
.41501 .32941 L
.41587 .32946 L
.41664 .32956 L
.41745 .32971 L
.41828 .32993 L
.4191 .33022 L
.41983 .33054 L
.42136 .33144 L
.42259 .33244 L
.42379 .33368 L
.42589 .33651 L
.42767 .33948 L
.4299 .3434 L
.43261 .3477 L
.4357 .35163 L
.43923 .35511 L
.44314 .35804 L
.44808 .36082 L
.45125 .36222 L
.45449 .3634 L
.46112 .36527 L
.46485 .36605 L
.469 .36673 L
.47267 .3672 L
.47682 .36759 L
.47919 .36775 L
Mistroke
.48138 .36787 L
.48263 .36792 L
.4838 .36796 L
.48503 .36799 L
.48637 .36802 L
.48756 .36803 L
.48868 .36803 L
.49078 .36802 L
.49304 .36797 L
.49432 .36793 L
.4955 .36789 L
.49767 .36778 L
.49973 .36766 L
.50355 .36736 L
.50761 .36695 L
.51194 .36641 L
.51942 .36516 L
.52592 .36371 L
.53313 .36164 L
.53905 .35946 L
.5504 .35368 L
.56844 .34036 L
.5765 .33478 L
.58111 .33214 L
.58634 .32964 L
.59174 .32756 L
.59687 .326 L
.60215 .32477 L
.6072 .32389 L
.60996 .32351 L
.61286 .32318 L
.61801 .32275 L
.62081 .32257 L
.62339 .32244 L
.6291 .32221 L
.6341 .32202 L
.63665 .32189 L
.63932 .32173 L
.64407 .32133 L
.6484 .32082 L
.65732 .31932 L
.66596 .31759 L
.67614 .31569 L
.6822 .31476 L
.68815 .31401 L
.69484 .31337 L
.69875 .31308 L
.70235 .31286 L
.70874 .31256 L
.71548 .31232 L
Mistroke
.7215 .3121 L
.72442 .31197 L
.72754 .31179 L
.73007 .3116 L
.7327 .31135 L
.73506 .31105 L
.73712 .31073 L
.74143 .30978 L
.74365 .30909 L
.74557 .30835 L
.74907 .30654 L
.75235 .30418 L
.76289 .29331 L
.76625 .2903 L
.77039 .28739 L
.77448 .28522 L
.77964 .28322 L
.78524 .28168 L
.79091 .28055 L
.80526 .27853 L
.81233 .27758 L
.82008 .27632 L
.82748 .27478 L
.83416 .27298 L
.84117 .27058 L
.8485 .2674 L
.86082 .26014 L
.87302 .24994 L
.88147 .24034 L
.88442 .23615 L
.88675 .23214 L
.88757 .23041 L
.88816 .2289 L
.88853 .22766 L
.88867 .22702 L
.88876 .22645 L
.88881 .22586 L
.88881 .22527 L
.88877 .22474 L
.88868 .22429 L
.88854 .22383 L
.88844 .22358 L
.88833 .22335 L
.88808 .22292 L
.88774 .2225 L
.88736 .22211 L
.88688 .22172 L
.88587 .22108 L
.88349 .21999 L
.8807 .21885 L
Mistroke
.87738 .21727 L
.87413 .21516 L
.87126 .21253 L
.8698 .2108 L
.86833 .20862 L
.86716 .2065 L
.86601 .20389 L
.86496 .20076 L
.86451 .19903 L
.86408 .19708 L
.86375 .19519 L
.86351 .19338 L
.86333 .1916 L
.86319 .18967 L
.86312 .18818 L
.86309 .18653 L
.8631 .18494 L
.86313 .18345 L
.86319 .1821 L
.86328 .18059 L
.8634 .17899 L
.86355 .17749 L
.86397 .17403 L
.86447 .17084 L
.86596 .16367 L
.86955 .15144 L
.87344 .14139 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P15o`<0AOl007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`14o`030?oo00<0@_l007co00@0ool02Ol00`3o0005o`030?oo04Co00<0ool00ol301?o
0P0Zo`00@ol303Oo0P0;o`801Ol00`3oo`14o`030?oo00Ko100:o`D000?o0?l0:Ol004;o00<0ool0
0P1:o`030?oo04Co00<0ool02_l:00Ko00<0ool0:?l0047o00<0ool00ol00`3oo`17o`030?oo04Co
00<0ool06_l00`3oo`0Xo`00@?l00`3oo`05o`<02Ol:03?o00<0ool0AOl00`3oo`0Ho`030?oo02Wo
000oo`030?oo00Wo2@09o`030?oo037o00<0ool0AOl00`3oo`0Ho`030?oo02Wo000no`030?oo01co
00<0ool0<Ol00`3oo`15o`030?oo01Oo00<0ool0:_l003go00<0ool07?l00`3oo`0bo`030?oo04Go
00<0ool05ol00`3oo`0Zo`00>ol201oo00<0ool0<_l00`3oo`15o`030?oo01Oo00<0ool0:_l003[o
00<0ool07_l00`3oo`0co`030?oo04Go00<0ool05_l00`3oo`0[o`00>Ol00`3oo`0Oo`030?oo03?o
00<0ool0AOl00`3oo`0Fo`030?oo02_o000go`808Ol00`3oo`0do`030?oo04Co0P0Ho`030?oo02_o
000fo`030?oo027o00<0ool0=?l00`3oo`14o`030?oo01Ko00<0ool0;?l003Go00<0ool08Ol00`3o
o`0Qo`801?l200?o1005o`030?oo04Co00<0ool05_l00`3oo`0/o`00<ol202Co00<0ool08?l0103o
o`08o`030?oo00Ko00<0ool0A?l00`3oo`0Fo`030?oo02co000ao`809_l00`3oo`0Po`040?oo00Wo
00<0ool01Ol204Co00<0ool05ol00`3oo`0/o`00;ol202Oo00<0ool06ol400;o00@0ool02_l00`3o
o`04o`030?oo04;o0P0Io`030?oo02co000^o`80:?l00`3oo`0Qo`040?oo00So00@0ool01Ol00`3o
o`11o`806_l00`3oo`0/o`00;Ol202Wo00<0ool08_l200[o0P06o`030?oo03ko0`0Lo`030?oo02co
000]o`030?oo02Oo00<0ool0=ol00`3oo`0no`030?oo01co00<0ool0;?l002go00<0ool09ol00`3o
o`0go`030?oo03go00<0ool07_l00`3oo`0[o`00;_l00`3oo`0Uo`030?oo03So00<0ool0??l00`3o
o`0Oo`030?oo02_o0007o`D00_l200Co0P0Ho`030?oo00So1@0Ho`030?oo00Oo0P04o`801?l202?o
00<0ool08_l200Co0P04o`803?l00`3oo`0Do`D01_l00`3oo`0Io`D00_l200Co0P03o`002Ol00`3o
o`07o`040?oo01So00<0ool02Ol00`3oo`0Go`030?oo00Oo00@0ool02?l0103oo`0Ro`030?oo027o
00@0ool02?l0103oo`0:o`030?oo01Oo00<0ool01ol00`3oo`0Jo`030?oo00Oo00@0ool00_l000Wo
00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3o
o`0Qo`040?oo00_o00<0ool02?l00`3oo`0Go`030?oo00So0P0Jo`030?oo00[o0@01o`7o00000ol0
000200Co00<0ool02?l201[o00<0ool01004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o
00<0ool08Ol0103oo`09o`802_l00`3oo`0Ho`030?oo00[o0P0Ho`030?oo00So0P03o`002Ol00`3o
o`08o`030?oo01Wo00<0ool02?l00`3oo`0Fo`030?oo00So00@0ool02Ol00`3oo`0Ro`030?oo027o
00@0ool02Ol00`3oo`08o`030?oo01Wo00<0ool03?l00`3oo`0Eo`030?oo00So00<0ool00_l000So
0P0:o`<06_l00`3oo`06o`805ol00`3oo`0:o`802_l302;o00<0ool08_l600Ko0`07o`030?oo01Wo
0P0>o`030?oo01Co0P0:o`<00_l0037o00<0ool07ol00`3oo`0ko`030?oo02?o0P03o`030?oo00go
00<0ool0:?l202Oo000bo`030?oo01go00<0ool0??l00`3oo`0So`030?oo00?o00<0ool02ol00`3o
o`0Yo`030?oo02Ko000bo`030?oo01co00<0ool0?Ol00`3oo`0Ro`030?oo00Go00<0ool02_l00`3o
o`0Xo`030?oo02Oo000co`030?oo01_o00<0ool03ol902Go00<0ool08_l00`3oo`06o`802Ol00`3o
o`0Xo`030?oo02So000do`030?oo01[o00<0ool02ol400Wo0P0So`030?oo02;o00<0ool02?l400?o
0P0[o`030?oo02So000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00?o00@0ool02?l00`3oo`06o`030?oo00Co0P03o`H02?l00`3oo`06o`030?l000Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o0006o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Wo`<06Ol400ko00<0ool02?l00`3oo`0Bo`L0
1_l00`3oo`0Qo`030?oo00Go00<0ool0:Ol00`3oo`05o`030?oo02;o00<0ool01ol003So00<0ool0
=Ol00`3oo`0>o`@01_l300?o00<0ool08Ol00`3oo`0ho`030?oo02go000io`@0<ol00`3oo`0<o`80
3Ol402?o00<0ool0=_l2033o000mo`80<_l00`3oo`08o`<04_l601ko00<0ool0<_l403;o000oo`<0
;ol00`3oo`06o`805Ol00`3oo`03o`806ol00`3oo`0bo`030?oo03Co0012o`<0;?l200?o100Go`03
0?oo00Go0`0Ho`030?oo02co1P0go`00AOl00`3oo`0Zo`D06_l00`3oo`08o`030?oo01Go00<0ool0
9Ol703go0016o`030?oo04So00<0ool02Ol00`3oo`0Do`030?oo02;o0`14o`00Aol00`3oo`17o`03
0?oo00[o0P0Do`030?oo027o00<0ool0AOl004So00<0ool0A_l00`3oo`0<o`030?oo017o00<0ool0
8?l00`3oo`16o`00B?l00`3oo`16o`030?oo00go00<0ool03ol00`3oo`0Po`030?oo04Oo0019o`03
0?oo04Go00<0ool03_l00`3oo`0>o`030?oo01oo00<0ool0B?l004Wo00<0ool0AOl00`3oo`0?o`03
0?oo00go00<0ool07_l00`3oo`19o`00B_l00`3oo`14o`030?oo013o0P0<o`030?oo01ko00<0ool0
B_l004[o00<0ool0<?l200Co0P03o`@01Ol00`3oo`0Bo`030?oo00Wo00<0ool05ol704go001;o`03
0?oo013o0`0Ko`040?oo00So00<0ool01_l00`3oo`0Co`030?oo00Oo00<0ool02_l>05Co001;o`03
0?oo00ko0P02o`<06Ol0103oo`09o`030?oo00Go0P0Eo`801_l200Oo1@1Ro`00C?l00`3oo`0;o`80
1_l201So00@0ool02_l00`3oo`04o`030?oo01Ko0P03o`X0Iol004co00<0ool02Ol200Wo00<0ool0
5_l0103oo`08o`040?oo00Go00<0ool05_l6073o001=o`030?oo00Ko0P0<o`030?oo013o1@000ol0
000:o`801_l00`3oo`0Co`<0M_l004ko00<0ool00ol200ko00<0ool03ol00`3oo`03o`030?oo01;o
00<0ool04_l00`3oo`1go`00Col5013o00<0ool03Ol200Oo00<0ool04Ol00`3oo`0@o`80N_l006Go
00<0ool02_l200Wo00<0ool04Ol00`3oo`0?o`030?oo07[o001Uo`030?oo00Oo0`0<o`030?oo013o
00<0ool03_l00`3oo`1ko`00IOl00`3oo`05o`804?l00`3oo`0?o`030?oo00co0P1no`00IOl200Go
00<0ool04?l00`3oo`0?o`030?oo00_o00<0ool0O_l006Ko0P001?l0000Do`030?oo00ko00<0ool0
2Ol2087o001Wo`<05ol300go00<0ool01_l308?o0024o`030?oo00[o00<0ool00ol308Ko0025oa80
ROl0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87365 .14329 m
.87528 .1382 L
.87645 .13429 L
.8769 .13247 L
.87706 .13168 L
.87716 .131 L
.87722 .1303 L
.87723 .12995 L
.87722 .12961 L
.87716 .12911 L
.87703 .12863 L
.87683 .12819 L
.87656 .12785 L
.87625 .12759 L
.87585 .12737 L
.87559 .12726 L
.87534 .12718 L
.87502 .12711 L
.87469 .12706 L
.87405 .127 L
.87365 .12699 L
.87327 .12699 L
.87252 .12702 L
.87176 .12707 L
.87085 .12716 L
.8698 .12729 L
.86493 .12802 L
.85823 .12902 L
.85418 .12951 L
.85182 .12975 L
.84954 .12995 L
.84733 .13011 L
.84526 .13023 L
.8429 .13033 L
.84165 .13037 L
.84032 .1304 L
.83811 .13042 L
.83602 .1304 L
.83372 .13035 L
.83242 .13031 L
.83123 .13025 L
.82898 .13013 L
.82659 .12996 L
.8223 .12955 L
.81772 .12897 L
.81277 .12818 L
.79695 .12461 L
.78421 .12092 L
.7818 .12026 L
.77952 .11972 L
Mistroke
.77851 .11952 L
.77763 .11939 L
.77689 .1193 L
.77614 .11925 L
.77574 .11925 L
.77532 .11926 L
.77491 .11929 L
.77454 .11934 L
.77393 .11947 L
.77335 .11969 L
.77304 .11984 L
.77273 .12004 L
.77248 .12023 L
.77223 .12047 L
.77181 .12098 L
.77148 .12153 L
.77123 .1221 L
.771 .12279 L
.77082 .1235 L
.77069 .12422 L
.77058 .12507 L
.77053 .12559 L
.77049 .12609 L
.77044 .12715 L
.77042 .12777 L
.77041 .12836 L
.77041 .12947 L
.77043 .13056 L
.77046 .13181 L
.77051 .13321 L
.77075 .13918 L
.7708 .14066 L
.77086 .14229 L
.7709 .14376 L
.77093 .14534 L
.77095 .14677 L
.77095 .14836 L
.77095 .14982 L
.77093 .15119 L
.77088 .15282 L
.77085 .15375 L
.77082 .1546 L
.77072 .15633 L
.77059 .15817 L
.77028 .16144 L
.7699 .16433 L
.76934 .16757 L
.76858 .17089 L
.76781 .17362 L
.76685 .17633 L
Mistroke
.76485 .1806 L
.76271 .18392 L
.75991 .18708 L
.75722 .18938 L
.75121 .1935 L
.74838 .19563 L
.74587 .19795 L
.74118 .20394 L
.73594 .21282 L
.72919 .22469 L
.72572 .23002 L
.72149 .23553 L
.71713 .24005 L
.71271 .24352 L
.71014 .24506 L
.70772 .24621 L
.70635 .24673 L
.70485 .24721 L
.70352 .24756 L
.70207 .24784 L
.70061 .24805 L
.69923 .24816 L
.69842 .24819 L
.69757 .2482 L
.69605 .24815 L
.69527 .24809 L
.69444 .248 L
.69294 .24779 L
.69116 .24744 L
.68954 .24704 L
.68666 .24613 L
.68356 .24492 L
.67806 .24225 L
.66726 .23602 L
.66255 .23355 L
.66063 .23274 L
.65964 .23239 L
.65862 .23209 L
.65758 .23186 L
.65665 .23172 L
.6557 .23166 L
.65487 .23168 L
.65393 .2318 L
.65342 .23191 L
.65295 .23204 L
.65211 .23236 L
.65125 .23279 L
.6504 .23335 L
.64951 .23408 L
.64797 .23573 L
Mistroke
.64624 .23826 L
.64473 .24108 L
.64222 .24716 L
.63964 .25495 L
.63496 .2712 L
.62923 .28951 L
.62552 .29875 L
.62174 .30592 L
.61787 .31118 L
.61552 .31349 L
.61323 .31518 L
.61207 .31585 L
.61094 .31638 L
.60991 .31678 L
.60879 .31712 L
.60755 .31738 L
.60638 .31753 L
.60511 .3176 L
.60438 .31759 L
.60369 .31756 L
.60245 .31743 L
.60109 .3172 L
.59979 .3169 L
.59856 .31655 L
.59364 .31458 L
.58804 .31148 L
.58183 .30738 L
.56913 .29806 L
.55542 .28846 L
.54293 .28129 L
.53187 .2766 L
.5265 .27488 L
.52102 .27345 L
.51035 .27128 L
.49106 .26696 L
.48155 .26443 L
.47878 .26381 L
.47574 .26321 L
.47304 .26279 L
.47148 .26259 L
.46997 .26243 L
.46854 .26231 L
.46778 .26226 L
.46696 .26221 L
.4655 .26216 L
.46412 .26213 L
.46328 .26213 L
.46235 .26214 L
.46147 .26216 L
.46065 .26219 L
Mistroke
.45908 .26227 L
.4574 .26239 L
.4559 .26253 L
.45427 .26272 L
.45127 .26314 L
.44426 .26453 L
.43677 .26654 L
.4298 .26878 L
.41462 .27433 L
.40834 .2766 L
.40296 .27829 L
.40028 .27897 L
.39798 .27942 L
.3968 .27959 L
.39622 .27966 L
.39561 .27971 L
.395 .27975 L
.39437 .27977 L
.39378 .27977 L
.39327 .27976 L
.39234 .27969 L
.39151 .27957 L
.39063 .2794 L
.38974 .27914 L
.38884 .27881 L
.38804 .27842 L
.38663 .27754 L
.38543 .27651 L
.38439 .27541 L
.38226 .27241 L
.37771 .26396 L
.37506 .25951 L
.37209 .25559 L
.36882 .25238 L
.36466 .24949 L
.36219 .24824 L
.35969 .24724 L
.35672 .24634 L
.35496 .24593 L
.3533 .24561 L
.35153 .24535 L
.34959 .24512 L
.34788 .24499 L
.34594 .24489 L
.34484 .24486 L
.34381 .24485 L
.34261 .24485 L
.34145 .24487 L
.33936 .24495 L
.33816 .24501 L
Mistroke
.33704 .24509 L
.33267 .24548 L
.32861 .24598 L
.31926 .24748 L
.30234 .25084 L
.29368 .25254 L
.28632 .25374 L
.28281 .25415 L
.28185 .25424 L
.28096 .25431 L
.27936 .2544 L
.27853 .25443 L
.27767 .25445 L
.27619 .25443 L
.27541 .2544 L
.27469 .25435 L
.27394 .25428 L
.27316 .25418 L
.27205 .25399 L
.27092 .25372 L
.26984 .25337 L
.26893 .25298 L
.26751 .25213 L
.26684 .25158 L
.2663 .25102 L
.26586 .25045 L
.26545 .24979 L
.26509 .249 L
.26482 .24822 L
.26471 .24779 L
.26461 .24729 L
.26448 .24636 L
.26443 .2454 L
.26445 .24448 L
.26452 .2435 L
.26464 .24255 L
.26478 .24167 L
.26498 .24069 L
.26627 .23625 L
.26835 .23101 L
.27098 .22538 L
.27595 .21567 L
.28745 .1883 L
.29429 .17258 L
.30291 .15622 L
.30452 .15264 L
.3052 .15082 L
.30568 .14923 L
.30587 .14845 L
.30601 .14764 L
Mistroke
.30607 .14721 L
.3061 .1468 L
.30611 .14645 L
.30611 .14607 L
.30607 .14544 L
.30602 .1451 L
.30595 .14481 L
.30578 .14426 L
.30557 .14378 L
.30531 .14336 L
.30498 .14294 L
.30456 .14253 L
.30411 .14219 L
.30316 .14167 L
.30254 .14143 L
.30192 .14123 L
.30113 .14104 L
.30035 .14089 L
.29942 .14076 L
.29892 .14071 L
.29835 .14066 L
.29736 .14061 L
.29623 .14057 L
.2951 .14057 L
.29401 .14059 L
.29271 .14064 L
.29142 .1407 L
.28889 .14086 L
.28265 .1414 L
.2767 .14195 L
.27008 .1425 L
.26654 .14273 L
.26466 .14283 L
.26263 .14292 L
.26078 .14298 L
.2591 .14302 L
.25741 .14304 L
.25582 .14304 L
.25401 .14302 L
.25301 .143 L
.2521 .14297 L
.25036 .1429 L
.24875 .1428 L
.24689 .14266 L
.24522 .14251 L
.24158 .14205 L
.23866 .14156 L
.23614 .14104 L
.23097 .13963 L
.22681 .13811 L
Mistroke
.22339 .13658 L
.21722 .13333 L
.21467 .13214 L
.21401 .1319 L
.21331 .13168 L
.21265 .13152 L
.21204 .13142 L
.2115 .13137 L
.21091 .13135 L
.21035 .13137 L
.20984 .13142 L
.20922 .13152 L
.20886 .1316 L
.20853 .13169 L
.20725 .13213 L
.20593 .13272 L
.2044 .13358 L
.20094 .13593 L
.18454 .14966 L
.16041 .16908 L
.14069 .1824 L
.13374 .18681 L
.13101 .18878 L
.12979 .1898 L
.12882 .19076 L
.12805 .19169 L
.1274 .19273 L
.12716 .1932 L
.12695 .19371 L
.12679 .1942 L
.12666 .19467 L
.12656 .19515 L
.12648 .19569 L
.12644 .1962 L
.12642 .1967 L
.12642 .1973 L
.12645 .19788 L
.12651 .19854 L
.12661 .19926 L
.12688 .20078 L
.12726 .20233 L
.12822 .20556 L
.13156 .21426 L
.14108 .23344 L
.14766 .2436 L
.1558 .25318 L
.16013 .25712 L
.16442 .26035 L
.17286 .26517 L
.19156 .27185 L
Mistroke
.19665 .27353 L
.20197 .27561 L
.21105 .28055 L
.21567 .28413 L
.21978 .28811 L
.2268 .297 L
.23352 .30782 L
.24023 .31907 L
.24308 .32321 L
.24627 .32703 L
.24818 .32887 L
.25001 .3303 L
.25181 .33144 L
.25284 .33196 L
.2538 .33237 L
.25555 .33295 L
.25653 .33318 L
.25749 .33335 L
.25838 .33346 L
.25923 .33352 L
.26017 .33354 L
.26118 .33352 L
.26221 .33345 L
.26335 .33331 L
.26444 .33314 L
.26545 .33294 L
.26798 .33231 L
.27047 .33152 L
.28161 .32682 L
.29261 .32178 L
.29737 .3199 L
.30225 .31836 L
.30473 .31779 L
.3061 .31754 L
.30736 .31736 L
.30849 .31725 L
.30951 .31718 L
.31067 .31716 L
.31174 .31719 L
.31278 .31727 L
.31391 .31741 L
.31506 .31763 L
.3161 .31789 L
.3179 .31851 L
.31973 .31938 L
.32164 .32059 L
.32351 .32212 L
.32651 .32544 L
.32928 .32962 L
.33156 .33388 L
Mistroke
.33631 .34367 L
.33863 .34766 L
.33996 .34951 L
.34148 .35122 L
.34292 .3525 L
.34463 .35365 L
.34543 .35407 L
.34632 .35445 L
.34712 .35474 L
.34801 .35498 L
.34898 .35518 L
.3499 .35531 L
.35094 .35539 L
.35154 .35541 L
.35212 .35541 L
.35318 .35536 L
.35418 .35527 L
.35532 .35513 L
.35655 .35491 L
.35877 .35441 L
.36094 .35379 L
.36616 .35194 L
.38962 .34109 L
.40157 .33602 L
.40701 .33429 L
.40841 .33394 L
.40987 .33362 L
.4112 .33338 L
.41239 .33322 L
.41302 .33315 L
.41367 .33309 L
.41439 .33305 L
.41502 .33302 L
.41561 .33302 L
.41624 .33303 L
.41682 .33306 L
.41734 .3331 L
.41829 .33322 L
.41882 .3333 L
.4193 .33339 L
.42024 .33362 L
.42121 .33392 L
.42301 .33467 L
.42459 .33557 L
.42731 .33769 L
.43235 .34322 L
.43835 .34989 L
.44159 .35275 L
.44564 .35564 L
.45086 .35846 L
Mistroke
.45654 .36071 L
.46244 .36241 L
.46562 .36312 L
.46925 .36378 L
.47248 .36425 L
.47603 .36466 L
.47947 .36496 L
.48265 .36516 L
.48455 .36524 L
.48633 .3653 L
.4883 .36534 L
.49043 .36535 L
.4915 .36535 L
.49266 .36534 L
.49479 .3653 L
.49685 .36524 L
.49876 .36516 L
.50254 .36494 L
.50593 .36467 L
.51336 .36385 L
.51743 .36325 L
.52107 .36262 L
.52883 .36092 L
.53515 .35913 L
.54126 .35696 L
.55115 .35233 L
.56039 .3467 L
.57648 .33632 L
.58639 .3314 L
.5924 .32912 L
.59816 .32738 L
.60375 .32605 L
.60903 .32506 L
.61451 .32427 L
.62044 .32361 L
.65792 .31884 L
.70496 .31169 L
.71739 .31096 L
.72382 .31054 L
.72682 .31029 L
.72948 .31002 L
.73437 .30935 L
.73677 .30891 L
.73917 .30837 L
.7428 .3073 L
.7464 .30586 L
.74983 .30403 L
.75271 .30212 L
.7588 .29717 L
.76436 .29257 L
Mistroke
.77128 .28806 L
.77527 .28612 L
.78016 .28426 L
.78508 .28284 L
.79009 .28169 L
.80272 .27953 L
.81017 .27836 L
.81732 .27709 L
.83092 .27376 L
.83801 .27131 L
.84535 .26809 L
.8575 .26089 L
.86709 .25312 L
.87582 .2439 L
.8815 .23603 L
.88349 .23245 L
.88432 .2306 L
.88488 .22905 L
.88509 .22831 L
.88526 .22755 L
.88533 .22715 L
.88537 .22678 L
.88541 .22645 L
.88543 .2261 L
.88543 .22578 L
.88543 .22544 L
.88537 .22486 L
.88528 .22432 L
.88515 .22385 L
.88497 .22336 L
.88476 .22291 L
.88428 .22212 L
.88362 .22132 L
.88277 .22047 L
.87894 .21733 L
.87637 .21504 L
.87359 .21191 L
.87135 .2085 L
.87027 .20639 L
.86919 .20384 L
.86836 .20139 L
.8676 .19855 L
.86724 .19689 L
.86695 .19532 L
.8667 .19367 L
.86651 .1921 L
.86636 .19061 L
.86624 .189 L
.86615 .18734 L
.8661 .18549 L
Mistroke
.86609 .18385 L
.86612 .18207 L
.86617 .18036 L
.86626 .17879 L
.8664 .17681 L
.86659 .17469 L
.86705 .17085 L
.8677 .1667 L
.86856 .16215 L
.87234 .14751 L
.87365 .14329 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P16o`<0A?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`15o`030?oo0080@_l007co00@0ool02Ol00`3o0005o`030?oo04Co0P04o`@04Ol302[o
0013o`80>?l200_o0P05o`030?oo04Co00<0ool01olA00;o00<0ool0:?l004;o1@1:o`030?oo04Co
00<0ool06_l00`3oo`0Xo`00@Ol00`3oo`03o`80B?l00`3oo`15o`030?oo01So00<0ool0:Ol0043o
00<0ool01_l600?o300co`030?oo04Go00<0ool06?l00`3oo`0Yo`00?_l200ko100;o`030?oo037o
00<0ool0AOl00`3oo`0Ho`030?oo02Wo000mo`030?oo01go00<0ool0<Ol00`3oo`15o`030?oo01So
00<0ool0:Ol003_o0P0Po`030?oo037o00<0ool0AOl00`3oo`0Ho`030?oo02Wo000jo`030?oo01oo
00<0ool0<_l00`3oo`15o`030?oo01So00<0ool0:Ol003Wo00<0ool08?l00`3oo`0bo`030?oo04Co
00<0ool06?l00`3oo`0Zo`00=ol202;o00<0ool0<ol00`3oo`14o`030?oo01So00<0ool0:_l003Ko
00<0ool08_l00`3oo`0co`030?oo04Co00<0ool05ol00`3oo`0[o`00=Ol00`3oo`0Ro`030?oo03Co
00<0ool0A?l00`3oo`0Go`030?oo02_o000co`809Ol00`3oo`0Po`801?l200?o1005o`030?oo04?o
0P0Io`030?oo02_o000bo`030?oo02Co00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0@ol00`3o
o`0Ho`030?oo02_o000ao`030?oo02Go00<0ool08?l0103oo`09o`030?oo00Go0P13o`806_l00`3o
o`0[o`00;ol202Oo00<0ool06ol400;o00@0ool02_l00`3oo`04o`030?oo04;o00<0ool06Ol00`3o
o`0[o`00;_l00`3oo`0Wo`030?oo027o00@0ool02?l0103oo`05o`030?oo047o00<0ool06_l00`3o
o`0[o`00;Ol00`3oo`0Xo`030?oo02;o0P0:o`801_l00`3oo`0oo`807Ol00`3oo`0[o`00;Ol00`3o
o`0Wo`030?oo03Oo00<0ool0?_l201ko00<0ool0:ol002go00<0ool09ol00`3oo`0go`030?oo03go
00<0ool07_l00`3oo`0[o`00;_l00`3oo`0Uo`030?oo03So00<0ool0?Ol00`3oo`0Oo`030?oo02[o
0007o`D00_l200Co0P0Ho`030?oo00So1@0Ho`030?oo00Oo0P04o`801?l202?o00<0ool08_l200Co
0P04o`803?l00`3oo`0Do`D01ol00`3oo`0Ho`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo
00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103o
o`0:o`030?oo01Oo00<0ool02?l00`3oo`0Io`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3o
o`0Fo`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o
00<0ool01ol00`3oo`0Ho`030?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`03
0?oo00So0P0Io`040?ooo`@01?l00`3oo`0Go`040?ooo`@00_l0103oo`09o`808ol00`3oo`0Qo`04
0?oo00Wo0P0:o`030?oo01So00<0ool02_l201So00<0ool02?l200?o0009o`030?oo00So00<0ool0
6Ol00`3oo`08o`030?oo01Ko00<0ool02?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo00So00<0ool06Ol00`3oo`0;o`030?oo01Ko00<0ool02?l00`3oo`02o`002?l200[o0`0Io`03
0?oo00Oo0P0Ho`030?oo00Wo0P0:o`<08_l00`3oo`0Ro`802_l300So00<0ool06?l200go00<0ool0
5Ol200[o0`02o`00<Ol00`3oo`0Oo`030?oo03_o00<0ool09Ol3017o00<0ool09ol00`3oo`0Vo`00
<Ol00`3oo`0No`030?oo03co00<0ool09?l200;o0P0=o`80:Ol00`3oo`0Wo`00<_l00`3oo`0Mo`03
0?oo03co00<0ool08ol00`3oo`04o`802_l00`3oo`0Xo`030?oo02So000co`030?oo01co00<0ool0
??l00`3oo`0So`030?oo00Ko00<0ool01_l00`3oo`0Xo`030?oo02Wo000do`030?oo01_o00<0ool0
2_l=02Go00<0ool08_l00`3oo`08o`@000Co0000:_l00`3oo`0Zo`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`05o`X0
2?l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?l000Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Wo`03
0?l002_o00<0ool01ol00`3oo`0Po`030?oo02;o00<0ool01?l00`3oo`0Yo`030?oo00Go00<0ool0
8_l00`3oo`07o`00=ol00`3oo`0eo`030?oo01;o1P07o`030?oo027o00<0ool0=ol202oo000ho`<0
=Ol00`3oo`0=o`@01_l400?o00<0ool08Ol00`3oo`0eo`80<Ol003_o0P0co`030?oo00[o0`0>o`@0
8_l00`3oo`0do`80<ol003go100`o`030?oo00Ko0`0Do`L07?l00`3oo`0`o`@0=Ol0047o0P0_o`03
0?oo00?o0P0Go`030?oo00Co0`0Io`030?oo02go0`0io`00@ol202ko1@0Io`030?oo00Oo0P0Go`03
0?oo02Ko1`0lo`00AOl00`3oo`19o`030?oo00Wo0P0Do`030?oo02Go0P13o`00A_l00`3oo`18o`03
0?oo00_o00<0ool04Ol00`3oo`0Ro`<0AOl004Oo00<0ool0Aol00`3oo`0<o`804Ol00`3oo`0Qo`03
0?oo04Ko0018o`030?oo04Ko00<0ool03_l00`3oo`0=o`030?oo027o00<0ool0Aol004Wo00<0ool0
AOl00`3oo`0?o`030?oo00co00<0ool08?l00`3oo`18o`00BOl00`3oo`15o`030?oo013o0P0;o`03
0?oo01oo0P1;o`00B_l00`3oo`14o`030?oo01;o0P09o`030?oo01go0P1=o`00B_l00`3oo`0`o`80
1?l200?o1005o`030?oo01Co00<0ool01Ol00`3oo`0Ao`d0Col004_o00<0ool0;_l0103oo`08o`03
0?oo00Ko00<0ool05Ol200Co00<0ool03?l605co001;o`030?oo00ko1P0Jo`040?oo00Wo00<0ool0
1Ol201So100:o`D0H_l004co00<0ool02Ol400Go0P0Io`040?oo00[o00<0ool01?l00`3oo`0Po`D0
Iol004go00<0ool01_l200[o00<0ool05ol0103oo`08o`040?oo00Go00<0ool06?l806co001>o`03
0?oo00?o0P0=o`030?oo01Oo0P0:o`801_l00`3oo`0Do`@0M?l004ko1P0@o`030?oo013o1P0Do`03
0?oo01;o0P1ho`00I?l00`3oo`0>o`801Ol201?o00<0ool04Ol00`3oo`1ho`00IOl00`3oo`0:o`<0
2Ol00`3oo`0@o`030?oo00oo0P1ko`00IOl00`3oo`08o`803Ol00`3oo`0?o`030?oo00ko00<0ool0
Nol006Ko00<0ool01Ol2013o00<0ool03_l00`3oo`0<o`80O_l006Ko0P05o`030?oo013o0P0?o`03
0?oo00Wo0`20o`00Iol601Go00<0ool03?l00`3oo`07o`80Pol008?o0`0;o`030?oo00Go0P25o`00
Q_l500Ko2027o`00Rol708ko002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`04
0?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87386 .14518 m
.87478 .14026 L
.87511 .13818 L
.87524 .13722 L
.87533 .13638 L
.87541 .13543 L
.87545 .13447 L
.87545 .13371 L
.87542 .13295 L
.87533 .13219 L
.8752 .13154 L
.87501 .13088 L
.87488 .13053 L
.87474 .13024 L
.87444 .12973 L
.87406 .12925 L
.87365 .12885 L
.87321 .12852 L
.87264 .12819 L
.87193 .1279 L
.87119 .12767 L
.87044 .1275 L
.86967 .12737 L
.8689 .12728 L
.868 .12722 L
.86696 .12717 L
.86642 .12716 L
.8658 .12715 L
.86463 .12716 L
.86343 .12719 L
.86226 .12723 L
.85691 .12748 L
.85377 .12763 L
.85026 .12777 L
.8483 .12783 L
.84724 .12786 L
.8461 .12788 L
.84503 .1279 L
.84405 .1279 L
.8421 .12791 L
.84011 .12789 L
.83898 .12787 L
.83792 .12785 L
.83597 .12779 L
.83388 .1277 L
.83169 .12759 L
.82968 .12746 L
.82514 .12708 L
.82093 .12662 L
.81697 .12611 L
Mistroke
.79971 .12297 L
.79324 .12157 L
.7879 .12047 L
.78529 .12002 L
.78401 .11985 L
.78335 .11977 L
.78275 .11971 L
.78174 .11964 L
.78122 .11962 L
.78069 .11961 L
.77979 .11963 L
.77901 .11969 L
.77826 .1198 L
.77752 .11996 L
.77692 .12014 L
.77631 .12039 L
.7757 .1207 L
.77509 .12111 L
.77462 .12151 L
.77414 .12201 L
.77338 .12309 L
.77302 .12377 L
.77272 .12445 L
.77226 .12581 L
.77186 .12744 L
.77167 .12843 L
.77149 .12959 L
.77135 .13078 L
.77121 .13212 L
.77101 .13467 L
.77086 .13722 L
.77056 .14383 L
.77038 .14727 L
.77017 .15051 L
.7696 .15683 L
.76919 .16011 L
.76866 .1635 L
.76758 .16871 L
.76606 .17395 L
.76428 .17844 L
.76213 .18251 L
.76012 .18548 L
.75778 .18829 L
.75224 .19365 L
.74712 .1987 L
.73654 .21346 L
.72951 .22438 L
.72597 .22928 L
.72171 .2344 L
.71778 .23833 L
Mistroke
.71324 .24192 L
.71061 .24356 L
.70813 .24482 L
.70559 .24586 L
.70312 .24661 L
.70171 .24694 L
.70042 .24717 L
.6997 .24727 L
.69893 .24737 L
.69751 .24748 L
.69617 .24753 L
.69476 .24752 L
.69354 .24747 L
.69221 .24736 L
.6914 .24726 L
.69064 .24716 L
.68921 .24692 L
.686 .24621 L
.6803 .24443 L
.67035 .24048 L
.66789 .23953 L
.66533 .23864 L
.66301 .23799 L
.66193 .23776 L
.66095 .23758 L
.66012 .23748 L
.65924 .2374 L
.65829 .23738 L
.65744 .2374 L
.65657 .23748 L
.65563 .23764 L
.65477 .23785 L
.65399 .2381 L
.65236 .23883 L
.65146 .23936 L
.65063 .23993 L
.64731 .24314 L
.64566 .24533 L
.64416 .24765 L
.64086 .25389 L
.63489 .2678 L
.62918 .28137 L
.62227 .29467 L
.61785 .30079 L
.61579 .303 L
.61345 .30507 L
.61116 .30668 L
.60895 .30785 L
.60769 .30836 L
.60653 .30875 L
Mistroke
.60589 .30893 L
.60518 .30909 L
.60387 .30933 L
.60268 .30946 L
.60155 .30951 L
.60029 .3095 L
.59892 .30941 L
.59823 .30934 L
.59747 .30923 L
.59607 .30897 L
.5934 .3083 L
.59082 .30743 L
.58802 .3063 L
.58284 .3038 L
.55742 .28894 L
.54392 .28228 L
.53731 .27974 L
.53029 .27759 L
.52426 .27616 L
.5181 .27502 L
.50774 .27349 L
.48972 .27024 L
.48003 .26816 L
.47723 .26764 L
.47417 .26714 L
.47145 .26678 L
.46999 .26662 L
.46842 .26648 L
.46704 .26638 L
.46631 .26633 L
.46551 .26629 L
.4641 .26624 L
.46277 .26621 L
.46129 .2662 L
.45988 .26622 L
.4586 .26626 L
.45719 .26632 L
.45561 .26642 L
.45386 .26656 L
.45025 .26695 L
.44685 .26745 L
.44361 .26802 L
.41905 .27447 L
.41285 .27627 L
.40741 .27767 L
.40458 .27827 L
.40316 .27853 L
.4017 .27875 L
.4005 .27889 L
.39988 .27895 L
Mistroke
.39923 .279 L
.39811 .27906 L
.3971 .27907 L
.39646 .27906 L
.39588 .27904 L
.39526 .279 L
.39462 .27893 L
.39407 .27886 L
.39357 .27879 L
.39245 .27856 L
.39148 .2783 L
.39063 .27802 L
.3888 .27721 L
.38731 .27633 L
.38577 .27519 L
.38288 .27238 L
.37725 .26524 L
.3712 .25833 L
.3674 .25515 L
.36515 .25365 L
.36257 .25222 L
.36013 .25111 L
.35725 .25006 L
.35445 .24927 L
.35169 .24867 L
.35 .24838 L
.3481 .24812 L
.34626 .24793 L
.34452 .24779 L
.34292 .2477 L
.34112 .24764 L
.3392 .24762 L
.33737 .24764 L
.3364 .24766 L
.33532 .24769 L
.33419 .24774 L
.33314 .24779 L
.32915 .24806 L
.32474 .24847 L
.31997 .24902 L
.30348 .25121 L
.29922 .25173 L
.2949 .25217 L
.29319 .2523 L
.29141 .25242 L
.29041 .25247 L
.28951 .25251 L
.28863 .25253 L
.2878 .25255 L
.28625 .25254 L
Mistroke
.2849 .25251 L
.28417 .25247 L
.28339 .25242 L
.28204 .25229 L
.28083 .25214 L
.27961 .25193 L
.27758 .25144 L
.27654 .2511 L
.27549 .25069 L
.27379 .2498 L
.27288 .24917 L
.2721 .24851 L
.27147 .24787 L
.27087 .24713 L
.27038 .24641 L
.26999 .24572 L
.26937 .24426 L
.26912 .24343 L
.2689 .2425 L
.26882 .24201 L
.26874 .24148 L
.26864 .2405 L
.26861 .23994 L
.26859 .23943 L
.26859 .23885 L
.2686 .23823 L
.26865 .23717 L
.2687 .23656 L
.26876 .23598 L
.26906 .23387 L
.26952 .23155 L
.27074 .2269 L
.27219 .22247 L
.27524 .21446 L
.28675 .18737 L
.29526 .17188 L
.29999 .16373 L
.30371 .15646 L
.30504 .15307 L
.30551 .15148 L
.30571 .15062 L
.30585 .14985 L
.3059 .14946 L
.30594 .14904 L
.30597 .14832 L
.30595 .14764 L
.30588 .14704 L
.30575 .14643 L
.30565 .1461 L
.30554 .14579 L
Mistroke
.30528 .14521 L
.30491 .14463 L
.30452 .14416 L
.30411 .14375 L
.30355 .14331 L
.30297 .14294 L
.30174 .14236 L
.30097 .14208 L
.30019 .14185 L
.29942 .14167 L
.29856 .1415 L
.2969 .14126 L
.29591 .14116 L
.29493 .14109 L
.29382 .14103 L
.29258 .14099 L
.29134 .14097 L
.29062 .14096 L
.28993 .14096 L
.2885 .14098 L
.28768 .141 L
.28691 .14102 L
.28411 .14111 L
.28103 .14125 L
.27358 .14162 L
.26983 .14179 L
.26817 .14185 L
.26637 .14191 L
.26468 .14195 L
.26312 .14198 L
.26163 .142 L
.26006 .14201 L
.25842 .14201 L
.25689 .14199 L
.25523 .14195 L
.2543 .14193 L
.25346 .1419 L
.25014 .14173 L
.2483 .1416 L
.24663 .14146 L
.24366 .14114 L
.24062 .14072 L
.2349 .13965 L
.2302 .13846 L
.22147 .13558 L
.21817 .13446 L
.21665 .13402 L
.21532 .13369 L
.21465 .13356 L
.21393 .13344 L
Mistroke
.21318 .13335 L
.21247 .13329 L
.21209 .13327 L
.21168 .13327 L
.21125 .13327 L
.21084 .13328 L
.21012 .13333 L
.2097 .13338 L
.20932 .13343 L
.20856 .13356 L
.20783 .13372 L
.20643 .13412 L
.20487 .1347 L
.2031 .13549 L
.1962 .13957 L
.17598 .15463 L
.15456 .17041 L
.14449 .17739 L
.1359 .18353 L
.13269 .18618 L
.13029 .1886 L
.12869 .19078 L
.128 .19206 L
.1275 .19328 L
.1273 .1939 L
.12713 .19458 L
.12699 .19524 L
.1269 .19587 L
.12683 .19644 L
.12679 .19709 L
.12676 .19778 L
.12677 .19845 L
.1268 .19925 L
.12686 .20002 L
.12694 .20074 L
.12705 .20154 L
.12738 .20341 L
.1278 .20524 L
.13036 .21341 L
.13397 .22221 L
.13949 .23323 L
.14591 .24356 L
.1531 .25251 L
.15726 .25664 L
.16194 .26051 L
.17072 .26604 L
.17927 .26997 L
.18877 .2736 L
.19866 .27766 L
.20365 .2802 L
Mistroke
.20892 .2835 L
.21757 .29075 L
.23284 .30943 L
.23907 .31741 L
.24242 .32107 L
.24611 .32434 L
.24796 .32565 L
.25005 .32688 L
.25202 .3278 L
.25392 .32849 L
.25489 .32876 L
.25593 .329 L
.25684 .32917 L
.25782 .32931 L
.2589 .32942 L
.25992 .32948 L
.26055 .3295 L
.26115 .3295 L
.26228 .32947 L
.26342 .3294 L
.26407 .32935 L
.26468 .32928 L
.2673 .3289 L
.26963 .32843 L
.27218 .3278 L
.28383 .32416 L
.28922 .32242 L
.29492 .3208 L
.29814 .32007 L
.30101 .31957 L
.30237 .31939 L
.30307 .31932 L
.30382 .31925 L
.30446 .31921 L
.30517 .31917 L
.30639 .31914 L
.30702 .31914 L
.3077 .31915 L
.30893 .31921 L
.3101 .31932 L
.31116 .31947 L
.31229 .31967 L
.31351 .31995 L
.31588 .32069 L
.31785 .32153 L
.31987 .32262 L
.32326 .32508 L
.32653 .32824 L
.3293 .33157 L
.33503 .33965 L
Mistroke
.33787 .34352 L
.34069 .34676 L
.34236 .34833 L
.34426 .3498 L
.3461 .35092 L
.34791 .35175 L
.34886 .3521 L
.34979 .35238 L
.35066 .3526 L
.3516 .35279 L
.35266 .35295 L
.35366 .35304 L
.35477 .3531 L
.35541 .35312 L
.35603 .35311 L
.35707 .35308 L
.35819 .353 L
.35917 .3529 L
.36026 .35276 L
.36278 .35232 L
.36521 .35177 L
.37673 .34814 L
.38979 .3432 L
.40189 .33908 L
.40499 .33824 L
.40815 .33753 L
.41073 .33708 L
.41215 .33689 L
.41282 .33682 L
.41344 .33676 L
.4146 .33668 L
.4152 .33666 L
.41583 .33664 L
.41698 .33664 L
.41799 .33668 L
.41896 .33675 L
.41998 .33687 L
.42091 .33701 L
.42175 .33717 L
.42357 .33763 L
.42547 .33829 L
.42736 .33913 L
.42905 .34005 L
.44088 .34897 L
.44462 .35161 L
.44911 .35426 L
.45349 .35633 L
.45879 .35829 L
.46488 .35994 L
.47113 .36115 L
Mistroke
.47496 .36169 L
.47914 .36212 L
.4828 .3624 L
.48491 .36251 L
.48593 .36256 L
.48688 .36259 L
.48875 .36264 L
.48974 .36266 L
.49079 .36268 L
.49256 .36269 L
.49451 .36267 L
.49625 .36264 L
.49812 .36259 L
.49988 .36253 L
.50149 .36245 L
.50532 .36222 L
.50945 .36188 L
.51678 .36103 L
.52417 .35983 L
.53071 .35844 L
.54179 .3552 L
.55218 .35092 L
.56223 .34564 L
.58004 .33596 L
.59111 .33135 L
.59671 .32953 L
.60211 .32806 L
.61295 .32581 L
.62262 .32434 L
.63267 .32302 L
.64273 .32153 L
.66146 .31766 L
.68087 .31364 L
.69188 .31199 L
.70418 .31068 L
.71554 .30974 L
.72544 .30881 L
.73052 .30815 L
.73555 .30725 L
.74067 .30596 L
.74492 .30449 L
.74864 .30284 L
.75242 .3008 L
.75899 .29658 L
.76623 .2919 L
.77394 .28788 L
.7783 .28612 L
.78338 .28446 L
.7955 .28159 L
.80728 .27939 L
Mistroke
.82051 .27658 L
.82748 .27464 L
.83507 .27197 L
.84186 .26896 L
.84783 .26574 L
.85877 .25816 L
.86864 .24895 L
.87532 .24089 L
.87809 .23677 L
.88001 .23335 L
.88076 .23177 L
.88141 .23015 L
.88187 .22871 L
.88216 .22746 L
.88227 .22681 L
.88233 .22622 L
.88237 .22559 L
.88238 .22524 L
.88237 .22493 L
.88234 .22431 L
.8823 .22397 L
.88225 .22366 L
.88214 .2231 L
.88199 .22251 L
.88163 .22147 L
.88112 .22036 L
.87992 .21831 L
.87703 .21394 L
.87536 .21111 L
.8736 .20751 L
.87265 .20516 L
.87185 .20281 L
.87117 .2004 L
.87053 .19765 L
.86998 .1946 L
.86958 .19161 L
.86941 .18988 L
.86928 .18822 L
.86917 .18636 L
.86913 .1853 L
.8691 .18431 L
.86908 .18263 L
.86908 .18104 L
.86911 .17928 L
.86914 .17828 L
.86918 .17737 L
.86927 .17552 L
.86941 .1735 L
.86973 .16982 L
.87049 .16369 L
Mistroke
.87154 .15713 L
.87386 .14518 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P17o`<0@ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`15o`800_l603go001lo`040?oo00Wo00<0o`001Ol00`3oo`15o`030?oo00Oo2003o`H0
;?l007go0P0;o`801Ol00`3oo`15o`030?oo00ko1005o`80:ol004;o1@1:o`030?oo04Go00<0ool0
6?l00`3oo`0Yo`00@_l0103oool404Oo00<0ool0AOl00`3oo`0Ho`030?oo02Wo0010o`802?lC03Co
00<0ool0A?l00`3oo`0Io`030?oo02Wo000no`807?l203?o00<0ool0A?l00`3oo`0Io`030?oo02Wo
000mo`030?oo01go00<0ool0<Ol00`3oo`14o`030?oo01Wo00<0ool0:Ol003_o0P0Po`030?oo037o
00<0ool0A?l00`3oo`0Ho`030?oo02[o000jo`030?oo01oo00<0ool0<_l00`3oo`14o`030?oo01So
00<0ool0:_l003Wo00<0ool08?l00`3oo`0bo`030?oo04Co00<0ool06?l00`3oo`0Zo`00=ol202;o
00<0ool0<ol00`3oo`14o`030?oo01So00<0ool0:_l003Ko00<0ool08_l00`3oo`0co`030?oo04Co
00<0ool06?l00`3oo`0Zo`00=Ol00`3oo`0Ro`030?oo03Co00<0ool0@ol00`3oo`0Io`030?oo02[o
000co`809Ol00`3oo`0Po`801?l200?o1005o`030?oo04?o00<0ool06Ol00`3oo`0Zo`00<_l00`3o
o`0To`030?oo023o00@0ool02?l00`3oo`06o`030?oo04?o00<0ool06Ol00`3oo`0Zo`00<?l202Oo
00<0ool08?l0103oo`09o`030?oo00Go0P13o`030?oo01[o00<0ool0:_l002oo00<0ool09_l00`3o
o`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0@_l00`3oo`0Jo`030?oo02[o000^o`030?oo02Oo
00<0ool08Ol0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Ko`030?oo02[o000]o`80:?l00`3o
o`0So`802_l200Ko00<0ool0@?l00`3oo`0Lo`030?oo02[o000]o`030?oo02Oo00<0ool0=ol00`3o
o`0no`807ol00`3oo`0Zo`00;Ol00`3oo`0Wo`030?oo03Oo00<0ool0?Ol00`3oo`0Oo`030?oo02[o
000]o`030?oo02Ko00<0ool0>?l00`3oo`0lo`030?oo027o00<0ool0:Ol000Oo1@02o`801?l201So
00<0ool02?l501So00<0ool01ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0<o`030?oo01Co
1@08o`030?oo01Oo1@02o`801?l200?o0009o`030?oo00Oo00@0ool05ol00`3oo`0:o`030?oo01Oo
00<0ool01ol0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00[o00<0ool05ol00`3o
o`08o`806_l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Oo
00<0ool01ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00So00<0ool05ol00`3o
o`09o`030?oo01So00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806Ol0103oool400Co
00<0ool05ol0103oool400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802_l00`3oo`0Ho`03
0?oo00[o00<0ool05ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool0
5_l00`3oo`08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3oo`0Ho`03
0?oo00[o00<0ool05ol00`3oo`08o`030?oo00;o0008o`802_l301Wo00<0ool01ol201So00<0ool0
2Ol200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`803?l00`3oo`0Fo`802_l300;o000`o`03
0?oo023o00<0ool0>ol00`3oo`0ho`030?oo02Oo00<0ool09ol0037o00<0ool07ol00`3oo`0ko`03
0?oo03Oo00<0ool09ol202Wo000bo`030?oo01ko00<0ool0>ol00`3oo`0Uo`H02ol00`3oo`0Xo`03
0?oo02So000bo`030?oo01ko00<0ool0>ol00`3oo`0To`030?oo00Co00<0ool01ol00`3oo`0Xo`03
0?oo02Wo000co`030?oo01go0P0>o`H0:?l00`3oo`0So`030?oo00Ko2@0[o`030?oo02Wo000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00;o00D0oooo0008o`03
0?oo00Ko00<0ool01Ol0103oool600So00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o0006o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`07o`003?l00`3oo`0Vo`040?oo02_o00<0ool01ol00`3oo`0Po`030?oo027o00<0ool01Ol00`3o
o`0Yo`030?oo00Co00<0ool08ol00`3oo`07o`00=_l203Oo00<0ool07ol00`3oo`0Qo`030?oo03Ko
0P0`o`00>?l203Ko00<0ool04Ol600Oo00<0ool08?l00`3oo`0eo`80<_l003[o0`0co`030?oo00ko
0`06o`P08_l00`3oo`0do`030?oo03;o000mo`80<_l00`3oo`09o`@04?l401oo00<0ool0<_l203Go
000oo`80<Ol200Co1@0Do`040?ooo`D06Ol00`3oo`0^o`D0=ol0047o0P0ao`@06Ol00`3oo`06o`80
5ol00`3oo`0Xo`H0??l004?o00<0ool0Bol00`3oo`08o`<04ol00`3oo`0Wo`80@_l004Co00<0ool0
B_l00`3oo`0;o`804Ol00`3oo`0So`@0A?l004Go00<0ool0BOl00`3oo`0=o`803_l00`3oo`0Ro`80
B?l004Ko00<0ool0B?l00`3oo`0?o`803?l00`3oo`0Qo`030?oo04So0017o`030?oo04Oo00<0ool0
4Ol200Wo00<0ool08_l00`3oo`18o`00Aol00`3oo`17o`030?oo01?o00<0ool01Ol2027o0`1;o`00
B?l00`3oo`16o`030?oo01Co0P03o`<07Ol504ko0019o`030?oo037o0P04o`800ol400Go00<0ool0
5Ol501Go2P1Co`00B_l00`3oo`0_o`040?oo00So00<0ool01_l00`3oo`0Zo`D0GOl004_o00<0ool0
3_l301go00@0ool02Ol00`3oo`05o`809_l506;o001<o`030?oo00Wo1@02o`<06_l0103oo`0:o`03
0?oo00Co00<0ool07ol606Oo001=o`801ol200Wo00<0ool06?l0103oo`08o`040?oo00Go00<0ool0
6_l506go001?o`L03?l00`3oo`0Ho`802_l200Ko00<0ool05_l407;o001So`030?oo02_o00<0ool0
5?l207Ko001To`030?oo013o1`0Co`030?oo017o0`1ho`00IOl00`3oo`0=o`801_l201;o00<0ool0
3ol207_o001Uo`030?oo00[o0`0:o`804?l00`3oo`0=o`80OOl006Ko0P07o`<03ol00`3oo`0=o`03
0?oo00_o0P1oo`00Iol801?o00<0ool03?l00`3oo`09o`80POl008?o0P0<o`030?oo00Oo0P23o`00
QOl300Wo00<0ool00_l508Go0028o`h0R_l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
1Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87407 .14708 m
.87415 .14582 L
.87421 .14471 L
.87426 .14347 L
.87429 .14233 L
.8743 .14131 L
.87429 .14041 L
.87426 .13944 L
.87421 .13846 L
.87412 .13752 L
.874 .13656 L
.87385 .13569 L
.87368 .13494 L
.87345 .13411 L
.87314 .13325 L
.87245 .13185 L
.872 .13118 L
.87152 .1306 L
.87049 .12963 L
.86917 .12875 L
.86833 .12833 L
.86748 .12798 L
.86594 .12748 L
.86404 .12704 L
.863 .12686 L
.86184 .12669 L
.85958 .12644 L
.84764 .12578 L
.84025 .12545 L
.83632 .12523 L
.83188 .12492 L
.82374 .12418 L
.81613 .12328 L
.80248 .12134 L
.79624 .12046 L
.79312 .12009 L
.79167 .11995 L
.79041 .11985 L
.78928 .11978 L
.78825 .11973 L
.78719 .11971 L
.7861 .11972 L
.78502 .11977 L
.78449 .11981 L
.78392 .11987 L
.78296 .11999 L
.7821 .12015 L
.78116 .12038 L
.7802 .12069 L
.7794 .12102 L
Mistroke
.7786 .12142 L
.77722 .12235 L
.7761 .1234 L
.77487 .125 L
.7739 .12678 L
.77315 .1286 L
.77245 .13079 L
.77145 .13526 L
.7707 .13994 L
.76916 .15173 L
.76826 .15782 L
.76724 .16344 L
.7658 .16955 L
.76423 .17465 L
.7607 .18279 L
.75867 .18626 L
.75627 .1897 L
.74728 .20065 L
.73531 .21683 L
.729 .2253 L
.72105 .23424 L
.7169 .23796 L
.71221 .24135 L
.70757 .24389 L
.70321 .24559 L
.70059 .24631 L
.69814 .24679 L
.69686 .24697 L
.69544 .24712 L
.69473 .24718 L
.69396 .24722 L
.69258 .24726 L
.69127 .24726 L
.69054 .24724 L
.68986 .24721 L
.68861 .24714 L
.68729 .24702 L
.68501 .24675 L
.68251 .24636 L
.67308 .24441 L
.6707 .24392 L
.66821 .24347 L
.66703 .24329 L
.66592 .24315 L
.66494 .24306 L
.66389 .24298 L
.66273 .24294 L
.66216 .24294 L
.66153 .24295 L
.66044 .24301 L
Mistroke
.65942 .24311 L
.65881 .24319 L
.65823 .24329 L
.65716 .24352 L
.65602 .24383 L
.6548 .24425 L
.65285 .24514 L
.65081 .24635 L
.64727 .24924 L
.64342 .2535 L
.63996 .25825 L
.63209 .27092 L
.62428 .28322 L
.62021 .2886 L
.6163 .29286 L
.61218 .29636 L
.60808 .29887 L
.60575 .29989 L
.60355 .3006 L
.60238 .30089 L
.60108 .30114 L
.60041 .30123 L
.5997 .30132 L
.59839 .30143 L
.59702 .30147 L
.59574 .30144 L
.595 .3014 L
.59432 .30135 L
.59353 .30128 L
.59279 .30119 L
.58987 .30069 L
.5882 .3003 L
.58664 .29989 L
.5751 .29563 L
.56314 .29025 L
.55012 .28477 L
.54388 .28258 L
.53804 .28085 L
.53242 .2795 L
.52746 .27854 L
.52237 .27776 L
.51713 .27712 L
.50815 .27622 L
.48914 .27373 L
.48003 .27218 L
.47483 .27139 L
.4698 .27079 L
.46708 .27055 L
.46566 .27046 L
.4641 .27037 L
Mistroke
.4632 .27033 L
.46237 .2703 L
.46157 .27028 L
.46071 .27026 L
.45922 .27025 L
.45837 .27025 L
.45756 .27026 L
.45604 .27029 L
.45458 .27034 L
.45327 .2704 L
.45182 .27049 L
.44854 .27076 L
.44545 .27109 L
.43909 .27199 L
.43219 .27324 L
.41997 .27579 L
.41449 .27689 L
.41162 .27741 L
.40902 .27781 L
.4067 .2781 L
.40571 .2782 L
.40466 .27829 L
.40352 .27836 L
.40246 .27841 L
.40128 .27843 L
.40021 .27842 L
.39914 .27838 L
.39805 .27831 L
.39713 .27821 L
.39615 .27808 L
.39512 .27791 L
.39404 .27767 L
.39217 .27715 L
.39035 .27648 L
.38871 .27573 L
.38578 .274 L
.3794 .26883 L
.3721 .2623 L
.36856 .25959 L
.36435 .25693 L
.35915 .25443 L
.35371 .25258 L
.35056 .2518 L
.34695 .25113 L
.34524 .25088 L
.34339 .25066 L
.34162 .25048 L
.33996 .25035 L
.33818 .25025 L
.33726 .2502 L
Mistroke
.33624 .25016 L
.3345 .25012 L
.33265 .25009 L
.33095 .25009 L
.32912 .25011 L
.32738 .25014 L
.32579 .25019 L
.31237 .25081 L
.30918 .25096 L
.30582 .2511 L
.3049 .25112 L
.30394 .25115 L
.30224 .25118 L
.30068 .25119 L
.29901 .25118 L
.29809 .25116 L
.29723 .25114 L
.29566 .25108 L
.29479 .25103 L
.29401 .25098 L
.29228 .25083 L
.29094 .25068 L
.28951 .25048 L
.28679 .24995 L
.28446 .24932 L
.28218 .24847 L
.28024 .24752 L
.27866 .24651 L
.27726 .24539 L
.27592 .24401 L
.27473 .24243 L
.27382 .2408 L
.27313 .23919 L
.27263 .23763 L
.27221 .23584 L
.27206 .23491 L
.27192 .23387 L
.27187 .23332 L
.27182 .23272 L
.27176 .23161 L
.27173 .23052 L
.27174 .2295 L
.27178 .22839 L
.27181 .22775 L
.27185 .22715 L
.27195 .22595 L
.2721 .22465 L
.27243 .22229 L
.27282 .22006 L
.27373 .21594 L
Mistroke
.27597 .20815 L
.27874 .20069 L
.28481 .18869 L
.29378 .17499 L
.29898 .16707 L
.30156 .16271 L
.30358 .15875 L
.30493 .1554 L
.30522 .15447 L
.30545 .15362 L
.30562 .15288 L
.30576 .15208 L
.30585 .15129 L
.30588 .15087 L
.30589 .15048 L
.30588 .14981 L
.30581 .14911 L
.30575 .14875 L
.30567 .14836 L
.30545 .14762 L
.30517 .14696 L
.30485 .14639 L
.3045 .14589 L
.30407 .14537 L
.30312 .14453 L
.30201 .1438 L
.30082 .14321 L
.29935 .14266 L
.29753 .14216 L
.29653 .14195 L
.2954 .14175 L
.29322 .14145 L
.29196 .14132 L
.29078 .14122 L
.28936 .14113 L
.28799 .14106 L
.28668 .14101 L
.28544 .14098 L
.28405 .14095 L
.28256 .14093 L
.28101 .14093 L
.28014 .14093 L
.27933 .14093 L
.27774 .14093 L
.27622 .14095 L
.27272 .14098 L
.27077 .14099 L
.26982 .141 L
.26894 .14101 L
.26747 .14101 L
Mistroke
.26589 .14102 L
.26507 .14101 L
.26417 .14101 L
.26253 .141 L
.26105 .14098 L
.25948 .14096 L
.25801 .14093 L
.25668 .14089 L
.25347 .14076 L
.25169 .14067 L
.25008 .14057 L
.24695 .14034 L
.24419 .14008 L
.23911 .13946 L
.23407 .13866 L
.22605 .13705 L
.22208 .13622 L
.2201 .13584 L
.21807 .13551 L
.21634 .1353 L
.21587 .13526 L
.21537 .13521 L
.21493 .13518 L
.21446 .13516 L
.21362 .13513 L
.21274 .13512 L
.21193 .13514 L
.21117 .13518 L
.2103 .13525 L
.20937 .13536 L
.20858 .13548 L
.20769 .13564 L
.20568 .13611 L
.20375 .13669 L
.2001 .1381 L
.19589 .14012 L
.17828 .15107 L
.15757 .16541 L
.14754 .17254 L
.13865 .17935 L
.13508 .18253 L
.13232 .18545 L
.13034 .18808 L
.12951 .18946 L
.12874 .191 L
.12816 .19249 L
.12773 .19392 L
.12754 .19472 L
.1274 .19548 L
.12726 .19638 L
Mistroke
.12717 .19722 L
.12712 .19797 L
.12709 .19878 L
.12708 .19948 L
.1271 .20027 L
.12714 .20113 L
.12721 .20207 L
.12731 .20298 L
.12741 .20383 L
.12773 .2058 L
.12811 .20766 L
.12925 .21204 L
.13222 .22062 L
.13705 .23131 L
.14279 .24139 L
.14933 .25043 L
.15719 .25867 L
.16455 .26443 L
.17315 .26952 L
.19279 .27851 L
.21004 .28866 L
.21792 .29525 L
.22601 .30324 L
.2333 .31077 L
.24017 .31706 L
.24401 .3199 L
.2483 .32237 L
.25026 .32325 L
.25237 .32403 L
.25422 .32458 L
.25628 .32504 L
.25754 .32526 L
.25875 .32542 L
.25985 .32554 L
.26106 .32562 L
.26169 .32565 L
.26239 .32567 L
.26313 .32568 L
.26382 .32568 L
.26508 .32565 L
.26578 .32562 L
.26644 .32559 L
.2677 .3255 L
.26889 .32539 L
.27162 .32507 L
.2767 .32427 L
.28204 .3233 L
.28788 .32225 L
.29055 .32182 L
.29336 .32144 L
Mistroke
.29573 .32119 L
.29707 .32108 L
.29831 .321 L
.29949 .32095 L
.30076 .32092 L
.30195 .32093 L
.30305 .32095 L
.30402 .321 L
.30508 .32108 L
.30618 .32118 L
.3072 .32131 L
.30947 .32169 L
.31149 .32216 L
.31364 .3228 L
.31584 .32363 L
.3199 .32565 L
.32335 .32791 L
.33041 .33402 L
.33616 .33975 L
.34238 .34516 L
.34574 .34735 L
.34763 .34831 L
.34973 .34919 L
.35175 .34983 L
.35367 .35029 L
.35462 .35047 L
.35567 .35062 L
.35669 .35074 L
.35764 .35082 L
.35887 .35088 L
.35957 .3509 L
.36022 .3509 L
.36144 .35088 L
.36274 .35083 L
.36394 .35074 L
.36507 .35064 L
.36769 .35031 L
.37062 .34982 L
.37335 .34928 L
.37968 .34778 L
.39147 .34461 L
.39785 .34298 L
.40365 .3417 L
.40835 .34089 L
.4109 .34057 L
.412 .34046 L
.41316 .34037 L
.41434 .3403 L
.41542 .34026 L
.41608 .34024 L
Mistroke
.41669 .34024 L
.41782 .34025 L
.41899 .34029 L
.42004 .34036 L
.42118 .34046 L
.42236 .3406 L
.42435 .34093 L
.42611 .34131 L
.4299 .34243 L
.43324 .34377 L
.43634 .34524 L
.44347 .34899 L
.45191 .35302 L
.4561 .35462 L
.46101 .35616 L
.46689 .35758 L
.46969 .35811 L
.47286 .35863 L
.47566 .359 L
.47873 .35934 L
.48206 .35962 L
.48357 .35972 L
.48522 .35981 L
.48688 .35989 L
.48843 .35995 L
.48991 .35999 L
.4915 .36002 L
.49292 .36003 L
.49426 .36003 L
.49573 .36002 L
.49731 .36 L
.49826 .35998 L
.49916 .35996 L
.50085 .3599 L
.50266 .35983 L
.50461 .35973 L
.50812 .3595 L
.51132 .35924 L
.51855 .35844 L
.52588 .35732 L
.53738 .35485 L
.54338 .35313 L
.54866 .35135 L
.58552 .33519 L
.59583 .33156 L
.60743 .3284 L
.62821 .32429 L
.66733 .31585 L
.71121 .30879 L
.72185 .3076 L
Mistroke
.72731 .30685 L
.73213 .30603 L
.74003 .30418 L
.74389 .30298 L
.74783 .3015 L
.76256 .2943 L
.77129 .29008 L
.78029 .28662 L
.79066 .28366 L
.8031 .28088 L
.81549 .27803 L
.82724 .27452 L
.83788 .27009 L
.84859 .26395 L
.85914 .25574 L
.86716 .24761 L
.87335 .23965 L
.87726 .23298 L
.87808 .23116 L
.8787 .22953 L
.87915 .22811 L
.8795 .22665 L
.87964 .22592 L
.87969 .22553 L
.87975 .22512 L
.87982 .22441 L
.87985 .22375 L
.87986 .22336 L
.87987 .22293 L
.87986 .22253 L
.87984 .22217 L
.87979 .22147 L
.87975 .22112 L
.87971 .22074 L
.87948 .21939 L
.87914 .2179 L
.87831 .21505 L
.8763 .20889 L
.87509 .20473 L
.87392 .19964 L
.87342 .19691 L
.87301 .1942 L
.87267 .19144 L
.87242 .18876 L
.87226 .18645 L
.87219 .18524 L
.87213 .1839 L
.87208 .18242 L
.87205 .18101 L
.87203 .17961 L
Mistroke
.87203 .17831 L
.87203 .17694 L
.87205 .17564 L
.87209 .17422 L
.87213 .17267 L
.87227 .16951 L
.87244 .16662 L
.87324 .15662 L
.87407 .14708 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P19o`<0@Ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`16o`800ol803Wo001lo`040?oo00Wo00<0o`001Ol00`3oo`15o`803?l<02go001mo`80
2ol200Go00<0ool0AOl00`3oo`0Fo`80;?l004?o0`1;o`030?oo04Go00<0ool06?l00`3oo`0Yo`00
@Ol300;o1@16o`030?oo04Go00<0ool06?l00`3oo`0Yo`00@?l00`3oo`08oa40=Ol00`3oo`15o`03
0?oo01So00<0ool0:Ol003ko0P0Lo`80<ol00`3oo`15o`030?oo01So00<0ool0:Ol003co0P0Oo`03
0?oo037o00<0ool0A?l00`3oo`0Io`030?oo02Wo000ko`030?oo01oo00<0ool0<Ol00`3oo`14o`03
0?oo01Wo00<0ool0:Ol003Wo0P0Ro`030?oo037o00<0ool0A?l00`3oo`0Io`030?oo02Wo000go`80
8ol00`3oo`0bo`030?oo04Co00<0ool06Ol00`3oo`0Yo`00=Ol202Go00<0ool0<_l00`3oo`14o`03
0?oo01Wo00<0ool0:Ol003Co00<0ool09?l00`3oo`0co`030?oo04?o00<0ool06_l00`3oo`0Yo`00
<ol00`3oo`0Uo`030?oo03?o00<0ool0@ol00`3oo`0Io`80:ol0037o0P0Wo`030?oo023o0P04o`80
0ol400Go00<0ool0@ol00`3oo`0Io`030?oo02[o000`o`030?oo02Ko00<0ool08?l0103oo`08o`03
0?oo00Ko00<0ool0@_l00`3oo`0Jo`030?oo02[o000_o`030?oo02Oo00<0ool08?l0103oo`09o`03
0?oo00Go0P13o`030?oo01[o00<0ool0:_l002oo00<0ool09_l00`3oo`0Ko`@00_l0103oo`0:o`03
0?oo00Co00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000^o`030?oo02Ko00<0ool08_l0103oo`08o`04
0?oo00Go00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000]o`80:?l00`3oo`0So`802_l200Ko00<0ool0
@?l00`3oo`0Mo`030?oo02Wo000]o`030?oo02Ko00<0ool0>?l00`3oo`0oo`030?oo01ko00<0ool0
:Ol002go00<0ool09_l00`3oo`0ho`030?oo03ko00<0ool07ol00`3oo`0Yo`00;Ol00`3oo`0Vo`03
0?oo03So00<0ool0?Ol00`3oo`0Po`030?oo02Wo0007o`D00_l200Co0P0Ho`030?oo00So1@0Go`03
0?oo00So0P04o`801?l202?o00<0ool08_l200Co0P04o`803?l00`3oo`0Do`D02Ol00`3oo`0Fo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool05_l00`3oo`08o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02?l00`3oo`0Go`030?oo00Wo00<0ool0
6?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Fo`03
0?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200[o00<0ool06?l00`3oo`09o`80
6Ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool05_l00`3oo`08o`04
0?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02?l00`3oo`0Io`030?oo00Wo0P0Io`03
0?oo00So00<0ool00_l000So0P0:o`<06Ol00`3oo`07o`806?l00`3oo`09o`802_l302;o00<0ool0
8_l200[o0`07o`030?oo01Wo0P0;o`030?oo01Oo0P0:o`<00_l0033o00<0ool08Ol00`3oo`0jo`03
0?oo03So00<0ool09_l00`3oo`0Xo`00<?l00`3oo`0Qo`030?oo03[o00<0ool0=ol00`3oo`0Wo`03
0?oo02So000ao`030?oo023o00<0ool0>_l00`3oo`0fo`030?oo02Oo00<0ool0:Ol0037o00<0ool0
8?l203_o00<0ool09_l500Wo0P0Yo`030?oo02[o000bo`030?oo023o0P0jo`030?oo02Go00<0ool0
0ol902[o00<0ool0:ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool00_l01@3oool000So00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`03o`80
00?o0?l01ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`040?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go
0P08o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Ko00@0ool0:ol00`3o
o`05o`808ol00`3oo`0Ro`030?oo00Co00<0ool0:Ol00`3oo`04o`030?oo02?o00<0ool01ol003Ko
00<0ool0=Ol00`3oo`0Po`030?oo027o00<0ool0=_l2033o000go`80=_l00`3oo`0Oo`030?oo027o
00<0ool0=?l203;o000io`80=Ol201;o1@08o`030?oo023o00<0ool0<ol203Co000ko`80=Ol00`3o
o`08o`L01?l:02;o00<0ool0<_l00`3oo`0do`00?Ol203Co0`000ol0000401Co1`0Ko`030?oo02go
1P0go`00?ol00`3oo`0co`<06Ol00`3oo`04o`@05ol00`3oo`0Zo`<0?Ol0043o0P1?o`030?oo00So
0`0Co`030?oo02So0`10o`00@_l204go00<0ool02ol200oo0P0Xo`<0@ol004Co00<0ool0B_l00`3o
o`0=o`803?l00`3oo`0Vo`80A_l004Go00<0ool0BOl00`3oo`0?o`@01ol00`3oo`0To`<0B?l004Ko
0P19o`030?oo01?o0P03o`809Ol204_o0018o`030?oo04Ko00<0ool05Ol402;o101=o`00BOl00`3o
o`15o`030?oo03Oo101Ao`00B_l00`3oo`0`o`801?l200?o1005o`030?oo03;o1@1Eo`00Bol00`3o
o`0^o`040?oo00So00<0ool01_l00`3oo`0/o`H0F_l004_o00<0ool0;_l0103oo`09o`030?oo00Go
0P0Xo`D0H?l004co0`07o`X07?l0103oo`0:o`030?oo00Co00<0ool08_l506Go001?o`L02_l201[o
00@0ool02?l0103oo`05o`030?oo01co1P1Zo`00H_l00`3oo`0Ho`802_l200Ko00<0ool06?l4073o
001So`030?oo02_o00<0ool05Ol307Co001To`030?oo02[o00<0ool04_l307Oo001Uo`030?oo00ko
200Co`030?oo013o0P1jo`00I_l200[o1008o`<04?l00`3oo`0=o`<0O?l006Oo0`05o`<03ol200ko
00<0ool02ol207oo001Yo`H05?l200co00<0ool01ol4087o0025o`802_l00`3oo`05o`80QOl008Oo
4P27o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3o
o`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So
0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87428 .14897 m
.87408 .14671 L
.8738 .1444 L
.87346 .14233 L
.87309 .14055 L
.87255 .13859 L
.87184 .13661 L
.87103 .13491 L
.87016 .13346 L
.86803 .13095 L
.86657 .12974 L
.86505 .12873 L
.86147 .12703 L
.85918 .12626 L
.8569 .12564 L
.85154 .12457 L
.84599 .12377 L
.83189 .12224 L
.81788 .12083 L
.81116 .12019 L
.80525 .1197 L
.80253 .11952 L
.80102 .11944 L
.79967 .11938 L
.79827 .11934 L
.79751 .11933 L
.7968 .11932 L
.79557 .11932 L
.79426 .11935 L
.79298 .1194 L
.79227 .11944 L
.79161 .11949 L
.79032 .11961 L
.78899 .11979 L
.78785 .11998 L
.7867 .12022 L
.78475 .12075 L
.78311 .12136 L
.78148 .12215 L
.77886 .12395 L
.77745 .12529 L
.77625 .12672 L
.7743 .12984 L
.77267 .13348 L
.77119 .13803 L
.76878 .14845 L
.76676 .15868 L
.76463 .16812 L
.76171 .17779 L
.75848 .18546 L
Mistroke
.75442 .19272 L
.74932 .20015 L
.73711 .21598 L
.72287 .23173 L
.71449 .23859 L
.70946 .24173 L
.70468 .24405 L
.7004 .24562 L
.69794 .24632 L
.69563 .24685 L
.69319 .2473 L
.69097 .2476 L
.68973 .24773 L
.6884 .24784 L
.68714 .24792 L
.68597 .24798 L
.68481 .24802 L
.68359 .24805 L
.68255 .24806 L
.68141 .24806 L
.68019 .24806 L
.67951 .24805 L
.6789 .24804 L
.67658 .248 L
.67554 .24798 L
.67445 .24797 L
.67326 .24796 L
.67217 .24795 L
.67116 .24796 L
.67022 .24797 L
.6692 .24799 L
.6681 .24803 L
.6675 .24807 L
.66685 .2481 L
.6657 .24819 L
.66447 .24832 L
.66312 .24849 L
.66075 .24891 L
.65854 .24944 L
.65643 .25009 L
.65452 .25082 L
.65029 .25294 L
.64643 .25549 L
.64236 .25879 L
.63506 .26594 L
.62627 .27527 L
.61777 .28328 L
.61342 .28658 L
.60846 .28953 L
.60581 .29076 L
Mistroke
.60328 .2917 L
.60057 .29247 L
.59902 .29281 L
.59759 .29305 L
.59632 .29322 L
.59496 .29336 L
.59367 .29344 L
.59249 .29348 L
.59176 .29348 L
.59108 .29348 L
.59034 .29346 L
.58955 .29343 L
.58822 .29334 L
.58746 .29328 L
.58675 .29321 L
.58405 .29287 L
.58156 .29245 L
.57587 .29121 L
.56431 .28802 L
.55181 .28452 L
.5412 .28213 L
.53525 .28112 L
.52978 .2804 L
.5246 .27988 L
.51928 .27949 L
.5101 .27897 L
.49046 .27742 L
.48032 .27618 L
.47521 .27555 L
.47037 .27503 L
.46586 .27464 L
.46355 .27449 L
.46097 .27435 L
.4596 .2743 L
.45835 .27426 L
.45689 .27423 L
.4555 .27421 L
.45421 .27421 L
.45301 .27422 L
.45169 .27424 L
.45027 .27427 L
.44873 .27432 L
.44731 .27438 L
.44408 .27456 L
.44123 .27476 L
.43821 .27502 L
.42731 .27621 L
.42181 .27685 L
.4161 .27744 L
.41447 .27758 L
Mistroke
.41294 .2777 L
.41147 .27779 L
.41013 .27785 L
.40888 .2779 L
.40758 .27792 L
.40628 .27792 L
.40556 .27791 L
.40488 .2779 L
.40369 .27785 L
.40303 .27782 L
.40244 .27778 L
.40024 .27758 L
.39904 .27742 L
.39775 .27722 L
.39549 .27677 L
.39339 .27622 L
.39124 .27554 L
.38752 .27403 L
.38082 .27041 L
.37338 .26569 L
.36498 .2608 L
.35588 .25687 L
.35032 .25517 L
.34723 .25443 L
.34384 .25374 L
.34093 .25326 L
.33764 .2528 L
.3317 .25219 L
.32874 .25196 L
.32549 .25174 L
.31895 .25137 L
.31291 .25101 L
.3076 .25057 L
.30447 .25023 L
.30162 .24983 L
.29872 .24933 L
.29572 .24866 L
.29323 .24796 L
.29075 .2471 L
.28671 .24524 L
.28481 .2441 L
.28314 .24291 L
.28039 .24039 L
.27796 .23732 L
.27682 .23544 L
.27592 .23361 L
.27519 .23182 L
.27461 .23009 L
.27417 .22848 L
.27378 .22676 L
Mistroke
.27346 .22486 L
.27325 .2231 L
.27316 .22212 L
.27311 .2212 L
.27306 .22019 L
.27305 .21963 L
.27304 .21911 L
.27305 .21816 L
.27307 .21726 L
.27311 .21628 L
.27317 .21523 L
.27326 .2142 L
.27338 .21312 L
.27364 .21121 L
.27429 .20787 L
.27524 .20436 L
.27656 .20069 L
.27828 .19693 L
.28216 .19041 L
.2923 .17722 L
.29797 .16984 L
.30076 .16575 L
.30296 .16199 L
.30449 .15873 L
.30512 .15697 L
.30554 .15541 L
.30571 .15456 L
.30581 .15378 L
.30585 .15339 L
.30588 .15295 L
.30589 .15251 L
.30588 .1521 L
.30586 .15174 L
.30583 .15136 L
.30573 .15071 L
.30557 .15002 L
.30533 .1493 L
.30506 .14868 L
.30476 .14811 L
.30406 .14711 L
.30312 .14611 L
.30187 .14514 L
.30059 .14437 L
.29905 .14364 L
.29582 .14256 L
.29375 .14206 L
.29162 .14166 L
.28957 .14135 L
.28727 .14107 L
.28466 .14083 L
Mistroke
.28214 .14064 L
.27905 .14047 L
.27616 .14034 L
.2656 .14004 L
.25978 .13987 L
.25444 .13968 L
.24899 .13941 L
.24326 .13903 L
.23397 .13819 L
.22924 .1377 L
.2244 .13724 L
.22227 .13707 L
.22112 .137 L
.22005 .13694 L
.21916 .1369 L
.21819 .13687 L
.21768 .13686 L
.21713 .13685 L
.21616 .13685 L
.21516 .13686 L
.21424 .13689 L
.21374 .13691 L
.21319 .13694 L
.2122 .137 L
.21126 .13708 L
.21028 .13719 L
.2085 .13742 L
.20641 .13779 L
.20443 .13821 L
.1998 .13952 L
.19193 .14258 L
.18284 .14709 L
.16303 .15896 L
.15299 .16576 L
.14357 .17292 L
.1362 .17978 L
.13347 .18303 L
.1314 .18607 L
.12975 .18922 L
.12907 .19093 L
.12847 .19276 L
.12806 .1944 L
.12776 .19597 L
.12763 .19688 L
.12753 .19772 L
.12745 .19871 L
.1274 .19965 L
.12738 .20055 L
.12738 .20153 L
.1274 .20247 L
Mistroke
.12745 .20334 L
.12752 .20437 L
.12763 .2055 L
.12791 .20762 L
.1283 .20987 L
.12883 .21233 L
.13007 .21691 L
.13343 .22622 L
.13746 .23481 L
.14266 .24371 L
.14905 .2524 L
.15697 .26068 L
.16497 .26702 L
.18205 .27682 L
.19855 .28536 L
.23135 .30863 L
.23962 .31426 L
.24408 .31674 L
.24894 .31889 L
.25331 .32034 L
.25585 .32099 L
.25829 .32148 L
.26055 .32184 L
.26265 .32209 L
.26499 .32229 L
.26618 .32237 L
.26749 .32243 L
.26878 .32247 L
.27 .3225 L
.27127 .32251 L
.27195 .32251 L
.27268 .32251 L
.27348 .3225 L
.27423 .32249 L
.27491 .32248 L
.27566 .32247 L
.27843 .3224 L
.28354 .32224 L
.28502 .3222 L
.28641 .32217 L
.28766 .32215 L
.28901 .32214 L
.29025 .32214 L
.29095 .32215 L
.2916 .32215 L
.29292 .32218 L
.29365 .3222 L
.29432 .32222 L
.29551 .32227 L
.2968 .32235 L
Mistroke
.29913 .32253 L
.30143 .32279 L
.30385 .32314 L
.30808 .32401 L
.31229 .32521 L
.31665 .32686 L
.32406 .33065 L
.33159 .33553 L
.33959 .34089 L
.34343 .34314 L
.34769 .34521 L
.34983 .34607 L
.3519 .34677 L
.3538 .34732 L
.35593 .34783 L
.35826 .34825 L
.35948 .34843 L
.36081 .34858 L
.36206 .3487 L
.36323 .34878 L
.36441 .34884 L
.36555 .34887 L
.36688 .34888 L
.36763 .34888 L
.36831 .34887 L
.36965 .34883 L
.37093 .34878 L
.37244 .34869 L
.37384 .34858 L
.37704 .34828 L
.38802 .3468 L
.39412 .34588 L
.39983 .34506 L
.40503 .34444 L
.4079 .34417 L
.40925 .34407 L
.4105 .34399 L
.41163 .34392 L
.41265 .34388 L
.41377 .34384 L
.41496 .34382 L
.4161 .34381 L
.41732 .34382 L
.41846 .34384 L
.41951 .34388 L
.42068 .34394 L
.42175 .34401 L
.42412 .34423 L
.42616 .34449 L
.42828 .34482 L
Mistroke
.43185 .34553 L
.43569 .34648 L
.45218 .35163 L
.45656 .35287 L
.46158 .35411 L
.46627 .35507 L
.47154 .35592 L
.47456 .35631 L
.478 .35668 L
.4812 .35694 L
.48427 .35714 L
.48576 .35722 L
.48734 .35728 L
.48871 .35733 L
.49019 .35737 L
.49179 .35739 L
.49329 .35741 L
.49421 .35741 L
.49508 .35741 L
.49672 .3574 L
.49818 .35738 L
.49976 .35734 L
.50125 .3573 L
.50262 .35725 L
.50595 .3571 L
.50911 .35691 L
.51532 .3564 L
.52195 .35565 L
.53324 .35386 L
.54299 .3517 L
.5538 .34857 L
.56342 .34518 L
.58118 .33839 L
.62009 .32685 L
.65958 .31732 L
.70116 .30881 L
.71239 .30726 L
.72263 .3058 L
.73201 .30416 L
.7401 .30228 L
.74827 .2998 L
.7568 .29663 L
.77477 .28962 L
.79462 .28374 L
.80527 .28106 L
.81529 .27832 L
.8249 .27511 L
.83512 .27064 L
.84482 .265 L
.85409 .25798 L
Mistroke
.86216 .25019 L
.86839 .2427 L
.8712 .23867 L
.8737 .23446 L
.87548 .23082 L
.87673 .22755 L
.87757 .22454 L
.8779 .22287 L
.87812 .22135 L
.87821 .22048 L
.87827 .21968 L
.8783 .21926 L
.87832 .2188 L
.87834 .21831 L
.87835 .21785 L
.87835 .217 L
.87834 .21622 L
.87832 .21535 L
.8783 .21491 L
.87827 .21442 L
.87813 .21248 L
.87795 .21065 L
.87742 .20627 L
.8763 .19725 L
.87581 .19239 L
.87537 .18686 L
.8751 .18189 L
.87498 .17897 L
.87489 .1762 L
.87485 .17467 L
.87481 .17298 L
.87477 .16991 L
.87473 .16691 L
.87471 .16415 L
.87469 .16261 L
.87467 .16096 L
.87466 .15952 L
.87463 .15801 L
.8746 .15627 L
.87456 .1547 L
.87451 .15331 L
.87444 .1518 L
.87428 .14897 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1:o`P0>ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`17o`802?l903;o001lo`040?oo00Wo00<0o`001Ol00`3oo`16o`030?oo017o100^o`00
OOl200_o0P05o`030?oo04Go00<0ool05_l00`3oo`0[o`00TOl00`3oo`15o`030?oo01Oo00<0ool0
:_l004;o3013o`030?oo04Go00<0ool05ol202_o000oo`<03?l=03Ko00<0ool0AOl00`3oo`0Ho`03
0?oo02Wo000mo`807?l203Co00<0ool0A?l00`3oo`0Io`030?oo02Wo000lo`030?oo01go0P0co`03
0?oo04Co00<0ool06Ol00`3oo`0Yo`00>_l2027o00<0ool0<Ol00`3oo`14o`030?oo01Wo00<0ool0
:Ol003So0P0So`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000go`030?oo02;o00<0ool0
<_l00`3oo`13o`030?oo01[o00<0ool0:Ol003Go0P0Uo`030?oo03;o00<0ool0@ol00`3oo`0Jo`03
0?oo02Wo000do`030?oo02Go00<0ool0<_l00`3oo`13o`030?oo01[o00<0ool0:Ol003;o0P0Wo`03
0?oo03?o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000ao`030?oo02Ko00<0ool08?l200Co0P03o`@0
1Ol00`3oo`12o`030?oo01_o00<0ool0:Ol0033o00<0ool09_l00`3oo`0Po`040?oo00So00<0ool0
1_l00`3oo`12o`030?oo01_o00<0ool0:Ol002oo00<0ool09_l00`3oo`0Qo`040?oo00Wo00<0ool0
1Ol204;o00<0ool07?l00`3oo`0Yo`00;_l00`3oo`0Wo`030?oo01_o1002o`040?oo00[o00<0ool0
1?l00`3oo`11o`030?oo01co00<0ool0:Ol002ko00<0ool09_l00`3oo`0Ro`040?oo00So00@0ool0
1Ol00`3oo`10o`030?oo01go00<0ool0:Ol002ko00<0ool09_l00`3oo`0So`802_l200Ko00<0ool0
@?l00`3oo`0Mo`030?oo02Wo000]o`030?oo02Ko00<0ool0>?l00`3oo`0oo`030?oo01oo00<0ool0
:?l002go00<0ool09Ol00`3oo`0io`030?oo03oo00<0ool07ol00`3oo`0Xo`00;Ol00`3oo`0Uo`03
0?oo03Wo00<0ool0?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803Ol00`3oo`0Co`D02Ol00`3oo`0Fo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02Ol00`3oo`0Ho`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00Wo00<0ool0
6?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Ko
00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802ol00`3oo`0Go`030?oo00Wo
00<0ool06?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool05_l00`3o
o`08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00Wo
00<0ool06?l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201So00<0ool02Ol200[o
0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`802ol00`3oo`0Go`802_l300;o000`o`030?oo02;o
00<0ool0>Ol00`3oo`0ho`030?oo02Go00<0ool0:Ol0033o00<0ool08_l00`3oo`0io`030?oo03Oo
00<0ool09_l00`3oo`0Yo`00<Ol00`3oo`0Qo`80>_l00`3oo`0fo`030?oo02Ko00<0ool0:_l0037o
00<0ool08ol00`3oo`0go`030?oo03?o0`0Xo`030?oo02_o000bo`030?oo02?o0`0fo`030?oo02Wo
2P0[o`030?oo02_o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`05
0?ooo`001?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Go1`03o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`0300?o00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Co00<0o`00
2?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Uo`050?ooo`00:ol00`3o
o`03o`809Ol00`3oo`0Ro`030?oo00Co00<0ool0:Ol00`3oo`03o`030?oo02Co00<0ool01ol003Go
00<0ool0=?l302;o00<0ool08Ol00`3oo`0eo`80<Ol003Ko00<0ool0=_l2023o00<0ool08?l00`3o
o`0do`80<ol003Oo0P0ho`807_l00`3oo`0Oo`030?oo03Co00<0ool0<ol003Wo0P0ho`030?oo00co
1@0:o`030?oo01oo00<0ool0<_l203Ko000ko`80=ol?00Co2`0Po`030?oo037o0P0ho`00?Ol205;o
2P0Fo`030?oo02go1@0jo`00?ol2053o00<0ool01ol600oo00<0ool0:ol303oo0011o`030?oo04go
00<0ool03Ol300_o00<0ool0:_l204;o0012o`80COl00`3oo`0@o`D000?o00000`0Yo`@0A?l004Co
00<0ool0B_l00`3oo`0Do`<0:Ol404So0015o`030?oo04Wo00<0ool0?_l204co0016o`80BOl00`3o
o`0io`D0C_l004So00<0ool0A_l00`3oo`0do`D0Dol004Wo00<0ool0<Ol200Co0P03o`@01Ol00`3o
o`0`o`@0F?l004[o0P0`o`040?oo00So00<0ool01_l00`3oo`0Zo`H0G?l004co0`0]o`040?oo00Wo
00<0ool01Ol202Ko1@1Ro`00Col>01oo00@0ool02_l00`3oo`04o`030?oo01oo1P1Wo`00GOl401_o
00@0ool02?l0103oo`05o`030?oo01_o101]o`00HOl201[o0P0:o`801_l00`3oo`0Io`80LOl006?o
00<0ool0:ol00`3oo`0Fo`<0Lol006Co00<0ool0:_l00`3oo`0Do`80M_l006Go0P0Zo`030?oo013o
101ho`00Iol200co2P0Bo`030?oo00co101lo`00JOl<00[o0`0?o`030?oo00[o0P20o`00P_l400_o
00<0ool01ol308;o0026o`L01?l:08Go002=o`D0S_l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co
001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo
00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go
00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87448 .15087 m
.87331 .14646 L
.87197 .14263 L
.87018 .13878 L
.86786 .13508 L
.86493 .13168 L
.86166 .12892 L
.8583 .1268 L
.85422 .12485 L
.84957 .12318 L
.84397 .12167 L
.83768 .12043 L
.83149 .11951 L
.82556 .11885 L
.82008 .1184 L
.81707 .11823 L
.81538 .11816 L
.81457 .11813 L
.81382 .1181 L
.81233 .11807 L
.81098 .11805 L
.80955 .11805 L
.80875 .11805 L
.80801 .11806 L
.80661 .11809 L
.8053 .11813 L
.80414 .11819 L
.80288 .11826 L
.80154 .11837 L
.80014 .1185 L
.79768 .11879 L
.7955 .11914 L
.79352 .11954 L
.78926 .12073 L
.78717 .12153 L
.78508 .1225 L
.78167 .12458 L
.77844 .12733 L
.7754 .13093 L
.77298 .13488 L
.77102 .13903 L
.76807 .14768 L
.76298 .16875 L
.7605 .1781 L
.75732 .18722 L
.75401 .19441 L
.74961 .20187 L
.73822 .21638 L
.72249 .23143 L
.71384 .2378 L
Mistroke
.70392 .24343 L
.69926 .24548 L
.69426 .24731 L
.68522 .24979 L
.66666 .25361 L
.66165 .25487 L
.6564 .2565 L
.64706 .26031 L
.62832 .27082 L
.61899 .27619 L
.61028 .28039 L
.60592 .28206 L
.6011 .28353 L
.59886 .28408 L
.59647 .28457 L
.59421 .28495 L
.59212 .28523 L
.58974 .28548 L
.58848 .28557 L
.58715 .28565 L
.58598 .2857 L
.58489 .28573 L
.58369 .28575 L
.5824 .28576 L
.58167 .28575 L
.58099 .28574 L
.58024 .28573 L
.57945 .28571 L
.57812 .28566 L
.57665 .28559 L
.57391 .28541 L
.57134 .28521 L
.55079 .2831 L
.54513 .28259 L
.53914 .28217 L
.53407 .28191 L
.5312 .2818 L
.52853 .28172 L
.52575 .28166 L
.52424 .28164 L
.52284 .28162 L
.52037 .28159 L
.51769 .28157 L
.51461 .28155 L
.51171 .28153 L
.50913 .2815 L
.50769 .28149 L
.50635 .28147 L
.50408 .28142 L
.50158 .28136 L
Mistroke
.49899 .28128 L
.49656 .28118 L
.49155 .28091 L
.48615 .28054 L
.46593 .27882 L
.46019 .27842 L
.45701 .27823 L
.45399 .27809 L
.45115 .27798 L
.44966 .27793 L
.44807 .27789 L
.44673 .27786 L
.44527 .27783 L
.44393 .27782 L
.44268 .27781 L
.44127 .2778 L
.43997 .2778 L
.43855 .2778 L
.43783 .2778 L
.43703 .2778 L
.43571 .27781 L
.43429 .27782 L
.43172 .27785 L
.42701 .2779 L
.42573 .27791 L
.42437 .27792 L
.42309 .27793 L
.42194 .27793 L
.42059 .27793 L
.41983 .27793 L
.41913 .27792 L
.41785 .27791 L
.41651 .27788 L
.41513 .27785 L
.41385 .27781 L
.41245 .27775 L
.41096 .27767 L
.40836 .27749 L
.40594 .27727 L
.40148 .27671 L
.39907 .2763 L
.39682 .27585 L
.39271 .27486 L
.38375 .27192 L
.36758 .26509 L
.35843 .26143 L
.34816 .25805 L
.32852 .25349 L
.31753 .25128 L
.30755 .24883 L
Mistroke
.29927 .24595 L
.29496 .24396 L
.29126 .24186 L
.28447 .23665 L
.28158 .23365 L
.27883 .2301 L
.27658 .22641 L
.27555 .22431 L
.2747 .2223 L
.27404 .22041 L
.27345 .21835 L
.27303 .21647 L
.27275 .21473 L
.27264 .21377 L
.27259 .21325 L
.27255 .21276 L
.27251 .21191 L
.27249 .21097 L
.27249 .21045 L
.2725 .20995 L
.27255 .20903 L
.27259 .20852 L
.27263 .20804 L
.27276 .20698 L
.27289 .20613 L
.27303 .20534 L
.27345 .20355 L
.27398 .20183 L
.27466 .2 L
.27618 .19681 L
.27818 .19351 L
.28261 .18771 L
.29462 .17465 L
.29951 .16886 L
.3017 .16583 L
.30361 .16267 L
.30439 .16108 L
.30508 .15936 L
.30532 .15861 L
.30554 .15783 L
.30571 .1571 L
.30583 .15643 L
.30591 .15572 L
.30594 .15535 L
.30596 .15496 L
.30596 .15424 L
.30591 .15361 L
.30583 .153 L
.3057 .15235 L
.30554 .15175 L
Mistroke
.30536 .15121 L
.30478 .14998 L
.30407 .14888 L
.3031 .14776 L
.30185 .14663 L
.3003 .14554 L
.29862 .14461 L
.29495 .14308 L
.29114 .14196 L
.28646 .14098 L
.28395 .14057 L
.28114 .1402 L
.27585 .13965 L
.2707 .13927 L
.26519 .13898 L
.26047 .13878 L
.25536 .13861 L
.24429 .13832 L
.2393 .13821 L
.23716 .13817 L
.23484 .13814 L
.23366 .13812 L
.2324 .13811 L
.23171 .1381 L
.23108 .1381 L
.22986 .13809 L
.22918 .13809 L
.22855 .13809 L
.22787 .1381 L
.22715 .1381 L
.22594 .13811 L
.22526 .13812 L
.22462 .13813 L
.22348 .13816 L
.22229 .13819 L
.22016 .13827 L
.21789 .13839 L
.21542 .13857 L
.21316 .13878 L
.21071 .13905 L
.20625 .13972 L
.20187 .14057 L
.19776 .14156 L
.18821 .14456 L
.17778 .14882 L
.15903 .15888 L
.14378 .17016 L
.14035 .17347 L
.13702 .17718 L
.13449 .18057 L
Mistroke
.13221 .18431 L
.13041 .18814 L
.12917 .19174 L
.1287 .19352 L
.12829 .19547 L
.12814 .19639 L
.12799 .19736 L
.12788 .19826 L
.1278 .19911 L
.12772 .20016 L
.12769 .20071 L
.12766 .20131 L
.12764 .2023 L
.12763 .20342 L
.12765 .20441 L
.12769 .20549 L
.12776 .20651 L
.12783 .20746 L
.12806 .20961 L
.12842 .21199 L
.12943 .21687 L
.13064 .22123 L
.13224 .22592 L
.13584 .23432 L
.14104 .24374 L
.14773 .25317 L
.15427 .26046 L
.16214 .26748 L
.17846 .27835 L
.19699 .28837 L
.21574 .29869 L
.23261 .30794 L
.24099 .3119 L
.2503 .31545 L
.25935 .31799 L
.26802 .31972 L
.28655 .3224 L
.29632 .324 L
.30639 .32628 L
.31672 .32956 L
.3258 .33322 L
.33416 .33693 L
.34309 .34073 L
.34771 .34243 L
.35218 .34385 L
.35665 .34501 L
.36088 .34589 L
.36501 .34654 L
.36949 .34704 L
.37186 .34723 L
Mistroke
.3741 .34737 L
.37659 .34748 L
.378 .34753 L
.37929 .34756 L
.38064 .34758 L
.38191 .34759 L
.38306 .3476 L
.3843 .3476 L
.38567 .34759 L
.38635 .34759 L
.38711 .34758 L
.38971 .34754 L
.39475 .34743 L
.39945 .34732 L
.4023 .34727 L
.40369 .34724 L
.40498 .34723 L
.40614 .34721 L
.40736 .3472 L
.40859 .3472 L
.40994 .3472 L
.41064 .34721 L
.41137 .34721 L
.41268 .34723 L
.4134 .34724 L
.41407 .34725 L
.41559 .34729 L
.41811 .34737 L
.42077 .3475 L
.42331 .34766 L
.42596 .34786 L
.43065 .34831 L
.44101 .34969 L
.45004 .35112 L
.46003 .35263 L
.46525 .3533 L
.47035 .35384 L
.47507 .35424 L
.47782 .35443 L
.48035 .35457 L
.48278 .35468 L
.48545 .35477 L
.48693 .35481 L
.48829 .35484 L
.4896 .35486 L
.49099 .35487 L
.49219 .35488 L
.49348 .35488 L
.4947 .35487 L
.49581 .35486 L
Mistroke
.49712 .35484 L
.49855 .35481 L
.50115 .35474 L
.50386 .35464 L
.50679 .3545 L
.51208 .35418 L
.51831 .35366 L
.52394 .35307 L
.53523 .35151 L
.54657 .34938 L
.56629 .34431 L
.58424 .33874 L
.62286 .32718 L
.6627 .3161 L
.70376 .3072 L
.72271 .30401 L
.74228 .29998 L
.76055 .29488 L
.78011 .28892 L
.80086 .28303 L
.8203 .27667 L
.8297 .27255 L
.83948 .26708 L
.85529 .25458 L
.86246 .24682 L
.86786 .23961 L
.87268 .23134 L
.87435 .22767 L
.87584 .22366 L
.87695 .21981 L
.87772 .21618 L
.87806 .21414 L
.8783 .21226 L
.87852 .21005 L
.8786 .20904 L
.87868 .20791 L
.87877 .20603 L
.87881 .20499 L
.87884 .204 L
.87886 .20307 L
.87887 .20221 L
.87887 .2012 L
.87887 .20024 L
.87886 .19919 L
.87885 .19806 L
.87883 .19698 L
.87881 .19599 L
.87873 .19351 L
.87864 .19113 L
.87812 .18119 L
Mistroke
.87776 .17596 L
.87728 .17014 L
.87678 .16517 L
.87612 .15992 L
.87535 .15509 L
.87448 .15087 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1<o`/0=_l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`18o`<02ol5037o001lo`040?oo00Wo00<0o`001Ol00`3oo`17o`030?oo017o0P0_o`00
OOl200_o0P05o`030?oo04Ko00<0ool05?l00`3oo`0/o`00TOl00`3oo`15o`030?oo01Ko00<0ool0
:ol004Co400mo`030?oo04Go00<0ool05ol00`3oo`0Zo`00@?l4013o1P0go`030?oo04Go00<0ool0
5ol00`3oo`0Zo`00?Ol301[o0P0eo`030?oo04Co00<0ool06Ol00`3oo`0Yo`00>ol201oo0P0co`03
0?oo04Co00<0ool06Ol00`3oo`0Yo`00>Ol202;o00<0ool0<Ol00`3oo`14o`030?oo01Wo00<0ool0
:Ol003Oo0P0To`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000fo`030?oo02?o0P0co`03
0?oo04?o00<0ool06ol00`3oo`0Xo`00=Ol00`3oo`0To`030?oo03;o00<0ool0@_l00`3oo`0Lo`03
0?oo02So000co`809_l203Co00<0ool0@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Ko00<0ool0
<ol00`3oo`12o`030?oo01co00<0ool0:?l0037o00<0ool09_l00`3oo`0Po`801?l200?o1005o`03
0?oo04;o00<0ool07?l00`3oo`0Xo`00<?l00`3oo`0Vo`030?oo023o00@0ool02?l00`3oo`06o`03
0?oo04;o00<0ool07?l00`3oo`0Xo`00;ol00`3oo`0Vo`030?oo027o00@0ool02Ol00`3oo`05o`80
@_l00`3oo`0Mo`030?oo02So000_o`030?oo02Go00<0ool07?l400;o00@0ool02_l00`3oo`04o`03
0?oo047o00<0ool07Ol00`3oo`0Xo`00;_l00`3oo`0Uo`030?oo02?o00@0ool02?l0103oo`05o`03
0?oo043o00<0ool07_l00`3oo`0Xo`00;_l00`3oo`0To`030?oo02Go0P0:o`801_l00`3oo`10o`03
0?oo01ko00<0ool0:?l002go0P0Vo`030?oo03Wo00<0ool0?ol00`3oo`0Oo`030?oo02So000]o`03
0?oo02Co0P0ko`030?oo03oo00<0ool07ol00`3oo`0Xo`00;Ol00`3oo`0To`030?oo03[o00<0ool0
?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`030?oo00Wo0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803Ol00`3oo`0Co`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02?l0103oo`0Ro`03
0?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool0
0_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`040?oo00_o00<0ool0
8?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00Wo00<0ool06?l00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Ko00<0ool00_l400;o
00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802ol00`3oo`0Go`030?oo00Wo00<0ool06?l00`3o
o`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00Wo
00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00So00<0ool06Ol00`3o
o`08o`030?oo00;o0008o`802_l301So00<0ool02?l201Wo00<0ool02?l200[o0`0Ro`030?oo02;o
0P0:o`<02?l00`3oo`0Ho`802_l00`3oo`0Ho`802_l300;o000`o`030?oo02?o00<0ool0>?l00`3o
o`0ho`030?oo02Go00<0ool0:Ol0033o00<0ool08ol203Wo00<0ool0=_l202Oo00<0ool0:_l0037o
00<0ool09?l00`3oo`0fo`030?oo03Go00<0ool09_l00`3oo`0[o`00<Ol00`3oo`0Uo`80=_l00`3o
o`0co`80:Ol00`3oo`0[o`00<_l00`3oo`0Vo`80=?l00`3oo`0`o`<0:_l00`3oo`0/o`003?oo00`0
2Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`04o`@01ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol400Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool09Ol01@3oool002_o00<0o`000`0Wo`030?oo02Ko0P000ol0o`0Zo`03
0?oo00;o00<0ool09Ol00`3oo`07o`00=Ol00`3oo`0bo`<09?l00`3oo`0To`80<ol00`3oo`0`o`00
=_l00`3oo`0do`808_l00`3oo`0Ro`80=?l00`3oo`0ao`00=ol00`3oo`0eo`808?l00`3oo`0Po`80
=?l203Co000ho`030?oo03Ko0`0Mo`030?oo01ko0P0co`<0=_l003Wo0P0ioa403?l00`3oo`0Lo`80
<ol203Wo000ko`80B?lF01?o0P0co`80>ol003go0P1Bo`030?oo00Oo4`0`o`D0?Ol003oo0P1@o`03
0?oo01;o0P0bo`@0@_l0047o0P1>o`030?oo04Co0P16o`00@ol204co00<0ool0@?l404So0015o`03
0?oo04Wo00<0ool0>ol504co0016o`80BOl00`3oo`0fo`D0DOl004So0`16o`030?oo03;o101Fo`00
Bol2033o0P04o`800ol400Go00<0ool0;_l405[o001=o`80;Ol0103oo`08o`030?oo00Ko00<0ool0
:_l405ko001?o`<0:_l0103oo`09o`030?oo00Go0P0Xo`<0H_l005;o1`0So`040?oo00[o00<0ool0
1?l00`3oo`0So`@0IOl005Wo1P0Mo`040?oo00So00@0ool01Ol00`3oo`0Oo`@0JOl005oo0P0Lo`80
2_l200Ko00<0ool07?l306go001Qo`80;_l00`3oo`0Ho`@0L?l006?o0`0[o`030?oo01Go0`1do`00
I_l202Wo00<0ool04Ol407Oo001Xo`<09_l00`3oo`0<o`D0Nol006_o5@0Ao`030?oo00So1020o`00
P?l800Wo00<0ool00_l608Co0028o`h0R_l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87469 .15276 m
.87085 .14472 L
.86557 .13669 L
.86248 .13304 L
.85886 .12948 L
.85155 .12405 L
.84696 .12157 L
.84166 .11939 L
.83914 .11857 L
.8364 .11782 L
.83398 .11728 L
.83135 .11679 L
.82876 .11642 L
.82637 .11617 L
.82522 .11608 L
.82395 .116 L
.82275 .11594 L
.82164 .11591 L
.82096 .11589 L
.82022 .11589 L
.81888 .11589 L
.81758 .11592 L
.81638 .11596 L
.81507 .11603 L
.81365 .11614 L
.81078 .11642 L
.80821 .11677 L
.80545 .11724 L
.80053 .11836 L
.79591 .11977 L
.79183 .12136 L
.78344 .12596 L
.77973 .12882 L
.77608 .13239 L
.7733 .13582 L
.77075 .13977 L
.76712 .14756 L
.76553 .15225 L
.76427 .1568 L
.76187 .16752 L
.75963 .17802 L
.75703 .18767 L
.75395 .19588 L
.74961 .20414 L
.74429 .21153 L
.73735 .21892 L
.72085 .23195 L
.7029 .24315 L
.68604 .25187 L
.66784 .25917 L
Mistroke
.64959 .26466 L
.6113 .27329 L
.59148 .27675 L
.57003 .27957 L
.54761 .2817 L
.5373 .2825 L
.52647 .28324 L
.51651 .28379 L
.51089 .28402 L
.50821 .2841 L
.50567 .28417 L
.50337 .28422 L
.50207 .28424 L
.50087 .28425 L
.49986 .28426 L
.49875 .28427 L
.49759 .28427 L
.4965 .28427 L
.4953 .28427 L
.49404 .28426 L
.49296 .28425 L
.49178 .28424 L
.48914 .28419 L
.48669 .28413 L
.48148 .28394 L
.47583 .28366 L
.46556 .28297 L
.44644 .28127 L
.42676 .27917 L
.41653 .2779 L
.40567 .27629 L
.38576 .27236 L
.36712 .26747 L
.34812 .26156 L
.32838 .25451 L
.31172 .24732 L
.30247 .24245 L
.29407 .23723 L
.28614 .23127 L
.27985 .22531 L
.27716 .22212 L
.2747 .21855 L
.27284 .21491 L
.2721 .21291 L
.27183 .212 L
.27159 .21104 L
.27142 .21017 L
.27135 .20972 L
.27128 .20922 L
.27119 .20836 L
Mistroke
.27115 .20755 L
.27114 .20706 L
.27115 .20654 L
.27116 .20604 L
.27119 .20558 L
.27128 .2047 L
.27135 .20422 L
.27142 .20378 L
.2718 .20209 L
.27208 .20115 L
.27238 .20028 L
.27388 .19702 L
.27567 .19413 L
.27807 .191 L
.28396 .18477 L
.29146 .17782 L
.29813 .17136 L
.30282 .16569 L
.30379 .16417 L
.30464 .16258 L
.30528 .16111 L
.3055 .16047 L
.30571 .15978 L
.30586 .15916 L
.30598 .15851 L
.30607 .15782 L
.30612 .15719 L
.30613 .15684 L
.30612 .15646 L
.30608 .15578 L
.30599 .15512 L
.30588 .15453 L
.30573 .15397 L
.30556 .15345 L
.30507 .1523 L
.30446 .15121 L
.30364 .15008 L
.30256 .14889 L
.30118 .14765 L
.29815 .14557 L
.29463 .14378 L
.29093 .14233 L
.28651 .14099 L
.28136 .13981 L
.27624 .13895 L
.27374 .13862 L
.27138 .13836 L
.2669 .13798 L
.2645 .13783 L
.26193 .13771 L
Mistroke
.26064 .13767 L
.25923 .13763 L
.2579 .13761 L
.25667 .13759 L
.25557 .13758 L
.25435 .13758 L
.25308 .13759 L
.2519 .1376 L
.25059 .13762 L
.24921 .13765 L
.24672 .13772 L
.24378 .13784 L
.24112 .13797 L
.23104 .13868 L
.22143 .13962 L
.21244 .14073 L
.20271 .14222 L
.19245 .14422 L
.18377 .14638 L
.1744 .14934 L
.16488 .15325 L
.15671 .15762 L
.14813 .16366 L
.14112 .1704 L
.13568 .17764 L
.13344 .18158 L
.13139 .18614 L
.13001 .19016 L
.12942 .19232 L
.1289 .19462 L
.12853 .19667 L
.12837 .19776 L
.12822 .19893 L
.1281 .20002 L
.12801 .20101 L
.12795 .20201 L
.1279 .20295 L
.12786 .20405 L
.12785 .20523 L
.12785 .20634 L
.12788 .20736 L
.12793 .20856 L
.12797 .20924 L
.12802 .20987 L
.12824 .21225 L
.12858 .21476 L
.12897 .21703 L
.13014 .22211 L
.13143 .22643 L
.13312 .23103 L
Mistroke
.13686 .23913 L
.14167 .24731 L
.14776 .25566 L
.16213 .27039 L
.17754 .28157 L
.19582 .2913 L
.23439 .30614 L
.27183 .31857 L
.30862 .32969 L
.34781 .34004 L
.36761 .34435 L
.38663 .34755 L
.40511 .34975 L
.41555 .35066 L
.42514 .35131 L
.43453 .35182 L
.44459 .35225 L
.45039 .35244 L
.45568 .35259 L
.45823 .35264 L
.46101 .3527 L
.46364 .35274 L
.46605 .35277 L
.46852 .3528 L
.46988 .35281 L
.47117 .35281 L
.47236 .35282 L
.47346 .35282 L
.47467 .35282 L
.47596 .35282 L
.47725 .35282 L
.47798 .35281 L
.47864 .35281 L
.47987 .3528 L
.48116 .35279 L
.4839 .35276 L
.48691 .35271 L
.48981 .35264 L
.49244 .35257 L
.49836 .35237 L
.50384 .35213 L
.50901 .35184 L
.51914 .35113 L
.52981 .35012 L
.54876 .34764 L
.55834 .346 L
.56857 .34393 L
.58681 .33943 L
.62658 .32693 L
.66472 .31512 L
Mistroke
.70257 .30611 L
.74317 .29823 L
.78177 .28951 L
.80221 .28349 L
.81131 .28023 L
.82068 .27629 L
.82907 .27208 L
.83789 .26671 L
.85206 .25526 L
.85895 .2479 L
.86452 .24067 L
.87008 .23177 L
.87425 .22322 L
.87726 .21514 L
.87848 .21095 L
.87956 .20638 L
.8804 .20176 L
.8807 .19968 L
.88096 .19738 L
.88119 .19482 L
.88128 .19342 L
.88135 .1921 L
.8814 .19087 L
.88143 .18974 L
.88144 .18911 L
.88144 .18843 L
.88144 .1872 L
.88143 .18617 L
.8814 .18523 L
.88136 .18415 L
.88132 .18311 L
.8812 .18127 L
.88104 .17926 L
.88081 .17704 L
.88051 .17468 L
.87982 .17041 L
.87879 .16557 L
.87758 .16104 L
.87469 .15276 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Col603Oo001lo`040?oo00So1@04o`80C?l400Ko100co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03?l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4Ol00`3oo`0^o`00OOl200_o0P05o`030?oo04Oo00<0ool04_l00`3oo`0]o`00TOl00`3oo`16o`03
0?oo01Co00<0ool0;?l004So3P0ko`030?oo04Go00<0ool05_l00`3oo`0[o`00@Ol700ko100go`03
0?oo04Go00<0ool05_l00`3oo`0[o`00??l501Wo0P0eo`030?oo04Co00<0ool06?l00`3oo`0Zo`00
>Ol3023o00<0ool0<_l00`3oo`14o`030?oo01So00<0ool0:_l003Oo0P0So`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Go0P0Vo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`03
0?oo02Ko00<0ool0<Ol00`3oo`13o`030?oo01_o00<0ool0:?l003?o00<0ool09_l203?o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`030?oo01co00<0ool0
:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07Ol00`3oo`0Wo`00<?l00`3oo`0Wo`03
0?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`030?oo02Ko00<0ool0
8?l0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Ko00<0ool0
8Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool09ol002ko00<0ool09Ol201ko1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0To`04
0?oo00So00@0ool01Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`80
2_l200Ko00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00
;_l00`3oo`0So`030?oo03[o00<0ool0?ol00`3oo`0Oo`030?oo02So0007o`D00_l200Co0P0Ho`03
0?oo00So1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01Oo0P08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol201Wo00<0ool0
2?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806ol00`3oo`06o`80
2_l302;o00<0ool08_l200[o0`07o`806_l200[o00<0ool06?l200[o0`02o`00;ol00`3oo`0Vo`03
0?oo03Ko00<0ool0=ol00`3oo`0Uo`030?oo02[o000`o`030?oo02Ko00<0ool0=Ol00`3oo`0eo`80
9ol00`3oo`0[o`00<?l00`3oo`0Wo`80=Ol00`3oo`0co`80:Ol00`3oo`0[o`00<Ol00`3oo`0Xo`80
<ol00`3oo`0bo`030?oo02So00<0ool0;?l0037o00<0ool0:_l2037o00<0ool0<?l202[o00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol302[o00<0ool0
:?l302[o00<0ool00P0Xo`030?oo00Oo000do`030?oo033o0`0Wo`030?oo02Ko0P0_o`030?oo03;o
000eo`80<ol202Go00<0ool08_l4033o00<0ool0<ol003Oo00<0ool0<_l4027o00<0ool07Ol503?o
00<0ool0=?l003So00<0ool0=Ol501co00<0ool06?l503Ko0P0go`00>Ol203[o1`0Eo`030?oo01?o
1@0io`80>Ol003_o00<0ool0?_l900co00<0ool02?l;03_o0`0ko`00??l204Oo5`14o`80?_l003ko
0P1Ao`030?oo04Wo0`10o`00@?l204oo00<0ool0AOl404?o0012o`80COl00`3oo`10o`D0Aol004Co
0`1:o`030?oo03_o1@1<o`00Aol204So00<0ool0=Ol6057o0019o`<0AOl00`3oo`0`o`D0Eol004co
0`0^o`801?l200?o1005o`030?oo02_o1@1Lo`00Col402Wo00@0ool02?l00`3oo`06o`030?oo02Oo
101Qo`00Dol302Ko00@0ool02Ol00`3oo`05o`809?l406Go001Fo`<08ol0103oo`0:o`030?oo00Co
00<0ool08?l306Wo001Io`@07ol0103oo`08o`040?oo00Go00<0ool07Ol306co001Mo`<07Ol200[o
0P06o`030?oo01_o0P1_o`00H?l302ko00<0ool06?l3077o001So`@0:_l00`3oo`0Fo`80M?l006Oo
100Vo`030?oo01;o101fo`00Jol5027o00<0ool03_l407[o001`o`D07?l00`3oo`07o`L0O_l007Go
9P25o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8749 .15466 m
.86328 .1383 L
.85648 .1305 L
.84887 .12326 L
.8447 .12008 L
.84083 .11768 L
.83894 .1167 L
.83685 .11578 L
.83488 .11505 L
.833 .11449 L
.83079 .11398 L
.82961 .11378 L
.82832 .1136 L
.82768 .11354 L
.82697 .11348 L
.82629 .11343 L
.82567 .11341 L
.82504 .11339 L
.82447 .11338 L
.82382 .11339 L
.82319 .1134 L
.82205 .11346 L
.82147 .1135 L
.82083 .11355 L
.81862 .1138 L
.81737 .114 L
.81601 .11424 L
.81354 .11478 L
.80855 .11619 L
.80387 .11785 L
.79336 .1226 L
.78464 .12765 L
.77654 .13376 L
.77064 .13999 L
.76798 .14385 L
.76595 .14763 L
.76447 .15116 L
.76314 .15526 L
.76202 .15978 L
.76115 .16419 L
.75964 .17355 L
.75784 .18382 L
.75662 .18915 L
.75493 .19485 L
.75304 .19981 L
.75094 .20419 L
.74522 .21274 L
.74139 .21687 L
.73758 .22026 L
.72143 .23087 L
Mistroke
.70451 .24139 L
.68888 .25285 L
.68066 .25855 L
.67244 .26326 L
.66766 .26544 L
.6623 .26737 L
.65988 .26807 L
.65728 .2687 L
.65503 .26915 L
.65256 .26956 L
.65114 .26975 L
.64983 .2699 L
.64859 .27002 L
.64722 .27013 L
.64599 .2702 L
.64483 .27025 L
.64361 .27029 L
.64228 .27032 L
.64093 .27032 L
.64016 .27032 L
.63944 .2703 L
.63799 .27027 L
.63718 .27024 L
.63645 .27021 L
.63368 .27006 L
.63108 .26987 L
.62148 .26895 L
.61644 .26844 L
.61109 .26799 L
.60965 .26789 L
.60829 .26781 L
.60697 .26775 L
.60575 .2677 L
.6046 .26766 L
.60338 .26764 L
.60216 .26763 L
.60082 .26764 L
.60015 .26765 L
.59944 .26766 L
.59817 .2677 L
.59682 .26776 L
.59534 .26785 L
.59269 .26808 L
.59025 .26835 L
.58554 .26908 L
.58128 .26994 L
.57182 .27245 L
.55135 .27897 L
.54232 .28151 L
.5323 .28374 L
Mistroke
.52763 .28455 L
.52258 .28529 L
.51335 .28629 L
.50787 .28672 L
.50263 .28705 L
.4979 .28728 L
.49272 .28747 L
.49003 .28754 L
.48862 .28758 L
.48712 .2876 L
.48581 .28762 L
.48512 .28763 L
.48438 .28764 L
.48307 .28765 L
.48186 .28765 L
.48059 .28765 L
.47922 .28764 L
.47793 .28763 L
.47673 .28761 L
.47566 .28759 L
.47448 .28756 L
.4721 .28749 L
.4693 .28737 L
.46676 .28723 L
.46109 .28678 L
.45619 .28623 L
.45113 .2855 L
.44207 .28377 L
.43262 .28151 L
.4154 .27718 L
.39362 .27324 L
.38301 .27188 L
.37174 .27042 L
.36174 .26877 L
.35636 .26759 L
.35157 .26629 L
.34379 .2635 L
.33628 .25976 L
.30557 .23831 L
.28751 .22667 L
.27951 .22078 L
.27631 .21792 L
.27383 .21522 L
.27177 .21232 L
.27093 .21077 L
.27058 .20997 L
.27026 .20912 L
.27002 .20836 L
.26984 .20766 L
.2697 .20696 L
Mistroke
.2696 .2063 L
.26953 .20552 L
.2695 .2048 L
.26952 .20401 L
.26959 .20317 L
.26971 .20235 L
.26986 .2016 L
.27009 .20074 L
.27036 .19991 L
.27177 .19685 L
.27283 .19512 L
.27399 .19352 L
.27711 .18991 L
.29103 .17789 L
.29807 .17201 L
.30123 .16891 L
.30358 .1661 L
.30459 .16457 L
.30531 .1632 L
.30563 .16247 L
.3059 .16169 L
.30603 .16126 L
.30613 .16086 L
.30621 .16049 L
.30627 .16009 L
.30634 .15945 L
.30637 .15885 L
.30637 .1585 L
.30635 .15814 L
.30629 .15749 L
.30617 .15682 L
.30601 .1562 L
.30579 .15552 L
.3055 .1548 L
.30483 .1535 L
.30388 .15212 L
.30167 .14968 L
.29894 .14743 L
.29599 .14547 L
.29264 .14364 L
.28871 .14186 L
.28509 .1405 L
.28095 .13924 L
.27836 .13858 L
.27591 .13806 L
.27333 .1376 L
.27097 .13726 L
.26861 .13699 L
.26637 .1368 L
.26512 .13672 L
Mistroke
.26398 .13667 L
.26271 .13662 L
.26207 .13661 L
.26137 .13659 L
.26007 .13658 L
.25886 .13659 L
.25822 .1366 L
.25752 .13661 L
.25679 .13663 L
.2561 .13666 L
.25367 .13677 L
.25229 .13686 L
.25103 .13696 L
.24632 .1374 L
.2413 .13802 L
.21854 .14162 L
.19663 .14499 L
.18747 .14635 L
.17801 .14799 L
.17298 .14908 L
.16852 .15026 L
.16027 .15324 L
.15634 .15517 L
.15225 .15767 L
.14827 .16069 L
.14475 .16397 L
.14173 .16739 L
.13872 .17149 L
.13425 .1795 L
.13221 .18442 L
.13047 .18989 L
.12934 .19468 L
.12885 .19744 L
.12849 .20005 L
.12836 .20125 L
.12824 .20253 L
.12816 .20364 L
.12809 .20484 L
.12806 .20557 L
.12804 .20625 L
.12802 .20758 L
.12802 .20878 L
.12804 .21006 L
.12809 .21129 L
.12817 .21262 L
.12826 .21386 L
.12837 .21499 L
.12871 .21774 L
.12913 .22032 L
.13009 .22471 L
Mistroke
.13122 .22867 L
.13456 .23727 L
.13925 .24608 L
.14444 .2539 L
.15575 .26751 L
.16244 .27411 L
.17008 .28052 L
.17783 .28582 L
.18538 .28994 L
.19399 .29354 L
.20368 .29651 L
.21296 .29864 L
.22189 .30038 L
.23156 .30233 L
.24169 .30484 L
.25016 .30756 L
.2593 .31122 L
.27865 .32071 L
.28715 .32482 L
.29664 .32874 L
.30617 .33176 L
.31534 .33382 L
.32053 .33469 L
.32531 .33536 L
.33618 .33661 L
.3461 .33774 L
.35648 .33925 L
.36757 .3415 L
.37746 .3441 L
.39655 .34978 L
.40495 .35194 L
.40948 .35289 L
.41438 .35373 L
.41717 .35411 L
.41972 .35441 L
.42235 .35465 L
.42479 .35484 L
.4273 .35498 L
.42863 .35503 L
.43004 .35508 L
.43128 .35511 L
.43244 .35513 L
.43371 .35514 L
.43508 .35514 L
.43584 .35513 L
.43664 .35513 L
.43808 .35511 L
.43888 .35509 L
.43963 .35507 L
.44136 .35502 L
Mistroke
.44424 .3549 L
.44736 .35474 L
.45866 .3539 L
.50184 .34992 L
.52109 .34849 L
.53948 .34713 L
.5583 .34539 L
.56902 .34406 L
.57864 .34251 L
.58867 .34042 L
.59762 .33806 L
.61708 .3314 L
.65453 .31774 L
.69102 .30761 L
.71107 .30273 L
.73029 .29889 L
.75165 .29561 L
.77149 .29256 L
.78089 .2907 L
.7904 .28831 L
.7994 .28543 L
.80876 .28171 L
.8242 .27394 L
.8397 .26377 L
.85354 .25117 L
.86 .24341 L
.86658 .23383 L
.87651 .21501 L
.8788 .20945 L
.88066 .20426 L
.88203 .1997 L
.88265 .19732 L
.88321 .19479 L
.88365 .19237 L
.88397 .19018 L
.88409 .18914 L
.88419 .18801 L
.88427 .18695 L
.88433 .18597 L
.88435 .1854 L
.88436 .18478 L
.88437 .18366 L
.88435 .18258 L
.88431 .18159 L
.88425 .18052 L
.88414 .17935 L
.884 .17813 L
.88384 .177 L
.88347 .17495 L
.88303 .17302 L
Mistroke
.88198 .1694 L
.88052 .1655 L
.87856 .16124 L
.8749 .15466 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Col803Go001lo`040?oo00So1@04o`80C_l200So0P0co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o0`0<o`030?oo033o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4?l00`3oo`0_o`00OOl200_o0P05o`030?oo04Oo00<0ool04Ol00`3oo`0^o`00TOl00`3oo`16o`03
0?oo01?o00<0ool0;Ol004[o3@0jo`030?oo04Go00<0ool05Ol00`3oo`0/o`00A?l600co0`0ho`03
0?oo04Co00<0ool05ol00`3oo`0[o`00?Ol701Go0P0fo`030?oo04Co00<0ool05ol00`3oo`0[o`00
>?l501ko0P0do`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=_l202Co0P0co`030?oo04?o00<0ool0
6Ol00`3oo`0Zo`00=?l202Oo00<0ool0<Ol00`3oo`12o`030?oo01_o00<0ool0:Ol003?o00<0ool0
9ol00`3oo`0ao`030?oo04;o00<0ool07?l00`3oo`0Xo`00<_l00`3oo`0Xo`030?oo037o00<0ool0
@_l00`3oo`0Lo`030?oo02So000ao`030?oo02So00<0ool0<_l00`3oo`12o`030?oo01go00<0ool0
9ol0033o00<0ool0:?l00`3oo`0co`030?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Xo`03
0?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Mo`80:?l002oo00<0ool09ol00`3oo`0Po`04
0?oo00So00<0ool01_l00`3oo`11o`030?oo01oo00<0ool09_l002oo00<0ool09Ol202?o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0Oo`030?oo02Ko000_o`030?oo02Co00<0ool07Ol400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07ol00`3oo`0Vo`00;_l00`3oo`0To`030?oo02Co00@0ool0
2?l0103oo`05o`030?oo047o00<0ool07ol00`3oo`0Vo`00;_l00`3oo`0So`030?oo02Ko0P0:o`80
1_l00`3oo`10o`030?oo01oo0P0Xo`00;_l00`3oo`0So`030?oo03[o00<0ool0@?l00`3oo`0Oo`03
0?oo02Oo000^o`030?oo02;o00<0ool0>ol00`3oo`10o`030?oo01oo00<0ool09ol002ko00<0ool0
8_l00`3oo`0ko`030?oo03oo00<0ool08?l00`3oo`0Wo`001ol500;o0P04o`806?l00`3oo`08o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool04_l500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool05_l00`3oo`08o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Ko0P09o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02_l00`3oo`0Eo`030?oo00Wo00<0ool0
6?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01So
00<0ool01002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`08o`03
0?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Wo00<0ool0
1Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo0P0Io`030?oo00So00<0ool0
6Ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201co00<0ool01Ol200[o0`0Ro`03
0?oo02;o0P0:o`<01ol201[o0P09o`030?oo01Wo0P0:o`<00_l002ko00<0ool0:?l203Ko00<0ool0
=ol00`3oo`0Uo`030?oo02[o000_o`030?oo02Wo0P0do`030?oo03Ko00<0ool09Ol00`3oo`0[o`00
;ol00`3oo`0[o`030?oo037o00<0ool0=?l202Oo00<0ool0;?l0033o00<0ool0:ol2037o00<0ool0
<ol00`3oo`0Vo`030?oo02go000ao`030?oo02co00<0ool0;_l00`3oo`0bo`030?oo02Oo00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o
00@0ool02?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol00`3oo`0Zo`03
0?oo02Wo00@0oooo0P0Vo`050?ooo`00:?l00`3oo`07o`00=?l00`3oo`0^o`80:_l00`3oo`0[o`80
:ol00`3oo`0ao`00=?l00`3oo`0`o`<09ol00`3oo`0Yo`80:ol203Co000eo`030?oo03;o1P0Qo`03
0?oo01;o5`0[o`80=_l003Ko0P0ho`P06Ol00`3oo`0?o`<0@?l203So000ho`030?oo03go1@0Do`03
0?oo00co0`11o`80>_l003Wo00<0ool0@Ol4013o00<0ool01ol504;o0P0lo`00>_l00`3oo`14o`@0
3?l00`3oo`02o`D0A?l303ko000ko`030?oo04Oo4@17o`80@Ol003co0P1Co`030?oo04;o1`13o`00
?_l504ko00<0ool0?Ol504[o0013o`D0BOl00`3oo`0ho`D0Col004So0`16o`030?oo03;o1P1Do`00
Bol404;o00<0ool0;_l405[o001?o`80;?l200Co0P03o`@01Ol00`3oo`0[o`<0G_l0057o0P0Yo`04
0?oo00So00<0ool01_l00`3oo`0Xo`<0HOl005?o0P0Wo`040?oo00Wo00<0ool01Ol202Ko0`1To`00
EOl202Go00@0ool02_l00`3oo`04o`030?oo02?o0P1Wo`00Eol202?o00@0ool02?l0103oo`05o`03
0?oo023o0`1Yo`00FOl3027o0P0:o`801_l00`3oo`0No`80K?l005co100ao`030?oo01Wo1@1^o`00
H?l902So00<0ool05Ol407?o001Yo`H08_l00`3oo`0Co`80Mol006oo0`0Oo`030?oo00[o2@1io`00
L_l201go00<0ool02P22o`00M?l501?o202<o`00NOlC00Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co
00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3o
o`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00
oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87511 .15655 m
.86098 .13991 L
.84619 .12246 L
.84274 .11886 L
.83914 .11565 L
.83742 .11436 L
.83579 .11332 L
.83434 .11254 L
.83272 .11184 L
.83186 .11154 L
.83092 .11128 L
.82992 .11106 L
.82947 .11098 L
.82897 .11091 L
.82848 .11086 L
.82802 .11082 L
.82711 .11078 L
.82626 .11079 L
.82535 .11084 L
.82478 .11089 L
.82425 .11095 L
.82304 .11114 L
.82194 .11137 L
.82077 .11166 L
.81631 .11314 L
.80784 .11702 L
.798 .12227 L
.78045 .13257 L
.77236 .13847 L
.76866 .14204 L
.76586 .1456 L
.76391 .14898 L
.76223 .15306 L
.76157 .15527 L
.76096 .15776 L
.76009 .16259 L
.75893 .1727 L
.75829 .17865 L
.75754 .18439 L
.757 .18765 L
.7563 .19116 L
.75476 .19712 L
.75271 .20284 L
.75037 .20769 L
.74773 .21187 L
.74494 .21535 L
.74155 .21868 L
.73761 .22174 L
.72897 .22669 L
.72087 .23049 L
Mistroke
.71341 .23439 L
.70581 .2396 L
.69771 .24693 L
.69003 .25481 L
.68144 .26298 L
.67654 .26682 L
.67103 .27024 L
.66591 .27255 L
.66289 .27354 L
.66142 .27393 L
.66001 .27424 L
.65872 .27448 L
.6573 .2747 L
.656 .27484 L
.65479 .27494 L
.65347 .27501 L
.65201 .27504 L
.65127 .27504 L
.65048 .27502 L
.64902 .27495 L
.64755 .27484 L
.6462 .27469 L
.64315 .27425 L
.64031 .27369 L
.63762 .27306 L
.62503 .26903 L
.61297 .2646 L
.60658 .26255 L
.60353 .26173 L
.60078 .26111 L
.59821 .26065 L
.59581 .26034 L
.5945 .26021 L
.5933 .26013 L
.592 .26008 L
.59134 .26007 L
.59061 .26007 L
.58993 .26009 L
.58921 .26011 L
.58794 .26019 L
.58723 .26025 L
.58657 .26031 L
.58509 .2605 L
.58262 .26093 L
.58002 .26155 L
.57518 .26313 L
.57068 .26504 L
.56242 .26937 L
.54548 .27893 L
.54081 .28117 L
Mistroke
.53574 .28326 L
.52637 .28618 L
.52112 .28735 L
.51604 .28824 L
.50623 .28949 L
.49551 .29045 L
.48942 .29088 L
.48643 .29106 L
.48368 .2912 L
.48114 .29132 L
.47837 .29141 L
.47694 .29145 L
.47561 .29148 L
.47488 .29149 L
.47421 .29149 L
.47345 .2915 L
.47273 .2915 L
.47155 .29149 L
.47031 .29148 L
.46926 .29146 L
.46811 .29142 L
.46688 .29137 L
.46557 .29131 L
.46322 .29115 L
.46072 .29092 L
.45805 .29059 L
.45282 .28968 L
.44815 .28854 L
.44395 .28724 L
.42804 .28081 L
.41917 .27743 L
.41397 .27585 L
.40882 .27461 L
.40314 .2736 L
.40001 .27319 L
.39657 .27283 L
.39337 .27257 L
.39029 .27238 L
.38749 .27225 L
.38443 .27215 L
.37786 .272 L
.37466 .27193 L
.37176 .27186 L
.36915 .27177 L
.36644 .27165 L
.36378 .2715 L
.36095 .27127 L
.35823 .27097 L
.35584 .27063 L
.3532 .27015 L
Mistroke
.35083 .2696 L
.3469 .26838 L
.34485 .26755 L
.34306 .26669 L
.33971 .26465 L
.33643 .26201 L
.33061 .25563 L
.30205 .22987 L
.28338 .22001 L
.27879 .21731 L
.27456 .21434 L
.27282 .21285 L
.27139 .21141 L
.27029 .21011 L
.26931 .20865 L
.26852 .20707 L
.2682 .20619 L
.26798 .20536 L
.26783 .20458 L
.26775 .20386 L
.26773 .20302 L
.26777 .20223 L
.26781 .20181 L
.26788 .20135 L
.26798 .20088 L
.26808 .20042 L
.26864 .19874 L
.26907 .19777 L
.26953 .19688 L
.27079 .19487 L
.27337 .19171 L
.27688 .18828 L
.2853 .18161 L
.29447 .17511 L
.29842 .17215 L
.30151 .16954 L
.30371 .16728 L
.30464 .16611 L
.30545 .16486 L
.3058 .16421 L
.30606 .16362 L
.30628 .16302 L
.30645 .16247 L
.30658 .16185 L
.30663 .16153 L
.30667 .16118 L
.30669 .16053 L
.30667 .15995 L
.3066 .15938 L
.30648 .15876 L
Mistroke
.3063 .1581 L
.30609 .15749 L
.30559 .1564 L
.30491 .15522 L
.30281 .15253 L
.30033 .15009 L
.29429 .14548 L
.28756 .14156 L
.28364 .13973 L
.27984 .13828 L
.27575 .13707 L
.27353 .13656 L
.27109 .13613 L
.26911 .13587 L
.26797 .13576 L
.26691 .13567 L
.26576 .13561 L
.26472 .13557 L
.2635 .13555 L
.26235 .13555 L
.26133 .13558 L
.26025 .13562 L
.25932 .13567 L
.25829 .13574 L
.25596 .13594 L
.25374 .1362 L
.24858 .13699 L
.24272 .13812 L
.23147 .14058 L
.22006 .14304 L
.20919 .14504 L
.19805 .14658 L
.19234 .14718 L
.1864 .14769 L
.1761 .14849 L
.17346 .14873 L
.17073 .14904 L
.16605 .14975 L
.16386 .1502 L
.16157 .15077 L
.15761 .15211 L
.15397 .15382 L
.15081 .1558 L
.14762 .15836 L
.14434 .16174 L
.14119 .16585 L
.13845 .17029 L
.13617 .17477 L
.13395 .18003 L
.13219 .18513 L
Mistroke
.13085 .18991 L
.12959 .19569 L
.12906 .19888 L
.12867 .20188 L
.12853 .20323 L
.1284 .20465 L
.12831 .20587 L
.12824 .20717 L
.12818 .20856 L
.12815 .20986 L
.12814 .21064 L
.12814 .21138 L
.12815 .21276 L
.12819 .21396 L
.12825 .21524 L
.12832 .21643 L
.12841 .21752 L
.12867 .22001 L
.12904 .22257 L
.12997 .22723 L
.13134 .23208 L
.13454 .24014 L
.13825 .24713 L
.14871 .2624 L
.16012 .27584 L
.16663 .28205 L
.17423 .28776 L
.1785 .29027 L
.18346 .29259 L
.18819 .29428 L
.1928 .29551 L
.19772 .29645 L
.20057 .29685 L
.20325 .29714 L
.20823 .29754 L
.21105 .2977 L
.21362 .29783 L
.21851 .29805 L
.22109 .29818 L
.22381 .29833 L
.22619 .2985 L
.22879 .29873 L
.23332 .29928 L
.23594 .2997 L
.23877 .30024 L
.24382 .30152 L
.24863 .30312 L
.25296 .30493 L
.26244 .31008 L
.27208 .31663 L
Mistroke
.28103 .32295 L
.28974 .32829 L
.29796 .33203 L
.30283 .33361 L
.30752 .33471 L
.31019 .33518 L
.31276 .33553 L
.3151 .33578 L
.31766 .33598 L
.31901 .33605 L
.3205 .33612 L
.32125 .33615 L
.32207 .33618 L
.32354 .33621 L
.32501 .33623 L
.32641 .33624 L
.32768 .33624 L
.32906 .33624 L
.33055 .33622 L
.33196 .33621 L
.33519 .33616 L
.33672 .33614 L
.33833 .33612 L
.33918 .33612 L
.3401 .33611 L
.34094 .3361 L
.34173 .3361 L
.34325 .3361 L
.34469 .33612 L
.34598 .33613 L
.34737 .33616 L
.34889 .33621 L
.35048 .33628 L
.35333 .33644 L
.3547 .33655 L
.35618 .33668 L
.35885 .33698 L
.36377 .33773 L
.36649 .33828 L
.36932 .33895 L
.37429 .3404 L
.38357 .34398 L
.39158 .34772 L
.40009 .35173 L
.40935 .35531 L
.41445 .35673 L
.41744 .35737 L
.42031 .35785 L
.423 .35819 L
.42551 .35842 L
Mistroke
.42692 .35851 L
.42767 .35855 L
.42847 .35859 L
.42995 .35863 L
.43133 .35865 L
.43264 .35865 L
.43409 .35863 L
.43562 .35858 L
.43705 .35852 L
.43965 .35837 L
.44245 .35814 L
.44894 .35742 L
.4549 .35656 L
.47695 .35251 L
.488 .35037 L
.49848 .34853 L
.50722 .34728 L
.51126 .34681 L
.51489 .34645 L
.52213 .34591 L
.52622 .34569 L
.53022 .34552 L
.53905 .34525 L
.54943 .34499 L
.55496 .34484 L
.56027 .34465 L
.56507 .34442 L
.57026 .34409 L
.5759 .34362 L
.58173 .34294 L
.5866 .3422 L
.5918 .3412 L
.60151 .33865 L
.60999 .33567 L
.62888 .32752 L
.66739 .31344 L
.70236 .30329 L
.7118 .30077 L
.7212 .29868 L
.7317 .29691 L
.73755 .29616 L
.744 .29547 L
.75497 .2945 L
.76658 .29347 L
.77305 .29275 L
.77871 .29194 L
.78416 .29095 L
.78904 .28983 L
.79798 .28705 L
.80541 .2839 L
Mistroke
.82017 .27556 L
.83363 .26655 L
.84622 .25618 L
.85856 .24211 L
.87153 .2223 L
.87758 .21122 L
.88296 .1996 L
.88482 .19464 L
.88565 .19203 L
.88628 .18966 L
.88674 .18754 L
.88705 .18566 L
.88718 .18466 L
.88727 .18362 L
.88733 .1826 L
.88735 .18202 L
.88735 .18149 L
.88734 .18064 L
.88729 .17975 L
.88723 .179 L
.88713 .17817 L
.887 .1773 L
.88682 .17637 L
.8864 .17471 L
.88587 .17305 L
.88525 .17149 L
.8838 .16856 L
.88018 .16299 L
.87511 .15655 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1Bo`@0=_l007co00@0ool0
2ol00`3oo`03o`030?oo053o0P04o`80=?l007co00@0ool02?l500Co0P1?o`802?l00`3oo`0ao`00
M_l400;o00@0ool02?l0103oo`05o`030?oo04co0P0;o`030?oo033o001lo`040?oo00Wo00<0o`00
1Ol00`3oo`1:o`803_l00`3oo`0_o`00OOl200_o0P05o`030?oo04So0P0Ao`030?oo02ko002Ao`03
0?oo04Oo00<0ool04Ol00`3oo`0^o`00Bol<03[o00<0ool0AOl201Go00<0ool0;Ol004So0`0<o`<0
=ol00`3oo`14o`030?oo01Ko00<0ool0;?l0043o200Bo`030?oo03Co00<0ool0@ol00`3oo`0Ho`03
0?oo02_o000go`T06ol00`3oo`0co`030?oo04?o00<0ool06?l00`3oo`0[o`00=?l302Go00<0ool0
<_l00`3oo`12o`030?oo01[o00<0ool0:_l003Co00<0ool09_l00`3oo`0ao`030?oo04;o00<0ool0
6ol00`3oo`0Yo`00<ol00`3oo`0Wo`030?oo037o00<0ool0@_l00`3oo`0Lo`030?oo02So000bo`03
0?oo02So00<0ool0<Ol00`3oo`12o`030?oo01go00<0ool09ol0037o00<0ool0:?l203?o00<0ool0
@Ol00`3oo`0Oo`030?oo02Ko000`o`030?oo02So0P0do`030?oo047o00<0ool07ol00`3oo`0Vo`00
<?l00`3oo`0Wo`030?oo023o0P04o`800ol400Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000_o`03
0?oo02Ko0P0Ro`040?oo00So00<0ool01_l00`3oo`11o`030?oo01oo0P0Wo`00;ol00`3oo`0Uo`03
0?oo02;o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0Po`030?oo02Go000_o`030?oo02Co00<0ool0
7Ol400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool07ol00`3oo`0Vo`00;_l00`3oo`0So`80
9_l0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000^o`030?oo02?o00<0ool0
9_l200[o0P06o`030?oo043o00<0ool08?l00`3oo`0Vo`00;_l00`3oo`0Ro`030?oo03_o00<0ool0
@?l00`3oo`0Po`030?oo02Ko000^o`030?oo02;o00<0ool0>ol00`3oo`10o`030?oo01oo00<0ool0
9ol002ko00<0ool08_l00`3oo`0ko`030?oo03oo00<0ool08?l00`3oo`0Wo`001ol500;o0P04o`80
6?l00`3oo`08o`D05Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool0
4Ol500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`03
0?oo00Wo00<0ool06?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`03
0?oo01Oo0P08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02ol00`3oo`0Do`03
0?oo00So00<0ool06Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Wo00<0o`000`02o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co0P0Go`03
0?oo00So00<0ool06Ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool0
6_l200Go00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`09o`806Ol00`3oo`07o`03
0?oo01[o00<0ool02?l00`3oo`02o`002?l200[o0`0Go`030?oo00Wo0P0No`801?l200[o0`0Ro`03
0?oo02;o0P0:o`<02?l00`3oo`0Ho`802?l00`3oo`0Jo`802_l300;o000^o`030?oo02_o00<0ool0
<_l00`3oo`0go`809Ol00`3oo`0[o`00;ol00`3oo`0[o`030?oo037o00<0ool0=Ol202Ko00<0ool0
;?l002oo00<0ool0;?l00`3oo`0`o`030?oo03?o0P0Xo`030?oo02co000`o`030?oo02co0P0`o`03
0?oo03;o00<0ool09ol00`3oo`0]o`00<?l00`3oo`0^o`030?oo02go00<0ool0<Ol00`3oo`0Wo`03
0?oo02ko000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`040?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`03o`040?oo00So00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool08ol00`3oo`03o`030?oo02Wo00<0ool0
:_l00`3oo`0Yo`030?oo00;o00<0ool09?l00`3oo`0202So00<0ool01ol003?o00<0ool0;ol00`3o
o`0Yo`030?oo013o2`0Bo`030?oo02Ko0P0do`00=?l00`3oo`0_o`80:Ol00`3oo`0?o`030?oo00Wo
0P0>o`80:?l00`3oo`0do`00=Ol00`3oo`0_o`@09_l00`3oo`0=o`803_l300[o00<0ool09ol00`3o
o`0eo`00=Ol00`3oo`0co`d06Ol00`3oo`0<o`030?oo017o1003o`<0:Ol00`3oo`0fo`00=_l00`3o
o`0oo`D05?l00`3oo`0:o`805ol502Wo0P0io`00=ol00`3oo`13o`804_l00`3oo`07o`<0AOl203_o
000ho`030?oo04Co0P0@o`030?oo00Ko00<0ool0A?l203go000io`80A_l300go00<0ool01P15o`80
?ol003_o0P17oa00A?l7047o000mo`D0Col00`3oo`0io`/0B?l004;o2P15o`030?oo03Ko0`1Co`00
C?l304;o00<0ool0<Ol505Ko001?o`030?oo03oo00<0ool0;_l305_o001@o`80:ol200Co0P03o`@0
1Ol00`3oo`0[o`<0G_l005;o0P0Xo`040?oo00So00<0ool01_l00`3oo`0Xo`<0HOl005Co00<0ool0
9Ol0103oo`09o`030?oo00Go0P0Wo`80I?l005Go0P0Uo`040?oo00[o00<0ool01?l00`3oo`0So`<0
I_l005Oo0P0So`040?oo00So00@0ool01Ol00`3oo`0Qo`80JOl005Wo00<0ool08Ol200[o0P06o`03
0?oo01ko0`1[o`00F_l303Co00<0ool07?l206ko001Moa408ol00`3oo`0Io`<0L?l006ko0`0Po`03
0?oo01Go101co`00LOl201ko00<0ool00olB07Oo001co`806_l808Wo001eo`804Ol700;o00<0ool0
S?l007Oo1009o`@02Ol00`3oo`2<o`00Nol900go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87532 .15845 m
.85869 .14153 L
.84351 .12167 L
.83783 .11407 L
.83634 .11238 L
.83497 .11104 L
.83374 .11003 L
.83244 .10919 L
.83175 .10885 L
.83097 .10854 L
.83063 .10844 L
.83026 .10834 L
.82958 .10821 L
.82917 .10817 L
.82871 .10814 L
.82828 .10813 L
.82787 .10815 L
.82746 .10818 L
.82708 .10823 L
.82664 .1083 L
.82622 .10839 L
.82459 .10886 L
.823 .10948 L
.81908 .11151 L
.80032 .1232 L
.78038 .13426 L
.77201 .13914 L
.76859 .14161 L
.76591 .14403 L
.76342 .1471 L
.76235 .14887 L
.76139 .15089 L
.76072 .15268 L
.7602 .15447 L
.75972 .15655 L
.75932 .15893 L
.75902 .16125 L
.75877 .16389 L
.75858 .1665 L
.75843 .16901 L
.7581 .17534 L
.7577 .18138 L
.75739 .18465 L
.75697 .18825 L
.75639 .19199 L
.75572 .19542 L
.75414 .20137 L
.75304 .20452 L
.75188 .20725 L
.74921 .21215 L
Mistroke
.7463 .21607 L
.74269 .21967 L
.73845 .22272 L
.73415 .22498 L
.72999 .22668 L
.72223 .22935 L
.71808 .23091 L
.71435 .23264 L
.71063 .23488 L
.70671 .23796 L
.70048 .24459 L
.69377 .25346 L
.68594 .26371 L
.68135 .26879 L
.67604 .27349 L
.67101 .27676 L
.66799 .2782 L
.66658 .27875 L
.66506 .27925 L
.66375 .27962 L
.6623 .27994 L
.66096 .28018 L
.6597 .28034 L
.65819 .28047 L
.65681 .28052 L
.65603 .28052 L
.65519 .2805 L
.65365 .28042 L
.65212 .28026 L
.65071 .28006 L
.64907 .27977 L
.64751 .27942 L
.64163 .27762 L
.63515 .27486 L
.62807 .27116 L
.61526 .26357 L
.6087 .25975 L
.60187 .25625 L
.59888 .25497 L
.5958 .25386 L
.59322 .25311 L
.59176 .25278 L
.59045 .25253 L
.58913 .25234 L
.5879 .25221 L
.58661 .25214 L
.58589 .25212 L
.58523 .25212 L
.58413 .25216 L
.58294 .25225 L
Mistroke
.5817 .2524 L
.58055 .25259 L
.57927 .25287 L
.57788 .25324 L
.57539 .2541 L
.57083 .25628 L
.56567 .25961 L
.56101 .26323 L
.5447 .27709 L
.54021 .28037 L
.53539 .28337 L
.53114 .28556 L
.52646 .2875 L
.5214 .28912 L
.51652 .29032 L
.51053 .29146 L
.5049 .29232 L
.49348 .29371 L
.48839 .29425 L
.48279 .29478 L
.48027 .29499 L
.47758 .29519 L
.47528 .29532 L
.47408 .29538 L
.4728 .29543 L
.47144 .29547 L
.47019 .29549 L
.46946 .2955 L
.46877 .2955 L
.4675 .2955 L
.46631 .29547 L
.46523 .29543 L
.46406 .29538 L
.46283 .2953 L
.46048 .29508 L
.45918 .29492 L
.45801 .29475 L
.45374 .29388 L
.45145 .29324 L
.44902 .29242 L
.44478 .29061 L
.43636 .28583 L
.42823 .28081 L
.42009 .27675 L
.41497 .27493 L
.41256 .27425 L
.40988 .27363 L
.40712 .27312 L
.40453 .27275 L
.40325 .2726 L
Mistroke
.40185 .27246 L
.40051 .27235 L
.39925 .27226 L
.39777 .27219 L
.39611 .27213 L
.39448 .2721 L
.39356 .27209 L
.39271 .27209 L
.3912 .2721 L
.38954 .27213 L
.38795 .27217 L
.38648 .27223 L
.38302 .27239 L
.37975 .27259 L
.3737 .27301 L
.36742 .27341 L
.36575 .27349 L
.364 .27356 L
.36316 .27359 L
.36237 .27361 L
.36092 .27363 L
.35958 .27364 L
.35819 .27362 L
.35703 .27358 L
.35578 .27352 L
.35448 .27343 L
.35311 .2733 L
.35187 .27314 L
.35077 .27297 L
.34863 .27252 L
.34646 .27189 L
.3444 .27106 L
.34265 .27013 L
.34115 .26912 L
.33986 .26805 L
.33775 .26573 L
.33584 .26283 L
.33407 .2593 L
.33064 .25119 L
.32655 .24336 L
.32382 .23959 L
.32023 .23578 L
.31542 .23189 L
.3103 .22868 L
.28654 .21834 L
.28082 .21594 L
.276 .21361 L
.27233 .21141 L
.27072 .21023 L
.2692 .20889 L
Mistroke
.26797 .20752 L
.26708 .20621 L
.26669 .20547 L
.2664 .20478 L
.26615 .20398 L
.26598 .2032 L
.26591 .20253 L
.26589 .20181 L
.26592 .20119 L
.26601 .20049 L
.26617 .19974 L
.26641 .19893 L
.26702 .19744 L
.26792 .19582 L
.26919 .19404 L
.27251 .19044 L
.28033 .18416 L
.29058 .17748 L
.29915 .17199 L
.30208 .16982 L
.30449 .16762 L
.30545 .1665 L
.30586 .16591 L
.30624 .16528 L
.30653 .16471 L
.30674 .16418 L
.3069 .16365 L
.307 .16315 L
.30706 .16264 L
.30708 .16216 L
.30705 .16162 L
.30697 .16105 L
.30684 .16047 L
.30664 .15986 L
.30616 .15874 L
.30556 .15769 L
.30489 .1567 L
.30297 .15438 L
.29071 .1439 L
.28717 .1415 L
.28323 .13917 L
.27967 .13744 L
.27768 .13664 L
.27579 .136 L
.27392 .13548 L
.27217 .1351 L
.27123 .13493 L
.2702 .13479 L
.26908 .13466 L
.26801 .13458 L
Mistroke
.26701 .13453 L
.2661 .13451 L
.26502 .13451 L
.264 .13454 L
.26303 .13459 L
.26213 .13466 L
.26006 .13487 L
.2577 .1352 L
.25548 .13559 L
.25024 .13674 L
.23992 .1395 L
.22653 .14318 L
.22038 .14469 L
.21352 .14617 L
.20767 .14721 L
.20146 .14808 L
.19811 .14844 L
.19509 .1487 L
.19226 .14889 L
.19075 .14897 L
.18935 .14902 L
.18783 .14907 L
.18646 .14911 L
.18501 .14913 L
.18421 .14913 L
.18347 .14914 L
.18225 .14914 L
.18093 .14913 L
.17958 .14911 L
.17833 .14909 L
.17314 .14898 L
.17179 .14896 L
.17106 .14895 L
.17038 .14894 L
.16975 .14893 L
.16908 .14893 L
.16793 .14893 L
.16684 .14894 L
.16623 .14896 L
.16566 .14897 L
.16454 .14902 L
.16392 .14906 L
.16337 .1491 L
.16239 .14918 L
.16135 .1493 L
.15951 .14957 L
.15769 .14996 L
.15583 .1505 L
.1543 .15108 L
.15267 .15184 L
Mistroke
.1498 .15364 L
.14688 .15618 L
.14429 .15915 L
.14204 .16236 L
.13734 .17131 L
.13365 .18121 L
.13189 .18735 L
.13056 .19305 L
.12994 .19623 L
.12937 .19966 L
.12896 .20264 L
.12862 .20579 L
.12849 .20733 L
.12839 .20876 L
.12832 .21004 L
.12826 .21142 L
.12822 .21271 L
.12821 .21407 L
.12821 .21522 L
.12823 .21647 L
.12825 .21717 L
.12828 .21791 L
.12834 .21921 L
.12844 .22058 L
.12857 .22207 L
.12889 .22466 L
.12926 .227 L
.13033 .23185 L
.13159 .23598 L
.13525 .24448 L
.14365 .25843 L
.15479 .2741 L
.16123 .28164 L
.16804 .28786 L
.1718 .29049 L
.17612 .29286 L
.1787 .29398 L
.18124 .29487 L
.18377 .29559 L
.18622 .29613 L
.1886 .29652 L
.18997 .29669 L
.19124 .29682 L
.19232 .2969 L
.19352 .29698 L
.19479 .29703 L
.19598 .29706 L
.19665 .29707 L
.19739 .29707 L
.19873 .29705 L
Mistroke
.20005 .29702 L
.20128 .29698 L
.20261 .29691 L
.20409 .29683 L
.20711 .29661 L
.21721 .29576 L
.21968 .29558 L
.22231 .29543 L
.22309 .2954 L
.22382 .29537 L
.22518 .29533 L
.22646 .29531 L
.22784 .29531 L
.2292 .29534 L
.22997 .29536 L
.23068 .29539 L
.23212 .29547 L
.23366 .29559 L
.23507 .29574 L
.2364 .29591 L
.23888 .29633 L
.24131 .29685 L
.24352 .29745 L
.24842 .2992 L
.25359 .30175 L
.25808 .30459 L
.26761 .31232 L
.27627 .32041 L
.28556 .32824 L
.28958 .33093 L
.29406 .33334 L
.29674 .33448 L
.29924 .33535 L
.30162 .33603 L
.30432 .33663 L
.30694 .33704 L
.3084 .33721 L
.30979 .33734 L
.31109 .33743 L
.3123 .33748 L
.31371 .33752 L
.31507 .33753 L
.31654 .33751 L
.31812 .33747 L
.31962 .3374 L
.321 .33731 L
.32442 .33705 L
.32769 .33673 L
.33907 .33543 L
.3422 .33511 L
Mistroke
.34554 .33481 L
.34716 .33469 L
.34868 .33459 L
.35002 .33452 L
.3515 .33446 L
.35296 .33443 L
.3545 .33441 L
.35591 .33442 L
.35723 .33445 L
.35876 .33452 L
.36015 .3346 L
.36167 .33473 L
.36324 .33489 L
.36615 .33531 L
.36879 .33582 L
.37161 .33651 L
.37453 .3374 L
.37953 .33937 L
.38399 .3416 L
.39183 .34646 L
.40014 .35212 L
.40475 .355 L
.40923 .35737 L
.41324 .35908 L
.41778 .36053 L
.42048 .36116 L
.42314 .36163 L
.42559 .36193 L
.42701 .36206 L
.42835 .36215 L
.42986 .36221 L
.43063 .36223 L
.43148 .36224 L
.433 .36223 L
.43443 .3622 L
.43531 .36216 L
.43615 .36212 L
.43773 .36201 L
.43946 .36187 L
.44135 .36167 L
.44813 .36071 L
.45452 .35949 L
.47888 .35323 L
.50061 .34712 L
.50889 .34519 L
.51215 .34457 L
.51556 .34404 L
.51879 .34365 L
.5218 .3434 L
.52348 .3433 L
Mistroke
.52504 .34323 L
.52591 .3432 L
.52687 .34317 L
.52779 .34316 L
.52865 .34314 L
.52953 .34314 L
.53048 .34313 L
.5314 .34314 L
.53224 .34314 L
.53325 .34315 L
.53437 .34317 L
.53645 .34321 L
.5387 .34327 L
.54117 .34335 L
.54582 .34353 L
.55544 .3439 L
.55819 .344 L
.56111 .34409 L
.56363 .34415 L
.56506 .34417 L
.5664 .34419 L
.56765 .34421 L
.569 .34422 L
.57028 .34422 L
.57146 .34421 L
.57252 .3442 L
.57368 .34418 L
.5749 .34416 L
.57603 .34413 L
.57734 .34408 L
.57857 .34403 L
.58088 .3439 L
.58333 .34372 L
.58596 .34347 L
.58854 .34317 L
.59129 .34276 L
.59611 .34183 L
.60064 .34067 L
.60467 .33939 L
.62293 .33122 L
.6606 .31512 L
.69455 .3043 L
.70374 .30123 L
.7131 .29856 L
.72237 .29659 L
.72802 .29572 L
.73349 .29509 L
.73893 .29463 L
.74417 .29431 L
.75001 .29405 L
Mistroke
.75635 .29385 L
.76259 .29365 L
.76594 .29352 L
.76901 .29337 L
.77432 .29303 L
.77704 .29279 L
.77987 .29248 L
.78262 .29211 L
.78544 .29166 L
.79024 .29065 L
.79502 .28928 L
.79984 .28747 L
.80743 .28356 L
.8136 .27947 L
.83607 .26299 L
.85022 .2495 L
.86541 .22933 L
.88019 .20556 L
.88528 .19603 L
.88736 .19149 L
.88901 .18711 L
.88962 .18498 L
.88987 .1839 L
.89008 .18278 L
.89022 .18177 L
.89031 .18088 L
.89035 .18 L
.89034 .17919 L
.89029 .17839 L
.8902 .17755 L
.89008 .17685 L
.88991 .17609 L
.88968 .17529 L
.88938 .17444 L
.8887 .17293 L
.88774 .17127 L
.88644 .16947 L
.88347 .16613 L
.87615 .15921 L
.87532 .15845 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1Co`<0=_l007go0P04o`801Ol300Co00<0ool0DOl200;o0P0eo`00
O?l0103oo`0;o`030?oo00?o00<0ool0Col200Go00<0ool0<ol007co00@0ool02?l500Co0P1>o`80
2?l00`3oo`0bo`00M_l400;o00@0ool02?l0103oo`05o`030?oo04co00<0ool02Ol00`3oo`0ao`00
O?l0103oo`09o`030?l000Go00<0ool0B_l200go00<0ool0<?l007go0P0;o`801Ol00`3oo`18o`80
4?l00`3oo`0_o`00D?l503co00<0ool0Aol00`3oo`0@o`030?oo02oo001=o`<01?l303[o00<0ool0
A_l00`3oo`0Bo`030?oo02ko001:o`<02_l203So00<0ool0A?l201Ko00<0ool0;Ol004Co1P0?o`03
0?oo03Go00<0ool0@ol00`3oo`0Go`030?oo02co000eo`l05_l00`3oo`0do`030?oo04;o0P0Jo`03
0?oo02_o000co`<09Ol00`3oo`0co`030?oo04;o00<0ool06Ol00`3oo`0[o`00<ol00`3oo`0Vo`80
<ol00`3oo`12o`030?oo01[o00<0ool0:_l003;o00<0ool0:?l00`3oo`0ao`030?oo047o00<0ool0
7?l202[o000ao`030?oo02Wo00<0ool0<Ol00`3oo`11o`030?oo01ko00<0ool09ol0033o00<0ool0
:Ol203?o00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000`o`030?oo02So0P0do`030?oo047o00<0ool0
7ol202Oo000`o`030?oo02Oo00<0ool08?l200Co0P03o`@01Ol00`3oo`11o`030?oo023o00<0ool0
9Ol002oo00<0ool09_l202;o00@0ool02?l00`3oo`06o`030?oo047o00<0ool08?l00`3oo`0Uo`00
;ol00`3oo`0To`809?l0103oo`09o`030?oo00Go0P12o`030?oo023o00<0ool09Ol002oo00<0ool0
8ol00`3oo`0No`@00_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Po`030?oo02Go000^o`03
0?oo02?o00<0ool09Ol0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Po`030?oo02Go000^o`03
0?oo02;o00<0ool09ol200[o0P06o`030?oo047o00<0ool08?l00`3oo`0Uo`00;_l00`3oo`0Qo`80
?Ol00`3oo`10o`030?oo023o00<0ool09_l002ko00<0ool08Ol00`3oo`0lo`030?oo043o00<0ool0
8?l00`3oo`0Vo`00;_l00`3oo`0Qo`80?Ol00`3oo`10o`030?oo01oo00<0ool09ol000Oo1@02o`80
1?l201So00<0ool02?l501Go0P0;o`801?l200Co0P0So`030?oo02;o0P04o`801?l2013o00<0ool0
4?l500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool0
5_l200Wo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0>o`030?oo01?o00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
6?l200Oo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`0;o`030?oo01Co00<0ool0
2?l00`3oo`0Io`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co
00<0ool06_l500;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`803Ol00`3oo`0Eo`030?oo00Oo
00<0ool06_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool07?l300;o
00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0:o`806?l00`3oo`07o`030?oo01[o
00<0ool02?l00`3oo`02o`002?l200[o0`0Go`030?oo00Wo0P0Qo`8000?o00002_l302;o00<0ool0
8_l200[o0`09o`030?oo01Oo0P08o`030?oo01[o0P0:o`<00_l002ko00<0ool0;_l2033o00<0ool0
=_l402?o00<0ool0;?l002oo00<0ool0;ol00`3oo`0]o`030?oo03Go00<0ool09Ol00`3oo`0/o`00
;ol00`3oo`0`o`030?oo02co00<0ool0<ol202Oo00<0ool0;Ol002oo00<0ool0<?l00`3oo`0/o`03
0?oo03;o00<0ool09_l00`3oo`0^o`00;ol00`3oo`0`o`030?oo02co00<0ool0<_l00`3oo`0Vo`03
0?oo02ko000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`0700So00<0ool01_l00`3oo`04o`03
0?l000So00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00P04o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01ol000co00<0ool08ol00`3oo`03o`030?oo02Wo00<0ool0:_l00`3o
o`0?o`801ol00`3oo`0>o`030?oo00?o00<0ool08ol00`3o000Zo`030?oo00Oo000co`030?oo02oo
00<0ool0:Ol00`3oo`0>o`030?oo00So0P0Do`030?oo02Co00<0ool0<ol003Co00<0ool0;_l00`3o
o`0Yo`030?oo00go00<0ool02ol2017o00<0ool09?l00`3oo`0do`00=?l00`3oo`0_o`80:Ol00`3o
o`0;o`804?l200ko00<0ool08ol203Oo000eo`030?oo02oo4`0Go`030?oo00Wo0P0Do`030?oo00Wo
0P0Uo`030?oo03Oo000fo`030?oo047o0`0Do`030?oo00So00<0ool05Ol400Go0P0Vo`030?oo03So
000fo`030?oo04Co00<0ool04Ol00`3oo`07o`030?oo01[o1P0Uo`80>ol003Oo00<0ool0A?l2017o
00<0ool01Ol204Ko0P0mo`00>?l00`3oo`15o`803ol00`3oo`02o`<0A_l203oo000io`80Aol300co
1@13o`H0@Ol003_o1@000ol0000:03Oo3P0io`h0Aol003oo0`0:o`<0@_l00`3oo`0do`<0EOl004oo
00<0ool0?ol00`3oo`0_o`D0F?l0053o00<0ool0?_l00`3oo`0/o`<0GOl0057o0P0Zo`801?l200?o
1005o`030?oo02Wo0`1Po`00Dol00`3oo`0Vo`040?oo00So00<0ool01_l00`3oo`0Vo`<0Hol005Co
00<0ool09Ol0103oo`09o`030?oo00Go0P0Uo`80I_l005Go00<0ool09?l0103oo`0:o`030?oo00Co
00<0ool08_l206So001Fo`030?oo02?o00@0ool02?l0103oo`05o`030?oo023o0P1Zo`00Eol00`3o
o`0So`802_l200Ko00<0ool07_l206co001Ho`<03Ol402Go00<0ool07?l206ko001Ko`h00_l5027o
00<0ool06_l2073o001`o`<07_l00`3oo`0Fo`@0L_l007?o00<0ool06olI07Ko001do`030?oo01Oo
102>o`00MOl201?o1003o`030?oo08co001go`030?oo00co1007o`030?oo08co001ho`@01_l400_o
00<0ool0S?l007_o1`0?o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87553 .16034 m
.86637 .15306 L
.8564 .14314 L
.84735 .13147 L
.84084 .12088 L
.83575 .11156 L
.83377 .10839 L
.83296 .10735 L
.83208 .10647 L
.83159 .1061 L
.83112 .10583 L
.83087 .10572 L
.83062 .10563 L
.83034 .10555 L
.83007 .1055 L
.8298 .10547 L
.8295 .10547 L
.82921 .10548 L
.82895 .10552 L
.82863 .10558 L
.82827 .10568 L
.8276 .10593 L
.82635 .10658 L
.82488 .10754 L
.82184 .10987 L
.7833 .13435 L
.773 .13896 L
.76878 .14113 L
.76549 .14327 L
.76394 .14456 L
.7625 .14605 L
.76128 .14768 L
.76019 .14967 L
.75946 .15149 L
.75914 .15251 L
.75885 .15364 L
.75863 .15467 L
.75843 .15584 L
.75814 .15803 L
.75802 .15934 L
.75793 .16061 L
.75786 .16198 L
.7578 .16354 L
.75778 .16438 L
.75776 .16516 L
.75775 .16602 L
.75773 .16696 L
.75772 .16784 L
.75772 .16866 L
.7577 .1703 L
Mistroke
.75769 .17202 L
.75768 .17366 L
.75767 .17515 L
.75764 .17679 L
.75762 .17765 L
.7576 .1786 L
.75754 .18054 L
.75747 .18234 L
.75738 .18404 L
.75714 .1875 L
.75681 .19076 L
.75643 .19367 L
.75591 .19677 L
.75522 .20005 L
.75432 .2034 L
.75333 .20638 L
.75229 .20897 L
.74983 .21369 L
.74835 .21588 L
.74666 .21796 L
.74354 .22095 L
.73983 .2235 L
.73758 .22465 L
.73549 .22552 L
.73072 .22701 L
.72222 .22889 L
.71824 .22995 L
.71632 .23061 L
.71427 .23149 L
.71081 .23352 L
.70777 .23605 L
.70434 .23988 L
.7013 .24416 L
.69431 .25601 L
.68753 .26715 L
.68329 .27289 L
.6789 .27766 L
.6737 .28181 L
.67074 .2835 L
.66917 .28423 L
.6674 .28489 L
.66596 .28533 L
.6644 .2857 L
.66352 .28586 L
.66272 .28598 L
.66112 .28614 L
.65965 .28621 L
.65802 .28618 L
.65647 .28607 L
.65502 .28588 L
Mistroke
.65337 .28559 L
.65179 .28524 L
.64879 .28435 L
.64551 .2831 L
.64189 .28142 L
.6292 .27346 L
.61429 .2618 L
.60158 .25211 L
.59565 .24843 L
.59265 .2469 L
.58957 .24563 L
.58694 .24482 L
.58548 .24449 L
.58413 .24427 L
.58291 .24414 L
.58181 .24408 L
.58056 .24409 L
.57938 .24417 L
.57817 .24432 L
.57749 .24444 L
.57685 .24458 L
.57548 .24494 L
.57424 .24536 L
.57204 .24632 L
.56972 .24762 L
.56496 .25117 L
.56017 .25584 L
.54398 .27481 L
.53688 .28179 L
.53302 .28481 L
.5288 .28749 L
.52405 .28984 L
.51947 .29156 L
.515 .29287 L
.50988 .29405 L
.49815 .2961 L
.48673 .29775 L
.48112 .29848 L
.4782 .29881 L
.47513 .29911 L
.47364 .29924 L
.47224 .29934 L
.4709 .29943 L
.46968 .29949 L
.46853 .29953 L
.46734 .29956 L
.46615 .29956 L
.46487 .29955 L
.46365 .29951 L
.46252 .29945 L
Mistroke
.46151 .29937 L
.46044 .29927 L
.45929 .29913 L
.4581 .29895 L
.45601 .29854 L
.45416 .29806 L
.45247 .29753 L
.44883 .29599 L
.44522 .2939 L
.44148 .29115 L
.43476 .28534 L
.43145 .28254 L
.4276 .27965 L
.42388 .27733 L
.41956 .27523 L
.41697 .27425 L
.41396 .27334 L
.41107 .27267 L
.40964 .27241 L
.40825 .27219 L
.40688 .27201 L
.40535 .27186 L
.40394 .27174 L
.40261 .27166 L
.40116 .27161 L
.39953 .27158 L
.39781 .27158 L
.39616 .27161 L
.39448 .27168 L
.39352 .27173 L
.39262 .27178 L
.38882 .27208 L
.38549 .27242 L
.38192 .27286 L
.37388 .27399 L
.36672 .27503 L
.36337 .27547 L
.36036 .2758 L
.35878 .27594 L
.35737 .27605 L
.35664 .27609 L
.35585 .27613 L
.35504 .27616 L
.35428 .27618 L
.35348 .27619 L
.35274 .27618 L
.35209 .27617 L
.3514 .27614 L
.35066 .2761 L
.3499 .27603 L
Mistroke
.34927 .27597 L
.3486 .27588 L
.34744 .27568 L
.34627 .27542 L
.34522 .27511 L
.34433 .27478 L
.34262 .27395 L
.34124 .27299 L
.33994 .27175 L
.33878 .27021 L
.33795 .26869 L
.3373 .26714 L
.33671 .26528 L
.33619 .26314 L
.33541 .25874 L
.33464 .25371 L
.3342 .25119 L
.3337 .24882 L
.33245 .24441 L
.33155 .24206 L
.33056 .23994 L
.32767 .23542 L
.32592 .23338 L
.32372 .23125 L
.31842 .22738 L
.31261 .2243 L
.30643 .22176 L
.29388 .21771 L
.28062 .21368 L
.27486 .21153 L
.272 .2102 L
.2697 .20892 L
.268 .20774 L
.26647 .20639 L
.26529 .20497 L
.26481 .20416 L
.26446 .20339 L
.26422 .20264 L
.26407 .20192 L
.26401 .20126 L
.26401 .20054 L
.2641 .19974 L
.26418 .19933 L
.26429 .19887 L
.26456 .19806 L
.26488 .1973 L
.26575 .1957 L
.26684 .19416 L
.26949 .19125 L
.2777 .18479 L
Mistroke
.29735 .17333 L
.3035 .16945 L
.30472 .16847 L
.30577 .16746 L
.30647 .16661 L
.30679 .16613 L
.30704 .16568 L
.30726 .16516 L
.3074 .16468 L
.30749 .16416 L
.30752 .16387 L
.30752 .1636 L
.30749 .16314 L
.30741 .16263 L
.30726 .16209 L
.30708 .16158 L
.30657 .16055 L
.30596 .15957 L
.3041 .15724 L
.29202 .14578 L
.28604 .14079 L
.28274 .13839 L
.27959 .13648 L
.27777 .13557 L
.27576 .13476 L
.27388 .13419 L
.27296 .13397 L
.27209 .1338 L
.27126 .13367 L
.27035 .13357 L
.26982 .13353 L
.26932 .13349 L
.26836 .13346 L
.26733 .13346 L
.26635 .1335 L
.26526 .13358 L
.26464 .13364 L
.26407 .13371 L
.26197 .13402 L
.25957 .1345 L
.25437 .13584 L
.24381 .13923 L
.22955 .14387 L
.21558 .14759 L
.20874 .14896 L
.20517 .14953 L
.20143 .15001 L
.19822 .15033 L
.19654 .15047 L
.19476 .15058 L
Mistroke
.19314 .15066 L
.19168 .1507 L
.19024 .15073 L
.18888 .15074 L
.18724 .15073 L
.18578 .1507 L
.18498 .15067 L
.18413 .15064 L
.1826 .15056 L
.17974 .15036 L
.17721 .15013 L
.17181 .14949 L
.16661 .14879 L
.16441 .14852 L
.16336 .14841 L
.16242 .14832 L
.16161 .14826 L
.16073 .14822 L
.15987 .14819 L
.15897 .14818 L
.15815 .1482 L
.15741 .14825 L
.15661 .14832 L
.15579 .14843 L
.15509 .14856 L
.15445 .1487 L
.15304 .14912 L
.15146 .14979 L
.15003 .15061 L
.14747 .15267 L
.1461 .15412 L
.14487 .15566 L
.14219 .15982 L
.13787 .16914 L
.1343 .17972 L
.13113 .19243 L
.1298 .19965 L
.1293 .20307 L
.12892 .20612 L
.12863 .20894 L
.12841 .21192 L
.12832 .21353 L
.12826 .21498 L
.12823 .21635 L
.12821 .21777 L
.12822 .21897 L
.12824 .22024 L
.12828 .22142 L
.12833 .22247 L
.12841 .22369 L
Mistroke
.12851 .22498 L
.12876 .22726 L
.1291 .2296 L
.12956 .23207 L
.13077 .23674 L
.13216 .24072 L
.13368 .24423 L
.15035 .27263 L
.15613 .2809 L
.16236 .28788 L
.16588 .29088 L
.16997 .29355 L
.17246 .29479 L
.17491 .29576 L
.1772 .29646 L
.17854 .29679 L
.17977 .29704 L
.18087 .29722 L
.18208 .29738 L
.18338 .29751 L
.18463 .2976 L
.18581 .29764 L
.18709 .29766 L
.1883 .29764 L
.18942 .2976 L
.19012 .29757 L
.19087 .29752 L
.19222 .29741 L
.19531 .29706 L
.19804 .29666 L
.20108 .29613 L
.21413 .29351 L
.21716 .29295 L
.22037 .29243 L
.22333 .29204 L
.22465 .2919 L
.22607 .29177 L
.22748 .29168 L
.22876 .29162 L
.23026 .29159 L
.23168 .29159 L
.23316 .29164 L
.23453 .29173 L
.23601 .29187 L
.23758 .29208 L
.23893 .29231 L
.24038 .29262 L
.24297 .29331 L
.24543 .29416 L
.24765 .29509 L
Mistroke
.25243 .29772 L
.2574 .30139 L
.26669 .31067 L
.27555 .32106 L
.27995 .32589 L
.28416 .32991 L
.28809 .33297 L
.29268 .3357 L
.29518 .33682 L
.29761 .33768 L
.30027 .33839 L
.3018 .3387 L
.30323 .33893 L
.30458 .3391 L
.30582 .33921 L
.30728 .3393 L
.30867 .33933 L
.30998 .33934 L
.3112 .33931 L
.31256 .33925 L
.31401 .33915 L
.31549 .33902 L
.3171 .33885 L
.32006 .33846 L
.32698 .33727 L
.34123 .33443 L
.34481 .33379 L
.34815 .33327 L
.35137 .33285 L
.35429 .33256 L
.3558 .33245 L
.35739 .33236 L
.35896 .33231 L
.35984 .33229 L
.36066 .33229 L
.36233 .33232 L
.36382 .33239 L
.36464 .33245 L
.36552 .33252 L
.36711 .33269 L
.36868 .33291 L
.37037 .33321 L
.3737 .33401 L
.37666 .33496 L
.37932 .33604 L
.38424 .33863 L
.38843 .34148 L
.39709 .34888 L
.40446 .35547 L
.40869 .35869 L
Mistroke
.41294 .36129 L
.4155 .36253 L
.41841 .36367 L
.42118 .36451 L
.42255 .36483 L
.42385 .3651 L
.42521 .36533 L
.42647 .3655 L
.4279 .36565 L
.42947 .36577 L
.431 .36583 L
.43187 .36585 L
.43269 .36585 L
.43422 .36583 L
.43587 .36576 L
.43734 .36566 L
.43898 .36552 L
.44235 .36511 L
.44554 .36461 L
.44856 .36404 L
.46308 .36033 L
.48798 .35179 L
.5071 .34471 L
.51429 .34243 L
.51742 .34166 L
.51913 .34132 L
.52067 .34107 L
.52228 .34087 L
.52317 .34079 L
.52403 .34072 L
.52483 .34068 L
.52558 .34064 L
.52602 .34063 L
.52642 .34062 L
.52688 .34061 L
.52732 .34061 L
.52812 .34061 L
.52899 .34062 L
.52974 .34064 L
.53058 .34067 L
.53254 .34076 L
.53446 .34089 L
.54267 .34161 L
.55301 .34269 L
.56417 .34378 L
.56731 .34404 L
.57064 .34427 L
.57348 .34444 L
.57499 .3445 L
.57658 .34456 L
Mistroke
.57795 .3446 L
.57872 .34462 L
.57944 .34463 L
.58069 .34464 L
.58202 .34463 L
.58277 .34463 L
.58357 .34461 L
.58432 .34459 L
.58501 .34457 L
.58631 .34451 L
.58767 .34444 L
.5901 .34424 L
.59233 .344 L
.59471 .34366 L
.59721 .34321 L
.60199 .34203 L
.60625 .34061 L
.61552 .33639 L
.63295 .32667 L
.671 .31134 L
.68791 .30554 L
.69609 .30232 L
.70514 .29895 L
.71432 .29631 L
.72013 .29513 L
.72308 .29466 L
.72596 .29428 L
.72866 .29399 L
.73168 .29373 L
.73448 .29355 L
.73712 .29342 L
.73861 .29336 L
.74002 .29332 L
.74158 .29328 L
.74326 .29326 L
.74411 .29325 L
.74504 .29324 L
.74673 .29324 L
.74839 .29324 L
.74993 .29325 L
.75162 .29327 L
.7534 .29329 L
.75659 .29334 L
.7603 .2934 L
.76209 .29342 L
.76372 .29344 L
.76523 .29346 L
.76665 .29347 L
.76813 .29347 L
.76973 .29347 L
Mistroke
.77135 .29345 L
.77216 .29344 L
.77305 .29342 L
.7746 .29338 L
.77604 .29333 L
.77868 .29319 L
.77998 .29311 L
.78138 .293 L
.78596 .29248 L
.78841 .29208 L
.79091 .29158 L
.79511 .29046 L
.79943 .28884 L
.803 .28708 L
.80979 .28243 L
.81959 .27377 L
.82961 .26616 L
.84145 .25637 L
.85578 .2399 L
.87329 .21539 L
.88176 .20255 L
.88905 .19041 L
.89134 .18579 L
.89225 .18352 L
.8926 .18247 L
.89286 .18154 L
.89306 .18071 L
.89322 .17983 L
.89332 .17898 L
.89334 .17853 L
.89335 .17812 L
.89333 .17741 L
.89326 .17678 L
.89313 .17607 L
.89295 .17542 L
.8926 .17451 L
.89217 .17367 L
.89083 .17184 L
.88908 .1701 L
.88671 .16819 L
.87553 .16034 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1Co`<0=_l007go0P04o`801Ol300Co00<0ool0D_l01@3oool003Go
001lo`040?oo00_o00<0ool00ol00`3oo`1@o`801?l00`3oo`0co`00O?l0103oo`08o`D01?l204oo
0P07o`030?oo03;o001fo`@00_l0103oo`08o`040?oo00Go00<0ool0COl00`3oo`07o`030?oo03;o
001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o`802ol00`3oo`0ao`00OOl200_o0P05o`030?oo04Wo
0P0>o`030?oo033o001Ao`D0>ol00`3oo`17o`804?l00`3oo`0`o`00C_l300Co0`0io`030?oo04Go
0P0Co`030?oo02oo001;o`<02_l00`3oo`0fo`030?oo04Co00<0ool04ol00`3oo`0_o`00B?l300ko
00<0ool0=Ol00`3oo`12o`805ol00`3oo`0^o`00=?l800Co200Bo`030?oo03Co00<0ool0@_l00`3o
o`0Go`80;_l003;o0P08o`D06_l00`3oo`0co`030?oo047o0P0Ko`030?oo02_o000bo`030?oo02Oo
00<0ool0<_l00`3oo`11o`030?oo01_o00<0ool0:_l0037o00<0ool0:Ol00`3oo`0ao`030?oo047o
00<0ool07?l00`3oo`0Yo`00<?l00`3oo`0Zo`030?oo037o00<0ool0@Ol00`3oo`0Mo`80:Ol0033o
00<0ool0:_l00`3oo`0ao`030?oo047o00<0ool07ol00`3oo`0Vo`00<?l00`3oo`0Yo`030?oo03;o
00<0ool0@Ol00`3oo`0Po`030?oo02Go000_o`030?oo02So0P0Qo`801?l200?o1005o`030?oo047o
00<0ool08Ol00`3oo`0To`00;ol00`3oo`0Vo`808_l0103oo`08o`030?oo00Ko00<0ool0@Ol00`3o
o`0Qo`030?oo02Co000_o`030?oo02Co0P0To`040?oo00Wo00<0ool01Ol204;o00<0ool08Ol00`3o
o`0To`00;_l00`3oo`0So`808?l400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool08Ol00`3o
o`0To`00;_l00`3oo`0Ro`030?oo02Ko00@0ool02?l0103oo`05o`030?oo047o00<0ool08?l00`3o
o`0Uo`00;_l00`3oo`0Qo`030?oo02So0P0:o`801_l00`3oo`11o`030?oo01oo00<0ool09_l002ko
00<0ool08Ol00`3oo`0lo`030?oo047o00<0ool07ol00`3oo`0Vo`00;_l00`3oo`0Qo`030?oo03co
00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo027o00<0ool0??l00`3oo`10o`030?oo01oo
00<0ool09ol000Oo1@02o`801?l201So00<0ool02?l501Go0P0;o`801?l200Co0P0So`030?oo02;o
0P04o`801?l2013o00<0ool04?l500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103o
o`0Go`030?oo00[o00<0ool05ol400Ko00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103o
o`0>o`030?oo01?o00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3o
o`0Eo`030?oo00[o00<0ool06ol300?o00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3o
o`0;o`805Ol00`3oo`08o`030?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`80
6?l00`3oo`02o`@01?l00`3oo`0Ko`L00_l00`3oo`07o`808ol00`3oo`0Qo`040?oo00Wo0P0=o`03
0?oo01Go00<0ool01ol00`3oo`0Jo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool0
2_l00`3oo`0Qo`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l201So00<0ool0
1_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`809?l200[o0`0Ro`03
0?oo02;o0P0:o`<01ol301Wo0P08o`030?oo01[o0P0:o`<00_l002ko00<0ool0<Ol00`3oo`0/o`03
0?oo03Go0`0Uo`030?oo02co000^o`030?oo037o0P0]o`030?oo03Co0P0Vo`030?oo02go000_o`03
0?oo037o00<0ool0:ol00`3oo`0co`030?oo02Go00<0ool0;_l002oo00<0ool0<_l00`3oo`0Zo`03
0?oo017o0`0No`030?oo02Go00<0ool0;ol0033o00<0ool0<Ol00`3oo`0Zo`030?oo00oo0`001?l0
000Ko`030?oo02Go00<0ool0<?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko0P07o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`05o`802?l200Oo00<0ool0
1_l00`3oo`04o`030?l000So00<0ool01Ol00`3oo`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02;o00<0ool01?l00`3oo`0Yo`80
:ol00`3oo`0=o`030?oo00Wo0P0>o`030?oo00Co00<0ool08_l00`3o000Zo`030?oo00Oo000bo`03
0?oo033o00<0ool0:Ol00`3oo`0<o`030?oo00co00<0ool04Ol00`3oo`0So`80=Ol003;o00<0ool0
<?l00`3oo`0Yo`030?oo00_o00<0ool03_l2017o00<0ool08_l00`3oo`0eo`00<ol00`3oo`0_o`80
:_l00`3oo`0:o`030?oo017o00<0ool03Ol00`3oo`0Ro`030?oo03Ko000co`030?oo033o0P07o``0
5_l00`3oo`09o`030?oo01?o00<0ool02ol00`3oo`0Ro`030?oo03Oo000do`030?oo033o200<o`80
5?l00`3oo`08o`030?oo01Go0P0:o`030?oo02;o00<0ool0>?l003Go00<0ool0AOl00`3oo`0Ao`03
0?oo00Ko0P0Jo`030?oo00Ko00<0ool08_l00`3oo`0io`00=_l00`3oo`15o`804Ol00`3oo`05o`03
0?oo01_o0`02o`<08ol203co000go`030?oo04Ko00<0ool03_l00`3oo`04o`030?oo01oo0`0So`80
?_l003So00<0ool02_l703Go00<0ool03Ol00`3oo`0403[o0P07o`D0@?l003Wo0`04o`D01ol203Co
0P0;o`D0=Ol90003o`0000H0AOl003_o1@0>o`80=?l;00;o00<0ool0<_l305Oo001@o`030?oo03ko
00<0ool0<?l205[o001Ao`030?oo03go00<0ool0;?l405co001Bo`030?oo02So0P04o`800ol400Go
00<0ool0:Ol3063o001Co`030?oo02Ko00@0ool02?l00`3oo`06o`030?oo02Oo0P1So`00E?l00`3o
o`0Uo`040?oo00Wo00<0ool01Ol202Go0`1Uo`00EOl00`3oo`0To`040?oo00[o00<0ool01?l00`3o
o`0Ro`80J?l005Ko00<0ool08ol0103oo`08o`040?oo00Go00<0ool08?l206[o001Go`030?oo02?o
0P0:o`801_l00`3oo`0No`80K?l005So00<0ool02Ol<027o00<0ool07?l206ko001Io`<01_l200co
00<0ool07_l00`3oo`0Jo`80L?l005_o1`0?o`<07Ol0103oool900go0`1bo`00M?l00`3oo`0Jo`@0
2Ol=07Go001eo`030?oo01Oo0`2>o`00M_l00`3oo`0Co`<00_l00`3oo`2<o`00Mol00`3oo`0?o`<0
1Ol00`3oo`2<o`00N?l00`3oo`0:o`@02?l00`3oo`2<o`00NOl400Go0`0<o`030?oo08co001lo`H0
3ol00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`04
0?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o
003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87574 .16224 m
.83816 .12008 L
.83428 .11015 L
.83292 .10628 L
.83237 .10488 L
.83188 .1039 L
.83161 .10347 L
.83131 .10314 L
.83116 .10301 L
.831 .10291 L
.83083 .10284 L
.83065 .10279 L
.83048 .10279 L
.8303 .10281 L
.83012 .10285 L
.82995 .10292 L
.82961 .1031 L
.82921 .10337 L
.82716 .10538 L
.82461 .10823 L
.81747 .11553 L
.80658 .12415 L
.7943 .13116 L
.78277 .13597 L
.77258 .13949 L
.76791 .14125 L
.76424 .14306 L
.76252 .14422 L
.76093 .14564 L
.75975 .14711 L
.75886 .14864 L
.75821 .15024 L
.7579 .15122 L
.75766 .1522 L
.75745 .15329 L
.75729 .15433 L
.75716 .15553 L
.7571 .15616 L
.75705 .15686 L
.75699 .15821 L
.75695 .15951 L
.75694 .16093 L
.75694 .16171 L
.75695 .16256 L
.75696 .1635 L
.75698 .16441 L
.75703 .16616 L
.75715 .16971 L
.75726 .17296 L
.75731 .1748 L
Mistroke
.75735 .17656 L
.75737 .17758 L
.75738 .17868 L
.75739 .17969 L
.75739 .18062 L
.75738 .18248 L
.75737 .18342 L
.75734 .18445 L
.75728 .18639 L
.7572 .18822 L
.7571 .18986 L
.75696 .19163 L
.75677 .19355 L
.75654 .19556 L
.756 .19911 L
.75536 .20227 L
.75464 .20512 L
.75372 .20807 L
.75255 .21109 L
.7499 .2161 L
.74821 .21844 L
.74652 .22036 L
.74319 .22317 L
.7412 .22441 L
.73932 .22534 L
.73705 .22622 L
.73492 .22684 L
.73255 .22734 L
.73004 .22772 L
.72787 .22796 L
.72587 .22814 L
.72355 .22834 L
.72146 .22856 L
.71948 .22885 L
.7185 .22903 L
.71744 .22927 L
.71531 .2299 L
.7134 .23071 L
.71168 .23169 L
.71017 .2328 L
.70694 .23608 L
.70439 .23972 L
.70168 .24459 L
.69605 .25666 L
.69289 .26344 L
.68907 .2708 L
.68527 .27691 L
.68075 .28262 L
.67802 .28532 L
.67481 .28783 L
Mistroke
.67334 .28876 L
.67172 .28963 L
.67017 .29033 L
.66869 .29087 L
.66711 .29132 L
.6662 .29153 L
.66536 .29168 L
.66378 .29188 L
.66207 .29197 L
.6605 .29195 L
.65878 .29181 L
.65713 .29156 L
.65559 .29125 L
.65212 .29022 L
.64889 .28893 L
.64147 .28484 L
.62782 .27421 L
.61204 .25913 L
.59849 .24646 L
.59244 .24179 L
.58709 .23857 L
.58451 .23743 L
.58322 .23697 L
.58182 .23656 L
.58041 .23626 L
.57964 .23614 L
.57894 .23607 L
.5777 .236 L
.57639 .23603 L
.5757 .23609 L
.57498 .23619 L
.57371 .23645 L
.57301 .23664 L
.57236 .23684 L
.57091 .23741 L
.56855 .23867 L
.5661 .24041 L
.56168 .24467 L
.55785 .24945 L
.55018 .26106 L
.5423 .27338 L
.53842 .27865 L
.53467 .28299 L
.5312 .28631 L
.52723 .28935 L
.52312 .29177 L
.51855 .29382 L
.51408 .29535 L
.50971 .29655 L
.48742 .30092 L
Mistroke
.48162 .30189 L
.47546 .3028 L
.47272 .30313 L
.47125 .30328 L
.46989 .3034 L
.46865 .3035 L
.46754 .30356 L
.46631 .30361 L
.46502 .30364 L
.46391 .30364 L
.46274 .30361 L
.46165 .30356 L
.46069 .30349 L
.45957 .30337 L
.4584 .30321 L
.45634 .30281 L
.45447 .3023 L
.45258 .30161 L
.44943 .30002 L
.44771 .29887 L
.44614 .29765 L
.44334 .29502 L
.43726 .28799 L
.4315 .28165 L
.42767 .27835 L
.42356 .27566 L
.41891 .27352 L
.41604 .27258 L
.41461 .27221 L
.41324 .27191 L
.41172 .27162 L
.4103 .27141 L
.40869 .27122 L
.40778 .27113 L
.40692 .27107 L
.40518 .27097 L
.40425 .27094 L
.40325 .27093 L
.40155 .27094 L
.40057 .27096 L
.39964 .271 L
.39799 .27109 L
.39617 .27122 L
.39279 .27156 L
.38632 .27249 L
.3717 .27539 L
.36462 .2769 L
.36111 .27759 L
.35761 .27821 L
.35473 .27862 L
Mistroke
.35318 .27879 L
.3518 .2789 L
.35058 .27897 L
.34952 .27899 L
.34834 .27898 L
.34729 .27892 L
.34674 .27887 L
.34614 .27879 L
.34555 .2787 L
.34501 .27859 L
.34409 .27836 L
.34318 .27805 L
.34225 .27762 L
.34138 .27709 L
.3407 .27657 L
.34005 .27592 L
.3391 .27461 L
.33867 .27377 L
.33835 .27295 L
.33809 .27213 L
.33786 .2712 L
.33769 .27026 L
.33757 .26936 L
.33747 .26837 L
.3374 .26728 L
.33738 .2667 L
.33737 .26604 L
.33736 .26484 L
.33738 .26369 L
.33742 .26241 L
.33746 .26118 L
.33751 .26002 L
.33764 .25715 L
.3377 .25556 L
.33773 .25477 L
.33774 .25405 L
.33777 .25276 L
.33777 .25138 L
.33776 .25066 L
.33774 .24988 L
.33768 .24845 L
.33759 .24714 L
.33747 .24592 L
.33731 .2446 L
.33708 .24318 L
.33678 .24172 L
.33644 .24036 L
.33562 .23787 L
.33465 .23563 L
.33354 .23365 L
Mistroke
.3303 .22951 L
.32829 .22764 L
.32583 .22576 L
.32063 .22276 L
.31477 .22029 L
.30883 .21837 L
.29333 .21462 L
.28586 .213 L
.27952 .21148 L
.27406 .20989 L
.26911 .20795 L
.2669 .20678 L
.26521 .20561 L
.26392 .20443 L
.2634 .20381 L
.26293 .2031 L
.26256 .20236 L
.26241 .20198 L
.26228 .20155 L
.26213 .20079 L
.26209 .20007 L
.26211 .19962 L
.26216 .19919 L
.26232 .19837 L
.26246 .19791 L
.26261 .19748 L
.26305 .1965 L
.264 .19494 L
.26537 .19319 L
.2691 .18962 L
.27914 .18278 L
.28941 .17728 L
.29986 .17212 L
.30361 .17004 L
.3049 .1692 L
.30604 .16833 L
.30692 .16747 L
.30723 .1671 L
.3075 .16669 L
.30771 .1663 L
.30785 .16594 L
.30795 .16556 L
.30801 .16514 L
.308 .1647 L
.30797 .16444 L
.30793 .1642 L
.3078 .16375 L
.30761 .16327 L
.30715 .1624 L
.30645 .1614 L
Mistroke
.29567 .1505 L
.28899 .14375 L
.28292 .13791 L
.28032 .13586 L
.27749 .13414 L
.2758 .1334 L
.27421 .13291 L
.27341 .13273 L
.27253 .13258 L
.27168 .13248 L
.27088 .13242 L
.27002 .13241 L
.26922 .13243 L
.26826 .13249 L
.26734 .13258 L
.26645 .13271 L
.26562 .13285 L
.26365 .13327 L
.2591 .13456 L
.25367 .13647 L
.24179 .14104 L
.22704 .14626 L
.21957 .14844 L
.21161 .15032 L
.20488 .15151 L
.20113 .15198 L
.1977 .1523 L
.19606 .15241 L
.19431 .1525 L
.19267 .15255 L
.19118 .15257 L
.18985 .15256 L
.18842 .15254 L
.18694 .15248 L
.18558 .15241 L
.18402 .15231 L
.18257 .15218 L
.17993 .15188 L
.1772 .15149 L
.17437 .15098 L
.16464 .14864 L
.16069 .14762 L
.15911 .14727 L
.15748 .14698 L
.15706 .14692 L
.15661 .14686 L
.15583 .14679 L
.15538 .14677 L
.15496 .14675 L
.15419 .14677 L
Mistroke
.15379 .14679 L
.15337 .14683 L
.15264 .14694 L
.15224 .14702 L
.15187 .14712 L
.15104 .14738 L
.15041 .14765 L
.14976 .14798 L
.14864 .14871 L
.14746 .1497 L
.14622 .15105 L
.14386 .15445 L
.14172 .15857 L
.13789 .16842 L
.13412 .18132 L
.13227 .18909 L
.13079 .19643 L
.12967 .20323 L
.1289 .20921 L
.12858 .21249 L
.12836 .2155 L
.12827 .2171 L
.12821 .21855 L
.12819 .21938 L
.12817 .22011 L
.12816 .22095 L
.12816 .22173 L
.12817 .22307 L
.12819 .22381 L
.1282 .2245 L
.12826 .22586 L
.1283 .2266 L
.12835 .22727 L
.12844 .22844 L
.12856 .22969 L
.12884 .23186 L
.12925 .23427 L
.12978 .23668 L
.13094 .24068 L
.13362 .24714 L
.13707 .25379 L
.14429 .26723 L
.14887 .27566 L
.15157 .28018 L
.15473 .28483 L
.1581 .28893 L
.16148 .29219 L
.16324 .29357 L
.16523 .29489 L
.16721 .29598 L
Mistroke
.1691 .29682 L
.17119 .29754 L
.17229 .29785 L
.17352 .29813 L
.17468 .29833 L
.17577 .29849 L
.17683 .29859 L
.17798 .29867 L
.17923 .29871 L
.17994 .29871 L
.1806 .29869 L
.18183 .29864 L
.18315 .29854 L
.18386 .29848 L
.18462 .29839 L
.18601 .2982 L
.18924 .29763 L
.19591 .296 L
.20247 .29404 L
.20844 .29218 L
.21507 .29024 L
.21812 .28946 L
.22137 .28874 L
.22442 .28819 L
.22576 .288 L
.22717 .28784 L
.22876 .2877 L
.23022 .28762 L
.23111 .28759 L
.23194 .28758 L
.23349 .28762 L
.23495 .2877 L
.23628 .28783 L
.23772 .28804 L
.23922 .28832 L
.24165 .28893 L
.24418 .2898 L
.24856 .29192 L
.25112 .29356 L
.25346 .29534 L
.25764 .29922 L
.27242 .31876 L
.27633 .32421 L
.28069 .32956 L
.28506 .33387 L
.28725 .33561 L
.28937 .33703 L
.29174 .33834 L
.29398 .33932 L
.29657 .34018 L
Mistroke
.29804 .34055 L
.29946 .34083 L
.30087 .34104 L
.3024 .3412 L
.30373 .34129 L
.30521 .34132 L
.30684 .3413 L
.30777 .34126 L
.30863 .34121 L
.31031 .34107 L
.31193 .34088 L
.31532 .34036 L
.31903 .33963 L
.32593 .33795 L
.34214 .33352 L
.34887 .33193 L
.35249 .33122 L
.3558 .3307 L
.35765 .33046 L
.35931 .33029 L
.36019 .33021 L
.36114 .33015 L
.36213 .33009 L
.36306 .33006 L
.36468 .33004 L
.36637 .33007 L
.36779 .33014 L
.36932 .33027 L
.37083 .33045 L
.37224 .33068 L
.37481 .33123 L
.37746 .33201 L
.38022 .33307 L
.38485 .33553 L
.38878 .33837 L
.40354 .35427 L
.40748 .35854 L
.41195 .36251 L
.41613 .36529 L
.41869 .36658 L
.42122 .36758 L
.42264 .36803 L
.42423 .36846 L
.42568 .36877 L
.42726 .36904 L
.42871 .36922 L
.42955 .36931 L
.43035 .36937 L
.43178 .36944 L
.43335 .36947 L
Mistroke
.43426 .36946 L
.43511 .36944 L
.43606 .3694 L
.43708 .36934 L
.43816 .36926 L
.4392 .36916 L
.44117 .36893 L
.44481 .36837 L
.44887 .36755 L
.45661 .36555 L
.46497 .36287 L
.47969 .35727 L
.50451 .3462 L
.51205 .34267 L
.51818 .34 L
.52091 .33904 L
.52214 .33869 L
.52322 .33844 L
.52433 .33825 L
.52484 .33818 L
.52538 .33811 L
.52597 .33806 L
.52651 .33803 L
.52711 .338 L
.52745 .338 L
.52777 .33799 L
.52845 .338 L
.52883 .33801 L
.5292 .33802 L
.52988 .33805 L
.53063 .3381 L
.53198 .33822 L
.53358 .3384 L
.54178 .33962 L
.55218 .34139 L
.56352 .34324 L
.56866 .34399 L
.57413 .34466 L
.57724 .34498 L
.58002 .34521 L
.58135 .3453 L
.58277 .34537 L
.5841 .34543 L
.5853 .34547 L
.58655 .34549 L
.58767 .34549 L
.58895 .34547 L
.59015 .34544 L
.59124 .34539 L
.59224 .34533 L
Mistroke
.59332 .34525 L
.59445 .34514 L
.59675 .34484 L
.59885 .34448 L
.60109 .344 L
.60344 .34336 L
.60771 .34184 L
.61204 .33984 L
.61969 .33543 L
.63713 .32475 L
.65509 .31641 L
.67374 .3099 L
.6901 .30349 L
.69744 .30012 L
.70171 .2983 L
.70601 .29669 L
.71033 .29536 L
.71462 .2943 L
.71979 .29336 L
.72282 .29295 L
.72581 .29264 L
.72733 .29252 L
.72878 .29242 L
.73011 .29235 L
.73154 .29228 L
.73312 .29223 L
.7346 .29219 L
.7362 .29217 L
.73712 .29216 L
.73798 .29216 L
.73952 .29217 L
.74099 .29219 L
.74232 .29221 L
.74377 .29225 L
.74706 .29236 L
.75011 .2925 L
.75706 .29287 L
.76444 .2933 L
.76776 .29346 L
.76954 .29354 L
.77117 .2936 L
.77267 .29364 L
.77402 .29367 L
.77549 .29369 L
.77704 .2937 L
.77846 .29369 L
.77993 .29366 L
.78117 .29362 L
.78251 .29356 L
.78391 .29348 L
Mistroke
.78538 .29336 L
.78795 .29309 L
.7904 .29272 L
.7926 .29228 L
.79641 .29125 L
.7984 .29052 L
.80038 .28966 L
.80357 .28789 L
.80654 .28573 L
.80935 .28316 L
.82529 .26766 L
.85293 .2409 L
.87291 .21436 L
.88303 .20029 L
.89157 .18778 L
.89438 .18292 L
.89534 .18087 L
.89566 .18004 L
.89595 .17918 L
.89614 .17844 L
.89627 .17778 L
.89634 .17712 L
.89636 .17645 L
.89633 .17611 L
.89629 .17575 L
.89615 .17512 L
.89594 .17454 L
.89567 .17403 L
.89534 .17352 L
.89491 .17298 L
.89393 .17206 L
.89146 .1704 L
.88328 .16626 L
.87574 .16224 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1Do`80=_l0097o00<0ool0Dol303Ko001mo`801?l200Go0`04o`030?oo05;o00@0ool0
=_l007co00@0ool02ol00`3oo`03o`030?oo057o00D0oooo000fo`00O?l0103oo`08o`D01?l2057o
00<0ool00ol00`3oo`0co`00M_l400;o00@0ool02?l0103oo`05o`030?oo04oo00<0ool01Ol00`3o
o`0bo`00O?l0103oo`09o`030?l000Go00<0ool0COl200Wo00<0ool0<Ol007go0P0;o`801Ol00`3o
o`1;o`802ol00`3oo`0ao`00D?l603_o00<0ool0BOl200ko00<0ool0<?l004ko0P06o`80>Ol00`3o
o`17o`804Ol00`3oo`0_o`00Bol300Wo00<0ool0=ol00`3oo`14o`<05?l00`3oo`0^o`00B?l300go
00<0ool0=_l00`3oo`12o`806?l00`3oo`0]o`00<ol600_o100Ao`030?oo03Go00<0ool0@Ol201[o
00<0ool0;?l003;o00<0ool01?l;01Ko00<0ool0=?l00`3oo`11o`030?oo01Wo00<0ool0;?l003;o
00<0ool09_l203Co00<0ool0@Ol00`3oo`0Jo`030?oo02_o000ao`030?oo02Wo00<0ool0<Ol00`3o
o`11o`030?oo01_o00<0ool0:_l0033o00<0ool0:_l00`3oo`0ao`030?oo047o00<0ool07?l202[o
000`o`030?oo02[o00<0ool0<Ol00`3oo`11o`030?oo01ko0P0Xo`00<?l00`3oo`0Xo`<0<ol00`3o
o`11o`030?oo023o0P0Vo`00;ol00`3oo`0Wo`808_l200Co0P03o`@01Ol00`3oo`11o`030?oo02;o
00<0ool08ol002oo00<0ool09Ol202?o00@0ool02?l00`3oo`06o`030?oo047o00<0ool08_l00`3o
o`0So`00;ol00`3oo`0To`030?oo02?o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0Qo`809Ol002oo
00<0ool08_l2023o1002o`040?oo00[o00<0ool01?l00`3oo`11o`030?oo027o00<0ool09?l002oo
00<0ool08Ol00`3oo`0Vo`040?oo00So00@0ool01Ol00`3oo`11o`030?oo023o00<0ool09Ol002ko
00<0ool08Ol00`3oo`0Xo`802_l200Ko00<0ool0@Ol00`3oo`0Po`030?oo02Go000^o`030?oo023o
00<0ool0?Ol00`3oo`11o`030?oo01oo00<0ool09_l002ko00<0ool08?l00`3oo`0mo`030?oo043o
00<0ool07ol00`3oo`0Wo`00;_l00`3oo`0Po`80?_l00`3oo`10o`030?oo01oo00<0ool09ol000Oo
1@02o`801?l201So00<0ool02?l501Co100:o`801?l200Co0P0So`030?oo02;o0P04o`801?l2013o
00<0ool04?l500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o
00<0ool06?l400Go00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0?o`030?oo01;o
00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o
00<0ool07?l300;o00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`0<o`030?oo01?o
00<0ool02?l00`3oo`0Io`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool0
0_l400Co00<0ool06ol80003o`3o00So0P0So`030?oo027o00@0ool02Ol200ko0P0Eo`030?oo00Oo
00<0ool06_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool08Ol0103o
0009o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00_o0P0Go`030?oo00Ko00<0ool06ol00`3o
o`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol202Co1008o`<08_l00`3oo`0Ro`802_l300Go
1`0Go`801ol00`3oo`0Ko`802_l300;o000^o`030?oo03;o00<0ool0:ol00`3oo`0do`<09_l00`3o
o`0/o`00;_l00`3oo`0co`80:ol00`3oo`0@o`<08Ol00`3oo`0Uo`030?oo02go000^o`030?oo03Co
00<0ool0:Ol00`3oo`0>o`800ol201ko00<0ool09Ol00`3oo`0^o`00;_l00`3oo`0do`030?oo02Wo
00<0ool03Ol00`3oo`05o`806ol00`3oo`0Uo`030?oo02oo000_o`030?oo03?o00<0ool0:Ol00`3o
o`0=o`030?oo00Oo00<0ool06?l00`3oo`0To`030?oo033o000<ool03009o`003?l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko0P07o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`03o`04
0?oo00So00<0o`001_l00`3oo`06o`030?oo00Go0P08o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?l000Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Ro`03
0?oo00Co00<0ool0:Ol202_o00<0ool02ol00`3oo`0<o`030?oo00co00<0ool01Ol00`3oo`0Qo`80
:ol00`3oo`07o`00<Ol00`3oo`0ao`030?oo02Wo00<0ool02_l00`3oo`0>o`030?oo01;o00<0ool0
8_l00`3oo`0do`00<_l00`3oo`0`o`030?oo02Wo00<0ool02Ol00`3oo`0@o`804_l00`3oo`0Qo`03
0?oo03Go000bo`030?oo033o00<0ool0:Ol00`3oo`09o`030?oo01;o00<0ool03_l00`3oo`0Qo`03
0?oo03Ko000co`030?oo02oo0P0:o`/05Ol00`3oo`08o`030?oo01Co00<0ool03Ol00`3oo`0Po`03
0?oo03Oo000co`030?oo033o0P04o`D02ol00`3oo`0Bo`030?oo00Oo00<0ool05_l00`3oo`0;o`03
0?oo023o00<0ool0>?l003Co00<0ool0<?l5017o0P0Bo`030?oo00Oo00<0ool05ol00`3oo`09o`03
0?oo023o00<0ool0>Ol003Go00<0ool0Aol00`3oo`0?o`030?oo00Ko00<0ool06Ol200So0P0Qo`03
0?oo03[o000fo`030?oo00go1`0co`030?oo00ko00<0ool01?l201ko00<0ool01?l00`3oo`0Oo`80
?Ol003Ko00<0ool02Ol400Ko0`0ao`030?oo00ko00<0ool00ol00`3oo`0Oo`H03Ol:00Oo100oo`00
=ol200Ko0`0<o`80<Ol00`3oo`0=o`H0<?l500[o1`13o`00>Ol6017o00<0ool0;ol200_o0`0bo`<0
FOl0057o00<0ool0<?l;00;o00<0ool0;_l205co001Bo`030?oo03co00<0ool0;?l205ko001Bo`03
0?oo02So0P04o`800ol400Go00<0ool0:_l2063o001Co`030?oo02Ko00@0ool02?l00`3oo`06o`03
0?oo02So0P1Ro`00E?l00`3oo`0Uo`040?oo00Wo00<0ool01Ol202Ko0`1To`00EOl00`3oo`0To`04
0?oo00[o00<0ool01?l00`3oo`0So`80Iol005Go00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
8Ol206Wo001Fo`030?oo017o1P0=o`802_l200Ko00<0ool07ol206_o001Go`030?oo00_o1@05o`<0
7ol00`3oo`0Mo`80KOl005So0P07o`@03Ol00`3oo`0Lo`030?oo00?o0`0Eo`80Kol005[o1`0Bo`03
0?oo01_o00@0oooo0`000ol0000500go0P1ao`00M?l00`3oo`0Jo`040?l000_o3@1co`00MOl00`3o
o`0Io`80SOl007Ko00<0ool05_l308ko001go`030?oo01?o0P02o`030?oo08co001ho`030?oo00ko
1004o`030?oo08co001io`030?oo00[o0`08o`030?oo08co001jo`<01Ol400_o00<0ool0S?l007co
1P0?o`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87595 .16413 m
.83548 .11929 L
.8336 .11314 L
.83251 .10819 L
.83213 .10589 L
.83187 .10381 L
.8316 .10125 L
.83158 .101 L
.83155 .10076 L
.83152 .10058 L
.83149 .10043 L
.83145 .1003 L
.83142 .10021 L
.83137 .10014 L
.83132 .1001 L
.83126 .10011 L
.83121 .10013 L
.83114 .1002 L
.83106 .10029 L
.8307 .10087 L
.82737 .10659 L
.82459 .1105 L
.82067 .11509 L
.81131 .12345 L
.80503 .12768 L
.79872 .13111 L
.784 .13679 L
.77128 .14008 L
.76871 .14074 L
.76623 .14145 L
.76417 .14217 L
.76246 .14291 L
.76112 .14366 L
.75987 .14457 L
.75928 .14511 L
.75871 .14573 L
.75784 .14697 L
.75747 .14767 L
.75712 .14848 L
.75686 .14923 L
.75662 .15009 L
.75639 .15117 L
.75623 .15226 L
.75612 .15338 L
.75605 .15448 L
.75601 .15573 L
.756 .15697 L
.75602 .15814 L
.75606 .15947 L
.75612 .16097 L
Mistroke
.75622 .16263 L
.75643 .16578 L
.75686 .17193 L
.75707 .17537 L
.75717 .17729 L
.75724 .17911 L
.75728 .18025 L
.75731 .18129 L
.75734 .18245 L
.75736 .18371 L
.75737 .18476 L
.75737 .1859 L
.75734 .18798 L
.75731 .18898 L
.75728 .19009 L
.75723 .19125 L
.75717 .19233 L
.75703 .19424 L
.75685 .19629 L
.75637 .2001 L
.75606 .20197 L
.75568 .20396 L
.75486 .20738 L
.75394 .21033 L
.75273 .21344 L
.7512 .2165 L
.74968 .21889 L
.74792 .22111 L
.74632 .22272 L
.7445 .22419 L
.7424 .22551 L
.74035 .22648 L
.73793 .22729 L
.73654 .22763 L
.73524 .22787 L
.73411 .22802 L
.73285 .22815 L
.73221 .22819 L
.73153 .22823 L
.73089 .22826 L
.7303 .22827 L
.72919 .22828 L
.72816 .22827 L
.72704 .22825 L
.72586 .22821 L
.72481 .22818 L
.7237 .22814 L
.72308 .22813 L
.72251 .22811 L
.72141 .22811 L
Mistroke
.72088 .22811 L
.72031 .22812 L
.71929 .22817 L
.71875 .22821 L
.71817 .22826 L
.71715 .2284 L
.71667 .22848 L
.71616 .22859 L
.71525 .22882 L
.7136 .2294 L
.7127 .22984 L
.71173 .23041 L
.70988 .23187 L
.70827 .23362 L
.70683 .23561 L
.70414 .24058 L
.70175 .24622 L
.69603 .26175 L
.69291 .26953 L
.68911 .27757 L
.68537 .28387 L
.68068 .28979 L
.67779 .29249 L
.67498 .29452 L
.67171 .29621 L
.66999 .29684 L
.66901 .29712 L
.66807 .29735 L
.66632 .29764 L
.66469 .29777 L
.66291 .29778 L
.66189 .29771 L
.66094 .29761 L
.65933 .29736 L
.65754 .29695 L
.65564 .29639 L
.65384 .29574 L
.6499 .29392 L
.64618 .29177 L
.63757 .28537 L
.62176 .26984 L
.60802 .25433 L
.59436 .23967 L
.58723 .23349 L
.58428 .23146 L
.58117 .22974 L
.57979 .22914 L
.57834 .22863 L
.57701 .22827 L
.57581 .22805 L
Mistroke
.57449 .22792 L
.57375 .22791 L
.57308 .22793 L
.57186 .22805 L
.5706 .22831 L
.56949 .22864 L
.56849 .22904 L
.5663 .23022 L
.56435 .23166 L
.56255 .23332 L
.55873 .23789 L
.55483 .244 L
.54707 .25916 L
.54006 .27312 L
.53352 .28344 L
.52956 .28802 L
.52567 .29145 L
.52163 .29415 L
.5176 .29618 L
.51288 .298 L
.5075 .29962 L
.48434 .30478 L
.47857 .30591 L
.47265 .3069 L
.46969 .3073 L
.46816 .30746 L
.46737 .30754 L
.46655 .3076 L
.46585 .30765 L
.4651 .30769 L
.46381 .30773 L
.46319 .30774 L
.46253 .30773 L
.46137 .3077 L
.46077 .30767 L
.46014 .30762 L
.45903 .30751 L
.45788 .30734 L
.45665 .3071 L
.45562 .30684 L
.45455 .30652 L
.45272 .30578 L
.45093 .30483 L
.44935 .30374 L
.44662 .30123 L
.44117 .29371 L
.43589 .28542 L
.43317 .28188 L
.42987 .27843 L
.42793 .27681 L
Mistroke
.42561 .27519 L
.42334 .2739 L
.42103 .27285 L
.41813 .27183 L
.41529 .27112 L
.41363 .27081 L
.41204 .27058 L
.41025 .2704 L
.40932 .27033 L
.40829 .27028 L
.40653 .27023 L
.40489 .27024 L
.40304 .2703 L
.40107 .27042 L
.39931 .27057 L
.39765 .27075 L
.39382 .27128 L
.38509 .27297 L
.3771 .27494 L
.36028 .27949 L
.35382 .281 L
.35092 .28151 L
.3494 .28172 L
.34805 .28184 L
.34689 .28191 L
.34585 .28191 L
.34497 .28188 L
.34408 .28179 L
.34317 .28163 L
.34229 .2814 L
.34161 .28114 L
.34092 .28079 L
.34026 .28034 L
.33965 .27978 L
.3392 .27924 L
.33879 .27857 L
.33843 .27776 L
.33827 .27727 L
.33815 .27681 L
.33805 .27635 L
.33796 .27584 L
.33786 .27484 L
.33782 .27386 L
.33782 .27328 L
.33784 .27272 L
.33791 .27154 L
.33798 .27085 L
.33805 .2702 L
.3384 .26767 L
.33885 .2651 L
Mistroke
.33998 .25932 L
.34101 .25356 L
.34145 .25031 L
.34164 .24849 L
.34171 .24762 L
.34176 .24682 L
.34182 .24541 L
.34184 .24389 L
.3418 .24242 L
.3417 .24087 L
.34154 .23941 L
.34133 .23809 L
.34104 .23667 L
.34064 .2352 L
.33977 .23282 L
.33851 .23034 L
.33711 .22827 L
.33531 .2262 L
.33127 .22286 L
.32856 .22121 L
.32535 .21963 L
.31882 .21722 L
.31443 .21599 L
.31008 .21498 L
.30157 .21338 L
.28417 .21064 L
.27636 .20916 L
.27245 .20822 L
.2692 .20723 L
.26655 .20621 L
.26444 .20515 L
.26289 .20411 L
.26159 .2029 L
.26104 .20217 L
.26065 .20148 L
.26049 .20109 L
.26035 .20066 L
.26025 .20025 L
.26019 .19987 L
.26016 .19916 L
.26024 .19837 L
.26032 .19797 L
.26044 .19753 L
.26074 .19672 L
.26114 .19591 L
.26159 .19514 L
.26292 .19337 L
.2663 .19009 L
.27514 .18404 L
.29815 .17303 L
Mistroke
.30245 .17112 L
.30554 .16953 L
.30654 .16889 L
.30739 .16822 L
.30777 .16785 L
.30805 .16751 L
.30826 .16716 L
.30841 .16684 L
.3085 .16652 L
.30854 .16617 L
.30852 .1658 L
.30844 .16545 L
.30833 .16511 L
.30818 .16479 L
.30773 .16405 L
.30615 .1622 L
.30406 .16016 L
.29237 .14842 L
.28733 .1424 L
.28242 .13663 L
.28109 .13531 L
.27969 .13409 L
.27835 .13314 L
.27706 .13243 L
.2758 .13192 L
.27506 .13171 L
.27437 .13156 L
.27359 .13144 L
.27314 .13139 L
.27271 .13137 L
.27191 .13136 L
.27102 .1314 L
.27014 .13149 L
.2693 .13161 L
.2683 .1318 L
.26731 .13203 L
.26328 .13327 L
.25191 .13797 L
.2379 .14395 L
.22929 .14719 L
.22083 .1499 L
.21266 .15199 L
.20838 .15286 L
.20391 .15358 L
.2002 .15402 L
.19813 .15421 L
.19622 .15434 L
.19449 .15442 L
.19293 .15446 L
.19116 .15447 L
Mistroke
.18951 .15444 L
.188 .15439 L
.18636 .1543 L
.18468 .15416 L
.18314 .15401 L
.18048 .15366 L
.17772 .15318 L
.17195 .15179 L
.16727 .15027 L
.15958 .14719 L
.15789 .14651 L
.1562 .14587 L
.15481 .14544 L
.15409 .14526 L
.15344 .14513 L
.15308 .14507 L
.1527 .14503 L
.15229 .14499 L
.15192 .14498 L
.15159 .14499 L
.15124 .14501 L
.15062 .1451 L
.15002 .14524 L
.1497 .14535 L
.14941 .14546 L
.14834 .14604 L
.1478 .14644 L
.14731 .14687 L
.14622 .14804 L
.14433 .15093 L
.14251 .15469 L
.14058 .15967 L
.13713 .17077 L
.13332 .18616 L
.13046 .2009 L
.12939 .20796 L
.12858 .21499 L
.12843 .21666 L
.1283 .21841 L
.12821 .21989 L
.12813 .22147 L
.1281 .2224 L
.12807 .22325 L
.12806 .22405 L
.12805 .22489 L
.12804 .22571 L
.12805 .22646 L
.12807 .22783 L
.12812 .22902 L
.12818 .23027 L
Mistroke
.12827 .23155 L
.12838 .2327 L
.12862 .23467 L
.12893 .23667 L
.12939 .23886 L
.12989 .24082 L
.13099 .24422 L
.14342 .27105 L
.14776 .28012 L
.15271 .28831 L
.15563 .29193 L
.15917 .29522 L
.16127 .29669 L
.16362 .29795 L
.16576 .2988 L
.16702 .29919 L
.16822 .29947 L
.16941 .29969 L
.17002 .29978 L
.1707 .29986 L
.17192 .29995 L
.17309 .29999 L
.17381 .29998 L
.1745 .29996 L
.17581 .29989 L
.17656 .29982 L
.17726 .29974 L
.17887 .29951 L
.1817 .29896 L
.18435 .29829 L
.19053 .2963 L
.20211 .29172 L
.21475 .28686 L
.22104 .28498 L
.22274 .28458 L
.22455 .28421 L
.2261 .28395 L
.22777 .28372 L
.22938 .28357 L
.23086 .28348 L
.23225 .28346 L
.23371 .28349 L
.235 .28358 L
.2362 .28371 L
.2375 .28391 L
.23887 .28418 L
.24171 .28499 L
.24328 .28558 L
.2447 .28622 L
.24773 .28792 L
Mistroke
.25036 .2898 L
.25485 .29398 L
.25928 .29939 L
.26719 .3119 L
.27518 .32545 L
.27862 .33041 L
.28258 .33502 L
.28499 .33726 L
.28732 .33902 L
.28977 .34049 L
.29222 .34161 L
.29461 .34241 L
.29602 .34276 L
.29738 .34301 L
.29884 .34321 L
.30021 .34333 L
.30175 .34339 L
.3034 .34337 L
.30477 .34331 L
.30625 .34318 L
.30768 .34302 L
.309 .34283 L
.31231 .34221 L
.31554 .34146 L
.3304 .33684 L
.33889 .33395 L
.34684 .33143 L
.35399 .32954 L
.35791 .32873 L
.3614 .32819 L
.36333 .32797 L
.36509 .32783 L
.3661 .32777 L
.36699 .32774 L
.36802 .32773 L
.36897 .32773 L
.37068 .3278 L
.37222 .32792 L
.37388 .32812 L
.37558 .32842 L
.37822 .32906 L
.38092 .32998 L
.38334 .33107 L
.38543 .33223 L
.39014 .33572 L
.39411 .33973 L
.40721 .35806 L
.41092 .36265 L
.41301 .36482 L
.41536 .36687 L
Mistroke
.41755 .36846 L
.42012 .36994 L
.42262 .37106 L
.42508 .37188 L
.4265 .37224 L
.42808 .37256 L
.42962 .37279 L
.43108 .37295 L
.43244 .37304 L
.43396 .37309 L
.43558 .37308 L
.43714 .37302 L
.43899 .37288 L
.4408 .37269 L
.44427 .37218 L
.44814 .37141 L
.45249 .37033 L
.47037 .36417 L
.48656 .35698 L
.51032 .3445 L
.51817 .33989 L
.52355 .33684 L
.52458 .33636 L
.5256 .33595 L
.52617 .33577 L
.52668 .33563 L
.5272 .33552 L
.52769 .33544 L
.52814 .33539 L
.52862 .33535 L
.52905 .33534 L
.52954 .33535 L
.52982 .33535 L
.53008 .33537 L
.53038 .33539 L
.53068 .33541 L
.53186 .33555 L
.53307 .33575 L
.53449 .33601 L
.53741 .33664 L
.54593 .33866 L
.56773 .34358 L
.57421 .34479 L
.5777 .34535 L
.58091 .3458 L
.58381 .34613 L
.58639 .34635 L
.58783 .34644 L
.58917 .34651 L
.5899 .34653 L
Mistroke
.59057 .34654 L
.5913 .34655 L
.59206 .34655 L
.59326 .34653 L
.59451 .34648 L
.59555 .34642 L
.59668 .34632 L
.59787 .34619 L
.59913 .34602 L
.60132 .34562 L
.60347 .3451 L
.60544 .3445 L
.60899 .34311 L
.61308 .34102 L
.61681 .33873 L
.62548 .33269 L
.66328 .31302 L
.69285 .30068 L
.69988 .29711 L
.70378 .29546 L
.70777 .29408 L
.71286 .29275 L
.71579 .29218 L
.71909 .29168 L
.72061 .2915 L
.72229 .29133 L
.7238 .29121 L
.72544 .2911 L
.72694 .29102 L
.72862 .29096 L
.73017 .29092 L
.73165 .29091 L
.73337 .29091 L
.73424 .29092 L
.73522 .29094 L
.73696 .29098 L
.73861 .29104 L
.74237 .29122 L
.74649 .2915 L
.76105 .2928 L
.76781 .29341 L
.77138 .2937 L
.7746 .29391 L
.77728 .29404 L
.77874 .29409 L
.78004 .29413 L
.78145 .29414 L
.78294 .29414 L
.78431 .29411 L
.78554 .29407 L
Mistroke
.78694 .29399 L
.78824 .2939 L
.78937 .29379 L
.79056 .29364 L
.79311 .29324 L
.79531 .29275 L
.79708 .29224 L
.79891 .29159 L
.80219 .29 L
.80383 .28895 L
.80519 .28792 L
.80781 .28537 L
.81134 .28051 L
.82416 .26707 L
.85543 .23533 L
.88002 .20373 L
.89007 .19072 L
.89658 .18172 L
.89849 .1784 L
.8988 .1777 L
.89905 .17701 L
.89917 .17663 L
.89925 .1763 L
.89935 .17569 L
.89937 .17539 L
.89936 .17509 L
.89929 .17457 L
.89921 .17428 L
.89911 .17401 L
.89899 .17378 L
.89884 .17354 L
.89849 .17311 L
.89803 .17269 L
.89696 .172 L
.8957 .1714 L
.89407 .17078 L
.89047 .1696 L
.87964 .16577 L
.87595 .16413 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Do`80
=_l0097o00<0ool0E?l203Ko002Ao`030?oo05Co0P0fo`00OOl200Co0P05o`<01?l00`3oo`1Co`03
0?l003Ko001lo`040?oo00_o00<0ool00ol00`3oo`1Bo`040?oo03Ko001lo`040?oo00So1@04o`80
DOl200Co00<0ool0<ol007Ko1002o`040?oo00So00@0ool01Ol00`3oo`1?o`030?oo00Go00<0ool0
<_l007co00@0ool02Ol00`3o0005o`030?oo04ko00<0ool01ol00`3oo`0ao`00Dol202So0P0;o`80
1Ol00`3oo`1<o`802ol00`3oo`0`o`00DOl30003o`0003[o00<0ool0B_l200ko00<0ool0;ol004oo
0P06o`030?oo03Oo00<0ool0Aol3017o00<0ool0;_l004go0P09o`030?oo03Ko00<0ool0@ol401Co
00<0ool0;_l003?o100Co`<02ol00`3oo`0fo`030?oo047o0P0Io`030?oo02go000bo`800_l300go
100?o`030?oo03Go00<0ool0@Ol00`3oo`0Io`030?oo02co000bo`030?oo00Co1004o`D05?l00`3o
o`0do`030?oo047o00<0ool06_l00`3oo`0[o`00<Ol00`3oo`09o`@06_l00`3oo`0co`030?oo047o
00<0ool06ol00`3oo`0Zo`00<Ol00`3oo`0Xo`030?oo03;o00<0ool0@Ol00`3oo`0Lo`030?oo02Wo
000ao`030?oo02Wo0P0bo`030?oo047o00<0ool07Ol00`3oo`0Xo`00<?l00`3oo`0Zo`80<_l00`3o
o`11o`030?oo01go0P0Yo`00<?l00`3oo`0Yo`80<ol00`3oo`11o`030?oo01oo100Uo`00<?l00`3o
o`0Wo`808Ol200Co0P03o`@01Ol00`3oo`11o`030?oo02;o0P0To`00;ol00`3oo`0Uo`<08_l0103o
o`08o`030?oo00Ko00<0ool0@Ol00`3oo`0Ro`809?l002oo00<0ool08ol202Go00@0ool02Ol00`3o
o`05o`80@_l00`3oo`0Ro`030?oo02?o000_o`030?oo027o0P0Qo`@00_l0103oo`0:o`030?oo00Co
00<0ool0@Ol00`3oo`0Qo`030?oo02Co000_o`030?oo023o00<0ool09ol0103oo`08o`040?oo00Go
00<0ool0@Ol00`3oo`0Po`030?oo02Go000^o`030?oo023o00<0ool0:Ol200[o0P06o`030?oo047o
00<0ool08?l00`3oo`0Uo`00;_l00`3oo`0Po`030?oo03go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko
000^o`030?oo023o00<0ool0?Ol00`3oo`11o`030?oo01oo00<0ool09_l002ko00<0ool08?l203ko
00<0ool0@Ol00`3oo`0No`030?oo02Oo0007o`D00_l200Co0P0Ho`030?oo00So1@0Eo`@02Ol200Co
0P04o`808ol00`3oo`0Ro`801?l200Co0P0@o`030?oo013o1@09o`030?oo01Ko1@02o`801?l200?o
0009o`030?oo00Oo00@0ool05ol00`3oo`0:o`030?oo01Wo1`001Ol0ool000So00@0ool08_l00`3o
o`0Qo`040?oo00So00@0ool03ol00`3oo`0Bo`030?oo00So00<0ool06Ol00`3oo`07o`040?oo00;o
0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo023o1@0;o`030?oo023o00<0ool08Ol0103o
o`0;o`030?oo00co00<0ool04ol00`3oo`07o`030?oo01[o00<0ool02_l1007o0Ol00003o`000080
1?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3oo`0Ko`@00_l00`3oo`0200So0P0So`030?oo027o
00@0ool02Ol200ko0P0Eo`030?oo00Oo00<0ool06_l00`3oo`08o`800ol000Wo00<0ool02?l00`3o
o`0Go`030?oo00[o00<0ool08Ol01P3oo`3o00Oo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool0
3?l201Ko00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`80
9?l200?o00<0ool01?l302;o00<0ool03ol400oo0P0:o`<01Ol701Oo0P07o`030?oo01_o0P0:o`<0
0_l002ko00<0ool0=Ol00`3oo`0Xo`030?oo00ko00@0oooo0P0Po`809_l00`3oo`0]o`00;_l00`3o
o`0eo`030?oo02So00<0ool03Ol00`3oo`04o`807_l00`3oo`0To`030?oo02ko000^o`030?oo03Go
00<0ool0:?l00`3oo`0<o`030?oo00Oo00<0ool06_l00`3oo`0To`030?oo02oo000^o`030?oo03Go
00<0ool0:?l00`3oo`0;o`030?oo00Wo00<0ool06?l00`3oo`0To`030?oo033o000_o`030?oo03Co
00<0ool0:?l00`3oo`0:o`030?oo00[o00<0ool06?l00`3oo`0So`030?oo037o000<ool03009o`00
3?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go00D0oooo0004o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Go0P08o`030?oo00Go
00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`07o`003?l00`3oo`0Qo`030?oo00Go00<0ool0:Ol00`3o000Zo`030?oo00Wo00<0ool03_l00`3o
o`0<o`030?oo00Go00<0ool08Ol202_o00<0ool01ol0037o00<0ool0<_l00`3oo`0Xo`030?oo00So
00<0ool04?l00`3oo`0Co`030?oo027o00<0ool0=?l0037o00<0ool0<_l00`3oo`0Xo`030?oo00So
00<0ool04Ol00`3oo`0Ao`030?oo027o00<0ool0=Ol003;o00<0ool0<?l200oo100Go`030?oo00Oo
00<0ool04ol00`3oo`0@o`030?oo023o00<0ool0=_l003;o00<0ool0<?l00`3oo`0:o`@01?l301Co
00<0ool01ol00`3oo`0Do`030?oo00ko00<0ool07ol203Wo000co`030?oo02oo0P07o`@02ol00`3o
o`0Ao`030?oo00Ko00<0ool05_l00`3oo`0=o`030?oo01ko00<0ool0>Ol003?o00<0ool0<?l8013o
0P0Ao`030?oo00Ko00<0ool05ol00`3oo`0;o`030?oo01ko00<0ool0>_l003Co00<0ool03ol703;o
00<0ool03ol00`3oo`05o`030?oo01Wo00<0ool02_l00`3oo`0Mo`030?oo03_o000do`030?oo00co
0`07o`030?oo033o00<0ool03_l00`3oo`04o`030?oo01_o00<0ool02?l00`3oo`0Mo`80?Ol003Go
00<0ool02?l300_o00<0ool0;ol00`3oo`0>o`030?oo00?o00<0ool07Ol00`3oo`06o`801ol>00Oo
100no`00=_l00`3oo`05o`803ol00`3oo`0_o`030?oo00go00@0oooo0P0Qo`<00ol200Oo0P0>o`P0
@Ol003Oo1`0Bo`030?oo02ko00<0ool03Ol402Ko0`05o`@0FOl0057o00<0ool0;_l200[o100_o`80
GOl005;o00<0ool0;_l;00?o00<0ool0:ol205oo001Co`030?oo02Oo0P04o`800ol400Go00<0ool0
:Ol2067o001Do`030?oo02Go00@0ool02?l00`3oo`06o`030?oo02Oo0P1So`00E?l00`3oo`0Uo`04
0?oo00Wo00<0ool01Ol202Ko0P1Uo`00EOl00`3oo`0To`040?oo00[o00<0ool01?l00`3oo`0So`80
Iol005Go00<0ool09?l0103oo`08o`040?oo00Go00<0ool08Ol206Wo001Fo`030?oo013o2@0;o`80
2_l200Ko00<0ool07ol206_o001Go`030?oo00_o1008o`807_l00`3oo`0No`030?oo06_o001Go`03
0?oo00Oo100>o`030?oo01_o00<0ool00_l601Go00<0ool0K?l005So0P05o`804ol00`3oo`0Jo`03
0?oo00801_l400ko0`1_o`00F_l601Go00<0ool06Ol300co1`04o`<0L_l007Ko00<0ool05ol201Go
1@1do`00M_l00`3oo`0Eo`8000?o0?l0SOl007Oo00<0ool04_l200?o00<0ool0S?l007So00<0ool0
3ol200Go00<0ool0S?l007Wo00<0ool03?l200Oo00<0ool0S?l007Wo0P0;o`802Ol00`3oo`2<o`00
Nol300Co100;o`030?oo08co001mo`D000?o00001Ol300Co00<0ool0S?l007co00@0ool02ol00`3o
o`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103o
o`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00
oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87616 .16603 m
.86938 .163 L
.86301 .15932 L
.85623 .15432 L
.84952 .14798 L
.84337 .14043 L
.83862 .13278 L
.83532 .12581 L
.8328 .11849 L
.83189 .11484 L
.83129 .11162 L
.83105 .10989 L
.83088 .10839 L
.83081 .10759 L
.83076 .10675 L
.83072 .10598 L
.8307 .1053 L
.83069 .10461 L
.83069 .10389 L
.83071 .10325 L
.83073 .10269 L
.8308 .10169 L
.83085 .10123 L
.8309 .10075 L
.83104 .09985 L
.83121 .09904 L
.83138 .09842 L
.83154 .09799 L
.83168 .09769 L
.83176 .09758 L
.83183 .09749 L
.83195 .09741 L
.83205 .09743 L
.83209 .09748 L
.83214 .09756 L
.83219 .09779 L
.8322 .09796 L
.83221 .09814 L
.8322 .09837 L
.83217 .09865 L
.8321 .09919 L
.83204 .09954 L
.83196 .09988 L
.83155 .10138 L
.83098 .10301 L
.83014 .10495 L
.82763 .1095 L
.82437 .11408 L
.81947 .11947 L
.81379 .12437 L
Mistroke
.80824 .12819 L
.80186 .13171 L
.79453 .13481 L
.7877 .13699 L
.78001 .13876 L
.77353 .13987 L
.76746 .1408 L
.76452 .14136 L
.76325 .14166 L
.76214 .14197 L
.76124 .14227 L
.76033 .14264 L
.75893 .14338 L
.75824 .14387 L
.75758 .14445 L
.75661 .14566 L
.75622 .14636 L
.75586 .14718 L
.75558 .14803 L
.75538 .14886 L
.75521 .14986 L
.75513 .15046 L
.75508 .15104 L
.75504 .15168 L
.75501 .15228 L
.755 .15299 L
.755 .15369 L
.75502 .15491 L
.75508 .15628 L
.75517 .15764 L
.75527 .15893 L
.75595 .16579 L
.7567 .17332 L
.75707 .17788 L
.75721 .1801 L
.75733 .18221 L
.7574 .184 L
.75745 .18593 L
.75748 .18808 L
.75748 .19005 L
.75744 .19198 L
.75736 .19408 L
.75724 .19626 L
.75708 .19826 L
.75688 .20016 L
.75665 .20193 L
.75613 .20515 L
.75541 .20846 L
.75445 .21182 L
.75333 .21485 L
Mistroke
.75193 .21779 L
.75044 .22027 L
.74893 .22225 L
.74701 .22423 L
.74475 .22597 L
.74367 .22663 L
.74248 .22723 L
.74028 .22807 L
.73897 .22843 L
.73774 .22868 L
.73709 .22878 L
.73638 .22887 L
.73563 .22894 L
.73494 .22898 L
.73427 .22901 L
.73364 .22901 L
.73246 .229 L
.73181 .22897 L
.7312 .22893 L
.72983 .22882 L
.72767 .22857 L
.72533 .22824 L
.72315 .22792 L
.72202 .22776 L
.72085 .22763 L
.72028 .22757 L
.71976 .22753 L
.71878 .22748 L
.7179 .22747 L
.71696 .2275 L
.716 .2276 L
.71553 .22767 L
.71501 .22778 L
.71413 .22801 L
.71334 .2283 L
.71257 .22867 L
.71186 .22908 L
.71059 .23005 L
.70931 .23138 L
.70799 .2332 L
.70662 .23566 L
.70544 .23828 L
.7029 .24548 L
.698 .262 L
.6951 .27089 L
.69148 .28011 L
.68784 .28739 L
.68318 .29423 L
.68024 .2974 L
.67737 .29977 L
Mistroke
.6757 .30086 L
.67383 .30184 L
.672 .30258 L
.67026 .30309 L
.66861 .30341 L
.66704 .30358 L
.66528 .30363 L
.66427 .30358 L
.66334 .3035 L
.66145 .30321 L
.66036 .30297 L
.65935 .30271 L
.65542 .3013 L
.65118 .29915 L
.64719 .2966 L
.63797 .28902 L
.62106 .27056 L
.59202 .23445 L
.58568 .22794 L
.58238 .2251 L
.57939 .22296 L
.57672 .22145 L
.57526 .22081 L
.57394 .22036 L
.57286 .22008 L
.57169 .21989 L
.57049 .21981 L
.56939 .21984 L
.56819 .22001 L
.5669 .22034 L
.56627 .22056 L
.5656 .22085 L
.56442 .22147 L
.5622 .22306 L
.56027 .22493 L
.55591 .23094 L
.55215 .2381 L
.54049 .26727 L
.53737 .27456 L
.53388 .28147 L
.53038 .28699 L
.52693 .29116 L
.52301 .29471 L
.51834 .29776 L
.51353 .30002 L
.50873 .30176 L
.49647 .30514 L
.48418 .30811 L
.47258 .31063 L
.46959 .31114 L
Mistroke
.46801 .31137 L
.46657 .31154 L
.46527 .31167 L
.46411 .31176 L
.46285 .31182 L
.46218 .31184 L
.46155 .31184 L
.46056 .31182 L
.45951 .31175 L
.45844 .31165 L
.45748 .31151 L
.4566 .31133 L
.45582 .31115 L
.45415 .3106 L
.45257 .30988 L
.45097 .30889 L
.44943 .30763 L
.44811 .30625 L
.43959 .29029 L
.43695 .28525 L
.43354 .28023 L
.42995 .27646 L
.42758 .27462 L
.42513 .27312 L
.42211 .27173 L
.42052 .27117 L
.41898 .27072 L
.41756 .27038 L
.41594 .27006 L
.41425 .26982 L
.4124 .26963 L
.4114 .26957 L
.41033 .26952 L
.40836 .2695 L
.40725 .26952 L
.40619 .26956 L
.40502 .26962 L
.40375 .26972 L
.40141 .26996 L
.39915 .27026 L
.39482 .27099 L
.38602 .27303 L
.37781 .27539 L
.36034 .28106 L
.35368 .28308 L
.35042 .28393 L
.34894 .28425 L
.34763 .2845 L
.3464 .2847 L
.34534 .28482 L
Mistroke
.34425 .28489 L
.34317 .2849 L
.34234 .28484 L
.34152 .28472 L
.34081 .28454 L
.34023 .28432 L
.33961 .28399 L
.33904 .28355 L
.33859 .28306 L
.33826 .28254 L
.33795 .28186 L
.33781 .28143 L
.33771 .28102 L
.33759 .28019 L
.33754 .27922 L
.33758 .27823 L
.33763 .27764 L
.33769 .27707 L
.33786 .27598 L
.33809 .27477 L
.33974 .26863 L
.34365 .25605 L
.34468 .25219 L
.34551 .24854 L
.34608 .24529 L
.34631 .24349 L
.34646 .24184 L
.34654 .24042 L
.34656 .23892 L
.34652 .23753 L
.34644 .23627 L
.34627 .23484 L
.346 .23332 L
.34566 .23192 L
.34527 .23066 L
.34414 .22802 L
.34281 .2258 L
.34088 .22343 L
.33869 .22142 L
.33608 .21955 L
.33331 .218 L
.32994 .2165 L
.32596 .2151 L
.32143 .21386 L
.31686 .21288 L
.30777 .21142 L
.29099 .20955 L
.28198 .20856 L
.2779 .20801 L
.27441 .20745 L
Mistroke
.27121 .20683 L
.26806 .20607 L
.26517 .20516 L
.26285 .20419 L
.26091 .20304 L
.26005 .20235 L
.25941 .20168 L
.25894 .20104 L
.25874 .20069 L
.25856 .20031 L
.25833 .1996 L
.25823 .19892 L
.25824 .19822 L
.25833 .19757 L
.25854 .1968 L
.25884 .19606 L
.25961 .19471 L
.26077 .19319 L
.26452 .18967 L
.26915 .1864 L
.28085 .1801 L
.29219 .17528 L
.30237 .17143 L
.30611 .16986 L
.30739 .16919 L
.30784 .1689 L
.30825 .16859 L
.30857 .1683 L
.3088 .16802 L
.30895 .16775 L
.30906 .16747 L
.3091 .16716 L
.30907 .16687 L
.30897 .16654 L
.30889 .16634 L
.30879 .16616 L
.30824 .1654 L
.30736 .16448 L
.30511 .16249 L
.29392 .15166 L
.28774 .14345 L
.28301 .13635 L
.2811 .13389 L
.28004 .13278 L
.27902 .13192 L
.27785 .13118 L
.2772 .13088 L
.27686 .13075 L
.27648 .13062 L
.27578 .13046 L
Mistroke
.2751 .13035 L
.2744 .13031 L
.27372 .13031 L
.27297 .13036 L
.27226 .13046 L
.27144 .13061 L
.27053 .13082 L
.26697 .132 L
.25648 .1368 L
.24305 .14322 L
.23457 .14686 L
.22608 .15003 L
.21747 .15268 L
.21291 .15383 L
.20811 .15483 L
.20411 .15549 L
.20201 .15577 L
.19979 .15601 L
.19777 .15619 L
.19595 .1563 L
.19414 .15637 L
.19245 .15641 L
.19063 .1564 L
.18901 .15636 L
.18718 .15626 L
.18548 .15613 L
.18394 .15598 L
.18255 .1558 L
.17952 .15532 L
.1766 .1547 L
.17357 .15389 L
.16804 .15193 L
.16298 .14956 L
.15896 .14731 L
.1559 .14548 L
.15317 .14398 L
.15193 .14344 L
.15127 .14323 L
.15096 .14315 L
.15068 .14309 L
.15016 .14302 L
.14969 .14301 L
.14921 .14305 L
.1487 .14316 L
.14842 .14325 L
.14816 .14337 L
.14768 .14363 L
.14718 .144 L
.14665 .1445 L
.14571 .14568 L
Mistroke
.14486 .14708 L
.14328 .15049 L
.14156 .15525 L
.13836 .16639 L
.13137 .19816 L
.13009 .20572 L
.12902 .2133 L
.12859 .21701 L
.12823 .22088 L
.12811 .2225 L
.12801 .22419 L
.12793 .22573 L
.12789 .2271 L
.12786 .22837 L
.12786 .22968 L
.12787 .23087 L
.1279 .23192 L
.12795 .23312 L
.12798 .23378 L
.12802 .23437 L
.12811 .2355 L
.12822 .23651 L
.12847 .23843 L
.12882 .2404 L
.12972 .244 L
.13076 .24705 L
.13206 .25019 L
.13471 .25609 L
.14029 .27019 L
.14396 .27959 L
.14827 .28837 L
.15118 .29271 L
.15288 .29474 L
.15484 .29663 L
.15661 .29801 L
.15869 .29928 L
.15965 .29974 L
.16071 .30018 L
.16172 .30053 L
.1627 .3008 L
.1638 .30105 L
.16445 .30116 L
.16505 .30124 L
.16631 .30136 L
.16698 .30139 L
.1677 .3014 L
.16892 .30137 L
.17009 .30129 L
.17139 .30114 L
.17282 .30091 L
Mistroke
.17441 .30059 L
.17617 .30015 L
.17948 .29913 L
.18602 .29655 L
.19879 .29041 L
.20583 .28696 L
.21342 .28359 L
.21722 .28216 L
.22079 .28102 L
.22394 .28021 L
.22555 .27988 L
.22731 .2796 L
.22892 .27941 L
.23041 .27931 L
.23197 .27927 L
.23284 .27928 L
.23363 .27931 L
.23494 .27942 L
.23635 .27961 L
.2378 .27988 L
.23913 .28022 L
.24063 .28069 L
.24222 .28131 L
.24501 .28273 L
.24761 .28446 L
.24991 .28636 L
.25475 .29165 L
.25939 .2985 L
.26689 .31285 L
.27374 .32648 L
.2769 .33172 L
.28054 .33656 L
.28279 .33892 L
.28495 .34076 L
.28724 .34232 L
.28952 .34351 L
.2921 .34448 L
.29345 .34484 L
.29498 .34515 L
.29632 .34534 L
.2978 .34546 L
.29922 .3455 L
.30055 .34548 L
.30226 .34538 L
.30388 .34521 L
.30569 .34495 L
.30767 .34458 L
.31499 .34264 L
.33133 .33656 L
.34029 .33301 L
Mistroke
.34969 .3296 L
.35768 .32727 L
.35981 .32677 L
.36205 .32631 L
.36395 .32599 L
.36597 .32571 L
.36792 .32552 L
.36968 .32541 L
.37133 .32538 L
.37303 .32541 L
.37452 .32551 L
.37589 .32565 L
.37735 .32587 L
.37888 .32617 L
.38175 .32698 L
.38329 .32756 L
.38468 .32818 L
.3873 .32962 L
.38951 .33116 L
.39338 .33469 L
.39702 .33913 L
.40399 .35078 L
.40754 .35731 L
.41102 .363 L
.41432 .36732 L
.41833 .37117 L
.42088 .37295 L
.4235 .37433 L
.42603 .3753 L
.42753 .37574 L
.42897 .37608 L
.4304 .37633 L
.43175 .37651 L
.43336 .37665 L
.43493 .37671 L
.43641 .37671 L
.43781 .37667 L
.43939 .37656 L
.44111 .37639 L
.44304 .37614 L
.44517 .37578 L
.44917 .37494 L
.45697 .37273 L
.47506 .36559 L
.49117 .35762 L
.51696 .34208 L
.52382 .33704 L
.52744 .33419 L
.52867 .33335 L
.52898 .33318 L
Mistroke
.52929 .33303 L
.52956 .33292 L
.52986 .33282 L
.53013 .33276 L
.53039 .33271 L
.53064 .33269 L
.53092 .33267 L
.53123 .33268 L
.53154 .3327 L
.53189 .33274 L
.53229 .3328 L
.53319 .333 L
.53431 .33329 L
.53681 .33405 L
.54333 .33613 L
.56368 .34216 L
.57013 .34384 L
.57706 .34545 L
.58335 .34666 L
.5863 .3471 L
.58889 .34741 L
.59024 .34753 L
.59148 .34762 L
.59277 .34769 L
.59348 .34772 L
.59413 .34773 L
.59539 .34773 L
.5961 .34772 L
.59675 .3477 L
.59743 .34767 L
.59804 .34763 L
.59938 .34751 L
.60054 .34736 L
.60159 .3472 L
.60388 .34671 L
.60583 .34615 L
.60783 .34543 L
.61137 .34377 L
.61458 .34184 L
.62195 .33642 L
.65814 .3146 L
.67608 .30786 L
.69043 .30046 L
.69678 .29662 L
.70076 .29461 L
.70504 .2929 L
.70958 .29155 L
.71239 .29093 L
.71515 .29045 L
.7176 .29013 L
Mistroke
.72043 .28986 L
.7219 .28975 L
.72351 .28967 L
.72507 .28961 L
.72652 .28958 L
.72812 .28957 L
.72966 .28958 L
.73137 .28962 L
.73324 .28968 L
.73513 .28977 L
.73714 .28988 L
.74081 .29016 L
.74752 .29081 L
.75497 .29169 L
.76355 .2928 L
.77137 .29375 L
.77462 .29409 L
.77796 .29438 L
.77965 .29449 L
.78146 .2946 L
.78306 .29466 L
.78453 .29469 L
.78596 .2947 L
.78749 .29468 L
.78888 .29463 L
.79013 .29455 L
.79133 .29445 L
.79239 .29433 L
.79463 .29398 L
.79581 .29374 L
.79686 .29348 L
.79906 .2928 L
.8008 .29207 L
.80247 .29119 L
.80393 .29021 L
.80532 .28907 L
.80738 .2868 L
.80839 .28532 L
.80921 .2839 L
.81043 .28124 L
.81207 .27681 L
.81288 .27483 L
.81331 .27399 L
.81373 .27329 L
.8146 .27221 L
.81574 .2712 L
.81913 .26908 L
.83078 .26049 L
.8408 .25031 L
.85221 .23703 L
Mistroke
.87656 .20709 L
.89792 .18141 L
.9009 .17741 L
.90181 .17587 L
.90198 .17549 L
.90214 .17512 L
.90225 .17479 L
.90232 .1745 L
.90238 .174 L
.90237 .17375 L
.90233 .17351 L
.90226 .1733 L
.90216 .17311 L
.90191 .17278 L
.90158 .1725 L
.90119 .17226 L
.90068 .17203 L
.90004 .17181 L
.89861 .17144 L
.89672 .17109 L
.89244 .17037 L
.8861 .16909 L
.87943 .1672 L
.87616 .16603 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Eo`030?oo03Co002Ao`03
0?oo05Go00<0ool0=?l0097o00<0ool0E?l203Ko002Ao`030?oo05Co0P0fo`00OOl200Co0P05o`<0
1?l00`3oo`1Do`80=_l007co00@0ool02ol00`3oo`03o`030?oo05?o00<0o`00=_l007co00@0ool0
2?l500Co0P1Bo`800_l00`3oo`0do`00M_l400;o00@0ool02?l0103oo`05o`030?oo053o00<0ool0
0_l00`3oo`0do`00O?l0103oo`09o`030?l000Go00<0ool0D?l00`3oo`03o`030?oo03?o001Co`<0
9ol200_o0P05o`030?oo04ko0P06o`030?oo03?o001Bo`800_l00`3oo`0ho`030?oo04_o0`09o`03
0?oo03;o001@o`801Ol00`3oo`0go`030?oo04Oo100<o`030?oo03;o001>o`801ol00`3oo`0go`03
0?oo04;o1@0Ao`030?oo037o000bo`@05_l200[o00<0ool0=_l00`3oo`11o`030?oo01Co00<0ool0
<Ol003;o00@0oooo0P0?o`D03Ol00`3oo`0eo`030?oo043o0P0Go`030?oo033o000bo`030?oo00?o
0`09o`<04ol00`3oo`0do`030?oo043o00<0ool05ol00`3oo`0_o`00<Ol00`3oo`06o`X05ol00`3o
o`0co`030?oo043o00<0ool06?l202oo000ao`030?oo02So00<0ool0<_l00`3oo`10o`030?oo01[o
0P0]o`00<?l00`3oo`0Zo`80<_l00`3oo`11o`030?oo01_o0P0[o`00<?l00`3oo`0Zo`80<_l00`3o
o`11o`030?oo01go100Wo`00<?l00`3oo`0Xo`<0<ol00`3oo`11o`030?oo027o0`0To`00;ol00`3o
o`0Vo`<08_l200Co0P03o`@01Ol00`3oo`11o`030?oo02?o0P0So`00;ol00`3oo`0To`809?l0103o
o`08o`030?oo00Ko00<0ool0@Ol00`3oo`0So`030?oo02;o000_o`030?oo02;o0P0Vo`040?oo00Wo
00<0ool01Ol204;o00<0ool08_l00`3oo`0So`00;ol00`3oo`0Qo`030?oo023o1002o`040?oo00[o
00<0ool01?l00`3oo`11o`030?oo027o00<0ool09?l002ko00<0ool08Ol00`3oo`0Wo`040?oo00So
00@0ool01Ol00`3oo`11o`030?oo027o00<0ool09?l002ko00<0ool08?l00`3oo`0Yo`802_l200Ko
00<0ool0@Ol00`3oo`0Po`030?oo02Go000^o`030?oo01oo00<0ool0?_l00`3oo`11o`030?oo01oo
00<0ool09_l002ko00<0ool07ol203oo00<0ool0@Ol00`3oo`0No`030?oo02Oo000^o`030?oo023o
0`0mo`030?oo047o00<0ool07_l00`3oo`0Wo`001ol500;o0P04o`806?l00`3oo`08o`D05_l800Co
0P04o`801?l202?o00<0ool08_l200Co0P04o`804?l00`3oo`0@o`D02Ol00`3oo`0Fo`D00_l200Co
0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0No`L02?l0103oo`0Ro`030?oo027o
00@0ool02?l0103oo`0?o`030?oo01;o00<0ool02?l00`3oo`0Io`030?oo00Oo00@0ool00_l000Wo
00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool08Ol00`3oo`0300Wo00<0ool08?l00`3oo`0Qo`04
0?oo00_o00<0ool03Ol00`3oo`0Bo`030?oo00Oo00<0ool06_l00`3oo`0:o`400Ol1o`0000?o0000
0P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01_o1002o`040?oo00;o0P05o`808ol00`3o
o`0>o`@03ol0103oo`09o`803ol00`3oo`0Co`030?oo00Ko00<0ool06ol00`3oo`08o`800ol000Wo
00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool08Ol0103oo`03o`030?oo00?o00<0ool08_l00`3o
o`0=o`030?oo00;o00<0ool03?l0103oo`09o`030?oo00go0P0Eo`030?oo00Ko00<0ool06ol00`3o
o`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol202Co0P05o`030?oo00;o0`0Ro`030?oo00co
00<0ool01?l00`3oo`0<o`802_l300Co2P0Eo`801ol00`3oo`0Ko`802_l300;o000^o`030?oo03Ko
00<0ool09ol00`3oo`0;o`030?oo00Ko00<0ool07Ol202Ko00<0ool0;Ol002ko00<0ool0=ol00`3o
o`0Vo`030?oo00[o00<0ool02?l00`3oo`0Ko`809_l00`3oo`0^o`00;_l00`3oo`0go`030?oo02Ko
00<0ool02_l00`3oo`09o`030?oo01[o00<0ool09?l00`3oo`0_o`00;_l00`3oo`0go`030?oo02Ko
00<0ool02_l00`3oo`0:o`030?oo01So00<0ool09?l00`3oo`0`o`00;_l00`3oo`0go`030?oo02Ko
00<0ool02Ol00`3oo`0<o`030?oo01Oo00<0ool08ol00`3oo`0ao`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1Ol0103oo`05o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Go0P08o`030?oo00Go00<0ool0
1_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`03
0?oo023o00<0ool01_l00`3oo`0Yo`040?oo02Wo00<0ool02?l00`3oo`0?o`030?oo00co00<0ool0
1Ol00`3oo`0Qo`030?oo02[o00<0ool01ol0033o00<0ool0=?l00`3oo`0Wo`030?oo00So00<0ool0
4?l00`3oo`0Co`030?oo023o00<0ool0=Ol0033o00<0ool0<ol00`3oo`0Xo`030?oo00Oo00<0ool0
4_l00`3oo`0Bo`030?oo01ko0P0ho`00<Ol00`3oo`0bo`030?oo00go1P0Eo`030?oo00Oo00<0ool0
4ol00`3oo`0@o`030?oo01ko00<0ool0>?l0037o00<0ool0<_l00`3oo`09o`@01Ol301?o00<0ool0
1_l00`3oo`0Eo`030?oo00oo00<0ool07Ol00`3oo`0io`00<_l00`3oo`0`o`030?oo00Ko100;o`80
4_l00`3oo`06o`030?oo01Ko00<0ool03Ol00`3oo`0No`030?oo03Wo000bo`030?oo017o1`0Ho`03
0?oo00?o0`0Ao`030?oo00oo00<0ool01_l00`3oo`0Fo`030?oo00go00<0ool07Ol00`3oo`0jo`00
<ol00`3oo`0=o`<01ol00`3oo`0Eo`H05Ol00`3oo`0>o`030?oo00Go00<0ool06?l00`3oo`0;o`03
0?oo01ko00<0ool0>_l003?o00<0ool02ol200_o00<0ool0;ol00`3oo`0>o`030?oo00Go00<0ool0
6Ol00`3oo`0:o`030?oo01go00<0ool0>ol003Co00<0ool02?l200ko00<0ool0;_l00`3oo`0>o`03
0?oo00Co00<0ool06ol200Wo00<0ool00olA00So0P0no`00=?l200Ko0`0Ao`030?oo02ko00<0ool0
3Ol00`3oo`03o`030?oo01ko00<0ool01_l00`3oo`03017o2010o`00=Ol300;o0P0Eo`030?oo02go
00<0ool03Ol00`3oo`0302;o00<0ool01?l0103o001Lo`00=ol301Oo00<0ool0;_l00`3oo`0<o`<0
9_l805ko001Bo`030?oo02ko00<0ool01Ol702co0P1Po`00D_l00`3oo`0Xo`801?l900Go00<0ool0
:?l206;o001Bo`030?oo02Oo00@0ool02?l00`3oo`06o`030?oo02Oo00<0ool0H_l005?o00<0ool0
9_l0103oo`09o`030?oo00Go0P0Vo`80IOl005?o00<0ool09_l0103oo`0:o`030?oo00Co00<0ool0
8ol206Oo001Do`030?oo01Co1`0:o`040?oo00So00@0ool01Ol00`3oo`0Ro`030?oo06Oo001Eo`03
0?oo013o1005o`<02Ol200[o0P06o`030?oo023o0P1Zo`00EOl00`3oo`0=o`<02ol00`3oo`0Ko`03
0?oo00Co0`0Go`80K?l005Ko00<0ool02_l200oo00<0ool06_l00`3oo`02o`800ol301;o0P1^o`00
Eol00`3oo`07o`804_l00`3oo`0Io`050?ooo`002?l200oo00<0ool0K_l005So2@0Eo`030?oo01So
00@0o`002ol600Oo0P1ao`00M_l00`3oo`0Ho`804ol807;o001go`030?oo01Oo00<0ool0S?l007So
00<0ool05?l308ko001ho`030?oo01;o0P02o`030?oo08co001io`030?oo00oo0P04o`030?oo08co
001jo`030?oo00co0P06o`030?oo08co001ko`030?oo00Wo0P08o`030?oo08co001ko`@00ol500?o
0`04o`030?oo08co001lo`030?l000<02Ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87637 .16792 m
.86888 .16506 L
.86189 .1614 L
.85448 .15627 L
.84723 .14959 L
.84068 .14152 L
.83574 .13327 L
.83245 .12569 L
.83108 .12148 L
.83013 .1177 L
.82944 .11398 L
.82918 .11199 L
.82901 .11023 L
.82892 .10866 L
.82888 .10702 L
.8289 .10555 L
.82896 .10425 L
.82902 .1035 L
.8291 .10271 L
.82928 .10137 L
.8295 .10018 L
.82974 .09918 L
.83038 .09726 L
.83073 .09649 L
.83114 .09581 L
.83133 .09555 L
.83154 .0953 L
.83173 .09512 L
.83192 .09496 L
.8321 .09486 L
.83228 .09477 L
.8326 .09472 L
.83278 .09474 L
.83294 .09479 L
.8331 .09489 L
.83326 .09504 L
.83339 .09521 L
.8335 .09539 L
.83361 .09563 L
.83371 .09592 L
.83379 .09622 L
.83385 .09653 L
.83389 .09689 L
.83392 .09732 L
.83393 .0981 L
.8339 .09859 L
.83386 .09907 L
.83372 .10003 L
.83354 .10098 L
.83291 .10331 L
Mistroke
.83197 .10582 L
.83084 .10824 L
.82731 .11399 L
.82283 .11939 L
.81763 .12423 L
.81054 .12927 L
.80332 .1331 L
.79499 .13629 L
.79076 .13749 L
.78618 .13851 L
.78228 .13918 L
.77822 .13972 L
.77458 .14008 L
.77142 .14032 L
.76857 .14051 L
.76576 .14071 L
.76457 .14082 L
.76338 .14094 L
.76143 .14122 L
.76045 .14142 L
.75961 .14163 L
.75875 .1419 L
.75792 .14224 L
.7571 .14269 L
.75642 .14318 L
.75589 .14368 L
.75539 .1443 L
.75498 .14499 L
.75461 .14584 L
.75446 .1463 L
.75432 .14683 L
.75421 .14736 L
.75413 .14787 L
.75402 .1489 L
.75398 .14953 L
.75396 .15013 L
.75397 .15129 L
.75402 .15262 L
.7541 .15386 L
.75423 .15529 L
.75458 .15843 L
.75543 .1648 L
.75644 .17267 L
.75694 .17732 L
.75715 .17965 L
.75732 .18183 L
.75745 .18388 L
.75757 .18614 L
.75765 .18842 L
.75768 .18966 L
Mistroke
.75769 .19083 L
.75769 .19288 L
.75766 .19475 L
.75758 .19688 L
.75746 .19888 L
.75731 .20071 L
.75711 .20263 L
.75665 .20599 L
.75591 .20978 L
.75506 .21299 L
.75397 .21615 L
.75256 .21927 L
.75086 .22212 L
.74908 .22438 L
.74725 .22614 L
.74548 .22745 L
.74446 .22805 L
.74331 .2286 L
.74229 .22901 L
.74117 .22937 L
.73995 .22967 L
.73879 .22987 L
.73811 .22995 L
.7374 .23001 L
.73611 .23005 L
.73545 .23005 L
.73475 .23002 L
.73348 .22993 L
.73271 .22985 L
.73198 .22975 L
.73063 .22953 L
.72604 .2285 L
.72375 .2279 L
.7214 .22731 L
.7203 .22707 L
.71929 .22688 L
.71841 .22674 L
.71747 .22664 L
.71662 .2266 L
.71574 .22662 L
.71494 .2267 L
.71425 .22682 L
.71382 .22693 L
.71338 .22708 L
.71262 .2274 L
.71184 .22785 L
.71102 .22847 L
.71034 .22912 L
.70964 .22993 L
.70846 .23172 L
Mistroke
.70725 .23422 L
.70601 .23753 L
.70388 .24505 L
.69956 .26322 L
.69711 .27237 L
.69391 .28214 L
.69177 .28747 L
.68959 .29204 L
.68468 .29984 L
.68171 .30319 L
.67874 .30572 L
.67694 .3069 L
.67524 .30778 L
.67428 .3082 L
.67322 .30858 L
.67221 .30888 L
.67125 .30911 L
.66947 .30938 L
.6685 .30946 L
.66746 .30949 L
.66635 .30945 L
.66528 .30936 L
.66325 .30904 L
.66132 .30857 L
.6595 .30797 L
.65527 .30608 L
.65116 .30364 L
.64732 .30085 L
.63845 .29286 L
.62218 .2738 L
.60374 .24835 L
.58834 .22768 L
.58182 .22037 L
.57616 .21541 L
.57324 .21354 L
.57165 .21277 L
.57023 .21224 L
.56907 .21194 L
.56781 .21174 L
.56654 .2117 L
.56538 .21181 L
.56413 .21209 L
.56344 .21232 L
.56281 .21258 L
.56168 .21318 L
.56051 .21398 L
.55837 .21595 L
.55653 .21819 L
.55242 .22528 L
.54895 .23357 L
Mistroke
.53836 .26683 L
.53552 .27499 L
.5323 .28262 L
.52899 .28867 L
.52569 .29319 L
.52183 .29706 L
.51716 .30035 L
.51229 .3028 L
.5074 .30469 L
.49479 .30849 L
.48223 .31189 L
.47647 .31337 L
.47067 .31469 L
.46839 .31513 L
.46605 .31551 L
.46475 .31568 L
.4636 .3158 L
.46247 .31588 L
.46144 .31593 L
.46033 .31594 L
.4592 .3159 L
.45827 .31583 L
.45728 .3157 L
.45625 .31551 L
.4552 .31524 L
.45426 .31492 L
.45343 .31456 L
.45174 .31359 L
.45086 .31292 L
.4501 .31223 L
.4488 .31075 L
.44756 .30892 L
.44394 .30042 L
.44212 .29487 L
.4402 .28948 L
.43821 .28493 L
.43556 .28036 L
.43376 .27798 L
.43183 .27593 L
.42991 .27428 L
.42758 .27269 L
.42503 .27136 L
.42363 .27078 L
.42206 .27023 L
.42053 .2698 L
.41907 .26947 L
.41755 .2692 L
.41607 .26901 L
.4143 .26885 L
.41235 .26876 L
Mistroke
.41045 .26876 L
.40869 .26882 L
.40655 .26897 L
.40533 .26909 L
.40418 .26922 L
.39974 .2699 L
.39523 .27084 L
.39092 .27192 L
.38095 .27495 L
.36166 .282 L
.35387 .28487 L
.35025 .28608 L
.34682 .28707 L
.34553 .28738 L
.34423 .28764 L
.34353 .28776 L
.34292 .28784 L
.34181 .28792 L
.34095 .28792 L
.34011 .28785 L
.33934 .28768 L
.33872 .28743 L
.3382 .28711 L
.33773 .28667 L
.33739 .28617 L
.33715 .28564 L
.337 .28509 L
.33691 .28442 L
.33689 .28365 L
.33694 .28286 L
.33708 .28185 L
.33728 .28082 L
.33786 .27869 L
.3399 .27302 L
.34548 .25949 L
.34847 .25135 L
.34968 .24734 L
.35058 .24371 L
.35093 .24194 L
.35119 .24032 L
.3514 .23866 L
.35155 .23687 L
.3516 .23508 L
.35157 .2335 L
.35151 .23264 L
.35142 .23172 L
.35116 .23009 L
.35087 .22878 L
.35045 .22737 L
.34991 .22596 L
Mistroke
.34931 .22468 L
.34799 .22254 L
.34623 .22041 L
.34373 .21818 L
.34055 .2161 L
.33708 .21441 L
.33357 .21309 L
.32901 .21178 L
.32365 .21064 L
.31826 .2098 L
.31307 .20918 L
.30334 .20835 L
.29265 .20767 L
.28751 .20735 L
.28216 .20697 L
.27779 .20658 L
.27332 .20607 L
.26937 .20547 L
.26568 .2047 L
.26277 .20386 L
.2616 .20343 L
.2605 .20295 L
.2596 .20248 L
.25873 .20194 L
.25805 .20141 L
.25753 .20091 L
.25702 .20029 L
.25663 .1996 L
.25639 .19893 L
.25628 .19827 L
.25629 .19754 L
.25634 .19712 L
.25643 .19672 L
.25672 .19589 L
.25694 .19542 L
.25717 .19498 L
.25835 .19331 L
.25988 .19167 L
.26452 .18797 L
.27111 .18399 L
.28501 .17762 L
.29684 .17338 L
.30222 .17166 L
.30641 .17027 L
.30765 .16978 L
.30864 .1693 L
.30906 .16903 L
.30935 .16878 L
.30955 .16853 L
.30965 .16829 L
Mistroke
.30968 .16807 L
.30964 .16782 L
.30952 .16754 L
.30934 .16727 L
.30887 .16675 L
.30815 .16611 L
.30311 .16226 L
.29608 .15575 L
.29013 .14807 L
.28556 .14004 L
.28355 .1361 L
.28179 .13301 L
.28098 .13187 L
.28009 .13088 L
.27959 .13045 L
.27902 .13005 L
.27848 .12975 L
.27794 .12954 L
.27739 .12938 L
.27684 .12929 L
.27632 .12925 L
.27572 .12926 L
.27504 .12932 L
.27426 .12945 L
.27349 .12963 L
.27275 .12984 L
.26886 .1314 L
.26448 .13358 L
.25168 .14042 L
.23751 .14727 L
.22887 .15079 L
.22039 .15368 L
.21586 .15497 L
.21094 .15617 L
.20688 .15697 L
.20459 .15735 L
.20246 .15765 L
.20032 .15791 L
.19838 .15809 L
.19633 .15824 L
.19416 .15833 L
.19235 .15837 L
.1904 .15836 L
.18859 .1583 L
.18694 .1582 L
.1849 .15802 L
.18275 .15777 L
.18077 .15746 L
.179 .15713 L
.17579 .15637 L
Mistroke
.1726 .1554 L
.16666 .15296 L
.16121 .14986 L
.15403 .14444 L
.1515 .14242 L
.15055 .14177 L
.1496 .14124 L
.14911 .14105 L
.14868 .14094 L
.14824 .1409 L
.148 .1409 L
.14778 .14093 L
.14756 .14099 L
.14736 .14106 L
.14715 .14116 L
.14692 .14129 L
.14647 .14165 L
.14607 .14208 L
.14537 .14309 L
.14462 .14455 L
.14324 .14815 L
.14032 .1585 L
.13674 .17383 L
.13294 .19219 L
.13003 .20858 L
.129 .21568 L
.12856 .21919 L
.12818 .22271 L
.12791 .22588 L
.12773 .22857 L
.12768 .22981 L
.12763 .23109 L
.12761 .23227 L
.1276 .23331 L
.12761 .23446 L
.12764 .23565 L
.12768 .23672 L
.12774 .23766 L
.12783 .23872 L
.12794 .23982 L
.1282 .2417 L
.12854 .24351 L
.12896 .24529 L
.12987 .24822 L
.13903 .27479 L
.14224 .28411 L
.14422 .28868 L
.14676 .29328 L
.14828 .29545 L
.15006 .29753 L
Mistroke
.1519 .29922 L
.1537 .3005 L
.15562 .30151 L
.15676 .30196 L
.15785 .3023 L
.15884 .30254 L
.15996 .30273 L
.16116 .30286 L
.16232 .3029 L
.16356 .30288 L
.16421 .30283 L
.16493 .30277 L
.16623 .30259 L
.16748 .30236 L
.17041 .30162 L
.17373 .30049 L
.18651 .2943 L
.20012 .28648 L
.21312 .27974 L
.2197 .27716 L
.22329 .27612 L
.22485 .27576 L
.22649 .27546 L
.22734 .27533 L
.22828 .27521 L
.2299 .27508 L
.2308 .27504 L
.23177 .27504 L
.23266 .27506 L
.23348 .27511 L
.23502 .27528 L
.23589 .27541 L
.23668 .27557 L
.23811 .27592 L
.2396 .2764 L
.24268 .27778 L
.24434 .27877 L
.24583 .27981 L
.2484 .28199 L
.25106 .28482 L
.25531 .29071 L
.25935 .29803 L
.26691 .31539 L
.2705 .32385 L
.274 .33102 L
.27746 .33667 L
.28156 .34152 L
.28389 .34351 L
.28622 .34503 L
.28753 .3457 L
Mistroke
.28895 .3463 L
.29057 .34683 L
.29211 .3472 L
.29364 .34745 L
.29512 .34759 L
.29649 .34764 L
.29801 .34762 L
.29972 .3475 L
.30059 .34741 L
.30157 .34728 L
.30503 .34663 L
.30841 .34576 L
.31218 .34457 L
.31933 .34185 L
.33798 .33361 L
.34724 .32966 L
.35547 .32663 L
.35953 .32537 L
.3638 .32429 L
.36609 .32383 L
.36815 .32349 L
.37007 .32325 L
.37206 .32309 L
.37402 .32302 L
.37509 .32302 L
.37606 .32305 L
.37789 .32319 L
.37958 .32341 L
.3805 .32357 L
.38147 .32378 L
.38316 .32423 L
.38491 .32483 L
.38675 .32562 L
.38962 .32725 L
.39204 .32906 L
.39445 .33133 L
.39684 .33414 L
.40058 .33986 L
.40367 .3459 L
.40965 .35931 L
.41277 .36541 L
.41604 .37036 L
.41811 .37277 L
.4205 .375 L
.42314 .37687 L
.42478 .37776 L
.42635 .37847 L
.42772 .37897 L
.42927 .37943 L
.43097 .37981 L
Mistroke
.43265 .38008 L
.43428 .38025 L
.43607 .38034 L
.43764 .38034 L
.43942 .38027 L
.44126 .38012 L
.44301 .37991 L
.44498 .37961 L
.44714 .37921 L
.456 .37685 L
.46478 .37368 L
.47296 .37017 L
.50842 .35041 L
.51949 .34259 L
.52696 .33626 L
.53093 .33193 L
.53184 .33076 L
.53217 .3304 L
.53232 .33027 L
.53245 .33017 L
.53259 .33009 L
.53275 .33004 L
.53292 .33 L
.5331 .33 L
.53329 .33002 L
.53348 .33005 L
.5337 .33011 L
.53396 .33019 L
.5375 .33171 L
.55175 .33739 L
.56255 .34115 L
.57512 .34503 L
.58113 .34661 L
.58658 .3478 L
.58886 .34821 L
.5912 .34855 L
.59239 .34869 L
.59368 .34882 L
.59483 .34891 L
.59588 .34896 L
.59703 .34899 L
.59823 .34899 L
.59934 .34895 L
.60032 .34889 L
.60145 .34878 L
.60265 .34862 L
.60375 .34842 L
.60474 .3482 L
.60667 .34766 L
.6087 .34691 L
Mistroke
.61261 .34489 L
.61589 .34266 L
.6194 .33986 L
.62612 .33407 L
.63477 .32718 L
.64552 .32032 L
.65627 .31507 L
.66583 .31125 L
.67428 .308 L
.68172 .30452 L
.68752 .30085 L
.69263 .29714 L
.69832 .29361 L
.70221 .29181 L
.7064 .29038 L
.70892 .28974 L
.71148 .28922 L
.71392 .28884 L
.71672 .28853 L
.71823 .2884 L
.71995 .2883 L
.72082 .28826 L
.72178 .28823 L
.72355 .28819 L
.72535 .28819 L
.72709 .28822 L
.72869 .28827 L
.73046 .28835 L
.73246 .28847 L
.73433 .2886 L
.73864 .28899 L
.74669 .28997 L
.76193 .29229 L
.7702 .29358 L
.77746 .29456 L
.78089 .29492 L
.78258 .29507 L
.78437 .29519 L
.78537 .29525 L
.78627 .29528 L
.78711 .29531 L
.78799 .29533 L
.78947 .29532 L
.79029 .29531 L
.79104 .29528 L
.7923 .29521 L
.79359 .2951 L
.79476 .29496 L
.79578 .2948 L
.79692 .29458 L
Mistroke
.7981 .2943 L
.80007 .29368 L
.8019 .29288 L
.8036 .29188 L
.80485 .29091 L
.80603 .28974 L
.80766 .28747 L
.80835 .28611 L
.80886 .28485 L
.80924 .28367 L
.80957 .2824 L
.80999 .28018 L
.81029 .27791 L
.81041 .27685 L
.81054 .27584 L
.81067 .27504 L
.81076 .27466 L
.81085 .27427 L
.81096 .27394 L
.81106 .27366 L
.8113 .27316 L
.81149 .27287 L
.81168 .27262 L
.8122 .27211 L
.81276 .2717 L
.81349 .27128 L
.81738 .26925 L
.82313 .26537 L
.83187 .25756 L
.8541 .23267 L
.88246 .19922 L
.89521 .18505 L
.90041 .1794 L
.90374 .1756 L
.90467 .17435 L
.90502 .17378 L
.90524 .17332 L
.90532 .1731 L
.90537 .17288 L
.90539 .17267 L
.90537 .17249 L
.90533 .17234 L
.90525 .1722 L
.90503 .17197 L
.90473 .17181 L
.90431 .17167 L
.90376 .17155 L
.90313 .17148 L
.90247 .17143 L
.90178 .17139 L
Mistroke
.90095 .17136 L
.89998 .17134 L
.898 .17129 L
.89677 .17125 L
.89557 .17121 L
.8944 .17115 L
.89306 .17106 L
.89048 .17083 L
.88895 .17066 L
.8872 .17042 L
.88349 .16976 L
.87987 .16893 L
.87637 .16792 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Eo`030?oo03Co002Ao`030?oo05Co0P0fo`00
TOl00`3oo`1Do`80=_l0097o00<0ool0E?l203Ko002Ao`030?oo05Co0P0fo`00OOl200Co0P05o`<0
1?l00`3oo`1Do`030?oo03Go001lo`040?oo00_o00<0ool00ol00`3oo`1Do`030?oo03Go001lo`04
0?oo00So1@04o`80Dol303Oo001fo`@00_l0103oo`08o`040?oo00Go00<0ool0DOl01@3oool003Ko
001lo`040?oo00Wo00<0o`001Ol00`3oo`1Ao`050?ooo`00=_l005?o100Vo`802ol200Go00<0ool0
Col200Co00<0ool0=?l005;o00D0oooo000jo`030?oo04go0P07o`030?oo03?o001@o`801Ol00`3o
o`0go`030?oo04Wo1009o`030?oo03?o000bo`<06Ol200Oo00<0ool0=ol00`3oo`11o`P03_l00`3o
o`0bo`00<_l0103o000Fo`802_l00`3oo`0fo`030?oo043o0P0Fo`030?oo037o000bo`040?ooo`80
4_l200go00<0ool0=Ol00`3oo`10o`030?oo01Ko00<0ool0<?l003;o00<0ool00ol00`3oo`0;o`@0
3ol00`3oo`0eo`030?oo043o00<0ool05ol00`3oo`0_o`00<Ol00`3oo`05o`@01?l501Co00<0ool0
=?l00`3oo`10o`030?oo01So0P0_o`00<Ol00`3oo`09o`D06Ol00`3oo`0co`030?oo043o00<0ool0
6_l202go000ao`030?oo02So00<0ool0<_l00`3oo`10o`030?oo01co0P0[o`00<?l00`3oo`0Zo`80
<_l00`3oo`11o`030?oo01go0P0Yo`00<?l00`3oo`0Xo`@0<_l00`3oo`11o`030?oo01oo1`0Ro`00
<?l00`3oo`0Vo`808_l200Co0P03o`@01Ol00`3oo`11o`030?oo02Co00<0ool08Ol0033o00<0ool0
8ol302?o00@0ool02?l00`3oo`06o`030?oo047o00<0ool08ol00`3oo`0Ro`00<?l00`3oo`0Po`<0
9_l0103oo`09o`030?oo00Go0P12o`030?oo02?o00<0ool08_l002oo00<0ool08?l00`3oo`0Qo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Ro`030?oo02?o000_o`030?oo01oo00<0ool0
:?l0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Po`809_l002oo00<0ool07_l00`3oo`0Zo`80
2_l200Ko00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000_o`030?oo01ko00<0ool0?_l00`3oo`11o`03
0?oo01ko00<0ool09ol002ko00<0ool07ol203oo00<0ool0@Ol00`3oo`0Mo`030?oo02So000^o`03
0?oo023o1P0jo`030?oo047o00<0ool07?l00`3oo`0Yo`001ol500;o0P04o`806?l00`3oo`08o`D0
6Ol;00Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0Ao`030?oo00oo1@08o`030?oo01Oo1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool05ol00`3oo`0:o`030?oo027o00<0ool00`06o`040?oo02;o
00<0ool03Ol4013o00@0ool02?l0103oo`0?o`030?oo01;o00<0ool01ol00`3oo`0Jo`030?oo00Oo
00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool08Ol0103oo`02o`801ol00`3o
o`0Po`030?oo00co00<0ool00_l00`3oo`0=o`040?oo00_o00<0ool03Ol00`3oo`0Bo`030?oo00Ko
00<0ool06ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`03
0?oo01_o1002o`040?oo00Co0P03o`808ol00`3oo`0<o`030?oo00?o00<0ool03?l0103oo`09o`80
4?l00`3oo`0Bo`030?oo00Go00<0ool07?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`03
0?oo00[o00<0ool08Ol0103oo`05o`050?ooo`009?l00`3oo`0:o`801ol00`3oo`0;o`040?oo00Wo
00<0ool03_l00`3oo`0Co`030?oo00Go00<0ool07?l00`3oo`08o`030?oo00;o0008o`802_l301Ko
0P0;o`809?l200Oo00<0ool00`0Ro`030?oo00[o00<0ool01ol00`3oo`0;o`802_l300?o1P03o`80
5Ol200Ko00<0ool07?l200[o0`02o`00;Ol00`3oo`0io`030?oo02Go00<0ool02_l00`3oo`08o`03
0?oo01co00<0ool00ol401go00<0ool0;_l002go00<0ool0>Ol00`3oo`0Uo`030?oo00[o00<0ool0
2?l00`3oo`0Lo`030?oo02?o00<0ool0;ol002go0P0jo`030?oo02Go00<0ool02_l00`3oo`09o`03
0?oo01[o00<0ool08ol00`3oo`0`o`00;_l00`3oo`0ho`030?oo02Go00<0ool02Ol00`3oo`0;o`03
0?oo01Wo00<0ool08_l00`3oo`0ao`00;_l00`3oo`0ho`030?oo02Go00<0ool02Ol00`3oo`0;o`03
0?oo01Wo00<0ool08_l00`3oo`0ao`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`050?ooo`00
1?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00@0ool01Ol00`3oo`06o`03
0?l000Ko00<0ool01_l00`3oo`06o`801ol00`3oo`05o`030?oo00Ko00<0ool01_l200Oo00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Po`030?oo00Ko00<0ool0
:Ol0103oo`0Yo`030?oo00So00<0ool03ol00`3oo`0<o`030?oo00Ko00<0ool08?l00`3oo`0Zo`03
0?oo00Oo000_o`030?oo03Go00<0ool09ol00`3oo`07o`030?oo013o00<0ool05Ol00`3oo`0No`80
=ol002oo00<0ool0=Ol00`3oo`0Wo`030?oo00Oo00<0ool04Ol00`3oo`0Do`030?oo01go00<0ool0
=ol002oo00<0ool0=?l00`3oo`0=o`L05?l00`3oo`07o`030?oo01;o00<0ool04_l00`3oo`0Mo`03
0?oo03So000`o`030?oo03?o00<0ool02_l300Ko0`0Bo`030?oo00Ko00<0ool05?l00`3oo`0Ao`03
0?oo01_o0P0ko`00<?l00`3oo`0Bo`P06Ol00`3oo`07o`<02ol2017o00<0ool01_l00`3oo`0Eo`03
0?oo013o00<0ool06ol00`3oo`0jo`00<Ol00`3oo`0@o`030?oo00Go0P0Go`030?oo00Ko0P0?o`03
0?oo00oo00<0ool01Ol00`3oo`0Go`030?oo00ko00<0ool07?l00`3oo`0jo`00<Ol00`3oo`0>o`80
2_l00`3oo`0Do`030?oo00;o100Bo`030?oo00ko00<0ool01Ol00`3oo`0Go`030?oo00ko00<0ool0
7?l00`3oo`0jo`00<_l00`3oo`0;o`803Ol00`3oo`0Co`D05ol00`3oo`0=o`030?oo00Co00<0ool0
6Ol00`3oo`0=o`030?oo00;o200Ao`80??l003?o00<0ool02?l200oo00<0ool0;ol00`3oo`0=o`03
0?oo00?o00<0ool06ol00`3oo`0;o`030?oo00<02?l600[o0P0mo`00<ol00`3oo`06o`804_l00`3o
o`0^o`030?oo00go00<0ool00_l00`3oo`0Mo`030?oo00Wo00@0ool04Ol:03oo000co`801Ol201Co
00<0ool0;ol00`3oo`0<o`040?ooo`808?l00`3oo`08o`<0Fol003Go1@0Go`030?oo02ko00<0ool0
3?l0103oo`0So`801ol205go001Bo`030?oo02ko00<0ool02ol302Ko0P02o`@0G_l005;o00<0ool0
:?l200Co0P03o`X0:Ol5063o001Co`030?oo02Ko00@0ool00ol30003o`0000<01Ol00`3oo`0Vo`80
I?l005?o00<0ool09_l0103oo`05o`<000?o0?l01_l202Go0P1Vo`00Dol00`3oo`0Ho`<02ol0103o
o`0:o`030?oo00Co00<0ool08ol00`3oo`1Vo`00Dol00`3oo`0Co`D00_l300Wo00@0ool02?l0103o
o`05o`030?oo027o0P1Yo`00E?l00`3oo`0@o`802_l200So0P0:o`801_l00`3oo`05o`806?l206_o
001Do`030?oo00go0`0=o`806ol00`3oo`05o`<05_l00`3oo`1[o`00EOl00`3oo`0:o`804_l00`3o
o`0Ho`030?oo00Co00@0oooo100Ao`030?oo06co001Fo`030?oo00Oo0P0Do`030?oo01So00<0ool0
0ol00`3oo`06o`<03Ol00`3oo`1]o`00Eol200Go0P0Go`030?oo01Oo00<0ool00_l00`3oo`0:o`80
2Ol2073o001Ho`H06Ol00`3oo`0Go`030?oo00803ol907;o001go`030?oo01Oo0`2<o`00N?l00`3o
o`0Eo`80S_l007So00<0ool04ol20003o`3o08go001io`030?oo013o0P03o`030?oo08co001io`03
0?oo00ko0P05o`030?oo08co001jo`030?oo00_o0P07o`030?oo08co001ko`040?l000Co0P000ol0
0002o`<01?l00`3oo`2<o`00O?l400;o1005o`030?oo00?o00<0ool0S?l007co00<0o`000`06o`D0
1?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0
S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<0
1_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80
SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`80
2_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 486},
WindowMargins->{{Automatic, 98}, {67, Automatic}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 190, 5, 465, "Title"],

Cell[CellGroupData[{
Cell[1954, 60, 32, 0, 53, "Section"],
Cell[1989, 62, 345, 9, 52, "Text"],
Cell[2337, 73, 320, 6, 52, "Text"],
Cell[2660, 81, 304, 6, 52, "Text"],
Cell[2967, 89, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3200, 98, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3382, 106, 48, 0, 45, "Subsection"],
Cell[3433, 108, 67, 1, 27, "Input"],
Cell[3503, 111, 103, 2, 27, "Input"],
Cell[3609, 115, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3774, 122, 46, 0, 29, "Subsection"],
Cell[3823, 124, 326, 7, 50, "Text"],
Cell[4152, 133, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4599, 145, 103, 2, 29, "Subsection"],
Cell[4705, 149, 105, 3, 32, "Text"],
Cell[4813, 154, 244, 6, 42, "Input"],
Cell[5060, 162, 123, 2, 27, "Input"],
Cell[5186, 166, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5469, 176, 78, 0, 42, "Subsubsection"],
Cell[5550, 178, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5720, 186, 60, 0, 29, "Subsection"],
Cell[5783, 188, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6040, 196, 130, 3, 46, "Subsection"],
Cell[6173, 201, 121, 2, 27, "Input"],
Cell[6297, 205, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6478, 213, 31, 0, 33, "Section"],
Cell[6512, 215, 101, 3, 33, "Text"],
Cell[6616, 220, 202, 3, 30, "Input"],
Cell[6821, 225, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[7012, 232, 44, 0, 33, "Section"],
Cell[7059, 234, 66, 0, 33, "Text"],
Cell[7128, 236, 1960, 30, 290, "Input"],

Cell[CellGroupData[{
Cell[9113, 270, 1950, 31, 290, "Input"],
Cell[11066, 303, 181, 3, 24, "Message"]
}, Open  ]],
Cell[11262, 309, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11363, 315, 32, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11420, 319, 80, 1, 30, "Input"],
Cell[11503, 322, 14621, 643, 152, 7832, 555, "GraphicsData", "PostScript", \
"Graphics"],
Cell[26127, 967, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26306, 976, 35, 0, 53, "Section"],
Cell[26344, 978, 176, 4, 52, "Text"],
Cell[26523, 984, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[27009, 995, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[27176, 1002, 23006, 1109, 152, 14814, 1002, "GraphicsData", \
"PostScript", "Graphics"],
Cell[50185, 2113, 21996, 1059, 152, 14082, 955, "GraphicsData", "PostScript", \
"Graphics"],
Cell[72184, 3174, 20937, 1002, 152, 13185, 900, "GraphicsData", "PostScript", \
"Graphics"],
Cell[93124, 4178, 20505, 983, 152, 12972, 884, "GraphicsData", "PostScript", \
"Graphics"],
Cell[113632, 5163, 20120, 971, 152, 12806, 875, "GraphicsData", "PostScript", \
"Graphics"],
Cell[133755, 6136, 19272, 925, 152, 12132, 831, "GraphicsData", "PostScript", \
"Graphics"],
Cell[153030, 7063, 18085, 856, 152, 11070, 763, "GraphicsData", "PostScript", \
"Graphics"],
Cell[171118, 7921, 16555, 766, 152, 9694, 675, "GraphicsData", "PostScript", \
"Graphics"],
Cell[187676, 8689, 14680, 645, 152, 7832, 555, "GraphicsData", "PostScript", \
"Graphics"],
Cell[202359, 9336, 15744, 713, 152, 8859, 622, "GraphicsData", "PostScript", \
"Graphics"],
Cell[218106, 10051, 17087, 791, 152, 10064, 698, "GraphicsData", \
"PostScript", "Graphics"],
Cell[235196, 10844, 18972, 905, 152, 11820, 811, "GraphicsData", \
"PostScript", "Graphics"],
Cell[254171, 11751, 20009, 960, 152, 12650, 863, "GraphicsData", \
"PostScript", "Graphics"],
Cell[274183, 12713, 20831, 1006, 152, 13322, 907, "GraphicsData", \
"PostScript", "Graphics"],
Cell[295017, 13721, 21169, 1014, 152, 13409, 912, "GraphicsData", \
"PostScript", "Graphics"],
Cell[316189, 14737, 21924, 1052, 152, 13962, 948, "GraphicsData", \
"PostScript", "Graphics"],
Cell[338116, 15791, 22726, 1094, 152, 14586, 988, "GraphicsData", \
"PostScript", "Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

