(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    344353,      16020]*)
(*NotebookOutlinePosition[    345008,      16043]*)
(*  CellTagsIndexPosition[    344964,      16039]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Fisher Linear Discriminant Direction, Full (40 dim \
subspace), Normals vs. Schizos, for Major Axis Rescaled, Full Population, on \
raw Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {1.00000000222222, 
          0.00000000074074, \(-0.00000000333333\), \(-0.15425630925926\), 
          0.05169293074074, \(-0.06879220814815\), \(-0.23349342185185\), 
          0.01942750444444, 0.02644838814815, 
          0.03411046925926, \(-0.00085310333333\), \(-0.08502326592593\), 
          0.00445768851852, \(-0.07443836148148\), \(-0.03757112740741\), \
\(-0.01314235111111\), 0.02496162851852, 
          0.01744490370370, \(-0.00924242481481\), \(-0.04486334851852\), \
\(-0.01167827814815\), \(-0.03341475777778\), \(-0.00899276037037\), \
\(-0.01361780259259\), 0.02253759407407, \(-0.00057942185185\), 
          0.00006088370370, \(-0.00850054296296\), \(-0.00318135962963\), 
          0.00169321407407, \(-0.00084379518519\), \(-0.00596311259259\), 
          0.00773305148148, \(-0.00597819814815\), 0.00465755111111, 
          0.00234933962963, 0.00598151148148, 0.00743163703704, 
          0.00069807370370, \(-0.00276610370370\), \(-0.00389088629630\), 
          0.00083397370370, 0.00164261629630, 0.00180199777778, 
          0.00444316000000, 
          0.00138173074074, \(-0.00039679962963\), \(-0.00024159888889\), \
\(-0.00377818703704\), 0.00526914814815, 0.00065971703704, 
          0.00101677000000, \(-0.00036303148148\), \(-0.00219449666667\), \
\(-0.00047574037037\), \(-0.00019208111111\), \(-0.00044444185185\), 
          0.00261664407407, 
          0.00049623222222, \(-0.00022200925926\), \(-0.00208886444444\), \
\(-0.00130737259259\), \(-0.00071002629630\), \(-0.00066462074074\), 
          0.00057646037037, \(-0.00057751666667\), 
          0.00056463481481, \(-0.00094097407407\), \(-0.00048834740741\), 
          0.00000291777778, 0.00016308296296, \(-0.00028143740741\), 
          0.00009326925926, \(-0.00091410851852\), \(-0.00007459814815\), \
\(-0.00054200000000\), 0.00063818074074, 0.00034150111111, 
          0.00067547740741, \(-0.00047642111111\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ {\(-0.00000407554949\), 0.00000127263632, 
          0.00000341973429, 0.07507025737867, \(-0.19955650914812\), 
          0.01390480536036, 
          0.01657951076676, \(-0.07053699909848\), \(-0.26779214174734\), 
          0.26333131918711, 0.09009234990905, 
          0.13402244274608, \(-0.10007322091237\), 0.13346209596251, 
          0.02049473850201, 0.08379012562328, 
          0.11821810735463, \(-0.06866357769443\), 
          0.24758567480288, \(-0.02661954987771\), \(-0.13486621527270\), 
          0.11370590469900, \(-0.12533732631352\), \(-0.19515927924857\), 
          0.01458723035432, 0.22075598081712, 0.22164060671392, 
          0.06678438749465, 0.03036038357309, 0.08599914864447, 
          0.10829151087157, 0.11807676737470, 
          0.07735221727880, \(-0.04956277104195\), 
          0.06056280641974, \(-0.04705178316792\), 0.22042448000070, 
          0.00506856423750, 0.02213678993220, \(-0.02632999037474\), 
          0.10945131056180, \(-0.10249289845493\), 
          0.03863140699425, \(-0.18955443197197\), 0.18252879094841, 
          0.01625844224468, 0.13666474127205, \(-0.13100040921165\), 
          0.05579122287710, \(-0.01234604257784\), \(-0.17571255153314\), 
          0.03885313499097, 0.03150741092074, 0.11243651108269, 
          0.02511711710076, 0.25657825344557, \(-0.00952558232226\), 
          0.16210917758840, \(-0.02731041393852\), 
          0.16864045498512, \(-0.03706793331623\), 
          0.05589957453102, \(-0.12496980391231\), 0.00089510303786, 
          0.00251214936159, \(-0.04115738530253\), \(-0.04269120422683\), 
          0.05725767926231, \(-0.03929388866886\), \(-0.00006536613112\), \
\(-0.11616560728366\), 0.10161683357152, 
          0.01721573306990, \(-0.07957901838393\), 
          0.06736439133939, \(-0.07423809303929\), 
          0.09678669759553, \(-0.02653506045251\), \(-0.01813096092838\), 
          0.02137394570252};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the mean", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87455 .15565 m
.8707 .14745 L
.86554 .13915 L
.86255 .13535 L
.85908 .13163 L
.85212 .12594 L
.84777 .12333 L
.84274 .12103 L
.84034 .12016 L
.83772 .11936 L
.83541 .11877 L
.83288 .11825 L
.8304 .11784 L
.82809 .11755 L
.82697 .11744 L
.82574 .11734 L
.82457 .11727 L
.82348 .11722 L
.82281 .11719 L
.82209 .11717 L
.82078 .11716 L
.8195 .11716 L
.81831 .11719 L
.81703 .11723 L
.81629 .11727 L
.81561 .11731 L
.81422 .1174 L
.81275 .11753 L
.81009 .11783 L
.80763 .11818 L
.80213 .11925 L
.79749 .12046 L
.79255 .12213 L
.78838 .12389 L
.78415 .1261 L
.77735 .13093 L
.77411 .13405 L
.77145 .13726 L
.7689 .14108 L
.76661 .14551 L
.76463 .15051 L
.7631 .1555 L
.76086 .16535 L
.75916 .17427 L
.75713 .18376 L
.75569 .18903 L
.75413 .19369 L
.75022 .20228 L
.74781 .20628 L
.74483 .21041 L
Mistroke
.73855 .21733 L
.72207 .23023 L
.70365 .24134 L
.68663 .25 L
.66821 .25744 L
.64984 .26301 L
.61099 .27174 L
.59091 .27528 L
.56941 .2782 L
.54883 .28032 L
.52953 .28193 L
.52429 .2823 L
.51861 .28265 L
.51364 .2829 L
.51092 .28302 L
.50838 .28311 L
.50592 .28319 L
.50367 .28324 L
.50117 .28329 L
.49986 .28331 L
.49914 .28332 L
.49848 .28332 L
.49739 .28333 L
.49622 .28333 L
.49514 .28333 L
.49412 .28333 L
.49293 .28333 L
.49183 .28332 L
.49063 .2833 L
.48936 .28328 L
.48676 .28323 L
.48429 .28317 L
.47966 .28301 L
.47467 .28279 L
.46922 .28251 L
.45036 .28128 L
.43075 .27979 L
.42055 .27884 L
.40969 .27753 L
.39945 .2759 L
.39006 .274 L
.37157 .26921 L
.35185 .26308 L
.33423 .25697 L
.31695 .24989 L
.3011 .24147 L
.29307 .23613 L
.28506 .2297 L
.27892 .22343 L
.27651 .22031 L
Mistroke
.27468 .21742 L
.27314 .21424 L
.27252 .21248 L
.27229 .21167 L
.27208 .21081 L
.27193 .21001 L
.27182 .20928 L
.27173 .20854 L
.27169 .20785 L
.27166 .20705 L
.27168 .20631 L
.27173 .2055 L
.27182 .20464 L
.27197 .20377 L
.27213 .20298 L
.27264 .2012 L
.27326 .19956 L
.27399 .198 L
.27608 .19454 L
.27897 .19078 L
.29178 .17773 L
.29844 .17073 L
.30136 .16708 L
.30378 .16329 L
.30458 .16169 L
.30526 .16 L
.30557 .15904 L
.30579 .15819 L
.30597 .15733 L
.30608 .15653 L
.30615 .15582 L
.30617 .15507 L
.30615 .15443 L
.30609 .15375 L
.30599 .15302 L
.30591 .15262 L
.30582 .15225 L
.30563 .15159 L
.30538 .1509 L
.30478 .14961 L
.30407 .14845 L
.30306 .14716 L
.30194 .14601 L
.29929 .14393 L
.29642 .14226 L
.29293 .14072 L
.28867 .13931 L
.28383 .13815 L
.2814 .13769 L
.27905 .13732 L
Mistroke
.27684 .13703 L
.27439 .13676 L
.27234 .13658 L
.27006 .13642 L
.26757 .13629 L
.2662 .13623 L
.26492 .13619 L
.26372 .13617 L
.26262 .13615 L
.26135 .13614 L
.26015 .13614 L
.25885 .13614 L
.25812 .13615 L
.25745 .13616 L
.25622 .13618 L
.25492 .13621 L
.25266 .13628 L
.25019 .13639 L
.24475 .13669 L
.23971 .13706 L
.23073 .1379 L
.22119 .13901 L
.21102 .14047 L
.19995 .14245 L
.18952 .14483 L
.17998 .14759 L
.17074 .15098 L
.16244 .15481 L
.15396 .15983 L
.14581 .16624 L
.13962 .17286 L
.13666 .17697 L
.13428 .18101 L
.1322 .1855 L
.1307 .18974 L
.13004 .19214 L
.12946 .19479 L
.12904 .19733 L
.12888 .19854 L
.12876 .19969 L
.12867 .20078 L
.1286 .20198 L
.12855 .20307 L
.12853 .20408 L
.12853 .20529 L
.12855 .20639 L
.12857 .20699 L
.1286 .20765 L
.12868 .20885 L
.12876 .20989 L
Mistroke
.12888 .21104 L
.12917 .21335 L
.12992 .21748 L
.13115 .22234 L
.13285 .22747 L
.13678 .23644 L
.14176 .24514 L
.14802 .25397 L
.16113 .26828 L
.17594 .28023 L
.19363 .29054 L
.21139 .29806 L
.22851 .30394 L
.26465 .31613 L
.3034 .32963 L
.34098 .34031 L
.3603 .34478 L
.37829 .34813 L
.38846 .3496 L
.3981 .35071 L
.4083 .35161 L
.41935 .35232 L
.42896 .35275 L
.43443 .35294 L
.43956 .35308 L
.44498 .3532 L
.44995 .35327 L
.4527 .3533 L
.45426 .35332 L
.45571 .35333 L
.45697 .35333 L
.45769 .35334 L
.45837 .35334 L
.45973 .35334 L
.4605 .35334 L
.4612 .35334 L
.46252 .35334 L
.46378 .35333 L
.4649 .35332 L
.46612 .35331 L
.46886 .35328 L
.4714 .35325 L
.47604 .35315 L
.48096 .35301 L
.48632 .35281 L
.49134 .35258 L
.50275 .35187 L
.51363 .35098 L
.52389 .34993 L
.54293 .34743 L
Mistroke
.56298 .34376 L
.58128 .33923 L
.61906 .3268 L
.65912 .31402 L
.6993 .30468 L
.73808 .29804 L
.75761 .29474 L
.77886 .29047 L
.79892 .28547 L
.81659 .27977 L
.83388 .27187 L
.84273 .26629 L
.85051 .26009 L
.85725 .2534 L
.86362 .24555 L
.87276 .23015 L
.87621 .22198 L
.87769 .21765 L
.879 .21307 L
.87994 .20903 L
.88074 .20458 L
.88102 .20261 L
.88126 .20051 L
.88145 .19851 L
.88158 .1967 L
.88163 .19561 L
.88167 .19459 L
.8817 .19347 L
.88172 .19226 L
.88172 .19166 L
.88172 .191 L
.8817 .18979 L
.88167 .18861 L
.88163 .18752 L
.88157 .1864 L
.8815 .18534 L
.88134 .18335 L
.88111 .1812 L
.88081 .17887 L
.87996 .17384 L
.87876 .16849 L
.87582 .15895 L
.87455 .15565 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`19o`@02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`030?oo00oo00<0ool0
;ol007go0P0;o`801Ol00`3oo`16o`804ol00`3oo`0^o`00TOl00`3oo`15o`030?oo01Co00<0ool0
;Ol004Ko4P0io`030?oo04Go00<0ool05Ol00`3oo`0/o`00@?l601;o0`0fo`030?oo04Co00<0ool0
5ol00`3oo`0[o`00?Ol301_o0P0do`030?oo04?o00<0ool06?l00`3oo`0[o`00>ol201oo0P0co`03
0?oo04?o00<0ool06Ol00`3oo`0Zo`00>Ol202;o00<0ool0<Ol00`3oo`13o`030?oo01Wo00<0ool0
:_l003Ko0`0To`030?oo037o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000do`809ol00`3oo`0ao`03
0?oo04;o00<0ool06ol00`3oo`0Yo`00<ol00`3oo`0Vo`80<ol00`3oo`12o`030?oo01_o00<0ool0
:Ol003;o00<0ool09ol00`3oo`0bo`030?oo04;o00<0ool07?l00`3oo`0Xo`00<Ol00`3oo`0Wo`03
0?oo03?o00<0ool0@Ol00`3oo`0Mo`030?oo02So000`o`030?oo02Oo00<0ool08?l200Co0P03o`@0
1Ol00`3oo`11o`030?oo01ko00<0ool09ol0033o00<0ool09_l00`3oo`0Po`040?oo00So00<0ool0
1_l00`3oo`11o`030?oo01ko00<0ool09ol002oo00<0ool09_l00`3oo`0Qo`040?oo00Wo00<0ool0
1Ol204;o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`030?oo01co1002o`040?oo00[o00<0ool0
1?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0So`040?oo00So00@0ool0
1Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`802_l200Ko00<0ool0
@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`0oo`030?oo023o00<0ool0
9ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;_l00`3oo`0So`03
0?oo03[o00<0ool0?_l00`3oo`0Qo`030?oo02Oo0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02_l00`3oo`0Eo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02Ol00`3oo`0Ho`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05ol00`3oo`07o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00Wo00<0ool0
6?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Oo
00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3oo`09o`03
0?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool0
1_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`09o`03
0?oo01So00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Jo`030?oo00Oo0P0:o`<0
8_l00`3oo`0Ro`802_l300So00<0ool06?l200_o00<0ool05ol200[o0`02o`00<?l00`3oo`0To`80
>?l00`3oo`0ho`030?oo02Go00<0ool0:Ol0033o00<0ool09_l00`3oo`0eo`030?oo03Ko0P0Wo`03
0?oo02[o000`o`030?oo02Oo0P0eo`030?oo03Co0P0Yo`030?oo02[o000ao`030?oo02So0P0co`03
0?oo03;o0P0Zo`030?oo02_o000ao`030?oo02[o0P0ao`030?oo033o0P0[o`030?oo02co000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01ol000co00<0ool09Ol01@3oool002_o0`0Zo`030?oo02Oo100Zo`030?oo00?o
00<0ool09?l00`3oo`07o`00=Ol00`3oo`0_o`@09_l00`3oo`0So`@0<_l2037o000eo`030?oo03?o
0P0To`030?oo023o0`0eo`030?oo037o000fo`030?oo03Co100Po`030?oo01co100fo`80=?l003Oo
00<0ool0=ol401co00<0ool06?l403So0P0fo`00>?l203_o1@0Go`030?oo00go2`0jo`80>?l003[o
00<0ool0?Ol<00_o00<0ool01?l904;o0`0jo`00>ol204Wo4P17o`@0?Ol003go0P1Bo`030?oo04Wo
0P11o`00?ol2053o00<0ool0A?l504?o0011o`80C_l00`3oo`0oo`D0B?l004?o0`1;o`030?oo03[o
1@1=o`00A_l404Oo00<0ool0=Ol505;o001:o`80AOl00`3oo`0`o`D0Eol004co0`0^o`801?l200?o
1005o`030?oo02[o1P1Lo`00Col202_o00@0ool02?l00`3oo`06o`030?oo02Ko101Ro`00DOl302So
00@0ool02Ol00`3oo`05o`808ol406Ko001Do`<09Ol0103oo`0:o`030?oo00Co00<0ool07_l406[o
001Go`@08Ol0103oo`08o`040?oo00Go00<0ool06ol306ko001Ko`<07ol200[o0P06o`030?oo01Wo
0P1ao`00G_l3033o00<0ool05_l307?o001Qo`@0;?l00`3oo`0Do`80M_l006Go100Xo`030?oo013o
101ho`00JOl502?o00<0ool02ol507co001^o`H07Ol00`3oo`06o`D0POl007Co9P26o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co
0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co
001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko
00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go
001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o
0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86893 .14149 m
.85657 .11439 L
.85267 .1039 L
.8517 .10166 L
.8508 .09993 L
.85 .0987 L
.84955 .09816 L
.84913 .09774 L
.84869 .09741 L
.84846 .09727 L
.84821 .09716 L
.84794 .09707 L
.84769 .09701 L
.84746 .09699 L
.8472 .097 L
.84673 .09708 L
.84647 .09716 L
.84622 .09727 L
.84574 .09754 L
.84527 .09788 L
.84425 .09882 L
.84327 .09998 L
.84088 .1035 L
.83442 .11495 L
.82701 .12747 L
.81952 .13723 L
.81478 .14169 L
.81008 .14491 L
.80724 .14635 L
.8045 .14743 L
.80136 .14832 L
.79972 .14866 L
.79787 .14895 L
.79622 .14913 L
.79468 .14924 L
.79296 .14931 L
.79208 .14932 L
.79111 .14931 L
.78945 .14927 L
.78765 .14917 L
.78592 .14903 L
.78433 .14887 L
.7772 .14787 L
.76937 .14648 L
.76248 .14524 L
.75869 .14463 L
.75512 .14413 L
.75318 .1439 L
.75108 .1437 L
.7492 .14357 L
Mistroke
.74748 .14348 L
.74573 .14344 L
.74417 .14345 L
.74241 .14352 L
.7408 .14364 L
.73996 .14372 L
.73906 .14384 L
.73728 .14413 L
.7357 .14449 L
.73428 .14489 L
.73129 .14606 L
.72975 .14688 L
.72845 .14773 L
.72652 .14936 L
.72471 .15147 L
.72393 .15266 L
.72319 .15404 L
.72264 .1553 L
.72213 .15677 L
.72174 .15822 L
.72145 .15963 L
.72122 .16118 L
.72108 .16291 L
.72102 .16459 L
.72104 .16622 L
.72112 .16774 L
.72127 .16942 L
.72153 .17148 L
.72186 .17345 L
.72288 .17812 L
.72421 .18284 L
.72598 .18807 L
.73394 .20648 L
.74263 .22376 L
.74628 .23143 L
.74777 .23507 L
.74834 .2367 L
.74876 .23809 L
.74904 .23921 L
.74926 .24036 L
.74938 .24143 L
.7494 .24246 L
.74936 .24297 L
.74929 .24342 L
.74918 .2439 L
.74903 .24433 L
.74868 .24502 L
.74844 .24536 L
.74818 .24566 L
.74765 .24612 L
Mistroke
.74706 .24648 L
.74632 .2468 L
.74541 .24706 L
.74491 .24716 L
.74433 .24724 L
.74319 .24733 L
.74196 .24733 L
.74073 .24728 L
.73924 .24714 L
.73845 .24704 L
.73755 .24691 L
.73422 .24631 L
.72544 .24416 L
.70626 .23849 L
.68624 .23269 L
.67667 .23031 L
.6683 .22863 L
.664 .22798 L
.66189 .22772 L
.65968 .2275 L
.65754 .22735 L
.6556 .22728 L
.65451 .22726 L
.65338 .22726 L
.65244 .22729 L
.65144 .22733 L
.64973 .22746 L
.64823 .22763 L
.64658 .22789 L
.64509 .2282 L
.6426 .22892 L
.64014 .22991 L
.63774 .23124 L
.63539 .233 L
.63323 .23515 L
.63144 .23742 L
.62869 .24205 L
.62614 .24767 L
.62127 .25977 L
.61824 .2663 L
.61439 .27302 L
.60995 .27906 L
.60532 .28403 L
.5949 .29205 L
.58917 .29519 L
.58303 .29783 L
.57785 .29957 L
.57238 .30094 L
.56943 .3015 L
.56685 .30187 L
Mistroke
.56548 .30203 L
.56402 .30216 L
.56337 .30221 L
.56269 .30225 L
.56149 .3023 L
.56043 .30232 L
.55931 .30231 L
.55829 .30229 L
.55738 .30224 L
.55635 .30217 L
.55526 .30206 L
.55427 .30193 L
.55337 .30179 L
.55154 .30143 L
.54995 .30102 L
.54663 .29984 L
.54389 .29854 L
.54106 .29692 L
.5294 .28933 L
.52197 .2849 L
.51306 .28032 L
.5044 .2767 L
.50065 .27544 L
.49661 .27432 L
.49459 .27387 L
.49245 .27348 L
.49129 .27331 L
.49023 .27318 L
.48924 .27307 L
.48819 .27298 L
.48763 .27295 L
.48712 .27292 L
.48614 .27288 L
.48559 .27287 L
.485 .27286 L
.48444 .27286 L
.48392 .27287 L
.4829 .27289 L
.48231 .27292 L
.48177 .27294 L
.48066 .27301 L
.47948 .27311 L
.47091 .27412 L
.46632 .27465 L
.46419 .27485 L
.4618 .27502 L
.46044 .2751 L
.45915 .27515 L
.4579 .27519 L
.45655 .27521 L
Mistroke
.45509 .27522 L
.4537 .2752 L
.45214 .27516 L
.45125 .27513 L
.45043 .2751 L
.44868 .275 L
.44704 .27488 L
.44319 .27452 L
.43968 .27409 L
.43585 .27352 L
.42204 .27074 L
.40683 .26673 L
.39443 .2632 L
.38846 .26161 L
.3835 .2605 L
.38112 .26007 L
.37996 .2599 L
.37875 .25975 L
.37807 .25968 L
.37746 .25963 L
.3769 .25959 L
.3763 .25956 L
.37527 .25953 L
.37473 .25953 L
.37422 .25953 L
.37328 .25957 L
.37241 .25963 L
.37165 .25971 L
.37083 .25981 L
.36906 .26013 L
.36747 .26052 L
.36175 .26233 L
.35849 .26332 L
.35657 .26379 L
.35467 .26416 L
.35361 .26432 L
.35243 .26446 L
.35174 .26453 L
.35108 .26458 L
.35047 .26461 L
.3498 .26464 L
.34857 .26465 L
.34793 .26464 L
.34722 .26462 L
.34593 .26455 L
.34469 .26444 L
.34329 .26428 L
.34193 .26408 L
.33927 .26358 L
.33628 .26284 L
Mistroke
.33288 .2618 L
.32581 .25904 L
.31888 .2556 L
.30592 .24739 L
.29336 .23718 L
.28315 .22669 L
.27786 .21971 L
.277 .21828 L
.27634 .21701 L
.27611 .21648 L
.27595 .21604 L
.27583 .21566 L
.27577 .21535 L
.27573 .21507 L
.27573 .21494 L
.27574 .21481 L
.27576 .21469 L
.27579 .21459 L
.27587 .21442 L
.27598 .21431 L
.2761 .21423 L
.27625 .21418 L
.27644 .21414 L
.27655 .21413 L
.27666 .21413 L
.27679 .21414 L
.27693 .21415 L
.27706 .21417 L
.27718 .21418 L
.27746 .21423 L
.27772 .21427 L
.278 .21431 L
.27827 .21435 L
.27851 .21436 L
.27875 .21436 L
.27889 .21435 L
.27901 .21434 L
.27923 .21429 L
.27946 .21421 L
.27957 .21415 L
.27969 .21408 L
.27989 .21391 L
.28011 .21365 L
.2803 .21335 L
.28046 .21301 L
.28059 .21265 L
.28072 .21218 L
.28084 .2116 L
.28092 .21103 L
.28099 .21044 L
Mistroke
.28105 .20972 L
.2811 .20886 L
.28113 .20797 L
.28115 .20707 L
.28116 .206 L
.28116 .20537 L
.28115 .20477 L
.28113 .20361 L
.28111 .20228 L
.28104 .19963 L
.28086 .19393 L
.28076 .19039 L
.28066 .1868 L
.28062 .18472 L
.28058 .18239 L
.28055 .18022 L
.28053 .17815 L
.28052 .17618 L
.28052 .17397 L
.28053 .17161 L
.28055 .16937 L
.28059 .16684 L
.28065 .16415 L
.28072 .16183 L
.2808 .15928 L
.28106 .15356 L
.28137 .14827 L
.28224 .13779 L
.28325 .12896 L
.28456 .12004 L
.28582 .1132 L
.2876 .10534 L
.28783 .10439 L
.28792 .10403 L
.28797 .10378 L
.28798 .10369 L
.28799 .10362 L
.28798 .10358 L
.28796 .10358 L
.2879 .10364 L
.28785 .10372 L
.28779 .10381 L
.28149 .11405 L
.27462 .12239 L
.26703 .1293 L
.26287 .13225 L
.25835 .13486 L
.25379 .13691 L
.24978 .1383 L
.2232 .14935 L
Mistroke
.20306 .1618 L
.1962 .16515 L
.18849 .16829 L
.17418 .17251 L
.16105 .17484 L
.15478 .17564 L
.14886 .17644 L
.14659 .17683 L
.14433 .1773 L
.14206 .17793 L
.14016 .17865 L
.13848 .1795 L
.13707 .18045 L
.13596 .18144 L
.1349 .18268 L
.13397 .18413 L
.13324 .18567 L
.1326 .18756 L
.1323 .18871 L
.13208 .18981 L
.13188 .19103 L
.13174 .1922 L
.13162 .19355 L
.13157 .19433 L
.13154 .19505 L
.1315 .19641 L
.13149 .19793 L
.13151 .19942 L
.13156 .20082 L
.13163 .20214 L
.13172 .20359 L
.13195 .20627 L
.13269 .21264 L
.13376 .21961 L
.13496 .22597 L
.13645 .23269 L
.13819 .23932 L
.1398 .24443 L
.1416 .24915 L
.1436 .25318 L
.14552 .25593 L
.14671 .25713 L
.14729 .25759 L
.14784 .25795 L
.14837 .25823 L
.14886 .25843 L
.14939 .2586 L
.14969 .25867 L
.14997 .25871 L
.15044 .25876 L
Mistroke
.15095 .25877 L
.15143 .25874 L
.15187 .25868 L
.15238 .25857 L
.15293 .25842 L
.15393 .25807 L
.15593 .25714 L
.15803 .25609 L
.15889 .2557 L
.15981 .25535 L
.16034 .25519 L
.16083 .25508 L
.16129 .255 L
.16178 .25494 L
.1622 .25493 L
.16264 .25494 L
.16306 .25498 L
.16345 .25505 L
.16391 .25516 L
.16441 .25533 L
.16532 .25574 L
.1663 .25634 L
.16739 .25718 L
.16945 .2592 L
.17484 .26595 L
.18202 .27518 L
.19046 .28414 L
.20072 .29215 L
.20681 .2957 L
.21372 .29891 L
.22736 .30345 L
.24021 .30651 L
.29234 .32117 L
.3146 .32908 L
.32653 .33285 L
.33679 .33613 L
.34582 .33978 L
.35031 .34227 L
.35405 .34495 L
.35766 .34826 L
.36127 .35244 L
.36732 .36138 L
.37341 .37112 L
.37986 .37947 L
.38352 .38284 L
.38571 .38437 L
.38781 .38552 L
.39006 .38642 L
.39137 .3868 L
.39261 .38705 L
Mistroke
.39329 .38715 L
.39395 .38722 L
.39517 .38728 L
.39587 .38728 L
.39652 .38725 L
.39723 .38719 L
.39798 .38709 L
.39917 .38688 L
.40048 .38656 L
.40287 .38574 L
.40522 .38463 L
.40739 .38336 L
.41227 .37961 L
.41727 .37446 L
.42484 .36395 L
.43167 .35121 L
.43699 .33896 L
.4392 .33372 L
.44016 .33167 L
.44099 .33012 L
.44169 .32904 L
.44244 .32814 L
.44285 .32776 L
.44324 .3275 L
.44361 .32731 L
.44402 .32718 L
.44444 .32713 L
.44491 .32716 L
.44537 .32727 L
.4458 .32743 L
.44633 .32769 L
.44694 .32807 L
.44813 .32898 L
.45444 .33518 L
.46291 .34296 L
.46739 .34634 L
.47258 .34958 L
.47545 .35108 L
.47872 .35252 L
.48175 .35363 L
.48469 .35449 L
.48633 .35487 L
.48813 .35522 L
.48978 .35548 L
.49131 .35565 L
.49296 .35578 L
.49391 .35582 L
.49479 .35584 L
.49657 .35582 L
.49757 .35577 L
Mistroke
.49849 .35571 L
.50005 .35556 L
.50149 .35536 L
.50307 .35509 L
.50475 .35473 L
.50814 .35378 L
.51117 .35268 L
.51769 .34948 L
.52433 .34504 L
.53513 .33534 L
.54386 .32601 L
.55249 .31739 L
.555 .31533 L
.5574 .31364 L
.55978 .31224 L
.56202 .31117 L
.56419 .31036 L
.56532 .31003 L
.56656 .30973 L
.56795 .30947 L
.56859 .30937 L
.56927 .30929 L
.5699 .30923 L
.57059 .30918 L
.57181 .30913 L
.5731 .30913 L
.57384 .30915 L
.57452 .30919 L
.5758 .30929 L
.57717 .30945 L
.5796 .30983 L
.58235 .31041 L
.58873 .31221 L
.59505 .31442 L
.62085 .32452 L
.6358 .32948 L
.64326 .33143 L
.65018 .33287 L
.65639 .33382 L
.65984 .33421 L
.66295 .33446 L
.66472 .33457 L
.66634 .33464 L
.66811 .33469 L
.669 .3347 L
.66997 .33471 L
.6716 .33469 L
.67333 .33465 L
.67495 .33458 L
.67642 .33449 L
Mistroke
.67801 .33437 L
.67949 .33423 L
.6828 .33382 L
.68596 .3333 L
.68926 .33261 L
.69496 .33103 L
.69981 .3292 L
.7047 .32679 L
.70945 .3237 L
.71369 .32002 L
.717 .31623 L
.71967 .31223 L
.722 .30767 L
.72521 .29924 L
.72653 .29544 L
.72804 .29161 L
.7297 .28835 L
.73157 .28577 L
.73275 .28456 L
.7342 .2834 L
.73574 .28247 L
.73733 .28177 L
.73838 .28142 L
.73943 .28114 L
.74042 .28094 L
.74154 .28077 L
.74215 .2807 L
.74283 .28064 L
.74356 .2806 L
.74426 .28058 L
.74556 .28058 L
.74631 .28061 L
.74701 .28064 L
.74849 .28076 L
.74926 .28085 L
.75013 .28096 L
.75324 .28147 L
.75713 .28232 L
.76102 .28336 L
.77735 .28907 L
.81018 .30259 L
.81899 .30576 L
.8237 .30721 L
.82803 .30835 L
.83197 .30917 L
.83415 .30952 L
.83612 .30977 L
.83777 .30992 L
.83955 .31003 L
.84138 .31007 L
Mistroke
.84308 .31003 L
.84488 .30992 L
.84588 .30982 L
.84681 .3097 L
.84849 .30942 L
.85025 .30905 L
.85323 .30818 L
.85645 .30691 L
.85979 .3052 L
.86331 .30292 L
.86951 .29772 L
.87595 .29075 L
.88262 .28203 L
.89444 .26391 L
.91578 .22655 L
.92262 .21284 L
.92518 .2067 L
.92622 .20368 L
.92695 .20107 L
.92723 .19981 L
.92745 .19852 L
.92754 .19781 L
.9276 .19716 L
.92765 .19597 L
.92763 .19493 L
.92755 .19402 L
.92739 .19305 L
.92713 .19207 L
.92682 .19118 L
.9264 .19026 L
.92542 .1887 L
.9242 .18729 L
.92286 .18607 L
.91916 .18348 L
.91411 .18068 L
.90173 .17403 L
.8928 .16815 L
.88452 .16119 L
.87577 .15152 L
.86893 .14149 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{42, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Ho`80<_l0097o00<0ool0
F?l3037o001Ho`030?oo03Ko00<0ool0Eol0103oo`0ao`00Eol203So00<0ool0E_l01@3oool0037o
001Go`809?l200Co0P05o`<01?l00`3oo`1Fo`030?oo00;o00<0ool0;_l005Ko00<0o`008ol0103o
o`0;o`030?oo00?o00<0ool0EOl00`3oo`03o`030?oo02ko001Eo`030?l002Co00@0ool02?l500Co
0P1Eo`030?oo00Go00<0ool0;Ol005Go00<0o`007_l400;o00@0ool02?l0103oo`05o`030?oo05Co
00<0ool01Ol00`3oo`0]o`00E?l0103oo`0To`040?oo00Wo00<0o`001Ol00`3oo`1Co`030?oo00Oo
00<0ool0;?l005?o00D0oooo000Uo`802ol200Go00<0ool0D_l00`3oo`08o`030?oo02co001@o`<0
1?l00`3oo`0go`030?oo05;o00<0ool02Ol00`3oo`0[o`00COl300Oo00<0ool0=ol00`3oo`1Ao`03
0?oo00[o00<0ool0:ol004_o0P0:o`030?oo03Oo00<0ool0D?l00`3oo`0<o`030?oo02[o0019o`80
2ol00`3oo`0ho`030?oo03[o3008o`803ol00`3oo`0Zo`00Aol200go00<0ool0>?l00`3oo`0io`80
2ol801;o00<0ool0:Ol004Ko00<0ool03Ol00`3oo`0ho`030?oo03So00<0ool09Ol00`3oo`0Yo`00
A?l2013o00<0ool0>?l00`3oo`0ho`030?oo02Ko0P0Yo`00@ol00`3oo`0@o`030?oo03So00<0ool0
=ol00`3oo`0Yo`030?oo02Ko0010o`<04ol00`3oo`0ho`030?oo03Oo00<0ool0:_l00`3oo`0Uo`00
?Ol301Ko00<0ool0>?l00`3oo`0go`030?oo02_o00<0ool09?l003Wo100Io`030?oo03So00<0ool0
=ol00`3oo`0/o`030?oo02?o000co`H07Ol00`3oo`0To`801?l200?o1005o`030?oo03So00<0ool0
;?l202?o000`o`<08ol00`3oo`0So`040?oo00So00<0ool01_l00`3oo`0ho`030?oo02ko0P0Qo`00
;ol202Go00<0ool08ol0103oo`09o`030?oo00Go0P0io`030?oo033o0P0Oo`00;ol00`3oo`0To`03
0?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`0io`030?oo037o00<0ool07?l002oo00<0ool0
9?l00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`0io`030?oo03;o00<0ool06ol002ko0P0Vo`03
0?oo02Co0P0:o`801_l00`3oo`0jo`030?oo037o00<0ool06ol002ko00<0ool09Ol00`3oo`0ho`03
0?oo03[o00<0ool0<Ol00`3oo`0Ko`00;_l202Ko00<0ool0>?l00`3oo`0ko`030?oo033o00<0ool0
6ol002oo00<0ool09?l00`3oo`0ho`030?oo03_o00<0ool0<?l00`3oo`0Ko`001ol500;o0P04o`80
6Ol00`3oo`07o`D06?l00`3oo`07o`801?l200Co0P0So`030?oo02;o0P04o`801?l200_o00<0ool0
5Ol501Ko00<0ool02Ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
5ol200So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool0
5Ol00`3oo`0<o`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool0
5ol200So00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`030?oo01Ko00<0ool0
5Ol00`3oo`0<o`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Wo00@0oooo1004o`03
0?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200go00<0ool05Ol00`3o
o`0Do`030?oo00go00<0ool02?l200?o0009o`030?oo00So00<0ool06?l00`3oo`09o`030?oo01So
00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00co00<0ool05Ol00`3o
o`0Do`030?oo00go00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Ko`030?oo00Ko
0P0:o`<08_l00`3oo`0Ro`T00ol300go00<0ool04ol201Go00<0ool03Ol200[o0`02o`00<?l00`3o
o`0Uo`030?oo03Ko00<0ool08Ol200So1@0>o`030?oo02Wo00<0ool07ol0033o00<0ool09_l00`3o
o`0eo`030?oo023o00<0ool03Ol200co0P0Zo`030?oo01oo000`o`030?oo02Oo00<0ool0=?l00`3o
o`0Oo`030?oo013o1009o`030?oo02Oo00<0ool08?l0037o00<0ool09ol203Co00<0ool07ol00`3o
o`0Do`@01Ol00`3oo`0Wo`030?oo023o000ao`030?oo02Wo00<0ool0<Ol00`3oo`0No`030?oo01Wo
1P0Xo`030?oo027o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`04
0?oo00?o0`08o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00;o0P02o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`03o`04
0?oo00So00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0So`P0:?l302co00<0ool0
7Ol00`3oo`09o`030?oo02Wo00<0ool03_l00`3oo`0Io`030?oo00Oo000jo`030?oo02Oo0P06o`P0
7Ol00`3oo`0Mo`030?oo04Ko00<0ool08ol003[o00<0ool0:Ol600So1P0Go`030?oo01co00<0ool0
A_l00`3oo`0To`00>ol00`3oo`0lo`805Ol00`3oo`0Ko`030?oo04Ko00<0ool09Ol003_o00<0ool0
?_l500So2@0Mo`030?oo04Ko00<0ool09Ol003co00<0ool0@_l800So0`0Jo`030?oo04Ko00<0ool0
9_l003go00<0ool0DOl00`3oo`0201So00<0ool07ol502;o00<0ool09_l003ko00<0ool0D?l00`3o
o`02o`805Ol00`3oo`0No`801Ol301ko00<0ool09ol003oo0P1@o`030?oo00Co00<0ool04?l2027o
00<0ool01ol401[o00<0ool09ol0047o00<0ool0COl00`3oo`05o`803ol00`3oo`0Po`030?oo00co
0P0Go`030?oo02So0012o`80COl00`3oo`07o`030?oo00[o0P0So`030?oo00ko0P0Do`030?oo02Wo
0014o`80Bol00`3oo`08o`801ol302Co00<0ool04Ol2013o0P0/o`00A_l404Oo00<0ool02_l702Oo
00<0ool04ol200go00<0ool0;?l004[o0`14o`030?oo03So00<0ool05Ol400Ko0`0_o`00COl402co
0P04o`800ol400Go00<0ool03Ol602Go00<0ool06Ol7037o001Ao`<0:?l0103oo`08o`030?oo00Ko
00<0ool02ol200Ko1@0Oo`030?oo05;o001Do`@09?l0103oo`09o`030?oo00Go0P0;o`030?oo00_o
0P0Lo`030?oo05?o001Ho`<08Ol0103oo`0:o`030?oo00Co00<0ool02Ol00`3oo`0>o`806Ol00`3o
o`1Do`00Fol301ko00@0ool02?l0103oo`05o`030?oo00So00<0ool04Ol301Co0P1Go`00G_l301co
0P02o`<01Ol200Ko00<0ool01ol00`3oo`0Eo`<03ol205Wo001Qo`<07?l200;o00<0ool02_l00`3o
o`07o`030?oo01So1`02o`H0Fol006Co0P0Jo`050?ooo`003?l00`3oo`06o`030?oo023o0`1Po`00
I_l201So00<0ool00_l200[o00<0ool01?l208Ko001Xo`<05?l00`3oo`05o`030?oo00Oo00<0ool0
0_l208So001[o`030?oo017o00<0ool01_l200Oo00<0ool00P2:o`00K?l00`3oo`0?o`030?oo00Wo
0P05o`030?l008co001/o`030?oo00oo00<0ool02ol808co001]o`030?oo00go00<0ool04Ol00`3o
o`2<o`00KOl00`3oo`0=o`030?oo017o00<0ool0S?l006ko00<0ool02ol00`3oo`0Bo`030?oo08co
001_o`030?oo00Wo00<0ool04ol00`3oo`2<o`00L?l00`3oo`07o`030?oo00801?l200Go0`04o`03
0?oo08co001ao`030?oo00Go00@0ool00_l00`3oo`09o`030?oo00?o00<0ool0S?l007;o00<0ool0
0ol01@3oool000;o00<0ool01_l500Co0P2=o`00L_l600Co00@0ool02?l0103oo`05o`030?oo08co
001eo`030?oo00Co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co
001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o
003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86963 .14326 m
.85769 .11749 L
.85353 .10742 L
.85244 .10524 L
.85142 .10352 L
.85051 .10228 L
.8495 .10127 L
.849 .1009 L
.84844 .10059 L
.84815 .10047 L
.84784 .10038 L
.84755 .10032 L
.84727 .10029 L
.84675 .10029 L
.84644 .10033 L
.84616 .10039 L
.84563 .10055 L
.84505 .10081 L
.84446 .10115 L
.84381 .10161 L
.84257 .10267 L
.84011 .10539 L
.83367 .11435 L
.82523 .12623 L
.82112 .13119 L
.8164 .13599 L
.81159 .13982 L
.8069 .14261 L
.80401 .1439 L
.80124 .14488 L
.79808 .1457 L
.79644 .14602 L
.7946 .14631 L
.79297 .14649 L
.79145 .14662 L
.79065 .14667 L
.78976 .14671 L
.78883 .14674 L
.78795 .14676 L
.78631 .14676 L
.7848 .14672 L
.78304 .14665 L
.78136 .14656 L
.77808 .1463 L
.77497 .146 L
.76778 .14522 L
.76416 .14485 L
.76075 .14454 L
.75887 .1444 L
.75789 .14434 L
Mistroke
.75685 .14429 L
.75503 .14422 L
.75336 .14419 L
.75167 .1442 L
.75015 .14424 L
.7485 .14433 L
.74769 .1444 L
.7468 .14449 L
.74534 .14467 L
.7438 .14492 L
.74115 .14552 L
.73988 .14589 L
.73855 .14636 L
.73627 .14737 L
.73423 .14857 L
.73252 .14986 L
.73068 .15168 L
.72923 .15358 L
.72791 .15593 L
.72736 .15722 L
.72684 .15869 L
.72647 .16005 L
.72617 .16138 L
.72591 .16297 L
.72574 .16448 L
.72565 .16595 L
.72561 .16734 L
.72562 .16891 L
.7257 .17061 L
.72586 .17253 L
.72608 .17433 L
.72678 .17851 L
.72768 .18252 L
.72874 .18642 L
.73551 .20506 L
.74217 .22056 L
.74509 .22789 L
.74616 .23107 L
.7469 .23373 L
.74714 .23483 L
.74731 .23581 L
.74744 .23678 L
.74751 .23779 L
.74751 .23874 L
.74744 .23955 L
.74728 .24042 L
.74705 .24118 L
.74673 .24188 L
.74629 .24255 L
.74584 .24306 L
Mistroke
.74526 .24356 L
.74455 .24402 L
.74366 .24443 L
.74272 .24474 L
.74177 .24495 L
.74061 .24512 L
.73943 .24521 L
.73829 .24523 L
.73697 .24521 L
.73544 .24511 L
.73464 .24504 L
.73372 .24495 L
.7304 .24449 L
.7231 .24308 L
.71547 .24129 L
.69522 .2362 L
.68507 .23386 L
.67601 .2321 L
.67165 .23143 L
.66772 .23095 L
.66581 .23077 L
.6638 .23062 L
.66274 .23056 L
.66164 .23051 L
.65972 .23047 L
.65784 .23048 L
.65682 .23052 L
.65591 .23056 L
.65419 .2307 L
.65262 .23088 L
.65086 .23115 L
.6493 .23147 L
.64603 .2324 L
.64372 .23333 L
.64174 .23436 L
.63957 .23578 L
.63776 .23726 L
.63449 .24075 L
.63148 .24513 L
.62607 .25539 L
.61972 .26705 L
.61597 .27253 L
.61193 .27737 L
.60309 .28529 L
.59802 .28868 L
.59241 .29172 L
.58746 .29389 L
.58216 .29577 L
.57658 .2973 L
.57162 .29829 L
Mistroke
.56872 .29872 L
.56734 .29887 L
.5661 .29899 L
.56502 .29908 L
.56385 .29915 L
.5627 .29919 L
.56208 .29921 L
.56151 .29922 L
.56042 .29922 L
.55982 .29921 L
.55929 .2992 L
.55827 .29916 L
.55733 .2991 L
.55626 .29902 L
.5553 .29892 L
.55322 .29863 L
.55161 .29833 L
.54991 .29794 L
.54664 .29696 L
.54071 .29449 L
.53492 .29156 L
.52043 .28429 L
.51159 .28048 L
.50195 .27712 L
.49722 .27585 L
.49476 .27531 L
.49215 .27484 L
.48981 .27452 L
.48879 .2744 L
.48769 .2743 L
.48672 .27422 L
.4857 .27416 L
.48462 .27411 L
.48362 .27409 L
.48308 .27408 L
.48251 .27408 L
.4815 .27409 L
.48093 .2741 L
.4804 .27411 L
.4792 .27416 L
.47815 .27422 L
.47719 .27428 L
.475 .27445 L
.46669 .27524 L
.46439 .27544 L
.46216 .2756 L
.46088 .27568 L
.45948 .27575 L
.45818 .2758 L
.45695 .27584 L
Mistroke
.45577 .27586 L
.45466 .27587 L
.45341 .27586 L
.45205 .27585 L
.45059 .27581 L
.44919 .27575 L
.44791 .27569 L
.44649 .2756 L
.44341 .27536 L
.43989 .27499 L
.43249 .27394 L
.42537 .27261 L
.41861 .27111 L
.39414 .26459 L
.38841 .2631 L
.38289 .26187 L
.38044 .26144 L
.37826 .26113 L
.37734 .26103 L
.3764 .26094 L
.37539 .26087 L
.37448 .26083 L
.37352 .26081 L
.37267 .26081 L
.3717 .26083 L
.37081 .26088 L
.37 .26094 L
.36925 .26102 L
.3676 .26123 L
.36417 .26187 L
.36045 .26268 L
.35691 .26336 L
.35589 .26352 L
.35478 .26367 L
.3538 .26378 L
.35272 .26387 L
.35208 .26392 L
.35149 .26395 L
.35032 .26399 L
.34965 .264 L
.34891 .26399 L
.34819 .26398 L
.34753 .26395 L
.34631 .26388 L
.3456 .26383 L
.34493 .26376 L
.34355 .2636 L
.34203 .26338 L
.33917 .26283 L
.33641 .26216 L
Mistroke
.32977 .26 L
.32307 .25713 L
.31683 .25387 L
.29379 .23721 L
.28458 .22809 L
.27908 .22137 L
.27724 .21856 L
.27665 .21748 L
.27627 .21665 L
.27614 .21628 L
.27603 .21593 L
.27599 .21575 L
.27596 .21559 L
.27593 .21532 L
.27593 .21508 L
.27594 .21495 L
.27597 .21485 L
.276 .21475 L
.27604 .21466 L
.27609 .21458 L
.27615 .21449 L
.27626 .21438 L
.2764 .21427 L
.27656 .21418 L
.27671 .21411 L
.2774 .21384 L
.27781 .21365 L
.27821 .2134 L
.2784 .21324 L
.27861 .21303 L
.27883 .21277 L
.27903 .21247 L
.27937 .21182 L
.27953 .2114 L
.2797 .2109 L
.27998 .2098 L
.2802 .20859 L
.28039 .2072 L
.28057 .20546 L
.28071 .20358 L
.28082 .20164 L
.28126 .18944 L
.28156 .18095 L
.28199 .17205 L
.28255 .16329 L
.28337 .15358 L
.28526 .1373 L
.28921 .11417 L
.28955 .11235 L
.28978 .11099 L
Mistroke
.28987 .11039 L
.2899 .11011 L
.28992 .10985 L
.28993 .10947 L
.28991 .10921 L
.28985 .10903 L
.2898 .10897 L
.28975 .10893 L
.28962 .10893 L
.28945 .10901 L
.28926 .10914 L
.28905 .10932 L
.28848 .10993 L
.27818 .12141 L
.26998 .12832 L
.26504 .13159 L
.26043 .13408 L
.25624 .1359 L
.2518 .13743 L
.24757 .13857 L
.24334 .13952 L
.23975 .14033 L
.2361 .14131 L
.23299 .14238 L
.2295 .14386 L
.2134 .15303 L
.2015 .15958 L
.18907 .16508 L
.1827 .16733 L
.17572 .16938 L
.16341 .17211 L
.15192 .17414 L
.14914 .1747 L
.14636 .17535 L
.14397 .17605 L
.14193 .17681 L
.14009 .17768 L
.13829 .1788 L
.13688 .17995 L
.13558 .18135 L
.13447 .18293 L
.1336 .18461 L
.13295 .18627 L
.13238 .18825 L
.13211 .18943 L
.1319 .19059 L
.13172 .19191 L
.13156 .19339 L
.1315 .19416 L
.13144 .19501 L
Mistroke
.1314 .19583 L
.13138 .1966 L
.13135 .1981 L
.13135 .19887 L
.13136 .19973 L
.1314 .20132 L
.13146 .20281 L
.13156 .20447 L
.1317 .20626 L
.13198 .20935 L
.13239 .21279 L
.13348 .22008 L
.13478 .22689 L
.13774 .23863 L
.13962 .24422 L
.14146 .24855 L
.14356 .25229 L
.14465 .25377 L
.14586 .25508 L
.14691 .25596 L
.14746 .25634 L
.14806 .25668 L
.14863 .25694 L
.14914 .25714 L
.14967 .2573 L
.15016 .25741 L
.15064 .25749 L
.15115 .25754 L
.15159 .25756 L
.15206 .25756 L
.15258 .25754 L
.15313 .25749 L
.15365 .25743 L
.15412 .25736 L
.15631 .25694 L
.15747 .25673 L
.1581 .25662 L
.15869 .25654 L
.1592 .25649 L
.15976 .25646 L
.16007 .25645 L
.16037 .25645 L
.16094 .25647 L
.16146 .25652 L
.16194 .25658 L
.16247 .25669 L
.16304 .25684 L
.16418 .25726 L
.16541 .25788 L
.16769 .25949 L
Mistroke
.17029 .26194 L
.17342 .26542 L
.18002 .27332 L
.18741 .28133 L
.19666 .28928 L
.20682 .29564 L
.21289 .29851 L
.21875 .30079 L
.23259 .30486 L
.28239 .31821 L
.32613 .33322 L
.33574 .33636 L
.34479 .34014 L
.34905 .34255 L
.35335 .34565 L
.36015 .35237 L
.36706 .3613 L
.37384 .37012 L
.37704 .37367 L
.38067 .37699 L
.38267 .37848 L
.38496 .37989 L
.38711 .38094 L
.38919 .38171 L
.39034 .38202 L
.39141 .38226 L
.3926 .38245 L
.39327 .38252 L
.39389 .38257 L
.39502 .3826 L
.39626 .38257 L
.39745 .38246 L
.39854 .38231 L
.39961 .3821 L
.40061 .38185 L
.40289 .38112 L
.40512 .38017 L
.40751 .37891 L
.41289 .37511 L
.4178 .37052 L
.42614 .3601 L
.43785 .34023 L
.44021 .33617 L
.44136 .33443 L
.44253 .33293 L
.44354 .33189 L
.44407 .33145 L
.44463 .33106 L
.44516 .33079 L
.44565 .3306 L
Mistroke
.44614 .33048 L
.44662 .33042 L
.44713 .33041 L
.4477 .33048 L
.44832 .33062 L
.44892 .33082 L
.45004 .33133 L
.45132 .33209 L
.45815 .33745 L
.46796 .34494 L
.47311 .34813 L
.47931 .35118 L
.48222 .35231 L
.48544 .35334 L
.48724 .35381 L
.48891 .35419 L
.4905 .35449 L
.49222 .35475 L
.49314 .35486 L
.49397 .35495 L
.49561 .35507 L
.49652 .35511 L
.49752 .35513 L
.49846 .35514 L
.49933 .35512 L
.50107 .35503 L
.50205 .35496 L
.50297 .35486 L
.50482 .35461 L
.5068 .35426 L
.51014 .35345 L
.51372 .35228 L
.52003 .34947 L
.52587 .34599 L
.53098 .34227 L
.54906 .32571 L
.55762 .31858 L
.56197 .31592 L
.56449 .31472 L
.56689 .31382 L
.56823 .31341 L
.56951 .31309 L
.5707 .31284 L
.57197 .31263 L
.57264 .31254 L
.57338 .31246 L
.57406 .3124 L
.57469 .31235 L
.57544 .31232 L
.57626 .31229 L
Mistroke
.57698 .31229 L
.57775 .3123 L
.57903 .31235 L
.58041 .31244 L
.58173 .31257 L
.58295 .31272 L
.586 .31322 L
.58896 .31383 L
.60062 .31708 L
.61421 .32157 L
.62818 .32591 L
.64155 .32919 L
.64831 .33041 L
.65177 .3309 L
.65554 .33133 L
.65766 .33152 L
.65963 .33167 L
.6615 .33177 L
.66246 .33181 L
.66348 .33185 L
.6645 .33188 L
.66543 .3319 L
.66631 .3319 L
.66724 .33191 L
.66886 .33189 L
.67061 .33184 L
.67236 .33177 L
.67397 .33168 L
.67572 .33154 L
.67756 .33137 L
.6806 .331 L
.68389 .33049 L
.68974 .32924 L
.69493 .32772 L
.70056 .32554 L
.70527 .32318 L
.71012 .32004 L
.71412 .3167 L
.71995 .30993 L
.72252 .30588 L
.72455 .30216 L
.72793 .29548 L
.72985 .29212 L
.73186 .28927 L
.73431 .2867 L
.7358 .28553 L
.73757 .28446 L
.73934 .28364 L
.74119 .28301 L
.74298 .28259 L
Mistroke
.74405 .2824 L
.74508 .28227 L
.74623 .28216 L
.74753 .2821 L
.7488 .28208 L
.75 .28211 L
.75138 .28218 L
.75218 .28224 L
.75294 .28231 L
.75625 .28273 L
.75978 .28336 L
.76316 .28412 L
.77128 .28636 L
.78752 .29193 L
.80555 .29856 L
.81287 .30104 L
.82054 .30328 L
.82482 .3043 L
.82863 .30502 L
.83043 .30529 L
.83237 .30553 L
.83418 .30569 L
.83584 .30579 L
.83764 .30583 L
.83928 .3058 L
.84106 .30571 L
.84195 .30563 L
.84291 .30553 L
.84452 .3053 L
.84621 .30497 L
.84922 .30419 L
.85227 .3031 L
.8551 .30181 L
.86024 .29869 L
.86606 .29392 L
.87131 .2885 L
.88292 .27327 L
.9045 .23779 L
.91319 .22165 L
.91621 .21539 L
.91882 .20924 L
.91976 .20662 L
.92055 .20401 L
.92085 .2028 L
.92109 .20171 L
.92125 .20076 L
.92139 .19976 L
.92148 .19872 L
.92152 .19767 L
.9215 .19673 L
Mistroke
.92143 .19589 L
.92129 .19492 L
.9211 .19408 L
.92082 .19314 L
.92049 .19228 L
.91967 .1907 L
.91854 .18912 L
.91588 .18645 L
.91222 .18368 L
.90738 .18052 L
.89841 .17463 L
.88835 .16665 L
.87788 .15543 L
.86963 .14326 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Ho`80
<_l0097o00<0ool0F?l00`3o000ao`00F?l203Oo00<0ool0Eol01@3oool0033o001Ho`808ol200Co
0P05o`<01?l00`3oo`1Fo`030?oo00;o00<0ool0;_l005Oo0P0So`040?oo00_o00<0ool00ol00`3o
o`1Eo`030?oo00Co00<0ool0;Ol005Ko00<0o`008ol0103oo`08o`D01?l205Go00<0ool01Ol00`3o
o`0]o`00EOl0103oo`0Mo`@00_l0103oo`08o`040?oo00Go00<0ool0E?l00`3oo`05o`030?oo02go
001Do`050?ooo`008ol0103oo`09o`030?l000Go00<0ool0Dol00`3oo`07o`030?oo02co001Co`05
0?ooo`009Ol200_o0P05o`030?oo05;o00<0ool02?l00`3oo`0/o`00DOl200Co00<0ool0=ol00`3o
o`1Ao`030?oo00[o00<0ool0:ol004go1006o`030?oo03Oo00<0ool0DOl00`3oo`0:o`030?oo02_o
0019o`@02_l00`3oo`0go`030?oo04ko0`0>o`030?oo02[o0018o`803Ol00`3oo`0go`030?oo03co
3002o`@04Ol00`3oo`0Zo`00A_l200oo00<0ool0=ol00`3oo`0jo`803?l301Go00<0ool0:Ol004Co
0P0Ao`030?oo03Oo00<0ool0>Ol202Go00<0ool0:Ol004;o0P0Co`030?oo03Oo00<0ool0>Ol00`3o
o`0Uo`030?oo02So0010o`805Ol00`3oo`0go`030?oo03Wo00<0ool09_l00`3oo`0Wo`00?_l201Oo
00<0ool0=ol00`3oo`0io`030?oo02Oo00<0ool09_l003co0P0Io`030?oo03Oo00<0ool0>Ol00`3o
o`0Xo`030?oo02Go000fo`H06ol00`3oo`0go`030?oo03Wo00<0ool0:Ol00`3oo`0To`00<_l4027o
00<0ool08ol200Co0P03o`@01Ol00`3oo`0io`030?oo02[o00<0ool08ol0033o0P0To`030?oo02?o
00@0ool02?l00`3oo`06o`030?oo03Wo00<0ool0:ol202?o000_o`809Ol00`3oo`0So`040?oo00Wo
00<0ool01Ol203[o00<0ool0;Ol2027o000_o`030?oo02Co00<0ool07Ol400;o00@0ool02_l00`3o
o`04o`030?oo03Wo00<0ool0;ol00`3oo`0No`00;ol00`3oo`0To`030?oo02?o00@0ool02?l0103o
o`05o`030?oo03Wo00<0ool0<?l00`3oo`0Mo`00;_l202Ko00<0ool09?l200[o0P06o`030?oo03[o
00<0ool0;ol201ko000^o`030?oo02Go00<0ool0>?l00`3oo`0jo`030?oo033o00<0ool07?l002ko
00<0ool09Ol00`3oo`0ho`030?oo03_o00<0ool0;_l201ko000^o`809_l00`3oo`0ho`030?oo03_o
00<0ool0;_l00`3oo`0Mo`001ol500;o0P04o`806Ol00`3oo`07o`D06?l00`3oo`07o`801?l200Co
0P0So`030?oo02;o0P04o`801?l200co00<0ool05?l501Co00<0ool02ol500;o0P04o`800ol000Wo
00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool05ol200So00@0ool02?l0103oo`0Ro`030?oo027o
00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool05?l00`3oo`0=o`030?oo00Oo00@0ool00_l000Wo
00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3o
o`0Qo`040?oo00_o00<0ool02_l00`3oo`0Eo`030?oo01?o00<0ool03_l00`3oo`0:o`400Ol1o`00
00?o00000P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Go`800_l400;o00@0ool02Ol202?o
00<0ool08Ol0103oo`09o`803Ol00`3oo`0Eo`030?oo01;o00<0ool03ol00`3oo`08o`800ol000Wo
00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool06?l00`3oo`06o`040?oo00Wo00<0ool08_l00`3o
o`0Qo`040?oo00Wo00<0ool03Ol00`3oo`0Do`030?oo01;o00<0ool03ol00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<03Ol00`3o
o`0Co`804ol00`3oo`0?o`802_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0So`X04Ol00`3o
o`0Xo`030?oo023o000`o`030?oo02Ko00<0ool0=Ol00`3oo`0Ro`802Ol500co00<0ool09ol00`3o
o`0Qo`00<?l00`3oo`0Wo`030?oo03Co00<0ool08Ol00`3oo`0>o`<02Ol00`3oo`0Wo`030?oo027o
000ao`030?oo02Oo0P0do`030?oo023o00<0ool04_l500Co00<0ool09_l00`3oo`0Ro`00<Ol00`3o
o`0Yo`030?oo037o00<0ool07ol00`3oo`0Ho`@0:Ol00`3oo`0Ro`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool00_l01@3oool000So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko00<0ool01_l00`3o
o`07o`003?l00`3oo`0To`H0:?l20003o`3o02_o00<0ool07Ol00`3oo`09o`030?oo02Wo00<0ool0
3Ol00`3oo`0Jo`030?oo00Oo000ho`80:Ol400Co2@0Mo`030?oo01go00<0ool0A?l00`3oo`0Uo`00
>_l00`3oo`0Yo`D02Ol201_o00<0ool07?l00`3oo`14o`030?oo02Ko000ko`030?oo03So100Go`03
0?oo01co00<0ool0A?l00`3oo`0Vo`00>ol00`3oo`0lo`L02Ol500;o00<0ool06ol00`3oo`14o`03
0?oo02Oo000lo`030?oo04;o2P03o`D06ol00`3oo`14o`030?oo02So000mo`030?oo057o00<0o`00
0P0Go`030?oo04Go00<0ool0:?l003ko0P1Ao`030?oo00;o0P0Do`030?oo023o2@0Lo`030?oo02Wo
0010o`030?oo04ko00<0ool01?l2017o00<0ool08?l200So0`0Ho`030?oo02[o0011o`80C_l00`3o
o`06o`030?oo00co0P0So`030?oo00[o100Do`030?oo02[o0013o`030?oo04_o00<0ool01ol300So
0`0To`030?oo00oo0P0@o`80;Ol004Co00<0ool0B_l00`3oo`0:o`P09ol00`3oo`0Ao`803Ol00`3o
o`0]o`00AOl304Wo00<0ool0>?l00`3oo`0Do`D01Ol3033o0018o`@0AOl00`3oo`0ho`030?oo01Wo
1P0bo`00C?l402go0P04o`800ol400Go00<0ool0=ol00`3oo`1Bo`00D?l502Oo00@0ool02?l00`3o
o`06o`030?oo00ko200Po`030?oo05?o001Eo`@08ol0103oo`09o`030?oo00Go0P0=o`802?l301co
00<0ool0E?l005Wo0`0Po`040?oo00[o00<0ool01?l00`3oo`0;o`030?oo00_o100Fo`80Eol005co
0P0No`040?oo00So00@0ool01Ol00`3oo`0:o`030?oo013o100@o`80FOl005ko0`0Lo`800ol200Go
0P06o`030?oo00Wo00<0ool05Ol400So101Ko`00HOl301go100<o`030?oo00So00<0ool06_l905ko
001To`806ol01@3oool000_o00<0ool01ol00`3oo`22o`00I_l201So00<0ool00ol200Wo00<0ool0
1_l00`3oo`23o`00J?l201Go00<0ool01_l00`3oo`06o`030?oo00Co0P26o`00J_l00`3oo`0Bo`03
0?oo00Oo00<0ool01Ol0103oool308So001[o`030?oo013o00<0ool02Ol400?o00@0ool0Rol006co
00<0ool03ol00`3oo`0<o`L0S?l006co00<0ool03_l00`3oo`0Ao`030?oo08co001]o`030?oo00co
00<0ool04_l00`3oo`2<o`00K_l00`3oo`0;o`030?oo01;o00<0ool0S?l006oo00<0ool02?l201Go
00<0ool0S?l0073o00<0ool01_l0103oool200Co0P05o`<01?l00`3oo`2<o`00LOl00`3oo`04o`05
0?ooo`000_l00`3oo`09o`030?oo00?o00<0ool0S?l007;o1P04o`040?oo00So1@04o`80SOl007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o
0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103o
o`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87034 .14503 m
.85881 .12058 L
.85439 .11094 L
.85318 .10882 L
.85204 .10712 L
.851 .10584 L
.84988 .10479 L
.84931 .10439 L
.84868 .10403 L
.8481 .10378 L
.84755 .10362 L
.84721 .10356 L
.84689 .10352 L
.84627 .1035 L
.84592 .10353 L
.84559 .10357 L
.84523 .10364 L
.84484 .10374 L
.84417 .10397 L
.84344 .1043 L
.84207 .10511 L
.83934 .10728 L
.82345 .12499 L
.8192 .12933 L
.81438 .13359 L
.80964 .13701 L
.80491 .13966 L
.80226 .14084 L
.7993 .14192 L
.79677 .14265 L
.79391 .14331 L
.79067 .14385 L
.78886 .14406 L
.78713 .14423 L
.78551 .14434 L
.78401 .14442 L
.78322 .14445 L
.78235 .14447 L
.78144 .14449 L
.78058 .14451 L
.77897 .14452 L
.77816 .14451 L
.77726 .14451 L
.77566 .14449 L
.77416 .14446 L
.77047 .14437 L
.76886 .14434 L
.76801 .14432 L
.7671 .1443 L
.76546 .14428 L
Mistroke
.76395 .14427 L
.76238 .14428 L
.76098 .14431 L
.75937 .14435 L
.75848 .14439 L
.75766 .14444 L
.75616 .14454 L
.7546 .14469 L
.75197 .14503 L
.75053 .14527 L
.74924 .14554 L
.74682 .14616 L
.74464 .14689 L
.74225 .14794 L
.74019 .14909 L
.73813 .15058 L
.73617 .15242 L
.7346 .15435 L
.7332 .15663 L
.73221 .15877 L
.73135 .16132 L
.73097 .16285 L
.73068 .16429 L
.73055 .16512 L
.73044 .16603 L
.73028 .16772 L
.73019 .16935 L
.73017 .17113 L
.73021 .17269 L
.7303 .17444 L
.73046 .17623 L
.73065 .17796 L
.73115 .18127 L
.73185 .18493 L
.7328 .18899 L
.73709 .20321 L
.74239 .21854 L
.74439 .22508 L
.74503 .22768 L
.74532 .22907 L
.74553 .23031 L
.74568 .23147 L
.74579 .23266 L
.74583 .23386 L
.7458 .23488 L
.74569 .23594 L
.74559 .2365 L
.74547 .237 L
.7452 .23786 L
.74483 .2387 L
Mistroke
.74436 .23948 L
.74375 .24023 L
.74307 .24086 L
.74237 .24136 L
.74144 .24188 L
.7403 .24234 L
.73973 .24252 L
.7391 .24269 L
.73848 .24283 L
.73788 .24294 L
.73667 .24311 L
.736 .24318 L
.73526 .24323 L
.73445 .24326 L
.73366 .24328 L
.73211 .24327 L
.73058 .2432 L
.72911 .2431 L
.72743 .24295 L
.72557 .24275 L
.71748 .24153 L
.70763 .23967 L
.69851 .23788 L
.68883 .23609 L
.68456 .23538 L
.68009 .23473 L
.67632 .23427 L
.67237 .2339 L
.67031 .23376 L
.66846 .23366 L
.66672 .23361 L
.66493 .23359 L
.66336 .23361 L
.66193 .23366 L
.6604 .23375 L
.6588 .23388 L
.65711 .23408 L
.65535 .23435 L
.6523 .23501 L
.64955 .23585 L
.64718 .2368 L
.64461 .23813 L
.64239 .23958 L
.63868 .24277 L
.63518 .2468 L
.63174 .2517 L
.62598 .26075 L
.61884 .27051 L
.61397 .27573 L
.60901 .28007 L
Mistroke
.60351 .28398 L
.59809 .2871 L
.59286 .28957 L
.58702 .29179 L
.58121 .29352 L
.57806 .29427 L
.57524 .29484 L
.57283 .29524 L
.57027 .2956 L
.5689 .29575 L
.56766 .29587 L
.56641 .29597 L
.56527 .29604 L
.56422 .2961 L
.5631 .29613 L
.56211 .29615 L
.56121 .29615 L
.56025 .29614 L
.55922 .29612 L
.55816 .29607 L
.5572 .29601 L
.55619 .29594 L
.55515 .29584 L
.55333 .29562 L
.55143 .29532 L
.5494 .29492 L
.54263 .29305 L
.52914 .28784 L
.52052 .28443 L
.51068 .2809 L
.50134 .27814 L
.49685 .27709 L
.49286 .27634 L
.4909 .27604 L
.4888 .27578 L
.48684 .27558 L
.48505 .27545 L
.48408 .27539 L
.48317 .27535 L
.48217 .27532 L
.48111 .2753 L
.48 .27529 L
.47894 .2753 L
.478 .27532 L
.47696 .27534 L
.4758 .27539 L
.4747 .27544 L
.47263 .27555 L
.46789 .27588 L
.46326 .2762 L
Mistroke
.46109 .27632 L
.45869 .27643 L
.45742 .27647 L
.45619 .2765 L
.45508 .27652 L
.45385 .27653 L
.45246 .27652 L
.45175 .27652 L
.45096 .2765 L
.44954 .27647 L
.44819 .27642 L
.44673 .27636 L
.44517 .27627 L
.44232 .27606 L
.43917 .27576 L
.43565 .27534 L
.42304 .27322 L
.41113 .27052 L
.38841 .2646 L
.38319 .26343 L
.38055 .26292 L
.3782 .26255 L
.37629 .26231 L
.37534 .26221 L
.37432 .26212 L
.37375 .26208 L
.37323 .26205 L
.37222 .262 L
.37132 .26198 L
.37038 .26198 L
.3694 .26199 L
.36852 .26202 L
.36806 .26204 L
.36756 .26206 L
.36656 .26213 L
.36474 .26229 L
.36304 .26247 L
.35874 .26297 L
.35679 .26317 L
.35573 .26326 L
.35458 .26334 L
.35356 .26339 L
.35297 .26342 L
.35242 .26343 L
.35142 .26345 L
.35033 .26345 L
.34919 .26342 L
.34808 .26337 L
.34707 .2633 L
.34594 .26321 L
Mistroke
.34334 .26289 L
.34183 .26265 L
.34044 .26239 L
.33488 .26102 L
.32912 .25907 L
.32275 .25634 L
.3111 .24989 L
.29921 .24143 L
.28834 .23183 L
.28129 .22419 L
.27743 .21893 L
.27682 .21788 L
.27636 .21693 L
.27619 .21649 L
.27604 .21605 L
.27599 .21588 L
.27595 .2157 L
.27592 .21553 L
.2759 .21539 L
.27589 .21513 L
.27589 .21501 L
.27589 .21488 L
.27591 .21474 L
.27593 .21462 L
.27599 .21441 L
.27606 .21422 L
.27614 .21406 L
.27637 .21369 L
.27753 .2121 L
.27791 .21143 L
.27827 .21066 L
.27893 .20874 L
.2793 .20734 L
.27963 .20581 L
.28034 .20147 L
.28093 .19667 L
.28159 .19051 L
.2851 .15923 L
.28788 .14075 L
.29036 .12682 L
.29119 .12221 L
.29155 .12011 L
.2918 .11841 L
.2919 .11758 L
.29194 .11718 L
.29197 .11678 L
.292 .11615 L
.292 .11565 L
.29196 .11523 L
.2919 .11489 L
Mistroke
.2918 .11461 L
.29173 .11448 L
.29166 .11438 L
.29148 .11425 L
.29136 .1142 L
.29125 .11419 L
.29099 .1142 L
.29071 .11428 L
.29054 .11436 L
.29034 .11445 L
.28996 .11467 L
.28894 .11539 L
.28376 .11973 L
.2769 .12514 L
.26818 .13072 L
.26374 .13297 L
.25953 .13476 L
.25188 .1372 L
.23767 .14021 L
.23406 .14112 L
.23016 .14237 L
.22267 .14561 L
.20383 .15547 L
.17823 .16583 L
.15462 .17164 L
.14918 .173 L
.14648 .17382 L
.14412 .17469 L
.14209 .1756 L
.1401 .17673 L
.13848 .17789 L
.13712 .17911 L
.13586 .18053 L
.13485 .18198 L
.13388 .18376 L
.13301 .18588 L
.13235 .1881 L
.13204 .18945 L
.1318 .19075 L
.13161 .19203 L
.13146 .19327 L
.13139 .19397 L
.13133 .19475 L
.13124 .19613 L
.13119 .19746 L
.13116 .19873 L
.13116 .20015 L
.13118 .20171 L
.1312 .20251 L
.13124 .20338 L
Mistroke
.13132 .20499 L
.13142 .20658 L
.13154 .20806 L
.13216 .21381 L
.13308 .21998 L
.13558 .23187 L
.13729 .23788 L
.13902 .24275 L
.14078 .24673 L
.14249 .2498 L
.14435 .25235 L
.14644 .2544 L
.14759 .25521 L
.14866 .2558 L
.14984 .25629 L
.15045 .25649 L
.1511 .25667 L
.15168 .25679 L
.1522 .25689 L
.15324 .25703 L
.15381 .25709 L
.15444 .25715 L
.1556 .25723 L
.15669 .25731 L
.15773 .25741 L
.15831 .25747 L
.15893 .25756 L
.16002 .25776 L
.16107 .25802 L
.16219 .25838 L
.16423 .25928 L
.16668 .2608 L
.16901 .26266 L
.17475 .2684 L
.18788 .28208 L
.19691 .28949 L
.20643 .29538 L
.21796 .30047 L
.22408 .30254 L
.23097 .3045 L
.28017 .31803 L
.32388 .33302 L
.33399 .33634 L
.33895 .33824 L
.3439 .34053 L
.35212 .34572 L
.35898 .35186 L
.37279 .36687 L
.37695 .37065 L
.37934 .37249 L
Mistroke
.38165 .37402 L
.38386 .37523 L
.38594 .37617 L
.3883 .37699 L
.38964 .37733 L
.39091 .37759 L
.39204 .37776 L
.3931 .37786 L
.39428 .37792 L
.39556 .37792 L
.39684 .37784 L
.39748 .37778 L
.3982 .37769 L
.39946 .37748 L
.40064 .37723 L
.40298 .37655 L
.40511 .37575 L
.40988 .37333 L
.41492 .36981 L
.42015 .36516 L
.4287 .35542 L
.43531 .34643 L
.44116 .33869 L
.44271 .33701 L
.44414 .33573 L
.44541 .33483 L
.44613 .33444 L
.4468 .33415 L
.44713 .33403 L
.44749 .33392 L
.44815 .33378 L
.44879 .3337 L
.44939 .33368 L
.45006 .33372 L
.45044 .33377 L
.45081 .33383 L
.45155 .334 L
.45237 .33425 L
.45396 .3349 L
.45552 .3357 L
.45942 .33808 L
.46797 .34369 L
.47742 .34889 L
.48272 .3511 L
.48562 .35209 L
.48887 .35298 L
.49041 .35333 L
.49207 .35366 L
.49351 .3539 L
.49508 .35411 L
Mistroke
.496 .35421 L
.49686 .35428 L
.49855 .35439 L
.50009 .35444 L
.50171 .35443 L
.50312 .35438 L
.50464 .35428 L
.50607 .35415 L
.50736 .35398 L
.51052 .35343 L
.51349 .35272 L
.5166 .35176 L
.51985 .35054 L
.5255 .34785 L
.53731 .34005 L
.54663 .3324 L
.55571 .32497 L
.56332 .32 L
.56543 .31893 L
.56772 .31795 L
.5699 .31717 L
.57194 .31657 L
.57433 .31605 L
.5756 .31583 L
.57697 .31565 L
.57817 .31553 L
.57931 .31545 L
.58057 .3154 L
.58128 .31538 L
.58194 .31538 L
.58316 .31539 L
.5843 .31544 L
.58557 .31551 L
.58693 .31562 L
.58951 .3159 L
.59188 .31625 L
.59781 .31736 L
.60448 .3189 L
.61715 .32216 L
.63062 .32539 L
.63731 .32672 L
.64344 .32772 L
.64992 .32851 L
.65348 .32882 L
.65509 .32893 L
.65685 .32903 L
.65861 .32911 L
.65958 .32914 L
.66048 .32916 L
.66209 .32919 L
Mistroke
.66383 .32919 L
.66474 .32918 L
.66574 .32916 L
.66678 .32914 L
.66774 .32911 L
.66946 .32903 L
.67129 .32892 L
.6746 .32864 L
.67805 .32824 L
.68408 .32726 L
.68732 .32657 L
.69028 .32582 L
.6967 .3238 L
.70188 .32167 L
.70624 .31945 L
.71056 .31675 L
.71465 .31359 L
.7209 .30722 L
.72559 .30101 L
.73065 .29398 L
.73303 .29123 L
.73594 .28858 L
.73919 .28648 L
.74119 .28555 L
.74321 .28483 L
.74442 .2845 L
.74562 .28422 L
.74674 .28402 L
.74799 .28384 L
.74865 .28376 L
.74939 .28369 L
.75007 .28364 L
.75073 .28361 L
.75151 .28358 L
.75225 .28357 L
.75311 .28357 L
.75394 .28358 L
.75537 .28364 L
.7567 .28372 L
.75828 .28386 L
.75981 .28404 L
.76263 .28444 L
.76581 .285 L
.781 .2888 L
.79572 .29334 L
.81165 .29811 L
.81622 .29927 L
.82048 .30019 L
.82422 .30086 L
.82612 .30113 L
Mistroke
.8282 .30137 L
.82912 .30145 L
.8301 .30153 L
.83187 .30162 L
.83285 .30165 L
.83386 .30166 L
.83473 .30166 L
.83567 .30163 L
.83734 .30154 L
.83827 .30146 L
.8391 .30138 L
.84072 .30116 L
.84222 .3009 L
.84511 .30022 L
.8482 .29921 L
.85136 .29786 L
.85425 .29632 L
.85928 .29289 L
.86453 .2883 L
.87609 .27471 L
.88633 .2596 L
.90547 .22665 L
.90913 .21937 L
.91191 .21316 L
.91296 .21045 L
.91388 .20775 L
.91455 .2054 L
.91481 .2043 L
.915 .20333 L
.91516 .20241 L
.91529 .20143 L
.91537 .20043 L
.91541 .19951 L
.9154 .19853 L
.91533 .19752 L
.91522 .19669 L
.91506 .1958 L
.91482 .19488 L
.91455 .19406 L
.91376 .19228 L
.91287 .19079 L
.91173 .18927 L
.90553 .1834 L
.89648 .17619 L
.88306 .16325 L
.87602 .15415 L
.87034 .14503 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1Go`@0<Ol0097o00<0ool0Eol0103oo`0ao`00OOl200Co0P05o`<01?l00`3oo`1Fo`03
0?oo00;o00<0ool0;_l005So0P0Ro`040?oo00_o00<0ool00ol00`3oo`1Eo`030?oo00?o00<0ool0
;_l005Oo00<0o`008_l0103oo`08o`D01?l205Ko00<0ool01?l00`3oo`0]o`00E_l0103oo`0Lo`@0
0_l0103oo`08o`040?oo00Go00<0ool0E?l00`3oo`05o`030?oo02go001Eo`050?ooo`008_l0103o
o`09o`030?l000Go00<0ool0Dol00`3oo`06o`030?oo02go001Co`801?l00`3oo`0Qo`802ol200Go
00<0ool0DOl200[o00<0ool0;?l0057o0P06o`030?oo03Go00<0ool0D?l00`3oo`0:o`030?oo02co
001>o`<01ol00`3oo`0fo`030?oo04oo00<0ool03?l00`3oo`0[o`00BOl500[o00<0ool0=_l00`3o
o`1;o`@03ol00`3oo`0[o`00Aol200oo00<0ool0=_l00`3oo`0no`h04ol00`3oo`0Zo`00AOl2017o
00<0ool0=_l00`3oo`0lo`<08Ol00`3oo`0Yo`00@ol201;o00<0ool0=ol00`3oo`0ko`808ol00`3o
o`0Yo`00@Ol201Co00<0ool0=ol00`3oo`0ko`030?oo02?o00<0ool0:?l003oo0P0Fo`030?oo03Oo
00<0ool0>_l00`3oo`0To`030?oo02So000mo`806?l00`3oo`0go`030?oo03[o00<0ool09Ol00`3o
o`0Wo`00>Ol401[o00<0ool0=ol00`3oo`0jo`030?oo02Ko0P0Wo`00=Ol401ko00<0ool0=ol00`3o
o`0jo`030?oo02So00<0ool09?l0037o100Ro`030?oo02?o0P04o`800ol400Go00<0ool0>_l00`3o
o`0Yo`030?oo02?o000`o`809?l00`3oo`0So`040?oo00So00<0ool01_l00`3oo`0jo`030?oo02[o
00<0ool08_l002oo0P0Uo`030?oo02?o00@0ool02Ol00`3oo`05o`80>ol00`3oo`0[o`030?oo027o
000_o`030?oo02Co00<0ool07Ol400;o00@0ool02_l00`3oo`04o`030?oo03[o00<0ool0;?l00`3o
o`0Po`00;ol00`3oo`0To`030?oo02?o00@0ool02?l0103oo`05o`030?oo03[o00<0ool0;Ol00`3o
o`0Oo`00;_l202Ko00<0ool09?l200[o0P06o`030?oo03_o00<0ool0;Ol00`3oo`0No`00;_l00`3o
o`0Uo`030?oo03So00<0ool0>ol00`3oo`0]o`030?oo01ko000^o`030?oo02Go00<0ool0>?l00`3o
o`0lo`030?oo02co00<0ool07_l002ko00<0ool09Ol00`3oo`0ho`030?oo03co00<0ool0;?l00`3o
o`0No`001ol500;o0P04o`806?l00`3oo`08o`D06?l00`3oo`07o`801?l200Co0P0So`030?oo02;o
0P04o`801?l200co00<0ool05?l501;o0P0>o`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01So
00<0ool02Ol00`3oo`0Go`802?l0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00co
00<0ool05Ol00`3oo`0Bo`030?oo00oo00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Ko
00<0ool02Ol00`3oo`0Go`030?oo00Oo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3o
o`0:o`030?oo01Go00<0ool04_l00`3oo`0?o`030?oo00[o0@01o`7o00000ol0000200Co00<0ool0
2?l201Wo00@0oooo1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol200go00<0ool05Ol00`3oo`0Ao`030?oo013o00<0ool02?l200?o0009o`030?oo00So00<0ool0
6?l00`3oo`09o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo00go00<0ool05?l00`3oo`0Ao`030?oo013o00<0ool02?l00`3oo`02o`002?l200[o0`0Io`03
0?oo00Oo0P0Ko`030?oo00Ko0P0:o`<08_l00`3oo`0Ro`802_l300go00<0ool04ol201;o00<0ool0
4?l200[o0`02o`00<?l00`3oo`0Uo`030?oo03Ko00<0ool0?_l00`3oo`0Vo`030?oo02;o000`o`03
0?oo02Ko00<0ool0=Ol00`3oo`0To`d03Ol00`3oo`0Vo`030?oo02;o000`o`030?oo02Oo00<0ool0
=?l00`3oo`0Ro`803Ol500Oo0P0Wo`030?oo02?o000`o`030?oo02So00<0ool0<ol00`3oo`0Qo`03
0?oo01;o200Xo`030?oo02?o000ao`030?oo02So0P0co`030?oo023o00<0ool0@_l00`3oo`0To`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l0103oool200?o00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o00@0ool02?l00`3o
o`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l602So100/o`030?oo01oo00<0ool01ol00`3o
o`0Yo`030?oo00co00<0ool06ol00`3oo`07o`00>?l00`3oo`0Zo`<00ol701oo00<0ool07_l00`3o
o`12o`030?oo02Ko000io`030?oo02co0`06o`H06_l00`3oo`0Mo`030?oo04?o00<0ool09_l003[o
00<0ool0>_l401Ko00<0ool07?l00`3oo`13o`030?oo02Oo000ko`030?oo03go0`0Co`030?oo01_o
00<0ool0A?l00`3oo`0Wo`00??l00`3oo`0ooaD07?l00`3oo`13o`030?oo02So000lo`030?oo05;o
00@0o`006?l204Co0P0[o`00?Ol00`3oo`1Ao`040?ooo`<05?l00`3oo`0Ro`P06Ol00`3oo`0[o`00
?_l2057o00<0ool01?l3013o00<0ool08Ol200So0`0Eo`030?oo02co0010o`030?oo04ko00<0ool0
1ol300[o0`0So`030?oo00_o0P0Bo`030?oo02go0011o`<0COl00`3oo`0:o`X09_l00`3oo`0=o`<0
3_l00`3oo`0^o`00A?l204_o00<0ool0>_l00`3oo`0@o`H01Ol3037o0016o`D0A_l00`3oo`0ho`80
6Ol603?o001;o`<0@ol00`3oo`0go`030?oo05;o001>o`@0:ol200Co0P03o`@01Ol00`3oo`0go`03
0?oo05;o001Bo`<09ol0103oo`08o`030?oo00Ko00<0ool04Ol4027o00<0ool0Dol005Go0`0To`04
0?oo00Wo00<0ool01Ol200oo1002o`H06Ol305Ko001Ho`<08Ol0103oo`0:o`030?oo00Co00<0ool0
3?l200co100Do`030?oo05Oo001Ko`<07_l0103oo`08o`040?oo00Go00<0ool02ol00`3oo`0@o`@0
3?l405[o001No`<07?l200[o0P06o`030?oo00[o00<0ool05Ol<05ko001Qo`807ol300co00<0ool0
2Ol00`3oo`20o`00Hol401[o0P001?l0000:o`030?oo00Oo0P23o`00Iol00`3oo`0Go`030?oo00?o
00<0ool01ol00`3oo`06o`030?oo08?o001Xo`805_l00`3oo`05o`030?oo00Ko00<0ool01?l208Ko
001Zo`030?oo01;o00<0ool01ol200Ko00<0ool00ol00`3oo`26o`00Jol00`3oo`0Ao`030?oo00Wo
1002o`030?oo00<0ROl006co00<0ool03ol00`3oo`0=o`H0S?l006go00<0ool03Ol00`3oo`0Ao`03
0?oo08co001^o`030?oo00_o00<0ool04_l00`3oo`2<o`00Kol00`3oo`09o`030?oo01?o00<0ool0
S?l0073o00<0ool01ol00`3oo`0Do`030?oo08co001`o`<01?l300?o0P04o`801Ol300Co00<0ool0
S?l007;o1P04o`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87104 .1468 m
.85993 .12368 L
.85525 .11447 L
.85393 .1124 L
.85267 .11071 L
.85151 .10942 L
.85025 .10832 L
.84962 .10787 L
.84892 .10746 L
.84826 .10716 L
.84765 .10694 L
.84728 .10684 L
.84692 .10676 L
.84654 .10669 L
.84612 .10665 L
.84575 .10663 L
.84535 .10663 L
.84462 .10668 L
.84389 .1068 L
.84347 .10689 L
.84307 .107 L
.84156 .10755 L
.84013 .10824 L
.83858 .10917 L
.83105 .11519 L
.82166 .12375 L
.81263 .131 L
.8079 .13408 L
.80335 .13651 L
.79792 .13876 L
.79502 .1397 L
.79176 .14057 L
.78636 .14163 L
.78334 .14206 L
.78048 .14239 L
.76903 .14337 L
.76335 .14397 L
.76021 .14443 L
.75736 .14498 L
.75481 .1456 L
.75248 .14632 L
.7484 .14801 L
.74629 .1492 L
.74447 .15044 L
.7425 .15212 L
.74088 .15384 L
.73951 .15565 L
.7382 .1578 L
.73705 .1603 L
.73616 .16287 L
Mistroke
.73553 .16543 L
.73525 .16693 L
.73505 .16832 L
.7349 .16966 L
.73479 .17115 L
.73472 .17258 L
.7347 .17391 L
.73471 .17553 L
.73477 .17729 L
.73487 .17897 L
.735 .18053 L
.73546 .18442 L
.73605 .18818 L
.73758 .19576 L
.73958 .20393 L
.74138 .21095 L
.74306 .218 L
.74366 .22101 L
.74393 .22257 L
.74413 .22398 L
.74428 .22525 L
.74438 .22636 L
.74444 .22755 L
.74446 .22877 L
.74442 .22998 L
.74433 .23104 L
.74416 .23215 L
.74404 .23269 L
.7439 .23326 L
.74361 .23417 L
.74322 .23509 L
.74233 .23659 L
.74167 .23739 L
.74084 .23818 L
.74001 .2388 L
.73904 .23939 L
.73808 .23985 L
.73693 .24029 L
.73579 .24064 L
.73466 .24091 L
.73351 .24112 L
.73222 .2413 L
.73093 .24143 L
.72971 .24151 L
.72822 .24157 L
.72735 .24158 L
.72653 .24159 L
.72498 .24156 L
.72328 .2415 L
.72161 .24141 L
Mistroke
.71976 .24128 L
.71629 .24097 L
.70791 .24 L
.70038 .23902 L
.69225 .23799 L
.68865 .23759 L
.68482 .23721 L
.68149 .23693 L
.6797 .23681 L
.67805 .23672 L
.67605 .23663 L
.67501 .2366 L
.6739 .23657 L
.6729 .23656 L
.67196 .23655 L
.67023 .23657 L
.66856 .23662 L
.66704 .2367 L
.66542 .23681 L
.66373 .23697 L
.66208 .23717 L
.66035 .23744 L
.65736 .23804 L
.65404 .23897 L
.65121 .24003 L
.64687 .24224 L
.64267 .24523 L
.63875 .24889 L
.63536 .25269 L
.62218 .26843 L
.61297 .27666 L
.60309 .28293 L
.59771 .28553 L
.59165 .28791 L
.58531 .28987 L
.57949 .29124 L
.57671 .29176 L
.57414 .29217 L
.57185 .29248 L
.56947 .29273 L
.56815 .29285 L
.56698 .29293 L
.5658 .293 L
.56474 .29305 L
.56415 .29308 L
.56351 .29309 L
.56287 .29311 L
.56221 .29312 L
.56163 .29312 L
.56101 .29312 L
Mistroke
.55991 .29311 L
.55887 .29309 L
.55829 .29307 L
.55775 .29304 L
.55666 .29299 L
.55552 .29291 L
.55349 .29273 L
.55167 .29252 L
.54739 .29185 L
.54338 .29102 L
.52758 .28655 L
.50934 .28119 L
.50103 .27918 L
.49671 .27832 L
.49221 .27757 L
.48991 .27727 L
.48741 .27699 L
.48517 .27679 L
.48309 .27666 L
.482 .2766 L
.48097 .27656 L
.48004 .27653 L
.47904 .27651 L
.47795 .2765 L
.4768 .27649 L
.47581 .2765 L
.47472 .27651 L
.47356 .27654 L
.47233 .27657 L
.47012 .27665 L
.46503 .27687 L
.46247 .27699 L
.46 .27708 L
.45772 .27715 L
.45641 .27718 L
.45519 .27719 L
.45402 .2772 L
.45291 .27721 L
.45168 .2772 L
.45034 .27718 L
.44891 .27715 L
.44754 .27711 L
.4463 .27706 L
.44492 .27699 L
.44198 .2768 L
.43877 .27653 L
.4328 .27586 L
.42658 .27492 L
.42062 .27384 L
.40948 .27139 L
Mistroke
.38805 .26602 L
.383 .26488 L
.37801 .26394 L
.37574 .2636 L
.37367 .26335 L
.37186 .26318 L
.37086 .2631 L
.36994 .26305 L
.36905 .263 L
.36809 .26297 L
.36757 .26295 L
.36703 .26294 L
.36607 .26293 L
.36559 .26293 L
.36507 .26293 L
.36414 .26294 L
.36357 .26295 L
.36304 .26296 L
.36203 .26298 L
.35846 .26307 L
.35757 .26309 L
.35661 .26311 L
.35605 .26312 L
.35553 .26312 L
.35452 .26312 L
.35345 .26311 L
.35284 .2631 L
.35226 .26309 L
.3512 .26305 L
.35004 .26299 L
.34898 .26292 L
.34797 .26284 L
.3456 .26258 L
.34294 .26219 L
.3404 .26172 L
.33434 .26017 L
.32901 .25835 L
.32308 .25585 L
.31217 .25003 L
.29018 .23343 L
.28269 .22586 L
.27988 .22253 L
.27774 .21962 L
.27651 .21757 L
.27606 .21659 L
.2759 .21616 L
.27578 .21578 L
.2757 .21545 L
.27564 .2151 L
.2756 .2148 L
Mistroke
.27559 .21453 L
.27559 .21422 L
.27561 .21395 L
.27563 .2138 L
.27566 .21363 L
.27572 .21334 L
.28241 .18735 L
.28783 .15781 L
.29071 .14329 L
.2929 .13221 L
.29366 .12785 L
.29393 .12591 L
.29404 .12495 L
.29412 .12412 L
.29416 .12345 L
.29418 .12279 L
.29417 .12213 L
.29413 .12158 L
.29408 .12117 L
.29399 .12076 L
.29387 .12039 L
.29373 .1201 L
.29354 .11982 L
.29341 .11969 L
.29328 .11959 L
.29303 .11945 L
.29272 .11936 L
.2924 .11932 L
.29208 .11934 L
.2917 .11939 L
.29126 .11951 L
.29074 .11968 L
.29013 .11993 L
.28889 .12052 L
.28615 .12207 L
.27812 .1268 L
.26905 .1314 L
.26045 .13471 L
.25566 .13611 L
.25138 .13713 L
.24312 .13866 L
.23883 .13941 L
.23487 .14019 L
.22721 .1423 L
.21984 .14518 L
.20152 .15376 L
.1771 .1633 L
.15594 .16925 L
.15092 .17075 L
.14608 .17253 L
Mistroke
.14211 .17451 L
.139 .17671 L
.13755 .17807 L
.13633 .17947 L
.13505 .1813 L
.13405 .18312 L
.13324 .18502 L
.13258 .18693 L
.13202 .1891 L
.13176 .19037 L
.13154 .1916 L
.13135 .19294 L
.13121 .19419 L
.13108 .19562 L
.13102 .19644 L
.13098 .1972 L
.13095 .19795 L
.13092 .19875 L
.1309 .19956 L
.13089 .20047 L
.13089 .20126 L
.1309 .20212 L
.13092 .20294 L
.13094 .2037 L
.131 .20516 L
.13107 .20654 L
.13131 .2097 L
.13163 .21283 L
.13206 .21618 L
.13317 .22281 L
.13445 .2287 L
.1361 .23472 L
.13807 .24041 L
.13995 .24464 L
.14183 .24798 L
.14399 .25094 L
.14516 .2522 L
.14642 .25335 L
.14883 .25503 L
.15111 .25615 L
.15388 .25715 L
.15642 .25793 L
.15887 .25874 L
.16157 .25985 L
.16418 .26124 L
.16665 .26287 L
.17905 .27422 L
.18652 .28134 L
.19436 .28778 L
.20353 .29378 L
Mistroke
.21308 .29851 L
.22384 .30246 L
.23596 .30588 L
.28358 .31986 L
.30613 .32787 L
.32608 .33423 L
.33458 .33716 L
.34279 .34072 L
.35097 .34562 L
.35793 .35112 L
.36541 .35796 L
.37278 .36437 L
.37619 .36688 L
.37995 .36921 L
.38199 .37026 L
.38429 .37125 L
.38641 .37198 L
.38846 .37254 L
.38962 .37278 L
.39089 .37299 L
.39206 .37313 L
.39314 .37322 L
.39382 .37325 L
.39445 .37326 L
.39564 .37325 L
.39693 .37318 L
.39766 .37312 L
.39833 .37304 L
.39947 .37288 L
.40053 .37269 L
.40295 .3721 L
.40532 .37135 L
.40784 .37034 L
.41343 .36739 L
.41853 .36386 L
.42737 .35606 L
.43421 .34898 L
.44059 .3425 L
.44392 .33976 L
.44545 .33877 L
.44686 .33804 L
.44827 .33749 L
.44902 .33728 L
.44941 .33719 L
.44983 .33711 L
.4506 .33701 L
.45133 .33696 L
.45206 .33696 L
.45273 .337 L
.45346 .33708 L
Mistroke
.45425 .33721 L
.45501 .33737 L
.45571 .33755 L
.45925 .33883 L
.46751 .34295 L
.47247 .3455 L
.47823 .34817 L
.48398 .35039 L
.48692 .35132 L
.48968 .35206 L
.49217 .35262 L
.49494 .35311 L
.49629 .3533 L
.49775 .35347 L
.49929 .35361 L
.50073 .3537 L
.50234 .35375 L
.50324 .35376 L
.50408 .35375 L
.50562 .35371 L
.50724 .35361 L
.5087 .35348 L
.51005 .35332 L
.51313 .35284 L
.5164 .35212 L
.51946 .35125 L
.52507 .34919 L
.53088 .34642 L
.53692 .3429 L
.54788 .33528 L
.55743 .32842 L
.56581 .32343 L
.57059 .32132 L
.57275 .32057 L
.5751 .31989 L
.57746 .31933 L
.57965 .31894 L
.58071 .31879 L
.58188 .31865 L
.58299 .31854 L
.58403 .31846 L
.58525 .31839 L
.58661 .31834 L
.58795 .31833 L
.58871 .31833 L
.58941 .31834 L
.59059 .31838 L
.5919 .31844 L
.59328 .31852 L
.59457 .31862 L
Mistroke
.59747 .31891 L
.60026 .31925 L
.60559 .32004 L
.61803 .32222 L
.63014 .32428 L
.63688 .32522 L
.64019 .32561 L
.64321 .32592 L
.64581 .32614 L
.64863 .32633 L
.65018 .32642 L
.65161 .32649 L
.65297 .32654 L
.65444 .32658 L
.6559 .32661 L
.65727 .32662 L
.6589 .32662 L
.66039 .3266 L
.66185 .32657 L
.66345 .32652 L
.66511 .32644 L
.66666 .32636 L
.66992 .32611 L
.6729 .32582 L
.67953 .3249 L
.68519 .32381 L
.69095 .32236 L
.70023 .31918 L
.70503 .31702 L
.7092 .31478 L
.71721 .30924 L
.72353 .30348 L
.72903 .29776 L
.73401 .29298 L
.73683 .29078 L
.74018 .28873 L
.74219 .28776 L
.74416 .28699 L
.74616 .28637 L
.7484 .28583 L
.75043 .28547 L
.75162 .28532 L
.75273 .2852 L
.75378 .28512 L
.75496 .28505 L
.7561 .28501 L
.75717 .285 L
.75844 .28501 L
.75966 .28504 L
.76102 .28511 L
Mistroke
.76252 .28521 L
.76404 .28534 L
.76567 .28551 L
.76868 .2859 L
.77533 .28703 L
.78169 .28837 L
.79658 .29201 L
.80972 .29513 L
.81619 .29637 L
.81979 .29692 L
.82309 .2973 L
.82485 .29745 L
.82651 .29754 L
.82797 .2976 L
.82956 .29762 L
.83126 .29759 L
.8322 .29755 L
.83304 .2975 L
.83467 .29736 L
.83619 .29718 L
.83759 .29697 L
.83905 .2967 L
.84161 .29608 L
.84432 .29524 L
.84721 .29408 L
.85305 .29091 L
.85905 .28637 L
.86942 .27546 L
.88996 .24512 L
.8981 .23086 L
.90482 .21734 L
.90693 .21212 L
.90782 .20945 L
.90849 .20704 L
.90878 .20575 L
.90903 .20439 L
.90912 .2038 L
.9092 .20318 L
.90929 .2021 L
.90933 .20154 L
.90934 .20094 L
.90933 .19989 L
.90927 .19888 L
.90916 .19798 L
.90902 .19717 L
.90883 .19632 L
.90837 .19482 L
.90766 .1931 L
.90684 .19157 L
.90456 .18832 L
Mistroke
.90184 .18522 L
.88743 .17042 L
.87179 .14816 L
.87104 .1468 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1Go`@0<Ol007go0P04o`801Ol300Co00<0ool0Eol00`3oo`02033o
001lo`040?oo00_o00<0ool00ol00`3oo`1Eo`801?l00`3oo`0^o`00FOl2027o00@0ool02?l500Co
0P1Do`801_l00`3oo`0^o`00Eol20003o`3o01[o1002o`040?oo00So00@0ool01Ol00`3oo`1Bo`03
0?oo00Oo00<0ool0;Ol005Go0P03o`030?oo01oo00@0ool02Ol00`3o0005o`030?oo057o00<0ool0
2?l00`3oo`0]o`00Dol200Go00<0ool08?l200_o0P05o`030?oo053o00<0ool02_l00`3oo`0/o`00
DOl200Ko00<0ool0=Ol00`3oo`1>o`803Ol00`3oo`0/o`00C?l500So00<0ool0=Ol00`3oo`1<o`80
4?l00`3oo`0[o`00B?l400go00<0ool0=Ol00`3oo`16o`H04_l00`3oo`0[o`00AOl3017o00<0ool0
=Ol00`3oo`10o`H06Ol00`3oo`0Zo`00@ol201Co00<0ool0=Ol00`3oo`0no`807ol00`3oo`0Zo`00
@Ol201Go00<0ool0=_l00`3oo`0mo`808Ol00`3oo`0Yo`00?ol201Oo00<0ool0=_l00`3oo`0lo`03
0?oo02;o00<0ool0:?l003go0P0Io`030?oo03Ko00<0ool0??l00`3oo`0Ro`030?oo02So000jo`<0
6ol00`3oo`0fo`030?oo03_o00<0ool09?l00`3oo`0Wo`00=ol301ko00<0ool0=_l00`3oo`0ko`03
0?oo02Go00<0ool09_l003;o1@0Qo`030?oo03Ko00<0ool0>ol00`3oo`0Vo`030?oo02Go000ao`03
0?oo02?o00<0ool08ol200Co0P03o`@01Ol00`3oo`0ko`030?oo02Oo00<0ool09?l0033o00<0ool0
9?l00`3oo`0Ro`040?oo00So00<0ool01_l00`3oo`0ko`030?oo02Oo00<0ool09?l002oo00<0ool0
9Ol00`3oo`0Ro`040?oo00Wo00<0ool01Ol203co00<0ool0:?l00`3oo`0So`00;ol00`3oo`0Uo`03
0?oo01co1002o`040?oo00[o00<0ool01?l00`3oo`0ko`030?oo02Wo00<0ool08_l002ko0P0Wo`03
0?oo02;o00@0ool02?l0103oo`05o`030?oo03_o00<0ool0:_l202;o000^o`030?oo02Go00<0ool0
9?l200[o0P06o`030?oo03co00<0ool0:_l00`3oo`0Po`00;_l00`3oo`0Uo`030?oo03So00<0ool0
??l00`3oo`0Zo`030?oo023o000^o`030?oo02Go00<0ool0>?l00`3oo`0lo`030?oo02[o00<0ool0
8?l002ko00<0ool09Ol00`3oo`0ho`030?oo03co00<0ool0:_l00`3oo`0Po`001ol500;o0P04o`80
6?l00`3oo`08o`D05ol00`3oo`08o`801?l200Co0P0So`030?oo02;o0P04o`801?l200go00<0ool0
4ol5017o00<0ool03_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
5ol00`3oo`07o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`03
0?oo017o00<0ool04?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`03
0?oo01Oo00<0ool01ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool0
5Ol00`3oo`0@o`030?oo017o00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806Ol0103o
ool400Co00<0ool05ol0103oool400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`803Ol00`3o
o`0Eo`030?oo013o00<0ool04Ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo
00<0ool06?l00`3oo`06o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool03?l00`3o
o`0Eo`030?oo00oo00<0ool04_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201_o
00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<03Ol00`3oo`0Co`804Ol00`3oo`0Ao`802_l300;o
000`o`030?oo02Go00<0ool0=_l00`3oo`0mo`809ol00`3oo`0So`00<?l00`3oo`0Vo`030?oo03Go
00<0ool0:Ol4013o00<0ool09_l00`3oo`0So`00<?l00`3oo`0Wo`80=Ol00`3oo`0Uo`@01?l900Ko
0P0Wo`030?oo02Co000ao`030?oo02So00<0ool0<_l00`3oo`0To`030?oo00oo1P0Yo`030?oo02Co
000ao`030?oo02Wo0P0bo`030?oo02;o0P10o`030?oo02Go000<ool03009o`003?l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00801?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3o
o`06o`030?oo00;o00D0oooo0008o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo02Go0`000ol0o`0Wo`@0;?l00`3oo`0Po`030?oo00Ko00<0ool0:Ol00`3oo`0:o`03
0?oo01go00<0ool01ol003Ko0`0[o`/08_l00`3oo`0Oo`030?oo043o00<0ool09ol003Wo00<0ool0
<_l701co00<0ool07_l00`3oo`10o`030?oo02So000jo`030?oo03So0P0Jo`030?oo01go00<0ool0
@?l00`3oo`0Yo`00>ol00`3oo`0io`H05?l00`3oo`0Lo`030?oo047o00<0ool0:Ol003co00<0ool0
?_lC0003o`3o01co00<0ool0@Ol00`3oo`0Zo`00?Ol00`3oo`1@o`D06?l204?o00<0ool0:ol003ko
00<0ool0D?l0103oool201?o0`0Vo`D06Ol00`3oo`0/o`00?ol00`3oo`1?o`030?oo00?o100>o`03
0?oo02Co1004o`D04_l202oo0010o`80Col00`3oo`07o`h09_l00`3oo`0;o`@03Ol00`3oo`0_o`00
@_l00`3oo`1<o`030?oo03[o00<0ool04?l600?o100bo`00@ol304_o00<0ool0>Ol00`3oo`0Fo`D0
=Ol004Ko0`18o`030?oo03So00<0ool0DOl004Wo0`15o`030?oo03Oo00<0ool0D_l004co0`0^o`80
1?l200?o1005o`030?oo03Go0P1Eo`00Col402Wo00@0ool02?l00`3oo`06o`030?oo03Co00<0ool0
EOl005?o0`0Vo`040?oo00Wo00<0ool01Ol201;o1`0Io`<0F?l005Ko0`0So`040?oo00[o00<0ool0
1?l00`3oo`0?o`801ol6017o0P1Ko`00FOl3023o00@0ool02?l0103oo`05o`030?oo00go0P0?oa40
GOl005co100Mo`802_l200Ko00<0ool03?l00`3oo`1mo`00H?l402go00<0ool02_l2083o001To`80
7?l400_o00<0ool02Ol00`3oo`20o`00I_l201[o00D0oooo000:o`030?oo00Oo0P23o`00J?l201Ko
0P05o`<01ol00`3oo`06o`030?oo08?o001Zo`030?oo01;o00<0ool02?l200Go00<0ool00ol308Ko
001[o`030?oo013o00<0ool02ol60003o`000080ROl006co00<0ool03_l00`3oo`0@o`@0S?l006go
00<0ool03?l00`3oo`0Bo`030?oo08co001^o`802_l201Go00<0ool0S?l0073o00<0ool01_l00`3o
o`0Eo`030?oo08co001ao`801?l201So00<0ool0S?l007?o1006o`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87174 .14857 m
.86638 .1378 L
.86106 .12677 L
.85849 .12192 L
.85575 .11747 L
.85312 .11411 L
.85194 .11293 L
.85062 .11184 L
.84913 .11089 L
.8483 .11048 L
.8475 .11017 L
.84675 .10994 L
.84605 .10979 L
.84565 .10972 L
.84522 .10967 L
.84443 .10961 L
.84394 .10961 L
.84347 .10963 L
.84305 .10966 L
.84259 .10971 L
.84208 .10978 L
.84153 .10988 L
.84055 .11011 L
.83873 .11069 L
.83668 .11156 L
.82839 .11651 L
.81988 .1225 L
.81021 .12888 L
.8005 .13395 L
.79141 .13735 L
.78616 .13881 L
.78125 .13994 L
.77178 .14188 L
.76692 .14297 L
.76198 .14435 L
.75945 .14523 L
.75679 .14633 L
.75244 .1487 L
.75048 .15008 L
.74853 .15174 L
.74544 .15521 L
.74402 .15734 L
.74271 .1598 L
.74176 .16206 L
.74088 .16472 L
.74014 .16775 L
.73982 .16953 L
.73958 .17121 L
.73941 .1728 L
.73934 .17365 L
Mistroke
.73927 .17457 L
.73922 .17546 L
.73919 .17628 L
.73915 .17789 L
.73915 .17953 L
.73916 .18037 L
.73918 .18129 L
.73924 .18294 L
.73933 .18449 L
.73961 .18803 L
.74002 .19185 L
.74194 .20514 L
.74286 .21151 L
.74326 .21482 L
.74341 .21639 L
.74351 .21779 L
.7436 .21927 L
.74363 .22002 L
.74364 .22081 L
.74365 .2216 L
.74364 .22232 L
.74361 .2231 L
.74357 .22391 L
.74352 .22459 L
.74345 .2253 L
.74327 .22656 L
.74305 .22772 L
.74279 .22875 L
.7425 .22969 L
.74213 .23066 L
.74132 .23228 L
.74011 .234 L
.73876 .23537 L
.73711 .2366 L
.73502 .23771 L
.73279 .23857 L
.73169 .23889 L
.73049 .2392 L
.72932 .23944 L
.72799 .23968 L
.72552 .24 L
.72407 .24013 L
.72325 .24019 L
.72246 .24024 L
.72089 .24032 L
.71943 .24036 L
.71801 .24038 L
.71647 .24039 L
.71513 .24038 L
.71364 .24036 L
Mistroke
.71201 .24032 L
.71023 .24026 L
.70696 .24013 L
.69882 .23972 L
.69477 .23952 L
.69102 .23936 L
.68929 .2393 L
.68767 .23925 L
.68582 .23921 L
.68415 .23919 L
.68231 .23918 L
.68064 .23919 L
.67883 .23922 L
.67793 .23925 L
.67694 .23928 L
.67518 .23937 L
.67357 .23947 L
.67184 .23961 L
.67004 .23979 L
.66711 .24017 L
.66401 .24071 L
.66139 .2413 L
.6587 .24205 L
.65416 .24372 L
.6493 .24617 L
.64468 .24926 L
.63716 .2558 L
.62953 .26323 L
.62162 .27017 L
.6136 .27584 L
.60398 .28099 L
.59865 .28323 L
.5929 .28522 L
.58793 .28663 L
.58265 .28786 L
.57722 .28883 L
.57232 .28948 L
.56943 .28977 L
.56804 .28988 L
.56676 .28996 L
.56553 .29003 L
.56443 .29008 L
.56322 .29012 L
.56257 .29013 L
.56195 .29014 L
.56091 .29016 L
.55981 .29016 L
.55879 .29015 L
.55786 .29013 L
.55678 .2901 L
Mistroke
.55562 .29006 L
.55355 .28995 L
.55141 .28979 L
.54916 .28958 L
.54517 .2891 L
.53805 .28795 L
.52987 .2863 L
.51043 .28203 L
.50079 .28016 L
.49193 .27882 L
.48763 .27835 L
.48545 .27815 L
.48311 .27798 L
.48177 .2779 L
.48055 .27784 L
.4794 .27779 L
.47815 .27775 L
.47702 .27772 L
.47597 .2777 L
.47487 .27768 L
.47368 .27767 L
.47253 .27767 L
.47147 .27767 L
.4703 .27768 L
.46971 .27769 L
.46906 .2777 L
.46677 .27773 L
.46461 .27778 L
.46248 .27782 L
.46018 .27786 L
.45905 .27788 L
.45797 .2779 L
.457 .27791 L
.45593 .27792 L
.45474 .27792 L
.45348 .27792 L
.45238 .27792 L
.45117 .2779 L
.44984 .27788 L
.44839 .27785 L
.44701 .2778 L
.44573 .27776 L
.44295 .27762 L
.43985 .27741 L
.43405 .27687 L
.42853 .27619 L
.42255 .27526 L
.4117 .27315 L
.38892 .26773 L
.36916 .26406 L
Mistroke
.36175 .26347 L
.35767 .26324 L
.35369 .26298 L
.35136 .26278 L
.34876 .26251 L
.344 .2618 L
.34154 .26133 L
.33886 .26072 L
.33383 .25933 L
.32343 .25537 L
.31159 .24916 L
.29974 .24119 L
.2904 .23352 L
.28366 .22702 L
.2786 .22114 L
.27699 .21882 L
.27584 .21669 L
.27561 .21613 L
.27543 .21563 L
.27529 .21515 L
.27519 .21472 L
.27512 .21432 L
.27507 .2139 L
.27504 .21349 L
.27504 .21325 L
.27504 .21303 L
.27506 .21262 L
.27508 .21241 L
.2751 .21218 L
.27516 .21176 L
.27524 .21137 L
.27546 .21044 L
.27578 .20935 L
.27746 .20454 L
.28229 .1891 L
.28898 .16435 L
.29251 .15056 L
.29416 .14364 L
.29539 .13787 L
.29582 .1355 L
.29617 .13323 L
.2963 .13214 L
.29635 .13159 L
.29639 .13102 L
.29643 .13052 L
.29644 .13007 L
.29645 .12926 L
.29643 .12853 L
.29638 .12792 L
.29628 .1273 L
Mistroke
.29622 .12698 L
.29614 .1267 L
.29599 .12624 L
.29578 .12579 L
.29555 .12543 L
.2953 .12514 L
.29503 .12489 L
.2947 .12468 L
.29436 .12452 L
.29402 .1244 L
.29359 .12432 L
.29308 .12427 L
.29256 .12427 L
.29205 .12431 L
.29149 .12439 L
.29094 .12449 L
.28957 .12483 L
.28638 .12591 L
.27106 .13182 L
.26116 .13483 L
.25207 .13682 L
.24279 .1383 L
.23483 .13957 L
.23055 .14044 L
.22648 .14147 L
.21715 .14457 L
.17613 .16072 L
.15469 .16757 L
.14966 .16948 L
.14486 .17178 L
.14094 .17433 L
.13792 .17709 L
.13664 .17861 L
.13541 .18037 L
.13357 .1839 L
.13268 .18629 L
.13198 .1887 L
.13148 .19099 L
.13126 .19232 L
.13107 .19359 L
.13092 .19485 L
.13081 .19602 L
.13071 .19734 L
.13063 .19877 L
.13058 .20018 L
.13057 .20098 L
.13056 .20173 L
.13057 .20312 L
.1306 .2046 L
.13065 .20587 L
Mistroke
.13072 .20724 L
.13089 .20975 L
.13118 .21275 L
.13152 .2155 L
.13243 .22113 L
.13367 .22694 L
.13651 .23643 L
.13857 .24141 L
.1407 .24544 L
.14274 .24849 L
.14506 .25123 L
.14961 .25501 L
.1523 .25664 L
.15488 .258 L
.16059 .261 L
.1663 .26469 L
.17924 .27572 L
.19414 .28794 L
.20327 .2937 L
.21389 .29882 L
.2235 .30235 L
.23445 .30557 L
.27955 .31902 L
.32054 .33298 L
.35635 .34966 L
.36404 .35538 L
.37259 .36136 L
.37698 .36388 L
.3813 .36587 L
.38532 .36725 L
.38765 .36783 L
.38878 .36806 L
.38983 .36823 L
.39084 .36837 L
.39196 .36849 L
.39314 .36857 L
.39427 .36861 L
.39533 .36862 L
.39647 .36859 L
.39754 .36853 L
.39851 .36844 L
.39977 .36829 L
.40092 .36812 L
.40353 .36759 L
.40583 .36698 L
.40826 .36618 L
.41305 .36416 L
.41743 .36185 L
.4268 .3556 L
.43428 .34983 L
Mistroke
.44142 .34462 L
.44491 .34259 L
.44686 .3417 L
.44867 .34106 L
.44948 .34083 L
.45033 .34063 L
.45107 .34049 L
.45187 .34037 L
.45273 .34028 L
.45354 .34024 L
.45403 .34023 L
.45451 .34023 L
.45539 .34027 L
.45627 .34034 L
.45678 .3404 L
.45725 .34047 L
.45928 .34084 L
.46109 .34129 L
.46309 .34189 L
.47275 .34552 L
.47771 .34743 L
.48334 .34937 L
.48885 .35094 L
.49423 .35209 L
.49724 .35254 L
.49894 .35274 L
.50052 .35289 L
.50185 .35298 L
.50332 .35305 L
.50486 .35308 L
.50629 .35308 L
.50712 .35307 L
.50801 .35304 L
.50886 .353 L
.50965 .35295 L
.51113 .35283 L
.5127 .35266 L
.51594 .35219 L
.51947 .35147 L
.52527 .34984 L
.53132 .34754 L
.54166 .34235 L
.56061 .33084 L
.56557 .32815 L
.57087 .3257 L
.57596 .32386 L
.58067 .3226 L
.5831 .32211 L
.5857 .3217 L
.58796 .32142 L
Mistroke
.58924 .3213 L
.59047 .32121 L
.5919 .32112 L
.59322 .32106 L
.59396 .32104 L
.59464 .32103 L
.5954 .32101 L
.59621 .32101 L
.59755 .32101 L
.59882 .32103 L
.60022 .32106 L
.60101 .32109 L
.60174 .32112 L
.60429 .32124 L
.60669 .32139 L
.61223 .32183 L
.62372 .32291 L
.62972 .32343 L
.63291 .32368 L
.63629 .3239 L
.63927 .32407 L
.64085 .32414 L
.64255 .32421 L
.64411 .32426 L
.64554 .32429 L
.64697 .32432 L
.64834 .32433 L
.64976 .32434 L
.65105 .32434 L
.65255 .32432 L
.65395 .32429 L
.65525 .32426 L
.65645 .32422 L
.65914 .3241 L
.66186 .32394 L
.66483 .32371 L
.67078 .32309 L
.67696 .32219 L
.68268 .32112 L
.69291 .31852 L
.70298 .31494 L
.71261 .31023 L
.72663 .30061 L
.73328 .29556 L
.73689 .29316 L
.74092 .29092 L
.74482 .28924 L
.74703 .2885 L
.7495 .28782 L
.75191 .2873 L
Mistroke
.75419 .28692 L
.75533 .28678 L
.75659 .28664 L
.75771 .28655 L
.75894 .28646 L
.7602 .2864 L
.76141 .28637 L
.76287 .28635 L
.76422 .28636 L
.76566 .28639 L
.76648 .28642 L
.76725 .28646 L
.77022 .28664 L
.77281 .28687 L
.77573 .28718 L
.78184 .28801 L
.79457 .29015 L
.80815 .29243 L
.81153 .29289 L
.81511 .29329 L
.81714 .29347 L
.81897 .2936 L
.81984 .29365 L
.82079 .2937 L
.82168 .29373 L
.8225 .29375 L
.82401 .29376 L
.82541 .29375 L
.82693 .2937 L
.82855 .2936 L
.82936 .29354 L
.83023 .29346 L
.83181 .29327 L
.83471 .29279 L
.83768 .29211 L
.84081 .29115 L
.84629 .28878 L
.85143 .28569 L
.856 .28218 L
.86598 .272 L
.876 .25871 L
.89234 .23205 L
.89775 .22137 L
.89997 .21624 L
.90174 .21121 L
.9024 .20882 L
.90268 .20759 L
.90293 .2063 L
.9031 .20515 L
.90322 .20412 L
Mistroke
.9033 .20316 L
.90333 .20216 L
.90333 .20123 L
.90329 .20026 L
.90321 .19936 L
.9031 .19856 L
.90294 .19762 L
.90272 .19662 L
.90219 .19485 L
.90147 .19303 L
.9005 .19104 L
.89799 .18703 L
.89211 .17932 L
.87174 .14857 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1Fo`D0<Ol007co00@0ool0
2ol00`3oo`03o`030?oo05Co0P04o`030?oo02oo001lo`040?oo00So1@04o`80E?l00`3oo`05o`03
0?oo02ko001fo`@00_l0103oo`08o`040?oo00Go00<0ool0D_l00`3oo`07o`030?oo02go001Ho`<0
8Ol0103oo`09o`030?l000Go00<0ool0D?l200_o00<0ool0;?l005Ko0P02o`030?oo023o0P0;o`80
1Ol00`3oo`1>o`803Ol00`3oo`0/o`00DOl500Co00<0ool0=?l00`3oo`1<o`804?l00`3oo`0[o`00
C?l500Wo00<0ool0=?l00`3oo`19o`<04_l00`3oo`0[o`00Aol500ko00<0ool0=?l00`3oo`15o`@0
5Ol00`3oo`0[o`00A?l301?o00<0ool0=?l00`3oo`11o`@06_l00`3oo`0Zo`00@Ol301Ko00<0ool0
=?l00`3oo`0oo`807_l00`3oo`0Zo`00?ol201So00<0ool0=Ol00`3oo`0oo`030?oo01ko00<0ool0
:Ol003co0`0Jo`030?oo03Go00<0ool0?Ol2027o00<0ool0:Ol003Wo0`0Lo`030?oo03Ko00<0ool0
?Ol00`3oo`0Qo`030?oo02So000go`807ol00`3oo`0fo`030?oo03go00<0ool08_l00`3oo`0Wo`00
=Ol2027o00<0ool0=_l00`3oo`0lo`030?oo02Co00<0ool09_l003;o0`0So`030?oo03Ko00<0ool0
??l00`3oo`0To`030?oo02Ko000ao`030?oo02Co00<0ool08_l200Co0P03o`@01Ol00`3oo`0lo`03
0?oo02Go00<0ool09Ol0033o00<0ool09?l00`3oo`0Ro`040?oo00So00<0ool01_l00`3oo`0lo`03
0?oo02Ko00<0ool09?l002oo00<0ool09Ol00`3oo`0Ro`040?oo00Wo00<0ool01Ol203go00<0ool0
9_l00`3oo`0To`00;ol00`3oo`0Uo`030?oo01co1002o`040?oo00[o00<0ool01?l00`3oo`0lo`03
0?oo02Oo00<0ool08ol002ko0P0Wo`030?oo02;o00@0ool02?l0103oo`05o`030?oo03co00<0ool0
:?l00`3oo`0Ro`00;_l00`3oo`0Vo`030?oo02?o0P0:o`801_l00`3oo`0lo`030?oo02So00<0ool0
8_l002ko00<0ool09Ol00`3oo`0ho`030?oo03co00<0ool0:Ol00`3oo`0Qo`00;_l00`3oo`0Uo`03
0?oo03So00<0ool0?Ol00`3oo`0Xo`030?oo027o000^o`030?oo02Co00<0ool0>Ol00`3oo`0mo`03
0?oo02So00<0ool08Ol000Oo1@02o`801?l201So00<0ool02?l501Oo00<0ool02?l200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0=o`030?oo01?o1@0@o`030?oo00oo1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool05ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`0?o`030?oo01;o00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool03ol00`3oo`0Bo`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201Wo00@0oooo1004o`030?oo01Oo0P02o`@00_l0103oo`09o`80
8ol00`3oo`0Qo`040?oo00Wo0P0=o`030?oo01Go00<0ool03_l00`3oo`0Co`030?oo00So0P03o`00
2Ol00`3oo`08o`030?oo01So00<0ool02Ol00`3oo`0Ho`030?oo00Ko00@0ool02Ol00`3oo`0Ro`03
0?oo027o00@0ool02Ol00`3oo`0<o`030?oo01Go00<0ool03_l00`3oo`0Co`030?oo00So00<0ool0
0_l000So0P0:o`<06?l00`3oo`08o`806ol00`3oo`06o`802_l302;o00<0ool08_l200[o0`0<o`03
0?oo01Co0P0?o`030?oo01?o0P0:o`<00_l0033o00<0ool09Ol00`3oo`0fo`030?oo03go00<0ool0
9Ol00`3oo`0To`00<?l00`3oo`0Vo`030?oo03Go00<0ool0??l00`3oo`0Uo`030?oo02Go000`o`03
0?oo02Oo00<0ool0=?l00`3oo`0Yoa<0:?l00`3oo`0Uo`00<?l00`3oo`0Xo`030?oo03?o00<0ool0
9_l303[o00<0ool09_l0037o00<0ool0:?l203?o00<0ool09?l203go00<0ool09_l000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool00ol400So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3oo`06o`03
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Oo000<o`030?oo02Go0P02o`030?oo02Wo0P0[o`030?oo027o00<0ool01Ol00`3o
o`0Yo`030?oo00Wo00<0ool07_l00`3oo`07o`00=_l00`3oo`0]o`D09_l00`3oo`0Po`030?oo03go
00<0ool0:Ol003Oo0P0bo`H08?l00`3oo`0Oo`030?oo03ko00<0ool0:Ol003Wo00<0ool0=Ol601[o
00<0ool07Ol2043o00<0ool0:_l003[o00<0ool0>_l401Ko00<0ool07?l00`3oo`0oo`030?oo02_o
000ko`030?oo03go4`03o`030?oo01[o0P11o`030?oo02co000lo`030?oo04ko1P0Io`80@_l00`3o
o`0]o`00?Ol205;o00<0o`000P0Co`<0@_l2033o000oo`030?oo04oo00<0ool00_l800Ko1@0Yo`/0
3ol203;o0010o`80Col00`3oo`0:o`H0;?l200_o1`06o`80=?l004;o00<0ool0C?l00`3oo`0jo`80
5?l603Ko0013o`<0Bol00`3oo`0io`030?oo053o0016o`<0B?l00`3oo`0go`80Dol004Wo0`15o`03
0?oo03Ko00<0ool0Dol004co100]o`801?l200?o1005o`030?oo03Co0P1Fo`00D?l402So00@0ool0
2?l00`3oo`06o`030?oo037o0`1Ho`00E?l402Co00@0ool02Ol00`3oo`05o`80<?l205_o001Ho`<0
8Ol0103oo`0:o`030?oo00Co00<0ool04_l<00_o1P1Mo`00Fol201oo00@0ool02?l0103oo`05o`03
0?oo00oo0`0<o`/0Hol005go0`0Mo`802_l200Ko00<0ool03_l00`3oo`1ko`00H?l202oo00<0ool0
3Ol00`3oo`1lo`00H_l202go00<0ool02ol207oo001To`807?l600Wo00<0ool02Ol2087o001Vo`80
6Ol200Go0P07o`030?oo00Oo0P23o`00J?l201Ko00<0ool01ol300Co00<0ool01?l308Go001Zo`03
0?oo017o0P0=o`/0R?l006_o0P0@o`030?oo017o00<0ool0S?l006go0P0=o`030?oo01;o00<0ool0
S?l006oo0P08o`<05Ol00`3oo`2<o`00LOl300;o0`0Ho`030?oo08co001do`<06_l00`3oo`2<o`00
OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool0
2?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`03
0?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
0ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool0
1Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00
OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87244 .15034 m
.86746 .14021 L
.86218 .12987 L
.85951 .12528 L
.85659 .12101 L
.85373 .1177 L
.851 .11537 L
.84936 .11432 L
.84758 .11346 L
.846 .11293 L
.84508 .11271 L
.84421 .11255 L
.84373 .11249 L
.84319 .11243 L
.84268 .1124 L
.8422 .11238 L
.84129 .11239 L
.84078 .11241 L
.84031 .11244 L
.83975 .11249 L
.83923 .11256 L
.83803 .11275 L
.83697 .11297 L
.83582 .11325 L
.83172 .11462 L
.82772 .11634 L
.8181 .12126 L
.78028 .13769 L
.77069 .14103 L
.76592 .14304 L
.76118 .14554 L
.75739 .14813 L
.75365 .15154 L
.75072 .15517 L
.74841 .15908 L
.74655 .16347 L
.7457 .16616 L
.74504 .16878 L
.74447 .17167 L
.74406 .1744 L
.74388 .17594 L
.74372 .17764 L
.7436 .17926 L
.74351 .18077 L
.74345 .18208 L
.74341 .18349 L
.74338 .18483 L
.74337 .18607 L
.74337 .18751 L
.74338 .1891 L
Mistroke
.7434 .1906 L
.74344 .19199 L
.74353 .19503 L
.74363 .19777 L
.74374 .20067 L
.7438 .20219 L
.74385 .20381 L
.74389 .2054 L
.74392 .20683 L
.74394 .20824 L
.74394 .20957 L
.74393 .21103 L
.7439 .21235 L
.74384 .21377 L
.7438 .21448 L
.74374 .21525 L
.74352 .21762 L
.74334 .21891 L
.74315 .22005 L
.74272 .22208 L
.74214 .2241 L
.74135 .22616 L
.74032 .22818 L
.73915 .22992 L
.73793 .23137 L
.73628 .2329 L
.73425 .23437 L
.73209 .23557 L
.7299 .23652 L
.72537 .23795 L
.72258 .23857 L
.7199 .23903 L
.71729 .23939 L
.71435 .23972 L
.70877 .24016 L
.69578 .2408 L
.68885 .24117 L
.68502 .24144 L
.68157 .24174 L
.67544 .24249 L
.67248 .24297 L
.66935 .24358 L
.66397 .24495 L
.65929 .24651 L
.65429 .24867 L
.65004 .25093 L
.64115 .25683 L
.62391 .26928 L
.61534 .27425 L
.60513 .27887 L
Mistroke
.59487 .28229 L
.58522 .28461 L
.58001 .28554 L
.57536 .28619 L
.57285 .28648 L
.57016 .28675 L
.56765 .28695 L
.56538 .28709 L
.56299 .2872 L
.56176 .28724 L
.56113 .28726 L
.56046 .28728 L
.55934 .2873 L
.5583 .28731 L
.55767 .28732 L
.55707 .28732 L
.55596 .28731 L
.55496 .2873 L
.55391 .28729 L
.55301 .28727 L
.55202 .28724 L
.54983 .28715 L
.54778 .28705 L
.54322 .28673 L
.53901 .28635 L
.52928 .2852 L
.50972 .28235 L
.4999 .28098 L
.49105 .27997 L
.48668 .27957 L
.482 .27924 L
.47986 .27912 L
.47761 .27901 L
.47551 .27894 L
.47359 .27888 L
.4714 .27883 L
.47019 .27881 L
.46904 .2788 L
.46805 .27879 L
.46699 .27878 L
.46583 .27877 L
.46475 .27876 L
.45994 .27875 L
.45875 .27874 L
.45747 .27874 L
.45606 .27873 L
.45474 .27871 L
.45329 .27869 L
.45191 .27867 L
.44929 .27861 L
Mistroke
.44684 .27853 L
.44413 .27842 L
.44112 .27825 L
.43787 .27803 L
.43191 .27748 L
.42627 .2768 L
.42091 .27601 L
.40867 .27375 L
.36969 .26522 L
.35301 .26288 L
.34826 .26213 L
.34297 .2611 L
.33283 .25833 L
.32751 .25641 L
.32162 .2539 L
.31086 .24824 L
.29258 .23537 L
.28483 .2284 L
.27949 .22265 L
.27739 .2199 L
.27597 .21762 L
.2754 .21647 L
.27492 .21527 L
.27472 .21463 L
.27457 .21404 L
.27446 .21348 L
.27441 .21317 L
.27438 .21289 L
.27433 .21241 L
.27431 .2119 L
.27431 .21139 L
.27434 .21084 L
.27439 .21032 L
.27446 .20982 L
.27463 .20886 L
.27493 .20768 L
.27527 .20653 L
.2763 .20366 L
.28206 .1897 L
.29066 .16729 L
.29459 .15563 L
.29733 .14596 L
.29821 .14183 L
.29838 .14079 L
.29854 .13972 L
.29866 .13877 L
.29873 .13792 L
.29878 .1371 L
.29881 .13625 L
.2988 .13554 L
Mistroke
.29876 .13484 L
.29869 .13424 L
.29861 .13371 L
.29849 .13316 L
.29834 .13261 L
.29812 .13203 L
.29787 .13154 L
.29757 .13105 L
.29719 .13059 L
.29685 .13026 L
.29645 .12995 L
.29597 .12965 L
.29547 .12942 L
.29499 .12925 L
.29443 .1291 L
.29381 .12898 L
.29319 .12891 L
.29258 .12887 L
.292 .12886 L
.2913 .12888 L
.29061 .12892 L
.28987 .12898 L
.28905 .12908 L
.28748 .12931 L
.28358 .13009 L
.27456 .13225 L
.25679 .13595 L
.23857 .13846 L
.23384 .13915 L
.22886 .14004 L
.21981 .14223 L
.20177 .14841 L
.1791 .15677 L
.16783 .16067 L
.15672 .16471 L
.14839 .16847 L
.14429 .17092 L
.14092 .1735 L
.13791 .17655 L
.13553 .17982 L
.13378 .18311 L
.13229 .18706 L
.13168 .18925 L
.13113 .19177 L
.1309 .19308 L
.13069 .19452 L
.13054 .19579 L
.1304 .19719 L
.1303 .1986 L
.13022 .19993 L
Mistroke
.13018 .20121 L
.13016 .20259 L
.13016 .20395 L
.13018 .20525 L
.13022 .20642 L
.13028 .20769 L
.13037 .20909 L
.13047 .21038 L
.13076 .21326 L
.1311 .21593 L
.13149 .21844 L
.13258 .22393 L
.13408 .22962 L
.13581 .23478 L
.13792 .23972 L
.1401 .24381 L
.14225 .2471 L
.14675 .25233 L
.15199 .25672 L
.16225 .2638 L
.1753 .27395 L
.18993 .28551 L
.20693 .29571 L
.2175 .30022 L
.22752 .30361 L
.2707 .3165 L
.31076 .33043 L
.34842 .34451 L
.36525 .35424 L
.37386 .35877 L
.37811 .36058 L
.38217 .36197 L
.38645 .36305 L
.38874 .36347 L
.38993 .36364 L
.39122 .36378 L
.39245 .36389 L
.3936 .36396 L
.39469 .36399 L
.39586 .36401 L
.3969 .36399 L
.39788 .36395 L
.39895 .36389 L
.4001 .36379 L
.40143 .36365 L
.40266 .36348 L
.40543 .36299 L
.40786 .36243 L
.41044 .36172 L
.41958 .35825 L
Mistroke
.43711 .3492 L
.444 .34603 L
.44584 .34536 L
.44778 .34475 L
.44946 .34432 L
.45127 .34396 L
.45234 .34379 L
.45333 .34367 L
.45428 .34359 L
.45529 .34353 L
.45621 .34351 L
.45706 .34351 L
.45801 .34354 L
.45903 .34359 L
.46001 .34367 L
.46108 .34379 L
.46307 .34407 L
.46691 .34482 L
.47603 .34718 L
.48126 .3486 L
.48721 .35006 L
.49301 .35121 L
.49561 .35161 L
.49849 .35197 L
.49982 .35211 L
.50125 .35223 L
.50249 .35231 L
.50384 .35238 L
.50537 .35243 L
.50681 .35245 L
.50822 .35244 L
.50953 .35241 L
.51093 .35236 L
.51244 .35226 L
.51386 .35215 L
.51515 .35202 L
.51832 .35161 L
.5213 .35109 L
.52714 .34972 L
.53336 .34776 L
.54424 .34322 L
.55354 .33857 L
.56394 .33328 L
.57336 .32916 L
.58228 .32626 L
.58687 .32518 L
.59202 .32428 L
.59477 .32391 L
.59731 .32363 L
.60027 .32338 L
Mistroke
.60311 .3232 L
.60576 .32307 L
.60823 .32299 L
.60953 .32296 L
.61097 .32293 L
.61248 .32291 L
.6139 .32289 L
.61519 .32288 L
.61657 .32288 L
.61788 .32287 L
.61908 .32287 L
.62049 .32287 L
.62128 .32287 L
.62202 .32288 L
.62346 .32288 L
.62481 .32288 L
.62611 .32288 L
.62751 .32288 L
.62831 .32288 L
.62907 .32288 L
.6305 .32288 L
.63122 .32287 L
.632 .32287 L
.63339 .32286 L
.63506 .32284 L
.63656 .32282 L
.6393 .32276 L
.64183 .32269 L
.64461 .32259 L
.64754 .32245 L
.65249 .32215 L
.65791 .32171 L
.66398 .32105 L
.66969 .32028 L
.68019 .31842 L
.69104 .31582 L
.70202 .31235 L
.71936 .30466 L
.73509 .29596 L
.74227 .29259 L
.7465 .29102 L
.75078 .28977 L
.755 .28885 L
.75751 .28843 L
.75991 .28811 L
.7622 .28788 L
.76432 .28772 L
.76553 .28765 L
.76684 .28759 L
.7681 .28755 L
Mistroke
.76928 .28753 L
.7707 .28751 L
.77207 .28751 L
.77331 .28753 L
.77468 .28756 L
.77618 .2876 L
.77778 .28767 L
.78071 .28782 L
.78603 .2882 L
.79202 .28871 L
.79853 .2893 L
.80194 .2896 L
.80553 .28987 L
.80865 .29007 L
.8103 .29015 L
.8112 .29019 L
.81204 .29022 L
.81365 .29027 L
.81509 .29029 L
.81644 .29029 L
.81789 .29027 L
.81937 .29024 L
.82071 .29018 L
.82225 .29009 L
.8237 .28997 L
.82623 .28969 L
.82895 .28928 L
.83188 .28868 L
.83493 .28787 L
.84024 .28593 L
.84604 .28294 L
.85128 .27934 L
.86061 .2707 L
.86924 .26028 L
.87775 .24801 L
.89035 .226 L
.89468 .21594 L
.89557 .21327 L
.89626 .21083 L
.89675 .20869 L
.89696 .20755 L
.89713 .2065 L
.89725 .20553 L
.89733 .20465 L
.89739 .20368 L
.89742 .20266 L
.89742 .20186 L
.89739 .20099 L
.89734 .2001 L
.89725 .19927 L
Mistroke
.89712 .1983 L
.89697 .19738 L
.8966 .19569 L
.89608 .19388 L
.89538 .19193 L
.89179 .18454 L
.88648 .17562 L
.87423 .15388 L
.87244 .15034 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1Fo`<0<ol007co00@0ool0
2ol00`3oo`03o`030?oo05Co0`02o`80<Ol007co00@0ool02?l500Co0P1Co`801ol00`3oo`0^o`00
M_l400;o00@0ool02?l0103oo`05o`030?oo053o0P09o`030?oo02ko001lo`040?oo00Wo00<0o`00
1Ol00`3oo`1>o`803?l00`3oo`0]o`00Eol5027o0P0;o`801Ol00`3oo`1;o`<03ol00`3oo`0/o`00
Dol400Co00<0ool0<ol00`3oo`19o`804_l00`3oo`0/o`00Bol800So00<0ool0<ol00`3oo`17o`80
5Ol00`3oo`0[o`00AOl6013o00<0ool0<ol00`3oo`15o`805ol00`3oo`0[o`00@ol201Ko00<0ool0
<ol00`3oo`12o`<06_l00`3oo`0Zo`00@?l301So00<0ool0<ol00`3oo`11o`030?oo01co00<0ool0
:Ol003go0`0Jo`030?oo03Co00<0ool0@?l00`3oo`0Mo`030?oo02Wo000jo`<07Ol00`3oo`0do`03
0?oo03oo00<0ool07_l00`3oo`0Yo`00=ol3023o00<0ool0=?l00`3oo`0oo`030?oo01oo00<0ool0
:?l003Go0P0Ro`030?oo03Go00<0ool0?_l00`3oo`0Po`030?oo02So000co`809?l00`3oo`0eo`03
0?oo03ko00<0ool08Ol00`3oo`0Wo`00<_l00`3oo`0To`030?oo03Go00<0ool0?_l00`3oo`0Qo`03
0?oo02Oo000`o`809_l00`3oo`0Ro`801?l200?o1005o`030?oo03go00<0ool08ol00`3oo`0Vo`00
<?l00`3oo`0Uo`030?oo027o00@0ool02?l00`3oo`06o`030?oo03go00<0ool09?l00`3oo`0Uo`00
;ol00`3oo`0Vo`030?oo027o00@0ool02Ol00`3oo`05o`80?_l00`3oo`0To`030?oo02Go000_o`03
0?oo02Go00<0ool07?l400;o00@0ool02_l00`3oo`04o`030?oo03go00<0ool09Ol00`3oo`0To`00
;ol00`3oo`0Uo`030?oo02;o00@0ool02?l0103oo`05o`030?oo03go00<0ool09Ol00`3oo`0To`00
;_l00`3oo`0Vo`030?oo02?o0P0:o`801_l00`3oo`0mo`030?oo02Ko00<0ool08ol002ko00<0ool0
9Ol00`3oo`0ho`030?oo03go00<0ool09_l00`3oo`0So`00;_l00`3oo`0Uo`030?oo03So00<0ool0
?Ol00`3oo`0Vo`030?oo02?o000^o`030?oo02Co00<0ool0>Ol00`3oo`0mo`030?oo02Ko00<0ool0
8ol000Oo1@02o`801?l201So00<0ool02?l501Ko0P0:o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200go00<0ool04ol500ko00<0ool04Ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`03
0?oo00[o00<0ool05_l200Wo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`03
0?oo01Go00<0ool03_l00`3oo`0Co`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`03
0?oo00[o00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
2_l00`3oo`0Eo`030?oo00go00<0ool05?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Ho`030?oo00;o1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol200go00<0ool05Ol00`3oo`0=o`030?oo01Co00<0ool02?l200?o0009o`030?oo00So00<0ool0
6?l00`3oo`09o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo00co00<0ool05Ol00`3oo`0<o`030?oo01Go00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`03
0?oo00So0P0Ko`030?oo00Ko0P0:o`<08_l00`3oo`0Ro`802_l300_o0P0Fo`803_l00`3oo`0Do`80
2_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0ko`809_l00`3oo`0Vo`00<?l00`3oo`0Vo`03
0?oo03Go00<0ool0>_l202Oo00<0ool09_l0033o00<0ool09ol00`3oo`0do`030?oo03?o1`0Xo`03
0?oo02Oo000ao`030?oo02Oo0P0do`030?oo02[o2@0_o`030?oo02Oo000ao`030?oo02Wo00<0ool0
<Ol00`3oo`0Wo`<0=ol00`3oo`0Xo`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`02o`<0
00?o0?l01ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3o0006o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Go
00D0oooo000Zo`<0:ol00`3oo`0Ro`030?oo00Co00<0ool0:Ol00`3oo`08o`030?oo01oo00<0ool0
1ol003Go0P0_o`H09Ol00`3oo`0Po`80?Ol00`3oo`0Zo`00=ol00`3oo`0bo`D08?l00`3oo`0No`80
?_l00`3oo`0[o`00>?l203Oo1@0Ko`030?oo01go00<0ool0?Ol00`3oo`0/o`00>_l00`3oo`0io`@0
5ol00`3oo`0Ko`80?ol00`3oo`0]o`00>ol00`3oo`0lo`L04?l00`3oo`0Io`80@?l00`3oo`0^o`00
??l204?o4@0Jo`030?oo03oo00<0ool0;ol003ko00<0ool0D?l900co1P10o`80<_l003oo0P1@o`03
0?oo00Ko300^o`d01ol403Co0011o`80C_l00`3oo`0lo`@03Ol703So0013o`030?oo04_o00<0ool0
>ol00`3oo`1>o`00A?l304[o00<0ool0>Ol2057o0017o`<0Aol00`3oo`0ho`030?oo057o001:o`<0
A?l00`3oo`0fo`80E?l004go0`0]o`801?l200?o1005o`030?oo03Co0P1Fo`00D?l302Wo00@0ool0
2?l00`3oo`06o`030?oo037o0`1Ho`00Dol302Ko00@0ool02Ol00`3oo`05o`80;?l605_o001Fo`<0
8ol0103oo`0:o`030?oo00Co00<0ool06?lC067o001Io`@07ol0103oo`08o`040?oo00Go00<0ool0
4Ol707Co001Mo`<07Ol200[o0P06o`030?oo00oo0P1ko`00H?l202oo00<0ool03_l00`3oo`1ko`00
H_l302co00<0ool03?l207ko001Uo`80:_l00`3oo`0:o`80P?l006Oo0P0Io`L02?l00`3oo`07o`<0
P_l006Wo0P0Eo`<01_l400Co00<0ool01Ol208Go001[o`804Ol200go3027o`00KOl00`3oo`0<o`80
4ol00`3oo`2<o`00K_l300Wo0P0Eo`030?oo08co001ao`T05ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0
S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool0
2Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool0
1Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o
0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87315 .15211 m
.86854 .14262 L
.8633 .13296 L
.86052 .12864 L
.85742 .12455 L
.85436 .12132 L
.85137 .11889 L
.84958 .11775 L
.84762 .11675 L
.84559 .11595 L
.84441 .11559 L
.84328 .11532 L
.84221 .11512 L
.84119 .11497 L
.84026 .11488 L
.83923 .11481 L
.83864 .1148 L
.8381 .11479 L
.83747 .11479 L
.83688 .11481 L
.8358 .11487 L
.8352 .11492 L
.83463 .11498 L
.83257 .11525 L
.8306 .11561 L
.82601 .11674 L
.81632 .12002 L
.79819 .12736 L
.7793 .13543 L
.77016 .14033 L
.76653 .14281 L
.76284 .14587 L
.75991 .14889 L
.75711 .15252 L
.75455 .1568 L
.75255 .16125 L
.75093 .16601 L
.74972 .17075 L
.7487 .17622 L
.74789 .18229 L
.74675 .19487 L
.74625 .20029 L
.74592 .2033 L
.74556 .20605 L
.74465 .21116 L
.74407 .21363 L
.74333 .21618 L
.74237 .21882 L
.7413 .22118 L
.74019 .2232 L
Mistroke
.73886 .22521 L
.73598 .22853 L
.7328 .23119 L
.7284 .23386 L
.72387 .23586 L
.719 .23749 L
.71346 .23892 L
.70288 .24098 L
.69101 .24296 L
.67868 .24533 L
.66779 .24825 L
.65872 .25159 L
.64253 .25974 L
.62398 .26964 L
.61512 .27352 L
.60487 .27714 L
.5948 .27988 L
.58538 .28181 L
.57565 .28324 L
.57033 .2838 L
.56556 .28419 L
.56101 .28446 L
.55849 .28457 L
.55622 .28465 L
.55411 .28471 L
.55292 .28473 L
.55183 .28474 L
.55088 .28475 L
.54983 .28476 L
.54874 .28476 L
.54772 .28476 L
.54653 .28476 L
.54585 .28475 L
.54523 .28475 L
.54408 .28473 L
.54286 .28471 L
.54075 .28466 L
.53841 .28459 L
.53321 .28437 L
.52825 .2841 L
.50848 .2826 L
.49011 .28108 L
.48053 .28047 L
.4717 .28007 L
.46662 .2799 L
.46198 .27977 L
.45687 .27964 L
.45137 .27948 L
.44654 .27929 L
.44113 .27901 L
Mistroke
.43611 .27867 L
.43134 .27825 L
.42042 .27694 L
.40861 .27498 L
.36773 .26594 L
.34951 .26221 L
.3403 .25996 L
.33159 .25725 L
.32214 .25349 L
.31316 .24906 L
.29467 .23704 L
.28703 .23071 L
.28061 .22436 L
.27669 .21939 L
.27515 .21675 L
.27462 .21558 L
.27417 .21436 L
.27398 .21373 L
.2738 .21304 L
.27368 .21241 L
.27359 .21182 L
.27355 .21149 L
.27352 .21114 L
.2735 .21079 L
.27348 .2104 L
.27348 .21006 L
.27348 .2097 L
.27352 .20904 L
.27358 .20837 L
.27363 .20799 L
.27369 .20763 L
.27397 .20627 L
.27432 .20499 L
.27479 .20359 L
.27729 .19795 L
.28124 .19058 L
.29186 .17069 L
.29648 .16058 L
.29831 .15586 L
.29963 .15183 L
.30016 .1499 L
.30062 .14792 L
.30082 .14687 L
.30097 .14593 L
.30108 .14507 L
.30116 .14419 L
.30121 .14332 L
.30122 .14286 L
.30122 .14244 L
.30119 .14165 L
Mistroke
.30112 .14094 L
.30107 .14055 L
.30101 .14015 L
.30086 .13946 L
.30065 .13876 L
.30037 .13803 L
.30009 .13745 L
.29978 .13694 L
.2991 .13606 L
.29823 .13525 L
.2971 .1345 L
.29599 .13398 L
.29532 .13373 L
.29466 .13353 L
.29402 .13337 L
.29328 .13322 L
.2926 .1331 L
.29193 .13301 L
.29109 .13292 L
.2903 .13286 L
.28942 .13282 L
.2889 .13281 L
.28842 .1328 L
.28761 .1328 L
.28669 .13281 L
.28571 .13284 L
.28476 .13288 L
.28368 .13295 L
.28253 .13303 L
.2804 .13321 L
.27538 .13375 L
.25465 .13626 L
.24473 .13732 L
.23569 .13835 L
.22775 .13951 L
.21911 .14122 L
.20949 .14372 L
.20017 .14659 L
.17816 .15413 L
.16784 .15791 L
.15741 .1622 L
.14868 .16676 L
.14197 .17172 L
.13897 .17476 L
.13657 .17786 L
.13435 .18161 L
.13338 .18366 L
.13247 .18598 L
.13172 .1883 L
.13114 .19051 L
Mistroke
.13068 .19265 L
.13028 .19506 L
.13014 .19612 L
.13001 .19727 L
.12992 .19828 L
.12983 .19938 L
.12976 .20063 L
.12971 .20183 L
.12969 .203 L
.12968 .20409 L
.12969 .20534 L
.12973 .20667 L
.12978 .2078 L
.12985 .20904 L
.12995 .21039 L
.13009 .21182 L
.13039 .2144 L
.13121 .21943 L
.1322 .22387 L
.13364 .22893 L
.13551 .23407 L
.13957 .24236 L
.14407 .24893 L
.14956 .25496 L
.16195 .26539 L
.17621 .27653 L
.19152 .28741 L
.20005 .29241 L
.20997 .2972 L
.22915 .30413 L
.27112 .31724 L
.31058 .33088 L
.3455 .34291 L
.36378 .35112 L
.37207 .35452 L
.37659 .35608 L
.38153 .35747 L
.38407 .35805 L
.38649 .35852 L
.38887 .35888 L
.39108 .35914 L
.39318 .35932 L
.39425 .35938 L
.39543 .35943 L
.39661 .35946 L
.39788 .35947 L
.39909 .35945 L
.40019 .35941 L
.40142 .35935 L
.40274 .35926 L
Mistroke
.4051 .35904 L
.4079 .35867 L
.41044 .35825 L
.41998 .356 L
.4296 .35301 L
.43926 .34994 L
.44311 .34888 L
.44719 .34796 L
.44952 .34755 L
.45164 .34724 L
.45266 .34712 L
.45377 .34701 L
.45483 .34693 L
.4558 .34687 L
.45679 .34682 L
.45785 .34679 L
.4588 .34678 L
.4598 .34678 L
.46091 .3468 L
.46194 .34684 L
.46308 .3469 L
.46431 .34697 L
.46682 .34719 L
.46922 .34746 L
.47384 .3481 L
.48253 .34949 L
.48735 .35023 L
.49267 .35095 L
.4955 .35126 L
.49812 .3515 L
.49938 .35159 L
.50077 .35168 L
.50207 .35176 L
.5033 .35181 L
.50403 .35183 L
.50481 .35186 L
.50618 .35188 L
.50696 .35189 L
.50779 .35189 L
.50926 .35187 L
.51063 .35184 L
.51191 .35179 L
.51331 .35172 L
.51481 .35162 L
.51747 .35139 L
.52035 .35105 L
.52555 .35023 L
.53055 .34917 L
.5351 .34799 L
.5543 .34116 L
Mistroke
.57419 .33318 L
.58372 .33004 L
.59293 .32767 L
.60383 .32562 L
.61409 .32424 L
.63534 .32203 L
.65515 .31965 L
.66625 .31792 L
.67664 .31596 L
.69849 .31055 L
.71719 .30424 L
.7327 .29809 L
.74021 .29526 L
.7486 .2926 L
.75367 .29131 L
.75851 .29032 L
.76359 .28951 L
.76861 .28892 L
.77352 .28851 L
.77638 .28834 L
.77911 .28821 L
.78201 .28811 L
.7847 .28804 L
.78612 .28802 L
.78768 .28799 L
.79083 .28795 L
.7962 .28791 L
.79774 .28789 L
.7992 .28787 L
.80053 .28785 L
.80193 .28783 L
.8045 .28777 L
.80686 .28769 L
.80938 .28757 L
.81214 .2874 L
.81494 .28717 L
.81796 .28684 L
.82078 .28644 L
.82336 .286 L
.82812 .28491 L
.83304 .28338 L
.83864 .28102 L
.84372 .27822 L
.85273 .27144 L
.86105 .263 L
.86922 .25267 L
.8817 .2328 L
.88651 .22299 L
.88854 .21789 L
.89002 .21329 L
Mistroke
.89053 .21128 L
.89098 .20918 L
.89117 .20811 L
.89134 .20694 L
.89147 .20587 L
.89156 .2049 L
.89162 .20401 L
.89166 .20308 L
.89168 .20216 L
.89168 .20115 L
.89165 .20028 L
.8916 .19936 L
.89152 .19849 L
.89144 .1977 L
.89118 .19583 L
.89079 .19377 L
.88968 .18958 L
.8868 .18176 L
.87764 .16169 L
.87315 .15211 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0E?l503?o001lo`040?oo00So1@04o`80DOl400Go0P0ao`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04ko0P0;o`030?oo02ko001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`80
3_l00`3oo`0]o`00OOl200_o0P05o`030?oo04[o0P0@o`030?oo02go001Co`T0=Ol00`3oo`18o`80
4ol00`3oo`0/o`00B_l900So00<0ool0<ol00`3oo`16o`805_l00`3oo`0[o`00AOl501;o00<0ool0
<_l00`3oo`13o`<06Ol00`3oo`0Zo`00@?l501Oo00<0ool0<_l00`3oo`12o`030?oo01[o00<0ool0
:_l003go0`0Ko`80=?l00`3oo`12o`030?oo01_o00<0ool0:Ol003_o0P0No`030?oo03?o00<0ool0
@Ol00`3oo`0Lo`030?oo02Wo000io`808?l00`3oo`0co`030?oo043o00<0ool07Ol00`3oo`0Yo`00
=ol2027o00<0ool0=?l00`3oo`10o`030?oo01ko00<0ool0:?l003Go0P0So`030?oo03Co00<0ool0
?ol00`3oo`0Oo`030?oo02So000co`809?l00`3oo`0eo`030?oo03oo00<0ool07ol00`3oo`0Xo`00
<Ol202Ko00<0ool0=Ol00`3oo`0oo`030?oo023o00<0ool09ol0033o00<0ool09Ol00`3oo`0Ro`80
1?l200?o1005o`030?oo03oo00<0ool08?l00`3oo`0Wo`00<?l00`3oo`0Uo`030?oo027o00@0ool0
2?l00`3oo`06o`030?oo03ko00<0ool08_l00`3oo`0Vo`00;ol00`3oo`0Uo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80?ol00`3oo`0Ro`030?oo02Ko000_o`030?oo02Go00<0ool07?l400;o00@0ool0
2_l00`3oo`04o`030?oo03ko00<0ool08_l00`3oo`0Vo`00;_l202Ko00<0ool08ol0103oo`08o`04
0?oo00Go00<0ool0?_l00`3oo`0So`030?oo02Go000^o`030?oo02Go00<0ool09?l200[o0P06o`03
0?oo03ko00<0ool08ol00`3oo`0Uo`00;_l00`3oo`0To`030?oo03Wo00<0ool0?_l00`3oo`0To`03
0?oo02Co000^o`030?oo02Co00<0ool0>Ol00`3oo`0no`030?oo02Co00<0ool09?l002ko00<0ool0
8ol203_o00<0ool0?_l00`3oo`0To`030?oo02Co0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D03Ol00`3oo`0Bo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`805_l00`3oo`0<o`030?oo01Go00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`802Ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`0<o`030?oo01Go00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3oo`0Go`040?ooo`@0
0_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0=o`030?oo01Go00<0ool03?l00`3oo`0Eo`03
0?oo00So0P03o`002Ol00`3oo`08o`030?oo01So00<0ool02Ol00`3oo`0Ho`030?oo00Ko00@0ool0
2Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0;o`030?oo01Ko00<0ool02ol00`3oo`0Fo`03
0?oo00So00<0ool00_l000So0P0:o`<06?l00`3oo`08o`806ol00`3oo`06o`802_l302;o00<0ool0
8_l200[o0`0:o`030?oo01Ko0P0=o`030?oo01Go0P0:o`<00_l0033o00<0ool09Ol00`3oo`0fo`03
0?oo03[o00<0ool09Ol00`3oo`0Wo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0>?l202So00<0ool0
9ol0037o00<0ool09_l00`3oo`0do`030?oo03Co100Yo`030?oo02So000ao`030?oo02Oo0P0do`03
0?oo02ko1P0]o`030?oo02So000bo`030?oo02So0P0bo`030?oo02_o0`0bo`030?oo02Wo000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`02o`800_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01ol000co00<0ool09Ol01@3oool002[o0`0[o`030?oo02?o0P04o`03
0?oo02Wo00<0ool01_l00`3oo`0Qo`030?oo00Oo000eo`80;ol502Ko00<0ool08Ol203[o00<0ool0
;?l003Oo00<0ool0<Ol402;o00<0ool07ol203_o00<0ool0;Ol003So00<0ool0=?l401ko00<0ool0
7Ol203co00<0ool0;_l003Wo00<0ool0=ol401[o00<0ool06_l303go00<0ool0;ol003[o00<0ool0
>_l601Co00<0ool06?l203oo00<0ool0<?l003_o00<0ool0?olE01Go1@0no`<0<ol003co0P1CoaH0
@Ol203Ko000no`80DOl00`3oo`14oa00>?l0043o0P1?o`030?oo03ko1P18o`00@_l00`3oo`1<o`03
0?oo03_o0`1>o`00@ol304_o00<0ool0>?l3057o0016o`@0Aol00`3oo`0eo`<0E?l004[o0`14o`03
0?oo03?o0P1Go`00COl302go0P04o`800ol400Go00<0ool0<?l305Wo001@o`<0:Ol0103oo`08o`03
0?oo00Ko00<0ool0;Ol305co001Co`<09_l0103oo`09o`030?oo00Go0P0Yo`D0Gol005Ko0P0To`04
0?oo00[o00<0ool01?l00`3oo`0No`X0I?l005So0`0Qo`040?oo00So00@0ool01Ol00`3oo`0Go`L0
K_l005_o0P0Po`802_l200Ko00<0ool04_l507Go001Mo`<0<Ol00`3oo`0?o`<0N_l0063o0`0^o`03
0?oo00go0P1mo`00Hol302_o00<0ool02_l307oo001Vo`<0:?l00`3oo`08o`80P_l006Wo0P0Fo``0
1?l00`3oo`04o`@0Q?l006_o0P0@o`@03?l;08So001]o`@02Ol301Co00<0ool0S?l0077o2@0Go`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`03
0?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87385 .15388 m
.86962 .14504 L
.86442 .13606 L
.86154 .13199 L
.85825 .12809 L
.85497 .12491 L
.85175 .12242 L
.8477 .12004 L
.84546 .11904 L
.84301 .11817 L
.84082 .11758 L
.83872 .11715 L
.83749 .11696 L
.83636 .11682 L
.83573 .11675 L
.83503 .11669 L
.83442 .11665 L
.83376 .11661 L
.83312 .11659 L
.83252 .11657 L
.83135 .11656 L
.83027 .11657 L
.82912 .11661 L
.82852 .11664 L
.82788 .11668 L
.82671 .11677 L
.82401 .11705 L
.8215 .1174 L
.81909 .11782 L
.81453 .11878 L
.80493 .12141 L
.79496 .12492 L
.78615 .12879 L
.77878 .13289 L
.7719 .13798 L
.76873 .14102 L
.76574 .14451 L
.76149 .15115 L
.75952 .15532 L
.75793 .15945 L
.75507 .16968 L
.753 .17999 L
.75127 .18932 L
.75019 .19454 L
.74904 .19929 L
.746 .20852 L
.74401 .21285 L
.74148 .21717 L
.73876 .22085 L
.73582 .22406 L
Mistroke
.72941 .22935 L
.72123 .23414 L
.71149 .23837 L
.69273 .24467 L
.65234 .2588 L
.63278 .26646 L
.61388 .27272 L
.59468 .27735 L
.58499 .27904 L
.57447 .28046 L
.56421 .28149 L
.55495 .28217 L
.54984 .28246 L
.5443 .28271 L
.53905 .28289 L
.53651 .28296 L
.53418 .28301 L
.53192 .28305 L
.52953 .28308 L
.52829 .28309 L
.52714 .2831 L
.52587 .28311 L
.52516 .28311 L
.52451 .28311 L
.52334 .28311 L
.52226 .28311 L
.52108 .2831 L
.51982 .2831 L
.51873 .28309 L
.51771 .28308 L
.51539 .28305 L
.51269 .283 L
.51014 .28295 L
.50534 .28282 L
.49457 .28242 L
.45619 .28061 L
.44588 .28008 L
.43607 .27944 L
.42574 .27851 L
.41428 .27708 L
.40416 .27541 L
.39368 .2733 L
.37566 .26901 L
.34047 .25957 L
.32221 .25287 L
.31196 .24793 L
.30287 .24266 L
.28723 .2312 L
.2803 .22449 L
.27784 .22154 L
Mistroke
.27567 .21841 L
.2748 .21685 L
.27409 .21538 L
.27353 .21395 L
.27313 .21263 L
.27285 .21138 L
.27274 .21068 L
.27266 .21005 L
.27261 .20935 L
.27259 .2086 L
.2726 .20788 L
.27264 .20722 L
.27272 .20649 L
.27283 .20572 L
.27298 .20498 L
.27313 .2043 L
.27363 .20262 L
.27425 .201 L
.27586 .19769 L
.27814 .1939 L
.28897 .17898 L
.29503 .17031 L
.29785 .16573 L
.30034 .16102 L
.30204 .15701 L
.30275 .15483 L
.30303 .15383 L
.30324 .15289 L
.3034 .15206 L
.30353 .15117 L
.30362 .15039 L
.30366 .14968 L
.30368 .14893 L
.30365 .14813 L
.30362 .14773 L
.30358 .14732 L
.30346 .14658 L
.30331 .14586 L
.30312 .14521 L
.30285 .14446 L
.30256 .14382 L
.30191 .14269 L
.30105 .14158 L
.30014 .14067 L
.29901 .13978 L
.2966 .1384 L
.2951 .13778 L
.29337 .13721 L
.29177 .13678 L
.28993 .1364 L
Mistroke
.28765 .13603 L
.28656 .13589 L
.28537 .13577 L
.28429 .13567 L
.28307 .13558 L
.28203 .13551 L
.28088 .13546 L
.2802 .13543 L
.27956 .13541 L
.2789 .13539 L
.2783 .13538 L
.27713 .13536 L
.27588 .13535 L
.27517 .13535 L
.27451 .13536 L
.27379 .13536 L
.27302 .13537 L
.27172 .1354 L
.27029 .13543 L
.26773 .13552 L
.26496 .13563 L
.25995 .1359 L
.24967 .1366 L
.23876 .13755 L
.2287 .13871 L
.21944 .14013 L
.21013 .14196 L
.2012 .14411 L
.18026 .15043 L
.17036 .15407 L
.16041 .15841 L
.15261 .16265 L
.14508 .16802 L
.14151 .17135 L
.13852 .17477 L
.13615 .17813 L
.13401 .18194 L
.13212 .18637 L
.13079 .19071 L
.13022 .19321 L
.1298 .19559 L
.12945 .19826 L
.12932 .19978 L
.12927 .20045 L
.12922 .20118 L
.12917 .20247 L
.12914 .20368 L
.12914 .20487 L
.12915 .20597 L
.12919 .20716 L
Mistroke
.12925 .20847 L
.12935 .20983 L
.12946 .21111 L
.12973 .21345 L
.13004 .21557 L
.13096 .22028 L
.13227 .22524 L
.13404 .23044 L
.13806 .23922 L
.14257 .24656 L
.14819 .25377 L
.16114 .26651 L
.17605 .27833 L
.19173 .28848 L
.20893 .2969 L
.22822 .30385 L
.2657 .31592 L
.30527 .32971 L
.34198 .3412 L
.36029 .34736 L
.37017 .35037 L
.37569 .35181 L
.38093 .35297 L
.38566 .35381 L
.39079 .3545 L
.39374 .35478 L
.39519 .35489 L
.39654 .35497 L
.39773 .35503 L
.39906 .35509 L
.40039 .35512 L
.40111 .35514 L
.4018 .35515 L
.40293 .35515 L
.40417 .35515 L
.40546 .35513 L
.40667 .3551 L
.4078 .35506 L
.40882 .35502 L
.41112 .35489 L
.41359 .35472 L
.41625 .35448 L
.42106 .35397 L
.43989 .3515 L
.44493 .35093 L
.4477 .35067 L
.45033 .35047 L
.45277 .35031 L
.45406 .35024 L
.45544 .35018 L
Mistroke
.45671 .35014 L
.45787 .3501 L
.45904 .35008 L
.46014 .35006 L
.46126 .35006 L
.46232 .35005 L
.46348 .35006 L
.46473 .35007 L
.46535 .35008 L
.46604 .35009 L
.46729 .35012 L
.46964 .3502 L
.47427 .35039 L
.47863 .35062 L
.48379 .35092 L
.48946 .35122 L
.49209 .35134 L
.49489 .35144 L
.49729 .35151 L
.49865 .35154 L
.49993 .35156 L
.50118 .35157 L
.50253 .35158 L
.50382 .35157 L
.505 .35156 L
.50607 .35155 L
.50725 .35152 L
.50848 .35149 L
.50964 .35145 L
.51226 .35133 L
.51466 .35118 L
.52005 .35073 L
.52548 .35007 L
.53135 .34915 L
.54186 .34696 L
.56176 .34131 L
.58224 .33476 L
.62317 .32406 L
.66437 .31544 L
.70393 .30627 L
.74222 .29613 L
.78104 .28884 L
.79185 .28748 L
.80212 .28613 L
.81121 .28464 L
.82079 .28244 L
.82564 .28096 L
.83065 .27908 L
.83927 .27479 L
.84787 .26887 L
Mistroke
.85668 .26076 L
.87024 .2435 L
.87643 .23309 L
.88092 .22367 L
.88261 .21929 L
.88407 .21471 L
.88511 .21048 L
.88546 .20864 L
.88577 .2067 L
.88591 .20561 L
.88604 .20441 L
.88614 .2033 L
.8862 .2023 L
.88623 .20169 L
.88626 .20112 L
.88627 .20051 L
.88628 .19985 L
.88629 .19927 L
.88628 .19866 L
.88626 .19756 L
.88622 .19654 L
.88615 .19545 L
.88607 .19441 L
.88598 .19345 L
.88571 .19123 L
.88535 .18886 L
.88449 .18461 L
.88351 .18065 L
.88069 .17141 L
.87385 .15388 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1Ao`X0<_l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1=o`<02_l00`3oo`0_o`00O?l0103oo`09o`030?l000Go00<0ool0Bol200ko00<0ool0
;_l007go0P0;o`801Ol00`3oo`19o`804Ol00`3oo`0]o`00TOl00`3oo`17o`805?l00`3oo`0/o`00
B?lC03Ko00<0ool0AOl201Ko00<0ool0;?l004?o1@0Co`80=?l00`3oo`14o`030?oo01Oo00<0ool0
:ol0043o0`0Io`030?oo03;o00<0ool0@ol00`3oo`0Io`030?oo02[o000mo`<07?l00`3oo`0bo`03
0?oo04;o00<0ool06_l00`3oo`0Zo`00>Ol401oo00<0ool0<_l00`3oo`12o`030?oo01_o00<0ool0
:Ol003Oo0P0So`030?oo03;o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000fo`030?oo02;o00<0ool0
<ol00`3oo`11o`030?oo01go00<0ool0:?l003Co0P0Uo`030?oo03?o00<0ool0@Ol00`3oo`0Mo`03
0?oo02So000bo`809ol00`3oo`0co`030?oo043o00<0ool07ol00`3oo`0Wo`00<Ol00`3oo`0Vo`03
0?oo03Co00<0ool0@?l00`3oo`0Oo`030?oo02Oo000`o`030?oo02Ko00<0ool08Ol200Co0P03o`@0
1Ol00`3oo`10o`030?oo01oo00<0ool09ol0033o00<0ool09Ol00`3oo`0Qo`040?oo00So00<0ool0
1_l00`3oo`10o`030?oo023o00<0ool09_l002oo00<0ool09Ol00`3oo`0Ro`040?oo00Wo00<0ool0
1Ol2047o00<0ool08?l00`3oo`0Vo`00;ol00`3oo`0Uo`030?oo01co1002o`040?oo00[o00<0ool0
1?l00`3oo`0oo`030?oo027o00<0ool09_l002ko00<0ool09Ol00`3oo`0So`040?oo00So00@0ool0
1Ol00`3oo`0oo`030?oo027o00<0ool09_l002ko00<0ool09?l00`3oo`0Uo`802_l200Ko00<0ool0
?ol00`3oo`0Qo`030?oo02Ko000^o`030?oo02Co00<0ool0>Ol00`3oo`0oo`030?oo027o00<0ool0
9_l002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08Ol00`3oo`0Vo`00;_l00`3oo`0So`03
0?oo03[o00<0ool0?_l00`3oo`0Ro`030?oo02Ko0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02ol00`3oo`0Do`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02ol00`3oo`0Fo`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l200Wo00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02ol00`3oo`0Fo`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05ol0103o
ool400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`803?l00`3oo`0Fo`030?oo00[o00<0ool0
5ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool06?l00`3oo`06o`04
0?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00[o00<0ool0
5ol00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`03
0?oo02;o0P0:o`<02Ol00`3oo`0Go`803?l00`3oo`0Fo`802_l300;o000_o`030?oo02Ko00<0ool0
=_l00`3oo`0io`809_l00`3oo`0Xo`00<?l00`3oo`0Vo`80=_l00`3oo`0go`80:?l00`3oo`0Xo`00
<?l00`3oo`0Xo`030?oo03?o00<0ool0=?l302Wo00<0ool0:Ol0037o00<0ool0:?l203?o00<0ool0
<_l202_o00<0ool0:_l0037o00<0ool0:_l00`3oo`0`o`030?oo02oo0`0/o`030?oo02_o000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`04o`@01ol00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Oo000<o`030?oo02Go00D0oooo000Zo`80;?l00`3oo`0To`<00_l00`3oo`0Yo`03
0?oo00Co00<0ool08ol00`3oo`07o`00=Ol00`3oo`0]o`@0:?l00`3oo`0Ro`80=ol00`3oo`0^o`00
=_l00`3oo`0`o`@09?l00`3oo`0Oo`<0>?l00`3oo`0_o`00=ol00`3oo`0co`D07ol00`3oo`0Mo`80
>_l00`3oo`0`o`00>?l00`3oo`0go`@06ol00`3oo`0Io`@0>ol00`3oo`0ao`00>Ol00`3oo`0jo`H0
5Ol00`3oo`0Eo`@0?Ol203Co000jo`80@?l?00Ko00<0ool03Ol803ko0`0fo`00??l204go5P11o`D0
>Ol003ko0P1Ao`030?oo04Go2@0no`00@?l204oo00<0ool0@?l504Oo0012o`<0C?l00`3oo`0lo`@0
C?l004Go0P1:o`030?oo03So101@o`00Aol304Oo00<0ool0=Ol305Co001:o`80AOl00`3oo`0ao`@0
Eol004co0`0^o`801?l200?o1005o`030?oo02_o1P1Ko`00Col202_o00@0ool02?l00`3oo`06o`03
0?oo02Oo101Qo`00DOl302So00@0ool02Ol00`3oo`05o`809?l406Go001Do`@09?l0103oo`0:o`03
0?oo00Co00<0ool07ol406Wo001Ho`@08?l0103oo`08o`040?oo00Go00<0ool07?l306go001Lo`<0
7_l200[o0P06o`030?oo01So101`o`00Gol302oo00<0ool05?l407Co001Ro`<0;?l00`3oo`0@o`@0
N?l006Go0`0Yo`030?oo00co101lo`00J?l202Oo00<0ool02_l2083o001Zo`<09?l00`3oo`04o`H0
P_l006go1@0=oaT0R?l007;o3@0Bo`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87455 .15565 m
.8707 .14745 L
.86554 .13915 L
.86255 .13535 L
.85908 .13163 L
.85212 .12594 L
.84777 .12333 L
.84274 .12103 L
.84034 .12016 L
.83772 .11936 L
.83541 .11877 L
.83288 .11825 L
.8304 .11784 L
.82809 .11755 L
.82697 .11744 L
.82574 .11734 L
.82457 .11727 L
.82348 .11722 L
.82281 .11719 L
.82209 .11717 L
.82078 .11716 L
.8195 .11716 L
.81831 .11719 L
.81703 .11723 L
.81629 .11727 L
.81561 .11731 L
.81422 .1174 L
.81275 .11753 L
.81009 .11783 L
.80763 .11818 L
.80213 .11925 L
.79749 .12046 L
.79255 .12213 L
.78838 .12389 L
.78415 .1261 L
.77735 .13093 L
.77411 .13405 L
.77145 .13726 L
.7689 .14108 L
.76661 .14551 L
.76463 .15051 L
.7631 .1555 L
.76086 .16535 L
.75916 .17427 L
.75713 .18376 L
.75569 .18903 L
.75413 .19369 L
.75022 .20228 L
.74781 .20628 L
.74483 .21041 L
Mistroke
.73855 .21733 L
.72207 .23023 L
.70365 .24134 L
.68663 .25 L
.66821 .25744 L
.64984 .26301 L
.61099 .27174 L
.59091 .27528 L
.56941 .2782 L
.54883 .28032 L
.52953 .28193 L
.52429 .2823 L
.51861 .28265 L
.51364 .2829 L
.51092 .28302 L
.50838 .28311 L
.50592 .28319 L
.50367 .28324 L
.50117 .28329 L
.49986 .28331 L
.49914 .28332 L
.49848 .28332 L
.49739 .28333 L
.49622 .28333 L
.49514 .28333 L
.49412 .28333 L
.49293 .28333 L
.49183 .28332 L
.49063 .2833 L
.48936 .28328 L
.48676 .28323 L
.48429 .28317 L
.47966 .28301 L
.47467 .28279 L
.46922 .28251 L
.45036 .28128 L
.43075 .27979 L
.42055 .27884 L
.40969 .27753 L
.39945 .2759 L
.39006 .274 L
.37157 .26921 L
.35185 .26308 L
.33423 .25697 L
.31695 .24989 L
.3011 .24147 L
.29307 .23613 L
.28506 .2297 L
.27892 .22343 L
.27651 .22031 L
Mistroke
.27468 .21742 L
.27314 .21424 L
.27252 .21248 L
.27229 .21167 L
.27208 .21081 L
.27193 .21001 L
.27182 .20928 L
.27173 .20854 L
.27169 .20785 L
.27166 .20705 L
.27168 .20631 L
.27173 .2055 L
.27182 .20464 L
.27197 .20377 L
.27213 .20298 L
.27264 .2012 L
.27326 .19956 L
.27399 .198 L
.27608 .19454 L
.27897 .19078 L
.29178 .17773 L
.29844 .17073 L
.30136 .16708 L
.30378 .16329 L
.30458 .16169 L
.30526 .16 L
.30557 .15904 L
.30579 .15819 L
.30597 .15733 L
.30608 .15653 L
.30615 .15582 L
.30617 .15507 L
.30615 .15443 L
.30609 .15375 L
.30599 .15302 L
.30591 .15262 L
.30582 .15225 L
.30563 .15159 L
.30538 .1509 L
.30478 .14961 L
.30407 .14845 L
.30306 .14716 L
.30194 .14601 L
.29929 .14393 L
.29642 .14226 L
.29293 .14072 L
.28867 .13931 L
.28383 .13815 L
.2814 .13769 L
.27905 .13732 L
Mistroke
.27684 .13703 L
.27439 .13676 L
.27234 .13658 L
.27006 .13642 L
.26757 .13629 L
.2662 .13623 L
.26492 .13619 L
.26372 .13617 L
.26262 .13615 L
.26135 .13614 L
.26015 .13614 L
.25885 .13614 L
.25812 .13615 L
.25745 .13616 L
.25622 .13618 L
.25492 .13621 L
.25266 .13628 L
.25019 .13639 L
.24475 .13669 L
.23971 .13706 L
.23073 .1379 L
.22119 .13901 L
.21102 .14047 L
.19995 .14245 L
.18952 .14483 L
.17998 .14759 L
.17074 .15098 L
.16244 .15481 L
.15396 .15983 L
.14581 .16624 L
.13962 .17286 L
.13666 .17697 L
.13428 .18101 L
.1322 .1855 L
.1307 .18974 L
.13004 .19214 L
.12946 .19479 L
.12904 .19733 L
.12888 .19854 L
.12876 .19969 L
.12867 .20078 L
.1286 .20198 L
.12855 .20307 L
.12853 .20408 L
.12853 .20529 L
.12855 .20639 L
.12857 .20699 L
.1286 .20765 L
.12868 .20885 L
.12876 .20989 L
Mistroke
.12888 .21104 L
.12917 .21335 L
.12992 .21748 L
.13115 .22234 L
.13285 .22747 L
.13678 .23644 L
.14176 .24514 L
.14802 .25397 L
.16113 .26828 L
.17594 .28023 L
.19363 .29054 L
.21139 .29806 L
.22851 .30394 L
.26465 .31613 L
.3034 .32963 L
.34098 .34031 L
.3603 .34478 L
.37829 .34813 L
.38846 .3496 L
.3981 .35071 L
.4083 .35161 L
.41935 .35232 L
.42896 .35275 L
.43443 .35294 L
.43956 .35308 L
.44498 .3532 L
.44995 .35327 L
.4527 .3533 L
.45426 .35332 L
.45571 .35333 L
.45697 .35333 L
.45769 .35334 L
.45837 .35334 L
.45973 .35334 L
.4605 .35334 L
.4612 .35334 L
.46252 .35334 L
.46378 .35333 L
.4649 .35332 L
.46612 .35331 L
.46886 .35328 L
.4714 .35325 L
.47604 .35315 L
.48096 .35301 L
.48632 .35281 L
.49134 .35258 L
.50275 .35187 L
.51363 .35098 L
.52389 .34993 L
.54293 .34743 L
Mistroke
.56298 .34376 L
.58128 .33923 L
.61906 .3268 L
.65912 .31402 L
.6993 .30468 L
.73808 .29804 L
.75761 .29474 L
.77886 .29047 L
.79892 .28547 L
.81659 .27977 L
.83388 .27187 L
.84273 .26629 L
.85051 .26009 L
.85725 .2534 L
.86362 .24555 L
.87276 .23015 L
.87621 .22198 L
.87769 .21765 L
.879 .21307 L
.87994 .20903 L
.88074 .20458 L
.88102 .20261 L
.88126 .20051 L
.88145 .19851 L
.88158 .1967 L
.88163 .19561 L
.88167 .19459 L
.8817 .19347 L
.88172 .19226 L
.88172 .19166 L
.88172 .191 L
.8817 .18979 L
.88167 .18861 L
.88163 .18752 L
.88157 .1864 L
.8815 .18534 L
.88134 .18335 L
.88111 .1812 L
.88081 .17887 L
.87996 .17384 L
.87876 .16849 L
.87582 .15895 L
.87455 .15565 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`19o`@02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`030?oo00oo00<0ool0
;ol007go0P0;o`801Ol00`3oo`16o`804ol00`3oo`0^o`00TOl00`3oo`15o`030?oo01Co00<0ool0
;Ol004Ko4P0io`030?oo04Go00<0ool05Ol00`3oo`0/o`00@?l601;o0`0fo`030?oo04Co00<0ool0
5ol00`3oo`0[o`00?Ol301_o0P0do`030?oo04?o00<0ool06?l00`3oo`0[o`00>ol201oo0P0co`03
0?oo04?o00<0ool06Ol00`3oo`0Zo`00>Ol202;o00<0ool0<Ol00`3oo`13o`030?oo01Wo00<0ool0
:_l003Ko0`0To`030?oo037o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000do`809ol00`3oo`0ao`03
0?oo04;o00<0ool06ol00`3oo`0Yo`00<ol00`3oo`0Vo`80<ol00`3oo`12o`030?oo01_o00<0ool0
:Ol003;o00<0ool09ol00`3oo`0bo`030?oo04;o00<0ool07?l00`3oo`0Xo`00<Ol00`3oo`0Wo`03
0?oo03?o00<0ool0@Ol00`3oo`0Mo`030?oo02So000`o`030?oo02Oo00<0ool08?l200Co0P03o`@0
1Ol00`3oo`11o`030?oo01ko00<0ool09ol0033o00<0ool09_l00`3oo`0Po`040?oo00So00<0ool0
1_l00`3oo`11o`030?oo01ko00<0ool09ol002oo00<0ool09_l00`3oo`0Qo`040?oo00Wo00<0ool0
1Ol204;o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`030?oo01co1002o`040?oo00[o00<0ool0
1?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0So`040?oo00So00@0ool0
1Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`802_l200Ko00<0ool0
@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`0oo`030?oo023o00<0ool0
9ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;_l00`3oo`0So`03
0?oo03[o00<0ool0?_l00`3oo`0Qo`030?oo02Oo0007o`D00_l200Co0P0Ho`030?oo00So1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02_l00`3oo`0Eo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02Ol00`3oo`0Ho`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05ol00`3oo`07o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00Wo00<0ool0
6?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Oo
00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3oo`09o`03
0?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool0
1_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`09o`03
0?oo01So00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Jo`030?oo00Oo0P0:o`<0
8_l00`3oo`0Ro`802_l300So00<0ool06?l200_o00<0ool05ol200[o0`02o`00<?l00`3oo`0To`80
>?l00`3oo`0ho`030?oo02Go00<0ool0:Ol0033o00<0ool09_l00`3oo`0eo`030?oo03Ko0P0Wo`03
0?oo02[o000`o`030?oo02Oo0P0eo`030?oo03Co0P0Yo`030?oo02[o000ao`030?oo02So0P0co`03
0?oo03;o0P0Zo`030?oo02_o000ao`030?oo02[o0P0ao`030?oo033o0P0[o`030?oo02co000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01ol000co00<0ool09Ol01@3oool002_o0`0Zo`030?oo02Oo100Zo`030?oo00?o
00<0ool09?l00`3oo`07o`00=Ol00`3oo`0_o`@09_l00`3oo`0So`@0<_l2037o000eo`030?oo03?o
0P0To`030?oo023o0`0eo`030?oo037o000fo`030?oo03Co100Po`030?oo01co100fo`80=?l003Oo
00<0ool0=ol401co00<0ool06?l403So0P0fo`00>?l203_o1@0Go`030?oo00go2`0jo`80>?l003[o
00<0ool0?Ol<00_o00<0ool01?l904;o0`0jo`00>ol204Wo4P17o`@0?Ol003go0P1Bo`030?oo04Wo
0P11o`00?ol2053o00<0ool0A?l504?o0011o`80C_l00`3oo`0oo`D0B?l004?o0`1;o`030?oo03[o
1@1=o`00A_l404Oo00<0ool0=Ol505;o001:o`80AOl00`3oo`0`o`D0Eol004co0`0^o`801?l200?o
1005o`030?oo02[o1P1Lo`00Col202_o00@0ool02?l00`3oo`06o`030?oo02Ko101Ro`00DOl302So
00@0ool02Ol00`3oo`05o`808ol406Ko001Do`<09Ol0103oo`0:o`030?oo00Co00<0ool07_l406[o
001Go`@08Ol0103oo`08o`040?oo00Go00<0ool06ol306ko001Ko`<07ol200[o0P06o`030?oo01Wo
0P1ao`00G_l3033o00<0ool05_l307?o001Qo`@0;?l00`3oo`0Do`80M_l006Go100Xo`030?oo013o
101ho`00JOl502?o00<0ool02ol507co001^o`H07Ol00`3oo`06o`D0POl007Co9P26o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co
0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co
001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko
00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go
001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o
0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87525 .15742 m
.87361 .15344 L
.87179 .14986 L
.86666 .14225 L
.86041 .13562 L
.85249 .12947 L
.84784 .12662 L
.84247 .12388 L
.8375 .12178 L
.83201 .11988 L
.82692 .11849 L
.82417 .11788 L
.82119 .11732 L
.81844 .11691 L
.81592 .11662 L
.8147 .11651 L
.81338 .11641 L
.81213 .11634 L
.81097 .11629 L
.80961 .11626 L
.80893 .11625 L
.80818 .11626 L
.80686 .11628 L
.80563 .11633 L
.80489 .11637 L
.80419 .11641 L
.80288 .11652 L
.80149 .11666 L
.79999 .11686 L
.79748 .11728 L
.79488 .11786 L
.7904 .11923 L
.78641 .12095 L
.78295 .12295 L
.7795 .12555 L
.77624 .12882 L
.77353 .13248 L
.77142 .13626 L
.76956 .14075 L
.76805 .14591 L
.76744 .14862 L
.76689 .15162 L
.76609 .15719 L
.76495 .16774 L
.76438 .17245 L
.76362 .17738 L
.76249 .18258 L
.76108 .1872 L
.75901 .19208 L
.75655 .19642 L
Mistroke
.75061 .20384 L
.74356 .21025 L
.73513 .21666 L
.71957 .2281 L
.70418 .24082 L
.69657 .24718 L
.68826 .25315 L
.68014 .25759 L
.67516 .25967 L
.67027 .26132 L
.66112 .26361 L
.65575 .2646 L
.65076 .26535 L
.60935 .26993 L
.57063 .27528 L
.54929 .27814 L
.52561 .28125 L
.51514 .28251 L
.50974 .28308 L
.5041 .28358 L
.49917 .28392 L
.49638 .28407 L
.49379 .28419 L
.49256 .28423 L
.49139 .28426 L
.49033 .28429 L
.48921 .28431 L
.48799 .28433 L
.48687 .28433 L
.48565 .28434 L
.48432 .28434 L
.48296 .28433 L
.4817 .28431 L
.48031 .28428 L
.47883 .28425 L
.47635 .28417 L
.4736 .28405 L
.46853 .28377 L
.46375 .28345 L
.44189 .28156 L
.42379 .27998 L
.41407 .27911 L
.40517 .2781 L
.39626 .27673 L
.38645 .27466 L
.36549 .26843 L
.32666 .2534 L
.30986 .24608 L
.30252 .2423 L
.29442 .23738 L
Mistroke
.28736 .23216 L
.28047 .22572 L
.27721 .22187 L
.27471 .21821 L
.27369 .21642 L
.27274 .21445 L
.272 .21257 L
.27146 .21085 L
.27123 .20991 L
.27102 .20888 L
.27087 .20785 L
.27081 .20728 L
.27076 .20676 L
.27072 .2058 L
.27073 .20493 L
.27075 .20444 L
.27079 .20392 L
.27089 .20298 L
.27103 .20215 L
.27122 .20124 L
.27176 .19942 L
.27243 .19777 L
.27319 .19625 L
.27558 .19271 L
.27837 .18962 L
.28612 .18324 L
.29388 .17793 L
.30136 .17251 L
.30427 .16988 L
.30546 .16857 L
.3064 .16736 L
.30711 .16627 L
.30775 .16507 L
.30822 .16384 L
.30841 .16317 L
.30855 .16254 L
.30864 .16188 L
.30868 .16127 L
.30868 .16059 L
.30865 .16021 L
.30862 .15987 L
.30852 .15924 L
.30837 .15856 L
.30818 .15791 L
.30797 .15732 L
.30747 .15621 L
.30678 .155 L
.30493 .15259 L
.30227 .15003 L
.29878 .14742 L
Mistroke
.29108 .14325 L
.2862 .1413 L
.28143 .13977 L
.27598 .13841 L
.27314 .13785 L
.27005 .13735 L
.26732 .13699 L
.26482 .13672 L
.2623 .13651 L
.25989 .13637 L
.25854 .13631 L
.2578 .13628 L
.2571 .13626 L
.25574 .13622 L
.25451 .13621 L
.2538 .1362 L
.25304 .1362 L
.25169 .13621 L
.25094 .13621 L
.25023 .13622 L
.24946 .13624 L
.24864 .13626 L
.24591 .13635 L
.24302 .13649 L
.23783 .13682 L
.22615 .1378 L
.21603 .1388 L
.20527 .14 L
.19503 .14143 L
.18591 .14319 L
.1775 .14548 L
.16902 .14874 L
.16085 .15306 L
.15271 .15881 L
.14574 .16519 L
.14015 .17166 L
.13484 .1796 L
.13117 .18728 L
.12977 .19139 L
.12913 .19378 L
.12866 .19598 L
.12829 .1983 L
.12813 .19959 L
.12801 .20082 L
.12793 .20195 L
.12789 .203 L
.12786 .20416 L
.12786 .20481 L
.12787 .2054 L
.1279 .20647 L
Mistroke
.12795 .20748 L
.12803 .20858 L
.12814 .20977 L
.12846 .2122 L
.12884 .21442 L
.12985 .21874 L
.1311 .22286 L
.13479 .23211 L
.14002 .24217 L
.152 .25981 L
.15905 .26794 L
.16726 .27572 L
.17545 .28192 L
.18334 .28672 L
.19957 .2941 L
.23526 .30621 L
.26901 .31831 L
.28827 .32514 L
.3061 .33099 L
.31539 .3337 L
.3254 .33626 L
.3439 .33981 L
.35318 .34107 L
.36349 .34222 L
.37467 .34336 L
.38483 .34452 L
.40463 .34762 L
.42617 .35204 L
.436 .35391 L
.4415 .35479 L
.44672 .35549 L
.4524 .35607 L
.45544 .35629 L
.45701 .35639 L
.45873 .35647 L
.4602 .35652 L
.46157 .35656 L
.46303 .35659 L
.46384 .3566 L
.46458 .35661 L
.4659 .35661 L
.46734 .35659 L
.4687 .35657 L
.46995 .35653 L
.47135 .35649 L
.47268 .35643 L
.47517 .35629 L
.47785 .3561 L
.48076 .35585 L
.49069 .35468 L
Mistroke
.51235 .35127 L
.52257 .34977 L
.53207 .34867 L
.54302 .34773 L
.55303 .34698 L
.56293 .34609 L
.56794 .3455 L
.57343 .34469 L
.58308 .34276 L
.59194 .34034 L
.61138 .33282 L
.62827 .32474 L
.64689 .31617 L
.66457 .30969 L
.68247 .30498 L
.69183 .30319 L
.70245 .30164 L
.71309 .3005 L
.72305 .29969 L
.73483 .29886 L
.74576 .298 L
.75553 .29694 L
.76088 .29618 L
.76564 .29537 L
.77613 .293 L
.7855 .29011 L
.80296 .28273 L
.81831 .27477 L
.83464 .26536 L
.8421 .26044 L
.84839 .25563 L
.85384 .25062 L
.85904 .24463 L
.86371 .23763 L
.86745 .23034 L
.87087 .22177 L
.8735 .2135 L
.87603 .20347 L
.87696 .19886 L
.8778 .19375 L
.87812 .19138 L
.8784 .18885 L
.87862 .18645 L
.87876 .18428 L
.87883 .183 L
.87887 .18182 L
.87888 .1812 L
.87889 .18053 L
.8789 .17982 L
.8789 .17916 L
Mistroke
.87889 .17795 L
.87886 .17684 L
.87883 .17623 L
.8788 .17557 L
.87873 .17436 L
.87865 .17317 L
.87854 .17206 L
.8783 .16998 L
.87797 .16777 L
.87753 .1654 L
.87649 .16119 L
.87525 .15743 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl403_o001lo`040?oo00So1@04o`80Bol400?o1@0fo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04So0P0<o`<0<ol007co00@0ool02Ol00`3o0005o`030?oo04Oo00<0ool0
3ol00`3oo`0`o`00OOl200_o0P05o`030?oo04Ko00<0ool04Ol00`3oo`0_o`00TOl00`3oo`15o`03
0?oo01?o00<0ool0;_l004Co4P0ko`030?oo04Go00<0ool05?l202ko000oo`D04_l303So00<0ool0
A?l00`3oo`0Go`030?oo02_o000jo`D06_l203Ko00<0ool0A?l00`3oo`0Go`030?oo02_o000ho`80
8Ol203Co00<0ool0A?l00`3oo`0Ho`030?oo02[o000fo`809Ol00`3oo`0ao`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol00`3oo`0Uo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`03
0?oo02Ko0P0bo`030?oo04?o00<0ool06ol00`3oo`0Xo`00<_l202Wo0P0bo`030?oo04?o00<0ool0
6ol00`3oo`0Xo`00<Ol00`3oo`0Yo`030?oo037o00<0ool0@ol00`3oo`0Ko`030?oo02So000ao`03
0?oo02So00<0ool0<_l00`3oo`13o`030?oo01_o00<0ool0:?l0033o00<0ool0:?l00`3oo`0Oo`80
1?l200?o1005o`030?oo04?o00<0ool06ol00`3oo`0Xo`00;ol00`3oo`0Wo`808Ol0103oo`08o`03
0?oo00Ko00<0ool0@ol00`3oo`0Ko`030?oo02So000^o`030?oo02Oo00<0ool08Ol0103oo`09o`03
0?oo00Go0P13o`030?oo01co00<0ool0:?l002ko00<0ool09_l00`3oo`0Lo`@00_l0103oo`0:o`03
0?oo00Co00<0ool0@_l00`3oo`0Lo`030?oo02So000^o`030?oo02Go00<0ool08ol0103oo`08o`04
0?oo00Go00<0ool0@Ol00`3oo`0Mo`030?oo02So000^o`030?oo02Co00<0ool09Ol200[o0P06o`03
0?oo047o00<0ool07Ol00`3oo`0Xo`00;_l00`3oo`0So`030?oo03[o00<0ool0@Ol00`3oo`0Mo`03
0?oo02So000^o`030?oo02?o00<0ool0>_l00`3oo`10o`030?oo01ko00<0ool0:?l002ko00<0ool0
8ol00`3oo`0jo`030?oo03oo00<0ool07ol00`3oo`0Xo`001ol500;o0P04o`806?l00`3oo`08o`D0
5_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`801?l200go0P0Do`D02Ol00`3oo`0Fo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool02?l00`3oo`0Io`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02?l00`3oo`0Go`030?oo00So00<0ool0
6Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Oo
00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200[o00<0ool06?l00`3oo`07o`03
0?oo01[o00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Oo00<0ool0
1ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Oo0P0Ko`030?oo00Oo00<0ool0
6_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201[o0P08o`802_l302;o00<0ool0
8_l200[o0`06o`030?oo01[o0P08o`030?oo01[o0P0:o`<00_l002oo00<0ool09_l00`3oo`0fo`03
0?oo03Ko00<0ool09Ol00`3oo`0[o`00<?l00`3oo`0Vo`030?oo03Go00<0ool0=Ol00`3oo`0Vo`03
0?oo02_o000`o`030?oo02Oo0P0eo`030?oo03Co00<0ool09_l00`3oo`0/o`00<Ol00`3oo`0Xo`80
<ol00`3oo`0co`030?oo02Oo00<0ool0;?l003;o00<0ool0:Ol2037o00<0ool0<Ol202Wo00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
0_l200;o00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol302[o
00<0ool0:Ol00`3oo`0202Oo00@0ool0:Ol00`3oo`07o`00=?l00`3oo`0`o`80:?l00`3oo`0Yo`<0
:ol00`3oo`0bo`00=Ol00`3oo`0ao`<09Ol00`3oo`0Po`T0;Ol00`3oo`0co`00=_l00`3oo`0co`<0
8_l00`3oo`0Ho`P0=?l203Ko000go`030?oo03Go0P0Po`030?oo01;o1P0jo`80>?l003So00<0ool0
=_l801So00<0ool01ol;03ko0P0jo`00>Ol203ko300<o`030?oo00?o1017o`80??l003_o00<0ool0
AolB04Wo0P0no`00??l205?o00<0ool0B_l2043o000no`80DOl00`3oo`15o`D0@_l0043o0`1>o`03
0?oo03oo1P17o`00@ol304_o00<0ool0=?l;04go0016o`<0B?l00`3oo`0_o`D0F?l004Wo0`15o`03
0?oo02_o101Mo`00C?l302ko0P04o`800ol400Go00<0ool09ol4067o001?o`<0:_l0103oo`08o`03
0?oo00Ko00<0ool09Ol206Go001Bo`<09ol0103oo`09o`030?oo00Go0P0To`80Iol005Go0P0Uo`04
0?oo00[o00<0ool01?l00`3oo`0Qo`80JOl005Oo0`0Ro`040?oo00So00@0ool01Ol00`3oo`0No`<0
Jol005[o0P0Qo`802_l200Ko00<0ool07?l206ko001Lo`D0<?l00`3oo`0Io`<0L?l0067o100/o`03
0?oo01Oo0P1co`00IOl602Ko00<0ool05?l307Go001[o`X07?l00`3oo`0>o`H0N?l007Go1@0Go`03
0?oo00Co2P1no`00N_l401?o1`28o`00O_lD08ko002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87595 .1592 m
.87455 .15554 L
.87287 .15227 L
.87062 .14885 L
.86778 .14535 L
.86066 .13864 L
.85287 .13299 L
.84315 .12724 L
.83156 .1217 L
.826 .11951 L
.82 .11752 L
.81484 .11615 L
.8119 .11552 L
.80919 .11505 L
.80649 .11469 L
.80507 .11456 L
.80435 .1145 L
.80358 .11445 L
.80284 .11441 L
.80216 .11439 L
.80084 .11436 L
.79965 .11437 L
.79841 .11442 L
.79728 .11449 L
.79609 .11459 L
.79498 .11473 L
.79398 .11487 L
.7916 .11534 L
.78942 .11594 L
.78719 .11673 L
.78494 .11776 L
.78125 .12008 L
.7794 .12163 L
.77781 .12325 L
.77612 .12537 L
.77472 .12752 L
.77274 .13158 L
.77189 .1339 L
.77113 .1365 L
.77049 .13936 L
.77002 .14215 L
.7697 .14472 L
.76956 .14618 L
.76945 .14757 L
.7693 .15008 L
.76925 .15142 L
.7692 .15286 L
.76919 .15359 L
.76917 .15438 L
.76915 .15582 L
Mistroke
.76914 .15722 L
.76914 .15852 L
.76914 .15985 L
.76914 .16109 L
.76915 .16237 L
.76915 .16377 L
.76915 .16498 L
.76914 .16607 L
.76913 .16732 L
.76911 .16849 L
.76908 .16987 L
.76904 .17113 L
.76897 .1725 L
.76889 .17393 L
.76878 .17533 L
.76865 .17661 L
.76827 .17947 L
.76781 .18185 L
.76729 .18395 L
.76658 .18617 L
.76569 .1884 L
.76374 .19204 L
.76147 .19515 L
.75848 .1983 L
.75476 .2014 L
.73646 .21257 L
.7264 .21877 L
.71685 .22615 L
.70963 .23345 L
.70291 .2418 L
.69727 .24905 L
.6916 .25514 L
.6883 .25788 L
.68436 .26046 L
.68043 .26241 L
.67647 .26389 L
.67189 .26515 L
.66914 .26571 L
.66648 .26615 L
.66362 .26652 L
.66093 .26678 L
.65793 .267 L
.65641 .26709 L
.65473 .26716 L
.65325 .26722 L
.65188 .26725 L
.65037 .26728 L
.64961 .26729 L
.64877 .2673 L
.64736 .26731 L
Mistroke
.64657 .26732 L
.64585 .26732 L
.64445 .26732 L
.64313 .26731 L
.64187 .2673 L
.64052 .26728 L
.63813 .26725 L
.63276 .26718 L
.63133 .26716 L
.62983 .26715 L
.62905 .26714 L
.62819 .26713 L
.62742 .26713 L
.6267 .26713 L
.62598 .26713 L
.62529 .26713 L
.62402 .26713 L
.62265 .26714 L
.62189 .26715 L
.62118 .26715 L
.61998 .26717 L
.61886 .2672 L
.61633 .26727 L
.61389 .26736 L
.6113 .26749 L
.60651 .26782 L
.60209 .26822 L
.59194 .26942 L
.56974 .27272 L
.52449 .28045 L
.51321 .28243 L
.50742 .28334 L
.50137 .28417 L
.49611 .28474 L
.49333 .28499 L
.49033 .28519 L
.48891 .28527 L
.48757 .28534 L
.48626 .28539 L
.48504 .28543 L
.48388 .28545 L
.48266 .28547 L
.48129 .28548 L
.48005 .28549 L
.47875 .28548 L
.47754 .28546 L
.47621 .28543 L
.47478 .28539 L
.47184 .28528 L
.46917 .28513 L
Mistroke
.46399 .28477 L
.45909 .28435 L
.4371 .28201 L
.41991 .28048 L
.41092 .27981 L
.40658 .27945 L
.4026 .27903 L
.39417 .2778 L
.38932 .27683 L
.38453 .27568 L
.36301 .26851 L
.32701 .25291 L
.31105 .24632 L
.29611 .23882 L
.2883 .23342 L
.2843 .23006 L
.28034 .22614 L
.27719 .22245 L
.27424 .21817 L
.27314 .21625 L
.27212 .21416 L
.27138 .21236 L
.27072 .21037 L
.27043 .20931 L
.27018 .20817 L
.27006 .20755 L
.26997 .20698 L
.26984 .20586 L
.26978 .2049 L
.26976 .20387 L
.2698 .2028 L
.26989 .20181 L
.26997 .20125 L
.27007 .20065 L
.2703 .19957 L
.27059 .19854 L
.27091 .1976 L
.27182 .19554 L
.27307 .19346 L
.27576 .19021 L
.27887 .18748 L
.28279 .18482 L
.2874 .18231 L
.29571 .17859 L
.30358 .1751 L
.30551 .17406 L
.3071 .17305 L
.30841 .17204 L
.30941 .17106 L
.31017 .17009 L
Mistroke
.31047 .16958 L
.31075 .16901 L
.31096 .16843 L
.31105 .16809 L
.31112 .16778 L
.31117 .16749 L
.3112 .16717 L
.31122 .16658 L
.31119 .16601 L
.3111 .16538 L
.31096 .16478 L
.31079 .16421 L
.31024 .16287 L
.30942 .16141 L
.30726 .15854 L
.30406 .15529 L
.29658 .14955 L
.28815 .1447 L
.28311 .1424 L
.27745 .14029 L
.27197 .13869 L
.26685 .13758 L
.26218 .13685 L
.25952 .13655 L
.25702 .13634 L
.25567 .13626 L
.25439 .13619 L
.25323 .13614 L
.25198 .13611 L
.25061 .13608 L
.24992 .13607 L
.24916 .13607 L
.24781 .13607 L
.24654 .13609 L
.24525 .13611 L
.24388 .13615 L
.24139 .13625 L
.23843 .13639 L
.23574 .13656 L
.22435 .13738 L
.20443 .13882 L
.19943 .1392 L
.19425 .13967 L
.18995 .14016 L
.18534 .14085 L
.18056 .14181 L
.1756 .14318 L
.17108 .14484 L
.16708 .14669 L
.15925 .15153 L
Mistroke
.15242 .1572 L
.14504 .16495 L
.13886 .17295 L
.13379 .18105 L
.13162 .18532 L
.12976 .18982 L
.12856 .1936 L
.12804 .19574 L
.12766 .19774 L
.12752 .19869 L
.1274 .19959 L
.12732 .2004 L
.12725 .20127 L
.12719 .20222 L
.12716 .2031 L
.12715 .20405 L
.12716 .2051 L
.1272 .20611 L
.12726 .20706 L
.12735 .20811 L
.12747 .20924 L
.12782 .21156 L
.12824 .21371 L
.12932 .21793 L
.13066 .22204 L
.13461 .23159 L
.14016 .24245 L
.15294 .2622 L
.1604 .27111 L
.16926 .27944 L
.17795 .28557 L
.1862 .28995 L
.19592 .29384 L
.20515 .2968 L
.23814 .30734 L
.27236 .32013 L
.28886 .3259 L
.30733 .33181 L
.31796 .33457 L
.32276 .33558 L
.32813 .33652 L
.33331 .33725 L
.33821 .33777 L
.34347 .33817 L
.34644 .33834 L
.34915 .33845 L
.35992 .33871 L
.36277 .33875 L
.36582 .3388 L
.36869 .33887 L
Mistroke
.37131 .33895 L
.37366 .33904 L
.37623 .33917 L
.38087 .3395 L
.38348 .33975 L
.38626 .34008 L
.39122 .34084 L
.39628 .34185 L
.40176 .34323 L
.41176 .34636 L
.43133 .35329 L
.44216 .3565 L
.4485 .35795 L
.45453 .35896 L
.45781 .35936 L
.45944 .35951 L
.46094 .35963 L
.46228 .35972 L
.46376 .35979 L
.46524 .35985 L
.46682 .35988 L
.46765 .35988 L
.46853 .35988 L
.47009 .35986 L
.47096 .35983 L
.47177 .3598 L
.4736 .35971 L
.47521 .35961 L
.47666 .35949 L
.4799 .35916 L
.48558 .35835 L
.49145 .35724 L
.50158 .35485 L
.51199 .35213 L
.5175 .35079 L
.52261 .3497 L
.5273 .34889 L
.52977 .34855 L
.53239 .34825 L
.53482 .34803 L
.53702 .34788 L
.53809 .34782 L
.53924 .34777 L
.54025 .34773 L
.54134 .3477 L
.54245 .34767 L
.54348 .34766 L
.54472 .34765 L
.54584 .34764 L
.54702 .34765 L
Mistroke
.54831 .34766 L
.54953 .34767 L
.55066 .34769 L
.55523 .34778 L
.55724 .34782 L
.55946 .34786 L
.56065 .34788 L
.56177 .34789 L
.56294 .34789 L
.5642 .34789 L
.56486 .34789 L
.56558 .34788 L
.56689 .34786 L
.56817 .34783 L
.56935 .3478 L
.57151 .34771 L
.57274 .34764 L
.57389 .34756 L
.5787 .34712 L
.58145 .34676 L
.58404 .34634 L
.58887 .34536 L
.59399 .34399 L
.59945 .34214 L
.60925 .3378 L
.61771 .33312 L
.63566 .32171 L
.64421 .31649 L
.65332 .31161 L
.66197 .30777 L
.67028 .30484 L
.67857 .30261 L
.68295 .30168 L
.68782 .30084 L
.69332 .30011 L
.69589 .29984 L
.69866 .2996 L
.70119 .29941 L
.70401 .29925 L
.70662 .29913 L
.70908 .29904 L
.71047 .299 L
.71195 .29897 L
.71337 .29895 L
.71465 .29893 L
.71539 .29892 L
.71618 .29891 L
.71704 .29891 L
.71784 .2989 L
.71928 .2989 L
Mistroke
.7201 .2989 L
.72086 .2989 L
.72235 .29891 L
.72376 .29892 L
.72641 .29893 L
.72932 .29896 L
.73242 .29899 L
.73367 .299 L
.73501 .299 L
.73648 .29901 L
.73787 .29901 L
.73915 .29901 L
.74049 .29901 L
.74164 .299 L
.74289 .29899 L
.74437 .29897 L
.74572 .29895 L
.74726 .29891 L
.74872 .29887 L
.75129 .29877 L
.75405 .29863 L
.7567 .29844 L
.75951 .2982 L
.76446 .29761 L
.7674 .29716 L
.77012 .29665 L
.77505 .29549 L
.77997 .29399 L
.78431 .29233 L
.79341 .28772 L
.8225 .26705 L
.83691 .25777 L
.84376 .25331 L
.85001 .24844 L
.85469 .24351 L
.85698 .24034 L
.85892 .23711 L
.86515 .22271 L
.87147 .20361 L
.87424 .1939 L
.8753 .18962 L
.8763 .18495 L
.87686 .18182 L
.87732 .1785 L
.87751 .1767 L
.87757 .17592 L
.87763 .17507 L
.87768 .17432 L
.87771 .17353 L
.87774 .1727 L
Mistroke
.87775 .17193 L
.87775 .17102 L
.87774 .17019 L
.8777 .16928 L
.87765 .16833 L
.87759 .16751 L
.87752 .16675 L
.8773 .16504 L
.87705 .16353 L
.87677 .16217 L
.87595 .1592 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0BOl903[o001lo`040?oo00So1@04o`80B?l300So0`0go`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Ko0P0=o`80=Ol007co00@0ool02Ol00`3o0005o`030?oo04Ko00<0ool0
3_l203?o001mo`802ol200Go00<0ool0AOl00`3oo`0Ao`030?oo033o002Ao`030?oo04Go00<0ool0
4_l2033o0012oa<0??l00`3oo`15o`030?oo01Co00<0ool0;Ol003co1P0Co`<0>Ol00`3oo`14o`03
0?oo01Ko00<0ool0;?l003Wo0`0Lo`80=ol00`3oo`14o`030?oo01Oo00<0ool0:ol003So00<0ool0
7ol00`3oo`0do`030?oo04Co00<0ool06?l00`3oo`0Zo`00=_l202?o00<0ool0<ol00`3oo`14o`03
0?oo01Wo00<0ool0:Ol003Co0P0Vo`030?oo03;o00<0ool0A?l00`3oo`0Io`030?oo02Wo000co`03
0?oo02Oo0P0bo`030?oo04Co00<0ool06_l00`3oo`0Xo`00<_l00`3oo`0Yo`030?oo033o00<0ool0
A?l00`3oo`0Jo`030?oo02So000ao`030?oo02[o00<0ool0<?l00`3oo`14o`030?oo01[o00<0ool0
:?l0037o00<0ool0:_l00`3oo`0`o`030?oo04Co00<0ool06_l00`3oo`0Xo`00<?l00`3oo`0Xo`<0
7ol200Co0P03o`@01Ol00`3oo`14o`030?oo01[o00<0ool0:?l002oo00<0ool09ol2027o00@0ool0
2?l00`3oo`06o`030?oo04Co00<0ool06_l00`3oo`0Xo`00;ol00`3oo`0Vo`030?oo027o00@0ool0
2Ol00`3oo`05o`80AOl00`3oo`0Jo`030?oo02So000_o`030?oo02Co0P0No`@00_l0103oo`0:o`03
0?oo00Co00<0ool0@ol00`3oo`0Ko`030?oo02So000^o`030?oo02Co00<0ool09?l0103oo`08o`04
0?oo00Go00<0ool0@ol00`3oo`0Jo`030?oo02Wo000^o`030?oo02?o00<0ool09_l200[o0P06o`03
0?oo04;o0P0Lo`030?oo02Wo000]o`809?l203co00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000]o`03
0?oo02?o00<0ool0>ol00`3oo`10o`030?oo01co00<0ool0:_l002go00<0ool08ol00`3oo`0ko`03
0?oo03ko0P0Oo`030?oo02[o0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`802ol200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0<o`805Ol500Oo00<0ool06?l500;o0P04o`800ol000Wo00<0ool0
1ol0103oo`0Go`030?oo00[o00<0ool05_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`04
0?oo00So00@0ool02_l00`3oo`0Go`030?oo00Oo00<0ool06_l00`3oo`07o`040?oo00;o0009o`03
0?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Ko00<0ool02?l0103oo`0;o`030?oo023o00<0ool0
8Ol0103oo`0;o`030?oo00Oo00<0ool06?l00`3oo`06o`030?oo01_o00<0ool02_l1007o0Ol00003
o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3oo`0Fo`030?oo00;o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol200Wo00<0ool06Ol00`3oo`06o`030?oo01_o00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00Ko0P0Lo`030?oo00Ko00<0ool06ol00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201[o00<0ool01ol200[o0`0Ro`030?oo02;o0P0:o`<01Ol00`3o
o`0Ko`802?l00`3oo`0Jo`802_l300;o000_o`030?oo02Go00<0ool0=ol00`3oo`0eo`030?oo02Go
00<0ool0;?l0033o00<0ool09Ol203Oo00<0ool0=?l00`3oo`0Vo`030?oo02co000`o`030?oo02Oo
0P0eo`030?oo03?o00<0ool09Ol202oo000ao`030?oo02So0P0co`030?oo03;o00<0ool09_l00`3o
o`0^o`00<_l00`3oo`0Yo`80<Ol00`3oo`0ao`030?oo02Oo00<0ool0;_l000coo`0<00Wo000<o`03
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00ol20003o`3o00Oo00<0ool0
1Ol00`3oo`06o`030?oo00Ko00@0o`001Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Oo000<o`030?oo02Co00<0ool00_l00`3oo`0Yo`80:ol00`3oo`0Yo`030?oo00?o00<0ool0
8ol202_o00<0ool01ol003Co00<0ool0;ol302So00<0ool0;?l302Go0P0fo`00=Ol00`3oo`0ao`80
9_l00`3oo`0Wo`D09ol00`3oo`0fo`00=Ol00`3oo`0co`<08ol00`3oo`0CoaD0:_l00`3oo`0go`00
=_l00`3oo`0eo`<08?l00`3oo`0=o`H0?Ol203[o000go`80>?l501_o00<0ool01ol604;o00<0ool0
>_l003Wo00<0ool0>_l801?o00<0ool00_l504Oo00<0ool0>ol003[o0P12oaP0B_l203ko000lo`03
0?oo05;o00<0ool0B_l2043o000mo`80D_l00`3oo`18o`80@_l003oo1@1=o`030?oo04Co1014o`00
A?l304[o00<0ool0;olF04Oo0017o`<0Aol00`3oo`0[o`@0GOl004[o0`14o`030?oo02Oo101Qo`00
COl202ko0P04o`800ol400Go00<0ool09Ol206Go001?o`80:ol0103oo`08o`030?oo00Ko00<0ool0
9?l00`3oo`1Uo`00DOl202Wo00@0ool02Ol00`3oo`05o`808ol206So001Co`<09_l0103oo`0:o`03
0?oo00Co00<0ool08Ol00`3oo`1Xo`00E_l402;o00@0ool02?l0103oo`05o`030?oo01oo0P1[o`00
F_l2027o0P0:o`801_l00`3oo`0Mo`80KOl005co1@0`o`030?oo01_o0P1_o`00HOl802So00<0ool0
6_l00`3oo`1_o`00J?l>01_o00<0ool06?l207;o001fo`806Ol00`3oo`0Do`@0M?l007So100Eo`03
0?oo00Co401ho`00O?l4017o00<0o`001028o`00P?l200ko0`2=o`00P_l>0003o`3o08go002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`04
0?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87666 .16097 m
.87616 .15933 L
.87549 .15763 L
.87395 .15469 L
.87156 .15135 L
.8689 .14844 L
.86164 .1423 L
.85324 .13652 L
.83043 .12324 L
.8195 .11806 L
.81364 .11576 L
.81043 .11468 L
.8074 .11381 L
.80472 .11315 L
.80183 .1126 L
.80058 .11241 L
.79926 .11224 L
.79802 .11213 L
.79692 .11206 L
.79572 .11202 L
.79463 .11201 L
.79344 .11205 L
.79222 .11214 L
.79117 .11225 L
.79021 .11239 L
.78918 .11258 L
.7881 .11282 L
.78586 .11351 L
.78358 .1145 L
.78157 .11567 L
.77987 .11695 L
.77704 .1199 L
.77582 .12163 L
.77463 .12373 L
.7737 .12583 L
.77285 .12829 L
.77214 .13103 L
.77187 .1324 L
.77165 .13372 L
.77148 .13497 L
.77132 .13635 L
.77121 .13763 L
.77113 .13883 L
.77107 .14013 L
.77103 .14159 L
.77102 .14313 L
.77104 .14459 L
.77108 .14618 L
.77112 .14708 L
.77116 .14792 L
Mistroke
.77136 .15113 L
.77189 .15715 L
.77254 .16356 L
.77285 .16683 L
.77299 .16858 L
.77311 .17019 L
.77319 .17163 L
.77322 .17237 L
.77325 .17315 L
.77328 .17401 L
.77329 .1748 L
.77329 .17557 L
.77329 .17627 L
.77326 .17749 L
.77319 .17881 L
.77314 .17946 L
.77308 .18015 L
.77293 .18138 L
.77276 .18252 L
.77252 .18374 L
.7719 .18607 L
.77114 .18811 L
.77027 .18989 L
.76916 .19165 L
.76795 .19322 L
.76525 .19589 L
.76174 .19846 L
.75729 .20095 L
.7362 .20989 L
.7303 .21269 L
.72434 .21607 L
.71508 .22331 L
.71137 .22736 L
.70793 .23206 L
.70293 .24105 L
.69884 .24958 L
.69469 .25672 L
.69226 .25976 L
.68924 .26259 L
.68741 .26392 L
.68533 .26515 L
.68121 .26693 L
.67893 .26763 L
.67635 .26824 L
.6738 .26868 L
.67139 .26897 L
.66987 .2691 L
.66845 .2692 L
.6677 .26923 L
.66686 .26927 L
Mistroke
.66598 .26929 L
.66515 .26931 L
.66437 .26931 L
.66362 .26931 L
.66219 .2693 L
.66065 .26926 L
.65977 .26923 L
.65895 .2692 L
.65581 .26902 L
.65289 .26881 L
.64631 .26818 L
.63471 .26683 L
.62494 .26578 L
.62226 .26555 L
.61979 .26538 L
.6176 .26525 L
.61639 .2652 L
.61527 .26516 L
.61414 .26513 L
.61291 .26511 L
.61172 .2651 L
.61105 .26509 L
.61044 .2651 L
.60941 .26511 L
.60829 .26513 L
.60713 .26517 L
.60605 .26521 L
.60389 .26532 L
.60193 .26545 L
.5975 .26586 L
.59255 .26644 L
.58698 .26723 L
.57617 .26898 L
.52671 .27893 L
.51545 .28153 L
.50391 .28398 L
.49844 .28495 L
.49327 .28568 L
.48862 .28618 L
.48597 .28638 L
.48351 .28653 L
.48215 .28658 L
.48091 .28662 L
.47954 .28665 L
.47885 .28666 L
.47809 .28667 L
.4768 .28667 L
.47543 .28666 L
.47412 .28664 L
.47293 .28661 L
Mistroke
.47169 .28657 L
.47054 .28652 L
.46793 .28638 L
.46318 .28604 L
.45811 .28556 L
.45257 .28494 L
.44168 .28355 L
.43165 .28232 L
.42296 .28149 L
.4185 .28118 L
.41384 .28092 L
.41002 .28074 L
.40592 .28052 L
.40372 .28037 L
.40167 .2802 L
.39754 .27975 L
.39363 .27917 L
.3893 .27833 L
.37959 .27574 L
.36801 .27158 L
.3284 .25292 L
.31533 .24761 L
.30863 .24516 L
.3009 .24182 L
.29194 .23662 L
.28347 .22986 L
.27635 .22196 L
.27308 .21697 L
.27178 .21448 L
.27076 .21214 L
.26994 .2098 L
.26938 .20768 L
.26914 .20648 L
.26897 .20536 L
.26885 .20418 L
.26882 .20352 L
.2688 .20292 L
.26881 .20181 L
.26888 .20078 L
.26901 .19968 L
.26912 .19906 L
.26923 .19851 L
.2695 .19742 L
.26988 .19625 L
.27077 .19422 L
.27183 .19242 L
.27298 .19089 L
.2744 .18935 L
.27614 .18781 L
.27969 .18541 L
Mistroke
.28345 .18353 L
.28806 .18181 L
.2925 .18055 L
.3021 .17849 L
.30618 .1776 L
.30793 .17713 L
.30936 .17667 L
.31078 .17608 L
.31141 .17575 L
.31202 .17536 L
.31244 .17503 L
.31282 .17466 L
.31316 .17424 L
.31341 .17382 L
.31359 .17339 L
.3137 .17298 L
.31377 .17248 L
.31377 .17199 L
.31371 .17151 L
.31362 .17105 L
.31347 .17052 L
.31326 .16994 L
.31263 .16865 L
.31183 .16736 L
.3093 .16414 L
.30236 .15742 L
.29258 .15002 L
.28225 .14397 L
.27613 .14121 L
.27035 .13917 L
.26526 .13781 L
.26234 .13722 L
.25958 .13676 L
.25668 .1364 L
.254 .13615 L
.2526 .13605 L
.25106 .13597 L
.25028 .13594 L
.24944 .13591 L
.24865 .13589 L
.24794 .13588 L
.24659 .13586 L
.24533 .13586 L
.24394 .13587 L
.24244 .1359 L
.2417 .13592 L
.24088 .13594 L
.23939 .13599 L
.23659 .13611 L
.22603 .13673 L
Mistroke
.20277 .13777 L
.19764 .13791 L
.19501 .13799 L
.19229 .1381 L
.18996 .13822 L
.18746 .1384 L
.18523 .1386 L
.18319 .13883 L
.18069 .1392 L
.17847 .13962 L
.17613 .14017 L
.17366 .14088 L
.16953 .14246 L
.16577 .14437 L
.15883 .14926 L
.15156 .15628 L
.13827 .1734 L
.13339 .18123 L
.12945 .18913 L
.12799 .19313 L
.12736 .19537 L
.12691 .19745 L
.12675 .19844 L
.12662 .19937 L
.12653 .20021 L
.12646 .20113 L
.1264 .20213 L
.12639 .20263 L
.12638 .20319 L
.1264 .20419 L
.12644 .20512 L
.1265 .20611 L
.12661 .20718 L
.12687 .20912 L
.12732 .21149 L
.12784 .21369 L
.13083 .22274 L
.13993 .24225 L
.15296 .26354 L
.1615 .2741 L
.16998 .28209 L
.17873 .28808 L
.18713 .29215 L
.19641 .29533 L
.20616 .29784 L
.24031 .30834 L
.27214 .32066 L
.28757 .32602 L
.29667 .32905 L
.30554 .3317 L
Mistroke
.31466 .3339 L
.31999 .33487 L
.32502 .33555 L
.32781 .33583 L
.33043 .33603 L
.33196 .33612 L
.33335 .33618 L
.33412 .33621 L
.33496 .33624 L
.33575 .33626 L
.33649 .33628 L
.33784 .33629 L
.33856 .3363 L
.33933 .3363 L
.34013 .3363 L
.3409 .33629 L
.34235 .33627 L
.3437 .33623 L
.34495 .3362 L
.34781 .33608 L
.35297 .33579 L
.35855 .33542 L
.36334 .33511 L
.36586 .33497 L
.36854 .33484 L
.37009 .33479 L
.37149 .33475 L
.3729 .33472 L
.37422 .33471 L
.37494 .33471 L
.3757 .33472 L
.37705 .33475 L
.3778 .33477 L
.37851 .33479 L
.3801 .33487 L
.38144 .33496 L
.38271 .33506 L
.38508 .33531 L
.38761 .33567 L
.39035 .33616 L
.39579 .33749 L
.40089 .33916 L
.41052 .34325 L
.42143 .34869 L
.43372 .35461 L
.44539 .3591 L
.4487 .36009 L
.45228 .36102 L
.4554 .36169 L
.45882 .36228 L
Mistroke
.46044 .36251 L
.46217 .36272 L
.46377 .36287 L
.46527 .36299 L
.46667 .36307 L
.46818 .36312 L
.46985 .36315 L
.47137 .36315 L
.47275 .36312 L
.47402 .36307 L
.47541 .36299 L
.4769 .36288 L
.47991 .36257 L
.48262 .3622 L
.48862 .36107 L
.49476 .35951 L
.50515 .35616 L
.51554 .35242 L
.52091 .35062 L
.52584 .34922 L
.53036 .34824 L
.5329 .34784 L
.53411 .34768 L
.53524 .34756 L
.53629 .34747 L
.53742 .34739 L
.53845 .34733 L
.5394 .3473 L
.54043 .34728 L
.54156 .34727 L
.54275 .34728 L
.54386 .34731 L
.54493 .34735 L
.5461 .34741 L
.54824 .34756 L
.5523 .34794 L
.56172 .34903 L
.56638 .34953 L
.56899 .34975 L
.57147 .34992 L
.57288 .34999 L
.57362 .35003 L
.57441 .35005 L
.57515 .35007 L
.57581 .35009 L
.57711 .3501 L
.57842 .35009 L
.57916 .35007 L
.57984 .35006 L
.58109 .35001 L
Mistroke
.5824 .34993 L
.58378 .34983 L
.58503 .34971 L
.58782 .34937 L
.59039 .34894 L
.59281 .34845 L
.59812 .34699 L
.60373 .34488 L
.61247 .34038 L
.62149 .33425 L
.63835 .32059 L
.64631 .31442 L
.65481 .3088 L
.6623 .30484 L
.66673 .30294 L
.67086 .30146 L
.67499 .30023 L
.67962 .29914 L
.68224 .29865 L
.6847 .29826 L
.68725 .29792 L
.68969 .29767 L
.69181 .29749 L
.69414 .29734 L
.69548 .29728 L
.69677 .29723 L
.69793 .2972 L
.69922 .29717 L
.7006 .29715 L
.70187 .29715 L
.70335 .29716 L
.70476 .29718 L
.70607 .2972 L
.70729 .29724 L
.71008 .29733 L
.72202 .29805 L
.73535 .29905 L
.74186 .29947 L
.74505 .29964 L
.74644 .2997 L
.74797 .29976 L
.74924 .2998 L
.7506 .29983 L
.75202 .29986 L
.75335 .29987 L
.75482 .29987 L
.75634 .29986 L
.75764 .29983 L
.75904 .29979 L
.76049 .29973 L
Mistroke
.76204 .29964 L
.7648 .29943 L
.76742 .29915 L
.76978 .29882 L
.77489 .29781 L
.77734 .29717 L
.77993 .29634 L
.78402 .29471 L
.78809 .29262 L
.79474 .28801 L
.80763 .2753 L
.81425 .26844 L
.82152 .26192 L
.83508 .25295 L
.84089 .24977 L
.84543 .24698 L
.84925 .24374 L
.85097 .24171 L
.85262 .23925 L
.8552 .23421 L
.85806 .2273 L
.86577 .20746 L
.86958 .19774 L
.87343 .18723 L
.87468 .18336 L
.8758 .17938 L
.87661 .17583 L
.87692 .17414 L
.87715 .17263 L
.87724 .17191 L
.87732 .17114 L
.87738 .17041 L
.87743 .16976 L
.87747 .169 L
.87749 .16821 L
.87749 .16748 L
.87748 .1668 L
.87745 .16606 L
.87739 .16529 L
.87733 .16463 L
.87725 .16392 L
.87699 .16237 L
.87666 .16097 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`19o`D0?_l007co00@0ool0
2ol00`3oo`03o`030?oo04Oo0`04o`<0>ol007co00@0ool02?l500Co0P17o`802Ol203Wo001fo`@0
0_l0103oo`08o`040?oo00Go00<0ool0A_l00`3oo`0:o`80=ol007co00@0ool02Ol00`3o0005o`03
0?oo04Go00<0ool03Ol203Go001mo`802ol200Go00<0ool0AOl00`3oo`0?o`030?oo03;o002Ao`03
0?oo04Go00<0ool04?l203;o000loaL0?_l00`3oo`15o`030?oo01;o0P0`o`00>Ol401Ko100jo`03
0?oo04Go00<0ool05?l202ko000ho`030?oo01co0P0ho`030?oo04Go00<0ool05_l00`3oo`0[o`00
=_l2027o00<0ool0=Ol00`3oo`15o`030?oo01Oo00<0ool0:_l003Go00<0ool08_l203Go00<0ool0
AOl00`3oo`0Go`030?oo02[o000do`030?oo02Go00<0ool0<_l00`3oo`15o`030?oo01So00<0ool0
:Ol003?o00<0ool09ol00`3oo`0ao`030?oo04Go00<0ool06Ol00`3oo`0Xo`00<_l00`3oo`0Yo`03
0?oo033o00<0ool0AOl00`3oo`0Io`030?oo02So000bo`030?oo02Wo0P0ao`030?oo04Go00<0ool0
6Ol00`3oo`0Xo`00<Ol00`3oo`0[o`030?oo02oo00<0ool0AOl00`3oo`0Io`030?oo02So000`o`03
0?oo02_o0P0Mo`801?l200?o1005o`030?oo04Go00<0ool06Ol00`3oo`0Xo`00;ol00`3oo`0Yo`<0
7_l0103oo`08o`030?oo00Ko00<0ool0AOl00`3oo`0Ho`030?oo02Wo000_o`030?oo02Go100Qo`04
0?oo00Wo00<0ool01Ol204Ko00<0ool06?l00`3oo`0Yo`00;_l00`3oo`0To`807ol400;o00@0ool0
2_l00`3oo`04o`030?oo04Go00<0ool06?l00`3oo`0Yo`00;_l00`3oo`0So`030?oo02Go00@0ool0
2?l0103oo`05o`030?oo04Co0P0Jo`030?oo02Wo000]o`809Ol00`3oo`0Vo`802_l200Ko00<0ool0
A?l00`3oo`0Ho`030?oo02[o000]o`030?oo02?o00<0ool0>ol00`3oo`12o`806ol00`3oo`0Zo`00
;Ol00`3oo`0So`030?oo03_o00<0ool0@?l201go00<0ool0:_l002go00<0ool08ol00`3oo`0ko`03
0?oo03go0`0No`030?oo02_o0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`802ol201Ko1@06o`030?oo01Wo1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool05_l200co00<0ool05_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`04
0?oo00So00@0ool02Ol00`3oo`0Ho`030?oo00Ko00<0ool06ol00`3oo`07o`040?oo00;o0009o`03
0?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Ko00<0ool02?l0103oo`0;o`030?oo023o00<0ool0
8Ol0103oo`0;o`030?oo00Go0P0Ko`030?oo00Go00<0ool07?l00`3oo`0:o`400Ol1o`0000?o0000
0P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o
00<0ool08Ol0103oo`09o`801ol00`3oo`0Ko`030?oo00Go00<0ool07?l00`3oo`08o`800ol000Wo
00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3o
o`0Qo`040?oo00Wo00<0ool01Ol00`3oo`0Lo`030?oo00Co00<0ool07Ol00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201[o00<0ool01ol200[o0`0Ro`030?oo02;o0P0:o`<01?l00`3o
o`0Lo`801_l00`3oo`0Lo`802_l300;o000_o`030?oo02Go0P0ho`030?oo03Co00<0ool09Ol00`3o
o`0]o`00<?l00`3oo`0Vo`030?oo03Go00<0ool0<ol00`3oo`0Uo`030?oo02ko000`o`030?oo02Oo
0P0eo`030?oo03?o00<0ool09?l00`3oo`0_o`00<Ol00`3oo`0Xo`80<ol00`3oo`0bo`030?oo02Co
0P0ao`00<_l00`3oo`0Yo`<0<?l00`3oo`0ao`030?oo02Co00<0ool0<Ol000coo`0<00Wo000<o`03
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Go
00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo02Co00<0ool00_l00`3oo`0Yo`<0:_l00`3oo`0Yo`030?oo00Co00<0ool08?l302co
00<0ool01ol003Co00<0ool0<?l202So00<0ool0;ol2027o00<0ool0=ol003Co00<0ool0<_l202Ko
00<0ool05Ol=00co00<0ool08?l00`3oo`0ho`00=Ol00`3oo`0co`809?l00`3oo`0?o`H03Ol<02;o
00<0ool0>Ol003Ko00<0ool0=?l202;o00<0ool02ol4043o00<0ool0>_l003Oo00<0ool0=Ol301oo
00<0ool01ol404?o00<0ool0>ol003So00<0ool0=ol<01?o00<0ool01`16o`030?oo03co000io`03
0?oo04;o2@08o`D0C?l00`3oo`0mo`00>_l204_o2002o`030?oo04[o0P10o`00??l305;o00<0ool0
BOl00`3oo`10o`00?ol2053o00<0ool0Aol204?o0011o`<0COl00`3oo`0[oa`0AOl004Co0`1:o`03
0?oo02Wo0P1Qo`00Aol304Oo00<0ool09ol206?o001:o`<0<?l200Co0P03o`@01Ol00`3oo`0Uo`80
IOl004go0`0/o`040?oo00So00<0ool01_l00`3oo`0So`80Iol0053o0`0Yo`040?oo00Wo00<0ool0
1Ol202?o00<0ool0Iol005?o0`0Vo`040?oo00[o00<0ool01?l00`3oo`0Qo`030?oo06So001Fo`<0
8ol0103oo`08o`040?oo00Go00<0ool07ol206_o001Io`<08Ol200[o0P06o`030?oo01ko00<0ool0
Jol005co100ao`030?oo01go00<0ool0K?l0063o5P0Ko`030?oo01_o0P1_o`00M_l201Wo00<0ool0
6Ol2077o001ho`<05_l00`3oo`0Ho`030?oo077o001ko`805?l00`3oo`03o``01Ol407Co001mo`80
4_l00`3oo`0300co1@1ho`00Ool2013o0`2<o`00POl300[o102>o`00Q?l:00?o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3o
o`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103o
o`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00
oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87736 .16274 m
.87718 .16195 L
.87699 .16125 L
.87643 .15972 L
.87578 .15835 L
.87503 .1571 L
.87293 .15439 L
.87002 .15154 L
.85362 .14004 L
.83127 .12594 L
.81801 .11826 L
.81149 .11502 L
.80562 .11256 L
.80298 .11164 L
.80015 .1108 L
.79754 .11018 L
.7964 .10996 L
.79518 .10976 L
.7938 .10959 L
.79252 .10949 L
.7913 .10944 L
.7902 .10944 L
.78903 .1095 L
.78838 .10955 L
.78778 .10962 L
.7866 .10981 L
.78537 .11008 L
.78415 .11045 L
.78306 .11085 L
.78193 .11137 L
.78075 .11204 L
.77891 .11338 L
.77718 .11511 L
.77581 .11694 L
.77456 .11917 L
.77358 .12152 L
.77287 .12387 L
.77256 .12521 L
.77231 .12651 L
.77212 .12774 L
.77196 .12907 L
.77183 .13056 L
.77174 .13196 L
.7717 .1335 L
.77169 .1344 L
.77169 .13524 L
.77172 .13675 L
.77179 .1382 L
.77187 .13952 L
.77198 .14098 L
Mistroke
.77232 .14428 L
.77273 .14738 L
.7753 .16193 L
.77654 .16869 L
.77702 .17175 L
.77737 .17448 L
.77751 .17585 L
.77761 .17707 L
.77769 .17839 L
.77773 .17976 L
.77772 .18104 L
.77769 .18175 L
.77766 .18239 L
.77756 .18353 L
.7774 .18471 L
.77719 .18583 L
.77689 .18699 L
.7762 .18897 L
.77569 .19004 L
.77508 .19111 L
.77376 .19288 L
.77245 .19423 L
.77083 .19554 L
.7674 .19762 L
.76498 .19874 L
.7625 .19971 L
.75741 .20135 L
.74649 .2043 L
.73428 .20797 L
.72763 .21063 L
.72205 .21353 L
.71708 .21695 L
.71302 .22064 L
.70949 .22495 L
.70685 .22928 L
.70456 .23431 L
.70268 .23983 L
.6999 .25018 L
.6984 .25521 L
.69745 .25776 L
.69645 .25998 L
.6943 .26354 L
.69295 .26518 L
.69131 .26675 L
.68964 .26801 L
.68791 .26903 L
.68622 .26982 L
.68426 .27054 L
.68225 .27108 L
.67991 .27154 L
Mistroke
.67868 .27171 L
.67732 .27186 L
.676 .27196 L
.67476 .27202 L
.67352 .27205 L
.67235 .27205 L
.67104 .27203 L
.66963 .27198 L
.66826 .27191 L
.66695 .27181 L
.66447 .27159 L
.65866 .27084 L
.6522 .26974 L
.63943 .26712 L
.6286 .26484 L
.62368 .26394 L
.61935 .26327 L
.6171 .26299 L
.61466 .26274 L
.61226 .26256 L
.61096 .26249 L
.61038 .26247 L
.60974 .26245 L
.6087 .26242 L
.60761 .26241 L
.6066 .26241 L
.60566 .26243 L
.60454 .26245 L
.60352 .26249 L
.60235 .26255 L
.60123 .26262 L
.59886 .26281 L
.59631 .26307 L
.59158 .26367 L
.57995 .26559 L
.53221 .27629 L
.5074 .28314 L
.50208 .28447 L
.49649 .28569 L
.49133 .2866 L
.48645 .28724 L
.48382 .2875 L
.48243 .28761 L
.48095 .2877 L
.4797 .28777 L
.47832 .28782 L
.47707 .28785 L
.47589 .28787 L
.47517 .28788 L
.47437 .28788 L
Mistroke
.47293 .28787 L
.47152 .28784 L
.47021 .2878 L
.46879 .28775 L
.46722 .28767 L
.46404 .28745 L
.45862 .28695 L
.45281 .28626 L
.44258 .28482 L
.43211 .28331 L
.42705 .28268 L
.42441 .28241 L
.42203 .28219 L
.41984 .28203 L
.41755 .28189 L
.41553 .2818 L
.41372 .28174 L
.41204 .2817 L
.41022 .28167 L
.40833 .28165 L
.40659 .28162 L
.40476 .28158 L
.40378 .28155 L
.40274 .28151 L
.40096 .28141 L
.39895 .28127 L
.39713 .2811 L
.3951 .28085 L
.39126 .28023 L
.3864 .27916 L
.3809 .27759 L
.3702 .27367 L
.33101 .25373 L
.31673 .24758 L
.31056 .24574 L
.30423 .24361 L
.29667 .24008 L
.29234 .23746 L
.28763 .23404 L
.27991 .22684 L
.27596 .22207 L
.27279 .21728 L
.27022 .21214 L
.26915 .2092 L
.26874 .20781 L
.26842 .20651 L
.26818 .20527 L
.26799 .20396 L
.26788 .20277 L
.26783 .2017 L
Mistroke
.26783 .20055 L
.26791 .19931 L
.26797 .1987 L
.26806 .19805 L
.26828 .1969 L
.26857 .1958 L
.26889 .19482 L
.26983 .19268 L
.27089 .19094 L
.27224 .18922 L
.27511 .18663 L
.277 .18537 L
.27893 .18434 L
.28102 .18342 L
.28337 .1826 L
.28606 .18186 L
.28863 .18133 L
.29004 .1811 L
.29158 .18089 L
.29293 .18073 L
.29436 .1806 L
.29564 .1805 L
.29704 .18042 L
.29829 .18036 L
.29945 .18033 L
.3005 .1803 L
.30163 .18029 L
.30266 .18028 L
.30359 .18027 L
.30455 .18027 L
.30554 .18028 L
.30651 .18028 L
.30754 .18028 L
.30852 .18027 L
.30905 .18026 L
.30952 .18025 L
.31039 .18023 L
.31116 .18019 L
.31188 .18014 L
.31249 .18008 L
.31312 .18 L
.31373 .1799 L
.31427 .17977 L
.31478 .1796 L
.3152 .17941 L
.31553 .17922 L
.31584 .17895 L
.31598 .17879 L
.31609 .17863 L
.31624 .1783 L
Mistroke
.3163 .1781 L
.31633 .17791 L
.31633 .17751 L
.31628 .17712 L
.31616 .17664 L
.31607 .17639 L
.31595 .1761 L
.31539 .175 L
.31466 .17385 L
.31007 .16837 L
.30265 .16115 L
.29294 .15305 L
.28735 .14903 L
.28122 .14519 L
.27577 .14231 L
.26967 .13975 L
.26698 .13883 L
.26407 .13798 L
.25875 .13681 L
.25601 .13638 L
.2545 .13619 L
.25307 .13604 L
.25173 .13592 L
.25027 .13581 L
.24894 .13574 L
.24772 .13568 L
.24626 .13564 L
.24488 .13561 L
.24338 .13561 L
.24253 .13561 L
.24175 .13561 L
.24019 .13564 L
.23931 .13566 L
.23849 .13568 L
.23537 .13579 L
.21036 .13676 L
.209 .13677 L
.20773 .13678 L
.20627 .13678 L
.20495 .13677 L
.20371 .13676 L
.20258 .13675 L
.20005 .13671 L
.19514 .13661 L
.19383 .13659 L
.19245 .13656 L
.19116 .13654 L
.19 .13653 L
.18876 .13653 L
.18761 .13653 L
Mistroke
.18638 .13654 L
.18569 .13656 L
.18506 .13657 L
.18378 .13662 L
.18264 .13668 L
.18134 .13677 L
.18014 .13688 L
.17805 .13713 L
.17619 .13744 L
.17419 .13787 L
.17215 .13845 L
.1702 .13914 L
.16817 .14002 L
.16463 .14201 L
.16058 .14504 L
.15699 .14841 L
.144 .16466 L
.1332 .18098 L
.13084 .18506 L
.12874 .18919 L
.12735 .19259 L
.12676 .19434 L
.12626 .19625 L
.12608 .19711 L
.12591 .19803 L
.1258 .19883 L
.1257 .1997 L
.12566 .20021 L
.12563 .20069 L
.12559 .20163 L
.12559 .2025 L
.12561 .20343 L
.12566 .20424 L
.12573 .20512 L
.12583 .20597 L
.12594 .20676 L
.12628 .20872 L
.12671 .21066 L
.12799 .21517 L
.12958 .2197 L
.13442 .23126 L
.14734 .25613 L
.15426 .26681 L
.16224 .27677 L
.16646 .28106 L
.17123 .28516 L
.1799 .29079 L
.18487 .29314 L
.19021 .2951 L
.19522 .29651 L
Mistroke
.19973 .29753 L
.20886 .29922 L
.21796 .30098 L
.22522 .30286 L
.2328 .30545 L
.2611 .31727 L
.2777 .32319 L
.29385 .32867 L
.30263 .33127 L
.31134 .33325 L
.31595 .33399 L
.3186 .33431 L
.32104 .33454 L
.32237 .33464 L
.32378 .33472 L
.32452 .33476 L
.32534 .33479 L
.3261 .33481 L
.32679 .33483 L
.32817 .33485 L
.32893 .33485 L
.32966 .33485 L
.33095 .33484 L
.33237 .3348 L
.33311 .33478 L
.33388 .33474 L
.33528 .33468 L
.33844 .33446 L
.3412 .33422 L
.34424 .3339 L
.35516 .33243 L
.36123 .33157 L
.36454 .33113 L
.36763 .33078 L
.37049 .33051 L
.37197 .3304 L
.37354 .33031 L
.37486 .33026 L
.37629 .33022 L
.37759 .33021 L
.3788 .33023 L
.38013 .33027 L
.38088 .33031 L
.38155 .33035 L
.38285 .33045 L
.38408 .33057 L
.3868 .33095 L
.38972 .33153 L
.39229 .33219 L
.39499 .33304 L
Mistroke
.39981 .33495 L
.40891 .33974 L
.41936 .34634 L
.43102 .35361 L
.443 .35978 L
.44996 .36251 L
.45371 .36369 L
.45774 .36472 L
.46114 .36541 L
.46295 .3657 L
.46491 .36596 L
.4667 .36615 L
.46834 .36628 L
.46991 .36637 L
.47156 .36642 L
.47327 .36643 L
.47423 .36641 L
.47511 .36639 L
.47673 .36631 L
.47843 .36619 L
.47995 .36605 L
.48136 .36589 L
.48455 .36541 L
.48793 .36475 L
.49097 .36401 L
.49758 .36201 L
.51782 .35333 L
.52653 .34965 L
.52892 .34882 L
.53114 .34816 L
.53313 .34767 L
.53523 .34725 L
.53635 .34708 L
.53758 .34692 L
.5387 .34682 L
.53974 .34675 L
.54038 .34672 L
.54098 .3467 L
.54212 .34668 L
.54275 .34669 L
.54334 .34671 L
.54399 .34673 L
.54468 .34677 L
.54574 .34684 L
.54689 .34694 L
.54898 .34719 L
.55386 .348 L
.56412 .35019 L
.56935 .35125 L
.5742 .35205 L
Mistroke
.57687 .3524 L
.5794 .35264 L
.5808 .35274 L
.58208 .35281 L
.58349 .35285 L
.5842 .35286 L
.58498 .35286 L
.58577 .35286 L
.58652 .35284 L
.58791 .35278 L
.58869 .35273 L
.58941 .35268 L
.59104 .35252 L
.59233 .35236 L
.59374 .35215 L
.59659 .3516 L
.59898 .35099 L
.60156 .3502 L
.60661 .34817 L
.61111 .34583 L
.61513 .34328 L
.63198 .3284 L
.64557 .31457 L
.65272 .30836 L
.66055 .30298 L
.66467 .30077 L
.66938 .29874 L
.67386 .29727 L
.67824 .29621 L
.68063 .29577 L
.68296 .29543 L
.68511 .29519 L
.6875 .29499 L
.68889 .29491 L
.69017 .29485 L
.69089 .29483 L
.69167 .29481 L
.69241 .2948 L
.6931 .29479 L
.69435 .29479 L
.69573 .2948 L
.69644 .29482 L
.6972 .29484 L
.69859 .29488 L
.69997 .29494 L
.70151 .29503 L
.7047 .29525 L
.71063 .2958 L
.72491 .29757 L
.73911 .29945 L
Mistroke
.74534 .30016 L
.7486 .30048 L
.75205 .30076 L
.75378 .30087 L
.7554 .30096 L
.75696 .30103 L
.75838 .30108 L
.75972 .30111 L
.76112 .30113 L
.76251 .30113 L
.76402 .3011 L
.76558 .30105 L
.76698 .30097 L
.76856 .30086 L
.77004 .30072 L
.7726 .30041 L
.77528 .29995 L
.77809 .2993 L
.78052 .29859 L
.78301 .29768 L
.78555 .29654 L
.7897 .29409 L
.79321 .29134 L
.79917 .28489 L
.80486 .27679 L
.81067 .26807 L
.81626 .26088 L
.82222 .25505 L
.82797 .25103 L
.83103 .2494 L
.83372 .24821 L
.83907 .24618 L
.84099 .24539 L
.8428 .24446 L
.84431 .2434 L
.84553 .2422 L
.84684 .24043 L
.84821 .23798 L
.85087 .23191 L
.85913 .21268 L
.86928 .19158 L
.8731 .18322 L
.87477 .17909 L
.87618 .17499 L
.87705 .17162 L
.87723 .17074 L
.87739 .16981 L
.8775 .16902 L
.87759 .16818 L
.87765 .16738 L
Mistroke
.87768 .16665 L
.87769 .16596 L
.87766 .16524 L
.87762 .16461 L
.87756 .16403 L
.87748 .1634 L
.87736 .16274 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`17o`P0?Ol007co00@0ool0
2ol00`3oo`03o`030?oo04Ko0P07o`80>ol007co00@0ool02?l500Co0P17o`030?oo00So0P0io`00
M_l400;o00@0ool02?l0103oo`05o`030?oo04Go00<0ool02ol203Oo001lo`040?oo00Wo00<0o`00
1Ol00`3oo`15o`030?oo00go00<0ool0=?l007go0P0;o`801Ol00`3oo`15o`030?oo00ko0P0do`00
TOl00`3oo`15o`030?oo013o00<0ool0<Ol003Wo6`0mo`030?oo04Go00<0ool04Ol2037o000go`<0
6Ol303_o00<0ool0AOl00`3oo`0Co`030?oo02ko000fo`030?oo01go0P0io`030?oo04Go00<0ool0
5?l202ko000eo`030?oo023o00<0ool0=_l00`3oo`15o`030?oo01Ko00<0ool0:ol003Co00<0ool0
8_l00`3oo`0eo`030?oo04Go00<0ool05ol00`3oo`0Zo`00<ol00`3oo`0To`80=Ol00`3oo`16o`03
0?oo01Oo00<0ool0:Ol003?o00<0ool09_l00`3oo`0bo`030?oo04Ko00<0ool05ol202[o000bo`03
0?oo02So00<0ool0<Ol00`3oo`16o`030?oo01So00<0ool0:?l0037o00<0ool0:_l00`3oo`0`o`03
0?oo04Ko00<0ool06?l00`3oo`0Xo`00<Ol00`3oo`0Zo`030?oo033o00<0ool0A_l00`3oo`0Ho`03
0?oo02So000`o`030?oo02co0P0Lo`801?l200?o1005o`030?oo04Ko00<0ool06?l00`3oo`0Xo`00
<?l00`3oo`0Xo`H06ol0103oo`08o`030?oo00Ko00<0ool0A_l00`3oo`0Go`030?oo02Wo000_o`03
0?oo02Co1@0Qo`040?oo00Wo00<0ool01Ol204Oo00<0ool05ol00`3oo`0Yo`00;_l00`3oo`0To`80
7ol400;o00@0ool02_l00`3oo`04o`030?oo04Ko00<0ool05ol00`3oo`0Yo`00;_l00`3oo`0So`03
0?oo02Go00@0ool02?l0103oo`05o`030?oo04Ko00<0ool05_l00`3oo`0Zo`00;Ol00`3oo`0So`03
0?oo02Oo0P0:o`801_l00`3oo`15o`030?oo01Oo00<0ool0:_l002go00<0ool08ol00`3oo`0ko`03
0?oo04?o0`0Ho`030?oo02_o000]o`030?oo02?o00<0ool0>ol00`3oo`10o`<06ol00`3oo`0[o`00
;Ol00`3oo`0So`030?oo03_o00<0ool0?Ol301go00<0ool0;?l000Oo1@02o`801?l201Oo00<0ool0
2Ol501Go00<0ool02_l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0;o`805_l500Go00<0ool0
6_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool01ol301[o00<0ool01?l00`3oo`0Mo`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool00ol201go00<0ool01?l00`3oo`0Mo`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l00`3o
o`02o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P05o`030?oo01go00<0ool00ol00`3o
o`0No`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool02_l00`3oo`0Go`030?oo00Oo
00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`03o`030?oo01ko00<0ool00ol00`3o
o`0No`030?oo00So00<0ool00_l000So0P0:o`<06?l00`3oo`08o`806_l00`3oo`07o`802_l302;o
00<0ool08_l200[o0`03o`030?oo01go0P04o`030?oo01ko0P0:o`<00_l002oo00<0ool09Ol00`3o
o`0go`030?oo03?o00<0ool09?l00`3oo`0_o`00<?l00`3oo`0Uo`030?oo03Ko00<0ool0<ol00`3o
o`0To`030?oo02oo000`o`030?oo02Ko0`0eo`030?oo03;o00<0ool09?l00`3oo`0`o`00<Ol00`3o
o`0Xo`80<ol00`3oo`0bo`030?oo02?o00<0ool0<Ol0037o00<0ool0:_l402oo00<0ool0<_l00`3o
o`0Qo`<0<ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko0P07o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go0P08o`030?oo00Go00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol0103o000Yo`03
0?oo02Wo00<0ool01Ol00`3oo`0No`040?oo02co00<0ool01ol003Co00<0ool0<Ol202Oo00<0ool0
5_l701Co00<0ool07Ol00`3oo`0ho`00=Ol00`3oo`0bo`809Ol00`3oo`0@o`H01_l500ko0P0Oo`03
0?oo03Wo000fo`030?oo03?o0P0So`030?oo00co100Ao`H01ol201oo00<0ool0>_l003Ko00<0ool0
=Ol2027o00<0ool02?l401_o1`0Qo`030?oo03[o000go`030?oo03Ko0P0Oo`030?oo00Co1016o`03
0?oo03_o000ho`030?oo03Oo2P0Eo`030?oo00@0BOl00`3oo`0lo`00>Ol00`3oo`0oo`P03Ol404co
00<0ool0?Ol003[o00<0ool0A_l=0003o`3o04co00<0ool0?_l003_o0`1Co`030?oo04[o00<0ool0
?ol003ko0P1Ao`030?oo02co2`0Ao`<0@Ol0043o101=o`030?oo02Wo0`0;o`T01?l504?o0014o`@0
BOl00`3oo`0Wo`805ol504Oo0018o`<0A_l00`3oo`0Vo`030?oo06?o001;o`80<?l200Co0P03o`@0
1Ol00`3oo`0To`80I_l004go0`0/o`040?oo00So00<0ool01_l00`3oo`0So`030?oo06Ko001@o`<0
:Ol0103oo`09o`030?oo00Go0P0So`030?oo06Oo001Co`@09Ol0103oo`0:o`030?oo00Co00<0ool0
8_l00`3oo`1Wo`00Eol202?o00@0ool02?l0103oo`05o`030?oo027o00<0ool0J?l005Wo0P0Bo`H0
2_l200[o0P06o`030?oo023o00<0ool0JOl005_o1`04o`L01Ol401_o00<0ool07ol00`3oo`1Zo`00
HOl5013o00<0ool06?l00`3oo`0Mo`80KOl007Oo0P0Ho`030?oo01co00<0ool0KOl007Wo0P0Fo`03
0?oo01[o0P1`o`00Nol00`3oo`0Co`030?oo00Co2@0<o`030?oo073o001lo`804ol0103oool400So
301co`00O_l2017o00@0o`00Rol0083o0P0?o`80SOl008;o0`09o`@0S_l008Go2@03o`030?oo08co
002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87806 .16451 m
.85399 .14356 L
.83091 .12807 L
.81776 .11923 L
.81053 .11484 L
.80384 .11132 L
.79785 .10882 L
.79461 .10781 L
.79311 .10743 L
.79173 .10716 L
.79061 .10698 L
.78942 .10684 L
.78813 .10675 L
.78698 .10674 L
.78582 .10679 L
.78518 .10685 L
.7846 .10692 L
.78356 .10711 L
.78249 .10738 L
.78145 .10773 L
.78052 .10813 L
.77958 .10863 L
.77859 .10926 L
.77707 .11054 L
.77566 .11217 L
.77456 .11389 L
.77358 .11598 L
.77318 .11709 L
.77284 .11818 L
.77233 .12037 L
.77211 .12163 L
.77193 .12303 L
.7718 .12441 L
.77173 .12571 L
.77168 .12719 L
.77169 .12885 L
.77171 .12971 L
.77175 .13063 L
.77186 .13233 L
.77202 .13417 L
.77225 .13619 L
.77281 .13996 L
.77423 .14721 L
.77781 .16134 L
.78089 .17347 L
.78147 .17636 L
.78189 .17895 L
.78205 .18032 L
.78212 .18102 L
.78218 .18175 L
Mistroke
.78224 .18301 L
.78225 .18414 L
.78223 .18482 L
.78219 .18543 L
.78214 .18609 L
.78205 .18677 L
.78186 .18786 L
.78172 .18846 L
.78158 .189 L
.78088 .19082 L
.78038 .19176 L
.77985 .19257 L
.77843 .19419 L
.77683 .1955 L
.77474 .19674 L
.77216 .19787 L
.76964 .1987 L
.76705 .19939 L
.76046 .20069 L
.74666 .20284 L
.73882 .20431 L
.73173 .20609 L
.72602 .20803 L
.72044 .21061 L
.71756 .21232 L
.71506 .2141 L
.71099 .21786 L
.70919 .22003 L
.7077 .22218 L
.70617 .22488 L
.705 .22746 L
.70404 .23007 L
.70318 .233 L
.70247 .23616 L
.70194 .23921 L
.70071 .25028 L
.70046 .25281 L
.70015 .25538 L
.69981 .25767 L
.69942 .25971 L
.69893 .26174 L
.69828 .26381 L
.69749 .26572 L
.69647 .26759 L
.69512 .26942 L
.69361 .27091 L
.69206 .27206 L
.69016 .27309 L
.68809 .27388 L
.68682 .27424 L
Mistroke
.68557 .27452 L
.6842 .27474 L
.68289 .2749 L
.68215 .27496 L
.68133 .27501 L
.68053 .27504 L
.67979 .27506 L
.67831 .27505 L
.67688 .275 L
.67557 .27492 L
.67412 .2748 L
.67077 .27441 L
.6676 .27391 L
.66143 .27266 L
.65472 .27099 L
.64289 .26757 L
.63018 .26372 L
.61997 .26109 L
.61534 .26024 L
.61308 .25993 L
.61179 .2598 L
.61062 .2597 L
.60942 .25962 L
.60883 .25958 L
.60817 .25956 L
.60752 .25954 L
.60681 .25952 L
.60615 .25952 L
.60554 .25952 L
.60492 .25952 L
.60435 .25953 L
.60373 .25955 L
.60306 .25957 L
.60193 .25963 L
.60068 .25971 L
.59823 .25993 L
.59593 .26018 L
.5905 .26096 L
.57966 .26296 L
.52728 .27642 L
.51462 .28055 L
.5035 .28409 L
.49389 .28665 L
.48849 .28773 L
.48349 .28845 L
.48084 .28873 L
.47834 .28891 L
.4769 .28899 L
.47558 .28904 L
.47485 .28906 L
Mistroke
.47405 .28908 L
.4726 .2891 L
.47119 .28909 L
.47038 .28908 L
.46963 .28906 L
.46811 .28902 L
.46725 .28898 L
.46646 .28894 L
.46352 .28876 L
.46072 .28853 L
.45456 .28784 L
.4489 .28705 L
.43665 .28506 L
.43142 .28422 L
.42618 .28347 L
.42392 .28319 L
.42164 .28295 L
.41974 .28278 L
.41779 .28265 L
.41685 .2826 L
.41597 .28256 L
.4152 .28254 L
.41436 .28252 L
.41359 .2825 L
.41278 .2825 L
.41209 .2825 L
.41134 .2825 L
.41048 .28251 L
.4097 .28252 L
.40794 .28257 L
.40632 .28261 L
.40479 .28265 L
.40399 .28266 L
.40311 .28267 L
.40231 .28267 L
.40156 .28267 L
.40074 .28266 L
.39982 .28263 L
.39935 .28261 L
.39885 .28259 L
.39792 .28254 L
.39607 .28239 L
.39424 .28218 L
.39059 .2816 L
.38838 .28113 L
.38591 .28053 L
.3812 .27914 L
.36938 .27455 L
.34637 .26253 L
.33588 .25629 L
Mistroke
.32702 .2512 L
.32079 .24833 L
.31768 .24732 L
.31485 .24662 L
.30956 .24551 L
.30617 .24461 L
.30268 .24343 L
.29862 .24166 L
.29395 .23909 L
.28482 .23229 L
.27705 .22411 L
.27325 .21879 L
.27033 .21354 L
.26921 .21097 L
.26823 .20822 L
.26778 .20665 L
.26744 .20524 L
.26718 .20384 L
.267 .20253 L
.26688 .20118 L
.26685 .20051 L
.26684 .19978 L
.26688 .19851 L
.26698 .19735 L
.26707 .19666 L
.26718 .19602 L
.26745 .19484 L
.26781 .19361 L
.26831 .19233 L
.26925 .19049 L
.27056 .18863 L
.2722 .18688 L
.27398 .18544 L
.27583 .18428 L
.27768 .18337 L
.27985 .18253 L
.28228 .18183 L
.28359 .18154 L
.28504 .18127 L
.28644 .18108 L
.28774 .18094 L
.28893 .18084 L
.29024 .18077 L
.29162 .18074 L
.29292 .18074 L
.29443 .18077 L
.29585 .18084 L
.29712 .18092 L
.2985 .18104 L
.3015 .18138 L
Mistroke
.30417 .18176 L
.31211 .18322 L
.31382 .18354 L
.3152 .18377 L
.31587 .18386 L
.31642 .18391 L
.31697 .18395 L
.31725 .18395 L
.31749 .18395 L
.31772 .18393 L
.31794 .18391 L
.31828 .18383 L
.31844 .18376 L
.31856 .18369 L
.31868 .1836 L
.31878 .18349 L
.31888 .18326 L
.31891 .18313 L
.31891 .18297 L
.31886 .18266 L
.31875 .18232 L
.31857 .1819 L
.31832 .18145 L
.3177 .18049 L
.31554 .17778 L
.30103 .16293 L
.29097 .15406 L
.27962 .14578 L
.27484 .14299 L
.26967 .14049 L
.26395 .13836 L
.25854 .13692 L
.25556 .13636 L
.25391 .13611 L
.25236 .13591 L
.2509 .13575 L
.24932 .13561 L
.24788 .13551 L
.24654 .13544 L
.2457 .13541 L
.24493 .13538 L
.24408 .13536 L
.24316 .13534 L
.24228 .13533 L
.24132 .13532 L
.24041 .13532 L
.23956 .13533 L
.23807 .13535 L
.23722 .13537 L
.23642 .13539 L
Mistroke
.23306 .13549 L
.2273 .13571 L
.22413 .13584 L
.22114 .13594 L
.21945 .13598 L
.21762 .13603 L
.21681 .13604 L
.21596 .13605 L
.21444 .13607 L
.21293 .13608 L
.21152 .13608 L
.20998 .13607 L
.20912 .13606 L
.20834 .13604 L
.20677 .13601 L
.20533 .13597 L
.20211 .13584 L
.19934 .1357 L
.19646 .13552 L
.19108 .13515 L
.18856 .13498 L
.18632 .13485 L
.18437 .13476 L
.18341 .13472 L
.18238 .1347 L
.18135 .13468 L
.18028 .13469 L
.17928 .13471 L
.1784 .13474 L
.17736 .1348 L
.17642 .13488 L
.1754 .135 L
.17435 .13515 L
.17263 .13548 L
.17081 .13596 L
.16903 .13658 L
.16715 .13742 L
.16388 .1394 L
.16082 .14191 L
.15804 .14471 L
.1452 .16217 L
.13417 .17886 L
.1295 .18633 L
.12752 .19001 L
.12612 .19328 L
.12559 .19492 L
.12517 .19668 L
.12499 .19769 L
.12492 .19819 L
.12487 .19866 L
Mistroke
.12483 .19907 L
.12479 .19954 L
.12477 .2 L
.12476 .20051 L
.12475 .20094 L
.12476 .20141 L
.12479 .20226 L
.12486 .20318 L
.12492 .20372 L
.12498 .20422 L
.12526 .20603 L
.12568 .20805 L
.12699 .21272 L
.1287 .21765 L
.13302 .22828 L
.14454 .25218 L
.15262 .26587 L
.161 .27721 L
.1699 .28616 L
.1751 .29005 L
.17994 .2929 L
.18506 .29523 L
.19064 .29708 L
.19584 .29829 L
.20056 .29909 L
.20897 .30012 L
.21718 .30119 L
.2212 .30193 L
.22539 .30294 L
.23234 .30527 L
.24689 .31189 L
.26114 .31792 L
.27724 .3236 L
.29312 .32895 L
.3018 .33136 L
.30703 .33244 L
.30961 .33286 L
.31209 .33319 L
.31429 .33342 L
.31543 .33351 L
.3167 .3336 L
.31809 .33367 L
.31943 .33371 L
.32074 .33374 L
.32197 .33374 L
.3234 .33373 L
.32475 .33369 L
.32598 .33364 L
.32732 .33357 L
.33036 .33333 L
Mistroke
.33319 .33304 L
.33879 .33225 L
.34488 .33116 L
.35589 .32884 L
.36251 .32747 L
.36571 .32688 L
.36864 .32641 L
.3712 .32607 L
.37255 .32592 L
.37399 .3258 L
.37542 .3257 L
.37676 .32564 L
.3775 .32562 L
.37827 .32562 L
.37965 .32563 L
.38098 .32568 L
.38173 .32573 L
.38242 .32578 L
.38367 .3259 L
.38499 .32608 L
.38726 .32648 L
.38974 .32707 L
.39235 .32788 L
.39515 .32897 L
.40018 .33146 L
.41017 .33807 L
.41987 .34555 L
.43098 .35381 L
.44398 .36168 L
.45194 .36525 L
.45937 .36764 L
.463 .36848 L
.46492 .36883 L
.46694 .36914 L
.46872 .36936 L
.47066 .36953 L
.47243 .36964 L
.47407 .36969 L
.47593 .3697 L
.47686 .36968 L
.47789 .36965 L
.4797 .36954 L
.48138 .36939 L
.48335 .36916 L
.48514 .3689 L
.48911 .36813 L
.49228 .36734 L
.49567 .36632 L
.50222 .36385 L
.52265 .35314 L
Mistroke
.52717 .35071 L
.5318 .34857 L
.53386 .34779 L
.53601 .34712 L
.53783 .34667 L
.53885 .34647 L
.53982 .34632 L
.54039 .34624 L
.54101 .34617 L
.5421 .34609 L
.54272 .34606 L
.54338 .34604 L
.54394 .34604 L
.54457 .34605 L
.54568 .3461 L
.54631 .34614 L
.54689 .3462 L
.54797 .34632 L
.54912 .34648 L
.55128 .34688 L
.55334 .34735 L
.55818 .34869 L
.56861 .35197 L
.57403 .35351 L
.57646 .3541 L
.57915 .35468 L
.58142 .35509 L
.58386 .35544 L
.58521 .35559 L
.58644 .3557 L
.58768 .35578 L
.58885 .35583 L
.59011 .35586 L
.59145 .35585 L
.59271 .3558 L
.59384 .35573 L
.59518 .35562 L
.59661 .35545 L
.59918 .35501 L
.60164 .35444 L
.60422 .35366 L
.60874 .35184 L
.61386 .349 L
.61827 .34584 L
.62703 .33756 L
.63447 .32872 L
.64666 .31336 L
.65355 .30599 L
.65702 .30291 L
.66029 .30042 L
Mistroke
.6636 .29831 L
.66735 .29635 L
.67132 .29477 L
.67573 .29352 L
.67815 .29303 L
.68043 .29268 L
.68173 .29253 L
.68316 .2924 L
.68382 .29234 L
.68453 .2923 L
.68582 .29223 L
.68703 .2922 L
.68835 .29218 L
.68961 .29218 L
.69078 .2922 L
.69218 .29225 L
.69298 .29228 L
.69373 .29232 L
.69662 .29252 L
.6997 .29282 L
.70262 .29315 L
.70953 .29414 L
.723 .29651 L
.73587 .29891 L
.74977 .30117 L
.75348 .30164 L
.75693 .302 L
.75996 .30226 L
.76165 .30237 L
.76318 .30244 L
.76459 .30249 L
.7659 .30252 L
.76731 .30252 L
.7688 .3025 L
.77033 .30245 L
.77176 .30236 L
.77302 .30226 L
.77436 .30212 L
.77594 .30192 L
.77738 .30168 L
.77998 .30113 L
.78262 .30037 L
.78529 .29935 L
.78747 .29829 L
.78963 .297 L
.7931 .29428 L
.79623 .29093 L
.79873 .28746 L
.8035 .27864 L
.81135 .26208 L
Mistroke
.81361 .25825 L
.81616 .25463 L
.81847 .25192 L
.82105 .24947 L
.82383 .24742 L
.82642 .24599 L
.82788 .24538 L
.82936 .24487 L
.83058 .24455 L
.83186 .24429 L
.83298 .24411 L
.83411 .24399 L
.83597 .24384 L
.83675 .24377 L
.83715 .24373 L
.83751 .24368 L
.83811 .24357 L
.83841 .2435 L
.83872 .2434 L
.83936 .24314 L
.8397 .24294 L
.84 .24273 L
.84053 .24224 L
.84107 .24159 L
.84209 .23989 L
.84309 .23777 L
.8617 .20049 L
.87331 .18033 L
.87535 .17616 L
.87697 .17211 L
.87732 .171 L
.8776 .16998 L
.87781 .16902 L
.87797 .16816 L
.87807 .16736 L
.87813 .16654 L
.87814 .16574 L
.87813 .1653 L
.8781 .16489 L
.87806 .16451 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`17o`H0?ol007go0P04o`801Ol300Co00<0ool0A_l200Go0P0mo`00
O?l0103oo`0;o`030?oo00?o00<0ool0AOl200So0P0ko`00O?l0103oo`08o`D01?l204Ko00<0ool0
2Ol00`3oo`0ho`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Go00<0ool02_l203So001lo`04
0?oo00Wo00<0o`001Ol00`3oo`15o`030?oo00co00<0ool0=Ol007go0P0;o`801Ol00`3oo`15o`03
0?oo00go00<0ool0=?l003[o1P0;o`80A?l00`3oo`15o`030?oo00ko0P0do`00>?l200Ko4`0no`03
0?oo04Go00<0ool04?l00`3oo`0ao`00=_l201_o0P0lo`030?oo04Go00<0ool04Ol2037o000fo`03
0?oo01co0P0jo`030?oo04Go00<0ool04ol00`3oo`0^o`00=Ol00`3oo`0Oo`80>?l00`3oo`15o`03
0?oo01Co00<0ool0;Ol003Co00<0ool08_l00`3oo`0eo`030?oo04Go00<0ool05Ol202go000do`03
0?oo02?o00<0ool0=?l00`3oo`16o`030?oo01Ko00<0ool0:_l003?o00<0ool09Ol00`3oo`0co`03
0?oo04Ko00<0ool05ol00`3oo`0Yo`00<_l00`3oo`0Wo`030?oo03;o00<0ool0A_l00`3oo`0Ho`03
0?oo02So000ao`030?oo02Wo00<0ool0<Ol00`3oo`17o`030?oo01Oo00<0ool0:?l0033o00<0ool0
:_l00`3oo`0ao`030?oo04Oo00<0ool05ol00`3oo`0Xo`00<?l00`3oo`0[o`030?oo01co0P04o`80
0ol400Go00<0ool0Aol00`3oo`0Fo`030?oo02Wo000_o`030?oo02go00<0ool06_l0103oo`08o`03
0?oo00Ko00<0ool0Aol201Oo00<0ool0:Ol002ko00<0ool09?l=01[o00@0ool02Ol00`3oo`05o`80
BOl00`3oo`0Eo`030?oo02Wo000^o`030?oo02?o0P0Po`@00_l0103oo`0:o`030?oo00Co00<0ool0
Aol201Ko00<0ool0:_l002go00<0ool08ol00`3oo`0Vo`040?oo00So00@0ool01Ol00`3oo`17o`03
0?oo01Go00<0ool0:_l002go00<0ool08ol00`3oo`0Wo`802_l200Ko00<0ool0Aol00`3oo`0Do`03
0?oo02_o000]o`030?oo02;o0P0mo`030?oo04Co0`0Go`030?oo02_o000]o`030?oo02;o0P0mo`03
0?oo03ko1`0Ho`030?oo02co000]o`030?oo02?o00<0ool0>ol00`3oo`0jo`@07ol00`3oo`0/o`00
1ol500;o0P04o`805ol00`3oo`09o`D05Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200Wo00<0ool05ol500Co00<0ool06ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`03
0?oo00_o00<0ool05Ol00`3oo`09o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
1Ol301co00<0ool01?l00`3oo`0Mo`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`03
0?oo00[o00<0ool05Ol00`3oo`09o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
0_l00`3oo`0Mo`030?oo00?o00<0ool07_l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Ho`030?oo00;o1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103o
o`09o`801?l00`3oo`0No`030?oo00?o00<0ool07_l00`3oo`08o`800ol000Wo00<0ool02?l00`3o
o`0Ho`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo
00<0ool00_l2023o00<0ool00_l00`3oo`0Oo`030?oo00So00<0ool00_l000So0P0:o`<06?l00`3o
o`08o`806_l00`3oo`07o`802_l302;o00<0ool08_l200[o0`02o`030?oo01ko0P04o`030?oo01ko
0P0:o`<00_l002oo00<0ool09Ol00`3oo`0go`030?oo03?o00<0ool08ol00`3oo`0`o`00<?l00`3o
o`0Uo`80=ol00`3oo`0bo`030?oo02Co00<0ool0<?l0033o00<0ool09ol00`3oo`0do`030?oo03;o
00<0ool08_l203?o000ao`030?oo02Oo0P0do`030?oo03;o00<0ool08?l303Co000ao`030?oo02Wo
100`o`030?oo03;o00<0ool07_l203Oo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`040?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`05o`80
2?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00;o00D0oooo
0008o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3o
o`02o`030?oo02Wo0`0Zo`030?oo02Wo00<0ool01_l00`3oo`0Ko`030?oo00;o00<0ool0:_l00`3o
o`07o`00<ol00`3oo`0ao`80:?l00`3oo`0Ao`d04ol201co00<0ool0>_l003Co00<0ool0<_l202Ko
00<0ool03Ol400go100?o`030?oo01_o00<0ool0>_l003Go00<0ool0<ol00`3oo`0So`030?oo00[o
0`0Eo`<02ol201co00<0ool0>ol003Go00<0ool0=?l202?o00<0ool01_l401_o1P03o`<07Ol00`3o
o`0ko`00=_l00`3oo`0eo`808Ol00`3oo`03o`<09Ol401ko00<0ool0??l003Oo00<0ool0=_l401go
00<0o`000`1:o`030?oo03co000ho`030?oo03Wo3P0?o`80C_l00`3oo`0lo`00>Ol00`3oo`16oa00
C_l00`3oo`0mo`00>_l205Go00<0ool0;?l701So00<0ool0?_l003co0P1Co`030?oo02Wo0`07o`L0
4?l00`3oo`0oo`00?_l604go00<0ool09ol2017o0`0;o`80@_l004Co1019o`030?oo02Ko00<0ool0
5?l;04Co0018o`<0A_l00`3oo`0Uo`030?oo06Co001;o`<0;ol200Co0P03o`@01Ol00`3oo`0Uo`03
0?oo06Co001>o`80;?l0103oo`08o`030?oo00Ko00<0ool09?l00`3oo`1Uo`00D?l202[o00@0ool0
2Ol00`3oo`05o`809?l00`3oo`1Vo`00D_l202So00@0ool02_l00`3oo`04o`030?oo02;o00<0ool0
Iol005Co0`0Fo`L02?l0103oo`08o`040?oo00Go00<0ool08Ol00`3oo`1Xo`00Eol400ko1007o`80
1ol200[o0P06o`030?oo023o00<0ool0JOl005_o3P0=o`806Ol00`3oo`0Oo`030?oo06[o001ho`80
5ol00`3oo`0No`030?oo06_o001jo`030?oo01Co00<0ool07Ol00`3oo`1/o`00Nol00`3oo`0Co`03
0?oo01co00<0ool0KOl007co00<0ool04_l00`3oo`05o`T03?l2073o001mo`030?oo017o00<0ool0
0_l300Wo0P09o`030?oo073o001no`030?oo013o00<0ool00P0>o`T0Lol007oo0P0@o`<0S?l0087o
0P0=o`80S_l008?o1007o`8000?o0?l0SOl008Oo1`03o`030?oo08co001mo`801?l200Go0`04o`03
0?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87876 .16628 m
.85437 .14709 L
.83014 .12997 L
.81644 .11962 L
.80895 .11437 L
.80206 .11008 L
.79592 .10694 L
.79262 .10562 L
.7911 .10512 L
.78971 .10472 L
.78848 .10444 L
.78738 .10423 L
.78618 .10407 L
.78494 .10397 L
.78379 .10396 L
.78315 .10399 L
.78258 .10404 L
.78155 .10418 L
.78051 .10442 L
.77949 .10475 L
.7786 .10513 L
.77767 .10564 L
.77675 .10628 L
.77536 .10758 L
.77466 .10844 L
.77406 .10933 L
.77309 .11119 L
.77238 .11311 L
.77207 .11427 L
.77178 .11559 L
.77156 .11691 L
.77141 .11818 L
.7713 .11959 L
.77125 .12114 L
.77124 .12251 L
.77129 .12406 L
.77139 .12578 L
.77147 .12676 L
.77156 .12767 L
.77202 .1312 L
.77261 .13459 L
.7734 .13838 L
.77521 .14543 L
.78021 .16122 L
.78283 .16908 L
.78515 .17674 L
.78598 .18016 L
.78629 .18174 L
.78652 .18315 L
.78667 .18437 L
Mistroke
.78677 .18561 L
.7868 .1863 L
.78681 .18692 L
.78678 .18805 L
.78667 .18909 L
.78649 .19012 L
.78628 .19095 L
.78598 .19182 L
.78557 .19269 L
.78505 .19356 L
.78386 .19497 L
.78302 .19569 L
.78215 .1963 L
.78111 .1969 L
.77984 .19748 L
.77743 .19831 L
.77596 .19869 L
.7745 .199 L
.77151 .19949 L
.76854 .19984 L
.76505 .20015 L
.76115 .20041 L
.75349 .20087 L
.74596 .20143 L
.74177 .20186 L
.73727 .20246 L
.73328 .20315 L
.72972 .20393 L
.72357 .20575 L
.72035 .20703 L
.71754 .2084 L
.71252 .21168 L
.71011 .21381 L
.70818 .21594 L
.70641 .21837 L
.705 .22082 L
.7038 .22352 L
.70275 .22665 L
.70207 .22946 L
.70162 .23211 L
.70143 .23352 L
.70128 .23508 L
.70115 .23672 L
.70107 .23826 L
.70102 .23996 L
.70101 .24092 L
.701 .24181 L
.70101 .24343 L
.70104 .24516 L
.70107 .24669 L
Mistroke
.70112 .24811 L
.70123 .25131 L
.70129 .25304 L
.70134 .25466 L
.70137 .25609 L
.70138 .25761 L
.70138 .25846 L
.70137 .25922 L
.70135 .26009 L
.70132 .26089 L
.70124 .26228 L
.70118 .26304 L
.70111 .26373 L
.70096 .265 L
.70077 .26616 L
.70029 .2682 L
.69996 .26927 L
.6996 .2702 L
.69861 .27214 L
.69764 .27349 L
.69637 .27479 L
.69565 .27537 L
.69482 .27592 L
.69315 .27677 L
.6921 .27717 L
.69107 .27748 L
.68988 .27776 L
.68853 .27798 L
.68782 .27806 L
.68704 .27813 L
.68628 .27817 L
.68556 .2782 L
.68416 .2782 L
.68344 .27818 L
.68263 .27814 L
.68113 .27804 L
.67971 .27789 L
.67641 .27741 L
.67322 .2768 L
.6701 .27608 L
.65675 .27208 L
.64292 .26696 L
.63138 .26246 L
.62224 .25926 L
.61756 .25797 L
.61527 .25747 L
.61287 .25705 L
.61162 .25688 L
.61049 .25675 L
.60935 .25666 L
Mistroke
.60827 .25659 L
.60766 .25656 L
.60701 .25654 L
.60645 .25653 L
.60583 .25652 L
.60517 .25652 L
.60445 .25654 L
.60377 .25656 L
.60314 .25658 L
.60201 .25664 L
.60079 .25673 L
.59855 .25694 L
.59578 .25729 L
.59317 .25768 L
.58213 .2598 L
.55638 .26611 L
.52731 .27518 L
.50351 .28395 L
.49751 .28596 L
.49114 .28775 L
.48805 .28846 L
.4851 .28904 L
.48245 .28946 L
.4795 .28983 L
.47815 .28997 L
.47668 .29009 L
.47528 .29018 L
.474 .29024 L
.47262 .29029 L
.4711 .29031 L
.47033 .29032 L
.46951 .29031 L
.46802 .29029 L
.46661 .29025 L
.46508 .29019 L
.46363 .29011 L
.46229 .29001 L
.45703 .28952 L
.45127 .28877 L
.44502 .28777 L
.43316 .28563 L
.42832 .28477 L
.42348 .28403 L
.42142 .28376 L
.41934 .28354 L
.41747 .28338 L
.41668 .28333 L
.41586 .28328 L
.41537 .28326 L
.41485 .28324 L
Mistroke
.41394 .28322 L
.41344 .28321 L
.41291 .28321 L
.41241 .28321 L
.41197 .28322 L
.41112 .28324 L
.41065 .28325 L
.41021 .28327 L
.40861 .28335 L
.40568 .28357 L
.40414 .28368 L
.4033 .28374 L
.40251 .28379 L
.40171 .28383 L
.40098 .28386 L
.40016 .28389 L
.3997 .28389 L
.39926 .2839 L
.39848 .28389 L
.39775 .28387 L
.39693 .28384 L
.39604 .28379 L
.39524 .28373 L
.39437 .28365 L
.39274 .28344 L
.39069 .28311 L
.38872 .28272 L
.38396 .28147 L
.37824 .27953 L
.36669 .27448 L
.34384 .26147 L
.33373 .25495 L
.3254 .24984 L
.32237 .24831 L
.32088 .24769 L
.31958 .24724 L
.31841 .24691 L
.31737 .24668 L
.31622 .24649 L
.31498 .24633 L
.3126 .24609 L
.31015 .24581 L
.30881 .2456 L
.30727 .2453 L
.30364 .24431 L
.30156 .24356 L
.29946 .24267 L
.29523 .2405 L
.28637 .23425 L
.28136 .22959 L
Mistroke
.27696 .22462 L
.27268 .21863 L
.27078 .21537 L
.26906 .21188 L
.26781 .20876 L
.26727 .20712 L
.26678 .20537 L
.26642 .20378 L
.26617 .20234 L
.26599 .20093 L
.26589 .19961 L
.26586 .19889 L
.26586 .19811 L
.26589 .1973 L
.26594 .19656 L
.26609 .19526 L
.26621 .19455 L
.26634 .1939 L
.26669 .19258 L
.26719 .19119 L
.26782 .18982 L
.26852 .18862 L
.27007 .18661 L
.27183 .18495 L
.27365 .18368 L
.27584 .18253 L
.27846 .18155 L
.27987 .18116 L
.28147 .18082 L
.28288 .18059 L
.28367 .18049 L
.28443 .18042 L
.28592 .18031 L
.28731 .18027 L
.28883 .18027 L
.28969 .1803 L
.29048 .18034 L
.29195 .18044 L
.29351 .18059 L
.2962 .18097 L
.29912 .1815 L
.3053 .18305 L
.31047 .18472 L
.31753 .1874 L
.31989 .18827 L
.3203 .1884 L
.32068 .18849 L
.32087 .18853 L
.32101 .18854 L
.32125 .18854 L
Mistroke
.32134 .18852 L
.32141 .18848 L
.32146 .18842 L
.32149 .18834 L
.32149 .18824 L
.32147 .18813 L
.32137 .18789 L
.32119 .18758 L
.32097 .18725 L
.32024 .18631 L
.31497 .18047 L
.29778 .1628 L
.28774 .15357 L
.27662 .14517 L
.27016 .14145 L
.26421 .13884 L
.26132 .13786 L
.25815 .13698 L
.25514 .13632 L
.25232 .13585 L
.25089 .13566 L
.24938 .13549 L
.24807 .13536 L
.24666 .13526 L
.24513 .13517 L
.2437 .1351 L
.24283 .13507 L
.242 .13505 L
.24044 .13503 L
.23906 .13502 L
.23759 .13502 L
.23634 .13504 L
.23495 .13506 L
.23346 .13509 L
.23185 .13513 L
.22893 .13522 L
.22553 .13532 L
.22374 .13538 L
.22182 .13542 L
.21999 .13546 L
.21833 .13549 L
.21667 .1355 L
.2151 .1355 L
.21358 .1355 L
.21198 .13548 L
.21063 .13545 L
.20915 .13541 L
.20756 .13536 L
.20588 .13528 L
.20289 .13511 L
Mistroke
.19985 .13488 L
.1967 .13459 L
.19131 .134 L
.18548 .1333 L
.18282 .13301 L
.1805 .13279 L
.17948 .13272 L
.17855 .13267 L
.17761 .13263 L
.17659 .13261 L
.17559 .13262 L
.17469 .13265 L
.17368 .13272 L
.17276 .13281 L
.17182 .13294 L
.17086 .13312 L
.16919 .13355 L
.16746 .13418 L
.16564 .13508 L
.16412 .13605 L
.1625 .1373 L
.15964 .14008 L
.15437 .14695 L
.14197 .16659 L
.13001 .18435 L
.12792 .18756 L
.12614 .19069 L
.125 .19325 L
.12453 .1947 L
.12421 .19606 L
.12408 .19681 L
.12399 .19753 L
.12393 .19818 L
.1239 .19891 L
.1239 .19971 L
.12391 .20013 L
.12393 .20059 L
.12399 .20143 L
.12408 .20224 L
.12422 .2032 L
.12439 .20415 L
.12479 .20602 L
.12612 .2107 L
.13476 .23263 L
.14104 .24625 L
.14887 .26114 L
.15712 .27403 L
.16538 .28396 L
.17029 .28855 L
.17561 .2925 L
Mistroke
.18104 .29557 L
.18417 .29695 L
.18702 .29798 L
.19215 .29936 L
.19494 .2999 L
.19755 .30029 L
.19992 .30057 L
.20208 .30076 L
.20682 .30106 L
.21074 .30126 L
.21467 .30149 L
.21664 .30165 L
.21874 .30186 L
.22227 .30237 L
.22422 .30275 L
.22623 .30322 L
.22972 .30427 L
.23614 .30692 L
.26316 .31929 L
.27851 .32461 L
.28576 .3271 L
.29434 .32976 L
.2986 .33084 L
.30103 .33136 L
.30341 .33179 L
.30567 .33213 L
.30819 .33243 L
.30932 .33254 L
.31054 .33264 L
.31171 .33271 L
.31278 .33276 L
.31413 .3328 L
.31558 .33281 L
.31684 .3328 L
.31823 .33277 L
.31974 .33271 L
.32059 .33266 L
.32138 .33261 L
.32438 .33234 L
.32721 .33199 L
.33032 .33151 L
.33609 .3304 L
.36041 .32393 L
.36682 .3224 L
.37033 .32174 L
.37198 .32149 L
.37349 .3213 L
.37481 .32116 L
.37621 .32106 L
.37701 .32102 L
Mistroke
.37775 .32099 L
.37914 .32098 L
.37985 .32099 L
.38059 .32102 L
.38191 .3211 L
.38333 .32124 L
.38413 .32135 L
.38487 .32146 L
.38743 .322 L
.38887 .32239 L
.39019 .32281 L
.39515 .32493 L
.39964 .32756 L
.40446 .33103 L
.42771 .35138 L
.44031 .3608 L
.44794 .36527 L
.45522 .36859 L
.46178 .37079 L
.46553 .3717 L
.46904 .37234 L
.47095 .37259 L
.47276 .37277 L
.47437 .37289 L
.47612 .37296 L
.47802 .37298 L
.48002 .37293 L
.48173 .37283 L
.48359 .37267 L
.48554 .37243 L
.48734 .37215 L
.49136 .37133 L
.49483 .3704 L
.49842 .36921 L
.51053 .36361 L
.53191 .34996 L
.53611 .34773 L
.53815 .34688 L
.54035 .34615 L
.5415 .34587 L
.54213 .34574 L
.54273 .34563 L
.54328 .34555 L
.54389 .34548 L
.54444 .34543 L
.54495 .3454 L
.54593 .34537 L
.54649 .34537 L
.54701 .34539 L
.54808 .34546 L
Mistroke
.54868 .34552 L
.54924 .34559 L
.55125 .34593 L
.55316 .34639 L
.55766 .34784 L
.56753 .35187 L
.57274 .35398 L
.57861 .35607 L
.58136 .35689 L
.58429 .35764 L
.58683 .35817 L
.58824 .3584 L
.58957 .35859 L
.59084 .35874 L
.59222 .35886 L
.59347 .35893 L
.59462 .35896 L
.59581 .35896 L
.59688 .35893 L
.59811 .35886 L
.59927 .35876 L
.60054 .35861 L
.60187 .3584 L
.60421 .3579 L
.60669 .35719 L
.60934 .35621 L
.61439 .35364 L
.61888 .3505 L
.6265 .34312 L
.6337 .33374 L
.64067 .32301 L
.65235 .30589 L
.65563 .30209 L
.65928 .29854 L
.66287 .29574 L
.66638 .2936 L
.67013 .29189 L
.67222 .29117 L
.67454 .29055 L
.67683 .29008 L
.678 .2899 L
.67909 .28976 L
.68008 .28966 L
.68119 .28957 L
.68233 .2895 L
.68356 .28946 L
.6848 .28944 L
.686 .28945 L
.6871 .28948 L
.68831 .28953 L
Mistroke
.68966 .28961 L
.69112 .28972 L
.69382 .28999 L
.6989 .29068 L
.70489 .29172 L
.73512 .29852 L
.74921 .30149 L
.75699 .3028 L
.76061 .30327 L
.76235 .30347 L
.76393 .30362 L
.76544 .30374 L
.76707 .30385 L
.76868 .30393 L
.77036 .30397 L
.77175 .30398 L
.77321 .30395 L
.77456 .3039 L
.77577 .30382 L
.77724 .30369 L
.77856 .30354 L
.77997 .30332 L
.78143 .30304 L
.784 .30239 L
.78619 .30165 L
.78849 .30063 L
.79046 .29952 L
.7924 .29815 L
.79435 .29644 L
.7972 .29309 L
.7997 .28898 L
.80191 .28409 L
.80369 .2791 L
.80665 .26932 L
.81003 .25911 L
.81198 .25453 L
.81435 .25016 L
.81662 .247 L
.81786 .24561 L
.81918 .24436 L
.82036 .24344 L
.82166 .24262 L
.82282 .24203 L
.82389 .24162 L
.82443 .24145 L
.82501 .2413 L
.82555 .2412 L
.82604 .24113 L
.82647 .24108 L
.82694 .24106 L
Mistroke
.82741 .24105 L
.82785 .24107 L
.82834 .24111 L
.8288 .24116 L
.82919 .24123 L
.8296 .24131 L
.83048 .24155 L
.83122 .2418 L
.83241 .2423 L
.83344 .24275 L
.83367 .24283 L
.83389 .24291 L
.83402 .24294 L
.83413 .24297 L
.83434 .243 L
.83452 .243 L
.8347 .24298 L
.83487 .24293 L
.83502 .24286 L
.8352 .24274 L
.83536 .2426 L
.83573 .24215 L
.8361 .24153 L
.83654 .24064 L
.83874 .23557 L
.86455 .19205 L
.87876 .16628 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`17o`@0@Ol0097o00<0ool0A_l0103oool303oo001mo`801?l200Go0`04o`030?oo04Go
00<0ool01Ol203go001lo`040?oo00_o00<0ool00ol00`3oo`15o`030?oo00Oo0P0ko`00O?l0103o
o`08o`D01?l204Ko00<0ool02Ol00`3oo`0ho`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Go
00<0ool02_l00`3oo`0go`00O?l0103oo`09o`030?l000Go00<0ool0AOl00`3oo`0;o`030?oo03Ko
001mo`802ol200Go00<0ool0AOl00`3oo`0<o`030?oo03Go000ho`/01?l704?o00<0ool0AOl00`3o
o`0=o`80=Ol003Oo00<0ool02Ol500Go1P0no`030?oo04Go00<0ool03ol00`3oo`0bo`00=_l00`3o
o`0Jo`80??l00`3oo`16o`030?oo00oo00<0ool0<Ol003Go00<0ool07Ol00`3oo`0io`030?oo04Ko
00<0ool04?l2037o000eo`030?oo01ko0P0io`030?oo04Ko00<0ool04_l00`3oo`0^o`00=?l00`3o
o`0Qo`030?oo03Ko00<0ool0Aol00`3oo`0Bo`030?oo02go000co`030?oo02?o00<0ool0=Ol00`3o
o`17o`030?oo01?o0P0]o`00<ol00`3oo`0To`030?oo03Co00<0ool0Aol00`3oo`0Eo`030?oo02[o
000bo`030?oo02Ko00<0ool0<ol00`3oo`18o`030?oo01Go00<0ool0:Ol0037o00<0ool0:?l00`3o
o`0bo`030?oo04So00<0ool05_l00`3oo`0Xo`00<?l00`3oo`0Zo`030?oo037o00<0ool0B?l00`3o
o`0Eo`030?oo02Wo000`o`030?oo02[o00<0ool07Ol200Co0P03o`@01Ol00`3oo`18o`030?oo01Go
00<0ool0:Ol002oo00<0ool09Ol300Co00<0ool06ol0103oo`08o`030?oo00Ko00<0ool0B?l201Go
00<0ool0:_l002oo00<0ool08_l400;o10000ol0000Ko`040?oo00Wo00<0ool01Ol204[o00<0ool0
4ol00`3oo`0Zo`00;_l00`3oo`0Ro`802Ol401Co1002o`040?oo00[o00<0ool01?l00`3oo`19o`03
0?oo01;o00<0ool0:ol002go00<0ool08ol00`3oo`0;o`030?oo01So00@0ool02?l0103oo`05o`03
0?oo04Wo00<0ool04_l00`3oo`0[o`00;Ol00`3oo`0Ro`030?oo02So0P0:o`801_l00`3oo`18o`80
4ol00`3oo`0/o`00;?l202Co00<0ool0??l00`3oo`16o`<04ol00`3oo`0]o`00;?l202Co00<0ool0
??l00`3oo`0lo`X05_l00`3oo`0]o`00;Ol00`3oo`0Ro`030?oo03co00<0ool0>?l401oo00<0ool0
;_l000Oo1@02o`801?l201Oo00<0ool02Ol501Go00<0ool02_l200Co0P04o`808ol00`3oo`0Ro`80
1?l200Co0P06o`806ol500?o00<0ool07?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`03
0?oo00_o00<0ool05Ol00`3oo`09o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
0ol201oo00<0ool00_l00`3oo`0Oo`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Do`03
0?oo00_o00<0ool05_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00D0oooo
000Po`030?oo00;o00<0ool07ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`03
0?oo00;o1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`80
0ol00`3oo`0Oo`050?ooo`008_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`030?oo00[o
00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool00_l00`3o
o`0Oo`050?ooo`008_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201Wo00<0ool0
2?l200[o0`0Ro`030?oo02;o0P0:o`<000?o0?l08?l200;o00<0ool08?l200[o0`02o`00;ol00`3o
o`0To`80>Ol00`3oo`0bo`030?oo02?o00<0ool0<Ol0033o00<0ool09Ol203Oo00<0ool0<_l00`3o
o`0Ro`030?oo03;o000`o`030?oo02Oo00<0ool0=?l00`3oo`0bo`030?oo02;o00<0ool0<_l0037o
00<0ool09ol203Co00<0ool0<_l00`3oo`0Lo`H0=Ol0037o00<0ool0:Ol502oo00<0ool0<_l00`3o
o`0Lo`030?oo03So000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`04
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l0103oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0
:Ol302[o00<0ool05?l;00[o00<0ool01_l00`3oo`0Jo`030?oo00?o00<0ool0:_l00`3oo`07o`00
<ol00`3oo`0ao`030?oo02Oo00<0ool03ol500_o0P0Ao`030?oo01[o00<0ool0>_l003Co00<0ool0
<Ol202Oo00<0ool02_l501;o0P0?o`030?oo01Wo00<0ool0>ol003Co00<0ool0<ol00`3oo`0To`03
0?oo00So0P0Io`803Ol00`3oo`0Io`030?oo03_o000eo`030?oo03?o0P0To`030?oo00Ko0P0Mo`@0
2?l00`3oo`0Jo`030?oo03_o000fo`030?oo03Co0P0Ro`030?oo00?o0`0So`T06ol00`3oo`0lo`00
=_l00`3oo`0fo`@07_l00`3o000304[o00<0ool0??l003Oo00<0ool0>Ol>013o0P1>o`030?oo03co
000ho`80Aol400Oo1P0_o`807Ol00`3oo`0mo`00>_l00`3oo`18o`L01Ol00`3oo`0Zo`@000?o0000
100Go`030?oo03go000ko`80E?l00`3oo`0Xo`802ol401;o00<0ool0?_l003go1@1?o`030?oo02Oo
00<0ool03ol400co0P11o`00@Ol904Oo00<0ool09_l00`3oo`0Do`d0@_l004[o0P15o`030?oo02Go
00<0ool0I?l004co0P0_o`801?l200?o1005o`030?oo02Co00<0ool0IOl004ko0`0[o`040?oo00So
00<0ool01_l00`3oo`0To`030?oo06Go001Ao`80:Ol0103oo`09o`030?oo00Go0P0To`030?oo06Ko
001Co`805ol800So00@0ool02_l00`3oo`04o`030?oo02;o00<0ool0Iol005Go0P0Ao`@02?l200Ko
00@0ool02?l0103oo`05o`030?oo027o00<0ool0J?l005Oo1009o`@03_l00`3oo`04o`802_l200Ko
00<0ool08Ol00`3oo`1Xo`00Fol901?o00<0ool05ol00`3oo`0Po`030?oo06Wo001ho`030?oo01Ko
00<0ool07ol00`3oo`1Zo`00NOl00`3oo`0Eo`030?oo01oo00<0ool0J_l007[o0P0Eo`030?oo00So
0`0Co`030?oo06_o001lo`030?oo01;o00<0ool01_l200;o0`0@o`030?oo06co001mo`030?oo017o
00<0ool01?l200Oo0P0=o`030?oo06go001no`804Ol00`3oo`02o`802ol400Oo0P1`o`00P?l200oo
00<0ool00P0Ao`L0L_l008;o00<0ool03?l00`3o002<o`00Pol00`3oo`0;o`80SOl008Co1005o`D0
S_l007go0P04o`800ol600?o00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`04
0?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`00
1Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0
S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87947 .16805 m
.85474 .15061 L
.82937 .13186 L
.81512 .12002 L
.80738 .1139 L
.80027 .10883 L
.79399 .10507 L
.79085 .10353 L
.78769 .10229 L
.78646 .10191 L
.78517 .10159 L
.78446 .10144 L
.78381 .10133 L
.78258 .10118 L
.78152 .10113 L
.78041 .10116 L
.77941 .10128 L
.77852 .10146 L
.7776 .10174 L
.77666 .10215 L
.77582 .10262 L
.77511 .10314 L
.77388 .10433 L
.77333 .10503 L
.77278 .10589 L
.77226 .10688 L
.77183 .10792 L
.7714 .1092 L
.77108 .11047 L
.77085 .11173 L
.77068 .11297 L
.77056 .1144 L
.7705 .11599 L
.7705 .11746 L
.77057 .11915 L
.77071 .12097 L
.77089 .12273 L
.77138 .12601 L
.77209 .12968 L
.77437 .13855 L
.77709 .14693 L
.78289 .16227 L
.78846 .17678 L
.79037 .18295 L
.79074 .18448 L
.79104 .18604 L
.79125 .18743 L
.79136 .18864 L
.7914 .18974 L
.79136 .19071 L
Mistroke
.79125 .19172 L
.79104 .19271 L
.79075 .19359 L
.7904 .19437 L
.79002 .19501 L
.78953 .19565 L
.78886 .19634 L
.78814 .19692 L
.78725 .19749 L
.78617 .19802 L
.7849 .1985 L
.78358 .19888 L
.7823 .19917 L
.78079 .19942 L
.7794 .1996 L
.77779 .19975 L
.77694 .19981 L
.776 .19987 L
.77424 .19994 L
.77251 .19998 L
.77062 .2 L
.76895 .2 L
.76707 .19999 L
.76291 .19993 L
.76064 .19988 L
.7582 .19984 L
.75603 .19981 L
.75478 .1998 L
.75361 .19979 L
.75154 .19979 L
.74931 .1998 L
.74715 .19983 L
.74479 .1999 L
.74256 .19999 L
.74051 .2001 L
.73668 .2004 L
.73478 .2006 L
.7327 .20086 L
.72817 .20161 L
.72422 .20251 L
.72014 .20376 L
.7162 .20538 L
.71308 .20706 L
.71048 .20885 L
.70795 .21106 L
.70565 .2137 L
.70387 .21643 L
.70237 .21956 L
.7014 .22242 L
.70097 .22411 L
Mistroke
.70064 .22573 L
.70049 .22664 L
.70035 .22764 L
.70024 .22853 L
.70015 .22949 L
.70008 .23045 L
.70003 .23133 L
.69998 .23306 L
.69997 .23411 L
.69998 .23508 L
.70001 .23615 L
.70005 .23729 L
.7001 .23829 L
.70017 .23921 L
.70034 .24133 L
.7021 .25508 L
.70257 .25861 L
.70278 .26048 L
.70295 .26218 L
.70306 .26358 L
.70314 .26505 L
.70316 .26582 L
.70318 .26662 L
.70317 .26799 L
.7031 .2693 L
.70304 .27001 L
.70297 .27063 L
.70279 .2718 L
.70257 .27285 L
.70224 .274 L
.70187 .27499 L
.70138 .276 L
.70077 .27697 L
.70021 .27768 L
.69954 .27838 L
.69815 .27945 L
.69718 .27999 L
.69618 .28042 L
.69522 .28074 L
.6941 .28102 L
.69294 .28122 L
.69178 .28135 L
.6907 .28141 L
.68948 .28143 L
.68809 .28138 L
.68737 .28134 L
.68656 .28127 L
.68509 .2811 L
.68368 .2809 L
.6808 .28036 L
Mistroke
.67806 .27973 L
.67161 .27787 L
.65712 .27242 L
.64428 .26667 L
.63124 .26053 L
.62575 .25809 L
.62116 .25629 L
.61732 .25505 L
.61536 .25454 L
.61331 .25411 L
.61217 .25392 L
.61114 .25378 L
.61009 .25366 L
.60912 .25358 L
.60813 .25352 L
.60704 .25349 L
.60607 .25348 L
.60504 .25349 L
.6039 .25353 L
.60284 .25359 L
.6016 .25368 L
.6004 .2538 L
.5957 .25443 L
.59012 .25544 L
.58343 .25689 L
.52545 .27465 L
.51289 .27982 L
.50049 .28492 L
.49503 .28692 L
.48932 .28869 L
.48441 .28989 L
.48181 .2904 L
.47895 .29084 L
.47741 .29103 L
.47598 .29118 L
.47463 .2913 L
.47313 .2914 L
.47179 .29147 L
.47052 .29151 L
.46917 .29154 L
.46772 .29154 L
.46614 .29152 L
.46524 .29149 L
.4644 .29146 L
.46287 .29139 L
.46123 .29128 L
.45809 .29101 L
.45523 .2907 L
.44368 .2889 L
.43347 .2869 L
Mistroke
.42358 .28502 L
.41932 .28438 L
.41711 .28413 L
.41618 .28405 L
.41519 .28398 L
.41431 .28393 L
.41353 .28391 L
.4128 .28389 L
.41204 .28389 L
.41138 .2839 L
.41078 .28392 L
.41013 .28395 L
.40944 .28399 L
.40804 .28411 L
.40662 .28428 L
.40412 .28462 L
.40282 .2848 L
.40138 .28497 L
.4007 .28504 L
.39996 .2851 L
.39932 .28514 L
.3986 .28518 L
.39817 .28519 L
.39776 .2852 L
.39728 .28521 L
.39682 .2852 L
.39598 .28518 L
.39549 .28516 L
.39504 .28514 L
.39423 .28508 L
.3933 .28499 L
.39134 .28472 L
.38935 .28435 L
.38741 .28391 L
.38265 .28253 L
.3728 .27862 L
.34947 .26566 L
.32887 .25165 L
.32557 .24949 L
.32258 .24778 L
.32138 .24722 L
.32021 .24676 L
.31926 .24648 L
.31875 .24636 L
.31827 .24627 L
.31799 .24622 L
.31769 .24618 L
.31716 .24613 L
.31687 .24611 L
.31655 .2461 L
Mistroke
.31625 .24609 L
.31597 .24609 L
.31543 .2461 L
.31512 .2461 L
.31483 .24611 L
.3137 .24616 L
.31319 .24618 L
.31262 .24619 L
.31197 .2462 L
.31133 .2462 L
.31098 .24619 L
.31059 .24618 L
.30985 .24614 L
.3091 .24607 L
.30837 .24599 L
.30678 .24573 L
.30521 .24538 L
.30335 .24485 L
.30125 .24411 L
.2974 .24236 L
.29282 .2397 L
.28753 .23584 L
.28183 .23067 L
.27706 .22538 L
.27255 .21917 L
.27059 .21592 L
.26872 .21229 L
.26729 .20892 L
.26626 .20586 L
.2658 .20416 L
.2654 .20232 L
.26522 .20133 L
.26509 .20042 L
.265 .19956 L
.26493 .19865 L
.26489 .19782 L
.26488 .19693 L
.26489 .1961 L
.26494 .19535 L
.26508 .19396 L
.26534 .19255 L
.26567 .19125 L
.26605 .19011 L
.26655 .18894 L
.26718 .18773 L
.26855 .18577 L
.27012 .18414 L
.2723 .18252 L
.27352 .18183 L
.27496 .18118 L
Mistroke
.27625 .1807 L
.2777 .18027 L
.27913 .17994 L
.28049 .17971 L
.28202 .17954 L
.2835 .17944 L
.28486 .17941 L
.28636 .17943 L
.28802 .17952 L
.28886 .1796 L
.28979 .17969 L
.29301 .18018 L
.29604 .18081 L
.29926 .18167 L
.30488 .18357 L
.31646 .18898 L
.3227 .19271 L
.32348 .1932 L
.32374 .19336 L
.32392 .19346 L
.32403 .1935 L
.32408 .19349 L
.32404 .1934 L
.32398 .19333 L
.3239 .19323 L
.32069 .1899 L
.3027 .17071 L
.29136 .159 L
.28026 .14906 L
.27408 .14455 L
.26732 .14065 L
.26431 .13928 L
.26111 .13804 L
.25833 .13716 L
.25528 .13638 L
.25216 .13577 L
.25072 .13554 L
.24917 .13534 L
.24763 .13517 L
.24622 .13504 L
.24488 .13494 L
.24346 .13485 L
.24178 .13478 L
.24023 .13474 L
.23855 .13471 L
.23759 .1347 L
.23669 .1347 L
.23507 .13471 L
.23358 .13472 L
.23185 .13475 L
Mistroke
.23021 .13478 L
.22454 .13491 L
.22316 .13493 L
.22167 .13496 L
.22004 .13498 L
.21851 .13499 L
.21683 .13499 L
.21599 .13499 L
.21508 .13498 L
.21345 .13495 L
.21195 .13492 L
.21044 .13487 L
.20907 .13482 L
.20602 .13466 L
.2031 .13445 L
.19996 .13415 L
.19387 .1334 L
.18406 .13178 L
.17923 .13099 L
.17794 .13081 L
.17675 .13066 L
.1757 .13056 L
.17514 .13051 L
.17462 .13048 L
.17362 .13043 L
.17273 .13042 L
.17193 .13044 L
.17108 .13049 L
.17064 .13053 L
.17017 .13058 L
.16924 .13073 L
.1684 .13091 L
.16764 .13112 L
.16605 .13173 L
.16465 .13246 L
.16318 .13348 L
.16167 .13479 L
.15921 .13753 L
.15653 .14128 L
.14631 .15888 L
.13364 .17846 L
.12848 .1853 L
.12647 .18809 L
.12501 .19043 L
.12442 .19157 L
.12389 .1928 L
.12365 .19348 L
.12347 .19411 L
.12332 .19473 L
.12319 .19539 L
Mistroke
.1231 .19608 L
.12304 .19675 L
.12302 .19716 L
.12301 .19756 L
.12302 .19832 L
.12304 .19872 L
.12307 .19916 L
.12315 .19996 L
.12327 .20085 L
.12344 .20189 L
.12432 .20557 L
.12576 .21022 L
.12958 .22056 L
.14209 .2494 L
.15063 .26559 L
.15578 .27381 L
.16093 .28081 L
.16606 .28663 L
.17181 .29186 L
.17716 .29559 L
.18214 .29821 L
.18684 .29999 L
.18949 .30075 L
.19196 .3013 L
.19453 .30174 L
.19684 .30202 L
.19811 .30214 L
.19948 .30224 L
.20076 .30231 L
.20193 .30236 L
.20319 .30239 L
.20381 .30241 L
.2045 .30241 L
.20569 .30242 L
.2068 .30241 L
.20746 .30241 L
.20808 .3024 L
.20923 .30239 L
.21047 .30237 L
.21176 .30235 L
.21278 .30233 L
.21389 .30232 L
.21488 .30232 L
.21579 .30233 L
.21685 .30234 L
.2178 .30237 L
.21883 .30241 L
.21988 .30246 L
.22076 .30253 L
.22155 .3026 L
Mistroke
.22327 .3028 L
.22481 .30304 L
.22622 .30331 L
.22925 .30411 L
.23236 .30523 L
.2436 .31115 L
.25661 .31748 L
.27063 .32246 L
.2849 .32737 L
.2889 .32863 L
.2935 .32988 L
.2957 .33039 L
.29811 .33087 L
.30044 .33126 L
.30264 .33155 L
.30386 .33168 L
.3052 .33179 L
.30658 .33189 L
.30734 .33192 L
.30805 .33195 L
.30938 .33198 L
.31061 .33199 L
.31205 .33197 L
.31343 .33193 L
.31415 .33189 L
.31496 .33184 L
.31661 .33172 L
.31963 .3314 L
.32323 .33086 L
.32668 .33021 L
.34008 .32665 L
.35297 .32232 L
.35936 .32018 L
.36522 .31843 L
.37031 .31723 L
.37168 .31698 L
.3731 .31675 L
.37443 .31658 L
.37562 .31646 L
.37692 .31637 L
.37829 .31632 L
.37955 .31632 L
.38073 .31637 L
.38209 .31648 L
.38334 .31663 L
.38469 .31685 L
.38612 .31717 L
.38875 .31794 L
.39119 .31891 L
.39669 .32197 L
Mistroke
.40144 .32553 L
.40588 .32947 L
.41649 .34014 L
.4279 .35149 L
.44169 .36283 L
.45031 .36821 L
.45849 .37204 L
.46251 .37347 L
.46691 .37469 L
.46893 .37514 L
.47108 .37553 L
.47291 .3758 L
.47492 .37602 L
.47705 .37617 L
.47903 .37625 L
.48118 .37624 L
.48238 .3762 L
.48346 .37615 L
.48544 .37599 L
.48655 .37586 L
.48756 .37573 L
.49135 .37508 L
.49469 .37428 L
.49826 .3732 L
.50515 .37044 L
.51216 .36674 L
.51897 .36231 L
.52949 .35432 L
.53394 .35091 L
.53827 .34799 L
.54042 .3468 L
.54273 .34579 L
.54373 .34544 L
.5448 .34515 L
.5458 .34493 L
.54673 .34479 L
.54766 .34471 L
.54866 .34468 L
.5496 .34471 L
.55047 .34478 L
.551 .34484 L
.5515 .34492 L
.55263 .34513 L
.55472 .34569 L
.5587 .34719 L
.56319 .34932 L
.57246 .35413 L
.57797 .35677 L
.5834 .35898 L
.58855 .36061 L
Mistroke
.59147 .3613 L
.59287 .36156 L
.59417 .36176 L
.59535 .36191 L
.59643 .36202 L
.59763 .3621 L
.59891 .36214 L
.60007 .36214 L
.60132 .3621 L
.60263 .36201 L
.60383 .36188 L
.6052 .36167 L
.60663 .36139 L
.60915 .36072 L
.61178 .35977 L
.61459 .35845 L
.61936 .35541 L
.6235 .35185 L
.6304 .34371 L
.63678 .33358 L
.64789 .31256 L
.65351 .30316 L
.65672 .29881 L
.65987 .29531 L
.6631 .29248 L
.66489 .29122 L
.66685 .29005 L
.66857 .2892 L
.67054 .28841 L
.6724 .28782 L
.67422 .28738 L
.6752 .28719 L
.67629 .28701 L
.67728 .28689 L
.67835 .28678 L
.67933 .28671 L
.68043 .28666 L
.68146 .28663 L
.68245 .28663 L
.68364 .28666 L
.68492 .28671 L
.68603 .28679 L
.68728 .28689 L
.69018 .28721 L
.693 .28763 L
.69916 .28878 L
.70643 .29045 L
.72103 .29433 L
.73634 .29854 L
.75093 .30216 L
Mistroke
.75876 .30374 L
.76207 .3043 L
.76559 .3048 L
.76754 .30502 L
.76931 .30519 L
.77097 .30532 L
.7727 .30541 L
.77446 .30547 L
.77608 .30547 L
.77699 .30545 L
.77781 .30542 L
.7787 .30538 L
.77961 .30531 L
.78117 .30515 L
.78255 .30496 L
.78408 .30468 L
.78548 .30436 L
.78782 .30365 L
.79017 .30267 L
.79252 .30135 L
.79475 .29966 L
.79644 .298 L
.79809 .29594 L
.79939 .29389 L
.80063 .29146 L
.80249 .28657 L
.8039 .28134 L
.80497 .27621 L
.80724 .26356 L
.80848 .25766 L
.8098 .25267 L
.81133 .2483 L
.81317 .24441 L
.81521 .24134 L
.81643 .24 L
.8176 .239 L
.81865 .23832 L
.81916 .23805 L
.81964 .23785 L
.82006 .2377 L
.82051 .23757 L
.82097 .23747 L
.82145 .23741 L
.82196 .23738 L
.82241 .23739 L
.8229 .23744 L
.8234 .23754 L
.82383 .23765 L
.82421 .23778 L
.82503 .23817 L
Mistroke
.82898 .24251 L
.82925 .24296 L
.82946 .24325 L
.82956 .24335 L
.82961 .24339 L
.82966 .24341 L
.82971 .24343 L
.82977 .24343 L
.82983 .24341 L
.82988 .24339 L
.82998 .2433 L
.83008 .24318 L
.83029 .24282 L
.83094 .24141 L
.83878 .22575 L
.85518 .2016 L
.87947 .16805 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`17o`80
@ol0097o00<0ool0AOl20003o`000080@?l0097o00<0ool0AOl00`3oo`04o`030?oo03go001mo`80
1?l200Go0`04o`030?oo04Go00<0ool01Ol00`3oo`0lo`00O?l0103oo`0;o`030?oo00?o00<0ool0
A?l200So0P0lo`00O?l0103oo`08o`D01?l204Go00<0ool02Ol00`3oo`0io`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Go00<0ool02Ol00`3oo`0ho`00O?l0103oo`09o`030?l000Go00<0ool0
AOl00`3oo`0:o`030?oo03Oo000ho`@0@Ol200_o0P05o`030?oo04Go00<0ool02ol00`3oo`0fo`00
=ol200?o4`12o`030?oo04Go00<0ool03?l00`3oo`0eo`00=_l00`3oo`0Eo`@0?ol00`3oo`15o`03
0?oo00go0P0eo`00=Ol00`3oo`0Jo`80?Ol00`3oo`15o`030?oo00oo00<0ool0<_l003Co00<0ool0
7Ol203_o00<0ool0A_l00`3oo`0?o`030?oo037o000do`030?oo01oo00<0ool0>?l00`3oo`16o`03
0?oo013o0P0ao`00<ol00`3oo`0Qo`030?oo03Oo00<0ool0Aol00`3oo`0Ao`030?oo02ko000co`03
0?oo02;o00<0ool0=_l00`3oo`17o`030?oo01;o0P0^o`00<_l00`3oo`0To`030?oo03Go00<0ool0
B?l00`3oo`0Co`030?oo02_o000ao`030?oo02Ko00<0ool0=?l00`3oo`18o`030?oo01Co0P0[o`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0B?l00`3oo`0Fo`030?oo02So000`o`030?oo02Wo00<0ool0
<_l00`3oo`19o`030?oo01Co00<0ool0:Ol0033o00<0ool0:_l00`3oo`0Mo`801?l200?o1005o`03
0?oo04Wo00<0ool05?l00`3oo`0Yo`00;ol00`3oo`0So`D01?l00`3oo`0Ko`040?oo00So00<0ool0
1_l00`3oo`19o`030?oo01?o00<0ool0:_l002ko00<0ool08_l300Co10000ol0o`0Ko`040?oo00Wo
00<0ool01Ol204_o00<0ool04Ol00`3oo`0[o`00;_l00`3oo`0Ro`030?oo00So0`0Eo`@00_l0103o
o`0:o`030?oo00Co00<0ool0B_l00`3oo`0Ao`030?oo02_o000]o`030?oo02;o00<0ool02ol201[o
00@0ool02?l0103oo`05o`030?oo04[o00<0ool04?l00`3oo`0/o`00;?l00`3oo`0So`030?oo00co
0P0Jo`802_l200Ko00<0ool0B_l00`3oo`0?o`030?oo02go000/o`030?oo02?o00<0ool0??l00`3o
o`18o`<04Ol00`3oo`0]o`00;?l00`3oo`0So`030?oo03co00<0ool0>?l@01?o00<0ool0;_l002go
00<0ool08_l00`3oo`0lo`030?oo03Ko0P0Ro`030?oo02oo0007o`D00_l200Co0P0Go`030?oo00Wo
1@0Eo`030?oo00[o0P04o`801?l202?o00<0ool08_l200Co0P04o`801?l201go1@02o`030?oo01go
1@02o`801?l200?o0009o`030?oo00Oo00@0ool05_l00`3oo`0;o`030?oo01Go00<0ool02Ol0103o
o`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00;o0P0Po`050?ooo`008_l00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Go00<0ool02Ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`040?oo027o00D0oooo000Ro`030?oo00[o0@01o`7o00000ol0
000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l00`3oo`02o`@00_l0103oo`09o`80
8ol00`3oo`0Qo`040?oo00Wo0P02o`030?oo023o00@0ool08ol00`3oo`08o`800ol000Wo00<0ool0
2?l00`3oo`0Go`030?oo00[o00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`04
0?oo00Wo00D0oooo000Ro`040?oo02?o00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So
0P0Io`030?oo00So0P0:o`<08_l00`3oo`0Ro`802_l30003o`3o023o0P000ol0o`0Ro`802_l300;o
000_o`030?oo02Co0P0io`030?oo03;o00<0ool08_l00`3oo`0bo`00<?l00`3oo`0Uo`030?oo03Ko
00<0ool0<_l00`3oo`0Qo`030?oo03?o000`o`030?oo02Ko0P0fo`030?oo03;o00<0ool07?l300;o
00<0ool0<ol0033o00<0ool0:?l203Co00<0ool0<_l00`3oo`0Ko`040?ooo`80=_l0037o00<0ool0
:?l602oo00<0ool0<_l00`3oo`0Ko`030?oo03Wo000<ool03009o`003?l00`3oo`05o`030?oo00Ko
00<0ool01_l00`3oo`06o`040?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko
00<0ool00ol900?o00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`040?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co
00<0ool00_l00`3oo`0Yo`<0:_l00`3oo`0Bo`802Ol200[o00<0ool01_l00`3oo`0Jo`030?oo00?o
00<0ool0:_l00`3oo`07o`00<ol00`3oo`0ao`030?oo02Oo00<0ool03ol300go0`0@o`030?oo01[o
00<0ool0>_l003Co00<0ool0<Ol00`3oo`0Vo`030?oo00co0`0Co`803ol00`3oo`0Ho`030?oo03_o
000eo`030?oo037o0P0Vo`030?oo00So100Ho`803?l201[o00<0ool0>ol003Go00<0ool0<ol202Co
00<0ool01Ol301ko0`09o`030?oo01Wo00<0ool0>ol003Ko00<0ool0=?l202;o00<0ool00_l302Co
0`06o`030?oo01Wo00<0ool0>ol003Ko00<0ool0=_l2023o00<0o`000P0Zo`L06_l00`3oo`0lo`00
=ol00`3oo`0go`d04ol204ko00<0ool0??l003So00<0ool0@ol600[o100[o`T06ol00`3oo`0lo`00
>Ol00`3oo`18o`X00ol00`3oo`0Xo`802?l301Oo00<0ool0?Ol003[o0P1Eo`030?oo02Oo00<0ool0
2ol401?o00<0ool0?Ol003co0`1Bo`030?oo02Ko00<0ool04?l400ko00<0ool0?_l003oo2P18o`03
0?oo02Go00<0ool05Ol600Go0`11o`00BOl204Ko00<0ool09Ol00`3oo`0Ko`H0@ol004_o0P0`o`80
1?l200?o1005o`030?oo02Co00<0ool0IOl004go0P0]o`040?oo00So00<0ool01_l00`3oo`0To`03
0?oo06Go001?o`<06_l900Oo00@0ool02Ol00`3oo`05o`809?l00`3oo`1Vo`00D_l201Co1009o`03
0?oo00Co00@0ool02_l00`3oo`04o`030?oo02?o00<0ool0I_l005Co0P0>o`@03_l00`3oo`03o`04
0?oo00So00@0ool01Ol00`3oo`0Ro`030?oo06Oo001Fo`D01ol201?o00<0ool00ol200[o0P06o`03
0?oo02;o00<0ool0Iol005_o1`0Fo`030?oo01Ko00<0ool08Ol00`3oo`1Xo`00NOl00`3oo`0Eo`03
0?oo023o00<0ool0JOl007[o00<0ool05?l00`3oo`0Po`030?oo06Wo001ko`030?oo01?o00<0ool0
2?l401?o00<0ool0J_l007co00<0ool04_l00`3oo`07o`030?oo00;o0P0@o`030?oo06_o001mo`03
0?oo017o00<0ool01Ol200Oo0P0=o`030?oo06co001no`030?oo013o00<0ool01?l00`3oo`09o`80
2_l00`3oo`1]o`00Ool2013o00<0ool00ol00`3oo`0<o`<01Ol2073o0021o`030?oo00go00@0oooo
0P0Bo`D0L_l008;o0P0=o`040?l008_o0024o`802ol208go001mo`801?l20003o`0000X0S_l007co
00@0ool02_l400?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go
00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co
001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o
003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88017 .16982 m
.85511 .15414 L
.82861 .13375 L
.81381 .12041 L
.79849 .10759 L
.79162 .10292 L
.78845 .10115 L
.78566 .09986 L
.78328 .09901 L
.78213 .0987 L
.78091 .09845 L
.77974 .09831 L
.77911 .09827 L
.77854 .09827 L
.77748 .09834 L
.77653 .0985 L
.77564 .09876 L
.77487 .09908 L
.77404 .09956 L
.77332 .1001 L
.77219 .10129 L
.77163 .1021 L
.77118 .10292 L
.77077 .10384 L
.7704 .10491 L
.77006 .10619 L
.76982 .10748 L
.76972 .10821 L
.76964 .10903 L
.76955 .1106 L
.76953 .11222 L
.76959 .11379 L
.7697 .11544 L
.7699 .11725 L
.77012 .11885 L
.77041 .12065 L
.77234 .129 L
.77497 .13745 L
.77799 .1456 L
.78636 .16512 L
.79023 .17382 L
.79316 .18095 L
.79433 .18421 L
.79517 .187 L
.79551 .18841 L
.79574 .18963 L
.79592 .1909 L
.79597 .19157 L
.79599 .19217 L
.79596 .19328 L
Mistroke
.7959 .19385 L
.79581 .19436 L
.79557 .19527 L
.79524 .19606 L
.795 .19647 L
.79473 .19689 L
.79414 .19757 L
.79339 .19821 L
.79241 .19881 L
.79147 .19923 L
.79049 .19957 L
.78953 .19982 L
.78839 .20005 L
.78706 .20023 L
.78634 .20031 L
.78553 .20037 L
.78401 .20045 L
.78251 .20048 L
.781 .20047 L
.77934 .20043 L
.77787 .20038 L
.7762 .20029 L
.76838 .19972 L
.75851 .19893 L
.75344 .19859 L
.75119 .19847 L
.74872 .19836 L
.74648 .19828 L
.7444 .19824 L
.74221 .19822 L
.73988 .19824 L
.73735 .1983 L
.73499 .19841 L
.7329 .19855 L
.73066 .19875 L
.72831 .19903 L
.72616 .19934 L
.72368 .1998 L
.72148 .20029 L
.71725 .20153 L
.71311 .20323 L
.71013 .20488 L
.70725 .20697 L
.70468 .20947 L
.7027 .21208 L
.70184 .21352 L
.70104 .2151 L
.69989 .21812 L
.6994 .21989 L
.69904 .22159 L
Mistroke
.69874 .22352 L
.69862 .22463 L
.69853 .22565 L
.69844 .22751 L
.69843 .22926 L
.69848 .23121 L
.69854 .23232 L
.69861 .23333 L
.69878 .23519 L
.69902 .23723 L
.69959 .24095 L
.70278 .25598 L
.70426 .26276 L
.70477 .26552 L
.7052 .26836 L
.70537 .26985 L
.70547 .27117 L
.70553 .27238 L
.70554 .27361 L
.70549 .27484 L
.70538 .27593 L
.7052 .27705 L
.70506 .27766 L
.70492 .27819 L
.70458 .27917 L
.70418 .28001 L
.70368 .28086 L
.70304 .28167 L
.70226 .28242 L
.70143 .28302 L
.70061 .28349 L
.69962 .28391 L
.69857 .28423 L
.6975 .28446 L
.69685 .28455 L
.69625 .28462 L
.69552 .28467 L
.69481 .2847 L
.6941 .2847 L
.69332 .28468 L
.69187 .28458 L
.69095 .28448 L
.69006 .28437 L
.68834 .28409 L
.68506 .28337 L
.68188 .28251 L
.67425 .27991 L
.64614 .26659 L
.63266 .25917 L
.62278 .25413 L
Mistroke
.61857 .25243 L
.61645 .25174 L
.61424 .25117 L
.61232 .25081 L
.61122 .25065 L
.61018 .25054 L
.60923 .25047 L
.60834 .25042 L
.6073 .2504 L
.6063 .2504 L
.60514 .25044 L
.60407 .25049 L
.60346 .25054 L
.6028 .25059 L
.60157 .25072 L
.59912 .25103 L
.59675 .25141 L
.58391 .25419 L
.55654 .26194 L
.52545 .27343 L
.51111 .2799 L
.49883 .28548 L
.4883 .28944 L
.48541 .29028 L
.48231 .29105 L
.47938 .29164 L
.47666 .29208 L
.47399 .2924 L
.47264 .29252 L
.47115 .29263 L
.46963 .2927 L
.46878 .29273 L
.468 .29275 L
.46648 .29277 L
.46504 .29276 L
.46368 .29272 L
.46223 .29267 L
.46099 .2926 L
.45963 .29251 L
.45656 .29224 L
.45369 .29191 L
.44714 .29094 L
.44128 .28985 L
.42904 .28724 L
.42009 .28544 L
.41787 .28509 L
.41592 .28483 L
.41429 .28468 L
.41342 .28461 L
.41265 .28458 L
Mistroke
.41182 .28455 L
.41107 .28455 L
.41028 .28457 L
.40985 .28458 L
.40947 .28461 L
.40871 .28466 L
.40792 .28475 L
.40654 .28494 L
.40399 .28545 L
.40165 .28595 L
.40049 .28617 L
.39921 .28636 L
.39852 .28644 L
.39814 .28647 L
.39773 .28651 L
.39698 .28655 L
.39626 .28657 L
.39551 .28657 L
.39466 .28654 L
.39382 .28649 L
.39302 .28642 L
.39227 .28634 L
.39143 .28623 L
.38962 .28591 L
.38727 .28538 L
.38496 .28474 L
.37922 .28273 L
.3673 .27717 L
.34325 .26213 L
.33301 .25463 L
.32486 .24866 L
.32224 .24702 L
.32108 .24643 L
.32049 .24618 L
.31997 .24599 L
.3195 .24585 L
.31909 .24576 L
.31864 .24568 L
.3184 .24566 L
.31817 .24564 L
.31778 .24563 L
.31743 .24564 L
.31704 .24567 L
.31663 .24572 L
.31618 .2458 L
.31576 .24588 L
.31478 .24609 L
.31378 .24631 L
.31276 .24648 L
.31214 .24657 L
Mistroke
.31181 .2466 L
.31144 .24664 L
.31102 .24666 L
.31061 .24668 L
.3102 .24669 L
.30982 .24669 L
.30907 .24666 L
.30862 .24663 L
.3082 .2466 L
.30731 .24649 L
.3063 .24632 L
.30434 .24586 L
.30208 .24514 L
.29755 .24316 L
.29216 .23994 L
.28584 .23506 L
.2804 .22981 L
.27491 .22326 L
.27042 .21646 L
.26716 .20997 L
.26584 .2065 L
.26492 .20339 L
.26453 .20169 L
.26436 .20081 L
.26421 .19988 L
.26401 .19821 L
.26391 .1967 L
.26389 .19535 L
.26395 .19394 L
.26408 .19265 L
.26427 .1915 L
.26457 .19021 L
.26499 .18886 L
.26548 .18766 L
.26601 .18659 L
.26672 .18543 L
.26761 .18426 L
.26848 .18331 L
.26949 .18239 L
.27049 .18164 L
.27166 .18091 L
.27392 .17984 L
.27542 .17933 L
.27685 .17895 L
.27767 .17878 L
.27858 .17863 L
.27946 .17852 L
.28029 .17843 L
.2818 .17835 L
.28351 .17834 L
Mistroke
.28439 .17837 L
.28534 .17843 L
.28709 .17859 L
.28803 .17871 L
.28893 .17885 L
.29062 .17915 L
.29444 .18005 L
.29774 .18106 L
.30129 .18236 L
.30845 .18562 L
.31514 .18945 L
.32383 .19577 L
.32596 .19774 L
.32647 .1983 L
.3266 .19847 L
.32666 .1986 L
.32666 .19865 L
.32657 .19865 L
.32639 .19855 L
.32614 .19838 L
.32453 .19704 L
.31772 .19026 L
.29848 .16887 L
.28682 .15649 L
.28041 .15051 L
.27442 .14572 L
.26885 .14205 L
.26374 .1394 L
.25799 .13719 L
.25478 .1363 L
.25171 .13564 L
.24865 .13514 L
.24575 .13481 L
.24407 .13467 L
.24253 .13457 L
.24168 .13452 L
.24074 .13448 L
.23903 .13442 L
.23809 .1344 L
.23721 .13438 L
.23555 .13437 L
.23376 .13437 L
.23274 .13437 L
.23181 .13438 L
.23023 .13439 L
.2285 .13442 L
.225 .13447 L
.22329 .13449 L
.22166 .13451 L
.22021 .13451 L
Mistroke
.21863 .13451 L
.21691 .13451 L
.21509 .13448 L
.21355 .13445 L
.21185 .1344 L
.21028 .13434 L
.20886 .13427 L
.20567 .13406 L
.20292 .13382 L
.20007 .13352 L
.18783 .13149 L
.1782 .12932 L
.17464 .12861 L
.17299 .12836 L
.1721 .12826 L
.17129 .12819 L
.17043 .12815 L
.16967 .12815 L
.16882 .12819 L
.16805 .12827 L
.16736 .12838 L
.16675 .12851 L
.16609 .1287 L
.16541 .12894 L
.16404 .12961 L
.16263 .13057 L
.16136 .13172 L
.16023 .13298 L
.15591 .13965 L
.1448 .16092 L
.13226 .17956 L
.12709 .18564 L
.12517 .18798 L
.12378 .18998 L
.12324 .19095 L
.12277 .19205 L
.12258 .19259 L
.12242 .19318 L
.12228 .19383 L
.12219 .19446 L
.12213 .19507 L
.1221 .19576 L
.12211 .19644 L
.12215 .19709 L
.12221 .19777 L
.12229 .19841 L
.12255 .19993 L
.12298 .20185 L
.1235 .20379 L
.1251 .20883 L
Mistroke
.12935 .22041 L
.14326 .25298 L
.15199 .26944 L
.16084 .28255 L
.16549 .288 L
.17071 .29301 L
.17574 .29682 L
.18045 .29955 L
.18544 .30166 L
.18827 .30253 L
.1909 .30315 L
.19224 .30339 L
.19364 .30361 L
.19484 .30376 L
.19611 .30389 L
.19748 .30399 L
.19873 .30406 L
.19949 .30408 L
.20021 .3041 L
.20154 .30411 L
.20269 .30409 L
.20392 .30406 L
.20507 .30401 L
.2061 .30396 L
.21083 .30363 L
.2151 .30329 L
.21621 .30322 L
.21737 .30315 L
.21793 .30313 L
.21844 .30311 L
.21939 .30308 L
.22026 .30307 L
.22116 .30307 L
.22191 .30308 L
.22272 .30311 L
.22358 .30317 L
.22448 .30324 L
.22531 .30334 L
.22604 .30344 L
.22745 .30369 L
.22893 .30403 L
.23148 .30485 L
.23416 .30601 L
.23658 .30729 L
.24195 .31061 L
.24741 .31388 L
.25356 .3169 L
.26635 .32156 L
.27956 .32614 L
.28733 .3286 L
Mistroke
.29211 .32979 L
.29449 .33026 L
.29677 .33063 L
.29899 .3309 L
.30014 .33101 L
.30142 .33111 L
.30273 .33118 L
.30415 .33123 L
.30553 .33125 L
.3068 .33123 L
.30809 .33119 L
.3093 .33114 L
.31065 .33104 L
.31211 .33091 L
.31459 .33062 L
.3174 .33019 L
.32365 .32888 L
.33047 .32696 L
.34298 .32248 L
.35648 .31711 L
.36361 .31453 L
.367 .3135 L
.37006 .31271 L
.37292 .31215 L
.37433 .31194 L
.37513 .31185 L
.37587 .31177 L
.37667 .31171 L
.3774 .31167 L
.37817 .31165 L
.37901 .31165 L
.37974 .31166 L
.38051 .3117 L
.38188 .31182 L
.3832 .31201 L
.38462 .31229 L
.38594 .31262 L
.38715 .31299 L
.38942 .31387 L
.3918 .31505 L
.39644 .31811 L
.4008 .3218 L
.41115 .33278 L
.42203 .3452 L
.43527 .35856 L
.4429 .36484 L
.45183 .37077 L
.45588 .37294 L
.46031 .37497 L
.46452 .37654 L
Mistroke
.46846 .37772 L
.47296 .3787 L
.47534 .37906 L
.47667 .37922 L
.47787 .37934 L
.47914 .37943 L
.48031 .37949 L
.48158 .37952 L
.48293 .37953 L
.48518 .37946 L
.48645 .37937 L
.48764 .37927 L
.48985 .37901 L
.49193 .37867 L
.49577 .37784 L
.4998 .37666 L
.50404 .37507 L
.51143 .37145 L
.5186 .36688 L
.52958 .35804 L
.53889 .34982 L
.54112 .34808 L
.54337 .34655 L
.54525 .3455 L
.54627 .34504 L
.54722 .34468 L
.54811 .34441 L
.54858 .3443 L
.54908 .34419 L
.54995 .34406 L
.55077 .344 L
.55158 .34398 L
.55235 .34402 L
.55321 .34411 L
.55413 .34427 L
.55514 .3445 L
.55625 .34484 L
.55834 .34564 L
.56249 .34774 L
.57272 .35402 L
.57833 .35736 L
.58376 .3602 L
.58875 .36236 L
.59143 .36331 L
.59431 .36415 L
.59582 .36451 L
.5972 .36478 L
.5986 .36501 L
.59989 .36517 L
.60132 .3653 L
Mistroke
.60207 .36534 L
.60287 .36537 L
.6042 .36536 L
.60565 .3653 L
.60694 .36518 L
.60833 .36498 L
.60977 .36471 L
.61109 .36439 L
.61341 .36365 L
.61581 .36264 L
.61854 .36117 L
.62101 .35951 L
.62534 .35573 L
.62909 .35145 L
.63226 .34699 L
.6385 .33574 L
.64889 .31186 L
.65364 .30193 L
.65823 .29459 L
.66067 .29165 L
.66351 .28897 L
.66488 .28793 L
.66641 .28695 L
.66932 .28552 L
.67118 .28488 L
.6722 .2846 L
.6733 .28435 L
.6743 .28416 L
.67542 .284 L
.67647 .28389 L
.67748 .28382 L
.67856 .28378 L
.67961 .28377 L
.68058 .28378 L
.68165 .28383 L
.68285 .2839 L
.68416 .28402 L
.6866 .28431 L
.68925 .28473 L
.69217 .28529 L
.69777 .28658 L
.71208 .29064 L
.74231 .3 L
.75852 .30424 L
.76299 .30518 L
.76708 .30591 L
.77063 .30641 L
.77255 .30663 L
.77431 .30679 L
.77582 .30689 L
Mistroke
.7774 .30696 L
.7791 .30699 L
.78061 .30697 L
.78214 .3069 L
.78375 .30676 L
.78521 .30658 L
.78652 .30636 L
.78871 .30585 L
.78989 .30549 L
.79096 .3051 L
.79297 .3042 L
.79494 .30301 L
.79674 .30159 L
.79844 .29984 L
.79984 .29797 L
.80095 .29609 L
.80207 .29368 L
.80307 .29083 L
.80385 .28794 L
.80444 .28512 L
.80493 .28209 L
.80531 .27915 L
.80598 .27222 L
.80651 .26541 L
.8068 .26187 L
.80716 .25818 L
.80756 .25474 L
.80798 .25177 L
.80902 .24646 L
.80962 .24414 L
.81035 .24186 L
.81104 .2401 L
.81185 .23839 L
.81263 .23706 L
.8135 .23587 L
.81446 .23488 L
.81489 .23452 L
.81536 .23418 L
.81583 .23391 L
.81627 .23372 L
.81668 .23358 L
.81711 .23348 L
.8175 .23343 L
.81786 .23342 L
.81824 .23345 L
.81865 .23352 L
.81908 .23365 L
.81947 .2338 L
.8199 .23403 L
.82028 .23429 L
Mistroke
.82093 .23484 L
.82157 .23555 L
.82219 .23646 L
.82275 .23751 L
.82314 .23845 L
.82349 .2395 L
.82376 .24057 L
.82393 .2415 L
.824 .24198 L
.82406 .24247 L
.82413 .24325 L
.82418 .24382 L
.82419 .24405 L
.8242 .24415 L
.82421 .24423 L
.82424 .24434 L
.82425 .24438 L
.82427 .2444 L
.8243 .24441 L
.82435 .24437 L
.82441 .24426 L
.82449 .24411 L
.82473 .24355 L
.82549 .24177 L
.82843 .23576 L
.84384 .21245 L
.86524 .18757 L
.87331 .179 L
.87668 .17523 L
.87885 .17242 L
.87966 .17107 L
.87995 .17045 L
.88017 .16982 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`15o`D0
@_l0097o00<0ool0A?l200Co0P10o`00TOl00`3oo`14o`030?oo00Go00<0ool0?Ol007go0P04o`80
1Ol300Co00<0ool0A?l00`3oo`06o`030?oo03co001lo`040?oo00_o00<0ool00ol00`3oo`14o`03
0?oo00Oo00<0ool0>ol007co00@0ool02?l500Co0P15o`030?oo00So00<0ool0>_l007Ko1002o`04
0?oo00So00@0ool01Ol00`3oo`14o`030?oo00Wo00<0ool0>Ol007co00@0ool02Ol00`3o0005o`03
0?oo04Go00<0ool02Ol203Wo000go`T0?Ol200_o0P05o`030?oo04Go00<0ool02ol00`3oo`0fo`00
=_l00`3oo`07o`l0@_l00`3oo`16o`030?oo00_o00<0ool0=Ol003Go00<0ool05_l403oo00<0ool0
A_l00`3oo`0<o`80=Ol003Go00<0ool06_l203go00<0ool0A_l00`3oo`0>o`030?oo03;o000do`03
0?oo01go00<0ool0>_l00`3oo`16o`030?oo00oo00<0ool0<Ol003Co00<0ool07_l00`3oo`0io`03
0?oo04Oo00<0ool03ol2037o000co`030?oo023o0P0io`030?oo04Oo00<0ool04Ol00`3oo`0^o`00
<ol00`3oo`0Ro`030?oo03Ko00<0ool0B?l00`3oo`0Ao`80;_l003;o00<0ool09?l00`3oo`0eo`03
0?oo04So00<0ool04ol00`3oo`0[o`00<Ol00`3oo`0Vo`030?oo03Co00<0ool0BOl00`3oo`0Co`03
0?oo02[o000ao`030?oo02Oo00<0ool0<ol00`3oo`19o`030?oo01Co0P0Zo`00<?l00`3oo`0Yo`03
0?oo03;o00<0ool0B_l00`3oo`0Do`80:Ol0033o00<0ool0:_l00`3oo`0Mo`801?l200?o1005o`03
0?oo04[o00<0ool05?l00`3oo`0Xo`00;ol00`3oo`0Ro`L00_l00`3oo`0Lo`040?oo00So00<0ool0
1_l00`3oo`1:o`030?oo01?o00<0ool0:Ol002ko00<0ool08_l200Ko0P000ol0o`0Lo`040?oo00Wo
00<0ool01Ol204_o0P0Co`030?oo02[o000]o`030?oo02;o0P09o`<05_l400;o00@0ool02_l00`3o
o`04o`030?oo04_o00<0ool04?l00`3oo`0[o`00;?l202Co00<0ool02_l201_o00@0ool02?l0103o
o`05o`030?oo04_o00<0ool03ol00`3oo`0/o`00;?l00`3oo`0So`030?oo00_o0P0Ko`802_l200Ko
00<0ool0Bol00`3oo`0>o`030?oo02go000/o`030?oo02?o00<0ool03Ol202go00<0ool0>Ol:00Oo
0P0?o`030?oo02ko000/o`030?oo02?o00<0ool0??l00`3oo`0fo`@02Ol8013o00<0ool0;_l002co
00<0ool08ol00`3oo`0lo`030?oo03Co0P0To`030?oo02oo0007o`D00_l200Co0P0Go`030?oo00Wo
1@0Eo`030?oo00[o0P04o`801?l202?o00<0ool08_l200Co0P04o`800ol201ko1@000ol0o`0Oo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Eo`030?oo00Wo00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l01P3oo`3o02;o00@0ool08ol00`3oo`07o`040?oo00;o
0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Go00<0ool02Ol0103oo`0;o`030?oo023o
00<0ool08Ol0103oo`0;o`030?l002;o00<0o`009?l00`3oo`0:o`400Ol1o`0000?o00000P04o`03
0?oo00So0P0Ho`030?oo00;o1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool0
8Ol0103oo`09o`8000?o00008_l00`3o000To`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo
00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol0103o
o`0So`809Ol00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201Wo00<0ool02?l200[o
0`0Ro`030?oo02;o0P0:o`@08_l202Go0P0:o`<00_l002oo00<0ool09?l203Wo00<0ool0<Ol00`3o
o`0Qo`030?oo03Co000`o`030?oo02Go00<0ool0=_l00`3oo`0ao`030?oo01co0`000ol0o`0fo`00
<?l00`3oo`0Vo`80=_l00`3oo`0ao`030?oo01_o0P001?l0000go`00<Ol00`3oo`0Wo`030?oo03?o
00<0ool0<_l00`3oo`0Jo`050?ooo`00>?l0037o00<0ool0:?l602oo00<0ool0<_l00`3oo`0Jo`05
0?ooo`00>?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l0103oool300;o00<0ool00_l30003
o`3o00Oo00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o0006o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0
:Ol302[o00<0ool03_l400go0P08o`030?oo00Ko00<0ool06Ol00`3oo`04o`030?oo02[o00<0ool0
1ol003?o00<0ool0<Ol00`3oo`0Wo`030?oo00_o0`0Co`804?l00`3oo`0Ho`030?oo03_o000do`03
0?oo037o0P0Wo`030?oo00So0`0Ho`803_l00`3oo`0Ho`030?oo03_o000do`030?oo03?o00<0ool0
9?l00`3oo`05o`<07Ol200co00<0ool06?l00`3oo`0ko`00=Ol00`3oo`0co`809?l00`3oo`03o`80
8_l200[o00<0ool06?l00`3oo`0ko`00=Ol00`3oo`0eo`030?oo027o00@0oooo0P0Vo`802?l00`3o
o`0Ho`030?oo03_o000fo`030?oo03Go0P0Qo`@0:_l200Ko00<0ool06?l00`3oo`0ko`00=_l00`3o
o`0go`@00ol601;o0`0/o`T06_l203go000go`80>_l400Ko0`0=o`800_l00`3oo`0Xo`801ol201[o
00<0ool0??l003Wo00<0ool0A?l>00?o00<0ool09ol00`3oo`09o`@05_l00`3oo`0lo`00>_l00`3o
o`1Do`030?oo02Oo00<0ool03Ol401;o00<0ool0??l003_o0P1Do`030?oo02Ko00<0ool04_l400go
00<0ool0?Ol003go3018o`030?oo02Go00<0ool05ol300So0P10o`00BOl204Ko00<0ool09Ol00`3o
o`0Jo`T0@Ol004_o0P0Ro`<02ol200Co0P03o`@01Ol00`3oo`0To`030?oo06Go001=o`030?oo01_o
1@02o`<01ol0103oo`08o`030?oo00Ko00<0ool09?l00`3oo`1Uo`00C_l301So0P0:o`030?oo00Co
00@0ool02Ol00`3oo`05o`809?l00`3oo`1Vo`00DOl3017o100=o`030?oo00?o00@0ool02_l00`3o
o`04o`030?oo02?o00<0ool0I_l005Co0`0<o`804_l00`3oo`02o`040?oo00So00@0ool01Ol00`3o
o`0Ro`030?oo06Oo001Go`@01?l401Go00<0ool00_l200[o0P06o`030?oo02;o00<0ool0Iol005_o
1@0Io`030?oo01Go00<0ool08Ol00`3oo`1Xo`00N_l00`3oo`0Do`030?oo027o00<0ool0J?l007_o
00<0ool04ol00`3oo`0Qo`030?oo06So001lo`030?oo01;o00<0ool02Ol401?o00<0ool0JOl007go
00<0ool04Ol00`3oo`07o`801?l00`3oo`0@o`030?oo06Wo001no`030?oo013o00<0ool01ol00`3o
o`04o`803ol00`3oo`1Zo`00O_l00`3oo`0@o`030?oo00Go0P09o`030?oo00_o00<0ool0Jol007oo
00<0ool03ol00`3oo`04o`030?oo00[o0`09o`030?oo06co0020o`030?oo00ko00<0ool00_l2013o
0`03o`<0Kol0087o0P0>o`050?ooo`005?l5077o0023o`803?l0103o002;o`00OOl200Co1003o`<0
1?l208go001lo`040?oo00Oo2`2>o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 485},
WindowMargins->{{Automatic, 78}, {91, Automatic}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 190, 5, 465, "Title"],

Cell[CellGroupData[{
Cell[1954, 60, 32, 0, 53, "Section"],
Cell[1989, 62, 345, 9, 52, "Text"],
Cell[2337, 73, 320, 6, 52, "Text"],
Cell[2660, 81, 304, 6, 52, "Text"],
Cell[2967, 89, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3200, 98, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3382, 106, 48, 0, 45, "Subsection"],
Cell[3433, 108, 67, 1, 27, "Input"],
Cell[3503, 111, 103, 2, 27, "Input"],
Cell[3609, 115, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3774, 122, 46, 0, 29, "Subsection"],
Cell[3823, 124, 326, 7, 50, "Text"],
Cell[4152, 133, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4599, 145, 103, 2, 29, "Subsection"],
Cell[4705, 149, 105, 3, 32, "Text"],
Cell[4813, 154, 244, 6, 42, "Input"],
Cell[5060, 162, 123, 2, 27, "Input"],
Cell[5186, 166, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5469, 176, 78, 0, 42, "Subsubsection"],
Cell[5550, 178, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5720, 186, 60, 0, 29, "Subsection"],
Cell[5783, 188, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6040, 196, 130, 3, 46, "Subsection"],
Cell[6173, 201, 121, 2, 27, "Input"],
Cell[6297, 205, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6478, 213, 31, 0, 33, "Section"],
Cell[6512, 215, 101, 3, 33, "Text"],
Cell[6616, 220, 202, 3, 30, "Input"],
Cell[6821, 225, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[7012, 232, 44, 0, 33, "Section"],
Cell[7059, 234, 66, 0, 33, "Text"],
Cell[7128, 236, 1965, 30, 290, "Input"],
Cell[9096, 268, 1976, 32, 290, "Input"],
Cell[11075, 302, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11176, 308, 32, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11233, 312, 80, 1, 30, "Input"],
Cell[11316, 315, 14657, 647, 152, 7884, 559, "GraphicsData", "PostScript", \
"Graphics"],
Cell[25976, 964, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26155, 973, 35, 0, 53, "Section"],
Cell[26193, 975, 176, 4, 52, "Text"],
Cell[26372, 981, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[26858, 992, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[27025, 999, 21282, 1016, 152, 13422, 913, "GraphicsData", "PostScript", \
"Graphics"],
Cell[48310, 2017, 20715, 988, 152, 13004, 887, "GraphicsData", "PostScript", \
"Graphics"],
Cell[69028, 3007, 20265, 968, 152, 12716, 869, "GraphicsData", "PostScript", \
"Graphics"],
Cell[89296, 3977, 19427, 923, 152, 12048, 826, "GraphicsData", "PostScript", \
"Graphics"],
Cell[108726, 4902, 18579, 879, 152, 11415, 785, "GraphicsData", "PostScript", \
"Graphics"],
Cell[127308, 5783, 17560, 822, 152, 10541, 729, "GraphicsData", "PostScript", \
"Graphics"],
Cell[144871, 6607, 16015, 729, 152, 9114, 638, "GraphicsData", "PostScript", \
"Graphics"],
Cell[160889, 7338, 15358, 688, 152, 8489, 597, "GraphicsData", "PostScript", \
"Graphics"],
Cell[176250, 8028, 14716, 649, 152, 7884, 559, "GraphicsData", "PostScript", \
"Graphics"],
Cell[190969, 8679, 15050, 670, 152, 8181, 579, "GraphicsData", "PostScript", \
"Graphics"],
Cell[206022, 9351, 17450, 822, 152, 10565, 731, "GraphicsData", "PostScript", \
"Graphics"],
Cell[223475, 10175, 18034, 855, 152, 11084, 763, "GraphicsData", \
"PostScript", "Graphics"],
Cell[241512, 11032, 19197, 923, 152, 12093, 829, "GraphicsData", \
"PostScript", "Graphics"],
Cell[260712, 11957, 19854, 957, 152, 12629, 862, "GraphicsData", \
"PostScript", "Graphics"],
Cell[280569, 12916, 20632, 997, 152, 13221, 900, "GraphicsData", \
"PostScript", "Graphics"],
Cell[301204, 13915, 21361, 1040, 152, 13832, 941, "GraphicsData", \
"PostScript", "Graphics"],
Cell[322568, 14957, 21733, 1057, 152, 14111, 957, "GraphicsData", \
"PostScript", "Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

