(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    278205,      12103]*)
(*NotebookOutlinePosition[    278859,      12126]*)
(*  CellTagsIndexPosition[    278815,      12122]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of PC1, for Major Axis Rescaled, Full Population, on Raw Data \
scale
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {0.99999999577465`, 
          0.00000000042254`, \(-0.00000000140845`\), \(-0.15265874436620`\), 
          0.05051707605634`, \(-0.06959121563380`\), \(-0.23566859253521`\), 
          0.01841770549296`, 0.02504751112676`, 
          0.03545001084507`, \(-0.00452289971831`\), \(-0.08314208704225`\), 
          0.00545232028169`, \(-0.07516635943662`\), \(-0.03967742253521`\), \
\(-0.01263550084507`\), 0.02432813154930`, 
          0.01841314887324`, \(-0.00793547971831`\), \(-0.04315542014085`\), \
\(-0.01110062661972`\), \(-0.03424806577465`\), \(-0.00997004577465`\), \
\(-0.01400951478873`\), 0.02235066478873`, \(-0.00055634633803`\), 
          0.00031299535211`, \(-0.00816849971831`\), \(-0.00305151535211`\), 
          0.00108292000000`, \(-0.00114177281690`\), \(-0.00645633408451`\), 
          0.00787526211268`, \(-0.00631870070423`\), 0.00468007873239`, 
          0.00181971619718`, 0.00642344676056`, 0.00725774338028`, 
          0.00033030056338`, \(-0.00271897239437`\), \(-0.00346696915493`\), 
          0.00029988901408`, 0.00166032788732`, 0.00120431774648`, 
          0.00509558619718`, 
          0.00156389971831`, \(-0.00014830422535`\), \(-0.00034171577465`\), \
\(-0.00357382661972`\), 0.00491082211268`, 0.00050473633803`, 
          0.00110795901408`, 
          0.00000952042254`, \(-0.00171832591549`\), \(-0.00025740760563`\), \
\(-0.00044140380282`\), \(-0.00035070605634`\), 0.00281390267606`, 
          0.00016366535211`, 
          0.00014455901408`, \(-0.00195638549296`\), \(-0.00106211915493`\), \
\(-0.00056562169014`\), \(-0.00060771816901`\), 
          0.00051153267606`, \(-0.00022239112676`\), 
          0.00035875732394`, \(-0.00069439746479`\), \(-0.00055557535211`\), 
          0.00008571070423`, \(-0.00018296197183`\), \(-0.00020264000000`\), 
          0.00002939464789`, \(-0.00077437957746`\), \(-0.00005848859155`\), \
\(-0.00063389633803`\), 0.00049846464789`, 0.00028082816901`, 
          0.00040762183099`, \(-0.00026705014085`\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ {0.00000008400919, 0.00000000266929, 0.00000007471587, 
          0.02366036580526, \(-0.33686300611536\), 0.26552985499673, 
          0.71159706670184, \(-0.09212998990006\), 
          0.36788627938139, \(-0.17975515034398\), 
          0.11533949184441, \(-0.08012944589057\), \(-0.08400704435231\), 
          0.21639318461344, 0.11019160921661, 0.01862571406697, 
          0.06522580640565, \(-0.06849828173244\), 
          0.06113098571537, \(-0.11258963348680\), 0.00718397435478, 
          0.07333370285521, 0.04936425010908, 
          0.05899709660051, \(-0.01263175581008\), 0.02704439991530, 
          0.00412472714858, \(-0.05017168649691\), \(-0.00515722617763\), 
          0.00664434011750, 0.01855680523859, 
          0.01712625628000, \(-0.01929957382938\), 0.03470442744668, 
          0.01499603027405, 0.01274059925867, \(-0.01784702755469\), 
          0.00634163489830, \(-0.01044793067820\), 
          0.00564762976611, \(-0.01762031222549\), 0.01825398416081, 
          0.01913996523226, 0.02005075305983, \(-0.01588916459942\), 
          0.00022600366413, \(-0.01530686550132\), 
          0.01193562023254, \(-0.01420971327829\), 0.00821340284740, 
          0.01039235849096, \(-0.00194398872269\), \(-0.01269901051971\), \
\(-0.01245671641619\), \(-0.01250278819588\), 
          0.00762433198554, \(-0.00018530608816\), 0.00459059247618, 
          0.00480759992932, \(-0.00790349503492\), \(-0.01119190649501\), \
\(-0.00904925089367\), 0.00343415508336, 0.00383113153417, 
          0.00898523882170, \(-0.00169608491171\), \(-0.00020245363363\), \
\(-0.00378829707628\), \(-0.00536364801414\), 0.00044508277057, 
          0.00299730290255, \(-0.00295083322657\), 
          0.00309142718734, \(-0.00595915038040\), 
          0.00191863026551, \(-0.00131476009871\), 0.00181570271675, 
          0.00430844017980, 
          0.00275743549949, \(-0.00530364216871\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the mean", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87469 .15276 m
.87085 .14472 L
.86557 .13669 L
.86248 .13304 L
.85886 .12948 L
.85155 .12405 L
.84696 .12157 L
.84166 .11939 L
.83914 .11857 L
.8364 .11782 L
.83398 .11728 L
.83135 .11679 L
.82876 .11642 L
.82637 .11617 L
.82522 .11608 L
.82395 .116 L
.82275 .11594 L
.82164 .11591 L
.82096 .11589 L
.82022 .11589 L
.81888 .11589 L
.81758 .11592 L
.81638 .11596 L
.81507 .11603 L
.81365 .11614 L
.81078 .11642 L
.80821 .11677 L
.80545 .11724 L
.80053 .11836 L
.79591 .11977 L
.79183 .12136 L
.78344 .12596 L
.77973 .12882 L
.77608 .13239 L
.7733 .13582 L
.77075 .13977 L
.76712 .14756 L
.76553 .15225 L
.76427 .1568 L
.76187 .16752 L
.75963 .17802 L
.75703 .18767 L
.75395 .19588 L
.74961 .20414 L
.74429 .21153 L
.73735 .21892 L
.72085 .23195 L
.7029 .24315 L
.68604 .25187 L
.66784 .25917 L
Mistroke
.64959 .26466 L
.6113 .27329 L
.59148 .27675 L
.57003 .27957 L
.54761 .2817 L
.5373 .2825 L
.52647 .28324 L
.51651 .28379 L
.51089 .28402 L
.50821 .2841 L
.50567 .28417 L
.50337 .28422 L
.50207 .28424 L
.50087 .28425 L
.49986 .28426 L
.49875 .28427 L
.49759 .28427 L
.4965 .28427 L
.4953 .28427 L
.49404 .28426 L
.49296 .28425 L
.49178 .28424 L
.48914 .28419 L
.48669 .28413 L
.48148 .28394 L
.47583 .28366 L
.46556 .28297 L
.44644 .28127 L
.42676 .27917 L
.41653 .2779 L
.40567 .27629 L
.38576 .27236 L
.36712 .26747 L
.34812 .26156 L
.32838 .25451 L
.31172 .24732 L
.30247 .24245 L
.29407 .23723 L
.28614 .23127 L
.27985 .22531 L
.27716 .22212 L
.2747 .21855 L
.27284 .21491 L
.2721 .21291 L
.27183 .212 L
.27159 .21104 L
.27142 .21017 L
.27135 .20972 L
.27128 .20922 L
.27119 .20836 L
Mistroke
.27115 .20755 L
.27114 .20706 L
.27115 .20654 L
.27116 .20604 L
.27119 .20558 L
.27128 .2047 L
.27135 .20422 L
.27142 .20378 L
.2718 .20209 L
.27208 .20115 L
.27238 .20028 L
.27388 .19702 L
.27567 .19413 L
.27807 .191 L
.28396 .18477 L
.29146 .17782 L
.29813 .17136 L
.30282 .16569 L
.30379 .16417 L
.30464 .16258 L
.30528 .16111 L
.3055 .16047 L
.30571 .15978 L
.30586 .15916 L
.30598 .15851 L
.30607 .15782 L
.30612 .15719 L
.30613 .15684 L
.30612 .15646 L
.30608 .15578 L
.30599 .15512 L
.30588 .15453 L
.30573 .15397 L
.30556 .15345 L
.30507 .1523 L
.30446 .15121 L
.30364 .15008 L
.30256 .14889 L
.30118 .14765 L
.29815 .14557 L
.29463 .14378 L
.29093 .14233 L
.28651 .14099 L
.28136 .13981 L
.27624 .13895 L
.27374 .13862 L
.27138 .13836 L
.2669 .13798 L
.2645 .13783 L
.26193 .13771 L
Mistroke
.26064 .13767 L
.25923 .13763 L
.2579 .13761 L
.25667 .13759 L
.25557 .13758 L
.25435 .13758 L
.25308 .13759 L
.2519 .1376 L
.25059 .13762 L
.24921 .13765 L
.24672 .13772 L
.24378 .13784 L
.24112 .13797 L
.23104 .13868 L
.22143 .13962 L
.21244 .14073 L
.20271 .14222 L
.19245 .14422 L
.18377 .14638 L
.1744 .14934 L
.16488 .15325 L
.15671 .15762 L
.14813 .16366 L
.14112 .1704 L
.13568 .17764 L
.13344 .18158 L
.13139 .18614 L
.13001 .19016 L
.12942 .19232 L
.1289 .19462 L
.12853 .19667 L
.12837 .19776 L
.12822 .19893 L
.1281 .20002 L
.12801 .20101 L
.12795 .20201 L
.1279 .20295 L
.12786 .20405 L
.12785 .20523 L
.12785 .20634 L
.12788 .20736 L
.12793 .20856 L
.12797 .20924 L
.12802 .20987 L
.12824 .21225 L
.12858 .21476 L
.12897 .21703 L
.13014 .22211 L
.13143 .22643 L
.13312 .23103 L
Mistroke
.13686 .23913 L
.14167 .24731 L
.14776 .25566 L
.16213 .27039 L
.17754 .28157 L
.19582 .2913 L
.23439 .30614 L
.27183 .31857 L
.30862 .32969 L
.34781 .34004 L
.36761 .34435 L
.38663 .34755 L
.40511 .34975 L
.41555 .35066 L
.42514 .35131 L
.43453 .35182 L
.44459 .35225 L
.45039 .35244 L
.45568 .35259 L
.45823 .35264 L
.46101 .3527 L
.46364 .35274 L
.46605 .35277 L
.46852 .3528 L
.46988 .35281 L
.47117 .35281 L
.47236 .35282 L
.47346 .35282 L
.47467 .35282 L
.47596 .35282 L
.47725 .35282 L
.47798 .35281 L
.47864 .35281 L
.47987 .3528 L
.48116 .35279 L
.4839 .35276 L
.48691 .35271 L
.48981 .35264 L
.49244 .35257 L
.49836 .35237 L
.50384 .35213 L
.50901 .35184 L
.51914 .35113 L
.52981 .35012 L
.54876 .34764 L
.55834 .346 L
.56857 .34393 L
.58681 .33943 L
.62658 .32693 L
.66472 .31512 L
Mistroke
.70257 .30611 L
.74317 .29823 L
.78177 .28951 L
.80221 .28349 L
.81131 .28023 L
.82068 .27629 L
.82907 .27208 L
.83789 .26671 L
.85206 .25526 L
.85895 .2479 L
.86452 .24067 L
.87008 .23177 L
.87425 .22322 L
.87726 .21514 L
.87848 .21095 L
.87956 .20638 L
.8804 .20176 L
.8807 .19968 L
.88096 .19738 L
.88119 .19482 L
.88128 .19342 L
.88135 .1921 L
.8814 .19087 L
.88143 .18974 L
.88144 .18911 L
.88144 .18843 L
.88144 .1872 L
.88143 .18617 L
.8814 .18523 L
.88136 .18415 L
.88132 .18311 L
.8812 .18127 L
.88104 .17926 L
.88081 .17704 L
.88051 .17468 L
.87982 .17041 L
.87879 .16557 L
.87758 .16104 L
.87469 .15276 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Col603Oo001lo`040?oo00So1@04o`80C?l400Ko100co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03?l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4Ol00`3oo`0^o`00OOl200_o0P05o`030?oo04Oo00<0ool04_l00`3oo`0]o`00TOl00`3oo`16o`03
0?oo01Co00<0ool0;?l004So3P0ko`030?oo04Go00<0ool05_l00`3oo`0[o`00@Ol700ko100go`03
0?oo04Go00<0ool05_l00`3oo`0[o`00??l501Wo0P0eo`030?oo04Co00<0ool06?l00`3oo`0Zo`00
>Ol3023o00<0ool0<_l00`3oo`14o`030?oo01So00<0ool0:_l003Oo0P0So`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Go0P0Vo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`03
0?oo02Ko00<0ool0<Ol00`3oo`13o`030?oo01_o00<0ool0:?l003?o00<0ool09_l203?o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`030?oo01co00<0ool0
:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07Ol00`3oo`0Wo`00<?l00`3oo`0Wo`03
0?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`030?oo02Ko00<0ool0
8?l0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Ko00<0ool0
8Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool09ol002ko00<0ool09Ol201ko1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0To`04
0?oo00So00@0ool01Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`80
2_l200Ko00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00
;_l00`3oo`0So`030?oo03[o00<0ool0?ol00`3oo`0Oo`030?oo02So0007o`D00_l200Co0P0Ho`03
0?oo00So1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01Oo0P08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol201Wo00<0ool0
2?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806ol00`3oo`06o`80
2_l302;o00<0ool08_l200[o0`07o`806_l200[o00<0ool06?l200[o0`02o`00;ol00`3oo`0Vo`03
0?oo03Ko00<0ool0=ol00`3oo`0Uo`030?oo02[o000`o`030?oo02Ko00<0ool0=Ol00`3oo`0eo`80
9ol00`3oo`0[o`00<?l00`3oo`0Wo`80=Ol00`3oo`0co`80:Ol00`3oo`0[o`00<Ol00`3oo`0Xo`80
<ol00`3oo`0bo`030?oo02So00<0ool0;?l0037o00<0ool0:_l2037o00<0ool0<?l202[o00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol302[o00<0ool0
:?l302[o00<0ool00P0Xo`030?oo00Oo000do`030?oo033o0`0Wo`030?oo02Ko0P0_o`030?oo03;o
000eo`80<ol202Go00<0ool08_l4033o00<0ool0<ol003Oo00<0ool0<_l4027o00<0ool07Ol503?o
00<0ool0=?l003So00<0ool0=Ol501co00<0ool06?l503Ko0P0go`00>Ol203[o1`0Eo`030?oo01?o
1@0io`80>Ol003_o00<0ool0?_l900co00<0ool02?l;03_o0`0ko`00??l204Oo5`14o`80?_l003ko
0P1Ao`030?oo04Wo0`10o`00@?l204oo00<0ool0AOl404?o0012o`80COl00`3oo`10o`D0Aol004Co
0`1:o`030?oo03_o1@1<o`00Aol204So00<0ool0=Ol6057o0019o`<0AOl00`3oo`0`o`D0Eol004co
0`0^o`801?l200?o1005o`030?oo02_o1@1Lo`00Col402Wo00@0ool02?l00`3oo`06o`030?oo02Oo
101Qo`00Dol302Ko00@0ool02Ol00`3oo`05o`809?l406Go001Fo`<08ol0103oo`0:o`030?oo00Co
00<0ool08?l306Wo001Io`@07ol0103oo`08o`040?oo00Go00<0ool07Ol306co001Mo`<07Ol200[o
0P06o`030?oo01_o0P1_o`00H?l302ko00<0ool06?l3077o001So`@0:_l00`3oo`0Fo`80M?l006Oo
100Vo`030?oo01;o101fo`00Jol5027o00<0ool03_l407[o001`o`D07?l00`3oo`07o`L0O_l007Go
9P25o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87799 .11635 m
.87527 .11239 L
.87246 .10893 L
.86874 .10514 L
.86503 .10201 L
.85653 .09667 L
.85139 .09438 L
.84628 .09266 L
.84082 .09136 L
.83801 .09088 L
.83637 .09066 L
.83486 .09049 L
.83357 .09038 L
.83216 .09028 L
.83139 .09023 L
.83067 .0902 L
.82924 .09016 L
.82789 .09015 L
.82667 .09016 L
.82526 .0902 L
.82393 .09026 L
.82249 .09035 L
.82092 .09049 L
.81775 .09087 L
.81484 .09135 L
.81215 .0919 L
.80624 .09352 L
.80019 .09581 L
.79511 .09835 L
.79074 .10108 L
.78645 .10441 L
.78224 .10853 L
.77647 .11644 L
.77401 .12124 L
.77212 .12601 L
.77069 .13067 L
.76944 .13596 L
.76845 .14166 L
.76773 .14708 L
.76673 .15691 L
.76564 .16763 L
.76485 .17365 L
.76392 .17913 L
.76281 .18426 L
.76136 .18957 L
.75812 .19841 L
.75344 .20757 L
.7474 .21648 L
.74101 .22409 L
.72461 .23962 L
Mistroke
.70699 .25315 L
.6898 .26404 L
.67216 .2728 L
.65219 .28013 L
.62949 .28618 L
.60774 .29077 L
.56563 .29799 L
.54352 .30091 L
.53234 .30208 L
.52603 .30264 L
.52006 .30309 L
.51459 .30343 L
.50865 .30372 L
.50603 .30382 L
.50325 .30391 L
.50087 .30396 L
.49826 .30401 L
.49678 .30403 L
.49598 .30404 L
.49523 .30405 L
.4939 .30406 L
.49245 .30406 L
.49164 .30406 L
.49087 .30406 L
.48943 .30405 L
.48863 .30404 L
.48789 .30403 L
.48621 .30401 L
.48359 .30396 L
.48072 .30389 L
.47493 .30368 L
.46911 .3034 L
.46361 .30306 L
.45333 .30228 L
.43047 .3 L
.40911 .29717 L
.3896 .29362 L
.37951 .29121 L
.37013 .28852 L
.34924 .2809 L
.31269 .26275 L
.29511 .25172 L
.28673 .24534 L
.27859 .23777 L
.27507 .23379 L
.27211 .22985 L
.26962 .22588 L
.26771 .22211 L
.2663 .21847 L
.26574 .21661 L
Mistroke
.26524 .21455 L
.26504 .2135 L
.26489 .21249 L
.26482 .21194 L
.26475 .21133 L
.26466 .21027 L
.26462 .20927 L
.2646 .2082 L
.26463 .20719 L
.26468 .20628 L
.26478 .20524 L
.2649 .20428 L
.26508 .20323 L
.26531 .20211 L
.26594 .19985 L
.26667 .19781 L
.26853 .19398 L
.27082 .19049 L
.27405 .18664 L
.27755 .18327 L
.28689 .17631 L
.30664 .1653 L
.31559 .16042 L
.32243 .156 L
.32522 .15375 L
.32736 .15164 L
.32909 .14943 L
.32979 .14826 L
.33037 .14701 L
.3306 .14636 L
.33077 .14575 L
.33091 .14509 L
.33096 .14476 L
.331 .14439 L
.33103 .14378 L
.33103 .14344 L
.33102 .14313 L
.33096 .14253 L
.33087 .14196 L
.33074 .14142 L
.33057 .14085 L
.33015 .13983 L
.3295 .13865 L
.32876 .1376 L
.32672 .13547 L
.32393 .13338 L
.32107 .13174 L
.31747 .13012 L
.3131 .1286 L
.30795 .12724 L
Mistroke
.30272 .12621 L
.2977 .12548 L
.29246 .12494 L
.28976 .12474 L
.28676 .12457 L
.28417 .12446 L
.28272 .12441 L
.28138 .12438 L
.28014 .12436 L
.27878 .12434 L
.27749 .12433 L
.2763 .12433 L
.2748 .12434 L
.27396 .12435 L
.27319 .12436 L
.27176 .12438 L
.27025 .12441 L
.26764 .12448 L
.26478 .12458 L
.25848 .12489 L
.25265 .12526 L
.24107 .1262 L
.23028 .12731 L
.21024 .13011 L
.19981 .13212 L
.18876 .13487 L
.17946 .13783 L
.16955 .14187 L
.15991 .14706 L
.15198 .15277 L
.14457 .16012 L
.14143 .16424 L
.13848 .16906 L
.13616 .17389 L
.13439 .17867 L
.13314 .18314 L
.13215 .18814 L
.13178 .19078 L
.13148 .19373 L
.13137 .19523 L
.13132 .19606 L
.13129 .19684 L
.13126 .19758 L
.13123 .19838 L
.13121 .19978 L
.13121 .20053 L
.13121 .20123 L
.13123 .20254 L
.13127 .20395 L
.13131 .20475 L
Mistroke
.13134 .20548 L
.13152 .20807 L
.13178 .21079 L
.13212 .2136 L
.13251 .21624 L
.13342 .22117 L
.13624 .23199 L
.13947 .24094 L
.14375 .25013 L
.15504 .26729 L
.16189 .27502 L
.17021 .2829 L
.1879 .29629 L
.20554 .30693 L
.24335 .3247 L
.28361 .34021 L
.32273 .35313 L
.34489 .35944 L
.36535 .3646 L
.3757 .3669 L
.387 .36909 L
.39769 .37081 L
.40775 .3721 L
.41825 .3731 L
.42385 .37351 L
.42988 .37386 L
.43622 .37415 L
.44226 .37435 L
.44769 .37448 L
.45054 .37453 L
.45361 .37458 L
.45647 .37461 L
.45909 .37464 L
.4616 .37466 L
.46286 .37467 L
.46425 .37467 L
.46504 .37468 L
.46579 .37468 L
.46719 .37468 L
.46797 .37468 L
.4687 .37468 L
.46949 .37468 L
.47033 .37468 L
.47173 .37468 L
.47303 .37468 L
.47447 .37467 L
.47601 .37466 L
.47906 .37464 L
.48078 .37462 L
.48238 .37461 L
Mistroke
.48556 .37456 L
.48844 .37451 L
.49394 .37438 L
.4998 .3742 L
.50532 .37396 L
.51114 .37365 L
.52156 .37289 L
.53275 .37174 L
.54498 .37005 L
.56661 .36589 L
.58914 .3599 L
.63161 .34446 L
.6724 .3288 L
.71258 .31614 L
.75512 .302 L
.77573 .2939 L
.79444 .28547 L
.81177 .2761 L
.82901 .2643 L
.84556 .24952 L
.85902 .23409 L
.87107 .21631 L
.87971 .19897 L
.88343 .18869 L
.88491 .18354 L
.88606 .17873 L
.88691 .17439 L
.88764 .16957 L
.88793 .16705 L
.88815 .16468 L
.88832 .16217 L
.88839 .16076 L
.88843 .15946 L
.88846 .1581 L
.88847 .15682 L
.88847 .15566 L
.88845 .15442 L
.88841 .15305 L
.88838 .15237 L
.88834 .15162 L
.88817 .14904 L
.88794 .14663 L
.88762 .14406 L
.88685 .13945 L
.8855 .13372 L
.88382 .12844 L
.88197 .12385 L
.87958 .11906 L
.87799 .11635 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1Co`<0=_l0097o00<0ool0C_l60003o`0000<0<_l0097o00<0ool0
C?l200co0P0`o`00TOl00`3oo`1:o`804?l202ko002Ao`030?oo04Wo00<0ool04_l202co002Ao`03
0?oo04So00<0ool05Ol00`3oo`0Yo`00OOl200Co0P05o`<01?l00`3oo`17o`030?oo01Ko00<0ool0
:Ol007co00@0ool02ol00`3oo`03o`030?oo04Ko00<0ool06?l00`3oo`0Xo`00O?l0103oo`08o`D0
1?l204Ko00<0ool06Ol00`3oo`0Xo`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Go00<0ool0
6_l00`3oo`0Wo`00AolH01go00@0ool02Ol00`3o0005o`030?oo04Go00<0ool06_l00`3oo`0Wo`00
@ol401So0P0Lo`802ol200Go00<0ool0AOl00`3oo`0Ko`030?oo02Ko000mo`H07_l202ko00<0ool0
A?l00`3oo`0Lo`030?oo02Ko000ko`809Ol202go00<0ool0A?l00`3oo`0Lo`030?oo02Ko000io`80
:?l202co00<0ool0A?l00`3oo`0Lo`030?oo02Ko000go`80:_l202co00<0ool0A?l00`3oo`0Mo`03
0?oo02Go000fo`030?oo02[o00<0ool0:ol00`3oo`14o`030?oo01go00<0ool09Ol003Co0P0/o`80
;Ol00`3oo`13o`030?oo01ko00<0ool09Ol003?o00<0ool0:_l202oo00<0ool0@ol00`3oo`0No`03
0?oo02Go000bo`030?oo02Wo0P0ao`030?oo04?o00<0ool07_l00`3oo`0Uo`00<Ol00`3oo`0Yo`03
0?oo037o00<0ool0@ol00`3oo`0No`030?oo02Go000`o`030?oo02So0P0do`030?oo04?o00<0ool0
7_l00`3oo`0Uo`00<?l00`3oo`0Vo`80=_l00`3oo`13o`030?oo01ko00<0ool09Ol0033o00<0ool0
9Ol00`3oo`0Ro`801?l200?o1005o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00;ol00`3oo`0To`80
9?l0103oo`08o`030?oo00Ko00<0ool0@ol00`3oo`0Mo`030?oo02Ko000_o`030?oo02?o00<0ool0
9?l0103oo`09o`030?oo00Go0P13o`030?oo01ko00<0ool09_l002oo00<0ool08_l00`3oo`0Oo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`030?oo02;o00<0ool0
9Ol0103oo`08o`040?oo00Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000^o`030?oo02;o00<0ool0
9ol200[o0P06o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Qo`030?oo03co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo027o00<0ool0??l00`3oo`11o`030?oo01go00<0ool0
:?l002ko00<0ool08Ol00`3oo`0lo`030?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`80
6?l00`3oo`08o`D05?l00`3oo`0;o`801?l200Co0P0So`030?oo02;o0P04o`801?l2013o00<0ool0
4?l500So00<0ool05ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
5?l00`3oo`0:o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03_l00`3oo`0Co`03
0?oo00So00<0ool06Ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`03
0?oo01Co00<0ool02_l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00_o00<0ool0
5?l00`3oo`07o`030?oo01[o00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806Ol0103o
ool400Co00<0ool05?l00`3oo`04o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0=o`03
0?oo01Go00<0ool01_l00`3oo`0Ko`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01So00<0ool0
2Ol00`3oo`0Eo`030?oo00Wo00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0<o`03
0?oo01Go00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<06Ol00`3oo`07o`80
5ol00`3oo`0:o`802_l302;o00<0ool08_l200[o0`0;o`030?oo01Go0P07o`030?oo01_o0P0:o`<0
0_l0033o00<0ool08Ol00`3oo`0jo`030?oo03[o0P0Qo`030?oo02co000`o`030?oo02;o00<0ool0
>Ol00`3oo`0io`030?oo023o00<0ool0;Ol0037o00<0ool08_l00`3oo`0ho`030?oo03So00<0ool0
8?l00`3oo`0^o`00<Ol00`3oo`0So`030?oo03Oo00<0ool0=ol00`3oo`0Po`030?oo02oo000bo`03
0?oo02?o0P0go`030?oo03Ko00<0ool08?l00`3oo`0`o`003?oo00`02Ol000co00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`03
0?oo02Co00<0ool00_l00`3oo`0Ro`801Ol00`3oo`0Zo`030?oo02Wo00<0ool01_l2027o0P0[o`03
0?oo00Oo000do`030?oo02So00<0ool0;ol00`3oo`0`o`808ol00`3oo`0do`00=?l00`3oo`0Yo`80
;ol00`3oo`0^o`809?l00`3oo`0eo`00=Ol00`3oo`0Zo`80;Ol00`3oo`0/o`809?l203So000fo`03
0?oo02_o0P0[o`030?oo02Wo0`0To`80>_l003Oo00<0ool0;?l202Wo00<0ool09ol202Go0P0lo`00
>?l00`3oo`0]o`809ol00`3oo`0To`<09Ol203ko000io`030?oo02ko0`0To`030?oo027o0`0Wo`03
0?oo03ko000jo`030?oo033o0`0Qo`030?oo01go100Xo`80@Ol003_o0P0co`807ol00`3oo`0Go`H0
:_l204?o000mo`030?oo03;o100Ko`030?oo017o1P0^o`80AOl003ko0P0fo``03ol00`3oo`0<o`D0
<Ol304Oo0010o`80@?lN03?o0`1:o`00@_l00`3oo`1<o`030?oo03co0`1=o`00@ol203So0P04o`80
0ol400Go00<0ool0>_l2053o0015o`80=Ol0103oo`08o`030?oo00Ko00<0ool0=ol305;o0017o`80
<ol0103oo`09o`030?oo00Go0P0eo`<0EOl004Wo0P0ao`040?oo00[o00<0ool01?l00`3oo`0`o`@0
F?l004_o0`0^o`040?oo00So00@0ool01Ol00`3oo`0/o`@0G?l004ko0`0/o`802_l200Ko00<0ool0
:Ol3063o001Ao`80?_l00`3oo`0Wo`80Hol005?o0`0ko`030?oo02Co0`1Uo`00E_l303So00<0ool0
8_l206So001Io`<0=Ol00`3oo`0Oo`<0J_l005co100ao`030?oo01co0`1]o`00H?l302ko00<0ool0
6Ol3073o001So`<0:ol00`3oo`0Fo`<0Lol006Ko100Wo`030?oo01?o0`1fo`00J_l402?o00<0ool0
4?l307Wo001^o`<08?l00`3oo`0;o`D0O?l0077o2@0Go`030?oo00Co1`21o`00N_lN08So001lo`04
0?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87758 .1209 m
.87497 .11691 L
.87225 .11341 L
.86868 .10954 L
.8651 .10634 L
.85689 .10082 L
.8519 .09842 L
.84694 .09659 L
.84163 .09516 L
.83888 .09461 L
.83581 .09414 L
.83317 .09385 L
.83167 .09374 L
.8303 .09366 L
.82903 .09361 L
.82763 .09359 L
.82631 .09359 L
.82509 .09361 L
.82429 .09364 L
.82356 .09367 L
.8219 .09378 L
.82044 .0939 L
.8189 .09406 L
.81625 .09441 L
.81336 .09492 L
.80711 .09645 L
.80151 .09839 L
.79665 .10058 L
.79172 .10339 L
.78773 .10623 L
.78387 .1096 L
.78077 .11292 L
.77776 .1169 L
.7735 .12462 L
.77168 .12922 L
.77013 .13429 L
.76906 .13879 L
.76812 .14384 L
.76657 .15549 L
.76531 .16651 L
.76457 .17202 L
.76359 .17792 L
.76243 .18338 L
.76113 .18831 L
.75794 .19729 L
.75349 .20624 L
.74806 .21448 L
.74122 .22272 L
.72705 .23624 L
.71037 .2492 L
Mistroke
.69124 .26147 L
.67175 .27107 L
.65252 .278 L
.63094 .28376 L
.61107 .28802 L
.58859 .29216 L
.56773 .29547 L
.54579 .29832 L
.53503 .29944 L
.52521 .30029 L
.51514 .30096 L
.5096 .30123 L
.50436 .30141 L
.5016 .30148 L
.50009 .30151 L
.49867 .30154 L
.49746 .30155 L
.49617 .30157 L
.49473 .30158 L
.49341 .30158 L
.49198 .30158 L
.49064 .30157 L
.48987 .30157 L
.48918 .30156 L
.48762 .30155 L
.48611 .30152 L
.48448 .30149 L
.48153 .30142 L
.47612 .30123 L
.47107 .30099 L
.45961 .30025 L
.44851 .29931 L
.43669 .29813 L
.41401 .29532 L
.40373 .29373 L
.39286 .29171 L
.37226 .28653 L
.35363 .28006 L
.31777 .26333 L
.30062 .25337 L
.29222 .24759 L
.28363 .24058 L
.27644 .23316 L
.27104 .22564 L
.26876 .22129 L
.26719 .21724 L
.26655 .21498 L
.26628 .21379 L
.26605 .21252 L
.26589 .21141 L
Mistroke
.26583 .21083 L
.26577 .2102 L
.26571 .20909 L
.26569 .20805 L
.26569 .20748 L
.26571 .20694 L
.26577 .20593 L
.26587 .20484 L
.26595 .20423 L
.26604 .20367 L
.26644 .20172 L
.26702 .19969 L
.26784 .1975 L
.26878 .1955 L
.27151 .19109 L
.27457 .18739 L
.27782 .18418 L
.28645 .17753 L
.30586 .16608 L
.31368 .16146 L
.31731 .15903 L
.32072 .15642 L
.3234 .15391 L
.32531 .15161 L
.32605 .15046 L
.32661 .1494 L
.32707 .1483 L
.32725 .14772 L
.3274 .1471 L
.32747 .14678 L
.32752 .14644 L
.32758 .14583 L
.3276 .14519 L
.32759 .14482 L
.32757 .14448 L
.32749 .14386 L
.32738 .14331 L
.3272 .14267 L
.32699 .14207 L
.32648 .14098 L
.32588 .13999 L
.32406 .13782 L
.32282 .13671 L
.32153 .13572 L
.31812 .13365 L
.31459 .13201 L
.3104 .1305 L
.30628 .12934 L
.30218 .12843 L
.29745 .12761 L
Mistroke
.29216 .12694 L
.28939 .12667 L
.2864 .12645 L
.28382 .1263 L
.28096 .12618 L
.27935 .12613 L
.27787 .12609 L
.27705 .12608 L
.27616 .12607 L
.27532 .12606 L
.27453 .12605 L
.27374 .12605 L
.27299 .12605 L
.2716 .12606 L
.27082 .12607 L
.2701 .12608 L
.26847 .1261 L
.26715 .12613 L
.26574 .12617 L
.2632 .12626 L
.25744 .12652 L
.2514 .12689 L
.24594 .1273 L
.22525 .12941 L
.21373 .13105 L
.20309 .13296 L
.19286 .1353 L
.18334 .13806 L
.17317 .14182 L
.16282 .14689 L
.15439 .15238 L
.1464 .1595 L
.14296 .16353 L
.13975 .16815 L
.13736 .17244 L
.13527 .17722 L
.1336 .18232 L
.13247 .1872 L
.13201 .18998 L
.13167 .19261 L
.13153 .19413 L
.13142 .19553 L
.13138 .19631 L
.13133 .19716 L
.1313 .19796 L
.13128 .19871 L
.13126 .20005 L
.13126 .20081 L
.13126 .20151 L
.13129 .2028 L
Mistroke
.13133 .20416 L
.13139 .20539 L
.13146 .20653 L
.13168 .20914 L
.13199 .21196 L
.13235 .21453 L
.1334 .22027 L
.1346 .22536 L
.13613 .23067 L
.13949 .2399 L
.14394 .2493 L
.14976 .25892 L
.16235 .27416 L
.17867 .28821 L
.19629 .29984 L
.21604 .31043 L
.25348 .32661 L
.29452 .34163 L
.33439 .35399 L
.35512 .35949 L
.37799 .36465 L
.38885 .36665 L
.40043 .3684 L
.4104 .36958 L
.42143 .37055 L
.4272 .37092 L
.43269 .37121 L
.43769 .37142 L
.44303 .3716 L
.44884 .37174 L
.45422 .37184 L
.45722 .37188 L
.46003 .37191 L
.46153 .37192 L
.46312 .37193 L
.46486 .37194 L
.46569 .37194 L
.46646 .37195 L
.46784 .37195 L
.46857 .37195 L
.46934 .37195 L
.47063 .37195 L
.47206 .37195 L
.47333 .37194 L
.47469 .37194 L
.47598 .37193 L
.47716 .37193 L
.47997 .3719 L
.48303 .37187 L
.48592 .37182 L
Mistroke
.48861 .37177 L
.49384 .37165 L
.49858 .3715 L
.50405 .37128 L
.50919 .37102 L
.52012 .37026 L
.52561 .36977 L
.53163 .36913 L
.55211 .36612 L
.5633 .36392 L
.57337 .36159 L
.59545 .35529 L
.63584 .34034 L
.67489 .32594 L
.71738 .3131 L
.75803 .29997 L
.79555 .28472 L
.81367 .27514 L
.83203 .26257 L
.84815 .24798 L
.86289 .23042 L
.86861 .22195 L
.87398 .21264 L
.87828 .20369 L
.88157 .19525 L
.88454 .18521 L
.88573 .17981 L
.88622 .17703 L
.88667 .17402 L
.887 .17137 L
.88723 .16894 L
.88733 .16776 L
.88741 .16648 L
.88748 .16528 L
.88752 .16414 L
.88755 .16345 L
.88757 .1627 L
.88759 .16139 L
.88759 .16064 L
.88759 .15985 L
.88758 .15918 L
.88756 .15844 L
.88752 .15712 L
.88749 .15638 L
.88746 .15571 L
.88729 .15315 L
.88712 .1512 L
.88687 .14905 L
.88621 .14472 L
.88539 .14074 L
Mistroke
.88445 .13707 L
.88165 .12903 L
.87967 .12471 L
.87758 .1209 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Bo`@0=_l0097o00<0ool0C_l400Co100bo`00
TOl00`3oo`1;o`<03?l2033o002Ao`030?oo04[o00<0ool03ol202ko002Ao`030?oo04Wo00<0ool0
4_l00`3oo`0[o`00OOl200Co0P05o`<01?l00`3oo`18o`030?oo01Co00<0ool0:_l007co00@0ool0
2ol00`3oo`03o`030?oo04Oo00<0ool05_l00`3oo`0Yo`00O?l0103oo`08o`D01?l204Oo00<0ool0
5ol00`3oo`0Yo`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Go00<0ool06Ol00`3oo`0Xo`00
BolA023o00@0ool02Ol00`3o0005o`030?oo04Go00<0ool06Ol00`3oo`0Xo`00A?l7017o0`0No`80
2ol200Go00<0ool0AOl00`3oo`0Jo`030?oo02Oo000no`H06ol2033o00<0ool0A?l00`3oo`0Ko`03
0?oo02Oo000lo`808ol202ko00<0ool0A?l00`3oo`0Lo`030?oo02Ko000jo`809_l202go00<0ool0
A?l00`3oo`0Lo`030?oo02Ko000ho`80:Ol00`3oo`0[o`030?oo04Co00<0ool07?l00`3oo`0Vo`00
=ol00`3oo`0Xo`80;Ol00`3oo`14o`030?oo01co00<0ool09_l003Go0P0[o`030?oo02co00<0ool0
@ol00`3oo`0Mo`809ol003?o0P0/o`030?oo02go00<0ool0@ol00`3oo`0No`030?oo02Go000bo`03
0?oo02[o0P0`o`030?oo04?o00<0ool07_l00`3oo`0Uo`00<_l00`3oo`0Yo`030?oo033o00<0ool0
@ol00`3oo`0No`030?oo02Go000ao`030?oo02Wo00<0ool0<Ol00`3oo`13o`030?oo01ko00<0ool0
9Ol0033o00<0ool0:?l203Co00<0ool0@ol00`3oo`0Mo`030?oo02Ko000_o`030?oo02Oo0P0Ro`80
1?l200?o1005o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00;ol00`3oo`0Uo`808ol0103oo`08o`03
0?oo00Ko00<0ool0@ol00`3oo`0Mo`030?oo02Ko000_o`030?oo02?o0P0Uo`040?oo00Wo00<0ool0
1Ol204?o00<0ool07_l00`3oo`0Vo`00;ol00`3oo`0So`030?oo01ko1002o`040?oo00[o00<0ool0
1?l00`3oo`12o`030?oo01ko00<0ool09_l002oo00<0ool08_l00`3oo`0Uo`040?oo00So00@0ool0
1Ol00`3oo`12o`030?oo01ko00<0ool09_l002ko0P0To`030?oo02Ko0P0:o`801_l00`3oo`11o`03
0?oo01ko00<0ool09ol002ko00<0ool08_l00`3oo`0ko`030?oo047o00<0ool07_l00`3oo`0Wo`00
;_l00`3oo`0Qo`80?Ol00`3oo`10o`030?oo01ko00<0ool0:?l002ko00<0ool08Ol00`3oo`0lo`03
0?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`806Ol00`3oo`07o`D05?l00`3oo`0;o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500Wo00<0ool05_l500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool05?l00`3oo`0:o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool03_l00`3oo`0Co`030?oo00So00<0ool06Ol00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`030?oo01Co0P0;o`040?oo00_o00<0ool0
8?l00`3oo`0Qo`040?oo00_o00<0ool02ol00`3oo`0Do`030?oo00So00<0ool06Ol00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Eo`030?oo00?o1002o`04
0?oo00Wo0P0So`030?oo027o00@0ool02Ol200go00<0ool05Ol00`3oo`07o`030?oo01[o00<0ool0
2?l200?o0009o`030?oo00So00<0ool06?l00`3oo`09o`030?oo01Go00<0ool02Ol0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00co00<0ool05Ol00`3oo`07o`030?oo01[o00<0ool0
2?l00`3oo`02o`002?l200[o0`0Io`030?oo00Oo0P0Ho`030?oo00Wo0P0:o`<08_l00`3oo`0Ro`80
2_l300_o00<0ool05Ol200So00<0ool06_l200[o0`02o`00<?l00`3oo`0Qo`030?oo03[o00<0ool0
>_l2027o00<0ool0;?l0037o00<0ool08Ol203[o00<0ool0>Ol00`3oo`0Po`030?oo02go000ao`03
0?oo02?o00<0ool0=ol00`3oo`0ho`030?oo023o00<0ool0;_l0037o00<0ool09?l00`3oo`0fo`03
0?oo03Ko0P0Ro`030?oo02oo000bo`030?oo02Co00<0ool0=Ol00`3oo`0eo`030?oo027o00<0ool0
<?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko0P07o`030?oo00Go00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool08Ol200Ko00<0ool0
:_l00`3oo`0Yo`030?oo00Co0P0So`80:ol00`3oo`07o`00=?l00`3oo`0Wo`80<Ol00`3oo`0_o`03
0?oo02?o00<0ool0=?l003Go00<0ool0:?l202oo00<0ool0;Ol202Go00<0ool0=Ol003Ko00<0ool0
:Ol202go00<0ool0:ol202Go0P0ho`00=ol00`3oo`0Zo`<0:_l00`3oo`0Xo`<09_l00`3oo`0ho`00
>?l00`3oo`0/o`80:?l00`3oo`0To`@09ol203_o000io`030?oo02go0`0Uo`030?oo027o0`0Yo`80
?Ol003[o00<0ool0;ol202?o00<0ool07Ol402Wo0`0oo`00>ol00`3oo`0`o`@07ol00`3oo`0Ho`D0
:ol204;o000lo`80=?l401_o00<0ool04_l602go0`14o`00?_l203Ko200Co`030?oo00go1@0ao`80
Aol0043o00<0ool0>ol:00Wo00<0ool00ol:03?o0`19o`00@Ol204Go3`0jo`<0C?l004?o0P1<o`03
0?oo03_o0P1?o`00AOl203Ko0P04o`800ol400Go00<0ool0>?l3057o0017o`80<ol0103oo`08o`03
0?oo00Ko00<0ool0=Ol305Co0019o`<0<?l0103oo`09o`030?oo00Go0P0bo`@0Eol004co0P0^o`04
0?oo00[o00<0ool01?l00`3oo`0]o`@0Fol004ko0`0[o`040?oo00So00@0ool01Ol00`3oo`0Zo`<0
Gol0057o0`0Yo`802_l200Ko00<0ool0:?l206;o001Do`80>ol00`3oo`0Uo`<0I?l005Ko0`0ho`03
0?oo02?o0P1Wo`00FOl303Go00<0ool08?l306Wo001Lo`80<ol00`3oo`0Mo`<0K?l005ko0`0`o`03
0?oo01_o0P1_o`00HOl202ko00<0ool06?l3077o001So`@0:_l00`3oo`0Eo`<0M?l006Oo1P0To`03
0?oo01;o0`1go`00KOl501oo00<0ool03?l607[o001bo`H06Ol00`3oo`08o`@0P?l007So9024o`00
OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool0
2?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`03
0?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
0ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool0
1Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00
OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87716 .12546 m
.87205 .11788 L
.86517 .11068 L
.86121 .10755 L
.85663 .10463 L
.85205 .1023 L
.84759 .10051 L
.84208 .09886 L
.83909 .0982 L
.83586 .09765 L
.8345 .09747 L
.83303 .09731 L
.83175 .09719 L
.83035 .09709 L
.8295 .09705 L
.82871 .09702 L
.82796 .09699 L
.82715 .09698 L
.82636 .09697 L
.82561 .09697 L
.82421 .097 L
.82293 .09705 L
.82153 .09713 L
.82007 .09724 L
.81872 .09737 L
.81629 .09767 L
.81371 .09809 L
.80793 .09938 L
.80284 .10097 L
.79813 .10287 L
.79343 .10526 L
.78585 .11051 L
.7822 .11392 L
.77915 .11742 L
.77626 .12155 L
.77359 .12641 L
.77163 .13104 L
.77012 .13556 L
.76882 .14053 L
.76771 .14602 L
.76617 .1563 L
.76495 .16582 L
.76421 .17105 L
.7633 .17653 L
.76114 .18613 L
.75957 .19129 L
.75784 .19597 L
.75286 .20608 L
.74682 .21499 L
.74008 .22283 L
Mistroke
.72564 .23613 L
.69253 .25899 L
.67415 .26815 L
.65329 .27574 L
.63096 .28168 L
.60959 .28621 L
.56825 .29318 L
.54647 .29591 L
.53531 .29704 L
.52335 .29801 L
.51755 .29838 L
.51206 .29867 L
.50714 .29887 L
.50436 .29895 L
.50179 .29902 L
.50064 .29904 L
.4994 .29906 L
.49833 .29907 L
.49716 .29909 L
.49586 .2991 L
.4945 .2991 L
.49323 .2991 L
.49206 .2991 L
.49071 .29909 L
.48944 .29908 L
.4883 .29907 L
.48706 .29905 L
.48429 .299 L
.48174 .29893 L
.47662 .29874 L
.47113 .29848 L
.46124 .29784 L
.43893 .2958 L
.41642 .29309 L
.40622 .29157 L
.39545 .28964 L
.37501 .28472 L
.35646 .27856 L
.32058 .26261 L
.30341 .25311 L
.29495 .2476 L
.28623 .24089 L
.27882 .23376 L
.27311 .22652 L
.27061 .22231 L
.26882 .21839 L
.26805 .2162 L
.26771 .21504 L
.26741 .21381 L
.26718 .21273 L
Mistroke
.26698 .21154 L
.26685 .21047 L
.26676 .20946 L
.26671 .20844 L
.2667 .20787 L
.2667 .20735 L
.26673 .20635 L
.2668 .20541 L
.26685 .20487 L
.26692 .20431 L
.26708 .2033 L
.26729 .20223 L
.26758 .20106 L
.26819 .19911 L
.26902 .19706 L
.27097 .19345 L
.27359 .18982 L
.27658 .18653 L
.28328 .18073 L
.29956 .17012 L
.3086 .1646 L
.31578 .15955 L
.31856 .15716 L
.32093 .15466 L
.32193 .15335 L
.32283 .15193 L
.32346 .15064 L
.32389 .14944 L
.32405 .14878 L
.32413 .14841 L
.32418 .14807 L
.32424 .14745 L
.32426 .14714 L
.32426 .14679 L
.32424 .1462 L
.32416 .14557 L
.32405 .14498 L
.32391 .14445 L
.32369 .14379 L
.32344 .14319 L
.3227 .14185 L
.32187 .1407 L
.32079 .13952 L
.31813 .13732 L
.3151 .13546 L
.3112 .13365 L
.30651 .132 L
.30202 .13079 L
.29755 .12985 L
.2924 .12904 L
Mistroke
.28941 .12868 L
.28665 .12841 L
.28397 .1282 L
.28143 .12804 L
.27876 .12791 L
.27734 .12786 L
.2758 .12782 L
.2745 .12779 L
.27307 .12777 L
.27172 .12776 L
.27047 .12776 L
.26923 .12776 L
.2681 .12777 L
.2668 .12779 L
.26557 .12781 L
.26276 .12788 L
.25978 .12798 L
.25447 .12824 L
.2488 .12859 L
.24262 .12907 L
.22169 .13134 L
.21012 .1331 L
.19962 .13514 L
.18925 .13772 L
.17981 .1407 L
.16978 .14476 L
.15981 .15009 L
.15195 .15568 L
.14465 .16271 L
.14178 .16629 L
.13912 .17028 L
.13701 .17414 L
.13531 .17795 L
.1337 .18261 L
.13258 .187 L
.1321 .18951 L
.13175 .19186 L
.13146 .19447 L
.13136 .19579 L
.13131 .19649 L
.13127 .19726 L
.13122 .19852 L
.13119 .19973 L
.13118 .2009 L
.13118 .202 L
.13121 .2033 L
.13125 .20448 L
.13133 .20585 L
.13142 .20714 L
.13163 .20948 L
Mistroke
.13187 .21163 L
.13262 .21647 L
.13364 .2214 L
.135 .22661 L
.13808 .23577 L
.1427 .24611 L
.14774 .25491 L
.15961 .27017 L
.16742 .2778 L
.17527 .2843 L
.21018 .30567 L
.24663 .32177 L
.28685 .33663 L
.32591 .34906 L
.34601 .35459 L
.36838 .35996 L
.37888 .36212 L
.39017 .36412 L
.40084 .36567 L
.41073 .36682 L
.42188 .36778 L
.42778 .36817 L
.43404 .36849 L
.43914 .3687 L
.44462 .36888 L
.45065 .36902 L
.45631 .36912 L
.45914 .36916 L
.46216 .36918 L
.46474 .3692 L
.46621 .36921 L
.46759 .36921 L
.46831 .36922 L
.46909 .36922 L
.46991 .36922 L
.47067 .36922 L
.47204 .36922 L
.47351 .36921 L
.47433 .36921 L
.47523 .36921 L
.47685 .3692 L
.47843 .36919 L
.4799 .36917 L
.48257 .36914 L
.48507 .36911 L
.49076 .369 L
.49631 .36884 L
.50227 .36862 L
.50809 .36834 L
.51349 .36801 L
Mistroke
.52569 .36701 L
.53642 .3658 L
.54779 .36414 L
.57095 .35949 L
.59214 .3537 L
.63104 .3398 L
.6724 .32475 L
.71326 .31273 L
.75243 .30095 L
.77379 .29348 L
.7927 .28586 L
.81171 .27653 L
.82835 .26607 L
.84408 .2532 L
.85919 .23686 L
.8704 .22066 L
.87561 .21098 L
.87964 .20166 L
.88278 .19223 L
.88403 .1874 L
.88513 .18205 L
.88585 .17741 L
.88616 .17479 L
.88638 .1724 L
.88654 .17016 L
.88661 .1689 L
.88665 .16773 L
.88668 .1667 L
.8867 .16557 L
.8867 .16439 L
.8867 .16329 L
.88667 .16202 L
.88663 .16067 L
.88656 .1594 L
.88649 .15825 L
.88626 .15541 L
.88596 .15275 L
.8852 .14797 L
.88408 .14287 L
.88275 .13827 L
.88104 .13353 L
.87716 .12546 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Ao`H0=Ol0097o00<0ool0
COl400Ko0`0bo`00TOl00`3oo`1;o`803Ol302oo002Ao`030?oo04[o00<0ool04?l00`3oo`0/o`00
OOl200Co0P05o`<01?l00`3oo`19o`030?oo01;o00<0ool0:ol007co00@0ool02ol00`3oo`03o`03
0?oo04So00<0ool04ol00`3oo`0[o`00O?l0103oo`08o`D01?l204So00<0ool05Ol00`3oo`0Zo`00
M_l400;o00@0ool02?l0103oo`05o`030?oo04Ko00<0ool05ol00`3oo`0Yo`00D_l402Ko00@0ool0
2Ol00`3o0005o`030?oo04Go00<0ool06Ol00`3oo`0Xo`00AOl=00?o200Po`802ol200Go00<0ool0
AOl00`3oo`0Io`030?oo02So0010o`D06?l203;o00<0ool0A?l00`3oo`0Ko`030?oo02Oo000mo`<0
7ol2033o00<0ool0A?l00`3oo`0Ko`030?oo02Oo000ko`809?l00`3oo`0]o`030?oo04Co00<0ool0
6ol00`3oo`0Wo`00>?l302Oo00<0ool0;?l00`3oo`14o`030?oo01co00<0ool09_l003Ko0P0Zo`03
0?oo02co00<0ool0A?l00`3oo`0Lo`030?oo02Ko000eo`030?oo02Wo0P0^o`030?oo04?o00<0ool0
7Ol00`3oo`0Vo`00=?l00`3oo`0Yo`80;ol00`3oo`13o`030?oo01go00<0ool09_l003?o00<0ool0
:Ol00`3oo`0_o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00<_l00`3oo`0Yo`030?oo033o00<0ool0
@ol00`3oo`0Mo`030?oo02Ko000ao`030?oo02So0P0co`030?oo04?o00<0ool07Ol00`3oo`0Vo`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0@ol00`3oo`0Mo`030?oo02Ko000`o`030?oo02Ko0P0Ro`80
1?l200?o1005o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00;ol00`3oo`0Vo`030?oo027o00@0ool0
2?l00`3oo`06o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00;ol00`3oo`0To`809?l0103oo`09o`03
0?oo00Go0P13o`030?oo01ko00<0ool09_l002oo00<0ool08ol00`3oo`0No`@00_l0103oo`0:o`03
0?oo00Co00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`030?oo02;o00<0ool09Ol0103oo`08o`04
0?oo00Go00<0ool0@_l00`3oo`0No`030?oo02Ko000^o`809?l00`3oo`0Vo`802_l200Ko00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02;o00<0ool0>ol00`3oo`11o`030?oo01ko00<0ool0
9ol002ko00<0ool08_l00`3oo`0ko`030?oo043o00<0ool07_l00`3oo`0Xo`00;_l00`3oo`0Qo`80
?Ol00`3oo`10o`030?oo01ko00<0ool0:?l000Oo1@02o`801?l201So0P09o`D05?l00`3oo`0;o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500So00<0ool05ol500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool05Ol00`3oo`09o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool03_l00`3oo`0Co`030?oo00So00<0ool06Ol00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`030?oo01Go00<0ool02Ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00_o00<0ool05?l00`3oo`08o`030?oo01Wo00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`806Ol0103oool400Co00<0ool05Ol00`3oo`03o`@0
0_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0=o`030?oo01Go00<0ool01ol00`3oo`0Jo`03
0?oo00So0P03o`002Ol00`3oo`08o`030?oo01So00<0ool02Ol00`3oo`0Eo`030?oo00Wo00@0ool0
2Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0;o`030?oo01Ko00<0ool01_l00`3oo`0Ko`03
0?oo00So00<0ool00_l000So0P0:o`<06?l00`3oo`08o`806?l00`3oo`09o`802_l302;o00<0ool0
8_l200[o0`0:o`030?oo01Ko0P08o`030?oo01[o0P0:o`<00_l0033o00<0ool08_l00`3oo`0io`03
0?oo03[o00<0ool08?l00`3oo`0/o`00<?l00`3oo`0So`030?oo03So00<0ool0>Ol00`3oo`0Qo`03
0?oo02co000`o`030?oo02Co00<0ool0=ol00`3oo`0go`808ol00`3oo`0]o`00<Ol00`3oo`0To`80
=ol00`3oo`0fo`030?oo02;o00<0ool0;_l0037o00<0ool09_l00`3oo`0do`030?oo03Co0P0To`03
0?oo02oo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`00
1?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`<01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go0P08o`030?oo00Go00<0ool01_l00`3oo`06o`040?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Uo`050?ooo`009Ol200Co00<0ool0:_l00`3o
o`0Yo`030?oo00Co00<0ool08_l302[o00<0ool01ol003Go00<0ool0:?l202oo00<0ool0;_l202Go
00<0ool0=?l003Go00<0ool0:_l202go00<0ool0;?l202Ko00<0ool0=Ol003Ko00<0ool0:ol202_o
00<0ool0:_l202Ko0P0ho`00=ol00`3oo`0/o`80:Ol00`3oo`0Wo`<09_l203[o000ho`030?oo02go
0P0Wo`030?oo02?o100Xo`030?oo03[o000io`030?oo02ko0`0To`030?oo01ko1@0Zo`80?Ol003[o
00<0ool0<?l4023o00<0ool07?l202go0P0oo`00>ol203Co0`0Mo`030?oo01Go1`0]o`80@Ol003go
00<0ool0=?l:01?o00<0ool03Ol803;o0P13o`00?_l203ko1P0=o`030?oo00So1@0go`<0AOl0043o
00<0ool0@OlH03[o0P18o`00@Ol204ko00<0ool0?ol304[o0013o`80C?l00`3oo`0ko`@0COl004Go
0P0fo`801?l200?o1005o`030?oo03Oo101Ao`00Aol303;o00@0ool02?l00`3oo`06o`030?oo03Co
0`1Eo`00B_l2033o00@0ool02Ol00`3oo`05o`80<Ol405So001<o`<0;Ol0103oo`0:o`030?oo00Co
00<0ool0;?l405co001?o`<0:_l0103oo`08o`040?oo00Go00<0ool0:Ol3063o001Bo`80:Ol200[o
0P06o`030?oo02Ko0`1So`00E?l303[o00<0ool09?l206Ko001Go`<0=ol00`3oo`0Qo`<0J?l005[o
0`0do`030?oo01ko0`1[o`00GOl4033o00<0ool07?l206ko001Qo`<0;Ol00`3oo`0Io`<0L?l006Co
0`0Zo`030?oo01Oo0P1co`00Iol402Ko00<0ool04_l507Go001[o`@08_l00`3oo`0<o`H0N_l006oo
1P0Lo`030?oo00So1020o`00MOl901?o2`24o`00O_lD08ko001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87675 .13001 m
.87185 .12235 L
.86523 .11501 L
.86142 .1118 L
.857 .10877 L
.85257 .10633 L
.84825 .10443 L
.84289 .10265 L
.83998 .10191 L
.83683 .10128 L
.83406 .10086 L
.83144 .10057 L
.82992 .10045 L
.82854 .10037 L
.82777 .10034 L
.82693 .10032 L
.82614 .1003 L
.8254 .1003 L
.82404 .10031 L
.82327 .10033 L
.82257 .10035 L
.82127 .10042 L
.8199 .10051 L
.81753 .10073 L
.81494 .10107 L
.81219 .10153 L
.80928 .10214 L
.80416 .10354 L
.79897 .10544 L
.79442 .10756 L
.78986 .11022 L
.78526 .1136 L
.77871 .12028 L
.77576 .12444 L
.77339 .12864 L
.77138 .13316 L
.76982 .13759 L
.76848 .1424 L
.76729 .14786 L
.76427 .16737 L
.76343 .17274 L
.76236 .1785 L
.76123 .18349 L
.7598 .18865 L
.75631 .19802 L
.75211 .20611 L
.74666 .21407 L
.73981 .22199 L
.72559 .23488 L
.70887 .24718 L
Mistroke
.68982 .25875 L
.67051 .26776 L
.65148 .27427 L
.63019 .27977 L
.61065 .28391 L
.58858 .2879 L
.56806 .29098 L
.54773 .29344 L
.53731 .29447 L
.52615 .29539 L
.52039 .29578 L
.51495 .29608 L
.51007 .2963 L
.50755 .2964 L
.50478 .29648 L
.50219 .29654 L
.49977 .29659 L
.49843 .2966 L
.49721 .29662 L
.49586 .29663 L
.49445 .29663 L
.49331 .29663 L
.49209 .29662 L
.49094 .29662 L
.4899 .29661 L
.48867 .29659 L
.48734 .29657 L
.48492 .29652 L
.48236 .29645 L
.47962 .29636 L
.47466 .29615 L
.46341 .29546 L
.45157 .29449 L
.44083 .29345 L
.41906 .29089 L
.40963 .28955 L
.39928 .2878 L
.38033 .28359 L
.36049 .27747 L
.34011 .26948 L
.32226 .26133 L
.30452 .25186 L
.28906 .24134 L
.28226 .23537 L
.27599 .2284 L
.27304 .22419 L
.27077 .2201 L
.26922 .21636 L
.26856 .21424 L
.26831 .21323 L
Mistroke
.26811 .21227 L
.26795 .21134 L
.26781 .21033 L
.26771 .20941 L
.26766 .20855 L
.26763 .20754 L
.26764 .20663 L
.26766 .20612 L
.2677 .20558 L
.2678 .20459 L
.26791 .20373 L
.26808 .20279 L
.26853 .20092 L
.26908 .1992 L
.26971 .19761 L
.27177 .19376 L
.27421 .19037 L
.28138 .1832 L
.29898 .17089 L
.30656 .16591 L
.31361 .16054 L
.31618 .1581 L
.31838 .1555 L
.31918 .1543 L
.31987 .15306 L
.32033 .152 L
.32053 .15145 L
.3207 .15085 L
.32079 .1505 L
.32086 .15018 L
.32095 .14954 L
.321 .14897 L
.32102 .14837 L
.32098 .14773 L
.32091 .14713 L
.32085 .14679 L
.32078 .14648 L
.32058 .1458 L
.32014 .14469 L
.31949 .14352 L
.31872 .14244 L
.31787 .14146 L
.31546 .13935 L
.31218 .13727 L
.30851 .13551 L
.3047 .1341 L
.30062 .1329 L
.29655 .13195 L
.29209 .13114 L
.28699 .13044 L
Mistroke
.28425 .13016 L
.28169 .12994 L
.27885 .12975 L
.27739 .12968 L
.27581 .12961 L
.27442 .12956 L
.27316 .12952 L
.27188 .12949 L
.27066 .12947 L
.26924 .12945 L
.26793 .12945 L
.26649 .12945 L
.26568 .12946 L
.26495 .12947 L
.26367 .12948 L
.26231 .12951 L
.25989 .12957 L
.25729 .12966 L
.25444 .12979 L
.24462 .1304 L
.23365 .13137 L
.22378 .13249 L
.2125 .13411 L
.20188 .13604 L
.19188 .13836 L
.18141 .14148 L
.17271 .14478 L
.16356 .14919 L
.15476 .15473 L
.14764 .1607 L
.14111 .16821 L
.1384 .17235 L
.13592 .17714 L
.13403 .1819 L
.13327 .18431 L
.13267 .18657 L
.13179 .19089 L
.13146 .19321 L
.13131 .19451 L
.1312 .1957 L
.1311 .19697 L
.13103 .19819 L
.13098 .19948 L
.13097 .20021 L
.13096 .20089 L
.13096 .20214 L
.13098 .20327 L
.13103 .20459 L
.13109 .20582 L
.13118 .2072 L
Mistroke
.13131 .20865 L
.13159 .21125 L
.13199 .21406 L
.13254 .21713 L
.13375 .2225 L
.13517 .22753 L
.13848 .2367 L
.14291 .24609 L
.14863 .25561 L
.16122 .27094 L
.16856 .27776 L
.17732 .28466 L
.21328 .30545 L
.25047 .32119 L
.29136 .33582 L
.33106 .34803 L
.37012 .35769 L
.39077 .36148 L
.40245 .36311 L
.41313 .36426 L
.42389 .36513 L
.43387 .3657 L
.43911 .36592 L
.44488 .36612 L
.45093 .36627 L
.45387 .36633 L
.4566 .36638 L
.4592 .36641 L
.46208 .36644 L
.46347 .36646 L
.46498 .36647 L
.46657 .36647 L
.46734 .36648 L
.46806 .36648 L
.46938 .36648 L
.47083 .36649 L
.47224 .36648 L
.47303 .36648 L
.47375 .36648 L
.47503 .36648 L
.47643 .36647 L
.47775 .36646 L
.47897 .36645 L
.48141 .36643 L
.48369 .3664 L
.4889 .36631 L
.49397 .36618 L
.49941 .366 L
.50518 .36575 L
.51053 .36545 L
Mistroke
.52263 .36455 L
.53325 .36345 L
.54452 .36193 L
.56751 .3576 L
.58861 .35212 L
.62733 .33878 L
.66829 .32406 L
.70882 .31237 L
.74782 .30144 L
.76918 .29457 L
.78818 .28759 L
.80736 .2791 L
.82424 .26959 L
.84032 .25774 L
.85593 .24236 L
.86768 .22675 L
.87326 .21723 L
.87753 .20823 L
.88083 .1994 L
.88344 .18988 L
.88443 .18484 L
.88512 .1801 L
.88541 .17735 L
.88561 .17484 L
.88569 .17344 L
.88573 .1727 L
.88576 .17192 L
.88578 .17124 L
.8858 .17049 L
.88582 .16917 L
.88582 .16787 L
.88581 .16668 L
.88577 .16536 L
.88574 .16463 L
.88571 .16396 L
.88565 .16273 L
.88556 .16141 L
.88534 .15901 L
.88507 .15665 L
.88471 .15417 L
.88389 .14975 L
.88264 .14469 L
.88118 .14006 L
.87675 .13001 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1@o`L0
=Ol0097o00<0ool0COl400Ko0`0bo`00TOl00`3oo`1;o`803Ol302oo001mo`801?l200Go0`04o`03
0?oo04[o00<0ool04?l00`3oo`0/o`00O?l0103oo`0;o`030?oo00?o00<0ool0B?l201Co00<0ool0
:ol007co00@0ool02?l500Co0P18o`030?oo01Co00<0ool0:ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`16o`030?oo01Ko00<0ool0:_l007co00@0ool02Ol00`3o0005o`030?oo04Ko00<0ool0
5ol00`3oo`0Yo`00BOlA02?o0P0;o`801Ol00`3oo`15o`030?oo01Wo00<0ool0:?l004?o1P0Ao`@0
<ol00`3oo`15o`030?oo01Wo00<0ool0:?l003ko1@0Ko`80<Ol00`3oo`14o`030?oo01_o00<0ool0
9ol003_o0`0Qo`80<?l00`3oo`14o`030?oo01_o00<0ool09ol003Wo0P0Uo`80;ol00`3oo`14o`03
0?oo01_o00<0ool09ol003Oo0P0Xo`030?oo02go00<0ool0A?l00`3oo`0Lo`030?oo02Ko000fo`03
0?oo02So00<0ool0;Ol00`3oo`14o`030?oo01co00<0ool09_l003Go00<0ool0:?l00`3oo`0^o`03
0?oo04Co00<0ool07?l00`3oo`0Vo`00<ol202[o0P0`o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00
<_l00`3oo`0Yo`030?oo033o00<0ool0@ol00`3oo`0Mo`030?oo02Ko000ao`030?oo02So0P0co`03
0?oo04?o00<0ool07Ol00`3oo`0Vo`00<?l00`3oo`0Xo`030?oo03?o00<0ool0@ol00`3oo`0Mo`03
0?oo02Ko000`o`030?oo02Ko0P0Ro`801?l200?o1005o`030?oo04;o00<0ool07_l00`3oo`0Vo`00
<?l00`3oo`0To`808ol0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`03
0?oo02Co00<0ool08ol0103oo`09o`030?oo00Go0P13o`030?oo01ko00<0ool09_l002oo00<0ool0
8ol00`3oo`0No`@00_l0103oo`0:o`030?oo00Co00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`03
0?oo02;o00<0ool09Ol0103oo`08o`040?oo00Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000^o`03
0?oo02?o00<0ool09_l200[o0P06o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Ro`03
0?oo03_o00<0ool0@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02;o00<0ool0>ol00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08_l00`3oo`0ko`030?oo043o00<0ool07ol00`3oo`0Wo`00
1ol500;o0P04o`806?l00`3oo`08o`D05Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200oo00<0ool04Ol500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`03
0?oo00Wo00<0ool05Ol00`3oo`09o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
3Ol00`3oo`0Do`030?oo00Wo00<0ool06?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
5_l00`3oo`09o`030?oo01Go00<0ool02Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`03
0?oo00[o00<0ool05Ol00`3oo`08o`030?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3o
o`08o`806Ol0103oool400Co00<0ool05_l00`3oo`02o`@00_l0103oo`09o`808ol00`3oo`0Qo`04
0?oo00Wo0P0=o`030?oo01Go00<0ool01ol00`3oo`0Jo`030?oo00So0P03o`002Ol00`3oo`08o`03
0?oo01So00<0ool02Ol00`3oo`0Fo`030?oo00So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool0
2Ol00`3oo`0;o`030?oo01Ko00<0ool01ol00`3oo`0Jo`030?oo00So00<0ool00_l000So0P0:o`<0
6?l00`3oo`08o`806Ol00`3oo`08o`802_l302;o00<0ool08_l200[o0`0:o`030?oo01Ko0P08o`03
0?oo01[o0P0:o`<00_l0033o00<0ool08ol00`3oo`0ho`030?oo03[o00<0ool08?l00`3oo`0/o`00
<?l00`3oo`0To`030?oo03Oo00<0ool0>Ol00`3oo`0Qo`030?oo02co000ao`030?oo02?o00<0ool0
=ol00`3oo`0go`808ol00`3oo`0]o`00<Ol00`3oo`0To`80=ol00`3oo`0eo`809?l00`3oo`0^o`00
<_l00`3oo`0Uo`80=Ol00`3oo`0do`030?oo02?o00<0ool0;ol000coo`0<00Wo000<o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool00P04o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01?l300So00<0ool01Ol00`3o
o`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo02Go00D0oooo000Wo`050?ooo`00;?l00`3oo`0Yo`030?oo00?o00<0ool08ol00`3o
000Zo`030?oo00Oo000do`030?oo02[o0P0^o`030?oo02go0P0Vo`80=Ol003Go00<0ool0:ol202co
00<0ool0:ol202Oo00<0ool0=Ol003Go00<0ool0;Ol302Wo00<0ool0:?l302So00<0ool0=_l003Ko
00<0ool0;ol202Oo00<0ool09?l402Wo0P0io`00=ol2037o0`0To`030?oo027o0`0[o`80>ol003Wo
0P0bo`<08Ol00`3oo`0Mo`@0;?l203go000ko`030?oo03;o100Mo`030?oo01So1@0^o`80?ol003co
0P0fo`@06Ol00`3oo`0Bo`H0<?l3047o000no`030?oo03Oo1`0Bo`030?oo00go1@0do`80A?l003oo
0P0nob80=_l304Ko0011o`030?oo04go00<0ool0@?l304Wo0012o`80COl00`3oo`0mo`<0C?l004Co
0P1;o`030?oo03Wo101?o`00A_l203Go0P04o`800ol400Go00<0ool0=_l305?o0018o`<0<Ol0103o
o`08o`030?oo00Ko00<0ool0<ol305Ko001;o`80;ol0103oo`09o`030?oo00Go0P0`o`@0FOl004go
0`0/o`040?oo00[o00<0ool01?l00`3oo`0[o`@0GOl0053o0`0Yo`040?oo00So00@0ool01Ol00`3o
o`0Xo`<0HOl005?o0P0Xo`802_l200Ko00<0ool09Ol306Co001Eo`<0>Ol00`3oo`0So`80Iol005So
0`0fo`030?oo023o0`1Yo`00Fol403;o00<0ool07Ol306co001Oo`@0;_l00`3oo`0Io`@0Kol006?o
0`0[o`030?oo01Ko0`1co`00I_l302So00<0ool04ol307Ko001Yo`@09?l00`3oo`0@o`<0NOl006go
100Po`030?oo00_o1@1lo`00LOl:01Ko00@0oooo2P21o`00NolJ08_o002Ao`030?oo08co001mo`80
1?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D0
1?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0
S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<0
1_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80
SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`80
2_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87634 .13456 m
.87165 .12683 L
.8653 .11935 L
.86163 .11605 L
.85737 .11291 L
.85309 .11036 L
.84891 .10836 L
.84371 .10643 L
.84087 .10562 L
.8378 .1049 L
.83509 .1044 L
.83252 .10404 L
.83104 .10388 L
.82967 .10376 L
.82892 .10371 L
.82809 .10366 L
.82737 .10363 L
.82659 .1036 L
.82583 .10358 L
.82514 .10357 L
.82377 .10357 L
.82253 .10359 L
.82122 .10364 L
.82054 .10367 L
.81981 .10371 L
.81849 .10381 L
.81551 .10412 L
.81277 .10452 L
.81021 .10499 L
.80548 .10612 L
.80093 .10755 L
.79613 .1095 L
.79172 .11174 L
.78731 .11453 L
.78357 .11748 L
.78046 .12046 L
.77529 .12706 L
.77305 .13094 L
.77101 .13535 L
.76817 .14388 L
.76686 .14941 L
.76583 .15478 L
.76416 .16508 L
.76244 .17528 L
.76129 .18078 L
.76002 .18582 L
.75717 .19441 L
.75302 .20328 L
.74731 .2121 L
.74083 .2198 L
Mistroke
.72666 .23275 L
.69382 .25483 L
.67576 .26371 L
.6553 .27111 L
.63337 .27695 L
.61237 .28147 L
.57182 .28826 L
.55039 .29083 L
.53939 .2919 L
.52761 .29287 L
.5219 .29325 L
.51651 .29356 L
.51126 .29381 L
.50896 .29389 L
.50643 .29398 L
.50422 .29404 L
.50185 .29409 L
.50055 .29411 L
.49935 .29413 L
.4982 .29414 L
.497 .29415 L
.49578 .29415 L
.49447 .29416 L
.49323 .29415 L
.49211 .29415 L
.49079 .29414 L
.49005 .29413 L
.48936 .29412 L
.48677 .29407 L
.48415 .29401 L
.48175 .29394 L
.47637 .29372 L
.47138 .29346 L
.46665 .29316 L
.44466 .29128 L
.42384 .28896 L
.40493 .28624 L
.38603 .28245 L
.36539 .27661 L
.32698 .26147 L
.31049 .25336 L
.30146 .24819 L
.29337 .24283 L
.28526 .23637 L
.27877 .22988 L
.27323 .22241 L
.27207 .22037 L
.27098 .21814 L
.27018 .21617 L
.26949 .21405 L
Mistroke
.26922 .21303 L
.269 .21207 L
.26882 .21113 L
.26869 .21026 L
.2686 .20944 L
.26853 .20857 L
.26849 .2076 L
.26849 .20673 L
.26851 .20623 L
.26854 .2057 L
.26862 .20476 L
.26868 .20423 L
.26875 .20375 L
.26895 .20264 L
.26941 .20081 L
.27007 .19887 L
.27094 .19688 L
.27209 .19476 L
.27471 .191 L
.28119 .18431 L
.29697 .17258 L
.30517 .16677 L
.30893 .16379 L
.31231 .16068 L
.3147 .15796 L
.31571 .15653 L
.31658 .15502 L
.31719 .15364 L
.31741 .15299 L
.31757 .1524 L
.3177 .15181 L
.3178 .15117 L
.31785 .15061 L
.31786 .15002 L
.31783 .14943 L
.31777 .1489 L
.31767 .14829 L
.31752 .14772 L
.31717 .14671 L
.31665 .14564 L
.31592 .1445 L
.31495 .14332 L
.31277 .14131 L
.30968 .13924 L
.30629 .13753 L
.30206 .13589 L
.2971 .13444 L
.29244 .1334 L
.28787 .13262 L
.28267 .13197 L
Mistroke
.27968 .13169 L
.27693 .13149 L
.27448 .13135 L
.27177 .13123 L
.27039 .13119 L
.26964 .13117 L
.26891 .13116 L
.26752 .13114 L
.26623 .13113 L
.26504 .13112 L
.26376 .13112 L
.26256 .13113 L
.26147 .13115 L
.26018 .13117 L
.25878 .1312 L
.25626 .13128 L
.25354 .13138 L
.25056 .13152 L
.24458 .13189 L
.23396 .13278 L
.22322 .13397 L
.21352 .13531 L
.20241 .13727 L
.19196 .13963 L
.1822 .14247 L
.17211 .14624 L
.1639 .15016 L
.15548 .15532 L
.14767 .16166 L
.14162 .16833 L
.13894 .17214 L
.13642 .17654 L
.13426 .18137 L
.1334 .18379 L
.13271 .18607 L
.13213 .18831 L
.13161 .19082 L
.13119 .19338 L
.13101 .1948 L
.13087 .19612 L
.13077 .19736 L
.1307 .19855 L
.13064 .19981 L
.13062 .20053 L
.13061 .20119 L
.13061 .20241 L
.13062 .20352 L
.13066 .2048 L
.13072 .20601 L
.13079 .2071 L
Mistroke
.13088 .20829 L
.13113 .21071 L
.13177 .21504 L
.1328 .22008 L
.13403 .22474 L
.13761 .23497 L
.14192 .24419 L
.15206 .25991 L
.1583 .26722 L
.16558 .27434 L
.18018 .28563 L
.21412 .3042 L
.25254 .31992 L
.29067 .33329 L
.33154 .34568 L
.35219 .35106 L
.37177 .35541 L
.39101 .3588 L
.40189 .36028 L
.41182 .36136 L
.42246 .36224 L
.43232 .36284 L
.4375 .36308 L
.4432 .3633 L
.44917 .36347 L
.45477 .3636 L
.4574 .36364 L
.46019 .36368 L
.46326 .36371 L
.46463 .36372 L
.46611 .36374 L
.46749 .36374 L
.46875 .36375 L
.46996 .36375 L
.47123 .36375 L
.47263 .36375 L
.47394 .36375 L
.47537 .36375 L
.47617 .36374 L
.4769 .36374 L
.4795 .36372 L
.4819 .36369 L
.48453 .36366 L
.48732 .36361 L
.492 .3635 L
.49716 .36334 L
.50283 .36312 L
.50821 .36285 L
.51829 .36217 L
.52911 .36118 L
Mistroke
.54084 .35973 L
.56183 .35613 L
.5837 .35087 L
.62529 .33709 L
.66495 .32309 L
.70419 .31198 L
.74608 .30097 L
.76666 .29485 L
.78546 .28847 L
.80406 .28094 L
.82091 .27236 L
.83829 .26063 L
.85258 .24766 L
.86558 .23166 L
.87492 .21553 L
.87864 .20674 L
.88138 .19829 L
.88261 .19338 L
.88352 .18882 L
.88426 .18375 L
.88453 .18118 L
.88466 .17972 L
.88475 .17836 L
.88482 .17716 L
.88488 .17584 L
.8849 .17516 L
.88492 .17451 L
.88493 .17384 L
.88494 .17311 L
.88494 .17247 L
.88494 .17177 L
.88492 .1705 L
.88489 .16926 L
.88485 .16811 L
.88478 .16688 L
.88471 .16573 L
.88453 .16357 L
.88427 .16124 L
.88393 .15873 L
.88296 .15341 L
.88157 .14787 L
.88008 .14322 L
.87811 .13826 L
.87634 .13456 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1?o`P0=Ol0097o00<0ool0COl200So0`0bo`00OOl200Co0P05o`<01?l00`3oo`1:o`<0
3Ol2033o001lo`040?oo00_o00<0ool00ol00`3oo`19o`030?oo013o0P0^o`00O?l0103oo`08o`D0
1?l204Wo00<0ool04ol00`3oo`0[o`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Oo00<0ool0
5?l00`3oo`0[o`00O?l0103oo`09o`030?l000Go00<0ool0A_l00`3oo`0Fo`030?oo02[o001>o`P0
9ol200_o0P05o`030?oo04Go00<0ool06?l00`3oo`0Yo`00@ol;00So1P0eo`030?oo04Go00<0ool0
6Ol00`3oo`0Xo`00@Ol201Wo0P0co`030?oo04Go00<0ool06Ol00`3oo`0Xo`00>ol601go0P0ao`03
0?oo04Co00<0ool06_l00`3oo`0Xo`00>Ol202Co0P0`o`030?oo04Co00<0ool06ol00`3oo`0Wo`00
=ol202Oo00<0ool0;_l00`3oo`13o`030?oo01co00<0ool09ol003Ko00<0ool09ol00`3oo`0^o`03
0?oo04?o00<0ool07?l00`3oo`0Wo`00=Ol00`3oo`0Wo`80<?l00`3oo`13o`030?oo01co00<0ool0
9ol003?o0P0Zo`030?oo02oo00<0ool0@ol00`3oo`0Mo`030?oo02Ko000bo`030?oo02Wo00<0ool0
<?l00`3oo`13o`030?oo01go00<0ool09_l0037o00<0ool0:?l203?o00<0ool0@ol00`3oo`0Mo`03
0?oo02Ko000`o`030?oo02Oo0P0eo`030?oo04;o00<0ool07_l00`3oo`0Vo`00<?l00`3oo`0Vo`03
0?oo027o0P04o`800ol400Go00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`030?oo02Ko00<0ool0
8Ol0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`030?oo02Go00<0ool0
8_l0103oo`09o`030?oo00Go0P13o`030?oo01ko00<0ool09_l002oo00<0ool09?l00`3oo`0Mo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@_l00`3oo`0No`030?oo02Ko000_o`030?oo02?o00<0ool0
9?l0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000^o`030?oo02?o00<0ool0
9_l200[o0P06o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Ro`80??l00`3oo`11o`03
0?oo01ko00<0ool09ol002ko00<0ool08_l00`3oo`0ko`030?oo043o00<0ool07ol00`3oo`0Wo`00
;_l00`3oo`0Ro`030?oo03_o00<0ool0@?l00`3oo`0No`030?oo02So0007o`D00_l200Co0P0Ho`03
0?oo00So1@0Eo`030?oo00[o0P04o`801?l202?o00<0ool08_l200Co0P04o`803ol00`3oo`0Ao`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Eo`03
0?oo00Wo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0=o`030?oo01Co00<0ool0
2?l00`3oo`0Io`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool0
5Ol00`3oo`09o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02_l00`3oo`0Eo`03
0?oo00So00<0ool06Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Io`040?ooo`@0
1?l00`3oo`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200go00<0ool0
5Ol00`3oo`08o`030?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool06?l00`3oo`09o`03
0?oo01Ko00<0ool02?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00_o00<0ool0
5_l00`3oo`07o`030?oo01[o00<0ool02?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Io`03
0?oo00So0P0:o`<08_l00`3oo`0Ro`802_l300Wo0P0Ho`802Ol00`3oo`0Io`802_l300;o000`o`03
0?oo02?o00<0ool0>?l00`3oo`0io`030?oo02;o00<0ool0:ol0033o00<0ool09?l00`3oo`0go`03
0?oo03Oo0P0To`030?oo02co000ao`030?oo02Co0P0go`030?oo03Ko00<0ool08ol00`3oo`0]o`00
<Ol00`3oo`0Vo`030?oo03Co00<0ool0=?l202Go00<0ool0;_l003;o00<0ool09_l00`3oo`0co`03
0?oo03?o00<0ool09?l00`3oo`0_o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`0200Co
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`04o`030?l000So00<0ool01Ol00`3oo`06o`030?oo00Ko
00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co
00<0ool00_l00`3oo`0Uo`800_l00`3oo`0Zo`030?oo02Wo00<0ool00_l202Go00<0o`00:_l00`3o
o`07o`00=?l00`3oo`0[o`80;Ol00`3oo`0/o`80:?l00`3oo`0co`00=Ol00`3oo`0/o`<0:_l00`3o
o`0Zo`80:?l203Ko000fo`030?oo02ko0P0Xo`030?oo02So0P0Yo`030?oo03Ko000go`80<?l302Go
00<0ool09?l402Wo0P0io`00>Ol00`3oo`0`o`<08_l00`3oo`0No`H0:ol203_o000jo`030?oo03;o
0P0Po`030?oo01So1P0^o`<0?Ol003_o00<0ool0<ol401co00<0ool04ol503;o0P10o`00??l203Oo
2`0Ao`030?oo00ko1@0do`<0@_l003ko0P10o`T02?l00`3oo`04o`X0=ol204Go0010o`80Aol?03ko
0`17o`00@_l204go00<0ool0@?l204[o0014o`80Bol00`3oo`0lo`@0C?l004Ko0P19o`030?oo03Ko
1P1@o`00B?l203?o0P04o`800ol400Go00<0ool0<_l405Ko001:o`80<?l0103oo`08o`030?oo00Ko
00<0ool0;_l405[o001<o`80;_l0103oo`09o`030?oo00Go0P0[o`@0G_l004ko0`0[o`040?oo00[o
00<0ool01?l00`3oo`0Wo`<0H_l0057o0`0Xo`040?oo00So00@0ool01Ol00`3oo`0Uo`80IOl005Co
100Uo`802_l200Ko00<0ool08_l306Oo001Ho`<0=_l00`3oo`0Po`80J_l005_o100bo`030?oo01go
0`1/o`00Gol402ko00<0ool06_l306oo001So`<0:ol00`3oo`0Ho`80L_l006Ko0P0Yo`030?oo01Go
0`1do`00J?l402Go00<0ool04?l507Oo001/o`D08?l00`3oo`0:o`H0O?l0077o1`0Io`030?oo00Ko
1022o`00N?lR08Ko002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87593 .13911 m
.87145 .1313 L
.86537 .12368 L
.86184 .1203 L
.85775 .11705 L
.85362 .11439 L
.84957 .11228 L
.84452 .11022 L
.84181 .10934 L
.83876 .10852 L
.83625 .10798 L
.8335 .1075 L
.83197 .10729 L
.83056 .10713 L
.82922 .107 L
.82778 .1069 L
.82706 .10685 L
.82638 .10682 L
.8251 .10678 L
.8239 .10676 L
.82261 .10677 L
.82192 .10678 L
.82117 .1068 L
.82039 .10682 L
.81966 .10686 L
.81835 .10694 L
.81694 .10705 L
.81419 .10735 L
.8116 .10773 L
.80681 .10869 L
.80222 .10995 L
.79736 .11168 L
.79256 .11387 L
.78836 .11628 L
.78168 .1214 L
.77845 .12472 L
.77581 .12805 L
.77342 .13175 L
.77123 .13599 L
.76813 .14425 L
.76668 .14964 L
.76554 .1549 L
.76373 .16502 L
.76193 .1751 L
.76077 .18054 L
.7595 .18554 L
.75667 .19404 L
.75256 .20282 L
.74691 .21152 L
.74046 .21909 L
Mistroke
.72631 .23177 L
.69352 .2533 L
.67556 .26192 L
.65524 .26912 L
.63347 .27485 L
.61265 .27932 L
.5936 .2828 L
.57248 .28596 L
.5512 .28842 L
.53937 .28953 L
.52859 .29038 L
.52344 .29072 L
.51785 .29105 L
.51304 .29128 L
.50774 .29148 L
.50485 .29156 L
.50214 .29162 L
.50064 .29164 L
.49928 .29166 L
.49856 .29167 L
.49777 .29167 L
.49696 .29168 L
.4962 .29168 L
.49481 .29168 L
.4935 .29168 L
.49232 .29167 L
.49105 .29166 L
.48965 .29164 L
.48818 .29162 L
.48555 .29156 L
.48286 .29148 L
.48031 .29139 L
.47552 .29117 L
.46476 .2905 L
.44542 .28881 L
.42478 .28653 L
.40606 .28389 L
.38735 .28021 L
.36684 .27458 L
.32854 .26013 L
.31207 .25239 L
.30304 .24741 L
.29492 .24223 L
.28673 .23597 L
.28011 .22967 L
.2744 .22242 L
.27206 .21828 L
.27115 .21623 L
.27046 .2143 L
.26998 .21262 L
Mistroke
.2696 .21082 L
.26952 .21033 L
.26945 .20979 L
.26934 .20882 L
.26929 .20787 L
.26927 .207 L
.26928 .20651 L
.26929 .20599 L
.26936 .20505 L
.26947 .20413 L
.2696 .2033 L
.26978 .20239 L
.27002 .2014 L
.27066 .19942 L
.27139 .19769 L
.27217 .19613 L
.27435 .19269 L
.27959 .18672 L
.29551 .17394 L
.30249 .16865 L
.30579 .16589 L
.3089 .16295 L
.31134 .16017 L
.31221 .15896 L
.31304 .15765 L
.31364 .15651 L
.31414 .15529 L
.31436 .15463 L
.31452 .15403 L
.31464 .15343 L
.31473 .15287 L
.31478 .15228 L
.31479 .15163 L
.31476 .15097 L
.31468 .15035 L
.31457 .14978 L
.31444 .14927 L
.31403 .14813 L
.31373 .14751 L
.31342 .14694 L
.31255 .14568 L
.31047 .14352 L
.30773 .14149 L
.30412 .13953 L
.30017 .1379 L
.29616 .13661 L
.29226 .13562 L
.28775 .13471 L
.28282 .13397 L
.27809 .13345 L
Mistroke
.27536 .13322 L
.27282 .13306 L
.27 .13292 L
.26858 .13287 L
.26701 .13283 L
.26563 .1328 L
.26437 .13279 L
.26367 .13278 L
.26291 .13278 L
.26152 .13278 L
.26023 .13278 L
.25885 .1328 L
.25755 .13282 L
.25637 .13285 L
.25381 .13292 L
.25146 .13301 L
.24615 .13328 L
.2415 .13359 L
.23662 .13397 L
.2267 .13495 L
.21617 .13627 L
.19736 .1396 L
.18707 .14217 L
.17612 .14575 L
.1662 .15003 L
.15762 .15485 L
.14989 .16052 L
.14357 .16665 L
.13791 .17423 L
.13537 .17885 L
.13347 .1833 L
.13209 .18758 L
.13148 .19003 L
.13103 .19229 L
.13069 .19443 L
.13054 .19556 L
.13041 .19677 L
.13031 .19783 L
.13023 .199 L
.13017 .20007 L
.13014 .20108 L
.13012 .20233 L
.13012 .20304 L
.13012 .2037 L
.13015 .20492 L
.13021 .20621 L
.13029 .20749 L
.13038 .20869 L
.13062 .21094 L
.13099 .21362 L
Mistroke
.13141 .21607 L
.13265 .22159 L
.13567 .23108 L
.13935 .23962 L
.14419 .24849 L
.15021 .25733 L
.15667 .26507 L
.16444 .27274 L
.18032 .28484 L
.19767 .29485 L
.21709 .30396 L
.25297 .31807 L
.29225 .33152 L
.33037 .3429 L
.3499 .34797 L
.37167 .35278 L
.38191 .35466 L
.39293 .35638 L
.40332 .35771 L
.41294 .35869 L
.42376 .35955 L
.42936 .35989 L
.43553 .36021 L
.44597 .36061 L
.4517 .36077 L
.45448 .36083 L
.45706 .36088 L
.45961 .36093 L
.46243 .36096 L
.46527 .36099 L
.46683 .361 L
.46829 .36101 L
.46959 .36102 L
.47097 .36102 L
.47214 .36102 L
.47343 .36102 L
.47482 .36102 L
.47559 .36101 L
.4763 .36101 L
.47768 .361 L
.47898 .36099 L
.48141 .36097 L
.48398 .36094 L
.4886 .36085 L
.49357 .36073 L
.49904 .36054 L
.50466 .36029 L
.50999 .36 L
.51996 .35929 L
.53065 .35826 L
Mistroke
.54226 .3568 L
.56299 .35319 L
.58462 .34796 L
.62583 .33444 L
.66519 .32098 L
.70418 .31039 L
.74583 .30012 L
.76632 .29443 L
.78507 .28849 L
.80366 .28145 L
.82052 .27337 L
.83794 .2622 L
.85225 .24965 L
.85889 .2423 L
.86522 .23399 L
.87446 .21808 L
.87802 .20963 L
.87957 .20512 L
.88096 .20031 L
.88198 .19603 L
.88286 .1913 L
.88318 .18921 L
.88346 .18696 L
.88366 .18502 L
.88384 .18287 L
.88392 .18157 L
.88398 .18035 L
.88402 .17905 L
.88404 .17832 L
.88405 .17766 L
.88406 .17647 L
.88406 .17518 L
.88403 .17395 L
.884 .17281 L
.88397 .17212 L
.88393 .17137 L
.88385 .17001 L
.88365 .16747 L
.88331 .1645 L
.88292 .16178 L
.88206 .15714 L
.88082 .1521 L
.87919 .14692 L
.87735 .14221 L
.87593 .13911 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1>o`X0=?l007go0P04o`801Ol300Co00<0ool0C?l200Wo100ao`00
O?l0103oo`0;o`030?oo00?o00<0ool0BOl300oo00<0ool0;_l007co00@0ool02?l500Co0P19o`03
0?oo017o0P0^o`00M_l400;o00@0ool02?l0103oo`05o`030?oo04Oo00<0ool05?l00`3oo`0[o`00
O?l0103oo`09o`030?l000Go00<0ool0Aol00`3oo`0Do`030?oo02_o001mo`802ol200Go00<0ool0
A_l00`3oo`0Fo`030?oo02[o0017oa80>?l00`3oo`15o`030?oo01Oo00<0ool0:_l004?o100Bo`<0
=Ol00`3oo`15o`030?oo01So00<0ool0:Ol003co1`0Io`80<ol00`3oo`14o`030?oo01Wo0P0Zo`00
>_l202;o0P0ao`030?oo04Co00<0ool06_l00`3oo`0Xo`00>?l202Go00<0ool0;ol00`3oo`13o`03
0?oo01_o00<0ool0:?l003Ko0P0Wo`030?oo02oo00<0ool0@ol00`3oo`0Lo`030?oo02Oo000eo`03
0?oo02Oo00<0ool0;ol00`3oo`13o`030?oo01co00<0ool09ol003?o0P0Yo`80<Ol00`3oo`13o`03
0?oo01co00<0ool09ol003;o00<0ool0:Ol00`3oo`0`o`030?oo04?o00<0ool07?l00`3oo`0Wo`00
<_l00`3oo`0Wo`80<ol00`3oo`13o`030?oo01go00<0ool09_l0037o00<0ool09ol00`3oo`0co`03
0?oo04;o00<0ool07_l00`3oo`0Vo`00<?l00`3oo`0Wo`030?oo023o0P04o`800ol400Go00<0ool0
@_l00`3oo`0No`030?oo02Ko000`o`030?oo02Go0P0Ro`040?oo00So00<0ool01_l00`3oo`12o`03
0?oo01ko00<0ool09_l002oo00<0ool09Ol00`3oo`0Ro`040?oo00Wo00<0ool01Ol204?o00<0ool0
7_l00`3oo`0Vo`00;ol00`3oo`0To`030?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`12o`03
0?oo01ko00<0ool09_l002ko00<0ool09?l00`3oo`0To`040?oo00So00@0ool01Ol00`3oo`11o`03
0?oo01ko00<0ool09ol002ko00<0ool08ol00`3oo`0Vo`802_l200Ko00<0ool0@Ol00`3oo`0No`03
0?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`10o`030?oo01oo00<0ool09ol002ko00<0ool0
8_l00`3oo`0ko`030?oo043o00<0ool07ol00`3oo`0Wo`00;_l00`3oo`0Ro`030?oo03_o00<0ool0
?ol00`3oo`0Oo`030?oo02So0007o`D00_l200Co0P0Ho`030?oo00So1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803ol00`3oo`0Ao`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Eo`030?oo00Wo00@0ool02?l0103oo`0Ro`03
0?oo027o00@0ool02?l0103oo`0=o`030?oo01Co00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool0
0_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05Ol200[o00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02?l00`3oo`0Io`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201Wo00@0oooo1004o`030?oo01Ko00<0ool00_l400;o00@0ool0
2Ol202?o00<0ool08Ol0103oo`09o`803?l00`3oo`0Fo`030?oo00So00<0ool06Ol00`3oo`08o`80
0ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool05_l00`3oo`08o`040?oo00Wo00<0ool0
8_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00So00<0ool06Ol00`3oo`08o`03
0?oo00;o0008o`802_l301Wo00<0ool01ol201Wo00<0ool02?l200[o0`0Ro`030?oo02;o0P0:o`<0
2Ol00`3oo`0Go`802Ol00`3oo`0Io`802_l300;o000`o`030?oo02?o0P0io`030?oo03Wo00<0ool0
8ol00`3oo`0Zo`00<Ol00`3oo`0To`030?oo03Ko00<0ool0=ol202Go00<0ool0:ol0037o00<0ool0
9Ol00`3oo`0eo`030?oo03Ko00<0ool09?l00`3oo`0/o`00<Ol00`3oo`0Vo`80=Ol00`3oo`0do`80
9_l00`3oo`0]o`00<_l00`3oo`0Wo`030?oo03;o00<0ool0<ol00`3oo`0Uo`030?oo02ko000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`040?ooo`800ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o
0P000ol0o`07o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00801?l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co00<0ool00_l00`3oo`0Vo`8000?o0?l0
:ol00`3oo`0Yo`040?ooo`809_l00`3o000Zo`030?oo00Oo000do`030?oo02co0P0/o`030?oo02[o
0`0Yo`030?oo03?o000eo`030?oo02go0P0Zo`030?oo02So0P0Zo`80=_l003Ko00<0ool0;_l302Oo
00<0ool09Ol302_o00<0ool0=_l003Oo00<0ool0<?l202Go00<0ool08_l302co0P0io`00>?l00`3o
o`0ao`D08?l00`3oo`0No`@0;Ol203_o000io`80=_l501_o00<0ool06?l602ko0`0mo`00>ol00`3o
o`0ho`D05_l00`3oo`0Co`D0<_l2043o000lo`80?Ol:00co00<0ool02Ol:03Co0`12o`00?_l204Go
600lo`80AOl0043o0P1?o`030?oo047o1017o`00@_l204go00<0ool0?Ol404_o0014o`<0B_l00`3o
o`0io`@0Col004Oo0P18o`030?oo03Go101Co`00BOl3037o0P04o`800ol400Go00<0ool0<_l305Oo
001<o`80;_l0103oo`08o`030?oo00Ko00<0ool0;_l405[o001>o`<0:ol0103oo`09o`030?oo00Go
0P0[o`@0G_l0057o0P0Yo`040?oo00[o00<0ool01?l00`3oo`0Wo`<0H_l005?o0`0Vo`040?oo00So
00@0ool01Ol00`3oo`0So`@0IOl005Ko0P0Uo`802_l200Ko00<0ool08?l306Wo001Ho`<0=_l00`3o
o`0Mo`<0K?l005_o0`0co`030?oo01[o0`1_o`00G_l402oo00<0ool06?l207;o001Ro`@0:ol00`3o
o`0Eo`<0M?l006Ko1P0Uo`030?oo013o1@1go`00K?l4027o00<0ool02ol507co001`o`H06ol00`3o
o`07o`@0POl007Ko4007o`h0QOl008Ko1`04o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103o
o`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go
00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103o
o`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co
001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87552 .14366 m
.87125 .13577 L
.86544 .12802 L
.86206 .12454 L
.85812 .12119 L
.85023 .11621 L
.84533 .114 L
.84265 .11303 L
.83973 .11215 L
.83713 .11149 L
.83421 .1109 L
.83152 .11049 L
.82897 .11019 L
.82769 .11009 L
.82701 .11004 L
.82629 .10999 L
.82496 .10993 L
.82373 .1099 L
.82243 .10988 L
.8212 .10989 L
.8201 .10992 L
.8189 .10997 L
.81759 .11004 L
.81615 .11015 L
.81326 .11045 L
.81059 .11083 L
.80813 .11127 L
.80311 .11245 L
.79784 .11413 L
.79313 .1161 L
.78903 .11823 L
.78208 .12311 L
.77902 .12598 L
.776 .12945 L
.77342 .13316 L
.77108 .13736 L
.76778 .14554 L
.76635 .15042 L
.76522 .15512 L
.76307 .16615 L
.761 .17692 L
.75852 .18682 L
.75552 .19528 L
.75126 .20385 L
.74602 .21157 L
.73916 .21934 L
.72281 .23315 L
.70488 .2451 L
.68794 .25447 L
.66959 .26238 L
Mistroke
.65115 .26829 L
.61225 .2773 L
.59205 .28093 L
.57024 .28402 L
.54751 .28643 L
.53705 .28733 L
.52603 .28813 L
.51588 .28871 L
.51014 .28895 L
.5074 .28904 L
.5048 .2891 L
.50245 .28915 L
.50112 .28917 L
.4999 .28919 L
.49887 .2892 L
.49773 .28921 L
.49655 .28921 L
.49544 .28921 L
.49421 .28921 L
.49292 .2892 L
.49181 .28919 L
.49061 .28918 L
.48931 .28915 L
.48792 .28913 L
.48541 .28907 L
.48269 .28898 L
.4798 .28887 L
.47464 .28862 L
.46296 .28784 L
.4419 .2859 L
.4216 .28359 L
.40345 .28092 L
.38409 .27691 L
.36345 .27106 L
.34356 .26413 L
.32595 .25709 L
.30855 .24884 L
.29303 .23932 L
.28615 .23393 L
.27961 .22759 L
.27662 .22397 L
.27397 .22005 L
.27219 .21665 L
.2714 .2147 L
.27082 .21288 L
.27058 .21195 L
.27036 .21096 L
.27022 .21012 L
.2701 .20919 L
.27004 .20867 L
Mistroke
.27001 .20819 L
.26998 .20765 L
.26997 .20713 L
.26999 .20621 L
.27001 .20569 L
.27005 .20521 L
.27015 .20427 L
.27029 .20339 L
.27065 .20175 L
.27119 .2 L
.27195 .19813 L
.27372 .19486 L
.27613 .19149 L
.28162 .18565 L
.29651 .17335 L
.30258 .16821 L
.30781 .16275 L
.30888 .16131 L
.30985 .1598 L
.31059 .15842 L
.31112 .15713 L
.31149 .15595 L
.31163 .1553 L
.31173 .1547 L
.31179 .15409 L
.31182 .15342 L
.31181 .15309 L
.31179 .15274 L
.31173 .15211 L
.31167 .15176 L
.3116 .15139 L
.31143 .15074 L
.31119 .15006 L
.31086 .14931 L
.31012 .14801 L
.30923 .14684 L
.30712 .14477 L
.30423 .14274 L
.30068 .14088 L
.29693 .13938 L
.29237 .13796 L
.28711 .13674 L
.28179 .13583 L
.27911 .13547 L
.27661 .13519 L
.27184 .13479 L
.26931 .13464 L
.26662 .13453 L
.26532 .13448 L
.26391 .13445 L
Mistroke
.26257 .13443 L
.26134 .13441 L
.26023 .13441 L
.259 .1344 L
.25772 .13441 L
.25653 .13442 L
.25513 .13445 L
.25381 .13448 L
.25133 .13455 L
.24866 .13465 L
.24576 .13479 L
.2358 .13547 L
.22664 .13634 L
.2158 .13766 L
.20565 .13923 L
.19659 .14099 L
.18688 .14337 L
.17752 .1463 L
.1676 .15032 L
.15789 .15553 L
.14971 .1614 L
.14246 .16843 L
.13942 .17226 L
.13693 .176 L
.13502 .17947 L
.13322 .18346 L
.13177 .18762 L
.13117 .18978 L
.13062 .19218 L
.13019 .19455 L
.12988 .19679 L
.12975 .19804 L
.12966 .19918 L
.12961 .19981 L
.12957 .2005 L
.12952 .20175 L
.1295 .20283 L
.1295 .20401 L
.12952 .20526 L
.12956 .20642 L
.12962 .20752 L
.12969 .20852 L
.1299 .21077 L
.13024 .21337 L
.13064 .21579 L
.13185 .22124 L
.13323 .226 L
.135 .23099 L
.13928 .2405 L
.14493 .25015 L
Mistroke
.15054 .25789 L
.15739 .26569 L
.17214 .27854 L
.18986 .28974 L
.22716 .30654 L
.26413 .32002 L
.30442 .33302 L
.34338 .34387 L
.3615 .34813 L
.3819 .35203 L
.40234 .35487 L
.41361 .35599 L
.42423 .3568 L
.43455 .35738 L
.44019 .35763 L
.44545 .35783 L
.45035 .35797 L
.45481 .35808 L
.45978 .35818 L
.46251 .35822 L
.46509 .35825 L
.46633 .35826 L
.46765 .35827 L
.4689 .35828 L
.47004 .35828 L
.47138 .35829 L
.47214 .35829 L
.47284 .35829 L
.47413 .35829 L
.4755 .35828 L
.4768 .35828 L
.4782 .35827 L
.47952 .35826 L
.48074 .35825 L
.48356 .35822 L
.48653 .35817 L
.49142 .35806 L
.49598 .35792 L
.50097 .35773 L
.5063 .35747 L
.5172 .35677 L
.52897 .3557 L
.54001 .35438 L
.55014 .35286 L
.56953 .34908 L
.59044 .34367 L
.62984 .3306 L
.66773 .31821 L
.70921 .30766 L
.74929 .2984 L
Mistroke
.78701 .28783 L
.80567 .28106 L
.82464 .27205 L
.83347 .26669 L
.84254 .26008 L
.84989 .25368 L
.85731 .24598 L
.86908 .22985 L
.8741 .22043 L
.87782 .21137 L
.8793 .20682 L
.88064 .20182 L
.88165 .19704 L
.88205 .19474 L
.88237 .19256 L
.88262 .19045 L
.88282 .1885 L
.88299 .18633 L
.88306 .18518 L
.88311 .18397 L
.88315 .18287 L
.88318 .18168 L
.88319 .18057 L
.88319 .17955 L
.88317 .17835 L
.88315 .17767 L
.88313 .17704 L
.88309 .17589 L
.88302 .17467 L
.88288 .17257 L
.88269 .17057 L
.88223 .16682 L
.88155 .16278 L
.8806 .15839 L
.87832 .15062 L
.87552 .14366 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1=o`/0=?l007co00@0ool0
2ol00`3oo`03o`030?oo04[o0`0;o`<0<Ol007co00@0ool02?l500Co0P1:o`030?oo00oo00<0ool0
;_l007Ko1002o`040?oo00So00@0ool01Ol00`3oo`18o`030?oo017o00<0ool0;Ol007co00@0ool0
2Ol00`3o0005o`030?oo04Oo00<0ool04ol00`3oo`0/o`00OOl200_o0P05o`030?oo04Ko00<0ool0
5Ol00`3oo`0[o`00C?l:03_o00<0ool0AOl00`3oo`0Fo`030?oo02_o0014o`P02_l403Oo00<0ool0
AOl00`3oo`0Go`030?oo02[o000oo`D05_l303Co00<0ool0A?l00`3oo`0Ho`030?oo02[o000jo`D0
7_l00`3oo`0ao`030?oo04Co00<0ool06Ol00`3oo`0Yo`00>?l202Co00<0ool0<?l00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Oo00<0ool09?l00`3oo`0`o`030?oo04?o00<0ool06ol00`3oo`0Xo`00
=Ol202Oo00<0ool0<?l00`3oo`13o`030?oo01_o00<0ool0:?l003?o0P0Yo`030?oo033o00<0ool0
@ol00`3oo`0Lo`030?oo02Oo000bo`030?oo02So00<0ool0<Ol00`3oo`12o`030?oo01go00<0ool0
9ol0037o00<0ool0:?l00`3oo`0bo`030?oo04;o00<0ool07Ol00`3oo`0Wo`00<Ol00`3oo`0Wo`03
0?oo03?o00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`030?oo02Oo00<0ool08?l200Co0P03o`@0
1Ol00`3oo`12o`030?oo01go00<0ool09ol002oo00<0ool09_l202;o00@0ool02?l00`3oo`06o`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Uo`030?oo02;o00@0ool02Ol00`3oo`05o`80
@_l00`3oo`0No`030?oo02Oo000_o`030?oo02Co00<0ool07Ol400;o00@0ool02_l00`3oo`04o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`030?oo02Co00@0ool02?l0103oo`05o`03
0?oo043o00<0ool07ol00`3oo`0Wo`00;_l00`3oo`0So`030?oo02Ko0P0:o`801_l00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08ol00`3oo`0jo`030?oo043o00<0ool07ol00`3oo`0Wo`00
;_l00`3oo`0Ro`80??l00`3oo`0oo`030?oo023o00<0ool09ol002ko00<0ool08_l00`3oo`0ko`03
0?oo03oo00<0ool08?l00`3oo`0Wo`001ol500;o0P04o`806?l00`3oo`08o`D05Ol00`3oo`0:o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool04_l500Wo00<0ool05_l500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool05Ol200[o00@0ool02?l0103oo`0Ro`03
0?oo027o00@0ool02?l0103oo`0=o`030?oo01Co00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool0
0_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`040?oo00_o00<0ool0
8?l00`3oo`0Qo`040?oo00_o00<0ool02_l00`3oo`0Eo`030?oo00Wo00<0ool06?l00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Fo`030?oo00;o1002o`04
0?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`08o`030?oo01Wo00<0ool0
2?l200?o0009o`030?oo00So00<0ool06?l00`3oo`09o`030?oo01Oo00<0ool01ol0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00[o00<0ool05ol00`3oo`08o`030?oo01Wo00<0ool0
2?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Jo`030?oo00Oo0P0:o`<08_l00`3oo`0Ro`80
2_l300Wo00<0ool05ol200Wo00<0ool06Ol200[o0`02o`00;ol00`3oo`0Uo`030?oo03Oo00<0ool0
>Ol00`3oo`0So`030?oo02[o000`o`030?oo02Go00<0ool0=_l00`3oo`0ho`030?oo02?o00<0ool0
:ol0037o00<0ool09Ol00`3oo`0eo`030?oo03Ko0P0Vo`030?oo02_o000ao`030?oo02Ko0P0eo`03
0?oo03Co0P0Wo`030?oo02co000bo`030?oo02Oo0P0co`030?oo03;o0P0Xo`030?oo02go000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00;o0P02o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
0_l200;o00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Uo`050?ooo`00:Ol302co00<0ool0
:Ol00`3oo`0202Oo00D0oooo000Xo`030?oo00Oo000eo`030?oo02co0P0[o`030?oo02Wo0`0[o`80
<ol003Go00<0ool0;_l302So00<0ool09ol202co0P0eo`00=_l00`3oo`0`o`809_l00`3oo`0So`@0
;Ol00`3oo`0eo`00=ol00`3oo`0ao`D08Ol00`3oo`0Mo`H0;ol203So000ho`030?oo03Go1@0Lo`03
0?oo01So1@0co`80>_l003Wo00<0ool0>Ol501Oo00<0ool04ol503Go0`0lo`00>_l203ko1@0Bo`03
0?oo00go1P0ho`80?ol003co0P11ob80>ol3047o000no`80DOl00`3oo`14o`@0A?l0043o0P1?o`03
0?oo047o0`18o`00@_l304co00<0ool0?Ol404_o0015o`80B_l00`3oo`0go`H0Col004Oo0`17o`03
0?oo03?o101Eo`00B_l2037o0P04o`800ol400Go00<0ool0;ol405Wo001<o`<0;Ol0103oo`08o`03
0?oo00Ko00<0ool0:ol405go001?o`80:ol0103oo`09o`030?oo00Go0P0Yo`<0HOl0057o0`0Xo`04
0?oo00[o00<0ool01?l00`3oo`0To`@0I?l005Co0`0Uo`040?oo00So00@0ool01Ol00`3oo`0Qo`<0
J?l005Oo100Ro`802_l200Ko00<0ool07_l306_o001Ko`<0<ol00`3oo`0Lo`80K_l005ko100_o`03
0?oo01Wo0`1`o`00H_l402_o00<0ool05ol207?o001Vo`@09ol00`3oo`0Bo`D0MOl006[o1@0Ro`03
0?oo00go1@1jo`00Kol501go00<0ool02Ol407oo001do`X04ol00`3oo`0908?o001noaH0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co
00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3o
o`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00
oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87511 .14821 m
.87105 .14025 L
.8655 .13235 L
.86227 .12879 L
.85849 .12534 L
.85089 .12013 L
.84615 .11779 L
.84354 .11674 L
.8407 .11577 L
.83816 .11504 L
.8353 .11436 L
.83263 .11386 L
.83016 .11349 L
.82891 .11335 L
.82755 .11321 L
.82624 .11311 L
.82505 .11303 L
.82438 .113 L
.82366 .11297 L
.82289 .11295 L
.82217 .11293 L
.82149 .11293 L
.82078 .11293 L
.81949 .11295 L
.81829 .11298 L
.817 .11304 L
.81579 .11312 L
.81469 .11321 L
.81199 .11349 L
.80945 .11385 L
.80481 .11473 L
.79983 .11604 L
.79485 .11778 L
.79044 .11977 L
.7833 .12414 L
.77977 .12706 L
.77685 .13003 L
.77416 .13338 L
.77166 .13727 L
.76805 .14499 L
.76632 .15009 L
.76496 .15512 L
.76286 .16491 L
.76109 .17384 L
.7589 .18331 L
.75735 .18858 L
.75569 .19325 L
.75165 .2019 L
.74622 .21021 L
.74 .21739 L
Mistroke
.72389 .23108 L
.69107 .25117 L
.67321 .25912 L
.65306 .26574 L
.63154 .27113 L
.61109 .27543 L
.59239 .27875 L
.57161 .28163 L
.55062 .2838 L
.53895 .28478 L
.52835 .28555 L
.52329 .28587 L
.51781 .28617 L
.51309 .28638 L
.5079 .28657 L
.50507 .28664 L
.50241 .28669 L
.50095 .28671 L
.49962 .28673 L
.49891 .28673 L
.49815 .28674 L
.49735 .28674 L
.4966 .28674 L
.49524 .28674 L
.49396 .28673 L
.49281 .28672 L
.49156 .28671 L
.4902 .28669 L
.48876 .28666 L
.48617 .2866 L
.48354 .28652 L
.48103 .28642 L
.47634 .2862 L
.46577 .28552 L
.44676 .28385 L
.4282 .28186 L
.40817 .27918 L
.38872 .2755 L
.36806 .27009 L
.34785 .26354 L
.33008 .25695 L
.31236 .24917 L
.29652 .24023 L
.28868 .23461 L
.28227 .22899 L
.27639 .22228 L
.27391 .21845 L
.27293 .21656 L
.27216 .21478 L
.27156 .21306 L
Mistroke
.27113 .21149 L
.27094 .2106 L
.2708 .20975 L
.27069 .20885 L
.27065 .20834 L
.27062 .20787 L
.2706 .20741 L
.2706 .20692 L
.27062 .20602 L
.27068 .20515 L
.27077 .20435 L
.27091 .20349 L
.2711 .20254 L
.27164 .20064 L
.27228 .19897 L
.27298 .19746 L
.27498 .19411 L
.27984 .18821 L
.29444 .17512 L
.30053 .16957 L
.30328 .16667 L
.30572 .16359 L
.30665 .16217 L
.30747 .1607 L
.30807 .15935 L
.3083 .15869 L
.30848 .15811 L
.30863 .15755 L
.30875 .15696 L
.30885 .15633 L
.30891 .15575 L
.30893 .15511 L
.30891 .15444 L
.30886 .15388 L
.30876 .15327 L
.3086 .15259 L
.30841 .15196 L
.30792 .15079 L
.30722 .14958 L
.30626 .1483 L
.30403 .14615 L
.30142 .14433 L
.29798 .14251 L
.29371 .1408 L
.28923 .13945 L
.28481 .13842 L
.28061 .13765 L
.27585 .137 L
.2734 .13674 L
.27076 .13651 L
Mistroke
.26847 .13635 L
.26594 .13621 L
.26464 .13615 L
.26323 .1361 L
.26175 .13606 L
.26035 .13603 L
.25897 .13602 L
.25765 .13601 L
.25647 .13601 L
.25519 .13602 L
.25439 .13602 L
.25365 .13603 L
.25226 .13606 L
.25076 .1361 L
.24914 .13614 L
.24624 .13626 L
.24357 .13639 L
.23345 .13708 L
.22288 .13811 L
.21328 .13932 L
.20303 .14094 L
.19341 .14287 L
.18472 .14507 L
.17507 .14818 L
.16645 .15174 L
.15754 .15649 L
.14896 .16257 L
.14244 .16878 L
.13945 .17236 L
.13667 .17636 L
.13455 .18005 L
.13259 .18427 L
.13116 .18824 L
.13012 .19205 L
.12962 .19447 L
.12926 .19672 L
.1291 .19801 L
.12897 .19922 L
.12887 .20049 L
.12883 .20117 L
.12879 .2019 L
.12875 .20311 L
.12874 .2038 L
.12874 .20445 L
.12875 .20562 L
.12879 .20688 L
.12885 .20812 L
.12893 .20929 L
.12914 .21147 L
.12943 .21366 L
Mistroke
.12981 .21598 L
.13068 .22012 L
.13189 .22453 L
.13351 .22932 L
.13763 .23871 L
.14221 .24683 L
.14842 .25566 L
.15484 .26316 L
.16868 .27565 L
.18582 .28681 L
.20259 .29514 L
.22178 .303 L
.25965 .31651 L
.2969 .32849 L
.33677 .33973 L
.35684 .34459 L
.3759 .34842 L
.39467 .35126 L
.40528 .35247 L
.41494 .35335 L
.42527 .35409 L
.43482 .35461 L
.44534 .35505 L
.45066 .35522 L
.45652 .35536 L
.45915 .35542 L
.46202 .35546 L
.46454 .3555 L
.46593 .35551 L
.46724 .35553 L
.4685 .35554 L
.46966 .35554 L
.47095 .35555 L
.47234 .35555 L
.47344 .35556 L
.47464 .35555 L
.4759 .35555 L
.47707 .35555 L
.47845 .35554 L
.47995 .35553 L
.48268 .3555 L
.48533 .35546 L
.48778 .35542 L
.49335 .35528 L
.49925 .35507 L
.50446 .35483 L
.50922 .35457 L
.51989 .35381 L
.52973 .35288 L
.53903 .35177 L
Mistroke
.55949 .34848 L
.57817 .34435 L
.61662 .33273 L
.65668 .31947 L
.69646 .309 L
.73521 .30074 L
.77597 .29108 L
.79568 .28518 L
.81344 .2785 L
.82967 .27037 L
.84569 .25916 L
.85318 .25226 L
.86054 .24401 L
.86614 .23637 L
.8714 .22753 L
.87553 .21868 L
.87853 .21016 L
.87981 .20539 L
.88076 .20097 L
.88154 .19605 L
.88183 .19356 L
.88197 .19213 L
.88207 .19082 L
.88215 .18967 L
.88221 .18841 L
.88226 .18714 L
.88228 .18645 L
.8823 .1858 L
.88231 .18459 L
.88231 .18349 L
.88231 .18288 L
.8823 .18222 L
.88226 .18101 L
.88222 .17982 L
.88216 .17874 L
.88199 .17631 L
.88178 .17406 L
.88153 .17194 L
.88072 .16681 L
.87972 .16211 L
.87833 .15699 L
.87652 .15168 L
.87511 .14821 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C?l<03Co001lo`040?oo00So1@04o`80Bol200co0`0ao`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool03ol00`3oo`0^o`00O?l0103oo`09o`030?l000Go00<0ool0
Aol201?o00<0ool0;Ol007go0P0;o`801Ol00`3oo`16o`030?oo01Co00<0ool0;?l0097o00<0ool0
AOl00`3oo`0Fo`030?oo02_o0016oa80>Ol00`3oo`15o`030?oo01Ko00<0ool0:ol0047o1@0Bo`@0
=Ol00`3oo`14o`030?oo01So00<0ool0:_l003co1@0Ko`030?oo03;o00<0ool0A?l00`3oo`0Ho`03
0?oo02[o000jo`808Ol00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00=ol302?o0P0bo`03
0?oo04?o00<0ool06ol00`3oo`0Xo`00=Ol202Oo00<0ool0<?l00`3oo`13o`030?oo01_o00<0ool0
:?l003Co00<0ool09_l203;o00<0ool0@_l00`3oo`0Lo`030?oo02So000co`030?oo02Oo00<0ool0
<Ol00`3oo`12o`030?oo01go00<0ool09ol0037o0P0Yo`030?oo03;o00<0ool0@_l00`3oo`0Mo`03
0?oo02Oo000ao`030?oo02Oo00<0ool0<ol00`3oo`12o`030?oo01go00<0ool09ol0033o00<0ool0
9ol00`3oo`0Po`801?l200?o1005o`030?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Wo`03
0?oo023o00@0ool02?l00`3oo`06o`030?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Vo`03
0?oo027o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0No`030?oo02Oo000_o`030?oo02Go00<0ool0
7?l400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Uo`03
0?oo02?o00@0ool02?l0103oo`05o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`03
0?oo02Go0P0:o`801_l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool08ol00`3oo`0jo`03
0?oo043o00<0ool07ol00`3oo`0Wo`00;_l00`3oo`0So`030?oo03[o00<0ool0?ol00`3oo`0Po`03
0?oo02Oo000^o`030?oo02;o0P0lo`030?oo03ko00<0ool08Ol00`3oo`0Wo`001ol500;o0P04o`80
6?l00`3oo`08o`D05Ol200_o0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00So00<0ool06Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802ol00`3o
o`0Go`030?oo00So00<0ool06Ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo
00<0ool05ol200So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`09o`030?oo01So
00<0ool02?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<06?l00`3oo`08o`806ol00`3o
o`06o`802_l302;o00<0ool08_l200[o0`08o`030?oo01So0P09o`030?oo01Wo0P0:o`<00_l002oo
00<0ool09Ol00`3oo`0go`030?oo03So00<0ool09?l00`3oo`0Zo`00<?l00`3oo`0Uo`80=ol00`3o
o`0fo`809_l00`3oo`0[o`00<?l00`3oo`0Wo`030?oo03Co00<0ool0=?l202Oo00<0ool0;?l0037o
00<0ool09ol203Co00<0ool0<_l202So00<0ool0;Ol0037o00<0ool0:Ol203;o00<0ool0<?l202Wo
00<0ool0;_l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`040?ooo`800ol00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`040?ooo`800ol00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co00<0ool00_l00`3o
o`0Xo`80;?l00`3oo`0Yo`@0:?l0103oo`0Yo`030?oo00Oo000do`80;ol202[o00<0ool09ol302co
0P0do`00=_l00`3oo`0^o`@09_l00`3oo`0To`<0;_l00`3oo`0do`00=ol00`3oo`0ao`@08_l00`3o
o`0Qo`<0<?l00`3oo`0eo`00=ol00`3oo`0eo`<07ol00`3oo`0Mo`@0<Ol203So000ho`030?oo03Oo
100Ko`030?oo01So1@0do`030?oo03So000io`80>ol:017o00<0ool03_l:03Ko0`0ko`00>ol204?o
2@08o`030?oo00Co2P0lo`@0?_l003go00<0ool0BOl?04Co0P12o`00?_l2057o00<0ool0A?l404Co
0010o`<0C_l00`3oo`0no`H0B?l004?o0P1<o`030?oo03[o101>o`00AOl304Wo00<0ool0=_l405;o
0018o`<0A_l00`3oo`0co`<0E_l004_o0P0`o`801?l200?o1005o`030?oo02oo101Io`00COl302co
00@0ool02?l00`3oo`06o`030?oo02Wo1P1Mo`00D?l302Wo00@0ool02Ol00`3oo`05o`809_l406?o
001Co`<09_l0103oo`0:o`030?oo00Co00<0ool08ol206Oo001Fo`<08ol0103oo`08o`040?oo00Go
00<0ool08?l306Wo001Io`<08Ol200[o0P06o`030?oo01ko0P1/o`00G?l4037o00<0ool06ol306ko
001Po`@0;Ol00`3oo`0Go`@0LOl006Co100Yo`030?oo01?o101eo`00J?l502Co00<0ool03ol407Wo
001]o`D07ol00`3oo`0:o`D0OOl007;o2P0Eo`030?oo00?o1`22o`00O?lK08Wo002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87469 .15276 m
.87085 .14472 L
.86557 .13669 L
.86248 .13304 L
.85886 .12948 L
.85155 .12405 L
.84696 .12157 L
.84166 .11939 L
.83914 .11857 L
.8364 .11782 L
.83398 .11728 L
.83135 .11679 L
.82876 .11642 L
.82637 .11617 L
.82522 .11608 L
.82395 .116 L
.82275 .11594 L
.82164 .11591 L
.82096 .11589 L
.82022 .11589 L
.81888 .11589 L
.81758 .11592 L
.81638 .11596 L
.81507 .11603 L
.81365 .11614 L
.81078 .11642 L
.80821 .11677 L
.80545 .11724 L
.80053 .11836 L
.79591 .11977 L
.79183 .12136 L
.78344 .12596 L
.77973 .12882 L
.77608 .13239 L
.7733 .13582 L
.77075 .13977 L
.76712 .14756 L
.76553 .15225 L
.76427 .1568 L
.76187 .16752 L
.75963 .17802 L
.75703 .18767 L
.75395 .19588 L
.74961 .20414 L
.74429 .21153 L
.73735 .21892 L
.72085 .23195 L
.7029 .24315 L
.68604 .25187 L
.66784 .25917 L
Mistroke
.64959 .26466 L
.6113 .27329 L
.59148 .27675 L
.57003 .27957 L
.54761 .2817 L
.5373 .2825 L
.52647 .28324 L
.51651 .28379 L
.51089 .28402 L
.50821 .2841 L
.50567 .28417 L
.50337 .28422 L
.50207 .28424 L
.50087 .28425 L
.49986 .28426 L
.49875 .28427 L
.49759 .28427 L
.4965 .28427 L
.4953 .28427 L
.49404 .28426 L
.49296 .28425 L
.49178 .28424 L
.48914 .28419 L
.48669 .28413 L
.48148 .28394 L
.47583 .28366 L
.46556 .28297 L
.44644 .28127 L
.42676 .27917 L
.41653 .2779 L
.40567 .27629 L
.38576 .27236 L
.36712 .26747 L
.34812 .26156 L
.32838 .25451 L
.31172 .24732 L
.30247 .24245 L
.29407 .23723 L
.28614 .23127 L
.27985 .22531 L
.27716 .22212 L
.2747 .21855 L
.27284 .21491 L
.2721 .21291 L
.27183 .212 L
.27159 .21104 L
.27142 .21017 L
.27135 .20972 L
.27128 .20922 L
.27119 .20836 L
Mistroke
.27115 .20755 L
.27114 .20706 L
.27115 .20654 L
.27116 .20604 L
.27119 .20558 L
.27128 .2047 L
.27135 .20422 L
.27142 .20378 L
.2718 .20209 L
.27208 .20115 L
.27238 .20028 L
.27388 .19702 L
.27567 .19413 L
.27807 .191 L
.28396 .18477 L
.29146 .17782 L
.29813 .17136 L
.30282 .16569 L
.30379 .16417 L
.30464 .16258 L
.30528 .16111 L
.3055 .16047 L
.30571 .15978 L
.30586 .15916 L
.30598 .15851 L
.30607 .15782 L
.30612 .15719 L
.30613 .15684 L
.30612 .15646 L
.30608 .15578 L
.30599 .15512 L
.30588 .15453 L
.30573 .15397 L
.30556 .15345 L
.30507 .1523 L
.30446 .15121 L
.30364 .15008 L
.30256 .14889 L
.30118 .14765 L
.29815 .14557 L
.29463 .14378 L
.29093 .14233 L
.28651 .14099 L
.28136 .13981 L
.27624 .13895 L
.27374 .13862 L
.27138 .13836 L
.2669 .13798 L
.2645 .13783 L
.26193 .13771 L
Mistroke
.26064 .13767 L
.25923 .13763 L
.2579 .13761 L
.25667 .13759 L
.25557 .13758 L
.25435 .13758 L
.25308 .13759 L
.2519 .1376 L
.25059 .13762 L
.24921 .13765 L
.24672 .13772 L
.24378 .13784 L
.24112 .13797 L
.23104 .13868 L
.22143 .13962 L
.21244 .14073 L
.20271 .14222 L
.19245 .14422 L
.18377 .14638 L
.1744 .14934 L
.16488 .15325 L
.15671 .15762 L
.14813 .16366 L
.14112 .1704 L
.13568 .17764 L
.13344 .18158 L
.13139 .18614 L
.13001 .19016 L
.12942 .19232 L
.1289 .19462 L
.12853 .19667 L
.12837 .19776 L
.12822 .19893 L
.1281 .20002 L
.12801 .20101 L
.12795 .20201 L
.1279 .20295 L
.12786 .20405 L
.12785 .20523 L
.12785 .20634 L
.12788 .20736 L
.12793 .20856 L
.12797 .20924 L
.12802 .20987 L
.12824 .21225 L
.12858 .21476 L
.12897 .21703 L
.13014 .22211 L
.13143 .22643 L
.13312 .23103 L
Mistroke
.13686 .23913 L
.14167 .24731 L
.14776 .25566 L
.16213 .27039 L
.17754 .28157 L
.19582 .2913 L
.23439 .30614 L
.27183 .31857 L
.30862 .32969 L
.34781 .34004 L
.36761 .34435 L
.38663 .34755 L
.40511 .34975 L
.41555 .35066 L
.42514 .35131 L
.43453 .35182 L
.44459 .35225 L
.45039 .35244 L
.45568 .35259 L
.45823 .35264 L
.46101 .3527 L
.46364 .35274 L
.46605 .35277 L
.46852 .3528 L
.46988 .35281 L
.47117 .35281 L
.47236 .35282 L
.47346 .35282 L
.47467 .35282 L
.47596 .35282 L
.47725 .35282 L
.47798 .35281 L
.47864 .35281 L
.47987 .3528 L
.48116 .35279 L
.4839 .35276 L
.48691 .35271 L
.48981 .35264 L
.49244 .35257 L
.49836 .35237 L
.50384 .35213 L
.50901 .35184 L
.51914 .35113 L
.52981 .35012 L
.54876 .34764 L
.55834 .346 L
.56857 .34393 L
.58681 .33943 L
.62658 .32693 L
.66472 .31512 L
Mistroke
.70257 .30611 L
.74317 .29823 L
.78177 .28951 L
.80221 .28349 L
.81131 .28023 L
.82068 .27629 L
.82907 .27208 L
.83789 .26671 L
.85206 .25526 L
.85895 .2479 L
.86452 .24067 L
.87008 .23177 L
.87425 .22322 L
.87726 .21514 L
.87848 .21095 L
.87956 .20638 L
.8804 .20176 L
.8807 .19968 L
.88096 .19738 L
.88119 .19482 L
.88128 .19342 L
.88135 .1921 L
.8814 .19087 L
.88143 .18974 L
.88144 .18911 L
.88144 .18843 L
.88144 .1872 L
.88143 .18617 L
.8814 .18523 L
.88136 .18415 L
.88132 .18311 L
.8812 .18127 L
.88104 .17926 L
.88081 .17704 L
.88051 .17468 L
.87982 .17041 L
.87879 .16557 L
.87758 .16104 L
.87469 .15276 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Col603Oo001lo`040?oo00So1@04o`80C?l400Ko100co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03?l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4Ol00`3oo`0^o`00OOl200_o0P05o`030?oo04Oo00<0ool04_l00`3oo`0]o`00TOl00`3oo`16o`03
0?oo01Co00<0ool0;?l004So3P0ko`030?oo04Go00<0ool05_l00`3oo`0[o`00@Ol700ko100go`03
0?oo04Go00<0ool05_l00`3oo`0[o`00??l501Wo0P0eo`030?oo04Co00<0ool06?l00`3oo`0Zo`00
>Ol3023o00<0ool0<_l00`3oo`14o`030?oo01So00<0ool0:_l003Oo0P0So`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Go0P0Vo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`03
0?oo02Ko00<0ool0<Ol00`3oo`13o`030?oo01_o00<0ool0:?l003?o00<0ool09_l203?o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`030?oo01co00<0ool0
:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07Ol00`3oo`0Wo`00<?l00`3oo`0Wo`03
0?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`030?oo02Ko00<0ool0
8?l0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Ko00<0ool0
8Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool09ol002ko00<0ool09Ol201ko1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0To`04
0?oo00So00@0ool01Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09?l00`3oo`0Uo`80
2_l200Ko00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo02?o00<0ool0>_l00`3oo`10o`03
0?oo01oo00<0ool09ol002ko00<0ool08ol00`3oo`0jo`030?oo03oo00<0ool08?l00`3oo`0Wo`00
;_l00`3oo`0So`030?oo03[o00<0ool0?ol00`3oo`0Oo`030?oo02So0007o`D00_l200Co0P0Ho`03
0?oo00So1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01Oo0P08o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol201Wo00<0ool0
2?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806ol00`3oo`06o`80
2_l302;o00<0ool08_l200[o0`07o`806_l200[o00<0ool06?l200[o0`02o`00;ol00`3oo`0Vo`03
0?oo03Ko00<0ool0=ol00`3oo`0Uo`030?oo02[o000`o`030?oo02Ko00<0ool0=Ol00`3oo`0eo`80
9ol00`3oo`0[o`00<?l00`3oo`0Wo`80=Ol00`3oo`0co`80:Ol00`3oo`0[o`00<Ol00`3oo`0Xo`80
<ol00`3oo`0bo`030?oo02So00<0ool0;?l0037o00<0ool0:_l2037o00<0ool0<?l202[o00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Co0`08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol302[o00<0ool0
:?l302[o00<0ool00P0Xo`030?oo00Oo000do`030?oo033o0`0Wo`030?oo02Ko0P0_o`030?oo03;o
000eo`80<ol202Go00<0ool08_l4033o00<0ool0<ol003Oo00<0ool0<_l4027o00<0ool07Ol503?o
00<0ool0=?l003So00<0ool0=Ol501co00<0ool06?l503Ko0P0go`00>Ol203[o1`0Eo`030?oo01?o
1@0io`80>Ol003_o00<0ool0?_l900co00<0ool02?l;03_o0`0ko`00??l204Oo5`14o`80?_l003ko
0P1Ao`030?oo04Wo0`10o`00@?l204oo00<0ool0AOl404?o0012o`80COl00`3oo`10o`D0Aol004Co
0`1:o`030?oo03_o1@1<o`00Aol204So00<0ool0=Ol6057o0019o`<0AOl00`3oo`0`o`D0Eol004co
0`0^o`801?l200?o1005o`030?oo02_o1@1Lo`00Col402Wo00@0ool02?l00`3oo`06o`030?oo02Oo
101Qo`00Dol302Ko00@0ool02Ol00`3oo`05o`809?l406Go001Fo`<08ol0103oo`0:o`030?oo00Co
00<0ool08?l306Wo001Io`@07ol0103oo`08o`040?oo00Go00<0ool07Ol306co001Mo`<07Ol200[o
0P06o`030?oo01_o0P1_o`00H?l302ko00<0ool06?l3077o001So`@0:_l00`3oo`0Fo`80M?l006Oo
100Vo`030?oo01;o101fo`00Jol5027o00<0ool03_l407[o001`o`D07?l00`3oo`07o`L0O_l007Go
9P25o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87428 .15731 m
.87065 .14919 L
.86564 .14102 L
.86269 .13729 L
.85923 .13362 L
.85221 .12798 L
.84777 .12536 L
.84263 .12302 L
.83749 .12128 L
.83495 .12062 L
.83253 .12009 L
.83014 .11966 L
.82793 .11934 L
.82545 .11907 L
.82406 .11895 L
.82275 .11887 L
.82148 .11881 L
.82012 .11876 L
.81883 .11875 L
.81767 .11875 L
.81695 .11876 L
.81629 .11877 L
.81556 .1188 L
.8148 .11883 L
.81351 .1189 L
.8121 .119 L
.80952 .11924 L
.80674 .11961 L
.80176 .12052 L
.79706 .12173 L
.79288 .12312 L
.78861 .12493 L
.78422 .1273 L
.78004 .13019 L
.77653 .1333 L
.77103 .14013 L
.76879 .14402 L
.76672 .14858 L
.76383 .15731 L
.76173 .16593 L
.75977 .17491 L
.75743 .18429 L
.75596 .18903 L
.75407 .19408 L
.74991 .20241 L
.74718 .20665 L
.74387 .21096 L
.73701 .21814 L
.72128 .23037 L
.70452 .24069 L
Mistroke
.68577 .25024 L
.66702 .25753 L
.64865 .26287 L
.60974 .27149 L
.58835 .27507 L
.56782 .27754 L
.54693 .27938 L
.52773 .28073 L
.52287 .28102 L
.51769 .28129 L
.51323 .28148 L
.5084 .28165 L
.50546 .28172 L
.50417 .28174 L
.50275 .28177 L
.50141 .28178 L
.50015 .2818 L
.49883 .2818 L
.49809 .28181 L
.4974 .28181 L
.4962 .2818 L
.49507 .2818 L
.49384 .28179 L
.49253 .28177 L
.49114 .28175 L
.48984 .28172 L
.48739 .28166 L
.48266 .28149 L
.47746 .28124 L
.46692 .28055 L
.44809 .27889 L
.427 .27669 L
.40825 .27422 L
.38894 .27065 L
.37051 .26613 L
.33442 .25504 L
.31692 .24824 L
.29882 .23901 L
.2912 .23404 L
.28374 .22812 L
.28073 .22523 L
.27792 .2221 L
.27573 .21916 L
.27405 .21634 L
.27328 .21473 L
.27261 .21299 L
.27214 .2114 L
.27198 .21069 L
.27184 .20992 L
.27173 .2091 L
Mistroke
.27165 .20833 L
.27163 .20786 L
.27161 .20742 L
.27161 .20658 L
.27166 .2057 L
.27175 .20489 L
.27181 .20442 L
.27188 .204 L
.27208 .20305 L
.27252 .20145 L
.27305 .19999 L
.27466 .19671 L
.27652 .19383 L
.27886 .1908 L
.29142 .17782 L
.29664 .17224 L
.29895 .16934 L
.30096 .16631 L
.30177 .16479 L
.30248 .16314 L
.30274 .16243 L
.30296 .16166 L
.30314 .16095 L
.30326 .1603 L
.30335 .15963 L
.3034 .15891 L
.30341 .15822 L
.30336 .15749 L
.30328 .15681 L
.30316 .15619 L
.30302 .15563 L
.30283 .15504 L
.30227 .15373 L
.30145 .15236 L
.30036 .15098 L
.29913 .14975 L
.29646 .14772 L
.29299 .14577 L
.2889 .14407 L
.28469 .14273 L
.27971 .14152 L
.27704 .14101 L
.27409 .14053 L
.2712 .14015 L
.26852 .13985 L
.26598 .13963 L
.26318 .13943 L
.26065 .1393 L
.25829 .13921 L
.25698 .13917 L
Mistroke
.2558 .13915 L
.25443 .13913 L
.25313 .13912 L
.25193 .13912 L
.25064 .13913 L
.24942 .13914 L
.24831 .13916 L
.24702 .13919 L
.24561 .13923 L
.24305 .13932 L
.23758 .13959 L
.23257 .13993 L
.22364 .14072 L
.21423 .14178 L
.20411 .14321 L
.19316 .14519 L
.18287 .14764 L
.17356 .15052 L
.16449 .15415 L
.15649 .15828 L
.14844 .16371 L
.14091 .17059 L
.13505 .1781 L
.13239 .18272 L
.13037 .18722 L
.12897 .19129 L
.12783 .19591 L
.12741 .19828 L
.12724 .19953 L
.12709 .20086 L
.12699 .20203 L
.12694 .20266 L
.1269 .20332 L
.12685 .2045 L
.12682 .20559 L
.12682 .20676 L
.12683 .20743 L
.12684 .20805 L
.1269 .20931 L
.12694 .21002 L
.12698 .21066 L
.12721 .21307 L
.12752 .21536 L
.12797 .21799 L
.1285 .22042 L
.13001 .22588 L
.13353 .23486 L
.13842 .24395 L
.1442 .25237 L
.15131 .26071 L
Mistroke
.16598 .27376 L
.1824 .28424 L
.20132 .29309 L
.23694 .30555 L
.27469 .31749 L
.31175 .32831 L
.35115 .33838 L
.37123 .34252 L
.38023 .34404 L
.39026 .34545 L
.40904 .34742 L
.41936 .34819 L
.42904 .34877 L
.43903 .34925 L
.44958 .34965 L
.45395 .34978 L
.45863 .3499 L
.46132 .34995 L
.46378 .34999 L
.46626 .35003 L
.46858 .35006 L
.46983 .35007 L
.47097 .35008 L
.47224 .35008 L
.4736 .35009 L
.47491 .35009 L
.47565 .35009 L
.47634 .35009 L
.4776 .35009 L
.47893 .35008 L
.48006 .35007 L
.48128 .35006 L
.48348 .35004 L
.4861 .35 L
.48851 .34995 L
.49355 .34982 L
.49906 .34962 L
.5048 .34935 L
.51014 .34904 L
.52099 .34823 L
.53116 .34723 L
.55002 .34472 L
.57003 .34093 L
.5882 .33642 L
.62583 .32478 L
.66183 .31393 L
.70144 .30484 L
.73995 .29807 L
.75948 .29447 L
.78056 .28993 L
Mistroke
.80048 .28468 L
.81789 .27867 L
.82687 .27465 L
.835 .2702 L
.84325 .26466 L
.85132 .25791 L
.85783 .2512 L
.86413 .2432 L
.86898 .23556 L
.87281 .22805 L
.87622 .21931 L
.87847 .21118 L
.87938 .20655 L
.87999 .20222 L
.88022 .19994 L
.88041 .1974 L
.88048 .19612 L
.8805 .19545 L
.88053 .19474 L
.88055 .19411 L
.88056 .19342 L
.88057 .19221 L
.88057 .1911 L
.88056 .19007 L
.88053 .18894 L
.88048 .18774 L
.88043 .18669 L
.88038 .18571 L
.88021 .18347 L
.87979 .17958 L
.87917 .17538 L
.87821 .17049 L
.87707 .16586 L
.87428 .15731 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1?o`P0=_l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`@02?l303?o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`803ol00`3oo`0`o`00
OOl200_o0P05o`030?oo04Oo00<0ool04?l2033o002Ao`030?oo04Ko00<0ool04ol00`3oo`0]o`00
TOl00`3oo`15o`030?oo01Go00<0ool0;?l004Co500io`030?oo04Go00<0ool05_l00`3oo`0[o`00
?_l601Co0P0go`030?oo04Co00<0ool05ol00`3oo`0[o`00??l201co0P0eo`030?oo04?o00<0ool0
6Ol00`3oo`0Zo`00>Ol301oo0P0do`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=ol202?o00<0ool0
<_l00`3oo`13o`030?oo01[o00<0ool0:Ol003Go0P0Uo`030?oo03;o00<0ool0@ol00`3oo`0Jo`03
0?oo02Wo000co`809ol00`3oo`0bo`030?oo04;o00<0ool07?l00`3oo`0Xo`00<_l00`3oo`0Wo`03
0?oo03;o00<0ool0@_l00`3oo`0Lo`030?oo02So000ao`030?oo02Ko0P0eo`030?oo04;o00<0ool0
7?l00`3oo`0Xo`00<?l00`3oo`0Wo`030?oo023o0P04o`800ol400Go00<0ool0@_l00`3oo`0Lo`03
0?oo02So000_o`030?oo02Oo00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`03
0?oo02Oo000_o`030?oo02Ko00<0ool08Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool0
9ol002ko00<0ool09_l00`3oo`0Lo`@00_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0No`03
0?oo02Oo000^o`030?oo02Go00<0ool08ol0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0No`03
0?oo02Oo000^o`030?oo02Co00<0ool09Ol200[o0P06o`030?oo043o00<0ool07ol00`3oo`0Wo`00
;Ol202Ko00<0ool0>Ol00`3oo`0oo`030?oo023o00<0ool09ol002go00<0ool09?l00`3oo`0jo`03
0?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?_l00`3oo`0Po`03
0?oo02So0007o`D00_l200Co0P0Go`030?oo00Wo1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool0
8_l200Co0P04o`803Ol00`3oo`0Co`D02Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01Ko00<0ool02ol00`3oo`0Fo`030?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`0<o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool0
2_l00`3oo`0Do`803?l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`09o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o00000ol0000200Co
00<0ool02?l201So00<0ool00_l400Co00<0ool05ol0103oool400;o00@0ool02Ol202?o00<0ool0
8Ol0103oo`09o`803?l00`3oo`0Fo`030?oo00Wo00<0ool06?l00`3oo`08o`800ol000Wo00<0ool0
2?l00`3oo`0Go`030?oo00[o00<0ool05ol200So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool0
2Ol00`3oo`09o`806Ol00`3oo`09o`030?oo01So00<0ool02?l00`3oo`02o`002?l200[o0`0Go`03
0?oo00Wo0P0Ko`030?oo00Ko0P0:o`<08_l00`3oo`0Ro`802_l300Oo0P0Jo`802_l00`3oo`0Ho`80
2_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0go`030?oo02Ko00<0ool0:Ol002oo00<0ool0
9ol203Ko00<0ool0=_l00`3oo`0Vo`030?oo02[o000`o`030?oo02So00<0ool0<ol00`3oo`0do`80
:?l00`3oo`0[o`00<?l00`3oo`0Yo`80<ol00`3oo`0bo`80:Ol00`3oo`0/o`00<Ol00`3oo`0Zo`80
<Ol00`3oo`0`o`80:ol00`3oo`0/o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01Ol300Oo
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00@0o`001Ol00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00?o
00@0ool02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o
00<0ool0:Ol00`3o000302Oo00<0ool09ol402[o00<0ool00_l00`3oo`0Uo`030?oo00Oo000do`03
0?oo03?o0`0To`030?oo02?o100bo`030?oo033o000eo`030?oo03Go100Po`030?oo023o0`0do`80
<ol003Ko0P0io`D06ol00`3oo`0Lo`@0=_l00`3oo`0co`00>?l00`3oo`0ko`D05_l00`3oo`0Ho`@0
>?l203Ko000io`030?oo03oo1P0@o`030?oo00go2`0jo`80>?l003[o0P15ob00@_l303[o000lo`80
Dol00`3oo`1;o`@0?Ol003ko0P1Ao`030?oo04Wo0P11o`00@?l304ko00<0ool0AOl404?o0013o`80
C?l00`3oo`0oo`H0Aol004Go0`19o`030?oo03[o1@1=o`00B?l204Oo00<0ool0=Ol505;o001:o`<0
A?l00`3oo`0`o`D0Eol004go0`0]o`801?l200?o1005o`030?oo02_o1@1Lo`00D?l402So00@0ool0
2?l00`3oo`06o`030?oo02Oo101Qo`00E?l302Go00@0ool02Ol00`3oo`05o`809?l406Go001Go`<0
8_l0103oo`0:o`030?oo00Co00<0ool08?l306Wo001Jo`<07ol0103oo`08o`040?oo00Go00<0ool0
7Ol306co001Mo`<07Ol200[o0P06o`030?oo01Wo101_o`00H?l402go00<0ool05_l307?o001To`@0
:Ol00`3oo`0Bo`@0M_l006So100Uo`030?oo00go1@1jo`00K?l:01_o00<0ool02?l507oo001fo``0
3_l<08Co0022o`h000?o0?l0SOl0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3o
o`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103o
o`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00
oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87387 .16187 m
.87045 .15367 L
.8657 .14536 L
.86291 .14153 L
.85961 .13776 L
.85286 .1319 L
.84859 .12914 L
.8436 .12664 L
.83858 .12474 L
.83372 .12339 L
.82919 .12249 L
.82655 .12211 L
.82405 .12183 L
.8227 .12172 L
.8214 .12164 L
.82023 .12158 L
.81896 .12153 L
.81823 .12152 L
.81757 .12151 L
.81681 .1215 L
.81609 .12151 L
.81481 .12153 L
.81409 .12155 L
.81343 .12157 L
.81212 .12164 L
.81089 .12172 L
.80858 .12191 L
.80612 .1222 L
.80346 .1226 L
.79877 .12354 L
.79458 .12468 L
.7902 .12622 L
.78567 .12828 L
.78132 .13085 L
.77762 .13366 L
.77167 .14003 L
.76923 .14371 L
.76695 .14808 L
.76377 .15652 L
.7615 .16496 L
.75715 .18312 L
.7557 .18794 L
.75391 .19287 L
.74991 .20118 L
.74455 .20905 L
.73731 .21679 L
.72171 .22882 L
.70384 .23954 L
.68715 .24787 L
.66927 .25486 L
Mistroke
.65137 .26014 L
.61374 .26861 L
.59433 .27204 L
.57331 .27473 L
.56277 .27573 L
.55127 .27667 L
.53044 .27813 L
.5199 .27873 L
.51479 .27897 L
.51015 .27914 L
.50764 .27921 L
.50527 .27927 L
.50405 .27929 L
.50277 .27931 L
.50209 .27932 L
.50135 .27933 L
.50005 .27934 L
.49935 .27934 L
.49861 .27934 L
.4973 .27934 L
.49656 .27934 L
.49588 .27933 L
.49513 .27933 L
.49433 .27932 L
.49289 .2793 L
.49152 .27928 L
.48897 .27922 L
.48655 .27915 L
.48429 .27906 L
.47913 .27882 L
.46869 .27815 L
.45001 .27654 L
.43155 .27469 L
.41198 .27231 L
.39163 .26881 L
.37213 .26431 L
.35355 .25927 L
.33391 .25322 L
.31584 .24638 L
.29956 .23828 L
.29194 .23352 L
.28436 .22777 L
.2781 .22154 L
.27548 .218 L
.27446 .21627 L
.27366 .21466 L
.27303 .21309 L
.27251 .21135 L
.27231 .21045 L
.27217 .2096 L
Mistroke
.2721 .20913 L
.27206 .20871 L
.27202 .20821 L
.272 .20774 L
.272 .20695 L
.27203 .20611 L
.27211 .20532 L
.27221 .2046 L
.27236 .20375 L
.27258 .20283 L
.27309 .20116 L
.27368 .19967 L
.27447 .19803 L
.27644 .19473 L
.28105 .18882 L
.28769 .1816 L
.29377 .1748 L
.29603 .17187 L
.29804 .16881 L
.29889 .16723 L
.29966 .16551 L
.30019 .16398 L
.30039 .16324 L
.30054 .16256 L
.30065 .1619 L
.30073 .1612 L
.30077 .16044 L
.30076 .15975 L
.30071 .15904 L
.30067 .15865 L
.30061 .15828 L
.30047 .15763 L
.30028 .15694 L
.29983 .15576 L
.29928 .15468 L
.29854 .15354 L
.29758 .15237 L
.29637 .15117 L
.29485 .14994 L
.29158 .14788 L
.28784 .14614 L
.28395 .14475 L
.27929 .14347 L
.27391 .14238 L
.27135 .14198 L
.26861 .14162 L
.26621 .14136 L
.26362 .14112 L
.26108 .14094 L
.25876 .14081 L
Mistroke
.25765 .14077 L
.25643 .14072 L
.25529 .14069 L
.25423 .14066 L
.25299 .14064 L
.25163 .14063 L
.25031 .14062 L
.24957 .14062 L
.24889 .14063 L
.24769 .14064 L
.24639 .14066 L
.24516 .14068 L
.24402 .14071 L
.24129 .14081 L
.23879 .14092 L
.23322 .14125 L
.22332 .14207 L
.21448 .14303 L
.2052 .14427 L
.196 .14578 L
.18612 .14783 L
.17687 .15032 L
.16859 .15314 L
.15958 .15711 L
.15179 .16159 L
.14408 .16743 L
.1371 .17471 L
.13222 .18192 L
.13003 .18628 L
.1284 .19045 L
.12728 .19423 L
.12677 .19644 L
.12639 .19847 L
.12607 .20065 L
.12593 .20185 L
.12583 .20299 L
.12576 .20405 L
.12571 .20503 L
.12567 .20611 L
.12566 .20726 L
.12567 .2084 L
.12571 .20966 L
.12577 .21084 L
.12585 .21194 L
.12605 .21402 L
.12635 .21624 L
.12681 .21885 L
.12734 .22129 L
.12859 .22583 L
.13166 .23392 L
Mistroke
.136 .24228 L
.14123 .25015 L
.14778 .25811 L
.16297 .27197 L
.17893 .28233 L
.1976 .29114 L
.2365 .30408 L
.27347 .31519 L
.30964 .32552 L
.34814 .33526 L
.36748 .33935 L
.37714 .34103 L
.38609 .34235 L
.40423 .34434 L
.41381 .34509 L
.42388 .34573 L
.43347 .34623 L
.44371 .34666 L
.45341 .347 L
.45763 .34711 L
.46227 .34722 L
.46466 .34726 L
.46722 .3473 L
.46864 .34732 L
.46993 .34733 L
.47125 .34734 L
.47247 .34735 L
.47359 .34735 L
.47479 .34736 L
.47599 .34736 L
.47732 .34736 L
.47846 .34735 L
.47969 .34735 L
.48086 .34734 L
.48192 .34733 L
.4845 .3473 L
.4869 .34726 L
.48954 .3472 L
.49236 .34713 L
.49812 .34693 L
.50336 .34669 L
.51358 .34607 L
.52296 .34531 L
.53382 .34419 L
.54392 .3429 L
.56255 .33979 L
.58279 .33522 L
.61975 .32431 L
.65858 .31281 L
.69726 .30413 L
Mistroke
.73497 .29809 L
.75603 .29476 L
.77499 .29126 L
.79464 .28691 L
.81217 .28193 L
.82065 .27884 L
.82933 .27496 L
.83777 .27024 L
.84626 .26425 L
.85368 .25765 L
.85976 .25098 L
.86943 .23661 L
.87334 .22838 L
.87611 .2206 L
.87724 .21649 L
.87824 .21189 L
.87864 .20956 L
.879 .20703 L
.87927 .20462 L
.87946 .2024 L
.87954 .20125 L
.8796 .20015 L
.87964 .19917 L
.87968 .1981 L
.8797 .19692 L
.87971 .19633 L
.87971 .19568 L
.8797 .19452 L
.87968 .19343 L
.87965 .19234 L
.87961 .19129 L
.8795 .18933 L
.87933 .1872 L
.87911 .18486 L
.87844 .1799 L
.87745 .17457 L
.87626 .16955 L
.87494 .16504 L
.87387 .16187 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo04co2`0eo`00O?l0103oo`09o`030?l000Go00<0ool0B_l200_o0P0co`00OOl200_o0P05o`03
0?oo04Oo0`0?o`030?oo033o002Ao`030?oo04Ko00<0ool04Ol2033o002Ao`030?oo04Go00<0ool0
5?l00`3oo`0]o`00Aol?03_o00<0ool0AOl00`3oo`0Eo`030?oo02co000oo`P03ol303So00<0ool0
A?l00`3oo`0Go`030?oo02_o000jo`D06_l203Ko00<0ool0A?l00`3oo`0Go`030?oo02_o000ho`80
8?l203Go00<0ool0@ol00`3oo`0Io`030?oo02[o000fo`808ol00`3oo`0co`030?oo04?o00<0ool0
6Ol00`3oo`0Zo`00=?l202Go0P0do`030?oo04;o00<0ool06ol00`3oo`0Yo`00<ol00`3oo`0Uo`03
0?oo03?o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000bo`030?oo02Ko00<0ool0<ol00`3oo`12o`03
0?oo01_o00<0ool0:Ol0037o00<0ool09_l00`3oo`0do`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<?l00`3oo`0Wo`030?oo023o0P04o`800ol400Go00<0ool0@Ol00`3oo`0Mo`030?oo02So000_o`03
0?oo02Oo00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0@Ol00`3oo`0Mo`030?oo02So000_o`03
0?oo02Ko00<0ool08Ol0103oo`09o`030?oo00Go0P12o`030?oo01go00<0ool0:?l002ko00<0ool0
9_l00`3oo`0Lo`@00_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Mo`030?oo02So000^o`03
0?oo02Go00<0ool08ol0103oo`08o`040?oo00Go00<0ool0@?l00`3oo`0No`030?oo02So000]o`03
0?oo02Go00<0ool09Ol200[o0P06o`030?oo043o00<0ool07_l202Wo000]o`030?oo02Go00<0ool0
>Ol00`3oo`0oo`030?oo01oo0P0Yo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?ol00`3oo`0Oo`03
0?oo02So000]o`030?oo02Co00<0ool0>_l00`3oo`0no`030?oo023o00<0ool0:?l000Oo1@02o`80
1?l201Oo00<0ool02Ol501Ko00<0ool02Ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0>o`03
0?oo01;o1@09o`030?oo01Ko1@02o`801?l200?o0009o`030?oo00Oo00@0ool05_l00`3oo`0;o`03
0?oo01Ko00<0ool02?l0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00co00<0ool0
5Ol00`3oo`09o`030?oo01So00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Co00<0ool0
2ol00`3oo`0Fo`802Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805ol00`3o
o`03o`@01?l00`3oo`0Go`040?ooo`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0:o`80
6Ol00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01So00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Oo0P0Ko`03
0?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201_o00<0ool0
1_l200[o0`0Ro`030?oo02;o0P0:o`<01_l00`3oo`0Jo`802_l00`3oo`0Ho`802_l300;o000_o`03
0?oo02Ko00<0ool0=_l00`3oo`0fo`030?oo02Oo00<0ool0:Ol002oo00<0ool09ol203Ko00<0ool0
=?l202Wo00<0ool0:_l0033o00<0ool0:?l203Co00<0ool0<_l202[o00<0ool0:ol0033o00<0ool0
:_l203;o00<0ool0<?l202co00<0ool0:ol0037o00<0ool0:ol302oo00<0ool0;_l202go00<0ool0
;?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko1005o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`<01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol0103oool402Go00<0ool0
9ol302_o00<0ool00ol00`3oo`0To`030?oo00Oo000do`030?oo03Go0P0So`030?oo02?o100co`03
0?oo02oo000eo`030?oo03Ko100Oo`030?oo01ko1@0eo`80<_l003Ko00<0ool0>Ol501[o00<0ool0
6Ol503So0P0do`00=ol00`3oo`0mo`D05Ol00`3oo`0Do`D0??l00`3oo`0do`00>?l00`3oo`11o`h0
1ol00`3oo`04oa00?ol203Oo000io`80Col>04_o100io`00>ol205Co00<0ool0COl203go000mo`80
D_l00`3oo`19o`@0?ol003oo0`1?o`030?oo04Co1@13o`00@_l304co00<0ool0?_l604So0015o`@0
B?l00`3oo`0fo`P0C_l004Wo0`15o`030?oo033o1P1Fo`00C?l304;o00<0ool0;?l405co001?o`@0
:_l200Co0P03o`@01Ol00`3oo`0Yo`<0H?l005?o0`0Vo`040?oo00So00<0ool01_l00`3oo`0Vo`<0
Hol005Ko0`0So`040?oo00Wo00<0ool01Ol202?o101Vo`00FOl401oo00@0ool02_l00`3oo`04o`03
0?oo01ko101Zo`00GOl301co00@0ool02?l0103oo`05o`030?oo01_o0`1^o`00H?l301[o0P0:o`80
1_l00`3oo`0Ho`<0LOl006?o100Zo`030?oo01Go0`1do`00Iol402Ko00<0ool04?l507Oo001[o`H0
8?l00`3oo`0;o`D0O?l0077o3@0Co`030?oo00;o2@21o`00O_lH08[o002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D0
1?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0
S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<0
1_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80
SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`80
2_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87346 .16642 m
.87024 .15814 L
.86577 .14969 L
.86312 .14578 L
.85998 .1419 L
.85352 .13582 L
.8494 .13293 L
.84456 .13026 L
.83968 .1282 L
.83491 .12669 L
.83044 .12563 L
.82784 .12516 L
.82536 .1248 L
.82272 .12451 L
.82146 .1244 L
.82028 .12432 L
.81893 .12424 L
.81822 .12421 L
.81745 .12418 L
.81672 .12417 L
.81604 .12415 L
.81475 .12415 L
.81354 .12416 L
.81225 .12419 L
.81103 .12424 L
.80992 .1243 L
.80862 .12439 L
.80721 .1245 L
.80466 .12478 L
.80203 .12516 L
.79927 .12565 L
.79444 .1268 L
.79022 .12814 L
.78646 .12966 L
.7826 .13163 L
.77877 .1341 L
.77558 .1367 L
.77253 .13982 L
.76962 .14363 L
.7673 .14751 L
.76509 .15223 L
.76334 .15696 L
.7607 .16634 L
.75634 .18388 L
.75488 .18851 L
.75308 .19332 L
.74902 .20146 L
.74633 .20559 L
.74346 .20929 L
.73731 .21572 L
Mistroke
.72106 .22798 L
.68838 .2456 L
.67088 .25243 L
.6512 .25816 L
.63172 .26275 L
.61059 .26712 L
.60068 .2689 L
.59017 .27051 L
.57101 .2727 L
.56134 .27354 L
.55091 .27432 L
.53212 .27559 L
.52181 .27619 L
.51642 .27646 L
.51348 .27658 L
.51072 .27668 L
.50814 .27675 L
.50577 .27681 L
.50452 .27683 L
.50315 .27685 L
.50177 .27686 L
.50102 .27687 L
.50032 .27687 L
.49907 .27688 L
.49789 .27688 L
.49683 .27687 L
.49568 .27686 L
.49442 .27685 L
.49309 .27683 L
.49071 .27678 L
.48818 .27671 L
.48578 .27663 L
.48126 .27643 L
.47107 .2758 L
.45268 .27426 L
.43475 .27253 L
.41551 .27034 L
.39675 .26741 L
.37651 .26316 L
.35639 .25814 L
.33867 .25315 L
.32091 .24713 L
.30453 .23988 L
.28877 .23044 L
.28169 .2247 L
.27843 .22134 L
.27596 .21817 L
.27414 .21504 L
.27345 .21344 L
.27287 .21167 L
Mistroke
.27265 .21074 L
.2725 .20989 L
.27243 .20942 L
.27237 .2089 L
.27234 .20842 L
.27232 .20797 L
.27231 .20751 L
.27231 .20708 L
.27235 .20627 L
.27239 .20582 L
.27244 .20541 L
.27258 .20446 L
.27274 .2037 L
.27295 .20288 L
.27342 .2014 L
.27493 .19801 L
.27712 .19441 L
.27954 .1911 L
.28522 .18433 L
.29126 .17711 L
.29541 .17102 L
.29627 .16939 L
.29703 .16764 L
.29759 .16602 L
.2978 .16526 L
.29795 .16456 L
.29807 .16385 L
.29815 .16318 L
.2982 .16249 L
.29821 .16174 L
.29818 .16109 L
.29812 .16042 L
.29803 .15985 L
.2979 .15923 L
.2977 .15851 L
.29747 .15787 L
.29682 .15646 L
.29603 .1552 L
.29513 .15405 L
.29272 .15177 L
.28992 .14986 L
.2863 .14802 L
.28189 .14635 L
.27694 .14495 L
.27208 .14394 L
.26649 .1431 L
.26362 .14279 L
.26044 .14251 L
.25762 .14233 L
.25505 .14221 L
Mistroke
.25363 .14216 L
.25288 .14214 L
.25208 .14212 L
.25139 .14211 L
.25063 .1421 L
.24929 .14209 L
.24795 .14208 L
.24672 .14209 L
.24605 .14209 L
.24532 .1421 L
.24398 .14213 L
.24275 .14216 L
.24143 .14219 L
.23908 .14228 L
.23361 .14256 L
.22867 .14291 L
.21778 .14393 L
.20837 .14505 L
.19847 .14651 L
.18938 .14818 L
.17963 .15046 L
.17052 .15321 L
.16243 .15634 L
.15374 .16071 L
.14634 .16559 L
.13918 .17191 L
.13572 .17582 L
.13292 .17967 L
.12874 .18727 L
.12709 .19152 L
.12639 .19374 L
.12576 .19615 L
.1253 .19831 L
.12496 .20029 L
.1247 .20231 L
.1246 .20322 L
.12452 .20422 L
.12445 .20531 L
.12439 .20647 L
.12437 .20755 L
.12436 .20856 L
.12438 .20975 L
.12442 .21084 L
.12449 .21209 L
.12458 .21329 L
.12485 .21577 L
.1252 .2181 L
.1261 .22244 L
.12741 .22705 L
.1292 .23196 L
Mistroke
.13333 .2405 L
.138 .24789 L
.15014 .26201 L
.15801 .26894 L
.16584 .27472 L
.1831 .28466 L
.20002 .29181 L
.23773 .30323 L
.27312 .31323 L
.31124 .32377 L
.34799 .33283 L
.36657 .33673 L
.38401 .33955 L
.39329 .34067 L
.40312 .34162 L
.41303 .34236 L
.42385 .343 L
.44333 .34387 L
.45427 .34425 L
.45902 .34439 L
.46426 .3445 L
.46662 .34454 L
.46915 .34458 L
.47056 .34459 L
.47184 .34461 L
.47313 .34461 L
.47435 .34462 L
.47548 .34462 L
.47667 .34463 L
.47769 .34462 L
.47883 .34462 L
.48004 .34462 L
.48134 .34461 L
.48258 .3446 L
.48371 .34458 L
.48626 .34454 L
.48864 .3445 L
.49404 .34435 L
.4993 .34415 L
.50493 .34388 L
.5153 .3432 L
.52648 .34223 L
.53605 .34119 L
.54651 .3398 L
.56598 .33642 L
.58634 .33166 L
.62427 .3205 L
.66057 .31032 L
.67953 .30597 L
.70053 .30209 L
Mistroke
.73954 .29673 L
.76051 .29377 L
.77939 .2906 L
.79893 .2866 L
.81633 .28178 L
.82471 .27866 L
.83324 .27466 L
.84136 .26981 L
.84963 .26349 L
.86224 .24989 L
.86783 .24136 L
.87192 .23326 L
.87355 .22929 L
.87503 .22502 L
.87611 .22129 L
.87706 .21724 L
.87751 .21496 L
.87786 .21285 L
.87816 .2107 L
.87839 .20868 L
.87859 .20644 L
.87867 .20525 L
.87874 .20397 L
.87877 .20332 L
.87879 .20262 L
.87881 .202 L
.87883 .20134 L
.87884 .20075 L
.87884 .20009 L
.87884 .19893 L
.87883 .19792 L
.87881 .19698 L
.87878 .19595 L
.87873 .19485 L
.87861 .19279 L
.87847 .19092 L
.87807 .18708 L
.87756 .18343 L
.87595 .17523 L
.87475 .17062 L
.87346 .16642 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o``0=Ol007go0P0;o`801Ol00`3oo`19o`80
3?l203?o002Ao`030?oo04Oo0P0@o`80<Ol0097o00<0ool0A_l00`3oo`0Bo`030?oo02ko001:o`T0
?_l00`3oo`15o`030?oo01Co00<0ool0;Ol0043o2P08o`H0>Ol00`3oo`14o`030?oo01Ko00<0ool0
;?l003go0`0Ho`80=ol00`3oo`14o`030?oo01Oo00<0ool0:ol003Wo100Lo`80=_l00`3oo`13o`03
0?oo01So00<0ool0:ol003Ko0`0Qo`80=Ol00`3oo`13o`030?oo01Wo00<0ool0:_l003Co0P0Uo`03
0?oo03?o00<0ool0@ol00`3oo`0Io`030?oo02[o000co`030?oo02Go00<0ool0<ol00`3oo`12o`03
0?oo01[o00<0ool0:_l003;o00<0ool09_l00`3oo`0co`030?oo04;o00<0ool06ol00`3oo`0Yo`00
<Ol00`3oo`0Vo`030?oo03Co00<0ool0@_l00`3oo`0Ko`030?oo02Wo000`o`030?oo02Oo00<0ool0
8?l200Co0P03o`@01Ol00`3oo`11o`030?oo01go00<0ool0:?l002oo00<0ool09ol00`3oo`0Po`04
0?oo00So00<0ool01_l00`3oo`11o`030?oo01go00<0ool0:?l002ko00<0ool09ol00`3oo`0Qo`04
0?oo00Wo00<0ool01Ol204;o00<0ool07Ol00`3oo`0Xo`00;_l00`3oo`0Vo`030?oo01co1002o`04
0?oo00[o00<0ool01?l00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool09_l00`3oo`0So`04
0?oo00So00@0ool01Ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool09Ol00`3oo`0Uo`80
2_l200Ko00<0ool0@?l00`3oo`0No`030?oo02So000]o`030?oo02Go00<0ool0>Ol00`3oo`0oo`03
0?oo01oo00<0ool0:?l002go00<0ool09?l00`3oo`0jo`030?oo03oo00<0ool07ol00`3oo`0Xo`00
;Ol00`3oo`0To`030?oo03[o00<0ool0?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Go`03
0?oo00Wo1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803Ol00`3oo`0Co`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Do`030?oo00_o00<0ool0
5_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Go`030?oo00?o
1004o`030?oo01Oo0P02o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0:o`806Ol00`3o
o`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool05_l00`3oo`0;o`030?oo01So
00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Oo0P0Ko`030?oo00So
00<0ool06Ol00`3oo`08o`030?oo00;o0008o`802_l301Ko00<0ool02_l201_o00<0ool01_l200[o
0`0Ro`030?oo02;o0P0:o`<01_l00`3oo`0Jo`802_l00`3oo`0Ho`802_l300;o000^o`030?oo02Oo
0P0go`030?oo03Go0P0Yo`030?oo02Wo000_o`030?oo02So0P0eo`030?oo03Co00<0ool0:?l00`3o
o`0Zo`00;ol00`3oo`0Zo`<0<_l00`3oo`0bo`80:ol00`3oo`0Zo`00<?l00`3oo`0/o`80<?l00`3o
o`0`o`80;?l00`3oo`0[o`00<?l00`3oo`0^o`80;_l00`3oo`0^o`80;_l00`3oo`0[o`003?oo00`0
2Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l500Co00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go1006o`030?oo00Ko00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Oo000<o`030?oo02?o00<0ool00ol00`3oo`0Yo`030?oo00;o1@0So`030?oo02?o1@000ol0
o`0Zo`030?oo00Co00<0ool08ol00`3oo`07o`00<ol00`3oo`0ho`@07ol00`3oo`0Oo`@0>?l00`3o
o`0^o`00=?l203co0P0Mo`030?oo01co0`0jo`80<Ol003Ko00<0ool0>ol901Co00<0ool04_l:03_o
0P0co`00=ol00`3oo`13o`X02_l00`3oo`08o`X0A?l00`3oo`0co`00>?l00`3oo`1<oaD0C?l203Ko
000io`80E_l00`3oo`1@o`@0>?l003_o0`1Co`030?oo04ko0P0lo`00?_l2057o00<0ool0B_l403ko
0010o`<0C_l00`3oo`15o`D0@_l004?o0`1;o`030?oo03[o2`17o`00A_l404Oo00<0ool0=Ol505;o
001:o`<0A?l00`3oo`0_o`H0Eol004go0`11o`030?oo02[o1@1Mo`00D?l302[o0P04o`800ol400Go
00<0ool09_l406;o001Co`<09_l0103oo`08o`030?oo00Ko00<0ool08ol306Ko001Fo`<08ol0103o
o`09o`030?oo00Go0P0Qo`<0JOl005Wo100Oo`040?oo00[o00<0ool01?l00`3oo`0Mo`<0K?l005go
100Ko`040?oo00So00@0ool01Ol00`3oo`0Io`@0Kol0067o1@0Go`802_l200Ko00<0ool05Ol407?o
001Vo`D09_l00`3oo`0Ao`@0Mol006_o1@0Qo`030?oo00co1@1ko`00L?l801Wo00<0ool01Ol7083o
001hob40Qol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co
00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3o
o`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00
oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87305 .17097 m
.87004 .16261 L
.86584 .15403 L
.86333 .15003 L
.86035 .14604 L
.85418 .13975 L
.85021 .13671 L
.84553 .13389 L
.84077 .13166 L
.8361 .12999 L
.8317 .12878 L
.82667 .12777 L
.82407 .12738 L
.82125 .12706 L
.81987 .12694 L
.81855 .12685 L
.81737 .12678 L
.81607 .12672 L
.8154 .1267 L
.81469 .12669 L
.81341 .12667 L
.81204 .12667 L
.81125 .12669 L
.81053 .1267 L
.80921 .12675 L
.80801 .12681 L
.80663 .1269 L
.80532 .12701 L
.80293 .12726 L
.80074 .12755 L
.79587 .12845 L
.79124 .12965 L
.7864 .13137 L
.78173 .13362 L
.77773 .13614 L
.7745 .13876 L
.77136 .142 L
.76851 .14576 L
.76625 .1496 L
.76298 .15716 L
.76151 .16169 L
.76033 .16592 L
.75794 .17552 L
.75522 .18556 L
.75227 .19367 L
.75022 .19802 L
.74802 .20189 L
.74278 .20902 L
.73706 .21488 L
.72178 .22627 L
Mistroke
.70435 .23623 L
.68791 .24409 L
.67157 .25031 L
.65329 .25561 L
.634 .26016 L
.61306 .26455 L
.60325 .26635 L
.59286 .26798 L
.57394 .27017 L
.56438 .27098 L
.55404 .27173 L
.53538 .27296 L
.52513 .27358 L
.51978 .27386 L
.51414 .27411 L
.50924 .27427 L
.50789 .27431 L
.50645 .27434 L
.50521 .27436 L
.50384 .27438 L
.50259 .2744 L
.50143 .27441 L
.5002 .27441 L
.49951 .27441 L
.49889 .27441 L
.49777 .27441 L
.49655 .2744 L
.4954 .27439 L
.49434 .27438 L
.49193 .27433 L
.48972 .27427 L
.48475 .27409 L
.48013 .27386 L
.47573 .2736 L
.45516 .27196 L
.43737 .27029 L
.4183 .26825 L
.39972 .26557 L
.37959 .26167 L
.35948 .25705 L
.34173 .25249 L
.32393 .24697 L
.3074 .24024 L
.29116 .23137 L
.28365 .22591 L
.28009 .2227 L
.27733 .21965 L
.2752 .21663 L
.27436 .21508 L
.2736 .21337 L
Mistroke
.27305 .21163 L
.27283 .21065 L
.27274 .21019 L
.27268 .20975 L
.27259 .20891 L
.27256 .20847 L
.27255 .20799 L
.27254 .20754 L
.27256 .20711 L
.27261 .20631 L
.27269 .20556 L
.27282 .20473 L
.27301 .20386 L
.27322 .20304 L
.27433 .19999 L
.27607 .19663 L
.27844 .19299 L
.28895 .17936 L
.2915 .17561 L
.29369 .17168 L
.29438 .1701 L
.29498 .16841 L
.29525 .16745 L
.29544 .16658 L
.29558 .16576 L
.29564 .16535 L
.29568 .16491 L
.29573 .1641 L
.29573 .16364 L
.29572 .16323 L
.29567 .16248 L
.29557 .16169 L
.29543 .161 L
.29526 .16036 L
.29475 .15894 L
.29439 .15817 L
.29399 .15746 L
.2931 .15617 L
.29195 .15484 L
.29047 .15347 L
.28748 .15136 L
.28368 .1494 L
.27959 .14783 L
.27553 .14663 L
.2707 .14554 L
.26796 .14506 L
.26535 .14467 L
.26065 .14413 L
.25813 .14392 L
.25543 .14375 L
Mistroke
.25401 .14367 L
.25251 .14361 L
.25173 .14359 L
.25087 .14356 L
.25008 .14354 L
.24935 .14353 L
.24793 .14351 L
.24715 .1435 L
.24641 .1435 L
.2451 .1435 L
.24435 .14351 L
.24367 .14352 L
.24229 .14354 L
.241 .14357 L
.2386 .14364 L
.23604 .14375 L
.2333 .14389 L
.22378 .1446 L
.21372 .14562 L
.20333 .14696 L
.19387 .14844 L
.18504 .15015 L
.17539 .15253 L
.16533 .1558 L
.15616 .15979 L
.14825 .16432 L
.14042 .17036 L
.13692 .1738 L
.13353 .1778 L
.13073 .18182 L
.12849 .18577 L
.12658 .19 L
.12518 .194 L
.12402 .19857 L
.12356 .20114 L
.12339 .20231 L
.12324 .2036 L
.12312 .20484 L
.12304 .20601 L
.12297 .20724 L
.12295 .20794 L
.12294 .20858 L
.12294 .20971 L
.12296 .21095 L
.12301 .21213 L
.12307 .21322 L
.12318 .21459 L
.12332 .21589 L
.12365 .21831 L
.12412 .22091 L
Mistroke
.12475 .2237 L
.12611 .22834 L
.12768 .23258 L
.13226 .24187 L
.13789 .25029 L
.14493 .25852 L
.15225 .26545 L
.16083 .27216 L
.17772 .28234 L
.19566 .29012 L
.21561 .29646 L
.25125 .30556 L
.28806 .31555 L
.32411 .32482 L
.34232 .32916 L
.36225 .33347 L
.37162 .33519 L
.38177 .33675 L
.39142 .33794 L
.40039 .3388 L
.41049 .33954 L
.42143 .34016 L
.44121 .34102 L
.45011 .34135 L
.45971 .34164 L
.46254 .34171 L
.46524 .34177 L
.46787 .34181 L
.47031 .34185 L
.47263 .34187 L
.47379 .34188 L
.47509 .34189 L
.47638 .34189 L
.47711 .34189 L
.47778 .34189 L
.47901 .34189 L
.48031 .34189 L
.48154 .34188 L
.48289 .34187 L
.48533 .34183 L
.48771 .34179 L
.48991 .34174 L
.4949 .3416 L
.50019 .34139 L
.509 .3409 L
.51864 .3402 L
.53786 .33823 L
.54853 .33678 L
.55821 .3352 L
.57651 .33145 L
Mistroke
.59621 .32636 L
.63388 .31536 L
.67055 .30611 L
.6912 .30214 L
.71084 .2993 L
.74857 .29511 L
.76773 .29266 L
.78854 .28942 L
.79795 .28767 L
.80788 .28548 L
.81699 .28298 L
.82518 .28011 L
.83394 .27615 L
.84291 .27077 L
.8502 .26511 L
.85716 .25823 L
.86266 .2514 L
.86765 .2436 L
.87145 .23595 L
.87414 .2288 L
.87529 .22487 L
.87628 .22063 L
.87701 .21658 L
.87749 .21285 L
.8777 .21055 L
.87785 .2084 L
.8779 .20727 L
.87795 .20602 L
.87796 .20539 L
.87797 .20471 L
.87798 .20407 L
.87798 .20348 L
.87798 .20232 L
.87796 .20125 L
.87793 .20007 L
.87791 .1994 L
.87788 .1988 L
.87775 .19648 L
.87758 .19428 L
.8773 .19151 L
.87691 .18844 L
.87586 .18219 L
.87456 .17637 L
.87305 .17097 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`T0=ol007go0P0;o`801Ol00`3oo`19o`<0
2Ol203Go002Ao`030?oo04Oo0P0>o`<0<_l0097o00<0ool0A_l00`3oo`0Ao`030?oo02oo002Ao`03
0?oo04Go00<0ool04ol00`3oo`0^o`00A?lA03co00<0ool0A?l00`3oo`0Eo`030?oo02go000lo`P0
4Ol303Wo00<0ool0@ol00`3oo`0Go`030?oo02co000io`<07?l203Oo00<0ool0@_l00`3oo`0Io`03
0?oo02_o000go`808?l203Ko00<0ool0@_l00`3oo`0Io`030?oo02_o000eo`808ol00`3oo`0do`03
0?oo04;o00<0ool06_l00`3oo`0Zo`00<ol202Go00<0ool0=?l00`3oo`12o`030?oo01[o00<0ool0
:_l003;o00<0ool09Ol00`3oo`0do`030?oo04;o00<0ool06_l00`3oo`0Zo`00<Ol00`3oo`0Vo`03
0?oo03Co00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000`o`030?oo02Ko00<0ool08Ol200Co0P03o`@0
1Ol00`3oo`11o`030?oo01co00<0ool0:Ol002oo00<0ool09_l00`3oo`0Qo`040?oo00So00<0ool0
1_l00`3oo`11o`030?oo01co00<0ool0:Ol002ko00<0ool09ol00`3oo`0Qo`040?oo00Wo00<0ool0
1Ol2047o00<0ool07Ol00`3oo`0Yo`00;_l00`3oo`0Vo`030?oo01co1002o`040?oo00[o00<0ool0
1?l00`3oo`10o`030?oo01go00<0ool0:Ol002go00<0ool09ol00`3oo`0Ro`040?oo00So00@0ool0
1Ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool09_l00`3oo`0To`802_l200Ko00<0ool0
@?l00`3oo`0No`030?oo02So000/o`030?oo02Ko00<0ool0>Ol00`3oo`0oo`030?oo01oo00<0ool0
:?l002co00<0ool09Ol00`3oo`0jo`030?oo03ko00<0ool08?l00`3oo`0Xo`00;?l00`3oo`0Uo`03
0?oo03[o00<0ool0?Ol00`3oo`0Qo`030?oo02So0007o`D00_l200Co0P0Fo`030?oo00[o1@0Fo`03
0?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803Ol00`3oo`0Co`D02Ol00`3oo`0Fo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Go00<0ool03?l00`3oo`0Fo`030?oo00So00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Co`030?oo00co00<0ool05_l200Wo00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Ko0P05o`@01?l00`3oo`0Go`040?ooo`@0
0_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0:o`806Ol00`3oo`09o`030?oo01So00<0ool0
2?l200?o0009o`030?oo00So00<0ool05_l00`3oo`0;o`030?oo01So00<0ool01_l0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00Oo0P0Ko`030?oo00So00<0ool06Ol00`3oo`08o`03
0?oo00;o0008o`802_l301Ko00<0ool02_l201_o0P07o`802_l302;o00<0ool08_l200[o0`06o`03
0?oo01[o0P0:o`030?oo01So0P0:o`<00_l002go00<0ool0:Ol203Ko00<0ool0=Ol202Wo00<0ool0
:Ol002ko00<0ool0:_l203Co00<0ool0<_l302[o00<0ool0:_l002ko00<0ool0;?l203;o00<0ool0
<?l202go00<0ool0:_l002oo00<0ool0;Ol2033o00<0ool0;_l202ko00<0ool0:ol0033o00<0ool0
;_l302go00<0ool0;?l2033o00<0ool0:ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0o`000P04o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l0103oool40003o`3o00Oo00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0So`03
0?oo00?o00<0ool0:Ol00`3oo`02o`D08ol00`3oo`0Ro`801Ol00`3oo`0Yo`030?oo00Co00<0ool0
8ol00`3oo`07o`00<ol00`3oo`0ho`H07Ol00`3oo`0No`@0>Ol00`3oo`0^o`00=?l00`3oo`0mo`D0
6?l00`3oo`0Go`L0??l00`3oo`0_o`00=Ol00`3oo`11o`X03_l00`3oo`0@o`L0@_l00`3oo`0`o`00
=_l00`3oo`1:ob40Aol203?o000go`80F?l00`3oo`1Do`<0=Ol003Wo0P1Fo`030?oo05;o0P0ho`00
>ol205Co00<0ool0COl503[o000mo`80D_l00`3oo`17o`H0?ol003oo101>o`030?oo04;o1@15o`00
@ol404[o00<0ool0>_l804[o0017o`<0Aol00`3oo`0bo`P0D_l004[o0`14o`030?oo02go1@1Jo`00
COl4043o00<0ool0:?l505oo001Ao`D09ol200Co0P03o`@01Ol00`3oo`0So`D0I?l005Ko100Ro`04
0?oo00So00<0ool01_l00`3oo`0Oo`@0JOl005[o0`0Oo`040?oo00Wo00<0ool01Ol201co101]o`00
GOl401_o00@0ool02_l00`3oo`04o`030?oo01So0`1ao`00HOl401Oo00@0ool02?l0103oo`05o`03
0?oo01Go0`1do`00IOl501?o0P0:o`801_l00`3oo`0Co`80Mol006[o0`0To`030?oo00oo101io`00
KOl801co00<0ool01?l;07go001eob<0R?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`04
0?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`00
1Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0
S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87264 .17552 m
.86984 .16709 L
.8659 .15837 L
.86354 .15428 L
.86072 .15019 L
.85484 .14367 L
.85102 .1405 L
.8465 .13751 L
.84186 .13512 L
.83729 .13328 L
.83296 .13192 L
.82797 .13073 L
.82539 .13025 L
.82258 .12983 L
.81988 .12952 L
.8174 .1293 L
.81605 .12921 L
.81478 .12915 L
.81363 .1291 L
.81239 .12907 L
.81104 .12906 L
.81035 .12906 L
.8096 .12907 L
.80828 .1291 L
.80705 .12914 L
.8058 .1292 L
.80463 .12928 L
.80245 .12946 L
.80011 .12972 L
.79761 .13009 L
.79246 .13113 L
.78995 .13181 L
.78727 .13267 L
.78278 .13451 L
.7789 .1366 L
.77545 .13898 L
.77208 .14197 L
.769 .14551 L
.76654 .14916 L
.76298 .15643 L
.76139 .16084 L
.76012 .16498 L
.75765 .17444 L
.75496 .1844 L
.75212 .19249 L
.75014 .19682 L
.74802 .20068 L
.74294 .20777 L
.73734 .21358 L
.7222 .22478 L
Mistroke
.70618 .2338 L
.68843 .24216 L
.67116 .24857 L
.65181 .25397 L
.63264 .25838 L
.6119 .26267 L
.6022 .26441 L
.59191 .26596 L
.57312 .26799 L
.56361 .26872 L
.55334 .2694 L
.53484 .27056 L
.52471 .27116 L
.51942 .27144 L
.51385 .27168 L
.51152 .27176 L
.50901 .27183 L
.50757 .27187 L
.50626 .27189 L
.50492 .27192 L
.50368 .27193 L
.50262 .27194 L
.50149 .27195 L
.50024 .27195 L
.4991 .27195 L
.49791 .27195 L
.49724 .27194 L
.49662 .27194 L
.49548 .27192 L
.49428 .2719 L
.4918 .27185 L
.48945 .27178 L
.48503 .27161 L
.48025 .27136 L
.47503 .27104 L
.45693 .26959 L
.43818 .26789 L
.41813 .26582 L
.39925 .26316 L
.38108 .25983 L
.34466 .25176 L
.32684 .24673 L
.30797 .23955 L
.29945 .23547 L
.29078 .23055 L
.28356 .22548 L
.2782 .22048 L
.2759 .21756 L
.27495 .21604 L
.27411 .21438 L
Mistroke
.27374 .21351 L
.27346 .21271 L
.27322 .21189 L
.27303 .21113 L
.27287 .21024 L
.27281 .20977 L
.27276 .20926 L
.27272 .20878 L
.27271 .20832 L
.2727 .20791 L
.27271 .20747 L
.27276 .20666 L
.27285 .20591 L
.27298 .20508 L
.27317 .20418 L
.27369 .20234 L
.27433 .20065 L
.27595 .19734 L
.27787 .19415 L
.2869 .1814 L
.28915 .17786 L
.29116 .17409 L
.2919 .17238 L
.29252 .1706 L
.29292 .1691 L
.29308 .16826 L
.2932 .16748 L
.29329 .16667 L
.29331 .16622 L
.29333 .16581 L
.29332 .16505 L
.29327 .16426 L
.29318 .16356 L
.29306 .16292 L
.2929 .16222 L
.29268 .16148 L
.29202 .15989 L
.29118 .15844 L
.29025 .15719 L
.28906 .1559 L
.28599 .15342 L
.28396 .15218 L
.28194 .15114 L
.27755 .14935 L
.27296 .14797 L
.26827 .1469 L
.26287 .14602 L
.26007 .14568 L
.25705 .14539 L
.25418 .14517 L
Mistroke
.25291 .1451 L
.25156 .14503 L
.25032 .14498 L
.24895 .14494 L
.24771 .14491 L
.24655 .14489 L
.24519 .14487 L
.24395 .14487 L
.24252 .14488 L
.24117 .1449 L
.24001 .14492 L
.23878 .14495 L
.23658 .14502 L
.23399 .14513 L
.23167 .14525 L
.2267 .14559 L
.22158 .14602 L
.21272 .14695 L
.20334 .14814 L
.19338 .14965 L
.18383 .15144 L
.17505 .15352 L
.16626 .15617 L
.15812 .15932 L
.14974 .16355 L
.14162 .16907 L
.13553 .17462 L
.12991 .18164 L
.12755 .18554 L
.12547 .18987 L
.12401 .19371 L
.12278 .19806 L
.12225 .20058 L
.12188 .2029 L
.12172 .20419 L
.12158 .2056 L
.12153 .20625 L
.12148 .20694 L
.12142 .20818 L
.12138 .20932 L
.12138 .21055 L
.12139 .21171 L
.12143 .21277 L
.1215 .21402 L
.12161 .21537 L
.12174 .21666 L
.12188 .21785 L
.12228 .2204 L
.12274 .22273 L
.12412 .22796 L
Mistroke
.1257 .23249 L
.12773 .23715 L
.13204 .2449 L
.13776 .2528 L
.15156 .26652 L
.16026 .27299 L
.16862 .27818 L
.18558 .28645 L
.20512 .29326 L
.22422 .2982 L
.24418 .30267 L
.27864 .31117 L
.31563 .3206 L
.3512 .3288 L
.36936 .33238 L
.37915 .33393 L
.38977 .33526 L
.40021 .33623 L
.40975 .33688 L
.41934 .33737 L
.42837 .33776 L
.44616 .33843 L
.45629 .33878 L
.46126 .33892 L
.46581 .33903 L
.46813 .33907 L
.47066 .33911 L
.47182 .33912 L
.47306 .33914 L
.47422 .33915 L
.47528 .33915 L
.47648 .33916 L
.47778 .33916 L
.47893 .33916 L
.48014 .33916 L
.48078 .33915 L
.48147 .33915 L
.48271 .33914 L
.48418 .33913 L
.48552 .33911 L
.48822 .33906 L
.49071 .339 L
.49634 .33882 L
.5013 .33861 L
.50656 .33832 L
.51591 .33768 L
.5259 .3368 L
.5437 .33474 L
.56399 .33144 L
.58307 .32723 L
Mistroke
.61855 .31741 L
.65588 .30754 L
.67362 .30369 L
.69314 .30033 L
.70393 .29891 L
.71388 .29783 L
.7335 .2961 L
.75347 .29431 L
.7721 .29226 L
.79097 .28973 L
.80844 .28661 L
.81741 .28438 L
.82688 .28121 L
.83511 .2775 L
.84232 .27327 L
.84954 .26785 L
.85567 .26203 L
.86157 .255 L
.86688 .24689 L
.87067 .23935 L
.87377 .23092 L
.87494 .2266 L
.87547 .2242 L
.8759 .22193 L
.87624 .21979 L
.87655 .21745 L
.87677 .21531 L
.87692 .21331 L
.87699 .21223 L
.87704 .21119 L
.87708 .21026 L
.8771 .20924 L
.87712 .20813 L
.87713 .20751 L
.87713 .20694 L
.87712 .20583 L
.8771 .20478 L
.87707 .20379 L
.87703 .20273 L
.87694 .20083 L
.87679 .19877 L
.87659 .19649 L
.87601 .19164 L
.87516 .18638 L
.873 .17683 L
.87264 .17552 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04[o2`0go`00
TOl00`3oo`18o`<02_l403?o002Ao`030?oo04Ko0P0Ao`030?oo033o002Ao`030?oo04Go00<0ool0
4_l00`3oo`0_o`00A_l=03ko00<0ool0A?l00`3oo`0Do`030?oo02ko000no`P03Ol303_o00<0ool0
@ol00`3oo`0Fo`030?oo02go000jo`@06?l303So00<0ool0@_l00`3oo`0Go`030?oo02go000ho`80
7_l203Oo00<0ool0@_l00`3oo`0Ho`030?oo02co000eo`<08Ol00`3oo`0eo`030?oo04;o00<0ool0
6Ol00`3oo`0[o`00<ol202Co0P0fo`030?oo04;o00<0ool06Ol00`3oo`0[o`00<Ol202Ko0P0fo`03
0?oo04;o00<0ool06_l00`3oo`0Zo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0@Ol00`3oo`0Ko`03
0?oo02[o000`o`030?oo02Ko00<0ool08Ol200Co0P03o`@01Ol00`3oo`11o`030?oo01co00<0ool0
:Ol002oo00<0ool09_l00`3oo`0Qo`040?oo00So00<0ool01_l00`3oo`11o`030?oo01co00<0ool0
:Ol002ko00<0ool09ol00`3oo`0Qo`040?oo00Wo00<0ool01Ol2047o00<0ool07Ol00`3oo`0Yo`00
;Ol00`3oo`0Wo`030?oo01co1002o`040?oo00[o00<0ool01?l00`3oo`10o`030?oo01go00<0ool0
:Ol002go00<0ool09_l00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`10o`030?oo01ko00<0ool0
:?l002co00<0ool09_l00`3oo`0Uo`802_l200Ko00<0ool0@?l00`3oo`0No`030?oo02So000/o`03
0?oo02Ko00<0ool0>Ol00`3oo`0oo`030?oo01oo00<0ool0:?l002co00<0ool09Ol00`3oo`0jo`03
0?oo03ko00<0ool08?l00`3oo`0Xo`00;?l00`3oo`0Uo`030?oo03[o00<0ool0?Ol00`3oo`0Qo`03
0?oo02So0007o`D00_l200Co0P0Fo`030?oo00[o1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool0
8_l200Co0P04o`803Ol00`3oo`0Co`D02Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01Go00<0ool03?l00`3oo`0Fo`030?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`0;o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool0
2_l00`3oo`0Co`030?oo00co00<0ool05_l200Wo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`08o`030?oo01Oo00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o00000ol0000200Co
00<0ool02?l201Ko00<0ool01?l400Co00<0ool06?l00`3oo`0400;o00@0ool02Ol202?o00<0ool0
8Ol0103oo`09o`802Ol201[o00<0ool02Ol00`3oo`0Ho`030?oo00So0P03o`002Ol00`3oo`08o`03
0?oo01Go00<0ool03?l00`3oo`0Io`030?oo00Go00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool0
2Ol00`3oo`07o`030?oo01[o00<0ool02?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<0
5_l00`3oo`0:o`807?l00`3oo`05o`802_l302;o00<0ool08_l200[o0`05o`807?l200[o00<0ool0
6?l200[o0`02o`00;Ol00`3oo`0Yo`80=_l00`3oo`0do`80:_l00`3oo`0Yo`00;Ol00`3oo`0[o`80
=?l00`3oo`0co`030?oo02[o00<0ool0:Ol002go00<0ool0;Ol203;o00<0ool0<Ol202co00<0ool0
:_l002ko00<0ool0;_l2033o00<0ool0;_l302ko00<0ool0:_l002oo00<0ool0;ol402co00<0ool0
;?l2033o00<0ool0:ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
0P07o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`D02?l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00D0
00?o0?l01ol00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02;o00<0ool01?l00`3oo`0Yo`030?oo00Ko
0`0Qo`030?oo01ko1@06o`030?oo02Wo00<0ool01?l00`3oo`0So`030?oo00Oo000bo`80??l901So
00<0ool06_l403go00<0ool0;_l003Co00<0ool0@_l501?o00<0ool04ol7043o00<0ool0;ol003Go
00<0ool0A_l:00Wo00<0ool01Ol>04Ko00<0ool0<?l003Ko00<0ool0ColA05;o0P0co`00=ol205So
00<0ool0EOl203Go000io`80E_l00`3oo`1Bo`<0=ol003_o0P1Do`030?oo053o0P0jo`00?Ol205;o
00<0ool0COl303co000oo`80D?l00`3oo`13o`X0?ol0047o101<o`030?oo03So2`19o`00AOl504Oo
00<0ool0;_l:05Co001:o`D0@_l00`3oo`0Yo`D0G_l004oo1@0mo`030?oo02Go101So`00E?l402Go
0P04o`800ol400Go00<0ool08Ol406Oo001Ho`<08Ol0103oo`08o`030?oo00Ko00<0ool07_l306_o
001Ko`<07_l0103oo`09o`030?oo00Go0P0Lo`<0K_l005ko100Jo`040?oo00[o00<0ool01?l00`3o
o`0Ho`<0LOl006;o1@0Eo`040?oo00So00@0ool01Ol00`3oo`0Eo`<0M?l006Oo1@0Ao`802_l200Ko
00<0ool04Ol407Oo001/o`H07ol00`3oo`0;o`H0Nol007;o3P0Ao`h0POl0083o4P2>o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103o
o`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go
00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co
001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o
003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87223 .18007 m
.86964 .17156 L
.86597 .1627 L
.86376 .15852 L
.8611 .15433 L
.8555 .1476 L
.85184 .14429 L
.84746 .14114 L
.84327 .13874 L
.83848 .13658 L
.8339 .13497 L
.82859 .13354 L
.82619 .13302 L
.82356 .13254 L
.82104 .13216 L
.81872 .13188 L
.81605 .13162 L
.81457 .13151 L
.81318 .13143 L
.81188 .13138 L
.81069 .13134 L
.80938 .13132 L
.808 .13131 L
.80661 .13133 L
.80583 .13135 L
.80512 .13138 L
.80374 .13144 L
.80244 .13152 L
.79956 .13177 L
.79795 .13196 L
.79648 .13216 L
.79186 .13302 L
.78932 .13364 L
.787 .13432 L
.78257 .13598 L
.77818 .13821 L
.77451 .14068 L
.77145 .14332 L
.7685 .14659 L
.76578 .15048 L
.76338 .15497 L
.76149 .15951 L
.75864 .16856 L
.75403 .18544 L
.75251 .18986 L
.75062 .19441 L
.74641 .20204 L
.74362 .20586 L
.74065 .20927 L
.73433 .21513 L
Mistroke
.71779 .22615 L
.68523 .24175 L
.66795 .24767 L
.64858 .25272 L
.62952 .25697 L
.60901 .26111 L
.5994 .26274 L
.58918 .26415 L
.58033 .2651 L
.57046 .26592 L
.54927 .26726 L
.53007 .26842 L
.5207 .26893 L
.51581 .26916 L
.51314 .26926 L
.51065 .26934 L
.5083 .2694 L
.50615 .26945 L
.50499 .26947 L
.50377 .26948 L
.50254 .26949 L
.50184 .26949 L
.5012 .26949 L
.50005 .26949 L
.49898 .26949 L
.49801 .26948 L
.49696 .26947 L
.49582 .26946 L
.49461 .26944 L
.49244 .26939 L
.48998 .26931 L
.48765 .26923 L
.48326 .26902 L
.47333 .2684 L
.4554 .26693 L
.41941 .26357 L
.40113 .26117 L
.38112 .25775 L
.3457 .25057 L
.3285 .2461 L
.30966 .23943 L
.29179 .23072 L
.28397 .22558 L
.28026 .2225 L
.27735 .21948 L
.27527 .21663 L
.27436 .21498 L
.27371 .21344 L
.27329 .21213 L
.27311 .21138 L
Mistroke
.27298 .21067 L
.27288 .20992 L
.27282 .20923 L
.27279 .20843 L
.27281 .20767 L
.27287 .20683 L
.27296 .20606 L
.27311 .20517 L
.27329 .20433 L
.27372 .20282 L
.27432 .20116 L
.27614 .19732 L
.27826 .1937 L
.28285 .18669 L
.28536 .18277 L
.2875 .17905 L
.28914 .17562 L
.28982 .17383 L
.29038 .17196 L
.29056 .17121 L
.29072 .1704 L
.29086 .16953 L
.29094 .16872 L
.29099 .16797 L
.29099 .1672 L
.29097 .16654 L
.29091 .16584 L
.2908 .16505 L
.29071 .1646 L
.29062 .1642 L
.29043 .16348 L
.29017 .16271 L
.2896 .16139 L
.28883 .16005 L
.28781 .15865 L
.2865 .15723 L
.28359 .15486 L
.28003 .15278 L
.27613 .15108 L
.27226 .14978 L
.26771 .14862 L
.26503 .14807 L
.26206 .14757 L
.25939 .1472 L
.25654 .14688 L
.25394 .14664 L
.25107 .14645 L
.24981 .14638 L
.24846 .14632 L
.24728 .14628 L
Mistroke
.24603 .14625 L
.24528 .14623 L
.24446 .14622 L
.24301 .1462 L
.2422 .1462 L
.24132 .1462 L
.2405 .14621 L
.23975 .14621 L
.23831 .14624 L
.2375 .14625 L
.23675 .14627 L
.23394 .14637 L
.23123 .1465 L
.22882 .14664 L
.22351 .14703 L
.21369 .148 L
.20506 .14906 L
.19525 .15047 L
.18609 .15204 L
.17704 .15394 L
.16872 .15614 L
.15956 .15926 L
.15012 .16356 L
.14203 .16856 L
.13537 .17409 L
.12922 .18117 L
.12664 .18512 L
.12424 .18974 L
.12254 .194 L
.12178 .19639 L
.12117 .19866 L
.1207 .20079 L
.12035 .20275 L
.12005 .20496 L
.11992 .20618 L
.11983 .20733 L
.11976 .20849 L
.11973 .20914 L
.11971 .20976 L
.11969 .211 L
.11969 .21171 L
.1197 .21235 L
.11973 .21349 L
.11979 .21473 L
.11988 .21591 L
.11998 .217 L
.12031 .21968 L
.12072 .2221 L
.12199 .22749 L
.1235 .23211 L
Mistroke
.12546 .23689 L
.12987 .24505 L
.13572 .25319 L
.1432 .26123 L
.15857 .27331 L
.16794 .27889 L
.17681 .28331 L
.19458 .2902 L
.21467 .29572 L
.25362 .30364 L
.28909 .31207 L
.32367 .32037 L
.3428 .32465 L
.36024 .32831 L
.37764 .3313 L
.38657 .33245 L
.39648 .33342 L
.40641 .3341 L
.41545 .33455 L
.43564 .3353 L
.44534 .33564 L
.45562 .33599 L
.46443 .33624 L
.46686 .33629 L
.46947 .33634 L
.47195 .33638 L
.47421 .33641 L
.47531 .33642 L
.47654 .33642 L
.47781 .33643 L
.47902 .33643 L
.48014 .33643 L
.48117 .33642 L
.4823 .33642 L
.48349 .33641 L
.4847 .33639 L
.486 .33638 L
.48834 .33633 L
.49113 .33627 L
.49367 .33619 L
.49824 .33602 L
.5032 .33579 L
.51218 .33522 L
.52143 .33448 L
.53151 .33348 L
.55161 .33087 L
.56995 .32757 L
.58951 .32298 L
.62724 .31264 L
.64595 .30788 L
Mistroke
.66364 .30391 L
.68224 .30047 L
.69999 .29806 L
.70949 .29714 L
.71978 .29636 L
.73783 .29524 L
.7569 .2939 L
.77783 .29201 L
.78737 .29099 L
.79749 .28973 L
.80691 .28826 L
.81533 .28652 L
.82428 .28399 L
.83225 .2809 L
.84053 .27655 L
.84864 .27081 L
.85502 .26493 L
.86102 .25789 L
.86577 .25079 L
.86932 .24404 L
.87092 .24032 L
.87238 .23631 L
.87345 .23282 L
.8744 .22899 L
.87488 .22663 L
.87526 .22444 L
.87556 .22237 L
.87582 .22007 L
.87601 .21799 L
.87614 .21602 L
.87619 .21492 L
.87623 .21392 L
.87624 .21336 L
.87626 .21275 L
.87627 .21164 L
.87628 .21066 L
.87627 .20977 L
.87626 .20877 L
.87623 .20772 L
.8762 .2068 L
.87616 .20594 L
.87606 .20396 L
.87593 .20216 L
.87579 .20051 L
.87538 .19676 L
.87482 .19269 L
.87405 .18823 L
.87223 .18007 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04go100ko`00
TOl00`3oo`19o`@01?l603Go002Ao`030?oo04Ko0`0>o`80<ol0097o00<0ool0A_l00`3oo`0@o`03
0?oo033o0019o`H0@_l00`3oo`14o`805?l00`3oo`0_o`00@Ol800Ko1P0lo`030?oo04?o00<0ool0
5Ol00`3oo`0^o`00>_l701Co0P0jo`030?oo04?o00<0ool05_l00`3oo`0]o`00>?l201go0P0ho`03
0?oo04?o00<0ool05ol00`3oo`0/o`00=Ol3023o0P0go`030?oo04;o00<0ool06?l00`3oo`0/o`00
<ol202Co00<0ool0=Ol00`3oo`11o`030?oo01[o00<0ool0:ol0037o0P0Vo`030?oo03Go00<0ool0
@Ol00`3oo`0Jo`030?oo02_o000`o`030?oo02Ko00<0ool0=Ol00`3oo`11o`030?oo01_o00<0ool0
:_l0033o00<0ool09Ol202?o0P04o`800ol400Go00<0ool0@Ol00`3oo`0Ko`030?oo02[o000_o`03
0?oo02Ko00<0ool08Ol0103oo`08o`030?oo00Ko00<0ool0@?l00`3oo`0Mo`030?oo02Wo000^o`03
0?oo02Ko00<0ool08_l0103oo`09o`030?oo00Go0P11o`030?oo01go00<0ool0:Ol002go00<0ool0
9ol00`3oo`0Lo`@00_l0103oo`0:o`030?oo00Co00<0ool0@?l00`3oo`0Mo`030?oo02Wo000]o`03
0?oo02Oo00<0ool08_l0103oo`08o`040?oo00Go00<0ool0@?l00`3oo`0Mo`030?oo02Wo000/o`03
0?oo02Oo00<0ool09?l200[o0P06o`030?oo03oo00<0ool07_l00`3oo`0Yo`00;?l00`3oo`0Vo`03
0?oo03Wo00<0ool0?_l00`3oo`0Oo`030?oo02Wo000/o`030?oo02Go00<0ool0>_l00`3oo`0no`03
0?oo01oo00<0ool0:Ol002_o00<0ool09_l00`3oo`0jo`030?oo03go00<0ool08Ol00`3oo`0Xo`00
1ol500;o0P04o`805Ol00`3oo`0;o`D05_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200go00<0ool04ol500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Do`03
0?oo00go00<0ool05_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
2ol00`3oo`0Fo`030?oo00Wo00<0ool06?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
4_l00`3oo`0=o`030?oo01Ko0P09o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
2?l00`3oo`0Go`030?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Fo`030?oo00Co1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol200Wo0P0Jo`030?oo00So00<0ool06Ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Eo`03
0?oo00co00<0ool06?l00`3oo`06o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool0
1_l201co00<0ool02?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05Ol00`3oo`0;o`80
6ol200Oo0P0:o`<08_l00`3oo`0Ro`802_l300Go00<0ool06ol200[o00<0ool06?l200[o0`02o`00
;?l00`3oo`0Zo`80=_l00`3oo`0do`80:_l00`3oo`0Yo`00;Ol00`3oo`0[o`80=?l00`3oo`0bo`80
;?l00`3oo`0Yo`00;Ol00`3oo`0]o`<0<Ol00`3oo`0`o`80;Ol00`3oo`0Zo`00;_l00`3oo`0_o`80
;ol00`3oo`0/o`@0;ol00`3oo`0Zo`00;_l00`3oo`0ao`@0:ol00`3oo`0Xo`@0<ol00`3oo`0Zo`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00?o1@07o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`000`03o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01ol000co00<0ool08_l00`3oo`04o`030?oo02Wo00<0ool02?l501go00<0ool0
7Ol500Oo00<0ool0:Ol00`3oo`05o`030?oo02;o00<0ool01ol003;o00<0ool0?ol501So00<0ool0
5_l703oo00<0ool0;Ol003?o00<0ool0@ol8013o00<0ool03Ol904Go00<0ool0;_l003Co0P1;ob00
COl00`3oo`0_o`00=_l00`3oo`1Ho`030?oo05So0P0bo`00=ol00`3oo`1Go`030?oo05Ko0P0do`00
>?l205Oo00<0ool0E?l203Ko000jo`80EOl00`3oo`1Bo`80>?l003co0P1Co`030?oo04go1@0jo`00
?_l404oo00<0ool0A?l903oo0012o`H0BOl00`3oo`0fo`h0B?l004So1@14o`030?oo02_o2`1Fo`00
COl4043o00<0ool09_l5067o001Ao`<0?Ol00`3oo`0Ro`@0I_l005Co0`0Vo`801?l200?o1005o`03
0?oo023o0P1Zo`00Eol4027o00@0ool02?l00`3oo`06o`030?oo01co101/o`00Fol501co00@0ool0
2Ol00`3oo`05o`806?l5073o001Po`D05ol0103oo`0:o`030?oo00Co00<0ool04ol407Go001Uo`@0
4ol0103oo`08o`040?oo00Go00<0ool04Ol207Wo001Yo`D03ol200[o0P06o`030?oo00go101ko`00
K_l?01Co00@0oooo301oo`00OOlH08_o002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87181 .18462 m
.86944 .17603 L
.86604 .16704 L
.86397 .16277 L
.86147 .15847 L
.85616 .15152 L
.85265 .14807 L
.84843 .14476 L
.84435 .14221 L
.83967 .13988 L
.83516 .1381 L
.8299 .13648 L
.82489 .1353 L
.82004 .13445 L
.81732 .13409 L
.81476 .13382 L
.81331 .1337 L
.81195 .13361 L
.81037 .13352 L
.80968 .13349 L
.80893 .13347 L
.80764 .13344 L
.80628 .13342 L
.80502 .13343 L
.80384 .13345 L
.80246 .1335 L
.80121 .13356 L
.79977 .13366 L
.79841 .13377 L
.79609 .13402 L
.79367 .13436 L
.78944 .13519 L
.7851 .13639 L
.78064 .13811 L
.77666 .14018 L
.77288 .14279 L
.77001 .14536 L
.76724 .14851 L
.76467 .15229 L
.76235 .1567 L
.75908 .16537 L
.754 .18354 L
.75243 .18834 L
.7505 .19321 L
.74856 .19721 L
.74615 .20127 L
.74082 .20806 L
.73481 .21368 L
.71852 .22448 L
.70059 .23354 L
Mistroke
.68347 .24069 L
.66612 .24629 L
.64923 .25053 L
.61231 .25839 L
.60268 .26009 L
.59222 .26161 L
.58219 .2627 L
.57281 .26346 L
.55252 .26467 L
.53365 .26577 L
.52449 .2663 L
.51594 .26671 L
.51125 .26688 L
.50861 .26695 L
.50735 .26698 L
.50617 .267 L
.5051 .26701 L
.50412 .26702 L
.50303 .26703 L
.50185 .26704 L
.50078 .26704 L
.49962 .26703 L
.4984 .26702 L
.49726 .26701 L
.49604 .26699 L
.49475 .26697 L
.49243 .26691 L
.48992 .26683 L
.48716 .26672 L
.47755 .26617 L
.45949 .26477 L
.43945 .26307 L
.42181 .26146 L
.4035 .25925 L
.38558 .25649 L
.34953 .25002 L
.33191 .24597 L
.31279 .23989 L
.30407 .23639 L
.29488 .23206 L
.28697 .22754 L
.28076 .22298 L
.27811 .22047 L
.27583 .21769 L
.2748 .21607 L
.27402 .21451 L
.27349 .21308 L
.27325 .21227 L
.27308 .21151 L
.27295 .21077 L
Mistroke
.27286 .20996 L
.27281 .20919 L
.27281 .20848 L
.27284 .20765 L
.27288 .20718 L
.27293 .20674 L
.27305 .20592 L
.27322 .20505 L
.27373 .20312 L
.27438 .20127 L
.27599 .19774 L
.28367 .18448 L
.28562 .18075 L
.28654 .17871 L
.2873 .17679 L
.28787 .17501 L
.28812 .17409 L
.28834 .17311 L
.28849 .17229 L
.28861 .1714 L
.28869 .1706 L
.28873 .16985 L
.28873 .16905 L
.28869 .16819 L
.28865 .16776 L
.2886 .1673 L
.28846 .16648 L
.28828 .16568 L
.28807 .16493 L
.28756 .16356 L
.28675 .162 L
.28583 .16062 L
.28453 .15909 L
.28311 .15774 L
.27991 .15542 L
.27567 .15321 L
.27092 .15143 L
.26824 .15064 L
.26527 .14991 L
.26232 .1493 L
.25958 .14884 L
.25674 .14845 L
.2541 .14815 L
.25098 .14788 L
.24936 .14776 L
.24759 .14767 L
.24604 .1476 L
.2446 .14755 L
.24381 .14753 L
.24308 .14751 L
Mistroke
.24223 .1475 L
.24145 .14749 L
.24014 .14748 L
.23869 .14749 L
.23797 .14749 L
.23719 .1475 L
.2358 .14753 L
.23442 .14757 L
.23311 .14761 L
.23069 .14771 L
.22533 .14804 L
.21969 .1485 L
.20961 .14959 L
.20097 .15073 L
.19118 .15222 L
.18219 .15383 L
.17388 .15562 L
.16467 .15814 L
.15549 .16142 L
.14619 .16589 L
.13807 .17125 L
.13159 .17711 L
.12846 .18073 L
.12548 .18499 L
.1231 .18925 L
.12125 .19346 L
.11991 .19749 L
.11934 .19962 L
.11883 .20198 L
.11841 .20446 L
.11824 .20583 L
.1181 .20711 L
.118 .20832 L
.11794 .20942 L
.11789 .2107 L
.11787 .21191 L
.11788 .21309 L
.11789 .21377 L
.11791 .2144 L
.11798 .21567 L
.11802 .21639 L
.11808 .21704 L
.11833 .21947 L
.11867 .22178 L
.11917 .22443 L
.11974 .22688 L
.12141 .23236 L
.1233 .23714 L
.12577 .24215 L
.13122 .25069 L
Mistroke
.13757 .25826 L
.14414 .26452 L
.15964 .2756 L
.16909 .28073 L
.17821 .28486 L
.18806 .28858 L
.19759 .29156 L
.2158 .29594 L
.25275 .30242 L
.28943 .31036 L
.32507 .31856 L
.3591 .32573 L
.37671 .32877 L
.38684 .33005 L
.39139 .33051 L
.39627 .33093 L
.40118 .33127 L
.40565 .33153 L
.41441 .33191 L
.42461 .33224 L
.43418 .33252 L
.45156 .33308 L
.46146 .3334 L
.46632 .33352 L
.47078 .33362 L
.47305 .33365 L
.47425 .33367 L
.47553 .33368 L
.47665 .33369 L
.47788 .33369 L
.479 .3337 L
.48006 .3337 L
.48128 .33369 L
.4819 .33369 L
.48257 .33369 L
.48377 .33368 L
.48488 .33367 L
.4874 .33363 L
.48882 .3336 L
.4901 .33357 L
.49518 .33342 L
.5007 .33319 L
.51069 .3326 L
.51986 .33188 L
.52984 .33091 L
.54972 .32843 L
.56786 .32531 L
.58725 .32095 L
.62472 .311 L
.64324 .30645 L
Mistroke
.66074 .30265 L
.67916 .29934 L
.68831 .29805 L
.69677 .29709 L
.7062 .29628 L
.71117 .29595 L
.71643 .29567 L
.73437 .29494 L
.75369 .29409 L
.77136 .29302 L
.79028 .29157 L
.80055 .29053 L
.80512 .28995 L
.81012 .28918 L
.81901 .28735 L
.82377 .28605 L
.82814 .28459 L
.83568 .28138 L
.84331 .277 L
.85032 .27166 L
.85711 .26492 L
.86308 .25715 L
.86753 .24952 L
.86935 .24565 L
.87104 .24141 L
.87228 .23764 L
.87335 .23361 L
.87412 .22994 L
.87445 .22797 L
.87475 .22584 L
.87497 .22386 L
.87514 .22204 L
.87527 .22019 L
.87532 .21927 L
.87535 .21842 L
.87539 .2174 L
.87541 .21629 L
.87543 .21531 L
.87543 .21426 L
.87542 .21332 L
.87541 .21244 L
.87539 .21146 L
.87535 .21041 L
.87531 .20928 L
.87524 .20805 L
.87509 .20579 L
.87469 .20147 L
.87418 .19729 L
.87348 .19279 L
.87181 .18462 L
Mistroke
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo04Wo2`0ho`00TOl00`3oo`17o`802ol303Go002Ao`030?oo04Go0P0@o`80<ol0097o00<0ool0
A?l00`3oo`0Bo`030?oo033o0013oa00?_l00`3oo`14o`030?oo01?o00<0ool0;ol003co1`0@o`<0
>ol00`3oo`13o`030?oo01Go00<0ool0;_l003Wo0`0Jo`80>Ol00`3oo`12o`030?oo01Oo00<0ool0
;Ol003Go100No`80>?l00`3oo`11o`030?oo01Wo00<0ool0;?l003?o0P0So`030?oo03Ko00<0ool0
@Ol00`3oo`0Io`030?oo02co000bo`030?oo02?o00<0ool0=_l00`3oo`11o`030?oo01[o00<0ool0
:ol0033o0P0Vo`030?oo03Ko00<0ool0@Ol00`3oo`0Jo`030?oo02_o000_o`030?oo02Ko00<0ool0
8_l200Co0P03o`@01Ol00`3oo`10o`030?oo01co00<0ool0:_l002oo00<0ool09_l00`3oo`0Qo`04
0?oo00So00<0ool01_l00`3oo`10o`030?oo01co00<0ool0:_l002ko00<0ool09_l00`3oo`0Ro`04
0?oo00Wo00<0ool01Ol2047o00<0ool07?l00`3oo`0Zo`00;Ol00`3oo`0Wo`030?oo01co1002o`04
0?oo00[o00<0ool01?l00`3oo`10o`030?oo01co00<0ool0:_l002co00<0ool09ol00`3oo`0So`04
0?oo00So00@0ool01Ol00`3oo`10o`030?oo01co00<0ool0:_l002co00<0ool09ol00`3oo`0To`80
2_l200Ko00<0ool0?ol00`3oo`0No`030?oo02Wo000[o`030?oo02Oo00<0ool0>Ol00`3oo`0oo`03
0?oo01ko00<0ool0:Ol002_o00<0ool09_l00`3oo`0jo`030?oo03ko00<0ool07ol00`3oo`0Yo`00
:ol00`3oo`0Vo`030?oo03[o00<0ool0?Ol00`3oo`0Po`030?oo02Wo0007o`D00_l200Co0P0Eo`03
0?oo00_o1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`803Ol00`3oo`0Co`D0
2?l00`3oo`0Go`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Co00<0ool03Ol00`3oo`0Fo`03
0?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0:o`806?l00`3oo`08o`03
0?oo01Wo00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01;o00<0ool03Ol00`3oo`0Fo`80
2Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Oo00<0ool06?l00`3oo`08o`03
0?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805Ol00`3oo`05o`@01?l00`3o
o`0Go`040?ooo`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P08o`806ol00`3oo`08o`03
0?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05?l00`3oo`0=o`030?oo01So0P07o`04
0?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool0
6Ol00`3oo`08o`030?oo00;o0008o`802_l301Co00<0ool03?l201co0P06o`802_l302;o00<0ool0
8_l200[o0`04o`807Ol200[o00<0ool06?l200[o0`02o`00;?l00`3oo`0[o`80=Ol00`3oo`0co`80
:ol00`3oo`0Yo`00;?l00`3oo`0]o`80<ol00`3oo`0ao`80;Ol00`3oo`0Yo`00;?l00`3oo`0_o`<0
<?l00`3oo`0_o`80;ol00`3oo`0Yo`00;Ol00`3oo`0ao`80;_l00`3oo`0/o`<0<?l00`3oo`0Zo`00
;_l00`3oo`0bo`@0:_l00`3oo`0Wo`D0<ol00`3oo`0Zo`003?oo00`02Ol000co00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko
00<0ool01_l00`3oo`04o`D01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko
00<0ool01_l00`3oo`03o`D01ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo027o
00<0ool01Ol00`3oo`0Yo`030?oo00Wo2P0Go`030?oo01[o0`0<o`030?oo02Wo00<0ool01Ol00`3o
o`0Ro`030?oo00Oo000ao`030?oo04Ko2P0=o`030?oo01;o2012o`030?oo02go000bo`030?oo04oo
2`02o`030?l00180BOl00`3oo`0^o`00<ol00`3oo`1Io`D0Fol00`3oo`0^o`00=?l205_o00<0ool0
FOl2037o000fo`030?oo05So00<0ool0Eol203?o000go`<0Eol00`3oo`1Eo`80=Ol003[o0`1Do`03
0?oo05Co00<0ool0=Ol003go0P1Bo`030?oo04oo1@0ho`00?ol404ko00<0ool0@ol<03go0013o`D0
BOl00`3oo`0boa40BOl004So1@14o`030?oo02[o201Jo`00COl503oo00<0ool09Ol506;o001Bo`@0
>ol00`3oo`0Qo`@0Iol005Ko1@0Ro`801?l200?o1005o`030?oo01go101[o`00Fol501co00@0ool0
2?l00`3oo`06o`030?oo01Wo101_o`00H?l501Oo00@0ool02Ol00`3oo`05o`805ol307?o001Uo`@0
4ol0103oo`0:o`030?oo00Co00<0ool04_l407Ko001Yo`D03_l0103oo`08o`040?oo00Go00<0ool0
3Ol507[o001^o`L02?l200[o0P06o`030?oo00Oo1P1oo`00MOlV08Go002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`03
0?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8714 .18917 m
.86924 .18051 L
.86611 .17137 L
.86418 .16702 L
.86184 .16261 L
.85682 .15544 L
.85346 .15186 L
.8494 .14838 L
.84544 .14568 L
.84086 .14318 L
.83642 .14123 L
.83121 .13942 L
.82622 .13806 L
.82137 .13703 L
.81605 .1362 L
.813 .13586 L
.81154 .13572 L
.81016 .13562 L
.80884 .13554 L
.80742 .13547 L
.80613 .13543 L
.80495 .1354 L
.80366 .13539 L
.80226 .13541 L
.80155 .13542 L
.8008 .13544 L
.79945 .1355 L
.79813 .13558 L
.79691 .13567 L
.7942 .13594 L
.79186 .13627 L
.78942 .1367 L
.78536 .13767 L
.78114 .13908 L
.77701 .14097 L
.77308 .14341 L
.77008 .14584 L
.76718 .14886 L
.76448 .15251 L
.76205 .1568 L
.75863 .1653 L
.75348 .18324 L
.75192 .188 L
.75001 .19281 L
.74809 .19676 L
.74571 .20077 L
.74042 .20744 L
.73444 .21294 L
.72628 .21871 L
.71817 .22342 L
Mistroke
.70071 .23196 L
.68449 .23858 L
.66849 .24372 L
.65062 .24818 L
.63189 .25229 L
.61176 .25639 L
.60236 .25803 L
.59238 .25942 L
.58373 .26034 L
.57408 .26109 L
.55324 .26224 L
.53432 .26329 L
.52511 .26383 L
.52032 .26408 L
.51526 .26431 L
.51086 .26445 L
.50966 .26449 L
.50836 .26452 L
.50724 .26454 L
.50602 .26456 L
.50489 .26457 L
.50385 .26458 L
.50274 .26458 L
.50156 .26458 L
.50046 .26458 L
.49942 .26457 L
.49848 .26456 L
.49747 .26455 L
.49524 .26451 L
.49283 .26444 L
.48848 .26428 L
.48327 .26402 L
.47849 .26372 L
.45824 .26215 L
.43892 .26059 L
.42162 .25909 L
.40471 .25719 L
.38605 .25456 L
.35065 .24892 L
.33341 .24533 L
.31421 .2397 L
.29536 .23203 L
.28679 .22739 L
.28259 .22458 L
.2792 .2218 L
.27665 .21913 L
.27556 .21768 L
.27459 .21612 L
.27393 .21478 L
.27364 .21406 L
Mistroke
.27337 .21327 L
.27317 .21255 L
.27302 .21188 L
.2729 .2112 L
.27282 .21055 L
.27276 .20973 L
.27275 .20898 L
.27275 .20856 L
.27278 .20811 L
.27285 .20727 L
.27295 .20648 L
.2731 .20564 L
.27346 .20412 L
.27402 .2023 L
.27468 .20058 L
.27652 .19656 L
.28005 .18994 L
.28368 .18272 L
.28516 .17895 L
.28579 .17682 L
.28602 .17585 L
.28622 .17485 L
.28636 .17399 L
.28646 .17307 L
.28652 .17223 L
.28654 .17146 L
.28653 .17056 L
.28647 .16975 L
.28642 .16931 L
.28636 .16883 L
.2862 .16797 L
.28598 .16709 L
.28573 .16627 L
.28498 .16447 L
.28401 .1628 L
.2827 .16108 L
.28125 .15958 L
.27966 .15824 L
.27616 .15596 L
.27192 .15395 L
.2675 .15239 L
.26221 .15102 L
.25909 .15041 L
.2562 .14994 L
.25329 .14956 L
.25055 .14928 L
.24767 .14905 L
.24604 .14895 L
.2445 .14887 L
.24318 .14882 L
Mistroke
.24175 .14877 L
.24046 .14875 L
.23926 .14873 L
.23798 .14872 L
.23657 .14873 L
.23511 .14874 L
.23374 .14877 L
.23228 .14881 L
.23069 .14886 L
.22783 .14899 L
.2226 .14933 L
.21673 .14985 L
.21152 .15042 L
.19264 .15305 L
.17627 .15603 L
.1673 .15816 L
.15896 .16068 L
.14994 .16423 L
.1409 .16906 L
.13366 .17436 L
.13008 .17769 L
.12701 .18112 L
.12446 .18454 L
.12203 .18851 L
.12029 .19203 L
.11872 .19599 L
.118 .19827 L
.11743 .2004 L
.11694 .20259 L
.11658 .20467 L
.11627 .20698 L
.11614 .20821 L
.11604 .20953 L
.116 .2102 L
.11597 .21093 L
.11595 .21162 L
.11593 .21226 L
.11593 .21347 L
.11595 .21475 L
.116 .21604 L
.11607 .21722 L
.11619 .21858 L
.11632 .21987 L
.11663 .22217 L
.11708 .22467 L
.11846 .23028 L
.1202 .23539 L
.12493 .24517 L
.13073 .25364 L
.13687 .26053 L
Mistroke
.1443 .26715 L
.1614 .27831 L
.17059 .28279 L
.18097 .28697 L
.19996 .29272 L
.21084 .29513 L
.22079 .29692 L
.25912 .30254 L
.29328 .30951 L
.33002 .31751 L
.34814 .32124 L
.36505 .32451 L
.37301 .32584 L
.3815 .32704 L
.38601 .32756 L
.39098 .32805 L
.39978 .32871 L
.4048 .32898 L
.40954 .32918 L
.41842 .32945 L
.4382 .32991 L
.45537 .33043 L
.46008 .33059 L
.46516 .33074 L
.46763 .3308 L
.46998 .33085 L
.47208 .33089 L
.4744 .33092 L
.4755 .33094 L
.47669 .33095 L
.47794 .33096 L
.47913 .33096 L
.48024 .33096 L
.48142 .33096 L
.48242 .33096 L
.48353 .33095 L
.4842 .33095 L
.48484 .33094 L
.48603 .33093 L
.48871 .33088 L
.49103 .33083 L
.49355 .33076 L
.4987 .33056 L
.50416 .33029 L
.51391 .32963 L
.5325 .32788 L
.54956 .32573 L
.56825 .32256 L
.58529 .31882 L
.62114 .3096 L
Mistroke
.65537 .30184 L
.67355 .29858 L
.68388 .29711 L
.6933 .29609 L
.69793 .2957 L
.70282 .29537 L
.70701 .29514 L
.71161 .29495 L
.71657 .2948 L
.7219 .29468 L
.72695 .2946 L
.73157 .29454 L
.74051 .29441 L
.74504 .29432 L
.75006 .29421 L
.76057 .29389 L
.7703 .29353 L
.77948 .29313 L
.7891 .29265 L
.7941 .29235 L
.79954 .29195 L
.80444 .29149 L
.80897 .29096 L
.81367 .29025 L
.8187 .28928 L
.82306 .28822 L
.82758 .28685 L
.83581 .28352 L
.84293 .27953 L
.84984 .27439 L
.8564 .268 L
.86168 .26134 L
.86568 .25494 L
.86925 .24752 L
.87168 .24052 L
.87272 .23643 L
.87345 .23268 L
.87378 .23057 L
.87406 .22827 L
.87427 .22607 L
.87435 .22503 L
.87442 .22403 L
.87447 .22308 L
.87452 .22204 L
.87455 .2211 L
.87457 .22021 L
.87458 .21926 L
.87459 .21819 L
.87459 .21708 L
.87457 .21602 L
Mistroke
.87455 .21503 L
.87452 .21408 L
.87444 .21229 L
.87433 .21033 L
.87417 .20816 L
.87372 .20352 L
.87307 .19846 L
.8714 .18917 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo04Oo3P0go`00TOl00`3oo`16o`030?oo00co0P0eo`00TOl00`3oo`14o`80
4Ol203?o0015o``0@?l00`3oo`14o`030?oo01;o0P0ao`00?Ol800_o1@0lo`030?oo04?o00<0ool0
5Ol00`3oo`0^o`00>_l301So00<0ool0>Ol00`3oo`12o`030?oo01Ko00<0ool0;_l003Go1@0Lo`80
>Ol00`3oo`11o`030?oo01So00<0ool0;Ol003?o0P0Ro`030?oo03Oo00<0ool0@Ol00`3oo`0Io`03
0?oo02co000ao`809Ol00`3oo`0fo`030?oo047o00<0ool06Ol00`3oo`0/o`00<?l00`3oo`0Uo`03
0?oo03Ko00<0ool0@Ol00`3oo`0Jo`030?oo02_o000_o`030?oo02Ko00<0ool08_l200Co0P03o`@0
1Ol00`3oo`10o`030?oo01_o00<0ool0:ol002ko00<0ool09_l202?o00@0ool02?l00`3oo`06o`03
0?oo043o00<0ool07?l00`3oo`0Zo`00;Ol00`3oo`0Wo`030?oo02;o00@0ool02Ol00`3oo`05o`80
@Ol00`3oo`0Lo`030?oo02[o000]o`030?oo02Oo00<0ool07?l400;o00@0ool02_l00`3oo`04o`03
0?oo043o00<0ool07?l00`3oo`0Zo`00;?l00`3oo`0Wo`030?oo02?o00@0ool02?l0103oo`05o`03
0?oo043o00<0ool07?l00`3oo`0Zo`00;?l00`3oo`0Wo`030?oo02Co0P0:o`801_l00`3oo`0oo`03
0?oo01go00<0ool0:_l002_o00<0ool09ol00`3oo`0io`030?oo03ko00<0ool07ol00`3oo`0Yo`00
:ol00`3oo`0Vo`80>ol00`3oo`0no`030?oo01oo00<0ool0:Ol002_o00<0ool09_l00`3oo`0jo`03
0?oo03go00<0ool08?l00`3oo`0Yo`001ol500;o0P04o`805?l00`3oo`0<o`D05_l00`3oo`09o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200co00<0ool05?l500So00<0ool05ol500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Co`030?oo00ko00<0ool05_l00`3oo`08o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool02_l00`3oo`0Go`030?oo00So00<0ool06Ol00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool04Ol00`3oo`0>o`030?oo01Ko0P09o`040?oo00_o00<0ool0
8?l00`3oo`0Qo`040?oo00_o00<0ool01ol00`3oo`0Ho`030?oo00So00<0ool06Ol00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Do`030?oo00Ko1004o`030?oo01Oo0P02o`@00_l0103o
o`09o`808ol00`3oo`0Qo`040?oo00Wo0P09o`030?oo01Wo00<0ool02?l00`3oo`0Io`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01Co00<0ool03Ol00`3oo`0Io`030?oo00Go00@0ool02Ol00`3o
o`0Ro`030?oo027o00@0ool02Ol00`3oo`05o`<07?l00`3oo`08o`030?oo01Wo00<0ool02?l00`3o
o`02o`002?l200[o0`0Do`030?oo00co0P0Lo`801_l200[o0`0Ro`030?oo02;o0P0:o`<00ol201ko
0P0:o`030?oo01So0P0:o`<00_l002_o00<0ool0;?l203Go00<0ool0<_l202co00<0ool0:Ol002_o
00<0ool0;_l203?o00<0ool0<?l202ko00<0ool0:Ol002co00<0ool0;ol3033o00<0ool0;?l402oo
00<0ool0:_l002co00<0ool0<_l202ko00<0ool0:Ol303?o00<0ool0:_l002go00<0ool0<ol402[o
00<0ool09ol203Ko00<0ool0:_l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Go0P08o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`P0
00?o0?l01ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01?l500Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Po`030?oo00Ko00<0ool0:Ol00`3o
o`0>o`X04_l00`3oo`0Go`L02ol00`3oo`0Yo`030?oo00Ko00<0ool08Ol00`3oo`07o`00<?l00`3o
o`1<o`X02?l00`3oo`09o`h0A_l00`3oo`0/o`00<Ol00`3oo`1Eoa@0Dol00`3oo`0]o`00<_l00`3o
o`1Lo`030?oo05_o00<0ool0;_l003?o00<0ool0Fol00`3oo`1Jo`030?oo02oo000do`80Fol00`3o
o`1Io`030?oo033o000fo`80FOl00`3oo`1Go`80<ol003So0`1Fo`030?oo05Go0P0eo`00>ol405;o
00<0ool0DOl403Oo000oo`@0C_l00`3oo`16o`/0>ol004?o1P18o`030?oo02oo5`16o`00BOl504?o
00<0ool0:Ol605go001>o`D0?_l00`3oo`0To`D0Hol005?o1@0io`030?oo023o101Xo`00F?l5023o
0P04o`800ol400Go00<0ool06ol506co001Mo`@06ol0103oo`08o`030?oo00Ko00<0ool05_l5077o
001Qo`D05_l0103oo`09o`030?oo00Go0P0Bo`D0M_l006Ko1@0Ao`040?oo00[o00<0ool01?l00`3o
o`0=o`@0Nol006_o1`0:o`040?oo00So00@0ool01Ol00`3oo`08o`D0Ool007;o6@03o`h0Q?l008Wo
1@03o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 479},
WindowMargins->{{0, Automatic}, {Automatic, 80}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 109, 4, 285, "Title"],

Cell[CellGroupData[{
Cell[1873, 59, 32, 0, 53, "Section"],
Cell[1908, 61, 345, 9, 52, "Text"],
Cell[2256, 72, 320, 6, 52, "Text"],
Cell[2579, 80, 304, 6, 52, "Text"],
Cell[2886, 88, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3119, 97, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3301, 105, 48, 0, 45, "Subsection"],
Cell[3352, 107, 67, 1, 27, "Input"],
Cell[3422, 110, 103, 2, 27, "Input"],
Cell[3528, 114, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3693, 121, 46, 0, 29, "Subsection"],
Cell[3742, 123, 326, 7, 50, "Text"],
Cell[4071, 132, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4518, 144, 103, 2, 29, "Subsection"],
Cell[4624, 148, 105, 3, 32, "Text"],
Cell[4732, 153, 244, 6, 42, "Input"],
Cell[4979, 161, 123, 2, 27, "Input"],
Cell[5105, 165, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5388, 175, 78, 0, 42, "Subsubsection"],
Cell[5469, 177, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5639, 185, 60, 0, 29, "Subsection"],
Cell[5702, 187, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[5959, 195, 130, 3, 46, "Subsection"],
Cell[6092, 200, 121, 2, 27, "Input"],
Cell[6216, 204, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6397, 212, 31, 0, 33, "Section"],
Cell[6431, 214, 101, 3, 33, "Text"],
Cell[6535, 219, 202, 3, 30, "Input"],
Cell[6740, 224, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6931, 231, 44, 0, 33, "Section"],
Cell[6978, 233, 66, 0, 33, "Text"],
Cell[7047, 235, 2051, 31, 290, "Input"],
Cell[9101, 268, 1957, 31, 290, "Input"],
Cell[11061, 301, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11162, 307, 32, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11219, 311, 80, 1, 30, "Input"],
Cell[11302, 314, 14621, 643, 152, 7832, 555, "GraphicsData", "PostScript", \
"Graphics"],
Cell[25926, 959, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26105, 968, 35, 0, 53, "Section"],
Cell[26143, 970, 176, 4, 52, "Text"],
Cell[26322, 976, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[26808, 987, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[26975, 994, 15199, 660, 152, 7983, 565, "GraphicsData", "PostScript", \
"Graphics"],
Cell[42177, 1656, 15235, 665, 152, 8071, 571, "GraphicsData", "PostScript", \
"Graphics"],
Cell[57415, 2323, 14986, 652, 152, 7850, 558, "GraphicsData", "PostScript", \
"Graphics"],
Cell[72404, 2977, 14980, 653, 152, 7901, 560, "GraphicsData", "PostScript", \
"Graphics"],
Cell[87387, 3632, 14961, 653, 152, 7906, 560, "GraphicsData", "PostScript", \
"Graphics"],
Cell[102351, 4287, 14918, 653, 152, 7912, 561, "GraphicsData", "PostScript", \
"Graphics"],
Cell[117272, 4942, 14729, 644, 152, 7763, 552, "GraphicsData", "PostScript", \
"Graphics"],
Cell[132004, 5588, 14784, 650, 152, 7891, 559, "GraphicsData", "PostScript", \
"Graphics"],
Cell[146791, 6240, 14680, 645, 152, 7832, 555, "GraphicsData", "PostScript", \
"Graphics"],
Cell[161474, 6887, 14534, 640, 152, 7746, 550, "GraphicsData", "PostScript", \
"Graphics"],
Cell[176011, 7529, 14579, 644, 152, 7824, 555, "GraphicsData", "PostScript", \
"Graphics"],
Cell[190593, 8175, 14577, 646, 152, 7854, 557, "GraphicsData", "PostScript", \
"Graphics"],
Cell[205173, 8823, 14573, 648, 152, 7891, 560, "GraphicsData", "PostScript", \
"Graphics"],
Cell[219749, 9473, 14563, 650, 152, 7925, 562, "GraphicsData", "PostScript", \
"Graphics"],
Cell[234315, 10125, 14575, 652, 152, 7957, 564, "GraphicsData", "PostScript", \
"Graphics"],
Cell[248893, 10779, 14583, 654, 152, 8006, 567, "GraphicsData", "PostScript", \
"Graphics"],
Cell[263479, 11435, 14674, 662, 152, 8129, 575, "GraphicsData", "PostScript", \
"Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

