(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    295439,      13177]*)
(*NotebookOutlinePosition[    296094,      13200]*)
(*  CellTagsIndexPosition[    296050,      13196]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Proj.of Normals, Ortho to Schizos, for Major Axis Rescaled, \
Full Population, on MAD Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {1.00000000000000, 0, 0, \(-0.15195084000000\), 
          0.04858573000000, \(-0.06803209000000\), \(-0.23511130000000\), 
          0.01804597000000, 0.02809804000000, 
          0.03525404000000, \(-0.00142181000000\), \(-0.08276291000000\), 
          0.00474653000000, \(-0.07421567000000\), \(-0.03859208000000\), \
\(-0.01290857000000\), 0.02544342000000, 
          0.01821591000000, \(-0.00840569000000\), \(-0.04382198000000\), \
\(-0.01090547000000\), \(-0.03409100000000\), \(-0.01027874000000\), \
\(-0.01428957000000\), 0.02276759000000, \(-0.00005591000000\), 
          0.00019325000000, \(-0.00798167000000\), \(-0.00412871000000\), 
          0.00156547000000, \(-0.00104142000000\), \(-0.00605098000000\), 
          0.00905652000000, \(-0.00692849000000\), 0.00479774000000, 
          0.00158902000000, 0.00656396000000, 0.00799122000000, 
          0.00007162000000, \(-0.00306726000000\), \(-0.00452232000000\), 
          0.00027030000000, 0.00232349000000, 0.00137900000000, 
          0.00553573000000, 
          0.00250358000000, \(-0.00026729000000\), \(-0.00033366000000\), \
\(-0.00403778000000\), 0.00521687000000, 0.00053708000000, 0.00114458000000, 
          0.00011893000000, \(-0.00205813000000\), \(-0.00035760000000\), \
\(-0.00067585000000\), \(-0.00014223000000\), 0.00350017000000, 
          0.00016588000000, 
          0.00012998000000, \(-0.00211975000000\), \(-0.00131863000000\), \
\(-0.00063710000000\), \(-0.00066186000000\), 
          0.00083729000000, \(-0.00021953000000\), 
          0.00039799000000, \(-0.00076859000000\), \(-0.00063800000000\), 
          0.00023349000000, \(-0.00025169000000\), \(-0.00033436000000\), 
          0.00015018000000, \(-0.00069026000000\), 
          0.00001612000000, \(-0.00063123000000\), 0.00066879000000, 
          0.00041001000000, 
          0.00057962000000, \(-0.00041244000000\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ {0, \(-0.00000028131843\), \(-0.00000063538197\), 
          0.35643814654014, \(-0.42068476838024\), \(-0.05133483572074\), \
\(-0.19473094269012\), 0.16866731374384, 0.16397887281617, 
          0.15256293915263, \(-0.60035883558650\), \(-0.07169870845208\), \
\(-0.07601675756502\), \(-0.10770683763924\), \(-0.02098345575201\), 
          0.09101389420037, 0.25390568639298, \(-0.02822819372170\), 
          0.07947424272362, \(-0.06169241994187\), 0.05676250326212, 
          0.11537478972116, 
          0.05591698722727, \(-0.06725257465094\), \(-0.01730396614977\), \
\(-0.01681574485966\), 0.00813187424717, \(-0.00389743591353\), 
          0.02030684900706, \(-0.00316052223392\), \(-0.04022484940983\), \
\(-0.04374617922211\), \(-0.04319006324623\), 
          0.09231660273958, \(-0.07447069453870\), \(-0.06309958784042\), \
\(-0.01002074876292\), 
          0.00293276259178, \(-0.08987962971648\), \(-0.01592798602315\), \
\(-0.03920713369387\), 0.08962826598974, 
          0.02425060036933, \(-0.04279281341795\), 0.02162533569188, 
          0.08293388005512, \(-0.06104006501578\), 
          0.03993353327999, \(-0.00003297827507\), 0.01820384392287, 
          0.04521698231243, \(-0.03227248557413\), 0.04903723937822, 
          0.07311112272516, 
          0.04441154260640, \(-0.03833513494025\), \(-0.02144061410800\), 
          0.04198225278679, \(-0.01768938884609\), 0.03621474451128, 
          0.02407768722701, 0.01560571907788, 
          0.02232027290428, \(-0.01732675919322\), \(-0.01104932990803\), 
          0.02326340335531, \(-0.04626953973266\), 
          0.02702584336371, \(-0.03134199117783\), \(-0.01022696479365\), \
\(-0.00318827390243\), 
          0.01271626937148, \(-0.01507931229601\), \(-0.01627188590456\), 
          0.00505656021143, \(-0.03657605946153\), 0.02961500483737, 
          0.00170565993620, \(-0.01754237370301\), \
\(-0.00435296198786\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the (coordinate wise) median", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87536 .15424 m
.87367 .14993 L
.87186 .14604 L
.86939 .14163 L
.86686 .13784 L
.86078 .13093 L
.85694 .12767 L
.85301 .12497 L
.84868 .12261 L
.84381 .12056 L
.83943 .11917 L
.83442 .11802 L
.83154 .11754 L
.82887 .11722 L
.82739 .11708 L
.82576 .11696 L
.82501 .11691 L
.82421 .11688 L
.82345 .11685 L
.82277 .11683 L
.82207 .11682 L
.82133 .11681 L
.82001 .11681 L
.81928 .11682 L
.81859 .11684 L
.81783 .11686 L
.81703 .1169 L
.81557 .11698 L
.8142 .11708 L
.81163 .11733 L
.80917 .11764 L
.80687 .118 L
.80175 .11906 L
.79692 .12041 L
.79196 .12222 L
.78786 .12413 L
.78383 .12645 L
.77748 .13141 L
.77439 .1347 L
.77191 .13797 L
.76958 .14178 L
.76743 .14622 L
.76576 .15059 L
.76444 .15481 L
.76231 .16348 L
.7603 .17329 L
.75795 .1837 L
.75657 .18857 L
.75503 .19303 L
.75146 .20093 L
Mistroke
.7467 .20838 L
.74134 .21466 L
.72648 .22712 L
.69258 .24738 L
.67337 .256 L
.65439 .26232 L
.61639 .27155 L
.59664 .27531 L
.58647 .27686 L
.57541 .27823 L
.55488 .28013 L
.53534 .28162 L
.51427 .28309 L
.50856 .28342 L
.50253 .2837 L
.4974 .28389 L
.49448 .28397 L
.49307 .284 L
.49176 .28402 L
.49049 .28404 L
.48934 .28405 L
.48806 .28406 L
.4867 .28406 L
.48538 .28406 L
.48464 .28406 L
.48396 .28406 L
.48271 .28405 L
.48139 .28404 L
.48026 .28402 L
.47905 .284 L
.47687 .28395 L
.47192 .2838 L
.46622 .28355 L
.46084 .28325 L
.45082 .28256 L
.43107 .28079 L
.41332 .27864 L
.40415 .27717 L
.39554 .2755 L
.37576 .27057 L
.33959 .25877 L
.32228 .25203 L
.30342 .2429 L
.2948 .23773 L
.28608 .23151 L
.27889 .22508 L
.2738 .2189 L
.27183 .21554 L
.27106 .21383 L
.27039 .21195 L
Mistroke
.26998 .21034 L
.26982 .20944 L
.26971 .20862 L
.26964 .20785 L
.26961 .20702 L
.26962 .20624 L
.26966 .20551 L
.26975 .2046 L
.26983 .20409 L
.26991 .20362 L
.27035 .20185 L
.27091 .20024 L
.27166 .19855 L
.27374 .19507 L
.27674 .1913 L
.29191 .17741 L
.29898 .17099 L
.30404 .1652 L
.30579 .16238 L
.30644 .16099 L
.30673 .16019 L
.30696 .15947 L
.30714 .15874 L
.30721 .15838 L
.30727 .15799 L
.30732 .15759 L
.30735 .15723 L
.30737 .15681 L
.30738 .15642 L
.30734 .15574 L
.3073 .15536 L
.30725 .15501 L
.30711 .15437 L
.30693 .15371 L
.30646 .15255 L
.30578 .15134 L
.30498 .15024 L
.3041 .14928 L
.30144 .14706 L
.29827 .14517 L
.29437 .14342 L
.29041 .14205 L
.2854 .14072 L
.28038 .1397 L
.2755 .13896 L
.27092 .13844 L
.26581 .13802 L
.26322 .13787 L
.26175 .13781 L
.26039 .13775 L
Mistroke
.25906 .13771 L
.2576 .13768 L
.2569 .13766 L
.25614 .13765 L
.25534 .13764 L
.25459 .13764 L
.25389 .13763 L
.25313 .13763 L
.25174 .13763 L
.2504 .13765 L
.24915 .13766 L
.24783 .13769 L
.24659 .13772 L
.24428 .13778 L
.24181 .13788 L
.23916 .138 L
.2336 .13833 L
.22782 .13877 L
.21763 .13974 L
.20815 .14087 L
.19805 .14238 L
.18857 .14417 L
.18001 .14622 L
.17052 .14916 L
.1621 .15257 L
.15345 .1572 L
.14521 .16325 L
.13903 .16951 L
.13606 .17341 L
.13367 .17723 L
.13172 .18105 L
.1302 .18476 L
.12893 .18877 L
.12838 .19095 L
.1279 .19329 L
.12751 .19576 L
.12726 .19806 L
.12715 .19935 L
.12708 .20056 L
.12705 .20126 L
.12703 .20189 L
.12702 .20263 L
.12701 .20333 L
.12702 .20449 L
.12706 .20572 L
.12712 .20689 L
.12718 .20795 L
.12729 .2092 L
.12743 .21055 L
.12776 .213 L
Mistroke
.12862 .21757 L
.12985 .22235 L
.13282 .2307 L
.13693 .23921 L
.14225 .24775 L
.15378 .26154 L
.16969 .27498 L
.18681 .28542 L
.22227 .30072 L
.26014 .31457 L
.29787 .3268 L
.33479 .33654 L
.3551 .3415 L
.37371 .34552 L
.39205 .34857 L
.4013 .34971 L
.41145 .35071 L
.43096 .35209 L
.44212 .35267 L
.45246 .35311 L
.45756 .35329 L
.463 .35344 L
.46768 .35354 L
.46896 .35356 L
.47034 .35358 L
.47155 .35359 L
.47284 .3536 L
.47355 .35361 L
.47433 .35361 L
.47571 .35362 L
.47648 .35362 L
.47732 .35362 L
.47811 .35362 L
.47883 .35362 L
.48016 .35361 L
.48091 .3536 L
.4816 .3536 L
.48302 .35358 L
.48453 .35356 L
.4874 .35351 L
.49011 .35344 L
.49518 .35327 L
.50011 .35304 L
.50551 .35273 L
.51633 .35187 L
.52622 .35082 L
.53539 .34963 L
.55591 .34613 L
.57436 .34189 L
.61239 .32983 L
Mistroke
.64825 .31761 L
.68853 .30708 L
.72747 .29956 L
.7471 .29616 L
.76825 .29207 L
.78673 .28771 L
.80657 .28174 L
.82481 .27424 L
.83435 .26907 L
.84282 .26343 L
.85091 .25679 L
.85783 .24979 L
.86843 .23522 L
.8726 .22716 L
.87603 .21837 L
.87818 .21074 L
.8791 .20633 L
.87978 .20215 L
.88029 .1979 L
.88049 .19546 L
.88064 .19318 L
.8807 .19193 L
.88074 .19075 L
.88076 .18968 L
.88078 .18852 L
.88078 .18792 L
.88078 .18725 L
.88078 .18655 L
.88077 .1859 L
.88075 .18473 L
.88073 .18407 L
.88071 .18347 L
.8806 .18121 L
.88046 .17911 L
.88026 .1768 L
.87999 .17433 L
.87932 .16972 L
.87853 .16555 L
.8772 .16007 L
.87536 .15424 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`<02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`804?l2033o001mo`80
2ol200Go00<0ool0A_l201Co00<0ool0;Ol0097o00<0ool0AOl00`3oo`0Eo`030?oo02co0017o`l0
>ol00`3oo`15o`030?oo01Ko00<0ool0:ol0043o1`0?o`@0=ol00`3oo`14o`030?oo01So00<0ool0
:_l003_o1@0Jo`80=Ol00`3oo`14o`030?oo01So00<0ool0:_l003Wo0P0Qo`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Ko0`0To`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`80
9ol00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Wo`030?oo037o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo0P0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Xo`030?oo03;o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02So00<0ool0
<ol00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool09ol00`3oo`0Po`801?l200?o1005o`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Wo`030?oo023o00@0ool02?l00`3oo`06o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`808ol0103oo`09o`030?oo00Go0P12o`03
0?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Co00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02?o00<0ool09_l200[o0P06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?ol00`3oo`0Po`030?oo02Oo000]o`03
0?oo02?o0P0lo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0So`030?oo03_o00<0ool0
?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Eo`802_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`09o`030?oo01So00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l200?o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`09o`030?oo01So00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00Wo0P0Io`030?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3o
o`0Ho`802_l00`3oo`0Ho`802_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0go`809ol00`3o
o`0Yo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0=Ol202So00<0ool0:_l0033o00<0ool09ol203Go
00<0ool0<ol202Wo00<0ool0:ol0037o00<0ool0:?l00`3oo`0bo`030?oo037o0P0[o`030?oo02_o
000ao`030?oo02Wo0P0bo`030?oo033o00<0ool0:_l00`3oo`0/o`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1ol000co00<0ool09?l00`3oo`02o`030?oo02Wo100Yo`030?oo02Wo0`0Yo`030?oo00;o00<0ool0
9Ol00`3oo`07o`00=?l00`3oo`0ao`<09_l00`3oo`0Uo`@0<?l00`3oo`0`o`00=Ol00`3oo`0co`<0
8ol00`3oo`0Qo`@0<_l203?o000fo`80=_l3023o00<0ool07Ol403?o0`0eo`00>?l00`3oo`0fo`<0
7Ol00`3oo`0Io`@0=Ol203So000io`030?oo03So1@0Ho`030?oo01?o1P0go`80>_l003[o0P0mo`P0
4?l00`3oo`04o`l0>_l303co000lo`<0@_lG04Oo0P0oo`00?ol2053o00<0ool0Aol4047o0011o`<0
COl00`3oo`11o`H0AOl004Co0P1;o`030?oo03co1@1;o`00A_l304So00<0ool0=ol5053o0019o`80
A_l00`3oo`0bo`D0EOl004_o0`13o`030?oo02ko101Jo`00C_l202go0P04o`800ol400Go00<0ool0
:_l405ko001@o`<0:Ol0103oo`08o`030?oo00Ko00<0ool09_l406;o001Co`<09_l0103oo`09o`03
0?oo00Go0P0To`<0I_l005Ko100Ro`040?oo00[o00<0ool01?l00`3oo`0Oo`@0JOl005[o0`0Oo`04
0?oo00So00@0ool01Ol00`3oo`0Lo`<0KOl005go0`0Mo`802_l200Ko00<0ool06Ol3073o001Po`@0
;Ol00`3oo`0Eo`@0Lol006Co100Yo`030?oo01;o0`1go`00J?l502Co00<0ool03ol307[o001]o`D0
7ol00`3oo`0=o`80OOl007;o0`0Lo`030?oo00Go201oo`00MOlA00Wo2P27o`00Q_l900;o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
1Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87756 .18363 m
.84172 .15342 L
.82744 .14111 L
.82148 .13667 L
.81852 .13485 L
.81585 .13352 L
.81341 .1326 L
.81204 .13221 L
.81076 .13193 L
.80967 .13175 L
.80905 .13167 L
.80846 .13161 L
.80783 .13156 L
.80726 .13153 L
.80694 .13152 L
.80659 .13152 L
.80594 .13152 L
.80528 .13154 L
.80465 .13158 L
.80408 .13163 L
.80345 .13169 L
.80203 .1319 L
.8007 .13217 L
.79821 .13285 L
.79551 .13387 L
.79264 .13531 L
.79002 .13698 L
.78724 .13919 L
.78431 .14209 L
.78175 .14525 L
.77954 .14856 L
.77572 .15612 L
.77404 .16049 L
.77245 .16544 L
.77001 .17469 L
.76763 .18448 L
.76625 .18925 L
.76453 .19385 L
.76351 .19593 L
.76245 .19775 L
.76129 .19936 L
.76012 .20072 L
.75749 .20298 L
.75592 .20397 L
.75411 .20487 L
.7522 .2056 L
.74998 .20623 L
.74774 .2067 L
.74555 .20702 L
.74417 .20717 L
Mistroke
.74285 .20728 L
.74135 .20737 L
.7405 .20741 L
.73972 .20744 L
.73895 .20746 L
.73825 .20748 L
.73746 .20749 L
.73662 .20749 L
.73519 .20749 L
.73436 .20748 L
.73358 .20747 L
.73207 .20744 L
.73044 .2074 L
.72744 .20729 L
.72043 .207 L
.71744 .20689 L
.7142 .20678 L
.71233 .20673 L
.71062 .20669 L
.70899 .20667 L
.70727 .20665 L
.70572 .20665 L
.70432 .20666 L
.7027 .20668 L
.70119 .20672 L
.69979 .20676 L
.69851 .20681 L
.69565 .20696 L
.6928 .20718 L
.68973 .20748 L
.68374 .2083 L
.68084 .20883 L
.67778 .20948 L
.67245 .21087 L
.66319 .214 L
.65399 .21776 L
.64388 .22183 L
.63386 .2251 L
.62353 .22757 L
.61164 .22961 L
.59811 .23147 L
.58524 .23333 L
.57383 .23549 L
.56289 .23844 L
.55212 .24254 L
.53554 .251 L
.51705 .26017 L
.50612 .2642 L
.50121 .26565 L
.49582 .26699 L
Mistroke
.49114 .26796 L
.48624 .26879 L
.48124 .26945 L
.47678 .26987 L
.47461 .27003 L
.47236 .27015 L
.47129 .2702 L
.47029 .27024 L
.46941 .27027 L
.46846 .27029 L
.46742 .27031 L
.46686 .27032 L
.46634 .27033 L
.46541 .27034 L
.46443 .27034 L
.46388 .27034 L
.46329 .27034 L
.46274 .27033 L
.46223 .27033 L
.46127 .27032 L
.46025 .2703 L
.45856 .27027 L
.45674 .27022 L
.45345 .27013 L
.44565 .2699 L
.44197 .26981 L
.4378 .26972 L
.43291 .26965 L
.42806 .26959 L
.42229 .26954 L
.41935 .26952 L
.41659 .26951 L
.4141 .26949 L
.41133 .26948 L
.40998 .26948 L
.40852 .26948 L
.40772 .26947 L
.40699 .26947 L
.40555 .26947 L
.40411 .26947 L
.40277 .26947 L
.40156 .26947 L
.40026 .26947 L
.39884 .26948 L
.39735 .26948 L
.39609 .26949 L
.39471 .26949 L
.39227 .26951 L
.38969 .26953 L
.38517 .26958 L
Mistroke
.3797 .26966 L
.3748 .26975 L
.37064 .26983 L
.36849 .26987 L
.36732 .26989 L
.36622 .2699 L
.36518 .26991 L
.36459 .26991 L
.36404 .26992 L
.36308 .26992 L
.36206 .26992 L
.36102 .26991 L
.36003 .2699 L
.35915 .26989 L
.35819 .26986 L
.35716 .26983 L
.35602 .26979 L
.35369 .26968 L
.35149 .26952 L
.34939 .26934 L
.34509 .26883 L
.34092 .26818 L
.33049 .26591 L
.31797 .26221 L
.30505 .25751 L
.27422 .2434 L
.25146 .22984 L
.24706 .22652 L
.24343 .22327 L
.24218 .22189 L
.24115 .22053 L
.24051 .21942 L
.24024 .21883 L
.24006 .21829 L
.23993 .21773 L
.23989 .21746 L
.23987 .21716 L
.23988 .21665 L
.23995 .21618 L
.24002 .21591 L
.2401 .21565 L
.24031 .21517 L
.24061 .21468 L
.24101 .21415 L
.24189 .21328 L
.24307 .21239 L
.24582 .21082 L
.26219 .20376 L
.27319 .19814 L
.2789 .19451 L
Mistroke
.28458 .19017 L
.28911 .18598 L
.29343 .18103 L
.29666 .17629 L
.299 .17176 L
.30002 .1692 L
.30076 .16685 L
.30135 .16428 L
.30156 .163 L
.30165 .16231 L
.30173 .1616 L
.30182 .16038 L
.30185 .15929 L
.30184 .15812 L
.30181 .15746 L
.30177 .15686 L
.30166 .15577 L
.30152 .1548 L
.30131 .1537 L
.30107 .15268 L
.30052 .15082 L
.29978 .14893 L
.29809 .14579 L
.29582 .14277 L
.29288 .13992 L
.28945 .13748 L
.28595 .13564 L
.28267 .13436 L
.27903 .13334 L
.27714 .13295 L
.27505 .13261 L
.27401 .13248 L
.27289 .13237 L
.27183 .13228 L
.27086 .13222 L
.26996 .13218 L
.26913 .13216 L
.26822 .13214 L
.26724 .13214 L
.26676 .13214 L
.26623 .13215 L
.26527 .13218 L
.26434 .13221 L
.26348 .13226 L
.26171 .13237 L
.26009 .1325 L
.25643 .13287 L
.2496 .1337 L
.23163 .13587 L
.22138 .13701 L
Mistroke
.20966 .13842 L
.19849 .1401 L
.18891 .1421 L
.18416 .14343 L
.18013 .14484 L
.17614 .14661 L
.17246 .14876 L
.16972 .15088 L
.16754 .15306 L
.16556 .1557 L
.16387 .15879 L
.16316 .16048 L
.16252 .16235 L
.16159 .16588 L
.16119 .16793 L
.16087 .1699 L
.16057 .17215 L
.16032 .17458 L
.15999 .1788 L
.15975 .18276 L
.15954 .18751 L
.15938 .19188 L
.15925 .19623 L
.1592 .19823 L
.15916 .20036 L
.15914 .20144 L
.15913 .20202 L
.15913 .20264 L
.15912 .20372 L
.15911 .20474 L
.15911 .20536 L
.15911 .20602 L
.15912 .20666 L
.15912 .20724 L
.15914 .20836 L
.15916 .20954 L
.15918 .21068 L
.15922 .21173 L
.15931 .21411 L
.15943 .21623 L
.15958 .21849 L
.15995 .22248 L
.1605 .22698 L
.16135 .23234 L
.16238 .23759 L
.1649 .24786 L
.16848 .25933 L
.17335 .2716 L
.17838 .28129 L
.18484 .29013 L
Mistroke
.18846 .29363 L
.19058 .29527 L
.19267 .29663 L
.19467 .29771 L
.19693 .29872 L
.19906 .29948 L
.20107 .30006 L
.20333 .30056 L
.20583 .30097 L
.2081 .30123 L
.20925 .30134 L
.21054 .30143 L
.21279 .30155 L
.21407 .3016 L
.21525 .30163 L
.21977 .30174 L
.22108 .30177 L
.22248 .30181 L
.22501 .30192 L
.22641 .302 L
.22773 .30209 L
.2292 .30221 L
.23074 .30238 L
.23335 .30273 L
.23573 .30316 L
.23827 .30375 L
.24102 .30455 L
.24614 .3066 L
.25079 .30918 L
.25479 .31204 L
.25893 .31568 L
.26749 .32547 L
.27466 .33552 L
.28079 .3443 L
.28739 .35247 L
.29129 .35628 L
.29498 .35913 L
.29883 .36139 L
.3026 .36302 L
.30467 .3637 L
.30665 .36423 L
.31041 .36499 L
.31255 .36531 L
.31454 .36556 L
.31909 .36598 L
.3268 .3666 L
.33115 .36706 L
.33512 .3676 L
.34369 .36925 L
.35181 .37127 L
Mistroke
.36177 .37379 L
.36633 .37477 L
.36881 .37522 L
.37152 .37564 L
.37407 .37596 L
.37644 .3762 L
.37873 .37637 L
.3799 .37644 L
.38118 .37649 L
.38257 .37653 L
.38385 .37654 L
.38455 .37655 L
.38532 .37654 L
.38672 .37652 L
.38805 .37648 L
.38945 .37642 L
.39199 .37626 L
.39476 .37602 L
.39776 .3757 L
.40321 .37495 L
.41556 .37273 L
.43674 .36863 L
.45734 .36581 L
.46662 .36453 L
.47193 .36358 L
.47686 .36249 L
.48698 .35966 L
.49711 .35613 L
.51849 .3476 L
.54168 .33911 L
.56242 .3337 L
.57302 .33134 L
.58249 .32882 L
.5921 .32544 L
.60145 .32126 L
.62302 .30944 L
.64226 .2993 L
.6475 .29707 L
.65311 .29506 L
.65841 .29355 L
.66326 .29253 L
.6659 .29211 L
.66837 .29182 L
.66976 .29169 L
.67109 .29159 L
.67182 .29154 L
.67262 .2915 L
.67401 .29145 L
.67526 .29142 L
.67665 .29141 L
Mistroke
.67735 .29142 L
.6781 .29143 L
.67946 .29146 L
.68083 .29152 L
.68213 .29158 L
.6846 .29176 L
.68746 .29202 L
.69055 .29238 L
.69623 .29319 L
.70972 .29565 L
.73595 .30122 L
.76075 .30621 L
.77318 .30826 L
.77922 .30906 L
.78541 .30968 L
.78704 .30981 L
.78855 .30991 L
.79 .30998 L
.79132 .31004 L
.79256 .31008 L
.79385 .31011 L
.79513 .31012 L
.79651 .31011 L
.79727 .31009 L
.79799 .31008 L
.79931 .31002 L
.80004 .30999 L
.80071 .30995 L
.8022 .30983 L
.80463 .30958 L
.80711 .30922 L
.80941 .30878 L
.81173 .30823 L
.81562 .307 L
.81948 .30535 L
.82334 .30318 L
.82951 .29846 L
.83475 .29327 L
.85835 .26719 L
.87406 .25112 L
.88237 .24221 L
.89012 .23281 L
.89532 .22478 L
.89638 .22266 L
.89733 .22041 L
.89803 .21834 L
.89827 .21743 L
.89849 .21647 L
.89866 .21558 L
.89877 .21477 L
Mistroke
.89885 .21397 L
.8989 .21322 L
.89891 .21235 L
.89889 .21156 L
.89882 .21069 L
.8987 .20978 L
.89856 .20897 L
.89836 .2081 L
.8979 .20655 L
.89716 .20472 L
.89632 .20306 L
.89314 .1984 L
.8892 .19403 L
.87756 .18363 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04ko00<0ool0
>ol004co300io`030?oo04[o10001?l0000jo`00@_l:00co00<0ool0=_l00`3oo`1:o`030?oo00Go
0P0ho`00?_l401Oo0P0fo`030?oo04So0P0:o`030?oo03Go000jo`@07Ol00`3oo`0co`030?oo04Oo
00<0ool02ol00`3oo`0do`00>Ol00`3oo`0Oo`030?oo03?o00<0ool0Aol00`3oo`0<o`030?oo03?o
000ho`808Ol203Co00<0ool0A_l00`3oo`0>o`030?oo03;o000ho`030?oo027o00<0ool0<_l00`3o
o`16o`030?oo00oo00<0ool0<Ol003Oo00<0ool08_l00`3oo`0bo`030?oo04Go00<0ool04Ol2037o
000go`030?oo02;o00<0ool0<_l00`3oo`15o`030?oo01?o00<0ool0;_l003Ko00<0ool08ol00`3o
o`0bo`030?oo04Go00<0ool05?l00`3oo`0]o`00=_l00`3oo`0Ro`030?oo03?o00<0ool0A?l00`3o
o`0Fo`030?oo02co000fo`030?oo027o00<0ool08?l200Co0P03o`@01Ol00`3oo`14o`030?oo01Oo
0P0/o`00=_l00`3oo`0Qo`030?oo01oo00@0ool02?l00`3oo`06o`030?oo04Co00<0ool06Ol00`3o
o`0Yo`00=_l00`3oo`0Po`808Ol0103oo`09o`030?oo00Go0P15o`030?oo01[o00<0ool0:?l003Ko
00<0ool07ol00`3oo`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0A?l00`3oo`0Ko`030?oo02Oo
000fo`030?oo01ko00<0ool08_l0103oo`08o`040?oo00Go00<0ool0@ol00`3oo`0Mo`030?oo02Ko
000fo`030?oo01co0P0Vo`802_l200Ko00<0ool0@ol00`3oo`0No`030?oo02Go000fo`030?oo01_o
00<0ool0>_l00`3oo`12o`808Ol00`3oo`0To`00=_l00`3oo`0Io`80?Ol00`3oo`11o`808_l00`3o
o`0To`00=_l00`3oo`0Go`80?ol00`3oo`0bo`@02?l402Co00<0ool08ol000Oo1@02o`801?l2023o
00<0ool01@0@o`804?l200Co0P04o`808ol00`3oo`0Ro`801?l200;o1P03o`X05?l500ko0P0Bo`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01oo00<0ool00_l00`3oo`0>o`804Ol0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`02o`D000Go0?oo000To`030?oo00oo00<0ool04_l00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool07Ol00`3oo`02o`030?oo00go0P0Bo`040?oo00_o00<0ool0
8?l00`3oo`0Qo`030?oo00<02Ol00`3oo`0Ro`030?oo00ko0P0Do`030?oo00[o0@01o`7o00000ol0
000200Co00<0ool02?l201go1004o`030?oo00go0P0<o`@00_l0103oo`09o`808ol00`3oo`0Qo`04
0?oo00Wo0P0Uo`030?oo00ko00<0ool04ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Oo`03
0?oo00;o00<0ool03_l00`3oo`0@o`040?oo00Wo00<0ool08_l00`3oo`0Oo`H02Ol00`3oo`0To`03
0?oo00ko00<0ool04ol00`3oo`08o`030?oo00;o0008o`802_l301oo00@0oooo0P0Ao`030?oo013o
0P0:o`<08_l00`3oo`0Lo`<00ol200[o0`0So`803ol00`3oo`0Co`802_l300;o000fo`030?oo01Go
0P11o`030?oo01Ko1P18o`030?oo02Go000go`030?oo01Ko0P0oo`030?oo00oo1`1=o`030?oo02Ko
000go`030?oo01So0P0mo`030?oo00co0`1Do`030?oo02Ko000go`030?oo01[o0P0ko`030?oo00[o
0P1Fo`030?oo02Oo000go`030?oo01co0P0io`030?oo00So0P1Go`030?oo02So000<ool03009o`00
3?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Co0P08o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`00
3?l00`3oo`0Yo`030?oo027o0`05o`030?oo02[o00<0ool00_l202Go00<0ool0:Ol00`3oo`06o`03
0?oo027o00<0ool01ol003Wo00<0ool08ol302oo00<0ool00P1Lo`030?oo02_o000io`030?oo02Ko
0`0/o`<0GOl00`3oo`0/o`00>_l00`3oo`0Xobd0G_l00`3oo`0]o`00>_l00`3oo`1Do`030?oo05_o
00<0ool0;_l003_o00<0ool0Dol00`3oo`1Jo`030?oo02oo000ko`030?oo05?o00<0ool0FOl00`3o
o`0`o`00??l00`3oo`1Bo`030?oo05Wo00<0ool0<?l003co00<0ool0D_l00`3oo`1Ho`030?oo037o
000mo`030?oo057o00<0ool0:?l;02Co00<0ool0<_l003ko0P1Ao`030?oo02Go0`0;o`<07ol203Go
000oo`@0C_l00`3oo`0Ro`<04Ol401[o00<0ool0=Ol004;o2P15o`030?oo023o0P0Ho`D04ol203So
001<o`80@ol00`3oo`0No`807ol700Wo0`0jo`00C_l00`3oo`0/o`801?l200?o1005o`030?oo01co
0P0Xo`X0??l004oo0P0[o`040?oo00So00<0ool01_l00`3oo`0Jo`80L?l0057o00<0ool0:?l0103o
o`09o`030?oo00Go0P0Io`80L_l005;o00<0ool09ol0103oo`0:o`030?oo00Co00<0ool05_l207Co
001Co`030?oo02Ko00@0ool02?l0103oo`05o`030?oo01Co0P1fo`00E?l00`3oo`0Vo`802_l200Ko
00<0ool04Ol307So001Do`030?oo03[o00<0ool03?l507_o001Eo`030?oo03Wo00<0ool02_l2083o
001Fo`030?oo03So00<0ool01ol308;o001Fo`030?oo03So00<0ool00ol408Go001Go`030?oo03Oo
00<0ool00`29o`00F?l00`3oo`0fo`<0S?l005Wo00<0ool0<ol308ko001Jo`030?oo02oo0`02o`03
0?oo08co001Ko`<0:Ol500Go00<0ool0S?l005ko200Mo`@02_l00`3oo`2<o`00I_l501;o1P0>o`03
0?oo08co001[o`@01_l:00Co0P05o`<01?l00`3oo`2<o`00Kol700Ko00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87729 .17995 m
.84313 .14986 L
.82832 .13822 L
.82482 .13594 L
.8213 .13396 L
.81814 .13251 L
.81532 .1315 L
.814 .13113 L
.81254 .13079 L
.81109 .13054 L
.81029 .13043 L
.80954 .13036 L
.80883 .1303 L
.80816 .13026 L
.80755 .13025 L
.80689 .13024 L
.80617 .13025 L
.80576 .13027 L
.8054 .13029 L
.80468 .13034 L
.804 .1304 L
.80271 .13056 L
.80134 .1308 L
.79886 .1314 L
.79615 .13232 L
.79316 .13366 L
.79011 .13545 L
.78727 .13755 L
.78221 .14265 L
.77962 .14618 L
.77741 .14986 L
.77523 .15428 L
.77315 .15948 L
.77022 .16902 L
.76792 .17823 L
.76543 .18775 L
.76377 .19263 L
.76192 .19669 L
.75973 .20015 L
.75735 .20282 L
.75585 .20411 L
.75407 .20534 L
.75041 .20717 L
.74836 .2079 L
.74594 .20857 L
.74323 .20914 L
.74054 .20955 L
.7379 .20984 L
.73539 .21005 L
.73396 .21014 L
Mistroke
.73237 .21023 L
.72941 .21034 L
.72783 .21039 L
.72636 .21043 L
.72298 .21049 L
.72001 .21054 L
.71671 .21059 L
.71371 .21065 L
.71091 .21073 L
.70795 .21084 L
.70478 .21099 L
.70136 .21121 L
.69769 .21151 L
.69451 .21184 L
.69118 .21227 L
.6851 .21327 L
.67974 .21444 L
.66996 .21727 L
.66028 .22089 L
.64334 .22752 L
.63381 .23046 L
.62246 .23306 L
.60966 .2352 L
.5973 .23695 L
.58461 .23892 L
.57326 .2412 L
.56381 .24375 L
.55436 .24708 L
.53667 .25504 L
.52723 .25938 L
.51659 .26362 L
.50592 .26698 L
.49608 .26927 L
.49142 .2701 L
.48647 .27082 L
.48194 .27132 L
.4779 .27164 L
.47578 .27177 L
.47388 .27186 L
.47183 .27194 L
.47082 .27196 L
.46972 .27199 L
.46876 .272 L
.4679 .27201 L
.46691 .27201 L
.46598 .27201 L
.46513 .27201 L
.46436 .272 L
.46351 .27199 L
.46261 .27198 L
Mistroke
.46076 .27194 L
.45879 .27189 L
.45524 .27177 L
.44699 .27146 L
.42845 .27091 L
.40742 .27046 L
.38613 .27004 L
.37765 .26988 L
.37344 .26979 L
.36915 .26969 L
.36697 .26963 L
.36462 .26954 L
.36053 .26936 L
.35852 .26924 L
.35637 .26908 L
.35241 .26873 L
.34738 .2681 L
.34234 .26728 L
.33187 .26495 L
.32033 .26154 L
.30667 .25656 L
.28061 .2447 L
.26822 .23791 L
.25691 .23075 L
.2527 .22765 L
.24907 .22458 L
.24652 .22196 L
.24483 .21968 L
.2442 .21847 L
.24395 .21781 L
.24378 .21722 L
.24369 .21673 L
.24364 .2162 L
.24364 .21567 L
.2437 .21518 L
.24383 .21462 L
.24401 .2141 L
.24423 .21364 L
.24453 .21315 L
.24539 .21208 L
.24643 .21111 L
.24891 .20937 L
.25174 .20778 L
.25969 .20393 L
.26901 .19932 L
.27921 .19324 L
.28429 .18951 L
.28941 .18503 L
.29321 .18097 L
.29669 .17634 L
Mistroke
.29822 .17383 L
.29943 .1715 L
.30038 .1693 L
.30122 .16688 L
.30152 .16584 L
.30179 .16472 L
.30199 .16376 L
.30216 .16272 L
.30231 .16156 L
.3024 .16046 L
.30244 .1594 L
.30243 .15843 L
.30238 .15729 L
.30232 .15666 L
.30226 .15607 L
.3021 .15502 L
.30188 .15392 L
.30138 .15208 L
.30065 .15013 L
.29969 .14815 L
.29845 .14615 L
.29574 .1429 L
.29247 .14009 L
.28909 .13794 L
.28509 .13607 L
.28297 .1353 L
.28065 .1346 L
.2786 .1341 L
.27672 .13372 L
.27481 .13342 L
.27303 .13319 L
.27107 .13301 L
.27005 .13294 L
.26895 .13289 L
.2684 .13287 L
.2678 .13285 L
.26724 .13284 L
.26672 .13283 L
.26574 .13282 L
.26525 .13283 L
.26471 .13283 L
.26371 .13285 L
.26278 .13288 L
.26176 .13292 L
.26068 .13298 L
.25687 .13326 L
.2531 .13362 L
.24964 .13401 L
.23162 .13616 L
.22148 .13733 L
Mistroke
.20994 .13875 L
.19899 .14039 L
.19375 .14137 L
.18913 .14238 L
.18052 .14485 L
.17628 .14649 L
.17265 .14827 L
.16916 .15045 L
.16636 .15271 L
.16377 .15547 L
.16152 .15871 L
.1599 .16193 L
.15872 .16508 L
.15771 .16877 L
.15691 .17285 L
.15639 .17657 L
.15603 .18013 L
.15573 .18413 L
.15561 .18639 L
.15551 .18846 L
.15543 .19069 L
.15536 .19278 L
.15532 .19497 L
.1553 .19614 L
.15529 .19679 L
.15529 .19739 L
.15528 .19844 L
.15528 .19942 L
.15528 .2005 L
.15529 .20164 L
.1553 .20263 L
.15531 .20356 L
.15533 .20459 L
.15536 .20569 L
.15544 .20798 L
.15556 .21049 L
.15573 .21313 L
.15593 .21561 L
.1564 .22009 L
.15709 .22499 L
.15909 .23508 L
.16227 .24667 L
.16589 .25705 L
.17075 .26843 L
.17634 .2787 L
.18235 .28695 L
.18601 .29072 L
.18957 .29362 L
.19395 .29631 L
.19619 .29737 L
Mistroke
.19837 .29822 L
.20046 .29889 L
.20278 .2995 L
.20695 .3003 L
.20938 .30062 L
.212 .3009 L
.21679 .30126 L
.22102 .30154 L
.22559 .3019 L
.22822 .30217 L
.23062 .30249 L
.23529 .30334 L
.23757 .30389 L
.24006 .30461 L
.245 .30648 L
.24944 .30873 L
.25347 .31129 L
.26197 .31854 L
.27035 .32796 L
.2845 .3453 L
.28838 .34925 L
.29262 .35287 L
.29671 .35564 L
.30059 .35767 L
.30473 .35927 L
.3069 .35993 L
.30933 .36054 L
.31348 .36133 L
.31796 .36194 L
.32645 .36283 L
.33457 .36383 L
.33889 .36453 L
.34361 .36546 L
.35195 .36742 L
.36208 .36995 L
.36704 .37104 L
.37171 .3719 L
.37664 .37258 L
.37929 .37284 L
.38065 .37295 L
.38212 .37304 L
.38343 .37311 L
.38462 .37316 L
.38583 .37319 L
.38698 .3732 L
.38833 .37321 L
.38957 .37319 L
.39095 .37316 L
.39165 .37314 L
.39242 .37311 L
Mistroke
.39395 .37304 L
.39542 .37295 L
.39819 .37274 L
.40448 .37207 L
.41584 .37037 L
.45502 .36471 L
.46537 .36359 L
.4705 .36289 L
.47531 .36209 L
.48465 .36003 L
.49535 .35693 L
.53984 .34108 L
.56122 .33533 L
.57091 .33314 L
.57954 .331 L
.59701 .32495 L
.60698 .3203 L
.61669 .3152 L
.63557 .30512 L
.64633 .30021 L
.65159 .29824 L
.65641 .29671 L
.6612 .2955 L
.66626 .29453 L
.66892 .29416 L
.6714 .2939 L
.6728 .29378 L
.67414 .29369 L
.67488 .29365 L
.67568 .29361 L
.67709 .29356 L
.67849 .29353 L
.67981 .29352 L
.68057 .29352 L
.68126 .29353 L
.68207 .29354 L
.68284 .29356 L
.68437 .29361 L
.6858 .29368 L
.68908 .29389 L
.69198 .29413 L
.69464 .2944 L
.70074 .29515 L
.72692 .29948 L
.75238 .3039 L
.76503 .30577 L
.77101 .30651 L
.77631 .30704 L
.77893 .30726 L
.78173 .30745 L
Mistroke
.78324 .30753 L
.78462 .30759 L
.78592 .30764 L
.78729 .30768 L
.78865 .3077 L
.79008 .30771 L
.79142 .3077 L
.79263 .30767 L
.79396 .30763 L
.7947 .3076 L
.79538 .30756 L
.79676 .30748 L
.79819 .30737 L
.80074 .3071 L
.80302 .30679 L
.8054 .30637 L
.80791 .30583 L
.81215 .30462 L
.81585 .30323 L
.82213 .29998 L
.82817 .29568 L
.85318 .27035 L
.86855 .2543 L
.88293 .23814 L
.88896 .23006 L
.89328 .22269 L
.89496 .21874 L
.89566 .21656 L
.89593 .21553 L
.89614 .21459 L
.8963 .21372 L
.89642 .21294 L
.89652 .21208 L
.89659 .21116 L
.89662 .21028 L
.89661 .20934 L
.89657 .20844 L
.89649 .20763 L
.89638 .20675 L
.89623 .20593 L
.89587 .20438 L
.89536 .20273 L
.89464 .20094 L
.89298 .19772 L
.89101 .19472 L
.87729 .17995 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04co1@0ko`00
C?l<03Wo00<0ool0B_l200Go0P0io`00@_l:00_o0P0ho`030?oo04Wo0P08o`80=ol003ko100Go`03
0?oo03Go00<0ool0B?l00`3oo`0:o`80=Ol003[o100Lo`80=Ol00`3oo`17o`030?oo00go00<0ool0
<_l003Wo00<0ool07ol00`3oo`0co`030?oo04Ko00<0ool03ol00`3oo`0ao`00>?l00`3oo`0Qo`03
0?oo03;o00<0ool0A_l00`3oo`0@o`030?oo033o000go`030?oo02;o00<0ool0<_l00`3oo`16o`03
0?oo017o00<0ool0;ol003Ko00<0ool08ol00`3oo`0bo`030?oo04Go00<0ool04ol00`3oo`0^o`00
=_l00`3oo`0So`030?oo03;o00<0ool0A?l00`3oo`0Eo`80;_l003Go00<0ool09?l00`3oo`0bo`03
0?oo04Co00<0ool05ol00`3oo`0[o`00=Ol00`3oo`0So`030?oo03?o00<0ool0A?l00`3oo`0Ho`03
0?oo02[o000eo`030?oo02;o0P0Qo`801?l200?o1005o`030?oo04Co00<0ool06Ol00`3oo`0Yo`00
=Ol00`3oo`0Ro`030?oo01oo00@0ool02?l00`3oo`06o`030?oo04Co00<0ool06_l00`3oo`0Xo`00
=Ol00`3oo`0Qo`030?oo023o00@0ool02Ol00`3oo`05o`80A?l00`3oo`0Lo`030?oo02Oo000eo`03
0?oo023o00<0ool06ol400;o00@0ool02_l00`3oo`04o`030?oo04?o00<0ool07Ol00`3oo`0Vo`00
=Ol00`3oo`0Oo`030?oo02;o00@0ool02?l0103oo`05o`030?oo04?o00<0ool07_l00`3oo`0Uo`00
=Ol00`3oo`0No`030?oo02Co0P0:o`801_l00`3oo`13o`030?oo01oo00<0ool09?l003Go00<0ool0
7?l203_o00<0ool0@_l00`3oo`0Po`030?oo02Co000eo`030?oo01[o0P0mo`030?oo047o0P0Ro`80
9Ol003Go00<0ool06?l203oo00<0ool0@?l202Co00<0ool08ol000Oo1@02o`801?l201oo00@0oooo
1@0@o`804?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P08o`P04ol500ko00<0ool04Ol500;o
0P04o`800ol000Wo00<0ool01ol0103oo`0No`030?oo00?o00<0ool03_l2017o00@0ool02?l0103o
o`0Ro`030?oo027o00@0ool02?l;01go00<0ool03_l00`3oo`0Co`030?oo00Oo00@0ool00_l000Wo
00<0ool02_l00`3oo`0Lo`030?oo00?o00<0ool03_l00`3oo`0@o`040?oo00_o00<0ool08?l00`3o
o`0Qo`040?oo00Co1@02o`030?oo02;o00<0ool03_l00`3oo`0Co`030?oo00[o0@01o`7o00000ol0
000200Co00<0ool02?l201go1004o`030?oo00ko0P0;o`@00_l0103oo`09o`808ol00`3oo`0Qo`04
0?oo00;o0P05o`809Ol00`3oo`0>o`030?oo01?o00<0ool02?l200?o0009o`030?oo00So00<0ool0
7_l00`3oo`03o`030?oo00oo0P0@o`040?oo00Wo00<0ool08_l00`3oo`0Qo`030?oo00<01ol00`3o
o`0To`030?oo00go00<0ool05?l00`3oo`08o`030?oo00;o0008o`802_l301ko00<0ool00_l201?o
00<0ool03_l200[o0`0Ro`030?oo02;o0`09o`<08ol200oo00<0ool04ol200[o0`02o`00=_l00`3o
o`0Go`030?oo03ko00<0ool07?l604;o00<0ool09Ol003Ko00<0ool06?l203ko00<0ool06Ol304Oo
00<0ool09_l003Ko00<0ool06_l00`3oo`0ko`030?oo01;o1`19o`030?oo02Oo000go`030?oo01[o
0P0ko`030?oo00go1@1?o`030?oo02So000go`030?oo01co0P0io`030?oo00[o0`1Do`030?oo02So
000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Go0`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`07o`003?l00`3oo`0Yo`030?oo027o1004o`030?oo02[o00<0ool01?l202?o00<0ool0
:Ol00`3oo`06o`030?oo027o00<0ool01ol003So00<0ool09Ol302ko00<0ool00_l205So0P0^o`00
>Ol00`3oo`0Wo`<0:ol00`3oo`0205Wo00<0ool0;_l003Wo00<0ool0:_lE01?o1P1Jo`030?oo02oo
000io`030?oo03oo4`03o`030?oo05Wo00<0ool0<?l003[o00<0ool0E?l00`3oo`1Ho`030?oo037o
000jo`030?oo05Co00<0ool0Eol00`3oo`0bo`00>ol00`3oo`1Co`030?oo05Oo00<0ool0<_l003co
00<0ool0D_l00`3oo`1Fo`030?oo03?o000mo`030?oo057o00<0ool0EOl00`3oo`0do`00?_l2057o
00<0ool09_l@01go0P0go`00@?l404go00<0ool08ol3013o1`0Eo`030?oo03Oo0014o`P0AOl00`3o
o`0Ro`030?oo01So1@0?o`030?oo03So001;o`<0@ol00`3oo`0Po`808?l?03_o001>o`80;Ol200Co
0P03o`@01Ol00`3oo`0No`80K?l0053o00<0ool0:Ol0103oo`08o`030?oo00Ko00<0ool07?l206ko
001Ao`030?oo02So00@0ool02Ol00`3oo`05o`806ol2073o001Bo`030?oo02Oo00@0ool02_l00`3o
o`04o`030?oo01So0P1bo`00D_l00`3oo`0Wo`040?oo00So00@0ool01Ol00`3oo`0Eo`<0M?l005?o
00<0ool09ol200[o0P06o`030?oo01?o0P1go`00E?l00`3oo`0jo`030?oo00ko1@1io`00EOl00`3o
o`0io`030?oo00[o101no`00E_l00`3oo`0ho`030?oo00Ko1022o`00Eol00`3oo`0go`030?oo00?o
0`26o`00F?l00`3oo`0fo`030?oo00<0ROl005Wo00<0ool0=Ol308co001Jo`030?oo037o102>o`00
Fol402co0`03o`030?oo08co001Oo`P07ol500Ko00<0ool0S?l006Oo100Eo`H02ol00`3oo`2<o`00
Jol400Oo2P0Ao`030?oo08co001_o`L01ol200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`04
0?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o
003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87701 .17628 m
.86145 .1611 L
.84454 .14631 L
.83684 .14037 L
.82921 .13534 L
.8251 .13311 L
.82151 .13151 L
.81802 .13029 L
.81635 .12983 L
.81479 .12948 L
.81331 .1292 L
.81196 .12902 L
.81119 .12893 L
.81047 .12887 L
.8097 .12882 L
.80927 .1288 L
.80886 .12879 L
.80847 .12878 L
.80805 .12878 L
.80728 .12878 L
.80653 .12881 L
.80585 .12884 L
.8051 .1289 L
.80428 .12898 L
.80263 .1292 L
.80117 .12947 L
.79984 .12977 L
.79684 .13066 L
.79408 .13176 L
.79149 .13306 L
.78589 .13697 L
.78322 .13951 L
.78043 .14278 L
.77822 .14595 L
.77603 .14974 L
.77259 .15749 L
.7696 .16677 L
.76689 .17738 L
.76548 .18303 L
.76404 .18813 L
.76246 .19275 L
.76074 .19665 L
.75883 .19997 L
.75643 .20306 L
.7536 .20569 L
.75179 .20699 L
.74996 .20807 L
.74789 .20907 L
.74579 .2099 L
.74151 .21119 L
Mistroke
.73893 .21177 L
.73642 .21223 L
.73039 .21305 L
.71981 .214 L
.70809 .21497 L
.70196 .21563 L
.69535 .21654 L
.68952 .21756 L
.68428 .2187 L
.67344 .22177 L
.66275 .22565 L
.64397 .23278 L
.63381 .23579 L
.62186 .23845 L
.59836 .2422 L
.58638 .24411 L
.57565 .24621 L
.56518 .24888 L
.55444 .25239 L
.53734 .25918 L
.52749 .26304 L
.51798 .26631 L
.50736 .26926 L
.49765 .27128 L
.49238 .27211 L
.48752 .27271 L
.48254 .27318 L
.47981 .27337 L
.47735 .2735 L
.47517 .27359 L
.47322 .27365 L
.47215 .27367 L
.4716 .27368 L
.47101 .27369 L
.46993 .2737 L
.46895 .2737 L
.46806 .2737 L
.46711 .27369 L
.46621 .27369 L
.4654 .27368 L
.46352 .27364 L
.46157 .27359 L
.45754 .27346 L
.45379 .2733 L
.44509 .27291 L
.40734 .27146 L
.3865 .27054 L
.37637 .26999 L
.36642 .26933 L
.35793 .26856 L
Mistroke
.35307 .26797 L
.34841 .26728 L
.33968 .26559 L
.32922 .2629 L
.30296 .25341 L
.27634 .24032 L
.26516 .23356 L
.2557 .22666 L
.25228 .22355 L
.24985 .2208 L
.24904 .21966 L
.24836 .21849 L
.24808 .21789 L
.24782 .21725 L
.24764 .21668 L
.24751 .21615 L
.24742 .21553 L
.24739 .21522 L
.24739 .21489 L
.24742 .2143 L
.2475 .21376 L
.24757 .21345 L
.24766 .21313 L
.24787 .21257 L
.24816 .21197 L
.24856 .21133 L
.24939 .21027 L
.25052 .20915 L
.25312 .20716 L
.26845 .19861 L
.27862 .19258 L
.28385 .18892 L
.28901 .18467 L
.29304 .18069 L
.29682 .17608 L
.29954 .17174 L
.30141 .16764 L
.30215 .16537 L
.30264 .16329 L
.30283 .16209 L
.30296 .16102 L
.30301 .16043 L
.30304 .1598 L
.30305 .15921 L
.30305 .15866 L
.30302 .15768 L
.30294 .15677 L
.30282 .15579 L
.30263 .15475 L
.30238 .15369 L
Mistroke
.30209 .1527 L
.30142 .1509 L
.30065 .1493 L
.29969 .14768 L
.29762 .14498 L
.29494 .1424 L
.29155 .13997 L
.28771 .13791 L
.2838 .13637 L
.28016 .1353 L
.27611 .13445 L
.2738 .1341 L
.2717 .13386 L
.27054 .13376 L
.26929 .13367 L
.26811 .1336 L
.26703 .13356 L
.26603 .13353 L
.2651 .13352 L
.26407 .13351 L
.2635 .13351 L
.26298 .13352 L
.26194 .13354 L
.26135 .13355 L
.26081 .13357 L
.25876 .13365 L
.25679 .13377 L
.25499 .13389 L
.25093 .13425 L
.24345 .13505 L
.22466 .13725 L
.20481 .13982 L
.19323 .14181 L
.18302 .14427 L
.17826 .1458 L
.17351 .14771 L
.16946 .14979 L
.16613 .15195 L
.16326 .15431 L
.16093 .15673 L
.15884 .15948 L
.15693 .16276 L
.15539 .16628 L
.15408 .17038 L
.15313 .17447 L
.15247 .17842 L
.1522 .18049 L
.15199 .18246 L
.15167 .18621 L
.15154 .18833 L
Mistroke
.15144 .1903 L
.15137 .19247 L
.15134 .19356 L
.15133 .19415 L
.15132 .19477 L
.1513 .19595 L
.1513 .19705 L
.1513 .19823 L
.1513 .1989 L
.15131 .19953 L
.15133 .20067 L
.15135 .20171 L
.15138 .20286 L
.15143 .20406 L
.15152 .20616 L
.15164 .20812 L
.15199 .21258 L
.15226 .21509 L
.15255 .21746 L
.15323 .22194 L
.15539 .2322 L
.15817 .24181 L
.162 .25232 L
.171 .27109 L
.17664 .2798 L
.18004 .28409 L
.18346 .28772 L
.18682 .29067 L
.19067 .29341 L
.19434 .29546 L
.1979 .297 L
.20196 .29833 L
.20651 .2994 L
.21095 .30014 L
.21512 .30067 L
.2202 .30122 L
.22569 .30187 L
.23089 .30268 L
.23558 .30369 L
.24015 .30501 L
.245 .30688 L
.24946 .30909 L
.25345 .31152 L
.26208 .31831 L
.27059 .32683 L
.27829 .33524 L
.2855 .3426 L
.29208 .34811 L
.29954 .35264 L
.30353 .35437 L
Mistroke
.30789 .35581 L
.31208 .35685 L
.31594 .35758 L
.32557 .35891 L
.33452 .3601 L
.34294 .36157 L
.35188 .36356 L
.36115 .36582 L
.36639 .36703 L
.37146 .36804 L
.37635 .36883 L
.37897 .36916 L
.38179 .36945 L
.38419 .36964 L
.38557 .36973 L
.38686 .36979 L
.38806 .36983 L
.38916 .36986 L
.39037 .36988 L
.39165 .36989 L
.39288 .36988 L
.39401 .36987 L
.39531 .36983 L
.39655 .36979 L
.39875 .36969 L
.40116 .36954 L
.40664 .36906 L
.41171 .36849 L
.45285 .36344 L
.46264 .36262 L
.46746 .36215 L
.47196 .36163 L
.48054 .36032 L
.49027 .35823 L
.51164 .35181 L
.53531 .34383 L
.57551 .33337 L
.58529 .33067 L
.59591 .32695 L
.61582 .31772 L
.63584 .30752 L
.6465 .3028 L
.65235 .30064 L
.65789 .29894 L
.66318 .29764 L
.66597 .29708 L
.66895 .29659 L
.67147 .29625 L
.67425 .29596 L
.67548 .29586 L
Mistroke
.67681 .29576 L
.67797 .29569 L
.67922 .29563 L
.68057 .29558 L
.68183 .29555 L
.68261 .29554 L
.68334 .29553 L
.68471 .29553 L
.68604 .29554 L
.68726 .29556 L
.68861 .2956 L
.69005 .29565 L
.69256 .29577 L
.69492 .29592 L
.70039 .29636 L
.71279 .29774 L
.72504 .29938 L
.7487 .30262 L
.76088 .30404 L
.76739 .30465 L
.77052 .30489 L
.77334 .30507 L
.7757 .30519 L
.77824 .3053 L
.77955 .30534 L
.78076 .30536 L
.78202 .30538 L
.78336 .30539 L
.78473 .30539 L
.7855 .30538 L
.7862 .30537 L
.78749 .30534 L
.78883 .30529 L
.79002 .30523 L
.79112 .30517 L
.79357 .30499 L
.79614 .30473 L
.79845 .30444 L
.80346 .30355 L
.80762 .30251 L
.81179 .30116 L
.81908 .29788 L
.82507 .2942 L
.83714 .28417 L
.85054 .27081 L
.8646 .25609 L
.88011 .23818 L
.88613 .22979 L
.88889 .22524 L
.89104 .221 L
Mistroke
.89259 .21713 L
.89325 .21498 L
.89373 .21302 L
.89403 .21138 L
.89415 .21046 L
.89424 .20959 L
.8943 .20869 L
.89433 .20786 L
.89433 .20696 L
.8943 .20601 L
.89423 .20507 L
.89418 .20455 L
.89412 .20406 L
.89381 .20223 L
.89341 .20058 L
.89286 .19883 L
.89135 .1952 L
.88912 .19119 L
.88385 .18395 L
.87767 .17697 L
.87701 .17628 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04_o200io`00
B_l=03[o00<0ool0B_l200Oo0P0go`00AOl500go0P0ho`030?oo04Wo00<0ool02Ol00`3oo`0do`00
?_l701Co0P0fo`030?oo04Oo0P0=o`80=?l003_o0`0Mo`030?oo03?o00<0ool0Aol00`3oo`0>o`03
0?oo037o000io`808?l00`3oo`0co`030?oo04Ko00<0ool04?l00`3oo`0`o`00>?l00`3oo`0Qo`03
0?oo03;o00<0ool0AOl00`3oo`0Bo`80<?l003Ko0P0To`030?oo03;o00<0ool0AOl00`3oo`0Do`03
0?oo02go000fo`030?oo02?o00<0ool0<_l00`3oo`15o`030?oo01Go00<0ool0;?l003Go00<0ool0
9?l00`3oo`0bo`030?oo04Co00<0ool05ol00`3oo`0[o`00=Ol00`3oo`0To`030?oo03;o00<0ool0
A?l00`3oo`0Ho`030?oo02[o000do`030?oo02Co00<0ool0<ol00`3oo`14o`030?oo01Wo00<0ool0
:Ol003Co00<0ool08ol00`3oo`0Po`801?l200?o1005o`030?oo04?o00<0ool06ol00`3oo`0Xo`00
=?l00`3oo`0So`030?oo01oo00@0ool02?l00`3oo`06o`030?oo04?o00<0ool06ol202Wo000do`03
0?oo02;o00<0ool08?l0103oo`09o`030?oo00Go0P14o`030?oo01go00<0ool09_l003Co00<0ool0
8Ol00`3oo`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0@ol00`3oo`0Mo`030?oo02Ko000do`03
0?oo023o00<0ool08_l0103oo`08o`040?oo00Go00<0ool0@ol00`3oo`0No`030?oo02Go000do`03
0?oo01ko0P0Vo`802_l200Ko00<0ool0@_l00`3oo`0Po`030?oo02Co000do`030?oo01co0P0lo`03
0?oo04;o00<0ool08?l00`3oo`0To`00=?l00`3oo`0Jo`80?_l00`3oo`11o`030?oo027o00<0ool0
9?l003Co00<0ool06Ol00`3oo`0no`030?oo043o0P0So`030?oo02Co0007o`D00_l200Co0P0No`03
0?oo00;o1@0@o`804?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0>o`804ol500go00<0ool0
4_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Mo`030?oo00Co00<0ool03ol2013o00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`05o`P05ol00`3oo`0=o`030?oo01Co00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo01_o00<0ool01?l00`3oo`0?o`030?oo00oo00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02_l701oo00<0ool03Ol00`3oo`0Do`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201go1004o`030?oo013o00<0ool02?l400;o00@0ool02Ol202?o
00<0ool08Ol0103oo`06o`D09Ol00`3oo`0=o`030?oo01Co00<0ool02?l200?o0009o`030?oo00So
00<0ool07Ol00`3oo`04o`030?oo017o00<0ool03Ol0103oo`09o`030?oo02;o00<0ool08Ol0103o
o`04o`800ol00`3oo`0To`030?oo00co00<0ool05Ol00`3oo`08o`030?oo00;o0008o`802_l301ko
00<0ool00_l201Co00<0ool03Ol200[o0`0Ro`030?oo02;o0P02o`<01Ol302?o0P0>o`030?oo01Co
0P0:o`<00_l003Go00<0ool06Ol00`3oo`0mo`030?oo02;o100mo`030?oo02Ko000eo`030?oo01[o
0P0mo`030?oo01oo0`11o`030?oo02Ko000fo`030?oo01_o0P0ko`030?oo01_o1013o`030?oo02Oo
000fo`030?oo01go0P0io`030?oo01?o2016o`030?oo02So000fo`030?oo01oo0P0go`030?oo013o
0`1=o`030?oo02Wo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go0P08o`030?oo00Ko00<0ool01Ol00`3oo`06o`<01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol0103oool40003o`3o00Oo00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool0:?l202Go1002o`030?oo02[o00<0ool0
1_l3023o00<0ool0:Ol00`3oo`05o`030?oo02;o00<0ool01ol003So00<0ool09ol302co00<0ool0
0ol305Ko00<0ool0;Ol003So00<0ool0:_l302Wo00<0ool00`1Io`030?oo02go000io`030?oo02co
3P0Ko`<0Fol00`3oo`0^o`00>Ol00`3oo`0joa`0G?l00`3oo`0_o`00>_l00`3oo`1Do`030?oo05So
0P0bo`00>_l00`3oo`1Do`030?oo05Oo00<0ool0<_l003_o00<0ool0Dol00`3oo`1Fo`030?oo03?o
000lo`030?oo05;o00<0ool0EOl00`3oo`0do`00?Ol00`3oo`1Ao`030?oo05Co00<0ool0=Ol003ko
0`1@o`030?oo02Wo2`0No`80>?l0047o1@1;o`030?oo02Co1P0:o`P05?l203[o0016o`D0A_l00`3o
o`0Ro`806?l700Wo100lo`00Bol304?o00<0ool08?l2027o2@10o`00C_l202go0P04o`800ol400Go
00<0ool07_l206co001@o`030?oo02Wo00@0ool02?l00`3oo`06o`030?oo01co0P1^o`00DOl00`3o
o`0Xo`040?oo00Wo00<0ool01Ol201_o0P1`o`00D_l00`3oo`0Wo`040?oo00[o00<0ool01?l00`3o
o`0Ho`80L_l005?o00<0ool09_l0103oo`08o`040?oo00Go00<0ool05Ol307Co001Do`030?oo02Ko
0P0:o`801_l00`3oo`0Bo`<0Mol005Go00<0ool0>Ol00`3oo`0@o`80N_l005Ko00<0ool0>?l00`3o
o`0<o`@0O?l005Oo00<0ool0=ol00`3oo`08o`@0P?l005Oo00<0ool0=ol00`3oo`05o`<0Q?l005So
0P0go`030?oo00;o0`27o`00F_l203Go00<0o`000P2:o`00G?l4033o0`2=o`00H?l602Co1P000ol0
o`2=o`00I_l501Oo2007o`030?oo08co001[o`@03Ol600oo00<0ool0S?l006oo3@0Eo`030?oo08co
001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103o
o`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go
00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103o
o`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co
001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87674 .17261 m
.86235 .15722 L
.84595 .14275 L
.83812 .13711 L
.83009 .13245 L
.82566 .13046 L
.82172 .12906 L
.81986 .12853 L
.81785 .12806 L
.81612 .12772 L
.81517 .12757 L
.81427 .12745 L
.81343 .12736 L
.81252 .12727 L
.81169 .12721 L
.81093 .12717 L
.81014 .12715 L
.80943 .12714 L
.80861 .12714 L
.80784 .12716 L
.80699 .1272 L
.80657 .12723 L
.8061 .12726 L
.8045 .12742 L
.80361 .12754 L
.80265 .12769 L
.80093 .12802 L
.79725 .12901 L
.79383 .13031 L
.79084 .13179 L
.78768 .13377 L
.78235 .13836 L
.77957 .14159 L
.77716 .14503 L
.77295 .15312 L
.77096 .15825 L
.76935 .16324 L
.76669 .17317 L
.76404 .18376 L
.76263 .18863 L
.76112 .19297 L
.75958 .19656 L
.7576 .20018 L
.75536 .20332 L
.7526 .20624 L
.74966 .20857 L
.74657 .21044 L
.74251 .21228 L
.73842 .21367 L
.7333 .21499 L
Mistroke
.72808 .21603 L
.70528 .21953 L
.69528 .22136 L
.68497 .22382 L
.67459 .22701 L
.66543 .23035 L
.64585 .23762 L
.63563 .24065 L
.62548 .24302 L
.61428 .24509 L
.60162 .24711 L
.58055 .25072 L
.56943 .25321 L
.55974 .25588 L
.53956 .26262 L
.52959 .26596 L
.51995 .26883 L
.51003 .27127 L
.50088 .27302 L
.49607 .27375 L
.49089 .27437 L
.4858 .27484 L
.48308 .27503 L
.4806 .27517 L
.47816 .27527 L
.47595 .27534 L
.4748 .27536 L
.47415 .27537 L
.47356 .27538 L
.47293 .27539 L
.47227 .27539 L
.47165 .2754 L
.47107 .2754 L
.47003 .27539 L
.46893 .27539 L
.46799 .27538 L
.46696 .27536 L
.4647 .27531 L
.46259 .27525 L
.45872 .27511 L
.45519 .27495 L
.44714 .27454 L
.42866 .2736 L
.4093 .27259 L
.39017 .27131 L
.37128 .2696 L
.36221 .26853 L
.35254 .26708 L
.34301 .26521 L
.33348 .26281 L
Mistroke
.31226 .25562 L
.28631 .24366 L
.26473 .23046 L
.25998 .22674 L
.25633 .22335 L
.25383 .22045 L
.25277 .21887 L
.25202 .21747 L
.25173 .2168 L
.25149 .21607 L
.2513 .21535 L
.25123 .21497 L
.25118 .21461 L
.25113 .21395 L
.25114 .21337 L
.25117 .21304 L
.25121 .2127 L
.25134 .21207 L
.2515 .21152 L
.25173 .21092 L
.25235 .20973 L
.25312 .20863 L
.25401 .20762 L
.25676 .20519 L
.25996 .20293 L
.26897 .19748 L
.27762 .19222 L
.28676 .18572 L
.29154 .18159 L
.29536 .17761 L
.29837 .17378 L
.2998 .17156 L
.30095 .16947 L
.30191 .16736 L
.30261 .16544 L
.30317 .1634 L
.30339 .16228 L
.30355 .16123 L
.30364 .16034 L
.30369 .1594 L
.3037 .15852 L
.30367 .15773 L
.3036 .15681 L
.30346 .15582 L
.30329 .15491 L
.3031 .15408 L
.30251 .15227 L
.30181 .15064 L
.30085 .14894 L
.29964 .1472 L
Mistroke
.2966 .14399 L
.29326 .14148 L
.2897 .13947 L
.2856 .13773 L
.28192 .13656 L
.27774 .13557 L
.27536 .13515 L
.27318 .13484 L
.27066 .13456 L
.2695 .13446 L
.26826 .13438 L
.26711 .13431 L
.26607 .13427 L
.26502 .13423 L
.26404 .13421 L
.26289 .1342 L
.26184 .1342 L
.26129 .1342 L
.26069 .13421 L
.26006 .13422 L
.25948 .13423 L
.25839 .13427 L
.25724 .13431 L
.25518 .13441 L
.25052 .13473 L
.24202 .13554 L
.22506 .13748 L
.21548 .13866 L
.20488 .14012 L
.1949 .1418 L
.18583 .14378 L
.17684 .14645 L
.17248 .14815 L
.1682 .15023 L
.16406 .15277 L
.16062 .15548 L
.15793 .15817 L
.15543 .16138 L
.15344 .16467 L
.15189 .16797 L
.1505 .17181 L
.14933 .17616 L
.14886 .17842 L
.14843 .18093 L
.14811 .18318 L
.14784 .18561 L
.14762 .18805 L
.14747 .19028 L
.14741 .19136 L
.14736 .19253 L
Mistroke
.14732 .19364 L
.1473 .19467 L
.14728 .19582 L
.14727 .19692 L
.14727 .19791 L
.14728 .19898 L
.1473 .20015 L
.14732 .20074 L
.14734 .20139 L
.14742 .20362 L
.14754 .2057 L
.1477 .20795 L
.1481 .21201 L
.14875 .21686 L
.14966 .22197 L
.15188 .2312 L
.1548 .24029 L
.15876 .2502 L
.16796 .26797 L
.1742 .27714 L
.17779 .28151 L
.18182 .28568 L
.18554 .2889 L
.18985 .29193 L
.19365 .29407 L
.19796 .29599 L
.20261 .29757 L
.20717 .29873 L
.21604 .30035 L
.2258 .30187 L
.23102 .30287 L
.2366 .30427 L
.24172 .30595 L
.24633 .30787 L
.25505 .31274 L
.26383 .31936 L
.2726 .32728 L
.28068 .33483 L
.28831 .34122 L
.2953 .34593 L
.30314 .34974 L
.30736 .35124 L
.31192 .3525 L
.32026 .35417 L
.33921 .35712 L
.35729 .36102 L
.36539 .36292 L
.37448 .36475 L
.37961 .36554 L
.38214 .36585 L
Mistroke
.38446 .36608 L
.38672 .36627 L
.38919 .36643 L
.39055 .36649 L
.39182 .36654 L
.39303 .36657 L
.39431 .36659 L
.39569 .3666 L
.39695 .3666 L
.39765 .36659 L
.39841 .36658 L
.3998 .36655 L
.4011 .36651 L
.40248 .36645 L
.40497 .36633 L
.4106 .36593 L
.41681 .36535 L
.43906 .36298 L
.45832 .36153 L
.46789 .36086 L
.47688 .35996 L
.4815 .35934 L
.48651 .35852 L
.49743 .3562 L
.54329 .34272 L
.56477 .33727 L
.58327 .33262 L
.60109 .3264 L
.62056 .31739 L
.64058 .30788 L
.64938 .30435 L
.659 .30127 L
.66429 .29997 L
.66923 .29901 L
.67399 .29831 L
.67667 .29801 L
.67911 .29779 L
.6805 .29769 L
.68197 .2976 L
.68324 .29754 L
.68461 .29748 L
.68539 .29745 L
.68609 .29743 L
.68748 .2974 L
.68826 .29739 L
.68911 .29738 L
.68991 .29738 L
.69066 .29738 L
.69198 .29739 L
.69322 .2974 L
Mistroke
.6946 .29743 L
.69608 .29747 L
.69913 .29758 L
.70195 .29772 L
.70741 .29806 L
.71334 .29852 L
.72427 .29952 L
.74969 .30192 L
.75577 .3024 L
.76211 .30281 L
.76538 .30297 L
.7671 .30304 L
.76892 .30311 L
.77059 .30316 L
.77209 .30319 L
.7736 .30321 L
.77501 .30322 L
.7763 .30322 L
.77772 .30321 L
.77843 .3032 L
.77919 .30319 L
.78055 .30316 L
.78183 .30312 L
.78316 .30306 L
.78551 .30293 L
.7882 .30274 L
.79066 .3025 L
.79349 .30216 L
.79603 .30178 L
.80072 .30088 L
.80555 .29966 L
.81366 .29675 L
.8204 .29339 L
.82706 .28915 L
.85249 .26655 L
.86753 .25039 L
.88034 .23421 L
.88546 .22613 L
.88766 .2219 L
.88956 .21746 L
.89031 .21529 L
.8909 .21326 L
.89136 .2113 L
.89153 .21044 L
.89169 .20951 L
.8918 .20869 L
.8919 .20781 L
.89198 .20689 L
.89203 .20603 L
.89205 .20502 L
Mistroke
.89205 .20408 L
.89202 .20323 L
.89196 .2023 L
.89187 .20143 L
.89178 .20063 L
.89148 .19882 L
.89112 .19724 L
.89067 .19557 L
.88939 .19202 L
.88756 .18811 L
.88261 .18016 L
.87674 .17261 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1=o`D0>_l007go0P0;o`801Ol00`3oo`1;o`<0
0ol303So001>o`P0>ol00`3oo`19o`802Ol203Ko0014o`X01ol403So00<0ool0B?l00`3oo`0;o`80
=?l003oo1@0Eo`80=_l00`3oo`17o`030?oo00ko0P0bo`00>ol401_o00<0ool0=?l00`3oo`16o`03
0?oo017o00<0ool0;ol003So0`0Po`80=?l00`3oo`15o`030?oo01?o00<0ool0;_l003Oo00<0ool0
8_l00`3oo`0bo`030?oo04Go00<0ool05?l00`3oo`0]o`00=_l00`3oo`0So`030?oo03;o00<0ool0
AOl00`3oo`0Eo`030?oo02co000eo`809Ol00`3oo`0bo`030?oo04Go00<0ool05_l00`3oo`0[o`00
=?l00`3oo`0Uo`030?oo03;o00<0ool0A?l00`3oo`0Ho`030?oo02[o000do`030?oo02Go00<0ool0
<_l00`3oo`14o`030?oo01Wo00<0ool0:Ol003Co00<0ool09?l203Co00<0ool0@ol00`3oo`0Ko`03
0?oo02So000co`030?oo02Go00<0ool07ol200Co0P03o`@01Ol00`3oo`13o`030?oo01_o00<0ool0
:?l003?o00<0ool09?l00`3oo`0Oo`040?oo00So00<0ool01_l00`3oo`13o`030?oo01co00<0ool0
9ol003?o00<0ool08ol00`3oo`0Po`040?oo00Wo00<0ool01Ol204Co00<0ool07Ol00`3oo`0Vo`00
<ol00`3oo`0Ro`030?oo01_o1002o`040?oo00[o00<0ool01?l00`3oo`12o`030?oo01ko00<0ool0
9_l003?o00<0ool08?l202Co00@0ool02?l0103oo`05o`030?oo04;o00<0ool07ol00`3oo`0Uo`00
<ol00`3oo`0No`809ol200[o0P06o`030?oo04;o00<0ool07ol00`3oo`0Uo`00<ol00`3oo`0Lo`80
?Ol00`3oo`12o`030?oo01oo0P0Vo`00<ol00`3oo`0Ko`030?oo03go00<0ool0@Ol00`3oo`0Qo`03
0?oo02Co000co`030?oo01[o00<0ool0?_l00`3oo`10o`030?oo02;o00<0ool09?l000Oo1@02o`80
1?l201go00<0ool00ol5017o00<0ool03_l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0>o`80
4ol500go00<0ool04_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Lo`030?oo00Go00<0ool0
4?l00`3oo`0>o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool02ol201Oo00<0ool0
3?l201Ko00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01[o00<0ool01Ol00`3oo`0@o`80
3ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Co1@0Io`030?oo00co00<0ool0
5Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Mo`@01?l00`3oo`0Ao`030?oo00Oo
1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol901ko00<0ool03?l00`3oo`0Eo`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01co00<0ool01Ol00`3oo`0Bo`030?oo00co00@0ool02Ol00`3o
o`0Ro`030?oo027o00@0ool02?l302Go00<0ool03?l00`3oo`0Eo`030?oo00So00<0ool00_l000So
0P0:o`<07Ol00`3oo`03o`805Ol00`3oo`0<o`802_l302;o00<0ool08_l200Oo0P001?l0000So`80
3Ol00`3oo`0Eo`802_l300;o000do`030?oo01_o0P0mo`030?oo02So0`0ho`030?oo02Ko000eo`03
0?oo01co0P0ko`030?oo02Go0`0jo`030?oo02Oo000eo`030?oo01ko0P0io`030?oo02?o0P0lo`03
0?oo02So000eo`030?oo023o0P0go`030?oo01go1P0mo`030?oo02Wo000fo`030?oo027o0P0eo`03
0?oo01Ko1`12o`030?oo02[o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Co00<0o`002?l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`0300?o00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l300Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`80
2?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Xo`80:?l20003o`3o02_o
00<0ool02_l401_o00<0ool0:Ol00`3oo`04o`030?oo02?o00<0ool01ol003Oo00<0ool0:Ol302_o
00<0ool01_l4057o00<0ool0;_l003So00<0ool0:ol302So00<0ool01?l205Co00<0ool0;ol003So
00<0ool0;_l8023o00@0oooo0`1Eo`030?oo033o000io`030?oo03Go400=o`L0Eol00`3oo`0ao`00
>_l00`3oo`14o`d00ol00`3oo`1Fo`80=?l003_o00<0ool0Dol00`3oo`1Eo`030?oo03Co000lo`03
0?oo05;o00<0ool0E?l00`3oo`0eo`00?Ol00`3oo`1Ao`030?oo05?o00<0ool0=_l003ko00<0ool0
D?l00`3oo`1Co`030?oo03Ko000oo`80D?l00`3oo`1?o`@0>Ol0047o1P1:o`030?oo02So500Ao`80
?Ol004Oo1016o`030?oo02?o1@0Doa40?ol004_o0`13o`030?oo02;o00<0ool0Iol004ko00<0ool0
;?l200Co0P03o`@01Ol00`3oo`0Po`80J_l004oo0P0[o`040?oo00So00<0ool01_l00`3oo`0No`80
K?l0057o0P0Yo`040?oo00Wo00<0ool01Ol201go0P1^o`00Dol00`3oo`0Vo`040?oo00[o00<0ool0
1?l00`3oo`0Io`<0L?l005?o00<0ool09_l0103oo`08o`040?oo00Go00<0ool05_l307?o001Do`03
0?oo02Ko0P0:o`801_l00`3oo`0Do`80M_l005Go00<0ool0>Ol00`3oo`0Ao`<0N?l005Ko00<0ool0
>?l00`3oo`0=o`@0Nol005Oo0P0ho`030?oo00[o0`1oo`00FOl00`3oo`0eo`030?oo00Ko1022o`00
F_l00`3oo`0do`030?oo00;o1026o`00Fol5037o1@2:o`00H?l802Oo0`2>o`00J?l201go2002o`03
0?oo08co001Zo`@03ol:00[o00<0ool0S?l006ko3`0Do`030?oo08co002Ao`030?oo08co001mo`80
1?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D0
1?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0
S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<0
1_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80
SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`80
2_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87646 .16893 m
.86325 .15334 L
.84736 .1392 L
.83884 .13352 L
.83098 .12957 L
.82653 .1279 L
.82253 .12676 L
.82034 .12628 L
.81798 .12587 L
.8169 .12572 L
.81577 .12559 L
.81479 .1255 L
.81428 .12546 L
.81374 .12543 L
.81275 .12538 L
.81183 .12535 L
.81088 .12534 L
.81003 .12535 L
.8092 .12537 L
.8083 .12541 L
.80746 .12547 L
.80669 .12553 L
.80496 .12572 L
.80309 .126 L
.80115 .12637 L
.79936 .1268 L
.79614 .12778 L
.79327 .12891 L
.78716 .13228 L
.78426 .13447 L
.78137 .13717 L
.77671 .143 L
.77445 .14677 L
.77253 .15065 L
.76886 .16048 L
.76407 .17886 L
.76278 .18385 L
.76128 .18898 L
.7598 .19318 L
.75795 .19743 L
.75562 .20157 L
.75303 .20507 L
.75011 .20808 L
.74706 .21053 L
.74341 .2128 L
.73896 .21494 L
.73386 .21685 L
.72872 .21839 L
.70882 .22299 L
.68699 .22854 L
Mistroke
.66631 .23575 L
.64751 .24255 L
.62764 .24785 L
.61568 .25017 L
.60454 .25206 L
.56194 .26046 L
.52418 .27085 L
.51363 .27318 L
.50371 .27489 L
.49832 .27562 L
.49348 .27615 L
.48819 .2766 L
.48532 .27678 L
.48265 .27691 L
.48039 .27699 L
.47829 .27705 L
.4771 .27707 L
.47602 .27709 L
.47483 .2771 L
.47424 .2771 L
.4736 .2771 L
.47296 .2771 L
.47237 .2771 L
.47126 .27709 L
.47007 .27708 L
.46941 .27707 L
.4688 .27706 L
.46666 .27701 L
.46442 .27695 L
.46047 .2768 L
.4568 .27662 L
.44844 .27616 L
.41146 .27381 L
.38976 .27187 L
.3705 .26942 L
.35277 .26633 L
.33567 .26218 L
.31588 .25549 L
.29161 .24446 L
.27063 .23179 L
.26596 .22825 L
.26181 .2246 L
.25873 .22128 L
.25667 .21838 L
.25594 .21699 L
.25563 .21627 L
.25535 .21552 L
.25516 .21487 L
.25501 .21415 L
.25491 .21352 L
Mistroke
.25487 .21292 L
.25487 .21221 L
.25492 .21153 L
.25502 .21093 L
.25517 .21028 L
.2554 .20958 L
.2557 .20885 L
.25642 .20753 L
.25728 .20632 L
.25841 .20501 L
.26095 .20264 L
.26713 .19812 L
.28451 .18652 L
.2892 .18286 L
.2938 .17872 L
.29752 .17466 L
.30033 .17082 L
.30249 .16684 L
.3033 .16475 L
.30364 .16368 L
.30392 .16254 L
.30412 .16155 L
.30425 .16066 L
.30433 .1598 L
.30437 .15891 L
.30437 .15794 L
.30431 .15705 L
.30426 .15655 L
.3042 .15609 L
.30401 .15507 L
.30381 .15425 L
.30355 .15338 L
.30296 .15188 L
.30217 .15032 L
.30111 .14867 L
.29875 .14595 L
.29607 .14367 L
.29256 .14144 L
.28829 .1394 L
.28393 .13787 L
.27969 .13676 L
.27747 .13631 L
.27537 .13595 L
.27142 .13543 L
.26919 .13522 L
.26715 .13507 L
.26608 .13502 L
.26492 .13497 L
.2637 .13493 L
.26257 .1349 L
Mistroke
.2614 .13489 L
.26082 .13489 L
.26017 .13489 L
.25903 .13489 L
.25797 .13491 L
.25733 .13492 L
.25672 .13494 L
.25558 .13498 L
.25303 .13509 L
.25082 .13521 L
.24877 .13535 L
.24414 .13572 L
.22715 .13751 L
.21686 .13875 L
.20668 .14014 L
.19738 .14163 L
.18723 .14369 L
.17753 .14637 L
.17258 .14815 L
.16826 .15004 L
.16135 .15404 L
.15799 .15665 L
.15526 .15927 L
.15272 .16227 L
.15041 .16572 L
.14871 .16892 L
.14717 .17257 L
.14584 .17668 L
.14487 .18072 L
.14442 .18312 L
.14402 .18569 L
.14375 .18791 L
.14353 .19038 L
.14337 .19269 L
.14331 .194 L
.14327 .19521 L
.14325 .19628 L
.14323 .19745 L
.14323 .19804 L
.14323 .19868 L
.14325 .19983 L
.14327 .20094 L
.14331 .20215 L
.14336 .2033 L
.14342 .20436 L
.14356 .20637 L
.14375 .20851 L
.1444 .21378 L
.14524 .21872 L
.14746 .22793 L
Mistroke
.15044 .23697 L
.15446 .24656 L
.16374 .26361 L
.16947 .27191 L
.1763 .28 L
.18339 .2865 L
.19039 .29128 L
.19446 .29342 L
.19893 .29534 L
.20731 .29797 L
.21671 .30006 L
.22733 .3022 L
.23213 .30334 L
.23715 .30477 L
.24594 .30807 L
.25504 .31282 L
.26454 .31927 L
.2822 .33315 L
.29051 .33891 L
.29832 .34322 L
.30669 .34659 L
.3112 .34795 L
.31602 .34914 L
.33334 .3523 L
.36856 .35975 L
.37316 .36068 L
.37825 .36158 L
.38281 .36223 L
.38774 .36277 L
.39049 .363 L
.39196 .36309 L
.39356 .36318 L
.39502 .36324 L
.39637 .36329 L
.39766 .36332 L
.39903 .36334 L
.40028 .36335 L
.40162 .36335 L
.40289 .36335 L
.40404 .36333 L
.40541 .3633 L
.40689 .36326 L
.40958 .36316 L
.41478 .36288 L
.41964 .36255 L
.43066 .3617 L
.4523 .36029 L
.46129 .35987 L
.47084 .35934 L
.47638 .35892 L
Mistroke
.48144 .35841 L
.49151 .357 L
.5031 .35471 L
.51428 .35196 L
.53633 .34588 L
.57579 .33598 L
.5948 .33029 L
.6151 .32195 L
.63478 .3129 L
.65299 .30574 L
.65784 .30424 L
.66314 .30282 L
.66774 .30179 L
.67286 .30086 L
.67831 .30009 L
.68133 .29976 L
.68418 .29951 L
.68667 .29934 L
.68931 .29919 L
.69071 .29913 L
.69223 .29907 L
.69354 .29903 L
.69493 .299 L
.69637 .29898 L
.6977 .29896 L
.69914 .29896 L
.69996 .29896 L
.70072 .29896 L
.70211 .29897 L
.70339 .29898 L
.70487 .299 L
.70628 .29903 L
.70889 .29909 L
.71132 .29916 L
.7169 .29938 L
.72947 .29999 L
.74146 .30061 L
.74714 .30087 L
.75309 .3011 L
.75619 .30119 L
.75781 .30123 L
.75955 .30126 L
.76037 .30127 L
.76114 .30128 L
.76258 .3013 L
.76393 .3013 L
.76538 .3013 L
.76668 .3013 L
.76789 .30129 L
.76922 .30127 L
Mistroke
.77063 .30124 L
.7721 .3012 L
.77349 .30115 L
.77607 .30104 L
.77844 .30091 L
.78061 .30075 L
.78543 .3003 L
.79049 .29964 L
.79574 .29871 L
.8005 .29762 L
.8047 .29643 L
.81223 .29369 L
.81966 .29014 L
.83383 .28086 L
.84666 .26992 L
.85981 .25667 L
.87348 .24028 L
.87988 .23092 L
.88465 .22224 L
.8865 .218 L
.88788 .21405 L
.88893 .20996 L
.88932 .20781 L
.88949 .2066 L
.88961 .20549 L
.88969 .20444 L
.88974 .20346 L
.88977 .20258 L
.88977 .20162 L
.88975 .20058 L
.88973 .20005 L
.88969 .19948 L
.88961 .19845 L
.88951 .1975 L
.88924 .19563 L
.88891 .19392 L
.88788 .19006 L
.88636 .18589 L
.88423 .1813 L
.87919 .17283 L
.87646 .16893 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o`P0>Ol007go0P0;o`801Ol00`3oo`1;o`03
0?oo00Go100eo`00C_l603go00<0ool0B?l300co00<0ool0<_l004Co2`05o`@0>Ol00`3oo`17o`03
0?oo00ko00<0ool0<Ol003oo1@0Do`80=ol00`3oo`16o`030?oo013o00<0ool0<?l003_o100Ko`80
=Ol00`3oo`16o`030?oo017o0P0`o`00>?l3027o00<0ool0<_l00`3oo`16o`030?oo01?o00<0ool0
;Ol003Ko0P0To`030?oo03;o00<0ool0AOl00`3oo`0Eo`030?oo02co000fo`030?oo02?o00<0ool0
<_l00`3oo`14o`030?oo01Oo00<0ool0:ol003Go00<0ool09?l00`3oo`0bo`030?oo04Co00<0ool0
6?l00`3oo`0Zo`00=?l00`3oo`0Uo`030?oo03;o00<0ool0A?l00`3oo`0Io`030?oo02Wo000co`80
9ol00`3oo`0bo`030?oo04Co00<0ool06_l00`3oo`0Xo`00<ol00`3oo`0Uo`030?oo03?o00<0ool0
@ol00`3oo`0Ko`030?oo02So000bo`030?oo02Ko00<0ool07ol200Co0P03o`@01Ol00`3oo`13o`03
0?oo01co00<0ool09ol003;o00<0ool09?l2027o00@0ool02?l00`3oo`06o`030?oo04?o00<0ool0
7?l00`3oo`0Wo`00<_l00`3oo`0So`030?oo027o00@0ool02Ol00`3oo`05o`80@ol00`3oo`0No`03
0?oo02Ko000bo`030?oo02;o00<0ool07?l400;o00@0ool02_l00`3oo`04o`030?oo04;o00<0ool0
7_l00`3oo`0Vo`00<_l00`3oo`0Po`809Ol0103oo`08o`040?oo00Go00<0ool0@_l00`3oo`0Oo`03
0?oo02Go000bo`030?oo01oo00<0ool09_l200[o0P06o`030?oo04;o00<0ool07ol00`3oo`0Uo`00
<_l00`3oo`0Mo`80?Ol00`3oo`11o`030?oo023o00<0ool09Ol003;o00<0ool07?l00`3oo`0mo`03
0?oo043o00<0ool08Ol00`3oo`0Uo`00<_l00`3oo`0Ko`030?oo03ko00<0ool0@?l00`3oo`0Qo`03
0?oo02Go0007o`D00_l200Co0P0Lo`030?oo00Co1@0Ao`803ol200Co0P04o`808ol00`3oo`0Ro`80
1?l200Co0P0>o`804ol500co00<0ool04ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ko`03
0?oo00Ko00<0ool04Ol00`3oo`0=o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
3?l00`3oo`0Eo`030?oo00co00<0ool05Ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
6Ol00`3oo`06o`030?oo017o0P0>o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
2?l201So00<0ool02ol00`3oo`0Fo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201co
1@04o`030?oo01;o00<0ool01_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`801ol401[o
00<0ool02ol00`3oo`0Fo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01co00<0ool01Ol00`3o
o`0Co`030?oo00_o00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol0103oool501ko00<0ool0
2ol00`3oo`0Fo`030?oo00So00<0ool00_l000So0P0:o`<07?l00`3oo`04o`805_l00`3oo`0;o`80
2_l302;o00<0ool08_l200Wo1@0Ro`803Ol00`3oo`0Eo`802_l300;o000co`030?oo01go00<0ool0
>ol00`3oo`0[o`80=Ol00`3oo`0Wo`00=?l00`3oo`0Mo`80>ol00`3oo`0Xo`<0=_l00`3oo`0Xo`00
=?l00`3oo`0Oo`030?oo03So00<0ool09_l203So00<0ool0:Ol003Go00<0ool07ol203So00<0ool0
8_l403Wo00<0ool0:_l003Go00<0ool08Ol203Ko00<0ool07Ol503co00<0ool0:ol000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko
00<0ool01Ol00`3oo`06o`040?ooo`<00_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Go1006o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Oo000<o`030?oo02Oo00<0o`00:Ol302co00<0ool04?l601?o00<0ool0:Ol00`3oo`04o`03
0?oo02?o00<0ool01ol003Oo00<0ool0:_l402Wo00<0ool03?l404_o00<0ool0;_l003Oo00<0ool0
;_l402Go00<0ool02Ol304ko00<0ool0;ol003So00<0ool0<Ol5023o00<0ool01Ol4053o00<0ool0
<?l003Wo00<0ool0=Ol>01;o00<0o`001@1Bo`80<ol003[o00<0ool0@_lD05Oo00<0ool0<ol003[o
00<0ool0E?l00`3oo`1Eo`030?oo03Co000ko`030?oo05?o00<0ool0E?l00`3oo`0eo`00??l00`3o
o`1Bo`030?oo05?o00<0ool0=_l003go0P1Bo`030?oo057o0P0io`00?ol304oo00<0ool0COl403_o
0012o`D0B_l00`3oo`0[oa<02Ol603oo0017o`<0Aol00`3oo`0Vo`D04ol:04Co001:o`@0@ol00`3o
o`0To`80I_l004ko00<0ool0;?l200Co0P03o`@01Ol00`3oo`0Ro`80J?l004oo0P0[o`040?oo00So
00<0ool01_l00`3oo`0Oo`<0J_l0057o0P0Yo`040?oo00Wo00<0ool01Ol201ko0P1]o`00Dol00`3o
o`0Vo`040?oo00[o00<0ool01?l00`3oo`0Jo`<0Kol005Co0P0Vo`040?oo00So00@0ool01Ol00`3o
o`0Ho`80L_l005Ko00<0ool09?l200[o0P06o`030?oo01Go0`1do`00Eol00`3oo`0go`030?oo01?o
0P1go`00F?l00`3oo`0fo`030?oo013o0`1io`00FOl00`3oo`0eo`030?oo00co101lo`00F_l203Go
00<0ool02?l4083o001Lo`H0;ol00`3oo`03o`D0Q?l006;o1@0Zo`030?l000<0ROl006Oo1@0Qo`H0
SOl006co100Ao``01?l00`3oo`2<o`00L?lA013o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`04
0?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`00
1Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0
S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87619 .16526 m
.87084 .15754 L
.86415 .14947 L
.85634 .14171 L
.84877 .13564 L
.83999 .13023 L
.83508 .12793 L
.83041 .12619 L
.82609 .12497 L
.82211 .12416 L
.82004 .12386 L
.81777 .12361 L
.81663 .12353 L
.816 .12349 L
.81541 .12346 L
.81486 .12344 L
.81426 .12342 L
.81321 .1234 L
.81267 .1234 L
.81211 .12341 L
.8111 .12343 L
.81002 .12348 L
.8094 .12351 L
.80883 .12355 L
.80687 .12373 L
.80479 .124 L
.80279 .12434 L
.80092 .12473 L
.79749 .12565 L
.79362 .12704 L
.79019 .12862 L
.78663 .13072 L
.78309 .13337 L
.77783 .13877 L
.77525 .14236 L
.77312 .146 L
.76978 .15348 L
.76696 .16223 L
.76438 .17242 L
.76203 .18209 L
.75959 .19037 L
.75621 .19853 L
.75401 .20237 L
.75131 .20609 L
.74542 .21182 L
.74163 .21447 L
.73773 .21669 L
.72781 .22096 L
.68807 .23348 L
.66909 .24043 L
Mistroke
.64892 .24757 L
.62825 .25301 L
.60773 .25693 L
.56653 .26452 L
.54672 .26901 L
.52496 .27378 L
.51446 .27567 L
.50871 .27653 L
.50336 .27722 L
.49843 .27775 L
.49399 .27814 L
.48912 .27846 L
.48648 .27859 L
.48399 .27869 L
.48288 .27873 L
.48168 .27876 L
.48065 .27878 L
.47953 .2788 L
.47829 .27881 L
.47762 .27882 L
.477 .27882 L
.47588 .27882 L
.4747 .27882 L
.47405 .27881 L
.47345 .27881 L
.4721 .27879 L
.47091 .27877 L
.46967 .27875 L
.46752 .27869 L
.46517 .27861 L
.45653 .2782 L
.4485 .27771 L
.43124 .27653 L
.41131 .27491 L
.40019 .27377 L
.38997 .27248 L
.37067 .26935 L
.35379 .2658 L
.3348 .26068 L
.31504 .25361 L
.29201 .24267 L
.27203 .22982 L
.26464 .2232 L
.26307 .22142 L
.26163 .21954 L
.26061 .21795 L
.25972 .21623 L
.25912 .21466 L
.25889 .21381 L
.2588 .2134 L
Mistroke
.25872 .21302 L
.25863 .21229 L
.25858 .21163 L
.25858 .21092 L
.2586 .21052 L
.25864 .21015 L
.25869 .20975 L
.25877 .20932 L
.25895 .20854 L
.25919 .20779 L
.25946 .2071 L
.26025 .20555 L
.26135 .20392 L
.26377 .20119 L
.2666 .19862 L
.27463 .19262 L
.29071 .18076 L
.29467 .1772 L
.29834 .1733 L
.30097 .16987 L
.30312 .16615 L
.30396 .16416 L
.30426 .16329 L
.30453 .16234 L
.30473 .16151 L
.30489 .16063 L
.30501 .1597 L
.30507 .15885 L
.30508 .15833 L
.30508 .15784 L
.30506 .15732 L
.30501 .15677 L
.30496 .15629 L
.30489 .15585 L
.30469 .15486 L
.30446 .15398 L
.30419 .15319 L
.30342 .15146 L
.3025 .14991 L
.30148 .14854 L
.29862 .14568 L
.29552 .14345 L
.29173 .1414 L
.28725 .13958 L
.28283 .13825 L
.27775 .13713 L
.27514 .13669 L
.27225 .13631 L
.26968 .13604 L
.26734 .13585 L
Mistroke
.26605 .13577 L
.26465 .1357 L
.26332 .13565 L
.2621 .13561 L
.26098 .13559 L
.25993 .13558 L
.25878 .13557 L
.25755 .13558 L
.25694 .13558 L
.25627 .13559 L
.25505 .13561 L
.25387 .13565 L
.25278 .13568 L
.25028 .13579 L
.248 .13591 L
.24287 .13627 L
.23358 .13711 L
.21569 .13916 L
.1965 .14204 L
.18551 .14434 L
.17567 .14713 L
.171 .14882 L
.16621 .15092 L
.16227 .15301 L
.1584 .15548 L
.15523 .15794 L
.15207 .16091 L
.14739 .16682 L
.1453 .17041 L
.14346 .17444 L
.14217 .17806 L
.14103 .18218 L
.14013 .18679 L
.13975 .18943 L
.1395 .19186 L
.13941 .19297 L
.13933 .19414 L
.13926 .19544 L
.13922 .19664 L
.13919 .19777 L
.13918 .19882 L
.13919 .19997 L
.13919 .20062 L
.13921 .20122 L
.13924 .20243 L
.13929 .20357 L
.13943 .20571 L
.13962 .20801 L
.1399 .2105 L
.14057 .21498 L
Mistroke
.14138 .21907 L
.14389 .2284 L
.14752 .23809 L
.15666 .25563 L
.16772 .27102 L
.17503 .27885 L
.18259 .28525 L
.18987 .29 L
.19829 .29406 L
.2069 .29707 L
.21674 .29969 L
.22732 .30228 L
.23697 .30506 L
.24586 .3084 L
.25496 .31282 L
.27324 .32431 L
.28206 .33014 L
.29042 .33509 L
.2987 .33914 L
.30647 .3421 L
.31504 .34458 L
.32446 .34668 L
.34397 .35047 L
.36306 .35469 L
.37255 .35672 L
.3814 .35829 L
.38585 .35891 L
.39075 .35945 L
.39301 .35965 L
.39544 .35982 L
.39772 .35995 L
.39982 .36005 L
.40096 .36009 L
.4022 .36013 L
.40346 .36015 L
.40479 .36018 L
.40596 .36019 L
.40722 .36019 L
.40841 .36019 L
.40949 .36019 L
.41087 .36017 L
.41214 .36015 L
.41354 .36012 L
.41503 .36009 L
.42028 .35991 L
.4594 .3584 L
.46424 .35825 L
.46952 .35806 L
.47452 .35782 L
.47918 .35752 L
Mistroke
.48462 .35707 L
.49048 .35643 L
.50102 .35487 L
.51206 .35272 L
.52412 .34996 L
.54572 .34462 L
.56573 .33983 L
.58656 .33435 L
.606 .32756 L
.62439 .31974 L
.64274 .31204 L
.65254 .30853 L
.66304 .30546 L
.66835 .3042 L
.67426 .30303 L
.67973 .30216 L
.6848 .30151 L
.69004 .30098 L
.69574 .30055 L
.70083 .30028 L
.7034 .30018 L
.70624 .30009 L
.70901 .30002 L
.71048 .29999 L
.71203 .29996 L
.7134 .29995 L
.7149 .29993 L
.71627 .29992 L
.71756 .29991 L
.71891 .2999 L
.72018 .2999 L
.72133 .29989 L
.72259 .29989 L
.72395 .29989 L
.72541 .2999 L
.72678 .2999 L
.72802 .2999 L
.73389 .29992 L
.73698 .29993 L
.73853 .29994 L
.73941 .29994 L
.74023 .29994 L
.74101 .29994 L
.74175 .29994 L
.74313 .29994 L
.74461 .29994 L
.74545 .29993 L
.74622 .29993 L
.74748 .29992 L
.74888 .29991 L
Mistroke
.75033 .29989 L
.75169 .29988 L
.75456 .29983 L
.75721 .29977 L
.76011 .29968 L
.76317 .29956 L
.76877 .29926 L
.77375 .29888 L
.77851 .2984 L
.78354 .29776 L
.7923 .29623 L
.79688 .29517 L
.80104 .29405 L
.81005 .29096 L
.81833 .28727 L
.82679 .28255 L
.84254 .2711 L
.85696 .25747 L
.87014 .24178 L
.87999 .2262 L
.88193 .22225 L
.88373 .21799 L
.88502 .21433 L
.88613 .21033 L
.88663 .20803 L
.88699 .20591 L
.88715 .20475 L
.88727 .20366 L
.88733 .20306 L
.88738 .2024 L
.88742 .20181 L
.88745 .20126 L
.88749 .20026 L
.8875 .19916 L
.88749 .19809 L
.88748 .19748 L
.88746 .19693 L
.8874 .19595 L
.88732 .19488 L
.8871 .19293 L
.88685 .19122 L
.88651 .18937 L
.8857 .186 L
.88358 .17965 L
.88016 .17216 L
.87619 .16526 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo04ko1@0io`00O?l0103oo`09o`030?l000Go00<0ool0Bol300Co100fo`00OOl200_o0P05o`03
0?oo04Wo0P0;o`80=?l0097o00<0ool0B?l00`3oo`0=o`030?oo037o0018oa00>Ol00`3oo`16o`80
4Ol2037o000oo`T04?l203Oo00<0ool0A_l00`3oo`0Bo`030?oo02ko000lo`<06ol203Go00<0ool0
AOl00`3oo`0Do`030?oo02go000io`<08?l00`3oo`0bo`030?oo04Co00<0ool05_l00`3oo`0/o`00
=ol202?o00<0ool0<_l00`3oo`14o`030?oo01Oo00<0ool0:ol003Ko00<0ool09?l00`3oo`0ao`03
0?oo04Co00<0ool06?l00`3oo`0Zo`00=?l202Oo00<0ool0<Ol00`3oo`14o`030?oo01Wo00<0ool0
:Ol003?o00<0ool09_l203?o00<0ool0A?l00`3oo`0Jo`030?oo02So000co`030?oo02Ko00<0ool0
<_l00`3oo`13o`030?oo01_o00<0ool0:?l003;o00<0ool09ol00`3oo`0bo`030?oo04?o00<0ool0
7?l00`3oo`0Wo`00<_l00`3oo`0Vo`030?oo01oo0P04o`800ol400Go00<0ool0@ol00`3oo`0Lo`03
0?oo02Oo000ao`030?oo02Ko00<0ool07ol0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0No`03
0?oo02Ko000ao`030?oo02Go00<0ool08?l0103oo`09o`030?oo00Go0P13o`030?oo01ko00<0ool0
9_l0037o00<0ool08ol201go1002o`040?oo00[o00<0ool01?l00`3oo`12o`030?oo01ko00<0ool0
9_l0037o00<0ool08_l00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`12o`030?oo01ko00<0ool0
9_l0037o00<0ool08?l202Oo0P0:o`801_l00`3oo`11o`030?oo01oo0P0Wo`00<Ol00`3oo`0No`80
?Ol00`3oo`11o`030?oo023o00<0ool09Ol0037o00<0ool07_l00`3oo`0lo`030?oo043o00<0ool0
8Ol00`3oo`0Uo`00<Ol00`3oo`0Mo`030?oo03go00<0ool0?ol00`3oo`0Qo`809ol000Oo1@02o`80
1?l201_o00<0ool01Ol501;o0P0>o`801?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool0
4_l500_o00<0ool05?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Jo`030?oo00Oo00<0ool0
4_l00`3oo`0<o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`03
0?oo00_o00<0ool05_l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool06?l00`3oo`07o`03
0?oo01;o0P0=o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02?l301Oo00<0ool0
2ol00`3oo`0Fo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201_o00<0o`000`04o`03
0?oo01?o00<0ool01Ol400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802?l301[o00<0ool0
2ol00`3oo`0Fo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01_o00<0ool01_l00`3oo`0Do`03
0?oo00[o00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`05o`807Ol00`3oo`0:o`03
0?oo01Oo00<0ool02?l00`3oo`02o`002?l200[o0`0Ko`030?oo00Go0P0Go`030?oo00[o0P0:o`<0
8_l00`3oo`0Ro`802_l300;o0`0No`803?l00`3oo`0Fo`802_l300;o000bo`030?oo01oo00<0ool0
>_l00`3oo`0ao`80;_l00`3oo`0Xo`00<ol00`3oo`0Oo`80>_l00`3oo`0]o`@0<?l00`3oo`0Xo`00
<ol00`3oo`0Qo`80>?l00`3oo`0Xo`D0<ol00`3oo`0Yo`00=?l00`3oo`0Ro`80=_l00`3oo`0Vo`80
=ol00`3oo`0Zo`00=?l00`3oo`0To`80=?l00`3oo`0So`<0>?l00`3oo`0[o`003?oo00`02Ol000co
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3o
o`05o`030?oo00Ko00@0oooo0P03o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Oo000<o`030?oo02Oo00<0o`00:?l402co00<0ool05ol600co00<0ool0:Ol00`3oo`03o`03
0?oo02Co00<0ool01ol003Ko00<0ool0:ol402Wo00<0ool04Ol604?o00<0ool0;ol003Oo00<0ool0
;_l502Co00<0ool03?l504Wo00<0ool0;ol003Oo00<0ool0<ol4023o00<0ool01ol504go00<0ool0
<?l003So00<0ool0=_l701Wo00<0ool00_l5057o00<0ool0<Ol003Wo00<0ool0??l?00So1`1Do`80
=?l003[o00<0ool0B_l800;o00<0ool0E?l203Ko000ko`80E?l00`3oo`1Bo`80>?l003go00<0ool0
DOl00`3oo`1@o`80>_l003ko00<0ool0D?l00`3oo`1>o`80??l003oo0`1?o`030?oo04Wo1@0no`00
@_l504[o00<0ool0<?lI04?o0017o`<0Aol00`3oo`0Yo`L0G?l004[o0P15o`030?oo02Oo0P1So`00
C?l202oo0P04o`800ol400Go00<0ool09?l306Go001>o`80;?l0103oo`08o`030?oo00Ko00<0ool0
8Ol306So001@o`80:_l0103oo`09o`030?oo00Go0P0Po`80Jol005;o0P0Xo`040?oo00[o00<0ool0
1?l00`3oo`0Mo`80KOl005Co0P0Vo`040?oo00So00@0ool01Ol00`3oo`0Ko`80Kol005Ko0P0Uo`80
2_l200Ko00<0ool05ol4077o001Ho`030?oo03Ko00<0ool04ol407Go001Io`030?oo03Go00<0ool0
4?l307Wo001Jo`@0<ol00`3oo`0<o`@0O?l005ko0`0`o`030?oo00So1020o`00HOl402co00<0ool0
1Ol308Co001Uo`D09ol00`3o000508Oo001Zo`H07?l708go001`oa`01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103o
o`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go
00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co
001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o
003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87591 .16159 m
.87118 .15371 L
.86505 .14559 L
.85772 .13802 L
.85018 .13208 L
.84561 .12924 L
.84115 .12695 L
.837 .12522 L
.83252 .12374 L
.83018 .12311 L
.82759 .12254 L
.8252 .12211 L
.82301 .1218 L
.82178 .12166 L
.82043 .12154 L
.81969 .12148 L
.81901 .12144 L
.81831 .1214 L
.81766 .12137 L
.81705 .12135 L
.81639 .12134 L
.81524 .12133 L
.81459 .12133 L
.81391 .12134 L
.81332 .12135 L
.81268 .12138 L
.81151 .12143 L
.81043 .12151 L
.808 .12174 L
.80588 .12202 L
.80365 .12239 L
.79988 .12323 L
.79585 .12444 L
.79176 .12605 L
.7876 .12817 L
.78395 .13056 L
.78086 .13309 L
.77808 .13587 L
.7754 .13919 L
.77129 .14605 L
.76921 .15074 L
.76751 .15547 L
.76486 .16492 L
.76053 .18313 L
.75916 .18794 L
.75747 .19289 L
.75365 .20112 L
.75105 .20518 L
.74822 .20871 L
.74196 .21457 L
Mistroke
.73405 .21979 L
.72424 .22465 L
.68465 .24013 L
.66704 .24692 L
.64782 .25334 L
.6263 .25873 L
.60514 .26283 L
.56289 .27022 L
.5422 .27393 L
.52251 .27713 L
.51267 .27845 L
.50339 .27943 L
.49831 .27985 L
.49565 .28003 L
.49284 .28019 L
.49049 .2803 L
.48792 .2804 L
.4856 .28047 L
.48446 .2805 L
.48343 .28051 L
.48226 .28053 L
.48101 .28054 L
.47986 .28055 L
.47878 .28055 L
.47752 .28055 L
.47637 .28054 L
.4751 .28053 L
.47377 .28051 L
.47131 .28045 L
.46902 .28039 L
.46385 .28018 L
.45542 .27973 L
.44643 .27912 L
.4276 .27763 L
.41743 .27669 L
.40646 .27547 L
.38607 .27239 L
.36785 .26861 L
.33323 .25893 L
.3143 .25172 L
.29284 .24104 L
.2827 .23471 L
.27439 .22846 L
.26761 .22181 L
.26504 .21832 L
.26395 .21639 L
.26318 .21465 L
.26272 .21319 L
.26252 .21236 L
.26239 .21159 L
Mistroke
.26231 .21078 L
.26228 .21005 L
.26228 .20965 L
.26229 .20921 L
.26237 .20841 L
.26248 .20767 L
.26263 .207 L
.26284 .20623 L
.26309 .20551 L
.26439 .20285 L
.26539 .20134 L
.26664 .19975 L
.26933 .19691 L
.27708 .19042 L
.28554 .18402 L
.29338 .17765 L
.29934 .17172 L
.30199 .16832 L
.30303 .1667 L
.30386 .16519 L
.3046 .16357 L
.30512 .1621 L
.30534 .16133 L
.30554 .16049 L
.30568 .15965 L
.30578 .15878 L
.30582 .15792 L
.3058 .15714 L
.30578 .15671 L
.30573 .15626 L
.30561 .15543 L
.30544 .15467 L
.30521 .15387 L
.30467 .15248 L
.3039 .15104 L
.30287 .14956 L
.30051 .14707 L
.29877 .14567 L
.29693 .14443 L
.29303 .14235 L
.2888 .14066 L
.28466 .13938 L
.27962 .13821 L
.27724 .13778 L
.27471 .13739 L
.27249 .13711 L
.27005 .13685 L
.2678 .13665 L
.26575 .13651 L
.26332 .13639 L
Mistroke
.26199 .13634 L
.26073 .1363 L
.25957 .13628 L
.2585 .13627 L
.25732 .13626 L
.25608 .13626 L
.25487 .13627 L
.25356 .13629 L
.25232 .13631 L
.25118 .13635 L
.24903 .13642 L
.24674 .13652 L
.24156 .13683 L
.23689 .13719 L
.21917 .13899 L
.19994 .14167 L
.18891 .14376 L
.17893 .14626 L
.16967 .14939 L
.16171 .15306 L
.15398 .15802 L
.15018 .16126 L
.14708 .1645 L
.14449 .1678 L
.14209 .17156 L
.14031 .17501 L
.13869 .17891 L
.13731 .18327 L
.13632 .18751 L
.13589 .19002 L
.13554 .19271 L
.13533 .19502 L
.13524 .19634 L
.13518 .19758 L
.13515 .19878 L
.13514 .19939 L
.13513 .20006 L
.13513 .20124 L
.13515 .20235 L
.13517 .20302 L
.1352 .20366 L
.13526 .20486 L
.13534 .20615 L
.13546 .20756 L
.13571 .20994 L
.13606 .21246 L
.13687 .21699 L
.13799 .22171 L
.13927 .2261 L
.14224 .23416 L
Mistroke
.14633 .24284 L
.15149 .2517 L
.16262 .26663 L
.1692 .27362 L
.17685 .28038 L
.18455 .28587 L
.19207 .29016 L
.20121 .29422 L
.21152 .29777 L
.23068 .30338 L
.24822 .30974 L
.2669 .31905 L
.28563 .32917 L
.29393 .33304 L
.30311 .33662 L
.3207 .34163 L
.33941 .34567 L
.3595 .35009 L
.378 .35392 L
.38745 .3554 L
.39285 .35604 L
.39787 .3565 L
.4023 .35679 L
.40484 .35692 L
.40721 .35702 L
.40972 .3571 L
.41202 .35715 L
.41329 .35717 L
.41396 .35718 L
.41467 .35718 L
.41598 .3572 L
.4172 .3572 L
.41837 .3572 L
.41964 .3572 L
.42098 .3572 L
.42222 .3572 L
.42339 .35719 L
.42446 .35718 L
.42685 .35715 L
.43728 .35701 L
.45615 .35681 L
.46114 .35676 L
.46387 .35672 L
.46643 .35668 L
.47096 .35659 L
.47336 .35652 L
.47595 .35644 L
.48175 .35617 L
.48728 .35581 L
.49227 .35539 L
Mistroke
.49774 .35481 L
.50871 .35329 L
.51869 .35155 L
.54094 .3469 L
.56239 .34202 L
.58128 .33724 L
.61799 .32437 L
.63767 .31643 L
.65645 .30994 L
.66587 .30736 L
.67489 .30534 L
.6854 .3035 L
.69511 .30223 L
.70525 .30125 L
.71616 .30048 L
.73602 .29943 L
.74616 .29889 L
.75698 .29816 L
.76702 .29723 L
.77598 .29609 L
.7857 .29443 L
.79437 .29245 L
.80349 .28978 L
.81279 .28629 L
.83028 .27725 L
.84581 .26588 L
.86005 .25183 L
.87131 .23702 L
.87586 .22938 L
.87986 .22107 L
.88168 .21628 L
.88303 .21189 L
.88357 .20978 L
.88407 .20748 L
.88447 .2053 L
.88476 .20328 L
.885 .20106 L
.8851 .1999 L
.88514 .1993 L
.88517 .19866 L
.88521 .19758 L
.88524 .19658 L
.88524 .19596 L
.88524 .19538 L
.88522 .19429 L
.88519 .19329 L
.88513 .19224 L
.88507 .19134 L
.88499 .19034 L
.88475 .18812 L
Mistroke
.88444 .18602 L
.88351 .18131 L
.88233 .17697 L
.87864 .16713 L
.87591 .16159 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo04co2P0fo`00O?l0103oo`09o`030?l000Go00<0ool0B_l200[o0P0do`00OOl200_o0P05o`03
0?oo04So0P0>o`80<_l0097o00<0ool0Aol00`3oo`0@o`80<?l004Co4`0jo`030?oo04Ko00<0ool0
4ol00`3oo`0]o`00@?l401?o0`0go`030?oo04Ko00<0ool05?l00`3oo`0/o`00>ol501[o0P0eo`03
0?oo04Go00<0ool05_l00`3oo`0[o`00>Ol2023o0P0do`030?oo04Co00<0ool05ol00`3oo`0[o`00
=ol202?o0P0co`030?oo04Co00<0ool06?l00`3oo`0Zo`00=Ol202Ko00<0ool0<Ol00`3oo`14o`03
0?oo01So00<0ool0:_l003?o0P0Xo`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000co`03
0?oo02Ko0P0co`030?oo04?o00<0ool06_l00`3oo`0Yo`00<_l00`3oo`0Wo`030?oo03;o00<0ool0
@ol00`3oo`0Ko`030?oo02So000ao`030?oo02Oo00<0ool0<ol00`3oo`13o`030?oo01_o00<0ool0
:?l0037o00<0ool09ol00`3oo`0Oo`801?l200?o1005o`030?oo04;o00<0ool07Ol00`3oo`0Wo`00
<?l00`3oo`0Vo`808Ol0103oo`08o`030?oo00Ko00<0ool0@_l00`3oo`0Mo`030?oo02Oo000`o`03
0?oo02Co0P0So`040?oo00Wo00<0ool01Ol204?o00<0ool07_l00`3oo`0Vo`00<?l00`3oo`0So`03
0?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`11o`030?oo01oo00<0ool09_l0033o00<0ool0
8_l00`3oo`0To`040?oo00So00@0ool01Ol00`3oo`11o`030?oo01oo00<0ool09_l0033o00<0ool0
8Ol00`3oo`0Vo`802_l200Ko00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000_o`030?oo027o00<0ool0
>ol00`3oo`10o`030?oo023o00<0ool09_l002oo00<0ool08?l00`3oo`0lo`030?oo043o00<0ool0
8?l00`3oo`0Vo`00;ol2023o00<0ool0?Ol00`3oo`0oo`030?oo027o00<0ool09_l000Oo1@02o`80
1?l201[o00<0ool01_l501?o00<0ool03?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0?o`03
0?oo017o1@0;o`030?oo01Co1@02o`801?l200?o0009o`030?oo00Oo00@0ool06Ol00`3oo`08o`03
0?oo01?o00<0ool02ol0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00go00<0ool0
5?l00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Oo00<0ool0
2?l00`3oo`0Do`030?oo00[o00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`0:o`03
0?oo01Go00<0ool02_l00`3oo`0Go`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201[o
00<0ool01004o`030?oo01Co0P05o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0:o`<0
6?l00`3oo`0:o`030?oo01Oo00<0ool02?l200?o0009o`030?oo00So00<0ool06_l00`3oo`07o`03
0?oo01Go00<0ool02Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Ko0`0Ko`03
0?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o0008o`802_l301[o00<0ool01_l201So00<0ool0
2Ol200[o0`0Ro`030?oo02;o0P0:o`<01?l201go0P0;o`030?oo01Oo0P0:o`<00_l0037o00<0ool0
8Ol203[o00<0ool0<_l302_o00<0ool0:Ol0037o00<0ool08ol00`3oo`0go`030?oo033o0P0^o`03
0?oo02Wo000bo`030?oo02?o00<0ool0=_l00`3oo`0]o`<0;ol00`3oo`0Zo`00<_l00`3oo`0To`80
=_l00`3oo`0[o`80<Ol00`3oo`0[o`00<ol00`3oo`0Uo`<0<ol00`3oo`0Xo`<0<_l00`3oo`0/o`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool00ol20003o`3o00Oo00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000<00ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00ol0103oo`08o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Oo00<0o`00:_l302_o00<0ool07?l600Oo
00<0ool0:Ol00`3oo`02o`030?oo02Go00<0ool01ol003Ko00<0ool0;Ol302So00<0ool05_l603go
00<0ool0<?l003Oo00<0ool0;ol302Go00<0ool04?l604;o00<0ool0<Ol003So00<0ool0<Ol4027o
00<0ool02ol504Ko0P0do`00>Ol00`3oo`0do`H06ol00`3oo`06o`D0B_l00`3oo`0do`00>_l00`3o
o`0io`X04Ol00`3oo`02o`@0C_l00`3oo`0eo`00>ol00`3oo`12oaH0D?l203So000lo`030?oo05;o
00<0ool0DOl00`3oo`0ho`00?Ol00`3oo`1Ao`030?oo04oo0P0ko`00?_l2057o00<0ool0B_l503go
0010o`<0C_l00`3oo`15o`D0@_l004?o101:o`030?oo03Go4017o`00Aol404Ko00<0ool0;Ol805Oo
001;o`80A?l00`3oo`0Xo`D0Gol004go0P0^o`801?l200?o1005o`030?oo02Go0`1To`00Col202_o
00@0ool02?l00`3oo`06o`030?oo02?o0P1Wo`00DOl202Wo00@0ool02Ol00`3oo`05o`808Ol306Wo
001Co`809ol0103oo`0:o`030?oo00Co00<0ool07_l206co001Eo`809Ol0103oo`08o`040?oo00Go
00<0ool06ol306ko001Go`809?l200[o0P06o`030?oo01Oo101ao`00FOl203Ko00<0ool05?l307Go
001Ko`@0<_l00`3oo`0@o`@0N?l005oo1@0]o`030?oo00co101lo`00I?l402Wo00<0ool02Ol3083o
001Xo`809ol00`3oo`04o`D0Pol006[o100So`L0R?l006ko3003oaD0S_l007Wo100Do`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87563 .15791 m
.87152 .14988 L
.86595 .14172 L
.85901 .13425 L
.8516 .12853 L
.84698 .12585 L
.84241 .12373 L
.83815 .12218 L
.8334 .12086 L
.83089 .12033 L
.8281 .11986 L
.82669 .11967 L
.82517 .11949 L
.82385 .11937 L
.8224 .11927 L
.82171 .11923 L
.82098 .11919 L
.82028 .11917 L
.81965 .11915 L
.81901 .11913 L
.81831 .11913 L
.81704 .11913 L
.8158 .11916 L
.81518 .11918 L
.81449 .11921 L
.81328 .11927 L
.81216 .11935 L
.80964 .11959 L
.80733 .1199 L
.80482 .12031 L
.80219 .12085 L
.79791 .12197 L
.79337 .12353 L
.78955 .12521 L
.78569 .12733 L
.77945 .13202 L
.77645 .13508 L
.77395 .13822 L
.77133 .14231 L
.76928 .14636 L
.76592 .15544 L
.76444 .16079 L
.76321 .166 L
.75904 .18417 L
.75764 .18894 L
.75589 .19387 L
.75193 .20212 L
.74927 .20624 L
.74639 .2099 L
.74013 .21611 L
Mistroke
.73239 .2219 L
.72298 .22752 L
.68542 .24532 L
.66808 .25225 L
.6488 .25848 L
.62747 .26379 L
.60691 .26798 L
.56512 .27464 L
.54398 .27738 L
.52396 .27974 L
.51405 .28073 L
.50472 .28149 L
.49961 .28181 L
.49692 .28195 L
.49409 .28207 L
.49171 .28215 L
.49045 .28219 L
.4891 .28222 L
.48787 .28225 L
.48674 .28226 L
.48566 .28228 L
.48453 .28229 L
.48327 .2823 L
.48257 .2823 L
.48192 .2823 L
.48072 .2823 L
.47946 .28229 L
.47834 .28228 L
.47728 .28226 L
.4749 .28222 L
.47233 .28215 L
.46999 .28207 L
.46476 .28185 L
.4555 .28131 L
.44679 .28069 L
.42916 .27917 L
.41913 .27813 L
.40968 .27697 L
.39048 .27377 L
.37031 .26908 L
.33469 .25819 L
.31615 .25093 L
.29604 .24075 L
.28728 .23522 L
.27882 .22879 L
.27526 .22554 L
.27232 .22243 L
.26995 .2194 L
.26819 .21655 L
.2674 .21493 L
Mistroke
.26683 .21345 L
.26636 .21182 L
.26617 .21092 L
.26605 .21007 L
.26598 .20925 L
.26595 .20838 L
.26598 .20757 L
.26604 .20683 L
.2661 .20638 L
.26616 .20596 L
.26636 .20503 L
.26658 .2042 L
.26687 .20334 L
.26824 .20041 L
.27003 .1977 L
.27569 .19159 L
.29008 .17966 L
.29738 .17322 L
.30073 .16968 L
.30329 .16637 L
.30439 .16459 L
.30522 .16297 L
.30591 .1612 L
.30618 .16027 L
.3063 .15977 L
.30639 .15932 L
.3065 .15856 L
.30657 .15777 L
.30658 .15704 L
.30655 .15635 L
.30651 .15597 L
.30646 .15557 L
.30633 .15485 L
.30613 .1541 L
.30585 .15328 L
.30517 .15184 L
.30437 .15057 L
.30322 .14917 L
.30194 .1479 L
.2992 .14582 L
.29561 .1438 L
.29127 .14197 L
.28674 .14053 L
.28241 .13946 L
.27751 .13854 L
.27197 .1378 L
.26915 .13752 L
.26604 .13729 L
.26338 .13714 L
.26188 .13707 L
Mistroke
.26053 .13703 L
.25926 .13699 L
.25789 .13697 L
.2566 .13695 L
.25541 .13694 L
.2541 .13695 L
.25286 .13696 L
.25175 .13697 L
.25054 .13699 L
.24924 .13702 L
.24785 .13706 L
.24537 .13715 L
.24052 .1374 L
.23618 .13769 L
.2256 .13859 L
.2157 .13967 L
.206 .14095 L
.19567 .14261 L
.18686 .1444 L
.17733 .14687 L
.16767 .15021 L
.15946 .15404 L
.15524 .15652 L
.15095 .15952 L
.14726 .16263 L
.14411 .16582 L
.13899 .17258 L
.13677 .17651 L
.13479 .18099 L
.13324 .18564 L
.13262 .18804 L
.13206 .19072 L
.13168 .19305 L
.13151 .19435 L
.13138 .19557 L
.13128 .19672 L
.13119 .19797 L
.13113 .19911 L
.13109 .20017 L
.13107 .20136 L
.13107 .2025 L
.13109 .20353 L
.13113 .20464 L
.13119 .20586 L
.13128 .20715 L
.1315 .20948 L
.13179 .21184 L
.13215 .21406 L
.13298 .21818 L
.13426 .22307 L
Mistroke
.13571 .2275 L
.1398 .23721 L
.14444 .24562 L
.14943 .25309 L
.16287 .26853 L
.17066 .27543 L
.17858 .28129 L
.19465 .29037 L
.21309 .29772 L
.23046 .30353 L
.26683 .31798 L
.30207 .33207 L
.32057 .33728 L
.34052 .34197 L
.35932 .34626 L
.36929 .34842 L
.38001 .35047 L
.39053 .35209 L
.40018 .35317 L
.40487 .35356 L
.40998 .3539 L
.41482 .35415 L
.41931 .35433 L
.42405 .35449 L
.42917 .35461 L
.43847 .3548 L
.44955 .35497 L
.45445 .35504 L
.45726 .35507 L
.4599 .3551 L
.46237 .35511 L
.46305 .35512 L
.46377 .35512 L
.46505 .35513 L
.46641 .35513 L
.46767 .35513 L
.46906 .35513 L
.46984 .35512 L
.47055 .35512 L
.47205 .35511 L
.47342 .3551 L
.47494 .35508 L
.47655 .35505 L
.47925 .35499 L
.48223 .35491 L
.48753 .3547 L
.49324 .35438 L
.49863 .35398 L
.50355 .35354 L
.51398 .35233 L
Mistroke
.52361 .35094 L
.54516 .34708 L
.58422 .33777 L
.60463 .33111 L
.62363 .32398 L
.66089 .3113 L
.68091 .3065 L
.70241 .30281 L
.72262 .30028 L
.74155 .29822 L
.76065 .29585 L
.78085 .29234 L
.79144 .28984 L
.80094 .28709 L
.81889 .2802 L
.83465 .27162 L
.85071 .25923 L
.86374 .24497 L
.86981 .23615 L
.87444 .22776 L
.87805 .21936 L
.88053 .21145 L
.88157 .20687 L
.88227 .20267 L
.88256 .20033 L
.88268 .1991 L
.88279 .19779 L
.88286 .19664 L
.8829 .19603 L
.88293 .19537 L
.88297 .19421 L
.88299 .19313 L
.883 .19202 L
.88299 .19101 L
.88296 .18984 L
.88292 .18874 L
.88287 .18771 L
.88281 .18677 L
.88264 .18463 L
.88241 .18245 L
.88208 .18005 L
.8812 .17518 L
.88017 .17084 L
.87881 .1662 L
.87563 .15791 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1?o`L0=ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1;o`@01_l403?o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`<03_l00`3oo`0`o`00
OOl200_o0P05o`030?oo04Oo00<0ool04?l2033o002Ao`030?oo04Ko0P0Do`030?oo02go0017oa00
>_l00`3oo`15o`030?oo01Go00<0ool0;?l003oo200@o`<0=ol00`3oo`15o`030?oo01Ko00<0ool0
:ol003go0P0Ko`80=Ol00`3oo`14o`030?oo01Oo00<0ool0:ol003So1@0Oo`030?oo03;o00<0ool0
A?l00`3oo`0Ho`030?oo02[o000fo`809?l203?o00<0ool0@ol00`3oo`0Io`030?oo02[o000eo`03
0?oo02Go00<0ool0<Ol00`3oo`13o`030?oo01[o00<0ool0:Ol003Co00<0ool09_l00`3oo`0ao`03
0?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Vo`80<ol00`3oo`13o`030?oo01[o00<0ool0
:Ol003;o00<0ool09ol00`3oo`0bo`030?oo04?o00<0ool06ol00`3oo`0Xo`00<?l202Wo00<0ool0
<ol00`3oo`12o`030?oo01go00<0ool09ol0033o00<0ool09ol2027o0P04o`800ol400Go00<0ool0
@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Oo00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0
@Ol00`3oo`0No`030?oo02Oo000_o`030?oo02Go0P0So`040?oo00Wo00<0ool01Ol204;o00<0ool0
7_l00`3oo`0Wo`00;ol00`3oo`0To`030?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`11o`03
0?oo01ko00<0ool09ol002oo00<0ool08ol00`3oo`0To`040?oo00So00@0ool01Ol00`3oo`11o`03
0?oo01ko00<0ool09ol002ko0P0To`030?oo02Ko0P0:o`801_l00`3oo`11o`030?oo01ko00<0ool0
9ol002ko00<0ool08_l00`3oo`0ko`030?oo043o00<0ool07ol00`3oo`0Wo`00;_l00`3oo`0Ro`03
0?oo03_o00<0ool0@?l00`3oo`0Oo`030?oo02Oo000^o`030?oo027o00<0ool0??l00`3oo`0oo`03
0?oo023o00<0ool09ol000Oo1@02o`801?l201So00<0ool02?l501Co00<0ool02ol200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0>o`030?oo01;o1@0:o`030?oo01Go1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool06?l00`3oo`09o`030?oo01Co00<0ool02_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Ko00<0ool02Ol00`3oo`0Eo`030?oo00Wo00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`09o`030?oo01Ko00<0ool02_l00`3oo`0Go`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201Wo00@0oooo1004o`030?oo01Go00<0ool00ol400;o00@0ool0
2Ol202?o00<0ool08Ol0103oo`09o`802ol00`3oo`0Go`030?oo00Wo00<0ool06?l00`3oo`08o`80
0ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo00<0ool05_l200Wo00@0ool02Ol00`3oo`0Ro`03
0?oo027o00@0ool02Ol00`3oo`09o`030?oo01So00<0ool02Ol00`3oo`0Ho`030?oo00So00<0ool0
0_l000So0P0:o`<06Ol00`3oo`07o`806_l00`3oo`07o`802_l302;o00<0ool08_l200[o0`07o`80
6_l200[o00<0ool06?l200[o0`02o`00<?l00`3oo`0To`030?oo03Oo00<0ool0=_l202So00<0ool0
:Ol0037o00<0ool09?l00`3oo`0fo`030?oo03Co0P0Yo`030?oo02[o000ao`030?oo02Go00<0ool0
=Ol00`3oo`0bo`80:_l00`3oo`0[o`00<_l00`3oo`0Uo`80=Ol00`3oo`0`o`80;?l00`3oo`0[o`00
<_l00`3oo`0Wo`<0<_l00`3oo`0]o`<0;Ol00`3oo`0/o`003?oo00`02Ol000co00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l00`3oo`04o`<02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01?l400Oo00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool0
9Ol01@3oool002_o0`0Zo`030?oo02?o1002o`030?oo02Wo00<0ool00_l00`3oo`0Uo`030?oo00Oo
000eo`030?oo02oo0`0Wo`030?oo023o0`0fo`030?oo033o000fo`030?oo037o0`0To`030?oo01co
100ho`030?oo037o000go`030?oo03?o100Po`030?oo01Oo1@0ko`030?oo03;o000ho`030?oo03Ko
1@0Ko`030?oo017o1P0no`80=Ol003Wo00<0ool0>_l701Co00<0ool01_l;04;o0P0go`00>_l2047o
3007o`X0B_l303Wo000lo`030?oo04[o1`000ol0o`1?o`80??l003go0P1Bo`030?oo04co0P0no`00
?ol2053o00<0ool0AOl7043o0011o`80C_l00`3oo`10o`D0Aol004?o101:o`030?oo03Go2`1<o`00
Aol404Ko00<0ool0<?l505Oo001;o`80A?l00`3oo`0Zo`H0G?l004go0P0^o`801?l200?o1005o`03
0?oo02Ko101Ro`00Col202_o00@0ool02?l00`3oo`06o`030?oo02Co0P1Vo`00DOl302So00@0ool0
2Ol00`3oo`05o`808_l306So001Do`809_l0103oo`0:o`030?oo00Co00<0ool07ol206_o001Fo`<0
8ol0103oo`08o`040?oo00Go00<0ool07?l306go001Io`808_l200[o0P06o`030?oo01[o0P1`o`00
Fol403;o00<0ool05_l407;o001Oo`D0;Ol00`3oo`0Bo`@0M_l006Co100Yo`030?oo00ko101jo`00
J?l202Oo00<0ool02ol307ko001Zo`H08Ol00`3oo`06o`D0POl0073o1`0Jo`030?l000H0Q_l007Oo
702=o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co
0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co
001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko
00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go
001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o
0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87536 .15424 m
.87367 .14993 L
.87186 .14604 L
.86939 .14163 L
.86686 .13784 L
.86078 .13093 L
.85694 .12767 L
.85301 .12497 L
.84868 .12261 L
.84381 .12056 L
.83943 .11917 L
.83442 .11802 L
.83154 .11754 L
.82887 .11722 L
.82739 .11708 L
.82576 .11696 L
.82501 .11691 L
.82421 .11688 L
.82345 .11685 L
.82277 .11683 L
.82207 .11682 L
.82133 .11681 L
.82001 .11681 L
.81928 .11682 L
.81859 .11684 L
.81783 .11686 L
.81703 .1169 L
.81557 .11698 L
.8142 .11708 L
.81163 .11733 L
.80917 .11764 L
.80687 .118 L
.80175 .11906 L
.79692 .12041 L
.79196 .12222 L
.78786 .12413 L
.78383 .12645 L
.77748 .13141 L
.77439 .1347 L
.77191 .13797 L
.76958 .14178 L
.76743 .14622 L
.76576 .15059 L
.76444 .15481 L
.76231 .16348 L
.7603 .17329 L
.75795 .1837 L
.75657 .18857 L
.75503 .19303 L
.75146 .20093 L
Mistroke
.7467 .20838 L
.74134 .21466 L
.72648 .22712 L
.69258 .24738 L
.67337 .256 L
.65439 .26232 L
.61639 .27155 L
.59664 .27531 L
.58647 .27686 L
.57541 .27823 L
.55488 .28013 L
.53534 .28162 L
.51427 .28309 L
.50856 .28342 L
.50253 .2837 L
.4974 .28389 L
.49448 .28397 L
.49307 .284 L
.49176 .28402 L
.49049 .28404 L
.48934 .28405 L
.48806 .28406 L
.4867 .28406 L
.48538 .28406 L
.48464 .28406 L
.48396 .28406 L
.48271 .28405 L
.48139 .28404 L
.48026 .28402 L
.47905 .284 L
.47687 .28395 L
.47192 .2838 L
.46622 .28355 L
.46084 .28325 L
.45082 .28256 L
.43107 .28079 L
.41332 .27864 L
.40415 .27717 L
.39554 .2755 L
.37576 .27057 L
.33959 .25877 L
.32228 .25203 L
.30342 .2429 L
.2948 .23773 L
.28608 .23151 L
.27889 .22508 L
.2738 .2189 L
.27183 .21554 L
.27106 .21383 L
.27039 .21195 L
Mistroke
.26998 .21034 L
.26982 .20944 L
.26971 .20862 L
.26964 .20785 L
.26961 .20702 L
.26962 .20624 L
.26966 .20551 L
.26975 .2046 L
.26983 .20409 L
.26991 .20362 L
.27035 .20185 L
.27091 .20024 L
.27166 .19855 L
.27374 .19507 L
.27674 .1913 L
.29191 .17741 L
.29898 .17099 L
.30404 .1652 L
.30579 .16238 L
.30644 .16099 L
.30673 .16019 L
.30696 .15947 L
.30714 .15874 L
.30721 .15838 L
.30727 .15799 L
.30732 .15759 L
.30735 .15723 L
.30737 .15681 L
.30738 .15642 L
.30734 .15574 L
.3073 .15536 L
.30725 .15501 L
.30711 .15437 L
.30693 .15371 L
.30646 .15255 L
.30578 .15134 L
.30498 .15024 L
.3041 .14928 L
.30144 .14706 L
.29827 .14517 L
.29437 .14342 L
.29041 .14205 L
.2854 .14072 L
.28038 .1397 L
.2755 .13896 L
.27092 .13844 L
.26581 .13802 L
.26322 .13787 L
.26175 .13781 L
.26039 .13775 L
Mistroke
.25906 .13771 L
.2576 .13768 L
.2569 .13766 L
.25614 .13765 L
.25534 .13764 L
.25459 .13764 L
.25389 .13763 L
.25313 .13763 L
.25174 .13763 L
.2504 .13765 L
.24915 .13766 L
.24783 .13769 L
.24659 .13772 L
.24428 .13778 L
.24181 .13788 L
.23916 .138 L
.2336 .13833 L
.22782 .13877 L
.21763 .13974 L
.20815 .14087 L
.19805 .14238 L
.18857 .14417 L
.18001 .14622 L
.17052 .14916 L
.1621 .15257 L
.15345 .1572 L
.14521 .16325 L
.13903 .16951 L
.13606 .17341 L
.13367 .17723 L
.13172 .18105 L
.1302 .18476 L
.12893 .18877 L
.12838 .19095 L
.1279 .19329 L
.12751 .19576 L
.12726 .19806 L
.12715 .19935 L
.12708 .20056 L
.12705 .20126 L
.12703 .20189 L
.12702 .20263 L
.12701 .20333 L
.12702 .20449 L
.12706 .20572 L
.12712 .20689 L
.12718 .20795 L
.12729 .2092 L
.12743 .21055 L
.12776 .213 L
Mistroke
.12862 .21757 L
.12985 .22235 L
.13282 .2307 L
.13693 .23921 L
.14225 .24775 L
.15378 .26154 L
.16969 .27498 L
.18681 .28542 L
.22227 .30072 L
.26014 .31457 L
.29787 .3268 L
.33479 .33654 L
.3551 .3415 L
.37371 .34552 L
.39205 .34857 L
.4013 .34971 L
.41145 .35071 L
.43096 .35209 L
.44212 .35267 L
.45246 .35311 L
.45756 .35329 L
.463 .35344 L
.46768 .35354 L
.46896 .35356 L
.47034 .35358 L
.47155 .35359 L
.47284 .3536 L
.47355 .35361 L
.47433 .35361 L
.47571 .35362 L
.47648 .35362 L
.47732 .35362 L
.47811 .35362 L
.47883 .35362 L
.48016 .35361 L
.48091 .3536 L
.4816 .3536 L
.48302 .35358 L
.48453 .35356 L
.4874 .35351 L
.49011 .35344 L
.49518 .35327 L
.50011 .35304 L
.50551 .35273 L
.51633 .35187 L
.52622 .35082 L
.53539 .34963 L
.55591 .34613 L
.57436 .34189 L
.61239 .32983 L
Mistroke
.64825 .31761 L
.68853 .30708 L
.72747 .29956 L
.7471 .29616 L
.76825 .29207 L
.78673 .28771 L
.80657 .28174 L
.82481 .27424 L
.83435 .26907 L
.84282 .26343 L
.85091 .25679 L
.85783 .24979 L
.86843 .23522 L
.8726 .22716 L
.87603 .21837 L
.87818 .21074 L
.8791 .20633 L
.87978 .20215 L
.88029 .1979 L
.88049 .19546 L
.88064 .19318 L
.8807 .19193 L
.88074 .19075 L
.88076 .18968 L
.88078 .18852 L
.88078 .18792 L
.88078 .18725 L
.88078 .18655 L
.88077 .1859 L
.88075 .18473 L
.88073 .18407 L
.88071 .18347 L
.8806 .18121 L
.88046 .17911 L
.88026 .1768 L
.87999 .17433 L
.87932 .16972 L
.87853 .16555 L
.8772 .16007 L
.87536 .15424 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`<02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`804?l2033o001mo`80
2ol200Go00<0ool0A_l201Co00<0ool0;Ol0097o00<0ool0AOl00`3oo`0Eo`030?oo02co0017o`l0
>ol00`3oo`15o`030?oo01Ko00<0ool0:ol0043o1`0?o`@0=ol00`3oo`14o`030?oo01So00<0ool0
:_l003_o1@0Jo`80=Ol00`3oo`14o`030?oo01So00<0ool0:_l003Wo0P0Qo`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Ko0`0To`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`80
9ol00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Wo`030?oo037o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo0P0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Xo`030?oo03;o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02So00<0ool0
<ol00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool09ol00`3oo`0Po`801?l200?o1005o`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Wo`030?oo023o00@0ool02?l00`3oo`06o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`808ol0103oo`09o`030?oo00Go0P12o`03
0?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Co00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02?o00<0ool09_l200[o0P06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?ol00`3oo`0Po`030?oo02Oo000]o`03
0?oo02?o0P0lo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0So`030?oo03_o00<0ool0
?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Eo`802_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`09o`030?oo01So00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l200?o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`09o`030?oo01So00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00Wo0P0Io`030?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3o
o`0Ho`802_l00`3oo`0Ho`802_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0go`809ol00`3o
o`0Yo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0=Ol202So00<0ool0:_l0033o00<0ool09ol203Go
00<0ool0<ol202Wo00<0ool0:ol0037o00<0ool0:?l00`3oo`0bo`030?oo037o0P0[o`030?oo02_o
000ao`030?oo02Wo0P0bo`030?oo033o00<0ool0:_l00`3oo`0/o`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1ol000co00<0ool09?l00`3oo`02o`030?oo02Wo100Yo`030?oo02Wo0`0Yo`030?oo00;o00<0ool0
9Ol00`3oo`07o`00=?l00`3oo`0ao`<09_l00`3oo`0Uo`@0<?l00`3oo`0`o`00=Ol00`3oo`0co`<0
8ol00`3oo`0Qo`@0<_l203?o000fo`80=_l3023o00<0ool07Ol403?o0`0eo`00>?l00`3oo`0fo`<0
7Ol00`3oo`0Io`@0=Ol203So000io`030?oo03So1@0Ho`030?oo01?o1P0go`80>_l003[o0P0mo`P0
4?l00`3oo`04o`l0>_l303co000lo`<0@_lG04Oo0P0oo`00?ol2053o00<0ool0Aol4047o0011o`<0
COl00`3oo`11o`H0AOl004Co0P1;o`030?oo03co1@1;o`00A_l304So00<0ool0=ol5053o0019o`80
A_l00`3oo`0bo`D0EOl004_o0`13o`030?oo02ko101Jo`00C_l202go0P04o`800ol400Go00<0ool0
:_l405ko001@o`<0:Ol0103oo`08o`030?oo00Ko00<0ool09_l406;o001Co`<09_l0103oo`09o`03
0?oo00Go0P0To`<0I_l005Ko100Ro`040?oo00[o00<0ool01?l00`3oo`0Oo`@0JOl005[o0`0Oo`04
0?oo00So00@0ool01Ol00`3oo`0Lo`<0KOl005go0`0Mo`802_l200Ko00<0ool06Ol3073o001Po`@0
;Ol00`3oo`0Eo`@0Lol006Co100Yo`030?oo01;o0`1go`00J?l502Co00<0ool03ol307[o001]o`D0
7ol00`3oo`0=o`80OOl007;o0`0Lo`030?oo00Go201oo`00MOlA00Wo2P27o`00Q_l900;o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
1Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87508 .15057 m
.87372 .14616 L
.87219 .14221 L
.87005 .13775 L
.86776 .13397 L
.86205 .12715 L
.85832 .12399 L
.85442 .12142 L
.85006 .1192 L
.84507 .11733 L
.84054 .1161 L
.83787 .11555 L
.8353 .11514 L
.83387 .11495 L
.8323 .11477 L
.83089 .11465 L
.82939 .11454 L
.82861 .11449 L
.82789 .11446 L
.82652 .11441 L
.82522 .11438 L
.82382 .11438 L
.82303 .11438 L
.82229 .11439 L
.82148 .11441 L
.82061 .11444 L
.81883 .11453 L
.8172 .11464 L
.81406 .11493 L
.8111 .1153 L
.80602 .11617 L
.80055 .11744 L
.79506 .11913 L
.79018 .12106 L
.78227 .12534 L
.77836 .12821 L
.77513 .13115 L
.77217 .13448 L
.76943 .13836 L
.76741 .14196 L
.76554 .14609 L
.76373 .15121 L
.76234 .15626 L
.76029 .16605 L
.75836 .17635 L
.7562 .18574 L
.75483 .19029 L
.75309 .195 L
.74833 .20441 L
.74267 .21234 L
Mistroke
.7353 .22035 L
.71989 .23371 L
.70255 .24643 L
.68635 .25602 L
.67783 .25995 L
.66825 .26358 L
.65005 .26893 L
.63094 .27365 L
.61033 .27829 L
.60086 .28009 L
.59052 .28164 L
.5854 .28223 L
.58048 .28271 L
.57104 .28338 L
.56616 .28364 L
.56073 .28387 L
.5496 .28424 L
.5296 .28486 L
.51864 .28524 L
.51273 .28544 L
.50727 .2856 L
.50259 .28572 L
.5002 .28577 L
.49761 .28581 L
.49613 .28583 L
.49476 .28584 L
.49346 .28585 L
.49208 .28586 L
.49084 .28586 L
.48968 .28586 L
.48841 .28585 L
.48704 .28584 L
.48572 .28583 L
.48432 .28581 L
.48179 .28576 L
.47906 .2857 L
.47605 .28561 L
.47065 .28539 L
.46557 .28514 L
.45616 .28453 L
.44614 .28373 L
.42606 .28165 L
.41642 .28034 L
.40608 .27857 L
.38767 .27424 L
.36747 .26792 L
.32951 .25364 L
.31255 .24589 L
.29725 .23692 L
.28989 .23146 L
Mistroke
.28364 .22577 L
.27828 .21934 L
.27611 .21578 L
.2752 .2139 L
.27441 .21188 L
.27408 .21084 L
.27382 .20985 L
.27361 .20889 L
.27346 .20801 L
.27335 .20717 L
.27328 .20628 L
.27324 .20539 L
.27324 .20488 L
.27325 .20441 L
.27327 .2039 L
.27331 .20335 L
.27336 .20283 L
.27343 .20235 L
.27357 .20144 L
.27378 .20048 L
.27435 .19848 L
.27521 .1963 L
.27716 .1927 L
.27989 .18888 L
.2861 .18217 L
.29987 .16948 L
.30297 .16623 L
.3056 .16288 L
.30658 .1613 L
.3073 .15984 L
.30777 .15856 L
.30796 .15785 L
.30809 .1572 L
.30817 .15658 L
.3082 .15602 L
.30819 .15538 L
.30813 .15479 L
.30804 .15424 L
.30792 .15375 L
.30775 .15321 L
.30752 .15265 L
.30683 .15139 L
.30596 .15026 L
.30381 .14827 L
.30242 .1473 L
.30078 .14633 L
.2974 .14474 L
.29321 .14323 L
.28805 .14183 L
.28271 .14073 L
Mistroke
.27737 .13989 L
.27241 .13931 L
.2669 .13884 L
.26402 .13866 L
.26084 .13851 L
.25924 .13845 L
.25752 .1384 L
.25589 .13836 L
.25441 .13834 L
.25295 .13832 L
.25159 .13832 L
.25011 .13832 L
.24928 .13832 L
.24854 .13833 L
.24716 .13834 L
.24568 .13837 L
.24302 .13843 L
.24038 .13851 L
.23757 .13863 L
.23256 .13889 L
.22671 .13927 L
.22148 .1397 L
.21037 .14084 L
.20011 .14223 L
.19063 .14388 L
.18071 .14613 L
.17237 .14856 L
.1634 .15192 L
.15442 .15629 L
.14672 .16117 L
.13911 .16753 L
.1354 .17148 L
.13239 .17534 L
.12999 .17902 L
.12776 .18321 L
.12585 .18783 L
.12514 .18998 L
.12448 .19237 L
.124 .19447 L
.12359 .19675 L
.12341 .19802 L
.12327 .19918 L
.12315 .20037 L
.12307 .20148 L
.12301 .2025 L
.12297 .20359 L
.12296 .2048 L
.12296 .2059 L
.12299 .20694 L
.12304 .20807 L
Mistroke
.12311 .20914 L
.12319 .21012 L
.12347 .21255 L
.1238 .21476 L
.12485 .21971 L
.12615 .22424 L
.12767 .22845 L
.13193 .23752 L
.13769 .24665 L
.15062 .2611 L
.15806 .26729 L
.16688 .2734 L
.20293 .29213 L
.2387 .30718 L
.25735 .31328 L
.27784 .31851 L
.31631 .32747 L
.35631 .33796 L
.37465 .34192 L
.3944 .34515 L
.415 .34773 L
.43473 .34976 L
.44514 .35063 L
.45516 .3513 L
.46073 .3516 L
.46587 .35182 L
.47163 .35201 L
.47482 .35209 L
.47786 .35215 L
.4791 .35216 L
.48043 .35218 L
.48159 .35219 L
.48286 .35221 L
.48426 .35221 L
.48503 .35222 L
.48574 .35222 L
.48701 .35222 L
.48838 .35222 L
.48982 .35222 L
.49063 .35222 L
.49138 .35221 L
.49276 .3522 L
.49421 .35219 L
.49672 .35215 L
.49948 .35211 L
.50438 .35199 L
.50961 .35183 L
.5189 .35141 L
.52429 .35108 L
.52924 .35072 L
Mistroke
.54034 .34964 L
.55004 .34834 L
.56035 .34652 L
.58154 .34134 L
.60101 .33521 L
.63668 .32348 L
.67623 .31243 L
.71491 .30239 L
.7505 .29369 L
.7891 .28366 L
.80718 .27816 L
.82649 .27048 L
.83648 .26514 L
.8451 .25933 L
.85303 .25257 L
.85951 .24553 L
.86458 .23857 L
.86901 .2307 L
.87098 .22632 L
.87275 .22164 L
.87415 .21719 L
.87522 .21311 L
.87689 .20462 L
.87755 .19985 L
.87803 .19526 L
.8782 .19317 L
.87835 .19094 L
.87845 .18899 L
.87854 .18689 L
.87858 .1857 L
.87861 .1846 L
.87863 .1835 L
.87864 .18245 L
.87865 .18123 L
.87865 .1801 L
.87863 .17884 L
.87862 .17814 L
.87861 .17751 L
.87857 .17594 L
.8785 .17422 L
.87841 .17242 L
.87831 .17073 L
.87807 .16771 L
.87773 .16447 L
.87721 .16067 L
.8766 .15711 L
.87508 .15057 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l:03Co001lo`040?oo00So1@04o`80C?l300[o0`0ao`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03_l202oo001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`80
4ol00`3oo`0/o`00OOl200_o0P05o`030?oo04Ko0P0Eo`030?oo02co002Ao`030?oo04Go00<0ool0
5_l00`3oo`0[o`00BOl;03go00<0ool0A?l00`3oo`0Ho`030?oo02[o0010o`T02ol503So00<0ool0
A?l00`3oo`0Io`030?oo02Wo000ko`D06Ol403Co00<0ool0@ol00`3oo`0Jo`030?oo02Wo000io`80
8Ol203?o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000go`809?l00`3oo`0ao`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol202Ko00<0ool0<Ol00`3oo`12o`030?oo01co00<0ool0:?l003?o0P0Xo`03
0?oo037o00<0ool0@_l00`3oo`0Lo`030?oo02So000bo`030?oo02So00<0ool0<Ol00`3oo`12o`03
0?oo01co00<0ool0:?l0037o00<0ool0:?l00`3oo`0bo`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<?l00`3oo`0Xo`030?oo03?o00<0ool0@Ol00`3oo`0Mo`030?oo02So000_o`030?oo02So00<0ool0
8?l200Co0P03o`@01Ol00`3oo`11o`030?oo01go00<0ool0:?l002ko00<0ool0:?l00`3oo`0Po`04
0?oo00So00<0ool01_l00`3oo`11o`030?oo01go00<0ool0:?l002go00<0ool0:?l00`3oo`0Qo`04
0?oo00Wo00<0ool01Ol204;o00<0ool07Ol00`3oo`0Xo`00;Ol00`3oo`0Wo`030?oo01co1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01go00<0ool0:?l002go00<0ool09_l00`3oo`0So`04
0?oo00So00@0ool01Ol00`3oo`10o`030?oo01ko00<0ool0:?l002co0P0Wo`030?oo02Go0P0:o`80
1_l00`3oo`10o`030?oo01ko00<0ool0:?l002co00<0ool09Ol203_o00<0ool0@?l00`3oo`0No`03
0?oo02So000/o`030?oo02Go00<0ool0>_l00`3oo`0oo`030?oo01oo00<0ool0:?l002co00<0ool0
9Ol00`3oo`0jo`030?oo03oo00<0ool07ol00`3oo`0Xo`001ol500;o0P04o`805_l00`3oo`0:o`D0
5_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool04_l500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Eo`030?oo00co00<0ool05_l200Wo00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02?l00`3oo`0Io`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Co`030?oo00co00<0ool05ol00`3oo`07o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00So00<0ool0
6Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Go`030?oo00?o1004o`030?oo01So
00<0ool01002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3oo`08o`03
0?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05_l00`3oo`0;o`030?oo01So00<0ool0
1_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`08o`03
0?oo01Wo00<0ool02?l00`3oo`02o`002?l200[o0`0Fo`030?oo00[o0P0Ko`801ol200[o0`0Ro`03
0?oo02;o0P0:o`<02?l00`3oo`0Ho`802Ol00`3oo`0Io`802_l300;o000^o`030?oo02So00<0ool0
=Ol00`3oo`0ho`030?oo02Co00<0ool0:_l002ko00<0ool0:Ol00`3oo`0do`030?oo03Oo00<0ool0
9?l00`3oo`0[o`00;ol00`3oo`0Yo`80=?l00`3oo`0eo`809_l00`3oo`0/o`00<?l00`3oo`0Zo`80
<_l00`3oo`0co`80:?l00`3oo`0/o`00<?l00`3oo`0/o`80<?l00`3oo`0bo`030?oo02Oo00<0ool0
;Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o
0P000ol0o`07o`030?oo00Go00<0ool01_l00`3oo`06o`040?ooo`800ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co00<0ool00_l00`3oo`0Yo`<0:_l00`3o
o`0Yo`030?oo00;o00<0ool09?l0103oo`0Yo`030?oo00Oo000do`030?oo033o0`0Wo`030?oo02_o
0`0Yo`030?oo03;o000eo`030?oo03;o0P0Uo`030?oo02Oo100Zo`80=Ol003Ko00<0ool0<ol5023o
00<0ool08_l502_o0`0go`00=ol203So100Lo`030?oo01ko100^o`80>_l003Wo0P0jo`806_l00`3o
o`0Lo`80;ol303co000ko`80>_l9017o00<0ool05Ol702oo0P0oo`00?Ol2047o3005oaP0<ol3047o
000oo`80Bol604Ko1014o`00@Ol204ko00<0ool0@Ol304So0013o`80C?l00`3oo`0mo`@0Bol004Go
0`19o`030?oo03Wo101?o`00B?l204Oo00<0ool0=Ol405?o001:o`<0A?l00`3oo`0ao`@0Eol004go
0P0^o`801?l200?o1005o`030?oo02go101Ko`00Col402Wo00@0ool02?l00`3oo`06o`030?oo02[o
0`1Oo`00Dol402Go00@0ool02Ol00`3oo`05o`809ol406;o001Go`@08Ol0103oo`0:o`030?oo00Co
00<0ool08ol306Ko001Ko`@07Ol0103oo`08o`040?oo00Go00<0ool08?l306Wo001Oo`@06_l200[o
0P06o`030?oo01co101/o`00Hol502Wo00<0ool06Ol3073o001Xo`D09?l00`3oo`0Eo`@0Lol006go
1@0Oo`030?oo017o101go`00L_l501[o00<0ool03ol207_o001go`H05?l00`3oo`09o`H0OOl007go
8023o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87481 .14689 m
.87377 .14239 L
.87253 .13838 L
.87085 .13422 L
.86866 .13009 L
.86588 .12616 L
.8628 .12287 L
.85965 .12027 L
.85583 .11786 L
.85147 .11581 L
.8462 .11406 L
.84339 .11338 L
.84024 .11279 L
.83868 .11256 L
.83717 .11237 L
.83567 .11221 L
.83429 .11209 L
.83289 .11199 L
.83155 .11192 L
.83013 .11187 L
.82932 .11185 L
.82858 .11184 L
.82733 .11184 L
.82594 .11185 L
.82449 .11189 L
.82313 .11194 L
.82151 .11203 L
.81997 .11213 L
.81707 .1124 L
.81396 .11277 L
.81057 .11328 L
.80487 .1144 L
.79965 .11574 L
.79414 .11753 L
.78853 .11984 L
.78338 .12252 L
.77848 .12578 L
.77476 .12891 L
.77119 .13268 L
.76814 .13684 L
.76573 .14106 L
.76394 .14507 L
.76234 .14962 L
.76094 .15477 L
.75976 .16036 L
.75803 .17054 L
.75633 .17998 L
.75515 .18512 L
.75384 .18968 L
.75056 .19815 L
Mistroke
.74643 .20571 L
.74094 .2134 L
.73416 .22124 L
.72067 .23473 L
.70555 .24835 L
.69746 .25466 L
.68858 .26042 L
.68011 .26472 L
.67024 .26858 L
.65181 .27386 L
.63133 .27889 L
.61338 .28316 L
.60449 .28495 L
.59973 .28574 L
.59458 .28645 L
.58999 .28695 L
.58752 .28717 L
.58481 .28737 L
.58231 .28752 L
.57999 .28764 L
.57763 .28774 L
.57537 .28781 L
.5741 .28784 L
.57272 .28787 L
.57157 .28788 L
.57029 .2879 L
.5689 .28791 L
.56814 .28792 L
.56743 .28792 L
.56615 .28793 L
.56478 .28792 L
.56349 .28792 L
.56209 .28791 L
.56076 .2879 L
.55953 .28789 L
.5547 .28784 L
.54899 .28775 L
.54284 .28767 L
.54002 .28763 L
.53706 .2876 L
.53455 .28758 L
.53314 .28757 L
.53181 .28757 L
.53053 .28756 L
.52915 .28756 L
.5279 .28756 L
.52674 .28756 L
.5255 .28756 L
.52413 .28756 L
.52344 .28756 L
Mistroke
.52271 .28756 L
.52138 .28757 L
.5188 .28758 L
.51637 .2876 L
.51181 .28763 L
.50676 .28767 L
.50531 .28768 L
.50395 .28769 L
.5027 .28769 L
.50139 .2877 L
.50069 .2877 L
.49991 .2877 L
.49855 .2877 L
.49779 .2877 L
.49696 .2877 L
.49619 .2877 L
.49548 .28769 L
.49412 .28769 L
.49265 .28768 L
.49125 .28766 L
.48996 .28765 L
.48736 .28761 L
.48497 .28756 L
.47955 .28741 L
.47474 .28722 L
.46965 .28698 L
.45884 .28631 L
.44717 .28537 L
.42702 .28318 L
.41651 .28163 L
.4073 .2799 L
.38961 .27535 L
.36956 .26851 L
.33172 .25327 L
.31609 .24597 L
.30229 .23789 L
.2896 .22773 L
.28634 .22431 L
.28354 .22087 L
.2812 .21741 L
.27939 .21404 L
.27861 .2122 L
.27791 .21019 L
.27742 .20835 L
.27725 .20753 L
.2771 .20665 L
.27698 .20569 L
.2769 .20479 L
.27686 .20425 L
.27684 .20373 L
Mistroke
.27684 .20276 L
.27688 .20173 L
.27696 .20078 L
.27702 .20029 L
.27709 .19974 L
.27729 .19865 L
.27774 .19678 L
.27828 .19509 L
.27994 .19132 L
.28198 .18789 L
.2843 .1847 L
.29748 .17113 L
.30283 .16588 L
.30514 .16328 L
.30708 .16066 L
.30784 .15935 L
.30817 .15867 L
.30847 .15795 L
.3087 .15731 L
.30885 .15672 L
.30897 .15615 L
.30903 .15561 L
.30906 .15507 L
.30904 .15451 L
.30899 .15404 L
.30889 .15353 L
.30873 .15298 L
.30851 .15241 L
.30796 .1514 L
.30719 .15038 L
.3061 .1493 L
.30472 .14824 L
.3032 .1473 L
.29998 .14573 L
.29589 .14425 L
.29107 .14292 L
.28617 .14188 L
.28039 .14094 L
.27394 .14016 L
.27117 .1399 L
.26816 .13966 L
.26554 .13948 L
.26266 .13932 L
.26091 .13924 L
.25929 .13918 L
.2576 .13913 L
.25608 .13908 L
.25468 .13905 L
.25317 .13903 L
.25233 .13902 L
Mistroke
.25157 .13901 L
.25008 .139 L
.24857 .139 L
.24716 .13901 L
.24589 .13902 L
.24452 .13903 L
.24303 .13906 L
.24145 .13909 L
.23865 .13917 L
.23369 .13935 L
.22833 .13963 L
.22268 .14001 L
.21674 .14049 L
.20633 .14159 L
.19673 .14294 L
.18654 .14483 L
.17616 .14743 L
.1667 .15055 L
.15711 .15468 L
.1484 .1595 L
.14104 .16467 L
.13384 .17117 L
.12786 .17844 L
.12505 .18293 L
.12287 .18734 L
.12187 .1898 L
.12106 .19218 L
.12043 .19438 L
.11988 .19676 L
.11964 .19799 L
.11942 .19936 L
.11925 .20063 L
.11912 .20181 L
.11901 .20319 L
.11896 .20391 L
.11893 .20468 L
.1189 .20598 L
.1189 .20672 L
.1189 .20741 L
.11894 .20867 L
.11897 .20931 L
.11901 .21001 L
.11911 .21125 L
.11923 .21242 L
.11958 .21504 L
.12011 .21786 L
.12142 .22298 L
.12299 .22758 L
.1271 .23648 L
.13199 .24435 L
Mistroke
.13799 .25189 L
.14511 .25897 L
.15943 .26944 L
.17809 .27935 L
.19685 .28829 L
.23238 .30509 L
.24176 .30865 L
.25069 .31135 L
.26028 .31356 L
.27121 .31544 L
.29045 .3182 L
.31165 .32207 L
.33328 .32767 L
.35326 .3333 L
.3728 .33781 L
.39332 .34124 L
.42927 .34636 L
.44854 .34871 L
.45407 .34923 L
.46001 .3497 L
.46567 .35007 L
.47088 .35034 L
.47631 .35056 L
.48146 .35072 L
.4869 .35085 L
.48996 .35091 L
.49277 .35095 L
.49743 .351 L
.50006 .35102 L
.50251 .35104 L
.50383 .35104 L
.50507 .35105 L
.50619 .35105 L
.5074 .35105 L
.50871 .35105 L
.51009 .35105 L
.5114 .35105 L
.51257 .35105 L
.51384 .35105 L
.51501 .35104 L
.5163 .35103 L
.51766 .35102 L
.52044 .35099 L
.52295 .35095 L
.52526 .35091 L
.52775 .35085 L
.53223 .3507 L
.53471 .35059 L
.53743 .35044 L
.54292 .35006 L
Mistroke
.54798 .34957 L
.55272 .34899 L
.56291 .34731 L
.57266 .34515 L
.59087 .33992 L
.62646 .32846 L
.66429 .31827 L
.70246 .30697 L
.74001 .29491 L
.77676 .28373 L
.79479 .27859 L
.81504 .27221 L
.83438 .26374 L
.84401 .25778 L
.852 .25131 L
.85897 .24388 L
.86429 .23625 L
.868 .22905 L
.86968 .2249 L
.87101 .22094 L
.87306 .21283 L
.87389 .20842 L
.87462 .20357 L
.8752 .19865 L
.87571 .19335 L
.87609 .18826 L
.87636 .18352 L
.87647 .18131 L
.87656 .17889 L
.87663 .17659 L
.87666 .17556 L
.87668 .17445 L
.8767 .17328 L
.87671 .17217 L
.87672 .17086 L
.87671 .16965 L
.87671 .1684 L
.87669 .16703 L
.87666 .16573 L
.87663 .16453 L
.87654 .16218 L
.87641 .15997 L
.87628 .15799 L
.87609 .15587 L
.87584 .15356 L
.87553 .15114 L
.87481 .14689 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1@o`P0=?l007co00@0ool0
2ol00`3oo`03o`030?oo04co1007o`@0<Ol007co00@0ool02?l500Co0P1:o`<03ol202oo001fo`@0
0_l0103oo`08o`040?oo00Go00<0ool0B?l00`3oo`0Bo`030?oo02co001lo`040?oo00Wo00<0o`00
1Ol00`3oo`17o`030?oo01Co00<0ool0:ol007go0P0;o`801Ol00`3oo`16o`030?oo01Ko00<0ool0
:_l0097o00<0ool0AOl00`3oo`0Go`030?oo02[o001<o`D0@?l00`3oo`14o`030?oo01Wo00<0ool0
:Ol004;o2P04o`T0>?l00`3oo`13o`030?oo01[o00<0ool0:Ol003co1P0Go`<0=Ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003[o0P0Po`80<ol00`3oo`12o`030?oo01_o00<0ool0:Ol003Oo0`0So`80
<_l00`3oo`12o`030?oo01_o00<0ool0:Ol003Go0P0Wo`030?oo033o00<0ool0@_l00`3oo`0Lo`03
0?oo02So000co`80:?l203;o00<0ool0@_l00`3oo`0Lo`030?oo02So000ao`80:_l00`3oo`0ao`03
0?oo04;o00<0ool07?l00`3oo`0Xo`00<?l00`3oo`0Yo`030?oo03;o00<0ool0@Ol00`3oo`0Mo`03
0?oo02So000_o`030?oo02Wo00<0ool0<ol00`3oo`11o`030?oo01go00<0ool0:?l002ko00<0ool0
:Ol00`3oo`0Po`801?l200?o1005o`030?oo047o00<0ool07Ol00`3oo`0Xo`00;_l00`3oo`0Xo`03
0?oo023o00@0ool02?l00`3oo`06o`030?oo047o00<0ool07Ol00`3oo`0Xo`00;Ol00`3oo`0Xo`03
0?oo027o00@0ool02Ol00`3oo`05o`80@Ol00`3oo`0No`030?oo02So000/o`030?oo02So00<0ool0
7?l400;o00@0ool02_l00`3oo`04o`030?oo043o00<0ool07_l00`3oo`0Xo`00;?l00`3oo`0Wo`03
0?oo02?o00@0ool02?l0103oo`05o`030?oo043o00<0ool07_l00`3oo`0Xo`00;?l00`3oo`0Wo`03
0?oo02Co0P0:o`801_l00`3oo`0oo`030?oo01ko00<0ool0:Ol002_o00<0ool09ol00`3oo`0io`03
0?oo03oo00<0ool07_l00`3oo`0Yo`00:ol00`3oo`0Wo`030?oo03Wo00<0ool0?_l00`3oo`0Oo`03
0?oo02Wo000[o`030?oo02Oo00<0ool0>Ol00`3oo`0no`030?oo01oo00<0ool0:Ol000Oo1@02o`80
1?l201Go00<0ool02ol501Oo00<0ool02?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0=o`03
0?oo01?o1@08o`030?oo01Oo1@02o`801?l200?o0009o`030?oo00Oo00@0ool05?l00`3oo`0=o`03
0?oo01So00<0ool01_l0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00co00<0ool0
5Ol00`3oo`08o`030?oo01Wo00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01;o00<0ool0
3Ol00`3oo`0Ho`030?oo00Ko00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`03
0?oo01Ko00<0ool02?l00`3oo`0Io`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Ko
00<0ool01?l400Co00<0ool06Ol00`3o000300;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`80
2ol00`3oo`0Go`030?oo00Oo00<0ool06_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Eo`03
0?oo00co00<0ool06_l00`3oo`04o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool0
2Ol00`3oo`0Ho`030?oo00Oo00<0ool06_l00`3oo`08o`030?oo00;o0008o`802_l301Go00<0ool0
2ol201go00<0ool01?l200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`802?l00`3oo`0Jo`80
2_l300;o000]o`030?oo02[o00<0ool0=?l00`3oo`0ho`030?oo02?o00<0ool0:ol002go00<0ool0
:ol00`3oo`0co`030?oo03Oo00<0ool08ol00`3oo`0/o`00;_l00`3oo`0[o`80<ol00`3oo`0fo`03
0?oo02Co00<0ool0;?l002oo00<0ool0;?l2037o00<0ool0=Ol00`3oo`0To`030?oo02go000_o`03
0?oo02ko0P0_o`030?oo03Co00<0ool09?l00`3oo`0^o`003?oo00`02Ol000co00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o
o`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`05o`030?oo00Ko00<0ool01_l01@3o
ool000Co00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Ro`80
1Ol00`3oo`0Yo`040?l002Wo00<0ool0:Ol00`3oo`04o`030?oo02;o00@0ool0:Ol00`3oo`07o`00
<ol00`3oo`0bo`<09_l00`3oo`0^o`809_l203Co000do`80=Ol202Co00<0ool0;?l202Ko0P0fo`00
=_l00`3oo`0do`<08Ol00`3oo`0Xo`@09_l203So000go`80=ol201oo00<0ool09?l402Oo0`0jo`00
>Ol203Oo100Ko`030?oo027o0`0Yo`80?Ol003_o0P0io`H05Ol00`3oo`0No`<0:?l403oo000mo`80
?Ol900co00<0ool06?l602Oo1013o`00?ol204Co9`0Yo`@0Aol0047o0P1>o`030?oo03ko0`1;o`00
@ol204co00<0ool0>ol304ko0015o`80B_l00`3oo`0go`@0DOl004Oo0P18o`030?oo03Co0`1Eo`00
BOl204Ko00<0ool0<Ol305So001;o`D0;Ol200Co0P03o`@01Ol00`3oo`0]o`@0Fol0053o1`0Uo`04
0?oo00So00<0ool01_l00`3oo`0Zo`<0Gol005Oo1@0Po`040?oo00Wo00<0ool01Ol202So0`1Ro`00
G?l401co00@0ool02_l00`3oo`04o`030?oo02?o101Uo`00H?l301Wo00@0ool02?l0103oo`05o`03
0?oo023o0`1Yo`00Hol401Ko0P0:o`801_l00`3oo`0Mo`<0K?l006Oo1@0Uo`030?oo01[o0`1_o`00
K?l601oo00<0ool05ol307;o001bo`D06_l00`3oo`0Co`@0MOl007Oo1@0Eo`030?oo00oo101io`00
O?l900co00<0ool02Ol607go0025oaP0Pol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool0
2ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00
O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00
oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87453 .14322 m
.87422 .14097 L
.87382 .13862 L
.87287 .13455 L
.87214 .13217 L
.87135 .13001 L
.86956 .12621 L
.86722 .12259 L
.86464 .11963 L
.86131 .1168 L
.85724 .1143 L
.85291 .11243 L
.85028 .11157 L
.84775 .1109 L
.8453 .11039 L
.84253 .10994 L
.84123 .10976 L
.83983 .10961 L
.83858 .10949 L
.83724 .10939 L
.83642 .10934 L
.83551 .10929 L
.83467 .10926 L
.8339 .10923 L
.83298 .10921 L
.83199 .1092 L
.83107 .10919 L
.83019 .1092 L
.82937 .10921 L
.82849 .10923 L
.8269 .10929 L
.82601 .10933 L
.82518 .10937 L
.82327 .1095 L
.81996 .1098 L
.81692 .11015 L
.81004 .11125 L
.80414 .11254 L
.798 .11425 L
.78765 .11824 L
.78196 .12125 L
.77711 .1245 L
.77326 .12773 L
.76958 .13161 L
.76654 .13572 L
.76417 .13983 L
.76213 .14441 L
.76042 .14956 L
.75918 .15449 L
.75825 .15927 L
Mistroke
.75687 .16831 L
.75539 .17803 L
.75435 .18329 L
.75319 .18793 L
.75185 .19215 L
.75012 .19658 L
.74619 .20428 L
.74094 .21217 L
.73422 .2206 L
.71939 .23739 L
.7045 .25327 L
.69731 .25979 L
.69029 .26495 L
.68247 .26939 L
.67326 .27322 L
.66461 .27595 L
.65513 .27842 L
.61962 .28722 L
.61122 .28917 L
.60221 .29081 L
.59733 .29145 L
.5945 .29174 L
.59182 .29196 L
.5893 .29211 L
.58698 .29222 L
.58568 .29227 L
.58427 .29231 L
.58291 .29234 L
.58166 .29235 L
.58034 .29236 L
.57907 .29236 L
.57793 .29236 L
.57668 .29234 L
.57531 .29232 L
.57385 .29229 L
.5712 .29222 L
.56641 .29204 L
.56105 .29178 L
.54885 .29109 L
.53773 .2905 L
.52737 .29008 L
.52225 .28993 L
.51669 .28982 L
.51114 .28973 L
.5052 .28966 L
.49999 .28959 L
.49442 .28951 L
.48863 .28938 L
.48321 .28922 L
.47731 .28899 L
Mistroke
.47074 .28867 L
.45892 .28791 L
.44799 .287 L
.43678 .28585 L
.42673 .28457 L
.40937 .28149 L
.40077 .27936 L
.3915 .27656 L
.37394 .27015 L
.33466 .25335 L
.3193 .24595 L
.3068 .23858 L
.29479 .22896 L
.28921 .22289 L
.28644 .21912 L
.2842 .21535 L
.28245 .21154 L
.28168 .20934 L
.28112 .20727 L
.28077 .20552 L
.28062 .20451 L
.28052 .20357 L
.28044 .20258 L
.28041 .20167 L
.28041 .20066 L
.28044 .19961 L
.28053 .19853 L
.28064 .19754 L
.28082 .19641 L
.28104 .19534 L
.28152 .19345 L
.2822 .19141 L
.28429 .18687 L
.2868 .18282 L
.29252 .17576 L
.29837 .16975 L
.30455 .1635 L
.30694 .16073 L
.30788 .15945 L
.30859 .15833 L
.30915 .15727 L
.3096 .15615 L
.30969 .15586 L
.30977 .15556 L
.30987 .15502 L
.30993 .15453 L
.30995 .15399 L
.30993 .1537 L
.30991 .15343 L
.30981 .15292 L
Mistroke
.30974 .15264 L
.30966 .15238 L
.30942 .15182 L
.30918 .15137 L
.30887 .15089 L
.30807 .14995 L
.30714 .14913 L
.30613 .14841 L
.30334 .14689 L
.30013 .14562 L
.29657 .14453 L
.29196 .14341 L
.28735 .14254 L
.28163 .14168 L
.27593 .14102 L
.27045 .14053 L
.2653 .14019 L
.26234 .14004 L
.25957 .13992 L
.25667 .13982 L
.25516 .13978 L
.25349 .13975 L
.25193 .13972 L
.25113 .13971 L
.25028 .1397 L
.24875 .13969 L
.24731 .13969 L
.2465 .13969 L
.24562 .13969 L
.24469 .13969 L
.24383 .1397 L
.2423 .13972 L
.24067 .13974 L
.23912 .13977 L
.23745 .13981 L
.23444 .13989 L
.2289 .14011 L
.22351 .1404 L
.21794 .14078 L
.20832 .14165 L
.19861 .14284 L
.18866 .1445 L
.17961 .14652 L
.17152 .14884 L
.16251 .15213 L
.15425 .15591 L
.14484 .16131 L
.13645 .16741 L
.129 .17439 L
.12555 .17844 L
Mistroke
.12234 .18298 L
.12003 .187 L
.11794 .1916 L
.11719 .19368 L
.1165 .19594 L
.116 .1979 L
.11555 .20009 L
.11517 .20261 L
.11502 .204 L
.11492 .20531 L
.11486 .20653 L
.11484 .20766 L
.11484 .2089 L
.11488 .21021 L
.11491 .21091 L
.11495 .21154 L
.11507 .21297 L
.11522 .21429 L
.11539 .21554 L
.11588 .21834 L
.11658 .22135 L
.11814 .2264 L
.12027 .23163 L
.1251 .24051 L
.13087 .24841 L
.13792 .25582 L
.14519 .26178 L
.15225 .2664 L
.1883 .2832 L
.22389 .3014 L
.233 .30569 L
.24164 .30896 L
.24643 .31038 L
.25091 .31144 L
.25599 .31238 L
.26163 .31313 L
.2664 .3136 L
.27161 .31398 L
.28123 .31454 L
.28723 .31491 L
.29293 .31535 L
.29806 .31586 L
.30361 .31656 L
.31372 .31832 L
.32463 .32089 L
.34603 .32721 L
.35684 .33032 L
.36708 .33286 L
.38616 .33641 L
.42284 .34218 L
Mistroke
.44197 .34549 L
.45082 .34674 L
.46081 .34784 L
.46658 .34833 L
.47197 .3487 L
.48272 .34925 L
.48821 .34946 L
.49421 .34966 L
.50607 .35003 L
.51669 .35036 L
.52176 .35052 L
.52635 .35064 L
.52888 .3507 L
.53017 .35072 L
.53155 .35074 L
.53226 .35075 L
.53292 .35075 L
.5342 .35076 L
.53488 .35076 L
.53562 .35076 L
.5364 .35076 L
.53711 .35076 L
.53837 .35074 L
.53956 .35073 L
.54062 .3507 L
.54178 .35067 L
.54436 .35058 L
.54674 .35046 L
.54909 .3503 L
.55124 .35013 L
.55609 .34961 L
.5606 .34897 L
.56491 .34823 L
.57463 .34607 L
.58527 .34308 L
.62358 .33094 L
.65925 .3222 L
.69951 .30956 L
.73443 .2956 L
.76809 .28305 L
.78718 .27732 L
.80942 .27096 L
.81974 .2675 L
.83045 .26308 L
.83992 .25806 L
.84771 .25272 L
.85505 .24609 L
.86059 .23936 L
.86525 .23146 L
.86706 .22732 L
Mistroke
.86868 .22267 L
.86987 .21827 L
.87078 .21402 L
.87202 .20581 L
.87254 .20108 L
.87296 .19658 L
.8738 .18593 L
.87455 .17503 L
.87487 .16951 L
.87499 .16707 L
.8751 .16437 L
.87514 .1632 L
.87518 .16194 L
.8752 .16085 L
.87523 .15967 L
.87524 .15834 L
.87525 .15708 L
.87525 .15585 L
.87524 .15472 L
.87522 .15349 L
.87519 .15234 L
.87515 .15108 L
.87509 .14973 L
.87501 .1484 L
.87491 .14701 L
.87469 .14456 L
.87453 .14322 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1=o`l0<?l007co00@0ool0
2ol00`3oo`03o`030?oo04_o0P0?o`030?oo02go001lo`040?oo00So1@04o`80B_l201;o00<0ool0
;?l007Ko1002o`040?oo00So00@0ool01Ol00`3oo`18o`030?oo01?o00<0ool0:ol007co00@0ool0
2Ol00`3o0005o`030?oo04Go0`0Go`030?oo02[o001mo`802ol200Go00<0ool0A?l00`3oo`0Ho`03
0?oo02[o002Ao`030?oo04Co00<0ool06?l202_o002Ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00
@?lI03So00<0ool0@ol00`3oo`0Jo`030?oo02Wo000ko`D06Ol303Go00<0ool0@_l00`3oo`0Ko`03
0?oo02Wo000io`808Ol203?o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000go`809Ol00`3oo`0`o`03
0?oo047o00<0ool07?l00`3oo`0Yo`00=?l302Oo00<0ool0<?l00`3oo`11o`030?oo01co00<0ool0
:Ol003;o0P0Yo`80<_l00`3oo`11o`030?oo01co00<0ool0:Ol0037o00<0ool0:Ol00`3oo`0ao`03
0?oo047o00<0ool07?l00`3oo`0Yo`00<?l00`3oo`0Yo`030?oo03;o00<0ool0@Ol00`3oo`0Lo`03
0?oo02Wo000_o`030?oo02So0P0eo`030?oo047o00<0ool07?l00`3oo`0Yo`00;_l00`3oo`0Xo`03
0?oo027o0P04o`800ol400Go00<0ool0@?l00`3oo`0Mo`030?oo02Wo000]o`030?oo02Wo00<0ool0
8?l0103oo`08o`030?oo00Ko00<0ool0@?l00`3oo`0Mo`030?oo02Wo000/o`030?oo02Wo00<0ool0
8Ol0103oo`09o`030?oo00Go0P11o`030?oo01go00<0ool0:Ol002_o00<0ool0:Ol00`3oo`0Lo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@?l00`3oo`0Mo`030?oo02Wo000[o`030?oo02Wo00<0ool0
8_l0103oo`08o`040?oo00Go00<0ool0?ol00`3oo`0No`030?oo02Wo000[o`030?oo02So00<0ool0
9?l200[o0P06o`030?oo03oo00<0ool07_l00`3oo`0Yo`00:_l00`3oo`0Yo`030?oo03So00<0ool0
?ol00`3oo`0No`030?oo02Wo000Zo`030?oo02Wo00<0ool0>?l00`3oo`0no`030?oo01oo00<0ool0
:Ol002[o00<0ool0:Ol00`3oo`0ho`030?oo03ko00<0ool07_l00`3oo`0Zo`001ol500;o0P04o`80
5?l00`3oo`0<o`D06?l00`3oo`07o`801?l200Co0P0So`030?oo02;o0P04o`801?l200go00<0ool0
4ol500Oo00<0ool06?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Co`030?oo00ko00<0ool0
6Ol00`3oo`05o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`03
0?oo00Oo00<0ool06_l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool04Ol00`3oo`0>o`03
0?oo01Wo00<0ool01Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo00<0ool0
5_l00`3oo`07o`030?oo01[o00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805?l200Oo
1004o`030?oo01[o1@02o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3o
o`07o`030?oo01[o00<0ool02?l200?o0009o`030?oo00So00<0ool05?l00`3oo`0=o`030?oo01[o
0P05o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00Oo
00<0ool06_l00`3oo`08o`030?oo00;o0008o`802_l301Co00<0ool03?l201ko00<0ool00ol200[o
0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`802?l00`3oo`0Jo`802_l300;o000[o`030?oo02go
00<0ool0<ol00`3oo`0io`030?oo02;o00<0ool0:ol002co00<0ool0;Ol00`3oo`0bo`030?oo03So
00<0ool08_l00`3oo`0/o`00;Ol00`3oo`0]o`030?oo037o00<0ool0=ol00`3oo`0Ro`030?oo02go
000]o`030?oo02ko0P0ao`030?oo03Ko00<0ool08ol00`3oo`0]o`00;_l00`3oo`0_o`80;ol00`3o
o`0eo`030?oo02?o00<0ool0;_l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko0`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko0P07o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00801?l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo027o00<0ool01Ol00`3o
o`0Yo`030?oo0080:?l00`3oo`0Yo`030?oo00Ko00<0ool08?l302[o00<0ool01ol0037o0P0fo`<0
9Ol00`3oo`0ao`030?oo023o0P0fo`00<ol203Oo0P0So`030?oo02oo0P0Ro`030?oo03Ko000eo`80
=ol3023o00<0ool0;Ol202;o0P0io`00=ol203So0P0No`030?oo02[o0`0Oo`D0>ol003Wo0P0ho`<0
6ol00`3oo`0Xo`807_l4043o000ko`80>Ol501Ko00<0ool09?l401ko0P14o`00?Ol203co100Bo`03
0?oo01oo1@0Oo`<0A_l003oo0P0noa000_l00`3oo`0Jo`D08Ol304Wo0011o`80C?lO02?o0`1<o`00
@ol204co00<0ool0>_l304oo0015o`80B_l00`3oo`0ho`80D_l004Oo00<0ool0Aol00`3oo`0fo`80
E?l004So0`16o`030?oo03Co0P1Fo`00Bol402ko0P04o`800ol400Go00<0ool0<Ol305So001?o`/0
8_l0103oo`08o`030?oo00Ko00<0ool0;Ol405_o001Jo`L06ol0103oo`09o`030?oo00Go0P0Zo`@0
Gol0067o0`0Ho`040?oo00[o00<0ool01?l00`3oo`0Vo`<0Hol006Co0`0Eo`040?oo00So00@0ool0
1Ol00`3oo`0Ro`@0I_l006Oo1@0Ao`802_l200Ko00<0ool07ol306[o001/o`@08Ol00`3oo`0Lo`<0
KOl0073o1@0Lo`030?oo01So101`o`00MOl501Oo00<0ool05Ol307Co001jo`P03ol00`3oo`0Bo`<0
Mol008;o3003o`030?oo00_o1`1jo`00S_lA087o002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`04
0?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87426 .13955 m
.87419 .13834 L
.87411 .13725 L
.87387 .13485 L
.87356 .13269 L
.87321 .13071 L
.87269 .12848 L
.87212 .1265 L
.87133 .1243 L
.87046 .12234 L
.86852 .119 L
.8673 .11738 L
.86584 .11577 L
.86415 .11423 L
.86238 .1129 L
.85865 .11075 L
.8564 .10977 L
.85423 .109 L
.85175 .10828 L
.84892 .10765 L
.84615 .10719 L
.8449 .10702 L
.8435 .10686 L
.84224 .10674 L
.84088 .10664 L
.83941 .10656 L
.83802 .1065 L
.83716 .10648 L
.83624 .10646 L
.83544 .10645 L
.83457 .10645 L
.83362 .10646 L
.83259 .10648 L
.83161 .10651 L
.8307 .10654 L
.82904 .10662 L
.8272 .10674 L
.82345 .10706 L
.81998 .10746 L
.81625 .10799 L
.80952 .10923 L
.80251 .11092 L
.79625 .11282 L
.78982 .11526 L
.78326 .1184 L
.77373 .12479 L
.76943 .12885 L
.76602 .133 L
.76321 .13741 L
.76076 .14252 L
Mistroke
.75899 .14758 L
.75774 .15241 L
.7568 .15729 L
.75612 .16174 L
.7549 .17167 L
.75424 .17664 L
.75339 .18175 L
.75248 .186 L
.75123 .19052 L
.74954 .19525 L
.74733 .20016 L
.74231 .20873 L
.71508 .24393 L
.70857 .25231 L
.70166 .26046 L
.69492 .26701 L
.68802 .27209 L
.68081 .27595 L
.67234 .27922 L
.66231 .28208 L
.65287 .28434 L
.6334 .28907 L
.62456 .29144 L
.61676 .29347 L
.61285 .29438 L
.60855 .29524 L
.60476 .29587 L
.60047 .29642 L
.59805 .29665 L
.59575 .29682 L
.59448 .2969 L
.59327 .29695 L
.59185 .29701 L
.59055 .29704 L
.5894 .29706 L
.58816 .29707 L
.5871 .29707 L
.58591 .29707 L
.58463 .29705 L
.58324 .29702 L
.58191 .29698 L
.58068 .29693 L
.57551 .29666 L
.56983 .29626 L
.55936 .29531 L
.53566 .2931 L
.5244 .29234 L
.51817 .29203 L
.51247 .29181 L
.5013 .29148 L
Mistroke
.49061 .29118 L
.48039 .2908 L
.4691 .29021 L
.45645 .2893 L
.44512 .28827 L
.43506 .28714 L
.42505 .28575 L
.41491 .28393 L
.40603 .28183 L
.3869 .27549 L
.36718 .26723 L
.3316 .25067 L
.31866 .2439 L
.3127 .24028 L
.30639 .23579 L
.3002 .23049 L
.29466 .22461 L
.29215 .2214 L
.28966 .21767 L
.28776 .21421 L
.28607 .21031 L
.28536 .2082 L
.28482 .20623 L
.28442 .20433 L
.28427 .20337 L
.28413 .20232 L
.28406 .20171 L
.28402 .20113 L
.28395 .20006 L
.28394 .19945 L
.28393 .1989 L
.28393 .19829 L
.28395 .19764 L
.28402 .19657 L
.28411 .19557 L
.28425 .19447 L
.28444 .19329 L
.28497 .19097 L
.28573 .18847 L
.28763 .18401 L
.29002 .17985 L
.29261 .17617 L
.30375 .16382 L
.30835 .15866 L
.30986 .1564 L
.31016 .15581 L
.31043 .1552 L
.31062 .15464 L
.31075 .15414 L
.3108 .15387 L
Mistroke
.31084 .15359 L
.31086 .15335 L
.31087 .15309 L
.31086 .15281 L
.31083 .15252 L
.31078 .15225 L
.31072 .152 L
.31057 .15154 L
.31033 .15105 L
.31005 .1506 L
.30973 .15019 L
.30892 .14938 L
.30799 .14868 L
.30669 .14791 L
.30529 .14724 L
.30199 .14601 L
.29766 .14483 L
.29327 .14391 L
.28812 .14306 L
.28275 .14235 L
.27735 .14178 L
.27087 .14125 L
.26479 .14087 L
.26197 .14074 L
.25897 .14062 L
.25637 .14054 L
.25356 .14047 L
.25199 .14044 L
.25055 .14041 L
.24918 .1404 L
.24774 .14038 L
.24616 .14037 L
.24469 .14037 L
.24308 .14038 L
.24218 .14038 L
.24136 .14039 L
.23986 .1404 L
.23846 .14042 L
.23529 .14048 L
.23223 .14056 L
.22898 .14066 L
.22321 .14092 L
.21808 .14121 L
.21228 .14163 L
.20618 .14219 L
.19564 .14347 L
.18553 .14521 L
.1768 .14726 L
.167 .15031 L
.15787 .154 L
Mistroke
.1491 .1584 L
.13996 .16402 L
.13178 .17024 L
.12499 .17671 L
.11879 .18449 L
.11603 .18909 L
.11391 .19365 L
.11313 .19574 L
.11242 .19799 L
.11191 .19997 L
.11146 .20212 L
.11126 .20333 L
.11111 .20445 L
.11098 .20558 L
.11089 .20665 L
.11082 .20782 L
.11078 .20911 L
.11078 .21037 L
.11079 .21107 L
.11081 .21171 L
.11088 .21293 L
.11097 .21407 L
.1111 .21532 L
.11128 .21666 L
.11176 .2194 L
.11233 .22189 L
.11378 .22665 L
.11553 .23108 L
.12058 .24044 L
.12751 .24956 L
.13436 .25631 L
.1426 .26247 L
.15179 .26758 L
.1609 .27152 L
.19785 .28674 L
.21609 .29697 L
.22534 .30225 L
.23038 .30483 L
.23517 .30698 L
.23978 .30871 L
.24488 .31021 L
.24718 .31074 L
.24964 .31122 L
.25198 .31159 L
.25415 .31187 L
.25637 .31209 L
.25752 .31218 L
.25879 .31227 L
.25997 .31234 L
.26106 .31239 L
Mistroke
.26212 .31242 L
.26325 .31246 L
.26459 .31248 L
.26535 .31249 L
.26605 .3125 L
.26735 .3125 L
.26873 .31249 L
.26997 .31248 L
.27114 .31247 L
.27381 .31242 L
.27927 .31229 L
.28219 .31222 L
.28379 .3122 L
.28458 .31219 L
.28531 .31218 L
.28662 .31217 L
.28804 .31216 L
.28951 .31216 L
.29025 .31217 L
.29106 .31218 L
.29239 .3122 L
.29382 .31223 L
.29517 .31228 L
.29639 .31232 L
.29907 .31246 L
.30152 .31263 L
.30421 .31287 L
.30705 .31318 L
.31178 .31384 L
.31693 .31476 L
.32779 .31737 L
.33745 .32027 L
.35871 .3269 L
.37789 .33131 L
.38858 .33305 L
.39825 .33445 L
.43442 .3413 L
.44377 .34321 L
.45421 .34498 L
.45992 .34575 L
.46638 .34646 L
.47806 .3474 L
.52237 .34993 L
.5272 .35027 L
.53233 .3506 L
.53465 .35074 L
.53709 .35086 L
.53918 .35095 L
.54034 .35099 L
.54143 .35102 L
Mistroke
.54207 .35104 L
.54276 .35105 L
.54398 .35107 L
.54521 .35107 L
.54636 .35107 L
.54706 .35106 L
.54773 .35105 L
.54898 .35103 L
.54968 .35101 L
.55032 .35098 L
.55178 .35092 L
.55397 .35079 L
.55638 .3506 L
.56129 .35005 L
.56634 .34925 L
.57118 .34828 L
.58035 .34597 L
.60085 .33944 L
.63554 .32974 L
.65288 .32621 L
.67322 .32101 L
.69273 .3141 L
.7099 .30655 L
.74097 .29106 L
.77734 .27672 L
.7987 .27082 L
.80959 .26782 L
.81947 .26471 L
.8288 .26115 L
.83831 .25654 L
.84729 .25076 L
.85446 .24448 L
.8601 .23763 L
.86264 .2335 L
.86461 .22952 L
.86626 .22522 L
.86745 .22121 L
.86849 .21649 L
.86922 .21191 L
.86973 .20746 L
.87009 .20334 L
.87078 .19371 L
.87228 .17482 L
.87426 .13955 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1Ao`/0<?l007go0P04o`801Ol300Co00<0ool0COl400[o0P0_o`00
O?l0103oo`0;o`030?oo00?o00<0ool0B_l3013o0P0]o`00O?l0103oo`08o`D01?l204Wo0P0Do`80
;?l007Ko1002o`040?oo00So00@0ool01Ol00`3oo`16o`805ol00`3oo`0Zo`00O?l0103oo`09o`03
0?l000Go00<0ool0AOl00`3oo`0Go`030?oo02[o001mo`802ol200Go00<0ool0A?l00`3oo`0Io`03
0?oo02Wo002Ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00TOl00`3oo`13o`030?oo01[o00<0ool0
:Ol004;o5P0io`030?oo04;o00<0ool06ol00`3oo`0Yo`00?Ol501Ko1@0do`030?oo047o00<0ool0
7?l00`3oo`0Yo`00=ol6023o0P0bo`030?oo047o00<0ool07?l00`3oo`0Yo`00=Ol202Oo00<0ool0
<?l00`3oo`11o`030?oo01co00<0ool0:Ol003Co00<0ool09ol00`3oo`0`o`030?oo047o00<0ool0
7?l00`3oo`0Yo`00<ol00`3oo`0Wo`80<_l00`3oo`11o`030?oo01co00<0ool0:Ol0037o0P0Yo`03
0?oo03;o00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000`o`030?oo02So00<0ool0<ol00`3oo`10o`03
0?oo01go00<0ool0:Ol002oo00<0ool0:?l00`3oo`0do`030?oo043o00<0ool07Ol00`3oo`0Yo`00
;_l00`3oo`0Xo`030?oo027o0P04o`800ol400Go00<0ool0@?l00`3oo`0Mo`030?oo02Wo000]o`03
0?oo02Wo00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0@?l00`3oo`0Mo`030?oo02Wo000/o`03
0?oo02Wo00<0ool08Ol0103oo`09o`030?oo00Go0P11o`030?oo01go00<0ool0:Ol002_o00<0ool0
:_l00`3oo`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0@?l00`3oo`0Lo`030?oo02[o000Zo`03
0?oo02[o0P0So`040?oo00So00@0ool01Ol00`3oo`0oo`030?oo01go00<0ool0:_l002[o00<0ool0
:_l00`3oo`0So`802_l200Ko00<0ool0?ol00`3oo`0Mo`030?oo02[o000Yo`80;?l00`3oo`0go`03
0?oo03ko00<0ool07_l00`3oo`0Zo`00:Ol00`3oo`0[o`030?oo03Oo00<0ool0?_l00`3oo`0No`03
0?oo02[o000Yo`030?oo02_o00<0ool0=ol00`3oo`0mo`030?oo01oo00<0ool0:_l000Oo1@02o`80
1?l201?o00<0ool03Ol501Wo0P07o`801?l200Co0P0So`030?oo02;o0P04o`801?l200go00<0ool0
4ol500Oo00<0ool06?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Bo`030?oo00oo00<0ool0
6_l00`3oo`04o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool02ol00`3oo`0Fo`03
0?oo00Oo00<0ool06_l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool04?l00`3oo`0?o`03
0?oo01_o00<0ool00ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00So00<0ool0
5ol00`3oo`07o`030?oo01[o00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`804ol00`3o
o`07o`@01?l00`3oo`0Ko`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`030?oo01Oo
00<0ool01_l00`3oo`0Ko`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01;o00<0ool03ol00`3o
o`0Lo`030?oo00;o00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`09o`030?oo01So
00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<04ol00`3oo`0=o`807_l00`3o
o`03o`802_l302;o00<0ool08_l200[o0`08o`030?oo01So0P08o`030?oo01[o0P0:o`<00_l002[o
00<0ool0;_l203Co00<0ool0>?l00`3oo`0Ro`030?oo02co000[o`030?oo02oo0P0bo`030?oo03So
00<0ool08_l00`3oo`0/o`00;?l00`3oo`0`o`030?oo02oo00<0ool0=ol00`3oo`0Ro`030?oo02go
000/o`030?oo037o0P0_o`030?oo03Ko00<0ool08ol00`3oo`0]o`00;Ol00`3oo`0bo`80;Ol00`3o
o`0eo`030?oo02;o0P0`o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1Ol200So00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`<01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?l000Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00@0o`001Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo023o00<0ool01_l00`3oo`0Yo`03
0?oo0080:?l00`3oo`0Yo`030?oo00Oo00<0ool07ol202_o00<0ool01ol0033o0P0go`809_l00`3o
o`0bo`030?oo01ko0P0go`00<_l203Oo0P0To`030?oo037o00<0ool07?l303Wo000do`030?oo03Ko
0P0Ro`030?oo033o00<0ool06ol203co000eo`<0=ol2023o00<0ool0;ol00`3oo`0Go`D0?_l003So
0P0go`D06ol00`3oo`0/o`<05_l404?o000jo`<0>Ol401Oo00<0ool0:_l201Oo0P17o`00?Ol203_o
0P0Eo`030?oo02?o1`0Go`80BOl003oo0`0jo`/02_l00`3oo`0Po`<07?l204_o0012o`80@olG01Co
0P0Lo`<0COl004Co0P1;o`030?oo00[o2@02o`T07?l2053o0016o`030?oo04So00<0ool04ol302;o
0P1Bo`00Aol00`3oo`17o`030?oo03Ko0P1Do`00B?l304Ko00<0ool0<ol305Ko001;o`D0;Ol200Co
0P03o`@01Ol00`3oo`0ao`80FOl0053o4P0Jo`040?oo00So00<0ool01_l00`3oo`0_o`80Fol006;o
0P0Ho`040?oo00Wo00<0ool01Ol202ko0P1Mo`00I?l401Co00@0ool02_l00`3oo`04o`030?oo02So
1@1Oo`00J?l500oo00@0ool02?l0103oo`05o`030?oo02?o1@1To`00KOl200ko0P0:o`801_l00`3o
o`0Po`<0JOl006oo2@0Io`030?oo01co101/o`00N?l501Co00<0ool06Ol3073o001mo`@04?l00`3o
o`0Fo`<0Lol0087o1P0:o`030?oo01Co0P1fo`00Qol;013o1P1ho`00TOlA07ko002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87398 .13587 m
.87399 .13464 L
.87399 .13353 L
.87396 .13226 L
.87391 .13108 L
.87385 .13001 L
.87378 .12903 L
.87354 .12688 L
.87339 .12579 L
.87322 .12479 L
.87274 .12258 L
.87215 .12054 L
.87137 .11846 L
.87046 .11658 L
.8693 .11466 L
.86812 .11311 L
.86669 .11158 L
.86512 .1102 L
.86354 .10907 L
.86006 .10719 L
.85781 .10629 L
.85525 .10549 L
.85292 .10492 L
.85021 .10442 L
.84861 .1042 L
.84712 .10402 L
.84536 .10386 L
.84453 .1038 L
.84364 .10375 L
.84287 .10371 L
.84202 .10367 L
.84049 .10363 L
.83962 .10362 L
.83868 .10362 L
.83778 .10362 L
.83694 .10363 L
.83541 .10367 L
.83376 .10374 L
.83218 .10382 L
.83073 .10392 L
.82712 .10423 L
.82365 .10462 L
.81664 .10567 L
.80899 .1072 L
.79604 .11087 L
.78912 .11353 L
.78308 .11639 L
.77737 .11975 L
.77253 .12331 L
.7677 .12784 L
Mistroke
.76402 .13236 L
.76111 .13704 L
.759 .1415 L
.75714 .14687 L
.75586 .15206 L
.75499 .15695 L
.75439 .16147 L
.75337 .17155 L
.75285 .17639 L
.75213 .18142 L
.75123 .18603 L
.75018 .1901 L
.74861 .19474 L
.74649 .19963 L
.74155 .20827 L
.71535 .24483 L
.70331 .26277 L
.69719 .27016 L
.69098 .27577 L
.68414 .28012 L
.6804 .28192 L
.67609 .28362 L
.66653 .28651 L
.65727 .28871 L
.62183 .29772 L
.61335 .29983 L
.60844 .30076 L
.60603 .30111 L
.60373 .30139 L
.60168 .30158 L
.59943 .30174 L
.59812 .30181 L
.59686 .30186 L
.59572 .30189 L
.59447 .30191 L
.59323 .30191 L
.59252 .30191 L
.59185 .30191 L
.59114 .30189 L
.59049 .30188 L
.589 .30184 L
.58778 .30179 L
.58644 .30172 L
.58369 .30156 L
.57786 .30106 L
.57217 .30045 L
.56125 .29904 L
.54926 .29739 L
.53685 .29583 L
.52628 .29476 L
Mistroke
.52079 .29433 L
.51495 .29395 L
.50467 .29346 L
.49483 .29308 L
.48324 .2926 L
.47008 .29189 L
.45744 .29097 L
.44595 .28991 L
.4341 .28852 L
.42388 .28699 L
.41547 .28536 L
.40697 .28322 L
.38961 .27711 L
.3692 .26799 L
.33343 .25036 L
.32016 .24302 L
.31058 .23642 L
.30573 .23223 L
.30025 .22654 L
.29521 .21992 L
.29122 .21281 L
.28963 .20889 L
.28894 .20676 L
.28835 .20449 L
.28794 .20246 L
.28776 .2013 L
.28762 .20022 L
.28753 .19922 L
.28747 .19831 L
.28742 .19726 L
.28741 .19627 L
.28743 .19518 L
.2875 .19398 L
.28755 .19338 L
.28761 .19273 L
.28775 .19157 L
.28815 .18932 L
.28863 .18729 L
.28935 .18496 L
.29029 .18248 L
.29243 .17811 L
.29779 .1702 L
.3031 .16407 L
.30846 .1582 L
.31015 .15604 L
.31083 .15499 L
.31113 .15444 L
.31137 .15394 L
.31154 .15349 L
.31162 .15325 L
Mistroke
.31169 .15301 L
.31177 .15259 L
.31181 .15221 L
.31181 .15185 L
.31178 .15151 L
.31171 .15115 L
.31158 .15078 L
.3114 .1504 L
.31114 .15 L
.31052 .14931 L
.30972 .14867 L
.3088 .14811 L
.30749 .14748 L
.30604 .14692 L
.30282 .14595 L
.29865 .14502 L
.29339 .14412 L
.28303 .14287 L
.27653 .1423 L
.27019 .14186 L
.26329 .14151 L
.25689 .14127 L
.25333 .14118 L
.25132 .14114 L
.24946 .14111 L
.24784 .14109 L
.2469 .14108 L
.24605 .14107 L
.24515 .14106 L
.24419 .14106 L
.24333 .14106 L
.24244 .14106 L
.24082 .14106 L
.23991 .14106 L
.23907 .14107 L
.23753 .14108 L
.23591 .1411 L
.23311 .14115 L
.23007 .14121 L
.22686 .14131 L
.22346 .14143 L
.21746 .14172 L
.21166 .14209 L
.20634 .14252 L
.19686 .14353 L
.19171 .14424 L
.18721 .14499 L
.17759 .14704 L
.16907 .1495 L
.16118 .15241 L
Mistroke
.1525 .15635 L
.14322 .16146 L
.12701 .17317 L
.11962 .18046 L
.11608 .18483 L
.11316 .18918 L
.11088 .19339 L
.10987 .19568 L
.10894 .19815 L
.10828 .20025 L
.10769 .2026 L
.10727 .20476 L
.10698 .20681 L
.10686 .208 L
.10677 .20928 L
.10673 .21049 L
.10671 .2116 L
.10673 .2129 L
.10676 .21364 L
.1068 .21431 L
.1069 .21556 L
.10705 .21689 L
.10745 .21952 L
.10796 .22197 L
.10959 .2275 L
.11148 .23217 L
.11355 .23632 L
.11924 .24506 L
.12626 .25297 L
.13351 .25908 L
.14203 .2644 L
.15162 .26869 L
.17035 .27473 L
.18821 .28113 L
.1974 .2855 L
.20681 .29082 L
.21599 .29659 L
.22548 .30246 L
.2301 .305 L
.2345 .30709 L
.23883 .30877 L
.24291 .30998 L
.24687 .31083 L
.24892 .31115 L
.2512 .31141 L
.2524 .31152 L
.25352 .31159 L
.25482 .31166 L
.25606 .3117 L
.25721 .31172 L
Mistroke
.25829 .31172 L
.25949 .31171 L
.26077 .31168 L
.26206 .31163 L
.26349 .31157 L
.26642 .31138 L
.27178 .31093 L
.27777 .31034 L
.28432 .30972 L
.28755 .30947 L
.29062 .30927 L
.29195 .3092 L
.29338 .30914 L
.29412 .30912 L
.29494 .30909 L
.29638 .30906 L
.29784 .30904 L
.29918 .30904 L
.30064 .30906 L
.30138 .30908 L
.30218 .3091 L
.30354 .30915 L
.30499 .30922 L
.30759 .3094 L
.31054 .30967 L
.31363 .31006 L
.31912 .31097 L
.32492 .31225 L
.33118 .31394 L
.35218 .32084 L
.3624 .32398 L
.37206 .32641 L
.38312 .3285 L
.39302 .32997 L
.42822 .33659 L
.43824 .33919 L
.44821 .34153 L
.45778 .34333 L
.46296 .3441 L
.46879 .34482 L
.48019 .34588 L
.49241 .34671 L
.51353 .34826 L
.52484 .34941 L
.5349 .35054 L
.53913 .35097 L
.54357 .35133 L
.54587 .35147 L
.54718 .35154 L
.54841 .35158 L
Mistroke
.54953 .35161 L
.55056 .35163 L
.55174 .35163 L
.55287 .35163 L
.55405 .35161 L
.55515 .35158 L
.55615 .35154 L
.55723 .35148 L
.55966 .35131 L
.56192 .35109 L
.56608 .35055 L
.57053 .34977 L
.58004 .3475 L
.59059 .34429 L
.61094 .33753 L
.62815 .33297 L
.64567 .32986 L
.66271 .32652 L
.67152 .32415 L
.68101 .321 L
.70039 .31259 L
.73035 .29525 L
.74607 .28602 L
.76228 .27843 L
.78064 .27229 L
.80233 .26669 L
.81315 .2638 L
.82455 .26014 L
.83472 .25591 L
.84314 .25131 L
.85138 .2452 L
.85746 .23884 L
.86001 .23528 L
.86233 .23123 L
.86424 .22686 L
.86558 .22267 L
.86652 .21855 L
.86692 .21618 L
.86722 .21395 L
.86763 .20981 L
.8678 .20746 L
.86792 .20529 L
.86828 .19696 L
.8685 .19219 L
.86876 .18759 L
.86912 .18264 L
.86964 .17703 L
.87082 .16675 L
.87244 .15421 L
.87312 .14835 L
Mistroke
.87341 .1455 L
.87363 .1429 L
.87375 .14127 L
.87386 .13949 L
.8739 .13863 L
.87393 .13772 L
.87396 .13676 L
.87398 .13587 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1Co`P0<Ol0097o00<0ool0C_l500So0P0_o`00OOl200Co0P05o`<01?l00`3oo`1;o`<0
3ol202go001lo`040?oo00_o00<0ool00ol00`3oo`18o`<04ol202co001lo`040?oo00So1@04o`80
Aol201Oo00<0ool0:_l007Ko1002o`040?oo00So00@0ool01Ol00`3oo`15o`030?oo01Oo0P0[o`00
O?l0103oo`09o`030?l000Go00<0ool0A?l00`3oo`0Io`030?oo02Wo001mo`802ol200Go00<0ool0
A?l00`3oo`0Io`030?oo02Wo002Ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00TOl00`3oo`12o`03
0?oo01_o00<0ool0:Ol004;o5@0jo`030?oo047o00<0ool07?l00`3oo`0Yo`00?Ol501Go1P0do`03
0?oo047o00<0ool07?l00`3oo`0Yo`00>?l5023o0P0bo`030?oo047o00<0ool07?l00`3oo`0Yo`00
=_l202Ko00<0ool0<?l00`3oo`11o`030?oo01co00<0ool0:Ol003Co0P0Xo`030?oo033o00<0ool0
@?l00`3oo`0Mo`030?oo02Wo000bo`80:Ol203;o00<0ool0@?l00`3oo`0Mo`030?oo02Wo000ao`03
0?oo02So00<0ool0<_l00`3oo`10o`030?oo01co00<0ool0:_l002oo0P0[o`030?oo03;o00<0ool0
@?l00`3oo`0Lo`030?oo02[o000^o`030?oo02[o00<0ool0<ol00`3oo`10o`030?oo01co00<0ool0
:_l002go00<0ool0:_l00`3oo`0Po`801?l200?o1005o`030?oo043o00<0ool07?l00`3oo`0Zo`00
;?l00`3oo`0Zo`030?oo023o00@0ool02?l00`3oo`06o`030?oo043o00<0ool07?l00`3oo`0Zo`00
:ol00`3oo`0[o`030?oo023o00@0ool02Ol00`3oo`05o`80@Ol00`3oo`0Lo`030?oo02[o000Zo`03
0?oo02_o00<0ool06ol400;o00@0ool02_l00`3oo`04o`030?oo03oo00<0ool07Ol00`3oo`0Zo`00
:_l00`3oo`0[o`030?oo027o00@0ool02?l0103oo`05o`030?oo03oo00<0ool07Ol00`3oo`0Zo`00
:Ol00`3oo`0/o`030?oo02;o0P0:o`801_l00`3oo`0oo`030?oo01go00<0ool0:_l002So0P0^o`03
0?oo03Ko00<0ool0?_l00`3oo`0Mo`030?oo02_o000Xo`030?oo02go00<0ool0=_l00`3oo`0no`03
0?oo01go00<0ool0:ol002So00<0ool0;Ol00`3oo`0fo`030?oo03go00<0ool07_l00`3oo`0[o`00
1ol500;o0P04o`804_l00`3oo`0>o`D06ol00`3oo`04o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200go00<0ool04ol500Ko00<0ool06Ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ao`03
0?oo013o00<0ool06ol00`3oo`03o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
2ol00`3oo`0Fo`030?oo00Ko00<0ool06ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
3ol00`3oo`0@o`030?oo01_o00<0ool00ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`03
0?oo00Wo00<0ool05_l00`3oo`06o`030?oo01_o00<0ool02_l1007o0Ol00003o`0000801?l00`3o
o`08o`804_l00`3oo`08o`@01?l00`3oo`0Ko`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo
0P0;o`030?oo01Oo00<0ool01_l00`3oo`0Ko`030?oo00So0P03o`002Ol00`3oo`08o`030?oo017o
00<0ool04?l00`3oo`0Lo`030?oo00;o00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3o
o`09o`030?oo01So00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<04_l00`3o
o`0>o`807ol200?o0P0:o`<08_l00`3oo`0Ro`802_l300So00<0ool06?l200Oo00<0ool06ol200[o
0`02o`00:Ol00`3oo`0ao`030?oo037o00<0ool0>Ol00`3oo`0Qo`030?oo02co000Zo`030?oo037o
00<0ool0<?l00`3oo`0ho`030?oo027o00<0ool0;Ol002[o00<0ool0<_l2033o00<0ool0=ol00`3o
o`0Ro`030?oo02go000[o`030?oo03?o00<0ool0;Ol00`3oo`0go`030?oo027o00<0ool0;_l002_o
00<0ool0=?l202go00<0ool0=_l00`3oo`0Po`80<Ol000coo`0<00Wo000<o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko00@0o`001Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`03
0?oo01oo0P08o`030?oo02Wo00@0oooo0P0Wo`030?oo02Wo00<0ool02?l00`3oo`0No`80:ol00`3o
o`07o`00<?l00`3oo`0go`809Ol00`3oo`0co`030?oo01co0`0go`00<Ol203Wo0P0So`030?oo03;o
00<0ool05ol603[o000co`<0>?l2027o00<0ool0<Ol00`3oo`0Eo`<0@?l003Ko0P0ho`807ol00`3o
o`0`o`030?oo01?o0`13o`00>?l403Ko0`0Lo`030?oo02oo00<0ool04_l204Ko000lo`<0=_l201[o
00<0ool0;?l301;o0`18o`00?ol203Ko1@0Eo`030?oo02_o00<0ool04Ol204_o0011o`80>Ol5013o
00<0ool09Ol601;o0P1=o`00@ol00`3oo`0koa409?l301Ko0P1?o`00A?l204_o300Fo`<06?l00`3o
o`1?o`00A_l00`3oo`18o`030?oo00Wo200:o`@06Ol205;o0017o`030?oo04Oo00<0ool04Ol:01_o
0P1Do`00B?l304Ko00<0ool0=Ol00`3oo`1Do`00BolF01co0P04o`800ol400Go00<0ool0<ol205Oo
001Qo`D05_l0103oo`08o`030?oo00Ko00<0ool0<Ol205Wo001Vo`805?l0103oo`09o`030?oo00Go
0P0`o`80Fol006So0`0Ao`040?oo00[o00<0ool01?l00`3oo`0/o`<0GOl006_o1@0<o`040?oo00So
00@0ool01Ol00`3oo`0Vo`H0H?l0073o1`06o`802_l200Ko00<0ool08Ol506Ko001go`@05_l00`3o
o`0Mo`@0Jol007_o100Bo`030?oo01Wo101_o`00Ool500go00<0ool05ol207?o0024o`H01ol00`3o
o`0Do`<0MOl008[o3`0<o`<0N?l0097o00<0ool01Ol<07_o002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87371 .1322 m
.8738 .13094 L
.87387 .1298 L
.87393 .12851 L
.87396 .12731 L
.87397 .12622 L
.87397 .12523 L
.87394 .12416 L
.87388 .12305 L
.8738 .12195 L
.87368 .1208 L
.87354 .11974 L
.87338 .11879 L
.87289 .11657 L
.87227 .11459 L
.87153 .11283 L
.87059 .11108 L
.86846 .10829 L
.86687 .10681 L
.86517 .10556 L
.86148 .10363 L
.85939 .10287 L
.85691 .10216 L
.85559 .10186 L
.85412 .10158 L
.85281 .10137 L
.85134 .10118 L
.84984 .10102 L
.8484 .1009 L
.847 .10081 L
.84548 .10074 L
.84409 .10071 L
.84277 .1007 L
.84128 .10071 L
.83966 .10074 L
.83879 .10077 L
.83785 .10081 L
.83614 .1009 L
.83407 .10103 L
.83216 .10119 L
.82793 .10164 L
.82384 .10218 L
.81649 .10342 L
.80846 .10518 L
.79486 .10924 L
.78761 .11213 L
.7813 .11523 L
.77535 .11884 L
.77034 .12263 L
.76539 .12744 L
Mistroke
.76165 .13221 L
.75875 .13714 L
.75668 .1418 L
.75575 .14446 L
.75491 .14738 L
.75429 .14993 L
.75373 .15275 L
.75297 .15785 L
.75266 .16063 L
.75242 .16321 L
.7517 .17262 L
.75123 .17779 L
.75088 .18064 L
.75049 .18322 L
.74957 .18782 L
.74825 .19254 L
.74674 .19663 L
.74503 .20038 L
.74016 .20877 L
.71564 .24502 L
.70491 .26418 L
.69921 .27291 L
.69578 .27705 L
.69226 .28046 L
.68856 .2833 L
.6842 .28591 L
.67991 .2879 L
.67571 .28945 L
.66688 .29194 L
.65719 .29409 L
.6235 .30282 L
.61972 .30389 L
.61557 .30492 L
.61155 .30572 L
.60769 .30629 L
.60548 .30652 L
.6042 .30662 L
.60297 .3067 L
.60234 .30673 L
.60166 .30676 L
.60045 .3068 L
.59976 .30682 L
.59912 .30683 L
.59841 .30683 L
.59764 .30683 L
.59637 .30681 L
.59565 .3068 L
.59497 .30678 L
.59372 .30673 L
.59238 .30666 L
Mistroke
.59001 .30651 L
.58735 .30629 L
.58157 .30565 L
.57588 .30486 L
.56485 .30307 L
.55269 .30093 L
.54003 .29884 L
.52832 .29724 L
.51805 .2962 L
.51313 .29582 L
.50779 .29548 L
.4979 .29498 L
.47471 .2938 L
.46085 .29284 L
.44806 .29168 L
.43642 .29034 L
.42671 .28891 L
.41662 .28696 L
.40793 .28468 L
.39982 .28193 L
.391 .27831 L
.37348 .27001 L
.33665 .25093 L
.32325 .24322 L
.31487 .23728 L
.3066 .22942 L
.302 .22382 L
.29942 .22012 L
.29712 .21627 L
.29507 .2122 L
.29321 .20749 L
.29253 .20533 L
.29193 .20301 L
.29151 .201 L
.29117 .19877 L
.29109 .19813 L
.29102 .19743 L
.29097 .19682 L
.29092 .19616 L
.29087 .19492 L
.29085 .19376 L
.29085 .19306 L
.29088 .19231 L
.29091 .1916 L
.29095 .19094 L
.29106 .1897 L
.29122 .18841 L
.29162 .18605 L
.29212 .18388 L
.2937 .1791 L
Mistroke
.29568 .17479 L
.29824 .17043 L
.30345 .1635 L
.3082 .15809 L
.31145 .15411 L
.312 .15324 L
.3124 .15246 L
.31257 .15206 L
.31263 .15185 L
.3127 .15163 L
.31277 .15126 L
.3128 .15092 L
.3128 .15072 L
.31279 .15054 L
.31276 .15034 L
.31271 .15014 L
.31259 .14981 L
.3125 .14963 L
.3124 .14946 L
.31193 .14889 L
.31159 .14859 L
.31122 .14832 L
.31026 .14777 L
.30916 .14729 L
.30773 .1468 L
.30596 .14632 L
.30244 .14558 L
.2978 .14485 L
.29295 .14425 L
.28729 .14369 L
.27608 .14285 L
.26938 .14247 L
.26296 .14219 L
.25578 .14195 L
.25173 .14186 L
.24983 .14182 L
.24805 .1418 L
.2464 .14178 L
.24489 .14176 L
.24322 .14175 L
.24142 .14174 L
.23971 .14174 L
.23874 .14174 L
.23784 .14174 L
.23621 .14175 L
.23447 .14177 L
.23287 .14179 L
.23115 .14181 L
.22806 .14187 L
.22508 .14195 L
Mistroke
.22237 .14203 L
.21628 .14229 L
.20983 .14265 L
.20338 .14314 L
.19765 .14371 L
.19232 .14435 L
.1868 .14518 L
.1774 .14707 L
.16878 .14948 L
.16103 .15229 L
.14453 .16034 L
.13473 .16644 L
.12587 .17301 L
.11811 .18002 L
.11092 .18858 L
.10792 .19338 L
.10569 .19802 L
.1047 .20066 L
.10397 .2031 L
.10364 .20446 L
.10333 .20595 L
.10309 .20737 L
.10291 .20867 L
.10279 .20992 L
.10271 .2111 L
.10266 .21235 L
.10265 .21306 L
.10265 .21372 L
.10268 .21482 L
.10275 .21603 L
.10285 .21729 L
.10299 .21849 L
.10342 .22121 L
.10397 .2237 L
.10571 .22928 L
.10783 .23421 L
.11069 .23942 L
.11782 .24905 L
.12582 .25671 L
.13379 .26223 L
.13841 .2647 L
.14355 .26694 L
.15321 .27008 L
.17257 .27488 L
.18381 .27839 L
.19419 .28283 L
.21213 .29362 L
.22147 .30003 L
.22993 .30524 L
.23453 .30749 L
Mistroke
.23877 .30912 L
.24124 .30985 L
.2436 .31041 L
.24615 .31085 L
.24761 .31104 L
.24896 .31117 L
.25009 .31125 L
.2513 .31131 L
.25246 .31134 L
.25353 .31135 L
.25479 .31134 L
.25551 .31132 L
.25618 .31129 L
.25742 .31123 L
.25874 .31113 L
.2614 .31089 L
.26395 .31059 L
.26991 .30971 L
.28122 .3078 L
.28756 .30684 L
.29068 .30644 L
.29352 .30614 L
.29597 .30592 L
.29862 .30574 L
.30008 .30567 L
.30142 .30562 L
.3027 .30559 L
.30405 .30558 L
.30541 .30559 L
.30687 .30562 L
.30823 .30566 L
.30948 .30572 L
.31093 .30582 L
.31251 .30594 L
.31536 .30624 L
.3183 .30665 L
.32144 .30719 L
.32705 .30843 L
.33724 .31141 L
.35922 .31902 L
.3699 .32209 L
.37981 .32423 L
.38999 .32585 L
.40078 .32731 L
.41056 .32885 L
.41942 .33072 L
.4357 .33535 L
.44377 .33773 L
.45304 .34009 L
.45868 .34126 L
Mistroke
.46405 .3422 L
.47509 .34366 L
.48077 .34422 L
.48704 .34476 L
.49845 .34565 L
.52174 .3481 L
.53264 .34976 L
.53772 .35057 L
.54229 .35124 L
.54632 .35174 L
.54845 .35195 L
.55073 .35213 L
.55192 .3522 L
.55302 .35226 L
.55412 .35229 L
.55517 .35232 L
.55634 .35233 L
.55743 .35232 L
.55862 .3523 L
.55923 .35228 L
.5599 .35225 L
.56099 .3522 L
.56219 .35212 L
.56462 .3519 L
.56689 .35163 L
.56902 .35132 L
.57389 .35042 L
.57918 .34916 L
.59795 .34325 L
.61675 .33718 L
.62664 .33479 L
.63547 .33328 L
.65187 .33109 L
.66169 .32932 L
.67124 .32688 L
.68074 .32365 L
.68933 .32002 L
.70729 .31025 L
.73597 .28979 L
.75087 .28034 L
.76693 .27314 L
.77716 .26983 L
.78755 .26709 L
.79926 .26433 L
.81195 .26122 L
.82311 .25794 L
.83465 .25346 L
.844 .24845 L
.85137 .24298 L
.85485 .23961 L
Mistroke
.85805 .23573 L
.86051 .2319 L
.86232 .22823 L
.86359 .22485 L
.86462 .22103 L
.86501 .21908 L
.86535 .21696 L
.8656 .21493 L
.86577 .21306 L
.86585 .21196 L
.86591 .21092 L
.86599 .20896 L
.86603 .20785 L
.86605 .20682 L
.86607 .20568 L
.86608 .20511 L
.86609 .20447 L
.8661 .2034 L
.8661 .20241 L
.86611 .20127 L
.86611 .20017 L
.86612 .19922 L
.86612 .1982 L
.86613 .19707 L
.86614 .19602 L
.86615 .19467 L
.86617 .19341 L
.8662 .19203 L
.86624 .19054 L
.86634 .18801 L
.86648 .18517 L
.86665 .18259 L
.86688 .17977 L
.86741 .17455 L
.86883 .16424 L
.87077 .15266 L
.87371 .1322 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Eo`H0
<Ol0097o00<0ool0Col600Ko0`0^o`00TOl00`3oo`1=o`803ol202co001mo`801?l200Go0`04o`03
0?oo04Wo100Bo`030?oo02[o001lo`040?oo00_o00<0ool00ol00`3oo`17o`805_l202_o001lo`04
0?oo00So1@04o`80Aol00`3oo`0Go`030?oo02Wo001fo`@00_l0103oo`08o`040?oo00Go00<0ool0
A?l201[o00<0ool0:Ol007co00@0ool02Ol00`3o0005o`030?oo04?o00<0ool06_l00`3oo`0Yo`00
OOl200_o0P05o`030?oo04;o00<0ool06ol00`3oo`0Yo`00TOl00`3oo`12o`030?oo01_o00<0ool0
:Ol0097o00<0ool0@Ol00`3oo`0Lo`030?oo02Wo0013oa@0>_l00`3oo`11o`030?oo01co00<0ool0
:Ol003go1P0Do`L0<ol00`3oo`11o`030?oo01_o00<0ool0:_l003So1@0Po`<0<Ol00`3oo`10o`03
0?oo01co00<0ool0:_l003Ko0P0Vo`80<Ol00`3oo`10o`030?oo01co00<0ool0:_l003Co0P0Xo`03
0?oo033o00<0ool0@?l00`3oo`0Lo`030?oo02[o000bo`80:Ol00`3oo`0ao`030?oo043o00<0ool0
7?l00`3oo`0Zo`00<?l202[o00<0ool0<_l00`3oo`10o`030?oo01co00<0ool0:_l002oo00<0ool0
:_l00`3oo`0bo`030?oo03oo00<0ool07Ol00`3oo`0Zo`00;_l00`3oo`0Zo`030?oo03?o00<0ool0
?ol00`3oo`0Lo`030?oo02_o000]o`030?oo02[o00<0ool08?l200Co0P03o`@01Ol00`3oo`0oo`03
0?oo01co00<0ool0:ol002_o0P0]o`030?oo01oo00@0ool02?l00`3oo`06o`030?oo03oo00<0ool0
7?l00`3oo`0[o`00:_l00`3oo`0/o`030?oo023o00@0ool02Ol00`3oo`05o`80@?l00`3oo`0Lo`03
0?oo02_o000Zo`030?oo02co00<0ool06_l400;o00@0ool02_l00`3oo`04o`030?oo03oo00<0ool0
7?l00`3oo`0[o`00:Ol00`3oo`0]o`030?oo023o00@0ool02?l0103oo`05o`030?oo03oo00<0ool0
7?l00`3oo`0[o`00:?l00`3oo`0^o`030?oo027o0P0:o`801_l00`3oo`0oo`030?oo01co00<0ool0
:ol002So00<0ool0;_l00`3oo`0eo`030?oo03ko00<0ool07Ol00`3oo`0[o`009ol00`3oo`0_o`03
0?oo03Go00<0ool0?_l00`3oo`0Mo`030?oo02_o000Wo`030?oo02oo00<0ool0=Ol00`3oo`0mo`03
0?oo01ko00<0ool0:ol000Oo1@02o`801?l2017o00<0ool03ol501_o00<0ool01?l200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0<o`030?oo01Co1@06o`030?oo01Wo1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool04?l00`3oo`0Ao`030?oo01co00<0ool00_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00[o00<0ool05ol00`3oo`06o`030?oo01_o00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo00ko00<0ool04Ol00`3oo`0Mo`050?ooo`000_l00`3oo`09o`030?oo023o
00<0ool08Ol0103oo`0;o`030?oo00So00<0ool05ol00`3oo`06o`030?oo01_o00<0ool02_l1007o
0Ol00003o`0000801?l00`3oo`08o`804Ol00`3oo`09o`@01?l00`3oo`0Ko`@00_l0103oo`09o`80
8ol00`3oo`0Qo`040?oo00Wo0P0:o`030?oo01So00<0ool01Ol201go00<0ool02?l200?o0009o`03
0?oo00So00<0ool04?l00`3oo`0Ao`030?oo01ko00@0ool00_l00`3oo`07o`030?oo02;o00<0ool0
8Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`05o`030?oo01co00<0ool02?l00`3oo`02o`00
2?l200[o0`0Ao`030?oo00oo0P0Po`040?ooo`802_l302;o00<0ool08_l200[o0`08o`030?oo01So
0P07o`030?oo01_o0P0:o`<00_l002So00<0ool0<_l00`3oo`0ao`030?oo03So00<0ool08Ol00`3o
o`0]o`00:?l00`3oo`0co`030?oo033o00<0ool0>?l00`3oo`0Qo`030?oo02go000Yo`030?oo03?o
0P0`o`030?oo03Oo00<0ool08Ol00`3oo`0^o`00:_l00`3oo`0do`80;_l00`3oo`0go`030?oo023o
00<0ool0;ol002[o00<0ool0=_l202co00<0ool0=_l00`3oo`0Po`030?oo033o000<ool03009o`00
3?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`03
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00801?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?oo00Go
00<0ool01Ol00`3oo`06o`030?oo00Go0`07o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool07_l00`3oo`08o`030?oo02Wo00<0ool00_l202Ko00<0ool0:Ol00`3o
o`08o`030?oo01[o0`000ol0o`0[o`030?oo00Oo000^o`80>ol202Co00<0ool0=?l00`3oo`0Go`<0
>ol0033o00<0ool0>_l202;o00<0ool0<ol00`3oo`0Eo`<0?_l0037o1@0io`030?oo01oo00<0ool0
<_l00`3oo`0@o`H0@Ol003Ko0P0ho`807ol00`3oo`0ao`030?oo00oo0P17o`00>?l303Oo0P0Mo`03
0?oo037o00<0ool03Ol204Wo000ko`<0=_l401Wo00<0ool0<?l00`3oo`0<o`80Bol003ko0P0ho`80
5ol00`3oo`0_o`030?oo00_o0P1=o`00@?l203So1@0Bo`030?oo02_o100=o`030?oo04go0012o`80
>ol:00So00<0ool0:?l3013o00<0ool0C_l004Co00<0ool0@_l>027o100Ao`80DOl004Go0P1:o`03
0?oo00?o1`0Fo`@05?l00`3oo`1Ao`00Aol204So00<0ool02_l600co100Go`030?oo05;o0019o`80
2_l=02oo00<0ool04?l<01Wo0P1Eo`00Bol:00go1@0Fo`801?l200?o1005o`030?oo03Co00<0ool0
EOl006Oo0`0Bo`040?oo00So00<0ool01_l00`3oo`0bo`80F?l006[o0`0?o`040?oo00Wo00<0ool0
1Ol2037o0P1Jo`00KOl300co00@0ool02_l00`3oo`04o`030?oo02ko0P1Lo`00L?l500Oo00@0ool0
2?l0103oo`05o`030?oo02Wo1@1No`00MOl70003o`0000[o0P06o`030?oo02Oo0P1So`00O?l201?o
00<0ool08?l706Go001no`D03_l00`3oo`0Ko`D0K?l008?o100:o`030?oo01Wo0P1ao`00Qol<01Ko
101co`00TOl00`3o000400go101go`00TOl00`3oo`04o`d0Nol0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0
S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87343 .12853 m
.87375 .12608 L
.8739 .12477 L
.87401 .12354 L
.8741 .12243 L
.87416 .12124 L
.8742 .12018 L
.87422 .11921 L
.87421 .1181 L
.87418 .11708 L
.87411 .11598 L
.874 .11484 L
.87384 .11368 L
.87365 .11262 L
.87341 .1116 L
.87317 .11071 L
.87255 .10896 L
.87171 .10724 L
.87076 .10577 L
.86973 .10451 L
.86832 .10317 L
.86685 .10208 L
.86504 .10102 L
.86289 .10008 L
.86063 .09933 L
.8584 .09878 L
.85709 .09853 L
.85561 .09829 L
.85429 .09812 L
.85277 .09796 L
.85134 .09785 L
.84996 .09777 L
.84846 .09772 L
.8476 .0977 L
.8468 .0977 L
.84544 .0977 L
.84391 .09772 L
.84229 .09778 L
.84075 .09785 L
.83887 .09796 L
.83679 .09812 L
.83293 .0985 L
.82542 .09952 L
.81637 .10118 L
.80793 .10316 L
.8008 .10519 L
.79335 .10774 L
.78092 .1133 L
.77422 .11731 L
.76862 .12155 L
Mistroke
.76428 .12571 L
.76031 .13057 L
.75703 .13595 L
.75467 .14126 L
.7531 .14618 L
.75242 .14899 L
.75191 .15159 L
.75149 .15423 L
.75113 .15706 L
.75083 .16016 L
.75062 .16302 L
.75027 .16916 L
.75014 .17181 L
.74997 .17468 L
.74979 .17715 L
.74956 .17976 L
.74924 .18247 L
.74887 .18495 L
.74836 .18764 L
.7477 .19041 L
.74618 .19521 L
.74432 .1996 L
.74225 .20361 L
.73645 .21273 L
.71228 .25174 L
.70265 .27207 L
.69992 .27686 L
.69677 .28138 L
.69381 .28477 L
.69018 .28798 L
.68631 .29059 L
.68245 .29258 L
.67765 .29448 L
.6728 .29597 L
.65378 .30014 L
.63717 .30417 L
.62885 .30675 L
.62146 .30904 L
.61793 .30997 L
.61411 .31078 L
.61204 .31112 L
.60971 .31141 L
.60867 .31152 L
.60753 .31161 L
.60644 .31168 L
.60544 .31173 L
.60425 .31177 L
.60364 .31179 L
.60297 .3118 L
.60176 .3118 L
Mistroke
.60061 .31178 L
.59991 .31176 L
.59924 .31174 L
.59796 .31168 L
.59656 .3116 L
.595 .31148 L
.58936 .31089 L
.58384 .3101 L
.56119 .30568 L
.53691 .30081 L
.53078 .29981 L
.52446 .29893 L
.51867 .29827 L
.51329 .29776 L
.50266 .29701 L
.49042 .29633 L
.47754 .29561 L
.46484 .29474 L
.44147 .2925 L
.43049 .291 L
.4201 .28911 L
.4122 .28718 L
.40413 .28458 L
.38778 .27739 L
.36722 .2667 L
.32983 .24581 L
.32033 .23936 L
.31382 .23339 L
.30617 .22413 L
.30147 .2167 L
.29769 .20863 L
.29616 .20406 L
.29554 .20169 L
.29498 .19904 L
.29462 .19675 L
.29447 .19546 L
.29437 .19428 L
.29429 .19318 L
.29425 .19198 L
.29423 .19086 L
.29424 .18981 L
.2943 .1885 L
.29434 .18776 L
.29439 .18709 L
.29452 .18585 L
.29469 .18456 L
.29508 .18233 L
.29564 .17994 L
.29732 .17484 L
.29935 .17041 L
Mistroke
.30431 .16259 L
.30902 .15679 L
.31223 .1529 L
.31285 .15202 L
.31335 .15117 L
.31353 .15078 L
.31366 .15043 L
.31375 .15013 L
.3138 .14982 L
.31382 .1495 L
.31379 .14922 L
.31376 .14907 L
.31371 .14891 L
.31358 .14863 L
.31341 .14839 L
.31319 .14814 L
.31267 .14773 L
.31231 .14751 L
.31188 .1473 L
.31096 .14693 L
.30964 .14654 L
.30821 .14621 L
.3052 .14569 L
.30125 .14519 L
.29247 .1444 L
.28146 .14369 L
.26783 .14305 L
.26061 .1428 L
.25677 .14269 L
.25269 .14259 L
.2491 .14252 L
.2472 .14249 L
.24517 .14247 L
.2433 .14245 L
.2416 .14244 L
.2399 .14243 L
.2383 .14242 L
.23668 .14242 L
.23495 .14243 L
.23332 .14244 L
.23184 .14245 L
.23012 .14247 L
.22825 .14249 L
.22487 .14255 L
.22141 .14264 L
.21777 .14276 L
.21142 .14303 L
.20527 .1434 L
.19974 .14383 L
.19008 .14488 L
Mistroke
.18529 .14558 L
.18029 .14647 L
.17087 .14873 L
.16263 .15145 L
.15503 .15464 L
.13686 .16455 L
.11975 .177 L
.1115 .18499 L
.10525 .19314 L
.10271 .19766 L
.10057 .20283 L
.09978 .20549 L
.09945 .20691 L
.09915 .20841 L
.09895 .20973 L
.09886 .21044 L
.09877 .21119 L
.09867 .21252 L
.09861 .21376 L
.09859 .21506 L
.0986 .21579 L
.09862 .21648 L
.09866 .21721 L
.0987 .21787 L
.09884 .21937 L
.09901 .22064 L
.09923 .22203 L
.09976 .22455 L
.10056 .22748 L
.10163 .23057 L
.10407 .23601 L
.10985 .24519 L
.1138 .24994 L
.11785 .25401 L
.12626 .26057 L
.13073 .26322 L
.13578 .26564 L
.1402 .26733 L
.14502 .26881 L
.15391 .27082 L
.16379 .27257 L
.17443 .27471 L
.18 .2762 L
.18609 .27824 L
.19654 .28301 L
.20608 .28882 L
.21574 .29579 L
.22437 .30207 L
.23228 .30683 L
.23651 .30871 L
Mistroke
.23893 .30953 L
.24115 .31014 L
.24365 .31065 L
.24508 .31086 L
.24642 .311 L
.2476 .31109 L
.24828 .31113 L
.24891 .31115 L
.24959 .31117 L
.25021 .31118 L
.25093 .31117 L
.25161 .31116 L
.25277 .31111 L
.25406 .31103 L
.25542 .31092 L
.25671 .31078 L
.26163 .31005 L
.26722 .30893 L
.27347 .30748 L
.28594 .30461 L
.29215 .30343 L
.29782 .30262 L
.2993 .30246 L
.3009 .3023 L
.30228 .30219 L
.3038 .3021 L
.3054 .30202 L
.30628 .30199 L
.3071 .30198 L
.30856 .30196 L
.31012 .30198 L
.311 .302 L
.31182 .30203 L
.31271 .30207 L
.31366 .30212 L
.31541 .30225 L
.31698 .3024 L
.32016 .3028 L
.32312 .30329 L
.32845 .30443 L
.33416 .306 L
.34537 .30984 L
.36482 .31675 L
.3755 .31961 L
.38513 .32145 L
.3961 .32299 L
.40579 .32428 L
.41429 .32575 L
.42288 .32784 L
.44082 .33379 L
Mistroke
.44958 .33659 L
.45985 .33922 L
.46516 .34028 L
.4704 .34115 L
.48062 .34246 L
.52451 .34746 L
.53527 .34958 L
.54473 .35149 L
.54875 .35217 L
.55094 .35248 L
.55298 .35272 L
.5552 .35291 L
.55638 .35299 L
.55699 .35302 L
.55765 .35305 L
.55881 .35309 L
.55992 .3531 L
.561 .3531 L
.562 .35308 L
.56307 .35305 L
.56405 .353 L
.56513 .35293 L
.56628 .35284 L
.56823 .35264 L
.57037 .35235 L
.57531 .35147 L
.57998 .35038 L
.59997 .34399 L
.61766 .33828 L
.62222 .33714 L
.62692 .33617 L
.6309 .33551 L
.63513 .33496 L
.63951 .3345 L
.64362 .33413 L
.64849 .33369 L
.65321 .33317 L
.65768 .33255 L
.66268 .33165 L
.67168 .3294 L
.68151 .32593 L
.69029 .32191 L
.70825 .31087 L
.723 .29883 L
.73528 .28797 L
.74856 .27827 L
.75635 .27399 L
.76595 .26993 L
.77651 .26657 L
.78698 .26395 L
Mistroke
.79995 .26111 L
.81225 .25829 L
.82319 .25525 L
.83452 .25105 L
.84455 .24581 L
.84926 .24251 L
.85312 .23916 L
.85634 .23564 L
.85881 .23221 L
.86085 .22849 L
.86177 .22634 L
.8625 .22429 L
.86302 .2225 L
.86347 .22054 L
.8638 .21874 L
.86404 .21707 L
.86423 .21523 L
.86431 .21418 L
.86437 .2132 L
.86441 .21215 L
.86444 .21119 L
.86446 .21007 L
.86446 .20901 L
.86446 .20792 L
.86445 .20731 L
.86444 .20674 L
.86442 .20569 L
.86439 .20458 L
.86424 .20045 L
.86416 .19826 L
.86408 .19588 L
.86405 .19449 L
.86402 .19321 L
.86401 .19258 L
.86401 .1919 L
.864 .19066 L
.864 .18952 L
.86401 .18844 L
.86403 .18724 L
.86407 .18594 L
.86411 .18457 L
.86418 .18324 L
.86433 .18072 L
.86451 .17833 L
.86473 .17608 L
.86536 .17083 L
.86726 .15947 L
.86954 .14844 L
.87343 .12853 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Ho`80<_l0097o00<0ool0
D?l90003o`0000<0;Ol0097o00<0ool0COl300ko0P0/o`00TOl00`3oo`1;o`804_l202_o001mo`80
1?l200Go0`04o`030?oo04Wo0P0Eo`030?oo02Wo001lo`040?oo00_o00<0ool00ol00`3oo`17o`80
5ol00`3oo`0Yo`00O?l0103oo`08o`D01?l204Ko0P0Io`030?oo02Wo001fo`@00_l0103oo`08o`04
0?oo00Go00<0ool0A?l00`3oo`0Io`030?oo02Wo001lo`040?oo00Wo00<0o`001Ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol007go0P0;o`801Ol00`3oo`12o`030?oo01_o00<0ool0:Ol0097o00<0ool0
@Ol00`3oo`0Lo`030?oo02Wo002Ao`030?oo047o00<0ool07?l00`3oo`0Yo`00AOl>03ko00<0ool0
@?l00`3oo`0Lo`030?oo02[o000ko`X03_l<03;o00<0ool0@?l00`3oo`0Lo`030?oo02[o000io`80
8ol2037o00<0ool0@?l00`3oo`0Lo`030?oo02[o000go`809_l00`3oo`0_o`030?oo043o00<0ool0
7?l00`3oo`0Zo`00=?l302Oo00<0ool0<?l00`3oo`0oo`030?oo01co00<0ool0:ol003;o0P0Yo`03
0?oo037o00<0ool0?ol00`3oo`0Lo`030?oo02_o000`o`80:_l00`3oo`0bo`030?oo03oo00<0ool0
7?l00`3oo`0[o`00;ol00`3oo`0Zo`030?oo03;o00<0ool0?ol00`3oo`0Lo`030?oo02_o000]o`80
;?l00`3oo`0co`030?oo03oo00<0ool07?l00`3oo`0[o`00;?l00`3oo`0/o`030?oo01oo0P04o`80
0ol400Go00<0ool0?ol00`3oo`0Lo`030?oo02_o000[o`030?oo02co00<0ool07ol0103oo`08o`03
0?oo00Ko00<0ool0?ol00`3oo`0Ko`030?oo02co000Zo`030?oo02go00<0ool07ol0103oo`09o`03
0?oo00Go0P10o`030?oo01_o00<0ool0;?l002Wo00<0ool0;_l00`3oo`0Io`@00_l0103oo`0:o`03
0?oo00Co00<0ool0?ol00`3oo`0Ko`030?oo02co000Xo`030?oo02oo00<0ool07ol0103oo`08o`04
0?oo00Go00<0ool0?_l00`3oo`0Lo`030?oo02co000Wo`030?oo033o00<0ool08?l200[o0P06o`03
0?oo03ko00<0ool07?l00`3oo`0/o`009ol00`3oo`0`o`030?oo03Co00<0ool0?Ol00`3oo`0Mo`03
0?oo02co000Vo`030?oo037o00<0ool0=?l00`3oo`0mo`030?oo01go00<0ool0;?l002Ko00<0ool0
<Ol00`3oo`0do`030?oo03go00<0ool07Ol00`3oo`0/o`001ol500;o0P04o`804?l00`3oo`0@o`D0
7Ol00`3oo`02o`801?l200Co0P0So`030?oo02;o0P04o`801?l200co00<0ool05?l500Go00<0ool0
6_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0?o`030?oo01;o00<0ool07Ol01@3oool000;o
00<0ool01_l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0:o`030?oo01Oo00<0ool01Ol00`3o
o`0Lo`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0=o`030?oo01;o00<0ool07_l0103o
o`02o`030?oo00Wo00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool01ol00`3oo`0Ho`030?oo00Go
00<0ool07?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0@o`030?oo00[o1004o`03
0?oo01_o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200[o00<0ool06?l00`3oo`05o`03
0?oo01co00<0ool02?l200?o0009o`030?oo00So00<0ool03ol00`3oo`0Bo`030?oo01oo00<0o`00
0_l00`3oo`07o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00So00<0ool06Ol00`3oo`05o`03
0?oo01co00<0ool02?l00`3oo`02o`002?l200[o0`0?o`030?oo017o0P0Ro`040?l000[o0`0Ro`03
0?oo02;o0P0:o`<02?l00`3oo`0Ho`801_l201go0P0:o`<00_l002Oo00<0ool0=?l00`3oo`0`o`03
0?oo03So00<0ool08Ol00`3oo`0]o`009ol00`3oo`0eo`80<?l00`3oo`0go`030?oo027o00<0ool0
;_l002So00<0ool0=_l00`3oo`0]o`030?oo03Oo00<0ool08?l00`3oo`0_o`00:?l00`3oo`0go`03
0?oo02co00<0ool0=_l00`3oo`0Po`030?oo033o000Yo`030?oo03Oo0P0/o`030?oo03Ko00<0ool0
7_l203?o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`00
2?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00P04o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0o`001_l00`3oo`05o`030?oo00Ko00<0ool00_l30003o`3o00Oo00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Mo`030?oo00Wo00<0ool0
:Ol00`3oo`02o`809_l00`3oo`0Yo`030?oo00So00<0ool05ol300Co00<0ool0:_l00`3oo`07o`00
;Ol00`3oo`0ko`809?l00`3oo`0co`030?oo01Go0`0no`00;_l303co0P0Ro`030?oo03?o00<0ool0
3ol6047o000ao`80??l2023o00<0ool0<_l00`3oo`0=o`<0Aol003?o1P0ho`807_l00`3oo`0bo`03
0?oo00co00<0ool0B?l003Wo0`0go`807?l00`3oo`0bo`030?oo00_o00<0ool0BOl003co0`0fo`80
6_l00`3oo`0ao`030?oo00[o0P1<o`00?ol203Ko0P0Ho`030?oo033o00<0ool02Ol204ko0011o`80
=_l401Co00<0ool0;ol00`3oo`09o`030?oo04ko0013o`80>?l800co00<0ool0;Ol200_o00<0ool0
Col004Go00<0ool0?Ol;0003o`3o02Wo1@0<o`030?oo053o0016o`030?oo04Oo2P0No`@04?l00`3o
o`1Ao`00Aol00`3oo`0?o`/0;Ol00`3oo`06o`@05ol301;o0P1Do`00B?l200[o1@0;o`80:ol00`3o
o`0:o`L03_l201Co00<0ool0E?l004[o2P0Bo`<05?l200Co0P03o`@01Ol00`3oo`0Ao`h05Ol00`3o
o`1Eo`00JOl3013o00@0ool02?l00`3oo`06o`030?oo03;o0P1Ho`00K?l200ko00@0ool02Ol00`3o
o`05o`80<Ol205[o001^o`P01_l0103oo`0:o`030?oo00Co00<0ool0;_l205co001fo`D000Go0?oo
0008o`040?oo00Go00<0ool0;?l205ko001ko`@02_l200Ko00<0ool0:Ol3063o001oo`<03ol00`3o
o`0Qo`P0Hol008;o0`0<o`030?oo01_o1P1[o`00QOl500Oo00<0ool06Ol2077o002:o`P06?l307?o
002Ao`X03Ol207Ko002Ao`030?oo00Oo3@1ho`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103o
o`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go
00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co
001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o
003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87316 .12485 m
.87364 .12236 L
.87406 .11978 L
.87437 .11743 L
.87447 .11643 L
.87456 .11538 L
.87463 .11417 L
.87466 .11308 L
.87466 .11194 L
.87464 .11131 L
.87462 .11072 L
.87454 .1097 L
.87443 .10879 L
.87427 .10777 L
.87407 .10684 L
.87356 .1051 L
.8732 .10419 L
.87282 .10339 L
.87197 .10196 L
.87099 .10072 L
.86965 .09943 L
.86822 .09839 L
.86644 .0974 L
.8643 .09652 L
.86203 .09585 L
.85977 .09537 L
.85845 .09516 L
.85694 .09497 L
.85546 .09483 L
.85404 .09473 L
.8527 .09467 L
.8514 .09463 L
.84994 .09462 L
.84833 .09463 L
.84668 .09467 L
.84584 .09471 L
.84491 .09475 L
.84163 .09496 L
.83953 .09515 L
.8375 .09535 L
.8336 .09582 L
.82449 .09725 L
.81633 .09891 L
.80741 .10113 L
.79251 .10598 L
.78459 .10935 L
.77773 .1129 L
.77133 .117 L
.76598 .12128 L
.76077 .12665 L
Mistroke
.75693 .13193 L
.75402 .13733 L
.75204 .1424 L
.75119 .14528 L
.75044 .14841 L
.74988 .15139 L
.74948 .15414 L
.7492 .15669 L
.74899 .15906 L
.74882 .16165 L
.74869 .16439 L
.74859 .16724 L
.74851 .1699 L
.74844 .17226 L
.74836 .17476 L
.74824 .17726 L
.74809 .17956 L
.74786 .18223 L
.74757 .18462 L
.74721 .18693 L
.74672 .1894 L
.74539 .19421 L
.74383 .1983 L
.74176 .20257 L
.73709 .21025 L
.7118 .25295 L
.70727 .2645 L
.70281 .27582 L
.69863 .28414 L
.69589 .28816 L
.693 .29142 L
.68965 .29428 L
.68613 .29656 L
.68267 .29827 L
.67863 .29984 L
.67397 .30124 L
.66878 .30248 L
.65923 .30438 L
.62692 .313 L
.62009 .31506 L
.61653 .31585 L
.61444 .3162 L
.61243 .31646 L
.61126 .31658 L
.61014 .31667 L
.60911 .31673 L
.60797 .31677 L
.60737 .31679 L
.60671 .3168 L
.606 .3168 L
Mistroke
.60534 .3168 L
.60413 .31677 L
.60344 .31675 L
.6028 .31672 L
.60158 .31665 L
.60042 .31657 L
.59771 .31633 L
.59254 .31566 L
.58681 .31468 L
.58031 .31336 L
.56801 .31049 L
.54507 .30491 L
.53311 .30243 L
.52245 .30071 L
.51715 .30005 L
.51145 .29947 L
.49897 .29854 L
.47426 .29703 L
.45941 .2959 L
.44596 .29457 L
.43483 .29316 L
.42388 .29132 L
.41397 .289 L
.40569 .2863 L
.3978 .28296 L
.3891 .27861 L
.3712 .26876 L
.33423 .24732 L
.3234 .23985 L
.31812 .23467 L
.31157 .22631 L
.30776 .22052 L
.30347 .21253 L
.30164 .20824 L
.29995 .20334 L
.29918 .20053 L
.29859 .19792 L
.29813 .19527 L
.29794 .19394 L
.29781 .19273 L
.2977 .19153 L
.29763 .1904 L
.29758 .1892 L
.29756 .18792 L
.29757 .18679 L
.2976 .18561 L
.29766 .18461 L
.29774 .18351 L
.29787 .18222 L
.29802 .18106 L
Mistroke
.29847 .17848 L
.299 .17615 L
.2996 .174 L
.30111 .16982 L
.30312 .16557 L
.30772 .15833 L
.31166 .15335 L
.31318 .15146 L
.3142 .15001 L
.31457 .14934 L
.31471 .14899 L
.3148 .1487 L
.31486 .1484 L
.31487 .14823 L
.31487 .14809 L
.31482 .14784 L
.31473 .14759 L
.31461 .1474 L
.31444 .14721 L
.31424 .14703 L
.31402 .14688 L
.31372 .14672 L
.31335 .14656 L
.31253 .14628 L
.31147 .14603 L
.31007 .14579 L
.30835 .14557 L
.3065 .14539 L
.30267 .14511 L
.29783 .14484 L
.27379 .14388 L
.25963 .14346 L
.25563 .14336 L
.25133 .14327 L
.24763 .14321 L
.24553 .14318 L
.24356 .14316 L
.24158 .14314 L
.23978 .14312 L
.23786 .14311 L
.2368 .14311 L
.2358 .14311 L
.23404 .14311 L
.23241 .14311 L
.23063 .14312 L
.22874 .14313 L
.22767 .14314 L
.2267 .14315 L
.2245 .14318 L
.22057 .14326 L
Mistroke
.21706 .14335 L
.21341 .14347 L
.20693 .14376 L
.20115 .14411 L
.19512 .14461 L
.18902 .14526 L
.18347 .14603 L
.17863 .14687 L
.1737 .14792 L
.16855 .14929 L
.16028 .15212 L
.15161 .15595 L
.13431 .16571 L
.12483 .1721 L
.11464 .18016 L
.10596 .18886 L
.10246 .19335 L
.09966 .19777 L
.09837 .20026 L
.09716 .20301 L
.09633 .20533 L
.09559 .20792 L
.09528 .20929 L
.095 .21076 L
.09488 .21158 L
.09478 .21233 L
.0947 .2131 L
.09464 .21381 L
.09456 .21516 L
.09454 .21587 L
.09453 .21663 L
.09455 .21791 L
.09458 .21864 L
.09463 .21933 L
.09474 .22058 L
.0949 .22191 L
.09531 .22431 L
.09598 .22717 L
.09677 .22975 L
.09901 .23528 L
.10174 .24035 L
.10777 .24868 L
.11561 .25645 L
.12418 .26245 L
.12931 .26509 L
.13424 .26707 L
.13953 .26867 L
.14446 .26978 L
.15023 .27072 L
.1558 .27142 L
Mistroke
.166 .27261 L
.17175 .27348 L
.17696 .27453 L
.1818 .27581 L
.18697 .27756 L
.19609 .28174 L
.2053 .2875 L
.21475 .29469 L
.22414 .30206 L
.22817 .30484 L
.23258 .30738 L
.23474 .30839 L
.23707 .3093 L
.23908 .30993 L
.24026 .31023 L
.24135 .31047 L
.24238 .31066 L
.24352 .31083 L
.24467 .31096 L
.2459 .31105 L
.247 .3111 L
.24819 .31111 L
.24933 .31109 L
.25037 .31104 L
.25168 .31095 L
.25309 .3108 L
.25566 .31044 L
.25849 .30991 L
.2617 .30919 L
.27347 .30585 L
.28578 .30224 L
.29199 .30069 L
.29777 .29954 L
.30098 .29904 L
.30437 .29864 L
.30631 .29847 L
.30725 .2984 L
.30812 .29836 L
.30974 .29829 L
.31058 .29827 L
.31148 .29825 L
.31294 .29826 L
.31451 .29829 L
.31597 .29836 L
.3173 .29844 L
.31885 .29857 L
.32053 .29875 L
.32355 .29916 L
.3263 .29965 L
.32923 .30028 L
Mistroke
.33443 .30166 L
.34592 .30562 L
.36831 .31382 L
.37894 .31659 L
.38491 .31774 L
.39039 .31858 L
.40008 .31981 L
.41002 .32115 L
.415 .32205 L
.41943 .32305 L
.42784 .32556 L
.44365 .33163 L
.45209 .33462 L
.46203 .3374 L
.46817 .33871 L
.47401 .33973 L
.48596 .34129 L
.53131 .34779 L
.53957 .34988 L
.54797 .35198 L
.55267 .35294 L
.55473 .35327 L
.55699 .35356 L
.55803 .35367 L
.55914 .35377 L
.5602 .35384 L
.56117 .35389 L
.56226 .35392 L
.56288 .35393 L
.56346 .35394 L
.56452 .35393 L
.56565 .3539 L
.56676 .35385 L
.56779 .35379 L
.56898 .35369 L
.57012 .35358 L
.5744 .35297 L
.57686 .3525 L
.57916 .35199 L
.58447 .3506 L
.60531 .34355 L
.61532 .34029 L
.62001 .33901 L
.6242 .33806 L
.62785 .33739 L
.63178 .33684 L
.63566 .33645 L
.63777 .33629 L
.63974 .33617 L
.64386 .33596 L
Mistroke
.6484 .33573 L
.65051 .3356 L
.65277 .33543 L
.65691 .33502 L
.66096 .33446 L
.66485 .33375 L
.66919 .33273 L
.67393 .33135 L
.68299 .32789 L
.69266 .32297 L
.70996 .31069 L
.72381 .29719 L
.73526 .28528 L
.74153 .27973 L
.74924 .27424 L
.75763 .26971 L
.76641 .26619 L
.77686 .26308 L
.78749 .26063 L
.80066 .25798 L
.81342 .25522 L
.82543 .25193 L
.83143 .24985 L
.83743 .24733 L
.84254 .24471 L
.84673 .24212 L
.85358 .23647 L
.85654 .23309 L
.85902 .22934 L
.86063 .22602 L
.86135 .2241 L
.8619 .22228 L
.86241 .22013 L
.8626 .21907 L
.86275 .21808 L
.86286 .21719 L
.86297 .21621 L
.86305 .21522 L
.86311 .21416 L
.86313 .21357 L
.86315 .213 L
.86316 .21193 L
.86316 .21133 L
.86316 .21077 L
.86315 .21016 L
.86313 .2095 L
.86311 .20892 L
.86308 .20829 L
.86303 .20715 L
.86289 .20499 L
Mistroke
.86254 .20055 L
.86215 .1956 L
.862 .19312 L
.86193 .19188 L
.86188 .19072 L
.86184 .18968 L
.86181 .18852 L
.86179 .18734 L
.86178 .18607 L
.86178 .18472 L
.8618 .18346 L
.86184 .18207 L
.86187 .18136 L
.86191 .18058 L
.86207 .17782 L
.8623 .17519 L
.86265 .17198 L
.86305 .16901 L
.86422 .16215 L
.87316 .12485 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1Co``0;Ol0097o00<0ool0
C_l500_o0P0/o`00TOl00`3oo`1<o`804Ol202_o002Ao`030?oo04[o0P0Do`030?oo02Wo001mo`80
1?l200Go0`04o`030?oo04So0P0Fo`030?oo02Wo001lo`040?oo00_o00<0ool00ol00`3oo`16o`80
6?l00`3oo`0Yo`00O?l0103oo`08o`D01?l204Ko00<0ool06?l00`3oo`0Yo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04?o0P0Ko`030?oo02Wo001lo`040?oo00Wo00<0o`001Ol00`3oo`12o`03
0?oo01_o00<0ool0:Ol007go0P0;o`801Ol00`3oo`11o`030?oo01co00<0ool0:Ol0097o00<0ool0
@Ol00`3oo`0Ko`030?oo02[o002Ao`030?oo043o00<0ool07?l00`3oo`0Zo`00TOl00`3oo`10o`03
0?oo01co00<0ool0:_l003co900ao`030?oo03oo00<0ool07Ol00`3oo`0Zo`00>?l402?o0P0`o`03
0?oo03oo00<0ool07?l00`3oo`0[o`00=_l202Ko0P0ao`030?oo03oo00<0ool07?l00`3oo`0[o`00
=?l202So00<0ool0<?l00`3oo`0oo`030?oo01co00<0ool0:ol003;o0P0Yo`030?oo037o00<0ool0
?ol00`3oo`0Ko`030?oo02co000`o`80:ol00`3oo`0ao`030?oo03oo00<0ool06ol00`3oo`0/o`00
;ol00`3oo`0Zo`030?oo03;o00<0ool0?ol00`3oo`0Ko`030?oo02co000]o`80;Ol00`3oo`0bo`03
0?oo03oo00<0ool06ol00`3oo`0/o`00:ol202ko00<0ool07ol200Co0P03o`@01Ol00`3oo`0oo`03
0?oo01_o00<0ool0;?l002[o00<0ool0;_l00`3oo`0No`040?oo00So00<0ool01_l00`3oo`0no`80
7Ol00`3oo`0/o`00:Ol00`3oo`0_o`030?oo01ko00@0ool02Ol00`3oo`05o`80?ol00`3oo`0Lo`03
0?oo02co000Yo`030?oo02oo00<0ool06?l400;o00@0ool02_l00`3oo`04o`030?oo03ko00<0ool0
7?l00`3oo`0/o`00:?l00`3oo`0`o`030?oo01ko00@0ool02?l0103oo`05o`030?oo03ko00<0ool0
7?l00`3oo`0/o`009ol00`3oo`0ao`030?oo01oo0P0:o`801_l00`3oo`0no`030?oo01co00<0ool0
;?l002Ko00<0ool0<_l00`3oo`0co`030?oo03go00<0ool07Ol00`3oo`0/o`009Ol203Co00<0ool0
<ol00`3oo`0mo`030?oo01go00<0ool0;?l002Go00<0ool0<ol00`3oo`0co`030?oo03co00<0ool0
7_l00`3oo`0/o`001ol500;o0P04o`803ol00`3oo`0Ao`D07_l0103oool200Co0P04o`808ol00`3o
o`0Ro`801?l200Co0P0<o`030?oo01Co1@05o`030?oo01[o1@02o`801?l200?o0009o`030?oo00Oo
00@0ool03_l00`3oo`0Co`030?oo01ko00@0ool00_l00`3oo`06o`040?oo02;o00<0ool08Ol0103o
o`08o`040?oo00[o00<0ool05ol00`3oo`05o`030?oo01co00<0ool01ol0103oo`02o`002Ol00`3o
o`0:o`030?oo00co00<0ool04ol00`3oo`0No`040?oo00;o00<0ool02Ol00`3oo`0Po`030?oo027o
00@0ool02ol00`3oo`08o`030?oo01Oo00<0ool01Ol00`3oo`0Lo`030?oo00[o0@01o`7o00000ol0
000200Co00<0ool02?l200oo00<0ool02ol400Co00<0ool06ol50005o`3oo`002Ol202?o00<0ool0
8Ol0103oo`09o`802_l00`3oo`0Ho`030?oo00Go00<0ool07?l00`3oo`08o`800ol000Wo00<0ool0
2?l00`3oo`0>o`030?oo01?o00<0ool07ol00`3o0002o`030?oo00Oo00<0ool08_l00`3oo`0Qo`04
0?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00Go00<0ool07?l00`3oo`08o`030?oo00;o0008o`80
2_l300ko00<0ool04_l202;o00@0o`002_l302;o00<0ool08_l200[o0`08o`030?oo01So0P06o`03
0?oo01co0P0:o`<00_l002Go00<0ool0=ol00`3oo`0_o`030?oo03So00<0ool08Ol00`3oo`0]o`00
9_l00`3oo`0go`80;ol00`3oo`0ho`030?oo023o00<0ool0;_l002Oo00<0ool0>?l00`3oo`0/o`03
0?oo03Oo00<0ool08?l00`3oo`0_o`009ol00`3oo`0io`030?oo02_o00<0ool0=ol00`3oo`0Oo`03
0?oo033o000Xo`030?oo03Wo00<0ool0:_l00`3oo`0fo`030?oo01go0`0co`003?oo00`02Ol000co
00<0ool01Ol00`3oo`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Go00<0ool01_l00`3o00050003o`3o00Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`07o`003?l00`3oo`0Ko`803?l00`3oo`0Yo`030?oo00;o00<0ool09Ol00`3oo`0Yo`03
0?oo00Wo00<0ool04ol300Oo00<0ool0:_l00`3oo`07o`00;?l203co0P0Uo`030?oo03Co00<0ool0
3_l6047o000^o`030?oo03_o0P0So`030?oo03Co00<0ool02ol304Oo000_o`@0>ol00`3oo`0Po`03
0?oo03?o00<0ool02_l204[o000co`L0=Ol2023o00<0ool0<ol00`3oo`09o`030?oo04[o000jo`D0
<_l201ko00<0ool0<_l00`3oo`09o`030?oo04_o000oo`80<_l201co00<0ool0<Ol00`3oo`08o`80
C_l0047o00<0ool0<Ol201[o00<0ool0<Ol00`3oo`07o`030?oo04ko0012o`030?oo03;o1@0Eo`03
0?oo037o00<0ool01_l00`3oo`1?o`00@ol203Oo1@0@o`030?oo033o00<0ool01ol00`3oo`1?o`00
AOl00`3oo`0io`P02?l00`3oo`0^o`802Ol00`3oo`1@o`00A_l00`3oo`0@o`/09Ol?02So0P0:o`03
0?oo057o0017o`030?oo00co0`0;o`80:ol00`3oo`04o`<07ol600[o0P1Do`00B?l200Wo0`0@o`@0
9ol00`3oo`07o`D05ol300oo00<0ool0E?l004[o2@0Go`804Ol200Co0P03o`@01Ol00`3oo`0<o`D0
3ol3017o00<0ool0EOl006co0`0=o`040?oo00So00<0ool01_l00`3oo`0Ao`D01_l401;o0P1Ho`00
Kol600Oo00@0ool02Ol00`3oo`05o`805ol601Go00<0ool0F?l007Go2002o`030?oo00So00<0ool0
1?l00`3oo`0`o`030?oo05Wo001lo`@02?l0103oo`05o`030?oo02ko0P1Lo`00OOl400So0P06o`03
0?oo02co0P1No`00POl200ko00<0ool09ol5063o0023o`@02_l00`3oo`0Mo`X0IOl008Oo2@000ol0
o`0Ko`<0Kol0093o1P0Fo`80L_l0097o00<0ool00_l400oo0`1do`00TOl00`3oo`06o`H01?l507Oo
002Ao`030?oo00co101lo`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 483},
WindowMargins->{{Automatic, 38}, {Automatic, 47}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 140, 4, 375, "Title"],

Cell[CellGroupData[{
Cell[1904, 59, 32, 0, 53, "Section"],
Cell[1939, 61, 345, 9, 52, "Text"],
Cell[2287, 72, 320, 6, 52, "Text"],
Cell[2610, 80, 304, 6, 52, "Text"],
Cell[2917, 88, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3150, 97, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3332, 105, 48, 0, 45, "Subsection"],
Cell[3383, 107, 67, 1, 27, "Input"],
Cell[3453, 110, 103, 2, 27, "Input"],
Cell[3559, 114, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3724, 121, 46, 0, 29, "Subsection"],
Cell[3773, 123, 326, 7, 50, "Text"],
Cell[4102, 132, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4549, 144, 103, 2, 29, "Subsection"],
Cell[4655, 148, 105, 3, 32, "Text"],
Cell[4763, 153, 244, 6, 42, "Input"],
Cell[5010, 161, 123, 2, 27, "Input"],
Cell[5136, 165, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5419, 175, 78, 0, 42, "Subsubsection"],
Cell[5500, 177, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5670, 185, 60, 0, 29, "Subsection"],
Cell[5733, 187, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[5990, 195, 130, 3, 46, "Subsection"],
Cell[6123, 200, 121, 2, 27, "Input"],
Cell[6247, 204, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6428, 212, 31, 0, 33, "Section"],
Cell[6462, 214, 101, 3, 33, "Text"],
Cell[6566, 219, 202, 3, 30, "Input"],
Cell[6771, 224, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6962, 231, 44, 0, 33, "Section"],
Cell[7009, 233, 66, 0, 33, "Text"],
Cell[7078, 235, 1929, 30, 270, "Input"],
Cell[9010, 267, 1963, 32, 290, "Input"],
Cell[10976, 301, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11077, 307, 52, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11154, 311, 80, 1, 30, "Input"],
Cell[11237, 314, 14601, 644, 152, 7832, 556, "GraphicsData", "PostScript", \
"Graphics"],
Cell[25841, 960, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26020, 969, 35, 0, 53, "Section"],
Cell[26058, 971, 176, 4, 52, "Text"],
Cell[26237, 977, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[26723, 988, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[26890, 995, 17497, 825, 152, 10629, 734, "GraphicsData", "PostScript", \
"Graphics"],
Cell[44390, 1822, 16400, 755, 152, 9527, 664, "GraphicsData", "PostScript", \
"Graphics"],
Cell[60793, 2579, 15975, 729, 20, 9131, 639, "GraphicsData", "PostScript", \
"Graphics"],
Cell[76771, 3310, 15822, 720, 20, 8986, 630, "GraphicsData", "PostScript", \
"Graphics"],
Cell[92596, 4032, 15483, 698, 20, 8639, 608, "GraphicsData", "PostScript", \
"Graphics"],
Cell[108082, 4732, 15525, 703, 20, 8725, 613, "GraphicsData", "PostScript", \
"Graphics"],
Cell[123610, 5437, 14896, 662, 20, 8084, 572, "GraphicsData", "PostScript", \
"Graphics"],
Cell[138509, 6101, 14751, 651, 20, 7919, 561, "GraphicsData", "PostScript", \
"Graphics"],
Cell[153263, 6754, 14660, 646, 20, 7832, 556, "GraphicsData", "PostScript", \
"Graphics"],
Cell[167926, 7402, 14794, 653, 20, 7946, 563, "GraphicsData", "PostScript", \
"Graphics"],
Cell[182723, 8057, 15624, 705, 20, 8747, 614, "GraphicsData", "PostScript", \
"Graphics"],
Cell[198350, 8764, 15307, 685, 20, 8422, 594, "GraphicsData", "PostScript", \
"Graphics"],
Cell[213660, 9451, 15605, 702, 20, 8696, 611, "GraphicsData", "PostScript", \
"Graphics"],
Cell[229268, 10155, 15910, 718, 20, 8936, 626, "GraphicsData", "PostScript", \
"Graphics"],
Cell[245181, 10875, 16439, 748, 20, 9384, 655, "GraphicsData", "PostScript", \
"Graphics"],
Cell[261623, 11625, 16679, 761, 20, 9555, 667, "GraphicsData", "PostScript", \
"Graphics"],
Cell[278305, 12388, 17082, 783, 20, 9902, 689, "GraphicsData", "PostScript", \
"Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

