(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    295716,      13214]*)
(*NotebookOutlinePosition[    296371,      13237]*)
(*  CellTagsIndexPosition[    296327,      13233]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Proj.of Schizos, Ortho to Normal, for Major Axis Rescaled, \
Full Population, on MAD Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {1.00000000000000, 0, 0, \(-0.15195084000000\), 
          0.04858573000000, \(-0.06803209000000\), \(-0.23511130000000\), 
          0.01804597000000, 0.02809804000000, 
          0.03525404000000, \(-0.00142181000000\), \(-0.08276291000000\), 
          0.00474653000000, \(-0.07421567000000\), \(-0.03859208000000\), \
\(-0.01290857000000\), 0.02544342000000, 
          0.01821591000000, \(-0.00840569000000\), \(-0.04382198000000\), \
\(-0.01090547000000\), \(-0.03409100000000\), \(-0.01027874000000\), \
\(-0.01428957000000\), 0.02276759000000, \(-0.00005591000000\), 
          0.00019325000000, \(-0.00798167000000\), \(-0.00412871000000\), 
          0.00156547000000, \(-0.00104142000000\), \(-0.00605098000000\), 
          0.00905652000000, \(-0.00692849000000\), 0.00479774000000, 
          0.00158902000000, 0.00656396000000, 0.00799122000000, 
          0.00007162000000, \(-0.00306726000000\), \(-0.00452232000000\), 
          0.00027030000000, 0.00232349000000, 0.00137900000000, 
          0.00553573000000, 
          0.00250358000000, \(-0.00026729000000\), \(-0.00033366000000\), \
\(-0.00403778000000\), 0.00521687000000, 0.00053708000000, 0.00114458000000, 
          0.00011893000000, \(-0.00205813000000\), \(-0.00035760000000\), \
\(-0.00067585000000\), \(-0.00014223000000\), 0.00350017000000, 
          0.00016588000000, 
          0.00012998000000, \(-0.00211975000000\), \(-0.00131863000000\), \
\(-0.00063710000000\), \(-0.00066186000000\), 
          0.00083729000000, \(-0.00021953000000\), 
          0.00039799000000, \(-0.00076859000000\), \(-0.00063800000000\), 
          0.00023349000000, \(-0.00025169000000\), \(-0.00033436000000\), 
          0.00015018000000, \(-0.00069026000000\), 
          0.00001612000000, \(-0.00063123000000\), 0.00066879000000, 
          0.00041001000000, 
          0.00057962000000, \(-0.00041244000000\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ {0, 0.00000009266148, \(-0.00000015265116\), 
          0.11691881460231, 
          0.00928321000026, \(-0.07902395468564\), \(-0.78124870467190\), 
          0.00125217223040, \(-0.33825335520485\), \(-0.09957028767671\), \
\(-0.17000929046086\), 0.12047562866866, 0.04936912063404, 
          0.07089221335272, \(-0.11011254078206\), \(-0.01762331654893\), \
\(-0.23282311470207\), \(-0.00121690047905\), 0.08757863620946, 
          0.02422549032222, 
          0.08658063910944, \(-0.05295258649932\), \(-0.10860126855395\), 
          0.07344289701922, \(-0.12915226834418\), 
          0.07434128752165, \(-0.06882902168474\), 
          0.01497264414622, \(-0.03932031995813\), \(-0.04755723557901\), \
\(-0.02250598878021\), 0.04529052001421, 0.03115145082496, 0.05335567973998, 
          0.00338154455927, 
          0.04190981592936, \(-0.04668054149707\), \(-0.05967648632536\), \
\(-0.03720003941127\), 0.02066884254217, 
          0.07975753484741, \(-0.03698582164967\), 
          0.00792127560574, \(-0.00086366967172\), \(-0.02605585785496\), \
\(-0.01027905760747\), 0.01465151583211, \(-0.02845571246573\), 
          0.08789185203286, \(-0.06331701674459\), 
          0.02868976787249, \(-0.02947901292262\), 0.00206249868899, 
          0.02743266416634, \(-0.03141107215562\), 0.03482404382283, 
          0.05400018148477, \(-0.02791523967292\), 0.01500337911438, 
          0.02310318859811, 0.00568853208321, 
          0.06815216587503, \(-0.00802660618368\), 0.04172426758825, 
          0.02405036007188, \(-0.01771878021727\), 
          0.04867443107439, \(-0.02694887759469\), 0.01014555900045, 
          0.03471610780945, 0.02912428155757, 
          0.01337502354730, \(-0.03080811799367\), \(-0.00793598844358\), 
          0.00833855295049, \(-0.01913418313933\), \(-0.01219337026053\), 
          0.02270090094485, 
          0.00130574369607, \(-0.00463757775290\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the (coordinate wise) median", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87536 .15424 m
.87367 .14993 L
.87186 .14604 L
.86939 .14163 L
.86686 .13784 L
.86078 .13093 L
.85694 .12767 L
.85301 .12497 L
.84868 .12261 L
.84381 .12056 L
.83943 .11917 L
.83442 .11802 L
.83154 .11754 L
.82887 .11722 L
.82739 .11708 L
.82576 .11696 L
.82501 .11691 L
.82421 .11688 L
.82345 .11685 L
.82277 .11683 L
.82207 .11682 L
.82133 .11681 L
.82001 .11681 L
.81928 .11682 L
.81859 .11684 L
.81783 .11686 L
.81703 .1169 L
.81557 .11698 L
.8142 .11708 L
.81163 .11733 L
.80917 .11764 L
.80687 .118 L
.80175 .11906 L
.79692 .12041 L
.79196 .12222 L
.78786 .12413 L
.78383 .12645 L
.77748 .13141 L
.77439 .1347 L
.77191 .13797 L
.76958 .14178 L
.76743 .14622 L
.76576 .15059 L
.76444 .15481 L
.76231 .16348 L
.7603 .17329 L
.75795 .1837 L
.75657 .18857 L
.75503 .19303 L
.75146 .20093 L
Mistroke
.7467 .20838 L
.74134 .21466 L
.72648 .22712 L
.69258 .24738 L
.67337 .256 L
.65439 .26232 L
.61639 .27155 L
.59664 .27531 L
.58647 .27686 L
.57541 .27823 L
.55488 .28013 L
.53534 .28162 L
.51427 .28309 L
.50856 .28342 L
.50253 .2837 L
.4974 .28389 L
.49448 .28397 L
.49307 .284 L
.49176 .28402 L
.49049 .28404 L
.48934 .28405 L
.48806 .28406 L
.4867 .28406 L
.48538 .28406 L
.48464 .28406 L
.48396 .28406 L
.48271 .28405 L
.48139 .28404 L
.48026 .28402 L
.47905 .284 L
.47687 .28395 L
.47192 .2838 L
.46622 .28355 L
.46084 .28325 L
.45082 .28256 L
.43107 .28079 L
.41332 .27864 L
.40415 .27717 L
.39554 .2755 L
.37576 .27057 L
.33959 .25877 L
.32228 .25203 L
.30342 .2429 L
.2948 .23773 L
.28608 .23151 L
.27889 .22508 L
.2738 .2189 L
.27183 .21554 L
.27106 .21383 L
.27039 .21195 L
Mistroke
.26998 .21034 L
.26982 .20944 L
.26971 .20862 L
.26964 .20785 L
.26961 .20702 L
.26962 .20624 L
.26966 .20551 L
.26975 .2046 L
.26983 .20409 L
.26991 .20362 L
.27035 .20185 L
.27091 .20024 L
.27166 .19855 L
.27374 .19507 L
.27674 .1913 L
.29191 .17741 L
.29898 .17099 L
.30404 .1652 L
.30579 .16238 L
.30644 .16099 L
.30673 .16019 L
.30696 .15947 L
.30714 .15874 L
.30721 .15838 L
.30727 .15799 L
.30732 .15759 L
.30735 .15723 L
.30737 .15681 L
.30738 .15642 L
.30734 .15574 L
.3073 .15536 L
.30725 .15501 L
.30711 .15437 L
.30693 .15371 L
.30646 .15255 L
.30578 .15134 L
.30498 .15024 L
.3041 .14928 L
.30144 .14706 L
.29827 .14517 L
.29437 .14342 L
.29041 .14205 L
.2854 .14072 L
.28038 .1397 L
.2755 .13896 L
.27092 .13844 L
.26581 .13802 L
.26322 .13787 L
.26175 .13781 L
.26039 .13775 L
Mistroke
.25906 .13771 L
.2576 .13768 L
.2569 .13766 L
.25614 .13765 L
.25534 .13764 L
.25459 .13764 L
.25389 .13763 L
.25313 .13763 L
.25174 .13763 L
.2504 .13765 L
.24915 .13766 L
.24783 .13769 L
.24659 .13772 L
.24428 .13778 L
.24181 .13788 L
.23916 .138 L
.2336 .13833 L
.22782 .13877 L
.21763 .13974 L
.20815 .14087 L
.19805 .14238 L
.18857 .14417 L
.18001 .14622 L
.17052 .14916 L
.1621 .15257 L
.15345 .1572 L
.14521 .16325 L
.13903 .16951 L
.13606 .17341 L
.13367 .17723 L
.13172 .18105 L
.1302 .18476 L
.12893 .18877 L
.12838 .19095 L
.1279 .19329 L
.12751 .19576 L
.12726 .19806 L
.12715 .19935 L
.12708 .20056 L
.12705 .20126 L
.12703 .20189 L
.12702 .20263 L
.12701 .20333 L
.12702 .20449 L
.12706 .20572 L
.12712 .20689 L
.12718 .20795 L
.12729 .2092 L
.12743 .21055 L
.12776 .213 L
Mistroke
.12862 .21757 L
.12985 .22235 L
.13282 .2307 L
.13693 .23921 L
.14225 .24775 L
.15378 .26154 L
.16969 .27498 L
.18681 .28542 L
.22227 .30072 L
.26014 .31457 L
.29787 .3268 L
.33479 .33654 L
.3551 .3415 L
.37371 .34552 L
.39205 .34857 L
.4013 .34971 L
.41145 .35071 L
.43096 .35209 L
.44212 .35267 L
.45246 .35311 L
.45756 .35329 L
.463 .35344 L
.46768 .35354 L
.46896 .35356 L
.47034 .35358 L
.47155 .35359 L
.47284 .3536 L
.47355 .35361 L
.47433 .35361 L
.47571 .35362 L
.47648 .35362 L
.47732 .35362 L
.47811 .35362 L
.47883 .35362 L
.48016 .35361 L
.48091 .3536 L
.4816 .3536 L
.48302 .35358 L
.48453 .35356 L
.4874 .35351 L
.49011 .35344 L
.49518 .35327 L
.50011 .35304 L
.50551 .35273 L
.51633 .35187 L
.52622 .35082 L
.53539 .34963 L
.55591 .34613 L
.57436 .34189 L
.61239 .32983 L
Mistroke
.64825 .31761 L
.68853 .30708 L
.72747 .29956 L
.7471 .29616 L
.76825 .29207 L
.78673 .28771 L
.80657 .28174 L
.82481 .27424 L
.83435 .26907 L
.84282 .26343 L
.85091 .25679 L
.85783 .24979 L
.86843 .23522 L
.8726 .22716 L
.87603 .21837 L
.87818 .21074 L
.8791 .20633 L
.87978 .20215 L
.88029 .1979 L
.88049 .19546 L
.88064 .19318 L
.8807 .19193 L
.88074 .19075 L
.88076 .18968 L
.88078 .18852 L
.88078 .18792 L
.88078 .18725 L
.88078 .18655 L
.88077 .1859 L
.88075 .18473 L
.88073 .18407 L
.88071 .18347 L
.8806 .18121 L
.88046 .17911 L
.88026 .1768 L
.87999 .17433 L
.87932 .16972 L
.87853 .16555 L
.8772 .16007 L
.87536 .15424 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`<02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`804?l2033o001mo`80
2ol200Go00<0ool0A_l201Co00<0ool0;Ol0097o00<0ool0AOl00`3oo`0Eo`030?oo02co0017o`l0
>ol00`3oo`15o`030?oo01Ko00<0ool0:ol0043o1`0?o`@0=ol00`3oo`14o`030?oo01So00<0ool0
:_l003_o1@0Jo`80=Ol00`3oo`14o`030?oo01So00<0ool0:_l003Wo0P0Qo`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Ko0`0To`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`80
9ol00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Wo`030?oo037o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo0P0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Xo`030?oo03;o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02So00<0ool0
<ol00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool09ol00`3oo`0Po`801?l200?o1005o`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Wo`030?oo023o00@0ool02?l00`3oo`06o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`808ol0103oo`09o`030?oo00Go0P12o`03
0?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Co00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02?o00<0ool09_l200[o0P06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?ol00`3oo`0Po`030?oo02Oo000]o`03
0?oo02?o0P0lo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0So`030?oo03_o00<0ool0
?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Eo`802_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`09o`030?oo01So00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l200?o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`09o`030?oo01So00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00Wo0P0Io`030?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3o
o`0Ho`802_l00`3oo`0Ho`802_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0go`809ol00`3o
o`0Yo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0=Ol202So00<0ool0:_l0033o00<0ool09ol203Go
00<0ool0<ol202Wo00<0ool0:ol0037o00<0ool0:?l00`3oo`0bo`030?oo037o0P0[o`030?oo02_o
000ao`030?oo02Wo0P0bo`030?oo033o00<0ool0:_l00`3oo`0/o`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1ol000co00<0ool09?l00`3oo`02o`030?oo02Wo100Yo`030?oo02Wo0`0Yo`030?oo00;o00<0ool0
9Ol00`3oo`07o`00=?l00`3oo`0ao`<09_l00`3oo`0Uo`@0<?l00`3oo`0`o`00=Ol00`3oo`0co`<0
8ol00`3oo`0Qo`@0<_l203?o000fo`80=_l3023o00<0ool07Ol403?o0`0eo`00>?l00`3oo`0fo`<0
7Ol00`3oo`0Io`@0=Ol203So000io`030?oo03So1@0Ho`030?oo01?o1P0go`80>_l003[o0P0mo`P0
4?l00`3oo`04o`l0>_l303co000lo`<0@_lG04Oo0P0oo`00?ol2053o00<0ool0Aol4047o0011o`<0
COl00`3oo`11o`H0AOl004Co0P1;o`030?oo03co1@1;o`00A_l304So00<0ool0=ol5053o0019o`80
A_l00`3oo`0bo`D0EOl004_o0`13o`030?oo02ko101Jo`00C_l202go0P04o`800ol400Go00<0ool0
:_l405ko001@o`<0:Ol0103oo`08o`030?oo00Ko00<0ool09_l406;o001Co`<09_l0103oo`09o`03
0?oo00Go0P0To`<0I_l005Ko100Ro`040?oo00[o00<0ool01?l00`3oo`0Oo`@0JOl005[o0`0Oo`04
0?oo00So00@0ool01Ol00`3oo`0Lo`<0KOl005go0`0Mo`802_l200Ko00<0ool06Ol3073o001Po`@0
;Ol00`3oo`0Eo`@0Lol006Co100Yo`030?oo01;o0`1go`00J?l502Co00<0ool03ol307[o001]o`D0
7ol00`3oo`0=o`80OOl007;o0`0Lo`030?oo00Go201oo`00MOlA00Wo2P27o`00Q_l900;o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
1Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8892 .19068 m
.88606 .18064 L
.88205 .17083 L
.87723 .16204 L
.87193 .15509 L
.86839 .15159 L
.86462 .14857 L
.85618 .14357 L
.85055 .14114 L
.84475 .13914 L
.83771 .13724 L
.8336 .13635 L
.82954 .1356 L
.82607 .13507 L
.82218 .13457 L
.81857 .13421 L
.81517 .13394 L
.81307 .13382 L
.81108 .13373 L
.80929 .13368 L
.80735 .13364 L
.80634 .13364 L
.80523 .13364 L
.80407 .13365 L
.80298 .13366 L
.80105 .13372 L
.79898 .1338 L
.79704 .13392 L
.79526 .13405 L
.79127 .13444 L
.78782 .1349 L
.78426 .13551 L
.7784 .13684 L
.77246 .13873 L
.76683 .14125 L
.76432 .1427 L
.76186 .14441 L
.75997 .14598 L
.7582 .14777 L
.75563 .15122 L
.75446 .15344 L
.75395 .15465 L
.7535 .15596 L
.75318 .15712 L
.75289 .15841 L
.75269 .1596 L
.75255 .16073 L
.75246 .16192 L
.75241 .16306 L
.75241 .16409 L
Mistroke
.75244 .16522 L
.75251 .16645 L
.75257 .16715 L
.75263 .16777 L
.75294 .17016 L
.75332 .17241 L
.75382 .17488 L
.75496 .17984 L
.75601 .18438 L
.75679 .18841 L
.75697 .18955 L
.75713 .19079 L
.75725 .19193 L
.75732 .19298 L
.75737 .19408 L
.75738 .19469 L
.75738 .19527 L
.75736 .19586 L
.75733 .19641 L
.75729 .197 L
.75723 .19762 L
.7571 .19858 L
.75692 .19959 L
.75648 .20137 L
.75613 .20243 L
.75574 .20341 L
.75487 .2052 L
.75273 .20835 L
.75123 .21004 L
.74964 .21155 L
.74558 .21465 L
.74127 .21722 L
.73237 .22132 L
.72196 .22495 L
.70162 .2301 L
.69134 .23198 L
.68257 .23325 L
.6654 .2354 L
.65657 .23673 L
.64805 .23833 L
.63031 .24262 L
.62039 .24532 L
.61136 .24767 L
.60281 .24947 L
.60037 .24987 L
.59806 .25019 L
.59601 .25042 L
.59375 .25063 L
.59255 .25071 L
.59142 .25077 L
Mistroke
.59023 .25082 L
.58956 .25085 L
.58894 .25086 L
.58785 .25088 L
.58665 .25088 L
.58552 .25086 L
.58446 .25084 L
.58381 .25082 L
.58312 .25079 L
.58185 .25073 L
.57947 .25057 L
.57665 .25032 L
.57404 .25004 L
.56409 .24866 L
.55307 .247 L
.54765 .24631 L
.54497 .24602 L
.5425 .2458 L
.54031 .24564 L
.5379 .2455 L
.53665 .24545 L
.53548 .24542 L
.53424 .24539 L
.53292 .24539 L
.53171 .24539 L
.53043 .24542 L
.52922 .24546 L
.52812 .24551 L
.52684 .24558 L
.52545 .24568 L
.52294 .24591 L
.52034 .24623 L
.5176 .24665 L
.51274 .24759 L
.50423 .24979 L
.49493 .25275 L
.48419 .25632 L
.47366 .25944 L
.46387 .26172 L
.45822 .26273 L
.45294 .2635 L
.44735 .26412 L
.44211 .26456 L
.43744 .26485 L
.43246 .26507 L
.42989 .26515 L
.42711 .26522 L
.42423 .26527 L
.42159 .26531 L
.41914 .26534 L
Mistroke
.41694 .26535 L
.41573 .26536 L
.41442 .26537 L
.4132 .26537 L
.41208 .26538 L
.41092 .26538 L
.40982 .26538 L
.40867 .26538 L
.40742 .26538 L
.40634 .26538 L
.40519 .26538 L
.40411 .26538 L
.40314 .26537 L
.4019 .26537 L
.40076 .26536 L
.39952 .26535 L
.39821 .26533 L
.39602 .26529 L
.39362 .26524 L
.39113 .26516 L
.38887 .26507 L
.38625 .26493 L
.38376 .26477 L
.37931 .26437 L
.37441 .26376 L
.36985 .26301 L
.36552 .26214 L
.35544 .25945 L
.34549 .25595 L
.33486 .25137 L
.31683 .242 L
.29974 .23157 L
.29302 .22672 L
.28971 .22395 L
.28689 .22123 L
.28457 .21857 L
.28245 .21552 L
.28167 .21412 L
.28095 .21259 L
.28042 .21124 L
.27995 .20973 L
.27974 .20884 L
.27957 .20801 L
.27945 .20721 L
.27936 .20634 L
.2793 .20547 L
.27928 .20465 L
.2793 .20391 L
.27934 .20309 L
.27943 .20219 L
Mistroke
.2795 .20167 L
.27958 .20119 L
.28003 .19914 L
.28059 .19734 L
.28138 .1954 L
.28347 .19155 L
.28612 .1879 L
.29276 .18124 L
.30226 .17446 L
.31378 .16837 L
.33291 .16101 L
.3363 .15985 L
.3389 .15885 L
.33982 .15841 L
.3402 .15818 L
.34034 .15807 L
.34044 .15797 L
.34058 .15779 L
.34064 .15763 L
.34061 .15747 L
.34049 .1573 L
.3403 .15715 L
.34005 .15702 L
.3397 .15688 L
.33923 .15673 L
.33796 .15642 L
.33644 .15615 L
.33254 .15559 L
.32771 .15503 L
.29656 .15199 L
.27396 .15005 L
.26254 .14921 L
.25757 .14889 L
.2522 .14859 L
.24735 .14836 L
.24228 .14817 L
.23954 .1481 L
.23708 .14805 L
.23465 .14802 L
.2324 .148 L
.23 .14801 L
.22751 .14805 L
.22544 .1481 L
.2232 .14818 L
.22087 .14829 L
.21844 .14844 L
.21424 .1488 L
.21016 .14929 L
.20605 .14998 L
.20249 .15076 L
Mistroke
.19947 .15161 L
.19446 .1535 L
.19204 .15471 L
.19004 .15591 L
.18659 .15849 L
.18352 .16147 L
.17881 .16716 L
.1736 .17332 L
.17016 .17653 L
.16652 .17921 L
.16261 .18149 L
.15782 .18369 L
.1479 .18703 L
.12733 .19242 L
.12269 .19391 L
.11815 .19574 L
.11583 .19691 L
.11389 .19809 L
.11213 .19938 L
.11068 .20068 L
.10937 .20216 L
.10837 .2036 L
.10748 .20531 L
.10709 .2063 L
.10679 .20726 L
.10655 .20823 L
.10644 .20877 L
.10636 .20929 L
.10626 .21021 L
.10619 .21126 L
.10618 .21185 L
.10618 .21242 L
.10623 .21351 L
.10628 .21413 L
.10634 .21471 L
.10653 .21605 L
.10675 .21721 L
.10703 .21846 L
.10765 .22075 L
.10961 .22613 L
.11469 .23625 L
.12708 .25441 L
.13397 .26239 L
.14152 .26966 L
.15608 .27983 L
.16501 .28442 L
.17544 .28904 L
.19556 .29717 L
.2149 .30453 L
.23536 .31116 L
Mistroke
.25477 .31589 L
.27326 .31957 L
.30888 .32575 L
.34726 .33098 L
.3684 .3345 L
.3785 .33633 L
.38733 .33783 L
.39165 .33847 L
.39625 .33906 L
.40054 .33949 L
.40244 .33964 L
.40447 .33977 L
.40561 .33983 L
.40664 .33987 L
.40769 .33991 L
.40867 .33994 L
.40981 .33996 L
.41041 .33996 L
.41105 .33997 L
.41213 .33997 L
.41333 .33996 L
.41456 .33994 L
.41569 .33991 L
.41695 .33987 L
.41828 .33981 L
.42054 .33969 L
.42305 .33952 L
.42834 .33905 L
.43349 .33848 L
.44348 .33714 L
.46639 .33333 L
.50096 .32572 L
.5086 .32401 L
.51594 .32274 L
.51983 .32225 L
.52422 .32183 L
.52681 .32165 L
.52935 .3215 L
.53186 .32138 L
.53428 .3213 L
.53917 .32116 L
.54373 .32106 L
.5487 .32093 L
.55151 .32083 L
.5541 .32072 L
.55839 .32046 L
.56079 .32026 L
.56301 .32004 L
.56739 .31946 L
.5719 .31864 L
Mistroke
.57657 .31747 L
.58068 .31613 L
.58937 .31225 L
.59716 .3079 L
.60468 .30377 L
.61221 .30048 L
.61679 .29904 L
.6189 .29852 L
.62121 .29805 L
.62368 .29766 L
.62603 .29739 L
.62749 .29728 L
.62821 .29723 L
.62888 .2972 L
.63015 .29715 L
.63153 .29713 L
.63289 .29714 L
.63417 .29717 L
.63559 .29723 L
.63638 .29728 L
.63711 .29733 L
.64021 .29761 L
.64317 .29798 L
.64886 .29894 L
.65401 .30003 L
.66528 .30281 L
.67462 .30506 L
.67909 .30592 L
.68159 .3063 L
.68382 .30657 L
.68503 .30668 L
.68616 .30676 L
.68735 .30681 L
.68801 .30683 L
.68862 .30684 L
.68964 .30684 L
.69074 .30682 L
.69189 .30677 L
.69296 .3067 L
.69401 .3066 L
.69501 .30648 L
.69686 .30621 L
.69884 .30584 L
.70099 .30534 L
.7103 .30226 L
.71913 .29866 L
.72876 .295 L
.73361 .29353 L
.73811 .29246 L
.74223 .29175 L
Mistroke
.74431 .2915 L
.74548 .29139 L
.7466 .2913 L
.7478 .29123 L
.74889 .29118 L
.75007 .29116 L
.75073 .29115 L
.75135 .29115 L
.75236 .29116 L
.75349 .29118 L
.75466 .29123 L
.75576 .29129 L
.75806 .29146 L
.76019 .29167 L
.76501 .29227 L
.77475 .29371 L
.77913 .29427 L
.78131 .29449 L
.78333 .29465 L
.78447 .29472 L
.78568 .29478 L
.78691 .29482 L
.78761 .29483 L
.78826 .29484 L
.78941 .29484 L
.79046 .29483 L
.79168 .29479 L
.79285 .29474 L
.79413 .29466 L
.79553 .29455 L
.79808 .29429 L
.8006 .29394 L
.80295 .29354 L
.8084 .29238 L
.81429 .29079 L
.83553 .28316 L
.85716 .27444 L
.87306 .2681 L
.87652 .26654 L
.8799 .26478 L
.88252 .26312 L
.88509 .26107 L
.88746 .25857 L
.8896 .25544 L
.89125 .25204 L
.89194 .2502 L
.89251 .24838 L
.89346 .2444 L
.89386 .24212 L
.8942 .23957 L
Mistroke
.89434 .23823 L
.89447 .23674 L
.89457 .2353 L
.89464 .23396 L
.89469 .23272 L
.89472 .23133 L
.89474 .22987 L
.89473 .22848 L
.89471 .22715 L
.89468 .2257 L
.89462 .22431 L
.89456 .22301 L
.8944 .22053 L
.89418 .21785 L
.89355 .21223 L
.8926 .20597 L
.89114 .1985 L
.8892 .19068 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 1}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo04Wo2`0ho`00TOl00`3oo`15o`@02ol403Co002Ao`030?oo04?o0P0Co`<0<Ol0097o00<0ool0
@_l00`3oo`0Fo`80;ol004;o5P0io`030?oo047o00<0ool06Ol00`3oo`0/o`00?ol301Ko1`0bo`03
0?oo043o00<0ool06ol00`3oo`0[o`00?_l201oo200Zo`030?oo043o00<0ool07?l00`3oo`0Zo`00
??l202Go100Zo`030?oo043o00<0ool07Ol00`3oo`0Yo`00>ol00`3oo`0So`80;_l00`3oo`10o`03
0?oo01ko00<0ool0:?l003_o00<0ool08Ol2033o00<0ool0@?l00`3oo`0No`030?oo02So000jo`03
0?oo023o0P0bo`030?oo043o00<0ool07ol00`3oo`0Wo`00>Ol2027o00<0ool07_l200Co0P03o`@0
1Ol00`3oo`10o`030?oo01oo00<0ool09ol003So00<0ool07ol2023o00@0ool02?l00`3oo`06o`03
0?oo043o00<0ool08?l00`3oo`0Vo`00=Ol3027o00<0ool08?l0103oo`09o`030?oo00Go0P12o`03
0?oo01oo00<0ool09_l003;o0`0So`030?oo01_o1002o`040?oo00[o00<0ool01?l00`3oo`11o`03
0?oo01oo00<0ool09_l002oo0`0Uo`030?oo02;o00@0ool02?l0103oo`05o`030?oo047o00<0ool0
8?l00`3oo`0Uo`00;?l302Oo00<0ool09?l200[o0P06o`030?oo047o00<0ool08?l00`3oo`0Uo`00
:_l202[o00<0ool0>?l00`3oo`11o`030?oo027o00<0ool09?l002Wo00<0ool0:_l00`3oo`0ho`03
0?oo047o00<0ool08Ol00`3oo`0To`00:?l00`3oo`0[o`030?oo03So00<0ool0@?l202?o00<0ool0
9?l000Oo1@02o`801?l201;o00<0ool03_l501So00<0ool01ol200Co0P04o`808ol00`3oo`0Ro`80
1?l200Co0P0?o`804_l500go00<0ool04_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ao`03
0?oo013o00<0ool06?l00`3oo`06o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool0
3_l00`3oo`0Co`030?oo00go00<0ool05?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
3ol00`3oo`0@o`030?oo01Wo00<0ool01Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`03
0?oo00Wo0`0Fo`030?oo00go00<0ool05?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Bo`030?oo00So1004o`030?oo01Wo1P02o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200[o
0P0Io`030?oo00go00<0ool05?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Bo`030?oo00oo
00<0ool06ol00`3oo`03o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool01_l301_o
00<0ool03Ol00`3oo`0Do`030?oo00So00<0ool00_l000So0P0:o`<04_l00`3oo`0>o`807Ol200Go
0P0:o`<08_l00`3oo`0Ro`802_l300?o0`0Mo`803ol00`3oo`0Co`802_l300;o000Zo`030?oo02ko
00<0ool0<ol00`3oo`0_o`D0<Ol202Go000Zo`030?oo02oo0P0co`030?oo02Ko2@0fo`030?oo02Co
000[o`030?oo033o0P0ao`030?oo02;o100oo`030?oo02Co000[o`030?oo03;o0P0_o`030?oo01oo
0`13o`030?oo02Co000/o`030?oo03?o0P0]o`030?oo00d03Ol504Ko00<0ool09?l000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`03o`@01ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko00<0ool0
1_l00`3oo`07o`003?l00`3oo`0Oo`030?oo00Oo00<0ool0:Ol00`3oo`0202;o0`03o`030?oo02Wo
00<0ool0:Ol00`3oo`0<o`030?oo01_o00<0ool01ol002oo00<0ool0=ol601Ko1P06o`030?oo06;o
0P0Xo`00<?l00`3oo`0loaH03?l00`3oo`1Qo`80:Ol0037o00<0ool0GOl00`3oo`1No`<0:ol003;o
00<0ool0G?l00`3oo`1Ko`<0;_l003?o0P1Lo`030?oo05Wo0P0ao`00=Ol00`3oo`1Io`030?oo05Oo
0P0co`00=_l305So00<0ool0EOl203Go000io`80E_l00`3oo`1Bo`<0=ol003_o0P1Do`030?oo03co
2@09o`@0>_l003go0P1Bo`030?oo03So1009o`T0?_l003oo0`1?o`030?oo01Wo3P0?o`80E?l004;o
0P1=o`030?oo01Oo0P0>o`<02Ol305Ko0014o`@0BOl00`3oo`0Fo`030?oo017o2@1Io`00B?l4037o
0P04o`800ol400Go00<0ool05Ol00`3oo`1do`00C?l202ko00@0ool02?l00`3oo`06o`030?oo01?o
0P1go`00C_l402[o00@0ool02Ol00`3oo`05o`803ol507Wo001Bo`L08ol0103oo`0:o`030?oo00Co
00<0ool03P1no`00FOl801_o00@0ool02?l0103oo`03o`D0S?l0067o1@0Go`802_l600;o00<0ool0
S?l006Ko2@0Fo`H01_l00`3oo`2<o`00Kol500co1@0<o`030?oo08co001do``04Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00
O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`03
0?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o
003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88747 .18613 m
.88428 .17632 L
.88015 .16671 L
.87511 .15809 L
.86956 .15133 L
.86589 .14795 L
.862 .14506 L
.85344 .14037 L
.84783 .13815 L
.84209 .13635 L
.83521 .13469 L
.83123 .13393 L
.8273 .13332 L
.82396 .13289 L
.82022 .13251 L
.81856 .13237 L
.81677 .13224 L
.81508 .13215 L
.81352 .13208 L
.81154 .13201 L
.81049 .13199 L
.80938 .13197 L
.80751 .13197 L
.80645 .13197 L
.80546 .13199 L
.8036 .13204 L
.80192 .13211 L
.79998 .13221 L
.79815 .13234 L
.79484 .13264 L
.79138 .13307 L
.78537 .13411 L
.78216 .13484 L
.77927 .13563 L
.77312 .13778 L
.76804 .14025 L
.76332 .14344 L
.76131 .14522 L
.75943 .14725 L
.75803 .14909 L
.75676 .15118 L
.75573 .15336 L
.75499 .15544 L
.75445 .15748 L
.75425 .15853 L
.75406 .1597 L
.75394 .16076 L
.75385 .16174 L
.75379 .16288 L
.75377 .16397 L
Mistroke
.75377 .16497 L
.7538 .16591 L
.75385 .16695 L
.75393 .16805 L
.75404 .16926 L
.75419 .17059 L
.75456 .17327 L
.75536 .17804 L
.75626 .18316 L
.75668 .18592 L
.75685 .18725 L
.75698 .18847 L
.75708 .18961 L
.75712 .19019 L
.75716 .19081 L
.75718 .19143 L
.7572 .192 L
.75721 .19267 L
.7572 .19329 L
.75717 .19428 L
.75711 .19535 L
.757 .19645 L
.75686 .19745 L
.75667 .19848 L
.75646 .19943 L
.75595 .20117 L
.75525 .203 L
.75431 .2049 L
.75201 .20829 L
.74925 .21124 L
.7456 .21424 L
.74104 .21719 L
.73027 .22241 L
.7196 .22632 L
.69998 .23181 L
.68133 .23558 L
.64487 .24222 L
.62501 .24729 L
.6154 .24982 L
.60677 .25181 L
.60247 .25262 L
.59849 .25322 L
.59434 .25369 L
.59214 .25386 L
.5909 .25393 L
.58977 .25399 L
.58862 .25403 L
.5874 .25406 L
.58635 .25407 L
.58519 .25408 L
Mistroke
.58395 .25406 L
.58324 .25405 L
.5826 .25403 L
.58135 .25399 L
.58015 .25394 L
.57785 .25381 L
.5757 .25366 L
.57075 .2532 L
.56139 .2521 L
.55125 .25091 L
.5459 .25041 L
.54286 .25019 L
.5414 .25011 L
.54002 .25004 L
.53877 .24999 L
.53742 .24995 L
.53619 .24993 L
.53506 .24992 L
.53384 .24991 L
.53249 .24993 L
.53182 .24994 L
.53109 .24996 L
.52978 .25 L
.52838 .25007 L
.52708 .25014 L
.52415 .25038 L
.52131 .25068 L
.51831 .25108 L
.51291 .25202 L
.50807 .25308 L
.48812 .25865 L
.47867 .26126 L
.46838 .2636 L
.46288 .26459 L
.45677 .26547 L
.45118 .26609 L
.44594 .26652 L
.44314 .26669 L
.44056 .26682 L
.43773 .26694 L
.43475 .26703 L
.43339 .26706 L
.43192 .26709 L
.43064 .26711 L
.42929 .26713 L
.42808 .26714 L
.42675 .26715 L
.42556 .26716 L
.42445 .26716 L
.42327 .26716 L
Mistroke
.422 .26716 L
.4208 .26715 L
.41971 .26715 L
.41765 .26713 L
.41548 .26711 L
.41106 .26705 L
.40628 .26696 L
.40144 .26684 L
.39689 .26668 L
.39284 .26649 L
.38842 .26621 L
.38363 .26579 L
.37848 .26519 L
.37356 .26445 L
.36904 .2636 L
.3594 .26124 L
.3504 .25838 L
.33035 .24992 L
.31224 .24032 L
.29744 .23114 L
.2911 .22642 L
.2859 .22163 L
.28347 .21881 L
.28126 .21557 L
.28045 .21408 L
.27971 .21246 L
.27919 .21103 L
.27873 .20944 L
.27855 .20859 L
.27841 .20779 L
.2783 .20693 L
.27822 .20601 L
.27818 .20514 L
.27819 .20431 L
.27822 .20357 L
.27829 .20275 L
.27841 .20185 L
.27858 .20085 L
.27882 .1998 L
.2791 .19881 L
.27976 .19691 L
.28056 .19509 L
.28261 .19154 L
.28502 .18831 L
.29225 .18124 L
.30161 .17469 L
.31272 .16875 L
.33039 .16121 L
.33356 .1598 L
.33484 .15913 L
Mistroke
.33572 .15854 L
.33605 .15826 L
.33619 .15812 L
.3363 .15796 L
.3364 .1578 L
.33646 .15765 L
.33648 .1575 L
.33648 .15737 L
.3364 .15714 L
.33632 .15702 L
.33622 .1569 L
.33594 .15667 L
.33553 .15642 L
.33506 .15621 L
.33454 .15601 L
.33306 .15556 L
.3312 .15514 L
.32879 .15468 L
.31711 .15301 L
.29989 .15107 L
.27767 .14895 L
.26695 .1481 L
.25567 .14736 L
.25112 .14713 L
.24632 .14693 L
.24361 .14684 L
.24118 .14678 L
.23878 .14674 L
.23653 .14671 L
.23443 .14671 L
.2322 .14672 L
.23014 .14675 L
.22828 .1468 L
.22627 .14687 L
.22412 .14696 L
.22193 .14709 L
.21995 .14724 L
.21603 .14761 L
.21267 .14805 L
.20905 .14865 L
.20587 .14933 L
.20079 .1508 L
.19673 .15241 L
.19286 .15445 L
.18934 .15687 L
.17785 .1678 L
.17235 .17242 L
.16504 .17686 L
.16038 .17898 L
.15568 .18077 L
Mistroke
.1456 .1839 L
.13381 .18727 L
.12833 .18907 L
.12368 .19091 L
.11957 .19297 L
.11588 .19548 L
.11425 .19693 L
.11292 .19838 L
.11169 .20007 L
.11062 .20202 L
.10994 .20374 L
.10964 .20475 L
.10942 .20571 L
.10926 .20657 L
.10913 .20753 L
.10904 .20856 L
.10901 .20955 L
.10901 .21048 L
.10905 .21147 L
.10911 .21233 L
.10922 .21328 L
.10936 .21433 L
.10956 .21546 L
.11001 .21754 L
.11158 .22272 L
.11356 .22764 L
.11941 .23889 L
.12598 .24897 L
.1326 .2575 L
.14616 .27082 L
.15393 .27639 L
.16294 .28159 L
.20268 .29901 L
.22271 .3064 L
.24083 .31187 L
.27761 .32021 L
.31342 .32685 L
.35328 .33298 L
.37455 .33685 L
.38354 .33845 L
.38818 .3392 L
.39307 .3399 L
.3978 .34045 L
.40208 .34083 L
.40414 .34097 L
.40523 .34103 L
.40639 .34109 L
.4074 .34113 L
.40852 .34116 L
.40954 .34119 L
Mistroke
.41049 .34121 L
.41164 .34122 L
.41229 .34122 L
.4129 .34122 L
.41401 .34121 L
.4152 .3412 L
.41638 .34117 L
.4175 .34114 L
.4196 .34106 L
.42212 .34094 L
.42447 .34079 L
.42989 .34036 L
.44013 .33928 L
.44999 .33802 L
.4721 .33465 L
.50609 .32802 L
.51414 .32665 L
.51887 .32602 L
.52343 .32554 L
.52758 .3252 L
.53233 .3249 L
.54245 .32444 L
.54737 .32422 L
.55197 .32399 L
.55676 .32366 L
.56192 .32317 L
.56703 .32249 L
.57229 .32151 L
.57698 .32034 L
.58106 .31905 L
.58916 .3157 L
.59629 .31203 L
.60471 .30756 L
.60903 .30552 L
.61315 .30383 L
.61703 .30252 L
.62146 .30135 L
.62397 .30085 L
.62636 .30046 L
.62868 .30018 L
.63 .30006 L
.63121 .29997 L
.63256 .29989 L
.63325 .29986 L
.63401 .29984 L
.63536 .29981 L
.63662 .29981 L
.6374 .29982 L
.63813 .29983 L
.63951 .29988 L
Mistroke
.64029 .29992 L
.641 .29995 L
.64264 .30006 L
.64518 .30029 L
.648 .3006 L
.65371 .30143 L
.66489 .3035 L
.67078 .30464 L
.67609 .30555 L
.68049 .30615 L
.6829 .30639 L
.68395 .30648 L
.68505 .30655 L
.68626 .3066 L
.68738 .30664 L
.68856 .30665 L
.68924 .30665 L
.68986 .30664 L
.69099 .3066 L
.69201 .30655 L
.69318 .30647 L
.69428 .30637 L
.69647 .3061 L
.69843 .30579 L
.70279 .30485 L
.7106 .30246 L
.71997 .29896 L
.72884 .29577 L
.73767 .29328 L
.74217 .29238 L
.74473 .29199 L
.74711 .29171 L
.74835 .29159 L
.7497 .29149 L
.75097 .29141 L
.75214 .29136 L
.75319 .29133 L
.75435 .29131 L
.75556 .2913 L
.7567 .29131 L
.75733 .29133 L
.75802 .29134 L
.75927 .29139 L
.76048 .29145 L
.76161 .29151 L
.76688 .29192 L
.77166 .29239 L
.77685 .29288 L
.77924 .29308 L
.78056 .29317 L
Mistroke
.78179 .29324 L
.78293 .2933 L
.78398 .29334 L
.78514 .29337 L
.7864 .29339 L
.78752 .2934 L
.78873 .29339 L
.78976 .29337 L
.7909 .29333 L
.79213 .29327 L
.79346 .29318 L
.7959 .29297 L
.79825 .2927 L
.80076 .29233 L
.80531 .29147 L
.81566 .28869 L
.82704 .28462 L
.84723 .27602 L
.86612 .26738 L
.87408 .26332 L
.87779 .26107 L
.88087 .25883 L
.88348 .25646 L
.88557 .25405 L
.88758 .25102 L
.88938 .24729 L
.8906 .24377 L
.89115 .24176 L
.89165 .2395 L
.89206 .23721 L
.89236 .23505 L
.89261 .23278 L
.89279 .23058 L
.89286 .22938 L
.89292 .22808 L
.89296 .22683 L
.89298 .22567 L
.89299 .2243 L
.89299 .22352 L
.89298 .22279 L
.89296 .22145 L
.89291 .22 L
.89286 .21861 L
.89277 .2171 L
.89258 .21433 L
.89227 .211 L
.89191 .20791 L
.89013 .19714 L
.88895 .1918 L
.88747 .18613 L
Mistroke
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo04So3`0eo`00TOl00`3oo`15o`<03ol4037o002Ao`030?oo04Co00<0ool05?l202oo002Ao`03
0?oo04;o0P0Io`030?oo02co0012oaL0>?l00`3oo`11o`806ol00`3oo`0[o`00@?l201Oo2@0_o`03
0?oo047o00<0ool06ol00`3oo`0Zo`00?_l202;o100[o`030?oo043o00<0ool07?l00`3oo`0Zo`00
?Ol00`3oo`0Ro`@0:ol00`3oo`10o`030?oo01go00<0ool0:Ol003co00<0ool08Ol202oo00<0ool0
@?l00`3oo`0No`030?oo02So000ko`030?oo027o00<0ool0;ol00`3oo`10o`030?oo01oo00<0ool0
9ol003Wo0P0Ro`80<_l00`3oo`10o`030?oo01oo00<0ool09ol003So00<0ool08Ol00`3oo`0No`80
1?l200?o1005o`030?oo043o00<0ool08?l00`3oo`0Vo`00=_l202;o0P0Po`040?oo00So00<0ool0
1_l00`3oo`10o`030?oo023o00<0ool09_l0037o1@0So`030?oo023o00@0ool02Ol00`3oo`05o`80
@_l00`3oo`0Po`030?oo02Go000_o`809ol00`3oo`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0Po`030?oo02Go000/o`<0:?l00`3oo`0Ro`040?oo00So00@0ool01Ol00`3oo`11o`03
0?oo023o00<0ool09Ol002[o0P0Zo`030?oo02Co0P0:o`801_l00`3oo`11o`030?oo023o00<0ool0
9Ol002[o00<0ool0:Ol00`3oo`0ho`030?oo047o00<0ool08?l00`3oo`0Uo`00:Ol00`3oo`0Zo`03
0?oo03So00<0ool0@?l202;o00<0ool09Ol002Wo00<0ool0:Ol203[o00<0ool0@?l00`3oo`0Ro`03
0?oo02Co0007o`D00_l200Co0P0Bo`803ol501So00<0ool01ol200Co0P04o`808ol00`3oo`0Ro`80
1?l200Co0P0@o`030?oo013o1@0=o`030?oo01;o1@02o`801?l200?o0009o`030?oo00Oo00@0ool0
4_l00`3oo`0?o`030?oo01So00<0ool01_l0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`04
0?oo00go0P0Eo`030?oo00go00<0ool05?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
4?l00`3oo`0?o`030?oo01So00<0ool01_l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`03
0?oo00[o00<0ool05Ol00`3oo`0=o`030?oo01Co00<0ool02_l1007o0Ol00003o`0000801?l00`3o
o`08o`804ol00`3oo`07o`@01?l00`3oo`0Io`H00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo
0P0;o`806?l00`3oo`0=o`030?oo01Co00<0ool02?l200?o0009o`030?oo00So00<0ool04_l00`3o
o`0?o`030?oo01[o00<0ool01?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00So
0P0Jo`030?oo00go00<0ool05?l00`3oo`08o`030?oo00;o0008o`802_l301?o00<0ool03Ol201go
0P05o`802_l302;o00<0ool08_l200[o0`04o`@06ol200oo00<0ool04ol200[o0`02o`00:_l00`3o
o`0^o`80=?l00`3oo`0`o`D0<?l00`3oo`0To`00:ol00`3oo`0_o`80<_l00`3oo`0[o`D0=Ol00`3o
o`0To`00:ol00`3oo`0ao`030?oo02oo00<0ool09_l503[o00<0ool09?l002co00<0ool0<Ol202oo
00<0ool08Ol503ko0P0Vo`00;?l00`3oo`0co`030?oo02co00<0ool07Ol404?o00<0ool09Ol000co
o`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l400Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l400Co00<0ool01?l=00Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`002?l00`3oo`06o`030?oo00Ko00<0ool01ol000co
00<0ool08?l00`3oo`06o`030?oo02Wo00@0oooo0`0Ro`D0:ol00`3oo`0Yo`030?oo00[o0P0No`03
0?oo00Oo000`o`030?oo03So0P0Mo`<01?l00`3oo`1Qo`030?oo02So000ao`030?oo03Wo2`09o`T0
1ol00`3oo`1Oo`80:ol0037o00<0ool0A?l9013o00<0ool0GOl202go000bo`030?oo05co00<0ool0
Fol202oo000co`<0Fol00`3oo`1Ho`<0<Ol003Ko0P1Io`030?oo05Go0`0do`00>?l205Oo00<0ool0
D_l303Oo000jo`80EOl00`3oo`1?o`<0>_l003co0P1Co`030?oo03go2`03o`@0?Ol003ko0P1Ao`03
0?oo03So1@0;o`<0@Ol0043o0`1>o`030?oo01ko1`0Ao`80E?l004?o0`1;o`030?oo01_o0`07o`@0
2ol205Ko0016o`@0Aol00`3oo`0Go`@03_l;05So001:o`@0;ol200Co0P03o`@01Ol00`3oo`0Eo`80
MOl004ko0`0[o`040?oo00So00<0ool01_l00`3oo`0Co`80Mol0057o0`0Xo`040?oo00Wo00<0ool0
1Ol201?o00<0ool0Mol005Co1P0Ro`040?oo00[o00<0ool01?l00`3oo`0=o`D0N_l005[o200Jo`04
0?oo00So00@0ool01Ol00`3o000=07oo001Ro`D05_l200[o0P05o`<0SOl006Oo1@0Po`@000?o0?l0
SOl006co1@0Ao`X01Ol00`3oo`2<o`00LOlA00oo00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88574 .18157 m
.88251 .17199 L
.87825 .16258 L
.873 .15415 L
.8672 .14756 L
.86338 .14431 L
.85938 .14155 L
.8507 .13718 L
.8451 .13516 L
.83944 .13357 L
.83272 .13214 L
.82913 .13156 L
.82506 .13103 L
.82166 .13069 L
.81984 .13055 L
.81787 .13042 L
.81597 .13032 L
.81424 .13026 L
.81249 .13022 L
.81083 .1302 L
.80921 .1302 L
.80749 .13023 L
.80602 .13027 L
.80441 .13033 L
.80267 .13043 L
.80082 .13055 L
.79749 .13086 L
.7937 .13133 L
.78967 .13199 L
.78621 .13271 L
.78267 .13361 L
.7764 .13571 L
.77121 .13811 L
.76661 .141 L
.76308 .14401 L
.75997 .14769 L
.75872 .14967 L
.75758 .15193 L
.75677 .15401 L
.75618 .15594 L
.75592 .15703 L
.75567 .15823 L
.7555 .15928 L
.75535 .16044 L
.75524 .16152 L
.75516 .16254 L
.75511 .16362 L
.75508 .16479 L
.75509 .16588 L
.75511 .16692 L
Mistroke
.75514 .16786 L
.75519 .16888 L
.75528 .17018 L
.75538 .17138 L
.75565 .17408 L
.75623 .17894 L
.7565 .18118 L
.75675 .18356 L
.75686 .1847 L
.75695 .18577 L
.75701 .18673 L
.75707 .18775 L
.75711 .18885 L
.75712 .18986 L
.75712 .19093 L
.75711 .19154 L
.75708 .19211 L
.75706 .19268 L
.75702 .19327 L
.75692 .19433 L
.75678 .19545 L
.75659 .19666 L
.75609 .19886 L
.75536 .20114 L
.75448 .20321 L
.75351 .20503 L
.75113 .20843 L
.74796 .21178 L
.74078 .21716 L
.73147 .22214 L
.72031 .22674 L
.70071 .23305 L
.68199 .23768 L
.64474 .24548 L
.62443 .25066 L
.6156 .25292 L
.60602 .25506 L
.60168 .25584 L
.597 .25651 L
.59484 .25675 L
.59257 .25696 L
.59062 .2571 L
.58848 .25721 L
.58721 .25726 L
.586 .25729 L
.58492 .25731 L
.58374 .25732 L
.58312 .25732 L
.58245 .25732 L
.58174 .25731 L
Mistroke
.58107 .2573 L
.57988 .25728 L
.57859 .25724 L
.57607 .25713 L
.57366 .25699 L
.56907 .25666 L
.55841 .2557 L
.55308 .25523 L
.54815 .25486 L
.54582 .25472 L
.54364 .2546 L
.54137 .25452 L
.54017 .25448 L
.53889 .25445 L
.5376 .25443 L
.53621 .25443 L
.53489 .25443 L
.53368 .25445 L
.53259 .25447 L
.53139 .25451 L
.53014 .25456 L
.52897 .25461 L
.52646 .25477 L
.52384 .25499 L
.5192 .25551 L
.51425 .25626 L
.5089 .25726 L
.488 .2623 L
.47773 .26469 L
.4678 .26655 L
.46221 .26736 L
.45609 .26805 L
.45338 .26829 L
.45051 .2685 L
.44538 .2688 L
.44266 .26891 L
.4401 .26899 L
.43781 .26904 L
.43653 .26906 L
.43533 .26907 L
.43466 .26908 L
.43394 .26908 L
.43263 .26909 L
.43137 .26909 L
.43021 .26909 L
.42895 .26908 L
.42824 .26908 L
.42759 .26908 L
.42626 .26906 L
.42485 .26904 L
Mistroke
.42246 .26901 L
.4199 .26895 L
.41531 .26884 L
.40694 .26855 L
.40238 .26834 L
.39815 .26811 L
.39315 .26776 L
.38859 .26737 L
.37923 .26621 L
.374 .26533 L
.36896 .26431 L
.35975 .26197 L
.34984 .25877 L
.33226 .25154 L
.31442 .24251 L
.29725 .23218 L
.29023 .22698 L
.2868 .22396 L
.28394 .221 L
.28164 .21812 L
.27989 .21537 L
.2784 .21219 L
.27779 .21037 L
.27758 .20953 L
.27738 .20861 L
.27724 .20774 L
.27714 .20694 L
.27708 .20616 L
.27705 .20534 L
.27706 .20443 L
.2771 .20357 L
.2772 .20263 L
.27727 .2021 L
.27735 .20162 L
.27778 .19972 L
.27811 .19865 L
.27846 .19765 L
.28026 .19387 L
.28239 .19064 L
.28533 .18712 L
.29296 .1803 L
.30296 .17369 L
.31266 .16851 L
.32293 .16362 L
.32692 .16168 L
.32971 .16013 L
.33071 .15945 L
.33115 .15911 L
.33154 .15876 L
.33182 .15846 L
Mistroke
.33205 .15814 L
.33221 .15786 L
.33229 .1576 L
.33232 .15729 L
.33228 .157 L
.33217 .15675 L
.33198 .15648 L
.33169 .15619 L
.33129 .15589 L
.33031 .15537 L
.329 .15485 L
.32722 .1543 L
.32305 .15332 L
.31657 .15216 L
.30933 .15109 L
.29409 .14922 L
.27561 .14738 L
.26459 .1465 L
.25468 .1459 L
.25001 .14568 L
.24758 .14559 L
.24505 .14552 L
.24287 .14547 L
.2405 .14543 L
.23839 .14541 L
.23644 .14541 L
.2343 .14543 L
.23323 .14545 L
.23207 .14547 L
.23004 .14553 L
.22818 .14561 L
.22605 .14571 L
.22414 .14584 L
.22002 .14619 L
.21684 .14656 L
.21356 .14705 L
.21032 .14766 L
.20756 .14829 L
.20274 .14973 L
.19825 .15156 L
.19148 .15545 L
.17824 .1658 L
.17162 .17009 L
.16298 .17427 L
.15344 .17775 L
.14387 .18077 L
.13398 .18405 L
.12913 .18599 L
.12442 .18832 L
.1203 .191 L
Mistroke
.11715 .19384 L
.11575 .1955 L
.11463 .19716 L
.11378 .19873 L
.11303 .20052 L
.1125 .20223 L
.11227 .20323 L
.11209 .20419 L
.11195 .20522 L
.11186 .20618 L
.1118 .20731 L
.11179 .2084 L
.11182 .20934 L
.11188 .21037 L
.11196 .21125 L
.11207 .21223 L
.11243 .21447 L
.11289 .21661 L
.11421 .22122 L
.11614 .22637 L
.1212 .23678 L
.13252 .2539 L
.13947 .26194 L
.14614 .26824 L
.16279 .27971 L
.18029 .28849 L
.20057 .29723 L
.2194 .30442 L
.23967 .31081 L
.27738 .32004 L
.31448 .32752 L
.35186 .33381 L
.37183 .33758 L
.38206 .33945 L
.38754 .34035 L
.39262 .34108 L
.39749 .34165 L
.40003 .3419 L
.40272 .34211 L
.40499 .34226 L
.4062 .34232 L
.4075 .34238 L
.40869 .34242 L
.40978 .34245 L
.41088 .34248 L
.41193 .34249 L
.41297 .3425 L
.41413 .3425 L
.41535 .3425 L
.4165 .34249 L
Mistroke
.41757 .34247 L
.41856 .34245 L
.4208 .34238 L
.42322 .34228 L
.42589 .34214 L
.43085 .34181 L
.44172 .34085 L
.45183 .33976 L
.47397 .33687 L
.4934 .33373 L
.5094 .33098 L
.51753 .32984 L
.52553 .32899 L
.53439 .32831 L
.54463 .32768 L
.54968 .32735 L
.55519 .32692 L
.56035 .3264 L
.56504 .32578 L
.56916 .3251 L
.57358 .32418 L
.58209 .32174 L
.59047 .31838 L
.59787 .31478 L
.60658 .31042 L
.611 .30843 L
.61517 .30678 L
.61915 .30546 L
.62363 .30426 L
.62617 .30372 L
.62857 .3033 L
.63088 .30297 L
.6334 .3027 L
.63474 .30259 L
.63597 .30251 L
.63722 .30244 L
.63841 .3024 L
.6398 .30237 L
.6411 .30236 L
.64185 .30236 L
.64253 .30237 L
.64331 .30238 L
.64406 .3024 L
.64546 .30245 L
.64695 .30252 L
.64962 .30269 L
.65249 .30293 L
.65565 .30325 L
.66645 .30463 L
.67162 .30532 L
Mistroke
.67629 .30587 L
.67879 .30611 L
.6815 .30632 L
.6829 .3064 L
.68419 .30646 L
.6849 .30648 L
.68555 .3065 L
.68626 .30651 L
.687 .30651 L
.68813 .30651 L
.68933 .30649 L
.69045 .30645 L
.69147 .3064 L
.69266 .30632 L
.69394 .30621 L
.69625 .30596 L
.69839 .30565 L
.70067 .30524 L
.70477 .30433 L
.71408 .30155 L
.73346 .29513 L
.73812 .29392 L
.74311 .29287 L
.74599 .2924 L
.7486 .29205 L
.75122 .29177 L
.7524 .29167 L
.75369 .29158 L
.75482 .29152 L
.75603 .29146 L
.7567 .29143 L
.75731 .29142 L
.7585 .29139 L
.75958 .29137 L
.76076 .29137 L
.76187 .29137 L
.7629 .29139 L
.76408 .29141 L
.76518 .29144 L
.7677 .29153 L
.77278 .29176 L
.77747 .29197 L
.77876 .29202 L
.78014 .29207 L
.78081 .29208 L
.78144 .2921 L
.78262 .29212 L
.78377 .29213 L
.78499 .29213 L
.78603 .29212 L
Mistroke
.78718 .2921 L
.78842 .29206 L
.78975 .292 L
.79101 .29193 L
.79217 .29185 L
.79451 .29165 L
.79669 .29141 L
.80169 .29064 L
.80619 .2897 L
.81106 .28845 L
.82109 .28516 L
.83203 .28072 L
.85104 .27174 L
.86817 .26256 L
.87528 .25792 L
.87859 .25525 L
.88133 .25256 L
.88362 .24976 L
.88574 .24639 L
.88737 .24298 L
.88861 .23951 L
.88956 .23593 L
.89036 .23165 L
.89067 .22936 L
.89081 .22806 L
.89092 .22681 L
.89102 .22562 L
.8911 .22431 L
.89116 .2231 L
.8912 .22196 L
.89123 .22073 L
.89124 .21942 L
.89124 .21828 L
.89123 .21702 L
.8912 .21565 L
.89117 .21488 L
.89114 .21417 L
.891 .21146 L
.89081 .20892 L
.89055 .20613 L
.88992 .20101 L
.88912 .19603 L
.88808 .19077 L
.88574 .18157 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04_o2P0go`00
TOl00`3oo`18o`<02_l503;o002Ao`030?oo04Go0`0Bo`80<?l0097o00<0ool0@ol201Oo00<0ool0
;Ol004Go3`0mo`030?oo04;o00<0ool06?l00`3oo`0/o`00@Ol400oo2@0do`030?oo047o0P0Ko`03
0?oo02_o0010o`030?oo01[o1P0^o`030?oo047o00<0ool06ol00`3oo`0Zo`00?_l202?o0P0/o`03
0?oo047o00<0ool07?l00`3oo`0Yo`00?Ol00`3oo`0So`80;?l00`3oo`10o`807_l00`3oo`0Yo`00
??l00`3oo`0Qo`<0;_l00`3oo`10o`030?oo01ko00<0ool0:?l003_o00<0ool08Ol00`3oo`0_o`03
0?oo043o00<0ool07_l00`3oo`0Xo`00>Ol202;o0P0bo`030?oo043o00<0ool07ol00`3oo`0Wo`00
=_l302;o0P0Po`801?l200?o1005o`030?oo047o00<0ool07_l00`3oo`0Wo`00=?l202?o0P0Qo`04
0?oo00So00<0ool01_l00`3oo`11o`030?oo01oo00<0ool09_l002oo1@0To`030?oo027o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0Oo`030?oo02Ko000^o`030?oo02Ko00<0ool07?l400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07ol00`3oo`0Vo`00;?l202Wo00<0ool08_l0103oo`08o`04
0?oo00Go00<0ool0@Ol00`3oo`0Po`030?oo02Go000[o`030?oo02So00<0ool09?l200[o0P06o`03
0?oo047o00<0ool08?l00`3oo`0Uo`00:_l00`3oo`0Xo`80>_l00`3oo`11o`030?oo023o00<0ool0
9Ol002Wo0P0Zo`030?oo03Wo00<0ool0@?l00`3oo`0Qo`030?oo02Go000Yo`030?oo02Wo00<0ool0
>Ol00`3oo`10o`030?oo027o00<0ool09Ol000Oo1@02o`801?l201?o00<0ool03Ol501Oo00<0ool0
2?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0?o`030?oo017o1@0<o`030?oo01?o1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool04_l00`3oo`0?o`030?oo01So00<0ool01_l0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`08o`040?oo00go00<0ool05?l00`3oo`0=o`030?oo01Co00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo013o0P0@o`030?oo01So00<0ool01_l0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`0=o`030?oo01Co00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`805?l00`3oo`06o`@01?l00`3oo`0Ho`8000?o0000
0P02o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o0P0Ho`030?oo00go00<0ool05?l00`3o
o`08o`800ol000Wo00<0ool02?l00`3oo`0Co`030?oo00ko00<0ool06_l00`3oo`04o`040?oo00Wo
00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02?l201[o00<0ool03Ol00`3oo`0Do`030?oo00So
00<0ool00_l000So0P0:o`<04ol00`3oo`0=o`807Ol200Go0P0:o`<08_l00`3oo`0Ro`802_l300Co
100Ko`803_l201Go0P0:o`<00_l002_o00<0ool0;Ol00`3oo`0co`030?oo037o100_o`030?oo02Go
000/o`030?oo02go0P0co`030?oo02oo0P0co`030?oo02Go000]o`030?oo02ko00<0ool0<?l00`3o
o`0[o`@0=Ol00`3oo`0Uo`00;Ol00`3oo`0_o`80<?l00`3oo`0Vo`D0>Ol00`3oo`0Uo`00;_l00`3o
o`0`o`80;_l00`3oo`0Qo`D0?Ol00`3oo`0Vo`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l0103o0005o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00_l:00;o00<0ool0
1_l00`3o000300?o00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool0
8Ol00`3oo`05o`030?oo02Wo00@0oooo0`0Vo`D02_l=013o00<0ool0:Ol00`3oo`07o`808Ol00`3o
o`07o`00<Ol00`3oo`0go`808?l5063o00<0ool0:ol003;o00<0ool0>?l501Ko1@04o`030?oo05co
0P0^o`00<_l00`3oo`0moaH02Ol00`3oo`1Ko`030?oo02ko000co`80G?l00`3oo`1Io`80<Ol003Go
0P1Jo`030?oo05Oo0P0co`00=ol00`3oo`1Go`030?oo05Co0`0eo`00>?l205Oo00<0ool0DOl303So
000jo`80EOl00`3oo`1?o`80>ol003co0P1Co`030?oo03go4P0mo`00?_l2057o00<0ool0>_l304oo
0010o`@0COl00`3oo`0go`<0D_l004Co1019o`030?oo01go2`0;o`@0EOl004So0P17o`030?oo01_o
0P0;o`/0FOl004[o0`0`o`801?l200?o1005o`030?oo01So0`1ao`00COl302co00@0ool02?l00`3o
o`06o`030?oo01Ko0P1do`00D?l402So00@0ool02Ol00`3oo`05o`805Ol207Ko001Do`@09?l0103o
o`0:o`030?oo00Co00<0ool04Ol307So001Ho`D07ol0103oo`08o`040?oo00Go00<0ool02ol607_o
001Mo`L06Ol200[o0P06o`030?oo00/0POl006Co200Qo`L0S?l006co100Go`H01?l00`3oo`2<o`00
L?l800Ko2@0:o`030?oo08co001go`L04ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool0
2?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`03
0?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
0ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool0
1Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00
OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88401 .17702 m
.88073 .16767 L
.87635 .15846 L
.87383 .15429 L
.87088 .1502 L
.86483 .1438 L
.86091 .14069 L
.85619 .13772 L
.85126 .1353 L
.84626 .13337 L
.84139 .13189 L
.83562 .13054 L
.83257 .12997 L
.82919 .12945 L
.82588 .12904 L
.82282 .12875 L
.82123 .12863 L
.81949 .12852 L
.81853 .12847 L
.81764 .12843 L
.81588 .12837 L
.81436 .12833 L
.81268 .12832 L
.81092 .12833 L
.80928 .12836 L
.80835 .12839 L
.80733 .12843 L
.80548 .12852 L
.80368 .12864 L
.80202 .12877 L
.79827 .12917 L
.79435 .12973 L
.78799 .13102 L
.78234 .13262 L
.77663 .13483 L
.77117 .13774 L
.76693 .14083 L
.76365 .14406 L
.7608 .14798 L
.75954 .15032 L
.7586 .15253 L
.7578 .15493 L
.75724 .1572 L
.75679 .15976 L
.75662 .16106 L
.75655 .16181 L
.75649 .16251 L
.75644 .16313 L
.7564 .16379 L
.75635 .16497 L
Mistroke
.75633 .16564 L
.75632 .16635 L
.75632 .16764 L
.75632 .16826 L
.75634 .16894 L
.75637 .17018 L
.75642 .17138 L
.75647 .17248 L
.75675 .17709 L
.75691 .1796 L
.75705 .18195 L
.75709 .18301 L
.75713 .18412 L
.75716 .18532 L
.75718 .18643 L
.75717 .18742 L
.75715 .18833 L
.75712 .18932 L
.75706 .19039 L
.75697 .19155 L
.75686 .19261 L
.75671 .19377 L
.75651 .19499 L
.75597 .19742 L
.75532 .19957 L
.75458 .2015 L
.75362 .20352 L
.75144 .20703 L
.74865 .21038 L
.74556 .21332 L
.73706 .21931 L
.72742 .22432 L
.71651 .22893 L
.67849 .24086 L
.64195 .2494 L
.62404 .25397 L
.61397 .25646 L
.60466 .25841 L
.60026 .25915 L
.59563 .25978 L
.5932 .26004 L
.59052 .26027 L
.58808 .26043 L
.5858 .26053 L
.58453 .26058 L
.58385 .2606 L
.58314 .26061 L
.58252 .26062 L
.58183 .26063 L
.58061 .26064 L
Mistroke
.57943 .26064 L
.57815 .26063 L
.57693 .26061 L
.57582 .26058 L
.57315 .2605 L
.57021 .26037 L
.5648 .26008 L
.55419 .25943 L
.55164 .25929 L
.54893 .25916 L
.54642 .25906 L
.54407 .25899 L
.54278 .25896 L
.54139 .25893 L
.5406 .25893 L
.53986 .25892 L
.53846 .25892 L
.53773 .25892 L
.53695 .25892 L
.53556 .25894 L
.53479 .25896 L
.53407 .25897 L
.53242 .25903 L
.53114 .25908 L
.52979 .25914 L
.52739 .25929 L
.52203 .25975 L
.51689 .26036 L
.51227 .26106 L
.50201 .26297 L
.48353 .26682 L
.47394 .26852 L
.46862 .26928 L
.46359 .26988 L
.45813 .27038 L
.45497 .27061 L
.45209 .27078 L
.44964 .27089 L
.44697 .27099 L
.44551 .27103 L
.44416 .27106 L
.44288 .27109 L
.44151 .2711 L
.44019 .27112 L
.43945 .27112 L
.43875 .27112 L
.43802 .27113 L
.43736 .27113 L
.43659 .27112 L
.43587 .27112 L
Mistroke
.43459 .27111 L
.43341 .2711 L
.43212 .27108 L
.43076 .27106 L
.42796 .271 L
.42537 .27093 L
.41954 .27073 L
.41012 .27029 L
.40498 .26998 L
.40028 .26964 L
.39092 .26877 L
.38598 .26816 L
.38068 .26737 L
.37074 .26545 L
.36159 .26313 L
.34308 .25683 L
.32381 .24828 L
.3071 .23942 L
.29889 .23443 L
.2908 .22871 L
.28719 .22569 L
.28409 .22269 L
.28143 .21962 L
.27936 .21666 L
.2778 .21374 L
.27717 .21223 L
.27661 .21052 L
.2764 .20969 L
.27621 .2088 L
.27613 .20833 L
.27605 .20781 L
.27595 .2069 L
.2759 .20603 L
.27588 .2051 L
.27591 .20422 L
.27598 .20341 L
.27608 .20253 L
.27622 .20171 L
.27642 .2008 L
.27667 .19982 L
.27736 .19782 L
.27817 .19599 L
.28023 .19246 L
.28278 .18914 L
.29024 .18202 L
.30042 .17494 L
.31133 .16873 L
.32035 .16389 L
.32346 .16207 L
.32597 .16031 L
Mistroke
.32694 .15946 L
.32734 .15902 L
.32768 .15857 L
.32792 .15816 L
.32807 .1578 L
.32815 .15744 L
.32816 .15711 L
.32811 .15677 L
.32801 .15646 L
.32782 .15612 L
.32754 .15577 L
.32718 .15542 L
.32671 .15506 L
.32564 .15444 L
.32438 .15387 L
.32298 .15335 L
.31902 .1522 L
.31348 .15098 L
.30747 .14991 L
.29124 .14764 L
.28285 .14669 L
.27365 .1458 L
.26559 .14515 L
.257 .1446 L
.25226 .14438 L
.24801 .14423 L
.246 .14418 L
.24384 .14414 L
.24183 .14412 L
.23999 .14411 L
.23791 .14412 L
.23598 .14415 L
.23489 .14417 L
.23391 .1442 L
.23172 .14427 L
.22977 .14436 L
.22769 .14448 L
.22367 .1448 L
.22032 .14516 L
.21688 .14564 L
.21115 .14674 L
.20619 .14809 L
.20208 .14954 L
.19405 .15346 L
.1866 .15818 L
.1549 .17351 L
.14378 .17731 L
.13368 .18132 L
.1297 .18331 L
.12585 .18566 L
Mistroke
.1222 .18855 L
.1194 .19154 L
.11815 .1933 L
.11715 .19501 L
.11625 .19697 L
.11586 .198 L
.1155 .19914 L
.11524 .20016 L
.11503 .20111 L
.11486 .2021 L
.11473 .20302 L
.11462 .20412 L
.11458 .2047 L
.11455 .20533 L
.11453 .20639 L
.11455 .20758 L
.1146 .20864 L
.11469 .20978 L
.1148 .21076 L
.11494 .21184 L
.11537 .21429 L
.11588 .21661 L
.1171 .22089 L
.11876 .22551 L
.12346 .23572 L
.12966 .24618 L
.1356 .25437 L
.14262 .2623 L
.15739 .27449 L
.16647 .28009 L
.17557 .28495 L
.21273 .30113 L
.25178 .31347 L
.28956 .32286 L
.32492 .32997 L
.36517 .33752 L
.38366 .34098 L
.38859 .34176 L
.39373 .34246 L
.39815 .34296 L
.40287 .34337 L
.40548 .34354 L
.40787 .34366 L
.40902 .34371 L
.41028 .34375 L
.41147 .34378 L
.41258 .3438 L
.4132 .34382 L
.41388 .34382 L
.41462 .34383 L
Mistroke
.41531 .34383 L
.41602 .34384 L
.41681 .34383 L
.41755 .34383 L
.41824 .34383 L
.41893 .34382 L
.41958 .34381 L
.4208 .34379 L
.42213 .34375 L
.42357 .34371 L
.42875 .3435 L
.43433 .34319 L
.44458 .34243 L
.46554 .34044 L
.48389 .33824 L
.50068 .33582 L
.51831 .33341 L
.52841 .33236 L
.53827 .33154 L
.54867 .3307 L
.55381 .33022 L
.55851 .32971 L
.56738 .32841 L
.5719 .32752 L
.57652 .32641 L
.58527 .32368 L
.59432 .31995 L
.60329 .31571 L
.61197 .31177 L
.62023 .30871 L
.62514 .30732 L
.62983 .30629 L
.63207 .30589 L
.63451 .30553 L
.63664 .30528 L
.639 .30506 L
.64044 .30495 L
.64178 .30487 L
.64307 .3048 L
.64445 .30475 L
.64581 .30472 L
.6471 .3047 L
.64825 .3047 L
.6495 .30471 L
.65085 .30473 L
.6521 .30475 L
.65348 .3048 L
.65496 .30486 L
.66071 .30517 L
.67084 .30591 L
Mistroke
.67591 .30624 L
.67843 .30636 L
.67985 .30642 L
.68117 .30645 L
.68187 .30647 L
.6826 .30648 L
.68322 .30649 L
.6839 .3065 L
.68463 .3065 L
.68541 .30649 L
.68614 .30648 L
.6868 .30647 L
.68806 .30644 L
.68877 .30641 L
.68942 .30638 L
.6906 .30632 L
.69186 .30623 L
.69412 .30603 L
.69622 .30579 L
.70093 .30506 L
.70586 .30402 L
.71035 .30287 L
.72938 .29703 L
.7389 .29447 L
.74344 .29349 L
.74762 .29275 L
.75204 .29215 L
.75431 .2919 L
.7568 .29169 L
.75883 .29155 L
.76099 .29143 L
.76218 .29138 L
.76327 .29134 L
.76539 .29128 L
.76655 .29126 L
.76779 .29124 L
.76885 .29123 L
.77001 .29122 L
.77125 .29121 L
.77195 .29121 L
.77259 .29121 L
.77385 .29121 L
.77501 .2912 L
.77609 .2912 L
.77724 .29119 L
.7779 .29119 L
.77852 .29118 L
.77968 .29117 L
.78079 .29115 L
.78181 .29113 L
Mistroke
.78293 .29111 L
.78412 .29107 L
.78656 .29097 L
.78887 .29083 L
.79096 .29067 L
.79325 .29045 L
.79842 .28976 L
.80328 .28885 L
.81233 .28652 L
.82215 .28309 L
.84232 .27386 L
.85936 .26435 L
.8671 .25935 L
.87396 .2541 L
.87875 .24939 L
.88291 .24367 L
.88474 .24018 L
.88618 .23659 L
.88726 .2331 L
.88818 .22902 L
.88856 .22678 L
.88891 .22423 L
.88906 .2229 L
.88919 .22145 L
.88928 .22018 L
.88937 .21877 L
.8894 .21809 L
.88943 .21735 L
.88947 .21602 L
.88948 .2153 L
.88949 .21462 L
.8895 .21334 L
.88949 .21206 L
.88946 .21087 L
.88942 .20956 L
.88936 .20816 L
.88925 .20611 L
.8891 .20414 L
.88875 .20043 L
.88824 .19639 L
.88754 .19198 L
.88592 .1841 L
.88401 .17702 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo04Wo3P0eo`00
TOl00`3oo`16o`<03_l4037o002Ao`030?oo04Go00<0ool04ol202oo002Ao`030?oo04Co00<0ool0
5_l00`3oo`0/o`00AOlA03_o00<0ool0@ol00`3oo`0Ho`030?oo02_o0011o`@04Ol903;o00<0ool0
@_l00`3oo`0Jo`030?oo02[o000oo`807_l402ko00<0ool0@Ol00`3oo`0Ko`030?oo02[o000no`03
0?oo027o0P0]o`030?oo047o00<0ool07?l00`3oo`0Yo`00?Ol00`3oo`0Ro`80;Ol00`3oo`11o`03
0?oo01go00<0ool0:?l003_o0P0So`<0;_l00`3oo`11o`030?oo01go00<0ool0:?l003Wo0P0So`80
<Ol00`3oo`11o`030?oo01ko00<0ool09ol003Oo0P0So`80<ol00`3oo`11o`030?oo01ko00<0ool0
9ol003Co0`0To`030?oo01oo0P04o`800ol400Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000ao`<0
9_l00`3oo`0Oo`040?oo00So00<0ool01_l00`3oo`11o`030?oo01oo00<0ool09_l002ko0`0Xo`03
0?oo023o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0Oo`030?oo02Ko000]o`030?oo02So00<0ool0
6ol400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool07ol00`3oo`0Vo`00:ol202[o00<0ool0
8_l0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Po`030?oo02Go000[o`030?oo02So00<0ool0
9?l200[o0P06o`030?oo047o00<0ool08?l00`3oo`0Uo`00:_l00`3oo`0Xo`030?oo03Wo00<0ool0
@Ol00`3oo`0Po`030?oo02Go000Zo`030?oo02So00<0ool0>Ol00`3oo`10o`030?oo027o00<0ool0
9Ol002[o00<0ool0:?l00`3oo`0io`030?oo043o00<0ool08Ol00`3oo`0Uo`001ol500;o0P04o`80
5?l00`3oo`0<o`D05ol00`3oo`08o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool0
4Ol500co00<0ool04ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Co`030?oo00ko00<0ool0
5ol00`3oo`07o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`03
0?oo00co00<0ool05Ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool04Ol00`3oo`0>o`03
0?oo01So00<0ool01_l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool0
5Ol00`3oo`0<o`030?oo01Go00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805Ol00`3o
o`05o`@01?l00`3oo`0Ho`030?oo00@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`80
6?l00`3oo`0<o`030?oo01Go00<0ool02?l200?o0009o`030?oo00So00<0ool05?l00`3oo`0=o`03
0?oo01Wo00<0ool01Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Oo0`0Jo`03
0?oo00co00<0ool05Ol00`3oo`08o`030?oo00;o0008o`802_l301Co00<0ool03?l201co0P06o`80
2_l302;o00<0ool08_l200[o0`04o`<07?l200ko00<0ool05?l200[o0`02o`00;?l00`3oo`0[o`03
0?oo03Co00<0ool0<ol202oo00<0ool09Ol002co00<0ool0;?l203Co00<0ool0<?l3033o0P0Wo`00
;Ol00`3oo`0]o`030?oo037o00<0ool0;_l203?o00<0ool09_l002go00<0ool0;_l2037o00<0ool0
:_l403Co00<0ool09ol002ko00<0ool0;ol202oo00<0ool09Ol503So00<0ool09ol000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko0P07o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l300Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co
00<0ool08_l00`3oo`04o`030?oo02Wo00<0o`000`0Wo`030?oo00Co1`0<o`H03?l00`3oo`0Yo`03
0?oo00Ko00<0ool08Ol00`3oo`07o`00<Ol00`3oo`0fo`@08Ol900Oo3016o`030?oo02co000bo`03
0?oo03Wo0P0Lo`<00_l00`3oo`1Ko`80;ol003?o00<0ool0>_l;00[o1`05o`030?oo05Wo0P0ao`00
=?l00`3oo`14o`X03?l00`3oo`1Go`80<ol003Go0P1Jo`030?oo05Go0P0eo`00=ol205So00<0ool0
Dol203Oo000io`80E_l00`3oo`1Ao`80>Ol003_o0P1Do`030?oo04go100ko`00?Ol205;o00<0ool0
?_l?03oo000oo`80D?l00`3oo`0ko`<0C_l0047o0P1>o`030?oo03Oo101Ao`00@ol304_o00<0ool0
<ol405Go0016o`<0B?l00`3oo`0MoaH0FOl004Wo100`o`801?l200?o1005o`030?oo01Wo101_o`00
COl402_o00@0ool02?l00`3oo`06o`030?oo01So00<0ool0LOl0057o1P0Uo`040?oo00Wo00<0ool0
1Ol201So00<0ool0L_l005Oo1@0Po`040?oo00[o00<0ool01?l00`3oo`0Eo`80MOl005co100Lo`04
0?oo00So00@0ool01Ol00`3oo`0Ao`@0Mol0063o1@0Ho`802_l200Ko00<0ool02?l907_o001Uo`H0
9_l00`3o000808Co001[o`D06_l908go001`o`@04Ol500Oo00<0ool0S?l007Co4@0<o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o
00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go
00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co
00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88228 .17246 m
.87896 .16334 L
.87445 .15434 L
.87184 .15026 L
.86877 .14626 L
.86247 .14003 L
.8584 .13704 L
.85356 .13423 L
.84854 .13198 L
.84352 .13024 L
.83867 .12894 L
.83299 .1278 L
.83001 .12734 L
.82672 .12694 L
.8251 .12678 L
.82354 .12665 L
.82201 .12655 L
.82058 .12646 L
.81902 .1264 L
.8182 .12637 L
.81732 .12635 L
.81584 .12632 L
.81422 .12632 L
.81276 .12633 L
.81119 .12637 L
.80971 .12642 L
.80835 .12649 L
.80667 .12659 L
.80511 .12671 L
.8016 .12706 L
.79853 .12747 L
.7953 .12801 L
.78978 .12921 L
.78484 .13066 L
.77977 .1326 L
.77479 .13512 L
.77045 .13803 L
.76698 .1411 L
.76384 .14484 L
.76239 .14709 L
.76126 .14922 L
.76031 .15139 L
.75948 .15376 L
.75875 .15644 L
.75825 .15897 L
.75807 .16012 L
.7579 .16137 L
.75767 .16364 L
.75757 .16504 L
.7575 .16637 L
Mistroke
.75745 .16757 L
.75743 .16825 L
.75742 .16888 L
.7574 .17015 L
.7574 .17079 L
.75739 .17149 L
.7574 .17274 L
.7574 .17389 L
.75741 .1746 L
.75742 .17526 L
.75743 .1765 L
.75744 .17785 L
.75745 .17852 L
.75745 .17927 L
.75745 .18051 L
.75745 .18166 L
.75743 .18272 L
.75741 .18385 L
.75736 .1851 L
.75734 .18572 L
.7573 .18639 L
.75722 .18758 L
.75713 .18868 L
.75687 .1909 L
.75657 .19287 L
.75614 .19495 L
.75556 .19715 L
.75416 .20096 L
.75247 .20428 L
.75017 .20772 L
.74719 .21117 L
.74058 .21685 L
.73115 .22276 L
.72174 .22747 L
.70193 .23563 L
.68487 .24146 L
.65031 .25057 L
.63048 .25555 L
.61192 .26004 L
.60343 .26172 L
.59882 .26247 L
.59451 .26303 L
.58981 .26351 L
.58731 .26369 L
.58462 .26385 L
.58343 .2639 L
.58214 .26395 L
.58099 .26399 L
.57978 .26402 L
.57868 .26404 L
Mistroke
.57747 .26406 L
.57636 .26406 L
.57532 .26407 L
.57407 .26406 L
.57336 .26406 L
.57272 .26405 L
.57147 .26404 L
.57029 .26402 L
.56762 .26396 L
.56472 .26388 L
.55971 .26372 L
.55438 .26356 L
.55203 .26349 L
.5495 .26343 L
.54805 .2634 L
.54673 .26339 L
.5454 .26337 L
.54414 .26336 L
.543 .26336 L
.54179 .26336 L
.54076 .26337 L
.53962 .26338 L
.5384 .2634 L
.53709 .26342 L
.53472 .26349 L
.53218 .26359 L
.52949 .26373 L
.52466 .26408 L
.51897 .26464 L
.51385 .26528 L
.49248 .26878 L
.48301 .27034 L
.47284 .27171 L
.46747 .27226 L
.46461 .2725 L
.46152 .27273 L
.45869 .27289 L
.45609 .27302 L
.45348 .27312 L
.45101 .27319 L
.44969 .27322 L
.44827 .27324 L
.44693 .27325 L
.4457 .27326 L
.44459 .27327 L
.44337 .27327 L
.4421 .27326 L
.44092 .27325 L
.44026 .27324 L
.43954 .27324 L
Mistroke
.43823 .27322 L
.4358 .27317 L
.43298 .27309 L
.43039 .27301 L
.42145 .27261 L
.41225 .27205 L
.40333 .27133 L
.39388 .27033 L
.38497 .26905 L
.37675 .26753 L
.35862 .26287 L
.3389 .2558 L
.31964 .2471 L
.30333 .23832 L
.29571 .2335 L
.28813 .22784 L
.28499 .22506 L
.28201 .22199 L
.27962 .21904 L
.27773 .21615 L
.27689 .21455 L
.27611 .21279 L
.27554 .21116 L
.27514 .20964 L
.27497 .20878 L
.27485 .20798 L
.27475 .20709 L
.27472 .20658 L
.2747 .20612 L
.27469 .20524 L
.27471 .20474 L
.27473 .20427 L
.27482 .20333 L
.27489 .2028 L
.27497 .20231 L
.27536 .20059 L
.27586 .19899 L
.27656 .19723 L
.2775 .19537 L
.27948 .19226 L
.28222 .18889 L
.28995 .18181 L
.29842 .17582 L
.30863 .16957 L
.31684 .16457 L
.31986 .16246 L
.32202 .16062 L
.3229 .15965 L
.32327 .15916 L
.32357 .15864 L
Mistroke
.32377 .15822 L
.32392 .15776 L
.32399 .15736 L
.32401 .15698 L
.32396 .15654 L
.32389 .15629 L
.32381 .15607 L
.32361 .15566 L
.32332 .15524 L
.32262 .15452 L
.32157 .15375 L
.32013 .15295 L
.31821 .15212 L
.3138 .15068 L
.30854 .14938 L
.30282 .14824 L
.29516 .14697 L
.28757 .14592 L
.27969 .14499 L
.27089 .14415 L
.26602 .14376 L
.26154 .14345 L
.25697 .14319 L
.25286 .14302 L
.25101 .14295 L
.24902 .1429 L
.24715 .14286 L
.24544 .14283 L
.24366 .14282 L
.24171 .14282 L
.24074 .14283 L
.23971 .14284 L
.23786 .14287 L
.23626 .14291 L
.23454 .14297 L
.23117 .14314 L
.22816 .14335 L
.22545 .14359 L
.22212 .14396 L
.21913 .14439 L
.2139 .14536 L
.20878 .14663 L
.20375 .14827 L
.19612 .15151 L
.18834 .15549 L
.17138 .16361 L
.15106 .1712 L
.14124 .17511 L
.13293 .17933 L
.12928 .18174 L
Mistroke
.12586 .18457 L
.12268 .18805 L
.12142 .18983 L
.12041 .19154 L
.11958 .19322 L
.11882 .19513 L
.1182 .19712 L
.11793 .19826 L
.11772 .19931 L
.11756 .20025 L
.11743 .20129 L
.11732 .2024 L
.11726 .20347 L
.11722 .20448 L
.11722 .20557 L
.11724 .20651 L
.1173 .20755 L
.11738 .2087 L
.11749 .20978 L
.11784 .21226 L
.1183 .2147 L
.11892 .21734 L
.12196 .22657 L
.12692 .23724 L
.13231 .24631 L
.14438 .26137 L
.15206 .26842 L
.15977 .2742 L
.19516 .29308 L
.23112 .30675 L
.2693 .31782 L
.30575 .32681 L
.34528 .33471 L
.36585 .33888 L
.37573 .34082 L
.38451 .34238 L
.39299 .34362 L
.39741 .34413 L
.40209 .34457 L
.40448 .34475 L
.40712 .34491 L
.40956 .34503 L
.41179 .34512 L
.41289 .34515 L
.4141 .34518 L
.41538 .3452 L
.4166 .34522 L
.41774 .34523 L
.41895 .34523 L
.41999 .34523 L
Mistroke
.42115 .34523 L
.42243 .34522 L
.42362 .3452 L
.42494 .34518 L
.42635 .34515 L
.4314 .345 L
.43644 .3448 L
.44107 .34457 L
.46325 .34309 L
.48279 .34131 L
.5022 .33903 L
.51994 .33688 L
.54061 .33482 L
.55052 .33381 L
.56115 .33246 L
.57093 .33071 L
.57948 .32858 L
.58833 .32565 L
.5962 .32247 L
.6139 .3147 L
.62198 .31176 L
.6264 .31045 L
.6312 .30927 L
.6367 .30822 L
.63937 .30782 L
.64188 .30751 L
.64446 .30725 L
.64682 .30706 L
.64946 .30689 L
.65092 .30682 L
.65229 .30676 L
.65356 .30672 L
.65472 .30669 L
.65601 .30667 L
.6574 .30665 L
.65872 .30663 L
.65947 .30663 L
.66017 .30663 L
.66144 .30663 L
.6628 .30663 L
.66405 .30664 L
.66541 .30666 L
.66785 .30668 L
.6702 .30671 L
.67237 .30673 L
.67362 .30674 L
.67495 .30675 L
.6757 .30675 L
.67641 .30675 L
.67774 .30675 L
Mistroke
.67842 .30675 L
.67914 .30674 L
.68042 .30672 L
.68113 .30671 L
.6818 .30669 L
.6833 .30665 L
.68451 .30661 L
.68576 .30656 L
.688 .30643 L
.69039 .30626 L
.69297 .30602 L
.69798 .30539 L
.70265 .30461 L
.71131 .30268 L
.73088 .29723 L
.74017 .29489 L
.74529 .29382 L
.75009 .29298 L
.75452 .29234 L
.75857 .29187 L
.76304 .29146 L
.76783 .29112 L
.77662 .29064 L
.78143 .29036 L
.78596 .29003 L
.79137 .28948 L
.7963 .28879 L
.80097 .28794 L
.80607 .28678 L
.81561 .28392 L
.8262 .27977 L
.84695 .2691 L
.8635 .25821 L
.87089 .25208 L
.87645 .24621 L
.88088 .23977 L
.88275 .23607 L
.8844 .23186 L
.88555 .22794 L
.88641 .22403 L
.88681 .22169 L
.88711 .21947 L
.88739 .21684 L
.88749 .21561 L
.88758 .21426 L
.88765 .21293 L
.8877 .21169 L
.88773 .2105 L
.88775 .20922 L
.88775 .20806 L
Mistroke
.88774 .20697 L
.88772 .20576 L
.88768 .20446 L
.88763 .20306 L
.88754 .20153 L
.88734 .19871 L
.88708 .19594 L
.88677 .19336 L
.88609 .18866 L
.88512 .18348 L
.88251 .1732 L
.88228 .17246 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`/0=Ol007go0P0;o`801Ol00`3oo`18o`@0
2ol203?o002Ao`030?oo04Ko0P0Ao`<0<?l0097o00<0ool0A?l201Ko00<0ool0;Ol004_o0`13o`03
0?oo04Co00<0ool05_l202go0014o`P00_l<03Oo00<0ool0@ol00`3oo`0Io`030?oo02[o0012o`80
5_l503;o00<0ool0@_l00`3oo`0Jo`030?oo02[o0010o`807Ol302oo00<0ool0@_l00`3oo`0Jo`03
0?oo02[o000mo`<08Ol202ko00<0ool0@Ol00`3oo`0Lo`030?oo02Wo000ko`809?l202ko00<0ool0
@Ol00`3oo`0Lo`030?oo02Wo000ho`<09Ol202oo00<0ool0@Ol00`3oo`0Mo`030?oo02So000fo`80
9ol00`3oo`0_o`030?oo047o00<0ool07Ol00`3oo`0Xo`00<ol302Oo0P0bo`030?oo047o00<0ool0
7_l00`3oo`0Wo`00<Ol202So0P0Po`801?l200?o1005o`030?oo047o00<0ool07_l00`3oo`0Wo`00
;ol202Wo00<0ool07ol0103oo`08o`030?oo00Ko00<0ool0@Ol00`3oo`0No`030?oo02Oo000^o`03
0?oo02So00<0ool08?l0103oo`09o`030?oo00Go0P12o`030?oo01oo00<0ool09_l002go00<0ool0
:?l00`3oo`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000/o`03
0?oo02So00<0ool08_l0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000[o`03
0?oo02Oo0P0Vo`802_l200Ko00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000[o`030?oo02Oo00<0ool0
>Ol00`3oo`10o`030?oo023o0P0Wo`00:ol00`3oo`0Wo`030?oo03Wo00<0ool0@?l00`3oo`0Qo`03
0?oo02Go000[o`030?oo02Oo00<0ool0>Ol00`3oo`10o`030?oo027o00<0ool09Ol000Oo1@02o`80
1?l201Go00<0ool02ol501Oo00<0ool02?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0?o`03
0?oo017o1@0<o`030?oo01?o1@02o`801?l200?o0009o`030?oo00Oo00@0ool05?l00`3oo`0=o`03
0?oo01Oo00<0ool01ol0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00go00<0ool0
5?l00`3oo`0<o`030?oo01Go00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01;o00<0ool0
3Ol00`3oo`0Go`030?oo00Oo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`80
5ol00`3oo`0<o`030?oo01Go00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805Ol00`3o
o`05o`@01?l00`3oo`0Ho`030?oo00@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`03
0?oo01Oo00<0ool02ol00`3oo`0Fo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Go00<0ool0
3?l00`3oo`0Io`030?oo00Go00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`08o`80
6_l00`3oo`0;o`030?oo01Ko00<0ool02?l00`3oo`02o`002?l200[o0`0Eo`030?oo00_o0P0Lo`03
0?oo00Go0P0:o`<08_l00`3oo`0Ro`802_l300Go0`0Ko`803Ol00`3oo`0Eo`802_l300;o000/o`03
0?oo02[o00<0ool0=Ol00`3oo`0do`80;Ol00`3oo`0Vo`00;Ol00`3oo`0Zo`80=Ol00`3oo`0bo`80
;_l00`3oo`0Wo`00;Ol00`3oo`0/o`80<ol00`3oo`0`o`80<?l00`3oo`0Wo`00;_l00`3oo`0]o`80
<Ol00`3oo`0/o`@0<_l00`3oo`0Wo`00;ol00`3oo`0^o`80;ol00`3oo`0Wo`D0=Ol00`3oo`0Xo`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o0006o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`04o`@01ol00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08_l00`3oo`04o`030?oo02Wo00<0o`000P0Xo`030?oo01co0P0;o`03
0?oo02Wo00<0ool01Ol00`3oo`0Ro`030?oo00Oo000bo`030?oo03Co100To`030?oo01Ko1P0oo`80
;ol003?o00<0ool0=ol202;o6@14o`030?oo02oo000do`030?oo03So1`0Eo`L0F_l203;o000eo`80
?olE00Ko00<0ool0E_l203Co000go`80F?l00`3oo`1Do`80=_l003Wo0P1Fo`030?oo05;o0P0ho`00
>ol205Co00<0ool0Col303[o000mo`80D_l00`3oo`1:o`D0?Ol003oo0P1@o`030?oo03oo2`12o`00
@Ol204ko00<0ool0>ol404go0013o`<0Bol00`3oo`0ho`<0DOl004Ko0P19o`030?oo03Co101Do`00
B?l304Ko00<0ool08?lD05So001;o`<0;ol200Co0P03o`@01Ol00`3oo`0Lo`@0K?l004ko100Zo`04
0?oo00So00<0ool01_l00`3oo`0Jo`80L?l005;o100Vo`040?oo00Wo00<0ool01Ol201Wo0P1bo`00
E_l5027o00@0ool02_l00`3oo`04o`030?oo01Go0`1do`00Fol401go00@0ool02?l0103oo`05o`03
0?oo01?o0P1go`00Gol301_o0P0:o`801_l00`3oo`0>o`D0NOl006;o100[o`030?oo00Wo1@1no`00
I_l402Oo00<0ool02@23o`00J_l701Wo2P2<o`00LOlI00Oo00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co
001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo
00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go
00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88055 .16791 m
.87718 .15902 L
.87255 .15021 L
.86984 .14622 L
.86665 .14231 L
.8601 .13627 L
.8559 .1334 L
.85093 .13075 L
.84583 .12867 L
.84078 .1271 L
.83595 .12598 L
.83036 .12505 L
.82745 .12471 L
.82582 .12456 L
.82426 .12443 L
.8228 .12434 L
.82119 .12426 L
.81972 .12421 L
.81834 .12418 L
.8168 .12417 L
.81602 .12417 L
.81516 .12418 L
.81364 .12421 L
.81221 .12426 L
.81052 .12434 L
.80898 .12444 L
.80552 .12474 L
.80256 .1251 L
.79936 .12558 L
.79366 .12675 L
.78837 .12822 L
.78308 .13017 L
.77889 .13216 L
.77468 .13471 L
.77079 .13775 L
.76731 .14135 L
.76485 .14473 L
.76269 .14871 L
.7617 .15109 L
.76094 .15331 L
.76025 .15576 L
.75966 .15849 L
.75922 .16106 L
.75891 .16346 L
.75868 .16578 L
.75849 .16825 L
.75803 .17756 L
.75789 .18021 L
.75772 .18269 L
.75752 .1849 L
Mistroke
.75725 .18727 L
.75693 .18949 L
.75656 .19147 L
.75609 .19356 L
.75548 .19577 L
.75411 .19958 L
.75248 .20299 L
.75015 .20672 L
.74755 .21001 L
.74029 .2168 L
.73177 .22262 L
.72176 .22814 L
.7032 .23667 L
.68406 .24405 L
.64757 .25448 L
.62846 .25928 L
.61834 .26172 L
.60756 .26405 L
.60257 .26497 L
.59733 .2658 L
.59245 .26642 L
.58789 .26688 L
.58356 .26721 L
.58112 .26735 L
.57889 .26746 L
.57642 .26755 L
.57512 .26759 L
.57373 .26762 L
.57252 .26765 L
.57118 .26767 L
.56996 .26768 L
.56881 .26769 L
.56752 .2677 L
.56686 .26771 L
.56613 .26771 L
.56485 .26771 L
.56364 .26771 L
.56295 .26771 L
.56222 .26771 L
.56091 .26771 L
.55795 .2677 L
.55671 .26769 L
.55537 .26769 L
.55411 .26769 L
.55297 .26769 L
.55162 .26769 L
.55085 .26769 L
.55015 .26769 L
.54885 .2677 L
.54748 .26771 L
Mistroke
.54626 .26772 L
.54514 .26773 L
.54263 .26777 L
.5403 .26783 L
.53809 .2679 L
.53316 .2681 L
.53038 .26826 L
.52781 .26842 L
.51678 .26942 L
.50655 .27067 L
.4876 .27322 L
.47758 .27432 L
.4725 .27476 L
.46957 .27496 L
.46689 .27513 L
.46459 .27524 L
.46208 .27535 L
.45943 .27543 L
.45823 .27546 L
.45692 .27549 L
.45571 .27551 L
.45457 .27552 L
.45337 .27553 L
.45208 .27554 L
.45094 .27554 L
.44975 .27554 L
.44873 .27553 L
.44761 .27552 L
.44624 .2755 L
.44498 .27548 L
.44213 .27542 L
.43964 .27534 L
.43701 .27525 L
.42758 .2748 L
.41777 .27414 L
.40726 .2732 L
.39768 .27209 L
.38897 .27078 L
.37886 .26884 L
.36916 .26652 L
.35019 .26069 L
.33308 .25408 L
.31639 .24644 L
.2993 .23701 L
.29059 .23108 L
.2835 .22507 L
.27807 .21882 L
.27603 .2155 L
.27511 .21357 L
.27443 .21176 L
Mistroke
.27416 .21087 L
.27392 .20992 L
.27376 .20909 L
.27362 .2082 L
.27356 .2077 L
.27352 .20724 L
.27347 .20634 L
.27346 .20584 L
.27347 .20528 L
.27349 .20476 L
.27353 .20428 L
.27363 .20334 L
.27378 .20245 L
.27395 .20166 L
.27417 .2008 L
.27481 .19886 L
.27559 .19708 L
.27738 .19398 L
.27992 .1906 L
.28664 .18398 L
.3052 .17083 L
.31296 .16549 L
.31565 .16333 L
.31692 .16215 L
.31792 .16108 L
.31874 .16001 L
.31931 .15905 L
.31952 .15856 L
.31969 .15806 L
.3198 .15756 L
.31984 .15728 L
.31985 .15702 L
.31985 .15676 L
.31982 .15648 L
.31973 .15599 L
.31957 .15551 L
.31936 .15507 L
.31907 .15461 L
.31867 .15411 L
.31762 .15313 L
.31642 .15228 L
.31509 .15153 L
.31134 .14992 L
.30698 .14853 L
.30215 .14731 L
.28983 .145 L
.28213 .14392 L
.27486 .1431 L
.26792 .14247 L
.26427 .1422 L
Mistroke
.26043 .14196 L
.25709 .1418 L
.25534 .14172 L
.25347 .14166 L
.25176 .14161 L
.25022 .14157 L
.24869 .14154 L
.24725 .14153 L
.24562 .14152 L
.24389 .14153 L
.24227 .14154 L
.24081 .14157 L
.2392 .14161 L
.23747 .14166 L
.23408 .14182 L
.23083 .14204 L
.22749 .14233 L
.2218 .14303 L
.21651 .14393 L
.21187 .14494 L
.20395 .14723 L
.17156 .15997 L
.15283 .16701 L
.14307 .17135 L
.13809 .17408 L
.13377 .17695 L
.13012 .17995 L
.12673 .18355 L
.1242 .18714 L
.1232 .18896 L
.12237 .19073 L
.12111 .19433 L
.12062 .19632 L
.12041 .19736 L
.12022 .19852 L
.12007 .19968 L
.11996 .20077 L
.11989 .20193 L
.11986 .20259 L
.11984 .20319 L
.11983 .20427 L
.11985 .20546 L
.1199 .20658 L
.11997 .20763 L
.12008 .20884 L
.12021 .20998 L
.12053 .21216 L
.12096 .21452 L
.12155 .2171 L
.12436 .22607 L
Mistroke
.12893 .23637 L
.13408 .2453 L
.1403 .25395 L
.14677 .26122 L
.16045 .27287 L
.17745 .28343 L
.19451 .29187 L
.23177 .30623 L
.26754 .31721 L
.30574 .3273 L
.34318 .33533 L
.36278 .33949 L
.37304 .34157 L
.38378 .34352 L
.39319 .3449 L
.39817 .34547 L
.40354 .34597 L
.40595 .34615 L
.40849 .34631 L
.41086 .34643 L
.41307 .34653 L
.41534 .3466 L
.41741 .34666 L
.41852 .34668 L
.41972 .3467 L
.421 .34672 L
.42221 .34673 L
.42345 .34674 L
.42458 .34674 L
.42589 .34674 L
.42714 .34674 L
.42829 .34673 L
.42936 .34672 L
.43181 .34669 L
.43454 .34665 L
.43702 .34659 L
.44264 .34644 L
.4541 .34601 L
.46486 .34548 L
.48536 .34409 L
.50313 .34239 L
.52249 .34026 L
.54122 .33823 L
.55973 .3358 L
.56991 .33392 L
.57887 .33173 L
.59614 .32581 L
.61426 .31827 L
.62352 .31488 L
.63261 .31225 L
Mistroke
.64252 .31021 L
.64792 .30942 L
.65381 .30878 L
.65924 .30833 L
.66428 .30801 L
.67561 .3074 L
.68066 .3071 L
.68536 .30674 L
.6907 .30622 L
.69547 .30563 L
.70477 .30407 L
.71448 .30194 L
.73197 .29752 L
.74106 .29539 L
.75088 .29347 L
.76015 .29208 L
.76857 .29109 L
.77816 .29011 L
.78831 .28893 L
.797 .2875 L
.80199 .28644 L
.8067 .28524 L
.82475 .27878 L
.83483 .27389 L
.84377 .26881 L
.85971 .25772 L
.86728 .25106 L
.873 .24482 L
.87769 .23813 L
.88111 .23133 L
.88258 .22734 L
.8837 .22346 L
.88467 .21898 L
.88506 .21662 L
.88541 .21396 L
.88565 .21164 L
.88575 .2104 L
.88584 .20906 L
.88591 .20775 L
.88595 .20656 L
.88599 .20535 L
.886 .2042 L
.88601 .20306 L
.886 .20185 L
.88598 .20081 L
.88595 .19967 L
.88591 .19842 L
.88584 .19708 L
.88568 .19464 L
.88544 .19179 L
Mistroke
.88509 .18867 L
.88423 .18296 L
.88313 .17741 L
.88187 .17234 L
.88055 .16791 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o`d0=?l007go0P0;o`801Ol00`3oo`18o`<0
3Ol3037o002Ao`030?oo04Ko0P0Co`030?oo02ko002Ao`030?oo04Go00<0ool05?l202ko0018o`d0
??l00`3oo`14o`030?oo01Oo00<0ool0:ol004Co100=o`H0=_l00`3oo`13o`030?oo01Wo00<0ool0
:_l0047o0`0Go`@0<_l00`3oo`12o`030?oo01_o00<0ool0:Ol003ko0`0No`80<?l00`3oo`12o`03
0?oo01_o00<0ool0:Ol003_o0`0Ro`80;ol00`3oo`12o`030?oo01co00<0ool0:?l003So0`0Uo`80
;ol00`3oo`11o`807_l00`3oo`0Xo`00=_l202Oo0P0`o`030?oo047o00<0ool07Ol00`3oo`0Xo`00
=?l202So0P0ao`030?oo047o00<0ool07_l00`3oo`0Wo`00<_l202Wo00<0ool0<Ol00`3oo`11o`03
0?oo01ko00<0ool09ol0033o0P0Yo`808?l200Co0P03o`@01Ol00`3oo`11o`030?oo01ko00<0ool0
9ol002ko0P0Yo`808Ol0103oo`08o`030?oo00Ko00<0ool0@Ol00`3oo`0No`030?oo02Oo000]o`03
0?oo02So00<0ool08Ol0103oo`09o`030?oo00Go0P12o`030?oo01oo00<0ool09_l002go00<0ool0
9ol00`3oo`0Lo`@00_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000/o`03
0?oo02Oo00<0ool08ol0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000/o`03
0?oo02Ko00<0ool09Ol200[o0P06o`030?oo047o00<0ool07ol00`3oo`0Vo`00:ol202So00<0ool0
>Ol00`3oo`10o`030?oo023o00<0ool09_l002_o00<0ool09_l00`3oo`0jo`030?oo043o00<0ool0
8?l00`3oo`0Vo`00:ol00`3oo`0Vo`030?oo03[o00<0ool0?ol00`3oo`0Qo`030?oo02Ko0007o`D0
0_l200Co0P0Eo`030?oo00_o1@0Fo`030?oo00Wo0P04o`801?l202?o00<0ool08_l200Co0P04o`80
3_l00`3oo`0Bo`D02ol00`3oo`0Do`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Go00<0ool0
3?l00`3oo`0Fo`802Ol0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00co00<0ool0
5Ol00`3oo`0;o`030?oo01Ko00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01?o00<0ool0
3?l00`3oo`0Go`030?oo00Oo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`03
0?oo01Ko00<0ool02ol00`3oo`0Fo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Ko
00<0ool01?l400Co00<0ool05ol200;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o
00<0ool05ol00`3oo`0;o`030?oo01Ko00<0ool02?l200?o0009o`030?oo00So00<0ool05_l00`3o
o`0;o`030?oo01Wo00<0ool01Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00So
0P0Jo`030?oo00_o00<0ool05_l00`3oo`08o`030?oo00;o0008o`802_l301Ko00<0ool02_l201co
00<0ool01Ol200[o0`0Ro`030?oo02;o0P0:o`<01_l201_o0P0<o`030?oo01Ko0P0:o`<00_l002ko
00<0ool0:?l00`3oo`0eo`030?oo03Go0P0[o`030?oo02Oo000^o`030?oo02Wo00<0ool0=?l00`3o
o`0bo`<0;Ol00`3oo`0Wo`00;_l00`3oo`0Zo`80=?l00`3oo`0_o`<0;ol00`3oo`0Xo`00;ol00`3o
o`0[o`80<_l00`3oo`0/o`<0<_l00`3oo`0Xo`00;ol00`3oo`0]o`<0;ol00`3oo`0Yo`<0<ol202_o
000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`000`03o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`07o`003?l00`3oo`0So`030?oo00?o00<0ool0:Ol0103o000Yo`030?oo01ko
1007o`030?oo02Wo00<0ool01?l00`3oo`0So`030?oo00Oo000co`030?oo03;o100Uo`030?oo01_o
0`0lo`80<?l003Co00<0ool0=Ol4027o00<0ool05?l703ko00<0ool0<?l003Go00<0ool0>?l201ko
6013o`80<ol003Ko00<0ool0>Ol;00ko1@000ol0o`1Go`030?oo03?o000go`80A?l>00Ko00<0ool0
E?l203Ko000io`80E_l00`3oo`1Ao`<0>?l003_o00<0ool0Dol00`3oo`1>o`<0>ol003co0P1Co`03
0?oo04So1P0no`00?_l3053o00<0ool0@Ol704Co0011o`80C_l00`3oo`0lo`D0Bol004?o0`1;o`03
0?oo03So101@o`00A_l204Wo00<0ool0<_l605Co0018o`<0A_l00`3oo`0Xo`X0F_l004_o0`0_o`80
1?l200?o1005o`030?oo027o1`1To`00C_l402[o00@0ool02?l00`3oo`06o`030?oo01co1@1[o`00
D_l302Oo00@0ool02Ol00`3oo`05o`806ol2073o001Eo`<09?l0103oo`0:o`030?oo00Co00<0ool0
6?l207;o001Ho`@08?l0103oo`08o`040?oo00Go00<0ool05Ol307Co001Lo`@07Ol200[o0P06o`03
0?oo01?o0P1go`00H?l502co00<0ool03_l507Wo001Uo`D09ol00`3oo`05o`T0O_l006[o1`0Po`03
0?oo00D0Qol0077o1@0@o`h0S?l007Ko400;o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87882 .16335 m
.87541 .15469 L
.87066 .14609 L
.86785 .14218 L
.86453 .13837 L
.85774 .1325 L
.8534 .12976 L
.84829 .12726 L
.84311 .12535 L
.83803 .12397 L
.83323 .12302 L
.83042 .12261 L
.82773 .12231 L
.82627 .12218 L
.82486 .12208 L
.82358 .122 L
.8222 .12194 L
.82141 .12191 L
.8207 .12189 L
.8193 .12187 L
.81851 .12187 L
.81766 .12187 L
.81685 .12188 L
.8161 .1219 L
.81466 .12194 L
.81328 .122 L
.81204 .12207 L
.8107 .12217 L
.80769 .12245 L
.80491 .12279 L
.79992 .12362 L
.79474 .12481 L
.78937 .12645 L
.78462 .12835 L
.77981 .13084 L
.77516 .13399 L
.77159 .13717 L
.76871 .14044 L
.76608 .14437 L
.76383 .14891 L
.76291 .15132 L
.76205 .15401 L
.76141 .15644 L
.76084 .15906 L
.75999 .16414 L
.75942 .16891 L
.75848 .17825 L
.75785 .18339 L
.75708 .18796 L
.75613 .19207 L
Mistroke
.75478 .19638 L
.75316 .20027 L
.75135 .2037 L
.74878 .20757 L
.74553 .21149 L
.73847 .21798 L
.7226 .22838 L
.70475 .23753 L
.68572 .24582 L
.64947 .25722 L
.62864 .26245 L
.60972 .26672 L
.60038 .26844 L
.59544 .2692 L
.59018 .26987 L
.58555 .27036 L
.5804 .27079 L
.57567 .2711 L
.57118 .27132 L
.55182 .2719 L
.54643 .27206 L
.54067 .27228 L
.53523 .27254 L
.53032 .27284 L
.51945 .27368 L
.50933 .27467 L
.49878 .27577 L
.48767 .27682 L
.48294 .27719 L
.47785 .27752 L
.47493 .27767 L
.47225 .27778 L
.46956 .27787 L
.46835 .27791 L
.46701 .27794 L
.4659 .27796 L
.4647 .27798 L
.46344 .27799 L
.46225 .278 L
.46113 .27801 L
.45995 .27801 L
.45895 .27801 L
.45784 .278 L
.45654 .27799 L
.45535 .27797 L
.45398 .27795 L
.45269 .27793 L
.44791 .2778 L
.44317 .27761 L
.438 .27736 L
Mistroke
.42772 .27669 L
.4184 .27592 L
.40985 .27503 L
.40001 .27374 L
.3907 .2722 L
.37308 .26828 L
.35416 .26265 L
.33708 .25643 L
.32033 .24929 L
.30285 .2403 L
.29367 .23454 L
.28589 .22865 L
.27955 .2225 L
.27695 .21924 L
.27578 .21748 L
.27468 .21555 L
.27379 .21366 L
.27314 .2119 L
.27267 .2102 L
.2725 .20934 L
.27242 .20887 L
.27235 .20838 L
.27226 .20753 L
.27222 .20674 L
.27221 .20589 L
.27225 .20496 L
.27232 .20415 L
.27244 .20329 L
.27259 .20247 L
.27276 .20173 L
.27335 .19981 L
.27408 .19804 L
.27637 .19408 L
.27906 .19063 L
.28251 .18704 L
.29876 .17428 L
.30569 .16919 L
.31169 .16404 L
.31289 .16274 L
.31395 .16139 L
.31467 .16025 L
.31524 .15905 L
.31546 .15842 L
.31559 .15785 L
.31567 .15731 L
.31569 .15675 L
.31566 .15625 L
.31559 .1558 L
.31547 .1553 L
.31527 .15478 L
Mistroke
.31498 .15419 L
.31464 .15365 L
.31363 .15247 L
.31246 .15146 L
.31094 .15044 L
.30766 .14877 L
.30317 .14709 L
.29789 .14558 L
.29225 .14431 L
.28043 .14236 L
.27341 .14154 L
.26678 .14095 L
.26296 .14069 L
.25951 .14051 L
.25761 .14042 L
.25557 .14035 L
.25365 .14029 L
.2519 .14026 L
.25022 .14024 L
.24865 .14023 L
.24725 .14023 L
.24574 .14023 L
.24411 .14026 L
.24329 .14027 L
.2424 .14029 L
.24082 .14034 L
.23938 .14039 L
.23664 .14052 L
.23374 .1407 L
.2287 .14114 L
.22347 .14175 L
.21807 .14259 L
.20799 .14473 L
.19078 .1498 L
.17182 .15621 L
.1621 .15976 L
.15165 .16417 L
.14245 .16913 L
.13506 .17462 L
.132 .17762 L
.12914 .18111 L
.12689 .18466 L
.12522 .18812 L
.12445 .19016 L
.12376 .19239 L
.12316 .19491 L
.12276 .1973 L
.12261 .1985 L
.12255 .19914 L
.12249 .19981 L
Mistroke
.12245 .2004 L
.12241 .20105 L
.12237 .20221 L
.12235 .20334 L
.12237 .20456 L
.12241 .20572 L
.12247 .20678 L
.12255 .20793 L
.12266 .20901 L
.12297 .21149 L
.12338 .21391 L
.12392 .21652 L
.12657 .22553 L
.12856 .23061 L
.13096 .23581 L
.13594 .24461 L
.14142 .25241 L
.14793 .25995 L
.16127 .27175 L
.17917 .28324 L
.19764 .2924 L
.23378 .30622 L
.27224 .31858 L
.30981 .32882 L
.34659 .33717 L
.3673 .34168 L
.38597 .34513 L
.39514 .3464 L
.39998 .34694 L
.40514 .34741 L
.40966 .34773 L
.4147 .34801 L
.41932 .3482 L
.42372 .34833 L
.42615 .34838 L
.42844 .34841 L
.43095 .34844 L
.43237 .34845 L
.43366 .34846 L
.43432 .34847 L
.43502 .34847 L
.43629 .34847 L
.43764 .34847 L
.43842 .34847 L
.43915 .34847 L
.44046 .34847 L
.44166 .34846 L
.44305 .34845 L
.44438 .34844 L
.4467 .34842 L
Mistroke
.44919 .3484 L
.45367 .34833 L
.459 .34823 L
.46383 .34811 L
.46894 .34795 L
.47431 .34775 L
.48378 .34729 L
.49377 .34664 L
.50454 .34575 L
.52529 .34361 L
.54455 .34128 L
.55421 .33988 L
.56464 .33804 L
.58297 .33354 L
.59167 .33069 L
.60102 .3272 L
.61985 .31983 L
.62857 .31686 L
.63807 .31418 L
.64886 .31185 L
.65897 .31022 L
.67931 .30774 L
.69729 .30527 L
.71617 .3016 L
.73629 .29707 L
.77503 .29032 L
.79252 .28741 L
.80244 .28514 L
.81174 .28236 L
.83065 .27436 L
.83998 .26912 L
.84945 .2628 L
.85746 .25644 L
.86508 .24913 L
.87115 .24186 L
.87558 .23501 L
.87746 .23138 L
.87922 .22734 L
.88052 .22374 L
.88171 .21965 L
.88272 .21513 L
.88314 .21276 L
.88351 .21016 L
.8838 .20768 L
.884 .20537 L
.88407 .20424 L
.88414 .20301 L
.88419 .20184 L
.88423 .20075 L
.88425 .19962 L
Mistroke
.88426 .19854 L
.88427 .19757 L
.88426 .1965 L
.88424 .19533 L
.8842 .19408 L
.88415 .1929 L
.8841 .19182 L
.88396 .18969 L
.88377 .18738 L
.88329 .18314 L
.88251 .17808 L
.88143 .17273 L
.87882 .16335 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00M_l400;o00@0ool02?l0103oo`05o`03
0?oo04co300do`00O?l0103oo`09o`030?l000Go00<0ool0BOl300co0`0ao`00OOl200_o0P05o`03
0?oo04Oo0P0Bo`030?oo02ko002Ao`030?oo04Ko00<0ool04ol00`3oo`0]o`00TOl00`3oo`15o`03
0?oo01Go00<0ool0;?l004Go4`0io`030?oo04Co00<0ool05ol00`3oo`0[o`00@Ol401?o100eo`03
0?oo04?o00<0ool06Ol00`3oo`0Zo`00?Ol401_o0P0co`030?oo04?o00<0ool06Ol00`3oo`0Zo`00
>ol2027o0P0ao`030?oo04;o00<0ool06ol00`3oo`0Yo`00>Ol202Co00<0ool0;ol00`3oo`12o`03
0?oo01_o00<0ool0:Ol003Ko0`0Vo`030?oo02oo00<0ool0@_l00`3oo`0Ko`030?oo02Wo000co`<0
:?l2037o00<0ool0@_l00`3oo`0Lo`030?oo02So000ao`80:ol00`3oo`0`o`030?oo04;o00<0ool0
7?l00`3oo`0Xo`00<?l00`3oo`0Yo`80<ol00`3oo`12o`030?oo01go00<0ool09ol002oo00<0ool0
:Ol00`3oo`0Oo`801?l200?o1005o`030?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Wo`80
8Ol0103oo`08o`030?oo00Ko00<0ool0@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Oo00<0ool0
8Ol0103oo`09o`030?oo00Go0P12o`030?oo01ko00<0ool09ol002go00<0ool09ol00`3oo`0Lo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000]o`030?oo02Ko00<0ool0
8ol0103oo`08o`040?oo00Go00<0ool0@Ol00`3oo`0Oo`030?oo02Ko000/o`030?oo02Ko00<0ool0
9Ol200[o0P06o`030?oo047o00<0ool07ol00`3oo`0Vo`00;?l00`3oo`0Uo`030?oo03[o00<0ool0
@?l00`3oo`0Po`030?oo02Ko000/o`030?oo02Go00<0ool0>_l00`3oo`10o`030?oo023o00<0ool0
9_l002co00<0ool09Ol00`3oo`0jo`030?oo03oo00<0ool08Ol00`3oo`0Vo`001ol500;o0P04o`80
5_l00`3oo`0:o`D05_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool0
4Ol500_o00<0ool05?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Eo`030?oo00co00<0ool0
5_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`03
0?oo00[o00<0ool05ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool04ol00`3oo`0<o`03
0?oo01Oo00<0ool01ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo0P0Go`03
0?oo00[o00<0ool05ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Go`030?oo00?o
1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool0
5ol00`3oo`0:o`030?oo01Oo00<0ool02?l200?o0009o`030?oo00So00<0ool05_l00`3oo`0;o`03
0?oo01So0P07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02?l201[o00<0ool0
2_l00`3oo`0Go`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`807?l00`3oo`05o`80
2_l302;o00<0ool08_l200[o0`07o`030?oo01Wo0P0;o`030?oo01Oo0P0:o`<00_l002ko00<0ool0
:?l00`3oo`0eo`030?oo03Ko0P0Yo`030?oo02So000^o`030?oo02Wo0P0eo`030?oo03Co0P0Zo`03
0?oo02Wo000_o`030?oo02[o00<0ool0<_l00`3oo`0bo`80;?l00`3oo`0Yo`00;ol00`3oo`0[o`80
<_l00`3oo`0`o`80;Ol00`3oo`0Zo`00<?l00`3oo`0/o`80<?l00`3oo`0]o`<0;ol00`3oo`0Zo`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o0006o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o1008o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08ol00`3oo`03o`030?oo02Wo100Yo`030?oo02;o1003o`030?oo02Wo
00<0ool00ol202Go00<0ool01ol003?o00<0ool0<_l402Go00<0ool07Ol503Oo0P0ao`00=?l00`3o
o`0eo`808ol00`3oo`0Io`@0>ol00`3oo`0ao`00=Ol00`3oo`0fo`@07ol00`3oo`0Co`H0?Ol203Co
000fo`030?oo03Wo1P0Io`040?oooa80@Ol203Ko000go`80?olM057o0P0ho`00>Ol205Ko00<0ool0
D?l203[o000ko`80E?l00`3oo`1<o`@0??l003go0P1Bo`030?oo04So1010o`00?ol304oo00<0ool0
A?l404Co0012o`80COl00`3oo`0no`H0B?l004Co0`1:o`030?oo03Wo1@1>o`00Aol204So00<0ool0
=?l505?o0019o`<0AOl00`3oo`0Zo`X0F?l004co0`0^o`801?l200?o1005o`030?oo02?o1`1Ro`00
Col402Wo00@0ool02?l00`3oo`06o`030?oo023o0`1Yo`00Dol302Ko00@0ool02Ol00`3oo`05o`80
7ol206co001Fo`<08ol0103oo`0:o`030?oo00Co00<0ool06ol306ko001Io`@07ol0103oo`08o`04
0?oo00Go00<0ool06Ol2077o001Mo`@07?l200[o0P06o`030?oo01Co1@1co`00HOl502_o00<0ool0
4Ol307So001Vo`D09_l00`3oo`0=o`@0Nol006_o1@0Qo`030?oo00Ko1`1oo`00L?l401go2@26o`00
M?lN08ko002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87709 .15879 m
.87363 .15037 L
.86876 .14197 L
.86585 .13815 L
.86242 .13443 L
.85537 .12874 L
.85089 .12611 L
.84566 .12377 L
.8404 .12203 L
.83779 .12137 L
.83529 .12083 L
.83282 .12039 L
.83052 .12007 L
.82793 .11978 L
.82648 .11966 L
.8251 .11957 L
.82377 .1195 L
.82234 .11945 L
.82099 .11942 L
.81976 .11941 L
.819 .11942 L
.81831 .11943 L
.81754 .11944 L
.81673 .11947 L
.81537 .11953 L
.81387 .11961 L
.81112 .11983 L
.80814 .12017 L
.80535 .12058 L
.80279 .12103 L
.79803 .1221 L
.79316 .12354 L
.78821 .12544 L
.78387 .12756 L
.77949 .13028 L
.77529 .13366 L
.7721 .13698 L
.76952 .14037 L
.76713 .1444 L
.76504 .14903 L
.76331 .1542 L
.76204 .15931 L
.76028 .16921 L
.75879 .17887 L
.75783 .18404 L
.75674 .18875 L
.75413 .19665 L
.75233 .20058 L
.75005 .20461 L
.74432 .2121 L
Mistroke
.73654 .21938 L
.71972 .23081 L
.70075 .24118 L
.6837 .24911 L
.6473 .26103 L
.6267 .26611 L
.60779 .2702 L
.59825 .27191 L
.5893 .27321 L
.57983 .27427 L
.56923 .27513 L
.52802 .27757 L
.5072 .27919 L
.49637 .27997 L
.49047 .28031 L
.48758 .28044 L
.48493 .28055 L
.4826 .28063 L
.48004 .2807 L
.47747 .28075 L
.47607 .28077 L
.47477 .28078 L
.47407 .28079 L
.47331 .28079 L
.47259 .28079 L
.47193 .28079 L
.47067 .28079 L
.46935 .28079 L
.46866 .28078 L
.46792 .28077 L
.46659 .28076 L
.46501 .28073 L
.46356 .2807 L
.45872 .28056 L
.45346 .28035 L
.44398 .27983 L
.43457 .27915 L
.42438 .27825 L
.41382 .27709 L
.40425 .27577 L
.38589 .27228 L
.36593 .26698 L
.34662 .26064 L
.32939 .25411 L
.31225 .24643 L
.29636 .23752 L
.28841 .232 L
.28077 .2254 L
.2779 .22231 L
.27531 .21896 L
Mistroke
.27353 .21604 L
.2727 .21435 L
.27207 .21278 L
.27152 .21103 L
.27131 .2101 L
.27121 .20963 L
.27113 .20911 L
.27106 .20862 L
.27101 .20817 L
.27094 .20728 L
.27092 .20647 L
.27094 .20562 L
.271 .20484 L
.2711 .204 L
.27123 .20321 L
.27138 .2025 L
.27188 .20074 L
.27252 .19909 L
.27447 .19547 L
.2769 .19216 L
.28416 .18484 L
.30016 .17195 L
.30354 .16904 L
.3066 .16605 L
.30886 .16337 L
.30973 .16209 L
.31038 .16095 L
.31091 .15979 L
.3111 .15927 L
.31126 .15872 L
.31139 .15813 L
.31148 .1576 L
.31153 .15704 L
.31153 .15642 L
.31148 .15581 L
.31139 .15525 L
.31122 .15462 L
.31101 .15404 L
.3105 .15301 L
.30974 .15193 L
.3087 .15081 L
.30756 .14983 L
.30436 .14778 L
.30065 .14606 L
.29659 .1446 L
.29194 .14329 L
.28638 .14205 L
.28017 .14099 L
.27415 .14021 L
.26838 .13965 L
Mistroke
.2631 .13928 L
.26018 .13914 L
.25865 .13907 L
.25703 .13902 L
.25562 .13899 L
.25408 .13896 L
.25268 .13894 L
.25138 .13893 L
.24988 .13893 L
.24905 .13893 L
.2483 .13894 L
.24684 .13896 L
.24548 .13898 L
.2439 .13902 L
.24245 .13907 L
.23924 .13921 L
.23667 .13935 L
.23391 .13954 L
.22847 .14002 L
.21869 .14122 L
.21003 .14265 L
.191 .14686 L
.1735 .15191 L
.16476 .15504 L
.15572 .15898 L
.14649 .16421 L
.14243 .16713 L
.139 .17004 L
.1332 .17646 L
.13072 .18018 L
.12856 .18437 L
.12707 .18821 L
.12602 .19193 L
.12556 .19413 L
.12523 .19616 L
.12509 .1973 L
.12496 .19854 L
.12487 .19972 L
.12481 .20081 L
.12477 .20201 L
.12476 .20315 L
.12477 .20418 L
.1248 .20531 L
.12486 .20654 L
.1249 .20722 L
.12495 .20784 L
.12519 .21021 L
.12549 .21241 L
.1259 .21478 L
.12686 .21908 L
Mistroke
.12833 .22415 L
.13025 .22941 L
.13449 .23844 L
.13938 .24656 L
.14537 .25458 L
.15789 .26729 L
.17485 .27959 L
.19273 .28927 L
.22855 .30373 L
.26665 .31683 L
.30432 .32803 L
.34102 .337 L
.36148 .3417 L
.38015 .34543 L
.38961 .34694 L
.3984 .34807 L
.40761 .34896 L
.41779 .34966 L
.42281 .34992 L
.42831 .35014 L
.43321 .35031 L
.43844 .35045 L
.4432 .35055 L
.44847 .35064 L
.45079 .35068 L
.45329 .3507 L
.45566 .35072 L
.45782 .35074 L
.45906 .35074 L
.46022 .35075 L
.46149 .35075 L
.46287 .35075 L
.46424 .35075 L
.46493 .35074 L
.46568 .35074 L
.46701 .35073 L
.46825 .35073 L
.4707 .3507 L
.47292 .35067 L
.47549 .35063 L
.47791 .35058 L
.48333 .35042 L
.48831 .35023 L
.49375 .34997 L
.49947 .34962 L
.50909 .3489 L
.51935 .34792 L
.53784 .3457 L
.54806 .34419 L
.55917 .34225 L
Mistroke
.579 .33766 L
.59759 .33177 L
.61679 .32464 L
.63537 .31827 L
.65647 .3128 L
.69667 .30532 L
.73669 .29752 L
.77513 .29035 L
.79415 .28629 L
.81125 .28124 L
.82061 .27758 L
.82933 .27345 L
.83836 .26829 L
.84744 .262 L
.85477 .25588 L
.86198 .24861 L
.86732 .24202 L
.87203 .23481 L
.87603 .22684 L
.8788 .21925 L
.88 .2149 L
.88089 .21075 L
.88165 .20608 L
.88197 .20341 L
.8822 .20095 L
.88235 .19881 L
.88241 .19758 L
.88246 .19642 L
.8825 .19519 L
.88252 .19406 L
.88252 .19344 L
.88252 .19276 L
.88252 .19151 L
.8825 .19033 L
.88247 .18926 L
.88242 .18801 L
.88237 .18684 L
.88223 .18474 L
.88205 .18245 L
.88143 .17716 L
.88102 .17448 L
.88049 .17155 L
.87801 .16161 L
.87709 .15879 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1?o`L0=ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`D01_l503;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`804?l00`3oo`0_o`00
OOl200_o0P05o`030?oo04Oo00<0ool04Ol202oo002Ao`030?oo04Ko00<0ool05?l00`3oo`0/o`00
C?l5043o00<0ool0AOl00`3oo`0Fo`030?oo02_o0012o`X01Ol803So00<0ool0A?l00`3oo`0Ho`03
0?oo02[o0010o`805ol303Go00<0ool0A?l00`3oo`0Ho`030?oo02[o000mo`<07?l203?o00<0ool0
@ol00`3oo`0Jo`030?oo02Wo000io`@08Ol00`3oo`0`o`030?oo04?o00<0ool06_l00`3oo`0Yo`00
=ol202Go00<0ool0<?l00`3oo`13o`030?oo01_o00<0ool0:?l003Co0`0Wo`030?oo033o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`80:_l00`3oo`0`o`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Xo`80<ol00`3oo`12o`030?oo01go00<0ool09ol0033o00<0ool0:?l00`3oo`0co`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Xo`030?oo023o0P04o`800ol400Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02So00<0ool08?l0103oo`08o`030?oo00Ko00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Oo00<0ool08Ol0103oo`09o`030?oo00Go0P12o`03
0?oo01ko00<0ool09ol002ko00<0ool09_l00`3oo`0Lo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000]o`030?oo02Ko00<0ool08ol0103oo`08o`040?oo00Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000]o`030?oo02Go00<0ool09Ol200[o0P06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0@?l00`3oo`0Oo`030?oo02Oo000]o`03
0?oo02Co00<0ool0>_l00`3oo`10o`030?oo01oo00<0ool09ol002go00<0ool09?l00`3oo`0jo`03
0?oo03oo00<0ool08?l00`3oo`0Wo`001ol500;o0P04o`805ol00`3oo`09o`D05_l00`3oo`09o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool04_l500[o00<0ool05Ol500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05_l00`3oo`08o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`030?oo00[o00<0ool05ol00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Ko00<0ool02?l0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo00<0ool05_l00`3oo`0:o`030?oo01Oo00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`805ol00`3oo`03o`@01?l00`3oo`0Go`040?ooo`@0
0_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`030?oo01Oo00<0ool02Ol00`3oo`0Ho`03
0?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool02_l00`3oo`0Go`802?l0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`09o`030?oo01So00<0ool0
2?l00`3oo`02o`002?l200[o0`0Go`030?oo00Wo0P0Ko`030?oo00Ko0P0:o`<08_l00`3oo`0Ro`80
2_l300Oo0P0Jo`802ol00`3oo`0Go`802_l300;o000^o`030?oo02Oo00<0ool0=_l00`3oo`0go`03
0?oo02Ko00<0ool0:Ol002oo00<0ool09ol00`3oo`0eo`030?oo03Go0P0Xo`030?oo02[o000_o`03
0?oo02So0P0eo`030?oo03?o0P0Zo`030?oo02[o000`o`030?oo02Wo0P0co`030?oo037o0P0[o`03
0?oo02_o000ao`030?oo02[o0`0`o`030?oo02oo0P0]o`030?oo02_o000<ool03009o`003?l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`040?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l300Ko00<0ool01_l00`3oo`05o`030?oo00Ko
00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co
00<0ool09?l00`3oo`02o`030?oo02Wo0`0Zo`030?oo02Ko100[o`030?oo00;o0P0Vo`030?oo00Oo
000do`030?oo033o0`0Wo`030?oo02;o100co`030?oo033o000eo`030?oo03;o0P0Uo`030?oo01co
1P0fo`030?oo037o000fo`030?oo03?o100Qo`030?oo01Wo0`0jo`80=?l003Oo00<0ool0=_l501co
00<0ool04Ol803[o0`0fo`00>?l203_o1`0Eo`030?oo00;o3`10o`80>Ol003[o0P10oaX0C?l303_o
000lo`80Dol00`3oo`1<o`80?_l003ko0`1@o`030?oo04Wo0`10o`00@Ol204ko00<0ool0A?l504?o
0013o`80C?l00`3oo`0no`H0B?l004Go0P1:o`030?oo03Wo1@1>o`00Aol304Oo00<0ool0=?l505?o
001:o`80AOl00`3oo`0_o`D0F?l004co0`0^o`801?l200?o1005o`030?oo02Wo1P1Mo`00Col202_o
00@0ool02?l00`3oo`06o`030?oo02Co1@1So`00DOl302So00@0ool02Ol00`3oo`05o`808Ol406So
001Do`<09Ol0103oo`0:o`030?oo00Co00<0ool07_l206co001Go`@08Ol0103oo`08o`040?oo00Go
00<0ool06ol306ko001Ko`@07_l200[o0P06o`030?oo01So0`1ao`00Gol502go00<0ool05?l407Co
001To`D0:?l00`3oo`0Ao`<0N?l006Wo1P0Ro`030?oo00ko0`1ko`00Kol301oo00<0ool01Ol907ko
001bo`P05_l908Oo001joaH000?o0?l0SOl0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool0
2ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00
O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00
oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87536 .15424 m
.87367 .14993 L
.87186 .14604 L
.86939 .14163 L
.86686 .13784 L
.86078 .13093 L
.85694 .12767 L
.85301 .12497 L
.84868 .12261 L
.84381 .12056 L
.83943 .11917 L
.83442 .11802 L
.83154 .11754 L
.82887 .11722 L
.82739 .11708 L
.82576 .11696 L
.82501 .11691 L
.82421 .11688 L
.82345 .11685 L
.82277 .11683 L
.82207 .11682 L
.82133 .11681 L
.82001 .11681 L
.81928 .11682 L
.81859 .11684 L
.81783 .11686 L
.81703 .1169 L
.81557 .11698 L
.8142 .11708 L
.81163 .11733 L
.80917 .11764 L
.80687 .118 L
.80175 .11906 L
.79692 .12041 L
.79196 .12222 L
.78786 .12413 L
.78383 .12645 L
.77748 .13141 L
.77439 .1347 L
.77191 .13797 L
.76958 .14178 L
.76743 .14622 L
.76576 .15059 L
.76444 .15481 L
.76231 .16348 L
.7603 .17329 L
.75795 .1837 L
.75657 .18857 L
.75503 .19303 L
.75146 .20093 L
Mistroke
.7467 .20838 L
.74134 .21466 L
.72648 .22712 L
.69258 .24738 L
.67337 .256 L
.65439 .26232 L
.61639 .27155 L
.59664 .27531 L
.58647 .27686 L
.57541 .27823 L
.55488 .28013 L
.53534 .28162 L
.51427 .28309 L
.50856 .28342 L
.50253 .2837 L
.4974 .28389 L
.49448 .28397 L
.49307 .284 L
.49176 .28402 L
.49049 .28404 L
.48934 .28405 L
.48806 .28406 L
.4867 .28406 L
.48538 .28406 L
.48464 .28406 L
.48396 .28406 L
.48271 .28405 L
.48139 .28404 L
.48026 .28402 L
.47905 .284 L
.47687 .28395 L
.47192 .2838 L
.46622 .28355 L
.46084 .28325 L
.45082 .28256 L
.43107 .28079 L
.41332 .27864 L
.40415 .27717 L
.39554 .2755 L
.37576 .27057 L
.33959 .25877 L
.32228 .25203 L
.30342 .2429 L
.2948 .23773 L
.28608 .23151 L
.27889 .22508 L
.2738 .2189 L
.27183 .21554 L
.27106 .21383 L
.27039 .21195 L
Mistroke
.26998 .21034 L
.26982 .20944 L
.26971 .20862 L
.26964 .20785 L
.26961 .20702 L
.26962 .20624 L
.26966 .20551 L
.26975 .2046 L
.26983 .20409 L
.26991 .20362 L
.27035 .20185 L
.27091 .20024 L
.27166 .19855 L
.27374 .19507 L
.27674 .1913 L
.29191 .17741 L
.29898 .17099 L
.30404 .1652 L
.30579 .16238 L
.30644 .16099 L
.30673 .16019 L
.30696 .15947 L
.30714 .15874 L
.30721 .15838 L
.30727 .15799 L
.30732 .15759 L
.30735 .15723 L
.30737 .15681 L
.30738 .15642 L
.30734 .15574 L
.3073 .15536 L
.30725 .15501 L
.30711 .15437 L
.30693 .15371 L
.30646 .15255 L
.30578 .15134 L
.30498 .15024 L
.3041 .14928 L
.30144 .14706 L
.29827 .14517 L
.29437 .14342 L
.29041 .14205 L
.2854 .14072 L
.28038 .1397 L
.2755 .13896 L
.27092 .13844 L
.26581 .13802 L
.26322 .13787 L
.26175 .13781 L
.26039 .13775 L
Mistroke
.25906 .13771 L
.2576 .13768 L
.2569 .13766 L
.25614 .13765 L
.25534 .13764 L
.25459 .13764 L
.25389 .13763 L
.25313 .13763 L
.25174 .13763 L
.2504 .13765 L
.24915 .13766 L
.24783 .13769 L
.24659 .13772 L
.24428 .13778 L
.24181 .13788 L
.23916 .138 L
.2336 .13833 L
.22782 .13877 L
.21763 .13974 L
.20815 .14087 L
.19805 .14238 L
.18857 .14417 L
.18001 .14622 L
.17052 .14916 L
.1621 .15257 L
.15345 .1572 L
.14521 .16325 L
.13903 .16951 L
.13606 .17341 L
.13367 .17723 L
.13172 .18105 L
.1302 .18476 L
.12893 .18877 L
.12838 .19095 L
.1279 .19329 L
.12751 .19576 L
.12726 .19806 L
.12715 .19935 L
.12708 .20056 L
.12705 .20126 L
.12703 .20189 L
.12702 .20263 L
.12701 .20333 L
.12702 .20449 L
.12706 .20572 L
.12712 .20689 L
.12718 .20795 L
.12729 .2092 L
.12743 .21055 L
.12776 .213 L
Mistroke
.12862 .21757 L
.12985 .22235 L
.13282 .2307 L
.13693 .23921 L
.14225 .24775 L
.15378 .26154 L
.16969 .27498 L
.18681 .28542 L
.22227 .30072 L
.26014 .31457 L
.29787 .3268 L
.33479 .33654 L
.3551 .3415 L
.37371 .34552 L
.39205 .34857 L
.4013 .34971 L
.41145 .35071 L
.43096 .35209 L
.44212 .35267 L
.45246 .35311 L
.45756 .35329 L
.463 .35344 L
.46768 .35354 L
.46896 .35356 L
.47034 .35358 L
.47155 .35359 L
.47284 .3536 L
.47355 .35361 L
.47433 .35361 L
.47571 .35362 L
.47648 .35362 L
.47732 .35362 L
.47811 .35362 L
.47883 .35362 L
.48016 .35361 L
.48091 .3536 L
.4816 .3536 L
.48302 .35358 L
.48453 .35356 L
.4874 .35351 L
.49011 .35344 L
.49518 .35327 L
.50011 .35304 L
.50551 .35273 L
.51633 .35187 L
.52622 .35082 L
.53539 .34963 L
.55591 .34613 L
.57436 .34189 L
.61239 .32983 L
Mistroke
.64825 .31761 L
.68853 .30708 L
.72747 .29956 L
.7471 .29616 L
.76825 .29207 L
.78673 .28771 L
.80657 .28174 L
.82481 .27424 L
.83435 .26907 L
.84282 .26343 L
.85091 .25679 L
.85783 .24979 L
.86843 .23522 L
.8726 .22716 L
.87603 .21837 L
.87818 .21074 L
.8791 .20633 L
.87978 .20215 L
.88029 .1979 L
.88049 .19546 L
.88064 .19318 L
.8807 .19193 L
.88074 .19075 L
.88076 .18968 L
.88078 .18852 L
.88078 .18792 L
.88078 .18725 L
.88078 .18655 L
.88077 .1859 L
.88075 .18473 L
.88073 .18407 L
.88071 .18347 L
.8806 .18121 L
.88046 .17911 L
.88026 .1768 L
.87999 .17433 L
.87932 .16972 L
.87853 .16555 L
.8772 .16007 L
.87536 .15424 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1>o`/0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1:o`<02ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`804?l2033o001mo`80
2ol200Go00<0ool0A_l201Co00<0ool0;Ol0097o00<0ool0AOl00`3oo`0Eo`030?oo02co0017o`l0
>ol00`3oo`15o`030?oo01Ko00<0ool0:ol0043o1`0?o`@0=ol00`3oo`14o`030?oo01So00<0ool0
:_l003_o1@0Jo`80=Ol00`3oo`14o`030?oo01So00<0ool0:_l003Wo0P0Qo`80<ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Ko0`0To`030?oo037o00<0ool0@ol00`3oo`0Jo`030?oo02Wo000do`80
9ol00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Wo`030?oo037o00<0ool0
@_l00`3oo`0Lo`030?oo02So000bo`030?oo02Oo0P0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Xo`030?oo03;o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02So00<0ool0
<ol00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool09ol00`3oo`0Po`801?l200?o1005o`03
0?oo04;o00<0ool07Ol00`3oo`0Wo`00;ol00`3oo`0Wo`030?oo023o00@0ool02?l00`3oo`06o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;ol00`3oo`0Uo`808ol0103oo`09o`030?oo00Go0P12o`03
0?oo01ko00<0ool09ol002ko00<0ool09Ol00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02Co00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
@Ol00`3oo`0No`030?oo02Oo000^o`030?oo02?o00<0ool09_l200[o0P06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;Ol00`3oo`0To`030?oo03[o00<0ool0?ol00`3oo`0Po`030?oo02Oo000]o`03
0?oo02?o0P0lo`030?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0So`030?oo03_o00<0ool0
?_l00`3oo`0Po`030?oo02So0007o`D00_l200Co0P0Go`030?oo00Wo1@0Eo`030?oo00[o0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02Ol00`3oo`0Fo`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Ko00<0ool02ol00`3oo`0Eo`802_l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00co00<0ool05Ol00`3oo`09o`030?oo01So00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0:o`030?oo01Go00<0ool02Ol00`3oo`0Ho`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l200?o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3oo`09o`030?oo01So00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00Wo0P0Io`030?oo00Wo00<0ool06?l00`3oo`08o`030?oo00;o
0008o`802_l301So00<0ool02?l201_o00<0ool01_l200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3o
o`0Ho`802_l00`3oo`0Ho`802_l300;o000_o`030?oo02Ko00<0ool0=_l00`3oo`0go`809ol00`3o
o`0Yo`00<?l00`3oo`0Vo`030?oo03Go00<0ool0=Ol202So00<0ool0:_l0033o00<0ool09ol203Go
00<0ool0<ol202Wo00<0ool0:ol0037o00<0ool0:?l00`3oo`0bo`030?oo037o0P0[o`030?oo02_o
000ao`030?oo02Wo0P0bo`030?oo033o00<0ool0:_l00`3oo`0/o`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`03o`@02?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1ol000co00<0ool09?l00`3oo`02o`030?oo02Wo100Yo`030?oo02Wo0`0Yo`030?oo00;o00<0ool0
9Ol00`3oo`07o`00=?l00`3oo`0ao`<09_l00`3oo`0Uo`@0<?l00`3oo`0`o`00=Ol00`3oo`0co`<0
8ol00`3oo`0Qo`@0<_l203?o000fo`80=_l3023o00<0ool07Ol403?o0`0eo`00>?l00`3oo`0fo`<0
7Ol00`3oo`0Io`@0=Ol203So000io`030?oo03So1@0Ho`030?oo01?o1P0go`80>_l003[o0P0mo`P0
4?l00`3oo`04o`l0>_l303co000lo`<0@_lG04Oo0P0oo`00?ol2053o00<0ool0Aol4047o0011o`<0
COl00`3oo`11o`H0AOl004Co0P1;o`030?oo03co1@1;o`00A_l304So00<0ool0=ol5053o0019o`80
A_l00`3oo`0bo`D0EOl004_o0`13o`030?oo02ko101Jo`00C_l202go0P04o`800ol400Go00<0ool0
:_l405ko001@o`<0:Ol0103oo`08o`030?oo00Ko00<0ool09_l406;o001Co`<09_l0103oo`09o`03
0?oo00Go0P0To`<0I_l005Ko100Ro`040?oo00[o00<0ool01?l00`3oo`0Oo`@0JOl005[o0`0Oo`04
0?oo00So00@0ool01Ol00`3oo`0Lo`<0KOl005go0`0Mo`802_l200Ko00<0ool06Ol3073o001Po`@0
;Ol00`3oo`0Eo`@0Lol006Co100Yo`030?oo01;o0`1go`00J?l502Co00<0ool03ol307[o001]o`D0
7ol00`3oo`0=o`80OOl007;o0`0Lo`030?oo00Go201oo`00MOlA00Wo2P27o`00Q_l900;o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
1Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80
SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00
OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`03
0?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00
O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<0
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87363 .14968 m
.87192 .1455 L
.87008 .14172 L
.86756 .13741 L
.86496 .13372 L
.85868 .12697 L
.85471 .12381 L
.85064 .12121 L
.84619 .11896 L
.84119 .11705 L
.83674 .11579 L
.83415 .11524 L
.83168 .11482 L
.83032 .11462 L
.82882 .11445 L
.8275 .11432 L
.82608 .11421 L
.82536 .11417 L
.82468 .11414 L
.8234 .11409 L
.8222 .11407 L
.82091 .11406 L
.82018 .11407 L
.8195 .11409 L
.81876 .11411 L
.81796 .11414 L
.81634 .11424 L
.81486 .11436 L
.81204 .11466 L
.80939 .11504 L
.8049 .11592 L
.80012 .1172 L
.79536 .11888 L
.79116 .12077 L
.78438 .12491 L
.781 .12767 L
.77818 .13047 L
.77555 .13366 L
.77304 .13737 L
.76926 .1448 L
.76734 .14976 L
.76575 .15469 L
.76312 .16435 L
.75826 .18266 L
.75477 .19253 L
.75066 .20109 L
.7457 .20877 L
.73936 .21625 L
.72376 .22971 L
.70725 .24099 L
Mistroke
.68839 .25192 L
.66958 .26042 L
.65133 .26642 L
.6126 .27544 L
.59263 .27917 L
.57057 .28246 L
.54856 .28496 L
.53794 .28595 L
.52803 .28674 L
.51831 .28735 L
.51289 .2876 L
.5105 .28768 L
.50798 .28776 L
.50665 .28779 L
.50518 .28782 L
.50384 .28784 L
.50261 .28786 L
.5019 .28787 L
.50116 .28787 L
.50042 .28788 L
.49961 .28788 L
.49891 .28788 L
.49815 .28788 L
.4968 .28787 L
.49544 .28786 L
.49467 .28785 L
.49397 .28784 L
.4913 .28779 L
.48888 .28772 L
.48632 .28763 L
.48119 .2874 L
.47653 .28713 L
.46613 .28634 L
.44768 .28456 L
.42749 .28226 L
.41877 .28107 L
.40917 .27952 L
.39136 .2757 L
.37229 .27031 L
.33526 .25793 L
.31912 .25165 L
.30113 .24277 L
.28395 .23115 L
.27998 .22769 L
.27621 .22388 L
.2734 .22048 L
.27103 .21683 L
.27013 .21506 L
.26933 .21311 L
.26905 .21226 L
Mistroke
.2688 .21135 L
.2686 .21049 L
.26846 .2097 L
.2684 .20922 L
.26834 .20871 L
.2683 .20815 L
.26827 .20764 L
.26826 .20716 L
.26827 .20664 L
.2683 .20615 L
.26833 .2057 L
.26843 .20486 L
.2685 .20438 L
.26859 .20394 L
.26879 .20305 L
.26906 .20209 L
.26972 .20032 L
.2705 .19866 L
.2724 .19557 L
.27828 .18869 L
.29181 .17587 L
.29521 .17233 L
.29836 .16851 L
.3007 .16494 L
.30151 .16337 L
.30221 .16171 L
.30249 .16089 L
.30272 .16012 L
.3029 .15938 L
.30303 .1587 L
.30312 .15806 L
.30319 .15739 L
.30322 .15672 L
.30321 .15599 L
.30316 .15521 L
.30307 .15451 L
.303 .15412 L
.30291 .15371 L
.30271 .15296 L
.3022 .15156 L
.30157 .1503 L
.30067 .14893 L
.29967 .1477 L
.29739 .14556 L
.29441 .14352 L
.29067 .14161 L
.28684 .14015 L
.28299 .13903 L
.27853 .13806 L
.27355 .13729 L
Mistroke
.27099 .137 L
.26816 .13675 L
.26547 .13658 L
.26418 .13652 L
.26296 .13646 L
.26178 .13642 L
.26069 .13639 L
.25948 .13637 L
.25881 .13636 L
.25817 .13635 L
.25691 .13634 L
.25573 .13633 L
.25444 .13634 L
.25371 .13634 L
.25305 .13634 L
.25186 .13636 L
.25074 .13637 L
.24819 .13642 L
.24569 .13649 L
.24301 .13657 L
.23362 .13693 L
.22243 .13744 L
.2119 .13805 L
.20669 .13844 L
.20116 .13894 L
.19564 .13958 L
.18963 .14046 L
.18455 .14142 L
.17919 .14267 L
.17432 .14409 L
.16993 .14564 L
.16149 .14951 L
.15314 .15491 L
.14599 .16135 L
.14248 .1654 L
.13956 .1694 L
.1348 .17791 L
.13284 .18268 L
.13119 .18791 L
.13064 .19014 L
.13014 .19257 L
.12979 .19468 L
.12948 .19699 L
.12934 .1984 L
.12924 .19971 L
.12917 .20097 L
.12914 .20167 L
.12912 .20231 L
.1291 .20352 L
.1291 .2042 L
Mistroke
.1291 .20482 L
.12913 .20597 L
.12918 .20719 L
.12925 .20838 L
.12936 .20966 L
.12961 .212 L
.12994 .21427 L
.13031 .21635 L
.1314 .22105 L
.1329 .22596 L
.13614 .23392 L
.14065 .2423 L
.15204 .25765 L
.15952 .2653 L
.16695 .27166 L
.18244 .28211 L
.20098 .29139 L
.23865 .3063 L
.2751 .31997 L
.31602 .33253 L
.35472 .34266 L
.3724 .34654 L
.39208 .34986 L
.41165 .35222 L
.4325 .35413 L
.45239 .3556 L
.45829 .35596 L
.46397 .35625 L
.46909 .35646 L
.47183 .35656 L
.4748 .35664 L
.47765 .3567 L
.4789 .35673 L
.48027 .35675 L
.48157 .35676 L
.48278 .35677 L
.48406 .35678 L
.48543 .35678 L
.48689 .35678 L
.48772 .35678 L
.48849 .35678 L
.48989 .35676 L
.4906 .35676 L
.49138 .35675 L
.49417 .3567 L
.49679 .35663 L
.49942 .35654 L
.50191 .35645 L
.50657 .35621 L
.51156 .35589 L
Mistroke
.51695 .35545 L
.52794 .35427 L
.53768 .35289 L
.55894 .34875 L
.59861 .33779 L
.63525 .32538 L
.65363 .31875 L
.67445 .31162 L
.69519 .30575 L
.71425 .3018 L
.7338 .29877 L
.75448 .29549 L
.7732 .29139 L
.79327 .28551 L
.81176 .27877 L
.82085 .2748 L
.83027 .26994 L
.83862 .26468 L
.84582 .2591 L
.85828 .24597 L
.86384 .23789 L
.86823 .22994 L
.87214 .22092 L
.8739 .21586 L
.87528 .21118 L
.87637 .20673 L
.87735 .20184 L
.8781 .19694 L
.87842 .19425 L
.87866 .19174 L
.87883 .18936 L
.8789 .18819 L
.87895 .1871 L
.87898 .18612 L
.87901 .18507 L
.87904 .1839 L
.87904 .18285 L
.87905 .18224 L
.87904 .18161 L
.87903 .18046 L
.87902 .17983 L
.879 .17923 L
.87895 .17788 L
.8789 .17669 L
.87884 .17561 L
.87866 .17317 L
.87838 .17033 L
.87799 .1673 L
.87706 .16186 L
.87567 .15602 L
Mistroke
.87363 .14968 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l903Go001lo`040?oo00So1@04o`80C?l300Wo0`0bo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04[o00<0ool03Ol2033o001lo`040?oo00Wo00<0o`001Ol00`3oo`17o`<0
4_l00`3oo`0]o`00OOl200_o0P05o`030?oo04Ko00<0ool05?l00`3oo`0/o`00TOl00`3oo`16o`03
0?oo01Go00<0ool0:ol0047o5`0io`030?oo04Go00<0ool05_l00`3oo`0[o`00>ol601Oo0P0go`03
0?oo04Co00<0ool06?l00`3oo`0Zo`00>?l301oo0P0eo`030?oo04Co00<0ool06?l00`3oo`0Zo`00
=_l202?o0P0do`030?oo04Co00<0ool06Ol00`3oo`0Yo`00=Ol00`3oo`0To`030?oo03;o00<0ool0
@ol00`3oo`0Jo`030?oo02Wo000do`030?oo02Go00<0ool0<_l00`3oo`13o`030?oo01[o00<0ool0
:Ol003;o0P0Xo`030?oo03;o00<0ool0@_l00`3oo`0Ko`030?oo02Wo000ao`030?oo02Oo0P0do`03
0?oo04;o00<0ool07?l00`3oo`0Xo`00<Ol00`3oo`0Wo`030?oo03?o00<0ool0@_l00`3oo`0Lo`03
0?oo02So000ao`030?oo02Ko00<0ool0=?l00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool0
9_l00`3oo`0Qo`801?l200?o1005o`030?oo047o00<0ool07Ol00`3oo`0Xo`00;ol00`3oo`0Vo`03
0?oo027o00@0ool02?l00`3oo`06o`030?oo047o00<0ool07Ol00`3oo`0Xo`00;ol00`3oo`0Vo`03
0?oo027o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0Mo`030?oo02So000^o`030?oo02Ko00<0ool0
7?l400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool07Ol00`3oo`0Xo`00;_l00`3oo`0Uo`03
0?oo02?o00@0ool02?l0103oo`05o`030?oo043o00<0ool07_l00`3oo`0Xo`00;_l00`3oo`0To`03
0?oo02Go0P0:o`801_l00`3oo`10o`030?oo01ko00<0ool0:?l002ko00<0ool08_l203co00<0ool0
?ol00`3oo`0Oo`030?oo02So000^o`030?oo02;o00<0ool0>ol00`3oo`0oo`030?oo01oo00<0ool0
:?l002ko00<0ool08_l00`3oo`0ko`030?oo03ko00<0ool08?l00`3oo`0Xo`001ol500;o0P04o`80
6?l00`3oo`08o`D05Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200ko00<0ool0
4_l500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool0
5Ol00`3oo`09o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`03
0?oo00So00<0ool06Ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`03
0?oo01Go0P0:o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`03
0?oo00So00<0ool06Ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o
1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802ol00`3o
o`0Go`030?oo00Oo00<0ool06_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Ho`030?oo00Wo
00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3o
o`0Ho`030?oo00Oo00<0ool06_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201[o
00<0ool01ol200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`802?l00`3oo`0Jo`802_l300;o
000`o`030?oo02Co00<0ool0=ol00`3oo`0ho`030?oo02?o00<0ool0:ol0033o00<0ool09Ol203Oo
00<0ool0=ol00`3oo`0To`030?oo02_o000ao`030?oo02Ko0P0eo`030?oo03Go0P0Vo`030?oo02co
000ao`030?oo02So0P0co`030?oo03Co00<0ool09_l00`3oo`0/o`00<_l00`3oo`0Yo`80<Ol00`3o
o`0bo`80:?l00`3oo`0]o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`04o`<02?l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`02o`800_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`050?ooo`00
1?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Go00D0oooo
000[o`<0:_l00`3oo`0Yo`030?oo00809ol0103oo`0Yo`030?oo00Oo000eo`030?oo02oo100Vo`03
0?oo02So100/o`030?oo037o000fo`030?oo03;o0`0So`030?oo02Co100^o`80=?l003Oo00<0ool0
=?l3023o00<0ool08?l4033o0P0fo`00>?l00`3oo`0fo`807_l00`3oo`0Lo`@0<ol00`3oo`0fo`00
>Ol00`3oo`0go`@06_l00`3oo`0Ho`@0=Ol203Wo000jo`80>ol601Co00<0ool04ol503Go100ko`00
??l203oo2@0;o`030?oo00Wo2P0go`<0?ol003ko0P16oaL0?_l304;o0010o`<0C_l00`3oo`13o`@0
AOl004?o0P1<o`030?oo03ko1@19o`00AOl304Wo00<0ool0>?l604ko0018o`80Aol00`3oo`0co`D0
E?l004[o0`14o`030?oo02ko1@1Io`00COl202ko0P04o`800ol400Go00<0ool0:_l405ko001?o`<0
:_l0103oo`08o`030?oo00Ko00<0ool09ol306;o001Bo`80:?l0103oo`09o`030?oo00Go0P0Uo`<0
IOl005Co0`0Uo`040?oo00[o00<0ool01?l00`3oo`0Ro`80J?l005Oo100Qo`040?oo00So00@0ool0
1Ol00`3oo`0Oo`<0J_l005_o100No`802_l200Ko00<0ool07?l306go001Oo`<0;ol00`3oo`0Io`<0
L?l006;o0`0/o`030?oo01Ko0`1co`00IOl402So00<0ool04_l407Ko001Yo`@09?l00`3oo`0>o`@0
N_l006go1@0Oo`030?oo00[o101no`00L_l:01Go00<0ool01_l408;o001loah0Q_l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0
S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8719 .14513 m
.87018 .14107 L
.86831 .1374 L
.86573 .1332 L
.86306 .12959 L
.85658 .12302 L
.85248 .11994 L
.84828 .11744 L
.84369 .11531 L
.83857 .11353 L
.83405 .11242 L
.83142 .11195 L
.82895 .11161 L
.82758 .11147 L
.82608 .11135 L
.8254 .11131 L
.82467 .11127 L
.82335 .11122 L
.82269 .1112 L
.82197 .1112 L
.82067 .1112 L
.81941 .11123 L
.81825 .11128 L
.8176 .11131 L
.81689 .11136 L
.81562 .11146 L
.81276 .11177 L
.8098 .11223 L
.80715 .11276 L
.80273 .1139 L
.79827 .11543 L
.79096 .11896 L
.78734 .12133 L
.78422 .12384 L
.78114 .12687 L
.77811 .13054 L
.77366 .13778 L
.76982 .14653 L
.76408 .16429 L
.76072 .17533 L
.75718 .18556 L
.75315 .19498 L
.7488 .20302 L
.73797 .21733 L
.73122 .22403 L
.72343 .23071 L
.69155 .25265 L
.67382 .26167 L
.65394 .26891 L
.63313 .27424 L
Mistroke
.61309 .27844 L
.5743 .2856 L
.55362 .28867 L
.54196 .29007 L
.53119 .29109 L
.5258 .29147 L
.52013 .29178 L
.51718 .2919 L
.51444 .29198 L
.51287 .29202 L
.51144 .29204 L
.51065 .29205 L
.50979 .29206 L
.50898 .29206 L
.50823 .29207 L
.507 .29207 L
.50567 .29206 L
.50442 .29204 L
.50328 .29203 L
.50201 .292 L
.50063 .29197 L
.49784 .29188 L
.4951 .29176 L
.49218 .29161 L
.48694 .29125 L
.47629 .29029 L
.46494 .28899 L
.4449 .28644 L
.42609 .28389 L
.41623 .28232 L
.40561 .28027 L
.38587 .27524 L
.36743 .26944 L
.3301 .25689 L
.31188 .24946 L
.29444 .23994 L
.27888 .22819 L
.2728 .22171 L
.27018 .21798 L
.26916 .21616 L
.26838 .21448 L
.26779 .2129 L
.26732 .21123 L
.26713 .21028 L
.267 .20939 L
.26693 .20859 L
.26689 .20773 L
.26691 .20689 L
.26693 .20641 L
.26697 .20596 L
Mistroke
.26703 .20549 L
.26709 .20507 L
.26727 .20411 L
.26747 .20329 L
.26773 .20241 L
.26833 .20081 L
.26991 .19776 L
.27214 .19453 L
.27723 .18872 L
.29021 .17523 L
.29324 .1714 L
.29586 .16732 L
.29752 .16385 L
.2982 .16189 L
.29846 .16095 L
.29866 .16009 L
.29882 .15928 L
.29894 .15839 L
.29902 .15751 L
.29905 .15703 L
.29906 .15658 L
.29905 .15577 L
.29901 .15503 L
.29893 .15423 L
.2988 .1534 L
.29866 .15271 L
.29847 .15198 L
.29805 .1507 L
.29741 .14923 L
.29671 .14795 L
.29474 .1453 L
.29341 .14393 L
.29206 .14277 L
.28928 .14085 L
.28599 .13916 L
.28259 .13785 L
.27929 .13692 L
.27741 .1365 L
.27531 .13611 L
.27327 .13581 L
.27138 .13559 L
.26945 .13541 L
.2676 .13528 L
.26563 .13518 L
.26451 .13514 L
.26347 .1351 L
.26262 .13508 L
.2617 .13507 L
.26069 .13505 L
.25974 .13504 L
Mistroke
.25867 .13504 L
.25765 .13504 L
.25672 .13504 L
.25571 .13504 L
.2546 .13505 L
.25358 .13506 L
.25124 .13508 L
.24692 .13514 L
.24439 .13517 L
.24164 .1352 L
.239 .13522 L
.23657 .13524 L
.23526 .13524 L
.23382 .13525 L
.23246 .13525 L
.23119 .13525 L
.22981 .13526 L
.22902 .13526 L
.22828 .13526 L
.22749 .13526 L
.22678 .13526 L
.22599 .13526 L
.22515 .13526 L
.22373 .13526 L
.2224 .13526 L
.22097 .13526 L
.22015 .13526 L
.21938 .13527 L
.21799 .13527 L
.21672 .13528 L
.21531 .13529 L
.21383 .13531 L
.21245 .13533 L
.21096 .13535 L
.20824 .13542 L
.20559 .13551 L
.20316 .13561 L
.20055 .13575 L
.19768 .13595 L
.19469 .13621 L
.19199 .1365 L
.18715 .13718 L
.18211 .13814 L
.17762 .13927 L
.17353 .14057 L
.16508 .1443 L
.16076 .1469 L
.15708 .14957 L
.15058 .1556 L
.14522 .16216 L
Mistroke
.14019 .17018 L
.13785 .17485 L
.13593 .17934 L
.13326 .18755 L
.13224 .19199 L
.13179 .19449 L
.13147 .19678 L
.13124 .19899 L
.13114 .20024 L
.13107 .20142 L
.13102 .20266 L
.13099 .20379 L
.13099 .20504 L
.13099 .20566 L
.13101 .20635 L
.13105 .20751 L
.13111 .20856 L
.1312 .20977 L
.1313 .21091 L
.13154 .21294 L
.13187 .21513 L
.13291 .22009 L
.13443 .22527 L
.13811 .2343 L
.1426 .24252 L
.14828 .25086 L
.16028 .26443 L
.16773 .27103 L
.17644 .27745 L
.19373 .28724 L
.23133 .30294 L
.26752 .31743 L
.30476 .33002 L
.34398 .34117 L
.36153 .34552 L
.38113 .34943 L
.40088 .3524 L
.42166 .35493 L
.44277 .35724 L
.45313 .35823 L
.46301 .35903 L
.46802 .35936 L
.47361 .35966 L
.47932 .3599 L
.48247 .35999 L
.48388 .36002 L
.48543 .36005 L
.48688 .36008 L
.48823 .36009 L
.48952 .3601 L
Mistroke
.49089 .36011 L
.49225 .36011 L
.49352 .36011 L
.49466 .3601 L
.4959 .36008 L
.49726 .36007 L
.49875 .36004 L
.50176 .35996 L
.50454 .35986 L
.50712 .35975 L
.51277 .35941 L
.51884 .35893 L
.5296 .35771 L
.53974 .35613 L
.54887 .35434 L
.58821 .34398 L
.62539 .33253 L
.66292 .31774 L
.68086 .31058 L
.6914 .30705 L
.70135 .30432 L
.71131 .30219 L
.72192 .30048 L
.7432 .29784 L
.75338 .29634 L
.76434 .29417 L
.77465 .2915 L
.78408 .28855 L
.80353 .28127 L
.82072 .27365 L
.83583 .26503 L
.84357 .25914 L
.85011 .25287 L
.85595 .24585 L
.86146 .23751 L
.86656 .22762 L
.87037 .21805 L
.87337 .2081 L
.87466 .20256 L
.87561 .19753 L
.87626 .19317 L
.87679 .18844 L
.87699 .18608 L
.87707 .18478 L
.87714 .18356 L
.8772 .18242 L
.87725 .18119 L
.87728 .18007 L
.8773 .17904 L
.87731 .17796 L
Mistroke
.87731 .17694 L
.8773 .17603 L
.87729 .17503 L
.87726 .17394 L
.87722 .17279 L
.87717 .1717 L
.87712 .17071 L
.87698 .16878 L
.8768 .16669 L
.87635 .16289 L
.87561 .15837 L
.87456 .15359 L
.87343 .14953 L
.8719 .14513 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1>o`T0=Ol007co00@0ool0
2ol00`3oo`03o`030?oo04co0P09o`80<ol007co00@0ool02?l500Co0P1;o`803Ol2037o001fo`@0
0_l0103oo`08o`040?oo00Go00<0ool0B?l2017o00<0ool0;_l007co00@0ool02Ol00`3o0005o`03
0?oo04Oo00<0ool04_l00`3oo`0]o`00OOl200_o0P05o`030?oo04Ko00<0ool05?l00`3oo`0/o`00
A?l@03go00<0ool0AOl00`3oo`0Fo`030?oo02_o000lo`T03_l503Wo00<0ool0AOl00`3oo`0Fo`03
0?oo02_o000io`<07?l203Oo00<0ool0A?l00`3oo`0Ho`030?oo02[o000go`808Ol00`3oo`0do`03
0?oo04Co00<0ool06?l00`3oo`0Zo`00=Ol202?o0P0eo`030?oo04Co00<0ool06Ol00`3oo`0Yo`00
=?l00`3oo`0To`030?oo03?o00<0ool0A?l00`3oo`0Io`030?oo02Wo000do`030?oo02Co00<0ool0
<ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o00<0ool09Ol00`3oo`0co`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00<_l00`3oo`0Vo`030?oo03?o00<0ool0@ol00`3oo`0Ko`030?oo02So000ao`03
0?oo02Ko00<0ool0=?l00`3oo`13o`030?oo01_o00<0ool0:?l0037o00<0ool09_l00`3oo`0do`03
0?oo04;o00<0ool07?l00`3oo`0Xo`00<?l00`3oo`0Vo`030?oo027o0P04o`800ol400Go00<0ool0
@_l00`3oo`0Lo`030?oo02So000`o`030?oo02Go00<0ool08Ol0103oo`08o`030?oo00Ko00<0ool0
@_l00`3oo`0Lo`030?oo02So000_o`030?oo02Go00<0ool08_l0103oo`09o`030?oo00Go0P12o`03
0?oo01go00<0ool0:?l002oo00<0ool09?l00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0
@Ol00`3oo`0Mo`030?oo02So000_o`030?oo02?o00<0ool09?l0103oo`08o`040?oo00Go00<0ool0
@?l00`3oo`0No`030?oo02So000_o`030?oo02;o00<0ool09_l200[o0P06o`030?oo043o00<0ool0
7_l00`3oo`0Xo`00;_l00`3oo`0Ro`030?oo03_o00<0ool0@?l00`3oo`0Mo`030?oo02Wo000^o`03
0?oo02;o00<0ool0>ol00`3oo`0oo`030?oo01ko00<0ool0:Ol002ko00<0ool08_l00`3oo`0ko`03
0?oo03oo00<0ool07_l00`3oo`0Yo`001ol500;o0P04o`806?l00`3oo`08o`D05?l200co0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803_l00`3oo`0Bo`D02?l00`3oo`0Go`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Eo`030?oo00Wo00@0ool02?l0103oo`0Ro`03
0?oo027o00@0ool02?l0103oo`0<o`030?oo01Go00<0ool02?l00`3oo`0Io`030?oo00Oo00@0ool0
0_l000Wo00<0ool02_l00`3oo`0Fo`030?oo00Wo00<0ool05Ol00`3oo`09o`040?oo00_o00<0ool0
8?l00`3oo`0Qo`040?oo00_o00<0ool02Ol00`3oo`0Fo`030?oo00Oo00<0ool06_l00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Fo`030?oo00;o1002o`04
0?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3oo`07o`030?oo01[o00<0ool0
2?l200?o0009o`030?oo00So00<0ool06?l00`3oo`09o`030?oo01Oo00<0ool01ol0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`06o`030?oo01_o00<0ool0
2?l00`3oo`02o`002?l200[o0`0Ho`030?oo00So0P0Jo`030?oo00Oo0P0:o`<08_l00`3oo`0Ro`80
2_l300So00<0ool06?l200So00<0ool06_l200[o0`02o`00<?l00`3oo`0To`80>?l00`3oo`0ho`03
0?oo02?o00<0ool0:ol0033o00<0ool09_l00`3oo`0eo`030?oo03Ko0P0Uo`030?oo02co000ao`03
0?oo02Ko00<0ool0=?l00`3oo`0eo`030?oo02Go00<0ool0;?l0037o00<0ool09ol203Co00<0ool0
<ol202Oo00<0ool0;Ol003;o00<0ool0:?l00`3oo`0ao`030?oo03;o00<0ool09_l00`3oo`0^o`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`02o`050?ooo`00
2?l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`02o`800_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00?o00@0ool02?l00`3oo`05o`030?oo00Ko00<0ool01_l0103oool200?o00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Vo`040?oo02Wo0`0/o`030?oo02Wo
00@0oooo0P0Vo`040?oo02Wo00<0ool01ol003Go00<0ool0;Ol402So00<0ool0:_l302[o00<0ool0
<_l003Ko00<0ool0<?l302Go00<0ool09ol302_o0P0eo`00=ol00`3oo`0bo`<08_l00`3oo`0So`@0
;?l203Oo000ho`030?oo03Co0P0Po`030?oo01oo100^o`80>Ol003Wo0P0fo`@07?l00`3oo`0Mo`80
<?l203_o000ko`80>?l401So00<0ool05_l702oo0`0mo`00?Ol203[o1`0Ao`030?oo00ko200do`80
@?l003oo0P0oo`T02?l00`3oo`0:o`@0>Ol304;o0011o`80A_lE03_o0P15o`00@ol304_o00<0ool0
?ol604Oo0016o`80BOl00`3oo`0go`P0COl004So0`16o`030?oo037o1P1Eo`00Bol204Co00<0ool0
;_l305_o001=o`<0;Ol200Co0P03o`@01Ol00`3oo`0/o`80G_l0053o0P0Zo`040?oo00So00<0ool0
1_l00`3oo`0Zo`80H?l005;o0`0Wo`040?oo00Wo00<0ool01Ol202So0`1Ro`00EOl302Co00@0ool0
2_l00`3oo`04o`030?oo02Go0P1Uo`00F?l4023o00@0ool02?l0103oo`05o`030?oo02;o0`1Wo`00
G?l301ko0P0:o`801_l00`3oo`0Po`80J_l005oo0`0_o`030?oo01go0`1/o`00H_l402_o00<0ool0
6Ol406oo001Vo`@09ol00`3oo`0Fo`<0Lol006[o1@0Ro`030?oo01?o0`1fo`00Kol501go00<0ool0
3ol407Wo001do`D06?l00`3oo`0<o`<0OOl007Wo1P0Bo`030?oo00Ko1P20o`00OolK08Ko002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`04
0?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0
S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87017 .14057 m
.86843 .13664 L
.86653 .13307 L
.86391 .12899 L
.86116 .12547 L
.85448 .11906 L
.85024 .11608 L
.84591 .11368 L
.8412 .11166 L
.83852 .11075 L
.83596 .11002 L
.83356 .10946 L
.83135 .10904 L
.82888 .10868 L
.82751 .10853 L
.82621 .10841 L
.82549 .10836 L
.82481 .10831 L
.82354 .10826 L
.82283 .10824 L
.82218 .10822 L
.82146 .10822 L
.82071 .10822 L
.81946 .10825 L
.8183 .1083 L
.81703 .10837 L
.81568 .10848 L
.81424 .10863 L
.81269 .10883 L
.80991 .1093 L
.80728 .10986 L
.80491 .11048 L
.80091 .11178 L
.79688 .11348 L
.79037 .11728 L
.78707 .11988 L
.78395 .12294 L
.77917 .12923 L
.77531 .13631 L
.7717 .14491 L
.76558 .16249 L
.75822 .1826 L
.75371 .19268 L
.74835 .20255 L
.73644 .21869 L
.72321 .23171 L
.69177 .255 L
.67388 .26463 L
.66352 .26892 L
.65389 .27217 L
Mistroke
.6352 .27699 L
.61446 .28127 L
.57289 .28952 L
.55308 .293 L
.54176 .29458 L
.53635 .29519 L
.53129 .29566 L
.52592 .29604 L
.52308 .29619 L
.52004 .29632 L
.51851 .29636 L
.5177 .29638 L
.51684 .2964 L
.5161 .29641 L
.51528 .29642 L
.51453 .29643 L
.51383 .29643 L
.51241 .29643 L
.51111 .29642 L
.51043 .29641 L
.50969 .2964 L
.50819 .29637 L
.50672 .29633 L
.50534 .29629 L
.50277 .29617 L
.5 .29602 L
.49702 .29582 L
.48675 .29483 L
.47689 .29356 L
.46626 .29196 L
.4455 .2887 L
.42536 .28557 L
.40699 .28199 L
.38757 .27674 L
.36917 .2707 L
.33457 .25925 L
.31768 .25309 L
.29796 .2433 L
.28897 .23756 L
.27998 .23066 L
.27292 .22378 L
.27023 .22043 L
.26827 .21739 L
.26745 .21582 L
.26671 .21411 L
.26637 .21319 L
.26611 .21233 L
.2659 .21147 L
.26573 .21066 L
.26563 .20998 L
Mistroke
.26555 .20924 L
.2655 .20843 L
.26549 .2077 L
.26552 .20693 L
.2656 .2061 L
.26571 .20532 L
.26585 .2046 L
.26634 .20284 L
.26698 .20119 L
.26859 .19812 L
.27086 .19483 L
.27382 .19124 L
.28566 .17793 L
.28854 .17404 L
.29115 .16978 L
.29219 .1677 L
.29303 .16572 L
.29365 .16396 L
.29419 .16203 L
.29439 .16115 L
.29457 .16021 L
.29469 .15939 L
.29479 .15851 L
.29487 .15758 L
.29489 .15707 L
.2949 .15661 L
.2949 .15577 L
.29486 .15489 L
.2948 .15408 L
.29469 .1532 L
.29456 .15238 L
.29442 .15164 L
.29407 .15027 L
.29361 .14887 L
.29299 .14738 L
.29222 .14587 L
.29053 .14335 L
.28813 .14075 L
.28563 .13877 L
.28308 .13726 L
.2816 .13657 L
.28017 .13601 L
.2788 .13555 L
.27728 .13513 L
.27453 .13456 L
.2729 .13431 L
.27106 .13411 L
.27012 .13402 L
.26909 .13395 L
.26819 .1339 L
Mistroke
.2672 .13385 L
.2662 .13382 L
.26526 .13379 L
.26435 .13377 L
.26338 .13376 L
.26249 .13375 L
.26165 .13374 L
.26071 .13374 L
.25969 .13373 L
.25871 .13373 L
.25764 .13373 L
.25661 .13373 L
.25565 .13373 L
.25511 .13373 L
.25451 .13373 L
.25388 .13373 L
.25328 .13373 L
.25222 .13372 L
.25106 .13372 L
.24966 .1337 L
.2483 .13369 L
.24569 .13364 L
.24318 .13358 L
.24036 .1335 L
.22908 .13303 L
.22378 .13277 L
.21793 .13251 L
.21505 .1324 L
.21201 .13231 L
.21075 .13228 L
.20941 .13225 L
.20794 .13223 L
.20657 .13222 L
.2052 .13221 L
.20372 .13222 L
.20233 .13224 L
.20105 .13226 L
.1999 .13229 L
.19864 .13233 L
.19612 .13246 L
.19457 .13255 L
.19312 .13267 L
.19042 .13293 L
.18759 .13329 L
.18458 .13377 L
.17939 .13493 L
.17487 .13633 L
.17037 .13819 L
.16586 .14064 L
.15856 .14617 L
Mistroke
.1524 .15281 L
.14741 .1598 L
.14293 .16761 L
.13885 .17649 L
.13688 .18174 L
.13538 .18658 L
.13417 .19148 L
.13371 .19388 L
.13334 .19613 L
.13307 .19825 L
.13285 .20054 L
.13277 .20177 L
.13274 .20239 L
.13271 .20307 L
.13268 .20427 L
.13267 .20539 L
.13268 .206 L
.13269 .20667 L
.13273 .20787 L
.13279 .20904 L
.13287 .21011 L
.13312 .21255 L
.13349 .21512 L
.13435 .21938 L
.13543 .22332 L
.13877 .23212 L
.14357 .2413 L
.14903 .24958 L
.16099 .26363 L
.17584 .27608 L
.19135 .28526 L
.22737 .30079 L
.26187 .31537 L
.30163 .32972 L
.33962 .34116 L
.35812 .34601 L
.37891 .35033 L
.4181 .35623 L
.43895 .35912 L
.45897 .36152 L
.46961 .36248 L
.47581 .36291 L
.47881 .36307 L
.48163 .36321 L
.48432 .36332 L
.48678 .3634 L
.48953 .36346 L
.49105 .36349 L
.49249 .36351 L
.49388 .36352 L
Mistroke
.49536 .36353 L
.49676 .36353 L
.49803 .36352 L
.49952 .36351 L
.50036 .3635 L
.50113 .36348 L
.50263 .36345 L
.50405 .36342 L
.50694 .36332 L
.50956 .36321 L
.5154 .36285 L
.52036 .36244 L
.52566 .36188 L
.53728 .36017 L
.54758 .35811 L
.58709 .3473 L
.62515 .33623 L
.64331 .32945 L
.66237 .32046 L
.68114 .31153 L
.69158 .30742 L
.69744 .30549 L
.70303 .30392 L
.70841 .30265 L
.71425 .30152 L
.72441 .30008 L
.73462 .29904 L
.7439 .29815 L
.74859 .29761 L
.7536 .29692 L
.76261 .29526 L
.77326 .29251 L
.78294 .28929 L
.80279 .2811 L
.82002 .27289 L
.83503 .26412 L
.84259 .25827 L
.84886 .25205 L
.85496 .24426 L
.85993 .23613 L
.86461 .22637 L
.86863 .21552 L
.87138 .20579 L
.87257 .20052 L
.8735 .19563 L
.8742 .19117 L
.8748 .18643 L
.875 .18437 L
.87519 .18221 L
.87532 .18038 L
Mistroke
.87543 .17839 L
.87548 .17722 L
.87552 .17613 L
.87554 .1751 L
.87557 .174 L
.87558 .17292 L
.87558 .17191 L
.87557 .171 L
.87555 .17002 L
.87553 .16895 L
.87549 .16777 L
.87543 .16654 L
.87537 .1654 L
.87522 .16335 L
.87502 .16116 L
.87448 .15687 L
.87376 .1528 L
.87294 .14916 L
.87182 .1452 L
.87017 .14057 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1@o`D0=ol007go0P04o`801Ol300Co00<0ool0COl300Go100co`00
O?l0103oo`0;o`030?oo00?o00<0ool0Bol200co0P0ao`00O?l0103oo`08o`D01?l204[o0P0@o`03
0?oo02ko001fo`@00_l0103oo`08o`040?oo00Go00<0ool0B?l00`3oo`0Ao`030?oo02go001lo`04
0?oo00Wo00<0o`001Ol00`3oo`17o`030?oo01?o00<0ool0;?l007go0P0;o`801Ol00`3oo`17o`03
0?oo01?o00<0ool0;?l003_o6`0ko`030?oo04Ko00<0ool05Ol00`3oo`0[o`00>Ol201_o0P0io`03
0?oo04Ko00<0ool05_l00`3oo`0Zo`00>?l00`3oo`0Mo`030?oo03Ko00<0ool0AOl00`3oo`0Go`03
0?oo02[o000fo`808Ol00`3oo`0eo`030?oo04Go00<0ool05ol00`3oo`0Zo`00=Ol00`3oo`0Qo`80
=_l00`3oo`15o`030?oo01So00<0ool0:Ol003Co00<0ool08ol00`3oo`0do`030?oo04Co00<0ool0
6Ol00`3oo`0Yo`00<ol00`3oo`0To`030?oo03Co00<0ool0A?l00`3oo`0Io`030?oo02Wo000co`03
0?oo02Co00<0ool0=?l00`3oo`13o`030?oo01[o00<0ool0:Ol003;o00<0ool09Ol00`3oo`0do`03
0?oo04?o00<0ool06_l00`3oo`0Yo`00<_l00`3oo`0To`030?oo03Go00<0ool0@ol00`3oo`0Jo`03
0?oo02Wo000ao`030?oo02Go00<0ool0=Ol00`3oo`12o`030?oo01_o00<0ool0:Ol0033o00<0ool0
9Ol00`3oo`0Ro`801?l200?o1005o`030?oo04;o00<0ool06ol00`3oo`0Yo`00<?l00`3oo`0To`03
0?oo02;o00@0ool02?l00`3oo`06o`030?oo047o00<0ool07?l00`3oo`0Yo`00<?l00`3oo`0So`03
0?oo02?o00@0ool02Ol00`3oo`05o`80@_l00`3oo`0Lo`030?oo02Wo000`o`030?oo02;o00<0ool0
7_l400;o00@0ool02_l00`3oo`04o`030?oo047o00<0ool07?l00`3oo`0Yo`00;ol00`3oo`0Ro`03
0?oo02Go00@0ool02?l0103oo`05o`030?oo043o00<0ool07Ol00`3oo`0Yo`00;ol00`3oo`0Ro`03
0?oo02Ko0P0:o`801_l00`3oo`10o`030?oo01go00<0ool0:Ol002oo00<0ool08Ol00`3oo`0ko`03
0?oo03oo00<0ool07_l00`3oo`0Yo`00;ol00`3oo`0Po`80?Ol00`3oo`0oo`030?oo01ko00<0ool0
:Ol002oo00<0ool08?l00`3oo`0lo`030?oo03ko00<0ool07_l00`3oo`0Zo`001ol500;o0P04o`80
6Ol00`3oo`07o`D05?l00`3oo`0;o`801?l200Co0P0So`030?oo02;o0P04o`801?l200go00<0ool0
4ol500Oo00<0ool06?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
5?l00`3oo`0:o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`03
0?oo00Oo00<0ool06_l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`03
0?oo01Go00<0ool02Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo00<0ool0
5_l00`3oo`07o`030?oo01[o00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806Ol0103o
ool400Co00<0ool05Ol00`3oo`03o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`03
0?oo01Oo00<0ool01_l00`3oo`0Ko`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Wo00<0ool0
2?l00`3oo`0Fo`030?oo00So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`09o`03
0?oo01So00<0ool01_l00`3oo`0Ko`030?oo00So00<0ool00_l000So0P0:o`<06Ol00`3oo`07o`80
6Ol00`3oo`08o`802_l302;o00<0ool08_l200[o0`08o`030?oo01So0P07o`030?oo01_o0P0:o`<0
0_l0033o00<0ool08ol00`3oo`0ho`030?oo03So00<0ool08_l00`3oo`0/o`00<Ol00`3oo`0So`03
0?oo03Oo00<0ool0=ol00`3oo`0Ro`030?oo02go000ao`030?oo02Co0P0go`030?oo03Go0P0Uo`03
0?oo02go000bo`030?oo02Go0P0eo`030?oo03Co00<0ool09?l00`3oo`0^o`00<ol00`3oo`0Vo`80
<ol00`3oo`0co`030?oo02Co00<0ool0;ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
0_l30003o`3o00Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Go00<0ool01_l00`3o
o`06o`050?ooo`001?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`03
0?oo02Ko00@0ool0:_l302_o00<0ool0:Ol00`3oo`02o`809Ol0103oo`0Yo`030?oo00Oo000eo`03
0?oo02ko0P0Yo`030?oo02co0P0Yo`030?oo03;o000fo`030?oo02oo0`0Vo`030?oo02[o0P0Yo`80
=Ol003Oo00<0ool0<Ol202Co00<0ool09ol302Wo0P0go`00>?l203?o100Po`030?oo02?o100Zo`80
>Ol003[o00<0ool0=?l401co00<0ool07_l502co0P0ko`00>ol203So0P0Jo`030?oo01[o100^o`<0
?Ol003go0P0ho`T04Ol00`3oo`0Fo`@0<?l2043o000oo`80?ol600_o00<0ool04_l4037o0`12o`00
@Ol204?o1007o`030?oo00[o200bo`<0AOl004?o0P15oa@0=_l404So0015o`80B_l00`3oo`0go`T0
C?l004Oo0`17o`030?oo03;o1@1Eo`00B_l304Co00<0ool0;_l405[o001=o`<0;Ol200Co0P03o`@0
1Ol00`3oo`0]o`030?oo05co001@o`<0:Ol0103oo`08o`030?oo00Ko00<0ool0:ol205oo001Co`<0
9_l0103oo`09o`030?oo00Go0P0Zo`80HOl005Ko0P0To`040?oo00[o00<0ool01?l00`3oo`0Wo`80
Hol005So0`0Qo`040?oo00So00@0ool01Ol00`3oo`0Uo`80IOl005_o0`0Oo`802_l200Ko00<0ool0
8_l306Oo001No`<0<?l00`3oo`0Po`80J_l0067o100/o`030?oo01go0`1/o`00IOl302Wo00<0ool0
6Ol406oo001Xo`809ol00`3oo`0Fo`<0Lol006[o100So`030?oo01?o0`1fo`00K_l801_o00<0ool0
3ol407Wo001fo`P04ol00`3oo`0;o`@0OOl007ko1@0>o`030?oo00Go1P21o`00PolF08Oo002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86844 .13602 m
.86668 .13221 L
.86476 .12875 L
.86208 .12477 L
.85926 .12135 L
.85238 .1151 L
.84801 .11222 L
.84355 .10991 L
.8387 .108 L
.83596 .10716 L
.83334 .10651 L
.8309 .10602 L
.82866 .10567 L
.82746 .10552 L
.82616 .10539 L
.82544 .10533 L
.82478 .10528 L
.82347 .10521 L
.82274 .10518 L
.82207 .10516 L
.82134 .10515 L
.82056 .10515 L
.81927 .10517 L
.81853 .1052 L
.81785 .10523 L
.81659 .10531 L
.81544 .10541 L
.8129 .10571 L
.81026 .10615 L
.80747 .10676 L
.80267 .10819 L
.79877 .10978 L
.79541 .11153 L
.79179 .1139 L
.78883 .11635 L
.78375 .12208 L
.78151 .12549 L
.77938 .12939 L
.76615 .16339 L
.75703 .1849 L
.74748 .20276 L
.73621 .21865 L
.72379 .23201 L
.69242 .2571 L
.67399 .26759 L
.66443 .27175 L
.65376 .27546 L
.63462 .28032 L
.61364 .28454 L
.57115 .29361 L
Mistroke
.56065 .29582 L
.54936 .29791 L
.54383 .29878 L
.53861 .29948 L
.53393 .3 L
.53127 .30024 L
.52879 .30044 L
.52615 .3006 L
.52374 .30072 L
.52251 .30077 L
.52117 .30081 L
.51975 .30085 L
.51841 .30087 L
.51726 .30088 L
.516 .30088 L
.51468 .30087 L
.51345 .30085 L
.51229 .30083 L
.51124 .3008 L
.50887 .30071 L
.50628 .30057 L
.50387 .30041 L
.49844 .29994 L
.49318 .29933 L
.48761 .29856 L
.44592 .29095 L
.42625 .28751 L
.40799 .28365 L
.3897 .27842 L
.37033 .27179 L
.33617 .26056 L
.3178 .25418 L
.29977 .24553 L
.29064 .23996 L
.28123 .23313 L
.27362 .22631 L
.2682 .21983 L
.2663 .21669 L
.26548 .21494 L
.26489 .21336 L
.26445 .21174 L
.26427 .21083 L
.26415 .20998 L
.26411 .20953 L
.26408 .2091 L
.26406 .2083 L
.26409 .20744 L
.26412 .20696 L
.26417 .20651 L
.26429 .20573 L
Mistroke
.26446 .2049 L
.26488 .20341 L
.26556 .20164 L
.26634 .20002 L
.27106 .19316 L
.27662 .18654 L
.28255 .17896 L
.28508 .17506 L
.2871 .17128 L
.28857 .16783 L
.28923 .16588 L
.28974 .16406 L
.29014 .16229 L
.29046 .1604 L
.29058 .15933 L
.29067 .15832 L
.29072 .15742 L
.29075 .15645 L
.29074 .15552 L
.29072 .15499 L
.2907 .1545 L
.29063 .15352 L
.29058 .15297 L
.29052 .15248 L
.29026 .15076 L
.28987 .14901 L
.28946 .14755 L
.28895 .14605 L
.28784 .14354 L
.28628 .14089 L
.28467 .13881 L
.28279 .13696 L
.28065 .13541 L
.27872 .13439 L
.27758 .13393 L
.27649 .13357 L
.27518 .13324 L
.2739 .13299 L
.27271 .13282 L
.27137 .13268 L
.27059 .13262 L
.26986 .13257 L
.269 .13253 L
.26817 .1325 L
.26738 .13248 L
.26664 .13247 L
.2658 .13246 L
.2649 .13245 L
.26331 .13244 L
.26245 .13243 L
Mistroke
.26151 .13242 L
.26066 .13242 L
.25971 .13241 L
.25798 .13238 L
.25691 .13236 L
.25573 .13233 L
.25351 .13226 L
.25122 .13216 L
.24904 .13205 L
.23934 .13137 L
.22904 .13044 L
.2173 .1294 L
.21387 .12914 L
.21063 .12894 L
.20775 .12879 L
.20614 .12872 L
.20465 .12867 L
.20335 .12864 L
.20196 .12862 L
.20043 .12861 L
.19905 .12861 L
.19762 .12864 L
.19608 .12868 L
.19465 .12874 L
.19334 .12881 L
.19106 .12899 L
.1886 .12925 L
.1861 .12959 L
.18383 .12999 L
.17921 .13111 L
.17674 .13191 L
.17453 .13275 L
.17035 .13478 L
.16626 .13741 L
.15973 .14332 L
.15457 .14979 L
.14505 .16637 L
.14058 .17652 L
.13886 .18111 L
.13726 .18605 L
.13597 .19089 L
.13507 .1953 L
.13474 .19742 L
.1346 .19854 L
.13447 .19973 L
.13436 .20087 L
.13428 .20191 L
.13423 .20294 L
.13419 .20392 L
.13416 .20506 L
Mistroke
.13416 .2063 L
.13418 .20692 L
.13419 .2076 L
.13425 .2088 L
.13433 .21001 L
.13443 .21114 L
.13468 .21327 L
.13507 .21577 L
.13554 .21807 L
.13692 .22325 L
.13841 .22752 L
.14032 .23207 L
.14448 .24012 L
.14984 .24853 L
.15645 .25713 L
.17041 .27102 L
.17892 .27729 L
.18739 .28238 L
.22295 .29838 L
.26081 .31509 L
.30077 .33012 L
.33872 .34208 L
.35697 .34703 L
.37772 .35141 L
.41693 .3578 L
.43749 .36123 L
.4574 .36419 L
.46807 .36541 L
.47393 .36594 L
.48031 .3664 L
.48595 .36671 L
.48919 .36684 L
.49068 .36688 L
.49225 .36692 L
.49359 .36695 L
.49505 .36698 L
.49638 .36699 L
.49762 .367 L
.49908 .36701 L
.50041 .367 L
.50195 .36699 L
.50339 .36697 L
.50479 .36695 L
.50629 .36691 L
.50899 .36682 L
.51218 .36667 L
.51507 .3665 L
.52125 .36601 L
.52697 .36538 L
.53667 .36391 L
Mistroke
.54678 .36182 L
.58785 .35007 L
.60747 .34475 L
.62805 .33889 L
.64799 .3309 L
.66612 .32099 L
.68475 .31095 L
.69602 .30623 L
.70157 .3044 L
.70688 .30294 L
.71194 .30182 L
.71744 .30085 L
.72243 .30019 L
.72703 .29971 L
.73797 .29889 L
.74325 .29852 L
.74805 .29811 L
.75246 .29763 L
.75708 .29698 L
.76169 .29616 L
.76671 .29504 L
.77728 .29188 L
.78705 .2881 L
.82302 .27025 L
.8312 .26536 L
.83931 .25951 L
.84625 .25299 L
.85203 .24582 L
.85721 .23742 L
.86146 .22859 L
.8654 .21814 L
.86876 .20656 L
.87111 .19575 L
.87209 .18985 L
.8728 .18457 L
.8733 .17967 L
.87347 .17741 L
.87361 .17537 L
.87372 .17304 L
.87377 .17183 L
.87379 .17122 L
.87381 .17057 L
.87383 .16947 L
.87384 .16845 L
.87385 .16783 L
.87385 .16725 L
.87384 .16616 L
.87383 .16524 L
.87381 .16425 L
.87378 .16332 L
Mistroke
.87375 .16247 L
.87364 .1604 L
.87351 .15845 L
.87333 .15645 L
.87312 .15459 L
.87251 .15039 L
.87179 .14673 L
.87097 .14341 L
.86844 .13602 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1Ao`<0>?l0097o00<0ool0COl50003o`0000<0=?l007go0P04o`801Ol300Co00<0ool0
Bol200_o0P0bo`00O?l0103oo`0;o`030?oo00?o00<0ool0BOl200oo00<0ool0;ol007co00@0ool0
2?l500Co0P19o`030?oo013o00<0ool0;_l007Ko1002o`040?oo00So00@0ool01Ol00`3oo`18o`03
0?oo017o00<0ool0;Ol007co00@0ool02Ol00`3o0005o`030?oo04Oo00<0ool04ol00`3oo`0/o`00
>olB033o0P0;o`801Ol00`3oo`17o`030?oo01Co00<0ool0:ol003Wo0`0Ao`X0>_l00`3oo`17o`03
0?oo01Co00<0ool0:ol003So00<0ool06ol203Wo00<0ool0A_l00`3oo`0Fo`030?oo02[o000go`03
0?oo01ko00<0ool0=_l00`3oo`16o`030?oo01Ko00<0ool0:_l003Ko00<0ool07ol00`3oo`0fo`03
0?oo04Go00<0ool05ol00`3oo`0Zo`00=Ol00`3oo`0Po`80=ol00`3oo`15o`030?oo01Oo00<0ool0
:_l003Co00<0ool08_l00`3oo`0eo`030?oo04Co00<0ool06Ol00`3oo`0Yo`00=?l00`3oo`0Ro`03
0?oo03Go00<0ool0A?l00`3oo`0Io`030?oo02Wo000co`030?oo02?o00<0ool0=Ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003?o00<0ool08ol00`3oo`0eo`030?oo04?o00<0ool06_l00`3oo`0Yo`00
<_l00`3oo`0So`030?oo03Ko00<0ool0@ol00`3oo`0Jo`030?oo02Wo000ao`030?oo02Co00<0ool0
=_l00`3oo`12o`030?oo01_o00<0ool0:Ol0037o00<0ool08ol00`3oo`0So`801?l200?o1005o`03
0?oo04;o00<0ool06ol00`3oo`0Yo`00<?l00`3oo`0To`030?oo02;o00@0ool02?l00`3oo`06o`03
0?oo04;o00<0ool06ol00`3oo`0Yo`00<?l00`3oo`0So`030?oo02?o00@0ool02Ol00`3oo`05o`80
@_l00`3oo`0Lo`030?oo02Wo000`o`030?oo02;o00<0ool07_l400;o00@0ool02_l00`3oo`04o`03
0?oo047o00<0ool07?l00`3oo`0Yo`00<?l00`3oo`0Ro`030?oo02Co00@0ool02?l0103oo`05o`03
0?oo043o00<0ool07?l00`3oo`0Zo`00;ol00`3oo`0Ro`030?oo02Ko0P0:o`801_l00`3oo`0oo`03
0?oo01go00<0ool0:_l002oo00<0ool08Ol00`3oo`0ko`030?oo03oo00<0ool07Ol00`3oo`0Zo`00
;ol00`3oo`0Po`030?oo03co00<0ool0?_l00`3oo`0No`030?oo02[o000_o`030?oo023o00<0ool0
??l00`3oo`0mo`030?oo01oo00<0ool0:_l000Oo1@02o`801?l201Wo00<0ool01ol501Co00<0ool0
2ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0=o`030?oo01?o1@07o`030?oo01So1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool06?l00`3oo`09o`030?oo01Co00<0ool02_l0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`08o`040?oo00_o00<0ool05_l00`3oo`07o`030?oo01[o00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Ko00<0ool02Ol00`3oo`0Do`030?oo00[o00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`030?oo01Ko00<0ool01_l00`3oo`0Ko`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201[o00<0ool01004o`030?oo01Go00<0ool0
0ol400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802ol00`3oo`0Go`030?oo00Ko00<0ool0
6ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Io`030?oo00So00<0ool05Ol00`3oo`09o`04
0?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00Ko00<0ool0
6ol00`3oo`08o`030?oo00;o0008o`802_l301Wo00<0ool01ol201So00<0ool02Ol200[o0`0Ro`03
0?oo02;o0P0:o`<02?l00`3oo`0Ho`801ol00`3oo`0Ko`802_l300;o000ao`030?oo027o00<0ool0
>Ol00`3oo`0ho`030?oo02;o00<0ool0;?l003;o00<0ool08Ol203Wo00<0ool0=ol00`3oo`0Ro`03
0?oo02go000bo`030?oo02?o0P0go`030?oo03Ko00<0ool08ol00`3oo`0]o`00<_l00`3oo`0Uo`03
0?oo03Co00<0ool0=Ol00`3oo`0So`030?oo02ko000co`030?oo02Go0P0do`030?oo03?o0P0Uo`03
0?oo02oo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00;o
00D0oooo0008o`030?oo00Ko00<0ool01Ol00`3oo`06o`040?ooo`<00_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Go0P08o`030?oo00Go00<0ool01_l00`3oo`06o`050?ooo`001?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Ko00@0ool0:Ol302co00<0ool0
:Ol00`3oo`04o`030?oo02;o00@0ool0:Ol00`3oo`07o`00=_l00`3oo`0[o`<0:_l00`3oo`0^o`80
9_l203Co000go`030?oo02go0`0Wo`030?oo02co0P0Vo`80=_l003So00<0ool0;ol302Co00<0ool0
:_l202Ko0P0ho`00>Ol00`3oo`0ao`<08Ol00`3oo`0Wo`<09_l203[o000jo`80=?l201oo00<0ool0
8ol402Oo0P0lo`00??l00`3oo`0co`@06ol00`3oo`0No`D0:Ol203ko000mo`80=ol501Ko00<0ool0
6_l402co0P10o`00?ol203[o1@0Ao`030?oo01Ko100^o`80@_l0047o0`0lo`D03?l00`3oo`0Bo`@0
;ol304Co0014o`80?ol600Ko00<0ool03?l602oo1017o`00A_l304;o2@03o`T0:ol:04_o0019o`80
A_l00`3o000302oo1@1Eo`00Bol304?o00<0ool0<?l205[o001>o`80;Ol200Co0P03o`@01Ol00`3o
o`0^o`80G?l0053o0`0Yo`040?oo00So00<0ool01_l00`3oo`0/o`80G_l005?o0`0Vo`040?oo00Wo
00<0ool01Ol202_o0P1Po`00E_l202Co00@0ool02_l00`3oo`04o`030?oo02Wo00<0ool0H?l005So
0`0Qo`040?oo00So00@0ool01Ol00`3oo`0Wo`80Hol005_o0`0Oo`802_l200Ko00<0ool09Ol206Go
001No`<0<?l00`3oo`0So`80Iol0067o100/o`030?oo027o0P1Yo`00IOl202[o00<0ool07?l506_o
001Wo`80:?l00`3oo`0Ho`@0L?l006Wo100To`030?oo01Ko0P1do`00KOl601ko00<0ool04ol307Ko
001co`D06Ol00`3oo`0?o`@0NOl007So1@0Do`030?oo00_o101mo`00OOl:00[o00<0ool01Ol6087o
0027oa80Qol0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3o
o`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103o
o`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00
oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86671 .13146 m
.86494 .12778 L
.86298 .12442 L
.86025 .12056 L
.85736 .11722 L
.85028 .11115 L
.84578 .10836 L
.84118 .10615 L
.8362 .10435 L
.83362 .10364 L
.83072 .103 L
.82824 .10258 L
.82687 .1024 L
.82556 .10225 L
.82435 .10215 L
.82303 .10207 L
.82184 .10202 L
.82073 .102 L
.82 .102 L
.81933 .10201 L
.81859 .10202 L
.81782 .10205 L
.81653 .10212 L
.8158 .10218 L
.81512 .10224 L
.81274 .10251 L
.81025 .10292 L
.80745 .10354 L
.8049 .10425 L
.80044 .10591 L
.797 .10761 L
.7936 .10978 L
.79048 .11233 L
.78795 .11497 L
.78421 .12023 L
.78246 .1235 L
.78099 .1267 L
.76765 .1618 L
.7487 .19949 L
.73809 .21577 L
.72537 .23121 L
.71079 .24532 L
.69223 .25981 L
.68287 .26579 L
.67273 .27126 L
.6542 .27858 L
.64496 .28117 L
.63492 .28346 L
.61647 .28708 L
.57379 .29673 L
Mistroke
.56329 .29929 L
.55202 .30176 L
.5465 .3028 L
.54128 .30365 L
.5366 .30428 L
.53394 .30457 L
.53145 .30481 L
.5288 .30502 L
.52638 .30517 L
.52514 .30523 L
.5238 .30528 L
.52237 .30532 L
.52102 .30535 L
.51987 .30536 L
.5186 .30537 L
.51727 .30536 L
.51603 .30534 L
.51487 .30531 L
.51381 .30528 L
.51141 .30517 L
.5088 .30501 L
.50637 .30483 L
.50088 .30426 L
.49557 .30354 L
.48995 .30262 L
.44819 .2936 L
.4287 .28977 L
.41054 .28574 L
.3922 .28029 L
.37269 .27334 L
.33851 .26205 L
.32029 .25619 L
.30216 .24802 L
.29278 .24258 L
.28288 .23579 L
.27463 .22893 L
.26846 .2224 L
.26616 .21923 L
.26429 .21588 L
.26357 .21411 L
.26326 .21315 L
.26302 .21227 L
.26286 .21152 L
.26273 .21072 L
.26268 .21027 L
.26264 .20984 L
.2626 .20905 L
.26261 .20828 L
.26263 .20784 L
.26266 .20743 L
Mistroke
.26276 .20661 L
.26283 .20615 L
.26291 .20573 L
.26332 .20418 L
.26384 .20272 L
.26545 .19948 L
.26778 .19591 L
.27329 .18869 L
.27864 .18124 L
.28265 .17393 L
.28435 .16961 L
.28496 .16765 L
.2855 .16555 L
.28593 .16347 L
.28623 .16158 L
.28634 .16067 L
.28643 .15969 L
.2865 .15876 L
.28655 .15791 L
.28658 .15693 L
.28659 .15602 L
.28658 .15503 L
.28654 .15398 L
.28647 .15294 L
.28638 .15187 L
.28614 .14999 L
.28584 .14822 L
.28545 .1464 L
.28458 .14336 L
.28334 .14017 L
.28208 .13771 L
.28076 .13575 L
.27929 .13411 L
.27797 .13305 L
.2772 .13258 L
.27644 .13222 L
.27554 .13189 L
.27464 .13164 L
.27379 .13148 L
.27329 .13141 L
.27281 .13135 L
.27223 .1313 L
.27166 .13126 L
.27113 .13123 L
.27055 .13121 L
.26988 .13119 L
.26953 .13118 L
.26914 .13117 L
.26774 .13115 L
.26631 .13114 L
Mistroke
.26551 .13113 L
.26462 .13111 L
.26382 .1311 L
.2629 .13107 L
.2612 .13101 L
.25944 .13093 L
.25742 .13082 L
.25344 .13052 L
.24791 .12997 L
.24211 .12928 L
.23003 .12768 L
.21763 .1261 L
.21403 .12571 L
.21065 .12539 L
.20761 .12514 L
.20596 .12503 L
.20444 .12494 L
.203 .12488 L
.2017 .12483 L
.20027 .12479 L
.19875 .12477 L
.19749 .12477 L
.19615 .12478 L
.19489 .12481 L
.19374 .12486 L
.19244 .12493 L
.19124 .12501 L
.18983 .12514 L
.18856 .12528 L
.18605 .12564 L
.18345 .12613 L
.18103 .12673 L
.17893 .12737 L
.17454 .12917 L
.17085 .13127 L
.16768 .13362 L
.16148 .14009 L
.1584 .14436 L
.15574 .14863 L
.14623 .16728 L
.14212 .17684 L
.13896 .18556 L
.13771 .18983 L
.13715 .19208 L
.13663 .19446 L
.13622 .19675 L
.13592 .19882 L
.1358 .19982 L
.13569 .2009 L
.13562 .20184 L
Mistroke
.13555 .20285 L
.1355 .204 L
.13548 .20507 L
.13547 .20573 L
.13548 .20635 L
.1355 .20751 L
.13556 .20874 L
.13563 .20987 L
.13574 .21111 L
.13589 .21243 L
.13621 .21458 L
.13658 .2166 L
.13771 .22119 L
.13912 .22555 L
.14101 .23029 L
.14571 .23967 L
.15628 .2553 L
.16956 .26931 L
.17789 .27577 L
.18605 .28086 L
.22226 .29752 L
.25712 .31364 L
.29812 .32991 L
.3372 .34281 L
.37301 .35182 L
.39256 .35529 L
.41335 .35892 L
.43242 .36262 L
.44272 .36461 L
.4543 .36661 L
.46529 .36817 L
.47142 .36888 L
.47825 .36952 L
.48463 .36999 L
.48745 .37015 L
.49057 .37029 L
.49352 .3704 L
.49494 .37044 L
.49628 .37047 L
.4978 .3705 L
.49919 .37051 L
.50068 .37052 L
.5023 .37052 L
.50384 .37051 L
.50525 .37049 L
.50687 .37046 L
.50841 .37042 L
.51112 .37033 L
.51405 .37019 L
.51723 .36999 L
Mistroke
.52057 .36972 L
.52645 .36911 L
.5329 .3682 L
.53877 .36715 L
.54925 .36474 L
.58832 .35292 L
.60914 .34751 L
.62002 .34475 L
.63118 .34149 L
.64129 .33768 L
.65006 .33343 L
.66686 .32309 L
.67605 .31714 L
.68636 .31115 L
.69562 .30678 L
.70112 .30471 L
.70622 .30312 L
.7111 .30189 L
.71645 .30085 L
.72182 .3001 L
.72474 .29979 L
.72745 .29955 L
.73243 .29923 L
.73692 .29901 L
.74167 .29881 L
.74435 .29868 L
.74684 .29854 L
.75178 .29817 L
.75428 .29792 L
.75696 .29759 L
.76179 .29683 L
.76617 .29592 L
.77679 .29278 L
.78681 .28871 L
.82226 .26974 L
.83722 .25984 L
.84438 .25333 L
.85017 .24622 L
.85536 .23764 L
.85956 .22853 L
.8635 .2175 L
.86683 .2053 L
.86907 .19452 L
.87001 .18879 L
.87073 .18356 L
.87126 .17887 L
.87148 .17648 L
.87168 .17399 L
.87182 .17195 L
.87194 .16974 L
Mistroke
.87203 .16778 L
.87206 .16682 L
.87208 .16596 L
.8721 .16487 L
.87211 .16427 L
.87212 .16371 L
.87212 .16272 L
.87212 .16165 L
.87211 .16109 L
.87211 .16049 L
.8721 .15985 L
.87208 .15926 L
.87205 .15821 L
.87201 .15709 L
.87189 .1549 L
.87174 .15285 L
.87156 .15101 L
.87133 .14903 L
.87076 .14526 L
.86994 .14127 L
.86903 .13786 L
.86777 .13411 L
.86671 .13146 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1>o`T0=Ol0097o00<0ool0C?l200Wo0`0bo`00OOl200Co0P05o`<01?l00`3oo`1:o`80
3_l00`3oo`0_o`00O?l0103oo`0;o`030?oo00?o00<0ool0BOl00`3oo`0?o`030?oo02ko001lo`04
0?oo00So1@04o`80BOl00`3oo`0Ao`030?oo02go001fo`@00_l0103oo`08o`040?oo00Go00<0ool0
B?l00`3oo`0Ao`030?oo02go000koa00<Ol0103oo`09o`030?l000Go00<0ool0Aol00`3oo`0Co`80
;Ol003Wo0P0@o`X0:?l200_o0P05o`030?oo04Oo00<0ool05?l00`3oo`0[o`00>?l00`3oo`0Jo`80
>_l00`3oo`16o`030?oo01Go00<0ool0:ol003Oo00<0ool07?l203Wo00<0ool0A_l00`3oo`0Fo`03
0?oo02[o000fo`030?oo01ko00<0ool0=ol00`3oo`16o`030?oo01Ko00<0ool0:_l003Ko00<0ool0
7_l00`3oo`0go`030?oo04Go00<0ool05ol00`3oo`0Zo`00=Ol00`3oo`0Po`030?oo03Ko00<0ool0
AOl00`3oo`0Go`030?oo02[o000do`030?oo027o00<0ool0=_l00`3oo`15o`030?oo01Oo00<0ool0
:_l003Co00<0ool08Ol00`3oo`0fo`030?oo04Go00<0ool05ol00`3oo`0Zo`00<ol00`3oo`0Ro`03
0?oo03Ko00<0ool0A?l00`3oo`0Ho`030?oo02[o000co`030?oo02;o00<0ool0=_l00`3oo`14o`03
0?oo01So00<0ool0:_l003;o00<0ool08_l00`3oo`0go`030?oo04?o00<0ool06Ol00`3oo`0Zo`00
<_l00`3oo`0Ro`030?oo03Oo00<0ool0@ol00`3oo`0Io`030?oo02[o000ao`030?oo02?o00<0ool0
8ol200Co0P03o`@01Ol00`3oo`12o`030?oo01[o00<0ool0:_l0037o00<0ool08ol00`3oo`0Ro`04
0?oo00So00<0ool01_l00`3oo`12o`030?oo01[o00<0ool0:_l0033o00<0ool08ol00`3oo`0So`04
0?oo00Wo00<0ool01Ol204;o00<0ool06ol00`3oo`0Zo`00<?l00`3oo`0Ro`030?oo01ko1002o`04
0?oo00[o00<0ool01?l00`3oo`11o`030?oo01_o00<0ool0:_l0033o00<0ool08Ol00`3oo`0Uo`04
0?oo00So00@0ool01Ol00`3oo`10o`030?oo01co00<0ool0:_l0033o00<0ool08Ol00`3oo`0Vo`80
2_l200Ko00<0ool0@?l00`3oo`0Lo`030?oo02[o000`o`030?oo023o00<0ool0>ol00`3oo`0oo`03
0?oo01go00<0ool0:_l0033o00<0ool07ol00`3oo`0lo`030?oo03oo00<0ool07?l00`3oo`0[o`00
<?l00`3oo`0No`80?_l00`3oo`0no`030?oo01go00<0ool0:ol000Oo1@02o`801?l201[o00<0ool0
1_l501?o00<0ool03?l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0=o`030?oo01?o1@06o`03
0?oo01Wo1@02o`801?l200?o0009o`030?oo00Oo00@0ool06Ol00`3oo`08o`030?oo01?o0P0<o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03?l00`3oo`0Eo`030?oo00Go00<0ool0
7?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05ol00`3oo`08o`030?oo01Co00<0ool0
2_l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00Wo00<0ool05_l00`3oo`05o`03
0?oo01co00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806_l00`3oo`0400Co00<0ool0
5?l00`3oo`04o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0;o`030?oo01Oo00<0ool0
1Ol00`3oo`0Lo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01[o00<0ool01ol00`3oo`0Eo`80
2_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`04o`03
0?oo01go00<0ool02?l00`3oo`02o`002?l200[o0`0Jo`030?oo00Ko0P0Io`030?oo00So0P0:o`<0
8_l00`3oo`0Ro`802_l300So00<0ool06?l200Ko00<0ool07?l200[o0`02o`00<Ol00`3oo`0Ro`03
0?oo03So00<0ool0>?l00`3oo`0Qo`030?oo02go000bo`030?oo02;o00<0ool0=ol00`3oo`0go`03
0?oo027o00<0ool0;_l003;o00<0ool08ol00`3oo`0fo`030?oo03Ko00<0ool08_l00`3oo`0^o`00
<ol00`3oo`0So`80=_l00`3oo`0fo`030?oo02;o00<0ool0;_l003Co00<0ool09?l203Co00<0ool0
=Ol00`3oo`0Ro`030?oo02oo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00?o00@0ool02?l00`3oo`06o`030?oo00Go00<0ool01_l0103oool200?o00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`05o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09ol00`3o000Xo`@0
;?l00`3oo`0Yo`030?oo00Go00<0ool08Ol00`3o000Zo`030?oo00Oo000go`030?oo02[o100Yo`03
0?oo02oo0P0To`80=Ol003So00<0ool0;Ol302Ko00<0ool0;Ol202Co0P0go`00>Ol00`3oo`0_o`<0
8ol00`3oo`0[o`809Ol00`3oo`0go`00>_l00`3oo`0ao`<08?l00`3oo`0Zo`030?oo02?o0P0jo`00
>ol00`3oo`0co`807_l00`3oo`0Xo`809?l203co000lo`80=Ol301_o00<0ool08_l602Co0P0no`00
?_l203Ko0P0Io`030?oo01ko100Xo`80@?l0043o0P0fo`@05Ol00`3oo`0Io`D0:_l204;o0012o`80
>?l5013o00<0ool04ol602co0`14o`00A?l203_o1@0;o`030?oo00oo100_o`<0Aol004Ko0`0mo`H0
1Ol00`3oo`0<o`<0:?l;04[o0019o`80@Ol600So1P0Wo`@0EOl004_o0P14o`X0:Ol305Wo001=o`80
;_l200Co0P03o`@01Ol00`3oo`0^o`80G?l004oo0`0Zo`040?oo00So00<0ool01_l00`3oo`0/o`80
G_l005;o0`0Wo`040?oo00Wo00<0ool01Ol202_o0P1Po`00EOl202Go00@0ool02_l00`3oo`04o`03
0?oo02Wo00<0ool0H?l005Oo0`0Ro`040?oo00So00@0ool01Ol00`3oo`0Wo`80Hol005[o0`0Po`80
2_l200Ko00<0ool09Ol206Go001Mo`80<_l00`3oo`0So`80Iol005oo0`0_o`030?oo027o0P1Yo`00
H_l202go00<0ool07_l306_o001To`@0:Ol00`3oo`0Ko`<0K_l006So1@0To`030?oo01Oo101ao`00
KOl501oo00<0ool04ol407Go001bo`D06_l00`3oo`0Ao`80NOl007Oo1@0Eo`030?oo00ko0`1ko`00
O?l700ko00<0ool03?l207ko0023o`D02Ol00`3oo`05o`L0P?l008So4@27o`00OOl200Co0P05o`<0
1?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`80
2ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0
S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`04
0?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86498 .12691 m
.86319 .12335 L
.86121 .1201 L
.85842 .11635 L
.85546 .1131 L
.84818 .10719 L
.84355 .10449 L
.83882 .10238 L
.83371 .1007 L
.83106 .10005 L
.82811 .09949 L
.82558 .09914 L
.82416 .09899 L
.82286 .09889 L
.82166 .09883 L
.82036 .09879 L
.81913 .09878 L
.81799 .0988 L
.81726 .09882 L
.81659 .09885 L
.81585 .0989 L
.81508 .09895 L
.81379 .09908 L
.8124 .09924 L
.80988 .09965 L
.8076 .10014 L
.8028 .10158 L
.80032 .10259 L
.7982 .10362 L
.79488 .10563 L
.79182 .10805 L
.78958 .11032 L
.78747 .11303 L
.78558 .11615 L
.78388 .11974 L
.78134 .12688 L
.77615 .14322 L
.76731 .16427 L
.74571 .20323 L
.7343 .22058 L
.72268 .23507 L
.70759 .25012 L
.69115 .26322 L
.67298 .27423 L
.66321 .27864 L
.6528 .28228 L
.64354 .28477 L
.63347 .28695 L
.6149 .29048 L
.57189 .30097 L
Mistroke
.56136 .30382 L
.55007 .30651 L
.54453 .30761 L
.53929 .30847 L
.53455 .30909 L
.5319 .30936 L
.52941 .30956 L
.52697 .30971 L
.5256 .30977 L
.52435 .30982 L
.52298 .30985 L
.52221 .30987 L
.5215 .30987 L
.52018 .30988 L
.51879 .30986 L
.51746 .30984 L
.51603 .30979 L
.51468 .30974 L
.51344 .30967 L
.51056 .30947 L
.50754 .3092 L
.50257 .30859 L
.49798 .30789 L
.48768 .30584 L
.44866 .29594 L
.43021 .2919 L
.40983 .28699 L
.38924 .28029 L
.37033 .27316 L
.33541 .26196 L
.32674 .25941 L
.31814 .25644 L
.30067 .24841 L
.28142 .23605 L
.27284 .22889 L
.26885 .22484 L
.26577 .22111 L
.26373 .21799 L
.2629 .21642 L
.26217 .21472 L
.26162 .213 L
.2614 .21208 L
.26132 .21164 L
.26125 .21122 L
.26116 .21043 L
.26112 .20971 L
.26111 .20894 L
.26112 .20851 L
.26115 .20811 L
.26122 .20736 L
Mistroke
.26135 .20656 L
.26151 .20581 L
.26168 .20513 L
.26224 .20346 L
.26293 .20188 L
.26702 .19517 L
.27192 .18819 L
.27636 .18086 L
.27802 .17741 L
.27952 .17362 L
.28019 .17158 L
.28082 .16932 L
.2813 .16726 L
.28167 .1653 L
.28201 .16299 L
.28223 .16089 L
.28232 .15973 L
.28238 .15865 L
.2824 .15809 L
.28242 .15748 L
.28243 .15684 L
.28243 .15624 L
.28243 .15523 L
.2824 .15418 L
.28237 .15329 L
.28232 .15233 L
.28225 .15131 L
.28215 .15023 L
.28195 .14834 L
.28166 .14626 L
.28129 .14413 L
.28054 .14062 L
.27974 .1377 L
.27898 .13542 L
.2782 .13355 L
.27749 .13224 L
.27707 .13165 L
.27666 .13118 L
.2762 .13079 L
.27593 .13061 L
.27566 .13046 L
.27539 .13034 L
.27508 .13023 L
.27478 .13014 L
.27449 .13008 L
.27416 .13002 L
.27378 .12997 L
.27357 .12995 L
.27335 .12994 L
.27313 .12992 L
Mistroke
.27292 .12991 L
.27249 .12989 L
.27225 .12989 L
.272 .12988 L
.27154 .12987 L
.27101 .12986 L
.2705 .12986 L
.26993 .12985 L
.26937 .12984 L
.26881 .12983 L
.26814 .12981 L
.26749 .12979 L
.26588 .12972 L
.26433 .12963 L
.26249 .12949 L
.25858 .1291 L
.25445 .12858 L
.2295 .12445 L
.21711 .12249 L
.21021 .12161 L
.207 .12129 L
.20409 .12106 L
.20246 .12095 L
.20075 .12087 L
.19987 .12084 L
.19906 .12081 L
.19817 .1208 L
.19722 .12079 L
.19564 .12079 L
.19485 .12081 L
.194 .12084 L
.19252 .12091 L
.19117 .121 L
.18973 .12113 L
.18841 .12128 L
.18601 .12165 L
.18355 .12217 L
.18102 .12288 L
.17892 .12363 L
.17679 .12456 L
.17325 .12659 L
.16951 .12956 L
.16648 .13275 L
.16077 .14104 L
.15595 .15016 L
.14687 .16949 L
.1432 .17795 L
.13985 .18697 L
.13841 .19196 L
.1379 .19413 L
Mistroke
.13743 .19652 L
.13709 .19877 L
.13685 .20086 L
.13675 .20202 L
.13669 .20307 L
.13665 .20429 L
.13663 .20544 L
.13664 .20657 L
.13666 .20721 L
.13668 .20781 L
.13675 .20902 L
.1368 .2097 L
.13685 .21032 L
.13711 .21253 L
.13748 .21484 L
.1386 .21974 L
.1403 .22508 L
.14443 .23465 L
.14911 .24311 L
.15534 .25245 L
.16164 .26033 L
.17515 .27299 L
.18314 .2784 L
.19223 .2834 L
.2268 .29908 L
.26347 .31681 L
.30346 .33252 L
.32343 .33946 L
.34356 .34599 L
.38176 .35475 L
.42218 .3626 L
.44173 .36701 L
.45203 .36909 L
.46221 .37082 L
.47279 .37223 L
.47841 .37283 L
.48456 .37335 L
.48783 .37357 L
.49083 .37373 L
.49385 .37387 L
.49671 .37397 L
.49833 .37401 L
.50004 .37404 L
.5015 .37405 L
.5031 .37406 L
.504 .37406 L
.50482 .37406 L
.50577 .37405 L
.50666 .37404 L
.50825 .37401 L
Mistroke
.50914 .37399 L
.50997 .37397 L
.51161 .37391 L
.51333 .37384 L
.51638 .37368 L
.51962 .37345 L
.52312 .37314 L
.52923 .37242 L
.53472 .37156 L
.54449 .36951 L
.5544 .36677 L
.59691 .35361 L
.61939 .34828 L
.62883 .3458 L
.63865 .3425 L
.64743 .33854 L
.65529 .33397 L
.67246 .32167 L
.69007 .3101 L
.69525 .30746 L
.70094 .30502 L
.70595 .30326 L
.71144 .30174 L
.71451 .30107 L
.71734 .30055 L
.71999 .30015 L
.72291 .29979 L
.72552 .29954 L
.72795 .29936 L
.7305 .29921 L
.73323 .2991 L
.73888 .29895 L
.74194 .29889 L
.7448 .29883 L
.74746 .29875 L
.74894 .2987 L
.75031 .29863 L
.75271 .29849 L
.75523 .29829 L
.75777 .29803 L
.76013 .29773 L
.76546 .2968 L
.77079 .2955 L
.77649 .29368 L
.78696 .28928 L
.79638 .28439 L
.81579 .27294 L
.83131 .26303 L
.83799 .25798 L
.8436 .25269 L
Mistroke
.84883 .24619 L
.85395 .23749 L
.85842 .22711 L
.86205 .21599 L
.86475 .20541 L
.86706 .19378 L
.86806 .18758 L
.86882 .18193 L
.86937 .17701 L
.86981 .17205 L
.86999 .16957 L
.87015 .16693 L
.87026 .16459 L
.87033 .16246 L
.87037 .1612 L
.87038 .16059 L
.87039 .16002 L
.8704 .15897 L
.8704 .15785 L
.8704 .15674 L
.87039 .15612 L
.87038 .15554 L
.87037 .15493 L
.87036 .15437 L
.87032 .15313 L
.87025 .15142 L
.87017 .14989 L
.87004 .14813 L
.86989 .14649 L
.86955 .1436 L
.86907 .14048 L
.86837 .13705 L
.8674 .13342 L
.86498 .12691 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1>o`T0
=Ol0097o00<0ool0C?l200Wo0P0co`00TOl00`3oo`1:o`803Ol00`3oo`0`o`00OOl200Co0P05o`<0
1?l00`3oo`1:o`030?oo00go00<0ool0;ol007co00@0ool02ol00`3oo`03o`030?oo04So0P0Ao`80
;ol007co00@0ool02?l500Co0P19o`030?oo01;o00<0ool0;?l003_o3@0^o`@00_l0103oo`08o`04
0?oo00Go00<0ool0Aol00`3oo`0Co`030?oo02co000jo`803?l502oo00@0ool02Ol00`3o0005o`03
0?oo04Oo00<0ool05?l00`3oo`0[o`00>Ol00`3oo`0Ao`T09ol200_o0P05o`030?oo04Oo00<0ool0
5?l00`3oo`0[o`00>?l00`3oo`0Jo`80>_l00`3oo`17o`030?oo01Co00<0ool0:ol003Oo00<0ool0
7?l00`3oo`0ho`030?oo04Ko00<0ool05Ol00`3oo`0[o`00=_l00`3oo`0Mo`030?oo03So00<0ool0
A_l00`3oo`0Fo`030?oo02[o000eo`030?oo01ko00<0ool0>?l00`3oo`16o`030?oo01Ko00<0ool0
:_l003Go00<0ool07ol00`3oo`0go`030?oo04Ko00<0ool05_l00`3oo`0Zo`00=Ol00`3oo`0Oo`03
0?oo03Oo00<0ool0AOl00`3oo`0Go`030?oo02[o000do`030?oo023o00<0ool0=ol00`3oo`15o`03
0?oo01Oo00<0ool0:_l003Co00<0ool08?l00`3oo`0go`030?oo04Co00<0ool06?l00`3oo`0Zo`00
=?l00`3oo`0Oo`80>Ol00`3oo`14o`030?oo01So00<0ool0:_l003?o00<0ool08?l00`3oo`0ho`03
0?oo04?o00<0ool06Ol00`3oo`0Zo`00<ol00`3oo`0Po`030?oo03So00<0ool0@ol00`3oo`0Io`03
0?oo02[o000bo`030?oo027o00<0ool09?l200Co0P03o`@01Ol00`3oo`12o`030?oo01[o00<0ool0
:_l003;o00<0ool08?l00`3oo`0To`040?oo00So00<0ool01_l00`3oo`12o`030?oo01[o00<0ool0
:_l0037o00<0ool08Ol00`3oo`0To`040?oo00Wo00<0ool01Ol204;o00<0ool06ol00`3oo`0Zo`00
<Ol00`3oo`0Qo`030?oo01ko1002o`040?oo00[o00<0ool01?l00`3oo`11o`030?oo01[o00<0ool0
:ol0033o00<0ool08Ol00`3oo`0Uo`040?oo00So00@0ool01Ol00`3oo`10o`030?oo01_o00<0ool0
:ol0033o00<0ool08?l00`3oo`0Wo`802_l200Ko00<0ool0@?l00`3oo`0Ko`030?oo02_o000`o`03
0?oo01oo00<0ool0??l00`3oo`0oo`030?oo01co00<0ool0:ol0033o00<0ool07_l00`3oo`0mo`03
0?oo03oo00<0ool07?l00`3oo`0[o`00<?l00`3oo`0No`030?oo03go00<0ool0?_l00`3oo`0Mo`03
0?oo02_o0007o`D00_l200Co0P0Jo`030?oo00Ko1@0Co`030?oo00co0P04o`801?l202?o00<0ool0
8_l200Co0P04o`803Ol00`3oo`0Co`D01_l00`3oo`0Io`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01Wo00<0ool02?l00`3oo`0Co`030?oo00_o00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`0;o`030?oo01Ko00<0ool01Ol00`3oo`0Lo`030?oo00Oo00@0ool00_l000Wo00<0ool0
2_l00`3oo`0Go`030?oo00So00<0ool04ol200co00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`09o`030?oo01Ko00<0ool01Ol00`3oo`0Lo`030?oo00[o0@01o`7o00000ol0000200Co
00<0ool02?l201[o00<0ool01004o`030?oo01Co00<0ool01?l400;o00@0ool02Ol202?o00<0ool0
8Ol0103oo`09o`802ol00`3oo`0Go`030?oo00Go00<0ool07?l00`3oo`08o`800ol000Wo00<0ool0
2?l00`3oo`0Jo`030?oo00Oo00<0ool05Ol00`3oo`09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`04
0?oo00Wo00<0ool02Ol00`3oo`0Ho`030?oo00Co00<0ool07Ol00`3oo`08o`030?oo00;o0008o`80
2_l301[o00<0ool01_l201So00<0ool02Ol200[o0`0Ro`030?oo02;o0P0:o`<02?l00`3oo`0Ho`80
1_l00`3oo`0Lo`802_l300;o000bo`030?oo023o00<0ool0>Ol00`3oo`0io`030?oo023o00<0ool0
;Ol003;o00<0ool08Ol00`3oo`0ho`030?oo03So00<0ool08?l00`3oo`0^o`00<ol00`3oo`0Qo`03
0?oo03Oo00<0ool0=ol00`3oo`0Qo`030?oo02ko000co`030?oo02;o0P0go`030?oo03Ko00<0ool0
8Ol00`3oo`0_o`00=?l00`3oo`0So`030?oo03Co00<0ool0=Ol00`3oo`0Qo`030?oo033o000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00?o00@0ool02?l00`3o
o`06o`030?oo00Go00<0ool01_l0103o0005o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l200Oo
00<0ool01Ol00`3oo`06o`030?oo00Ko00@0o`001Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Oo000<o`030?oo02Oo00<0o`009_l300;o00<0ool0:_l00`3oo`0Yo`030?oo00Go
0P0Ro`80:ol00`3oo`07o`00=_l00`3oo`0Yo`D0:_l00`3oo`0`o`030?oo02;o00<0ool0=?l003Oo
00<0ool0;Ol302Oo00<0ool0;ol00`3oo`0Qo`80=ol003So0P0`o`<09?l00`3oo`0]o`808ol00`3o
o`0go`00>_l00`3oo`0`o`<08Ol00`3oo`0[o`808ol203[o000ko`030?oo03;o0P0Oo`030?oo02Wo
0P0So`80??l003co0P0do`<07?l00`3oo`0Wo`808ol203ko000no`<0=?l301Wo00<0ool08_l502?o
0P10o`00@Ol203Go100Eo`030?oo01ko100Wo`030?oo043o0013o`80=ol5013o00<0ool06_l402Wo
0P13o`00AOl203[o100<o`030?oo01Ko100Yo`@0AOl004Oo0P0lo`@02?l00`3oo`0Bo`@08?l=04Wo
0019o`80?_l300Go00<0ool03?l6027o101Eo`00Bol203oo1P0:o`@09?l305Wo001=o`80;_l200Co
0P03o`@01Ol;02Oo00<0ool0F_l004oo0P0[o`040?oo00So00<0ool01_l00`3oo`0]o`80GOl0057o
0`0Xo`040?oo00Wo00<0ool01Ol202co0P1Oo`00E?l202Ko00@0ool02_l00`3oo`04o`030?oo02[o
00<0ool0Gol005Ko0`0So`040?oo00So00@0ool01Ol00`3oo`0Xo`80H_l005Wo0P0Ro`802_l200Ko
00<0ool09ol00`3oo`1Ro`00Fol303?o00<0ool09_l00`3oo`1So`00G_l3033o00<0ool09Ol00`3o
o`1To`00HOl302go00<0ool08ol206Oo001To`80:ol00`3oo`0No`D0JOl006Ko100Wo`030?oo01_o
0`1^o`00J_l502;o00<0ool06?l3077o001_o`D07Ol00`3oo`0Eo`<0M?l007Co1P0Go`030?oo017o
101go`00N_l501;o00<0ool03Ol407_o001oo`@03_l00`3oo`0:o`<0Ool008?o1@09o`030?oo00Go
1@22o`00OOl200Co0P03oa40Qol007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So
1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3o
o`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o
0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go
0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go
0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86325 .12235 m
.86145 .11892 L
.85943 .11577 L
.85659 .11213 L
.85356 .10898 L
.84609 .10323 L
.84131 .10063 L
.83645 .09861 L
.83121 .09705 L
.8285 .09646 L
.82693 .09618 L
.82549 .09597 L
.82426 .09582 L
.82292 .09569 L
.82219 .09564 L
.82151 .09559 L
.82016 .09553 L
.81891 .0955 L
.81777 .09551 L
.81647 .09554 L
.81526 .0956 L
.81395 .09569 L
.81254 .09583 L
.81108 .09602 L
.80975 .09624 L
.80742 .09671 L
.80502 .09733 L
.80235 .09821 L
.79997 .09919 L
.79596 .10134 L
.79305 .10345 L
.79034 .10606 L
.78804 .10905 L
.78634 .11206 L
.78508 .11504 L
.78413 .1179 L
.7824 .12481 L
.78059 .13269 L
.77789 .14193 L
.77422 .15141 L
.76991 .16053 L
.74764 .19884 L
.73663 .21671 L
.72623 .23173 L
.7128 .24716 L
.69721 .26129 L
.67961 .27364 L
.66998 .27886 L
.65958 .28328 L
.64968 .28645 L
Mistroke
.63879 .28906 L
.61846 .2929 L
.57585 .30355 L
.55596 .30941 L
.55046 .3108 L
.54463 .31207 L
.53955 .31298 L
.53408 .31372 L
.53105 .31402 L
.52829 .31421 L
.52695 .31429 L
.5255 .31434 L
.52412 .31438 L
.52285 .3144 L
.52214 .3144 L
.52138 .3144 L
.52 .31439 L
.51919 .31437 L
.51835 .31434 L
.51684 .31428 L
.51525 .3142 L
.51378 .3141 L
.51044 .3138 L
.50753 .31346 L
.50478 .31308 L
.49967 .31221 L
.48836 .30966 L
.44994 .29851 L
.43208 .29416 L
.41199 .28906 L
.39181 .28225 L
.37286 .27482 L
.33835 .26356 L
.32138 .25867 L
.31087 .25458 L
.30047 .24949 L
.28088 .23703 L
.27224 .22996 L
.26832 .22615 L
.26489 .22222 L
.26256 .2189 L
.2616 .2172 L
.26078 .21541 L
.26021 .21379 L
.26 .21304 L
.25985 .21235 L
.25973 .21167 L
.25964 .21093 L
.2596 .21028 L
.25958 .2096 L
Mistroke
.25961 .20881 L
.25967 .2081 L
.25972 .20771 L
.25978 .20728 L
.25993 .20651 L
.26033 .20502 L
.26083 .20365 L
.26235 .20047 L
.26443 .19697 L
.26884 .19005 L
.27115 .18603 L
.27337 .18146 L
.27504 .17719 L
.27578 .17484 L
.27648 .17226 L
.27701 .16985 L
.27741 .16762 L
.27773 .16536 L
.27796 .16322 L
.27806 .16214 L
.27813 .16098 L
.27819 .15988 L
.27823 .15888 L
.27826 .15778 L
.27827 .15658 L
.27827 .15598 L
.27827 .15534 L
.27825 .1542 L
.27824 .15357 L
.27822 .153 L
.27817 .15172 L
.27805 .14945 L
.2779 .14745 L
.27773 .14543 L
.2765 .13398 L
.27621 .13076 L
.27618 .13045 L
.27615 .13015 L
.27612 .12991 L
.27608 .12968 L
.27604 .12948 L
.276 .12932 L
.27594 .12918 L
.27589 .12907 L
.27583 .12898 L
.27575 .12889 L
.27566 .12881 L
.27555 .12875 L
.27545 .12871 L
.27533 .12867 L
Mistroke
.27519 .12864 L
.27504 .12862 L
.27485 .1286 L
.27463 .12859 L
.27443 .12858 L
.27417 .12857 L
.27387 .12857 L
.27351 .12857 L
.27314 .12856 L
.27277 .12856 L
.27254 .12855 L
.27229 .12855 L
.27182 .12853 L
.27123 .12851 L
.27064 .12848 L
.26944 .1284 L
.26799 .12827 L
.26435 .12784 L
.25938 .12708 L
.23531 .12226 L
.22109 .11945 L
.21408 .11828 L
.20777 .11744 L
.20438 .1171 L
.20268 .11696 L
.20173 .1169 L
.20088 .11685 L
.20001 .1168 L
.19912 .11677 L
.1975 .11673 L
.1966 .11672 L
.19577 .11672 L
.19488 .11674 L
.19394 .11676 L
.19246 .11684 L
.19164 .11689 L
.1909 .11696 L
.18956 .1171 L
.18818 .11728 L
.18589 .1177 L
.1835 .11829 L
.18126 .11902 L
.17897 .11999 L
.17709 .12099 L
.17521 .12219 L
.17202 .12486 L
.16939 .12776 L
.16392 .13634 L
.15952 .14541 L
.15003 .16601 L
Mistroke
.14586 .17497 L
.14183 .18455 L
.14031 .18896 L
.1396 .19137 L
.13902 .19364 L
.13856 .19575 L
.13817 .19805 L
.13803 .19906 L
.13791 .20013 L
.13782 .20107 L
.13775 .20206 L
.13769 .2032 L
.13766 .20424 L
.13765 .20536 L
.13767 .20659 L
.13772 .20766 L
.13779 .20881 L
.13789 .20989 L
.138 .21087 L
.13835 .21328 L
.13885 .21591 L
.14009 .22069 L
.14167 .22534 L
.14342 .22969 L
.14824 .23943 L
.15411 .24907 L
.16026 .25747 L
.16658 .26461 L
.17406 .27135 L
.1889 .28093 L
.22578 .29819 L
.2612 .3161 L
.30025 .33214 L
.32163 .33989 L
.33958 .34603 L
.37621 .35504 L
.41588 .36309 L
.43478 .36793 L
.45343 .37228 L
.46509 .37438 L
.47703 .37599 L
.48387 .37667 L
.49031 .37715 L
.49369 .37734 L
.49742 .37749 L
.4993 .37754 L
.50033 .37757 L
.50131 .37759 L
.50222 .3776 L
.50321 .37761 L
Mistroke
.50494 .37762 L
.5058 .37762 L
.50671 .37761 L
.50833 .3776 L
.51007 .37756 L
.51105 .37754 L
.51196 .37751 L
.51509 .37738 L
.51683 .37728 L
.51841 .37718 L
.52131 .37696 L
.52441 .37667 L
.52989 .37599 L
.53601 .37498 L
.54223 .37367 L
.55277 .37076 L
.59139 .35804 L
.61468 .35261 L
.62466 .35036 L
.63498 .3475 L
.64427 .34389 L
.65232 .33955 L
.66912 .32689 L
.68621 .31355 L
.69132 .31029 L
.69692 .30721 L
.7019 .30493 L
.70737 .3029 L
.71314 .30128 L
.71859 .30019 L
.72192 .29971 L
.72507 .29938 L
.72788 .29917 L
.72947 .29908 L
.73094 .29902 L
.73233 .29897 L
.73306 .29896 L
.73382 .29894 L
.73523 .29892 L
.73652 .29891 L
.73775 .2989 L
.7389 .2989 L
.74016 .29891 L
.7415 .29891 L
.74292 .29892 L
.74367 .29892 L
.74446 .29892 L
.74515 .29893 L
.74591 .29893 L
.74724 .29892 L
Mistroke
.74853 .29891 L
.74925 .2989 L
.74991 .29889 L
.75119 .29887 L
.7524 .29883 L
.75368 .29878 L
.75506 .29871 L
.75755 .29854 L
.75997 .2983 L
.76218 .29803 L
.76719 .29715 L
.76971 .29658 L
.77246 .29583 L
.78197 .29236 L
.79082 .28803 L
.8264 .26549 L
.8339 .26013 L
.841 .25407 L
.8468 .24741 L
.85149 .23986 L
.85559 .23078 L
.85896 .22088 L
.86212 .20887 L
.86482 .19534 L
.8666 .18348 L
.86735 .17702 L
.86788 .17144 L
.86828 .16588 L
.86844 .16298 L
.86856 .16034 L
.8686 .15918 L
.86863 .15798 L
.86865 .15695 L
.86867 .15587 L
.86869 .15472 L
.86869 .15367 L
.86869 .15303 L
.86869 .15243 L
.86869 .1519 L
.86868 .15132 L
.86866 .15028 L
.86864 .14934 L
.86861 .14831 L
.86857 .14725 L
.86848 .14548 L
.86835 .14357 L
.86821 .14189 L
.86803 .14012 L
.86759 .13695 L
.86685 .13303 L
Mistroke
.86589 .12932 L
.86478 .12598 L
.86325 .12235 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1>o`P0=_l0097o00<0ool0
Bol300So0`0co`00TOl00`3oo`1:o`030?oo00co0P0ao`00TOl00`3oo`1:o`030?oo00ko00<0ool0
;_l007go0P04o`801Ol300Co00<0ool0BOl00`3oo`0?o`030?oo02ko001lo`040?oo00_o00<0ool0
0ol00`3oo`18o`030?oo017o00<0ool0;Ol003_o3P0co`040?oo00So1@04o`80BOl00`3oo`0Bo`03
0?oo02co000jo`803Ol502So1002o`040?oo00So00@0ool01Ol00`3oo`18o`030?oo01;o00<0ool0
;?l003Wo00<0ool04_l400?o00<0ool09?l0103oo`09o`030?l000Go00<0ool0B?l00`3oo`0Co`03
0?oo02_o000ho`030?oo01Oo100Wo`802ol200Go00<0ool0Aol00`3oo`0Do`030?oo02_o000ho`03
0?oo01[o00<0ool0>Ol00`3oo`17o`030?oo01Co00<0ool0:ol003Oo00<0ool06ol00`3oo`0io`03
0?oo04Ko00<0ool05Ol00`3oo`0[o`00=_l00`3oo`0Lo`030?oo03Wo00<0ool0A_l00`3oo`0Eo`03
0?oo02_o000fo`030?oo01co00<0ool0>Ol00`3oo`16o`030?oo01Go0P0/o`00=Ol00`3oo`0Mo`03
0?oo03Wo00<0ool0AOl00`3oo`0Go`030?oo02[o000eo`030?oo01go0P0jo`030?oo04Go00<0ool0
5ol00`3oo`0Zo`00=?l00`3oo`0Oo`030?oo03So00<0ool0A?l00`3oo`0Ho`030?oo02[o000do`03
0?oo01ko0P0jo`030?oo04Co00<0ool06?l00`3oo`0Zo`00<ol00`3oo`0Oo`030?oo03Wo00<0ool0
@ol00`3oo`0Io`030?oo02[o000co`030?oo01oo00<0ool0>Ol00`3oo`13o`030?oo01So00<0ool0
:ol003;o00<0ool08?l00`3oo`0io`030?oo04;o00<0ool06Ol00`3oo`0[o`00<_l00`3oo`0Po`03
0?oo02Go0P04o`800ol400Go00<0ool0@_l00`3oo`0Io`030?oo02_o000bo`030?oo023o00<0ool0
9?l0103oo`08o`030?oo00Ko00<0ool0@Ol00`3oo`0Jo`030?oo02_o000ao`030?oo023o00<0ool0
9Ol0103oo`09o`030?oo00Go0P11o`030?oo01_o00<0ool0:ol0037o00<0ool08?l00`3oo`0Oo`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@?l00`3oo`0Ko`030?oo02_o000ao`030?oo01oo00<0ool0
9_l0103oo`08o`040?oo00Go00<0ool0?ol00`3oo`0Lo`030?oo02_o000`o`030?oo023o00<0ool0
9ol200[o0P06o`030?oo03oo00<0ool07?l00`3oo`0[o`00<?l00`3oo`0Oo`030?oo03co00<0ool0
?_l00`3oo`0Mo`030?oo02_o000`o`030?oo01ko00<0ool0?Ol00`3oo`0mo`030?oo01ko00<0ool0
:ol0033o00<0ool07_l00`3oo`0mo`030?oo03go00<0ool07Ol00`3oo`0/o`001ol500;o0P04o`80
6_l00`3oo`06o`D04ol00`3oo`0<o`801?l200Co0P0So`030?oo02;o0P04o`801?l200co00<0ool0
5?l500Go00<0ool06_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Io`030?oo00So00<0ool0
4ol00`3oo`0;o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool02ol00`3oo`0Fo`03
0?oo00Go00<0ool07?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05ol00`3oo`08o`03
0?oo01?o00<0ool02ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00So00<0ool0
5ol00`3oo`04o`030?oo01go00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806ol00`3o
000300Co00<0ool04ol00`3oo`05o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0:o`03
0?oo01So00<0ool01?l00`3oo`0Mo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01[o00<0ool0
1ol00`3oo`0Do`030?oo00[o00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`09o`03
0?oo01So00<0ool01?l00`3oo`0Mo`030?oo00So00<0ool00_l000So0P0:o`<06ol00`3oo`05o`80
5ol00`3oo`0:o`802_l302;o00<0ool08_l200[o0`08o`030?oo01So0P05o`030?oo01go0P0:o`<0
0_l003;o00<0ool07ol00`3oo`0jo`030?oo03Wo00<0ool07ol00`3oo`0^o`00<ol00`3oo`0Oo`03
0?oo03Wo00<0ool0>?l00`3oo`0Po`030?oo02ko000co`030?oo023o00<0ool0>?l00`3oo`0go`03
0?oo023o00<0ool0;ol003?o00<0ool08Ol203So00<0ool0=_l00`3oo`0Qo`030?oo02oo000do`03
0?oo02;o0P0fo`030?oo03Go00<0ool08Ol00`3oo`0`o`003?oo00`02Ol000co00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`05o`030?oo00Ko
00<0o`000P04o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`03
0?oo02Oo00<0o`009ol30003o`3o02_o00<0ool0:Ol00`3oo`06o`030?oo023o0P0[o`030?oo00Oo
000go`030?oo02Wo0P0/o`030?oo037o00<0ool08?l203Ko000ho`030?oo02[o0`0Yo`030?oo02oo
0P0Ro`030?oo03Ko000io`030?oo02co0`0Vo`030?oo02go0P0Ro`80>Ol003[o00<0ool0;_l302?o
00<0ool0;?l00`3oo`0Qo`030?oo03Wo000ko`80<Ol3023o00<0ool0:ol00`3oo`0Po`80??l003go
0P0bo`807_l00`3oo`0Yo`808_l00`3oo`0lo`00?ol203;o100Jo`030?oo02Ko0`0Ro`80?ol0047o
0P0do`@05_l00`3oo`0Oo`L08ol2047o0013o`80=_l201Co00<0ool06ol402So0P13o`00AOl203Ko
100@o`030?oo01Oo100Yo`<0AOl004Oo0P0ho`@03?l00`3oo`0Co`@07ol?04Oo0019o`80>_l400So
00<0ool04?l3023o0`1Fo`00Bol203co0`05o`030?oo00go0`0Po`<0FOl004go0P0^o`801?l200?o
1`02o`030?oo00[o0`0Ro`030?oo05[o001?o`80:ol0103oo`08o`030?oo00Co3`0To`030?oo05_o
001Ao`80:Ol0103oo`09o`030?oo00Go0P0]o`80G_l005?o0P0Wo`040?oo00[o00<0ool01?l00`3o
o`0Zo`80H?l005Go0`0To`040?oo00So00@0ool01Ol00`3oo`0Yo`030?oo063o001Ho`808ol200[o
0P06o`030?oo02So00<0ool0HOl005[o0`0do`030?oo02Oo00<0ool0H_l005go0`0ao`030?oo02Ko
00<0ool0Hol0063o0`0^o`030?oo02Co0P1Vo`00Hol202co00<0ool08_l206So001Uo`@0:?l00`3o
o`0Oo`<0J_l006Wo1@0So`030?oo01[o1@1]o`00K_l501ko00<0ool05ol307;o001co`D06Ol00`3o
o`0Do`<0MOl007So1@0Do`030?oo013o101ho`00OOl4013o00<0ool03Ol307co0021o`803_l00`3o
o`0:o`<0Ool007go0P04o`H000Co00001?l00`3oo`05o`D0P_l007co00@0ool02Ol@08Oo001lo`04
0?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`00
1Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`04
0?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0
S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86152 .1178 m
.8597 .11449 L
.85766 .11145 L
.85476 .10792 L
.85166 .10485 L
.84399 .09928 L
.83908 .09677 L
.83408 .09485 L
.83134 .09403 L
.82872 .0934 L
.82595 .09287 L
.82437 .09264 L
.82287 .09246 L
.82161 .09234 L
.82023 .09225 L
.81898 .0922 L
.81782 .09217 L
.81646 .09218 L
.81523 .09221 L
.81456 .09224 L
.81383 .09229 L
.81252 .09239 L
.81111 .09255 L
.80982 .09273 L
.80696 .09326 L
.80429 .09393 L
.80159 .09482 L
.79924 .09579 L
.79717 .09683 L
.79372 .09905 L
.79107 .10138 L
.78871 .1042 L
.78773 .10572 L
.78682 .10741 L
.78616 .10891 L
.78554 .11061 L
.78466 .11378 L
.78429 .11557 L
.784 .11727 L
.7831 .12386 L
.78255 .12786 L
.78176 .13239 L
.78072 .13687 L
.77945 .14116 L
.77613 .14972 L
.77137 .1593 L
.74757 .19777 L
.7365 .21629 L
.72671 .23184 L
.71386 .24796 L
Mistroke
.69815 .26311 L
.68923 .27018 L
.67939 .27682 L
.67005 .28201 L
.66145 .28583 L
.65137 .28924 L
.64039 .29194 L
.61995 .29572 L
.57832 .30643 L
.55727 .31328 L
.54801 .31579 L
.54309 .31688 L
.53782 .3178 L
.53512 .31818 L
.53212 .31851 L
.5308 .31862 L
.52938 .31873 L
.52805 .31881 L
.52681 .31886 L
.52553 .31891 L
.52413 .31893 L
.52277 .31894 L
.52131 .31892 L
.51995 .31889 L
.51866 .31884 L
.51751 .31878 L
.51626 .3187 L
.51344 .31847 L
.51035 .31812 L
.50411 .31715 L
.49837 .31597 L
.49305 .31464 L
.45375 .30192 L
.4159 .29169 L
.37629 .27693 L
.34197 .26525 L
.3257 .26108 L
.3156 .25765 L
.30529 .25314 L
.28548 .24163 L
.27591 .23461 L
.26715 .22679 L
.26392 .22322 L
.26121 .21955 L
.26019 .21783 L
.25941 .21623 L
.25886 .21483 L
.25842 .21334 L
.25825 .21254 L
.25814 .2118 L
Mistroke
.25809 .21137 L
.25806 .21099 L
.25803 .21054 L
.25803 .21012 L
.25803 .20972 L
.25805 .20929 L
.25811 .20853 L
.25817 .20806 L
.25824 .20762 L
.25841 .2068 L
.25883 .20532 L
.25932 .20396 L
.26079 .20083 L
.26436 .19453 L
.26649 .19069 L
.26841 .18675 L
.27003 .18282 L
.27152 .17824 L
.27258 .17384 L
.2733 .16962 L
.27357 .16748 L
.27379 .16516 L
.27387 .16408 L
.27394 .16295 L
.274 .16189 L
.27404 .16093 L
.27408 .15978 L
.2741 .15856 L
.27412 .15743 L
.27412 .15637 L
.27411 .15514 L
.2741 .15402 L
.27408 .15281 L
.27405 .15153 L
.27389 .14678 L
.27381 .14458 L
.27374 .14264 L
.27372 .14177 L
.2737 .14098 L
.27369 .14009 L
.27368 .1393 L
.27368 .13846 L
.27369 .13771 L
.27371 .13692 L
.27374 .13613 L
.27377 .13546 L
.27382 .13479 L
.27393 .13365 L
.27401 .133 L
.2741 .13243 L
Mistroke
.2743 .13145 L
.27454 .13054 L
.27479 .12981 L
.27508 .12914 L
.27541 .12856 L
.27596 .12784 L
.27636 .12746 L
.27647 .12737 L
.27651 .12734 L
.27653 .12731 L
.27653 .12729 L
.27651 .12728 L
.27647 .12727 L
.2764 .12727 L
.2763 .12727 L
.27616 .12727 L
.27598 .12727 L
.27577 .12727 L
.27554 .12726 L
.2753 .12726 L
.27498 .12724 L
.27465 .12723 L
.27376 .12716 L
.27275 .12706 L
.2698 .12665 L
.26265 .1253 L
.2522 .12292 L
.23924 .11977 L
.22431 .11633 L
.21811 .11508 L
.21164 .11395 L
.20804 .11344 L
.20486 .11307 L
.20337 .11292 L
.20178 .1128 L
.2003 .1127 L
.19896 .11264 L
.19759 .1126 L
.19614 .11259 L
.19481 .11262 L
.19362 .11266 L
.19227 .11274 L
.19083 .11288 L
.18951 .11304 L
.18833 .11323 L
.18585 .11376 L
.18376 .11437 L
.1815 .11525 L
.17954 .11624 L
.17769 .11742 L
Mistroke
.17588 .11882 L
.17301 .12176 L
.17033 .12546 L
.16774 .13008 L
.15883 .15046 L
.14959 .16986 L
.14512 .17908 L
.14305 .18383 L
.1412 .18881 L
.13995 .19314 L
.13941 .1956 L
.13902 .19791 L
.13888 .19894 L
.13876 .20003 L
.13868 .20095 L
.13862 .20197 L
.13857 .20307 L
.13855 .20425 L
.13856 .20537 L
.1386 .20638 L
.13866 .20751 L
.13874 .20857 L
.13886 .20973 L
.13902 .21098 L
.13944 .21353 L
.13994 .21588 L
.14117 .22043 L
.14264 .22477 L
.14678 .23435 L
.15227 .24447 L
.16392 .26079 L
.17058 .26762 L
.17868 .27405 L
.19484 .28321 L
.21385 .29202 L
.25052 .31105 L
.29053 .32917 L
.32931 .34378 L
.36679 .35447 L
.40636 .36259 L
.42521 .36768 L
.44629 .37359 L
.4575 .37614 L
.47064 .37845 L
.47639 .37924 L
.48275 .37995 L
.48645 .3803 L
.48986 .38057 L
.49339 .3808 L
.49659 .38096 L
Mistroke
.49823 .38103 L
.49999 .38109 L
.50166 .38114 L
.50318 .38116 L
.5048 .38118 L
.50657 .38119 L
.50746 .38119 L
.50841 .38118 L
.51012 .38116 L
.5117 .38112 L
.51342 .38107 L
.51502 .38101 L
.5165 .38093 L
.51926 .38077 L
.52215 .38054 L
.52849 .37985 L
.5342 .37898 L
.54017 .37778 L
.54631 .37624 L
.56621 .36947 L
.57667 .36562 L
.58841 .36186 L
.61153 .35641 L
.6227 .35415 L
.6325 .35181 L
.64262 .34839 L
.64757 .34609 L
.65258 .34322 L
.66932 .32968 L
.67808 .32166 L
.68706 .31428 L
.69576 .30849 L
.70586 .3036 L
.71126 .30176 L
.71426 .30095 L
.71709 .30032 L
.71975 .29984 L
.72262 .29943 L
.72523 .29915 L
.72762 .29897 L
.72897 .29889 L
.73042 .29883 L
.73179 .29879 L
.73303 .29876 L
.73448 .29875 L
.73529 .29875 L
.73604 .29875 L
.73742 .29877 L
.73811 .29878 L
.73888 .29879 L
Mistroke
.74482 .29894 L
.74622 .29898 L
.74771 .29901 L
.74911 .29904 L
.75037 .29906 L
.75163 .29906 L
.75299 .29906 L
.75426 .29905 L
.75543 .29902 L
.7567 .29898 L
.75808 .29891 L
.75953 .29882 L
.76087 .29871 L
.76326 .29846 L
.76543 .29815 L
.77028 .2972 L
.77296 .2965 L
.77544 .29575 L
.78105 .29365 L
.79122 .28859 L
.8109 .27584 L
.82723 .26421 L
.83966 .25426 L
.84528 .24809 L
.84971 .24129 L
.85374 .23254 L
.85702 .2228 L
.86012 .2106 L
.86282 .19654 L
.86476 .18305 L
.86556 .17576 L
.86613 .16938 L
.8665 .16413 L
.86678 .15873 L
.86687 .15617 L
.86691 .15479 L
.86694 .15352 L
.86697 .15236 L
.86697 .15171 L
.86698 .15112 L
.86699 .15009 L
.86699 .14901 L
.86699 .14794 L
.86698 .14734 L
.86698 .14679 L
.86697 .14621 L
.86696 .14568 L
.86693 .14453 L
.8669 .14355 L
.86686 .14251 L
Mistroke
.86677 .14066 L
.86663 .13855 L
.86644 .13636 L
.86623 .1345 L
.86596 .13252 L
.86533 .12902 L
.86453 .12575 L
.86327 .12184 L
.86173 .11825 L
.86152 .1178 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`1=o`P0=ol0097o00<0ool0C?l200Oo0`0do`00
TOl00`3oo`1:o`803?l203;o002Ao`030?oo04Wo00<0ool03_l00`3oo`0_o`00TOl00`3oo`19o`03
0?oo00oo00<0ool0;_l003oo0`0ko`801?l200Go0`04o`030?oo04So00<0ool04Ol00`3oo`0]o`00
>ol400;o2@0bo`040?oo00_o00<0ool00ol00`3oo`18o`030?oo017o00<0ool0;Ol003[o0P0>o`@0
;_l0103oo`08o`D01?l204Wo00<0ool04_l00`3oo`0/o`00>_l00`3oo`0Ao`<09Ol400;o00@0ool0
2?l0103oo`05o`030?oo04So00<0ool04_l00`3oo`0/o`00>Ol00`3oo`0Eo`D09_l0103oo`09o`03
0?l000Go00<0ool0B?l00`3oo`0Bo`030?oo02co000ho`030?oo01Wo0P0Wo`802ol200Go00<0ool0
B?l00`3oo`0Co`030?oo02_o000ho`030?oo01Wo00<0ool0>_l00`3oo`17o`030?oo01Co00<0ool0
:ol003Oo00<0ool06_l00`3oo`0jo`030?oo04Oo00<0ool05?l00`3oo`0[o`00=ol00`3oo`0Jo`03
0?oo03[o00<0ool0Aol00`3oo`0Do`030?oo02_o000go`030?oo01[o00<0ool0>_l00`3oo`16o`03
0?oo01Go00<0ool0:ol003Ko00<0ool06ol00`3oo`0jo`030?oo04Ko00<0ool05Ol00`3oo`0[o`00
=_l00`3oo`0Ko`030?oo03[o00<0ool0A_l00`3oo`0Eo`030?oo02_o000eo`030?oo01co00<0ool0
>_l00`3oo`15o`030?oo01Ko00<0ool0:ol003Go00<0ool07?l00`3oo`0jo`030?oo04Go00<0ool0
5_l00`3oo`0[o`00=?l00`3oo`0Mo`030?oo03[o00<0ool0A?l00`3oo`0Go`030?oo02_o000do`03
0?oo01go00<0ool0>_l00`3oo`14o`030?oo01Oo00<0ool0:ol003?o00<0ool07_l00`3oo`0jo`03
0?oo04?o00<0ool06?l00`3oo`0[o`00<_l00`3oo`0Oo`030?oo02Ko0P04o`800ol400Go00<0ool0
@_l00`3oo`0Io`030?oo02_o000bo`030?oo01oo00<0ool09Ol0103oo`08o`030?oo00Ko00<0ool0
@Ol00`3oo`0Jo`030?oo02_o000bo`030?oo01ko00<0ool09_l0103oo`09o`030?oo00Go0P12o`03
0?oo01[o00<0ool0:ol0037o00<0ool07ol00`3oo`0Po`@00_l0103oo`0:o`030?oo00Co00<0ool0
@?l00`3oo`0Ko`030?oo02_o000ao`030?oo01ko00<0ool09ol0103oo`08o`040?oo00Go00<0ool0
?ol00`3oo`0Lo`030?oo02_o000ao`030?oo01ko00<0ool0:?l200[o0P06o`030?oo03oo00<0ool0
6ol00`3oo`0/o`00<?l2023o00<0ool0??l00`3oo`0no`030?oo01co00<0ool0;?l0033o00<0ool0
7_l00`3oo`0mo`030?oo03go00<0ool07Ol00`3oo`0/o`00<?l00`3oo`0Mo`030?oo03ko00<0ool0
?Ol00`3oo`0Mo`030?oo02co0007o`D00_l200Co0P0Jo`030?oo00Ko1@0Bo`030?oo00go0P04o`80
1?l202?o00<0ool08_l200Co0P04o`803?l00`3oo`0Do`D01?l00`3oo`0Ko`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Wo0P09o`030?oo01;o00<0ool03?l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00_o00<0ool05_l00`3oo`04o`030?oo01go00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01So00<0ool01ol00`3oo`0Bo`803Ol0103oo`0;o`030?oo023o00<0ool0
8Ol0103oo`0;o`030?oo00So00<0ool05ol00`3oo`04o`030?oo01go00<0ool02_l1007o0Ol00003
o`0000801?l00`3oo`08o`806ol00`3o000300Co00<0ool04ol00`3oo`05o`@00_l0103oo`09o`80
8ol00`3oo`0Qo`040?oo00Wo0P0:o`030?oo01So00<0ool00ol00`3oo`0No`030?oo00So0P03o`00
2Ol00`3oo`08o`030?oo01[o00<0ool01ol00`3oo`0Do`030?oo00[o00@0ool02Ol00`3oo`0Ro`03
0?oo027o00@0ool02Ol00`3oo`09o`030?oo01So00<0ool00ol00`3oo`0No`030?oo00So00<0ool0
0_l000So0P0:o`<06ol00`3oo`05o`805ol00`3oo`0:o`802_l302;o00<0ool08_l200[o0`08o`03
0?oo01So0P05o`030?oo01go0P0:o`<00_l003;o00<0ool07ol00`3oo`0jo`030?oo03Wo00<0ool0
7ol00`3oo`0^o`00<ol00`3oo`0Oo`030?oo03Wo00<0ool0>?l00`3oo`0Po`030?oo02ko000co`03
0?oo023o00<0ool0>?l00`3oo`0go`030?oo023o00<0ool0;ol003Co00<0ool08?l203So00<0ool0
=_l00`3oo`0Po`80<Ol003Co00<0ool08_l203Ko00<0ool0=Ol00`3oo`0Po`030?oo037o000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Co00<0o`002?l00`3o
o`06o`030?oo00Go00<0ool01_l0103o0005o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l200Oo
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Oo000<o`030?oo02Oo00<0o`009_l200?o00<0ool0:_l00`3oo`0Yo`030?oo00Ko
00<0ool08?l202_o00<0ool01ol003Oo00<0ool09ol402co00<0ool0<Ol00`3oo`0Qo`030?oo03Co
000ho`030?oo02[o0`0Yo`030?oo033o00<0ool08Ol00`3oo`0eo`00>Ol00`3oo`0/o`@09Ol00`3o
o`0_o`030?oo023o0P0ho`00>_l2033o0`0Ro`030?oo02go0P0Qo`80>_l003co00<0ool0<?l301oo
00<0ool0;?l00`3oo`0Po`030?oo03[o000mo`80<ol201go00<0ool0:ol00`3oo`0Oo`80?Ol003oo
0P0co`<06_l00`3oo`0Yo`808?l203oo0011o`80=?l201So00<0ool09?l5027o00<0ool0?ol004?o
0P0do`<05Ol00`3oo`0Po`@09?l204;o0015o`80=Ol4017o00<0ool06ol502Go0`14o`00Aol203Oo
0`0>o`030?oo01Oo100Ko`l0Aol004Wo0P0ho`<02ol00`3oo`0Co`@07?l305Ko001;o`80>Ol400Oo
00<0ool04?l301oo00<0ool0Eol004go0`0]o`801?l200?o1P03o`030?oo00ko0P0Po`80F_l0053o
0P0Zo`040?oo00So00<0ool00ol20003o`3o00[o1@0Po`80G?l005;o0P0Xo`040?oo00Wo00<0ool0
1?l=02Co00<0ool0G?l005Co0P0Vo`040?oo00[o00<0ool01?l00`3oo`0/o`030?oo05go001Fo`80
9?l0103oo`08o`040?oo00Go00<0ool0:ol00`3oo`1No`00F?l302;o0P0:o`801_l00`3oo`0Zo`03
0?oo05oo001Ko`80=?l00`3oo`0Yo`030?oo063o001Mo`<0<Ol00`3oo`0Wo`80Hol0063o0P0_o`03
0?oo02Ko00<0ool0Hol006;o0`0/o`030?oo02Go00<0ool0I?l006Go0`0Yo`030?oo027o101Wo`00
J?l402Go00<0ool07ol206_o001/o`@08Ol00`3oo`0Io`H0KOl0073o1P0Ko`030?oo01Ko0`1co`00
M_l501Ko00<0ool04ol307Ko001ko`@04_l00`3oo`0@o`<0NOl007oo0P0@o`030?oo00go0`1lo`00
OOl200;o1005o`<01?l00`3oo`0;o`80Ool007co00@0ool01?l800Go00<0ool01Ol6087o001lo`04
0?oo00So4@27o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`03
0?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
0ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool0
1Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00
OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 484},
WindowMargins->{{Automatic, 58}, {Automatic, 69}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 139, 4, 375, "Title"],

Cell[CellGroupData[{
Cell[1903, 59, 32, 0, 53, "Section"],
Cell[1938, 61, 345, 9, 52, "Text"],
Cell[2286, 72, 320, 6, 52, "Text"],
Cell[2609, 80, 304, 6, 52, "Text"],
Cell[2916, 88, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3149, 97, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3331, 105, 48, 0, 45, "Subsection"],
Cell[3382, 107, 67, 1, 27, "Input"],
Cell[3452, 110, 103, 2, 27, "Input"],
Cell[3558, 114, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3723, 121, 46, 0, 29, "Subsection"],
Cell[3772, 123, 326, 7, 50, "Text"],
Cell[4101, 132, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4548, 144, 103, 2, 29, "Subsection"],
Cell[4654, 148, 105, 3, 32, "Text"],
Cell[4762, 153, 244, 6, 42, "Input"],
Cell[5009, 161, 123, 2, 27, "Input"],
Cell[5135, 165, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5418, 175, 78, 0, 42, "Subsubsection"],
Cell[5499, 177, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5669, 185, 60, 0, 29, "Subsection"],
Cell[5732, 187, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[5989, 195, 130, 3, 46, "Subsection"],
Cell[6122, 200, 121, 2, 27, "Input"],
Cell[6246, 204, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6427, 212, 31, 0, 33, "Section"],
Cell[6461, 214, 101, 3, 33, "Text"],
Cell[6565, 219, 202, 3, 30, "Input"],
Cell[6770, 224, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6961, 231, 44, 0, 33, "Section"],
Cell[7008, 233, 66, 0, 33, "Text"],
Cell[7077, 235, 1929, 30, 270, "Input"],
Cell[9009, 267, 1955, 32, 290, "Input"],
Cell[10967, 301, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11068, 307, 52, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11145, 311, 80, 1, 30, "Input"],
Cell[11228, 314, 14601, 644, 152, 7832, 556, "GraphicsData", "PostScript", \
"Graphics"],
Cell[25832, 960, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26011, 969, 35, 0, 53, "Section"],
Cell[26049, 971, 176, 4, 52, "Text"],
Cell[26228, 977, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[26714, 988, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[26881, 995, 17260, 825, 153, 10688, 738, "GraphicsData", "PostScript", \
"Graphics"],
Cell[44144, 1822, 16924, 807, 152, 10387, 720, "GraphicsData", "PostScript", \
"Graphics"],
Cell[61071, 2631, 16873, 800, 152, 10276, 713, "GraphicsData", "PostScript", \
"Graphics"],
Cell[77947, 3433, 16914, 800, 20, 10280, 712, "GraphicsData", "PostScript", \
"Graphics"],
Cell[94864, 4235, 16403, 769, 20, 9793, 681, "GraphicsData", "PostScript", \
"Graphics"],
Cell[111270, 5006, 15552, 711, 20, 8886, 623, "GraphicsData", "PostScript", \
"Graphics"],
Cell[126825, 5719, 14914, 669, 20, 8207, 580, "GraphicsData", "PostScript", \
"Graphics"],
Cell[141742, 6390, 14683, 650, 20, 7887, 560, "GraphicsData", "PostScript", \
"Graphics"],
Cell[156428, 7042, 14660, 646, 20, 7832, 556, "GraphicsData", "PostScript", \
"Graphics"],
Cell[171091, 7690, 14896, 659, 20, 8003, 568, "GraphicsData", "PostScript", \
"Graphics"],
Cell[185990, 8351, 15191, 673, 20, 8225, 581, "GraphicsData", "PostScript", \
"Graphics"],
Cell[201184, 9026, 15345, 680, 20, 8314, 587, "GraphicsData", "PostScript", \
"Graphics"],
Cell[216532, 9708, 15262, 670, 20, 8150, 576, "GraphicsData", "PostScript", \
"Graphics"],
Cell[231797, 10380, 15512, 684, 20, 8364, 590, "GraphicsData", "PostScript", \
"Graphics"],
Cell[247312, 11066, 15780, 696, 20, 8539, 601, "GraphicsData", "PostScript", \
"Graphics"],
Cell[263095, 11764, 16179, 717, 20, 8849, 621, "GraphicsData", "PostScript", \
"Graphics"],
Cell[279277, 12483, 16387, 725, 20, 8972, 628, "GraphicsData", "PostScript", \
"Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

