(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    358632,      16840]*)
(*NotebookOutlinePosition[    359287,      16863]*)
(*  CellTagsIndexPosition[    359243,      16859]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Proj.of Females, Ortho to Males, for Major Axis Rescaled, \
Schizo Population, on Sph Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {0.99999998988257, \(-0.00000000004249\), \
\(-0.00000000815207\), \(-0.14890885667999\), 
          0.04631615579661, \(-0.06867955424860\), \(-0.23594605970587\), 
          0.01859252456040, 0.02438649565581, 
          0.03413432177527, \(-0.00469090595776\), \(-0.08295111916888\), 
          0.00533664633423, \(-0.07524301148546\), \(-0.04136179149529\), \
\(-0.01210875268342\), 0.02309310511489, 
          0.01833272513018, \(-0.00846486752268\), \(-0.04392924956181\), \
\(-0.01098615476735\), \(-0.03598997698790\), \(-0.01058517671750\), \
\(-0.01401422746211\), 0.02257020063291, 0.00033072899653, 
          0.00023590085341, \(-0.00869156702959\), \(-0.00345348275585\), 
          0.00030597192151, \(-0.00137193187445\), \(-0.00667969066486\), 
          0.00904127017269, \(-0.00655340249989\), 0.00501447359205, 
          0.00258552073912, 0.00648801952449, 0.00823373129342, 
          0.00043275898306, \(-0.00298887850287\), \(-0.00332827922859\), \
\(-0.00044356465178\), 0.00196544293964, 0.00124098121236, 0.00592167003307, 
          0.00226352247340, \(-0.00014285700056\), \(-0.00058604947765\), \
\(-0.00428359723348\), 0.00503898752002, 0.00075457255761, 0.00107280526054, 
          0.00042732990809, \(-0.00194148383409\), \(-0.00026021270502\), \
\(-0.00015924393146\), \(-0.00055896473400\), 0.00357139004531, 
          0.00018521478115, 
          0.00014204963106, \(-0.00227702206002\), \(-0.00142480374493\), \
\(-0.00062619308861\), \(-0.00062940968098\), 
          0.00077624011619, \(-0.00019058121174\), 
          0.00041251634176, \(-0.00069431013873\), \(-0.00088148948557\), 
          0.00021433450379, \(-0.00037807112229\), \(-0.00026326849121\), 
          0.00019032930334, \(-0.00108451226204\), 
          0.00002689323115, \(-0.00072626836349\), 0.00058142750668, 
          0.00047562210024, 
          0.00043195050971, \(-0.00011112357174\)};\)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(axis\  = \ {0.00000580421318, 0.00000063283691, 0.00000266569574, 
          0.06191757218265, \(-0.23036285244482\), \(-0.09425388078541\), \
\(-0.08348880216425\), \(-0.09970476813023\), 0.21003647206820, 
          0.06982091874489, 0.13999207726686, 0.15103396307198, 
          0.15031573284590, \(-0.01537792114230\), \(-0.08555906752556\), 
          0.01240356272939, 0.05231352357553, \(-0.05112768505175\), 
          0.03035515302992, \(-0.07836326900063\), 
          0.13125990402343, \(-0.04310087124098\), 
          0.14428799383809, \(-0.18156732503443\), \(-0.02698087222507\), 
          0.04165978688936, 0.15043996431348, 0.24919154864383, 
          0.06740249410363, 
          0.22788289116638, \(-0.18344814434978\), \(-0.21903078720913\), \
\(-0.04856512463805\), \(-0.00262334688644\), \(-0.04499630935318\), 
          0.43426329037237, \(-0.00310463263787\), \(-0.04455108064842\), 
          0.00394471832423, \(-0.12453471357998\), 
          0.04876663905111, \(-0.05428706536103\), \(-0.05755996652706\), \
\(-0.11258046178048\), 0.10894889784346, \(-0.12043170578442\), 
          0.16976261577114, 0.01908354000589, 
          0.01162295114694, \(-0.06797862242406\), 0.06888038691709, 
          0.09259171499657, 0.14544939424741, \(-0.08756622977094\), 
          0.04673652944153, \(-0.01709549646658\), 
          0.10367493668457, \(-0.05029871627848\), \(-0.03432265574094\), \
\(-0.09526792332072\), \(-0.01308567666581\), \(-0.09854620491074\), 
          0.04096730895577, \(-0.16085883124028\), \(-0.01524967655959\), \
\(-0.04810142664946\), \(-0.14594417235688\), 
          0.09519451012774, \(-0.03309318273081\), 
          0.00268459254321, \(-0.04055543053180\), \(-0.08142501511506\), \
\(-0.05476589401771\), 0.06594181245148, 0.05868658651286, 
          0.08110599648398, \(-0.02746592338270\), \(-0.10842648499120\), \
\(-0.10677875950918\), 0.06120640610578};\)\)], "Input"],

Cell[BoxData[
    \(General::"spell1" \(\(:\)\(\ \)\) 
      "Possible spelling error: new symbol name \"\!\(axis\)\" is similar to \
existing symbol \"\!\(Axis\)\"."\)], "Message"]
}, Open  ]],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the L1 median", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87312 .15188 m
.86926 .14386 L
.86393 .13584 L
.8608 .13219 L
.85712 .12864 L
.84968 .12322 L
.845 .12076 L
.8396 .11859 L
.83702 .11778 L
.83423 .11705 L
.83177 .11652 L
.82908 .11605 L
.82645 .1157 L
.82402 .11547 L
.82284 .11538 L
.82155 .11532 L
.82034 .11527 L
.8192 .11525 L
.81851 .11524 L
.81776 .11525 L
.8164 .11527 L
.81508 .11531 L
.81386 .11537 L
.81254 .11546 L
.8111 .11558 L
.8082 .11591 L
.80561 .1163 L
.80282 .11683 L
.79788 .11805 L
.79326 .11957 L
.78919 .12126 L
.78093 .12614 L
.77732 .12914 L
.77384 .13287 L
.77123 .13646 L
.7689 .14056 L
.76722 .1443 L
.7657 .14863 L
.76437 .15353 L
.76322 .15898 L
.76158 .16914 L
.75992 .17956 L
.75875 .18531 L
.75741 .19047 L
.75414 .19945 L
.75203 .2037 L
.74934 .20814 L
.74329 .21582 L
.73678 .22219 L
.71999 .23483 L
Mistroke
.70239 .24567 L
.6858 .25422 L
.66896 .2608 L
.64987 .26611 L
.60818 .27454 L
.58735 .27803 L
.56776 .28057 L
.54812 .2825 L
.53806 .28333 L
.52734 .28411 L
.52185 .28446 L
.51668 .28474 L
.51167 .28497 L
.50947 .28505 L
.50705 .28513 L
.50495 .28519 L
.50268 .28523 L
.50144 .28525 L
.5003 .28527 L
.49921 .28527 L
.49806 .28528 L
.49689 .28528 L
.49564 .28528 L
.49446 .28527 L
.49339 .28527 L
.49213 .28525 L
.49076 .28523 L
.48828 .28517 L
.48586 .2851 L
.48325 .285 L
.47853 .28477 L
.46777 .28402 L
.44607 .2819 L
.4262 .27964 L
.40523 .2768 L
.38512 .27297 L
.36631 .26808 L
.3297 .25555 L
.31263 .24816 L
.29698 .23985 L
.28116 .22874 L
.2744 .22208 L
.27171 .21859 L
.27053 .21672 L
.26945 .21468 L
.26863 .21275 L
.26834 .2119 L
.26806 .21096 L
.26786 .21013 L
.26769 .20924 L
Mistroke
.26762 .20875 L
.26756 .2083 L
.26748 .20743 L
.26745 .20658 L
.26746 .2058 L
.26751 .20495 L
.26762 .20404 L
.26776 .20317 L
.26794 .20236 L
.26838 .20084 L
.26907 .19905 L
.27 .19718 L
.27214 .19385 L
.27885 .18647 L
.29392 .17344 L
.29962 .16794 L
.30201 .16509 L
.30396 .16214 L
.30469 .16066 L
.30501 .15988 L
.30529 .15906 L
.30549 .15831 L
.30562 .15763 L
.30571 .15698 L
.30576 .15631 L
.30576 .15568 L
.30572 .15502 L
.30563 .1544 L
.30552 .15384 L
.30534 .1532 L
.30509 .15251 L
.30449 .15129 L
.30367 .15006 L
.30255 .14876 L
.30124 .14756 L
.29961 .14634 L
.29616 .14435 L
.29239 .14272 L
.2878 .14123 L
.28319 .14008 L
.27784 .1391 L
.27475 .13867 L
.27191 .13835 L
.26935 .13812 L
.26662 .13792 L
.26519 .13784 L
.26363 .13776 L
.26218 .13771 L
.26087 .13767 L
.25953 .13764 L
Mistroke
.25827 .13762 L
.25694 .13761 L
.25618 .1376 L
.25549 .1376 L
.25434 .13761 L
.25307 .13762 L
.25175 .13765 L
.25051 .13767 L
.24781 .13775 L
.24532 .13785 L
.23971 .13815 L
.22968 .13889 L
.21115 .14075 L
.20062 .14216 L
.19069 .14388 L
.18184 .1459 L
.17247 .14874 L
.16362 .15233 L
.15453 .15729 L
.146 .16365 L
.13916 .17065 L
.13627 .17445 L
.13351 .17882 L
.13144 .18286 L
.12962 .18733 L
.12834 .19147 L
.12776 .19387 L
.12732 .19616 L
.127 .19827 L
.12678 .20022 L
.12669 .20125 L
.12662 .20237 L
.12656 .20355 L
.12653 .20466 L
.12653 .20589 L
.12653 .20651 L
.12655 .20719 L
.1266 .2084 L
.12666 .20954 L
.12677 .21088 L
.12689 .2121 L
.12726 .21485 L
.12767 .21722 L
.12822 .21979 L
.12947 .22445 L
.13289 .23353 L
.137 .24162 L
.14283 .25064 L
.14909 .25849 L
.16235 .27131 L
Mistroke
.17861 .28254 L
.19507 .29081 L
.23253 .30432 L
.26816 .31585 L
.30666 .32791 L
.34417 .33806 L
.36346 .34243 L
.3738 .34442 L
.38492 .34623 L
.39491 .34757 L
.406 .34873 L
.41614 .34954 L
.42568 .35011 L
.43583 .35055 L
.44095 .35073 L
.44655 .35088 L
.45136 .35099 L
.45651 .35108 L
.45902 .35111 L
.46138 .35114 L
.46352 .35116 L
.46582 .35117 L
.46713 .35118 L
.46833 .35119 L
.46953 .35119 L
.47067 .35119 L
.472 .35119 L
.47323 .35119 L
.47458 .35119 L
.47527 .35118 L
.47602 .35118 L
.47742 .35117 L
.4787 .35116 L
.4816 .35113 L
.48415 .3511 L
.48684 .35105 L
.49164 .35094 L
.4961 .35081 L
.5062 .35039 L
.51145 .3501 L
.51713 .34973 L
.52738 .34888 L
.54641 .34669 L
.55601 .34525 L
.56652 .34337 L
.58752 .33863 L
.62628 .32655 L
.66302 .31448 L
.7036 .30483 L
.74334 .29758 L
Mistroke
.78115 .2896 L
.7999 .28436 L
.81898 .27708 L
.8279 .27256 L
.83711 .26679 L
.84463 .26104 L
.85227 .25392 L
.86452 .2386 L
.86978 .22951 L
.87371 .22075 L
.87528 .21636 L
.87673 .21153 L
.87778 .20723 L
.87866 .20262 L
.87925 .19843 L
.8795 .19603 L
.87959 .19493 L
.87968 .19376 L
.87974 .19277 L
.87979 .19168 L
.87983 .19068 L
.87986 .18975 L
.87988 .18865 L
.87989 .18765 L
.87988 .18649 L
.87987 .18539 L
.87984 .18439 L
.8798 .18332 L
.8797 .18137 L
.87953 .17905 L
.87932 .17692 L
.87871 .17228 L
.87792 .16786 L
.876 .16012 L
.87312 .15188 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l803Ko001lo`040?oo00So1@04o`80C?l300Oo0`0do`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0=o`<0<Ol007co00@0ool02Ol00`3o0005o`030?oo04Oo0P0Ao`03
0?oo02oo001mo`802ol200Go00<0ool0A_l00`3oo`0Bo`030?oo02ko002Ao`030?oo04Go00<0ool0
5?l00`3oo`0]o`00Aol@03[o00<0ool0AOl00`3oo`0Eo`030?oo02co0010o`L04?l303Oo00<0ool0
A?l00`3oo`0Go`030?oo02_o000ko`D06_l203Go00<0ool0A?l00`3oo`0Ho`030?oo02[o000io`80
8Ol00`3oo`0bo`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=ol202?o0P0co`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol202Ko00<0ool0<Ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o0P0Xo`03
0?oo037o00<0ool0@ol00`3oo`0Ko`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`03
0?oo01co00<0ool0:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02Oo00<0ool0
8?l200Co0P03o`@01Ol00`3oo`12o`030?oo01co00<0ool0:?l002oo00<0ool09_l202;o00@0ool0
2?l00`3oo`06o`030?oo04;o00<0ool07?l00`3oo`0Xo`00;_l00`3oo`0Vo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0No`030?oo02Oo000^o`030?oo02Go00<0ool07Ol400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`030?oo02Co00@0ool0
2?l0103oo`05o`030?oo047o00<0ool07_l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo02Ko0P0:o`80
1_l00`3oo`11o`030?oo01go0P0Yo`00;Ol00`3oo`0So`030?oo03_o00<0ool0@?l00`3oo`0No`03
0?oo02So000]o`030?oo02?o00<0ool0>ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool0
8ol00`3oo`0ko`030?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`805ol00`3oo`09o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool0
2Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`08o`03
0?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3o
o`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3o
o`08o`030?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Ko
0P09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00So
00<0ool06Ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201[o00<0ool01ol200[o
0`0Ro`030?oo02;o0P0:o`<02Ol00`3oo`0Go`802Ol00`3oo`0Io`802_l300;o000_o`030?oo02Go
0P0ho`030?oo03So0P0Uo`030?oo02[o000_o`030?oo02Oo00<0ool0=Ol00`3oo`0go`030?oo02Co
00<0ool0:ol0033o00<0ool09ol00`3oo`0do`030?oo03Go0P0Vo`030?oo02co000`o`030?oo02So
0P0do`030?oo03?o0P0Wo`030?oo02go000ao`030?oo02Wo0P0bo`030?oo03;o00<0ool09ol00`3o
o`0]o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00@0oooo
0`02o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02Wo0`0Zo`03
0?oo02Wo00@0o`00:?l01@3oool002So00<0ool01ol003Co00<0ool0<?l302Oo00<0ool09ol402co
0P0co`00=Ol00`3oo`0bo`809Ol00`3oo`0So`@0;ol00`3oo`0co`00=_l00`3oo`0co`@08Ol00`3o
o`0Oo`@0<_l00`3oo`0do`00=ol00`3oo`0fo`D07?l00`3oo`0Ko`@0=?l203Oo000ho`030?oo03[o
1P0Fo`030?oo01;o2@0eo`<0>Ol003Wo0P10o`/02ol00`3oo`0=o`D0??l203co000ko`030?oo04So
6`0no`<0?_l003co0P1Co`030?oo04Wo0P11o`00?_l3053o00<0ool0AOl404?o0011o`<0COl00`3o
o`10o`D0Aol004Co1019o`030?oo03_o1@1<o`00B?l304Ko00<0ool0=_l5057o001;o`80A?l00`3o
o`0ao`D0E_l004go0`0]o`801?l200?o1005o`030?oo02_o1P1Ko`00D?l202[o00@0ool02?l00`3o
o`06o`030?oo02Oo101Qo`00D_l302Oo00@0ool02Ol00`3oo`05o`809?l406Go001Eo`<09?l0103o
o`0:o`030?oo00Co00<0ool08?l306Wo001Ho`@08?l0103oo`08o`040?oo00Go00<0ool07Ol306co
001Lo`@07Ol200[o0P06o`030?oo01_o0P1_o`00H?l502co00<0ool06?l3077o001Uo`@0:?l00`3o
o`0Fo`80M?l006Wo0P0Vo`030?oo017o1@1fo`00Jol6023o00<0ool02_l707_o001ao`/05Ol00`3o
o`03o`L0P_l007co6`29o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85388 .1495 m
.85564 .14493 L
.8571 .1406 L
.8578 .13814 L
.85834 .13583 L
.85858 .13457 L
.8588 .13321 L
.85895 .13199 L
.85904 .13088 L
.85909 .1298 L
.8591 .12866 L
.85907 .12761 L
.85899 .12666 L
.85886 .12558 L
.85876 .12498 L
.85865 .12444 L
.8581 .12244 L
.8577 .12142 L
.85728 .12049 L
.85632 .11886 L
.85506 .11727 L
.85344 .11576 L
.85164 .11451 L
.84976 .11352 L
.84748 .11265 L
.84611 .11226 L
.84481 .11196 L
.8434 .1117 L
.84205 .11151 L
.84127 .11143 L
.84052 .11136 L
.83973 .11131 L
.83885 .11126 L
.83812 .11124 L
.83734 .11122 L
.83592 .11123 L
.83511 .11125 L
.83436 .11128 L
.83352 .11132 L
.83263 .11138 L
.83102 .11151 L
.82952 .11166 L
.82607 .1121 L
.81969 .1132 L
.80771 .11595 L
.7969 .11891 L
.78637 .12233 L
.78104 .12451 L
.77672 .12675 L
.77486 .12795 L
Mistroke
.77301 .12936 L
.77013 .13227 L
.76889 .134 L
.76779 .13603 L
.76736 .13702 L
.76697 .13811 L
.76668 .13909 L
.76642 .14017 L
.76622 .14127 L
.76608 .14229 L
.76598 .14336 L
.76592 .14439 L
.76591 .14562 L
.76594 .1468 L
.76602 .14812 L
.76617 .14956 L
.76637 .15101 L
.76661 .15238 L
.76731 .15555 L
.76912 .1616 L
.77537 .17625 L
.78214 .18938 L
.78557 .19608 L
.7882 .20188 L
.78912 .20432 L
.78986 .20677 L
.79015 .20799 L
.79028 .20862 L
.79039 .20929 L
.79051 .21043 L
.79056 .21146 L
.79054 .21242 L
.79043 .21344 L
.79025 .21436 L
.79002 .21519 L
.78975 .21592 L
.78938 .21669 L
.78844 .21819 L
.7871 .21969 L
.78534 .22117 L
.78128 .22358 L
.77854 .22482 L
.7757 .22591 L
.7683 .22824 L
.73638 .23543 L
.71783 .23881 L
.70952 .24007 L
.70611 .2405 L
.70254 .24088 L
.70105 .24101 L
Mistroke
.6995 .24113 L
.69819 .24121 L
.6968 .24128 L
.69528 .24133 L
.69388 .24135 L
.69256 .24134 L
.69137 .24131 L
.68998 .24125 L
.68871 .24116 L
.68761 .24105 L
.68645 .24091 L
.68397 .24051 L
.68182 .24004 L
.6739 .23723 L
.66984 .23541 L
.66594 .23377 L
.66412 .23313 L
.66213 .23255 L
.66106 .23229 L
.65986 .23206 L
.65874 .23189 L
.65767 .23177 L
.65667 .2317 L
.6556 .23167 L
.65467 .23167 L
.65364 .23172 L
.65251 .23182 L
.65127 .23198 L
.65009 .23219 L
.64901 .23243 L
.64658 .23313 L
.64395 .23412 L
.63913 .23658 L
.63387 .24019 L
.62815 .24527 L
.61859 .25671 L
.61095 .269 L
.59919 .29319 L
.59647 .29831 L
.59494 .30074 L
.59345 .30277 L
.59198 .30443 L
.59029 .30598 L
.58865 .30717 L
.58704 .30808 L
.58505 .30893 L
.58306 .30954 L
.58185 .30981 L
.58071 .31001 L
.58006 .31011 L
Mistroke
.57934 .3102 L
.57865 .31027 L
.57799 .31033 L
.57665 .31041 L
.57592 .31044 L
.57512 .31046 L
.57427 .31046 L
.57345 .31046 L
.5727 .31045 L
.57187 .31042 L
.57034 .31036 L
.56866 .31026 L
.56552 .31001 L
.55835 .3092 L
.55121 .3082 L
.5375 .30588 L
.52408 .30294 L
.51783 .30117 L
.51145 .29895 L
.50132 .29422 L
.49269 .28902 L
.48238 .28272 L
.47678 .27994 L
.47416 .27885 L
.47112 .27779 L
.46849 .27702 L
.46553 .27632 L
.46385 .276 L
.46227 .27575 L
.46075 .27555 L
.45909 .27538 L
.45737 .27525 L
.45574 .27517 L
.45378 .27514 L
.45196 .27516 L
.45003 .27523 L
.44893 .2753 L
.44791 .27538 L
.44601 .27556 L
.44398 .2758 L
.43977 .27647 L
.43572 .27731 L
.42806 .27938 L
.41128 .28567 L
.39743 .29199 L
.39237 .29422 L
.38998 .29515 L
.38767 .29593 L
.38653 .29624 L
.38595 .29638 L
Mistroke
.38535 .2965 L
.38435 .29666 L
.38346 .29672 L
.38255 .29672 L
.38174 .29662 L
.38106 .29647 L
.38036 .29623 L
.37967 .29587 L
.37897 .29538 L
.3783 .29473 L
.37772 .29401 L
.37679 .29244 L
.37633 .2914 L
.37593 .29035 L
.37472 .28594 L
.37355 .27995 L
.37078 .26511 L
.36849 .25592 L
.36559 .24727 L
.36227 .2399 L
.35788 .23269 L
.35218 .22609 L
.34899 .22336 L
.34528 .22086 L
.34197 .21915 L
.3401 .21838 L
.3383 .21775 L
.33661 .21727 L
.33475 .21684 L
.33305 .21655 L
.33147 .21634 L
.32981 .21621 L
.32806 .21614 L
.32656 .21615 L
.32493 .21622 L
.32318 .21636 L
.32221 .21647 L
.32133 .21659 L
.31804 .21717 L
.31507 .21788 L
.31196 .21878 L
.30665 .22066 L
.29674 .22463 L
.29453 .22536 L
.2934 .22567 L
.29287 .22579 L
.29239 .22588 L
.29149 .226 L
.29104 .22604 L
.29057 .22605 L
Mistroke
.28977 .22602 L
.28908 .22592 L
.28839 .22575 L
.2877 .22548 L
.28714 .22519 L
.28655 .2248 L
.28532 .22365 L
.28426 .22226 L
.2825 .21899 L
.28071 .2146 L
.27652 .20327 L
.27443 .19837 L
.27196 .19363 L
.26958 .1902 L
.26828 .18882 L
.26711 .18788 L
.2665 .18753 L
.26618 .18738 L
.26589 .18727 L
.26539 .18715 L
.26513 .18711 L
.26486 .1871 L
.26458 .18712 L
.26429 .18718 L
.26403 .18727 L
.2638 .18738 L
.2636 .1875 L
.26339 .18766 L
.26319 .18784 L
.263 .18806 L
.26267 .18854 L
.26241 .18907 L
.2622 .18967 L
.26205 .19028 L
.26198 .19068 L
.26192 .19107 L
.26186 .19185 L
.26185 .19231 L
.26186 .19274 L
.26188 .19323 L
.26192 .19376 L
.26203 .19463 L
.2622 .19558 L
.26242 .1965 L
.26266 .19735 L
.26343 .19947 L
.26441 .20146 L
.26556 .20331 L
.26682 .20493 L
.26841 .20655 L
Mistroke
.27032 .20804 L
.27135 .20866 L
.27253 .20924 L
.27369 .20969 L
.27479 .21001 L
.27592 .21023 L
.27697 .21034 L
.2782 .21037 L
.27937 .21028 L
.28041 .21012 L
.28155 .20985 L
.28275 .20947 L
.28388 .209 L
.28588 .20793 L
.28801 .20644 L
.29044 .20424 L
.29265 .20172 L
.29646 .19593 L
.29821 .19245 L
.29969 .18893 L
.30124 .18445 L
.30244 .18014 L
.3034 .1757 L
.30418 .17093 L
.30448 .16842 L
.3047 .16609 L
.3048 .16477 L
.30488 .16353 L
.30495 .16211 L
.30499 .16082 L
.30502 .15883 L
.30501 .15666 L
.30494 .15443 L
.30483 .15238 L
.30469 .1505 L
.30452 .14881 L
.30404 .14515 L
.30337 .14146 L
.30247 .13775 L
.30148 .13458 L
.30045 .13195 L
.29923 .12941 L
.29786 .12716 L
.29628 .1252 L
.29556 .12449 L
.29479 .12385 L
.29405 .12335 L
.29338 .12298 L
.2927 .12269 L
.29206 .12248 L
Mistroke
.29132 .12233 L
.29063 .12226 L
.28989 .12227 L
.28952 .12231 L
.28911 .12237 L
.28845 .12252 L
.28777 .12273 L
.28657 .12324 L
.28529 .12395 L
.28392 .12487 L
.27915 .12845 L
.27775 .12937 L
.27615 .13022 L
.27538 .13056 L
.27455 .13087 L
.2738 .13109 L
.27298 .13128 L
.27215 .13142 L
.27136 .13151 L
.27055 .13155 L
.26975 .13155 L
.26878 .1315 L
.26786 .1314 L
.2668 .13124 L
.26563 .13101 L
.26304 .13033 L
.26 .1293 L
.25377 .12675 L
.23982 .12053 L
.2325 .11755 L
.22427 .11472 L
.21943 .11339 L
.215 .11242 L
.21287 .11205 L
.21056 .11173 L
.20839 .11151 L
.2064 .11138 L
.20437 .11132 L
.2022 .11134 L
.20018 .11145 L
.19835 .11163 L
.19625 .11193 L
.19401 .11238 L
.19193 .11292 L
.19004 .11352 L
.18606 .11519 L
.18265 .11712 L
.17898 .11983 L
.17577 .12285 L
.17089 .12899 L
Mistroke
.1664 .137 L
.16261 .14652 L
.15976 .15618 L
.15585 .17441 L
.15237 .19377 L
.14839 .21248 L
.1458 .22175 L
.14322 .22936 L
.13811 .24152 L
.13589 .24625 L
.13423 .24993 L
.13313 .25286 L
.13277 .25416 L
.13263 .25484 L
.13254 .25545 L
.1325 .25607 L
.13251 .25662 L
.13255 .2571 L
.13265 .25758 L
.1328 .25808 L
.1329 .25832 L
.13303 .25857 L
.13327 .25897 L
.13359 .25937 L
.13401 .25977 L
.13446 .26012 L
.13547 .2607 L
.13613 .26098 L
.1369 .26125 L
.13845 .26163 L
.13928 .26178 L
.14023 .2619 L
.14117 .262 L
.14207 .26206 L
.14292 .2621 L
.14387 .26213 L
.1449 .26214 L
.14588 .26212 L
.14696 .26209 L
.14751 .26207 L
.14813 .26203 L
.15028 .26188 L
.15285 .26163 L
.15528 .26133 L
.16417 .25981 L
.17121 .25828 L
.17444 .25755 L
.17753 .25694 L
.17905 .25669 L
.18037 .25654 L
Mistroke
.18099 .25648 L
.18166 .25644 L
.18227 .25642 L
.18284 .25642 L
.18347 .25643 L
.18406 .25646 L
.18471 .25652 L
.1854 .2566 L
.18614 .25671 L
.18685 .25684 L
.18823 .25716 L
.19089 .25795 L
.19363 .25895 L
.20108 .26211 L
.2201 .27163 L
.23188 .27919 L
.23682 .28315 L
.24175 .28793 L
.24568 .29268 L
.24923 .29817 L
.25221 .30438 L
.25431 .31049 L
.25586 .3168 L
.25693 .32282 L
.25846 .33383 L
.25887 .33668 L
.25937 .33962 L
.26047 .34452 L
.26122 .34693 L
.26202 .34894 L
.26313 .35107 L
.26436 .35282 L
.26553 .35408 L
.26693 .35519 L
.26767 .35564 L
.26839 .35602 L
.26984 .35658 L
.27073 .35683 L
.27171 .35702 L
.27257 .35713 L
.27356 .3572 L
.27466 .35721 L
.2753 .35718 L
.27589 .35714 L
.27706 .35701 L
.2782 .35683 L
.28061 .35629 L
.28296 .3556 L
.28865 .35337 L
.30105 .34683 L
Mistroke
.32359 .33331 L
.32954 .33026 L
.33495 .32794 L
.3405 .32613 L
.34341 .32542 L
.34503 .32511 L
.34652 .32487 L
.34779 .32469 L
.34919 .32454 L
.3506 .32442 L
.35137 .32438 L
.35209 .32434 L
.35343 .3243 L
.35465 .32428 L
.35607 .3243 L
.35741 .32434 L
.35862 .3244 L
.35992 .32448 L
.36225 .32468 L
.36762 .32532 L
.3916 .32933 L
.40344 .33108 L
.40926 .33189 L
.41415 .33265 L
.41807 .33343 L
.42178 .33447 L
.42369 .33519 L
.42537 .33598 L
.42827 .33778 L
.43074 .33984 L
.43299 .34208 L
.43854 .34826 L
.44488 .35464 L
.4538 .36138 L
.45997 .36473 L
.46622 .36722 L
.46939 .36818 L
.47296 .36903 L
.47462 .36935 L
.4764 .36963 L
.47808 .36985 L
.47966 .37 L
.48139 .37012 L
.48231 .37016 L
.4833 .37019 L
.48434 .3702 L
.48533 .3702 L
.48717 .37014 L
.48889 .37003 L
.49048 .36988 L
Mistroke
.49221 .36966 L
.49404 .36938 L
.49775 .36859 L
.50115 .36764 L
.50477 .36634 L
.50852 .36468 L
.51441 .36132 L
.51997 .35717 L
.52827 .34875 L
.53534 .33951 L
.54143 .33164 L
.54423 .32868 L
.54747 .32595 L
.5493 .3247 L
.55145 .32347 L
.55561 .32169 L
.55818 .32089 L
.56068 .3203 L
.56212 .32002 L
.56375 .31976 L
.56535 .31956 L
.56685 .3194 L
.56839 .31928 L
.56988 .31919 L
.57149 .31912 L
.57328 .31908 L
.57488 .31907 L
.5766 .31909 L
.57809 .31912 L
.57976 .31918 L
.58184 .31927 L
.58385 .31939 L
.5877 .31966 L
.59665 .32051 L
.61327 .32239 L
.62138 .32323 L
.62526 .32357 L
.62875 .32382 L
.63233 .32402 L
.63418 .32409 L
.63609 .32414 L
.63703 .32415 L
.63802 .32416 L
.63979 .32415 L
.64085 .32414 L
.64185 .32411 L
.64273 .32408 L
.64368 .32404 L
.64522 .32395 L
.64664 .32384 L
Mistroke
.64978 .32351 L
.65271 .32306 L
.65573 .32246 L
.66085 .32102 L
.66518 .31934 L
.68324 .30812 L
.68683 .30574 L
.6908 .30349 L
.69314 .3024 L
.69535 .30155 L
.69752 .30091 L
.69874 .30062 L
.69988 .30041 L
.70098 .30024 L
.70203 .30013 L
.70316 .30005 L
.7044 .30002 L
.70551 .30003 L
.7067 .30009 L
.70773 .30017 L
.70887 .30031 L
.71024 .30052 L
.7115 .30076 L
.71442 .30147 L
.71991 .30332 L
.73248 .30932 L
.74488 .31626 L
.75691 .32238 L
.7602 .3237 L
.76319 .32468 L
.76578 .32532 L
.7672 .32557 L
.76847 .32573 L
.76956 .32581 L
.7707 .32583 L
.77175 .32579 L
.77269 .3257 L
.77376 .32552 L
.77489 .32525 L
.77592 .32491 L
.77684 .32454 L
.77858 .32358 L
.78037 .32223 L
.78208 .32048 L
.78354 .31854 L
.78604 .31387 L
.78724 .31081 L
.78822 .30773 L
.78987 .30098 L
.79129 .29331 L
Mistroke
.7924 .28653 L
.79361 .27964 L
.79484 .27421 L
.79566 .27139 L
.79654 .26896 L
.7977 .26643 L
.79892 .26438 L
.80055 .26235 L
.80233 .26077 L
.80334 .26008 L
.80449 .25945 L
.80554 .25898 L
.80675 .25856 L
.80752 .25834 L
.80826 .25817 L
.80897 .25803 L
.80977 .2579 L
.81051 .25781 L
.81122 .25774 L
.81198 .25769 L
.81282 .25766 L
.81359 .25765 L
.81443 .25765 L
.81597 .25771 L
.81685 .25776 L
.8177 .25783 L
.81968 .25803 L
.82316 .2585 L
.82713 .25917 L
.83508 .26069 L
.84293 .26218 L
.85102 .26347 L
.85335 .26376 L
.85555 .264 L
.85749 .26418 L
.85961 .26433 L
.86178 .26443 L
.86296 .26447 L
.86404 .26448 L
.86597 .26447 L
.86799 .2644 L
.86911 .26433 L
.87031 .26423 L
.87245 .26399 L
.87446 .26367 L
.87627 .26331 L
.87969 .26235 L
.88264 .26118 L
.88555 .25959 L
.88828 .25753 L
Mistroke
.89052 .25513 L
.89147 .2538 L
.89236 .25225 L
.89301 .25085 L
.89359 .24928 L
.89401 .24774 L
.89429 .24628 L
.89449 .24479 L
.89458 .24336 L
.89459 .24175 L
.8945 .23998 L
.89441 .23908 L
.89429 .23809 L
.89398 .23625 L
.89358 .23444 L
.89313 .23274 L
.89074 .22621 L
.88715 .2192 L
.87621 .20389 L
.86542 .19195 L
.85674 .18259 L
.85321 .17812 L
.85088 .17422 L
.85003 .17224 L
.84967 .17114 L
.84941 .17014 L
.84921 .16904 L
.84912 .16842 L
.84907 .16784 L
.84904 .16734 L
.84902 .16678 L
.84902 .16619 L
.84904 .16564 L
.84911 .16466 L
.84924 .16361 L
.84939 .16271 L
.84959 .16171 L
.8509 .15725 L
.85388 .1495 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{42, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00?ol703Oo0P04o`801Ol300Co00<0ool0Dol703;o000mo`<0
1_l303?o00@0ool02ol00`3oo`03o`030?oo04go1P07o`80<?l003_o0P0<o`80<Ol0103oo`08o`D0
1?l204co0P0>o`030?oo02ko000jo`030?oo00ko0`09o`@06ol400;o00@0ool02?l0103oo`05o`03
0?oo04Wo0P0Ao`030?oo02go000io`030?oo01;o0`06o`030?oo00808?l0103oo`09o`030?l000Go
00<0ool0A_l301?o00<0ool0;Ol003Wo00<0ool05Ol600Co00<0ool07ol200_o0P05o`030?oo04Go
0P0Eo`030?oo02go000ho`030?oo023o0P0do`030?oo04Co00<0ool05Ol00`3oo`0]o`00>?l00`3o
o`0Qo`030?oo03;o00<0ool0@ol201Oo00<0ool0;Ol003Oo00<0ool08_l00`3oo`0bo`030?oo04?o
00<0ool05Ol00`3oo`0^o`00=ol00`3oo`0Ro`030?oo03;o00<0ool0@ol00`3oo`0Eo`030?oo02ko
000go`030?oo02?o00<0ool0<Ol00`3oo`13o`030?oo01Go00<0ool0;_l003Ko00<0ool09?l00`3o
o`0ao`030?oo04?o00<0ool05Ol00`3oo`0^o`00=_l00`3oo`0To`030?oo037o00<0ool0A?l00`3o
o`0Co`030?oo02oo000fo`030?oo02Co00<0ool0<Ol00`3oo`14o`030?oo01?o00<0ool0;ol003Ko
00<0ool09?l00`3oo`0ao`030?oo04Go00<0ool04Ol00`3oo`0`o`00=Ol00`3oo`0To`80<ol00`3o
o`15o`030?oo017o00<0ool0<?l003Go00<0ool09?l00`3oo`0bo`030?oo04Ko00<0ool04?l2037o
000eo`030?oo02Co00<0ool07_l200Co0P03o`@01Ol00`3oo`16o`030?oo017o00<0ool0;ol003Go
00<0ool09?l00`3oo`0Mo`040?oo00So00<0ool01_l00`3oo`17o`030?oo013o00<0ool0;ol003Go
00<0ool09?l00`3oo`0Mo`040?oo00Wo00<0ool01Ol204So00<0ool04Ol00`3oo`0^o`00=?l00`3o
o`0Ko`802?l00`3oo`0Go`@00_l0103oo`0:o`030?oo00Co00<0ool0B?l00`3oo`0Ao`030?oo02go
000do`030?oo01[o00<0o`001ol00`3oo`0No`040?oo00So00@0ool01Ol00`3oo`18o`030?oo01;o
00<0ool0;?l003Co00<0ool06_l0103oo`06o`030?oo01oo0P0:o`801_l00`3oo`18o`030?oo01?o
00<0ool0:ol003Co00<0ool06_l01@3oool000Co00<0ool0=?l00`3oo`18o`030?oo01Co00<0ool0
:_l003Co00<0ool06ol0103oo`03o`030?oo03Go00<0ool0BOl00`3oo`0Do`030?oo02Wo000co`03
0?oo01co0P001?l0ool203Oo00<0ool0BOl201Ko00<0ool0:?l000Oo1@02o`801?l201go00<0ool0
0ol501Go1P07o`801?l200Co0P0So`030?oo02;o0P04o`801?l201[o00<0ool01_l500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Lo`030?oo00Go00<0ool06?l00`3oo`06o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool06Ol00`3oo`08o`030?oo00[o00<0ool0
5ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool06Ol00`3oo`06o`030?oo01So00<0ool0
1_l700So00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool05_l200[o00<0ool02_l00`3oo`0Go`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201co1@04o`030?oo01Wo00<0o`001P02o`04
0?ooo`801?l202?o00<0ool08Ol0103oo`09o`806?l200_o00<0ool02ol00`3oo`0Fo`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01_o00<0ool01_l00`3oo`0Io`8000?o00000ol0103oo`05o`05
0?ooo`009?l00`3oo`0Qo`040?oo00Wo00<0ool05_l00`3oo`0;o`030?oo00_o00<0ool05_l00`3o
o`08o`030?oo00;o0008o`802_l301_o00<0ool01Ol201co0`05o`801ol20004o`0002;o00<0ool0
8_l200[o0`0Ao`D03Ol200ko00<0ool05?l200[o0`02o`00<_l00`3oo`0fo`030?oo02?o00<0ool0
9?l501Go100So`030?oo02Co000ao`030?oo03Oo00<0ool08ol00`3oo`0Ro`801?l300ko1@0Wo`03
0?oo02Co000ao`030?oo03So00<0ool08_l00`3oo`0Po`802Ol300Go1P0/o`030?oo02Co000`o`03
0?oo03Wo00<0ool08_l00`3oo`0Oo`030?oo00co1@0bo`030?oo02Co000`o`030?oo03[o00<0ool0
8Ol00`3oo`0Oo`030?oo04?o00<0ool09?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool00ol0103oo`08o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Po`03
0?oo00Ko200To`030?oo00Oo00<0ool08?l00`3oo`0Lo`030?oo00[o00<0ool08_l800ko00<0ool0
6ol00`3oo`07o`00;ol900So0P0/o`030?oo023o00<0ool06ol00`3oo`0^o`801ol400So0`0Xo`00
@_l202[o00<0ool08?l00`3oo`0Ko`030?oo02ko00<0ool02_l902[o0014o`80:?l00`3oo`0Po`03
0?oo01[o00<0ool0;_l00`3oo`0no`00A_l00`3oo`0Uo`030?oo023o00<0ool06Ol00`3oo`0_o`03
0?oo03ko0017o`809_l00`3oo`0=o`/01ol00`3oo`0Io`030?oo02oo00<0ool0?_l004Wo00<0ool0
8ol00`3oo`0:o`<02ol200Go00<0ool06?l00`3oo`0`o`030?oo03ko001:o`030?oo02;o00<0ool0
1ol3013o0P03o`030?oo01So00<0ool0<?l00`3oo`0no`00Bol202;o00<0ool01Ol201Go0P000ol0
o`0Ho`030?oo033o00<0ool0?ol004go00<0ool07ol200Co0P0Io`806Ol00`3oo`0`o`030?oo03oo
001>o`030?oo01oo1@0Lo`805ol00`3oo`0ao`030?oo03oo001>o`030?oo043o00@0o`005Ol00`3o
o`0Ho`@05Ol00`3oo`0oo`00Col00`3oo`0oo`040?ooo`@04?l00`3oo`0Go`<00ol301;o00<0ool0
?ol004oo00<0ool0?ol00`3oo`05o`L02?l201Oo0P09o`803ol00`3oo`10o`00Col00`3oo`0[o`80
1?l200?o1005o`030?oo00co200Go`803Ol00`3oo`0<o`030?oo043o001@o`030?oo02Wo00@0ool0
2?l00`3oo`06o`030?oo02Wo0P0@o`803?l00`3oo`10o`00D?l00`3oo`0Yo`040?oo00Wo00<0ool0
1Ol200oo1P0Do`030?oo01;o0`08o`030?oo047o001@o`030?oo02Wo00@0ool02_l00`3oo`04o`03
0?oo00_o1005o`X01ol301So0P05o`80@ol0053o00<0ool04ol;00_o00@0ool02?l0103oo`05o`03
0?oo00Wo0P0Co`L07Ol604Co001@o`030?oo013o0`0;o`D01ol200[o0P06o`030?oo00So00<0ool0
POl0053o00<0ool03ol00`3oo`0Ao`L05?l00`3oo`08o`030?oo087o001@o`030?oo00go0P0Jo`80
4ol00`3oo`07o`030?oo08;o001@o`030?oo00co00<0ool07?l2017o00<0ool01_l00`3oo`23o`00
DOl00`3oo`09o`808Ol00`3oo`0>o`030?oo00Go00<0ool0Q?l0057o00<0ool01_l302?o00<0ool0
3_l00`3oo`04o`030?oo08Go001Ao`801Ol202Oo00<0ool03Ol00`3oo`03o`030?oo08Ko001Bo`H0
:_l00`3oo`0<o`030?oo00?o00<0ool0Q_l005Co0P0]o`030?oo00_o00@0oooo0P29o`00Q?l00`3o
o`0:o`040?oo08_o0025o`@02?l308co0029o`T0S_l007go0P04o`801Ol300Co00<0ool0S?l007co
00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3o
o`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00
oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85629 .14979 m
.85759 .14527 L
.85862 .141 L
.85909 .13855 L
.8594 .13634 L
.85952 .13513 L
.85958 .13445 L
.85961 .13382 L
.85965 .13269 L
.85965 .1315 L
.85961 .13037 L
.85953 .12936 L
.85939 .12821 L
.8592 .12714 L
.85876 .12531 L
.85844 .12431 L
.85809 .12341 L
.85713 .12146 L
.85606 .11982 L
.85459 .11812 L
.85297 .11669 L
.85125 .11552 L
.84917 .11443 L
.84677 .11351 L
.84556 .11314 L
.84436 .11284 L
.84322 .1126 L
.84195 .11238 L
.84087 .11223 L
.83965 .1121 L
.8383 .112 L
.83755 .11195 L
.83684 .11193 L
.83556 .1119 L
.83415 .1119 L
.83339 .11192 L
.83257 .11195 L
.83109 .11202 L
.83017 .11208 L
.8293 .11215 L
.82766 .1123 L
.82428 .11271 L
.82056 .11328 L
.80777 .11594 L
.79669 .1189 L
.78571 .12258 L
.78085 .1247 L
.77625 .12731 L
.77429 .12873 L
.77259 .13019 L
Mistroke
.7699 .13327 L
.76873 .13511 L
.76782 .13698 L
.76715 .13878 L
.76661 .14084 L
.76639 .14197 L
.76622 .14324 L
.76615 .1439 L
.76609 .14462 L
.76604 .14595 L
.76604 .1473 L
.76609 .14862 L
.76617 .14984 L
.76631 .15118 L
.76668 .1537 L
.76725 .15656 L
.76889 .16264 L
.77314 .17407 L
.77857 .18626 L
.78359 .19801 L
.78453 .20072 L
.78531 .20346 L
.78565 .20497 L
.78587 .2063 L
.78595 .20692 L
.78602 .20759 L
.78606 .20821 L
.78608 .20877 L
.78606 .2098 L
.78599 .21073 L
.78585 .21172 L
.78561 .21275 L
.78528 .2138 L
.78487 .21475 L
.78387 .21646 L
.78262 .218 L
.78118 .21937 L
.77914 .22091 L
.77696 .22224 L
.77165 .22475 L
.76476 .22724 L
.73518 .23494 L
.71999 .23822 L
.7122 .23972 L
.70539 .24085 L
.70179 .24135 L
.69998 .24156 L
.69811 .24175 L
.69659 .24189 L
.69497 .242 L
Mistroke
.69354 .24207 L
.69224 .24211 L
.69087 .24213 L
.6896 .24211 L
.68849 .24208 L
.68732 .24202 L
.68608 .24193 L
.6848 .2418 L
.68258 .24148 L
.67471 .23953 L
.66678 .23704 L
.6647 .23651 L
.66239 .23606 L
.66103 .23587 L
.65977 .23573 L
.65915 .23569 L
.65847 .23565 L
.65782 .23563 L
.65722 .23562 L
.65608 .23563 L
.655 .23569 L
.65379 .23579 L
.6531 .23587 L
.65247 .23596 L
.64989 .23646 L
.64857 .23679 L
.64711 .23723 L
.64158 .23949 L
.63645 .24244 L
.63053 .24689 L
.62511 .25198 L
.61621 .26266 L
.60807 .27521 L
.60177 .28629 L
.59572 .29598 L
.59392 .29826 L
.59215 .30015 L
.59034 .30175 L
.58857 .30302 L
.58679 .30403 L
.58479 .30491 L
.58358 .30533 L
.58241 .30567 L
.58123 .30595 L
.5801 .30618 L
.57897 .30636 L
.57791 .30649 L
.5767 .30661 L
.57537 .30671 L
.57405 .30677 L
Mistroke
.57329 .30679 L
.57258 .3068 L
.57125 .3068 L
.56981 .30678 L
.56856 .30674 L
.56718 .30668 L
.56464 .30652 L
.56151 .30627 L
.55856 .30599 L
.54427 .30421 L
.53056 .30196 L
.51831 .29921 L
.51205 .29733 L
.50675 .29538 L
.4886 .28618 L
.47897 .28137 L
.47362 .27932 L
.47036 .27833 L
.46724 .27756 L
.46367 .2769 L
.46164 .27662 L
.45967 .27641 L
.45782 .27628 L
.45608 .2762 L
.45413 .27616 L
.45314 .27617 L
.45203 .27619 L
.45007 .27626 L
.44825 .27638 L
.44623 .27656 L
.44408 .27681 L
.44041 .27736 L
.43635 .27814 L
.42781 .28032 L
.41987 .2829 L
.40607 .28828 L
.39974 .29085 L
.39388 .29298 L
.39118 .29375 L
.38977 .29405 L
.38912 .29416 L
.38852 .29424 L
.38743 .29433 L
.3865 .29434 L
.38554 .29427 L
.38502 .2942 L
.38456 .2941 L
.38372 .29385 L
.38331 .29368 L
.38288 .29348 L
Mistroke
.38152 .29257 L
.38082 .2919 L
.38012 .29107 L
.37896 .28927 L
.37791 .28705 L
.377 .2846 L
.37538 .27917 L
.37205 .26679 L
.36961 .25901 L
.36675 .25167 L
.36331 .2447 L
.35887 .23786 L
.35625 .23466 L
.3533 .23163 L
.35059 .22931 L
.34755 .22715 L
.34457 .22544 L
.34177 .22415 L
.3389 .22311 L
.33615 .22237 L
.33457 .22205 L
.3329 .22178 L
.33147 .22161 L
.3299 .22148 L
.32821 .22142 L
.32727 .22141 L
.32641 .22142 L
.32474 .22149 L
.32318 .22161 L
.32168 .22177 L
.32009 .22199 L
.31725 .22249 L
.3111 .22401 L
.30558 .22571 L
.30044 .22731 L
.2985 .22784 L
.29657 .22826 L
.29602 .22835 L
.29552 .22842 L
.29461 .22851 L
.29378 .22855 L
.29293 .22853 L
.29224 .22847 L
.2915 .22835 L
.29076 .22816 L
.2901 .22793 L
.28897 .22737 L
.28837 .22697 L
.28783 .22656 L
.28557 .2241 L
Mistroke
.28439 .22233 L
.28336 .22049 L
.27952 .21214 L
.27462 .20145 L
.27233 .19734 L
.26981 .19369 L
.26871 .19241 L
.26757 .19132 L
.26705 .19091 L
.26657 .19058 L
.26615 .19035 L
.26571 .19015 L
.26528 .19001 L
.26484 .18994 L
.26446 .18994 L
.26414 .19001 L
.26379 .19016 L
.26351 .19037 L
.26325 .19067 L
.26313 .19084 L
.26302 .19106 L
.26293 .19128 L
.26285 .19154 L
.26278 .1918 L
.26274 .19206 L
.26269 .19258 L
.26268 .19319 L
.26271 .19352 L
.26275 .19389 L
.26286 .19459 L
.26296 .19503 L
.26308 .19546 L
.26335 .19629 L
.26401 .19782 L
.26482 .19926 L
.26594 .20082 L
.26738 .20238 L
.269 .20374 L
.27094 .20495 L
.27276 .20576 L
.27385 .20611 L
.2749 .20635 L
.2755 .20646 L
.27616 .20654 L
.27679 .20659 L
.27738 .20661 L
.27852 .20659 L
.27974 .20646 L
.28038 .20635 L
.28107 .20621 L
Mistroke
.28233 .20586 L
.28384 .20529 L
.28522 .20464 L
.28765 .20314 L
.28986 .20137 L
.2924 .1988 L
.29458 .19604 L
.2982 .18998 L
.29992 .18619 L
.30133 .18238 L
.30243 .17874 L
.30341 .1747 L
.30387 .17234 L
.30422 .17017 L
.30451 .1681 L
.30475 .16589 L
.30493 .1636 L
.30504 .16149 L
.3051 .15925 L
.3051 .15797 L
.30509 .15679 L
.30502 .15469 L
.30491 .15279 L
.30473 .15075 L
.30449 .14865 L
.30396 .14532 L
.30324 .14198 L
.30242 .13905 L
.30158 .1366 L
.30047 .134 L
.29916 .13153 L
.29783 .12954 L
.29655 .12802 L
.29516 .12673 L
.29444 .12619 L
.29363 .12569 L
.29289 .12532 L
.2922 .12504 L
.29154 .12483 L
.29083 .12467 L
.29039 .1246 L
.29 .12456 L
.28956 .12454 L
.2891 .12453 L
.28833 .12458 L
.28789 .12463 L
.28748 .1247 L
.28673 .12487 L
.28604 .12506 L
.28451 .12564 L
Mistroke
.27802 .12916 L
.27505 .13067 L
.27327 .13136 L
.27243 .13163 L
.27152 .13186 L
.27073 .13203 L
.26984 .13217 L
.26901 .13226 L
.26822 .13231 L
.26726 .13233 L
.26635 .1323 L
.26528 .13223 L
.26423 .13211 L
.26315 .13195 L
.26198 .13172 L
.25977 .1312 L
.25427 .12946 L
.24143 .12444 L
.23332 .12134 L
.22513 .11864 L
.22132 .11758 L
.21722 .11663 L
.21485 .11618 L
.2127 .11584 L
.21053 .11556 L
.20854 .11536 L
.20672 .11524 L
.20482 .11518 L
.20293 .11519 L
.20087 .11527 L
.19911 .11541 L
.19725 .11564 L
.19552 .11592 L
.19396 .11625 L
.19055 .11719 L
.18751 .11833 L
.18432 .11988 L
.18111 .12188 L
.17615 .126 L
.17133 .13161 L
.16675 .13901 L
.16312 .14697 L
.16037 .15469 L
.1578 .16353 L
.15299 .18394 L
.14886 .20142 L
.14421 .21701 L
.13953 .22913 L
.135 .23932 L
.13316 .24378 L
Mistroke
.13251 .24571 L
.13229 .2465 L
.13208 .24733 L
.13193 .24808 L
.13183 .24875 L
.13178 .24937 L
.13175 .25001 L
.13177 .25064 L
.13183 .25121 L
.13195 .25181 L
.13204 .25214 L
.13213 .25243 L
.13236 .25298 L
.13261 .25347 L
.13297 .25401 L
.13336 .25448 L
.13431 .25537 L
.13556 .25619 L
.1368 .2568 L
.13808 .2573 L
.13963 .25776 L
.14141 .25817 L
.14306 .25846 L
.14468 .25868 L
.14652 .25886 L
.14759 .25894 L
.14857 .259 L
.14967 .25905 L
.15072 .25908 L
.15167 .2591 L
.1527 .25911 L
.15382 .25911 L
.15439 .2591 L
.15501 .25909 L
.1561 .25907 L
.15712 .25904 L
.1591 .25896 L
.16087 .25886 L
.16469 .25862 L
.16852 .25834 L
.1704 .25822 L
.17139 .25816 L
.1723 .25811 L
.17313 .25808 L
.17387 .25806 L
.17427 .25805 L
.17468 .25804 L
.17509 .25804 L
.17553 .25804 L
.17624 .25805 L
Mistroke
.17699 .25807 L
.17768 .25811 L
.1783 .25815 L
.17896 .25821 L
.17957 .25828 L
.18095 .25847 L
.18229 .25873 L
.18375 .25907 L
.18921 .26088 L
.20619 .26843 L
.21687 .2738 L
.22894 .28104 L
.23418 .28489 L
.23931 .28939 L
.24359 .29399 L
.24699 .29854 L
.24978 .3032 L
.25231 .30856 L
.25435 .31414 L
.25608 .32014 L
.25862 .3308 L
.25988 .33569 L
.2612 .33985 L
.26304 .3441 L
.26418 .34605 L
.26555 .34788 L
.26681 .34921 L
.26812 .35029 L
.26967 .35127 L
.27149 .35208 L
.27256 .35242 L
.27359 .35267 L
.2742 .35279 L
.27477 .35288 L
.27544 .35295 L
.27608 .35301 L
.27665 .35303 L
.27727 .35305 L
.27841 .35303 L
.27967 .35294 L
.28039 .35287 L
.28109 .35278 L
.28366 .35232 L
.2861 .35172 L
.29186 .3498 L
.31656 .33771 L
.32719 .33253 L
.33267 .33032 L
.3355 .32934 L
.33846 .32847 L
Mistroke
.34109 .32782 L
.34395 .32726 L
.34521 .32706 L
.34655 .32688 L
.34781 .32674 L
.34898 .32664 L
.35026 .32656 L
.35164 .32649 L
.35287 .32646 L
.35417 .32646 L
.35534 .32647 L
.35644 .32651 L
.35765 .32656 L
.35895 .32664 L
.3618 .32688 L
.36495 .32723 L
.37075 .32806 L
.38184 .32994 L
.40592 .33359 L
.41075 .33424 L
.4154 .33495 L
.41902 .33565 L
.42262 .33657 L
.42599 .33779 L
.42888 .33922 L
.43184 .34112 L
.43495 .34358 L
.44802 .35472 L
.45782 .36084 L
.46434 .36374 L
.46767 .36489 L
.4709 .36582 L
.47404 .36654 L
.47696 .36706 L
.47854 .36728 L
.48027 .36747 L
.48122 .36756 L
.4821 .36763 L
.48386 .36772 L
.48474 .36775 L
.4857 .36776 L
.48744 .36775 L
.48913 .36769 L
.49068 .36759 L
.49236 .36744 L
.4942 .36722 L
.49786 .3666 L
.50113 .36584 L
.50461 .36481 L
.51064 .36242 L
Mistroke
.51607 .35953 L
.52068 .35647 L
.52967 .34861 L
.53751 .33996 L
.54384 .33315 L
.54735 .33007 L
.55137 .3273 L
.55528 .32529 L
.55769 .32434 L
.56007 .32358 L
.56233 .32298 L
.56457 .32251 L
.56727 .32207 L
.56984 .32175 L
.57126 .32161 L
.57282 .3215 L
.57432 .32141 L
.57572 .32134 L
.57738 .32129 L
.57834 .32127 L
.57923 .32126 L
.58089 .32126 L
.58269 .32128 L
.58368 .3213 L
.5846 .32132 L
.58669 .32138 L
.58856 .32146 L
.59056 .32156 L
.598 .32204 L
.61304 .32323 L
.61955 .3237 L
.62293 .3239 L
.62647 .32407 L
.62825 .32414 L
.62919 .32417 L
.63019 .32419 L
.63194 .32422 L
.63355 .32423 L
.63521 .32423 L
.63693 .3242 L
.63837 .32416 L
.63993 .32409 L
.64171 .32399 L
.6433 .32388 L
.64676 .32352 L
.64984 .32308 L
.65265 .32255 L
.65777 .32125 L
.66217 .31973 L
.67118 .31536 L
Mistroke
.67963 .31009 L
.68417 .30724 L
.68854 .30481 L
.69273 .30296 L
.69519 .30213 L
.69752 .30152 L
.69953 .30115 L
.70069 .30099 L
.7018 .30088 L
.70298 .3008 L
.70407 .30077 L
.70529 .30077 L
.70659 .30083 L
.70788 .30093 L
.70912 .30106 L
.71147 .30143 L
.71406 .30198 L
.71694 .30275 L
.72738 .30668 L
.74016 .31277 L
.74593 .31556 L
.75203 .31825 L
.75505 .31942 L
.75785 .32037 L
.76034 .32107 L
.76291 .32163 L
.76366 .32176 L
.76436 .32186 L
.76565 .322 L
.76684 .32207 L
.76809 .32208 L
.76874 .32206 L
.76945 .32202 L
.77017 .32194 L
.77083 .32185 L
.772 .32163 L
.77321 .3213 L
.77444 .32087 L
.77553 .32038 L
.77787 .31898 L
.77977 .3174 L
.78162 .31538 L
.78444 .31112 L
.78584 .30831 L
.78703 .30546 L
.79362 .28207 L
.79545 .27607 L
.79659 .27307 L
.79779 .27047 L
.79904 .26828 L
Mistroke
.80056 .26617 L
.8021 .26451 L
.80368 .2632 L
.80555 .26203 L
.80746 .26115 L
.80864 .26074 L
.80976 .26043 L
.81112 .26013 L
.81246 .25992 L
.81381 .25977 L
.8153 .25966 L
.81677 .25962 L
.81813 .25962 L
.819 .25964 L
.8198 .25967 L
.8207 .25972 L
.82166 .25978 L
.8252 .26009 L
.84057 .26208 L
.84529 .26264 L
.84763 .26289 L
.84979 .26308 L
.85172 .26322 L
.85383 .26335 L
.85595 .26343 L
.8571 .26345 L
.85817 .26347 L
.86 .26345 L
.86196 .26339 L
.86398 .26327 L
.86582 .26311 L
.86752 .26291 L
.86904 .26268 L
.87231 .26201 L
.87406 .26154 L
.87562 .26104 L
.87893 .2597 L
.88157 .25824 L
.88405 .25644 L
.88592 .25469 L
.88763 .2526 L
.88896 .25043 L
.88993 .24831 L
.8904 .24695 L
.89076 .24565 L
.89107 .24418 L
.89119 .24343 L
.8913 .24258 L
.89142 .24114 L
.89146 .23955 L
Mistroke
.89142 .23801 L
.89131 .23659 L
.8911 .23488 L
.89082 .23325 L
.8901 .23015 L
.88905 .22677 L
.88762 .22307 L
.88077 .21049 L
.87076 .19705 L
.86208 .18656 L
.85875 .18222 L
.85594 .17788 L
.85489 .17585 L
.85408 .17396 L
.85351 .17228 L
.85328 .17141 L
.85307 .17046 L
.85296 .16987 L
.85288 .16933 L
.85276 .16826 L
.8527 .16728 L
.85268 .16624 L
.85269 .1657 L
.85272 .16512 L
.8528 .16406 L
.85286 .16341 L
.85294 .16279 L
.85313 .16161 L
.85356 .15947 L
.85405 .15744 L
.85541 .15269 L
.85629 .14979 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1Co`H0<ol003oo1`0fo`04
0?oo00_o00<0ool00ol00`3oo`1=o`H01Ol3037o000mo`801_l403?o00@0ool02?l500Co0P1<o`80
3Ol2033o000lo`030?oo00[o0`0Zo`@00_l0103oo`08o`040?oo00Go00<0ool0BOl2013o0P0_o`00
>_l2013o0P09o`@08Ol0103oo`09o`030?l000Go00<0ool0A_l301?o00<0ool0;Ol003Wo00<0ool0
4_l400;o0`03o`808Ol200_o0P05o`030?oo04Go00<0ool05?l00`3oo`0]o`00>Ol00`3oo`0Eo`@0
1_l00`3oo`0co`030?oo04Co00<0ool05Ol00`3oo`0]o`00>?l00`3oo`0Qo`030?oo03;o00<0ool0
A?l00`3oo`0Eo`030?oo02go000ho`030?oo027o00<0ool0<_l00`3oo`13o`030?oo01Ko00<0ool0
;Ol003Oo00<0ool08_l00`3oo`0bo`030?oo04?o00<0ool05_l00`3oo`0]o`00=ol00`3oo`0Ro`03
0?oo03;o00<0ool0@ol00`3oo`0Eo`030?oo02ko000fo`030?oo02Co00<0ool0<Ol00`3oo`13o`03
0?oo01Go00<0ool0;_l003Ko00<0ool09?l00`3oo`0ao`030?oo04Co00<0ool05?l00`3oo`0^o`00
=Ol00`3oo`0Uo`030?oo037o00<0ool0A?l00`3oo`0Co`80<?l003Go00<0ool09Ol00`3oo`0ao`03
0?oo04Co00<0ool04ol00`3oo`0_o`00=Ol00`3oo`0Uo`030?oo037o00<0ool0A?l00`3oo`0Co`03
0?oo02oo000eo`030?oo02Co00<0ool0<_l00`3oo`15o`030?oo01;o0P0`o`00=?l00`3oo`0Uo`03
0?oo01ko0P04o`800ol400Go00<0ool0AOl00`3oo`0Co`030?oo02ko000do`030?oo02Go00<0ool0
7Ol0103oo`08o`030?oo00Ko00<0ool0A_l00`3oo`0Bo`030?oo02ko000do`030?oo02Go00<0ool0
7Ol0103oo`09o`030?oo00Go0P17o`030?oo01?o00<0ool0;Ol003Co00<0ool09?l00`3oo`0Ho`@0
0_l0103oo`0:o`030?oo00Co00<0ool0Aol00`3oo`0Co`030?oo02co000do`030?oo01[o0`07o`03
0?oo01ko00@0ool02?l0103oo`05o`030?oo04Oo00<0ool05?l00`3oo`0[o`00<ol00`3oo`0Ko`03
0?l000Oo00<0ool07ol200[o0P06o`030?oo04So00<0ool04ol00`3oo`0[o`00<ol00`3oo`0Lo`03
0?l000Co0P0fo`030?oo04So00<0ool05?l00`3oo`0Zo`00<ol00`3oo`0Lo`8000?o0?l00_l00`3o
o`0eo`030?oo04So00<0ool05Ol00`3oo`0Yo`00<ol00`3oo`0Mo`H0>?l00`3oo`18o`805ol00`3o
o`0Xo`001ol500;o0P04o`807?l00`3oo`04o`D06?l00`3oo`07o`801?l200Co0P0So`030?oo02;o
0P04o`801?l201Wo00<0ool01ol500[o00<0ool05Ol500;o0P04o`800ol000Wo00<0ool01ol0103o
o`0Ko`030?oo00Ko00<0ool06?l00`3oo`06o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So
00@0ool05ol200_o00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3o
o`0Io`030?oo00Ko00<0ool06?l00`3oo`06o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o
00<0ool05Ol00`3oo`0:o`030?oo00_o00<0ool05_l00`3oo`0:o`400Ol1o`0000?o00000P04o`03
0?oo00So0P0Lo`D01?l00`3oo`0Io`030?l000<00_l00`3o000300Oo0P0So`030?oo027o00@0ool0
2Ol201Oo00<0ool02ol00`3oo`0;o`030?oo01Ko00<0ool02?l200?o0009o`030?oo00So00<0ool0
6_l00`3oo`07o`030?oo01Wo00<0ool01?l50003o`0000801?l00`3oo`0Ro`030?oo027o00@0ool0
2Ol00`3oo`0Do`803_l00`3oo`0<o`030?oo01Go00<0ool02?l00`3oo`02o`002?l200[o0`0Jo`03
0?oo00Ko0P0Lo`H00_l200Ko0P02o`<08_l00`3oo`0Ro`802_l3017o0`0?o`803_l00`3oo`0Do`80
2_l300;o000`o`030?oo03Ko00<0ool09Ol00`3oo`0no`@08_l00`3oo`0Uo`00<?l00`3oo`0go`03
0?oo02Co00<0ool09?l5017o100Vo`809_l002oo00<0ool0>Ol00`3oo`0So`030?oo02;o0`04o`@0
2?l502_o00<0ool09?l002oo00<0ool0>Ol00`3oo`0So`030?oo027o00<0ool02Ol8033o00<0ool0
9?l002oo00<0ool0>_l00`3oo`0Ro`030?oo01oo0P13o`030?oo02Go000<ool03009o`003?l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`030?oo00Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00_l200;o00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08Ol=02Oo00<0ool01ol00`3oo`0Po`030?oo01co0P0;o`030?oo02Wo
00<0ool02_l201ko00<0ool01ol003go0P0_o`030?oo023o00<0ool06ol00`3oo`0`o`d00ol402[o
000oo`80;Ol00`3oo`0Po`030?oo01[o00<0ool0;ol200co100^o`00@Ol302[o00<0ool08?l00`3o
o`0Jo`030?oo02oo00<0ool0?Ol004Co0P0Xo`030?oo023o00<0ool06Ol00`3oo`0_o`030?oo03ko
0016o`809ol00`3oo`0=o`X02?l00`3oo`0Io`030?oo02oo00<0ool0?_l004So00<0ool09?l00`3o
o`0;o`802_l300Go00<0ool06Ol00`3oo`0_o`030?oo03ko0019o`809Ol00`3oo`07o`<03ol00`3o
o`02o`030?oo01So00<0ool0<?l00`3oo`0no`00Bol00`3oo`0Ro`030?oo00Go0P0Co`800_l00`3o
o`0Ho`030?oo033o00<0ool0?_l004co00<0ool08_l00`3oo`0401Oo0`0Io`030?oo033o00<0ool0
?ol004co00<0ool08_l401co0`0Fo`030?oo037o00<0ool0?ol004go00<0ool0@Ol00`3oo`04017o
00<0ool0<_l00`3oo`0oo`00C_l00`3oo`10o`030?oo00Co1`09o`806_l8017o00<0ool0@?l004oo
00<0ool0?ol00`3oo`0;o`T06Ol300So0P0?o`030?oo043o001?o`030?oo02_o0P04o`800ol400Go
00<0ool0:ol200go0P0<o`80@_l004oo00<0ool0:_l0103oo`08o`030?oo00Ko00<0ool0:_l00`3o
o`0?o`<02?l00`3oo`12o`00D?l00`3oo`0Yo`040?oo00Wo00<0ool01Ol202[o00<0ool04ol200Go
0P14o`00D?l00`3oo`0Yo`040?oo00[o00<0ool01?l00`3oo`0=o`l02?l501So1P15o`00D?l00`3o
o`0Do`H03ol0103oo`08o`040?oo00Go00<0ool02ol200oo2@1Wo`00D?l00`3oo`0Ao`@01Ol700Wo
0P0:o`801_l00`3oo`09o`80POl0057o00<0ool03_l2013o1P0Go`030?oo00So00<0ool0POl0057o
00<0ool03?l201So100Co`030?oo00Oo00<0ool0P_l0057o00<0ool02_l201ko00<0ool04?l00`3o
o`06o`030?oo08?o001Ao`030?oo00So0P0Qo`030?oo00oo00<0ool01Ol00`3oo`24o`00D_l00`3o
o`05o`809?l00`3oo`0>o`030?oo00Co00<0ool0QOl005?o1`0Wo`803_l00`3oo`03o`030?oo08Ko
0023o`030?oo00_o00<0ool00_l00`3oo`27o`00Q?l200_o00<0ool00P2:o`00Q_l200Wo00<0o`00
S?l008So2`2=o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`04
0?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o
003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85869 .15009 m
.85954 .14561 L
.85988 .14345 L
.86014 .14141 L
.86026 .14026 L
.86036 .13901 L
.86043 .13788 L
.86046 .13684 L
.86047 .13573 L
.86044 .13452 L
.86041 .13392 L
.86037 .13327 L
.86026 .13212 L
.86011 .13094 L
.85992 .12986 L
.85964 .12861 L
.85933 .1275 L
.85863 .12553 L
.85768 .12355 L
.85665 .12189 L
.85535 .12023 L
.8525 .11762 L
.85059 .11637 L
.84864 .11537 L
.84633 .11445 L
.84366 .11367 L
.84225 .11336 L
.84068 .11308 L
.83914 .11287 L
.83769 .11272 L
.83636 .11263 L
.8356 .11259 L
.83487 .11256 L
.8341 .11255 L
.83339 .11254 L
.8326 .11254 L
.83177 .11255 L
.83032 .11259 L
.829 .11266 L
.82745 .11276 L
.82598 .11288 L
.82045 .11355 L
.81437 .11457 L
.80783 .11594 L
.79697 .11875 L
.78616 .12241 L
.78137 .12457 L
.77681 .12723 L
.77485 .12867 L
.77315 .13016 L
Mistroke
.77042 .13326 L
.76922 .13511 L
.76826 .13698 L
.76755 .13876 L
.76693 .1408 L
.76668 .14191 L
.76645 .14317 L
.76636 .14382 L
.76628 .14453 L
.76616 .14583 L
.7661 .14716 L
.76609 .14845 L
.76612 .14964 L
.76619 .15095 L
.76629 .1522 L
.76642 .1534 L
.76682 .15619 L
.76736 .15899 L
.76808 .16208 L
.77152 .17315 L
.7764 .18614 L
.77859 .19211 L
.78019 .19721 L
.78082 .19966 L
.78108 .20092 L
.78131 .20221 L
.78141 .20286 L
.7815 .20356 L
.78161 .20477 L
.78166 .20596 L
.78164 .20705 L
.78161 .20764 L
.78155 .20828 L
.78148 .20887 L
.78139 .20941 L
.78118 .21041 L
.78088 .21145 L
.78048 .21252 L
.78002 .21349 L
.77868 .2156 L
.77729 .21721 L
.77575 .21862 L
.7714 .22161 L
.76609 .22426 L
.7594 .22689 L
.73154 .23507 L
.71594 .23891 L
.70839 .24055 L
.70109 .24191 L
.69824 .24234 L
Mistroke
.69533 .2427 L
.69373 .24286 L
.69231 .24298 L
.69165 .24302 L
.69093 .24306 L
.68964 .24312 L
.68841 .24315 L
.68732 .24315 L
.68673 .24315 L
.6861 .24313 L
.68497 .24309 L
.68393 .24303 L
.68285 .24295 L
.68097 .24275 L
.67888 .24246 L
.67695 .24213 L
.67338 .24143 L
.66921 .24058 L
.66733 .24024 L
.66522 .23992 L
.66433 .23981 L
.66336 .23971 L
.66252 .23963 L
.66158 .23957 L
.66056 .23953 L
.65948 .23951 L
.65846 .23952 L
.6575 .23955 L
.65634 .23963 L
.65523 .23973 L
.65422 .23985 L
.65311 .24002 L
.65057 .24053 L
.64818 .24116 L
.64332 .24293 L
.63799 .24561 L
.6284 .25239 L
.61778 .26302 L
.60938 .27383 L
.60213 .28428 L
.5985 .28929 L
.59514 .29335 L
.59191 .2965 L
.59014 .29789 L
.58819 .29916 L
.58647 .30008 L
.58449 .30094 L
.5826 .30158 L
.58076 .30207 L
.57856 .3025 L
Mistroke
.57727 .30269 L
.57605 .30284 L
.57495 .30294 L
.57372 .30303 L
.5724 .30309 L
.57113 .30313 L
.56988 .30315 L
.56867 .30315 L
.56757 .30314 L
.56634 .30311 L
.56499 .30307 L
.56352 .30301 L
.56083 .30286 L
.55403 .30232 L
.54738 .30165 L
.53481 .30007 L
.52232 .29797 L
.51644 .29667 L
.51037 .295 L
.50046 .29142 L
.49174 .28746 L
.48123 .2827 L
.47561 .28063 L
.47301 .27982 L
.47003 .27904 L
.46747 .27848 L
.46461 .27798 L
.463 .27775 L
.4615 .27757 L
.46006 .27744 L
.45849 .27732 L
.45687 .27724 L
.45535 .2772 L
.45351 .27718 L
.45182 .27721 L
.45004 .27728 L
.44902 .27734 L
.44808 .27741 L
.44448 .27775 L
.44259 .27799 L
.44078 .27825 L
.43738 .27884 L
.42969 .28057 L
.41624 .28466 L
.40312 .28923 L
.39983 .29027 L
.39695 .29106 L
.39456 .29158 L
.39394 .29169 L
.39329 .29178 L
Mistroke
.39217 .29191 L
.39112 .29197 L
.39021 .29197 L
.38919 .2919 L
.38828 .29176 L
.38747 .29158 L
.38675 .29135 L
.38599 .29104 L
.38523 .29065 L
.38374 .28959 L
.38246 .28831 L
.38116 .28658 L
.37986 .28436 L
.37567 .27432 L
.37125 .26282 L
.36829 .25629 L
.36454 .24949 L
.36077 .244 L
.35613 .23871 L
.35302 .23588 L
.35 .23359 L
.34704 .23174 L
.34368 .23004 L
.34058 .22882 L
.33763 .22794 L
.33602 .22756 L
.33449 .22726 L
.33272 .22699 L
.33111 .22682 L
.32964 .22671 L
.32807 .22664 L
.32659 .22663 L
.32525 .22666 L
.32376 .22673 L
.32216 .22685 L
.3205 .22702 L
.31898 .22721 L
.3132 .22824 L
.30753 .2295 L
.30518 .23003 L
.30278 .23053 L
.30173 .23072 L
.30065 .23088 L
.29974 .231 L
.2988 .23109 L
.29791 .23114 L
.29713 .23115 L
.29636 .23113 L
.29565 .23108 L
.29486 .23098 L
Mistroke
.29414 .23085 L
.29338 .23066 L
.29259 .2304 L
.29101 .22966 L
.29025 .22919 L
.28944 .2286 L
.28684 .22599 L
.28546 .22414 L
.28402 .22189 L
.28146 .21729 L
.27636 .2074 L
.27152 .19923 L
.26917 .19609 L
.26693 .1938 L
.26643 .19341 L
.26593 .19307 L
.26552 .19284 L
.26511 .19267 L
.2649 .1926 L
.26471 .19257 L
.26452 .19255 L
.26433 .19255 L
.26417 .19258 L
.26404 .19262 L
.26382 .19276 L
.26367 .19293 L
.26356 .19318 L
.26352 .19332 L
.26349 .19349 L
.26347 .19382 L
.26351 .1942 L
.26359 .19459 L
.26371 .19497 L
.26387 .19541 L
.26443 .19648 L
.26529 .19769 L
.26635 .19884 L
.26752 .19984 L
.26883 .20075 L
.27047 .20162 L
.27136 .202 L
.27237 .20234 L
.27336 .20261 L
.2743 .2028 L
.27526 .20293 L
.27618 .203 L
.2772 .20302 L
.27832 .20297 L
.2793 .20286 L
.28023 .2027 L
Mistroke
.28126 .20247 L
.28238 .20214 L
.28485 .20114 L
.28755 .19961 L
.29004 .19773 L
.29228 .19561 L
.29619 .1907 L
.29815 .18748 L
.29975 .1843 L
.30237 .17739 L
.30338 .17369 L
.30386 .17152 L
.30423 .16952 L
.3045 .16779 L
.30475 .16591 L
.30495 .16392 L
.30508 .16204 L
.30516 .16029 L
.30519 .15845 L
.30518 .15689 L
.30512 .15519 L
.30499 .15323 L
.30491 .15227 L
.3048 .15122 L
.30429 .14774 L
.30392 .14587 L
.30347 .14394 L
.30249 .14068 L
.30132 .13771 L
.30007 .13521 L
.29874 .13311 L
.29742 .13142 L
.29597 .12995 L
.29459 .12886 L
.29303 .12794 L
.29215 .12755 L
.29131 .12726 L
.29049 .12705 L
.28962 .12689 L
.2888 .12681 L
.28806 .12678 L
.2876 .12679 L
.28717 .12681 L
.28671 .12684 L
.28622 .1269 L
.2854 .12703 L
.2845 .12723 L
.28291 .12767 L
.28147 .12818 L
.27823 .12952 L
Mistroke
.27541 .13072 L
.27219 .13189 L
.2704 .13239 L
.26867 .13275 L
.26768 .1329 L
.26675 .133 L
.26571 .13307 L
.26518 .13309 L
.26459 .1331 L
.26357 .13309 L
.26261 .13304 L
.26149 .13294 L
.26039 .13282 L
.25837 .13249 L
.25607 .13201 L
.25035 .13043 L
.2436 .12818 L
.22989 .12357 L
.22266 .12155 L
.21902 .12071 L
.21563 .12006 L
.21172 .11949 L
.20967 .11927 L
.20852 .11918 L
.20749 .11912 L
.20566 .11904 L
.2037 .11903 L
.20187 .11908 L
.2002 .11918 L
.19864 .11933 L
.19698 .11955 L
.19404 .12009 L
.19228 .12052 L
.19065 .12099 L
.18767 .12206 L
.18465 .12346 L
.18197 .12498 L
.17637 .12925 L
.17165 .1343 L
.16722 .1407 L
.16361 .14749 L
.16074 .1542 L
.15029 .18835 L
.14045 .21757 L
.1358 .22842 L
.13256 .23638 L
.13201 .23807 L
.13154 .23977 L
.13135 .24062 L
.13119 .24152 L
Mistroke
.13109 .24231 L
.13103 .24303 L
.13101 .24374 L
.13102 .24439 L
.13107 .24495 L
.13115 .24555 L
.13127 .24618 L
.13145 .24683 L
.13164 .24737 L
.13189 .24794 L
.13258 .24912 L
.13337 .25012 L
.13443 .25113 L
.13574 .25211 L
.13835 .25354 L
.14004 .25425 L
.14172 .25483 L
.14552 .25587 L
.14993 .25672 L
.15418 .25732 L
.15805 .25773 L
.16489 .25834 L
.16834 .25866 L
.17138 .25903 L
.17446 .25954 L
.17716 .26014 L
.18242 .26179 L
.19717 .26817 L
.21715 .27788 L
.22651 .28329 L
.23593 .2901 L
.24062 .2944 L
.24444 .29866 L
.25052 .30778 L
.25296 .31272 L
.25523 .31824 L
.25918 .32917 L
.26114 .33408 L
.26323 .33827 L
.26539 .34155 L
.26814 .3445 L
.26974 .34574 L
.27165 .34689 L
.27268 .34737 L
.27383 .34781 L
.27486 .34814 L
.27602 .34843 L
.27715 .34864 L
.27823 .34879 L
.27941 .34889 L
Mistroke
.28008 .34892 L
.28069 .34893 L
.28182 .34892 L
.28307 .34886 L
.28428 .34876 L
.28541 .34863 L
.28795 .34822 L
.29035 .3477 L
.29593 .3461 L
.30644 .34212 L
.31896 .33675 L
.32436 .33455 L
.33021 .33241 L
.33511 .33093 L
.34032 .32972 L
.34317 .32924 L
.34442 .32907 L
.34578 .32892 L
.34699 .3288 L
.34828 .32871 L
.34964 .32864 L
.35091 .3286 L
.35164 .32858 L
.35242 .32858 L
.35381 .32859 L
.35459 .32861 L
.35532 .32863 L
.35612 .32867 L
.35698 .32871 L
.36001 .32894 L
.36281 .32922 L
.36849 .32998 L
.37399 .33086 L
.39693 .33457 L
.40621 .33579 L
.41519 .337 L
.41966 .33777 L
.42342 .33863 L
.42675 .33965 L
.43005 .34098 L
.43649 .34462 L
.44403 .34989 L
.4523 .35529 L
.46183 .36005 L
.46779 .36222 L
.47111 .36317 L
.47475 .36401 L
.47787 .36456 L
.47955 .3648 L
.48138 .365 L
Mistroke
.48306 .36515 L
.48461 .36524 L
.4861 .3653 L
.48766 .36532 L
.48929 .36531 L
.49021 .36528 L
.49105 .36524 L
.4926 .36515 L
.49424 .365 L
.49706 .36466 L
.50012 .36413 L
.50333 .36342 L
.5067 .36247 L
.51252 .36032 L
.51795 .35769 L
.52271 .35483 L
.53063 .34884 L
.53866 .34149 L
.54572 .33513 L
.55244 .33038 L
.55664 .32824 L
.56085 .32662 L
.56552 .32533 L
.56835 .32475 L
.57111 .32431 L
.57415 .32394 L
.57702 .32369 L
.57866 .32358 L
.58047 .32348 L
.5822 .32341 L
.58383 .32337 L
.58546 .32334 L
.58723 .32333 L
.58893 .32333 L
.5905 .32334 L
.59235 .32337 L
.5934 .32339 L
.59439 .32341 L
.59815 .32352 L
.60575 .32382 L
.61403 .32417 L
.61794 .32431 L
.62006 .32437 L
.62202 .32442 L
.62369 .32445 L
.62546 .32447 L
.62646 .32447 L
.62741 .32447 L
.62917 .32446 L
.63082 .32444 L
Mistroke
.63256 .3244 L
.63402 .32435 L
.63562 .32428 L
.63732 .32418 L
.63911 .32405 L
.64229 .32375 L
.64507 .3234 L
.64801 .32293 L
.65366 .32171 L
.65855 .32024 L
.66285 .31861 L
.67965 .30966 L
.68819 .30519 L
.69267 .30345 L
.6953 .30268 L
.69782 .30212 L
.69921 .30189 L
.70055 .30171 L
.70177 .3016 L
.70247 .30155 L
.70312 .30152 L
.70389 .3015 L
.70461 .30149 L
.70527 .30149 L
.70599 .30151 L
.70678 .30154 L
.70762 .30159 L
.70834 .30165 L
.70914 .30172 L
.71192 .3021 L
.71351 .30238 L
.71497 .30269 L
.72064 .30419 L
.73303 .30874 L
.74494 .31359 L
.75088 .31573 L
.75626 .31726 L
.75888 .3178 L
.76031 .31803 L
.76161 .31819 L
.76293 .3183 L
.76366 .31834 L
.76434 .31836 L
.76505 .31836 L
.76569 .31834 L
.76642 .31831 L
.7671 .31825 L
.76813 .31813 L
.76924 .31794 L
.77038 .31769 L
Mistroke
.77141 .31739 L
.7732 .31673 L
.77503 .31582 L
.77706 .3145 L
.77886 .31301 L
.78194 .30959 L
.78464 .30547 L
.78726 .30027 L
.79177 .28863 L
.79557 .2787 L
.79772 .27427 L
.80042 .27008 L
.80199 .26824 L
.80383 .26653 L
.80572 .26515 L
.8076 .26409 L
.80959 .26324 L
.8107 .26286 L
.81192 .26252 L
.81303 .26226 L
.81429 .26202 L
.81538 .26185 L
.81661 .2617 L
.81741 .26162 L
.81817 .26156 L
.81971 .26148 L
.82114 .26144 L
.82272 .26143 L
.82363 .26144 L
.82462 .26146 L
.82558 .26148 L
.82647 .26152 L
.83004 .2617 L
.83693 .26217 L
.84086 .26243 L
.84282 .26254 L
.84464 .26263 L
.84642 .2627 L
.8483 .26275 L
.85019 .26278 L
.85127 .26278 L
.85228 .26277 L
.85401 .26273 L
.85559 .26267 L
.85732 .26257 L
.85911 .26243 L
.86067 .26227 L
.86231 .26206 L
.8652 .26157 L
.8685 .2608 L
Mistroke
.87134 .25991 L
.87447 .25863 L
.87712 .25723 L
.87974 .25542 L
.88217 .25323 L
.88398 .25109 L
.88539 .24891 L
.88663 .24633 L
.88718 .2448 L
.88759 .24335 L
.88792 .24187 L
.88815 .24045 L
.88829 .23914 L
.88837 .2377 L
.88838 .2361 L
.88836 .23529 L
.88831 .23439 L
.88816 .23277 L
.88796 .23125 L
.88769 .2297 L
.88738 .22822 L
.88665 .22544 L
.88442 .21916 L
.88119 .2124 L
.87771 .20638 L
.86927 .19398 L
.86304 .18504 L
.86046 .18078 L
.8584 .17652 L
.8576 .17429 L
.85725 .17312 L
.85695 .17188 L
.85672 .17073 L
.85656 .16969 L
.85644 .16863 L
.85637 .16764 L
.85633 .16662 L
.85633 .16553 L
.85635 .1646 L
.85641 .16357 L
.8565 .16245 L
.85664 .16124 L
.85695 .15902 L
.85869 .15009 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1Do`<0=Ol007co00@0ool0
2ol00`3oo`03o`030?oo04go1`03o`@0<Ol003oo1`0fo`040?oo00So1@04o`80C?l200go0P0`o`00
?Ol300Ko100/o`@00_l0103oo`08o`040?oo00Go00<0ool0BOl2013o0P0_o`00??l00`3oo`0;o`80
2ol302;o00@0ool02Ol00`3o0005o`030?oo04Ko0`0Co`030?oo02go000jo`804?l400Go0P02o`<0
8Ol200_o0P05o`030?oo04Go0P0Eo`030?oo02go000io`030?oo01Co1@06o`030?oo03?o00<0ool0
A?l00`3oo`0Eo`030?oo02go000ho`030?oo027o00<0ool0<_l00`3oo`14o`030?oo01Go00<0ool0
;Ol003So00<0ool08Ol00`3oo`0bo`030?oo04?o0P0Go`030?oo02go000go`030?oo02;o00<0ool0
<_l00`3oo`13o`030?oo01Ko00<0ool0;Ol003Oo00<0ool08_l00`3oo`0bo`030?oo04?o00<0ool0
5_l00`3oo`0]o`00=_l00`3oo`0To`030?oo037o00<0ool0@ol00`3oo`0Fo`030?oo02go000fo`03
0?oo02Co00<0ool0<Ol00`3oo`13o`030?oo01Go00<0ool0;_l003Ko00<0ool09?l00`3oo`0ao`03
0?oo04Co00<0ool05?l00`3oo`0^o`00=Ol00`3oo`0Uo`030?oo037o00<0ool0A?l00`3oo`0Do`03
0?oo02ko000eo`030?oo02Co0P0co`030?oo04Co00<0ool05?l00`3oo`0^o`00=Ol00`3oo`0To`03
0?oo03;o00<0ool0AOl00`3oo`0Co`030?oo02ko000do`030?oo02Go00<0ool07_l200Co0P03o`@0
1Ol00`3oo`15o`030?oo01Co00<0ool0;Ol003Co00<0ool09Ol00`3oo`0Mo`040?oo00So00<0ool0
1_l00`3oo`15o`030?oo01Co00<0ool0;Ol003Co00<0ool09Ol00`3oo`0Mo`040?oo00Wo00<0ool0
1Ol204Oo00<0ool04ol00`3oo`0]o`00<ol00`3oo`0Uo`030?oo01So1002o`040?oo00[o00<0ool0
1?l00`3oo`16o`030?oo01Co00<0ool0;?l003?o00<0ool07?l00`3oo`05o`030?oo01oo00@0ool0
2?l0103oo`05o`030?oo04Oo00<0ool05?l00`3oo`0[o`00<ol00`3oo`0Lo`801_l00`3oo`0Po`80
2_l200Ko00<0ool0Aol00`3oo`0Eo`030?oo02[o000bo`030?oo01go0`04o`030?oo03Go00<0ool0
Aol00`3oo`0Fo`030?oo02Wo000bo`030?oo01ko1P0ho`030?oo04Oo00<0ool05_l00`3oo`0Yo`00
<_l00`3oo`0Po`030?oo03Wo00<0ool0Aol00`3oo`0Go`030?oo02So0007o`D00_l200Co0P0Lo`03
0?oo00Co1@0Go`030?oo00So0P04o`801?l202?o00<0ool08_l200Co0P04o`805ol00`3oo`09o`D0
2Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01[o00<0ool01ol00`3oo`0Go`03
0?oo00Oo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0Fo`030?oo00_o00<0ool0
2_l00`3oo`0Go`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Ho`030?oo00Oo00<0ool0
6?l00`3oo`06o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool04ol200go00<0ool0
2_l00`3oo`0Go`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201_o00<0o`000`04o`03
0?oo01So00<0ool01002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol201Go0P0>o`030?oo00_o
00<0ool05_l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Io`030?oo00So00<0ool06Ol00`3o
o`05o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool04_l2013o00<0ool02ol00`3o
o`0Fo`030?oo00So00<0ool00_l000So0P0:o`<06Ol00`3oo`07o`807?l200Go2005o`<08_l00`3o
o`0Ro`802_l3013o0P0Ao`803_l00`3oo`0Do`802_l300;o000_o`030?oo02Oo1P08o`80:?l00`3o
o`0mo`@08ol00`3oo`0Uo`00;ol00`3oo`0go`030?oo02Go00<0ool0>Ol402Oo00<0ool09Ol002ko
0P0jo`030?oo02Co00<0ool09Ol600Wo1@0[o`030?oo02Go000^o`030?oo03[o00<0ool08ol00`3o
o`0So`801_l9033o00<0ool09Ol002ko0P0ko`030?oo02?o00<0ool08Ol2043o00<0ool09_l000co
o`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko0`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08ol802[o00<0ool01_l00`3oo`0Qo`030?oo01go0P0:o`030?oo02Wo
00<0ool02?l2023o00<0ool01ol003[o0`0ao`030?oo023o00<0ool07?l00`3oo`0`oa40;?l003go
0P0_o`030?oo023o00<0ool06ol00`3oo`0_o`<0??l003oo0`0]o`030?oo01oo00<0ool06ol00`3o
o`0^o`80?_l004;o0P0[o`030?oo01oo00<0ool06_l00`3oo`0_o`030?oo03go0014o`80:_l00`3o
o`0>o`L02Ol00`3oo`0Io`030?oo033o00<0ool0?Ol004Ko0P0Xo`030?oo00[o1@06o`@01Ol00`3o
o`0Io`030?oo02oo00<0ool0?_l004So00<0ool09_l00`3oo`05o`@03ol200?o00<0ool06?l00`3o
o`0`o`030?oo03ko0019o`030?oo02Go00<0ool00ol201Go0P000ol0o`0Ho`030?oo033o00<0ool0
?ol004[o00<0ool09Ol501Wo0`0Go`030?oo037o00<0ool0?ol004_o0P14o`030?l000804ol203;o
00<0ool0@?l004go00<0ool0@Ol00`3oo`02o`H03?l00`3oo`0bo`030?oo043o001>o`030?oo043o
00<0ool02?l<01co200Ao`030?oo043o001>o`030?oo043o00<0ool0;_l200So0P0>o`030?oo047o
001?o`030?oo02_o0P04o`800ol400Go00<0ool0:ol300co0P0;o`030?oo04;o001?o`030?oo02[o
00@0ool02?l00`3oo`06o`030?oo02Wo0P0Ao`@01_l204Co001?o`030?oo02[o00@0ool02Ol00`3o
o`05o`80:Ol00`3oo`0Eo`L0AOl004oo00<0ool0:_l0103oo`0:o`030?oo00Co00<0ool09?l406Co
001@o`030?oo02Wo00@0ool02?l0103oo`05o`030?oo00go5`1Xo`00D?l00`3oo`0Co`T03_l200[o
0P06o`030?oo00[o0`1oo`00DOl00`3oo`0=o`D02Ol801[o00<0ool02_l00`3oo`1oo`00D_l00`3o
o`0;o`030?oo01Co1@0Eo`030?oo00So0P22o`00D_l00`3oo`09o`807?l301;o00<0ool01ol00`3o
o`22o`00Dol00`3oo`06o`808Ol2013o00<0ool01_l00`3oo`23o`00E?l802Go00<0ool03Ol00`3o
o`05o`030?oo08Co0022o`030?oo00co00<0ool01?l00`3oo`25o`00Pol200co00<0ool00_l208So
0025o`802_l00`3o000208[o0027o`D01?l308go002;o`D000?o0?l0SOl0097o00<0ool0S?l007go
0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So
1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3o
o`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o
0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go
0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go
0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8611 .15039 m
.8613 .14825 L
.86141 .14707 L
.86149 .14595 L
.86155 .14491 L
.86161 .14379 L
.86164 .14276 L
.86166 .14182 L
.86166 .1407 L
.86165 .13967 L
.8616 .13853 L
.86152 .13732 L
.86147 .13673 L
.8614 .13607 L
.86125 .1349 L
.86087 .13274 L
.8606 .13157 L
.86031 .13048 L
.85964 .12851 L
.85876 .1265 L
.85761 .12444 L
.85643 .12276 L
.85501 .12112 L
.85203 .11856 L
.85007 .11731 L
.84815 .11632 L
.84576 .11535 L
.84336 .1146 L
.84102 .11405 L
.83877 .11366 L
.83756 .1135 L
.8362 .11336 L
.83552 .1133 L
.83477 .11325 L
.83339 .11318 L
.83266 .11316 L
.83186 .11314 L
.83042 .11313 L
.82954 .11315 L
.82869 .11317 L
.82793 .11319 L
.8271 .11323 L
.82538 .11333 L
.82355 .11348 L
.82025 .11383 L
.81434 .11469 L
.80789 .11593 L
.79649 .11882 L
.78702 .12208 L
.78235 .1242 L
Mistroke
.7778 .12685 L
.7741 .12975 L
.77124 .13286 L
.76994 .13472 L
.76891 .13655 L
.76796 .1387 L
.76754 .13985 L
.76716 .14112 L
.76687 .14227 L
.76664 .14335 L
.76645 .14448 L
.76631 .14554 L
.76618 .1468 L
.76613 .14748 L
.76609 .14821 L
.76606 .14946 L
.76605 .15018 L
.76606 .15086 L
.7661 .15203 L
.76617 .15333 L
.76628 .15471 L
.76642 .15601 L
.76674 .15839 L
.76718 .16098 L
.76851 .16706 L
.77002 .17267 L
.77344 .18399 L
.77515 .1899 L
.77641 .19512 L
.77688 .19771 L
.77708 .19912 L
.77721 .20038 L
.77726 .20108 L
.7773 .20172 L
.77732 .20242 L
.77733 .20315 L
.7773 .20435 L
.77725 .20502 L
.77719 .20564 L
.77705 .20669 L
.77683 .20778 L
.7766 .20868 L
.7763 .20965 L
.77542 .21172 L
.77434 .21355 L
.77293 .21538 L
.77139 .21697 L
.76704 .22032 L
.76228 .22304 L
.75641 .22572 L
Mistroke
.73074 .23451 L
.71736 .23838 L
.7043 .24173 L
.69867 .24292 L
.69375 .24376 L
.69173 .24403 L
.68966 .24425 L
.6885 .24435 L
.68747 .24442 L
.68645 .24448 L
.68551 .24452 L
.68449 .24455 L
.68342 .24456 L
.68242 .24455 L
.68152 .24453 L
.68048 .2445 L
.67989 .24447 L
.67935 .24445 L
.67733 .24431 L
.67338 .24397 L
.67127 .24376 L
.66897 .24356 L
.668 .24349 L
.66696 .24342 L
.66599 .24337 L
.66508 .24333 L
.66421 .24331 L
.66328 .24329 L
.66223 .24329 L
.66126 .24331 L
.66016 .24335 L
.65913 .24341 L
.65798 .2435 L
.65675 .24363 L
.65464 .24392 L
.65228 .24437 L
.64978 .24496 L
.64701 .24579 L
.64185 .24779 L
.63677 .25035 L
.63207 .25325 L
.62181 .26137 L
.61199 .27141 L
.60424 .28043 L
.59683 .2886 L
.593 .29201 L
.59091 .29355 L
.58862 .29498 L
.58642 .2961 L
.58433 .29697 L
Mistroke
.58231 .29765 L
.57998 .29826 L
.57766 .29872 L
.57644 .29891 L
.57509 .29908 L
.5738 .29921 L
.5726 .29931 L
.57136 .29939 L
.57017 .29945 L
.56877 .2995 L
.56798 .29952 L
.56725 .29953 L
.56582 .29953 L
.56444 .29952 L
.56365 .29951 L
.56281 .29949 L
.56128 .29945 L
.55962 .29939 L
.55776 .29931 L
.55168 .29894 L
.54499 .29841 L
.53287 .29718 L
.52039 .29539 L
.51381 .29409 L
.50805 .29267 L
.49814 .28948 L
.48794 .28553 L
.47823 .28203 L
.47231 .28038 L
.46936 .27972 L
.46653 .2792 L
.46396 .27883 L
.46114 .27852 L
.4595 .27839 L
.45793 .27829 L
.45651 .27823 L
.45495 .2782 L
.4534 .27819 L
.45169 .27822 L
.44988 .27828 L
.44817 .27838 L
.44656 .2785 L
.44509 .27863 L
.44174 .27901 L
.43483 .28013 L
.42752 .28173 L
.415 .28516 L
.40845 .28706 L
.40513 .28795 L
.40221 .28865 L
Mistroke
.39968 .28914 L
.39862 .28931 L
.3975 .28945 L
.39632 .28955 L
.39523 .2896 L
.39465 .28961 L
.39404 .2896 L
.39298 .28953 L
.39202 .28941 L
.39102 .28922 L
.39011 .28898 L
.3893 .2887 L
.38753 .28786 L
.38669 .28733 L
.3858 .28667 L
.38417 .28514 L
.38253 .28317 L
.3797 .27878 L
.37396 .26774 L
.36743 .2563 L
.36343 .25072 L
.35846 .24512 L
.35287 .24032 L
.34955 .23811 L
.34642 .23637 L
.34358 .23508 L
.34043 .23393 L
.33705 .23299 L
.33552 .23266 L
.33383 .23236 L
.33238 .23216 L
.33083 .23198 L
.32997 .23191 L
.32919 .23186 L
.32766 .23178 L
.32635 .23175 L
.32492 .23175 L
.32344 .23179 L
.32208 .23185 L
.32061 .23194 L
.31907 .23207 L
.31637 .23235 L
.31061 .2331 L
.30837 .2334 L
.30607 .23366 L
.3048 .23378 L
.30368 .23386 L
.30257 .23391 L
.30155 .23392 L
.30061 .2339 L
Mistroke
.29977 .23385 L
.29883 .23376 L
.29797 .23364 L
.29719 .23349 L
.29648 .23332 L
.29495 .23283 L
.29347 .23218 L
.29191 .23127 L
.28896 .22882 L
.28604 .22544 L
.28337 .22159 L
.27842 .21342 L
.27309 .20472 L
.27035 .20077 L
.26805 .19789 L
.26625 .19602 L
.26562 .19547 L
.26505 .19506 L
.26481 .19493 L
.2646 .19484 L
.26451 .19481 L
.26443 .1948 L
.26432 .19481 L
.26427 .19483 L
.26425 .19487 L
.26423 .19492 L
.26423 .19499 L
.26427 .19513 L
.26432 .19523 L
.26438 .19533 L
.2647 .19576 L
.26518 .19625 L
.26655 .19727 L
.26758 .19786 L
.26869 .19838 L
.27008 .19888 L
.27173 .19932 L
.2726 .19948 L
.27312 .19955 L
.2736 .19961 L
.27411 .19965 L
.27468 .19969 L
.27521 .1997 L
.27573 .1997 L
.27672 .19967 L
.27767 .19959 L
.27873 .19945 L
.27987 .19923 L
.28103 .19894 L
.28229 .19855 L
Mistroke
.28459 .19762 L
.28733 .19613 L
.28981 .19439 L
.29398 .19044 L
.29616 .18774 L
.29803 .18496 L
.30097 .17928 L
.30231 .17583 L
.30332 .17258 L
.3041 .16938 L
.30473 .16582 L
.30495 .16402 L
.30505 .16302 L
.30513 .16208 L
.30523 .16027 L
.30527 .15863 L
.30527 .1571 L
.30521 .1555 L
.30513 .15415 L
.30499 .15267 L
.3048 .15113 L
.30454 .1495 L
.30394 .14667 L
.30311 .14379 L
.302 .14089 L
.30064 .13818 L
.29922 .13596 L
.2978 .13419 L
.29637 .13277 L
.29464 .13143 L
.293 .13048 L
.29221 .13012 L
.29136 .12979 L
.29061 .12955 L
.2898 .12935 L
.2893 .12925 L
.28883 .12917 L
.28791 .12906 L
.28707 .129 L
.28618 .12899 L
.28568 .12901 L
.28522 .12903 L
.28471 .12907 L
.28418 .12913 L
.28328 .12925 L
.2823 .12942 L
.27877 .13028 L
.27547 .13128 L
.27182 .13237 L
.2701 .13282 L
Mistroke
.26824 .13322 L
.26662 .1335 L
.26573 .13362 L
.26477 .13372 L
.26425 .13377 L
.26366 .13381 L
.26263 .13386 L
.26203 .13387 L
.26139 .13387 L
.26022 .13385 L
.25964 .13383 L
.259 .13379 L
.25783 .1337 L
.25666 .13357 L
.25556 .13343 L
.25092 .13259 L
.24559 .13128 L
.23296 .12768 L
.22533 .12567 L
.22145 .12479 L
.21782 .12409 L
.21438 .12356 L
.21262 .12333 L
.21073 .12314 L
.20964 .12305 L
.20864 .12298 L
.20768 .12293 L
.20668 .12289 L
.20495 .12286 L
.20397 .12287 L
.20306 .12289 L
.20134 .12296 L
.19972 .12309 L
.19829 .12324 L
.19674 .12345 L
.19334 .1241 L
.19028 .12494 L
.18749 .12592 L
.18496 .12703 L
.17953 .13019 L
.1747 .13411 L
.17004 .13917 L
.16564 .14544 L
.16205 .15188 L
.15528 .16784 L
.14501 .19683 L
.13593 .21888 L
.13225 .22801 L
.13159 .22999 L
.13103 .23201 L
Mistroke
.13077 .23313 L
.13057 .23412 L
.13043 .23509 L
.13033 .23599 L
.1303 .23643 L
.13027 .23691 L
.13026 .2374 L
.13027 .23786 L
.1303 .23865 L
.13034 .23909 L
.13039 .23949 L
.13053 .24032 L
.13063 .24078 L
.13074 .2412 L
.13102 .24208 L
.13134 .24289 L
.13207 .24428 L
.13289 .24547 L
.13395 .2467 L
.13524 .24792 L
.13806 .24996 L
.14108 .25162 L
.14882 .25466 L
.1637 .25865 L
.17526 .26221 L
.18143 .26479 L
.18796 .2678 L
.20639 .27642 L
.21715 .28174 L
.22729 .28752 L
.23571 .29346 L
.24351 .30075 L
.25028 .30957 L
.25519 .31815 L
.26016 .32779 L
.26268 .33208 L
.26519 .3356 L
.26791 .33855 L
.27079 .34088 L
.27251 .34194 L
.27419 .34277 L
.27614 .34353 L
.27726 .34388 L
.27832 .34416 L
.27933 .34437 L
.28047 .34457 L
.28163 .34472 L
.28228 .34478 L
.28289 .34482 L
.28403 .34488 L
Mistroke
.28509 .3449 L
.28634 .34488 L
.28753 .34483 L
.28887 .34473 L
.29031 .34458 L
.29294 .34421 L
.29583 .34367 L
.29908 .34293 L
.31133 .33929 L
.32293 .33548 L
.33306 .33268 L
.33578 .33209 L
.33866 .33158 L
.34134 .33119 L
.34251 .33106 L
.34379 .33093 L
.34493 .33083 L
.34615 .33075 L
.34744 .33069 L
.34864 .33065 L
.34932 .33063 L
.35005 .33062 L
.35139 .33063 L
.35269 .33065 L
.35389 .33069 L
.35503 .33074 L
.3561 .33081 L
.35854 .33099 L
.36094 .33123 L
.36354 .33154 L
.38637 .3352 L
.40767 .33813 L
.41746 .3394 L
.42237 .34025 L
.4265 .3412 L
.43027 .34234 L
.4342 .34386 L
.44133 .3474 L
.45743 .35579 L
.46209 .35772 L
.46744 .35955 L
.47067 .36046 L
.47371 .36118 L
.47665 .36176 L
.47984 .36225 L
.48144 .36245 L
.48321 .36262 L
.48483 .36274 L
.48635 .36282 L
.48723 .36285 L
Mistroke
.48815 .36287 L
.4898 .36288 L
.49072 .36287 L
.49158 .36285 L
.49251 .36282 L
.49352 .36277 L
.4952 .36265 L
.49701 .36248 L
.50028 .36205 L
.50342 .36148 L
.50626 .36084 L
.51176 .35923 L
.51736 .35706 L
.52655 .35225 L
.54376 .33977 L
.55169 .33436 L
.55648 .3318 L
.56128 .32981 L
.56629 .32826 L
.57119 .32716 L
.57405 .32667 L
.57727 .32623 L
.58042 .32589 L
.58339 .32564 L
.58653 .32545 L
.58945 .32531 L
.59109 .32525 L
.59289 .3252 L
.5946 .32516 L
.5962 .32514 L
.59767 .32512 L
.59906 .3251 L
.60059 .32509 L
.60225 .32509 L
.60398 .32508 L
.60563 .32508 L
.60712 .32508 L
.60874 .32508 L
.60967 .32509 L
.61068 .32509 L
.61252 .32508 L
.61429 .32508 L
.61593 .32507 L
.61769 .32506 L
.61869 .32505 L
.61962 .32503 L
.62149 .325 L
.62347 .32496 L
.62519 .3249 L
.62704 .32484 L
Mistroke
.63037 .32467 L
.63354 .32446 L
.63642 .3242 L
.64205 .32352 L
.6451 .32302 L
.64782 .32249 L
.65746 .31984 L
.66218 .31805 L
.66711 .31585 L
.67656 .31101 L
.68545 .30662 L
.68993 .30485 L
.69419 .30356 L
.69664 .303 L
.69929 .30256 L
.70068 .3024 L
.70201 .30228 L
.70277 .30224 L
.70347 .3022 L
.70429 .30218 L
.70507 .30217 L
.70632 .30218 L
.70767 .30222 L
.70895 .3023 L
.71012 .3024 L
.71152 .30255 L
.71304 .30275 L
.71584 .30322 L
.72142 .30447 L
.72676 .30597 L
.73908 .30995 L
.74522 .31185 L
.75089 .31333 L
.75395 .31395 L
.75553 .3142 L
.75719 .31442 L
.75856 .31455 L
.75933 .3146 L
.76005 .31464 L
.7613 .31466 L
.76262 .31464 L
.76388 .31457 L
.76501 .31446 L
.76629 .31428 L
.76747 .31406 L
.76951 .31352 L
.77164 .31275 L
.77389 .31166 L
.77613 .31024 L
.77984 .30706 L
Mistroke
.78328 .30301 L
.78617 .29866 L
.79198 .28773 L
.7945 .28281 L
.79737 .2778 L
.80014 .27386 L
.80344 .27029 L
.80681 .26772 L
.80893 .26652 L
.81106 .26558 L
.81313 .26486 L
.81513 .26432 L
.81744 .26384 L
.82 .26345 L
.8227 .26318 L
.82411 .26308 L
.82569 .26299 L
.82653 .26295 L
.82744 .26292 L
.8283 .26289 L
.82912 .26287 L
.82993 .26285 L
.83082 .26284 L
.83239 .26282 L
.83557 .26279 L
.83715 .26278 L
.83862 .26276 L
.84033 .26274 L
.84124 .26272 L
.8422 .2627 L
.84375 .26266 L
.84546 .2626 L
.84715 .26252 L
.84869 .26243 L
.85035 .26231 L
.85214 .26214 L
.8553 .26177 L
.85823 .2613 L
.86108 .26072 L
.86401 .25995 L
.86896 .25816 L
.87147 .25695 L
.87399 .25545 L
.87798 .25224 L
.87993 .25009 L
.88167 .24761 L
.8829 .24534 L
.88396 .2427 L
.88442 .24118 L
.88478 .23971 L
Mistroke
.88503 .23836 L
.88523 .23687 L
.88531 .23608 L
.88537 .23521 L
.88541 .23429 L
.88542 .23343 L
.88539 .23187 L
.88534 .23098 L
.88528 .23017 L
.88512 .22858 L
.88491 .22711 L
.88425 .22374 L
.88346 .22075 L
.88244 .21759 L
.87687 .20511 L
.86991 .19298 L
.86473 .18375 L
.86256 .1789 L
.86175 .17664 L
.86112 .17452 L
.86079 .17315 L
.8605 .17168 L
.86028 .17033 L
.86013 .16908 L
.86007 .16843 L
.86002 .16773 L
.85997 .16696 L
.85994 .16623 L
.85993 .16545 L
.85992 .16463 L
.85993 .16393 L
.85994 .16316 L
.85998 .16233 L
.86001 .16154 L
.86011 .16005 L
.86042 .15666 L
.8611 .15039 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl=03;o001lo`040?oo00So1@04o`80C?l200go00<0ool0;ol0043o1P0`o`@0
0_l0103oo`08o`040?oo00Go00<0ool0BOl2013o00<0ool0;_l003go0`05o`L0<?l0103oo`09o`03
0?l000Go00<0ool0A_l301?o00<0ool0;Ol003_o0P0?o`801ol602;o0P0;o`801Ol00`3oo`15o`03
0?oo01Co00<0ool0;Ol003[o00<0ool04Ol700Go0P0eo`030?oo04Co0P0Fo`80;_l003Wo00<0ool0
7ol00`3oo`0co`030?oo04Co00<0ool05_l00`3oo`0/o`00>Ol00`3oo`0Po`030?oo03;o00<0ool0
@ol201So00<0ool0;?l003So00<0ool08Ol00`3oo`0bo`030?oo04?o00<0ool05ol00`3oo`0/o`00
=ol00`3oo`0Ro`80<ol00`3oo`13o`030?oo01Oo00<0ool0;?l003Oo00<0ool08ol00`3oo`0ao`03
0?oo04?o00<0ool05ol00`3oo`0/o`00=_l00`3oo`0To`030?oo037o00<0ool0@ol00`3oo`0Fo`03
0?oo02go000fo`030?oo02Co00<0ool0<Ol00`3oo`13o`805ol00`3oo`0]o`00=Ol00`3oo`0Uo`03
0?oo037o00<0ool0A?l00`3oo`0Eo`030?oo02go000eo`030?oo02Go00<0ool0<Ol00`3oo`14o`03
0?oo01Go00<0ool0;Ol003Go00<0ool09?l00`3oo`0bo`030?oo04Co00<0ool05Ol00`3oo`0]o`00
=?l00`3oo`0Uo`030?oo01ko0P04o`800ol400Go00<0ool0A?l00`3oo`0Eo`80;_l003Co00<0ool0
9Ol00`3oo`0Mo`040?oo00So00<0ool01_l00`3oo`15o`030?oo01Go00<0ool0;?l003?o00<0ool0
9_l00`3oo`0Mo`040?oo00Wo00<0ool01Ol204Ko00<0ool05_l00`3oo`0[o`00<ol00`3oo`0To`80
6_l400;o00@0ool02_l00`3oo`04o`030?oo04Ko00<0ool05Ol00`3oo`0[o`00<ol00`3oo`0To`03
0?oo01oo00@0ool02?l0103oo`05o`030?oo04Ko00<0ool05_l00`3oo`0Zo`00<_l00`3oo`0Mo`03
0?oo00Co00<0ool08Ol200[o0P06o`030?oo04Ko00<0ool05_l00`3oo`0Zo`00<_l00`3oo`0Mo`80
0_l303So00<0ool0A_l00`3oo`0Go`030?oo02Wo000bo`030?oo01ko100jo`030?oo04Ko00<0ool0
5ol00`3oo`0Yo`00<Ol00`3oo`0Po`030?oo03[o00<0ool0A_l00`3oo`0Ho`030?oo02So0007o`D0
0_l200Co0P0Ko`030?oo00Go1@0Go`030?oo00So0P04o`801?l202?o00<0ool08_l200Co0P04o`80
5_l00`3oo`0:o`D02Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01[o00<0ool0
1ol00`3oo`0Ho`030?oo00Ko00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0Do`80
3_l00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Oo00<0ool0
2?l00`3oo`0Ho`030?oo00Ko00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`0Bo`03
0?oo00go00<0ool02_l00`3oo`0Go`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201[o
00<0ool01004o`030?oo01Wo00<0o`000`02o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol201Co
00<0ool03_l00`3oo`0;o`030?oo01Ko00<0ool02?l200?o0009o`030?oo00So00<0ool06?l00`3o
o`09o`030?oo01[o00<0ool01?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo017o
0P0Ao`030?oo00_o00<0ool05_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201co
00<0ool01Ol200[o0`0Ro`030?oo02;o0P0:o`<03ol201;o0P0=o`030?oo01Go0P0:o`<00_l002ko
0P0Yo`801Ol602_o00<0ool0??l402?o00<0ool09_l002ko00<0ool0:Ol600Ko0`0Xo`030?oo03Wo
0`0Wo`030?oo02Ko000^o`030?oo03Oo0P0Wo`030?oo03Go100Zo`030?oo02Ko000^o`80>_l00`3o
o`0To`030?oo02Ko1`02o`H0;_l00`3oo`0Vo`00;ol00`3oo`0io`030?oo02?o00<0ool08_l400Ko
0`0co`030?oo02Oo000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00<00ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01Ol200So
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09_l402_o00<0ool01_l00`3o
o`0Qo`030?oo01ko0P09o`030?oo02Wo00<0ool01_l202;o00<0ool01ol003Wo0P0co`030?oo023o
00<0ool07Ol00`3oo`0co`/0;_l003_o0P0ao`030?oo023o00<0ool07?l00`3oo`0`o`@0>Ol003go
0P0`o`030?oo01oo00<0ool06ol00`3oo`0_o`80?Ol003oo0P0_o`030?oo01ko00<0ool06_l00`3o
o`0_o`030?oo03go0011o`<0;?l00`3oo`0Ao`802ol00`3oo`0Io`030?oo033o00<0ool0?Ol004Co
0P0[o`030?oo00So3`06o`030?oo01So00<0ool0<Ol00`3oo`0mo`00A_l202Wo0P07o`803ol400;o
00<0ool05ol00`3oo`0ao`030?oo03ko0018o`030?oo02Oo200Eo`<06?l00`3oo`0ao`030?oo03oo
0019o`030?oo02So0P0Ko`<05Ol00`3oo`0bo`030?oo03oo001:o`030?oo04Co00<0ool0100?o`<0
<ol00`3oo`10o`00Bol204Co00<0ool01?l?03Ko00<0ool0@?l004go00<0ool0@Ol00`3oo`0`o`P0
4?l00`3oo`11o`00COl00`3oo`11o`030?oo02go0`08o`<03?l00`3oo`12o`00C_l00`3oo`0/o`80
1?l200?o1005o`030?oo02_o0P0>o`@01_l204Go001?o`030?oo02[o00@0ool02?l00`3oo`06o`03
0?oo02[o00<0ool04_l704Ko001?o`030?oo02[o00@0ool02Ol00`3oo`05o`80:?l306;o001@o`03
0?oo02Wo00@0ool02_l00`3oo`04o`030?oo02?o101Uo`00D?l00`3oo`0Yo`040?oo00So00@0ool0
1Ol00`3oo`0@oa<0JOl0057o00<0ool04ol401;o0P0:o`801_l00`3oo`0=o`<0O?l0057o00<0ool0
3ol500?o100Ro`030?oo00[o0`1oo`00D_l00`3oo`0;o`<03?l;01Oo00<0ool02Ol00`3oo`20o`00
Dol200So0`0Jo`@04ol00`3oo`08o`030?oo087o001Do`T08Ol2017o00<0ool01_l208Co0020o`80
3ol00`3oo`05o`030?oo08Co0022o`803Ol00`3oo`03o`80Qol008Co0P0;o`040?ooo`80ROl008Ko
1007o`@0Rol008[o202>o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3o
o`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So
0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8635 .15069 m
.8635 .14958 L
.86349 .14857 L
.86347 .1474 L
.86344 .1463 L
.8634 .14528 L
.86334 .14434 L
.86318 .14223 L
.86307 .14113 L
.86294 .14012 L
.86258 .13783 L
.86212 .13565 L
.86149 .13336 L
.86073 .13122 L
.85974 .12899 L
.85872 .12711 L
.85747 .12523 L
.85467 .12202 L
.85156 .11949 L
.84937 .11814 L
.84721 .11707 L
.84468 .11606 L
.84184 .1152 L
.83925 .11462 L
.83673 .1142 L
.83525 .11402 L
.83388 .11389 L
.83312 .11384 L
.83228 .11378 L
.83148 .11374 L
.83074 .11372 L
.82948 .11369 L
.82811 .11368 L
.8268 .1137 L
.8256 .11374 L
.82425 .1138 L
.82278 .11389 L
.81978 .11414 L
.81681 .11448 L
.81363 .11492 L
.80795 .11593 L
.79669 .11869 L
.79077 .12062 L
.78571 .12267 L
.78132 .1249 L
.77763 .12727 L
.77426 .13009 L
.77257 .13188 L
.77117 .13365 L
.76895 .13737 L
Mistroke
.76802 .13951 L
.76722 .14193 L
.76691 .14309 L
.76663 .14437 L
.76642 .14553 L
.76624 .14678 L
.76611 .14793 L
.766 .14922 L
.76594 .15041 L
.7659 .15154 L
.7659 .15277 L
.76592 .15411 L
.76597 .1554 L
.76605 .15661 L
.76613 .15771 L
.76625 .15893 L
.76654 .16144 L
.76747 .1672 L
.76857 .17269 L
.77113 .1842 L
.77173 .18713 L
.77229 .19018 L
.7727 .19292 L
.77286 .19423 L
.77297 .19542 L
.77305 .19652 L
.77311 .19765 L
.77314 .19878 L
.77313 .19999 L
.77309 .20098 L
.77301 .20202 L
.7729 .20298 L
.77277 .20385 L
.77257 .20493 L
.77234 .20591 L
.77166 .20805 L
.77087 .20985 L
.76984 .21169 L
.76713 .21515 L
.76393 .21808 L
.75964 .22106 L
.75436 .224 L
.73161 .23329 L
.70662 .24149 L
.69707 .24404 L
.6925 .24503 L
.68797 .24579 L
.68568 .24608 L
.68323 .24633 L
.68217 .24641 L
Mistroke
.68104 .24648 L
.67901 .24658 L
.67796 .24662 L
.67682 .24665 L
.67574 .24667 L
.67474 .24669 L
.67272 .24671 L
.67172 .24672 L
.6708 .24673 L
.66988 .24674 L
.66888 .24676 L
.66786 .24678 L
.66677 .24681 L
.66584 .24685 L
.66483 .24689 L
.66299 .24701 L
.66064 .24721 L
.65837 .24748 L
.65629 .24779 L
.65398 .24821 L
.64915 .24939 L
.64448 .25091 L
.63928 .25307 L
.63366 .25596 L
.62254 .26337 L
.61318 .27124 L
.60483 .27899 L
.59659 .28614 L
.59263 .28895 L
.58879 .29114 L
.5852 .29272 L
.5831 .29344 L
.58108 .29402 L
.57889 .29454 L
.57682 .29493 L
.57445 .29528 L
.57309 .29544 L
.57181 .29556 L
.57044 .29567 L
.56919 .29575 L
.56777 .29583 L
.56697 .29586 L
.56624 .29588 L
.56487 .29592 L
.56415 .29593 L
.56338 .29594 L
.56208 .29595 L
.56063 .29595 L
.5592 .29593 L
.55783 .29591 L
Mistroke
.55659 .29588 L
.55524 .29585 L
.55216 .29574 L
.54928 .29562 L
.53834 .29497 L
.5326 .29452 L
.52654 .29394 L
.51608 .29259 L
.51008 .29152 L
.50461 .2903 L
.49475 .28756 L
.48518 .28455 L
.47971 .28294 L
.47431 .28158 L
.46902 .28051 L
.46591 .28003 L
.46298 .27968 L
.46046 .27945 L
.459 .27935 L
.45763 .27928 L
.45617 .27922 L
.45482 .27919 L
.45326 .27918 L
.45177 .27919 L
.45026 .27923 L
.44862 .27929 L
.44706 .27938 L
.44563 .27948 L
.44228 .27977 L
.43874 .28019 L
.43209 .28123 L
.42497 .28264 L
.41318 .28526 L
.40794 .28632 L
.40577 .28669 L
.40348 .287 L
.40227 .28712 L
.40116 .2872 L
.40003 .28725 L
.39884 .28727 L
.39823 .28726 L
.39758 .28723 L
.39697 .28719 L
.39642 .28713 L
.3954 .287 L
.39434 .28681 L
.3934 .28658 L
.3924 .28628 L
.39067 .2856 L
.38905 .28475 L
Mistroke
.38763 .28381 L
.38453 .2811 L
.38125 .2773 L
.37438 .26762 L
.36683 .2576 L
.35819 .24889 L
.3528 .2449 L
.34981 .2431 L
.34655 .24144 L
.34375 .24025 L
.34063 .23916 L
.33774 .23835 L
.33506 .23776 L
.33232 .2373 L
.33088 .23711 L
.32936 .23695 L
.32776 .23682 L
.32626 .23674 L
.32493 .23669 L
.3235 .23666 L
.32227 .23665 L
.32094 .23665 L
.31956 .23668 L
.31829 .23671 L
.31374 .23688 L
.31247 .23693 L
.31181 .23695 L
.31112 .23697 L
.31052 .23698 L
.30987 .23699 L
.30873 .237 L
.30765 .237 L
.30651 .23697 L
.30546 .23692 L
.30452 .23686 L
.30344 .23675 L
.30245 .23662 L
.30139 .23644 L
.30029 .23621 L
.29812 .23558 L
.29623 .23482 L
.29464 .234 L
.29295 .23293 L
.28968 .23023 L
.28413 .22384 L
.27244 .20689 L
.26797 .20075 L
.26638 .19869 L
.26539 .19738 L
.2652 .19709 L
Mistroke
.26506 .19686 L
.26499 .19669 L
.26495 .19655 L
.26495 .19649 L
.26497 .19643 L
.26505 .19636 L
.26511 .19634 L
.26519 .19632 L
.26529 .19631 L
.26541 .19631 L
.26566 .19633 L
.26582 .19635 L
.26599 .19638 L
.26674 .1965 L
.26769 .19665 L
.26872 .19678 L
.26931 .19685 L
.26989 .1969 L
.27054 .19694 L
.27129 .19698 L
.27209 .19699 L
.27285 .19698 L
.27329 .19696 L
.27377 .19694 L
.27468 .19687 L
.27549 .19678 L
.27641 .19665 L
.27834 .19627 L
.28021 .19577 L
.282 .19517 L
.28431 .19418 L
.28652 .19302 L
.29062 .19021 L
.29476 .18629 L
.29676 .18388 L
.29873 .18103 L
.30043 .17807 L
.30179 .1752 L
.30288 .1724 L
.30385 .16924 L
.30453 .16635 L
.30498 .16363 L
.30516 .16212 L
.30527 .16076 L
.30535 .15919 L
.30537 .15772 L
.30534 .15626 L
.3053 .15544 L
.30524 .15468 L
.30509 .15316 L
Mistroke
.30499 .15233 L
.30487 .15157 L
.30432 .14883 L
.30362 .14636 L
.30261 .14372 L
.30151 .14147 L
.30012 .13924 L
.29845 .13716 L
.29685 .13559 L
.29505 .1342 L
.29343 .13323 L
.29159 .1324 L
.29053 .13203 L
.28955 .13175 L
.2884 .1315 L
.28787 .13141 L
.2873 .13133 L
.28682 .13127 L
.28628 .13122 L
.28533 .13116 L
.2848 .13115 L
.28422 .13115 L
.28367 .13115 L
.28316 .13117 L
.28225 .13122 L
.28127 .13131 L
.2795 .13153 L
.27738 .13188 L
.27545 .13225 L
.27143 .13309 L
.2692 .13354 L
.26708 .13391 L
.265 .13422 L
.26306 .13443 L
.26193 .13453 L
.26085 .13459 L
.25969 .13463 L
.25902 .13465 L
.2584 .13465 L
.25729 .13464 L
.25623 .13461 L
.25526 .13456 L
.2542 .13449 L
.25302 .13439 L
.25175 .13425 L
.24943 .13395 L
.2439 .13299 L
.23839 .13181 L
.23202 .13037 L
.22473 .12882 L
Mistroke
.22123 .12817 L
.21747 .12758 L
.21424 .12717 L
.21237 .12699 L
.21066 .12685 L
.20903 .12676 L
.20732 .12669 L
.20561 .12667 L
.20465 .12668 L
.20375 .1267 L
.20212 .12676 L
.20066 .12686 L
.19897 .12702 L
.1974 .12722 L
.19449 .12771 L
.19148 .12842 L
.18897 .12917 L
.18628 .13017 L
.18059 .13301 L
.17568 .13643 L
.17088 .14082 L
.16619 .14639 L
.1589 .15806 L
.14795 .18205 L
.13702 .20714 L
.13284 .21703 L
.13143 .22096 L
.13078 .22309 L
.13029 .22501 L
.12992 .22684 L
.12976 .22783 L
.12965 .22872 L
.12957 .22964 L
.12954 .23016 L
.12952 .23063 L
.12952 .23148 L
.12955 .23237 L
.12962 .23324 L
.12967 .23372 L
.12973 .23416 L
.12988 .235 L
.13006 .23578 L
.13058 .23747 L
.13135 .23924 L
.13218 .24073 L
.13306 .24202 L
.13544 .24475 L
.13808 .24706 L
.14088 .24907 L
.15526 .25642 L
Mistroke
.1679 .26185 L
.19647 .27573 L
.21619 .28504 L
.23411 .29554 L
.24252 .30241 L
.2492 .30956 L
.25526 .31756 L
.26121 .3258 L
.26678 .33213 L
.27035 .33511 L
.27403 .33739 L
.27578 .33823 L
.27771 .339 L
.27944 .33956 L
.28135 .34006 L
.28246 .34029 L
.28349 .34048 L
.28455 .34063 L
.28554 .34075 L
.28672 .34086 L
.28735 .34091 L
.28802 .34095 L
.28915 .34099 L
.29042 .341 L
.29173 .34099 L
.29295 .34095 L
.29432 .34087 L
.29578 .34076 L
.29853 .34046 L
.30106 .34012 L
.31123 .33819 L
.32318 .33552 L
.32833 .33447 L
.33391 .33354 L
.33638 .33321 L
.33902 .33292 L
.34029 .33281 L
.3415 .33272 L
.34257 .33266 L
.34375 .3326 L
.34499 .33256 L
.34631 .33253 L
.34754 .33252 L
.34869 .33253 L
.35004 .33256 L
.35127 .3326 L
.35263 .33267 L
.35407 .33275 L
.35669 .33296 L
.3595 .33325 L
Mistroke
.3646 .33391 L
.37643 .3358 L
.39833 .33917 L
.40917 .34046 L
.4147 .3411 L
.41958 .34173 L
.42795 .34321 L
.43211 .34425 L
.43648 .3456 L
.45205 .35182 L
.46169 .35545 L
.4673 .35716 L
.47365 .35868 L
.47641 .3592 L
.47944 .35967 L
.48119 .35989 L
.48282 .36006 L
.48448 .3602 L
.48604 .36031 L
.48736 .36037 L
.48878 .36042 L
.49036 .36045 L
.4918 .36044 L
.4933 .36041 L
.49415 .36038 L
.49493 .36035 L
.49636 .36026 L
.49787 .36014 L
.50094 .35982 L
.50382 .3594 L
.50912 .35835 L
.51463 .35684 L
.52034 .35482 L
.52977 .35045 L
.53766 .34593 L
.55325 .33671 L
.55769 .33461 L
.56262 .33266 L
.56714 .33122 L
.5723 .32993 L
.57805 .32884 L
.58374 .32805 L
.59084 .32735 L
.59757 .32688 L
.61113 .32619 L
.6171 .32588 L
.62361 .32546 L
.62957 .32494 L
.635 .32431 L
.64053 .32346 L
Mistroke
.64623 .32232 L
.65185 .32086 L
.65747 .31905 L
.66692 .31521 L
.67623 .31086 L
.68498 .30705 L
.68888 .30565 L
.6931 .30443 L
.69535 .30392 L
.69787 .30345 L
.69899 .30329 L
.7002 .30314 L
.70126 .30302 L
.70241 .30293 L
.70374 .30285 L
.70499 .3028 L
.70576 .30278 L
.70649 .30277 L
.70786 .30279 L
.70918 .30283 L
.71041 .30289 L
.71176 .30299 L
.7132 .30312 L
.71573 .30341 L
.71811 .30375 L
.72361 .30476 L
.73592 .30762 L
.74196 .30901 L
.74459 .30955 L
.74746 .31007 L
.74998 .31045 L
.75266 .31077 L
.75341 .31083 L
.75412 .31089 L
.75544 .31096 L
.75666 .31101 L
.75798 .31101 L
.75872 .311 L
.75942 .31098 L
.7607 .3109 L
.76141 .31084 L
.76206 .31077 L
.76351 .31058 L
.76466 .31038 L
.76589 .31012 L
.76833 .30944 L
.77049 .30864 L
.7724 .30776 L
.77646 .30531 L
.77993 .30249 L
Mistroke
.783 .29939 L
.79424 .28425 L
.79725 .2802 L
.80067 .27621 L
.80386 .27318 L
.80769 .27039 L
.81188 .2682 L
.81435 .26723 L
.81677 .26646 L
.82141 .26537 L
.82417 .26489 L
.82676 .26452 L
.83986 .26314 L
.84588 .2624 L
.84906 .2619 L
.85245 .26126 L
.85551 .26054 L
.85825 .25977 L
.86336 .25792 L
.86594 .25671 L
.86859 .25522 L
.87293 .25204 L
.87508 .24995 L
.8768 .24789 L
.87842 .24552 L
.87988 .24277 L
.88095 .24007 L
.88133 .23884 L
.8817 .23747 L
.88197 .23624 L
.8822 .23491 L
.88239 .23347 L
.88252 .23212 L
.88259 .23073 L
.88261 .22994 L
.88262 .22919 L
.88258 .22767 L
.88254 .22681 L
.88248 .22602 L
.88236 .22468 L
.88218 .22322 L
.88168 .22027 L
.88095 .217 L
.87998 .21354 L
.87784 .20744 L
.86868 .18679 L
.86693 .18239 L
.86546 .17786 L
.86486 .17553 L
.86433 .17297 L
Mistroke
.86396 .17065 L
.86371 .16846 L
.86362 .1674 L
.86353 .16624 L
.86347 .16513 L
.86343 .16411 L
.86341 .16318 L
.86339 .16215 L
.86338 .16107 L
.86338 .16005 L
.86339 .15893 L
.8634 .15774 L
.86344 .15559 L
.8635 .15069 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl<03?o001lo`040?oo00So1@04o`80C?l200co0P0ao`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0@o`030?oo02ko000oo`L0=_l0103oo`09o`030?l000Go00<0ool0
A_l301?o00<0ool0;Ol003go0P07o`D02ol302Co0P0;o`801Ol00`3oo`15o`030?oo01Co00<0ool0
;Ol003co00<0ool03?l;00?o0P0fo`030?oo04Go00<0ool05Ol00`3oo`0/o`00>_l201ko0P0eo`03
0?oo04Co00<0ool05_l00`3oo`0/o`00>Ol00`3oo`0Oo`80=?l00`3oo`14o`030?oo01Ko00<0ool0
;?l003So00<0ool08Ol00`3oo`0bo`030?oo04?o00<0ool05ol00`3oo`0/o`00=ol00`3oo`0Ro`03
0?oo03;o00<0ool0@ol00`3oo`0Go`030?oo02co000go`030?oo02?o00<0ool0<Ol00`3oo`13o`03
0?oo01Oo00<0ool0;?l003Ko00<0ool09?l00`3oo`0ao`030?oo04?o00<0ool05ol00`3oo`0/o`00
=_l00`3oo`0To`030?oo037o00<0ool0@ol00`3oo`0Go`030?oo02co000eo`030?oo02Go00<0ool0
<Ol00`3oo`14o`030?oo01Ko00<0ool0;?l003Go00<0ool09Ol00`3oo`0ao`030?oo04Co00<0ool0
5_l00`3oo`0/o`00=?l00`3oo`0Uo`030?oo03;o00<0ool0A?l00`3oo`0Fo`030?oo02co000do`03
0?oo02Go00<0ool07_l200Co0P03o`@01Ol00`3oo`14o`030?oo01Ko0P0]o`00<ol00`3oo`0Uo`03
0?oo01ko00@0ool02?l00`3oo`06o`030?oo04Go00<0ool05_l00`3oo`0[o`00<ol00`3oo`0To`80
8?l0103oo`09o`030?oo00Go0P16o`030?oo01Ko00<0ool0:ol003?o00<0ool09?l00`3oo`0Io`@0
0_l0103oo`0:o`030?oo00Co00<0ool0AOl00`3oo`0Go`030?oo02[o000bo`030?oo02Co00<0ool0
8?l0103oo`08o`040?oo00Go00<0ool0AOl00`3oo`0Go`030?oo02[o000bo`030?oo01go00<0ool0
0_l202Co0P0:o`801_l00`3oo`15o`030?oo01So00<0ool0:Ol0037o00<0ool07_l503[o00<0ool0
AOl00`3oo`0Ho`030?oo02Wo000ao`030?oo01oo00<0ool0>ol00`3oo`15o`030?oo01So00<0ool0
:Ol0033o00<0ool08?l00`3oo`0ko`030?oo04Go00<0ool06Ol00`3oo`0Xo`001ol500;o0P04o`80
6_l00`3oo`06o`D05_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`801?l201Go00<0ool0
2ol500Wo00<0ool05_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool0
5ol00`3oo`07o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool04ol00`3oo`0>o`03
0?oo00[o00<0ool05ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05_l00`3oo`09o`03
0?oo01Oo00<0ool01ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo017o00<0ool0
3_l00`3oo`0:o`030?oo01Oo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3o
o`02o`@01?l00`3oo`0Ho`030?oo00@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0Bo`80
4Ol00`3oo`0:o`030?oo01Oo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`03
0?oo01Wo00<0ool01Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00oo0P0Co`03
0?oo00[o00<0ool05ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201co00<0ool0
1Ol200[o0`0Ro`030?oo02;o0P0:o`<03Ol201Co0P0<o`030?oo01Ko0P0:o`<00_l002ko00<0ool0
:?l00`3oo`0eo`030?oo03co0P0To`030?oo02Oo000^o`030?oo02Wo0`03o`@0;Ol00`3oo`0io`<0
9_l00`3oo`0Wo`00;_l00`3oo`0/o`<00ol502Wo00<0ool0=Ol402Wo00<0ool09ol002oo00<0ool0
=_l202Oo00<0ool0<Ol402go00<0ool09ol0033o00<0ool0=_l202Ko00<0ool09_l;033o00<0ool0
:?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00P04o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko0`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Oo000<o`030?oo02Go0`000ol0o`0Zo`030?oo00Ko00<0ool08Ol00`3o
o`0Po`801ol00`3oo`0Yo`030?oo00Go0P0So`030?oo00Oo000go`80=Ol00`3oo`0Po`030?oo01ko
0P0eo`P0;ol003Wo0P0co`030?oo023o00<0ool07Ol00`3oo`0ao`@0=ol003_o0P0bo`030?oo01oo
00<0ool07?l00`3oo`0_o`<0>ol003go0P0ao`030?oo01ko00<0ool06ol00`3oo`0_o`030?oo03co
000oo`<0;ol00`3oo`0Mo`030?oo01Wo0P0ao`030?oo03go0012o`80;_l00`3oo`09o`d01_l00`3o
o`0Ho`030?oo037o00<0ool0?Ol004Co0P0]o`030?oo00?o1@0=o`@00_l00`3oo`0Go`030?oo037o
00<0ool0?_l004Ko0P0/o`D05_l301Oo0P0co`030?oo03oo0018o`80Aol6013o0P0eo`030?oo03oo
001:o`030?oo04Co00<0ool00olA03Go00<0ool0@?l004_o00<0ool0@ol00`3oo`18o`030?oo047o
001<o`030?oo04;o00<0ool0<?l800oo00<0ool0@_l004go00<0ool0@Ol00`3oo`0]o`<02?l400Wo
0P15o`00C_l00`3oo`0/o`801?l200?o1005o`030?oo02[o0`0?o`T0Aol004oo00<0ool0:_l0103o
o`08o`030?oo00Ko00<0ool0:?l206;o001?o`030?oo02[o00@0ool02Ol00`3oo`05o`809ol206Co
001@o`030?oo02Wo00@0ool02_l00`3oo`04o`030?oo02?o0`1Vo`00DOl00`3oo`0Xo`040?oo00So
00@0ool01Ol00`3oo`0Fo`d0JOl005;o00<0ool0:?l200[o0P06o`030?oo00oo1`1fo`00D_l00`3o
o`0=o``08ol00`3oo`0=o`80OOl005?o0`07o`D03?l501ko00<0ool02ol207oo001Eo`P05_l901Go
00<0ool02Ol2087o001lo`D04?l00`3oo`08o`030?oo087o0021o`803_l00`3oo`06o`80Q?l008?o
0P0<o`030?oo00?o0`26o`00QOl200[o00<0ool00`29o`00Qol=08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86591 .15098 m
.86568 .14888 L
.86539 .14664 L
.8647 .14263 L
.86416 .14023 L
.86354 .138 L
.8621 .13398 L
.86012 .13001 L
.85784 .12665 L
.85484 .12338 L
.85109 .12042 L
.84706 .11817 L
.8446 .11714 L
.84224 .11633 L
.83997 .1157 L
.83739 .11515 L
.83489 .11474 L
.8325 .11447 L
.83103 .11434 L
.83024 .11429 L
.82941 .11425 L
.82868 .11422 L
.82787 .11419 L
.82644 .11417 L
.82504 .11418 L
.82425 .11419 L
.82353 .11421 L
.82213 .11426 L
.82079 .11434 L
.81925 .11444 L
.81761 .11458 L
.81465 .1149 L
.80801 .11593 L
.80265 .11704 L
.79788 .11827 L
.79296 .1198 L
.78789 .12175 L
.78027 .12575 L
.77669 .12837 L
.77378 .13112 L
.77103 .13458 L
.76973 .13671 L
.76868 .1388 L
.76784 .14089 L
.76717 .14293 L
.7666 .14516 L
.76636 .14637 L
.76613 .14771 L
.76595 .14894 L
.76582 .15009 L
Mistroke
.76572 .15121 L
.76564 .15242 L
.76558 .15362 L
.76555 .15477 L
.76554 .15581 L
.76555 .15695 L
.76558 .15821 L
.76564 .15962 L
.76572 .16109 L
.76582 .16248 L
.76633 .16769 L
.76793 .17963 L
.7683 .18242 L
.76865 .18533 L
.76891 .18798 L
.76901 .18926 L
.76908 .19041 L
.76913 .19144 L
.76916 .19251 L
.76918 .19357 L
.76917 .19472 L
.76914 .19587 L
.76907 .19708 L
.76898 .19821 L
.76887 .19921 L
.7687 .20036 L
.76851 .2014 L
.76797 .20369 L
.76734 .20563 L
.76651 .2076 L
.76429 .21142 L
.76162 .21465 L
.75782 .21811 L
.75354 .22118 L
.73359 .2313 L
.71207 .23986 L
.70096 .24376 L
.69162 .2465 L
.68329 .24828 L
.67877 .24897 L
.67463 .24948 L
.67013 .24996 L
.66516 .25053 L
.66037 .2512 L
.65568 .25205 L
.651 .25314 L
.64578 .25466 L
.63649 .25826 L
.62551 .264 L
.61593 .27021 L
Mistroke
.60716 .27648 L
.59832 .28253 L
.59412 .285 L
.59009 .28701 L
.58601 .28864 L
.58218 .28983 L
.57797 .2908 L
.57557 .29122 L
.57325 .29155 L
.57119 .29178 L
.56897 .29198 L
.56648 .29216 L
.56413 .29228 L
.56278 .29233 L
.56154 .29237 L
.56009 .29241 L
.55871 .29243 L
.55742 .29245 L
.55622 .29246 L
.55489 .29246 L
.55348 .29246 L
.55205 .29245 L
.55051 .29243 L
.54904 .29241 L
.54769 .29239 L
.54455 .29231 L
.54122 .29222 L
.53525 .29199 L
.52988 .29172 L
.52381 .29132 L
.51825 .29084 L
.51325 .29028 L
.50877 .28966 L
.49903 .28789 L
.48921 .28563 L
.47833 .28309 L
.47278 .282 L
.46729 .28115 L
.46483 .28085 L
.46217 .28058 L
.46062 .28046 L
.45921 .28036 L
.45777 .28028 L
.45641 .28022 L
.45501 .28017 L
.45346 .28014 L
.45194 .28013 L
.4503 .28014 L
.44877 .28017 L
.44732 .28022 L
Mistroke
.44601 .28027 L
.44459 .28035 L
.44132 .28057 L
.43786 .28088 L
.4317 .28161 L
.42005 .28333 L
.41527 .28404 L
.41285 .28435 L
.41035 .28463 L
.409 .28475 L
.4078 .28484 L
.40661 .28491 L
.40551 .28495 L
.40442 .28497 L
.40343 .28497 L
.40236 .28493 L
.40126 .28487 L
.40019 .28477 L
.39907 .28464 L
.39804 .28447 L
.39711 .28428 L
.39491 .2837 L
.3929 .28298 L
.38951 .28128 L
.38649 .27923 L
.37975 .27314 L
.37243 .26551 L
.36344 .25703 L
.35886 .25346 L
.35375 .25012 L
.34874 .24746 L
.34406 .24549 L
.33911 .24388 L
.33656 .24322 L
.33381 .24264 L
.33127 .24219 L
.32896 .24186 L
.32448 .24138 L
.31946 .24101 L
.31679 .24085 L
.31433 .24068 L
.31212 .2405 L
.30978 .24025 L
.30772 .23996 L
.30586 .23962 L
.30385 .23915 L
.302 .23861 L
.29869 .23732 L
.2953 .23547 L
.29178 .23294 L
Mistroke
.28464 .22588 L
.27844 .2182 L
.27283 .2106 L
.26815 .20386 L
.26665 .20138 L
.26614 .20036 L
.26597 .19995 L
.26582 .19953 L
.26573 .19918 L
.26567 .19888 L
.26564 .19858 L
.26565 .19827 L
.2657 .19798 L
.2658 .1977 L
.26592 .19748 L
.2661 .19726 L
.26633 .19704 L
.26664 .19682 L
.26734 .19647 L
.26819 .19616 L
.26912 .19589 L
.27169 .19525 L
.27518 .19434 L
.27841 .1933 L
.28215 .19179 L
.28626 .18965 L
.2901 .18708 L
.29432 .18341 L
.29784 .17938 L
.30107 .17437 L
.3024 .17166 L
.30361 .16855 L
.30442 .1658 L
.30496 .16324 L
.30517 .1619 L
.30526 .16113 L
.30533 .16044 L
.30539 .1597 L
.30543 .15892 L
.30545 .1582 L
.30546 .15751 L
.30544 .15628 L
.30536 .15499 L
.30525 .15391 L
.30508 .15273 L
.30486 .15149 L
.30455 .15019 L
.30386 .14793 L
.30295 .14572 L
.30179 .14355 L
Mistroke
.29924 .14013 L
.29742 .13838 L
.29556 .13698 L
.29378 .13593 L
.29177 .13499 L
.2899 .13434 L
.2878 .13381 L
.28671 .13361 L
.28554 .13344 L
.28497 .13338 L
.28443 .13333 L
.2834 .13326 L
.28243 .13322 L
.2814 .1332 L
.28051 .13321 L
.27953 .13324 L
.27848 .13329 L
.27734 .13337 L
.27526 .13356 L
.27093 .13407 L
.26623 .13467 L
.26392 .13494 L
.2617 .13514 L
.25969 .13529 L
.25854 .13535 L
.25747 .13539 L
.25634 .13542 L
.25527 .13543 L
.25408 .13543 L
.25341 .13542 L
.25279 .13541 L
.25155 .13537 L
.25083 .13534 L
.25017 .1353 L
.24761 .13512 L
.2447 .13484 L
.24199 .13452 L
.23567 .13361 L
.22453 .13188 L
.21867 .13113 L
.2153 .1308 L
.21378 .13068 L
.21214 .13058 L
.21121 .13053 L
.21035 .13049 L
.20953 .13047 L
.20866 .13045 L
.20715 .13043 L
.20549 .13045 L
.20458 .13047 L
Mistroke
.20374 .1305 L
.20277 .13055 L
.20185 .13061 L
.20023 .13075 L
.19848 .13094 L
.19541 .1314 L
.1922 .13207 L
.18924 .13288 L
.18661 .13377 L
.18124 .13612 L
.17593 .13931 L
.17171 .14256 L
.16741 .14664 L
.15896 .15723 L
.1522 .16827 L
.1414 .18903 L
.13646 .19928 L
.13216 .20935 L
.13055 .21407 L
.12989 .2165 L
.1296 .21775 L
.12934 .21905 L
.12915 .22014 L
.12899 .22132 L
.12888 .22238 L
.12881 .22335 L
.12878 .22435 L
.12877 .22524 L
.1288 .22626 L
.12887 .22721 L
.12897 .22809 L
.12911 .22905 L
.12929 .23004 L
.1295 .23095 L
.12996 .23255 L
.13059 .23424 L
.13247 .23791 L
.13469 .24108 L
.14024 .24677 L
.14626 .25145 L
.16059 .2604 L
.18843 .27583 L
.20698 .28448 L
.21761 .28943 L
.22855 .29515 L
.24624 .30769 L
.2592 .32042 L
.26545 .32624 L
.26909 .32907 L
.27275 .33142 L
Mistroke
.27638 .33328 L
.28059 .33492 L
.28253 .3355 L
.28464 .33604 L
.2865 .33642 L
.28858 .33676 L
.29089 .33705 L
.29217 .33717 L
.29337 .33726 L
.29451 .33732 L
.29555 .33736 L
.29676 .33739 L
.29791 .3374 L
.29918 .3374 L
.30053 .33737 L
.30168 .33734 L
.30296 .33728 L
.30583 .33711 L
.30852 .3369 L
.31786 .33593 L
.32795 .33485 L
.3307 .33461 L
.3336 .3344 L
.335 .33432 L
.33632 .33426 L
.33749 .33421 L
.33877 .33418 L
.34001 .33415 L
.34132 .33414 L
.34253 .33414 L
.34365 .33415 L
.34498 .33418 L
.34619 .33422 L
.34758 .33428 L
.3489 .33435 L
.3512 .33451 L
.35364 .33472 L
.35804 .33521 L
.36813 .33668 L
.40834 .34251 L
.44418 .3486 L
.4611 .35355 L
.46626 .35486 L
.47203 .35608 L
.47492 .35659 L
.4777 .35701 L
.48022 .35733 L
.48301 .35762 L
.48445 .35773 L
.486 .35784 L
Mistroke
.48686 .35789 L
.48765 .35792 L
.48919 .35798 L
.4906 .35801 L
.49214 .35802 L
.49359 .35801 L
.49493 .35798 L
.49648 .35792 L
.49792 .35784 L
.49951 .35773 L
.50121 .35758 L
.50465 .35719 L
.50777 .35673 L
.51305 .3557 L
.51863 .35427 L
.52918 .35063 L
.54844 .34179 L
.55723 .33791 L
.56577 .33485 L
.57538 .3323 L
.58079 .3312 L
.58677 .33021 L
.60927 .32757 L
.62067 .32633 L
.62682 .32553 L
.63252 .32466 L
.64309 .32254 L
.65214 .32005 L
.67124 .31276 L
.67925 .30946 L
.68804 .30639 L
.6931 .30505 L
.69536 .30457 L
.69787 .30412 L
.70028 .30378 L
.70255 .30354 L
.70381 .30344 L
.70497 .30336 L
.70632 .3033 L
.7076 .30326 L
.70887 .30325 L
.70959 .30324 L
.71027 .30325 L
.71164 .30327 L
.71241 .3033 L
.71312 .30333 L
.71433 .30339 L
.71565 .30347 L
.71833 .30368 L
.72454 .30438 L
Mistroke
.73037 .30519 L
.736 .30601 L
.73894 .30641 L
.74204 .3068 L
.74474 .30707 L
.74616 .3072 L
.74767 .30731 L
.74905 .30738 L
.7503 .30743 L
.75155 .30746 L
.75273 .30747 L
.75383 .30746 L
.75505 .30742 L
.75631 .30735 L
.75748 .30727 L
.75856 .30716 L
.75955 .30705 L
.76174 .30672 L
.76391 .30627 L
.76623 .30566 L
.77074 .30403 L
.77524 .30174 L
.77924 .29909 L
.78617 .29316 L
.7932 .28601 L
.80065 .27877 L
.80781 .27346 L
.8123 .27104 L
.81682 .26916 L
.82556 .26659 L
.83589 .26437 L
.84732 .26172 L
.85366 .2597 L
.85926 .25733 L
.86422 .25453 L
.86826 .25152 L
.87227 .24749 L
.87394 .24532 L
.87532 .24318 L
.87748 .23879 L
.87836 .23634 L
.87909 .23363 L
.87939 .23224 L
.87965 .23071 L
.87984 .22925 L
.87997 .22789 L
.88005 .22666 L
.8801 .2253 L
.88011 .22387 L
.88008 .22252 L
Mistroke
.88002 .22133 L
.87993 .22003 L
.8798 .21867 L
.87966 .2174 L
.8789 .21274 L
.87776 .20774 L
.8763 .20254 L
.87316 .19286 L
.87019 .18326 L
.86897 .17841 L
.86796 .17313 L
.86762 .17081 L
.86731 .16831 L
.86688 .16375 L
.86591 .15098 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l;03?o001lo`040?oo00So1@04o`80Bol400[o0P0bo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool03_l2033o001lo`040?oo00Wo00<0o`001Ol00`3oo`17o`80
4ol00`3oo`0]o`00?ol803Ko0P0;o`801Ol00`3oo`15o`805Ol00`3oo`0]o`00?Ol300Oo2002o`T0
=ol00`3oo`15o`030?oo01Go00<0ool0;?l003co00<0ool03ol300Wo0P0eo`030?oo04Co00<0ool0
5_l00`3oo`0/o`00>Ol301oo00<0ool0<ol00`3oo`14o`030?oo01Ko0P0]o`00>?l00`3oo`0Qo`03
0?oo03;o00<0ool0@ol201Wo00<0ool0:ol003So00<0ool08Ol203?o00<0ool0@ol00`3oo`0Ho`03
0?oo02_o000go`030?oo02?o00<0ool0<Ol00`3oo`13o`030?oo01So00<0ool0:ol003Ko00<0ool0
9?l00`3oo`0ao`030?oo04?o00<0ool06?l00`3oo`0[o`00=Ol00`3oo`0Uo`030?oo037o00<0ool0
@ol00`3oo`0Ho`030?oo02_o000eo`030?oo02Go00<0ool0<Ol00`3oo`13o`030?oo01So00<0ool0
:ol003Co00<0ool09Ol00`3oo`0bo`030?oo04?o00<0ool06?l00`3oo`0[o`00<ol00`3oo`0Vo`03
0?oo03;o00<0ool0@ol00`3oo`0Ho`030?oo02_o000co`030?oo02Ko00<0ool07_l200Co0P03o`@0
1Ol00`3oo`14o`030?oo01Oo00<0ool0:ol003;o00<0ool09_l00`3oo`0No`040?oo00So00<0ool0
1_l00`3oo`14o`030?oo01So00<0ool0:_l003;o00<0ool09Ol00`3oo`0Oo`040?oo00Wo00<0ool0
1Ol204Go00<0ool06?l00`3oo`0Zo`00<Ol00`3oo`0Uo`030?oo01[o1002o`040?oo00[o00<0ool0
1?l00`3oo`14o`030?oo01So00<0ool0:_l0037o00<0ool09?l00`3oo`0Qo`040?oo00So00@0ool0
1Ol00`3oo`14o`030?oo01Wo00<0ool0:Ol0037o00<0ool08?l402Go0P0:o`801_l00`3oo`14o`03
0?oo01Wo00<0ool0:Ol0033o00<0ool07ol203go00<0ool0A?l00`3oo`0Io`030?oo02Wo000`o`03
0?oo01oo00<0ool0??l00`3oo`14o`030?oo01[o00<0ool0:?l002oo00<0ool08Ol00`3oo`0ko`03
0?oo04Co00<0ool06_l00`3oo`0Xo`001ol500;o0P04o`806Ol00`3oo`07o`D05Ol00`3oo`0:o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l201?o00<0ool03Ol500Wo00<0ool05_l500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Go`030?oo00[o00<0ool05_l00`3oo`08o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool04_l00`3oo`0?o`030?oo00Wo00<0ool06?l00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05Ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00oo00<0ool04?l00`3oo`09o`030?oo01So00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3oo`0Ho`030?oo00@0
0_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0@o`804ol00`3oo`0:o`030?oo01Oo00<0ool0
2?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`03
0?oo02;o00<0ool08Ol0103oo`09o`030?oo00go0P0Eo`030?oo00[o00<0ool05ol00`3oo`08o`03
0?oo00;o0008o`802_l301Oo00<0ool02Ol201_o0P07o`802_l302;o00<0ool08_l200[o0`0;o`80
5_l200co00<0ool05_l200[o0`02o`00;_l00`3oo`0Xo`030?oo03Go00<0ool0>_l202Go00<0ool0
:?l002ko00<0ool0:Ol00`3oo`0do`030?oo03Oo0`0Wo`030?oo02So000_o`030?oo02Wo1@0ao`03
0?oo03Co0`0Zo`030?oo02So000_o`030?oo02go200Zo`030?oo037o0`0/o`030?oo02Wo000`o`80
=Ol302Oo00<0ool0;ol202ko0P0[o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00@00_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01Ol200So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09Ol200;o
00<0ool0:Ol00`3oo`05o`030?oo02;o00<0ool08Ol200Ko00<0ool0:Ol00`3oo`04o`030?oo02?o
00<0ool01ol003Ko00<0ool0=?l202;o00<0ool07ol203Oo100`o`00=ol203Ko00<0ool07ol00`3o
o`0Mo`80<ol603Co000io`80=Ol00`3oo`0No`030?oo01_o0P0do`030?oo03So000ko`80=?l00`3o
o`0Mo`030?oo01[o00<0ool0<Ol303_o000mo`<0<Ol00`3oo`0Mo`030?oo01Wo00<0ool0<Ol00`3o
o`0lo`00@?l2033o0P09o``02?l00`3oo`0Ho`030?oo037o00<0ool0?Ol004;o0P0`o`T03?l70003
o`3o01Oo0P0co`030?oo03ko0014o`80B_l201Go0`0do`030?oo03oo0016o`80BOlF03Ko00<0ool0
@?l004So00<0ool0A_l00`3oo`18o`030?oo047o0019o`80A_l00`3oo`17o`030?oo04;o001;o`80
A?l00`3oo`0ao`P02ol304Go001=o`030?oo047o00<0ool0;_l300So2`18o`00C_l00`3oo`0/o`80
1?l200?o1005o`030?oo02_o0`1No`00Col00`3oo`0Zo`040?oo00So00<0ool01_l00`3oo`0Yo`80
HOl004oo00<0ool0:_l0103oo`09o`030?oo00Go0P0Xo`80Hol0053o00<0ool0:Ol0103oo`0:o`03
0?oo00Co00<0ool08_l506Go001Ao`80:Ol0103oo`08o`040?oo00Go00<0ool07?l606[o001Co`03
0?oo02Oo0P0:o`801_l00`3oo`0Co`T0L?l005Co0P0>o`H09ol00`3oo`0@o`<0NOl005Ko3P06o`H0
8Ol00`3oo`0;o`D0O?l0073o1P0Ko`030?oo00Wo0P21o`00M_l701Co00<0ool01ol208?o001mo`H0
3_l00`3oo`05o`80QOl008?o0P0<o`030?oo00?o0P27o`00QOl700Go00<0o`000`29o`00Rol808go
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86831 .15128 m
.86734 .14698 L
.86622 .14304 L
.86471 .13886 L
.86271 .1346 L
.86014 .13045 L
.85725 .12689 L
.85426 .12404 L
.85062 .12136 L
.84644 .11907 L
.84141 .11711 L
.83872 .11634 L
.83572 .11568 L
.83424 .11542 L
.83281 .11521 L
.83139 .11503 L
.83009 .1149 L
.82886 .11479 L
.82751 .11471 L
.82677 .11467 L
.82608 .11464 L
.82473 .11461 L
.82356 .1146 L
.82226 .1146 L
.82091 .11463 L
.81965 .11468 L
.81893 .11472 L
.81814 .11476 L
.81672 .11486 L
.81405 .1151 L
.81093 .11548 L
.80807 .11592 L
.8031 .11689 L
.79789 .11819 L
.78908 .12126 L
.78444 .12348 L
.78001 .12619 L
.77664 .12884 L
.77343 .1321 L
.77068 .13587 L
.76859 .13982 L
.76714 .14368 L
.7665 .14598 L
.76601 .14818 L
.76559 .15062 L
.76529 .15296 L
.76515 .15433 L
.76506 .1556 L
.76501 .1563 L
.76497 .15708 L
Mistroke
.76494 .15776 L
.76491 .15849 L
.76489 .15927 L
.76487 .16001 L
.76486 .16139 L
.76486 .16218 L
.76486 .16301 L
.76487 .16373 L
.76488 .16452 L
.76491 .16593 L
.76496 .16746 L
.76507 .17021 L
.76534 .17566 L
.76548 .17862 L
.76558 .18137 L
.76562 .18278 L
.76564 .18357 L
.76565 .18429 L
.76565 .18503 L
.76565 .18584 L
.76565 .18667 L
.76563 .18745 L
.76562 .18816 L
.7656 .18883 L
.76554 .19008 L
.76546 .1914 L
.76533 .19282 L
.76505 .19518 L
.76484 .19649 L
.76462 .19769 L
.76406 .20007 L
.76329 .20253 L
.76147 .20675 L
.75927 .21036 L
.75644 .21387 L
.75284 .21738 L
.74504 .22316 L
.72356 .23465 L
.70316 .24366 L
.68669 .2495 L
.67859 .25159 L
.66993 .25336 L
.66067 .25516 L
.65015 .2576 L
.63966 .26082 L
.63007 .26458 L
.61218 .27336 L
.60303 .27813 L
.59359 .28242 L
.58541 .2852 L
Mistroke
.58059 .28639 L
.57592 .28727 L
.57096 .28794 L
.56845 .28821 L
.56608 .28842 L
.56157 .28872 L
.55913 .28885 L
.5565 .28896 L
.55361 .28905 L
.55046 .28913 L
.549 .28915 L
.54746 .28918 L
.54612 .28919 L
.54468 .28921 L
.54328 .28922 L
.54201 .28922 L
.54072 .28923 L
.53951 .28923 L
.53809 .28923 L
.53679 .28922 L
.53536 .28921 L
.53464 .28921 L
.53384 .2892 L
.53117 .28917 L
.52966 .28914 L
.52827 .28912 L
.52547 .28905 L
.52251 .28895 L
.51722 .28871 L
.51253 .28841 L
.50717 .28795 L
.50147 .28731 L
.49127 .2858 L
.48167 .28418 L
.4709 .28253 L
.46762 .28212 L
.46445 .28178 L
.46132 .2815 L
.45838 .28129 L
.45557 .28114 L
.454 .28108 L
.45256 .28104 L
.45179 .28102 L
.45098 .281 L
.45008 .28099 L
.44926 .28098 L
.44778 .28098 L
.44693 .28098 L
.44614 .28099 L
.44462 .28101 L
Mistroke
.44321 .28104 L
.44167 .28109 L
.44004 .28115 L
.43431 .28144 L
.42391 .28218 L
.41881 .28253 L
.41738 .28261 L
.41604 .28267 L
.41484 .28272 L
.41357 .28276 L
.41285 .28277 L
.41219 .28278 L
.41093 .28279 L
.41023 .28278 L
.40948 .28277 L
.40877 .28275 L
.40813 .28273 L
.40699 .28267 L
.40576 .28259 L
.40449 .28248 L
.40333 .28234 L
.40101 .28199 L
.39892 .28156 L
.39668 .28097 L
.39436 .28021 L
.39054 .2786 L
.38641 .27638 L
.37114 .26567 L
.36256 .25977 L
.35281 .2543 L
.3475 .25193 L
.3416 .24979 L
.33131 .24709 L
.32054 .24513 L
.31555 .24424 L
.31115 .24329 L
.3073 .24222 L
.30339 .24081 L
.29959 .23905 L
.29555 .23667 L
.28838 .23111 L
.28112 .22384 L
.27499 .21659 L
.27034 .21024 L
.26832 .20692 L
.26763 .2056 L
.26704 .20424 L
.26678 .20351 L
.26658 .20285 L
.26644 .20225 L
Mistroke
.26638 .20196 L
.26634 .20164 L
.2663 .20131 L
.26628 .20101 L
.26628 .20068 L
.2663 .20037 L
.26636 .19984 L
.26642 .19954 L
.26649 .19928 L
.26664 .19884 L
.26685 .19839 L
.26734 .1976 L
.26812 .19672 L
.26902 .19593 L
.27141 .19435 L
.27462 .19263 L
.28172 .18894 L
.28602 .1864 L
.29008 .18357 L
.29373 .18055 L
.29734 .17687 L
.30018 .1732 L
.30233 .16961 L
.30333 .16748 L
.30412 .16544 L
.30448 .1643 L
.30482 .16305 L
.30505 .16202 L
.30526 .16088 L
.3054 .15982 L
.3055 .15882 L
.30555 .15778 L
.30555 .15666 L
.30551 .15569 L
.30543 .15468 L
.30532 .15383 L
.30516 .15291 L
.3049 .15175 L
.30459 .15069 L
.30388 .14877 L
.30291 .14682 L
.30163 .14485 L
.30033 .14327 L
.29879 .14175 L
.29565 .13943 L
.29382 .13842 L
.29198 .1376 L
.29025 .13696 L
.28835 .1364 L
.28622 .13592 L
Mistroke
.2842 .13558 L
.28295 .13542 L
.28175 .1353 L
.28067 .13521 L
.2795 .13515 L
.27846 .13511 L
.27749 .13509 L
.27642 .13508 L
.27526 .13509 L
.27405 .13511 L
.2729 .13515 L
.27186 .13519 L
.27073 .13524 L
.26249 .13577 L
.25816 .13602 L
.25697 .13608 L
.2557 .13613 L
.25461 .13617 L
.25343 .1362 L
.25275 .13621 L
.25213 .13622 L
.25091 .13624 L
.25022 .13624 L
.24948 .13624 L
.24877 .13623 L
.24811 .13623 L
.24694 .13621 L
.24564 .13618 L
.24427 .13613 L
.24299 .13608 L
.24015 .13593 L
.23741 .13575 L
.23216 .13535 L
.22577 .13482 L
.22293 .1346 L
.21983 .13439 L
.21837 .13431 L
.21683 .13423 L
.21539 .13417 L
.21405 .13413 L
.21279 .1341 L
.21143 .13408 L
.20993 .13407 L
.20856 .13409 L
.20702 .13412 L
.2056 .13418 L
.20404 .13426 L
.20239 .13438 L
.19963 .13465 L
.19661 .13507 L
Mistroke
.19389 .13555 L
.19142 .1361 L
.186 .13767 L
.18118 .13956 L
.17678 .14176 L
.16782 .14787 L
.15938 .15598 L
.14656 .17303 L
.14048 .18302 L
.13486 .19369 L
.13271 .19848 L
.13081 .20346 L
.12955 .20762 L
.12898 .20996 L
.12857 .21211 L
.12841 .21315 L
.12827 .21429 L
.12815 .21544 L
.12811 .21606 L
.12807 .21664 L
.12803 .21771 L
.12802 .21867 L
.12804 .21973 L
.1281 .22084 L
.12818 .22176 L
.12829 .22274 L
.12855 .22449 L
.12899 .22655 L
.12951 .22839 L
.13102 .23238 L
.1332 .23654 L
.13834 .24361 L
.14408 .24961 L
.15094 .2556 L
.16528 .26623 L
.19744 .28406 L
.23483 .3015 L
.25211 .31269 L
.25937 .31796 L
.26715 .32323 L
.27546 .32773 L
.28363 .33087 L
.28782 .33202 L
.29251 .33301 L
.29471 .33337 L
.29706 .3337 L
.30135 .33415 L
.30393 .33435 L
.3064 .33449 L
.30863 .33459 L
Mistroke
.31105 .33466 L
.31231 .3347 L
.3137 .33472 L
.31505 .33474 L
.31651 .33476 L
.31911 .33478 L
.32155 .3348 L
.32404 .33482 L
.32634 .33484 L
.3275 .33486 L
.32874 .33488 L
.33135 .33493 L
.33375 .335 L
.33601 .33509 L
.34022 .33532 L
.34488 .33568 L
.35 .3362 L
.36042 .33764 L
.37956 .34086 L
.42093 .34587 L
.45675 .35147 L
.46687 .35331 L
.47248 .35416 L
.47563 .35456 L
.47864 .35489 L
.48155 .35515 L
.48309 .35527 L
.48472 .35537 L
.4861 .35544 L
.48763 .35551 L
.48901 .35555 L
.49031 .35558 L
.49174 .3556 L
.49309 .3556 L
.4943 .35559 L
.49562 .35557 L
.49705 .35552 L
.49857 .35546 L
.50129 .35531 L
.50414 .35508 L
.50683 .3548 L
.51182 .35413 L
.51708 .3532 L
.52267 .35197 L
.54071 .3465 L
.56039 .33963 L
.5793 .33447 L
.58955 .33236 L
.60112 .33033 L
.62242 .32669 L
Mistroke
.64367 .32186 L
.65365 .3188 L
.66419 .31507 L
.67435 .3113 L
.68404 .30804 L
.69295 .30571 L
.69811 .30473 L
.70044 .30438 L
.70297 .30406 L
.70543 .30382 L
.7077 .30364 L
.70989 .30351 L
.711 .30347 L
.71221 .30343 L
.71357 .30339 L
.71481 .30338 L
.7155 .30337 L
.71625 .30337 L
.71761 .30337 L
.71889 .30339 L
.72025 .30341 L
.72141 .30344 L
.72269 .30348 L
.72845 .30371 L
.73392 .30395 L
.7363 .30405 L
.73883 .30413 L
.73957 .30415 L
.74027 .30416 L
.74158 .30419 L
.7428 .3042 L
.74412 .3042 L
.7454 .30419 L
.74659 .30417 L
.74789 .30413 L
.74862 .3041 L
.74928 .30407 L
.7506 .30399 L
.75203 .30389 L
.7546 .30365 L
.75696 .30334 L
.75943 .30294 L
.76376 .30199 L
.76856 .30053 L
.77289 .29882 L
.78218 .29384 L
.79667 .28367 L
.80488 .27817 L
.81296 .27377 L
.82169 .27012 L
Mistroke
.83174 .26668 L
.84286 .26284 L
.8529 .2584 L
.85754 .25571 L
.86209 .25245 L
.8662 .24873 L
.87007 .24414 L
.87282 .23979 L
.87408 .23726 L
.87512 .23479 L
.87594 .23243 L
.87668 .22982 L
.87722 .22741 L
.8776 .22515 L
.87776 .22395 L
.8779 .22269 L
.878 .2216 L
.87808 .22042 L
.87814 .21916 L
.87818 .21798 L
.87818 .21726 L
.87819 .21658 L
.87817 .2153 L
.87814 .21418 L
.87808 .21299 L
.87802 .21198 L
.87794 .21087 L
.87771 .20841 L
.87743 .20612 L
.87582 .19636 L
.87384 .18628 L
.86831 .15128 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l:03Co001lo`040?oo00So1@04o`80C?l300Wo0`0bo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04So0`0?o`80<?l007co00@0ool02Ol00`3o0005o`030?oo04Oo00<0ool0
4_l00`3oo`0]o`00OOl200_o0P05o`030?oo04Ko00<0ool04ol00`3oo`0]o`00@?l900[o100jo`03
0?oo04Go00<0ool05Ol00`3oo`0/o`00?Ol300Wo2P04o`@0=_l00`3oo`15o`030?oo01Ko00<0ool0
:ol003_o0P0Mo`80=Ol00`3oo`14o`030?oo01Oo00<0ool0:ol003Wo0P0Po`80=?l00`3oo`13o`80
6Ol00`3oo`0[o`00>?l00`3oo`0Qo`030?oo03;o00<0ool0@ol00`3oo`0Ho`030?oo02_o000go`03
0?oo02?o00<0ool0<Ol00`3oo`13o`030?oo01So00<0ool0:ol003Ko00<0ool09?l00`3oo`0ao`03
0?oo04?o00<0ool06?l00`3oo`0[o`00=Ol00`3oo`0Uo`030?oo037o00<0ool0@ol00`3oo`0Ho`03
0?oo02_o000do`030?oo02Go0P0co`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=?l00`3oo`0Uo`03
0?oo03;o00<0ool0@ol00`3oo`0Io`030?oo02[o000co`030?oo02Ko00<0ool0<_l00`3oo`13o`03
0?oo01Wo00<0ool0:_l003;o00<0ool09_l00`3oo`0Oo`801?l200?o1005o`030?oo04?o00<0ool0
6Ol00`3oo`0Zo`00<Ol00`3oo`0Wo`030?oo01ko00@0ool02?l00`3oo`06o`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00<Ol00`3oo`0Uo`808Ol0103oo`09o`030?oo00Go0P14o`030?oo01[o00<0ool0
:Ol0033o00<0ool09?l201go1002o`040?oo00[o00<0ool01?l00`3oo`13o`030?oo01[o00<0ool0
:Ol0033o00<0ool08ol00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`13o`030?oo01[o00<0ool0
:Ol002oo00<0ool08_l202Oo0P0:o`801_l00`3oo`13o`030?oo01[o00<0ool0:Ol002oo00<0ool0
8?l203go00<0ool0@ol00`3oo`0Jo`030?oo02Wo000^o`030?oo027o00<0ool0??l00`3oo`13o`03
0?oo01[o00<0ool0:Ol002ko00<0ool08Ol203go00<0ool0@_l00`3oo`0Lo`030?oo02So0007o`D0
0_l200Co0P0Ho`030?oo00So1@0Eo`030?oo00[o0P04o`801?l202?o00<0ool08_l200Co0P04o`80
4Ol00`3oo`0?o`D02Ol00`3oo`0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool0
2_l00`3oo`0Fo`030?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0@o`03
0?oo017o00<0ool02Ol00`3oo`0Ho`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`03
0?oo00[o00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
3Ol00`3oo`0Bo`030?oo00Wo00<0ool06?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Ho`030?oo00;o1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol200oo00<0ool04ol00`3oo`09o`030?oo01So00<0ool02?l200?o0009o`030?oo00So00<0ool0
5ol00`3oo`0:o`030?oo01So00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo00go00<0ool05?l00`3oo`09o`030?oo01So00<0ool02?l00`3oo`02o`002?l200[o0`0Go`03
0?oo00Wo0P0Ko`030?oo00Ko0P0:o`<08_l00`3oo`0Ro`802_l300_o0P0Fo`802ol00`3oo`0Go`80
2_l300;o000^o`030?oo02Oo0P0go`030?oo03[o0P0Uo`030?oo02So000_o`030?oo02So00<0ool0
=?l00`3oo`0go`<09_l00`3oo`0Yo`00<?l00`3oo`0Xo`030?oo03?o00<0ool0=Ol202Wo00<0ool0
:Ol0033o00<0ool0:Ol5033o00<0ool0<_l302[o00<0ool0:_l0037o00<0ool0;Ol502_o00<0ool0
<?l202go00<0ool0:_l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00@0oooo0P03o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l0103oool200?o
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00?o1P06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01Ol200So
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09_l0103oo`0[o`030?oo00?o
0P0Uo`030?oo02Co0P03o`030?oo02Wo00@0oooo100Uo`030?oo00Oo000fo`030?oo03?o0P0So`03
0?oo027o0`0co`80<ol003Oo00<0ool0=?l00`3oo`0Po`030?oo01ko0`0co`<0=Ol003So0P0eo`80
8?l00`3oo`0Lo`80<ol303So000jo`80=Ol00`3oo`0Mo`030?oo01[o0P0co`80>ol003co0P0do`03
0?oo01co00<0ool06?l203Co00<0ool0>ol003ko0P0co`@01ol:00Wo00<0ool05_l203Co0P0no`00
@?l00`3oo`0co`P02_l600?o00<0ool04ol303Go00<0ool0?_l0047o0P1;oaT0=ol00`3oo`0oo`00
@ol204co00<0ool0BOl00`3oo`10o`00AOl204[o00<0ool0Aol204?o0017o`80B?l00`3oo`14o`<0
AOl004Wo0P16o`030?oo03;o4P18o`00Bol204Co00<0ool0;?l605[o001=o`80;_l200Co0P03o`@0
1Ol00`3oo`0Zo`80H?l004oo00<0ool0:_l0103oo`08o`030?oo00Ko00<0ool09ol306;o001@o`80
:_l0103oo`09o`030?oo00Go0P0Uo`<0IOl005;o0P0Xo`040?oo00[o00<0ool01?l00`3oo`0Po`@0
J?l005Co00<0ool09Ol0103oo`08o`040?oo00Go00<0ool06_l606co001Eo`030?oo02Go0P0:o`80
1_l00`3oo`0Fo`@0L_l005Ko2@0bo`030?oo017o1@1fo`00G_l@02?o00<0ool03ol207_o001^o`P0
6ol00`3oo`0<o`<0OOl007Ko200Co`030?oo00Wo0`20o`00O_l500ko00<0ool01Ol408?o0023o`H0
2?l0103oool408Oo0029o``0Rol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co
001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo
00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go
00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87072 .15158 m
.86929 .14732 L
.86774 .14345 L
.86559 .13903 L
.86332 .13522 L
.85768 .12825 L
.854 .12498 L
.85015 .12229 L
.84585 .11997 L
.84095 .11801 L
.83652 .11672 L
.83391 .11615 L
.83142 .11572 L
.83003 .11553 L
.8285 .11535 L
.82714 .11522 L
.82568 .11511 L
.82493 .11507 L
.82423 .11503 L
.8229 .11498 L
.82166 .11496 L
.82031 .11496 L
.81955 .11496 L
.81884 .11498 L
.81806 .115 L
.81722 .11503 L
.81551 .11513 L
.81395 .11524 L
.81096 .11554 L
.80814 .11592 L
.80331 .11678 L
.79814 .11805 L
.79297 .11971 L
.78839 .12159 L
.78104 .12576 L
.77745 .12857 L
.77452 .13146 L
.77189 .13477 L
.76952 .13865 L
.76771 .14263 L
.7664 .1465 L
.76523 .15131 L
.76473 .15414 L
.76435 .15681 L
.76384 .16184 L
.76366 .16442 L
.76349 .16728 L
.76324 .17234 L
.76312 .17501 L
.76296 .17784 L
Mistroke
.76279 .18054 L
.76259 .18296 L
.76208 .18758 L
.76167 .19025 L
.76112 .19308 L
.76053 .19551 L
.7598 .198 L
.75805 .20258 L
.75601 .20655 L
.75343 .21046 L
.75069 .21382 L
.74329 .22077 L
.7269 .23175 L
.70717 .24241 L
.68999 .25027 L
.67363 .25585 L
.65507 .2606 L
.61719 .27191 L
.60734 .27536 L
.5972 .27864 L
.58779 .28115 L
.58313 .28216 L
.57883 .28294 L
.56984 .2842 L
.56475 .28473 L
.55985 .28515 L
.54935 .28581 L
.54335 .28609 L
.53781 .28631 L
.53268 .28648 L
.52987 .28656 L
.52721 .28661 L
.52477 .28666 L
.52348 .28668 L
.52211 .28669 L
.52098 .2867 L
.51972 .28671 L
.51858 .28671 L
.51752 .28671 L
.51625 .2867 L
.51487 .28669 L
.51417 .28669 L
.51343 .28668 L
.5121 .28666 L
.50952 .2866 L
.50719 .28652 L
.50448 .28642 L
.50156 .28627 L
.49627 .28592 L
.47631 .28396 L
Mistroke
.4653 .28284 L
.45909 .28231 L
.45312 .2819 L
.44749 .28161 L
.44431 .28149 L
.44136 .2814 L
.43621 .28129 L
.43336 .28124 L
.4308 .2812 L
.42581 .28112 L
.42305 .28106 L
.42055 .28099 L
.41838 .28091 L
.41604 .2808 L
.41363 .28064 L
.41141 .28047 L
.40665 .27993 L
.40403 .27953 L
.40163 .27909 L
.39738 .27812 L
.39275 .27677 L
.38477 .27378 L
.37618 .26987 L
.3374 .25358 L
.3201 .24859 L
.31092 .24553 L
.30271 .24191 L
.2948 .23728 L
.28668 .2312 L
.27877 .22384 L
.27269 .21677 L
.27039 .21342 L
.26847 .20988 L
.26775 .2081 L
.26727 .20649 L
.26709 .20562 L
.26697 .20484 L
.26693 .20441 L
.2669 .20395 L
.26689 .20352 L
.26689 .20312 L
.26694 .2024 L
.26698 .202 L
.26704 .20164 L
.26718 .20097 L
.26737 .20028 L
.26782 .1991 L
.26848 .19783 L
.26937 .1965 L
.27053 .1951 L
Mistroke
.27313 .1926 L
.28765 .18197 L
.29566 .1753 L
.29913 .17161 L
.30086 .16942 L
.30223 .16738 L
.30328 .16554 L
.30421 .16353 L
.30463 .16243 L
.30495 .16141 L
.30522 .16037 L
.30542 .1594 L
.30555 .1585 L
.3056 .15799 L
.30563 .15753 L
.30566 .15672 L
.30564 .15587 L
.30557 .15496 L
.30546 .15413 L
.30528 .15323 L
.30502 .15228 L
.30442 .15066 L
.30356 .14898 L
.30254 .14748 L
.30143 .14617 L
.29894 .14391 L
.29579 .14186 L
.29205 .14012 L
.28829 .13887 L
.28621 .13834 L
.28387 .13785 L
.28136 .13744 L
.27898 .13715 L
.27668 .13693 L
.27553 .13685 L
.27447 .13678 L
.27248 .13669 L
.27143 .13665 L
.27031 .13662 L
.26919 .1366 L
.26799 .13659 L
.26685 .13659 L
.26581 .1366 L
.26459 .13661 L
.2639 .13662 L
.26326 .13663 L
.26084 .13668 L
.2557 .13682 L
.25013 .13699 L
.24757 .13706 L
Mistroke
.24485 .13712 L
.24252 .13716 L
.2412 .13718 L
.23996 .1372 L
.23875 .13721 L
.23744 .13722 L
.23625 .13722 L
.23514 .13723 L
.23395 .13723 L
.23263 .13723 L
.23196 .13723 L
.23124 .13723 L
.22993 .13723 L
.22878 .13723 L
.22752 .13723 L
.2262 .13723 L
.22497 .13723 L
.22381 .13724 L
.22276 .13724 L
.22154 .13725 L
.22038 .13726 L
.21914 .13727 L
.2178 .13729 L
.21653 .13732 L
.21538 .13735 L
.21402 .13739 L
.21254 .13744 L
.20984 .13757 L
.20702 .13774 L
.20403 .13799 L
.19866 .13862 L
.19381 .13943 L
.18932 .14041 L
.18448 .14176 L
.1794 .14355 L
.17022 .14789 L
.16098 .15399 L
.15229 .16162 L
.1451 .1696 L
.13867 .17844 L
.13368 .18718 L
.13029 .19522 L
.12889 .1998 L
.12838 .20197 L
.12798 .20404 L
.12781 .2051 L
.12764 .20626 L
.12752 .20733 L
.12743 .20829 L
.12735 .20944 L
Mistroke
.12732 .21007 L
.1273 .21066 L
.12728 .21171 L
.12728 .21285 L
.12731 .21383 L
.12737 .21492 L
.12745 .21605 L
.12756 .2171 L
.12781 .21897 L
.12817 .22097 L
.1287 .22331 L
.12942 .22583 L
.13107 .23034 L
.13523 .23844 L
.14023 .24571 L
.14654 .25314 L
.16075 .26629 L
.17523 .27651 L
.19307 .28599 L
.2323 .30241 L
.25041 .31093 L
.26646 .3189 L
.28263 .32551 L
.29227 .3284 L
.30136 .33047 L
.32043 .33354 L
.34105 .33643 L
.36033 .33972 L
.38209 .34354 L
.39224 .34501 L
.40312 .34629 L
.42231 .34797 L
.44155 .34957 L
.45171 .35056 L
.4628 .35166 L
.46853 .35217 L
.47402 .35259 L
.47902 .35289 L
.48187 .35303 L
.48314 .35308 L
.4845 .35312 L
.4858 .35316 L
.48699 .35318 L
.48819 .3532 L
.48932 .35321 L
.49064 .35322 L
.49186 .35322 L
.4932 .35321 L
.49463 .3532 L
.4959 .35317 L
Mistroke
.49726 .35314 L
.49973 .35305 L
.50266 .35291 L
.50534 .35273 L
.51048 .3523 L
.51591 .35169 L
.52547 .35026 L
.53647 .34809 L
.54615 .34584 L
.58547 .33609 L
.62534 .32651 L
.6654 .31414 L
.67479 .31114 L
.68496 .30824 L
.69391 .30616 L
.70384 .30442 L
.7092 .30373 L
.71422 .30321 L
.72419 .30244 L
.73322 .30187 L
.74292 .30112 L
.74805 .30057 L
.75269 .29992 L
.76163 .29822 L
.76625 .29705 L
.7712 .29556 L
.78086 .29196 L
.81595 .27543 L
.83278 .26787 L
.84172 .26337 L
.85083 .25776 L
.85958 .25057 L
.86325 .24667 L
.86634 .24277 L
.87125 .23458 L
.87325 .22998 L
.87495 .2249 L
.87558 .2225 L
.87615 .21993 L
.87657 .2177 L
.87693 .21529 L
.87726 .21244 L
.87738 .21114 L
.87748 .20976 L
.87756 .20852 L
.87762 .20715 L
.87766 .20592 L
.87769 .20476 L
.8777 .20358 L
.8777 .20291 L
Mistroke
.8777 .20229 L
.87769 .20115 L
.87767 .19995 L
.87764 .19888 L
.8776 .19788 L
.8775 .19559 L
.87735 .19314 L
.87719 .1909 L
.87674 .18585 L
.87579 .17741 L
.87466 .16951 L
.87306 .1608 L
.87072 .15158 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l803Ko001lo`040?oo00So1@04o`80Bol400So100bo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool03_l00`3oo`0_o`00O?l0103oo`09o`030?l000Go00<0ool0
Aol201;o00<0ool0;_l007go0P0;o`801Ol00`3oo`16o`030?oo01?o00<0ool0;Ol0097o00<0ool0
AOl00`3oo`0Eo`030?oo02co0010oaT0>?l00`3oo`14o`030?oo01Oo00<0ool0:ol003go0`0Io`80
=_l00`3oo`14o`030?oo01Oo00<0ool0:ol003_o0P0No`80=?l00`3oo`13o`030?oo01So00<0ool0
:ol003So0`0Qo`030?oo03;o00<0ool0@ol00`3oo`0Io`030?oo02[o000fo`809?l203?o00<0ool0
@ol00`3oo`0Io`030?oo02[o000eo`030?oo02Go00<0ool0<Ol00`3oo`13o`030?oo01Wo00<0ool0
:_l003Co00<0ool09_l00`3oo`0ao`030?oo04?o00<0ool06_l00`3oo`0Yo`00<ol00`3oo`0Vo`80
<ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o00<0ool09_l00`3oo`0bo`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00<_l00`3oo`0Vo`80=?l00`3oo`13o`030?oo01[o00<0ool0:Ol0037o00<0ool0
9_l00`3oo`0Po`801?l200?o1005o`030?oo04;o00<0ool06ol00`3oo`0Yo`00<?l00`3oo`0Vo`03
0?oo023o00@0ool02?l00`3oo`06o`030?oo04;o00<0ool06ol00`3oo`0Yo`00;ol00`3oo`0Vo`03
0?oo027o00@0ool02Ol00`3oo`05o`80@ol00`3oo`0Ko`030?oo02Wo000_o`030?oo02Co0P0No`@0
0_l0103oo`0:o`030?oo00Co00<0ool0@_l00`3oo`0Lo`030?oo02So000^o`030?oo02Co00<0ool0
9?l0103oo`08o`040?oo00Go00<0ool0@_l00`3oo`0Lo`030?oo02So000^o`030?oo02;o0P0Xo`80
2_l200Ko00<0ool0@_l00`3oo`0Lo`030?oo02So000^o`030?oo02;o00<0ool0>ol00`3oo`12o`03
0?oo01co00<0ool0:?l002ko00<0ool08_l00`3oo`0ko`030?oo047o00<0ool07Ol00`3oo`0Xo`00
;Ol202?o0P0mo`030?oo047o00<0ool07Ol00`3oo`0Xo`001ol500;o0P04o`805ol00`3oo`09o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l2013o00<0ool04?l500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03_l00`3oo`0Co`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Ko00<0ool0
2?l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00_o00<0ool05?l00`3oo`09o`03
0?oo01So00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3o
o`0Go`040?ooo`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P0=o`030?oo01Go00<0ool0
2Ol00`3oo`0Ho`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool02_l00`3oo`0Ho`03
0?oo00Ko00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0;o`030?oo01Ko00<0ool0
2?l00`3oo`0Io`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806ol00`3oo`06o`80
2_l302;o00<0ool08_l200[o0`09o`806?l200[o00<0ool06?l200[o0`02o`00;_l00`3oo`0Wo`03
0?oo03Ko00<0ool0>Ol00`3oo`0To`030?oo02Wo000_o`030?oo02Oo00<0ool0=Ol00`3oo`0go`80
9_l00`3oo`0Zo`00;ol00`3oo`0Xo`80=Ol00`3oo`0eo`80:?l00`3oo`0Zo`00<?l00`3oo`0Yo`80
<ol00`3oo`0co`80:Ol00`3oo`0[o`00<Ol00`3oo`0Zo`80<Ol00`3oo`0ao`80:_l00`3oo`0/o`00
3?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l300Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00@0oooo0P03o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00ol0103oo`08o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co00<0ool00_l00`3oo`0Yo`030?oo00<09ol00`3o
o`0Yo`@0:?l00`3oo`02o`030?oo02Go00<0ool01ol003Co0P0do`809Ol00`3oo`0Uo`@0;ol203;o
000fo`030?oo03?o0`0Ro`030?oo02;o0`0ao`80=?l003Oo00<0ool0=Ol301oo00<0ool07_l403;o
0P0fo`00>?l203So0P0Mo`030?oo01_o0`0do`80>?l003[o00<0ool0=ol201_o00<0ool06?l303Go
0P0jo`00>ol203Wo400;o`030?oo01Co100fo`80??l003go0P17o`P00ol00`3oo`0<o`P0=ol303ko
000oo`80COlB03go0P11o`00@Ol204ko00<0ool0A_l304?o0013o`<0Bol00`3oo`14o`80A_l004Ko
0P19o`030?oo03ko1P18o`00B?l304Ko00<0ool0<_l<04ko001;o`80A?l00`3oo`0]o`D0F_l004go
0P0^o`801?l200?o1005o`030?oo02[o0`1Oo`00Col202_o00@0ool02?l00`3oo`06o`030?oo02Oo
0`1Ro`00DOl302So00@0ool02Ol00`3oo`05o`809?l406Go001Do`809_l0103oo`0:o`030?oo00Co
00<0ool08?l306Wo001Fo`809?l0103oo`08o`040?oo00Go00<0ool07Ol306co001Ho`L07_l200[o
0P06o`030?oo01Wo101_o`00Gol502go00<0ool05Ol407?o001To`D0:?l00`3oo`0Ao`@0Mol006Wo
1@0So`030?oo00co1@1ko`00K_l801_o00<0ool02?l4083o001fo`d03_l00`3oo`05o`<0Q?l008?o
5P27o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co
00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o
0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go
0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So
00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3o
o`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87312 .15188 m
.86926 .14386 L
.86393 .13584 L
.8608 .13219 L
.85712 .12864 L
.84968 .12322 L
.845 .12076 L
.8396 .11859 L
.83702 .11778 L
.83423 .11705 L
.83177 .11652 L
.82908 .11605 L
.82645 .1157 L
.82402 .11547 L
.82284 .11538 L
.82155 .11532 L
.82034 .11527 L
.8192 .11525 L
.81851 .11524 L
.81776 .11525 L
.8164 .11527 L
.81508 .11531 L
.81386 .11537 L
.81254 .11546 L
.8111 .11558 L
.8082 .11591 L
.80561 .1163 L
.80282 .11683 L
.79788 .11805 L
.79326 .11957 L
.78919 .12126 L
.78093 .12614 L
.77732 .12914 L
.77384 .13287 L
.77123 .13646 L
.7689 .14056 L
.76722 .1443 L
.7657 .14863 L
.76437 .15353 L
.76322 .15898 L
.76158 .16914 L
.75992 .17956 L
.75875 .18531 L
.75741 .19047 L
.75414 .19945 L
.75203 .2037 L
.74934 .20814 L
.74329 .21582 L
.73678 .22219 L
.71999 .23483 L
Mistroke
.70239 .24567 L
.6858 .25422 L
.66896 .2608 L
.64987 .26611 L
.60818 .27454 L
.58735 .27803 L
.56776 .28057 L
.54812 .2825 L
.53806 .28333 L
.52734 .28411 L
.52185 .28446 L
.51668 .28474 L
.51167 .28497 L
.50947 .28505 L
.50705 .28513 L
.50495 .28519 L
.50268 .28523 L
.50144 .28525 L
.5003 .28527 L
.49921 .28527 L
.49806 .28528 L
.49689 .28528 L
.49564 .28528 L
.49446 .28527 L
.49339 .28527 L
.49213 .28525 L
.49076 .28523 L
.48828 .28517 L
.48586 .2851 L
.48325 .285 L
.47853 .28477 L
.46777 .28402 L
.44607 .2819 L
.4262 .27964 L
.40523 .2768 L
.38512 .27297 L
.36631 .26808 L
.3297 .25555 L
.31263 .24816 L
.29698 .23985 L
.28116 .22874 L
.2744 .22208 L
.27171 .21859 L
.27053 .21672 L
.26945 .21468 L
.26863 .21275 L
.26834 .2119 L
.26806 .21096 L
.26786 .21013 L
.26769 .20924 L
Mistroke
.26762 .20875 L
.26756 .2083 L
.26748 .20743 L
.26745 .20658 L
.26746 .2058 L
.26751 .20495 L
.26762 .20404 L
.26776 .20317 L
.26794 .20236 L
.26838 .20084 L
.26907 .19905 L
.27 .19718 L
.27214 .19385 L
.27885 .18647 L
.29392 .17344 L
.29962 .16794 L
.30201 .16509 L
.30396 .16214 L
.30469 .16066 L
.30501 .15988 L
.30529 .15906 L
.30549 .15831 L
.30562 .15763 L
.30571 .15698 L
.30576 .15631 L
.30576 .15568 L
.30572 .15502 L
.30563 .1544 L
.30552 .15384 L
.30534 .1532 L
.30509 .15251 L
.30449 .15129 L
.30367 .15006 L
.30255 .14876 L
.30124 .14756 L
.29961 .14634 L
.29616 .14435 L
.29239 .14272 L
.2878 .14123 L
.28319 .14008 L
.27784 .1391 L
.27475 .13867 L
.27191 .13835 L
.26935 .13812 L
.26662 .13792 L
.26519 .13784 L
.26363 .13776 L
.26218 .13771 L
.26087 .13767 L
.25953 .13764 L
Mistroke
.25827 .13762 L
.25694 .13761 L
.25618 .1376 L
.25549 .1376 L
.25434 .13761 L
.25307 .13762 L
.25175 .13765 L
.25051 .13767 L
.24781 .13775 L
.24532 .13785 L
.23971 .13815 L
.22968 .13889 L
.21115 .14075 L
.20062 .14216 L
.19069 .14388 L
.18184 .1459 L
.17247 .14874 L
.16362 .15233 L
.15453 .15729 L
.146 .16365 L
.13916 .17065 L
.13627 .17445 L
.13351 .17882 L
.13144 .18286 L
.12962 .18733 L
.12834 .19147 L
.12776 .19387 L
.12732 .19616 L
.127 .19827 L
.12678 .20022 L
.12669 .20125 L
.12662 .20237 L
.12656 .20355 L
.12653 .20466 L
.12653 .20589 L
.12653 .20651 L
.12655 .20719 L
.1266 .2084 L
.12666 .20954 L
.12677 .21088 L
.12689 .2121 L
.12726 .21485 L
.12767 .21722 L
.12822 .21979 L
.12947 .22445 L
.13289 .23353 L
.137 .24162 L
.14283 .25064 L
.14909 .25849 L
.16235 .27131 L
Mistroke
.17861 .28254 L
.19507 .29081 L
.23253 .30432 L
.26816 .31585 L
.30666 .32791 L
.34417 .33806 L
.36346 .34243 L
.3738 .34442 L
.38492 .34623 L
.39491 .34757 L
.406 .34873 L
.41614 .34954 L
.42568 .35011 L
.43583 .35055 L
.44095 .35073 L
.44655 .35088 L
.45136 .35099 L
.45651 .35108 L
.45902 .35111 L
.46138 .35114 L
.46352 .35116 L
.46582 .35117 L
.46713 .35118 L
.46833 .35119 L
.46953 .35119 L
.47067 .35119 L
.472 .35119 L
.47323 .35119 L
.47458 .35119 L
.47527 .35118 L
.47602 .35118 L
.47742 .35117 L
.4787 .35116 L
.4816 .35113 L
.48415 .3511 L
.48684 .35105 L
.49164 .35094 L
.4961 .35081 L
.5062 .35039 L
.51145 .3501 L
.51713 .34973 L
.52738 .34888 L
.54641 .34669 L
.55601 .34525 L
.56652 .34337 L
.58752 .33863 L
.62628 .32655 L
.66302 .31448 L
.7036 .30483 L
.74334 .29758 L
Mistroke
.78115 .2896 L
.7999 .28436 L
.81898 .27708 L
.8279 .27256 L
.83711 .26679 L
.84463 .26104 L
.85227 .25392 L
.86452 .2386 L
.86978 .22951 L
.87371 .22075 L
.87528 .21636 L
.87673 .21153 L
.87778 .20723 L
.87866 .20262 L
.87925 .19843 L
.8795 .19603 L
.87959 .19493 L
.87968 .19376 L
.87974 .19277 L
.87979 .19168 L
.87983 .19068 L
.87986 .18975 L
.87988 .18865 L
.87989 .18765 L
.87988 .18649 L
.87987 .18539 L
.87984 .18439 L
.8798 .18332 L
.8797 .18137 L
.87953 .17905 L
.87932 .17692 L
.87871 .17228 L
.87792 .16786 L
.876 .16012 L
.87312 .15188 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l803Ko001lo`040?oo00So1@04o`80C?l300Oo0`0do`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0=o`<0<Ol007co00@0ool02Ol00`3o0005o`030?oo04Oo0P0Ao`03
0?oo02oo001mo`802ol200Go00<0ool0A_l00`3oo`0Bo`030?oo02ko002Ao`030?oo04Go00<0ool0
5?l00`3oo`0]o`00Aol@03[o00<0ool0AOl00`3oo`0Eo`030?oo02co0010o`L04?l303Oo00<0ool0
A?l00`3oo`0Go`030?oo02_o000ko`D06_l203Go00<0ool0A?l00`3oo`0Ho`030?oo02[o000io`80
8Ol00`3oo`0bo`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=ol202?o0P0co`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol202Ko00<0ool0<Ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o0P0Xo`03
0?oo037o00<0ool0@ol00`3oo`0Ko`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`03
0?oo01co00<0ool0:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02Oo00<0ool0
8?l200Co0P03o`@01Ol00`3oo`12o`030?oo01co00<0ool0:?l002oo00<0ool09_l202;o00@0ool0
2?l00`3oo`06o`030?oo04;o00<0ool07?l00`3oo`0Xo`00;_l00`3oo`0Vo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0No`030?oo02Oo000^o`030?oo02Go00<0ool07Ol400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`030?oo02Co00@0ool0
2?l0103oo`05o`030?oo047o00<0ool07_l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo02Ko0P0:o`80
1_l00`3oo`11o`030?oo01go0P0Yo`00;Ol00`3oo`0So`030?oo03_o00<0ool0@?l00`3oo`0No`03
0?oo02So000]o`030?oo02?o00<0ool0>ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool0
8ol00`3oo`0ko`030?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`805ol00`3oo`09o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool0
2Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`08o`03
0?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3o
o`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3o
o`08o`030?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Ko
0P09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00So
00<0ool06Ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201[o00<0ool01ol200[o
0`0Ro`030?oo02;o0P0:o`<02Ol00`3oo`0Go`802Ol00`3oo`0Io`802_l300;o000_o`030?oo02Go
0P0ho`030?oo03So0P0Uo`030?oo02[o000_o`030?oo02Oo00<0ool0=Ol00`3oo`0go`030?oo02Co
00<0ool0:ol0033o00<0ool09ol00`3oo`0do`030?oo03Go0P0Vo`030?oo02co000`o`030?oo02So
0P0do`030?oo03?o0P0Wo`030?oo02go000ao`030?oo02Wo0P0bo`030?oo03;o00<0ool09ol00`3o
o`0]o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00@0oooo
0`02o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02Wo0`0Zo`03
0?oo02Wo00@0o`00:?l01@3oool002So00<0ool01ol003Co00<0ool0<?l302Oo00<0ool09ol402co
0P0co`00=Ol00`3oo`0bo`809Ol00`3oo`0So`@0;ol00`3oo`0co`00=_l00`3oo`0co`@08Ol00`3o
o`0Oo`@0<_l00`3oo`0do`00=ol00`3oo`0fo`D07?l00`3oo`0Ko`@0=?l203Oo000ho`030?oo03[o
1P0Fo`030?oo01;o2@0eo`<0>Ol003Wo0P10o`/02ol00`3oo`0=o`D0??l203co000ko`030?oo04So
6`0no`<0?_l003co0P1Co`030?oo04Wo0P11o`00?_l3053o00<0ool0AOl404?o0011o`<0COl00`3o
o`10o`D0Aol004Co1019o`030?oo03_o1@1<o`00B?l304Ko00<0ool0=_l5057o001;o`80A?l00`3o
o`0ao`D0E_l004go0`0]o`801?l200?o1005o`030?oo02_o1P1Ko`00D?l202[o00@0ool02?l00`3o
o`06o`030?oo02Oo101Qo`00D_l302Oo00@0ool02Ol00`3oo`05o`809?l406Go001Eo`<09?l0103o
o`0:o`030?oo00Co00<0ool08?l306Wo001Ho`@08?l0103oo`08o`040?oo00Go00<0ool07Ol306co
001Lo`@07Ol200[o0P06o`030?oo01_o0P1_o`00H?l502co00<0ool06?l3077o001Uo`@0:?l00`3o
o`0Fo`80M?l006Wo0P0Vo`030?oo017o1@1fo`00Jol6023o00<0ool02_l707_o001ao`/05Ol00`3o
o`03o`L0P_l007co6`29o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87552 .15217 m
.87078 .14426 L
.86454 .13646 L
.85702 .12948 L
.8492 .12416 L
.8444 .12166 L
.83966 .11967 L
.83526 .11821 L
.83034 .11697 L
.82774 .11647 L
.82485 .11603 L
.82339 .11586 L
.8218 .11571 L
.82031 .1156 L
.81964 .11556 L
.81893 .11553 L
.81822 .1155 L
.81757 .11549 L
.81631 .11547 L
.8156 .11547 L
.81484 .11548 L
.81411 .11549 L
.81345 .11552 L
.81224 .11557 L
.81092 .11566 L
.80954 .11577 L
.80826 .11591 L
.80564 .11626 L
.80324 .11668 L
.79793 .11796 L
.79344 .11946 L
.78891 .12145 L
.78183 .12578 L
.77827 .12874 L
.77526 .13182 L
.77003 .13909 L
.76789 .14314 L
.76587 .14782 L
.76058 .16519 L
.75773 .17591 L
.75459 .18591 L
.75114 .19456 L
.74662 .20339 L
.73521 .21917 L
.72273 .23141 L
.70757 .24318 L
.6918 .25335 L
.67524 .26194 L
.66547 .26587 L
.65637 .26872 L
Mistroke
.65167 .26989 L
.64665 .27092 L
.64229 .27164 L
.63756 .27227 L
.63195 .27282 L
.62664 .27319 L
.62143 .27345 L
.61661 .27364 L
.6072 .27395 L
.597 .27436 L
.59109 .27468 L
.5857 .27505 L
.57515 .27595 L
.5343 .28076 L
.51599 .28287 L
.50542 .28393 L
.50021 .28438 L
.49541 .28473 L
.49062 .28502 L
.48818 .28514 L
.48558 .28524 L
.48421 .28528 L
.48297 .2853 L
.48173 .28533 L
.48055 .28534 L
.47989 .28535 L
.47919 .28535 L
.47793 .28535 L
.47723 .28534 L
.47658 .28533 L
.47587 .28532 L
.4751 .28531 L
.47369 .28527 L
.47238 .28523 L
.46944 .28509 L
.46687 .28493 L
.46416 .2847 L
.45953 .2842 L
.45449 .28349 L
.44502 .28172 L
.43644 .2798 L
.42631 .2775 L
.41657 .27562 L
.40737 .2743 L
.40207 .27373 L
.39717 .27332 L
.38569 .27261 L
.38065 .27233 L
.37516 .27199 L
.37017 .27158 L
Mistroke
.36561 .27109 L
.36091 .27043 L
.35603 .26952 L
.35089 .26826 L
.34619 .2668 L
.33774 .26338 L
.30507 .24558 L
.29497 .24005 L
.28581 .23458 L
.27851 .22923 L
.27504 .22595 L
.27238 .22273 L
.27132 .22115 L
.27034 .2194 L
.26962 .21784 L
.26899 .21611 L
.26867 .21503 L
.26843 .21402 L
.26825 .21305 L
.26811 .212 L
.26803 .21105 L
.26799 .21015 L
.26799 .20916 L
.26803 .20808 L
.26813 .20701 L
.26826 .20601 L
.26845 .2049 L
.26871 .20372 L
.26942 .20127 L
.27025 .19903 L
.27234 .19468 L
.27484 .19061 L
.28129 .18242 L
.28901 .17454 L
.30118 .16316 L
.30333 .16074 L
.3048 .15864 L
.30536 .15755 L
.30558 .15695 L
.30567 .15667 L
.30573 .15641 L
.30583 .15591 L
.30587 .15548 L
.30587 .15501 L
.30582 .15451 L
.30573 .15408 L
.30561 .15368 L
.30544 .15325 L
.3052 .1528 L
.30457 .15188 L
Mistroke
.30381 .15107 L
.30187 .14955 L
.29949 .14815 L
.29233 .14515 L
.28794 .14376 L
.28263 .14235 L
.27781 .14128 L
.27241 .14028 L
.26718 .13952 L
.26235 .13899 L
.25952 .13876 L
.25696 .1386 L
.25554 .13853 L
.2548 .1385 L
.25401 .13848 L
.25258 .13844 L
.25127 .13843 L
.25006 .13842 L
.24878 .13843 L
.24757 .13845 L
.24647 .13848 L
.2452 .13852 L
.24382 .13859 L
.24134 .13874 L
.23873 .13895 L
.23589 .13924 L
.23031 .13999 L
.22072 .14172 L
.18361 .14877 L
.175 .15014 L
.1658 .15193 L
.16061 .15324 L
.15603 .15468 L
.14765 .15831 L
.14378 .16062 L
.13996 .16352 L
.13698 .16635 L
.1341 .16981 L
.13155 .17379 L
.12936 .1784 L
.12855 .18057 L
.12781 .18298 L
.12721 .18533 L
.12676 .18751 L
.12637 .18994 L
.12621 .19121 L
.12606 .19259 L
.12596 .19389 L
.12588 .19509 L
.12583 .19624 L
Mistroke
.12579 .19747 L
.12578 .19881 L
.12578 .19956 L
.12579 .20025 L
.12582 .2016 L
.12588 .20288 L
.12597 .20438 L
.12608 .20577 L
.12641 .20893 L
.12675 .21146 L
.12714 .21383 L
.12825 .21926 L
.1314 .23016 L
.13547 .24023 L
.14082 .25029 L
.15288 .26631 L
.16017 .27343 L
.16856 .28011 L
.18463 .28986 L
.19391 .29417 L
.2041 .2981 L
.22311 .30359 L
.24281 .30762 L
.26438 .31151 L
.27405 .31355 L
.28425 .31614 L
.30205 .32205 L
.32057 .32969 L
.34064 .33778 L
.35011 .34094 L
.36027 .34376 L
.37877 .34754 L
.38795 .34891 L
.39801 .35012 L
.40771 .35103 L
.41247 .35138 L
.41684 .35165 L
.42186 .35188 L
.42464 .35198 L
.42728 .35205 L
.42848 .35208 L
.42974 .3521 L
.43083 .35211 L
.43202 .35212 L
.43331 .35213 L
.43469 .35213 L
.436 .35213 L
.4372 .35212 L
.4386 .35211 L
.43938 .3521 L
Mistroke
.44009 .35209 L
.44277 .35203 L
.44591 .35194 L
.44886 .35184 L
.45418 .35158 L
.45991 .35124 L
.48077 .34964 L
.49142 .34884 L
.49718 .34848 L
.50261 .3482 L
.50773 .348 L
.51044 .34791 L
.51332 .34784 L
.51574 .34779 L
.51842 .34774 L
.52318 .34768 L
.52812 .34763 L
.53275 .34758 L
.53763 .34751 L
.54023 .34745 L
.54301 .34738 L
.5459 .34729 L
.54863 .34718 L
.55374 .3469 L
.55847 .34656 L
.56357 .34607 L
.5691 .3454 L
.57487 .3445 L
.58492 .34242 L
.59399 .33989 L
.60329 .33665 L
.61929 .32989 L
.6365 .32251 L
.65646 .31584 L
.6768 .31105 L
.69591 .30696 L
.73406 .29631 L
.74414 .29354 L
.75525 .29107 L
.766 .28934 L
.77607 .28819 L
.78564 .28726 L
.79581 .28609 L
.80153 .28519 L
.80662 .28413 L
.81159 .2828 L
.81613 .28126 L
.82467 .27727 L
.82868 .27479 L
.83287 .27171 L
Mistroke
.84543 .25897 L
.85652 .24368 L
.8673 .22714 L
.87211 .21915 L
.87678 .21018 L
.87861 .20592 L
.88028 .20126 L
.88155 .19671 L
.88241 .19248 L
.88273 .19031 L
.88299 .18789 L
.88309 .18667 L
.88313 .18599 L
.88316 .18534 L
.8832 .18409 L
.88322 .18294 L
.88321 .18186 L
.88319 .1807 L
.88314 .17971 L
.88308 .17862 L
.88299 .17745 L
.88287 .17618 L
.88258 .17388 L
.88217 .17138 L
.8816 .16865 L
.8801 .16318 L
.8783 .15821 L
.87628 .15369 L
.87552 .15217 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l703Oo001lo`040?oo00So1@04o`80C?l300Ko100do`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0=o`030?oo037o001lo`040?oo00Wo00<0o`001Ol00`3oo`17o`80
4?l2037o001mo`802ol200Go00<0ool0Aol00`3oo`0Ao`030?oo02ko002Ao`030?oo04Ko00<0ool0
4ol00`3oo`0]o`00Bol803ko00<0ool0A?l201Oo00<0ool0;?l004Co1`08o`D0>Ol00`3oo`14o`03
0?oo01Oo00<0ool0:ol003oo1@0Do`<0=_l00`3oo`14o`030?oo01So00<0ool0:_l003[o1@0Lo`80
=?l00`3oo`13o`030?oo01Wo00<0ool0:_l003Ko100Ro`80<ol00`3oo`13o`030?oo01[o0P0Zo`00
=Ol00`3oo`0Uo`030?oo037o00<0ool0@_l00`3oo`0Lo`030?oo02So000bo`<09ol203?o00<0ool0
@_l00`3oo`0Lo`030?oo02So000ao`030?oo02So00<0ool0<_l00`3oo`12o`030?oo01go00<0ool0
9ol0033o00<0ool09ol203Go00<0ool0@_l00`3oo`0Mo`030?oo02Oo000_o`030?oo02Oo00<0ool0
=Ol00`3oo`11o`030?oo01ko00<0ool09ol002ko00<0ool09ol00`3oo`0Ro`801?l200?o1005o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Vo`030?oo02;o00@0ool02?l00`3oo`06o`03
0?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0Uo`030?oo02?o00@0ool02Ol00`3oo`05o`80
@Ol00`3oo`0Oo`030?oo02Oo000]o`030?oo02Ko00<0ool07Ol400;o00@0ool02_l00`3oo`04o`03
0?oo043o00<0ool07ol00`3oo`0Wo`00;Ol00`3oo`0Uo`030?oo02Co00@0ool02?l0103oo`05o`03
0?oo03oo00<0ool08?l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo02Ko0P0:o`801_l00`3oo`0oo`03
0?oo023o00<0ool09ol002go00<0ool08ol00`3oo`0ko`030?oo03oo00<0ool08?l00`3oo`0Wo`00
;Ol00`3oo`0So`030?oo03_o00<0ool0?_l00`3oo`0Qo`030?oo02Oo000]o`030?oo02?o00<0ool0
>ol00`3oo`0no`030?oo023o00<0ool0:?l000Oo1@02o`801?l201Oo00<0ool02Ol501Go00<0ool0
2_l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0=o`030?oo01?o1@09o`030?oo01Ko1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool05_l00`3oo`0;o`030?oo01Go00<0ool02Ol0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`08o`040?oo00_o00<0ool05_l00`3oo`08o`030?oo01Wo00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Eo`030?oo00Wo00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`09o`030?oo01Ko00<0ool02?l00`3oo`0Io`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05Ol200Co
1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200_o00<0ool05ol00`3oo`07o`030?oo01[o
00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Ko00<0ool02?l0103o
o`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Wo00<0ool06?l00`3oo`07o`030?oo01[o
00<0ool02?l00`3oo`02o`002?l200[o0`0Go`030?oo00Wo0P0Io`802Ol200[o0`0Ro`030?oo02;o
0P0:o`<02?l00`3oo`0Ho`802?l00`3oo`0Jo`802_l300;o000^o`030?oo02Ko00<0ool0=ol00`3o
o`0ho`030?oo02;o00<0ool0;?l002oo00<0ool09_l203Oo00<0ool0=ol00`3oo`0So`030?oo02co
000`o`030?oo02Oo00<0ool0=?l00`3oo`0eo`809Ol00`3oo`0]o`00<?l00`3oo`0Xo`80=?l00`3o
o`0do`030?oo02Co00<0ool0;_l0037o00<0ool0:Ol00`3oo`0ao`030?oo03?o00<0ool09?l00`3o
o`0_o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool00_l200;o00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`03o`8000?o0?l01ol00`3oo`05o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0
9ol00`3o000/o`030?oo02Wo00@0oooo0P0Vo`040?oo02Wo00<0ool01ol003?o00<0ool0;Ol202co
00<0ool0:ol202[o00<0ool0<_l003Co00<0ool0;_l402So00<0ool0:?l302co00<0ool0<_l003Go
00<0ool0<Ol402Co00<0ool09Ol302ko00<0ool0<ol003Ko00<0ool0=?l=01Oo00<0ool05_l?033o
00<0ool0=?l003Oo00<0ool0@?l501;o00<0ool03?l:03co0`0go`00>?l00`3oo`14o`D03Ol00`3o
o`04o`P0A?l203[o000io`80BOlD04So100lo`00>ol205Co00<0ool0@_l:043o000mo`030?oo057o
00<0ool0?ol304[o000no`<0D?l00`3oo`0jo`D0COl0047o101<o`030?oo03Ko101Bo`00AOl504Oo
00<0ool0<ol305Ko001:o`D0@_l00`3oo`0_o`@0FOl004oo100Zo`801?l200?o1005o`030?oo02_o
101Mo`00Dol302Ko00@0ool02?l00`3oo`06o`030?oo02So0`1Qo`00E_l402;o00@0ool02Ol00`3o
o`05o`809Ol406Co001Jo`@07_l0103oo`0:o`030?oo00Co00<0ool08?l406So001No`807?l0103o
o`08o`040?oo00Go00<0ool07_l206co001Po`<06_l200[o0P06o`030?oo01_o0`1^o`00Hol202co
00<0ool06Ol2077o001Uo`<0:Ol00`3oo`0Go`80Lol006So0P0Wo`030?oo01Go0P1eo`00J_l402?o
00<0ool03_l707Oo001^o`H06OlE07ko001doaT01?l00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3o
o`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So
0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87793 .15247 m
.8723 .14467 L
.86515 .13708 L
.85692 .13031 L
.84873 .12509 L
.84384 .12258 L
.83909 .12053 L
.83467 .11895 L
.82988 .1176 L
.82737 .11703 L
.82457 .11651 L
.82202 .11614 L
.81961 .11588 L
.81815 .11577 L
.81743 .11573 L
.81676 .1157 L
.81552 .11565 L
.81486 .11564 L
.81416 .11564 L
.8134 .11564 L
.81258 .11565 L
.8118 .11568 L
.81109 .11571 L
.80974 .11579 L
.80832 .1159 L
.80581 .1162 L
.80346 .11657 L
.79914 .11755 L
.79463 .11898 L
.78998 .12098 L
.78616 .12309 L
.78238 .12568 L
.77638 .13116 L
.7711 .13798 L
.7665 .14621 L
.75978 .1626 L
.75241 .18109 L
.74254 .20074 L
.73182 .21757 L
.72147 .23053 L
.70904 .24256 L
.69249 .25509 L
.67475 .26566 L
.66966 .26811 L
.66424 .27038 L
.65914 .27215 L
.65449 .27345 L
.65222 .27397 L
.64975 .27445 L
.64739 .27482 L
Mistroke
.64525 .27509 L
.64275 .27533 L
.64137 .27542 L
.64008 .27549 L
.63887 .27553 L
.63776 .27555 L
.63654 .27556 L
.63525 .27556 L
.63411 .27553 L
.63307 .2755 L
.63187 .27546 L
.63072 .2754 L
.62645 .27511 L
.62174 .27467 L
.61652 .2741 L
.61128 .27352 L
.60554 .27293 L
.60041 .2725 L
.59759 .27231 L
.59497 .27216 L
.59219 .27205 L
.59072 .272 L
.58997 .27198 L
.58915 .27197 L
.58772 .27195 L
.58637 .27194 L
.58505 .27194 L
.58384 .27195 L
.58238 .27198 L
.58101 .27201 L
.57952 .27206 L
.57792 .27213 L
.57468 .2723 L
.57176 .27251 L
.56663 .27298 L
.5613 .27359 L
.55202 .2749 L
.53202 .27825 L
.51287 .2815 L
.49435 .28436 L
.48929 .285 L
.48461 .2855 L
.48207 .28571 L
.47979 .28587 L
.47858 .28594 L
.47727 .28601 L
.47656 .28603 L
.47591 .28606 L
.47462 .28609 L
.47344 .2861 L
Mistroke
.47219 .2861 L
.47101 .28609 L
.46994 .28606 L
.4687 .28602 L
.46736 .28595 L
.4661 .28586 L
.46495 .28576 L
.46245 .28548 L
.46022 .28516 L
.45534 .28421 L
.45114 .28311 L
.44677 .28174 L
.43892 .27894 L
.43001 .27585 L
.4248 .27436 L
.41963 .2732 L
.41719 .27277 L
.41454 .27238 L
.41314 .27221 L
.41157 .27205 L
.41011 .27193 L
.40877 .27183 L
.40752 .27176 L
.40619 .2717 L
.40486 .27166 L
.40338 .27164 L
.40209 .27164 L
.40071 .27165 L
.3994 .27167 L
.39821 .27171 L
.39689 .27176 L
.39549 .27183 L
.39298 .27198 L
.38729 .27245 L
.37713 .27349 L
.37179 .27401 L
.36924 .27421 L
.36692 .27435 L
.36566 .2744 L
.36433 .27445 L
.36357 .27446 L
.36287 .27447 L
.36157 .27447 L
.36039 .27444 L
.35932 .2744 L
.35816 .27434 L
.35695 .27425 L
.35495 .27403 L
.35286 .27371 L
.35094 .27332 L
Mistroke
.34924 .2729 L
.34519 .27156 L
.34151 .26993 L
.33406 .26555 L
.32601 .25991 L
.30986 .24942 L
.29077 .23952 L
.28301 .23524 L
.27936 .2328 L
.27592 .22997 L
.27426 .22827 L
.27291 .22665 L
.27077 .22322 L
.26997 .22142 L
.26962 .22044 L
.2693 .21937 L
.26906 .21842 L
.26884 .21735 L
.26869 .21636 L
.26858 .21541 L
.2685 .21423 L
.26848 .21356 L
.26848 .21293 L
.2685 .21175 L
.26859 .21049 L
.26871 .20934 L
.26886 .20824 L
.26934 .2057 L
.27006 .20291 L
.2709 .2003 L
.27337 .19435 L
.27923 .18411 L
.28573 .17542 L
.29982 .16121 L
.30233 .15901 L
.30437 .15708 L
.30505 .15633 L
.30534 .15594 L
.30557 .1556 L
.30574 .1553 L
.30586 .15504 L
.30594 .15475 L
.30599 .15446 L
.30598 .1542 L
.30594 .15394 L
.30585 .1537 L
.30573 .15348 L
.30555 .15322 L
.3053 .15295 L
.30471 .15246 L
Mistroke
.30397 .15199 L
.30302 .15149 L
.30086 .15054 L
.29427 .1482 L
.28478 .14536 L
.27429 .14271 L
.26939 .14167 L
.26387 .14068 L
.25793 .13988 L
.2547 .13956 L
.25326 .13945 L
.25169 .13935 L
.2501 .13928 L
.24924 .13925 L
.24844 .13923 L
.24701 .13921 L
.24551 .13921 L
.24466 .13923 L
.24375 .13925 L
.2429 .13928 L
.24211 .13932 L
.24065 .1394 L
.23911 .13952 L
.23654 .13978 L
.23385 .14014 L
.22918 .141 L
.22391 .14229 L
.21935 .14364 L
.21082 .14652 L
.20631 .148 L
.20209 .14924 L
.19816 .15019 L
.19448 .15089 L
.19028 .15146 L
.18801 .15168 L
.18558 .15186 L
.18339 .15198 L
.18215 .15203 L
.18098 .15207 L
.17883 .15212 L
.17773 .15215 L
.17651 .15216 L
.17381 .1522 L
.17125 .15223 L
.16891 .15226 L
.1676 .15229 L
.1664 .15232 L
.16527 .15236 L
.16406 .1524 L
.1619 .15251 L
Mistroke
.15936 .15269 L
.15707 .15291 L
.15494 .15318 L
.15268 .15354 L
.14871 .15439 L
.14664 .15499 L
.14478 .15562 L
.1408 .1574 L
.13732 .15958 L
.13407 .16243 L
.13151 .16555 L
.1295 .1689 L
.12804 .17222 L
.1268 .17614 L
.1263 .17827 L
.12586 .18065 L
.12553 .18297 L
.12531 .18515 L
.12521 .18643 L
.12514 .18763 L
.12508 .18897 L
.12505 .18972 L
.12504 .19041 L
.12502 .19181 L
.12502 .19261 L
.12503 .19335 L
.12506 .19472 L
.12508 .19541 L
.12511 .19618 L
.12518 .19773 L
.12527 .19922 L
.12549 .20203 L
.1258 .20509 L
.12622 .20846 L
.12718 .21457 L
.12831 .22026 L
.13156 .23253 L
.13572 .24371 L
.14043 .25324 L
.14659 .26278 L
.15366 .27126 L
.16228 .27938 L
.17199 .28659 L
.18105 .292 L
.18949 .29614 L
.19863 .29977 L
.20837 .30271 L
.21324 .30384 L
.21791 .30474 L
.22269 .30548 L
Mistroke
.22719 .30604 L
.23233 .30655 L
.23802 .30698 L
.24996 .30764 L
.2559 .30796 L
.26132 .30831 L
.26662 .30874 L
.27158 .30927 L
.27673 .30998 L
.2822 .31096 L
.28769 .31224 L
.29332 .31389 L
.30297 .31762 L
.33735 .33644 L
.34652 .34079 L
.35535 .34417 L
.36563 .34715 L
.37505 .34916 L
.38422 .35065 L
.39414 .35191 L
.40288 .35278 L
.40762 .35317 L
.41292 .35351 L
.41546 .35365 L
.41821 .35376 L
.41975 .35381 L
.42115 .35385 L
.42257 .35388 L
.42391 .35391 L
.42517 .35392 L
.4265 .35393 L
.42764 .35393 L
.42891 .35392 L
.43027 .35391 L
.43174 .35389 L
.43313 .35386 L
.43441 .35382 L
.43747 .35371 L
.44033 .35356 L
.44683 .35311 L
.45255 .35256 L
.45861 .35186 L
.48065 .34854 L
.49136 .34697 L
.49655 .34637 L
.49949 .34609 L
.50215 .34589 L
.50339 .34582 L
.50472 .34574 L
.50588 .34569 L
Mistroke
.50715 .34565 L
.5085 .34561 L
.50924 .3456 L
.50993 .34559 L
.51118 .34558 L
.51188 .34558 L
.51252 .34558 L
.51369 .34559 L
.51479 .3456 L
.51599 .34563 L
.51729 .34567 L
.51995 .34577 L
.52239 .34589 L
.53261 .34659 L
.54244 .3473 L
.54731 .34759 L
.55014 .34773 L
.55273 .34782 L
.55413 .34786 L
.55546 .34789 L
.55687 .34792 L
.55767 .34792 L
.5584 .34793 L
.55975 .34793 L
.56097 .34792 L
.56238 .3479 L
.5637 .34787 L
.56504 .34783 L
.5665 .34777 L
.56942 .3476 L
.57211 .34739 L
.57458 .34714 L
.58004 .34641 L
.58568 .34533 L
.59081 .34403 L
.59535 .34261 L
.60493 .33869 L
.62083 .33004 L
.62853 .32577 L
.63724 .32159 L
.64603 .31825 L
.65458 .31579 L
.6637 .31379 L
.67394 .31194 L
.69261 .30831 L
.70274 .30552 L
.71167 .30236 L
.73122 .29406 L
.74009 .29067 L
.74507 .28912 L
Mistroke
.74982 .28793 L
.7551 .28694 L
.75796 .28654 L
.75944 .28638 L
.76106 .28622 L
.76255 .28611 L
.76394 .28602 L
.76527 .28595 L
.7667 .28589 L
.76797 .28586 L
.76918 .28583 L
.77051 .28582 L
.77194 .28582 L
.77334 .28583 L
.77462 .28585 L
.7761 .28588 L
.7775 .28592 L
.78272 .28611 L
.78573 .28623 L
.78891 .28634 L
.79022 .28638 L
.79162 .28642 L
.79242 .28644 L
.79316 .28645 L
.79459 .28647 L
.79603 .28647 L
.79736 .28646 L
.79882 .28643 L
.79963 .28641 L
.80036 .28638 L
.8017 .28632 L
.80313 .28623 L
.8057 .28601 L
.80711 .28585 L
.80863 .28565 L
.81165 .28512 L
.81437 .2845 L
.81682 .28379 L
.82146 .28201 L
.82612 .27951 L
.83015 .27661 L
.83355 .27352 L
.83919 .26677 L
.84427 .25875 L
.8627 .22788 L
.8748 .21261 L
.87967 .2054 L
.88188 .20133 L
.88387 .19678 L
.8847 .19437 L
Mistroke
.88536 .19206 L
.88589 .18977 L
.88627 .18762 L
.88654 .18555 L
.88663 .1845 L
.88671 .18334 L
.88675 .18217 L
.88677 .1809 L
.88675 .17971 L
.8867 .17861 L
.88662 .17738 L
.8865 .17606 L
.88617 .17369 L
.8857 .17113 L
.88501 .16832 L
.88338 .16335 L
.88141 .15879 L
.87793 .15247 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l603So001lo`040?oo00So1@04o`80C?l300Ko0`0eo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0<o`80<ol007co00@0ool02Ol00`3o0005o`030?oo04So00<0ool0
3_l2037o001mo`802ol200Go00<0ool0A_l201?o00<0ool0;_l0097o00<0ool0AOl00`3oo`0Do`80
;_l0097o00<0ool0AOl00`3oo`0Fo`030?oo02_o0017o`l0>ol00`3oo`14o`030?oo01So00<0ool0
:_l004Co0`0?o`<0>?l00`3oo`13o`030?oo01[o00<0ool0:Ol0043o100Eo`<0=Ol00`3oo`13o`03
0?oo01[o00<0ool0:Ol003Co300Lo`80<ol00`3oo`13o`030?oo01_o00<0ool0:?l0037o0`0Xo`80
<ol00`3oo`12o`030?oo01co00<0ool0:?l0033o00<0ool0:?l203Co00<0ool0@_l00`3oo`0Mo`03
0?oo02Oo000_o`030?oo02So00<0ool0=?l00`3oo`12o`030?oo01go00<0ool09ol002ko00<0ool0
:Ol00`3oo`0do`030?oo04;o00<0ool07_l00`3oo`0Vo`00;_l00`3oo`0Xo`030?oo03Go00<0ool0
@Ol00`3oo`0Oo`030?oo02Ko000]o`030?oo02So00<0ool08_l200Co0P03o`@01Ol00`3oo`11o`03
0?oo01oo00<0ool09_l002go00<0ool09ol00`3oo`0Ro`040?oo00So00<0ool01_l00`3oo`10o`03
0?oo023o00<0ool09_l002go00<0ool09_l00`3oo`0So`040?oo00Wo00<0ool01Ol2047o00<0ool0
8?l00`3oo`0Vo`00;Ol00`3oo`0Uo`030?oo01ko1002o`040?oo00[o00<0ool01?l00`3oo`0oo`03
0?oo027o00<0ool09_l002go00<0ool09Ol00`3oo`0To`040?oo00So00@0ool01Ol00`3oo`0oo`03
0?oo027o00<0ool09_l002go00<0ool09?l00`3oo`0Vo`802_l200Ko00<0ool0?_l00`3oo`0Ro`03
0?oo02Ko000]o`030?oo02Co00<0ool0>_l00`3oo`0no`030?oo02;o00<0ool09_l002go00<0ool0
8ol203co00<0ool0?Ol00`3oo`0Ro`030?oo02Oo000]o`030?oo02?o00<0ool0>ol00`3oo`0lo`03
0?oo02;o00<0ool0:?l000Oo1@02o`801?l201Oo00<0ool02Ol501Go00<0ool02_l200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0;o`030?oo01Go1@09o`030?oo01Ko1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool05_l00`3oo`0;o`030?oo01Go00<0ool02Ol0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00[o00<0ool05ol00`3oo`08o`030?oo01Wo00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Co00<0ool02ol00`3oo`0Eo`030?oo00Wo00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`07o`030?oo01So00<0ool01ol00`3oo`0Jo`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05Ol00`3oo`03o`@00_l0103o
o`09o`808ol00`3oo`0Qo`040?oo00Wo0P09o`030?oo01Wo00<0ool01ol00`3oo`0Jo`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02Ol00`3o
o`0Ro`030?oo027o00@0ool02Ol00`3oo`07o`030?oo01[o00<0ool01_l00`3oo`0Ko`030?oo00So
00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806?l00`3oo`09o`802_l302;o00<0ool08_l200[o
0`07o`030?oo01Wo0P07o`030?oo01_o0P0:o`<00_l002ko00<0ool09?l203[o00<0ool0=ol00`3o
o`0Ro`030?oo02go000_o`030?oo02Go0P0ho`030?oo03Ko00<0ool08ol00`3oo`0]o`00;ol00`3o
o`0Wo`030?oo03Go00<0ool0=Ol00`3oo`0So`030?oo02ko000`o`030?oo02Oo0P0eo`030?oo03Co
00<0ool09?l00`3oo`0^o`00<?l00`3oo`0Yo`80<ol00`3oo`0co`030?oo02Co00<0ool0;ol000co
o`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`040?ooo`800ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Co
00<0o`002?l00`3oo`05o`030?oo00Ko00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0So`030?oo00?o00<0ool09_l0103oo`0/o`03
0?oo02Wo00<0ool00_l202Go00@0ool0:Ol00`3oo`07o`00<_l00`3oo`0]o`80;Ol00`3oo`0/o`80
:_l00`3oo`0ao`00<ol00`3oo`0^o`80:ol00`3oo`0[o`030?oo02Wo00<0ool0<_l003?o00<0ool0
<?l00`3oo`0Xo`030?oo02So0`0[o`030?oo03?o000do`030?oo033o1@000ol0000?01?o00<0ool0
3?lA00So0`0]o`030?oo03Co000eo`80=Ol00`3oo`0?o`804Ol00`3oo`08o`@04Ol802oo00<0ool0
=Ol003Oo00<0ool0A_l400go2`1;o`030?oo03Ko000ho`030?oo04Wo1004o`H0DOl403Wo000io`80
C?l500Go00<0ool0?OlB03go000ko`80E?l00`3oo`0ko`80Col003go0P1Bo`030?oo03Wo0P1Ao`00
?ol304oo00<0ool0=ol205?o0012o`<0C?l00`3oo`0do`<0EOl004Go3010o`030?oo037o0`1Ho`00
DOl702Go0P04o`800ol400Go00<0ool0:_l705_o001Ho`<08Ol0103oo`08o`030?oo00Ko00<0ool0
9Ol506;o001Ko`807ol0103oo`09o`030?oo00Go0P0So`<0Iol005go0P0Mo`040?oo00[o00<0ool0
1?l00`3oo`0Po`80J_l005oo0P0Ko`040?oo00So00@0ool01Ol00`3oo`0No`80K?l0067o0P0Jo`80
2_l200Ko00<0ool07Ol00`3oo`1/o`00Hol202co00<0ool06ol206oo001Uo`80:_l00`3oo`0Io`80
LOl006Oo0P0Xo`030?oo01Oo0P1co`00JOl302Go1P0@o`@0MOl006co1@0Go`X01?lA07Wo001ao`P0
2Ol600Wo00<0ool0S?l007Wo2@0?o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88033 .15277 m
.87382 .14508 L
.86576 .1377 L
.84826 .12602 L
.83863 .12143 L
.83322 .11941 L
.82804 .11786 L
.82314 .11677 L
.82036 .11632 L
.81903 .11615 L
.81778 .11602 L
.81664 .11592 L
.8156 .11585 L
.81445 .11579 L
.81321 .11576 L
.81202 .11575 L
.81135 .11576 L
.81073 .11578 L
.80959 .11582 L
.80838 .1159 L
.80719 .116 L
.80588 .11615 L
.80326 .11654 L
.80087 .11703 L
.79867 .1176 L
.79384 .11925 L
.78895 .12159 L
.78467 .12429 L
.78092 .12724 L
.77473 .13368 L
.76951 .14102 L
.76472 .14946 L
.75661 .16556 L
.73742 .19991 L
.72746 .21737 L
.71909 .23101 L
.70898 .24316 L
.694 .25626 L
.68533 .26259 L
.67548 .26885 L
.66599 .27375 L
.66189 .27543 L
.6574 .27693 L
.65512 .27753 L
.65268 .27807 L
.65133 .27831 L
.6501 .2785 L
.64886 .27866 L
.6477 .27878 L
.64664 .27886 L
Mistroke
.64551 .27893 L
.64425 .27897 L
.64312 .27898 L
.64203 .27897 L
.64086 .27893 L
.63976 .27887 L
.63874 .2788 L
.63628 .27855 L
.63397 .27824 L
.62863 .27727 L
.61741 .27462 L
.60575 .27186 L
.59983 .2707 L
.59428 .26983 L
.58922 .26924 L
.58667 .26902 L
.58387 .26884 L
.58241 .26877 L
.58108 .26872 L
.57963 .26869 L
.5789 .26867 L
.57811 .26867 L
.57677 .26867 L
.57534 .26869 L
.57401 .26872 L
.57281 .26876 L
.57147 .26882 L
.57023 .26888 L
.56744 .26908 L
.56477 .26932 L
.56195 .26963 L
.55263 .27104 L
.53335 .27512 L
.51486 .27937 L
.50434 .28177 L
.49276 .28426 L
.48783 .2852 L
.4826 .28605 L
.47966 .28644 L
.47702 .28671 L
.47573 .28682 L
.47454 .2869 L
.47331 .28697 L
.47267 .28699 L
.47199 .28701 L
.47079 .28703 L
.46971 .28702 L
.46852 .28699 L
.46793 .28697 L
.46729 .28693 L
Mistroke
.46622 .28686 L
.46508 .28675 L
.46307 .28649 L
.46103 .28613 L
.45913 .2857 L
.45567 .28467 L
.45178 .28316 L
.44831 .28154 L
.44036 .27741 L
.43277 .27386 L
.4286 .27233 L
.4237 .27098 L
.42109 .27045 L
.41962 .27021 L
.4182 .27001 L
.41686 .26986 L
.41539 .26973 L
.41403 .26964 L
.41275 .26958 L
.41148 .26955 L
.41027 .26954 L
.40894 .26956 L
.40748 .26961 L
.40603 .26968 L
.40448 .26979 L
.4017 .27006 L
.39861 .27045 L
.39528 .27098 L
.38929 .27216 L
.37608 .27538 L
.37041 .27678 L
.36477 .27796 L
.36199 .27841 L
.36046 .2786 L
.35974 .27867 L
.35907 .27873 L
.35847 .27877 L
.35782 .27881 L
.3567 .27885 L
.35608 .27886 L
.35542 .27885 L
.35481 .27884 L
.35427 .27881 L
.35318 .27873 L
.35265 .27868 L
.35207 .2786 L
.35012 .27826 L
.34906 .27801 L
.34794 .27768 L
.34597 .27697 L
Mistroke
.34225 .27509 L
.33897 .27288 L
.30522 .2481 L
.28671 .23973 L
.2822 .23744 L
.27791 .23472 L
.27609 .23326 L
.27453 .23177 L
.27208 .22864 L
.27109 .22687 L
.27024 .22484 L
.2699 .2238 L
.26963 .22278 L
.26943 .22186 L
.26924 .22081 L
.26909 .21962 L
.26903 .21895 L
.26899 .21832 L
.26894 .21716 L
.26895 .2159 L
.26897 .21517 L
.26901 .21437 L
.26907 .2136 L
.26914 .21289 L
.2693 .21151 L
.26952 .21002 L
.27009 .20714 L
.2709 .20394 L
.27284 .19799 L
.2791 .18431 L
.28576 .17351 L
.29322 .16412 L
.29997 .15769 L
.30433 .1546 L
.30556 .15384 L
.30593 .15359 L
.30604 .15348 L
.30608 .15343 L
.30611 .15339 L
.30611 .15331 L
.30606 .15325 L
.30602 .15322 L
.30596 .15319 L
.3058 .15313 L
.30561 .15308 L
.30537 .15302 L
.30479 .15291 L
.29415 .15049 L
.27298 .14456 L
.26617 .1429 L
Mistroke
.25885 .14144 L
.25583 .14096 L
.25262 .14054 L
.2508 .14035 L
.24914 .14022 L
.24835 .14016 L
.2475 .14011 L
.24595 .14004 L
.24452 .14 L
.24299 .13999 L
.24156 .14001 L
.24027 .14005 L
.23885 .14012 L
.23805 .14018 L
.23732 .14024 L
.23432 .14058 L
.23282 .14081 L
.23143 .14105 L
.2289 .14159 L
.22355 .14313 L
.21925 .14476 L
.21556 .1464 L
.20789 .15018 L
.20447 .15175 L
.2008 .15318 L
.19899 .15375 L
.19725 .15421 L
.19568 .15454 L
.1939 .15484 L
.19288 .15497 L
.1919 .15506 L
.19086 .15514 L
.19027 .15517 L
.18972 .1552 L
.18878 .15522 L
.18775 .15522 L
.18665 .1552 L
.18561 .15517 L
.18501 .15514 L
.18436 .1551 L
.18315 .15501 L
.18085 .15479 L
.17547 .15409 L
.16503 .15248 L
.16237 .15211 L
.15985 .15179 L
.1574 .15154 L
.15516 .15137 L
.15404 .1513 L
.15286 .15126 L
Mistroke
.15156 .15123 L
.1504 .15122 L
.14917 .15125 L
.14849 .15128 L
.14786 .15131 L
.14673 .15139 L
.14556 .15152 L
.14439 .15167 L
.14317 .15188 L
.14106 .15237 L
.13869 .15313 L
.13668 .15402 L
.13513 .15489 L
.13355 .156 L
.13213 .15723 L
.13075 .15871 L
.12959 .16024 L
.12864 .16176 L
.12709 .16508 L
.12638 .16714 L
.12583 .16916 L
.12533 .17149 L
.12491 .17411 L
.12474 .17547 L
.12459 .17699 L
.12448 .17845 L
.12439 .17982 L
.12434 .18108 L
.1243 .18249 L
.12428 .1832 L
.12427 .18397 L
.12427 .18539 L
.12427 .18616 L
.12428 .1869 L
.12431 .18829 L
.12435 .18981 L
.12442 .19146 L
.12458 .19449 L
.12478 .19731 L
.12539 .20395 L
.12627 .21109 L
.12833 .22332 L
.12988 .23023 L
.13162 .23659 L
.13366 .24278 L
.13626 .24931 L
.14209 .26051 L
.14907 .27037 L
.15645 .2784 L
.17345 .29165 L
Mistroke
.18318 .29713 L
.19207 .30107 L
.19642 .30262 L
.20118 .30405 L
.20542 .30506 L
.20778 .30553 L
.21004 .30591 L
.21228 .30623 L
.21476 .30652 L
.21705 .30673 L
.2192 .30689 L
.22126 .30699 L
.22245 .30704 L
.22357 .30707 L
.22477 .3071 L
.22587 .30711 L
.22716 .30712 L
.22839 .30712 L
.22961 .30711 L
.23095 .30709 L
.23222 .30707 L
.23339 .30703 L
.23616 .30694 L
.23913 .30682 L
.25129 .30618 L
.25646 .30593 L
.25931 .30582 L
.262 .30574 L
.26353 .30571 L
.26439 .3057 L
.26518 .30569 L
.26658 .30568 L
.26806 .30568 L
.26943 .3057 L
.27071 .30573 L
.27211 .30577 L
.27361 .30583 L
.27519 .30591 L
.27661 .30601 L
.27975 .30629 L
.28261 .30663 L
.28526 .30704 L
.29102 .30826 L
.29415 .30914 L
.29696 .31007 L
.30643 .31431 L
.31573 .32011 L
.33289 .33285 L
.3405 .33803 L
.34897 .34285 L
Mistroke
.35815 .34684 L
.36691 .34959 L
.37177 .35075 L
.37693 .35176 L
.38626 .35319 L
.39518 .35423 L
.40002 .35471 L
.40527 .35514 L
.41004 .35546 L
.41262 .3556 L
.41543 .35572 L
.41678 .35576 L
.41807 .3558 L
.41923 .35583 L
.4205 .35585 L
.42171 .35586 L
.42301 .35587 L
.42412 .35587 L
.42535 .35586 L
.42686 .35584 L
.42828 .35581 L
.42984 .35576 L
.43152 .3557 L
.43465 .35554 L
.4375 .35535 L
.44337 .35483 L
.44892 .35417 L
.46117 .35216 L
.47381 .34945 L
.48406 .34703 L
.49417 .34486 L
.49702 .34436 L
.49967 .34396 L
.50221 .34365 L
.50332 .34354 L
.50454 .34344 L
.50561 .34336 L
.50675 .34329 L
.50795 .34324 L
.50908 .34321 L
.50971 .3432 L
.5104 .34319 L
.51105 .34319 L
.51165 .3432 L
.5128 .34322 L
.51344 .34323 L
.51403 .34325 L
.51512 .3433 L
.51633 .34337 L
.5188 .34356 L
Mistroke
.52331 .34404 L
.53407 .34561 L
.54485 .34732 L
.55037 .34809 L
.55556 .34869 L
.55825 .34895 L
.56123 .34919 L
.56422 .34937 L
.56585 .34944 L
.56658 .34947 L
.56738 .34949 L
.56868 .34951 L
.57004 .34952 L
.5713 .34952 L
.57246 .3495 L
.57383 .34947 L
.57507 .34942 L
.57648 .34935 L
.57781 .34926 L
.58054 .34901 L
.58337 .34867 L
.58572 .34829 L
.58824 .34781 L
.59361 .34644 L
.59829 .3448 L
.60668 .34072 L
.61384 .33607 L
.62676 .32678 L
.634 .32246 L
.63828 .32039 L
.64252 .31868 L
.64669 .3173 L
.65138 .31603 L
.65997 .3143 L
.68016 .31142 L
.68567 .31046 L
.69146 .30919 L
.70157 .30604 L
.70637 .30405 L
.71147 .3016 L
.72052 .29658 L
.72922 .29159 L
.73762 .28745 L
.74195 .28577 L
.74678 .28434 L
.74947 .28374 L
.75236 .28325 L
.75488 .28295 L
.75633 .28283 L
.75769 .28275 L
Mistroke
.75913 .28269 L
.75996 .28267 L
.76072 .28266 L
.76213 .28267 L
.76284 .28268 L
.76363 .2827 L
.76501 .28276 L
.7665 .28285 L
.76922 .28306 L
.77266 .28343 L
.77597 .28385 L
.78226 .28479 L
.79403 .28656 L
.79687 .28689 L
.79989 .28718 L
.8016 .28731 L
.80234 .28735 L
.80315 .28739 L
.80462 .28744 L
.80542 .28746 L
.80616 .28746 L
.80746 .28745 L
.80886 .28741 L
.81016 .28734 L
.81135 .28724 L
.8124 .28713 L
.81355 .28699 L
.81582 .2866 L
.8182 .28605 L
.82065 .28528 L
.8248 .28346 L
.82692 .28221 L
.82904 .28069 L
.83251 .27751 L
.83606 .27306 L
.8388 .26852 L
.84316 .25874 L
.8466 .24936 L
.85018 .24056 L
.8548 .23234 L
.85971 .22615 L
.86608 .21986 L
.87388 .21271 L
.88094 .20517 L
.88362 .20154 L
.88608 .19745 L
.88717 .19521 L
.88821 .19267 L
.88894 .19043 L
.88956 .18802 L
Mistroke
.88997 .18586 L
.89014 .18462 L
.89026 .18346 L
.89033 .18238 L
.89038 .1814 L
.89039 .18026 L
.89037 .17918 L
.89031 .178 L
.89021 .17675 L
.89009 .17568 L
.88993 .17451 L
.88943 .17189 L
.8888 .16944 L
.88728 .16506 L
.88551 .16114 L
.88033 .15277 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl603Wo001lo`040?oo00So1@04o`80C?l300Co1@0eo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0<o`80<ol007co00@0ool02Ol00`3o0005o`030?oo04Oo0P0@o`03
0?oo033o001mo`802ol200Go00<0ool0A_l00`3oo`0Ao`80<?l0097o00<0ool0A_l00`3oo`0Co`80
;_l0097o00<0ool0AOl00`3oo`0Fo`030?oo02_o0018o`X0?ol00`3oo`14o`030?oo01So00<0ool0
:_l004Go0`0:o`@0>ol00`3oo`13o`030?oo01[o00<0ool0:Ol004?o0P0Ao`<0>?l00`3oo`13o`03
0?oo01_o00<0ool0:?l0033o3004o`<05_l503?o00<0ool0@ol00`3oo`0Lo`030?oo02Oo000_o`03
0?oo00[o1@0Jo`80=?l00`3oo`12o`030?oo01go00<0ool09ol002ko00<0ool0:Ol00`3oo`0do`03
0?oo04;o00<0ool07_l00`3oo`0Vo`00;Ol202[o00<0ool0=Ol00`3oo`11o`030?oo023o00<0ool0
9Ol002go00<0ool0:Ol00`3oo`0eo`030?oo047o00<0ool08?l00`3oo`0Uo`00;Ol00`3oo`0Xo`03
0?oo03Ko00<0ool0@?l00`3oo`0Qo`030?oo02Go000]o`030?oo02So00<0ool08_l200Co0P03o`@0
1Ol00`3oo`10o`030?oo027o00<0ool09Ol002go00<0ool09ol00`3oo`0Ro`040?oo00So00<0ool0
1_l00`3oo`0oo`030?oo02;o00<0ool09Ol002go00<0ool09ol00`3oo`0Ro`040?oo00Wo00<0ool0
1Ol2043o00<0ool08_l00`3oo`0Uo`00;Ol00`3oo`0Vo`030?oo01go1002o`040?oo00[o00<0ool0
1?l00`3oo`0no`030?oo02?o00<0ool09Ol002go00<0ool09Ol00`3oo`0To`040?oo00So00@0ool0
1Ol00`3oo`0no`030?oo02?o00<0ool09Ol002go00<0ool09Ol00`3oo`0Uo`802_l200Ko00<0ool0
?Ol00`3oo`0So`809ol002go00<0ool09?l00`3oo`0jo`030?oo03go00<0ool08ol00`3oo`0Vo`00
;Ol00`3oo`0To`030?oo03[o00<0ool0??l00`3oo`0To`030?oo02Ko000]o`030?oo02Co00<0ool0
>_l00`3oo`0ko`030?oo02Co00<0ool09ol000Oo1@02o`801?l201Oo00<0ool02Ol501Go00<0ool0
2_l200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P0:o`030?oo01Ko1@09o`030?oo01Ko1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool05_l00`3oo`0;o`030?oo01Go00<0ool02Ol0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`08o`040?oo00Wo00<0ool06?l00`3oo`08o`030?oo01Wo00<0ool0
1ol0103oo`02o`002Ol00`3oo`0:o`030?oo01Co00<0ool02ol00`3oo`0Eo`030?oo00Wo00@0ool0
2ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`06o`030?oo01Wo00<0ool01ol00`3oo`0Jo`03
0?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05Ol00`3o
o`03o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P08o`030?oo01[o00<0ool01_l00`3o
o`0Ko`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo00<0ool02_l00`3oo`0Eo`030?oo00Wo
00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`07o`030?oo01[o00<0ool01Ol00`3o
o`0Lo`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806?l00`3oo`09o`802_l302;o
00<0ool08_l200[o0`06o`030?oo01[o0P06o`030?oo01co0P0:o`<00_l002ko00<0ool09?l00`3o
o`0io`030?oo03Oo00<0ool08Ol00`3oo`0^o`00;ol00`3oo`0So`80>_l00`3oo`0fo`030?oo02;o
00<0ool0;_l002oo00<0ool09Ol303Oo00<0ool0=Ol00`3oo`0Ro`030?oo02oo000_o`030?oo02So
0P0eo`030?oo03Co00<0ool08ol00`3oo`0_o`00<?l00`3oo`0Yo`80<ol00`3oo`0co`030?oo02?o
00<0ool0<?l000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko0P07o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00@0oooo0P03o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01Ol200So00<0ool01Ol00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool08_l00`3oo`04o`030?oo02Ko00@0ool0
;?l00`3oo`0Yo`030?oo00Co00<0ool08_l0103oo`0Yo`030?oo00Oo000ao`030?oo02ko00<0ool0
;?l00`3oo`0^o`80:?l00`3oo`0ao`00<_l00`3oo`0^o`80;?l00`3oo`0/o`80:Ol00`3oo`0bo`00
<ol00`3oo`0_o`030?oo00ko1`0Do`030?oo00go2P0Do`030?oo02Wo00<0ool0<_l003Co00<0ool0
;ol00`3oo`09o`@01_l5013o00<0ool02Ol400[o100>o`80:ol00`3oo`0co`00=?l00`3oo`0`o`80
1?l500oo0P0>o`030?oo00?o1P0Bo`D01Ol402co00<0ool0=?l003Go00<0ool0<Ol401Ko00<0ool0
2ol601go1P0^o`030?oo03Go000fo`80C?l300Ko1@10o``02?l203So000ho`80C?l700Co00<0ool0
>ol300co200jo`00>_l00`3oo`1Do`030?oo03So0`1Ao`00>ol00`3oo`1Co`030?oo03Oo00<0ool0
D_l003co0P1Co`030?oo03Ko00<0ool0Dol003ko101?o`030?oo03?o0`1Fo`00@_lH03Oo00<0ool0
<Ol205Wo001Jo`808Ol200Co0P03o`@01Ol00`3oo`0Zo`L0Fol005co0P0No`040?oo00So00<0ool0
1_l00`3oo`0Vo`@0H_l005ko00<0ool06ol0103oo`09o`030?oo00Go0P0So`@0I_l005oo00<0ool0
6_l0103oo`0:o`030?oo00Co00<0ool07ol306[o001Po`806_l0103oo`08o`040?oo00Go00<0ool0
7_l00`3oo`1[o`00H_l201Wo0P0:o`801_l00`3oo`0Mo`030?oo06co001To`030?oo02[o00<0ool0
7?l00`3oo`1]o`00IOl202[o00<0ool06_l2073o001Wo`030?oo02Oo00<0ool06?l207;o001Xo`80
9?l=00oo0P1do`00J_l401[o1P03o`030?oo00Oo3`1fo`00K_l6017o0`09o`030?oo08co001doa40
3?l00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool0
2?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`03
0?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`80
0ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool0
1Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00
OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88274 .15307 m
.87533 .14549 L
.86637 .13832 L
.84779 .12696 L
.83806 .12229 L
.8327 .12015 L
.82761 .11845 L
.82282 .1172 L
.82011 .11666 L
.8176 .11627 L
.81548 .11604 L
.81427 .11594 L
.81315 .11588 L
.81196 .11584 L
.81088 .11583 L
.81028 .11583 L
.80963 .11585 L
.80844 .11589 L
.80784 .11593 L
.80719 .11598 L
.80605 .11608 L
.80483 .11623 L
.80349 .11643 L
.80111 .1169 L
.79892 .11745 L
.79494 .11879 L
.79084 .12066 L
.78635 .12335 L
.7824 .12637 L
.77581 .133 L
.77032 .14024 L
.76503 .14844 L
.75543 .16381 L
.7332 .19753 L
.72397 .21566 L
.72097 .22277 L
.71782 .22997 L
.71449 .23616 L
.70987 .24275 L
.7034 .24997 L
.69588 .25703 L
.68655 .2647 L
.6756 .27244 L
.67049 .27551 L
.66511 .27828 L
.66056 .28016 L
.65798 .28101 L
.65562 .28165 L
.6532 .28215 L
.65193 .28235 L
Mistroke
.65058 .28252 L
.64997 .28258 L
.6493 .28263 L
.64809 .28269 L
.64691 .28272 L
.64583 .28271 L
.64476 .28266 L
.64376 .2826 L
.64266 .2825 L
.64148 .28236 L
.63906 .28196 L
.63684 .28149 L
.63171 .28008 L
.62595 .27811 L
.61486 .27392 L
.60289 .26984 L
.59681 .26818 L
.59118 .26695 L
.58528 .266 L
.5822 .26564 L
.58052 .26549 L
.579 .26537 L
.57742 .26528 L
.57661 .26525 L
.57575 .26521 L
.57421 .26518 L
.57277 .26516 L
.57197 .26517 L
.57109 .26518 L
.57018 .2652 L
.56933 .26522 L
.56783 .26529 L
.56624 .26538 L
.56344 .26561 L
.56055 .26593 L
.55555 .26667 L
.55045 .26766 L
.54511 .26891 L
.52768 .27381 L
.50796 .27963 L
.49844 .28239 L
.48787 .28524 L
.48303 .28635 L
.48043 .28687 L
.47803 .28728 L
.47585 .28759 L
.47472 .28772 L
.47355 .28783 L
.47257 .28791 L
.47151 .28797 L
Mistroke
.47057 .288 L
.46969 .28801 L
.46864 .28799 L
.46805 .28797 L
.46751 .28795 L
.46654 .28787 L
.46552 .28777 L
.46453 .28763 L
.46362 .28748 L
.46195 .28711 L
.46004 .28655 L
.45833 .28592 L
.45462 .28412 L
.44791 .27969 L
.44013 .2743 L
.43585 .27189 L
.43138 .26995 L
.4293 .26924 L
.427 .2686 L
.42437 .26803 L
.42184 .26764 L
.42053 .2675 L
.41914 .26738 L
.41792 .26732 L
.41658 .26728 L
.41511 .26728 L
.41351 .26733 L
.41198 .26741 L
.41058 .26752 L
.4078 .26784 L
.40475 .26832 L
.3991 .26952 L
.39234 .27143 L
.38527 .27384 L
.37329 .27845 L
.36776 .28054 L
.36291 .28214 L
.36042 .28282 L
.35916 .28311 L
.35785 .28336 L
.35668 .28354 L
.35565 .28366 L
.35463 .28373 L
.3537 .28376 L
.35285 .28375 L
.35193 .28369 L
.35099 .28359 L
.35014 .28344 L
.34936 .28327 L
.34866 .28307 L
Mistroke
.34713 .28253 L
.34554 .28178 L
.34386 .28076 L
.34088 .27845 L
.32732 .26456 L
.32281 .26049 L
.31748 .2564 L
.30678 .25018 L
.2867 .24219 L
.28191 .24009 L
.27953 .23881 L
.27749 .2375 L
.27576 .23614 L
.2741 .23452 L
.2728 .23289 L
.27179 .23125 L
.27131 .2303 L
.27086 .22923 L
.27019 .22718 L
.26988 .22584 L
.26976 .22516 L
.26966 .22451 L
.26951 .22327 L
.26945 .2226 L
.26941 .22188 L
.26937 .22049 L
.26938 .21969 L
.2694 .21894 L
.26947 .21753 L
.2696 .216 L
.26981 .21427 L
.27005 .21262 L
.27081 .20874 L
.27266 .20174 L
.2749 .19514 L
.28127 .18047 L
.28826 .16811 L
.29457 .15952 L
.2978 .15609 L
.30048 .15383 L
.30176 .15297 L
.30281 .15239 L
.30384 .15196 L
.30408 .15188 L
.30433 .15181 L
.30454 .15176 L
.30475 .15172 L
.30492 .1517 L
.3051 .15168 L
.30538 .15168 L
Mistroke
.30552 .15169 L
.30566 .15171 L
.30578 .15174 L
.30587 .15177 L
.30603 .15185 L
.3061 .1519 L
.30615 .15196 L
.30621 .15207 L
.30624 .1522 L
.30622 .15232 L
.30616 .15246 L
.30607 .1526 L
.30595 .15272 L
.30578 .15286 L
.30555 .153 L
.30526 .15315 L
.30494 .15328 L
.30454 .1534 L
.30405 .15351 L
.3038 .15356 L
.30352 .1536 L
.30299 .15367 L
.30243 .15371 L
.30187 .15373 L
.30121 .15373 L
.30044 .1537 L
.3 .15367 L
.29957 .15364 L
.29856 .15353 L
.29653 .15324 L
.2945 .15286 L
.29214 .15233 L
.28216 .14956 L
.26972 .14583 L
.25832 .14293 L
.25517 .14229 L
.25183 .14172 L
.24871 .14129 L
.24591 .14101 L
.24438 .1409 L
.24277 .14081 L
.24142 .14077 L
.23995 .14076 L
.23839 .14079 L
.23753 .14082 L
.23675 .14086 L
.23525 .14097 L
.23385 .14112 L
.23125 .14149 L
.22893 .14194 L
Mistroke
.22648 .14256 L
.22399 .14334 L
.21887 .14548 L
.21457 .14785 L
.20755 .15253 L
.20428 .1547 L
.20127 .15641 L
.19961 .15717 L
.19807 .15776 L
.1963 .15826 L
.19531 .15848 L
.19434 .15864 L
.19358 .15873 L
.19272 .1588 L
.1918 .15885 L
.19092 .15885 L
.19009 .15884 L
.18931 .1588 L
.18845 .15873 L
.18751 .15863 L
.18549 .15833 L
.18354 .15796 L
.17884 .15679 L
.1592 .15071 L
.1537 .14929 L
.15078 .14868 L
.14812 .14825 L
.14698 .1481 L
.14592 .14799 L
.14473 .1479 L
.14365 .14784 L
.1425 .14782 L
.14145 .14784 L
.14033 .1479 L
.13972 .14795 L
.13918 .14801 L
.13809 .14818 L
.13698 .1484 L
.13606 .14864 L
.1351 .14894 L
.13411 .14933 L
.1331 .14981 L
.13142 .15085 L
.1299 .15212 L
.12864 .15353 L
.12764 .15496 L
.1267 .15669 L
.12594 .15847 L
.12524 .16061 L
.12466 .16303 L
Mistroke
.12424 .16545 L
.12407 .16667 L
.12394 .16785 L
.12374 .17014 L
.12366 .17146 L
.1236 .17272 L
.12355 .17412 L
.12353 .17491 L
.12352 .17564 L
.1235 .17702 L
.1235 .17832 L
.12352 .17978 L
.12355 .18136 L
.12356 .18216 L
.12359 .18304 L
.12364 .18466 L
.12377 .18774 L
.12413 .19393 L
.12456 .19984 L
.12594 .21368 L
.1269 .22077 L
.12825 .22859 L
.12973 .23534 L
.13165 .24232 L
.13615 .25431 L
.13929 .26062 L
.14261 .26619 L
.1514 .27761 L
.16101 .28682 L
.17044 .29386 L
.17872 .29884 L
.18744 .30303 L
.19151 .30459 L
.19582 .30597 L
.1982 .3066 L
.20039 .3071 L
.20263 .30753 L
.20473 .30785 L
.20669 .30809 L
.20881 .30829 L
.21004 .30837 L
.21121 .30843 L
.21228 .30847 L
.21344 .3085 L
.2145 .30851 L
.21566 .3085 L
.2169 .30849 L
.21807 .30845 L
.21918 .30841 L
.22021 .30836 L
Mistroke
.22256 .30821 L
.22503 .30801 L
.22785 .30774 L
.23377 .30704 L
.24513 .30544 L
.25196 .30445 L
.25926 .30349 L
.2655 .30283 L
.269 .30255 L
.27069 .30245 L
.27223 .30237 L
.27378 .30231 L
.27545 .30227 L
.27691 .30226 L
.27844 .30227 L
.27979 .3023 L
.28105 .30235 L
.28242 .30242 L
.28388 .30253 L
.28541 .30267 L
.28705 .30286 L
.28998 .3033 L
.29276 .30387 L
.29526 .3045 L
.30079 .3064 L
.30551 .30862 L
.31525 .31511 L
.33269 .33095 L
.34042 .33762 L
.34919 .34382 L
.35338 .34622 L
.3579 .34841 L
.36609 .35144 L
.3709 .35275 L
.37544 .35374 L
.38394 .35513 L
.39286 .35621 L
.39793 .35671 L
.40286 .35712 L
.40759 .35745 L
.40972 .35757 L
.41209 .35768 L
.41345 .35773 L
.41473 .35778 L
.41605 .35781 L
.4173 .35784 L
.41876 .35785 L
.4201 .35786 L
.42156 .35786 L
.42238 .35785 L
Mistroke
.42315 .35785 L
.42466 .35782 L
.42611 .35778 L
.42741 .35774 L
.42885 .35767 L
.4321 .35749 L
.43515 .35726 L
.44078 .3567 L
.44684 .35587 L
.45951 .35343 L
.47287 .34991 L
.48435 .34638 L
.49512 .34318 L
.49922 .34218 L
.50142 .34174 L
.50346 .34139 L
.50461 .34123 L
.50585 .34108 L
.50688 .34098 L
.508 .34089 L
.50906 .34083 L
.51019 .34079 L
.51123 .34077 L
.51221 .34078 L
.51329 .3408 L
.5143 .34085 L
.51542 .34092 L
.51661 .34102 L
.51885 .34127 L
.52091 .34156 L
.5295 .34325 L
.5406 .34592 L
.55171 .34845 L
.55754 .34956 L
.56302 .35041 L
.566 .35079 L
.56871 .35106 L
.57168 .35129 L
.5733 .35138 L
.57482 .35144 L
.57621 .35148 L
.57773 .35149 L
.57848 .35148 L
.57929 .35147 L
.58074 .35143 L
.58209 .35136 L
.58349 .35127 L
.58467 .35117 L
.58594 .35104 L
.58876 .35065 L
Mistroke
.59165 .35011 L
.59426 .34947 L
.59654 .34879 L
.60071 .3472 L
.60475 .34516 L
.61118 .34074 L
.62396 .32912 L
.63019 .32417 L
.63397 .32178 L
.63782 .31979 L
.64164 .3182 L
.64608 .31677 L
.65033 .31573 L
.65444 .31496 L
.66439 .31369 L
.66977 .31318 L
.67546 .31261 L
.68044 .31199 L
.6858 .31109 L
.69057 .31 L
.6949 .30871 L
.70392 .3049 L
.71168 .30031 L
.72041 .29408 L
.72848 .28829 L
.73687 .28342 L
.74136 .28155 L
.744 .28072 L
.74654 .28009 L
.74787 .27983 L
.74935 .27959 L
.75076 .27941 L
.75209 .27929 L
.75342 .2792 L
.75465 .27916 L
.75608 .27915 L
.75744 .27917 L
.75895 .27924 L
.75972 .27929 L
.76057 .27935 L
.76352 .27967 L
.76704 .28019 L
.7704 .28082 L
.78203 .2836 L
.79462 .28676 L
.80106 .28806 L
.80279 .28834 L
.80463 .28859 L
.80621 .28877 L
.80703 .28885 L
Mistroke
.8079 .28893 L
.80943 .28902 L
.81084 .28907 L
.81165 .28908 L
.81242 .28907 L
.81382 .28903 L
.81512 .28895 L
.81629 .28883 L
.81755 .28866 L
.81885 .28844 L
.82139 .28782 L
.82275 .28738 L
.82396 .28692 L
.82621 .28584 L
.82816 .28465 L
.83026 .28305 L
.83234 .28103 L
.8341 .27891 L
.83558 .27673 L
.83804 .27202 L
.83982 .26737 L
.84137 .2621 L
.84275 .2563 L
.8439 .25099 L
.84515 .24554 L
.84649 .24081 L
.84801 .23669 L
.85012 .23248 L
.85248 .22903 L
.85559 .22557 L
.85955 .22213 L
.87582 .21067 L
.88031 .20703 L
.88474 .20268 L
.88831 .19815 L
.89092 .19365 L
.89193 .19134 L
.8928 .1888 L
.89314 .18755 L
.89341 .18636 L
.89361 .18529 L
.89379 .1841 L
.89393 .18278 L
.89399 .18205 L
.89402 .18137 L
.89404 .18014 L
.89402 .17883 L
.89398 .17809 L
.89391 .17728 L
.89384 .17652 L
Mistroke
.89375 .17581 L
.89354 .17447 L
.89326 .17305 L
.89265 .17068 L
.89181 .16813 L
.88983 .16361 L
.88678 .15842 L
.88344 .15391 L
.88274 .15307 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl503[o001lo`040?oo00So1@04o`80C?l300Co100fo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0;o`80=?l007co00@0ool02Ol00`3o0005o`030?oo04So00<0ool0
3Ol203;o001mo`802ol200Go00<0ool0Aol00`3oo`0@o`80<?l0097o00<0ool0A_l00`3oo`0Co`80
;_l0097o00<0ool0AOl00`3oo`0Fo`030?oo02_o0018o`P0@Ol00`3oo`14o`030?oo01So00<0ool0
:_l004Ko0P08o`H0>ol00`3oo`13o`030?oo01[o00<0ool0:Ol002oo1`0?o`030?oo00ko0`0ho`03
0?oo04?o00<0ool06ol00`3oo`0Xo`00;_l00`3oo`05o`<02_l201Co1@0co`030?oo04;o00<0ool0
7Ol00`3oo`0Wo`00;Ol00`3oo`09o`<01?l301So00<0ool0<ol00`3oo`11o`030?oo01ko00<0ool0
9ol002go00<0ool03?l501Wo00<0ool0=?l00`3oo`11o`030?oo01oo00<0ool09_l002go00<0ool0
:Ol00`3oo`0eo`030?oo043o00<0ool08Ol00`3oo`0Uo`00;?l202[o00<0ool0=_l00`3oo`0oo`03
0?oo02?o00<0ool09?l002co00<0ool0:Ol00`3oo`0fo`030?oo03oo00<0ool08ol00`3oo`0To`00
;?l00`3oo`0Xo`030?oo02?o0P04o`800ol400Go00<0ool0?_l00`3oo`0To`030?oo02Co000/o`03
0?oo02So00<0ool08_l0103oo`08o`030?oo00Ko00<0ool0?_l00`3oo`0To`030?oo02Co000/o`03
0?oo02Oo00<0ool08ol0103oo`09o`030?oo00Go0P0no`030?oo02Go00<0ool09?l002co00<0ool0
9ol00`3oo`0Mo`@00_l0103oo`0:o`030?oo00Co00<0ool0?Ol00`3oo`0Uo`030?oo02Co000]o`03
0?oo02Go00<0ool09?l0103oo`08o`040?oo00Go00<0ool0??l00`3oo`0Vo`030?oo02Co000]o`03
0?oo02Go00<0ool09Ol200[o0P06o`030?oo03co00<0ool09Ol00`3oo`0Uo`00;Ol00`3oo`0Uo`03
0?oo03Wo00<0ool0>ol00`3oo`0Vo`030?oo02Go000]o`030?oo02Co00<0ool0>_l00`3oo`0jo`03
0?oo02Ko00<0ool09_l002go00<0ool09?l00`3oo`0jo`030?oo03[o00<0ool09_l00`3oo`0Vo`00
1ol500;o0P04o`805ol00`3oo`09o`D05_l00`3oo`09o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200Wo00<0ool05ol500Wo0P0Go`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Ko00<0ool0
2ol00`3oo`0Eo`030?oo00Wo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`08o`03
0?oo01Wo00<0ool02?l00`3oo`0Io`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Do`03
0?oo00_o00<0ool05Ol00`3oo`09o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
1Ol00`3oo`0Jo`030?oo00Ko0P0Lo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Oo
00<0ool00ol400Co00<0ool05Ol00`3oo`03o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo
0P07o`030?oo01_o00<0ool01Ol00`3oo`0Lo`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Oo
00<0ool02_l00`3oo`0Eo`030?oo00Wo00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3o
o`06o`030?oo01_o00<0ool00ol201oo00<0ool02?l00`3oo`02o`002?l200[o0`0Go`030?oo00Wo
0P0Go`802ol200[o0`0Ro`030?oo02;o0P0:o`<01Ol00`3oo`0Ko`801?l00`3oo`0No`802_l300;o
000^o`030?oo02?o00<0ool0>_l00`3oo`0fo`030?oo027o00<0ool0;ol002ko00<0ool08ol203_o
00<0ool0=_l00`3oo`0Po`030?oo033o000_o`030?oo02?o0`0io`030?oo03Go00<0ool08Ol00`3o
o`0`o`00;ol00`3oo`0Vo`80=ol00`3oo`0do`030?oo02;o00<0ool0<?l002oo00<0ool0:?l203Go
00<0ool0<ol00`3oo`0Ro`030?oo037o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`801ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00801?l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l0103o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool08_l00`3o
o`04o`030?oo02Go00D0oooo000/o`030?oo02Wo00<0ool01Ol00`3oo`0Qo`040?oo02Wo00<0ool0
1ol0037o00<0ool0;Ol202ko00<0ool0;ol202Ko00<0ool0<_l0037o00<0ool0;ol00`3oo`0[o`03
0?oo00_o2P0Io`030?oo02Ko00<0ool0<_l003;o00<0ool0;ol00`3oo`0?o`T04_l00`3oo`07o`@0
2_l501;o0P0Yo`030?oo03;o000co`030?oo02ko00<0ool02ol400Wo0`0?o`030?oo00?o100Co`<0
3_l00`3oo`0Xo`030?oo03?o000do`030?oo02ko00<0ool02?l2013o00<0ool03?l0103oool201[o
0`0:o`030?oo02Wo00<0ool0<ol003Go00<0ool0;_l200Go0`0Co`030?oo00_o100Oo`801?l401?o
2@0?o`030?oo03Co000eo`030?oo033o1@0Go`030?oo00Ko1@0To`@05Ol200Wo100:o`030?oo03Go
000fo`030?oo04co2004o`030?oo03Wo0P0?o`X0>?l003Oo0P1Ho`030?oo03So00<0ool0DOl003Wo
00<0ool0EOl00`3oo`0fo`80E?l003[o0P1Eo`030?oo03Go00<0ool0E?l003co0P0@o`d0=_l00`3o
o`0do`030?oo05Go000no`D01?l700go0P0do`030?oo03;o0P1Ho`00@ol401Ko00<0ool07Ol200Co
0P03o`@01Ol00`3oo`0/o`H0F_l005ko0P0Lo`040?oo00So00<0ool01_l00`3oo`0To`P0H?l0063o
00<0ool06Ol0103oo`09o`030?oo00Go0P0So`80J?l0067o00<0ool06?l0103oo`0:o`030?oo00Co
00<0ool08?l206[o001Ro`030?oo01Oo00@0ool02?l0103oo`05o`030?oo01ko0P1/o`00Hol00`3o
o`0Go`802_l200Ko00<0ool07Ol00`3oo`1/o`00I?l00`3oo`0Zo`030?oo01co00<0ool0KOl006Go
0P0Zo`030?oo01_o00<0ool0K_l006Oo00<0ool09ol601Ko0P1ao`00J?l00`3oo`0So`@01Ol301;o
00<0ool0LOl006Wo0`0Oo`<00ol00`3oo`06o`L02Ol207Co001/o`<06Ol300Ko00<0ool03Ol907Ko
001_o`T02?l800Wo00<0ool0S?l007So200Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3o
o`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3o
o`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88514 .15336 m
.86698 .13894 L
.84732 .12789 L
.83824 .12349 L
.82873 .11958 L
.82327 .11784 L
.82068 .11718 L
.81828 .11668 L
.81587 .1163 L
.8146 .11614 L
.81325 .11601 L
.81208 .11593 L
.81099 .11589 L
.80979 .11587 L
.80912 .11588 L
.8085 .11589 L
.80733 .11594 L
.80622 .11602 L
.80523 .11612 L
.80417 .11626 L
.80179 .11667 L
.79963 .11718 L
.79577 .11843 L
.79185 .12019 L
.7878 .12259 L
.78423 .12522 L
.77692 .13232 L
.75407 .16261 L
.74032 .17933 L
.73321 .18874 L
.72677 .19897 L
.72249 .20788 L
.71933 .21713 L
.71744 .22445 L
.71549 .23161 L
.71428 .23502 L
.71259 .23874 L
.70846 .24544 L
.70148 .2538 L
.69306 .2621 L
.68289 .27077 L
.67278 .27808 L
.66838 .28074 L
.66378 .28309 L
.66141 .28409 L
.65882 .28501 L
.6565 .28567 L
.65434 .28613 L
.65314 .28631 L
.65248 .2864 L
Mistroke
.65186 .28646 L
.65067 .28655 L
.64958 .28658 L
.64896 .28659 L
.64829 .28657 L
.64711 .28651 L
.64645 .28646 L
.64583 .2864 L
.64443 .28621 L
.6432 .286 L
.64204 .28575 L
.63985 .28518 L
.63485 .28343 L
.61486 .27372 L
.60308 .26847 L
.59635 .26605 L
.58996 .26422 L
.58455 .26303 L
.58153 .26252 L
.5788 .26215 L
.576 .26187 L
.57443 .26175 L
.57299 .26167 L
.57148 .26161 L
.57012 .26158 L
.56858 .26158 L
.56714 .26161 L
.56581 .26165 L
.5644 .26173 L
.5631 .26182 L
.56192 .26192 L
.55911 .26223 L
.55654 .2626 L
.55167 .26352 L
.54673 .26475 L
.53141 .26982 L
.51452 .27597 L
.49248 .2838 L
.48238 .28691 L
.47975 .28758 L
.47707 .28816 L
.47486 .28855 L
.47365 .28872 L
.47253 .28885 L
.47151 .28893 L
.47042 .28899 L
.46945 .28901 L
.46858 .289 L
.46765 .28895 L
.46666 .28887 L
Mistroke
.46617 .28881 L
.46565 .28873 L
.46473 .28856 L
.463 .28812 L
.46143 .28756 L
.45863 .28615 L
.45246 .28123 L
.44633 .27539 L
.44329 .27283 L
.43972 .27032 L
.43768 .26914 L
.43531 .26797 L
.43303 .26706 L
.43076 .26633 L
.42943 .26598 L
.42795 .26566 L
.42669 .26544 L
.42526 .26524 L
.42451 .26515 L
.42368 .26508 L
.4228 .26502 L
.42197 .26498 L
.42032 .26495 L
.41878 .26498 L
.4173 .26505 L
.41591 .26516 L
.41435 .26533 L
.41267 .26556 L
.40916 .2662 L
.40591 .26697 L
.39853 .26926 L
.39051 .27246 L
.37668 .2792 L
.36465 .2853 L
.36202 .28647 L
.35967 .28738 L
.35753 .28807 L
.35566 .28853 L
.35475 .28869 L
.35382 .2888 L
.3533 .28884 L
.35281 .28886 L
.35193 .28886 L
.35101 .28878 L
.35018 .28865 L
.3493 .28844 L
.3484 .28814 L
.34757 .28779 L
.3467 .28733 L
.34518 .28631 L
Mistroke
.34183 .28316 L
.32862 .26698 L
.32392 .26238 L
.31832 .25796 L
.30703 .25167 L
.29561 .24751 L
.28956 .24566 L
.28426 .24389 L
.27991 .24203 L
.27774 .24079 L
.27594 .23949 L
.27446 .23814 L
.2733 .23676 L
.27215 .235 L
.27128 .23314 L
.27063 .2312 L
.27035 .23002 L
.27014 .22886 L
.26997 .22759 L
.26986 .22638 L
.26979 .22492 L
.26979 .22349 L
.26982 .22221 L
.2699 .22082 L
.27001 .21959 L
.27015 .2182 L
.27062 .21499 L
.2712 .21183 L
.27302 .20433 L
.27521 .19709 L
.28144 .18069 L
.28822 .16668 L
.29438 .15683 L
.29759 .15292 L
.30019 .15053 L
.30147 .14966 L
.30209 .14932 L
.30272 .14906 L
.30322 .1489 L
.3035 .14884 L
.30374 .1488 L
.30415 .14877 L
.30455 .14879 L
.30475 .14882 L
.30493 .14887 L
.30524 .14898 L
.30553 .14915 L
.30568 .14927 L
.3058 .14938 L
.30598 .1496 L
Mistroke
.30614 .14986 L
.30626 .15017 L
.30633 .15047 L
.30636 .15065 L
.30637 .15085 L
.30637 .15104 L
.30635 .15122 L
.30629 .15157 L
.30617 .15195 L
.3061 .15214 L
.306 .15235 L
.30578 .15272 L
.30545 .15315 L
.30508 .15352 L
.30473 .15382 L
.30431 .15411 L
.30343 .15457 L
.3028 .15482 L
.30216 .15501 L
.30154 .15516 L
.30082 .15528 L
.30006 .15537 L
.2996 .1554 L
.29916 .15542 L
.29868 .15543 L
.29823 .15543 L
.2977 .15542 L
.29718 .15539 L
.29621 .15532 L
.29528 .15522 L
.29302 .15487 L
.29073 .1544 L
.28849 .15386 L
.28295 .15228 L
.27157 .14862 L
.26454 .14641 L
.25684 .14428 L
.2531 .14341 L
.24959 .14272 L
.24648 .14222 L
.24476 .142 L
.24317 .14183 L
.24172 .1417 L
.24018 .14161 L
.2385 .14155 L
.23699 .14154 L
.23544 .14157 L
.23458 .14161 L
.23379 .14166 L
.23236 .14178 L
Mistroke
.23087 .14195 L
.22952 .14216 L
.2281 .14242 L
.22561 .14302 L
.22331 .14373 L
.22129 .1445 L
.21699 .14663 L
.21287 .1494 L
.20601 .1554 L
.20331 .15786 L
.20192 .15902 L
.20045 .16009 L
.19914 .16089 L
.19766 .16161 L
.19699 .16187 L
.19626 .1621 L
.19556 .16228 L
.19489 .16242 L
.19448 .16248 L
.19404 .16253 L
.19354 .16258 L
.19308 .1626 L
.19258 .16261 L
.19204 .1626 L
.19152 .16258 L
.19103 .16254 L
.19015 .16244 L
.18915 .16229 L
.18702 .16181 L
.18494 .16119 L
.18258 .16036 L
.17799 .15845 L
.15638 .14845 L
.15039 .14613 L
.14724 .14511 L
.14438 .14434 L
.14184 .14381 L
.14046 .1436 L
.13982 .14352 L
.13924 .14346 L
.13824 .14339 L
.13717 .14336 L
.13617 .14338 L
.13514 .14345 L
.13416 .14359 L
.13368 .14367 L
.13317 .14379 L
.13229 .14403 L
.13151 .14431 L
.12989 .14511 L
Mistroke
.12906 .14567 L
.12835 .14625 L
.1272 .14745 L
.12612 .149 L
.12526 .15067 L
.1246 .15241 L
.12404 .1544 L
.12361 .15644 L
.12331 .15839 L
.12307 .16063 L
.12297 .16184 L
.12289 .16321 L
.12285 .16392 L
.12282 .16469 L
.12277 .1661 L
.12274 .16754 L
.12273 .16893 L
.12273 .1702 L
.12274 .17162 L
.12276 .17311 L
.12279 .17451 L
.12287 .17773 L
.12309 .18379 L
.12341 .19068 L
.12384 .19849 L
.1243 .20579 L
.12494 .21372 L
.12584 .22214 L
.12698 .22996 L
.12854 .23787 L
.1304 .24502 L
.13296 .25262 L
.1363 .26031 L
.1433 .27219 L
.15263 .28345 L
.16247 .29231 L
.17153 .29873 L
.18147 .30427 L
.18616 .30634 L
.19027 .30784 L
.19441 .30903 L
.1963 .30946 L
.19831 .30983 L
.20048 .31014 L
.20145 .31025 L
.20251 .31034 L
.20365 .31042 L
.20473 .31048 L
.20539 .3105 L
.20603 .31051 L
Mistroke
.2066 .31052 L
.20723 .31052 L
.20828 .31051 L
.20928 .31048 L
.21038 .31043 L
.2116 .31036 L
.21413 .31015 L
.21654 .30989 L
.22116 .30922 L
.22634 .3083 L
.25254 .30256 L
.26483 .30017 L
.26818 .29965 L
.27168 .29918 L
.27464 .29886 L
.27628 .29872 L
.27783 .29861 L
.27958 .29851 L
.28116 .29846 L
.28284 .29844 L
.28378 .29844 L
.28465 .29845 L
.28622 .29851 L
.28762 .29859 L
.28921 .29872 L
.29069 .29888 L
.29351 .2993 L
.29602 .29983 L
.29869 .30055 L
.3014 .30147 L
.30663 .30389 L
.31117 .30674 L
.32055 .31498 L
.33542 .33164 L
.3435 .33976 L
.35246 .34675 L
.35709 .34952 L
.36222 .35198 L
.36689 .35374 L
.37125 .35502 L
.37973 .35678 L
.38451 .35748 L
.38895 .35801 L
.39807 .35891 L
.4027 .35928 L
.40728 .35957 L
.41 .35971 L
.41146 .35976 L
.41302 .35982 L
.41436 .35985 L
Mistroke
.41585 .35988 L
.41723 .35989 L
.41853 .35989 L
.41932 .35989 L
.42018 .35989 L
.42177 .35986 L
.42333 .35983 L
.4248 .35978 L
.4264 .35971 L
.42818 .35962 L
.43186 .35936 L
.43494 .35907 L
.43835 .35869 L
.44535 .35765 L
.45188 .35639 L
.45793 .35496 L
.48392 .34636 L
.49476 .3422 L
.49953 .34058 L
.50154 .33999 L
.50369 .33944 L
.50546 .33906 L
.50733 .33874 L
.50835 .3386 L
.50929 .3385 L
.51022 .33843 L
.51111 .33838 L
.51207 .33835 L
.51309 .33834 L
.51405 .33837 L
.51493 .33841 L
.51598 .33849 L
.51711 .3386 L
.5192 .3389 L
.52128 .33928 L
.52357 .33979 L
.52788 .34093 L
.53868 .34435 L
.55148 .34833 L
.56412 .35146 L
.57032 .35257 L
.57342 .35299 L
.57677 .35333 L
.5785 .35346 L
.58006 .35354 L
.58087 .35358 L
.58175 .3536 L
.58266 .35361 L
.58349 .35362 L
.58429 .35361 L
Mistroke
.58503 .35359 L
.5864 .35354 L
.58716 .3535 L
.58785 .35345 L
.58939 .35332 L
.59061 .35318 L
.59189 .353 L
.59413 .35261 L
.59668 .35201 L
.59888 .35134 L
.60329 .34953 L
.60701 .34737 L
.61004 .34509 L
.61306 .34229 L
.61854 .33601 L
.62321 .33036 L
.62865 .32486 L
.63178 .3224 L
.63556 .32005 L
.63934 .31827 L
.64314 .31692 L
.64559 .31625 L
.64798 .3157 L
.65265 .31491 L
.65537 .31458 L
.65792 .31433 L
.66384 .31391 L
.6688 .31363 L
.67354 .31332 L
.67614 .31311 L
.67896 .31282 L
.68383 .31211 L
.68659 .31157 L
.68948 .31087 L
.69452 .30923 L
.69946 .30703 L
.70376 .30458 L
.7213 .29017 L
.72897 .28373 L
.73325 .28082 L
.73569 .27944 L
.73808 .2783 L
.74045 .27738 L
.74311 .27658 L
.74431 .2763 L
.74561 .27604 L
.74675 .27586 L
.74799 .27571 L
.74926 .2756 L
.75045 .27554 L
Mistroke
.75159 .27551 L
.75282 .27552 L
.75418 .27558 L
.75547 .27566 L
.7569 .2758 L
.75845 .276 L
.76163 .27653 L
.76516 .2773 L
.77182 .27915 L
.78483 .28354 L
.79178 .28593 L
.79915 .28821 L
.80264 .28913 L
.80589 .28986 L
.80873 .29036 L
.81029 .29058 L
.81173 .29074 L
.81251 .29081 L
.81334 .29087 L
.81477 .29093 L
.81557 .29094 L
.8164 .29093 L
.81786 .29087 L
.81918 .29076 L
.82038 .2906 L
.82166 .29038 L
.82299 .29006 L
.82438 .28965 L
.82583 .2891 L
.8283 .28785 L
.83047 .28634 L
.83229 .28468 L
.83514 .28103 L
.83647 .27867 L
.83752 .27634 L
.83836 .27405 L
.83915 .27144 L
.84026 .26643 L
.84078 .26325 L
.84116 .26033 L
.84183 .25381 L
.84233 .2484 L
.84266 .24545 L
.84302 .2428 L
.84342 .24043 L
.84394 .23801 L
.84462 .23558 L
.84537 .23349 L
.84619 .23168 L
.84721 .22983 L
Mistroke
.84948 .22678 L
.85096 .22524 L
.85279 .22365 L
.85707 .22068 L
.86633 .21571 L
.87596 .21059 L
.88556 .20389 L
.88984 .19971 L
.89179 .19732 L
.8936 .19462 L
.895 .19203 L
.89603 .18959 L
.89645 .18839 L
.89683 .18707 L
.89713 .1858 L
.89735 .18464 L
.89749 .18369 L
.89759 .18279 L
.89767 .18183 L
.89771 .18078 L
.89771 .17986 L
.89769 .17888 L
.89763 .17796 L
.89755 .17712 L
.89741 .17604 L
.89725 .17505 L
.89676 .17282 L
.89619 .17086 L
.89546 .16881 L
.89358 .16477 L
.8914 .16112 L
.88514 .15336 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl503[o001lo`040?oo00So1@04o`80Bol400?o1@0fo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool02_l00`3oo`0co`00O?l0103oo`09o`030?l000Go00<0ool0
B?l00`3oo`0<o`80<ol007go0P0;o`801Ol00`3oo`17o`030?oo00oo0P0ao`00TOl00`3oo`16o`03
0?oo01;o0P0_o`00TOl00`3oo`15o`030?oo01Go0P0]o`00B?l704;o00<0ool0A?l00`3oo`0Ho`03
0?oo02[o000^o`L04Ol300Ko1@0mo`030?oo04Co00<0ool06Ol202[o000]o`030?oo00Go0P0=o`80
3_l200Ko0P0co`030?oo04?o00<0ool07?l00`3oo`0Wo`00;Ol00`3oo`07o`802_l00`3oo`0@o`<0
0_l303?o00<0ool0@_l00`3oo`0No`030?oo02Ko000/o`030?oo00[o0P07o`030?oo01Co100do`03
0?oo047o00<0ool08?l00`3oo`0Uo`00;?l00`3oo`0<o`<00_l201Oo00<0ool0=Ol00`3oo`11o`03
0?oo027o00<0ool09?l002co00<0ool03_l401So00<0ool0=Ol00`3oo`10o`030?oo02;o00<0ool0
9?l002co00<0ool0:Ol00`3oo`0fo`030?oo03oo00<0ool09?l00`3oo`0So`00;?l00`3oo`0Yo`03
0?oo03Ko00<0ool0?_l00`3oo`0Uo`030?oo02?o000/o`030?oo02So00<0ool08ol200Co0P03o`@0
1Ol00`3oo`0mo`030?oo02Ko00<0ool08ol002co00<0ool0:?l00`3oo`0Ro`040?oo00So00<0ool0
1_l00`3oo`0lo`030?oo02Oo00<0ool08ol002co00<0ool09ol00`3oo`0So`040?oo00Wo00<0ool0
1Ol203go00<0ool09ol00`3oo`0So`00;?l00`3oo`0Wo`030?oo01go1002o`040?oo00[o00<0ool0
1?l00`3oo`0ko`030?oo02So00<0ool08ol002co00<0ool09ol00`3oo`0So`040?oo00So00@0ool0
1Ol00`3oo`0ko`030?oo02So00<0ool08ol002co00<0ool09_l00`3oo`0Uo`802_l200Ko00<0ool0
>_l00`3oo`0Xo`809Ol002co00<0ool09_l00`3oo`0io`030?oo03Wo00<0ool0:Ol00`3oo`0To`00
;Ol00`3oo`0Uo`030?oo03Wo00<0ool0>Ol00`3oo`0Xo`030?oo02Go000]o`030?oo02Co00<0ool0
>_l00`3oo`0ho`030?oo02So00<0ool09_l000Oo1@02o`801?l201Oo00<0ool02Ol501Ko00<0ool0
2Ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co0P08o`030?oo01So1@0:o`030?oo01Go1@02o`80
1?l200?o0009o`030?oo00Oo00@0ool05_l00`3oo`0;o`030?oo01Ko00<0ool02?l0103oo`08o`04
0?oo02;o00<0ool08Ol0103oo`08o`040?oo00Ko00<0ool06ol00`3oo`08o`806_l00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool02Ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00Co00<0ool06ol00`3oo`05o`<07?l00`3oo`0:o`40
0Ol1o`0000?o00000P04o`030?oo00So0P0Go`030?oo00?o1004o`030?oo01Go00<0ool00ol400;o
00@0ool02Ol202?o00<0ool08Ol0103oo`09o`801_l00`3oo`0Lo`030?oo00?o0P0Oo`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01Ko00<0ool02ol00`3oo`0Eo`030?oo00Wo00@0ool02Ol00`3o
o`0Ro`030?oo027o00@0ool02Ol00`3oo`05o`030?oo01co00<0ool00_l00`3oo`0Oo`030?oo00So
00<0ool00_l000So0P0:o`<05_l00`3oo`0:o`805ol00`3oo`0:o`802_l302;o00<0ool08_l200[o
0`05o`030?oo01_o0P03o`030?oo01oo0P0:o`<00_l002go00<0ool08ol00`3oo`0ko`030?oo03Ko
00<0ool08?l00`3oo`0`o`00;_l00`3oo`0So`030?oo03[o00<0ool0=Ol00`3oo`0Po`030?oo037o
000^o`030?oo02Co00<0ool0>Ol00`3oo`0eo`030?oo023o00<0ool0<Ol002ko00<0ool09?l403So
00<0ool0=?l00`3oo`0Qo`030?oo037o000^o`030?oo02So0`0eo`030?oo03Co00<0ool08Ol00`3o
o`0ao`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool00P04o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0Qo`030?oo00Go00<0ool09Ol200;o
00<0ool0:_l00`3oo`0Yo`030?oo00Go00<0ool08Ol00`3o000Zo`030?oo00Oo000`o`030?oo02oo
00<0ool0;?l00`3oo`0<o`P07?l00`3oo`0To`030?oo03;o000ao`030?oo02ko00<0ool04ol701;o
00<0ool02?l400So100Go`030?oo02Go00<0ool0<_l0037o00<0ool0;ol00`3oo`0?o`<01ol2013o
00<0ool01?l4013o0P0Do`030?oo02Ko00<0ool0<_l003;o00<0ool0;ol00`3oo`0<o`803?l00`3o
o`0=o`030?oo00;o0P0Fo`804?l202Wo00<0ool0<_l003?o00<0ool0;_l00`3oo`0:o`803ol200go
00<0o`000P0Jo`803?l2013o200Bo`030?oo03?o000co`030?oo02oo00<0ool01ol201?o00<0ool0
2_l201oo0P09o`030?oo00ko0`07o`<03ol00`3oo`0co`00=?l00`3oo`0_o`030?oo00?o0`0Eo`03
0?oo00Oo100Ro`<00ol3013o00<0ool02ol200co00<0ool0=?l003Go0P0`o`H06?l900?o00<0ool0
8_l5017o00<0ool03_l500Go0P0go`00=ol00`3oo`1Go`030?oo03Oo00<0ool05?l603So000ho`03
0?oo05Ko00<0ool0=_l00`3oo`1Co`00>Ol201Ko2`0eo`030?oo03Go00<0ool0E?l003_o0P0Ao`<0
2ol203?o00<0ool0=?l00`3oo`1Eo`00?Ol200_o100@o`030?oo033o00<0ool0<Ol305So000oo`/0
5Ol201co0P04o`800ol400Go00<0ool0;_l305_o001Qo`030?oo01So00@0ool02?l00`3oo`06o`03
0?oo02Co2P1No`00H_l00`3oo`0Go`040?oo00Wo00<0ool01Ol202?o0P1Xo`00Hol00`3oo`0Fo`04
0?oo00[o00<0ool01?l00`3oo`0Po`80J_l006?o00<0ool05_l0103oo`08o`040?oo00Go00<0ool0
7_l206co001To`030?oo01Ko0P0:o`801_l00`3oo`0Mo`030?oo06co001Uo`030?oo02Wo00<0ool0
7Ol00`3oo`1/o`00I_l00`3oo`0Xo`040?oo01_o00<0ool0KOl006Oo00<0ool09_l:01Ko00<0ool0
KOl006So00<0ool08ol20003o`3o00Oo0`0Ao`80L?l006Wo0P0Po`<00ol00`3oo`09o`<03?l207;o
001[o`806_l400Ko00<0ool03?l600;o101do`00KOl5017o100:o`030?oo017o101go`00L_lA00ko
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co
00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o
0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103o
o`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88755 .15366 m
.86759 .13956 L
.84685 .12883 L
.83766 .12437 L
.82827 .12022 L
.82293 .1183 L
.81808 .11697 L
.81573 .1165 L
.81449 .11631 L
.81318 .11614 L
.81204 .11602 L
.81139 .11597 L
.81078 .11594 L
.80971 .1159 L
.80857 .11589 L
.80795 .11589 L
.80738 .11591 L
.80631 .11597 L
.80516 .11607 L
.80392 .11622 L
.80284 .11639 L
.8017 .1166 L
.79969 .11708 L
.79737 .1178 L
.79525 .11862 L
.79059 .12098 L
.78671 .12359 L
.78338 .12632 L
.7765 .1334 L
.75064 .16361 L
.73603 .17853 L
.72826 .18784 L
.72496 .19256 L
.72233 .19694 L
.72018 .20121 L
.7183 .20582 L
.71691 .21035 L
.71634 .21276 L
.71591 .21496 L
.71555 .21728 L
.71529 .21942 L
.71492 .2233 L
.7146 .22729 L
.71439 .22935 L
.71411 .23153 L
.71379 .23346 L
.71341 .23522 L
.71238 .23869 L
.71165 .24058 L
.7107 .24267 L
Mistroke
.70851 .24659 L
.70263 .2546 L
.69538 .26258 L
.68514 .27229 L
.67482 .28061 L
.66564 .28641 L
.66318 .28761 L
.66063 .28867 L
.65846 .28939 L
.65614 .28998 L
.65487 .29021 L
.6537 .29037 L
.65259 .29047 L
.65142 .29052 L
.65037 .29052 L
.64939 .29047 L
.64831 .29038 L
.64716 .29022 L
.646 .29001 L
.64472 .28972 L
.64215 .28895 L
.63746 .287 L
.62775 .28147 L
.6178 .2752 L
.60467 .26781 L
.59785 .26468 L
.59146 .26227 L
.58563 .26052 L
.57955 .25916 L
.57645 .25866 L
.5736 .25832 L
.572 .25818 L
.57055 .25808 L
.56976 .25803 L
.5689 .258 L
.56736 .25796 L
.56592 .25796 L
.56461 .25798 L
.5632 .25803 L
.56175 .25812 L
.5605 .25821 L
.55935 .25833 L
.55682 .25864 L
.55414 .25909 L
.5518 .25956 L
.54688 .26085 L
.53805 .26401 L
.53027 .26736 L
.49415 .2829 L
.48341 .28701 L
Mistroke
.47852 .28854 L
.4765 .28906 L
.47438 .28951 L
.47329 .2897 L
.4723 .28984 L
.47137 .28993 L
.47042 .29 L
.4699 .29002 L
.46934 .29003 L
.46837 .29 L
.46745 .28992 L
.46662 .28981 L
.46573 .28964 L
.46493 .28943 L
.46407 .28915 L
.46318 .28879 L
.46146 .28786 L
.45998 .28679 L
.45483 .28124 L
.44884 .27379 L
.44585 .27075 L
.44213 .26779 L
.43973 .26632 L
.43736 .26514 L
.43492 .26421 L
.43354 .26378 L
.43219 .26344 L
.43086 .26316 L
.42938 .26292 L
.428 .26276 L
.4267 .26266 L
.42526 .2626 L
.42364 .2626 L
.42191 .26267 L
.42026 .26281 L
.41868 .26299 L
.41716 .26322 L
.41425 .26378 L
.41105 .26457 L
.4075 .26565 L
.39986 .2686 L
.39172 .27254 L
.37789 .28066 L
.3653 .28867 L
.35988 .29172 L
.35728 .2929 L
.3561 .29333 L
.35507 .29364 L
.35414 .29386 L
.35329 .294 L
Mistroke
.35242 .29407 L
.35195 .29408 L
.35152 .29406 L
.35071 .29397 L
.35026 .29389 L
.34985 .29378 L
.34904 .2935 L
.3482 .29311 L
.34678 .29216 L
.34549 .29098 L
.34312 .28809 L
.33742 .27933 L
.33052 .26987 L
.32257 .26204 L
.31737 .25831 L
.31222 .25543 L
.30679 .25307 L
.30071 .25102 L
.2889 .24797 L
.28347 .24644 L
.28075 .24544 L
.2785 .24437 L
.27662 .24322 L
.27484 .24174 L
.27341 .24012 L
.27232 .23839 L
.27186 .23744 L
.27143 .23638 L
.27112 .23541 L
.27083 .23432 L
.27058 .23308 L
.2704 .23186 L
.27031 .23109 L
.27025 .23034 L
.27018 .22888 L
.27017 .22754 L
.2702 .22606 L
.27027 .22475 L
.27039 .22327 L
.27056 .22162 L
.2708 .21978 L
.27136 .21632 L
.27335 .20724 L
.27595 .19798 L
.28207 .1803 L
.28868 .1648 L
.29225 .15786 L
.29541 .15266 L
.29814 .14905 L
.29945 .14766 L
Mistroke
.30074 .14656 L
.30175 .14592 L
.30228 .14567 L
.30276 .14551 L
.30317 .14542 L
.30353 .14538 L
.30392 .14539 L
.30427 .14545 L
.30462 .14557 L
.30497 .14576 L
.30527 .14599 L
.30551 .14624 L
.30576 .14656 L
.30598 .14695 L
.30618 .14741 L
.30632 .14788 L
.30638 .14813 L
.30643 .1484 L
.30649 .14893 L
.30651 .14946 L
.30649 .14996 L
.30643 .15051 L
.30637 .15083 L
.30631 .15112 L
.30615 .15172 L
.30592 .15235 L
.3054 .15338 L
.30503 .15393 L
.30464 .15442 L
.30379 .15524 L
.30329 .15562 L
.30271 .156 L
.30214 .1563 L
.30158 .15655 L
.30039 .15695 L
.2997 .15712 L
.2989 .15726 L
.29801 .15736 L
.29715 .15741 L
.29613 .15742 L
.2956 .1574 L
.29499 .15737 L
.29389 .15728 L
.29281 .15715 L
.29023 .1567 L
.28721 .156 L
.27498 .15213 L
.26138 .14747 L
.2545 .14542 L
.24829 .1439 L
Mistroke
.24467 .14322 L
.24133 .14274 L
.23952 .14255 L
.23789 .14242 L
.23613 .14233 L
.23525 .14231 L
.2343 .14231 L
.23335 .14232 L
.23245 .14235 L
.23081 .14245 L
.22991 .14254 L
.22908 .14263 L
.22726 .1429 L
.22585 .14317 L
.22434 .14353 L
.22143 .14444 L
.21886 .14551 L
.21668 .14665 L
.21231 .14969 L
.20906 .15272 L
.20614 .15604 L
.20147 .16185 L
.20021 .1632 L
.19901 .16429 L
.1979 .16509 L
.19726 .16546 L
.19664 .16575 L
.19604 .16598 L
.19547 .16615 L
.19483 .16629 L
.19411 .16639 L
.19339 .16642 L
.19259 .16639 L
.19179 .16631 L
.19103 .16618 L
.19023 .166 L
.18946 .16579 L
.18795 .16528 L
.18423 .16363 L
.16497 .15208 L
.15265 .14492 L
.14687 .14208 L
.14196 .1401 L
.13961 .13935 L
.13843 .13903 L
.13721 .13875 L
.13619 .13856 L
.1351 .1384 L
.13415 .13832 L
.13329 .13828 L
Mistroke
.1328 .13829 L
.13228 .13831 L
.13138 .13839 L
.13054 .13853 L
.1298 .13872 L
.12897 .13901 L
.12826 .13933 L
.1275 .13978 L
.12683 .14028 L
.1257 .14142 L
.1252 .14208 L
.12469 .14289 L
.12393 .1445 L
.12335 .1462 L
.12305 .14732 L
.12281 .14844 L
.12245 .15071 L
.1223 .15206 L
.12219 .15336 L
.12209 .15485 L
.12205 .15562 L
.12201 .15649 L
.12197 .15785 L
.12195 .15914 L
.12194 .1606 L
.12194 .16143 L
.12194 .16219 L
.12195 .16377 L
.12197 .16467 L
.12198 .16551 L
.12205 .16879 L
.12244 .18181 L
.12269 .18964 L
.12297 .19837 L
.1233 .2068 L
.1235 .21085 L
.12373 .21462 L
.12433 .22212 L
.12475 .22607 L
.12529 .23026 L
.12658 .23795 L
.12813 .24478 L
.13046 .25258 L
.13364 .26065 L
.13723 .26779 L
.14108 .27403 L
.14976 .285 L
.16022 .29483 L
.17041 .30218 L
.1758 .30538 L
Mistroke
.18055 .30779 L
.18474 .30959 L
.18919 .31111 L
.19145 .31171 L
.19357 .31216 L
.19581 .31251 L
.19707 .31265 L
.19769 .31271 L
.19825 .31275 L
.19923 .31281 L
.20029 .31285 L
.20128 .31286 L
.20223 .31285 L
.20335 .31281 L
.2044 .31276 L
.20557 .31267 L
.20684 .31255 L
.20925 .31225 L
.2116 .31187 L
.21732 .31071 L
.24111 .30404 L
.25561 .29985 L
.26372 .29777 L
.27128 .29616 L
.27534 .29546 L
.27744 .29516 L
.27963 .2949 L
.28145 .29472 L
.28241 .29464 L
.28342 .29457 L
.28519 .29448 L
.28681 .29444 L
.28854 .29444 L
.28951 .29447 L
.2904 .2945 L
.292 .29461 L
.29279 .29468 L
.29366 .29477 L
.29514 .29496 L
.2967 .29522 L
.29944 .29583 L
.30231 .29669 L
.30531 .29788 L
.3104 .30067 L
.31494 .30413 L
.31887 .3079 L
.33337 .32649 L
.34102 .33632 L
.34956 .34515 L
.35407 .3488 L
Mistroke
.3591 .35208 L
.3637 .35443 L
.36798 .35614 L
.37235 .35748 L
.37631 .35841 L
.38085 .35923 L
.38518 .35983 L
.39367 .36073 L
.3986 .36115 L
.40351 .3615 L
.40791 .36174 L
.40916 .36179 L
.4105 .36184 L
.41178 .36188 L
.41295 .36191 L
.41428 .36193 L
.41573 .36195 L
.41709 .36195 L
.41839 .36194 L
.41993 .36192 L
.42137 .36188 L
.42297 .36183 L
.4247 .36175 L
.42799 .36155 L
.43119 .36129 L
.43736 .36058 L
.44419 .35949 L
.45173 .35789 L
.46514 .35405 L
.47681 .34966 L
.49749 .34034 L
.50211 .33843 L
.50449 .33759 L
.50664 .33695 L
.50843 .33652 L
.50934 .33634 L
.51032 .33618 L
.51087 .33611 L
.51139 .33605 L
.5124 .33596 L
.51331 .33592 L
.51429 .33591 L
.51479 .33591 L
.51532 .33593 L
.51627 .33599 L
.51731 .33609 L
.51845 .33624 L
.52053 .33663 L
.52253 .33711 L
.52735 .33861 L
Mistroke
.53297 .34073 L
.55649 .34977 L
.56296 .35178 L
.56986 .35355 L
.57344 .3543 L
.57731 .35497 L
.57898 .3552 L
.58075 .35542 L
.58227 .35557 L
.58388 .35569 L
.58464 .35574 L
.58547 .35578 L
.58692 .35582 L
.58773 .35583 L
.58859 .35583 L
.58939 .35582 L
.59011 .3558 L
.59142 .35573 L
.59281 .35562 L
.59424 .35547 L
.59552 .35529 L
.5979 .35483 L
.59995 .35431 L
.60217 .35356 L
.60415 .35274 L
.60756 .35083 L
.61083 .34827 L
.61377 .34519 L
.61619 .342 L
.62432 .32937 L
.62681 .32621 L
.62988 .32309 L
.6332 .32055 L
.63666 .31858 L
.63857 .31774 L
.64046 .31703 L
.64421 .31596 L
.64643 .3155 L
.64854 .31514 L
.65093 .31484 L
.65352 .31459 L
.6644 .31416 L
.66707 .31411 L
.66853 .31408 L
.66991 .31404 L
.67233 .31395 L
.67361 .31389 L
.67498 .31381 L
.67737 .31363 L
.67986 .31338 L
Mistroke
.68197 .31309 L
.68424 .31271 L
.68667 .31218 L
.68922 .3115 L
.69367 .3099 L
.69858 .30745 L
.70298 .30453 L
.71069 .29774 L
.7187 .28914 L
.7231 .28446 L
.72735 .28043 L
.73166 .2771 L
.73585 .27467 L
.73836 .2736 L
.74085 .2728 L
.74225 .27247 L
.74357 .27222 L
.74506 .27201 L
.74591 .27193 L
.74668 .27188 L
.74799 .27183 L
.74941 .27183 L
.75063 .27188 L
.752 .27199 L
.75349 .27216 L
.75513 .2724 L
.75815 .273 L
.76436 .27471 L
.77029 .2768 L
.78502 .28293 L
.79324 .28637 L
.80082 .28923 L
.80703 .29115 L
.81039 .29195 L
.81335 .29249 L
.81498 .2927 L
.81645 .29284 L
.81803 .29292 L
.81889 .29293 L
.81966 .29292 L
.82104 .29285 L
.82248 .2927 L
.8238 .29248 L
.82497 .29222 L
.82713 .29152 L
.82827 .29102 L
.82926 .29051 L
.83107 .28934 L
.83262 .28802 L
.83424 .28624 L
Mistroke
.83555 .28438 L
.83678 .2821 L
.83788 .27945 L
.83867 .27692 L
.83934 .27403 L
.83982 .27117 L
.84014 .26847 L
.84028 .26689 L
.84038 .2654 L
.84046 .26376 L
.8405 .26284 L
.84052 .26199 L
.84056 .26029 L
.84057 .25933 L
.84058 .25845 L
.84058 .25683 L
.84057 .25601 L
.84056 .25512 L
.84055 .25347 L
.84052 .25195 L
.8405 .25017 L
.84048 .24859 L
.84047 .24709 L
.84047 .24572 L
.84049 .24429 L
.8405 .2435 L
.84052 .24278 L
.84057 .24155 L
.84063 .24042 L
.84072 .23921 L
.84084 .23795 L
.841 .23667 L
.84118 .2355 L
.84163 .23342 L
.84191 .23237 L
.84228 .23126 L
.84317 .22916 L
.8442 .22738 L
.84534 .22585 L
.84691 .2242 L
.84857 .22283 L
.85306 .22009 L
.86309 .21587 L
.87549 .2109 L
.88149 .2079 L
.8868 .20461 L
.89108 .20125 L
.89503 .19717 L
.89804 .19282 L
.89935 .19021 L
Mistroke
.90028 .18768 L
.90067 .18629 L
.90096 .18495 L
.90116 .18374 L
.90131 .18241 L
.90136 .18172 L
.90139 .18096 L
.9014 .18016 L
.90138 .17941 L
.9013 .17807 L
.90112 .17661 L
.90091 .17535 L
.90066 .17418 L
.89989 .17152 L
.8989 .16896 L
.89759 .16627 L
.8946 .16155 L
.89129 .15747 L
.88755 .15366 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl403_o001lo`040?oo00So1@04o`80C?l300?o100go`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0:o`030?oo03Co001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`03
0?oo00_o0P0do`00OOl200_o0P05o`030?oo04Oo00<0ool03_l203;o002Ao`030?oo04Ko00<0ool0
4Ol2033o000^o`<0H?l00`3oo`15o`030?oo01Co0P0^o`00;Ol200;o0`0Do`D0A?l00`3oo`14o`03
0?oo01Oo0P0/o`00;?l200Ko00<0ool03ol300Co100:o`<0<ol00`3oo`13o`030?oo01[o0P0Zo`00
;?l00`3oo`06o`803Ol200_o0`07o`030?l003?o00<0ool0@_l00`3oo`0Mo`80:?l002co00<0ool0
2?l200[o00<0ool03_l300?o00@0ool0<ol00`3oo`12o`030?oo01oo00<0ool09Ol002co00<0ool0
2_l200Oo00<0ool04_l703?o00<0ool0@Ol00`3oo`0Qo`030?oo02Co000/o`030?oo00co0P04o`03
0?oo01Go00<0ool0=Ol00`3oo`10o`030?oo02;o00<0ool09?l002co00<0ool03_l00`3oo`0201Ko
00<0ool0=_l00`3oo`0oo`030?oo02?o00<0ool09?l002co00<0ool03ol301Oo00<0ool0=_l00`3o
o`0no`030?oo02Go0P0To`00;?l00`3oo`0Yo`030?oo03Ko00<0ool0?Ol00`3oo`0Wo`030?oo02;o
000/o`030?oo02So00<0ool08ol200Co0P03o`@01Ol00`3oo`0lo`030?oo02So00<0ool08_l002co
00<0ool0:?l00`3oo`0Ro`040?oo00So00<0ool01_l00`3oo`0ko`030?oo02Wo00<0ool08_l002co
00<0ool0:?l00`3oo`0Ro`040?oo00Wo00<0ool01Ol203_o00<0ool0:_l00`3oo`0Ro`00;?l00`3o
o`0Wo`030?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`0io`030?oo02_o00<0ool08_l002co
00<0ool09ol00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`0ho`030?oo02co00<0ool08_l002co
00<0ool09_l00`3oo`0Uo`802_l200Ko00<0ool0>?l00`3oo`0[o`030?oo02?o000/o`030?oo02Ko
00<0ool0>Ol00`3oo`0ho`030?oo02_o00<0ool08ol002co00<0ool09_l00`3oo`0io`030?oo03Oo
00<0ool0:_l202Ko000/o`030?oo02Go00<0ool0>_l00`3oo`0go`030?oo02Wo00<0ool09_l000Oo
1@02o`801?l201Ko00<0ool02_l501Ko00<0ool02Ol200Co0P04o`808ol00`3oo`0Ro`801?l200Co
0P06o`030?oo01[o1@09o`805ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Eo`030?oo00co
00<0ool05_l00`3oo`08o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool01Ol00`3o
o`0Lo`030?oo00Oo0P0Ko`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Co`030?oo00co
00<0ool05_l00`3oo`08o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool00ol00`3o
o`0Lo`030?oo00Co0`0Mo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Oo00<0ool0
0ol400Co00<0ool05Ol200Co1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200Ko00<0ool0
7?l00`3oo`02o`808?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Fo`030?oo00_o00<0ool0
5Ol00`3oo`09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool01Ol00`3oo`0Lo`05
0?ooo`008_l00`3oo`08o`030?oo00;o0008o`802_l301Ko00<0ool02_l201Oo00<0ool02_l200[o
0`0Ro`030?oo02;o0P0:o`<01Ol00`3oo`0Ko`800_l00`3oo`0Po`802_l300;o000]o`030?oo02?o
00<0ool0>ol00`3oo`0eo`808?l203?o000]o`030?oo02Co00<0ool0>_l00`3oo`0eo`030?oo01oo
00<0ool0<_l002go00<0ool09?l00`3oo`0jo`030?oo03Go00<0ool07ol00`3oo`0bo`00;_l00`3o
o`0To`030?oo03Wo00<0ool0=Ol00`3oo`0Oo`030?oo03;o000^o`030?oo02Go100go`030?oo03Co
00<0ool08?l00`3oo`0bo`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1Ol200So00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3o000200Co00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0o`001_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo023o00<0ool01_l00`3o
o`0Uo`050?ooo`00;?l00`3oo`0;o`P05_l00`3oo`05o`030?oo027o00<0o`00:_l00`3oo`07o`00
;ol00`3oo`0_o`030?oo01Oo100Bo`030?oo00Wo0P08o`@06Ol00`3oo`0To`030?oo03;o000`o`03
0?oo02oo00<0ool04ol300?o0`0@o`030?oo00Oo0P0>o`030?oo01Go00<0ool09Ol00`3oo`0bo`00
<?l00`3oo`0`o`030?oo013o0P09o`030?oo00go00<0ool01?l3017o0P0Eo`030?oo02Go00<0ool0
<_l0037o00<0ool0<?l00`3oo`0=o`803?l200go00<0ool00_l201Ko0P0Bo`030?oo00_o1P0Eo`03
0?oo03;o000bo`030?oo02oo00<0ool02ol200oo00<0ool02ol00`3o000201[o0P0>o`803?l200Ko
0`0Bo`030?oo03;o000bo`030?oo033o00<0ool02?l201;o00<0ool02_l201oo00<0ool02_l00`3o
o`0;o`030?oo00Wo0P0?o`030?oo03?o000co`030?oo033o00<0ool01Ol201Go00<0ool01_l4027o
0P09o`030?oo00_o00<0ool03?l300co00<0ool0<ol003Co00<0ool0<?l00`3oo`03o`030?oo01Ko
0P02o`@00ol00`3oo`0Qo`801?l300go00<0ool04?l400Oo00<0ool0=?l003Go00<0ool0;ol200;o
0P0Jo`@01_l00`3oo`0So`@03ol00`3oo`0Eo`L0=ol003Ko00<0ool06_l:00_o0`0Vo`030?oo03Ko
00<0ool0Dol003Oo0P0Eo`D02_l203;o00<0ool0=Ol00`3oo`1Do`00>Ol00`3oo`0?o`<04Ol00`3o
o`0_o`030?oo03?o0P1Go`00>_l300[o100Eo`030?oo02ko00<0ool0<Ol205Wo000mo`801Ol301[o
00<0ool06Ol200Co0P03o`@01Ol00`3oo`0_o`80Fol003oo1@0No`030?oo01Oo00@0ool02?l00`3o
o`06o`030?oo02?o301Mo`00H_l00`3oo`0Go`040?oo00Wo00<0ool01Ol202;o0`1Xo`00Hol00`3o
o`0Fo`040?oo00[o00<0ool01?l00`3oo`0Oo`80Jol006Co00<0ool05Ol0103oo`08o`040?oo00Go
00<0ool07_l00`3oo`1[o`00IOl00`3oo`0Eo`802_l200Ko00<0ool07_l00`3oo`1[o`00IOl00`3o
o`0Yo`030?oo01go00<0ool0K?l006Ko00<0ool0:?l601Wo00<0ool0KOl006Oo00<0ool09Ol300Go
100Do`030?oo06ko001Xo`030?oo02;o0P02o`030?oo00Oo0P0Bo`030?oo06ko001Yo`808?l200Co
00<0ool02Ol200oo00<0ool0Kol006_o00<0ool06ol200Ko00<0ool02ol300[o0P1bo`00K?l201So
0`08o`030?oo00ko2`1co`00K_l700[o1`0;o`030?oo08co001eo`X04_l00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88995 .15396 m
.8682 .14018 L
.84638 .12976 L
.83634 .12487 L
.82631 .12022 L
.82155 .11838 L
.81921 .11763 L
.81705 .11704 L
.81513 .11661 L
.81301 .11625 L
.81191 .1161 L
.81088 .116 L
.80979 .11592 L
.80918 .1159 L
.80863 .11588 L
.80756 .11588 L
.8066 .11591 L
.80549 .11598 L
.80445 .11608 L
.80349 .1162 L
.8026 .11634 L
.80063 .11676 L
.79865 .11732 L
.79651 .11808 L
.79232 .12008 L
.78877 .12233 L
.78515 .12517 L
.77906 .13112 L
.75025 .1614 L
.7413 .16895 L
.73163 .17767 L
.72357 .18652 L
.72024 .19107 L
.71767 .19531 L
.71568 .19934 L
.71404 .20372 L
.71339 .20598 L
.71291 .20806 L
.71254 .21023 L
.71237 .21142 L
.71226 .2125 L
.71217 .21363 L
.71213 .21419 L
.7121 .2148 L
.71206 .21588 L
.71205 .21688 L
.71205 .21748 L
.71206 .21805 L
.71208 .2191 L
.71211 .21969 L
Mistroke
.71213 .22022 L
.71219 .22143 L
.7125 .22555 L
.71263 .2273 L
.71269 .22822 L
.71275 .2292 L
.71279 .23019 L
.71281 .23111 L
.71282 .23165 L
.71282 .23216 L
.7128 .23311 L
.71279 .23362 L
.71276 .23416 L
.71269 .23512 L
.71264 .23566 L
.71258 .23617 L
.71241 .23732 L
.71203 .23927 L
.71154 .24108 L
.70995 .24527 L
.70803 .24896 L
.70533 .2532 L
.69811 .26235 L
.67851 .2816 L
.67253 .28627 L
.66702 .28989 L
.66221 .29232 L
.65957 .29332 L
.65833 .29368 L
.65717 .29397 L
.65605 .29419 L
.65484 .29436 L
.65375 .29446 L
.65274 .2945 L
.65166 .29449 L
.65047 .29441 L
.64987 .29434 L
.64922 .29425 L
.64806 .29403 L
.64703 .29379 L
.64591 .29346 L
.64364 .29263 L
.63951 .29061 L
.62864 .2832 L
.61716 .27463 L
.60352 .26582 L
.5964 .26214 L
.58974 .25931 L
.58369 .25727 L
.5774 .2557 L
Mistroke
.57422 .25512 L
.5713 .25473 L
.56966 .25456 L
.56819 .25445 L
.56738 .2544 L
.56651 .25436 L
.56494 .25431 L
.56416 .25431 L
.56335 .25431 L
.56191 .25434 L
.56039 .25442 L
.55955 .25447 L
.55877 .25454 L
.55614 .25483 L
.55472 .25505 L
.55346 .25527 L
.54884 .25633 L
.54486 .25759 L
.53709 .2609 L
.52927 .26494 L
.51325 .27323 L
.49089 .28412 L
.4809 .28839 L
.47838 .28928 L
.47587 .29005 L
.47385 .29055 L
.47276 .29076 L
.47178 .2909 L
.47089 .29099 L
.46996 .29104 L
.46911 .29104 L
.46836 .29099 L
.4677 .29092 L
.467 .29079 L
.4663 .29061 L
.46566 .2904 L
.46495 .29011 L
.4643 .28978 L
.46314 .28902 L
.46197 .28801 L
.46077 .2867 L
.45606 .27928 L
.45346 .27494 L
.45061 .27094 L
.44748 .26752 L
.44352 .26441 L
.44117 .26307 L
.4388 .26203 L
.43599 .26114 L
.43432 .26077 L
Mistroke
.43274 .26051 L
.43129 .26035 L
.42964 .26025 L
.42879 .26022 L
.42786 .26023 L
.42615 .26029 L
.42441 .26043 L
.42272 .26064 L
.42117 .26088 L
.41946 .26122 L
.41545 .26223 L
.41106 .26367 L
.40297 .26713 L
.38707 .27639 L
.37406 .28577 L
.3631 .29402 L
.35835 .29714 L
.35633 .29819 L
.35465 .29887 L
.35377 .29912 L
.35298 .29927 L
.35218 .29934 L
.35174 .29934 L
.35135 .29931 L
.35061 .29918 L
.35021 .29907 L
.34984 .29893 L
.34911 .29857 L
.34836 .29806 L
.34717 .29693 L
.34595 .29539 L
.3438 .29185 L
.33889 .28262 L
.33548 .27696 L
.33164 .27169 L
.32747 .26709 L
.32233 .26266 L
.31716 .25927 L
.31208 .25671 L
.30573 .25436 L
.29965 .25272 L
.29283 .25132 L
.28704 .25019 L
.28461 .24963 L
.28219 .24896 L
.28008 .24822 L
.27833 .24744 L
.27648 .24634 L
.2756 .24567 L
.27475 .2449 L
Mistroke
.27407 .24416 L
.27339 .24328 L
.27236 .24153 L
.27192 .24051 L
.2715 .23931 L
.27118 .2381 L
.27094 .23691 L
.27077 .23577 L
.27064 .23446 L
.27056 .23302 L
.27053 .23159 L
.27057 .22995 L
.27066 .22813 L
.2708 .22652 L
.271 .22468 L
.27163 .22034 L
.27243 .21602 L
.27658 .19935 L
.28254 .1806 L
.2896 .16219 L
.29318 .1544 L
.29669 .14809 L
.29943 .14433 L
.30012 .1436 L
.30083 .14295 L
.30142 .14251 L
.30202 .14215 L
.3023 .14201 L
.30258 .14191 L
.30307 .14178 L
.30351 .14175 L
.30396 .1418 L
.30433 .14193 L
.30471 .14213 L
.30503 .1424 L
.30531 .1427 L
.3056 .14312 L
.30585 .14357 L
.30605 .14402 L
.30623 .14456 L
.30639 .14519 L
.30646 .14555 L
.30652 .1459 L
.3066 .14655 L
.30665 .14728 L
.30665 .14798 L
.30663 .14866 L
.30657 .14927 L
.30649 .14995 L
.30635 .15069 L
Mistroke
.30617 .15147 L
.30572 .15284 L
.30537 .15366 L
.30499 .15438 L
.30453 .15513 L
.30395 .15589 L
.30281 .15705 L
.30209 .1576 L
.30136 .15806 L
.3006 .15845 L
.29982 .15877 L
.29908 .15901 L
.29823 .15922 L
.29725 .15939 L
.29673 .15945 L
.29614 .1595 L
.29506 .15954 L
.294 .15952 L
.2929 .15945 L
.29167 .15932 L
.29046 .15914 L
.28933 .15893 L
.28341 .15743 L
.2694 .15248 L
.25585 .14766 L
.24831 .14547 L
.24468 .14461 L
.24141 .14397 L
.23807 .14348 L
.23628 .14329 L
.2346 .14317 L
.23281 .14309 L
.23119 .14308 L
.22946 .14314 L
.22851 .1432 L
.22766 .14327 L
.22618 .14344 L
.22486 .14365 L
.22345 .14394 L
.222 .1443 L
.21968 .14503 L
.2173 .14603 L
.21475 .14742 L
.21263 .14887 L
.20855 .15264 L
.20553 .1565 L
.20321 .16018 L
.19946 .16649 L
.19859 .16765 L
.19761 .16868 L
Mistroke
.19703 .16917 L
.19648 .16953 L
.1959 .16983 L
.19532 .17004 L
.19477 .17017 L
.19425 .17024 L
.19362 .17025 L
.19299 .17021 L
.19239 .17012 L
.19181 .16999 L
.19115 .16979 L
.19042 .16954 L
.18714 .16793 L
.18289 .16522 L
.15921 .14751 L
.14707 .13943 L
.14213 .13668 L
.1395 .13541 L
.13716 .13444 L
.13492 .13367 L
.13302 .13317 L
.13208 .133 L
.13114 .13288 L
.13024 .13282 L
.12975 .13282 L
.12931 .13284 L
.12888 .13287 L
.12843 .13293 L
.12764 .1331 L
.12693 .13333 L
.12631 .1336 L
.12569 .13397 L
.12506 .13444 L
.12446 .13503 L
.12396 .13566 L
.12356 .13627 L
.12316 .137 L
.12285 .1377 L
.12255 .1385 L
.12229 .13935 L
.12208 .14016 L
.12174 .14188 L
.12157 .143 L
.12145 .14407 L
.12134 .14531 L
.12125 .14667 L
.12119 .1479 L
.12116 .1491 L
.12113 .15045 L
.12112 .15194 L
Mistroke
.12112 .1534 L
.12113 .15425 L
.12114 .15506 L
.12117 .15672 L
.1212 .15856 L
.12138 .1653 L
.12173 .18225 L
.1218 .18989 L
.12183 .19419 L
.12186 .1983 L
.12189 .20065 L
.12192 .20321 L
.12196 .20555 L
.122 .20771 L
.12212 .212 L
.1222 .21417 L
.1223 .21656 L
.12257 .22114 L
.12289 .22531 L
.12337 .23011 L
.12396 .23463 L
.12535 .24253 L
.12743 .25091 L
.13031 .25935 L
.13414 .26783 L
.14279 .28151 L
.14807 .28782 L
.15315 .29302 L
.16478 .3026 L
.17441 .30869 L
.17931 .31117 L
.18352 .31291 L
.18545 .31357 L
.1875 .31417 L
.18926 .3146 L
.19118 .31496 L
.19223 .31511 L
.19321 .31522 L
.19426 .31531 L
.19485 .31535 L
.19539 .31537 L
.19636 .31539 L
.19743 .31538 L
.19846 .31535 L
.19942 .31529 L
.2004 .31521 L
.20149 .31509 L
.2035 .31481 L
.20749 .31403 L
.21257 .31271 L
Mistroke
.21771 .31114 L
.24616 .30101 L
.26205 .29572 L
.26958 .29361 L
.2764 .29203 L
.27965 .29143 L
.28316 .2909 L
.28495 .2907 L
.2866 .29054 L
.28831 .29043 L
.29013 .29036 L
.2916 .29034 L
.29318 .29037 L
.2948 .29046 L
.29628 .29059 L
.29764 .29075 L
.29886 .29094 L
.3015 .29149 L
.3044 .29234 L
.30694 .29333 L
.3096 .29466 L
.31226 .29631 L
.31643 .29966 L
.32064 .30407 L
.33624 .32742 L
.34373 .33837 L
.35114 .3469 L
.35911 .35348 L
.36338 .35603 L
.36792 .35813 L
.3722 .35961 L
.3762 .36063 L
.37976 .36132 L
.38361 .36189 L
.39166 .36276 L
.39656 .36318 L
.40131 .36353 L
.40385 .36368 L
.40672 .36383 L
.40835 .36389 L
.40988 .36394 L
.41138 .36398 L
.41215 .36399 L
.41301 .36401 L
.41456 .36402 L
.41601 .36402 L
.41678 .36401 L
.41763 .364 L
.41852 .36398 L
.41936 .36396 L
Mistroke
.42094 .36391 L
.4224 .36385 L
.42575 .36366 L
.42889 .36341 L
.43243 .36304 L
.43981 .36199 L
.44685 .36061 L
.4535 .35897 L
.46825 .35412 L
.48271 .34773 L
.49369 .34196 L
.50331 .3369 L
.50763 .33505 L
.50995 .3343 L
.51099 .33403 L
.51207 .3338 L
.51299 .33366 L
.51347 .3336 L
.51398 .33354 L
.51489 .33349 L
.51573 .33347 L
.51624 .33347 L
.5168 .3335 L
.51732 .33353 L
.51781 .33358 L
.51874 .33369 L
.51974 .33386 L
.52164 .33429 L
.52344 .33481 L
.52775 .33643 L
.53664 .34054 L
.54915 .34648 L
.56228 .35184 L
.5687 .35399 L
.57472 .35567 L
.57801 .35643 L
.58149 .35711 L
.58458 .35757 L
.58604 .35775 L
.58734 .35788 L
.58879 .35799 L
.59031 .35807 L
.59169 .35811 L
.59297 .35811 L
.59442 .35807 L
.59573 .35799 L
.5964 .35793 L
.59712 .35786 L
.59856 .35767 L
.59991 .35743 L
Mistroke
.6011 .35717 L
.60365 .35644 L
.60572 .35563 L
.60773 .35462 L
.60951 .35347 L
.61105 .35226 L
.61359 .34969 L
.61588 .34655 L
.61798 .3428 L
.62128 .33566 L
.62455 .3291 L
.62655 .326 L
.62905 .323 L
.63066 .32148 L
.63233 .32016 L
.63582 .31807 L
.63778 .3172 L
.64 .31642 L
.64222 .31582 L
.64432 .31538 L
.64692 .31499 L
.64842 .31482 L
.64984 .3147 L
.6511 .31461 L
.6525 .31454 L
.65321 .31452 L
.65399 .3145 L
.65539 .31447 L
.6567 .31446 L
.65746 .31445 L
.65817 .31446 L
.65961 .31447 L
.66042 .31448 L
.66117 .31449 L
.66399 .31453 L
.66537 .31455 L
.66662 .31457 L
.66809 .31458 L
.66889 .31459 L
.66965 .31459 L
.67048 .31458 L
.67124 .31457 L
.67207 .31456 L
.67295 .31454 L
.67366 .31452 L
.67442 .31449 L
.67579 .31442 L
.6774 .31431 L
.67887 .31418 L
.68162 .31383 L
Mistroke
.68412 .31338 L
.6868 .31275 L
.68958 .31188 L
.69406 .30997 L
.69872 .30715 L
.70292 .30378 L
.70709 .2996 L
.71415 .29102 L
.72175 .28136 L
.72616 .27661 L
.72844 .27457 L
.73063 .27288 L
.73278 .2715 L
.73525 .27024 L
.73784 .26925 L
.73932 .26883 L
.74074 .26853 L
.74206 .26832 L
.74333 .26819 L
.74451 .26812 L
.74581 .2681 L
.74725 .26815 L
.74882 .26827 L
.75033 .26846 L
.75173 .26869 L
.75479 .26935 L
.7577 .27016 L
.7646 .27264 L
.77803 .27882 L
.79237 .28592 L
.80019 .28945 L
.80731 .29219 L
.81117 .29338 L
.81314 .29389 L
.81518 .29433 L
.81684 .29461 L
.8177 .29474 L
.81862 .29484 L
.82019 .29496 L
.82162 .295 L
.82303 .29496 L
.82381 .29491 L
.82452 .29483 L
.8259 .29461 L
.82666 .29445 L
.82732 .29428 L
.82847 .29391 L
.82966 .29342 L
.83166 .29231 L
.83348 .29089 L
Mistroke
.83518 .28904 L
.83654 .287 L
.83759 .28491 L
.83858 .28221 L
.83904 .28058 L
.83939 .279 L
.83967 .27745 L
.83988 .27596 L
.84005 .27436 L
.84013 .2735 L
.84019 .27256 L
.84027 .27079 L
.84031 .26911 L
.84031 .26813 L
.84031 .26724 L
.84029 .26621 L
.84026 .26523 L
.84022 .26427 L
.84018 .26326 L
.84007 .26143 L
.83975 .2573 L
.83903 .25005 L
.83843 .24381 L
.83832 .24244 L
.83823 .24102 L
.83815 .23949 L
.83812 .23813 L
.83811 .23677 L
.83814 .23554 L
.8382 .23447 L
.83829 .23335 L
.83844 .23219 L
.83865 .23097 L
.83893 .22976 L
.83926 .22869 L
.84001 .22691 L
.84051 .22598 L
.84103 .22518 L
.84222 .22373 L
.84351 .22252 L
.84514 .22133 L
.84712 .22018 L
.851 .21849 L
.85599 .21684 L
.86775 .21349 L
.88101 .20883 L
.88757 .20559 L
.89393 .20132 L
.89636 .19921 L
.89867 .1968 L
Mistroke
.90043 .19458 L
.902 .19211 L
.90314 .18983 L
.90364 .18859 L
.9041 .18724 L
.90444 .18599 L
.90469 .18484 L
.90488 .18367 L
.905 .18255 L
.90508 .18125 L
.90507 .17987 L
.905 .17856 L
.90486 .17737 L
.90463 .17596 L
.90429 .17445 L
.90343 .17172 L
.90245 .16938 L
.90114 .16686 L
.8978 .16196 L
.89399 .15768 L
.88995 .15396 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl403_o001lo`040?oo00So1@04o`80Bol300?o100ho`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool02?l203Ko001lo`040?oo00Wo00<0o`001Ol00`3oo`18o`03
0?oo00_o0P0do`00OOl200_o0P05o`030?oo04Oo00<0ool03_l203;o000]o`@0H?l00`3oo`16o`03
0?oo017o0P0`o`00;?l00`3oo`02o`80G_l00`3oo`15o`030?oo01Co0P0^o`00;?l00`3oo`04o`03
0?oo02Ko0P0co`030?oo04Co00<0ool05ol202co000/o`030?oo00Go0P0?o`/02ol303?o00<0ool0
@ol00`3oo`0Jo`80:_l002co00<0ool01ol00`3oo`0;o`802_l200So00@0ool0<ol00`3oo`12o`03
0?oo01go0P0Xo`00;?l00`3oo`08o`802_l00`3oo`0<o`@01?l0103oo`0co`030?oo047o00<0ool0
8?l00`3oo`0Uo`00;?l00`3oo`0:o`030?oo00Ko00<0ool04Ol20004o`3oo`80<ol00`3oo`10o`03
0?oo02;o00<0ool09?l002co00<0ool02ol200Ko00<0ool04ol503Co00<0ool0?ol00`3oo`0So`03
0?oo02Co000/o`030?oo00go00<0ool00_l00`3oo`0Eo`030?oo03Go00<0ool0?_l00`3oo`0Uo`03
0?oo02?o000/o`030?oo00ko00<0ool00P0Go`030?oo03Go00<0ool0?Ol00`3oo`0Wo`030?oo02;o
000/o`030?oo00oo0`0Go`030?oo03Ko00<0ool0??l00`3oo`0Yo`030?oo027o000/o`030?oo02Wo
00<0ool08_l200Co0P03o`@01Ol00`3oo`0ko`030?oo02[o00<0ool08Ol002co00<0ool0:?l00`3o
o`0Ro`040?oo00So00<0ool01_l00`3oo`0jo`030?oo02_o00<0ool08Ol002co00<0ool0:?l00`3o
o`0Ro`040?oo00Wo00<0ool01Ol203[o00<0ool0;?l00`3oo`0Qo`00;?l00`3oo`0Xo`030?oo01co
1002o`040?oo00[o00<0ool01?l00`3oo`0ho`030?oo02go00<0ool08Ol002co00<0ool09ol00`3o
o`0So`040?oo00So00@0ool01Ol00`3oo`0go`030?oo02ko00<0ool08Ol002co00<0ool09ol00`3o
o`0To`802_l200Ko00<0ool0=_l00`3oo`0^o`030?oo02;o000/o`030?oo02Ko00<0ool0>Ol00`3o
o`0fo`030?oo02go0P0To`00;?l00`3oo`0Vo`030?oo03Wo00<0ool0=_l00`3oo`0/o`030?oo02Co
000/o`030?oo02Ko00<0ool0>Ol00`3oo`0eo`030?oo02_o0P0Wo`001ol500;o0P04o`805_l00`3o
o`0:o`D05ol00`3oo`08o`801?l200Co0P0So`030?oo02;o0P04o`801?l200Go00<0ool06ol500So
0`0Go`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Go00<0ool03?l00`3oo`0Fo`030?oo00So
00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`04o`030?oo01go00<0ool01Ol301co
00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01?o00<0ool03?l00`3oo`0Fo`030?oo00So
00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`02o`030?oo01go00<0ool00ol201oo
00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`805_l00`3oo`04o`@01?l00`3oo`0Fo`03
0?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200Go00<0ool07Ol00`3oo`03027o
00<0ool02?l200?o0009o`030?oo00So00<0ool05Ol00`3oo`0<o`030?oo01Ko00<0ool02?l0103o
o`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00Co00<0ool07Ol0103o000So`030?oo00So
00<0ool00_l000So0P0:o`<05Ol00`3oo`0;o`805ol200_o0P0:o`<08_l00`3oo`0Ro`802_l300Co
00<0ool07?l20003o`3o02;o0P0:o`<00_l002co00<0ool09?l00`3oo`0ko`030?oo03Go00<0ool0
7ol00`3oo`0bo`00;?l00`3oo`0To`80??l00`3oo`0eo`030?oo01ko00<0ool0<ol002go00<0ool0
9?l00`3oo`0jo`030?oo03Go00<0ool07_l00`3oo`0co`00;Ol00`3oo`0To`030?oo03[o00<0ool0
=Ol00`3oo`0No`80=?l002go00<0ool09Ol203[o00<0ool0=?l00`3oo`0Po`030?oo03;o000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`@01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`03o`H01_l00`3oo`06o`030?oo00Ko00<0ool01_l200Oo00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo01oo00<0ool01ol00`3oo`0To`800ol00`3oo`0Zo`030?oo00So0`06o`D04ol00`3o
o`06o`030?oo023o00<0o`00:_l00`3oo`07o`00;_l00`3oo`0`o`030?oo01Ko200?o`030?oo00Ko
0P0>o`806Ol00`3oo`0So`030?oo03;o000_o`030?oo033o00<0ool05?l00`3oo`05o`803_l00`3o
o`04o`804_l00`3oo`0Eo`030?oo02Co00<0ool0<_l002oo00<0ool0<Ol00`3oo`0@o`<02_l00`3o
o`0;o`030?oo00;o0P0Eo`805?l00`3oo`09o`L05Ol00`3oo`0bo`00<?l00`3oo`0ao`030?oo00go
0P0=o`030?oo00_o00<0ool00P0Io`030?oo013o00<0ool02Ol00`3oo`05o`804ol00`3oo`0bo`00
<?l00`3oo`0bo`030?oo00[o0P0@o`030?oo00[o0`0Lo`030?oo00ko00<0ool02?l200[o0P0Ao`03
0?oo03;o000ao`030?oo037o00<0ool02Ol00`3oo`0Ao`030?oo00So0P0Oo`803Ol00`3oo`08o`03
0?oo00co00<0ool03_l00`3oo`0bo`00<_l00`3oo`0ao`030?oo00Oo00<0ool04_l00`3oo`05o`<0
00?o0?l08?l00`3oo`09o`030?oo00So00<0ool03_l200go0P0do`00<ol00`3oo`0`o`030?oo00Ko
00<0ool05?l200?o0P04o`030?oo023o0P08o`030?oo00Wo00<0ool04?l400Wo00<0ool0<ol003Co
00<0ool07Ol900[o00<0ool00ol201So1006o`030?oo02;o0P04o`802ol00`3oo`0Eo`801_l00`3o
o`0do`00=Ol00`3oo`0Ho`@02Ol300So00D0oooo000To`030?oo02?o1@0=o`030?oo01Oo1P0go`00
=_l00`3oo`0Eo`804?l00`3oo`05o`@09Ol00`3oo`0co`80Eol003Oo00<0ool04Ol301?o00<0ool0
;Ol00`3oo`0bo`030?oo05Oo000ho`803Ol401Oo00<0ool0;?l00`3oo`0ao`030?oo05So000jo`80
2Ol201_o00<0ool06?l200Co0P03o`@01Ol00`3oo`0_o`80Fol003co2@0No`030?oo01Ko00@0ool0
2?l00`3oo`06o`030?oo02?o301Mo`00Hol00`3oo`0Fo`040?oo00Wo00<0ool01Ol202;o0P1Yo`00
I?l00`3oo`0Eo`040?oo00[o00<0ool01?l00`3oo`0Oo`80Jol006Co00<0ool05Ol0103oo`08o`04
0?oo00Go00<0ool07_l00`3oo`1[o`00IOl00`3oo`0Eo`802_l200Ko00<0ool07_l00`3oo`1[o`00
I_l00`3oo`0Xo`030?l000<06ol00`3oo`1[o`00I_l00`3oo`0Wo`@00ol201So00<0ool0K?l006Oo
00<0ool09Ol00`3o0007o`805_l00`3oo`1/o`00J?l00`3oo`0Ro`800_l00`3oo`07o`804ol00`3o
o`1]o`00JOl00`3oo`0Oo`801?l00`3oo`09o`<03ol00`3oo`1^o`00J_l00`3oo`0Lo`801_l00`3o
o`0<o`803?l2073o001[o`806Ol300So00<0ool03_l500Go0`1ao`00KOl4013o1@0;o`030?oo01?o
1@1do`00LOl@013o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co
00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co
00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o
0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103o
o`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.89236 .15426 m
.84591 .13069 L
.82734 .1215 L
.82265 .11938 L
.82009 .11837 L
.81767 .11754 L
.81541 .1169 L
.81422 .11662 L
.81296 .11637 L
.81191 .11619 L
.81074 .11604 L
.80968 .11594 L
.80869 .11588 L
.80762 .11585 L
.80663 .11586 L
.80555 .11591 L
.80495 .11595 L
.8044 .116 L
.8034 .11613 L
.80232 .11631 L
.80038 .11674 L
.7985 .1173 L
.79654 .11804 L
.79311 .11972 L
.78929 .12218 L
.78587 .12491 L
.77862 .13219 L
.7495 .15983 L
.72965 .17457 L
.72447 .1792 L
.7195 .18442 L
.71582 .18914 L
.71263 .19438 L
.71145 .19682 L
.71042 .1994 L
.70965 .20178 L
.70911 .20399 L
.70873 .20604 L
.70857 .20718 L
.70847 .20821 L
.70839 .20934 L
.70834 .21055 L
.70833 .21168 L
.70836 .21271 L
.70841 .21382 L
.7085 .215 L
.70862 .21609 L
.70874 .21709 L
.70956 .22169 L
.71052 .22601 L
Mistroke
.71103 .22832 L
.71145 .23047 L
.71175 .23244 L
.71188 .23355 L
.71197 .23458 L
.71201 .23544 L
.71203 .2364 L
.71202 .2374 L
.71197 .23834 L
.71188 .2393 L
.71177 .24023 L
.71163 .24106 L
.71145 .24198 L
.71091 .24407 L
.71024 .24606 L
.70829 .25041 L
.70529 .25552 L
.69679 .26673 L
.68736 .27704 L
.67669 .28703 L
.67115 .2914 L
.66828 .29336 L
.66528 .29513 L
.66258 .29646 L
.66017 .2974 L
.65902 .29775 L
.65779 .29805 L
.65664 .29827 L
.65559 .29841 L
.65441 .29849 L
.65331 .2985 L
.65231 .29845 L
.65124 .29833 L
.65007 .29813 L
.64884 .29783 L
.64662 .29708 L
.64456 .29615 L
.64261 .29509 L
.63804 .29201 L
.61472 .2724 L
.60129 .26301 L
.59356 .25875 L
.58628 .25554 L
.58012 .25343 L
.57702 .25259 L
.57371 .25185 L
.57186 .25151 L
.57018 .25125 L
.56863 .25105 L
.56695 .25087 L
Mistroke
.56547 .25076 L
.56411 .25069 L
.56271 .25065 L
.56122 .25065 L
.5604 .25066 L
.55965 .25069 L
.55825 .25077 L
.55679 .25089 L
.55525 .25106 L
.5526 .25147 L
.55029 .25196 L
.54543 .25339 L
.54091 .25527 L
.52715 .26335 L
.49134 .28381 L
.48105 .2889 L
.47658 .29069 L
.47478 .29126 L
.47293 .29173 L
.47197 .29191 L
.47111 .29201 L
.47024 .29207 L
.46976 .29208 L
.46934 .29206 L
.46892 .29203 L
.46854 .29199 L
.46813 .29192 L
.46771 .29182 L
.46728 .2917 L
.46689 .29156 L
.46618 .29125 L
.4656 .29091 L
.46501 .29049 L
.46403 .28958 L
.46307 .28837 L
.46209 .28679 L
.45867 .27915 L
.4567 .27473 L
.45445 .27058 L
.45117 .2661 L
.44907 .26398 L
.44653 .26199 L
.44391 .26045 L
.44253 .25981 L
.4412 .2593 L
.43996 .2589 L
.43855 .25854 L
.43708 .25824 L
.43547 .25802 L
.43374 .25788 L
Mistroke
.43209 .25784 L
.43121 .25785 L
.43022 .25789 L
.42918 .25797 L
.42818 .25806 L
.42632 .25831 L
.42458 .25862 L
.42054 .25959 L
.41684 .26077 L
.41271 .26238 L
.40501 .26611 L
.38685 .27805 L
.37211 .29024 L
.36596 .29565 L
.36041 .30034 L
.35824 .30197 L
.35619 .30329 L
.35463 .30408 L
.35382 .30438 L
.3531 .30456 L
.35245 .30464 L
.35178 .30463 L
.35118 .30453 L
.35065 .30436 L
.35019 .30414 L
.34971 .30382 L
.34922 .30341 L
.34877 .30295 L
.34781 .30167 L
.34695 .30021 L
.345 .29605 L
.34067 .28621 L
.33751 .28028 L
.33386 .27477 L
.32866 .26878 L
.32314 .26411 L
.31739 .26057 L
.31395 .25894 L
.3107 .25767 L
.30751 .25665 L
.304 .25573 L
.29767 .2545 L
.29073 .2535 L
.28763 .25305 L
.28499 .2526 L
.28246 .25205 L
.28027 .2514 L
.27819 .25055 L
.27712 .24998 L
.27621 .24938 L
Mistroke
.27484 .24823 L
.27357 .24673 L
.27303 .24587 L
.27251 .24486 L
.27212 .24392 L
.27176 .24282 L
.27145 .24159 L
.2712 .24019 L
.27103 .23881 L
.27093 .23743 L
.2709 .2366 L
.27088 .23571 L
.2709 .23404 L
.271 .23215 L
.27108 .23104 L
.27117 .22998 L
.27166 .22601 L
.27228 .22214 L
.27423 .21265 L
.28536 .17407 L
.29174 .15659 L
.29499 .1491 L
.29777 .14376 L
.2992 .14151 L
.30043 .13994 L
.30104 .13929 L
.30168 .13874 L
.30201 .1385 L
.30232 .13832 L
.30259 .13819 L
.30289 .13808 L
.30333 .13798 L
.30373 .13798 L
.30395 .13802 L
.30417 .13809 L
.30455 .13829 L
.30492 .13859 L
.30524 .13895 L
.30555 .13944 L
.30586 .14005 L
.30609 .14067 L
.30631 .14142 L
.3065 .14228 L
.30664 .14315 L
.30673 .144 L
.30678 .14481 L
.30681 .14578 L
.30679 .14673 L
.30677 .14727 L
.30674 .14776 L
Mistroke
.30668 .14833 L
.30662 .14888 L
.30648 .14984 L
.30628 .15088 L
.30601 .15202 L
.30564 .15319 L
.30482 .15518 L
.30391 .15675 L
.30273 .15824 L
.30141 .15944 L
.30003 .16032 L
.2991 .16076 L
.29819 .16109 L
.29712 .16138 L
.29656 .16149 L
.29591 .16159 L
.29479 .1617 L
.29414 .16173 L
.29352 .16174 L
.29235 .1617 L
.29105 .16158 L
.28974 .16141 L
.28826 .16115 L
.28543 .16049 L
.27963 .15871 L
.26474 .153 L
.25751 .15024 L
.24964 .14756 L
.24585 .14645 L
.24229 .14555 L
.23912 .14488 L
.23582 .14435 L
.23399 .14413 L
.23237 .14398 L
.23077 .14389 L
.22929 .14385 L
.2276 .14387 L
.22608 .14395 L
.22446 .1441 L
.2228 .14434 L
.22129 .14463 L
.21973 .14502 L
.21706 .14592 L
.21416 .14728 L
.21178 .14879 L
.20956 .15058 L
.20773 .15242 L
.20435 .157 L
.20211 .16124 L
.20028 .16556 L
Mistroke
.19889 .16901 L
.19823 .17045 L
.1976 .17161 L
.19685 .17268 L
.19642 .17313 L
.19594 .17352 L
.19546 .1738 L
.19499 .17398 L
.19454 .17407 L
.19402 .17411 L
.19349 .17407 L
.19297 .17397 L
.19237 .1738 L
.19168 .17354 L
.19014 .17275 L
.18856 .17175 L
.18115 .16579 L
.15659 .14422 L
.14518 .13547 L
.13966 .13185 L
.13698 .13032 L
.13436 .12901 L
.13221 .12812 L
.13114 .12776 L
.13006 .12746 L
.12912 .12726 L
.12831 .12714 L
.12754 .12709 L
.12685 .1271 L
.12619 .12717 L
.12554 .1273 L
.12502 .12747 L
.12448 .12771 L
.12394 .12804 L
.12341 .12846 L
.12296 .12893 L
.12258 .12941 L
.1222 .13002 L
.12183 .13075 L
.12129 .13225 L
.12103 .13328 L
.12082 .13434 L
.12067 .13535 L
.12053 .13651 L
.12044 .1376 L
.12036 .13886 L
.12033 .13952 L
.12031 .14024 L
.12028 .14158 L
.12027 .14297 L
Mistroke
.12027 .14434 L
.12028 .14561 L
.12031 .14702 L
.12048 .15356 L
.12058 .15726 L
.12069 .16132 L
.12076 .16504 L
.12079 .16719 L
.12081 .16923 L
.12083 .17113 L
.12083 .17288 L
.12084 .17493 L
.12083 .17688 L
.12082 .17896 L
.1208 .18119 L
.12078 .1831 L
.12075 .18523 L
.1206 .19458 L
.12052 .19962 L
.1205 .20108 L
.12049 .20247 L
.12048 .20373 L
.12047 .20508 L
.12046 .20638 L
.12046 .20755 L
.12047 .20984 L
.12047 .21112 L
.12049 .21251 L
.12051 .21383 L
.12053 .21503 L
.12059 .21741 L
.12067 .21967 L
.12088 .22388 L
.12103 .22623 L
.1212 .22839 L
.12168 .23326 L
.12217 .23713 L
.12284 .24134 L
.12462 .24969 L
.12726 .25852 L
.13092 .2675 L
.13511 .27542 L
.13948 .28211 L
.14957 .29407 L
.15943 .30289 L
.16975 .3101 L
.175 .31305 L
.17948 .31514 L
.18358 .31663 L
.18557 .31717 L
Mistroke
.18772 .31762 L
.18884 .3178 L
.18985 .31792 L
.19042 .31797 L
.19103 .31801 L
.19161 .31804 L
.19215 .31805 L
.19312 .31805 L
.19367 .31804 L
.1942 .31802 L
.19475 .31799 L
.19526 .31795 L
.19643 .31783 L
.1974 .31771 L
.19845 .31754 L
.20042 .31716 L
.2052 .3159 L
.21663 .31178 L
.24589 .29951 L
.2636 .29276 L
.27324 .28972 L
.27782 .28851 L
.28191 .28761 L
.28571 .28693 L
.28742 .28669 L
.2892 .28649 L
.29106 .28633 L
.29276 .28623 L
.29378 .2862 L
.29473 .28619 L
.29647 .28623 L
.29732 .28627 L
.29824 .28633 L
.2992 .28642 L
.30007 .28652 L
.30161 .28675 L
.30321 .28706 L
.30503 .28753 L
.30671 .28806 L
.30969 .28929 L
.31212 .29059 L
.31455 .29221 L
.31859 .29573 L
.32252 .30027 L
.32649 .30605 L
.34114 .33253 L
.34799 .34301 L
.35181 .34774 L
.35601 .35206 L
.36024 .35557 L
Mistroke
.36411 .35811 L
.36798 .36011 L
.3716 .36154 L
.37562 .36272 L
.37929 .3635 L
.3833 .36412 L
.38758 .36462 L
.39565 .36531 L
.40052 .36565 L
.40306 .3658 L
.40552 .36592 L
.40791 .36601 L
.40915 .36604 L
.41055 .36607 L
.41199 .36609 L
.4128 .3661 L
.41356 .3661 L
.41498 .3661 L
.41651 .36608 L
.41782 .36605 L
.41924 .36601 L
.42061 .36595 L
.42187 .36589 L
.42504 .36569 L
.42812 .36542 L
.43479 .36461 L
.43872 .36398 L
.44246 .36327 L
.45078 .36131 L
.45844 .35903 L
.47192 .35385 L
.48507 .3473 L
.49662 .34045 L
.50194 .33714 L
.50636 .33455 L
.51009 .3327 L
.51206 .33195 L
.51297 .33167 L
.51381 .33145 L
.51458 .33129 L
.51527 .33118 L
.51604 .33109 L
.51687 .33104 L
.51766 .33103 L
.51852 .33107 L
.51935 .33115 L
.52011 .33126 L
.52093 .33142 L
.52168 .3316 L
.52343 .33212 L
Mistroke
.52678 .33346 L
.53564 .33812 L
.54711 .34451 L
.55952 .35066 L
.57232 .35575 L
.57945 .35792 L
.58569 .35935 L
.58737 .35964 L
.58911 .3599 L
.59059 .36009 L
.59214 .36024 L
.593 .36031 L
.59377 .36035 L
.59525 .36041 L
.59606 .36042 L
.59693 .36042 L
.59774 .3604 L
.59848 .36036 L
.59987 .36026 L
.60113 .36012 L
.60223 .35995 L
.60338 .35973 L
.60458 .35944 L
.6058 .35908 L
.60784 .3583 L
.60976 .35732 L
.6114 .35624 L
.61404 .35383 L
.61529 .35229 L
.61632 .35075 L
.61827 .34694 L
.61954 .34365 L
.62077 .33986 L
.62325 .33215 L
.62492 .32817 L
.62688 .32471 L
.62903 .32197 L
.63043 .3206 L
.63184 .31944 L
.63366 .31825 L
.63552 .31729 L
.63731 .31655 L
.63938 .31589 L
.64178 .31533 L
.64305 .31511 L
.64447 .31491 L
.64572 .31478 L
.64642 .31472 L
.64709 .31467 L
.64841 .31459 L
Mistroke
.64964 .31455 L
.65083 .31453 L
.65213 .31452 L
.65337 .31453 L
.65452 .31456 L
.65581 .31459 L
.65723 .31465 L
.66013 .31478 L
.66574 .31506 L
.66736 .31512 L
.66816 .31515 L
.66889 .31517 L
.67026 .3152 L
.67175 .31521 L
.67303 .3152 L
.67422 .31518 L
.67552 .31513 L
.67691 .31505 L
.67834 .31493 L
.6797 .31479 L
.6822 .31441 L
.6845 .31394 L
.68659 .31337 L
.69114 .31163 L
.69362 .31033 L
.69583 .30893 L
.70053 .30509 L
.70439 .30096 L
.70824 .29599 L
.71522 .28557 L
.72171 .27635 L
.72557 .27199 L
.72953 .26861 L
.73186 .26713 L
.73413 .26603 L
.73546 .26553 L
.73674 .26513 L
.73812 .26481 L
.73887 .26467 L
.73968 .26455 L
.74098 .26442 L
.74221 .26437 L
.7436 .26437 L
.74511 .26446 L
.74589 .26454 L
.74675 .26465 L
.74833 .2649 L
.75139 .26559 L
.75441 .26649 L
.75773 .26769 L
Mistroke
.76396 .2704 L
.78024 .27905 L
.79632 .28789 L
.80315 .29124 L
.81011 .29413 L
.81356 .2953 L
.81534 .29581 L
.81721 .29627 L
.81887 .29661 L
.82034 .29684 L
.82172 .297 L
.82312 .2971 L
.82458 .29711 L
.82587 .29705 L
.82728 .29689 L
.82856 .29664 L
.82969 .29633 L
.83068 .29597 L
.83276 .29493 L
.83374 .29425 L
.8346 .29354 L
.83555 .29259 L
.83634 .29164 L
.83774 .28942 L
.83839 .28806 L
.83892 .28667 L
.83937 .28524 L
.83973 .28382 L
.84007 .28212 L
.8403 .2805 L
.84047 .27892 L
.84057 .27743 L
.84064 .27574 L
.84065 .27393 L
.84061 .27205 L
.84056 .27098 L
.84051 .27 L
.84037 .26808 L
.8402 .26624 L
.83971 .26209 L
.83903 .25756 L
.83776 .25026 L
.83645 .24288 L
.83596 .23974 L
.83574 .23811 L
.83558 .23664 L
.83547 .23542 L
.83539 .23416 L
.83534 .23282 L
.83534 .23166 L
Mistroke
.8354 .2304 L
.8355 .2293 L
.83568 .22815 L
.83579 .22759 L
.83594 .227 L
.83624 .22605 L
.83663 .22508 L
.83752 .22349 L
.83818 .22263 L
.83901 .22175 L
.8408 .22034 L
.84291 .21917 L
.84517 .21823 L
.86931 .21273 L
.87707 .21069 L
.88506 .2079 L
.8914 .20492 L
.89753 .20099 L
.90043 .19854 L
.90274 .19615 L
.90497 .19322 L
.90662 .19031 L
.90721 .18896 L
.90774 .18751 L
.9081 .18626 L
.90841 .18486 L
.90855 .18408 L
.90864 .18335 L
.90872 .18255 L
.90876 .18171 L
.90877 .18029 L
.90873 .17948 L
.90867 .17873 L
.90848 .17728 L
.90833 .17646 L
.90816 .17571 L
.90732 .17288 L
.90626 .17035 L
.90486 .16771 L
.90108 .16244 L
.8958 .15709 L
.89236 .15426 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0COl403_o001lo`040?oo00So1@04o`80Bol300?o100ho`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo00<0ool02?l203Ko000]o`80COl0103oo`09o`030?l000Go00<0ool0
BOl00`3oo`0:o`80=?l002co00<0ool00P1<o`802ol200Go00<0ool0B?l00`3oo`0=o`80<_l002co
00<0ool00_l00`3oo`1Mo`030?oo04Oo00<0ool04?l2033o000/o`030?oo00?o00<0ool09ol203?o
00<0ool0A_l00`3oo`0Co`80;_l002_o0P06o`809_l303?o00<0ool0AOl00`3oo`0Fo`80;?l002_o
00<0ool01ol00`3oo`0=o`T03Ol00`3o000co`030?oo04Co00<0ool06Ol00`3oo`0Yo`00;?l00`3o
o`07o`030?oo00_o00<0ool01ol400So00@0ool0<ol00`3oo`12o`807Ol202Wo000/o`030?oo00So
00<0ool02Ol00`3oo`0<o`801_l0103oo`0co`030?oo047o00<0ool07ol202Oo000/o`030?oo00Wo
0P08o`030?oo00oo0P03o`030?oo0080<ol00`3oo`10o`030?oo02;o0P0Uo`00;?l00`3oo`0;o`03
0?oo00Co00<0ool04_l703Co00<0ool0?ol00`3oo`0To`030?oo02?o000/o`030?oo00co00<0ool0
0ol00`3oo`0Do`<0=_l00`3oo`0no`030?oo02Ko00<0ool08_l002co00<0ool03Ol200;o0P0Fo`03
0?oo03Ko00<0ool0??l202[o00<0ool08Ol002co00<0ool03ol00`3o000Fo`030?oo03Oo00<0ool0
>ol00`3oo`0[o`030?oo023o000/o`030?oo00oo0`0Fo`030?oo02?o0P04o`800ol400Go00<0ool0
>_l00`3oo`0/o`030?oo023o000/o`030?oo02So00<0ool08_l0103oo`08o`030?oo00Ko00<0ool0
>?l202oo00<0ool08?l002co00<0ool09ol00`3oo`0So`040?oo00Wo00<0ool01Ol203So00<0ool0
;ol00`3oo`0Po`00;?l00`3oo`0Wo`030?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`0go`03
0?oo02oo00<0ool08?l002co00<0ool09ol00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`0fo`03
0?oo033o00<0ool08?l002co00<0ool09ol00`3oo`0To`802_l200Ko00<0ool0=Ol00`3oo`0`o`03
0?oo027o000/o`030?oo02Ko00<0ool0>Ol00`3oo`0eo`030?oo02oo0P0So`00;?l00`3oo`0Vo`03
0?oo03Wo00<0ool0=?l00`3oo`0_o`030?oo02?o000/o`030?oo02Ko00<0ool0>Ol00`3oo`0do`03
0?oo02go0P0Vo`001ol500;o0P04o`805_l00`3oo`0:o`D05_l00`3oo`09o`801?l200Co0P0So`03
0?oo02;o0P04o`801?l200Co00<0ool07?l500[o0P0Fo`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01Go00<0ool03?l00`3oo`0Fo`030?oo00So00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`03o`030?oo01ko00<0ool01_l401[o00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`03
0?oo01?o00<0ool03?l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol01@3oool0023o00<0ool00_l401ko00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`80
5_l00`3oo`04o`@01?l00`3oo`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol200Co00<0ool07_l00`3o000202;o00<0ool02?l200?o0009o`030?oo00So00<0ool05Ol00`3o
o`0<o`030?oo01Ko00<0ool02?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo00?o
00<0ool07_l202Go00<0ool02?l00`3oo`02o`002?l200[o0`0Eo`030?oo00_o0P0Ho`030?oo00Wo
0P0:o`<08_l00`3oo`0Ro`802_l300?o0P0No`<09?l200[o0`02o`00;?l00`3oo`0Uo`030?oo03[o
00<0ool0=Ol00`3oo`0No`030?oo03?o000/o`030?oo02Go00<0ool0>_l00`3oo`0eo`030?oo01ko
00<0ool0<ol002co00<0ool09Ol00`3oo`0jo`030?oo03Go00<0ool07_l00`3oo`0co`00;?l00`3o
o`0Uo`030?oo03[o00<0ool0=Ol00`3oo`0No`030?oo03?o000]o`030?oo02Co00<0ool0>_l00`3o
o`0do`030?oo01oo00<0ool0<ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`02o`D02?l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3o000300?o00<0ool01002o`03
0?oo00Ko00<0ool01_l00`3oo`06o`801ol00`3oo`05o`030?oo00Ko00<0ool01_l200Oo00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0No`030?oo00So00<0ool0
8Ol400Co00<0ool05Ol600oo00<0ool01_l00`3oo`0<o`804_l00`3oo`06o`030?oo023o00<0o`00
:_l00`3oo`07o`00;Ol00`3oo`0`o`030?oo01Ko0`05o`803_l00`3oo`05o`030?oo00oo00<0ool0
6?l00`3oo`0Ro`030?oo03;o000^o`030?oo033o0P0Do`802_l200co00<0ool01?l00`3oo`0Ao`80
5ol00`3oo`07o`H05_l00`3oo`0bo`00;_l00`3oo`0bo`030?oo00oo0P0=o`030?oo00[o00<0ool0
0_l201Ko00<0ool04ol00`3oo`07o`030?oo00Co0`0Co`030?oo03;o000_o`030?oo037o00<0ool0
3Ol200oo00<0ool02_l00`3oo`0201Wo00<0ool04Ol00`3oo`06o`802_l00`3oo`0@o`030?oo03;o
000_o`030?oo03;o00<0ool02ol00`3oo`0@o`030?oo00Wo0`0Lo`030?oo00oo00<0ool01_l00`3o
o`0;o`804?l00`3oo`0bo`00<?l00`3oo`0bo`030?oo00Wo00<0ool04_l00`3oo`07o`807ol200ko
00<0ool01ol00`3oo`0=o`030?oo00go00<0ool0<_l0037o00<0ool0<Ol00`3oo`08o`030?oo01?o
00<0ool01?l30003o`3o023o00<0ool02ol00`3oo`06o`030?oo00oo0P0=o`030?oo03;o000bo`03
0?oo023o2P07o`030?oo00Ko00<0ool05?l200?o0P04o`030?oo023o00<0ool02Ol00`3oo`06o`03
0?oo01;o0P0:o`80=?l003;o00<0ool07?l400[o00<0ool01?l00`3oo`05o`030?oo01Ko1006o`03
0?oo027o00<0ool01_l200Wo00<0ool05?l200So00<0ool0<ol003?o00<0ool06Ol200oo00<0ool0
1?l00`3oo`03o`030?oo027o00<0ool08_l200Co0P0:o`030?oo01Oo1@001?l0000eo`00=?l201Ko
0`0Bo`030?oo00Co00<0ool00P0To`030?oo02Co100<o`030?oo01_o0P0ho`00=_l00`3oo`0@o`<0
5_l00`3oo`03o`@09Ol00`3oo`0co`030?oo05Ko000go`803_l201Wo00<0ool01?l00`3oo`0Uo`03
0?oo03;o00<0ool0Eol003Wo00<0ool02Ol201co00<0ool05ol200Co0P03o`@01Ol00`3oo`0_o`<0
F_l003[o0P07o`807_l00`3oo`0Fo`040?oo00So00<0ool01_l00`3oo`0Ro`d0GOl003co1`0Qo`03
0?oo01Go00@0ool02Ol00`3oo`05o`808Ol306Wo001To`030?oo01Go00@0ool02_l00`3oo`04o`03
0?oo01oo00<0ool0J_l006Go00<0ool05?l0103oo`08o`040?oo00Go00<0ool07_l00`3oo`1[o`00
IOl00`3oo`0Eo`802_l200Ko00<0ool07_l00`3oo`1[o`00I_l00`3oo`0Xo`P06?l00`3oo`1/o`00
Iol00`3oo`0Wo`030?oo00Go0P0Fo`030?oo06co001Wo`030?oo02Ko0P09o`805?l00`3oo`1/o`00
J?l00`3oo`0So`8000?o0?l02_l00`3oo`0@o`030?oo06go001Yo`030?oo023o0P03o`030?oo00[o
0P0@o`030?oo06go001Zo`030?oo01go0P05o`030?oo00co0P0=o`030?oo06ko001[o`030?oo01Wo
0`07o`030?oo00ko0`09o`80L?l006co0P0Fo`<02_l00`3oo`0Ao`X0LOl006ko1P0:o`H03Ol00`3o
o`2<o`00M?l:01?o00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co
00@0ool02ol00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103o
o`05o`030?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3o
o`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00
oolQo`00oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{496, 481},
WindowMargins->{{Automatic, 16}, {Automatic, 25}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 140, 4, 330, "Title"],

Cell[CellGroupData[{
Cell[1904, 59, 32, 0, 53, "Section"],
Cell[1939, 61, 345, 9, 52, "Text"],
Cell[2287, 72, 320, 6, 52, "Text"],
Cell[2610, 80, 304, 6, 52, "Text"],
Cell[2917, 88, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3150, 97, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3332, 105, 48, 0, 45, "Subsection"],
Cell[3383, 107, 67, 1, 27, "Input"],
Cell[3453, 110, 103, 2, 27, "Input"],
Cell[3559, 114, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3724, 121, 46, 0, 29, "Subsection"],
Cell[3773, 123, 326, 7, 50, "Text"],
Cell[4102, 132, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4549, 144, 103, 2, 29, "Subsection"],
Cell[4655, 148, 105, 3, 32, "Text"],
Cell[4763, 153, 244, 6, 42, "Input"],
Cell[5010, 161, 123, 2, 27, "Input"],
Cell[5136, 165, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5419, 175, 78, 0, 42, "Subsubsection"],
Cell[5500, 177, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5670, 185, 60, 0, 29, "Subsection"],
Cell[5733, 187, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[5990, 195, 130, 3, 46, "Subsection"],
Cell[6123, 200, 121, 2, 27, "Input"],
Cell[6247, 204, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6428, 212, 31, 0, 33, "Section"],
Cell[6462, 214, 101, 3, 33, "Text"],
Cell[6566, 219, 202, 3, 30, "Input"],
Cell[6771, 224, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6962, 231, 44, 0, 33, "Section"],
Cell[7009, 233, 66, 0, 33, "Text"],
Cell[7078, 235, 1951, 30, 290, "Input"],

Cell[CellGroupData[{
Cell[9054, 269, 1955, 30, 290, "Input"],
Cell[11012, 301, 181, 3, 70, "Message"]
}, Open  ]],
Cell[11208, 307, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11309, 313, 37, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11371, 317, 80, 1, 30, "Input"],
Cell[11454, 320, 14554, 640, 152, 7748, 551, "GraphicsData", "PostScript", \
"Graphics"],
Cell[26011, 962, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26190, 971, 35, 0, 53, "Section"],
Cell[26228, 973, 176, 4, 52, "Text"],
Cell[26407, 979, 461, 7, 109, "Text"],

Cell[CellGroupData[{
Cell[26893, 990, 142, 3, 50, "Input"],

Cell[CellGroupData[{
Cell[27060, 997, 22874, 1117, 152, 15010, 1014, "GraphicsData", "PostScript", \
"Graphics"],
Cell[49937, 2116, 22573, 1108, 152, 14890, 1007, "GraphicsData", \
"PostScript", "Graphics"],
Cell[72513, 3226, 21784, 1066, 152, 14300, 968, "GraphicsData", "PostScript", \
"Graphics"],
Cell[94300, 4294, 21484, 1057, 152, 14182, 961, "GraphicsData", "PostScript", \
"Graphics"],
Cell[115787, 5353, 20207, 982, 152, 13014, 887, "GraphicsData", "PostScript", \
"Graphics"],
Cell[135997, 6337, 18486, 879, 152, 11431, 786, "GraphicsData", "PostScript", \
"Graphics"],
Cell[154486, 7218, 17837, 844, 152, 10903, 752, "GraphicsData", "PostScript", \
"Graphics"],
Cell[172326, 8064, 15849, 721, 152, 9001, 631, "GraphicsData", "PostScript", \
"Graphics"],
Cell[188178, 8787, 14613, 642, 152, 7748, 551, "GraphicsData", "PostScript", \
"Graphics"],
Cell[202794, 9431, 15362, 687, 152, 8465, 596, "GraphicsData", "PostScript", \
"Graphics"],
Cell[218159, 10120, 17633, 830, 152, 10683, 738, "GraphicsData", \
"PostScript", "Graphics"],
Cell[235795, 10952, 18541, 880, 152, 11458, 787, "GraphicsData", \
"PostScript", "Graphics"],
Cell[254339, 11834, 19387, 927, 152, 12121, 831, "GraphicsData", \
"PostScript", "Graphics"],
Cell[273729, 12763, 19965, 952, 152, 12497, 854, "GraphicsData", \
"PostScript", "Graphics"],
Cell[293697, 13717, 20739, 993, 152, 13064, 892, "GraphicsData", \
"PostScript", "Graphics"],
Cell[314439, 14712, 21767, 1048, 152, 13894, 945, "GraphicsData", \
"PostScript", "Graphics"],
Cell[336209, 15762, 22371, 1072, 152, 14263, 966, "GraphicsData", \
"PostScript", "Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

