(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    334335,      15532]*)
(*NotebookOutlinePosition[    334989,      15555]*)
(*  CellTagsIndexPosition[    334945,      15551]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Visualization of Proj.of Males, Ortho to Females, for Major Axis Rescaled, \
Schizo Population, on Sph Data scale,
\
\>", "Title"],

Cell[CellGroupData[{

Cell["Documentation", "Section"],

Cell[TextData[{
  "This ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook allows visualization of a direction in the feature space, using \
an animation of shapes at ",
  StyleBox["(mean + t*axis)", "Output"],
  ", for various values of t around zero.  As input, we require both a mean \
vector and an axis vector."
}], "Text"],

Cell[TextData[{
  "To use, simply paste in the mean and axis in the ",
  StyleBox["\"Input data: mean and axis\"", "Subsubsection"],
  " section, and evaluate the entire notebook (single left click on the \
rightmost blue bar, and shift+Enter) (also from the menus, \
Kernel->Evaluation->Evaluate Notebook)."
}], "Text"],

Cell[TextData[{
  "To adjust the range of the plot, adjust the values for \[Delta] and \
\[Alpha] in the ",
  StyleBox["\"Plot options\"", "Subsubsection"],
  " section, and evaluate all cells from there on (click and drag on a blue \
bar to highlight multiple cells, and press shift+Enter)."
}], "Text"],

Cell["\<\
A warning: You may need to adjust the scale of perturbations off \
mean (the range of t values); this is the third argument to the FourierAnim \
function in the final plot.\
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Setup subroutines",
  Evaluatable->False]], "Section",
  PageWidth->Infinity,
  ShowSpecialCharacters->False,
  FormatType->TextForm],

Cell[CellGroupData[{

Cell["Read in required libraries", "Subsection"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"];\)\)], "Input"],

Cell["Needs[\"Graphics`Animation`\"];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(\(Needs["\<LinearAlgebra`Orthogonalization`\>"];\)\(\ \)\( (*\ 
      for\ Normalize\ *) \)\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["plotFourierPt subroutine", "Subsection"],

Cell[TextData[{
  "Return the {x,y} coord corresponding to s (between 0 and 2Pi), using \
maxorder coefficients of c.\nAssumes c holds at least 4*maxorder real numbers \
in the order: ",
  Cell[BoxData[
      \(TraditionalForm\`Re(c\_1), \ Im(c\_1), \ Re(c\_\(-1\)), \ 
      Im(c\_\(-1\)), \ Re(c\_2), \  ... \)]]
}], "Text"],

Cell[BoxData[
    \(\(plotFourierPt[c_List, \ maxorder_, \ 
          s_]\  := \ \((\ \n\t\tcplxPt = \ 
            Sum[\ \((c[\([4  n - 3]\)]\  + \ c[\([4  n - 2]\)] I)\)\ Exp[
                    n\ I\ s]\  + \((c[\([4  n - 1]\)]\  + \ 
                      c[\([4  n]\)] I)\)\ Exp[\(-n\)\ I\ s], \ {n, 1, \ 
                maxorder}\ ]; \n\t\t{Re[cplxPt], \ 
            Im[cplxPt]}\ \ )\);\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["The main plotting routines", "Subsection",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
These functions return a Graphics object; use ShowPlot to actually \
display them\
\>", "Text"],

Cell["\<\
plotFourier[c_List, maxorder_] := ParametricPlot[Evaluate[plotFourierPt[c, \
maxorder, s]], {s, 0, 2Pi},
  Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["plotFourier[c_List] := plotFourier[c, Length[c]/4];", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell["\<\
multiplotFourier[c_List, maxorder_] := \
ParametricPlot[Evaluate[Map[plotFourierPt[#, maxorder, s]&, c]],
  {s, 0, 2Pi}, Evaluate[plotFourierOptions], DisplayFunction->Identity];\
\>", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[CellGroupData[{

Cell["This is just a macro to set the right DisplayFunction", "Subsubsection"],

Cell[BoxData[
    \(\(ShowPlot[g_Graphics]\  := \ 
        Show[g, \ DisplayFunction -> $DisplayFunction];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Animate along an axis through the mean", "Subsection"],

Cell[BoxData[
    \(FourierAnim[mymean_List, \ offset_List, \ length_\ , numframes_]\  := \ 
      Table[plotFourier[mymean\  + \ t\ offset], \ {t, \ \(-length\), \ 
          length, \ 2\ length/numframes}]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Subroutine to convert from matlab format (flat, w00z) to the format \
plotFourier likes (flat, cplx)\
\>", "Subsection"],

Cell["cvtw00z[w_,x_,y_,z_] := { w+z, y-x, w-z, y+x }/2;", "Input",
  PageWidth->Infinity,
  ShowSpecialCharacters->False],

Cell[BoxData[
    \(\(Flatw00zToFlatCplx[c_List]\  := \ 
        Flatten[Apply[cvtw00z, \ Partition[c, \ 4], \ {1}]];\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot options", "Section"],

Cell["\<\
After modifying plotrange, reevaluate both cells, then reevaluate \
the plot.\
\>", "Text"],

Cell[BoxData[
    \(\(plotrange\  = \ {\ {\(-\[Delta]\), \ \[Delta]}, \ {\(-\[Alpha]\)\ \
\[Delta], \ \[Alpha]\ \[Delta]}\ } /. \ {\[Delta] \[Rule] 1.5, \ \[Alpha] -> 
              1/2};\)\)], "Input"],

Cell[BoxData[
    \(\(plotFourierOptions\  = \ {PlotRange\  -> \ plotrange, \ 
          AspectRatio -> Automatic, \ PlotPoints\  -> \ 50};\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Input data: mean and axis", "Section"],

Cell["These come from Part 1 of the file Eigenvector.txt", "Text"],

Cell[BoxData[
    \(\(mean\  = \ {0.99999998988257, \(-0.00000000004249\), \
\(-0.00000000815207\), \(-0.14890885667999\), 
          0.04631615579661, \(-0.06867955424860\), \(-0.23594605970587\), 
          0.01859252456040, 0.02438649565581, 
          0.03413432177527, \(-0.00469090595776\), \(-0.08295111916888\), 
          0.00533664633423, \(-0.07524301148546\), \(-0.04136179149529\), \
\(-0.01210875268342\), 0.02309310511489, 
          0.01833272513018, \(-0.00846486752268\), \(-0.04392924956181\), \
\(-0.01098615476735\), \(-0.03598997698790\), \(-0.01058517671750\), \
\(-0.01401422746211\), 0.02257020063291, 0.00033072899653, 
          0.00023590085341, \(-0.00869156702959\), \(-0.00345348275585\), 
          0.00030597192151, \(-0.00137193187445\), \(-0.00667969066486\), 
          0.00904127017269, \(-0.00655340249989\), 0.00501447359205, 
          0.00258552073912, 0.00648801952449, 0.00823373129342, 
          0.00043275898306, \(-0.00298887850287\), \(-0.00332827922859\), \
\(-0.00044356465178\), 0.00196544293964, 0.00124098121236, 0.00592167003307, 
          0.00226352247340, \(-0.00014285700056\), \(-0.00058604947765\), \
\(-0.00428359723348\), 0.00503898752002, 0.00075457255761, 0.00107280526054, 
          0.00042732990809, \(-0.00194148383409\), \(-0.00026021270502\), \
\(-0.00015924393146\), \(-0.00055896473400\), 0.00357139004531, 
          0.00018521478115, 
          0.00014204963106, \(-0.00227702206002\), \(-0.00142480374493\), \
\(-0.00062619308861\), \(-0.00062940968098\), 
          0.00077624011619, \(-0.00019058121174\), 
          0.00041251634176, \(-0.00069431013873\), \(-0.00088148948557\), 
          0.00021433450379, \(-0.00037807112229\), \(-0.00026326849121\), 
          0.00019032930334, \(-0.00108451226204\), 
          0.00002689323115, \(-0.00072626836349\), 0.00058142750668, 
          0.00047562210024, 
          0.00043195050971, \(-0.00011112357174\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ {\(-0.00000182545561\), 
          0.00000003347761, \(-0.00000086426077\), \(-0.12641255131961\), 
          0.13843838679534, 
          0.02381068942784, \(-0.23568583349728\), \(-0.01676967114582\), 
          0.30521203706262, \(-0.16229214840010\), 
          0.20358038855350, \(-0.16564811681689\), 
          0.15946992862629, \(-0.21406000385948\), 
          0.19796013350408, \(-0.20832100220807\), 
          0.31852192484948, \(-0.03934593967932\), 
          0.29339901416011, \(-0.20545979268099\), 
          0.00542495941528, \(-0.09684499056066\), 
          0.11743111554211, \(-0.07300330316572\), 0.09458175524481, 
          0.01867053753123, 
          0.13505373465020, \(-0.17881895985230\), \(-0.06257497816274\), 
          0.01105680212600, 0.14490545033690, 
          0.01735532296091, \(-0.01161291094039\), 
          0.00172257356290, \(-0.05113430316693\), 
          0.04908153691813, \(-0.09766329325836\), 
          0.06296518684102, \(-0.06166989399147\), 
          0.00635759598599, \(-0.16544139374472\), 0.01195195396767, 
          0.05799382195157, 
          0.03689831095595, \(-0.02492765724809\), \(-0.01960707849731\), \
\(-0.18890329383082\), \(-0.13260297756412\), \(-0.08822859106046\), 
          0.01650947063666, \(-0.01824420297468\), 
          0.14369173720252, \(-0.05757151132515\), \(-0.03061972384856\), \
\(-0.01498725987633\), \(-0.03986241019106\), 0.03293442229801, 
          0.05624633371076, 0.16348419636895, 
          0.01463375680904, \(-0.03006945866230\), \(-0.03706035246328\), \
\(-0.06315473024030\), 0.06832286483973, 
          0.03815147468789, \(-0.04997569309160\), 
          0.02867175759177, \(-0.05716912610179\), \(-0.07116054165142\), 
          0.02062888640725, \(-0.02694813919728\), 0.04996998613338, 
          0.00490397820756, \(-0.07296354201024\), \(-0.08090430803446\), \
\(-0.03329873875197\), 0.05080425267299, 0.04179764195603, 
          0.02843368403103, \(-0.02064339591632\)};\)\)], "Input"],

Cell[BoxData[
    \(\(axis\  = \ Normalize[axis];\)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot the L1 median", "Section"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowPlot[plotFourier[Flatw00zToFlatCplx[mean]]]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87312 .15188 m
.86926 .14386 L
.86393 .13584 L
.8608 .13219 L
.85712 .12864 L
.84968 .12322 L
.845 .12076 L
.8396 .11859 L
.83702 .11778 L
.83423 .11705 L
.83177 .11652 L
.82908 .11605 L
.82645 .1157 L
.82402 .11547 L
.82284 .11538 L
.82155 .11532 L
.82034 .11527 L
.8192 .11525 L
.81851 .11524 L
.81776 .11525 L
.8164 .11527 L
.81508 .11531 L
.81386 .11537 L
.81254 .11546 L
.8111 .11558 L
.8082 .11591 L
.80561 .1163 L
.80282 .11683 L
.79788 .11805 L
.79326 .11957 L
.78919 .12126 L
.78093 .12614 L
.77732 .12914 L
.77384 .13287 L
.77123 .13646 L
.7689 .14056 L
.76722 .1443 L
.7657 .14863 L
.76437 .15353 L
.76322 .15898 L
.76158 .16914 L
.75992 .17956 L
.75875 .18531 L
.75741 .19047 L
.75414 .19945 L
.75203 .2037 L
.74934 .20814 L
.74329 .21582 L
.73678 .22219 L
.71999 .23483 L
Mistroke
.70239 .24567 L
.6858 .25422 L
.66896 .2608 L
.64987 .26611 L
.60818 .27454 L
.58735 .27803 L
.56776 .28057 L
.54812 .2825 L
.53806 .28333 L
.52734 .28411 L
.52185 .28446 L
.51668 .28474 L
.51167 .28497 L
.50947 .28505 L
.50705 .28513 L
.50495 .28519 L
.50268 .28523 L
.50144 .28525 L
.5003 .28527 L
.49921 .28527 L
.49806 .28528 L
.49689 .28528 L
.49564 .28528 L
.49446 .28527 L
.49339 .28527 L
.49213 .28525 L
.49076 .28523 L
.48828 .28517 L
.48586 .2851 L
.48325 .285 L
.47853 .28477 L
.46777 .28402 L
.44607 .2819 L
.4262 .27964 L
.40523 .2768 L
.38512 .27297 L
.36631 .26808 L
.3297 .25555 L
.31263 .24816 L
.29698 .23985 L
.28116 .22874 L
.2744 .22208 L
.27171 .21859 L
.27053 .21672 L
.26945 .21468 L
.26863 .21275 L
.26834 .2119 L
.26806 .21096 L
.26786 .21013 L
.26769 .20924 L
Mistroke
.26762 .20875 L
.26756 .2083 L
.26748 .20743 L
.26745 .20658 L
.26746 .2058 L
.26751 .20495 L
.26762 .20404 L
.26776 .20317 L
.26794 .20236 L
.26838 .20084 L
.26907 .19905 L
.27 .19718 L
.27214 .19385 L
.27885 .18647 L
.29392 .17344 L
.29962 .16794 L
.30201 .16509 L
.30396 .16214 L
.30469 .16066 L
.30501 .15988 L
.30529 .15906 L
.30549 .15831 L
.30562 .15763 L
.30571 .15698 L
.30576 .15631 L
.30576 .15568 L
.30572 .15502 L
.30563 .1544 L
.30552 .15384 L
.30534 .1532 L
.30509 .15251 L
.30449 .15129 L
.30367 .15006 L
.30255 .14876 L
.30124 .14756 L
.29961 .14634 L
.29616 .14435 L
.29239 .14272 L
.2878 .14123 L
.28319 .14008 L
.27784 .1391 L
.27475 .13867 L
.27191 .13835 L
.26935 .13812 L
.26662 .13792 L
.26519 .13784 L
.26363 .13776 L
.26218 .13771 L
.26087 .13767 L
.25953 .13764 L
Mistroke
.25827 .13762 L
.25694 .13761 L
.25618 .1376 L
.25549 .1376 L
.25434 .13761 L
.25307 .13762 L
.25175 .13765 L
.25051 .13767 L
.24781 .13775 L
.24532 .13785 L
.23971 .13815 L
.22968 .13889 L
.21115 .14075 L
.20062 .14216 L
.19069 .14388 L
.18184 .1459 L
.17247 .14874 L
.16362 .15233 L
.15453 .15729 L
.146 .16365 L
.13916 .17065 L
.13627 .17445 L
.13351 .17882 L
.13144 .18286 L
.12962 .18733 L
.12834 .19147 L
.12776 .19387 L
.12732 .19616 L
.127 .19827 L
.12678 .20022 L
.12669 .20125 L
.12662 .20237 L
.12656 .20355 L
.12653 .20466 L
.12653 .20589 L
.12653 .20651 L
.12655 .20719 L
.1266 .2084 L
.12666 .20954 L
.12677 .21088 L
.12689 .2121 L
.12726 .21485 L
.12767 .21722 L
.12822 .21979 L
.12947 .22445 L
.13289 .23353 L
.137 .24162 L
.14283 .25064 L
.14909 .25849 L
.16235 .27131 L
Mistroke
.17861 .28254 L
.19507 .29081 L
.23253 .30432 L
.26816 .31585 L
.30666 .32791 L
.34417 .33806 L
.36346 .34243 L
.3738 .34442 L
.38492 .34623 L
.39491 .34757 L
.406 .34873 L
.41614 .34954 L
.42568 .35011 L
.43583 .35055 L
.44095 .35073 L
.44655 .35088 L
.45136 .35099 L
.45651 .35108 L
.45902 .35111 L
.46138 .35114 L
.46352 .35116 L
.46582 .35117 L
.46713 .35118 L
.46833 .35119 L
.46953 .35119 L
.47067 .35119 L
.472 .35119 L
.47323 .35119 L
.47458 .35119 L
.47527 .35118 L
.47602 .35118 L
.47742 .35117 L
.4787 .35116 L
.4816 .35113 L
.48415 .3511 L
.48684 .35105 L
.49164 .35094 L
.4961 .35081 L
.5062 .35039 L
.51145 .3501 L
.51713 .34973 L
.52738 .34888 L
.54641 .34669 L
.55601 .34525 L
.56652 .34337 L
.58752 .33863 L
.62628 .32655 L
.66302 .31448 L
.7036 .30483 L
.74334 .29758 L
Mistroke
.78115 .2896 L
.7999 .28436 L
.81898 .27708 L
.8279 .27256 L
.83711 .26679 L
.84463 .26104 L
.85227 .25392 L
.86452 .2386 L
.86978 .22951 L
.87371 .22075 L
.87528 .21636 L
.87673 .21153 L
.87778 .20723 L
.87866 .20262 L
.87925 .19843 L
.8795 .19603 L
.87959 .19493 L
.87968 .19376 L
.87974 .19277 L
.87979 .19168 L
.87983 .19068 L
.87986 .18975 L
.87988 .18865 L
.87989 .18765 L
.87988 .18649 L
.87987 .18539 L
.87984 .18439 L
.8798 .18332 L
.8797 .18137 L
.87953 .17905 L
.87932 .17692 L
.87871 .17228 L
.87792 .16786 L
.876 .16012 L
.87312 .15188 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 1}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l803Ko001lo`040?oo00So1@04o`80C?l300Oo0`0do`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0=o`<0<Ol007co00@0ool02Ol00`3o0005o`030?oo04Oo0P0Ao`03
0?oo02oo001mo`802ol200Go00<0ool0A_l00`3oo`0Bo`030?oo02ko002Ao`030?oo04Go00<0ool0
5?l00`3oo`0]o`00Aol@03[o00<0ool0AOl00`3oo`0Eo`030?oo02co0010o`L04?l303Oo00<0ool0
A?l00`3oo`0Go`030?oo02_o000ko`D06_l203Go00<0ool0A?l00`3oo`0Ho`030?oo02[o000io`80
8Ol00`3oo`0bo`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=ol202?o0P0co`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol202Ko00<0ool0<Ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o0P0Xo`03
0?oo037o00<0ool0@ol00`3oo`0Ko`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`03
0?oo01co00<0ool0:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02Oo00<0ool0
8?l200Co0P03o`@01Ol00`3oo`12o`030?oo01co00<0ool0:?l002oo00<0ool09_l202;o00@0ool0
2?l00`3oo`06o`030?oo04;o00<0ool07?l00`3oo`0Xo`00;_l00`3oo`0Vo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0No`030?oo02Oo000^o`030?oo02Go00<0ool07Ol400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`030?oo02Co00@0ool0
2?l0103oo`05o`030?oo047o00<0ool07_l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo02Ko0P0:o`80
1_l00`3oo`11o`030?oo01go0P0Yo`00;Ol00`3oo`0So`030?oo03_o00<0ool0@?l00`3oo`0No`03
0?oo02So000]o`030?oo02?o00<0ool0>ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool0
8ol00`3oo`0ko`030?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`805ol00`3oo`09o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool0
2Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`08o`03
0?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3o
o`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3o
o`08o`030?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Ko
0P09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00So
00<0ool06Ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201[o00<0ool01ol200[o
0`0Ro`030?oo02;o0P0:o`<02Ol00`3oo`0Go`802Ol00`3oo`0Io`802_l300;o000_o`030?oo02Go
0P0ho`030?oo03So0P0Uo`030?oo02[o000_o`030?oo02Oo00<0ool0=Ol00`3oo`0go`030?oo02Co
00<0ool0:ol0033o00<0ool09ol00`3oo`0do`030?oo03Go0P0Vo`030?oo02co000`o`030?oo02So
0P0do`030?oo03?o0P0Wo`030?oo02go000ao`030?oo02Wo0P0bo`030?oo03;o00<0ool09ol00`3o
o`0]o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00@0oooo
0`02o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02Wo0`0Zo`03
0?oo02Wo00@0o`00:?l01@3oool002So00<0ool01ol003Co00<0ool0<?l302Oo00<0ool09ol402co
0P0co`00=Ol00`3oo`0bo`809Ol00`3oo`0So`@0;ol00`3oo`0co`00=_l00`3oo`0co`@08Ol00`3o
o`0Oo`@0<_l00`3oo`0do`00=ol00`3oo`0fo`D07?l00`3oo`0Ko`@0=?l203Oo000ho`030?oo03[o
1P0Fo`030?oo01;o2@0eo`<0>Ol003Wo0P10o`/02ol00`3oo`0=o`D0??l203co000ko`030?oo04So
6`0no`<0?_l003co0P1Co`030?oo04Wo0P11o`00?_l3053o00<0ool0AOl404?o0011o`<0COl00`3o
o`10o`D0Aol004Co1019o`030?oo03_o1@1<o`00B?l304Ko00<0ool0=_l5057o001;o`80A?l00`3o
o`0ao`D0E_l004go0`0]o`801?l200?o1005o`030?oo02_o1P1Ko`00D?l202[o00@0ool02?l00`3o
o`06o`030?oo02Oo101Qo`00D_l302Oo00@0ool02Ol00`3oo`05o`809?l406Go001Eo`<09?l0103o
o`0:o`030?oo00Co00<0ool08?l306Wo001Ho`@08?l0103oo`08o`040?oo00Go00<0ool07Ol306co
001Lo`@07Ol200[o0P06o`030?oo01_o0P1_o`00H?l502co00<0ool06?l3077o001Uo`@0:?l00`3o
o`0Fo`80M?l006Wo0P0Vo`030?oo017o1@1fo`00Jol6023o00<0ool02_l707_o001ao`/05Ol00`3o
o`03o`L0P_l007co6`29o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Animate the axis", "Section"],

Cell["\<\
The arguments to FourierAnim are the axis along which to move, the \
distance to move along that axis (+ and -), and the number of frames to \
generate.\
\>", "Text"],

Cell["\<\
Evaluating this cell generates many frames of output.  To view the animation, \
double-click on the second blue bar (from the left) in the output; this will \
shrink all the frames of output to show just the first frame.  Now \
double-click on the graphic; this should start playing the animation.  \
Animation controls should appear at the left of the bottom status bar; the \
second button makes the animation loop \"ping-pong\" style.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ShowAnimation[
      FourierAnim[Flatw00zToFlatCplx[mean], \ Flatw00zToFlatCplx[axis], \ 
        0.1, \ 16]]\)], "Input"],

Cell[CellGroupData[{

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85515 .1309 m
.85479 .12963 L
.85428 .12829 L
.85366 .12706 L
.85297 .12597 L
.85203 .12473 L
.85082 .12344 L
.84815 .12127 L
.84659 .12029 L
.84469 .11931 L
.84268 .11845 L
.84034 .11765 L
.83799 .11703 L
.83572 .11658 L
.83459 .11641 L
.83333 .11625 L
.83211 .11613 L
.83096 .11605 L
.83024 .11601 L
.82947 .11599 L
.82808 .11597 L
.82729 .11598 L
.82655 .116 L
.82573 .11603 L
.82485 .11608 L
.82408 .11613 L
.82325 .1162 L
.82172 .11637 L
.81882 .11681 L
.81594 .1174 L
.81288 .1182 L
.80746 .12008 L
.80153 .12291 L
.79639 .12617 L
.79143 .13033 L
.78918 .13271 L
.78705 .13538 L
.78549 .13774 L
.78427 .13994 L
.78323 .14222 L
.78245 .14436 L
.78177 .14689 L
.7815 .14823 L
.78139 .14892 L
.78129 .14965 L
.78116 .15089 L
.78109 .15205 L
.78107 .15276 L
.78107 .15343 L
.7811 .15469 L
Mistroke
.78114 .15536 L
.78118 .156 L
.78131 .15719 L
.78149 .15846 L
.78174 .15983 L
.78232 .1623 L
.78299 .16454 L
.78492 .16962 L
.78738 .17488 L
.79181 .18358 L
.79404 .18837 L
.79582 .19295 L
.79708 .19729 L
.7976 .19977 L
.79779 .20097 L
.79794 .20209 L
.79804 .20304 L
.79812 .20406 L
.79818 .20519 L
.7982 .20624 L
.7982 .20727 L
.79817 .2084 L
.7981 .20958 L
.79801 .21068 L
.79786 .21195 L
.79766 .21329 L
.79719 .21568 L
.79653 .21824 L
.79563 .22101 L
.79358 .22581 L
.7912 .22999 L
.7885 .23369 L
.78575 .23665 L
.78413 .23809 L
.78229 .23948 L
.77875 .24151 L
.7769 .24226 L
.77591 .24259 L
.77485 .24289 L
.7739 .2431 L
.77285 .24329 L
.77189 .24342 L
.77098 .24351 L
.77041 .24355 L
.76979 .24357 L
.76865 .24358 L
.76752 .24354 L
.76647 .24347 L
.76532 .24335 L
.76403 .24316 L
Mistroke
.76137 .24266 L
.7561 .24124 L
.7501 .23924 L
.7384 .23524 L
.73241 .23355 L
.72905 .23279 L
.72585 .23221 L
.72401 .23195 L
.72201 .23173 L
.72113 .23166 L
.7202 .23159 L
.71852 .23151 L
.71684 .23148 L
.71527 .2315 L
.71437 .23153 L
.71355 .23157 L
.71261 .23164 L
.71172 .23172 L
.71025 .23188 L
.70864 .23212 L
.7054 .23275 L
.70244 .23354 L
.69971 .23444 L
.69376 .23703 L
.68772 .24058 L
.6781 .24819 L
.66907 .25706 L
.66464 .26149 L
.66059 .26512 L
.65696 .26778 L
.65504 .26891 L
.65289 .26996 L
.6508 .27074 L
.64878 .27131 L
.64764 .27155 L
.64657 .27173 L
.64597 .27182 L
.64531 .27189 L
.64468 .27196 L
.64409 .272 L
.64291 .27207 L
.64183 .27209 L
.64064 .27209 L
.63932 .27205 L
.63814 .27199 L
.63687 .2719 L
.63454 .27167 L
.62863 .27088 L
.62274 .26994 L
.61598 .26886 L
Mistroke
.6123 .26832 L
.60834 .26781 L
.6048 .26743 L
.60292 .26726 L
.6009 .2671 L
.59904 .26698 L
.59735 .26689 L
.59565 .26683 L
.59405 .26678 L
.59244 .26676 L
.59074 .26677 L
.58915 .26679 L
.58772 .26683 L
.58604 .2669 L
.58425 .26701 L
.58104 .26727 L
.57825 .26757 L
.57526 .26798 L
.56948 .269 L
.55998 .27123 L
.55035 .27373 L
.54584 .27472 L
.54168 .27541 L
.53934 .27569 L
.53815 .2758 L
.53683 .2759 L
.5355 .27598 L
.53474 .27601 L
.53404 .27603 L
.53343 .27604 L
.53275 .27605 L
.53203 .27605 L
.53137 .27604 L
.53008 .27602 L
.52886 .27598 L
.52775 .27593 L
.52653 .27585 L
.52374 .27564 L
.5211 .27539 L
.47736 .26963 L
.46859 .2688 L
.46177 .26811 L
.45073 .26652 L
.44776 .26612 L
.44603 .26592 L
.44431 .26576 L
.44329 .26568 L
.44215 .26561 L
.44108 .26555 L
.44007 .26552 L
Mistroke
.43909 .26549 L
.43798 .26548 L
.43679 .26549 L
.43565 .26551 L
.4344 .26555 L
.43367 .26559 L
.43299 .26562 L
.43034 .26582 L
.42783 .26608 L
.42505 .26644 L
.419 .26751 L
.41198 .26916 L
.38608 .27814 L
.3764 .28173 L
.374 .28246 L
.37157 .28308 L
.37043 .28331 L
.36937 .28349 L
.36845 .28361 L
.36747 .28369 L
.36652 .28373 L
.36565 .28373 L
.36471 .28369 L
.36372 .28358 L
.36323 .28351 L
.3627 .28341 L
.36174 .28319 L
.35997 .28262 L
.35832 .28189 L
.35657 .28092 L
.35339 .27868 L
.33636 .26253 L
.32633 .25391 L
.31462 .24597 L
.30925 .24312 L
.30382 .24073 L
.29524 .23792 L
.28833 .23632 L
.2869 .23594 L
.2863 .23575 L
.28579 .23556 L
.28499 .23517 L
.28464 .23493 L
.28433 .23465 L
.2841 .23438 L
.28392 .2341 L
.28376 .23375 L
.28364 .23339 L
.28356 .23302 L
.28352 .23265 L
Mistroke
.2835 .23221 L
.28351 .23169 L
.28355 .23108 L
.28359 .23071 L
.28363 .23035 L
.28387 .22885 L
.28453 .2254 L
.28538 .22092 L
.28586 .21805 L
.28629 .21508 L
.28665 .21212 L
.28698 .20868 L
.2871 .2071 L
.2872 .20538 L
.28728 .20389 L
.28734 .20221 L
.28739 .20017 L
.28741 .19913 L
.28741 .19799 L
.2874 .19593 L
.28736 .194 L
.28732 .19282 L
.28727 .1917 L
.28716 .18959 L
.287 .18732 L
.28678 .18486 L
.28631 .18069 L
.28569 .17635 L
.28426 .16897 L
.28249 .16216 L
.28072 .15689 L
.27868 .15216 L
.27779 .15053 L
.27692 .14918 L
.27653 .14868 L
.27618 .1483 L
.27587 .14801 L
.2756 .14782 L
.27536 .1477 L
.27513 .14764 L
.27493 .14764 L
.27473 .14772 L
.27455 .1479 L
.27442 .14815 L
.2743 .14852 L
.27424 .14896 L
.27421 .14944 L
.27422 .14994 L
.27425 .15055 L
.27434 .15127 L
Mistroke
.27445 .15203 L
.27462 .15293 L
.27503 .15472 L
.27616 .15852 L
.27989 .16792 L
.28367 .17555 L
.28606 .17986 L
.28634 .18038 L
.2864 .18054 L
.28642 .18059 L
.28642 .18062 L
.2864 .18062 L
.28636 .18061 L
.28631 .18056 L
.28623 .18049 L
.28578 .18001 L
.28179 .17557 L
.27334 .16696 L
.2608 .15591 L
.24842 .14694 L
.24189 .14309 L
.23856 .14142 L
.23518 .13995 L
.23242 .13895 L
.23101 .13852 L
.22954 .13813 L
.22822 .13785 L
.22706 .13764 L
.22592 .13749 L
.22487 .1374 L
.22383 .13735 L
.22277 .13736 L
.22189 .13742 L
.22095 .13752 L
.21997 .13768 L
.21895 .13792 L
.21803 .1382 L
.21722 .13851 L
.21542 .13938 L
.21391 .14037 L
.21089 .14316 L
.20843 .14635 L
.2063 .14975 L
.20151 .15823 L
.19859 .16293 L
.1955 .16717 L
.18859 .17434 L
.18433 .17765 L
.1794 .18079 L
.16169 .18876 L
Mistroke
.15345 .19177 L
.1465 .19463 L
.14373 .19611 L
.1425 .19692 L
.14134 .19783 L
.14044 .19869 L
.13973 .19951 L
.13915 .20033 L
.13862 .20125 L
.13819 .2022 L
.13787 .20311 L
.13759 .20408 L
.13738 .20503 L
.13717 .2062 L
.137 .20755 L
.13687 .20881 L
.13677 .21023 L
.1367 .21168 L
.13667 .21253 L
.13664 .21332 L
.13662 .21409 L
.13661 .21494 L
.1366 .21576 L
.13659 .21655 L
.13659 .21704 L
.13659 .21758 L
.1366 .21857 L
.13661 .21956 L
.13663 .22049 L
.13665 .22153 L
.13669 .22269 L
.13679 .22513 L
.13693 .2274 L
.13712 .22996 L
.13763 .23486 L
.13837 .23989 L
.13929 .2447 L
.1406 .25002 L
.14242 .25593 L
.14476 .26201 L
.14741 .26763 L
.15356 .27768 L
.16016 .28555 L
.16794 .29232 L
.17255 .29532 L
.17691 .29758 L
.1812 .29932 L
.18348 .30006 L
.18592 .30073 L
.18807 .30122 L
Mistroke
.19045 .30165 L
.19265 .30197 L
.19472 .30219 L
.19577 .30228 L
.19692 .30236 L
.19801 .30242 L
.19903 .30247 L
.19995 .30249 L
.20097 .30251 L
.20205 .30252 L
.20306 .30251 L
.20419 .3025 L
.20538 .30247 L
.20641 .30243 L
.20755 .30238 L
.21011 .30223 L
.21253 .30204 L
.2325 .2997 L
.24347 .29837 L
.24614 .29811 L
.24899 .29787 L
.25062 .29776 L
.2521 .29767 L
.25359 .29761 L
.25499 .29756 L
.25575 .29755 L
.25644 .29754 L
.25723 .29753 L
.25797 .29753 L
.25929 .29755 L
.26003 .29757 L
.2607 .2976 L
.26213 .29766 L
.26293 .29772 L
.26368 .29777 L
.26527 .29792 L
.26677 .2981 L
.26941 .2985 L
.27179 .29898 L
.27702 .30045 L
.28183 .30236 L
.28619 .30458 L
.29584 .31111 L
.31471 .32721 L
.32453 .33474 L
.32899 .33756 L
.33398 .34019 L
.33894 .3422 L
.34135 .34296 L
.34364 .34355 L
Mistroke
.34491 .34382 L
.34628 .34407 L
.34744 .34425 L
.34871 .34441 L
.35009 .34454 L
.35157 .34463 L
.35294 .34467 L
.35422 .34467 L
.35563 .34463 L
.35717 .34454 L
.35863 .34441 L
.35999 .34426 L
.36258 .34387 L
.36535 .34333 L
.37115 .34183 L
.37766 .33966 L
.40372 .32911 L
.41498 .32519 L
.42118 .32354 L
.42408 .32292 L
.42678 .32246 L
.42934 .32211 L
.43077 .32196 L
.43207 .32185 L
.43315 .32178 L
.43433 .32173 L
.43555 .3217 L
.43668 .3217 L
.43798 .32173 L
.43919 .32179 L
.44027 .32186 L
.44143 .32197 L
.44267 .32211 L
.44397 .3223 L
.44625 .32272 L
.4486 .32327 L
.45106 .32399 L
.45532 .32557 L
.46275 .32928 L
.47809 .33819 L
.48283 .34049 L
.48838 .34273 L
.49412 .34455 L
.4998 .34592 L
.50277 .34648 L
.50602 .34698 L
.50888 .34733 L
.51212 .34762 L
.51396 .34775 L
.51566 .34783 L
Mistroke
.51656 .34787 L
.51755 .3479 L
.51859 .34792 L
.51956 .34793 L
.52137 .34793 L
.52302 .34791 L
.52394 .34789 L
.52494 .34785 L
.52676 .34777 L
.52845 .34766 L
.53005 .34754 L
.53367 .34719 L
.53719 .34673 L
.54094 .34613 L
.54761 .34472 L
.55348 .34312 L
.56564 .33856 L
.57539 .33352 L
.58506 .32739 L
.59309 .3222 L
.59683 .32016 L
.60086 .31841 L
.60305 .31769 L
.60552 .31705 L
.60664 .31683 L
.60785 .31662 L
.60902 .31647 L
.61011 .31635 L
.61121 .31626 L
.6124 .3162 L
.61361 .31617 L
.61431 .31617 L
.61497 .31617 L
.61619 .3162 L
.61732 .31626 L
.61864 .31634 L
.61993 .31645 L
.62234 .31671 L
.62465 .31703 L
.6301 .31796 L
.6541 .3233 L
.66073 .32465 L
.66428 .32527 L
.66753 .32576 L
.67047 .32612 L
.67308 .32637 L
.67454 .32647 L
.67591 .32654 L
.67733 .32659 L
.67812 .3266 L
Mistroke
.67885 .3266 L
.68017 .32659 L
.68158 .32654 L
.68288 .32646 L
.68406 .32636 L
.68533 .32622 L
.6867 .32603 L
.68938 .32553 L
.69192 .3249 L
.69452 .32407 L
.69897 .32222 L
.70719 .31757 L
.71459 .31296 L
.7185 .31077 L
.72287 .30871 L
.72683 .30724 L
.72916 .30655 L
.73144 .306 L
.73416 .30548 L
.73671 .30514 L
.73817 .305 L
.73954 .3049 L
.74034 .30486 L
.74106 .30483 L
.74191 .30481 L
.74271 .3048 L
.74407 .3048 L
.74552 .30484 L
.74677 .30489 L
.74815 .30498 L
.74965 .3051 L
.75127 .30526 L
.75422 .30563 L
.76049 .30668 L
.7663 .30792 L
.77914 .31102 L
.78467 .31227 L
.78959 .31319 L
.79196 .31353 L
.79315 .31366 L
.79442 .31378 L
.79509 .31382 L
.79571 .31386 L
.79636 .31388 L
.79704 .3139 L
.79767 .3139 L
.79826 .3139 L
.79934 .31386 L
.80033 .31379 L
.80137 .31368 L
Mistroke
.80233 .31354 L
.80318 .31338 L
.80416 .31316 L
.8052 .31287 L
.80702 .31222 L
.80881 .31136 L
.8106 .31024 L
.81205 .30911 L
.81351 .30771 L
.81585 .30482 L
.81778 .30163 L
.81974 .29728 L
.82126 .2928 L
.82253 .28784 L
.82361 .28215 L
.8244 .27649 L
.82494 .27116 L
.8254 .26473 L
.82559 .26104 L
.82573 .25762 L
.82592 .25131 L
.82607 .24461 L
.82621 .23839 L
.82637 .23252 L
.82649 .22931 L
.82665 .22583 L
.82703 .2198 L
.82753 .21417 L
.82825 .20827 L
.82911 .20282 L
.83005 .198 L
.83295 .18693 L
.83632 .17746 L
.84531 .15896 L
.84943 .15155 L
.85142 .14777 L
.85303 .14434 L
.85422 .14126 L
.85471 .13968 L
.85511 .13802 L
.85523 .13735 L
.85534 .13664 L
.85543 .13587 L
.85549 .13515 L
.85551 .13448 L
.85551 .13379 L
.85548 .1332 L
.85542 .13256 L
.85537 .13214 L
.8553 .1317 L
Mistroke
.85515 .1309 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 1}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0D_l503Go001lo`040?oo00So1@04o`80Col400Co100bo`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04go00<0ool02Ol2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o`80
3Ol2033o001mo`802ol200Go00<0ool0B_l00`3oo`0>o`030?oo02ko002Ao`030?oo04Wo00<0ool0
3ol00`3oo`0^o`00AOl504Oo00<0ool0B?l2017o00<0ool0;_l004Co0P04o`80AOl00`3oo`18o`03
0?oo013o00<0ool0;_l004?o0P07o`80@ol00`3oo`18o`030?oo00oo00<0ool0;ol004;o00<0ool0
2Ol00`3oo`03o`80>ol00`3oo`17o`030?oo013o00<0ool0;ol004;o00<0ool02_l200?o0`0jo`03
0?oo04Oo00<0ool03ol00`3oo`0`o`00@Ol00`3oo`0=o`040?ooo`80>_l00`3oo`17o`030?oo00oo
00<0ool0<?l0047o00<0ool03_l00`3oo`0203[o00<0ool0Aol00`3oo`0>o`030?oo037o0010o`03
0?oo013o00<0ool00P0io`030?oo04So00<0ool03Ol00`3oo`0ao`00@?l00`3oo`0Ao`040?l003Wo
00<0ool0B?l00`3oo`0<o`030?oo03;o000oo`030?oo01?o00<0o`00>Ol00`3oo`18o`030?oo00co
00<0ool0<_l003ko00<0ool05Ol202Go0P04o`800ol400Go00<0ool0BOl00`3oo`0:o`030?oo03?o
000mo`030?oo01Oo0P0So`040?oo00So00<0ool01_l00`3oo`1:o`030?oo00Wo00<0ool0<ol003_o
0P0Ko`030?oo027o00@0ool02Ol00`3oo`05o`80Bol00`3oo`09o`030?oo03?o000io`807Ol00`3o
o`0Ko`@00_l0103oo`0:o`030?oo00Co00<0ool0Bol00`3oo`07o`030?oo03Co000fo`<07ol00`3o
o`0Qo`040?oo00So00@0ool01Ol00`3oo`1;o`030?oo00Oo00<0ool0=?l003;o100Ro`030?oo02;o
0P0:o`801_l00`3oo`1;o`030?oo00Ko00<0ool0=Ol0037o0P0Uo`030?oo03Ko00<0ool0C?l00`3o
o`05o`030?oo03Go000`o`809_l00`3oo`0fo`030?oo04co00<0ool01Ol00`3oo`0eo`00<?l00`3o
o`0Uo`030?oo03Ko00<0ool0C?l00`3oo`05o`030?oo03Go0007o`D00_l200Co0P0Jo`030?oo00Ko
1@0Jo`030?oo00Go0P04o`801?l202?o00<0ool08_l200Co0P04o`807?l00`3oo`04o`D08_l500;o
0P04o`800ol000Wo00<0ool01ol0103oo`0Io`030?oo00So00<0ool06_l00`3oo`04o`040?oo00So
00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool06ol00`3oo`05o`809_l00`3oo`07o`040?oo00;o
0009o`030?oo00[o00<0ool05ol00`3oo`08o`030?oo01[o00<0ool01?l0103oo`0;o`030?oo023o
00<0ool08Ol0103oo`0;o`030?oo01Wo00<0ool01Ol202Ko00<0ool02_l1007o0Ol00003o`000080
1?l00`3oo`08o`806_l00`3oo`0400Co00<0ool06Ol00`3o000300;o00@0ool02Ol202?o00<0ool0
8Ol0103oo`09o`806ol00`3oo`05o`030?l002Ko00<0ool02?l200?o0009o`030?oo00So00<0ool0
6Ol00`3oo`08o`030?oo01Wo00<0ool01Ol0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo01[o00<0ool01Ol00`3o000Vo`030?oo00So00<0ool00_l000So0P0:o`<06Ol00`3oo`07o`80
6ol00`3oo`06o`802_l302;o00<0ool08_l200[o0`0Jo`030?oo00Go0`0Uo`802_l300;o000`o`03
0?oo02Co00<0ool0=ol00`3oo`0co`L04?l00`3oo`06o`030?oo03Ko000`o`030?oo02Co0`0go`03
0?oo03;o00<0ool01Ol400_o00<0ool01ol00`3oo`0fo`00<?l00`3oo`0Wo`80=Ol00`3oo`0^o`@0
3?l300Oo00<0ool02?l00`3oo`0fo`00<Ol00`3oo`0Xo`<0<_l00`3oo`0]o`030?oo017o1`0;o`03
0?oo03Ko000ao`030?oo02_o00<0ool0;ol00`3oo`0]o`030?oo02?o00<0ool0=_l000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Co00<0o`002?l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l300Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08_l00`3oo`04o`030?oo02So0P0/o`030?oo02Wo00<0ool09ol00`3o
000/o`030?oo00Oo000bo`030?oo033o00<0ool0:Ol00`3oo`0Xo`030?oo02So00<0ool0=_l003?o
00<0ool0<?l00`3oo`0Do`P03?l00`3oo`0Co`@04?l00`3oo`0Yo`030?oo03Ko000co`030?oo037o
00<0ool03_l500So2@03o`030?oo00ko1P03o`P01_l202co00<0ool0=_l003Co00<0ool0<Ol00`3o
o`09o`@05_l600[o100Ao`L0;Ol00`3oo`0fo`00=?l00`3oo`0bo`030?oo00Ko0P0Mo`030?oo00X0
BOl00`3oo`0fo`00=Ol00`3oo`0bo`801?l201oo00<0ool0Dol00`3oo`0fo`00=_l00`3oo`0bo`D0
8Ol00`3oo`1Co`030?oo03Ko000go`030?oo05Oo00<0ool0D_l00`3oo`0go`00>?l00`3oo`1Fo`03
0?oo05;o00<0ool0=ol003Wo00<0ool0EOl00`3oo`1Bo`030?oo03Oo000jo`@02Ol?03_o00<0ool0
D_l00`3oo`0go`00?_l900oo00<0ool0>?l00`3oo`1Ao`030?oo03So001Go`80>?l00`3oo`0io``0
2ol203[o001Io`808_l200Co0P03o`@01Ol00`3oo`0go`803?l300Oo0P0ko`00Fol00`3oo`0No`04
0?oo00So00<0ool01_l00`3oo`0eo`804Ol703go001Lo`030?oo01go00@0ool02Ol00`3oo`05o`80
6?l;017o0P1Go`00GOl00`3oo`0Lo`P01_l00`3oo`04o`030?oo01Go0P0;o`@02ol205Wo001No`03
0?oo01So1002o`040?ooo`<00_l0103oo`05o`030?oo01Co00<0ool03ol;05_o001Oo`030?oo01Co
0`04o`801ol20003o`0000Ko00<0ool04_l207So001Po`804_l201;o0P07o`030?oo017o00<0ool0
N?l006;o0P0>o`805_l200Go00<0ool03ol207_o001To`030?oo00Oo100Jo`030?oo00;o00<0ool0
3Ol207go001Uo`T07ol30003o`3o00Wo1@1oo`00T?l<08Co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85739 .13352 m
.85697 .13215 L
.8564 .13071 L
.85501 .12821 L
.85393 .12674 L
.85275 .12541 L
.85012 .12309 L
.84675 .12094 L
.84312 .11928 L
.84104 .11855 L
.83861 .11788 L
.83733 .11759 L
.83593 .11732 L
.83456 .11711 L
.8333 .11695 L
.83188 .11681 L
.83109 .11675 L
.83033 .11671 L
.82898 .11667 L
.8282 .11666 L
.82749 .11666 L
.8267 .11667 L
.82596 .11669 L
.82515 .11673 L
.82427 .11679 L
.82247 .11695 L
.82082 .11715 L
.81778 .11765 L
.81448 .11839 L
.80857 .12026 L
.80307 .1227 L
.79769 .1259 L
.79359 .12907 L
.78969 .13295 L
.78664 .13701 L
.7853 .13932 L
.78426 .14152 L
.78346 .14359 L
.78289 .14549 L
.78242 .14761 L
.78224 .14879 L
.78211 .1499 L
.78202 .151 L
.78199 .15163 L
.78197 .15222 L
.78197 .1534 L
.78199 .15407 L
.78202 .15467 L
.7821 .15575 L
.78222 .15693 L
Mistroke
.78238 .15804 L
.78255 .15906 L
.78368 .16382 L
.78533 .16885 L
.78889 .1781 L
.79067 .18302 L
.79152 .18574 L
.79222 .18831 L
.79277 .19079 L
.79318 .19307 L
.79335 .19429 L
.79351 .19562 L
.79357 .19636 L
.79363 .19704 L
.79367 .19769 L
.7937 .19839 L
.79374 .19952 L
.79374 .20058 L
.79372 .20185 L
.79367 .20301 L
.79359 .20416 L
.79347 .20542 L
.7933 .20673 L
.79312 .20795 L
.79262 .21057 L
.79203 .21296 L
.79025 .21823 L
.78823 .22266 L
.78565 .22699 L
.78301 .23046 L
.78029 .23329 L
.77704 .23587 L
.77511 .23706 L
.77329 .23796 L
.7715 .23867 L
.76947 .23928 L
.76842 .23952 L
.76727 .23972 L
.76617 .23987 L
.76515 .23997 L
.76418 .24002 L
.76327 .24005 L
.76225 .24005 L
.76115 .24001 L
.75998 .23993 L
.75886 .23983 L
.75672 .23955 L
.75425 .23913 L
.75157 .23858 L
.74657 .2374 L
Mistroke
.74036 .23585 L
.73444 .23451 L
.73171 .23399 L
.7287 .2335 L
.72731 .23331 L
.72583 .23314 L
.72443 .233 L
.72313 .2329 L
.72153 .23281 L
.72068 .23278 L
.71978 .23275 L
.71827 .23275 L
.71741 .23276 L
.71661 .23278 L
.71512 .23285 L
.71375 .23296 L
.71218 .23311 L
.7107 .23331 L
.70801 .23376 L
.70517 .2344 L
.70017 .23593 L
.69495 .23811 L
.68945 .24109 L
.67927 .24842 L
.67106 .25572 L
.66239 .26341 L
.65829 .2664 L
.6545 .26857 L
.65266 .26939 L
.65062 .27014 L
.64864 .27071 L
.64676 .27112 L
.64578 .27129 L
.6447 .27144 L
.6435 .27157 L
.64236 .27166 L
.64174 .27169 L
.64105 .27172 L
.63979 .27175 L
.63853 .27174 L
.63735 .27172 L
.63598 .27166 L
.63521 .27162 L
.63448 .27157 L
.63169 .27134 L
.62621 .27074 L
.61994 .26995 L
.61293 .26911 L
.60951 .26876 L
.60624 .26847 L
Mistroke
.6033 .26827 L
.60178 .26818 L
.6001 .2681 L
.59854 .26805 L
.59707 .26801 L
.59553 .26799 L
.59388 .26798 L
.59236 .26799 L
.59092 .26802 L
.58964 .26807 L
.58824 .26813 L
.58646 .26822 L
.5848 .26834 L
.58175 .26862 L
.57852 .26899 L
.57513 .26948 L
.56415 .27166 L
.5539 .27413 L
.54916 .27518 L
.54489 .27596 L
.54274 .27627 L
.54038 .27655 L
.53907 .27667 L
.53786 .27676 L
.53669 .27684 L
.53543 .2769 L
.53417 .27694 L
.53346 .27695 L
.5328 .27696 L
.53158 .27696 L
.53028 .27695 L
.52958 .27693 L
.52883 .27691 L
.52747 .27686 L
.52583 .27677 L
.52433 .27668 L
.51913 .27624 L
.51344 .27561 L
.50293 .2743 L
.48022 .27172 L
.4515 .26863 L
.44822 .26818 L
.4464 .26796 L
.44437 .26774 L
.44233 .26756 L
.44039 .26744 L
.43942 .26739 L
.43835 .26735 L
.43732 .26732 L
.43635 .2673 L
Mistroke
.43512 .2673 L
.43394 .26731 L
.43326 .26732 L
.43262 .26734 L
.43118 .26739 L
.42976 .26747 L
.42818 .26757 L
.42522 .26784 L
.42239 .26818 L
.4193 .26863 L
.41358 .26968 L
.40204 .27255 L
.38992 .27632 L
.37959 .2796 L
.37504 .28079 L
.37315 .28119 L
.37118 .28151 L
.37009 .28164 L
.3691 .28173 L
.36817 .28179 L
.3672 .28181 L
.3662 .28179 L
.36528 .28173 L
.36428 .28162 L
.36372 .28154 L
.36322 .28146 L
.36123 .28099 L
.36012 .28063 L
.3591 .28024 L
.35525 .27825 L
.35154 .27566 L
.34724 .27209 L
.33753 .2633 L
.32787 .25523 L
.31681 .24764 L
.31095 .24439 L
.30476 .24153 L
.29512 .23814 L
.29131 .23705 L
.28793 .23604 L
.28666 .23558 L
.28552 .23505 L
.2847 .23454 L
.28396 .23391 L
.2836 .23348 L
.28332 .23305 L
.28305 .2325 L
.28295 .23223 L
.28285 .23193 L
.28271 .23135 L
Mistroke
.28262 .23077 L
.28258 .23039 L
.28255 .23002 L
.28253 .22928 L
.28253 .22855 L
.28257 .22772 L
.28262 .22696 L
.28269 .22608 L
.28319 .22166 L
.28384 .21617 L
.28415 .21311 L
.28442 .21008 L
.28463 .20725 L
.28481 .20399 L
.28488 .20225 L
.28493 .20059 L
.28497 .19881 L
.28499 .19687 L
.28498 .19488 L
.28496 .19301 L
.28493 .19193 L
.28491 .19095 L
.28482 .18874 L
.28471 .18662 L
.28456 .18432 L
.28422 .1802 L
.28381 .17646 L
.28329 .17264 L
.28221 .16632 L
.27985 .15657 L
.27866 .15306 L
.2782 .15195 L
.27785 .15123 L
.27771 .15099 L
.27759 .15083 L
.27748 .15072 L
.27741 .15069 L
.27735 .15074 L
.27732 .15084 L
.27732 .15104 L
.27735 .15132 L
.2774 .15161 L
.27748 .15197 L
.27768 .15277 L
.27844 .15517 L
.2846 .16927 L
.2862 .17248 L
.28725 .17465 L
.28754 .17538 L
.28759 .17554 L
Mistroke
.28763 .17567 L
.28765 .17586 L
.28764 .17593 L
.28761 .17598 L
.28757 .17601 L
.28751 .17602 L
.28743 .176 L
.28732 .17595 L
.28719 .17588 L
.28706 .17579 L
.27451 .16459 L
.26352 .15572 L
.25182 .1477 L
.24031 .14157 L
.23714 .14027 L
.23424 .13927 L
.23175 .13856 L
.22912 .138 L
.22778 .1378 L
.22656 .13766 L
.2253 .13758 L
.22461 .13756 L
.22398 .13756 L
.22296 .1376 L
.22186 .13769 L
.22076 .13785 L
.21976 .13804 L
.21861 .13834 L
.21756 .13868 L
.2157 .13947 L
.21383 .14054 L
.21193 .14193 L
.209 .14472 L
.20647 .14776 L
.20087 .15554 L
.19433 .16382 L
.18673 .17097 L
.17827 .17673 L
.16825 .18188 L
.15188 .18895 L
.14837 .19062 L
.14509 .19241 L
.14266 .19402 L
.14057 .19578 L
.13959 .19681 L
.13868 .19797 L
.13802 .19899 L
.13743 .20014 L
.13694 .20132 L
.13658 .20245 L
Mistroke
.13629 .20363 L
.13606 .2048 L
.13589 .20604 L
.13576 .20726 L
.13571 .20798 L
.13567 .20866 L
.13563 .20942 L
.13562 .20984 L
.1356 .21024 L
.13559 .211 L
.13558 .21184 L
.13558 .21263 L
.13559 .2134 L
.13559 .21384 L
.1356 .21432 L
.13562 .21518 L
.13565 .21613 L
.13569 .21721 L
.13589 .22076 L
.13605 .22287 L
.13623 .22488 L
.13723 .23305 L
.13805 .23778 L
.13918 .24302 L
.14202 .25273 L
.14432 .2587 L
.14688 .26421 L
.15264 .27391 L
.15943 .28237 L
.1674 .28966 L
.17462 .29444 L
.17883 .29658 L
.18281 .29821 L
.18717 .29962 L
.1895 .30022 L
.19199 .30076 L
.19419 .30115 L
.19664 .30151 L
.19889 .30176 L
.20103 .30195 L
.2021 .30202 L
.20328 .30209 L
.20432 .30214 L
.20544 .30218 L
.20656 .3022 L
.20758 .30222 L
.20861 .30223 L
.20958 .30223 L
.21073 .30222 L
.21196 .3022 L
Mistroke
.21301 .30218 L
.21417 .30214 L
.21679 .30204 L
.21927 .30191 L
.23917 .30039 L
.24474 .30001 L
.24755 .29986 L
.24917 .29979 L
.25064 .29974 L
.25192 .29971 L
.25325 .29969 L
.25459 .29968 L
.25535 .29969 L
.25605 .29969 L
.25732 .29972 L
.25848 .29975 L
.2598 .2998 L
.26105 .29987 L
.26374 .30009 L
.26508 .30023 L
.26655 .30041 L
.2715 .30127 L
.27424 .30192 L
.27713 .30276 L
.28227 .30468 L
.28735 .30713 L
.2921 .30991 L
.30094 .31615 L
.31992 .33097 L
.32867 .33666 L
.33342 .33915 L
.33854 .34129 L
.34125 .34219 L
.34426 .343 L
.34702 .34356 L
.34963 .34394 L
.35101 .34408 L
.35233 .34418 L
.35352 .34424 L
.35482 .34428 L
.35623 .34427 L
.35774 .34423 L
.35903 .34416 L
.36047 .34405 L
.362 .34389 L
.36362 .34368 L
.36653 .3432 L
.37316 .34169 L
.38538 .33784 L
.39793 .33332 L
Mistroke
.4097 .32943 L
.41475 .32804 L
.42 .32684 L
.42269 .32635 L
.4256 .3259 L
.4282 .3256 L
.42945 .32549 L
.4306 .32541 L
.43191 .32533 L
.4326 .32531 L
.43332 .32528 L
.43453 .32527 L
.43586 .32528 L
.43701 .32531 L
.43825 .32536 L
.4394 .32544 L
.44045 .32553 L
.44276 .3258 L
.44523 .32619 L
.44774 .32671 L
.45001 .32728 L
.45864 .33034 L
.46731 .33446 L
.47566 .3386 L
.48036 .34069 L
.4857 .34274 L
.49073 .34434 L
.4967 .34583 L
.49946 .34638 L
.50252 .34689 L
.50551 .34731 L
.50829 .34761 L
.51154 .34788 L
.51335 .34799 L
.51507 .34807 L
.51669 .34812 L
.51819 .34814 L
.51984 .34815 L
.5216 .34813 L
.52338 .34808 L
.52439 .34805 L
.52531 .348 L
.52879 .34779 L
.53074 .34762 L
.53287 .3474 L
.53672 .34692 L
.54357 .34573 L
.55063 .34406 L
.56244 .34013 L
.57257 .33549 L
Mistroke
.58203 .33015 L
.58995 .32546 L
.59704 .32187 L
.60086 .32042 L
.60288 .31981 L
.60512 .31926 L
.60643 .319 L
.60766 .31879 L
.60884 .31862 L
.61015 .31847 L
.61134 .31837 L
.61248 .31829 L
.6137 .31824 L
.6144 .31822 L
.61504 .31821 L
.61637 .3182 L
.61712 .31822 L
.61781 .31823 L
.61917 .31829 L
.62047 .31836 L
.62202 .31847 L
.62346 .31859 L
.62679 .31896 L
.63914 .32087 L
.65058 .32288 L
.65686 .32388 L
.66267 .32463 L
.66588 .32495 L
.66766 .32509 L
.66929 .32519 L
.67064 .32525 L
.67211 .32529 L
.67284 .3253 L
.67362 .32531 L
.67502 .3253 L
.6764 .32526 L
.67717 .32523 L
.67789 .32519 L
.67917 .3251 L
.68053 .32497 L
.68305 .32465 L
.68526 .32427 L
.6898 .32317 L
.69431 .32162 L
.70188 .31806 L
.70984 .31359 L
.71434 .31117 L
.71878 .30907 L
.72316 .30736 L
.72557 .30658 L
Mistroke
.7282 .30587 L
.73048 .30537 L
.73305 .30491 L
.7342 .30475 L
.73545 .3046 L
.73656 .30448 L
.73776 .30438 L
.73899 .3043 L
.74015 .30425 L
.74154 .30421 L
.74282 .3042 L
.74416 .30421 L
.74563 .30424 L
.74703 .3043 L
.74832 .30438 L
.75064 .30456 L
.75321 .30482 L
.75845 .30552 L
.77096 .30779 L
.77694 .30896 L
.78229 .3099 L
.78691 .31055 L
.7882 .31069 L
.7894 .31079 L
.79046 .31087 L
.79158 .31093 L
.79273 .31097 L
.79338 .31098 L
.79397 .31098 L
.79516 .31096 L
.79581 .31093 L
.7964 .3109 L
.79744 .31082 L
.79855 .31069 L
.79957 .31054 L
.8005 .31038 L
.80229 .30996 L
.80414 .30938 L
.80581 .30871 L
.80728 .30798 L
.81046 .30589 L
.81193 .30463 L
.81346 .30308 L
.8161 .29972 L
.81856 .29553 L
.82058 .29106 L
.82217 .28663 L
.82363 .2816 L
.82493 .27597 L
.82675 .26534 L
Mistroke
.82754 .25923 L
.82818 .25344 L
.82944 .24021 L
.83072 .22765 L
.83229 .216 L
.83418 .20586 L
.83682 .19526 L
.83995 .18553 L
.84389 .17554 L
.85211 .1579 L
.85484 .15172 L
.85605 .14849 L
.85696 .14556 L
.85735 .14393 L
.85764 .14246 L
.85776 .14169 L
.85786 .14085 L
.85791 .14039 L
.85794 .13997 L
.85798 .13914 L
.858 .13846 L
.85799 .13782 L
.85796 .13712 L
.8579 .13638 L
.85781 .13561 L
.85769 .13488 L
.85739 .13352 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1@o`X0<ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1=o`802_l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`030?oo00_o0P0`o`00
OOl200_o0P05o`030?oo04_o00<0ool03Ol00`3oo`0^o`00TOl00`3oo`1:o`030?oo00ko0P0_o`00
AOl504Oo00<0ool0BOl00`3oo`0@o`030?oo02go0014o`801?l204Go00<0ool0B?l00`3oo`0Ao`03
0?oo02go0013o`030?oo00Ko0P13o`030?oo04So00<0ool04?l202oo0012o`030?oo00Wo0P06o`03
0?oo03So00<0ool0B?l00`3oo`0@o`030?oo02ko0011o`030?oo00co0P03o`80>_l00`3oo`18o`03
0?oo013o00<0ool0;_l0047o00<0ool03_l01@3oool003[o00<0ool0B?l00`3oo`0?o`030?oo02oo
0010o`030?oo013o0P000ol0o`0io`030?oo04So00<0ool03ol00`3oo`0_o`00?ol00`3oo`0Co`03
0?l003Wo00<0ool0B?l00`3oo`0>o`030?oo033o000no`030?oo01Go0P0io`030?oo04So00<0ool0
3_l00`3oo`0`o`00?Ol00`3oo`0Go`80>?l00`3oo`18o`030?oo00go00<0ool0<Ol003co00<0ool0
6?l202Co0P04o`800ol400Go00<0ool0BOl00`3oo`0<o`030?oo037o000jo`806ol00`3oo`0Ro`04
0?oo00So00<0ool01_l00`3oo`19o`030?oo00co00<0ool0<Ol003So0P0Mo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80Bol00`3oo`0:o`030?oo03;o000fo`807ol00`3oo`0Lo`@00_l0103oo`0:o`03
0?oo00Co00<0ool0B_l00`3oo`0:o`030?oo03;o000co`<08Ol00`3oo`0Ro`040?oo00So00@0ool0
1Ol00`3oo`1:o`030?oo00[o00<0ool0<_l0037o0P0To`030?oo02?o0P0:o`801_l00`3oo`1:o`80
2_l00`3oo`0co`00<?l202Go00<0ool0=ol00`3oo`1;o`030?oo00So00<0ool0<ol0033o00<0ool0
9?l00`3oo`0go`030?oo04_o00<0ool01ol00`3oo`0do`00<?l00`3oo`0To`030?oo03Oo00<0ool0
Bol00`3oo`07o`030?oo03Co0007o`D00_l200Co0P0Jo`030?oo00Ko1@0Io`030?oo00Ko0P04o`80
1?l202?o00<0ool08_l200Co0P04o`806_l00`3oo`06o`D08_l500;o0P04o`800ol000Wo00<0ool0
1ol0103oo`0Io`030?oo00So00<0ool06Ol00`3oo`05o`040?oo00So00@0ool08_l00`3oo`0Qo`04
0?oo00So00@0ool06Ol00`3oo`08o`030?oo02Co00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`03
0?oo01Oo00<0ool02?l00`3oo`0Io`030?oo00Go00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`0Go`030?oo00So00<0ool09?l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Jo`030?oo00@01?l00`3oo`0Io`030?l000<00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo
0P0Jo`030?oo00So00<0ool09?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Io`030?oo00So
00<0ool06Ol00`3oo`05o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool06?l00`3o
o`08o`809_l00`3oo`08o`030?oo00;o0008o`802_l301Wo00<0ool01ol201_o00<0ool01_l200[o
0`0Ro`030?oo02;o0P0:o`<05ol00`3oo`09o`809Ol200[o0`02o`00<?l00`3oo`0To`030?oo03Oo
00<0ool0=Ol400oo00<0ool02Ol00`3oo`0eo`00<?l00`3oo`0To`80>?l00`3oo`0bo`@00ol500So
0P0<o`030?oo03Go000`o`030?oo02Ko0`0eo`030?oo033o0P0<o`T03Ol00`3oo`0eo`00<Ol00`3o
o`0Xo`80<ol00`3oo`0_o`030?oo02;o00<0ool0=Ol0037o00<0ool0:_l3033o00<0ool0;Ol202Go
00<0ool0=Ol000coo`0<00Wo000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`00
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?l000Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01Ol200So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0So`030?oo00?o00<0ool0:Ol00`3oo`0Zo`03
0?oo02Wo0P0Yo`80;?l00`3oo`07o`00<ol00`3oo`0^o`030?oo02[o00<0ool0:Ol00`3oo`0Xo`03
0?oo03Go000co`030?oo02oo00<0ool0:Ol00`3oo`0Wo`80:_l00`3oo`0fo`00=?l00`3oo`0_o`80
4Ol?00Wo00<0ool04?l>00Oo0P0/o`030?oo03Ko000do`030?oo037o00<0ool02_l400oo2`0<o`D0
3_l702ko00<0ool0=_l003Go00<0ool0<Ol00`3oo`06o`<07?l00`3o000;04So00<0ool0=_l003Go
00<0ool0<_l801oo00<0ool0Dol00`3oo`0fo`00=_l00`3oo`1Ho`030?oo05?o00<0ool0=_l003Oo
00<0ool0Eol00`3oo`1Bo`030?oo03Oo000ho`030?oo05Ko00<0ool0D_l00`3oo`0go`00>Ol305Go
00<0ool0DOl00`3oo`0ho`00??l300[o2P0no`030?oo057o00<0ool0>?l003oo2P0:o`<0>ol00`3o
o`0ko`D04?l00`3oo`0io`00E_l303So00<0ool0>?l400?o1`08o`80??l005Wo00<0ool08Ol200Co
0P03o`@01Ol00`3oo`0eo`<03_l903go001Jo`808?l0103oo`08o`030?oo00Ko00<0ool0<ol205Oo
001Lo`030?oo01go00@0ool02Ol00`3oo`05o`806Ol8017o0P1Io`00GOl00`3oo`0Lo`040?oo00[o
00<0ool01?l00`3oo`0Eo`@01ol600Wo0P1Ko`00G_l201_o2@04o`040?oo00Go00<0ool04ol2017o
2@1Mo`00H?l00`3oo`0Do`@00_l200Co1002o`801_l00`3oo`0Bo`030?oo07Oo001Qo`030?oo013o
0`0@o`802?l00`3oo`0Ao`030?oo07So001Ro`803Ol301Go00<0ool01Ol00`3oo`0?o`80Nol006Co
0`07o`<06Ol200Go00<0ool03?l307go001Vo`P07_l600Wo1@20o`00T?l;08Go002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103o
o`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go
00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co
0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103o
o`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co
001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.85964 .13615 m
.85916 .13468 L
.85852 .13313 L
.85704 .13044 L
.85473 .12743 L
.85209 .12491 L
.8488 .12257 L
.84685 .12148 L
.84462 .12043 L
.84256 .11963 L
.84016 .11886 L
.83787 .11828 L
.83564 .11784 L
.83436 .11765 L
.83297 .11747 L
.83215 .11739 L
.83141 .11733 L
.83065 .11728 L
.82995 .11724 L
.82856 .1172 L
.82783 .11719 L
.82705 .11719 L
.82573 .11723 L
.82427 .1173 L
.82285 .11741 L
.82156 .11755 L
.81864 .11796 L
.81593 .11849 L
.81339 .11912 L
.80747 .12108 L
.8023 .12346 L
.79712 .12662 L
.79234 .13051 L
.78902 .13409 L
.78656 .13758 L
.78452 .14157 L
.78376 .14363 L
.78311 .1459 L
.78289 .14695 L
.78271 .14793 L
.78258 .14884 L
.78247 .14979 L
.78238 .15083 L
.78234 .1518 L
.78232 .15283 L
.78232 .15343 L
.78234 .15398 L
.78238 .15496 L
.78244 .15589 L
.78252 .15672 L
Mistroke
.78262 .15763 L
.78291 .15967 L
.78326 .16157 L
.78518 .16955 L
.78741 .17821 L
.78835 .18288 L
.78877 .18559 L
.78893 .1869 L
.78906 .18815 L
.78915 .18928 L
.78923 .19053 L
.78928 .19166 L
.78931 .19272 L
.78931 .19386 L
.78929 .19513 L
.78927 .19576 L
.78924 .19645 L
.78916 .1977 L
.78905 .19904 L
.78891 .2003 L
.7885 .20314 L
.78795 .2059 L
.78721 .20882 L
.78548 .21398 L
.78349 .21844 L
.78107 .22266 L
.77806 .22675 L
.77487 .23004 L
.77167 .23253 L
.76822 .23445 L
.76619 .23528 L
.76426 .23588 L
.76242 .23632 L
.76136 .23651 L
.76036 .23666 L
.75948 .23676 L
.75849 .23684 L
.75745 .2369 L
.75645 .23694 L
.75536 .23695 L
.75413 .23693 L
.75351 .2369 L
.75284 .23687 L
.75162 .2368 L
.7492 .23659 L
.74694 .23634 L
.74157 .23561 L
.73561 .23478 L
.73307 .23447 L
.7303 .23419 L
Mistroke
.72869 .23405 L
.72722 .23395 L
.72572 .23386 L
.72429 .23381 L
.72301 .23378 L
.72163 .23378 L
.72033 .2338 L
.71915 .23383 L
.71776 .2339 L
.71625 .23401 L
.71482 .23415 L
.71351 .2343 L
.71058 .23474 L
.70786 .23529 L
.70176 .237 L
.69598 .2393 L
.69004 .24237 L
.67127 .2562 L
.66266 .26307 L
.65804 .26611 L
.65363 .26835 L
.6496 .26981 L
.64724 .27042 L
.64502 .27084 L
.6438 .27102 L
.64265 .27116 L
.64136 .27128 L
.63996 .27137 L
.63925 .2714 L
.63846 .27143 L
.63772 .27144 L
.63702 .27145 L
.63569 .27145 L
.63501 .27144 L
.63425 .27142 L
.63288 .27137 L
.63157 .27131 L
.62904 .27116 L
.62314 .27069 L
.61183 .26972 L
.60855 .2695 L
.60544 .26934 L
.60408 .26928 L
.60264 .26923 L
.60107 .26919 L
.59962 .26917 L
.59807 .26915 L
.59641 .26916 L
.59485 .26918 L
.59344 .26921 L
Mistroke
.59179 .26927 L
.59 .26935 L
.5868 .26956 L
.58411 .2698 L
.5812 .27011 L
.57549 .27091 L
.56585 .27274 L
.55585 .27496 L
.55115 .27593 L
.54683 .27669 L
.54425 .27707 L
.54189 .27735 L
.53925 .2776 L
.53779 .2777 L
.53641 .27778 L
.53522 .27783 L
.53395 .27786 L
.53286 .27788 L
.53166 .27789 L
.53035 .27788 L
.52961 .27787 L
.52894 .27786 L
.52761 .27783 L
.52635 .27778 L
.52386 .27766 L
.5215 .27751 L
.51598 .27707 L
.50446 .2759 L
.48429 .2739 L
.45267 .27078 L
.44904 .27029 L
.44496 .2698 L
.44287 .26958 L
.44082 .26941 L
.43892 .26927 L
.43782 .26921 L
.43677 .26917 L
.43565 .26913 L
.43505 .26911 L
.4344 .2691 L
.43371 .26909 L
.43304 .26908 L
.43178 .26908 L
.43058 .2691 L
.42945 .26912 L
.42819 .26917 L
.42683 .26923 L
.4253 .26932 L
.42362 .26945 L
.4205 .26974 L
Mistroke
.41448 .27054 L
.40865 .27156 L
.40315 .27271 L
.39091 .27575 L
.38515 .27724 L
.37926 .27862 L
.3767 .27912 L
.37407 .27954 L
.37302 .27967 L
.37191 .27978 L
.37128 .27983 L
.37071 .27987 L
.3696 .27992 L
.36856 .27993 L
.36747 .27991 L
.36645 .27985 L
.36554 .27977 L
.36448 .27963 L
.36335 .27944 L
.36133 .27895 L
.35923 .27826 L
.35704 .27732 L
.35312 .27513 L
.3372 .26262 L
.32696 .25463 L
.31651 .24782 L
.30715 .24299 L
.29782 .23927 L
.29023 .23669 L
.2886 .23609 L
.28703 .23543 L
.28581 .23482 L
.28469 .23412 L
.28368 .23327 L
.28292 .23238 L
.28255 .23182 L
.28226 .23126 L
.28198 .2306 L
.28173 .22983 L
.28156 .22913 L
.28148 .22875 L
.28141 .22832 L
.28136 .22794 L
.28131 .2275 L
.28124 .2267 L
.2812 .22588 L
.28118 .22494 L
.28118 .22401 L
.28119 .22311 L
.28178 .21326 L
Mistroke
.28199 .21016 L
.28221 .20661 L
.28237 .2034 L
.28245 .20153 L
.28251 .19975 L
.28257 .19792 L
.2826 .19623 L
.28263 .19441 L
.28264 .1934 L
.28264 .19244 L
.28264 .19075 L
.28263 .18918 L
.28261 .18746 L
.28257 .18564 L
.28253 .18413 L
.28247 .18252 L
.28234 .17965 L
.28215 .17633 L
.28194 .1734 L
.28098 .16349 L
.28046 .15919 L
.28012 .15632 L
.28006 .15573 L
.28001 .15518 L
.27998 .15474 L
.27997 .15439 L
.27997 .1541 L
.27998 .15396 L
.28 .15385 L
.28005 .15369 L
.28009 .15363 L
.28013 .1536 L
.28022 .1536 L
.28032 .15365 L
.28045 .15377 L
.28061 .15396 L
.28136 .15513 L
.28391 .15986 L
.28711 .16604 L
.28829 .16846 L
.28876 .16953 L
.28893 .16998 L
.28905 .17035 L
.28913 .17064 L
.28917 .17087 L
.28918 .17108 L
.28916 .17126 L
.28908 .17138 L
.28903 .17143 L
.28895 .17145 L
Mistroke
.28878 .17146 L
.28857 .17141 L
.2883 .17131 L
.28795 .17113 L
.28718 .17066 L
.28466 .16884 L
.27745 .16326 L
.25694 .14911 L
.2441 .14244 L
.24068 .14105 L
.23716 .13983 L
.23422 .13899 L
.23265 .13862 L
.23119 .13833 L
.22986 .13812 L
.22917 .13802 L
.22845 .13794 L
.2272 .13783 L
.22607 .13778 L
.22496 .13777 L
.22395 .1378 L
.22286 .13787 L
.22172 .138 L
.22059 .13818 L
.2194 .13843 L
.2171 .13912 L
.21509 .13996 L
.21332 .1409 L
.2099 .14328 L
.20648 .14638 L
.20049 .15291 L
.18768 .16549 L
.17971 .17112 L
.17012 .17649 L
.15149 .1855 L
.148 .1874 L
.14465 .18949 L
.14212 .19137 L
.13988 .19344 L
.13874 .19474 L
.13784 .19596 L
.13711 .19713 L
.13644 .19844 L
.13589 .19977 L
.13547 .20103 L
.13515 .20228 L
.13488 .20365 L
.13477 .20434 L
.13467 .2051 L
.13459 .20579 L
Mistroke
.13453 .20653 L
.13448 .20721 L
.13444 .20797 L
.13442 .20868 L
.1344 .20935 L
.1344 .21016 L
.1344 .21103 L
.13441 .21178 L
.13444 .21262 L
.13448 .21355 L
.13453 .21456 L
.13465 .21643 L
.13499 .22027 L
.13545 .22413 L
.13671 .23194 L
.13763 .23636 L
.13887 .2414 L
.14228 .25188 L
.1466 .26158 L
.15167 .27019 L
.15869 .27923 L
.1662 .28644 L
.1734 .29164 L
.18158 .29594 L
.18621 .29775 L
.19063 .29912 L
.19466 .30009 L
.19909 .30091 L
.20144 .30125 L
.20405 .30155 L
.2068 .3018 L
.2094 .30198 L
.21067 .30205 L
.21187 .3021 L
.21297 .30214 L
.21415 .30218 L
.21543 .3022 L
.21662 .30222 L
.21791 .30224 L
.21865 .30224 L
.21934 .30224 L
.22061 .30224 L
.22132 .30223 L
.22196 .30223 L
.22322 .30221 L
.22442 .30219 L
.22711 .30214 L
.23006 .30206 L
.23513 .3019 L
.23792 .30182 L
Mistroke
.24054 .30174 L
.24311 .30168 L
.24423 .30166 L
.24545 .30164 L
.24679 .30163 L
.24806 .30162 L
.24876 .30162 L
.2495 .30162 L
.25084 .30163 L
.25195 .30164 L
.25315 .30167 L
.25428 .3017 L
.25531 .30173 L
.25769 .30184 L
.25888 .30192 L
.26019 .30201 L
.26281 .30225 L
.26563 .30258 L
.27067 .30343 L
.27548 .30458 L
.27983 .30594 L
.28947 .31016 L
.29947 .31619 L
.31811 .32943 L
.32663 .33486 L
.33626 .33965 L
.34171 .34159 L
.34439 .34233 L
.34685 .34288 L
.34942 .34332 L
.35185 .34363 L
.3532 .34375 L
.35444 .34384 L
.35581 .3439 L
.35727 .34392 L
.35873 .34392 L
.36006 .34388 L
.3608 .34384 L
.36161 .3438 L
.36308 .34369 L
.36592 .34338 L
.36857 .34301 L
.37471 .34182 L
.38131 .34017 L
.39271 .33683 L
.40474 .33331 L
.41486 .33087 L
.41759 .33034 L
.42046 .32986 L
.42313 .32949 L
Mistroke
.42555 .32922 L
.42686 .3291 L
.4283 .329 L
.42973 .32892 L
.4305 .32888 L
.43122 .32886 L
.43254 .32884 L
.43373 .32883 L
.4351 .32885 L
.43637 .32889 L
.43751 .32895 L
.4387 .32903 L
.44081 .32922 L
.44304 .3295 L
.44545 .32988 L
.45023 .3309 L
.45517 .33231 L
.46371 .33545 L
.48055 .34215 L
.48566 .3438 L
.49149 .34536 L
.49738 .34658 L
.50306 .34744 L
.50601 .34777 L
.50921 .34804 L
.51199 .34821 L
.5136 .34828 L
.51511 .34833 L
.51595 .34834 L
.51687 .34836 L
.51855 .34836 L
.5202 .34835 L
.52173 .34832 L
.52268 .34829 L
.52357 .34825 L
.52523 .34817 L
.52703 .34806 L
.52898 .34791 L
.53206 .34762 L
.53543 .34721 L
.54217 .34612 L
.54826 .3448 L
.55375 .34333 L
.56572 .33908 L
.57572 .33442 L
.59187 .32599 L
.59602 .32417 L
.59998 .32274 L
.60426 .32157 L
.60662 .32109 L
Mistroke
.60923 .32068 L
.61138 .32043 L
.61262 .32032 L
.61379 .32024 L
.61505 .32018 L
.61621 .32014 L
.61757 .32011 L
.61887 .32011 L
.62 .32012 L
.62122 .32015 L
.62227 .32019 L
.62343 .32024 L
.62607 .3204 L
.62859 .32059 L
.6392 .32171 L
.65081 .32306 L
.65423 .32341 L
.65743 .32369 L
.66026 .32388 L
.6617 .32397 L
.66327 .32404 L
.66474 .32408 L
.66609 .32411 L
.66751 .32412 L
.6683 .32412 L
.66901 .32412 L
.67027 .32409 L
.67163 .32404 L
.67289 .32398 L
.67405 .3239 L
.67668 .32366 L
.67908 .32335 L
.68165 .32291 L
.68433 .32234 L
.68955 .32086 L
.69408 .31917 L
.71074 .31113 L
.71872 .30773 L
.72347 .30618 L
.72798 .30506 L
.73064 .30455 L
.73358 .30412 L
.73496 .30397 L
.73643 .30383 L
.7377 .30373 L
.7391 .30365 L
.74051 .3036 L
.74181 .30357 L
.74253 .30356 L
.74331 .30356 L
Mistroke
.74475 .30357 L
.7461 .3036 L
.74755 .30365 L
.74879 .30371 L
.75016 .3038 L
.75282 .304 L
.75575 .30429 L
.76165 .30501 L
.77205 .3065 L
.77772 .30727 L
.78074 .30762 L
.7822 .30776 L
.78353 .30787 L
.78475 .30796 L
.78607 .30803 L
.78675 .30806 L
.78738 .30809 L
.78804 .3081 L
.78875 .30811 L
.78997 .30811 L
.79112 .30809 L
.79213 .30805 L
.79323 .30798 L
.79443 .30787 L
.79551 .30775 L
.79789 .30736 L
.79991 .30689 L
.80197 .30626 L
.80534 .30484 L
.80715 .30385 L
.80876 .3028 L
.81214 .30001 L
.81493 .29697 L
.81783 .2929 L
.82028 .28856 L
.82409 .27956 L
.8273 .26893 L
.82951 .25931 L
.83159 .24839 L
.83582 .22453 L
.84073 .20346 L
.847 .18442 L
.85412 .16681 L
.8571 .15918 L
.85845 .15518 L
.85946 .15151 L
.85983 .14987 L
.86014 .14816 L
.86027 .14727 L
.86039 .14629 L
Mistroke
.86047 .14541 L
.86053 .1446 L
.86056 .1438 L
.86058 .14305 L
.86057 .14238 L
.86054 .14166 L
.86049 .14087 L
.86041 .14004 L
.86033 .13934 L
.8602 .13857 L
.85997 .1374 L
.85964 .13615 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1Ao`T0<ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1=o`<02?l203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`030?oo00_o0P0`o`00
OOl200_o0P05o`030?oo04[o0P0?o`030?oo02ko002Ao`030?oo04Wo00<0ool04?l00`3oo`0]o`00
A_l504Ko00<0ool0BOl00`3oo`0@o`030?oo02go0014o`801Ol304?o00<0ool0B?l00`3oo`0Ao`03
0?oo02go0012o`802_l2047o00<0ool0B?l00`3oo`0Ao`030?oo02go0011o`030?oo00co00<0ool0
?_l00`3oo`18o`030?oo017o00<0ool0;Ol0047o00<0ool03Ol00`3oo`02o`030?oo03So00<0ool0
B?l00`3oo`0Ao`030?oo02go0010o`030?oo00oo0P02o`030?oo03So00<0ool0B?l00`3oo`0Ao`03
0?oo02go000oo`030?oo01;o00@0o`00>Ol00`3oo`18o`030?oo013o00<0ool0;_l003ko00<0ool0
5?l20003o`3o03Oo00<0ool0B?l00`3oo`0@o`030?oo02ko000mo`030?oo01Ko0`0ho`030?oo04So
00<0ool04?l00`3oo`0^o`00>ol201Wo00@0o`00=ol00`3oo`18o`030?oo013o00<0ool0;_l003Wo
0P0Ko`809Ol200Co0P03o`@01Ol00`3oo`19o`030?oo00ko00<0ool0;ol003Oo0P0No`030?oo02;o
00@0ool02?l00`3oo`06o`030?oo04Wo00<0ool03_l00`3oo`0_o`00=Ol2023o00<0ool08_l0103o
o`09o`030?oo00Go0P1:o`030?oo00go00<0ool0<?l003?o0P0Ro`030?oo01co1002o`040?oo00[o
00<0ool01?l00`3oo`19o`030?oo00go00<0ool0<?l0037o0P0To`030?oo02;o00@0ool02?l0103o
o`05o`030?oo04Wo00<0ool03Ol00`3oo`0`o`00<?l202Go00<0ool08ol200[o0P06o`030?oo04Wo
00<0ool03?l00`3oo`0ao`00<?l00`3oo`0To`030?oo03Oo00<0ool0BOl00`3oo`0<o`030?oo037o
000_o`809_l00`3oo`0go`030?oo04Wo00<0ool02ol00`3oo`0bo`00;ol00`3oo`0Uo`030?oo03Oo
00<0ool0BOl00`3oo`0;o`030?oo03;o0007o`D00_l200Co0P0Io`030?oo00Oo1@0Io`030?oo00Ko
0P04o`801?l202?o00<0ool08_l200Co0P04o`806Ol00`3oo`07o`D08_l500;o0P04o`800ol000Wo
00<0ool01ol0103oo`0Ho`030?oo00Wo00<0ool06?l00`3oo`06o`040?oo00So00@0ool08_l00`3o
o`0Qo`040?oo00So00@0ool05ol00`3oo`0:o`030?l002Co00<0ool01ol0103oo`02o`002Ol00`3o
o`0:o`030?oo01Ko00<0ool02Ol00`3oo`0Ho`030?oo00Ko00@0ool02ol00`3oo`0Po`030?oo027o
00@0ool02ol00`3oo`0Eo`030?oo00[o0P0Uo`030?oo00[o0@01o`7o00000ol0000200Co00<0ool0
2?l201Wo00@0oooo1004o`030?oo01So00<0ool01002o`040?oo00Wo0P0So`030?oo027o00@0ool0
2Ol201So00<0ool02_l202Go00<0ool02?l200?o0009o`030?oo00So00<0ool06Ol00`3oo`08o`03
0?oo01So00<0ool01_l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`030?oo01Ko00<0ool0
2ol202Go00<0ool02?l00`3oo`02o`002?l200[o0`0Io`030?oo00Oo0P0Jo`030?oo00Oo0P0:o`<0
8_l00`3oo`0Ro`802_l301Go00<0ool02ol302Co0P0:o`<00_l0033o00<0ool08ol203Wo00<0ool0
AOl200oo00<0ool0<ol0033o00<0ool09?l403Ko00<0ool0=?l<0003o`0000804Ol00`3oo`0co`00
<?l00`3oo`0Xo`80=?l00`3oo`0ao`<03?l201?o00<0ool0=?l0033o00<0ool0:_l203;o00<0ool0
;ol202Co00<0ool0=?l0037o00<0ool0:ol2033o00<0ool0;_l00`3oo`0To`030?oo03Co000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`04o`030?l000So00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`001_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`07o`003?l00`3oo`0So`030?oo00?o00<0ool0:?l202co00<0ool0:Ol202Wo
0P0/o`030?oo00Oo000bo`030?oo033o00<0ool0:Ol00`3oo`0Xo`80:_l00`3oo`0eo`00<ol00`3o
o`0`o`030?oo02So00<0ool09ol00`3oo`0Zo`030?oo03Go000do`030?oo033o00<0ool04?l:00go
00<0ool04_l:00So0`0]o`030?oo03Go000eo`030?oo033o00<0ool02ol400[o30000ol0o`0>o`D0
2_l8033o00<0ool0=Ol003Go00<0ool0<Ol200Oo100Joa40A_l00`3oo`0fo`00=_l00`3oo`0bo`L0
7ol00`3oo`1Co`030?oo03Ko000go`030?oo05Oo00<0ool0Dol00`3oo`0fo`00>?l00`3oo`1Fo`03
0?oo05;o00<0ool0=ol003Wo0P1Fo`030?oo057o00<0ool0>?l003_o0P1Do`030?oo053o00<0ool0
>Ol003go0`1Ao`030?oo053o00<0ool0>Ol0043o5@0lo`030?oo03_o1`0<o`80??l005Go0`0io`03
0?oo03Ko1@07o`d0?Ol005So0P0So`801?l200?o1005o`030?oo03Co0P1Fo`00F_l00`3oo`0Oo`04
0?oo00So00<0ool01_l00`3oo`0bo`80F?l005_o00<0ool07_l0103oo`09o`030?oo00Go0P0ao`80
F_l005co0P0No`040?oo00[o00<0ool01?l00`3oo`0Go`l01ol305co001No`807?l0103oo`08o`04
0?oo00Go00<0ool05?l300oo1`1Oo`00H?l00`3oo`0Ho`P01_l200Ko00<0ool04_l207So001Qo`03
0?oo01;o1@08o`<02ol00`3oo`0Ao`030?oo07So001Ro`803ol3013o0P09o`030?oo00oo0P1ko`00
I?l300So100Eo`801ol00`3oo`0=o`80OOl006Oo200Ko`D00_l00`3oo`09o`@0Ool008oo3P23o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0
S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool0
2Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool0
1Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o
0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86189 .13877 m
.86064 .13555 L
.85908 .13268 L
.85692 .12969 L
.85406 .12674 L
.85048 .12397 L
.84656 .12173 L
.84263 .12009 L
.84026 .11933 L
.83798 .11874 L
.83556 .11826 L
.8342 .11804 L
.83289 .11788 L
.83165 .11775 L
.8303 .11765 L
.82905 .11759 L
.82788 .11756 L
.82715 .11756 L
.82636 .11756 L
.82492 .1176 L
.82351 .11768 L
.82221 .11779 L
.8208 .11794 L
.81925 .11814 L
.81613 .11869 L
.81306 .11941 L
.8103 .12021 L
.80436 .12249 L
.79929 .12515 L
.79498 .12808 L
.79158 .131 L
.78842 .13449 L
.78581 .13835 L
.78485 .14018 L
.784 .14213 L
.78339 .14385 L
.78285 .14575 L
.78247 .1475 L
.78221 .14914 L
.78209 .15014 L
.782 .1511 L
.78195 .15196 L
.78191 .15289 L
.78189 .15391 L
.7819 .155 L
.78193 .15593 L
.78198 .15696 L
.78205 .15804 L
.78213 .15905 L
.78236 .16135 L
Mistroke
.78292 .1656 L
.78409 .17388 L
.78439 .17641 L
.78462 .17882 L
.78478 .18101 L
.78484 .18226 L
.78489 .18341 L
.78492 .18467 L
.78493 .18584 L
.78493 .18652 L
.78492 .18714 L
.78491 .18786 L
.78489 .18854 L
.78486 .1892 L
.78483 .1899 L
.78475 .19117 L
.78464 .19254 L
.78449 .19405 L
.78418 .1964 L
.78375 .19898 L
.78257 .20412 L
.78111 .20877 L
.77943 .2129 L
.77707 .2175 L
.7746 .2213 L
.77138 .22514 L
.76801 .22817 L
.76474 .23034 L
.76089 .23217 L
.75884 .23287 L
.75657 .23347 L
.75435 .23391 L
.75335 .23406 L
.75227 .2342 L
.75126 .2343 L
.75014 .2344 L
.74911 .23447 L
.74813 .23452 L
.74696 .23456 L
.74588 .23458 L
.74468 .23459 L
.74339 .23459 L
.74234 .23458 L
.74119 .23456 L
.73911 .23451 L
.73681 .23445 L
.73424 .23438 L
.73292 .23435 L
.73219 .23434 L
.7315 .23433 L
Mistroke
.7307 .23432 L
.72998 .23432 L
.72929 .23432 L
.72854 .23432 L
.7272 .23433 L
.72576 .23437 L
.72439 .23441 L
.72315 .23447 L
.72156 .23457 L
.72009 .23468 L
.71677 .23503 L
.71383 .23547 L
.71073 .23606 L
.70519 .23748 L
.70009 .23923 L
.689 .2446 L
.67155 .25666 L
.66323 .26259 L
.65875 .2653 L
.65449 .2674 L
.64993 .26907 L
.64744 .26975 L
.64467 .27033 L
.64222 .2707 L
.64095 .27085 L
.63955 .27098 L
.63834 .27107 L
.63702 .27115 L
.63576 .27119 L
.6346 .27122 L
.63328 .27124 L
.63261 .27124 L
.63187 .27124 L
.63054 .27123 L
.62929 .2712 L
.6278 .27116 L
.62641 .27111 L
.62325 .27098 L
.61727 .27069 L
.6117 .27044 L
.61026 .27039 L
.6087 .27034 L
.60735 .2703 L
.60586 .27028 L
.60428 .27025 L
.60341 .27025 L
.60261 .27025 L
.60116 .27025 L
.59961 .27026 L
.59817 .27029 L
Mistroke
.5966 .27033 L
.59513 .27039 L
.59377 .27045 L
.59121 .27059 L
.58852 .27079 L
.58254 .2714 L
.5771 .27214 L
.56541 .27427 L
.55503 .27636 L
.55008 .27724 L
.5455 .27791 L
.54135 .27836 L
.53901 .27855 L
.53687 .27868 L
.53563 .27874 L
.53448 .27878 L
.53322 .27881 L
.53258 .27882 L
.53188 .27883 L
.53068 .27884 L
.52939 .27883 L
.52817 .27882 L
.52705 .2788 L
.52582 .27877 L
.52465 .27873 L
.52197 .27862 L
.51645 .27829 L
.5063 .2775 L
.48389 .27567 L
.45127 .27258 L
.44736 .27208 L
.44299 .27158 L
.44077 .27137 L
.43861 .27119 L
.43659 .27105 L
.43439 .27094 L
.43325 .27089 L
.43197 .27086 L
.4308 .27083 L
.42968 .27082 L
.42835 .27082 L
.42711 .27084 L
.42642 .27085 L
.42566 .27087 L
.42427 .27091 L
.42272 .27098 L
.42129 .27106 L
.41805 .27129 L
.41498 .27158 L
.41206 .27191 L
Mistroke
.39097 .27548 L
.38548 .27652 L
.38256 .27702 L
.37988 .27743 L
.37744 .27774 L
.37619 .27788 L
.37487 .27799 L
.37379 .27806 L
.3726 .27812 L
.37154 .27815 L
.37056 .27815 L
.36951 .27813 L
.36837 .27807 L
.3678 .27803 L
.3672 .27798 L
.36613 .27786 L
.36506 .27771 L
.36407 .27754 L
.36224 .27713 L
.3601 .27652 L
.35816 .27581 L
.35379 .27375 L
.34528 .26823 L
.33549 .26087 L
.29724 .23931 L
.29323 .23779 L
.2894 .23625 L
.28655 .23486 L
.28408 .23322 L
.28296 .23219 L
.28192 .23091 L
.28116 .2296 L
.28086 .22894 L
.28058 .22822 L
.28035 .22751 L
.28013 .22671 L
.27982 .22518 L
.27969 .22426 L
.27957 .22322 L
.27949 .22219 L
.27943 .22121 L
.27939 .2202 L
.27937 .21921 L
.27936 .21809 L
.27937 .21684 L
.27938 .2162 L
.27939 .21549 L
.27943 .21415 L
.27952 .21154 L
.28005 .20064 L
Mistroke
.28035 .19395 L
.28062 .18675 L
.2808 .18014 L
.28093 .17453 L
.28105 .16935 L
.28113 .16702 L
.28123 .16469 L
.28134 .16291 L
.28141 .162 L
.28149 .16123 L
.28156 .16057 L
.28165 .15989 L
.28185 .15882 L
.28198 .15827 L
.28211 .15782 L
.28228 .15739 L
.28247 .157 L
.28268 .1567 L
.28279 .15658 L
.28291 .15646 L
.28302 .15639 L
.28315 .15632 L
.28338 .15625 L
.28363 .15623 L
.28378 .15625 L
.28392 .15628 L
.28418 .15638 L
.28448 .15652 L
.28501 .1569 L
.28563 .15746 L
.2883 .16082 L
.29033 .16422 L
.29069 .165 L
.29085 .16538 L
.29096 .16571 L
.29101 .16589 L
.29105 .16607 L
.29108 .16621 L
.29109 .16634 L
.29108 .16656 L
.29107 .16666 L
.29103 .16675 L
.29099 .16683 L
.29094 .16689 L
.29081 .16697 L
.29062 .16701 L
.2905 .16702 L
.29037 .16701 L
.29012 .16696 L
.2898 .16687 L
Mistroke
.28901 .16654 L
.28675 .16531 L
.28379 .16345 L
.27623 .15852 L
.26556 .15194 L
.25266 .14512 L
.24616 .14233 L
.24017 .14026 L
.23764 .13955 L
.23502 .13894 L
.23368 .13868 L
.23224 .13844 L
.23104 .13827 L
.22976 .13813 L
.2285 .13803 L
.22732 .13798 L
.2262 .13797 L
.22518 .13798 L
.2241 .13804 L
.22293 .13814 L
.22174 .13828 L
.22065 .13846 L
.21872 .13889 L
.21702 .13938 L
.2134 .14085 L
.20994 .14279 L
.20645 .14527 L
.20036 .15043 L
.18511 .16291 L
.166 .17403 L
.15719 .17854 L
.14885 .1832 L
.14264 .18758 L
.13984 .19019 L
.13771 .19273 L
.13615 .19517 L
.13505 .19749 L
.13457 .19882 L
.13413 .20031 L
.13379 .20175 L
.13355 .20313 L
.13344 .20393 L
.13334 .2048 L
.13328 .20555 L
.13322 .20638 L
.13319 .20686 L
.13318 .20729 L
.13316 .2078 L
.13315 .20829 L
.13314 .20916 L
Mistroke
.13315 .20966 L
.13316 .21012 L
.13318 .21096 L
.13321 .21176 L
.13325 .21265 L
.13332 .21362 L
.13349 .21564 L
.13369 .21754 L
.1348 .22503 L
.13559 .22905 L
.13664 .23362 L
.13953 .24348 L
.1431 .25267 L
.14846 .2631 L
.15472 .27227 L
.16163 .28005 L
.16968 .28695 L
.17698 .29169 L
.1853 .2957 L
.1899 .29739 L
.19434 .29872 L
.19843 .29972 L
.20285 .30058 L
.20774 .30132 L
.21235 .30184 L
.21522 .30209 L
.21795 .30229 L
.22312 .30257 L
.22778 .30275 L
.23223 .30287 L
.23741 .30298 L
.24215 .30309 L
.2474 .30326 L
.25036 .30339 L
.25308 .30354 L
.25798 .30391 L
.26043 .30415 L
.26311 .30447 L
.26814 .30525 L
.27345 .30637 L
.27838 .30774 L
.28283 .30926 L
.29303 .31386 L
.30261 .3194 L
.32163 .33151 L
.33026 .33624 L
.33491 .33837 L
.33989 .34027 L
.34251 .34109 L
.34542 .34187 L
Mistroke
.34811 .34246 L
.35058 .3429 L
.35314 .34324 L
.35456 .34338 L
.35591 .34348 L
.35716 .34355 L
.35832 .3436 L
.3596 .34362 L
.36096 .34363 L
.36239 .3436 L
.3637 .34355 L
.36515 .34347 L
.36667 .34336 L
.36926 .34311 L
.3721 .34274 L
.37859 .34165 L
.38469 .34036 L
.40641 .33534 L
.4123 .33421 L
.41549 .33369 L
.41845 .33328 L
.42115 .33296 L
.42359 .33273 L
.42627 .33255 L
.42773 .33247 L
.42911 .33243 L
.42985 .33241 L
.43054 .3324 L
.43182 .33239 L
.4332 .3324 L
.43396 .33242 L
.43467 .33244 L
.43584 .33249 L
.43711 .33256 L
.43964 .33275 L
.44238 .33305 L
.44526 .33346 L
.45029 .33441 L
.45971 .33691 L
.46805 .33961 L
.47626 .34229 L
.48555 .34489 L
.49075 .34605 L
.4968 .34712 L
.49956 .34751 L
.5026 .34786 L
.50555 .34814 L
.50828 .34834 L
.50967 .34842 L
.51119 .34848 L
Mistroke
.5128 .34854 L
.51434 .34857 L
.51579 .34859 L
.51733 .34859 L
.51865 .34858 L
.52011 .34855 L
.521 .34853 L
.52184 .3485 L
.52342 .34843 L
.52513 .34834 L
.52699 .34821 L
.53035 .34793 L
.53345 .34759 L
.53961 .3467 L
.54627 .34541 L
.55904 .3419 L
.56973 .33786 L
.57858 .33386 L
.5881 .32936 L
.59261 .32741 L
.59674 .32585 L
.60059 .32462 L
.60488 .32354 L
.60732 .32306 L
.60961 .32268 L
.61184 .32239 L
.61428 .32215 L
.61559 .32204 L
.6168 .32197 L
.61804 .3219 L
.6192 .32186 L
.62056 .32182 L
.62129 .3218 L
.62207 .32179 L
.62281 .32179 L
.62352 .32178 L
.62485 .32179 L
.62627 .32181 L
.627 .32182 L
.62779 .32184 L
.63054 .32193 L
.6338 .32207 L
.63687 .32223 L
.6427 .32256 L
.64579 .32273 L
.64907 .3229 L
.65191 .32301 L
.65341 .32307 L
.65502 .32311 L
.65649 .32314 L
Mistroke
.65782 .32315 L
.65909 .32316 L
.66042 .32315 L
.66172 .32314 L
.6629 .32311 L
.66424 .32307 L
.6655 .32301 L
.66771 .32288 L
.67007 .32269 L
.67262 .32241 L
.67526 .32205 L
.67985 .3212 L
.68451 .32003 L
.6886 .31875 L
.69741 .31528 L
.71316 .30847 L
.7175 .30693 L
.72229 .30551 L
.72699 .30444 L
.73139 .3037 L
.73417 .30336 L
.73676 .30312 L
.73814 .30303 L
.73966 .30295 L
.74042 .30292 L
.74125 .30289 L
.74203 .30287 L
.74275 .30286 L
.74416 .30285 L
.74546 .30285 L
.7469 .30287 L
.74763 .30289 L
.74844 .30291 L
.74979 .30296 L
.75125 .30302 L
.75389 .30317 L
.75974 .30362 L
.76513 .30411 L
.77144 .30469 L
.77413 .30492 L
.777 .30512 L
.77832 .30519 L
.77969 .30526 L
.78086 .3053 L
.78212 .30533 L
.7833 .30535 L
.78454 .30535 L
.78571 .30533 L
.78675 .3053 L
.78737 .30527 L
Mistroke
.78802 .30524 L
.78917 .30516 L
.7904 .30506 L
.79173 .30491 L
.79406 .30457 L
.79618 .30415 L
.80009 .30305 L
.8022 .30225 L
.80409 .30139 L
.80773 .29931 L
.81133 .29659 L
.81715 .29039 L
.8217 .28352 L
.82611 .27463 L
.82957 .26567 L
.83596 .24395 L
.84735 .2024 L
.85343 .18476 L
.8595 .16767 L
.86071 .16372 L
.86181 .1595 L
.86222 .15761 L
.8626 .15561 L
.86288 .15375 L
.86307 .15205 L
.86315 .15118 L
.86321 .15027 L
.86325 .14936 L
.86326 .14885 L
.86327 .14837 L
.86326 .1475 L
.86324 .14671 L
.86319 .14581 L
.86312 .14495 L
.86304 .14419 L
.86294 .14339 L
.8627 .14196 L
.86236 .14043 L
.86189 .13877 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1@o`X0<ol007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1=o`802_l2037o001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`030?oo00co00<0ool0
;_l007go0P0;o`801Ol00`3oo`1:o`803ol00`3oo`0^o`00TOl00`3oo`19o`030?oo013o00<0ool0
;Ol004Ko1@16o`030?oo04Wo00<0ool04?l202ko0013o`<01Ol404;o00<0ool0B?l00`3oo`0Bo`03
0?oo02co0012o`030?oo00[o00<0ool0?ol00`3oo`18o`030?oo01;o00<0ool0;?l0047o00<0ool0
3?l203oo00<0ool0B?l00`3oo`0Bo`030?oo02co0011o`030?oo00ko00<0ool0??l00`3oo`17o`80
5?l00`3oo`0/o`00@?l00`3oo`0@o`800_l00`3oo`0go`030?oo04Oo0P0Do`030?oo02co000no`80
5Ol403So00<0ool0B?l00`3oo`0Bo`030?oo02co000mo`030?oo01Ko100go`030?oo04So00<0ool0
4Ol00`3oo`0]o`00>ol201Wo00@0o`00=ol00`3oo`18o`030?oo017o00<0ool0;Ol003Wo0P0Ko`03
0?oo03So00<0ool0B?l00`3oo`0Ao`030?oo02go000go`807Ol00`3oo`0To`801?l200?o1005o`03
0?oo04So00<0ool04Ol00`3oo`0]o`00=Ol201oo00<0ool08ol0103oo`08o`030?oo00Ko00<0ool0
B?l00`3oo`0@o`030?oo02ko000co`808Ol00`3oo`0So`040?oo00Wo00<0ool01Ol204Wo00<0ool0
4?l00`3oo`0^o`00<Ol202?o00<0ool07Ol400;o00@0ool02_l00`3oo`04o`030?oo04So00<0ool0
4?l00`3oo`0^o`00<Ol00`3oo`0Ro`030?oo02?o00@0ool02?l0103oo`05o`030?oo04So00<0ool0
4?l00`3oo`0^o`00<?l00`3oo`0So`030?oo02Co0P0:o`801_l00`3oo`18o`030?oo00oo00<0ool0
;ol002oo00<0ool09?l00`3oo`0ho`030?oo04So00<0ool03ol00`3oo`0_o`00;ol00`3oo`0To`03
0?oo03So00<0ool0B?l00`3oo`0>o`030?oo033o000_o`030?oo02Co00<0ool0>?l00`3oo`18o`03
0?oo00ko00<0ool0<?l000Oo1@02o`801?l201Wo00<0ool01ol501So00<0ool01ol200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0Go`030?oo00Wo1P0Qo`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01So00<0ool02Ol00`3oo`0Ho`030?oo00Ko00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`0Fo`030?oo00_o00@0ool08ol00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool0
5_l00`3oo`09o`030?oo01So00<0ool01_l0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`03
0?oo01?o00<0ool03?l0103oo`0So`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Wo
00@0oooo1004o`030?oo01So00<0ool01002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol201Ko
00<0ool03?l0103oo`0So`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01So00<0ool02Ol00`3o
o`0Ho`030?oo00Ko00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`0Do`030?oo00go
00<0o`009?l00`3oo`08o`030?oo00;o0008o`802_l301Wo00<0ool01ol201[o00<0ool01ol200[o
0`0Ro`030?oo02;o0P0:o`<04ol00`3oo`0=o`8000?o0?l08_l200[o0`02o`00<?l00`3oo`0So`80
>Ol00`3oo`11o`<04ol00`3oo`0bo`00<?l00`3oo`0To`<0=ol00`3oo`0eo`d05Ol00`3oo`0bo`00
<Ol00`3oo`0Vo`80=Ol00`3oo`0bo`<08Ol00`3oo`0co`00<Ol00`3oo`0Xo`80<ol00`3oo`0`o`80
9?l00`3oo`0co`00<Ol00`3oo`0Zo`80<Ol00`3oo`0_o`030?oo02Co00<0ool0<ol000coo`0<00Wo
000<o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00?o0P000ol0o`07o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`040?l000Go00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02So0P0/o`030?oo02Wo0P0Zo`03
0?oo02[o00<0ool01ol003?o00<0ool0;ol00`3oo`0Yo`030?oo02So0P0Zo`030?oo03Go000do`03
0?oo02oo00<0ool0:?l00`3oo`0Wo`030?oo02[o00<0ool0=Ol003Co00<0ool0<?l202So00<0ool0
5Ol700Oo100]o`030?oo03Go000eo`030?oo037o0P0<oa802?l00`3oo`0=o`P01ol7033o00<0ool0
=_l003Ko00<0ool0<_l<01;o2`09o`@0A_l00`3oo`0fo`00=_l00`3oo`1Ho`030?oo00T0B_l00`3o
o`0fo`00=ol205So00<0ool0D_l00`3oo`0go`00>Ol00`3oo`1Eo`030?oo05;o00<0ool0=ol003[o
0P1Eo`030?oo053o0P0jo`00??l205?o00<0ool0Col00`3oo`0jo`00?_l404oo00<0ool0C_l00`3o
o`0ko`00@_l@03oo00<0ool0>Ol;00Oo0`0no`00D_l503[o00<0ool0=_l300_o2010o`00Eol00`3o
o`0So`801?l200?o1005o`030?oo03?o0`1Fo`00F?l3027o00@0ool02?l00`3oo`06o`030?oo033o
0`1Io`00Fol201oo00@0ool02Ol00`3oo`05o`80<?l00`3oo`1Jo`00GOl00`3oo`0Lo`040?oo00[o
00<0ool01?l00`3oo`0IoaH0GOl005ko0P0Lo`040?oo00So00@0ool01Ol00`3oo`0Fo`<0Lol0063o
00<0ool06_l200[o0P06o`030?oo01?o0`1fo`00HOl00`3oo`0Eo``03?l00`3oo`0@o`<0NOl006;o
0P0Ao`@03?l200[o00<0ool03_l207co001To`@01ol601;o0P08o`030?oo00_o0`1no`00J?l701[o
1@03o`030?oo00So0`21o`00S_l>08Co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co
001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So
00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko
00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo
8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86413 .14139 m
.86276 .13797 L
.86112 .13491 L
.8589 .13172 L
.85604 .12856 L
.85249 .12556 L
.84865 .1231 L
.84483 .12124 L
.84032 .11964 L
.83779 .11898 L
.83535 .11848 L
.83394 .11825 L
.8326 .11808 L
.83103 .11792 L
.83027 .11785 L
.82958 .11781 L
.82884 .11777 L
.82814 .11775 L
.8274 .11773 L
.8266 .11772 L
.8259 .11772 L
.82515 .11773 L
.82382 .11778 L
.82306 .11782 L
.82226 .11788 L
.8208 .11801 L
.81906 .11821 L
.81748 .11844 L
.81473 .11896 L
.81221 .11955 L
.80669 .12127 L
.80211 .12321 L
.79762 .12565 L
.79398 .12813 L
.79061 .13099 L
.78748 .13438 L
.78515 .13768 L
.78344 .14091 L
.78222 .14401 L
.78168 .14584 L
.78127 .14753 L
.7809 .1495 L
.78077 .15041 L
.78064 .1514 L
.78054 .1524 L
.78046 .15332 L
.7804 .15426 L
.78035 .15514 L
.78032 .15617 L
.7803 .15672 L
Mistroke
.78029 .15731 L
.78028 .1584 L
.78028 .1594 L
.78029 .15995 L
.78029 .16053 L
.78031 .16158 L
.78034 .16272 L
.78038 .16399 L
.78053 .16846 L
.7806 .17064 L
.78063 .17172 L
.78065 .17273 L
.78066 .1739 L
.78067 .17452 L
.78067 .17518 L
.78068 .17625 L
.78067 .17743 L
.78065 .17855 L
.78062 .17958 L
.78058 .18072 L
.78053 .18192 L
.78041 .18396 L
.78023 .18621 L
.77995 .18876 L
.77957 .19153 L
.77864 .19654 L
.77742 .20134 L
.77602 .20563 L
.77419 .21007 L
.77181 .21461 L
.76894 .21889 L
.76591 .22238 L
.7627 .22525 L
.75887 .2278 L
.75496 .2297 L
.75109 .23103 L
.74742 .23194 L
.74317 .23269 L
.73852 .23328 L
.73397 .23373 L
.7239 .2348 L
.71868 .23559 L
.7129 .23676 L
.70796 .23807 L
.70271 .23982 L
.69349 .24383 L
.67456 .25532 L
.66541 .26127 L
.66049 .26407 L
.65585 .26629 L
Mistroke
.65165 .2679 L
.64699 .26924 L
.64427 .26982 L
.64162 .27026 L
.63918 .27058 L
.63779 .27072 L
.63647 .27084 L
.63523 .27093 L
.63388 .27101 L
.63275 .27106 L
.63149 .27111 L
.63011 .27115 L
.6294 .27117 L
.62864 .27119 L
.62727 .2712 L
.62597 .27122 L
.62466 .27122 L
.62323 .27122 L
.62186 .27121 L
.62059 .27121 L
.61816 .27119 L
.61557 .27117 L
.61423 .27116 L
.61276 .27116 L
.61142 .27115 L
.61017 .27116 L
.60868 .27116 L
.60733 .27118 L
.60658 .27119 L
.60576 .2712 L
.60429 .27123 L
.60159 .27131 L
.60023 .27136 L
.59874 .27142 L
.59365 .27172 L
.59053 .27197 L
.58771 .27224 L
.58147 .27299 L
.57065 .27471 L
.56038 .2766 L
.55493 .27756 L
.54986 .27834 L
.54512 .27894 L
.54262 .2792 L
.53996 .27942 L
.5377 .27956 L
.5365 .27963 L
.53521 .27969 L
.53401 .27973 L
.53292 .27976 L
Mistroke
.53182 .27979 L
.53077 .2798 L
.52959 .27981 L
.52849 .27982 L
.52727 .27981 L
.52595 .2798 L
.52457 .27978 L
.52325 .27975 L
.52075 .27968 L
.51621 .2795 L
.5112 .27924 L
.48964 .27787 L
.47862 .27716 L
.46912 .27646 L
.46123 .27572 L
.45302 .27478 L
.4448 .27378 L
.44008 .27328 L
.4377 .27306 L
.43547 .27289 L
.43352 .27276 L
.43135 .27265 L
.43015 .2726 L
.42902 .27256 L
.42792 .27253 L
.42675 .27251 L
.42557 .27249 L
.42429 .27249 L
.42306 .27249 L
.42192 .27251 L
.42066 .27253 L
.41926 .27257 L
.41779 .27263 L
.41641 .27269 L
.41353 .27285 L
.41085 .27304 L
.40474 .2736 L
.3937 .2749 L
.38872 .2755 L
.38412 .276 L
.38165 .27622 L
.38036 .27631 L
.37901 .2764 L
.37777 .27646 L
.37666 .2765 L
.37555 .27652 L
.37452 .27653 L
.3734 .27652 L
.37223 .27649 L
.37124 .27645 L
Mistroke
.37017 .27638 L
.36902 .27628 L
.36781 .27614 L
.36565 .27582 L
.36351 .27539 L
.36153 .27488 L
.35785 .27367 L
.35354 .27178 L
.34953 .26963 L
.34017 .26363 L
.30167 .2414 L
.29329 .23785 L
.28937 .23604 L
.28621 .23429 L
.28389 .23265 L
.28183 .23067 L
.28084 .22941 L
.28002 .22811 L
.2794 .22686 L
.27881 .22538 L
.27832 .22378 L
.27791 .22195 L
.27776 .22109 L
.27762 .22014 L
.27741 .21833 L
.27732 .21719 L
.27728 .21658 L
.27725 .21592 L
.27722 .21533 L
.2772 .21467 L
.27717 .21349 L
.27716 .21231 L
.27717 .21101 L
.27719 .20976 L
.27722 .20859 L
.27726 .20729 L
.27732 .20587 L
.27746 .20327 L
.27789 .19715 L
.27912 .18463 L
.28062 .1738 L
.28162 .16871 L
.28214 .16661 L
.28267 .16485 L
.28369 .16222 L
.28427 .16112 L
.28493 .16015 L
.28527 .15975 L
.28564 .15937 L
.28598 .15909 L
Mistroke
.28635 .15885 L
.28672 .15866 L
.28706 .15853 L
.2874 .15844 L
.28773 .15839 L
.28806 .15838 L
.28841 .15839 L
.28874 .15844 L
.28904 .1585 L
.28939 .1586 L
.28976 .15874 L
.29042 .15907 L
.29108 .15948 L
.29172 .15998 L
.29221 .16044 L
.29266 .16094 L
.293 .16138 L
.29322 .16175 L
.29331 .16192 L
.29336 .16207 L
.2934 .16221 L
.29341 .16235 L
.2934 .16249 L
.29335 .1626 L
.29328 .16268 L
.29317 .16274 L
.29301 .16278 L
.29292 .16279 L
.2928 .1628 L
.29257 .16278 L
.29231 .16273 L
.29199 .16266 L
.29159 .16254 L
.29076 .16223 L
.28826 .16109 L
.28152 .15753 L
.26275 .14788 L
.25057 .14282 L
.24373 .14063 L
.24055 .1398 L
.23765 .13918 L
.23516 .13875 L
.2339 .13858 L
.23255 .13842 L
.23179 .13835 L
.2311 .13829 L
.22975 .13821 L
.22854 .13816 L
.22728 .13815 L
.22607 .13818 L
Mistroke
.2248 .13825 L
.22362 .13834 L
.22255 .13846 L
.22133 .13863 L
.22003 .13886 L
.21771 .13939 L
.21557 .14003 L
.21332 .14085 L
.20941 .14264 L
.20247 .14681 L
.18641 .15781 L
.16919 .16776 L
.15981 .17276 L
.15051 .17815 L
.14306 .18358 L
.13791 .18888 L
.13603 .19159 L
.13441 .19466 L
.13382 .19611 L
.13328 .19768 L
.13289 .19906 L
.13254 .2006 L
.13227 .20221 L
.13216 .20304 L
.13206 .20394 L
.13198 .20479 L
.13193 .20557 L
.13189 .20636 L
.13187 .20712 L
.13186 .20802 L
.13186 .20897 L
.13188 .2098 L
.13192 .21071 L
.13197 .21171 L
.13205 .21279 L
.13223 .21479 L
.13276 .21874 L
.13344 .22265 L
.13521 .23032 L
.13791 .23918 L
.14175 .24888 L
.14635 .25796 L
.15137 .26584 L
.15756 .27365 L
.16489 .28092 L
.17362 .28755 L
.1821 .29239 L
.19046 .29595 L
.19516 .29753 L
.19968 .2988 L
Mistroke
.20865 .30074 L
.21705 .30202 L
.2266 .30306 L
.23697 .30392 L
.24681 .3047 L
.25581 .3056 L
.26044 .30619 L
.26528 .30696 L
.27048 .30798 L
.27525 .30914 L
.28368 .31179 L
.29272 .31555 L
.31035 .32498 L
.33 .33558 L
.3354 .3379 L
.34121 .33997 L
.34624 .34137 L
.34916 .34201 L
.35185 .34249 L
.35458 .34287 L
.35718 .34314 L
.35862 .34325 L
.35995 .34333 L
.36072 .34336 L
.36142 .34339 L
.36222 .34341 L
.36298 .34342 L
.36427 .34343 L
.36564 .34342 L
.3668 .34339 L
.36808 .34334 L
.36947 .34327 L
.37096 .34317 L
.37365 .34295 L
.37956 .34227 L
.38506 .34146 L
.3976 .33932 L
.40327 .33837 L
.40934 .33746 L
.41454 .33681 L
.41745 .33652 L
.42014 .3363 L
.42278 .33613 L
.42424 .33606 L
.42556 .33601 L
.42682 .33597 L
.42817 .33595 L
.42945 .33594 L
.43061 .33594 L
.43173 .33596 L
Mistroke
.43277 .33598 L
.4339 .33602 L
.43511 .33607 L
.43755 .33622 L
.43975 .3364 L
.44473 .33696 L
.44999 .33779 L
.45924 .33975 L
.4769 .34418 L
.48196 .34532 L
.48764 .34643 L
.49275 .34726 L
.49553 .34764 L
.49858 .34799 L
.50158 .34828 L
.50437 .3485 L
.50573 .34858 L
.50721 .34866 L
.50862 .34872 L
.50994 .34877 L
.51143 .34881 L
.51223 .34882 L
.51308 .34883 L
.51398 .34884 L
.51484 .34884 L
.51561 .34884 L
.51645 .34883 L
.51798 .34881 L
.51884 .34879 L
.51962 .34877 L
.52105 .34872 L
.52257 .34865 L
.52554 .34847 L
.52827 .34825 L
.53443 .34759 L
.53971 .34682 L
.5454 .34576 L
.55709 .34284 L
.56695 .33958 L
.58661 .3316 L
.59481 .32849 L
.60373 .32589 L
.60858 .32488 L
.61123 .32444 L
.61413 .32403 L
.61685 .32372 L
.61938 .32349 L
.62203 .3233 L
.62447 .32316 L
.6271 .32305 L
Mistroke
.62962 .32297 L
.63229 .32291 L
.63381 .32288 L
.63522 .32286 L
.6405 .3228 L
.64309 .32277 L
.64545 .32274 L
.64804 .32269 L
.65081 .32262 L
.65381 .32252 L
.65663 .32238 L
.6595 .3222 L
.66213 .32198 L
.66795 .32133 L
.67264 .32058 L
.67753 .31957 L
.68598 .31722 L
.7042 .31034 L
.71355 .30699 L
.72274 .30449 L
.72758 .30354 L
.73292 .30279 L
.73603 .30248 L
.73757 .30235 L
.739 .30226 L
.74028 .30219 L
.7417 .30212 L
.74313 .30207 L
.74387 .30205 L
.74465 .30203 L
.74605 .30201 L
.74736 .302 L
.74816 .302 L
.74891 .302 L
.75032 .30201 L
.75167 .30203 L
.75292 .30206 L
.75573 .30213 L
.76065 .30231 L
.7658 .30252 L
.76846 .30262 L
.76991 .30266 L
.77127 .3027 L
.77253 .30273 L
.7739 .30275 L
.77513 .30276 L
.77626 .30276 L
.77758 .30276 L
.77881 .30274 L
.77951 .30272 L
Mistroke
.78015 .3027 L
.78158 .30264 L
.78286 .30258 L
.78425 .30248 L
.787 .30222 L
.78937 .30191 L
.79187 .30149 L
.79624 .30045 L
.80029 .29909 L
.80384 .29751 L
.80748 .29546 L
.81125 .29279 L
.81817 .28616 L
.8239 .27857 L
.82853 .27076 L
.83728 .25114 L
.85091 .21114 L
.85742 .1918 L
.86256 .17554 L
.86367 .17137 L
.86468 .16693 L
.86506 .16492 L
.86541 .16281 L
.86568 .16085 L
.86587 .15908 L
.86596 .15803 L
.86602 .15705 L
.86607 .15601 L
.86609 .15543 L
.8661 .15489 L
.86611 .154 L
.8661 .15302 L
.86606 .152 L
.86601 .15105 L
.86597 .15052 L
.86592 .14993 L
.86581 .14888 L
.86553 .14691 L
.86506 .14464 L
.86451 .14256 L
.86413 .14139 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1Ao`P0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1=o`<02?l203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`1<o`030?oo00_o00<0ool0
;ol007go0P0;o`801Ol00`3oo`1:o`803ol00`3oo`0^o`00TOl00`3oo`19o`030?oo013o0P0^o`00
A_l504Ko00<0ool0B?l00`3oo`0Bo`030?oo02co0013o`<01Ol404;o00<0ool0B?l00`3oo`0Bo`03
0?oo02co0012o`030?oo00[o0P10o`030?oo04Oo0P0Eo`030?oo02_o0011o`030?oo00go0P0no`03
0?oo04Oo00<0ool05?l00`3oo`0[o`00?ol201;o0P0lo`030?oo04Oo00<0ool05?l00`3oo`0[o`00
?Ol201Ko0`0io`030?oo04Oo00<0ool05?l00`3oo`0[o`00??l00`3oo`0Ho`@0=_l00`3oo`17o`03
0?oo01Co00<0ool0:ol003[o0P0Ko`040?l003Ko00<0ool0Aol00`3oo`0Do`030?oo02_o000ho`80
7?l203Wo00<0ool0Aol00`3oo`0Do`030?oo02_o000fo`807_l00`3oo`0ho`030?oo04Oo00<0ool0
4ol00`3oo`0/o`00=Ol00`3oo`0No`030?oo02Co0P04o`800ol400Go00<0ool0Aol00`3oo`0Co`03
0?oo02co000do`030?oo01oo00<0ool08ol0103oo`08o`030?oo00Ko00<0ool0Aol00`3oo`0Co`03
0?oo02co000bo`808_l00`3oo`0So`040?oo00Wo00<0ool01Ol204So00<0ool04ol00`3oo`0/o`00
<Ol00`3oo`0Ro`030?oo01go1002o`040?oo00[o00<0ool01?l00`3oo`17o`030?oo01;o00<0ool0
;Ol0033o00<0ool08ol00`3oo`0So`040?oo00So00@0ool01Ol00`3oo`17o`030?oo01;o00<0ool0
;Ol002oo00<0ool08ol00`3oo`0Uo`802_l200Ko00<0ool0Aol00`3oo`0Bo`030?oo02go000_o`03
0?oo02?o00<0ool0>Ol00`3oo`17o`030?oo01;o00<0ool0;Ol002oo00<0ool08ol00`3oo`0io`03
0?oo04Ko00<0ool04_l00`3oo`0^o`00;ol00`3oo`0So`030?oo03Wo00<0ool0A_l00`3oo`0Bo`03
0?oo02ko0007o`D00_l200Co0P0Io`030?oo00Oo1@0Go`030?oo00So0P04o`801?l202?o00<0ool0
8_l200Co0P04o`805Ol00`3oo`0;o`D00_l00`3oo`0Mo`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01So00<0ool02Ol00`3oo`0Go`030?oo00Oo00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`0Do`030?oo00go00<0ool00_l00`3oo`0Oo`030?oo00Oo00@0ool00_l000Wo00<0ool0
2_l00`3oo`0Fo`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00_o00<0ool08?l00`3oo`0Qo`04
0?oo00_o00<0ool04_l00`3oo`0=o`030?oo00;o00<0ool07ol00`3oo`0:o`400Ol1o`0000?o0000
0P04o`030?oo00So0P0Io`040?ooo`@01?l00`3oo`0Go`040?ooo`@00_l0103oo`09o`808ol00`3o
o`0Qo`040?oo00Wo0P0Do`030?oo00ko00D0oooo000Ro`030?oo00So0P03o`002Ol00`3oo`08o`03
0?oo01So00<0ool02Ol00`3oo`0Go`802?l0103oo`09o`030?oo02;o00<0ool08Ol0103oo`09o`03
0?oo017o0P0Ao`050?ooo`008_l00`3oo`08o`030?oo00;o0008o`802_l301So00<0ool02?l201[o
00<0ool01ol200[o0`0Ro`030?oo02;o0P0:o`<04?l00`3oo`0@o`800_l00`3oo`0Po`802_l300;o
000_o`030?oo02Co0P0io`030?oo03go100Go`030?oo037o000`o`030?oo02Go0P0go`030?oo03Go
200Ko`030?oo037o000`o`030?oo02Oo0P0eo`030?oo03;o0`0Ro`030?oo03;o000`o`030?oo02Wo
00<0ool0<_l00`3oo`0`o`809Ol00`3oo`0bo`00<Ol00`3oo`0Yo`80<_l00`3oo`0^o`809_l00`3o
o`0co`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool00ol0103oo`08o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?l000Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l200Oo00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`07o`003?l00`3oo`0So`030?oo00?o00<0ool09ol302co00<0ool0
:Ol202[o00<0ool0:_l00`3oo`07o`00<ol00`3oo`0^o`80:ol00`3oo`0Yo`030?oo02Wo00<0ool0
=?l003Co00<0ool0;ol202Wo00<0ool09_l302co00<0ool0=?l003Go00<0ool0<?l202Oo00<0ool0
8_l402ko00<0ool0=Ol003Go00<0ool0<_l200[o3P0=o`030?oo017o4@0bo`030?oo03Go000fo`03
0?oo03?o2P0>o`h03Ol604?o00<0ool0=Ol003Oo00<0ool0Eol>04So00<0ool0=_l003So0P1Go`03
0?oo05;o00<0ool0=ol003[o00<0ool0E?l00`3oo`1@o`80>_l003_o0P1Do`030?oo04oo00<0ool0
>_l003go0P1Bo`030?oo04ko00<0ool0>ol003oo1@1=o`030?oo04_o0`0no`00A?l804Go00<0ool0
=olD047o001<o`P0?Ol00`3oo`0do`<0EOl005Co100Uo`801?l200?o1005o`030?oo03;o0P1Ho`00
F?l202;o00@0ool02?l00`3oo`06o`030?oo033o0P1Jo`00F_l2023o00@0ool02Ol00`3oo`05o`80
;_l305co001Lo`807_l0103oo`0:o`030?oo00Co00<0ool07Ol@05oo001No`030?oo01_o00@0ool0
2?l0103oo`05o`030?oo01Oo1P1_o`00Gol201co0P0:o`801_l00`3oo`0Eo`80MOl0067o0P0^o`03
0?oo01?o0P1go`00Hol301;o3@0<o`030?oo013o0`1io`00I_l300So1`0=o`<02Ol00`3oo`0=o`<0
O?l006Wo200Go`D01?l00`3oo`08o`D0Ool008go3`24o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<0
1?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P2=o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02Ol00`3o0005o`030?oo08co001mo`80
2ol200Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0
S?l007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001lo`04
0?oo00So0`06o`030?oo08co001lo`040?oo00Wo00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co003oob7o003oob7o003oob7o003oob7o003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86638 .14401 m
.86488 .14039 L
.86315 .13715 L
.86088 .13376 L
.85801 .13038 L
.85451 .12715 L
.85075 .12447 L
.84703 .12239 L
.84266 .12053 L
.83784 .11909 L
.83499 .11848 L
.83225 .11806 L
.83075 .1179 L
.83001 .11783 L
.82931 .11778 L
.82802 .1177 L
.82733 .11767 L
.8266 .11765 L
.8258 .11764 L
.82493 .11764 L
.82411 .11765 L
.82336 .11767 L
.82268 .1177 L
.82193 .11774 L
.82042 .11785 L
.81899 .11799 L
.81764 .11815 L
.81512 .11854 L
.81243 .11908 L
.80957 .11979 L
.8046 .12139 L
.80027 .12321 L
.79631 .1253 L
.79244 .12784 L
.78661 .13307 L
.78396 .13643 L
.78173 .14019 L
.78024 .14359 L
.77902 .14748 L
.77855 .14946 L
.77813 .15167 L
.77782 .15366 L
.77755 .15582 L
.77717 .16003 L
.77691 .16406 L
.77636 .17364 L
.77615 .1765 L
.7759 .17924 L
.77529 .18445 L
.7745 .18935 L
Mistroke
.77356 .19386 L
.77226 .1987 L
.77056 .20367 L
.76864 .20808 L
.76655 .21196 L
.76191 .21833 L
.75904 .22124 L
.7556 .224 L
.75169 .22643 L
.74777 .22832 L
.74016 .23096 L
.7313 .23317 L
.72185 .23534 L
.71152 .23818 L
.70019 .24238 L
.69002 .24729 L
.67103 .2583 L
.66211 .26309 L
.6582 .26488 L
.65387 .26656 L
.64995 .26783 L
.64566 .26892 L
.64101 .26982 L
.63655 .27045 L
.63398 .27072 L
.63114 .27096 L
.62588 .27129 L
.62062 .27152 L
.61568 .27171 L
.61042 .27191 L
.60474 .27219 L
.59936 .27254 L
.59446 .27294 L
.5827 .27429 L
.57196 .27592 L
.55344 .27891 L
.54846 .27955 L
.54308 .2801 L
.54059 .28031 L
.53795 .28049 L
.53546 .28062 L
.53319 .28072 L
.53075 .28079 L
.5294 .28082 L
.52813 .28084 L
.52694 .28085 L
.52584 .28086 L
.52463 .28086 L
.52335 .28086 L
.52205 .28085 L
Mistroke
.52066 .28084 L
.51933 .28082 L
.51812 .2808 L
.51515 .28073 L
.51231 .28065 L
.50149 .28023 L
.48991 .27968 L
.4797 .27911 L
.47064 .27849 L
.46077 .27761 L
.4521 .27664 L
.4435 .27563 L
.43864 .27511 L
.43396 .27469 L
.42985 .2744 L
.42769 .27428 L
.42531 .27417 L
.42394 .27412 L
.42267 .27408 L
.42144 .27405 L
.42013 .27403 L
.41895 .27401 L
.41784 .27401 L
.41662 .27401 L
.41529 .27401 L
.41395 .27403 L
.41273 .27405 L
.41132 .27407 L
.40999 .27411 L
.405 .27429 L
.39372 .27485 L
.38852 .27509 L
.38595 .27518 L
.38524 .27519 L
.3845 .27521 L
.38315 .27523 L
.38241 .27524 L
.3816 .27524 L
.38084 .27524 L
.38014 .27524 L
.37883 .27522 L
.3781 .2752 L
.37745 .27518 L
.37617 .27512 L
.37481 .27504 L
.37237 .27485 L
.37013 .27459 L
.36777 .27424 L
.3636 .27338 L
.35876 .27196 L
Mistroke
.35442 .27029 L
.34499 .26556 L
.33579 .26009 L
.29917 .24059 L
.29071 .23654 L
.28423 .2325 L
.28159 .23017 L
.27925 .22734 L
.27815 .22555 L
.27726 .22371 L
.27658 .22195 L
.27597 .21989 L
.27553 .21787 L
.27534 .21677 L
.27517 .21556 L
.27505 .21448 L
.27494 .21327 L
.27487 .21214 L
.27482 .21106 L
.2748 .20992 L
.27479 .20881 L
.2748 .20781 L
.27483 .2067 L
.27487 .20548 L
.27495 .20417 L
.27513 .20178 L
.27539 .1992 L
.27574 .19639 L
.27657 .19129 L
.27782 .18535 L
.27919 .18018 L
.28057 .17589 L
.2822 .17178 L
.28369 .16869 L
.28538 .16586 L
.28742 .16325 L
.28836 .1623 L
.28934 .16146 L
.2902 .16085 L
.29113 .16031 L
.29194 .15993 L
.29265 .15967 L
.29338 .15948 L
.29374 .1594 L
.29411 .15934 L
.29427 .15932 L
.29443 .15931 L
.29472 .15928 L
.29497 .15927 L
.29519 .15927 L
Mistroke
.29538 .15927 L
.29554 .15927 L
.2957 .15928 L
.29585 .15929 L
.29596 .1593 L
.29605 .15931 L
.2961 .15932 L
.29613 .15932 L
.29614 .15931 L
.29613 .15931 L
.29609 .15929 L
.29585 .15921 L
.29487 .15886 L
.29105 .15729 L
.27599 .15068 L
.26501 .14623 L
.25852 .14391 L
.25229 .14197 L
.24633 .14043 L
.24324 .13977 L
.24002 .1392 L
.23736 .13883 L
.23597 .13867 L
.2345 .13854 L
.23316 .13844 L
.23195 .13838 L
.23075 .13834 L
.22962 .13832 L
.22852 .13833 L
.22732 .13836 L
.22608 .13842 L
.22492 .1385 L
.22387 .13859 L
.22291 .1387 L
.22078 .139 L
.21867 .13938 L
.21642 .1399 L
.21204 .14121 L
.20368 .14472 L
.19621 .14861 L
.17847 .15827 L
.15974 .16824 L
.15203 .17288 L
.14457 .17834 L
.1416 .18101 L
.1388 .184 L
.13671 .18669 L
.13479 .18973 L
.13309 .19329 L
.13246 .19498 L
Mistroke
.13189 .19681 L
.13144 .19866 L
.13111 .20037 L
.13098 .2012 L
.13085 .20211 L
.13076 .20297 L
.13069 .20379 L
.13062 .20474 L
.13058 .20561 L
.13055 .20656 L
.13055 .2071 L
.13055 .2076 L
.13055 .20844 L
.13058 .20937 L
.13062 .21035 L
.13068 .21127 L
.13087 .21342 L
.1311 .21541 L
.13185 .22001 L
.13278 .22427 L
.13405 .22904 L
.13742 .2389 L
.14163 .24837 L
.14632 .25682 L
.15283 .26617 L
.15974 .27402 L
.16666 .28033 L
.17483 .28625 L
.18348 .29113 L
.193 .29522 L
.20233 .29824 L
.21106 .3004 L
.22138 .30238 L
.23138 .30392 L
.25145 .30668 L
.26016 .30809 L
.26946 .30998 L
.27932 .31262 L
.28845 .31577 L
.30829 .32473 L
.3271 .33388 L
.33808 .33824 L
.34344 .33992 L
.34857 .34121 L
.35351 .34216 L
.35632 .34258 L
.35893 .34289 L
.36114 .34309 L
.36358 .34326 L
.36482 .34332 L
Mistroke
.36615 .34337 L
.36741 .3434 L
.36856 .34341 L
.36991 .34342 L
.37066 .34342 L
.37135 .34341 L
.37268 .34338 L
.37394 .34335 L
.37542 .3433 L
.37678 .34323 L
.37987 .34305 L
.38517 .34263 L
.3908 .34209 L
.40125 .34099 L
.40699 .34044 L
.41241 .34 L
.41483 .33984 L
.4174 .3397 L
.41873 .33964 L
.42019 .33958 L
.42153 .33954 L
.42276 .33951 L
.42389 .33949 L
.42507 .33948 L
.42625 .33947 L
.42753 .33947 L
.42864 .33948 L
.42983 .3395 L
.43094 .33953 L
.43195 .33956 L
.43432 .33966 L
.4369 .3398 L
.43957 .34 L
.44204 .34022 L
.451 .34132 L
.46049 .34289 L
.47061 .34477 L
.48033 .34647 L
.48565 .34726 L
.49148 .34799 L
.49445 .34829 L
.49728 .34854 L
.50008 .34874 L
.50269 .34889 L
.50521 .349 L
.50649 .34904 L
.5079 .34908 L
.50932 .34911 L
.51013 .34912 L
.51086 .34912 L
Mistroke
.51229 .34913 L
.51365 .34912 L
.51497 .3491 L
.5164 .34907 L
.51775 .34904 L
.519 .34899 L
.52189 .34886 L
.52495 .34866 L
.53073 .34815 L
.53701 .34737 L
.54236 .34652 L
.54796 .34544 L
.55769 .34308 L
.57761 .33656 L
.58729 .33303 L
.59742 .32969 L
.60671 .32727 L
.61218 .32617 L
.61743 .32531 L
.62698 .32414 L
.63757 .32318 L
.64848 .32223 L
.65418 .32162 L
.66015 .32085 L
.6707 .31898 L
.68001 .31667 L
.69835 .31066 L
.70799 .30745 L
.71338 .30586 L
.71853 .30456 L
.72369 .30347 L
.72849 .30265 L
.73393 .30194 L
.73697 .30163 L
.73986 .30139 L
.74233 .30123 L
.74496 .30109 L
.74723 .30099 L
.74969 .3009 L
.75232 .30084 L
.75478 .30078 L
.76037 .3007 L
.76306 .30065 L
.7659 .3006 L
.76855 .30053 L
.77094 .30045 L
.77373 .30032 L
.7767 .30014 L
.77946 .29991 L
.78196 .29965 L
Mistroke
.78639 .29902 L
.7911 .29809 L
.79582 .2968 L
.80005 .2953 L
.80412 .29348 L
.80834 .29115 L
.81551 .28599 L
.82189 .27991 L
.82729 .27349 L
.83723 .25811 L
.84646 .23916 L
.85452 .219 L
.86093 .20073 L
.86595 .18358 L
.86776 .17525 L
.86814 .173 L
.86848 .1706 L
.86873 .16851 L
.86893 .1663 L
.86903 .1649 L
.86907 .16411 L
.8691 .16336 L
.86912 .16268 L
.86914 .16206 L
.86914 .16135 L
.86914 .16067 L
.86914 .15994 L
.86912 .15916 L
.86909 .15843 L
.86906 .15777 L
.86901 .15699 L
.86895 .15615 L
.8688 .15463 L
.86864 .15332 L
.86842 .15189 L
.86785 .14903 L
.86717 .14641 L
.86638 .14401 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1@o`T0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1<o`<02Ol203;o001lo`040?oo00Wo00<0o`001Ol00`3oo`1;o`030?oo00co0P0`o`00
OOl200_o0P05o`030?oo04[o00<0ool03ol00`3oo`0]o`00TOl00`3oo`19o`030?oo013o00<0ool0
;Ol004Ko1P15o`030?oo04So00<0ool04_l00`3oo`0/o`00A?l300Co1@11o`030?oo04Oo00<0ool0
5?l00`3oo`0[o`00@_l200co0`0no`030?oo04Oo00<0ool05?l00`3oo`0[o`00@?l2017o0P0lo`03
0?oo04Oo00<0ool05?l00`3oo`0[o`00?_l201Go0`0io`030?oo04Ko00<0ool05_l00`3oo`0Zo`00
??l201[o0P0go`030?oo04Ko00<0ool05_l00`3oo`0Zo`00>_l201co100eo`030?oo04Ko00<0ool0
5_l00`3oo`0Zo`00>?l201ko00<0ool0=_l00`3oo`16o`030?oo01Ko00<0ool0:_l003Ko0P0Oo`03
0?oo03Oo00<0ool0A_l00`3oo`0Fo`030?oo02[o000do`808Ol00`3oo`0go`030?oo04Ko00<0ool0
5_l00`3oo`0Zo`00<ol00`3oo`0Po`030?oo02Co0P04o`800ol400Go00<0ool0A_l00`3oo`0Eo`03
0?oo02_o000bo`030?oo027o00<0ool08ol0103oo`08o`030?oo00Ko00<0ool0A_l00`3oo`0Eo`03
0?oo02_o000`o`809?l00`3oo`0So`040?oo00Wo00<0ool01Ol204Oo00<0ool05Ol00`3oo`0[o`00
<?l00`3oo`0Ro`030?oo01ko1002o`040?oo00[o00<0ool01?l00`3oo`16o`030?oo01Go00<0ool0
:ol002oo00<0ool08ol00`3oo`0To`040?oo00So00@0ool01Ol00`3oo`16o`030?oo01Go00<0ool0
:ol002oo00<0ool08ol00`3oo`0Uo`802_l200Ko00<0ool0AOl00`3oo`0Eo`030?oo02co000^o`80
9Ol00`3oo`0io`030?oo04Go00<0ool05Ol00`3oo`0/o`00;_l00`3oo`0To`030?oo03Wo00<0ool0
A?l00`3oo`0Fo`030?oo02co000^o`030?oo02Co00<0ool0>Ol00`3oo`14o`030?oo01Ko00<0ool0
;?l000Oo1@02o`801?l201So00<0ool02?l501Oo00<0ool02?l200Co0P04o`808ol00`3oo`0Ro`80
1?l200Co0P0Do`030?oo00co1@04o`030?oo01_o1@02o`801?l200?o0009o`030?oo00Oo00@0ool0
5ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`04
0?oo01;o00<0ool03ol00`3oo`04o`030?oo01go00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`03
0?oo01Go00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`0?o`030?oo013o00<0ool00ol00`3oo`0No`030?oo00[o0@01o`7o00000ol0000200Co
00<0ool02?l201Wo00@0oooo1004o`030?oo01Oo00@0oooo1002o`040?oo00Wo0P0So`030?oo027o
00@0ool02Ol2017o0P0Bo`030?oo00?o00<0ool07_l00`3oo`08o`800ol000Wo00<0ool02?l00`3o
o`0Ho`030?oo00Wo00<0ool05ol00`3oo`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo
00<0ool03ol00`3oo`0Bo`030?oo00?o00<0ool07_l00`3oo`08o`030?oo00;o0008o`802_l301So
00<0ool02?l201[o00<0ool01ol200[o0`0Ro`030?oo02;o0P0:o`<03Ol201Co0P04o`030?oo01ko
0P0:o`<00_l002oo00<0ool09?l303So00<0ool0??l201co00<0ool0;ol0033o00<0ool09_l203Ko
00<0ool0=ol501go00<0ool0<?l0033o00<0ool0:?l00`3oo`0co`030?oo03Co0`0Ro`030?oo033o
000`o`030?oo02Wo0P0co`030?oo037o0`0To`030?oo037o000ao`030?oo02[o0P0ao`030?oo02oo
0P0Wo`030?oo037o000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`04
0?oo00Go00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`03o`8000?o0?l01ol00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l0103o0005o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0o`001_l00`3o
o`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02?o00<0ool00ol00`3o
o`0Xo`80;?l00`3oo`0Yo`80:_l202_o00<0ool01ol003?o00<0ool0;ol202[o00<0ool0:?l202co
00<0ool0<ol003Co00<0ool0<?l202So00<0ool09Ol302go00<0ool0=?l003Go00<0ool0<Ol202Ko
00<0ool08Ol4033o00<0ool0=?l003Ko00<0ool0<_l300So2@0Bo`030?oo01;o3`0co`030?oo03Go
000go`030?oo03Co2009o`T02Ol00`3oo`0>o`@0@Ol00`3oo`0fo`00>?l00`3oo`1=oaX0A?l00`3o
o`0go`00>Ol00`3oo`1Eo`030?oo057o00<0ool0>?l003[o00<0ool0E?l00`3oo`1@o`030?oo03Wo
000ko`<0Dol00`3oo`1>o`80??l003ko0P1Ao`030?oo04_o0`0no`00@?l504co00<0ool0@Ol:047o
0015o`L0AOl00`3oo`0ho`T0Bol004co1P0oo`030?oo03?o1@1Do`00D_l302So0P04o`800ol400Go
00<0ool0<?l305Wo001Eo`@08ol0103oo`08o`030?oo00Ko00<0ool0;_l205co001Io`808Ol0103o
o`09o`030?oo00Go0P0Zo`D0G_l005_o0P0Oo`040?oo00[o00<0ool01?l00`3oo`0Po`T0Hol005go
0`0Lo`040?oo00So00@0ool01Ol00`3oo`0Jo`H0K?l0063o0P0Ko`802_l200Ko00<0ool05Ol507;o
001Ro`80;Ol00`3oo`0Co`80Mol006Co0P0[o`030?oo013o0`1io`00I_l300[o4P0<o`030?oo00ko
0P1lo`00JOl:01;o1@07o`030?oo00Wo1@1no`00R_lC08?o002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.86863 .14663 m
.867 .14282 L
.86519 .13938 L
.86265 .1355 L
.85998 .1322 L
.85346 .12627 L
.84928 .12357 L
.845 .12143 L
.8403 .11968 L
.83783 .11898 L
.83505 .11834 L
.83263 .11792 L
.83128 .11773 L
.82998 .11757 L
.82878 .11746 L
.82746 .11737 L
.82626 .11731 L
.82513 .11727 L
.82439 .11726 L
.8237 .11726 L
.82295 .11726 L
.82215 .11728 L
.8208 .11733 L
.82004 .11737 L
.81933 .11742 L
.8168 .11765 L
.81538 .11782 L
.8141 .11801 L
.81126 .11851 L
.80823 .11919 L
.80291 .12078 L
.79849 .12252 L
.79463 .12442 L
.79043 .12699 L
.78699 .12963 L
.78405 .13244 L
.78129 .13577 L
.77899 .13939 L
.77703 .14348 L
.77558 .14761 L
.77451 .15174 L
.77374 .15564 L
.77307 .16004 L
.77193 .16959 L
.77131 .17487 L
.77054 .18061 L
.76962 .18603 L
.76857 .19093 L
.7658 .20013 L
.76387 .20474 L
Mistroke
.76139 .20942 L
.75583 .21686 L
.75244 .22016 L
.74894 .22291 L
.74171 .22725 L
.73308 .23108 L
.72307 .23479 L
.68411 .25205 L
.6751 .25684 L
.66553 .26153 L
.65719 .26493 L
.64789 .26777 L
.64276 .26893 L
.63792 .2698 L
.62807 .27109 L
.6188 .27195 L
.60875 .27281 L
.58764 .27508 L
.56843 .27788 L
.55835 .27934 L
.54782 .28061 L
.54346 .28102 L
.53878 .28137 L
.53607 .28154 L
.53361 .28167 L
.52877 .28187 L
.52634 .28194 L
.52367 .28199 L
.52232 .28201 L
.52087 .28203 L
.51949 .28204 L
.51824 .28205 L
.51699 .28205 L
.51584 .28205 L
.51457 .28205 L
.51323 .28204 L
.51206 .28204 L
.51096 .28203 L
.50846 .282 L
.50577 .28195 L
.50293 .2819 L
.49789 .28177 L
.49252 .28159 L
.48673 .28136 L
.47649 .28082 L
.46753 .28016 L
.4481 .27815 L
.40885 .27514 L
.39966 .27494 L
.3946 .27484 L
Mistroke
.38984 .27469 L
.38718 .27458 L
.38431 .27443 L
.38172 .27425 L
.37934 .27405 L
.37473 .27352 L
.36991 .27276 L
.36488 .27167 L
.36024 .27038 L
.35174 .26728 L
.33285 .25787 L
.29726 .23986 L
.2889 .23528 L
.28232 .23051 L
.27957 .2278 L
.27709 .22461 L
.2759 .22265 L
.27493 .22069 L
.27414 .2187 L
.27353 .21675 L
.2731 .215 L
.27275 .21304 L
.27261 .21196 L
.27255 .21137 L
.27249 .21081 L
.27242 .2098 L
.27238 .2087 L
.27236 .20813 L
.27236 .20751 L
.27238 .20638 L
.27242 .20527 L
.27248 .20423 L
.27258 .20311 L
.27272 .20186 L
.27309 .19932 L
.27352 .19707 L
.274 .19502 L
.27534 .19043 L
.27856 .18251 L
.28301 .17475 L
.28752 .16889 L
.29259 .16384 L
.29647 .16064 L
.29781 .1595 L
.29837 .15895 L
.29876 .15848 L
.29891 .15825 L
.29904 .158 L
.29909 .15786 L
.29913 .15773 L
Mistroke
.29915 .15762 L
.29917 .15749 L
.29916 .15727 L
.29914 .15716 L
.29911 .15703 L
.29902 .15682 L
.2989 .15661 L
.29874 .15641 L
.29854 .15619 L
.29807 .15577 L
.2968 .15494 L
.2922 .15272 L
.27616 .14682 L
.26464 .14337 L
.25339 .14072 L
.24779 .13973 L
.24496 .13932 L
.24196 .13897 L
.2404 .13882 L
.23899 .1387 L
.23757 .13861 L
.23627 .13854 L
.23555 .13851 L
.23476 .13848 L
.23398 .13846 L
.23314 .13845 L
.23242 .13844 L
.23164 .13844 L
.23091 .13845 L
.23024 .13846 L
.22906 .1385 L
.2278 .13856 L
.22663 .13863 L
.22557 .13871 L
.22301 .13896 L
.22064 .13927 L
.21604 .14009 L
.21119 .14125 L
.20292 .14387 L
.19389 .14737 L
.17508 .15573 L
.16523 .1606 L
.15622 .16568 L
.14819 .17114 L
.14176 .17673 L
.1367 .18265 L
.13444 .18617 L
.13274 .18948 L
.13127 .19326 L
.13062 .19545 L
Mistroke
.13014 .1975 L
.1298 .19932 L
.12952 .20135 L
.12941 .20238 L
.12933 .20349 L
.12927 .20455 L
.12923 .20552 L
.12922 .20651 L
.12922 .20745 L
.12924 .20848 L
.12929 .2096 L
.12936 .21072 L
.12946 .2119 L
.12969 .21403 L
.12996 .21602 L
.13028 .21789 L
.13118 .22216 L
.13388 .23125 L
.13786 .241 L
.14266 .25018 L
.14889 .25969 L
.15509 .26735 L
.16268 .27495 L
.17879 .28662 L
.18832 .29151 L
.19755 .29525 L
.21667 .30091 L
.23464 .30479 L
.25486 .30879 L
.27343 .3132 L
.29117 .31891 L
.30993 .32651 L
.32884 .33435 L
.33727 .33735 L
.34664 .34005 L
.3512 .3411 L
.35613 .34203 L
.36138 .34279 L
.36632 .3433 L
.36912 .34351 L
.3706 .3436 L
.37217 .34368 L
.37355 .34373 L
.37507 .34378 L
.37645 .34382 L
.37775 .34384 L
.37911 .34386 L
.3804 .34386 L
.38156 .34386 L
.38283 .34386 L
Mistroke
.38421 .34385 L
.38498 .34384 L
.38567 .34383 L
.38832 .34377 L
.39923 .34342 L
.40519 .34322 L
.4081 .34313 L
.41075 .34305 L
.41301 .34301 L
.4142 .34298 L
.41546 .34297 L
.41681 .34295 L
.41805 .34294 L
.41921 .34293 L
.42049 .34293 L
.42186 .34294 L
.42312 .34295 L
.42449 .34296 L
.42593 .34299 L
.42835 .34305 L
.43098 .34313 L
.43366 .34325 L
.43618 .34338 L
.44083 .34368 L
.45109 .34461 L
.46922 .34675 L
.47881 .34784 L
.48417 .34836 L
.48927 .34878 L
.49369 .34907 L
.49599 .34919 L
.49852 .3493 L
.49997 .34935 L
.50131 .34939 L
.50271 .34943 L
.50398 .34945 L
.50524 .34947 L
.50642 .34947 L
.50768 .34948 L
.50905 .34947 L
.51043 .34946 L
.51169 .34944 L
.51315 .34941 L
.51453 .34937 L
.51701 .34929 L
.5197 .34916 L
.52267 .34898 L
.52583 .34874 L
.53149 .3482 L
.5373 .34748 L
Mistroke
.54268 .34666 L
.55463 .34431 L
.57503 .33872 L
.59252 .33321 L
.63129 .32432 L
.65188 .32087 L
.67 .31703 L
.68867 .3118 L
.70838 .30605 L
.71745 .3039 L
.72733 .30207 L
.7326 .30131 L
.73847 .30062 L
.74889 .29968 L
.76021 .29885 L
.76571 .29841 L
.77074 .29794 L
.77987 .29676 L
.78453 .29594 L
.78937 .29486 L
.7979 .2923 L
.80547 .28913 L
.81328 .28476 L
.82116 .27905 L
.83452 .26566 L
.84651 .24865 L
.85583 .23104 L
.86289 .21408 L
.86619 .2044 L
.86876 .19535 L
.87056 .1873 L
.87133 .18278 L
.87164 .18061 L
.87188 .17863 L
.87206 .17678 L
.87222 .17477 L
.87229 .17367 L
.87234 .17264 L
.87238 .17167 L
.87241 .17063 L
.87242 .16966 L
.87242 .16861 L
.87241 .16762 L
.87239 .16671 L
.87236 .16589 L
.87232 .16499 L
.8722 .16317 L
.8721 .16204 L
.87199 .16096 L
.87174 .15896 L
Mistroke
.87094 .15449 L
.86998 .15069 L
.86863 .14663 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0S?l007co00@0ool02?l500Co0P1?o`X0=?l007Ko1002o`040?oo00So00@0ool0
1Ol00`3oo`1<o`802_l3037o001lo`040?oo00Wo00<0o`001Ol00`3oo`1:o`803ol00`3oo`0^o`00
OOl200_o0P05o`030?oo04Wo00<0ool03ol00`3oo`0^o`00TOl00`3oo`18o`030?oo017o00<0ool0
;Ol004Oo1P14o`030?oo04Oo00<0ool04ol00`3oo`0/o`00@ol400Ko1010o`030?oo04Oo00<0ool0
5?l00`3oo`0[o`00@Ol200ko0`0mo`030?oo04Ko00<0ool05Ol00`3oo`0[o`00?_l301?o100io`03
0?oo04Ko00<0ool05_l00`3oo`0Zo`00??l201[o0P0go`030?oo04Ko00<0ool05_l00`3oo`0Zo`00
>Ol301ko0P0eo`030?oo04Go00<0ool05ol00`3oo`0Zo`00=ol2027o0P0eo`030?oo04Go00<0ool0
5ol00`3oo`0Zo`00=_l00`3oo`0Po`030?oo03Go00<0ool0AOl00`3oo`0Go`80:ol003Go00<0ool0
8?l00`3oo`0fo`030?oo04Go00<0ool06?l00`3oo`0Yo`00<ol202?o00<0ool0=_l00`3oo`15o`03
0?oo01So00<0ool0:Ol003;o00<0ool08_l00`3oo`0So`801?l200?o1005o`030?oo04Go00<0ool0
6?l00`3oo`0Yo`00<Ol00`3oo`0So`030?oo02;o00@0ool02?l00`3oo`06o`030?oo04Co00<0ool0
6?l00`3oo`0Zo`00<?l00`3oo`0So`030?oo02?o00@0ool02Ol00`3oo`05o`80AOl00`3oo`0Ho`03
0?oo02[o000_o`030?oo02Co00<0ool07Ol400;o00@0ool02_l00`3oo`04o`030?oo04Co00<0ool0
6?l00`3oo`0Zo`00;ol00`3oo`0So`030?oo02Co00@0ool02?l0103oo`05o`030?oo04Co00<0ool0
6?l00`3oo`0Zo`00;_l00`3oo`0So`030?oo02Ko0P0:o`801_l00`3oo`14o`030?oo01So00<0ool0
:_l002ko00<0ool08ol00`3oo`0jo`030?oo04?o00<0ool06Ol00`3oo`0Zo`00;_l00`3oo`0So`03
0?oo03[o00<0ool0@ol00`3oo`0Ho`030?oo02_o000^o`030?oo02?o00<0ool0>_l00`3oo`13o`03
0?oo01So00<0ool0:ol000Oo1@02o`801?l201So00<0ool02?l501Ko00<0ool02Ol200Co0P04o`80
8ol00`3oo`0Ro`801?l200Co0P0Bo`030?oo00ko1@06o`030?oo01Wo1@02o`801?l200?o0009o`03
0?oo00Oo00@0ool05ol00`3oo`0:o`030?oo01Ko00<0ool02?l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo013o00<0ool04Ol00`3oo`05o`030?oo01co00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01Go00<0ool02_l00`3oo`0Fo`030?oo00So00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`0>o`030?oo017o00<0ool01Ol00`3oo`0Lo`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201So00<0ool00_l400Co00<0ool05_l200?o1002o`040?oo00Wo
0P0So`030?oo027o00@0ool02Ol2013o00<0ool04_l00`3oo`04o`030?oo01go00<0ool02?l200?o
0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Oo00<0ool01ol0103oo`09o`030?oo02;o
00<0ool08Ol0103oo`09o`030?oo00ko00<0ool04ol00`3oo`04o`030?oo01go00<0ool02?l00`3o
o`02o`002?l200[o0`0Ho`030?oo00So0P0Jo`030?oo00Oo0P0:o`<08_l00`3oo`0Ro`802_l300co
0P0Eo`801Ol00`3oo`0Mo`802_l300;o000_o`030?oo02Go00<0ool0=ol00`3oo`0jo`<07_l00`3o
o`0^o`00;ol00`3oo`0Vo`80=ol00`3oo`0go`<08Ol00`3oo`0^o`00<?l00`3oo`0Wo`80=Ol00`3o
o`0eo`808ol00`3oo`0_o`00<?l00`3oo`0Yo`80<ol00`3oo`0co`809Ol00`3oo`0_o`00<Ol00`3o
o`0Zo`80<Ol00`3oo`0ao`809_l00`3oo`0`o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1?l300So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00@0oooo0P03o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02?o
00<0ool00ol00`3oo`0Yo`030?oo02[o00<0ool0:Ol0103o000Xo`030?l002[o00<0ool01ol003?o
00<0ool0;ol202[o00<0ool0:?l302_o00<0ool0<ol003Co00<0ool0<?l202So00<0ool09_l202go
00<0ool0=?l003Go00<0ool0<Ol402Co00<0ool08Ol502ko00<0ool0=Ol003Ko00<0ool0=?l601ko
00<0ool05ol:03?o00<0ool0=Ol003Oo00<0ool0>?lD00_o00<0ool03_l903co00<0ool0=_l003So
00<0ool0Bol=00Go2P13o`80>Ol003Wo0P1Fo`030?l000@0Bol203_o000ko`80E?l00`3oo`1=o`80
?Ol003go0P1Bo`030?oo04_o0P0oo`00?ol304oo00<0ool0A_l5047o0012o`80COl00`3oo`0ko`/0
A_l004Co1019o`030?oo03Go1P1Ao`00B?l504Co00<0ool0<_l305Oo001=o`D0:ol200Co0P03o`@0
1Ol00`3oo`0^o`@0F_l005;o1@0Uo`040?oo00So00<0ool01_l00`3oo`0/o`80G_l005Oo100Qo`04
0?oo00Wo00<0ool01Ol202Wo101Po`00Fol201oo00@0ool02_l00`3oo`04o`030?oo02;o1P1To`00
GOl301co00@0ool02?l0103oo`05o`030?oo01go1@1Zo`00H?l201_o0P0:o`801_l00`3oo`0Ho`D0
Kol006;o0P0]o`030?oo01Co101do`00I?l302[o00<0ool04_l207So001Wo`@09_l00`3oo`0>o`@0
N_l006_o6`0;o`030?oo00[o101no`00Q_lH08;o002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`03
0?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87087 .14926 m
.86912 .14524 L
.86722 .14162 L
.86463 .13752 L
.86196 .13402 L
.85554 .12767 L
.85148 .12472 L
.84734 .12233 L
.8428 .12029 L
.84021 .11936 L
.83771 .1186 L
.83318 .11756 L
.83072 .11715 L
.82804 .11682 L
.82655 .11668 L
.82516 .11659 L
.82436 .11655 L
.82363 .11653 L
.82278 .1165 L
.82199 .11649 L
.82129 .11649 L
.82054 .1165 L
.81921 .11653 L
.81847 .11656 L
.81777 .11659 L
.8162 .1167 L
.8148 .11682 L
.81347 .11696 L
.811 .11729 L
.80837 .11775 L
.80555 .11835 L
.80086 .11963 L
.79665 .12112 L
.7923 .12306 L
.78798 .1255 L
.78411 .12826 L
.78051 .13154 L
.77784 .13462 L
.77533 .1383 L
.77323 .14231 L
.77159 .14637 L
.77038 .15023 L
.7693 .15462 L
.76746 .1651 L
.766 .17524 L
.76525 .17999 L
.76429 .18504 L
.76204 .19389 L
.76025 .199 L
.7582 .2036 L
Mistroke
.75584 .20785 L
.7534 .21145 L
.74754 .21807 L
.74017 .22411 L
.72484 .23316 L
.68859 .2513 L
.67189 .25915 L
.66223 .26292 L
.653 .26583 L
.64361 .26815 L
.6333 .27012 L
.61165 .27324 L
.59102 .27605 L
.56874 .27917 L
.55837 .28047 L
.54735 .28162 L
.53789 .28238 L
.52742 .28299 L
.52227 .28321 L
.51945 .2833 L
.51684 .28337 L
.51441 .28343 L
.51221 .28347 L
.50977 .28351 L
.50843 .28352 L
.50716 .28353 L
.50591 .28354 L
.50519 .28354 L
.50452 .28354 L
.50318 .28354 L
.50243 .28354 L
.50174 .28354 L
.50051 .28353 L
.49918 .28352 L
.49793 .28351 L
.49676 .28349 L
.49419 .28345 L
.49135 .28338 L
.4884 .28329 L
.48564 .28319 L
.47574 .28268 L
.47081 .28234 L
.46546 .2819 L
.44737 .28006 L
.40737 .27608 L
.39823 .27525 L
.38857 .27419 L
.37809 .27262 L
.36861 .27063 L
.35897 .26793 L
Mistroke
.34864 .26423 L
.33043 .25624 L
.31145 .24717 L
.2945 .23841 L
.28684 .23358 L
.28316 .23081 L
.2796 .22765 L
.27662 .22442 L
.27433 .2213 L
.27248 .218 L
.27117 .21475 L
.27065 .21292 L
.27029 .21118 L
.27013 .21017 L
.27003 .20925 L
.26998 .20873 L
.26995 .20817 L
.26991 .20714 L
.26992 .20615 L
.26994 .20558 L
.26997 .20504 L
.27001 .20448 L
.27006 .20397 L
.2702 .2028 L
.27036 .20186 L
.27056 .20083 L
.27108 .19874 L
.2724 .19496 L
.27429 .19097 L
.27874 .18406 L
.28495 .17678 L
.29192 .16992 L
.29715 .16488 L
.29935 .16252 L
.30033 .1613 L
.3011 .16019 L
.30168 .15916 L
.30193 .15863 L
.30213 .15808 L
.30221 .15782 L
.30227 .15758 L
.30235 .15713 L
.3024 .15671 L
.3024 .15627 L
.30237 .15586 L
.30229 .15541 L
.30223 .15518 L
.30216 .15495 L
.30199 .15452 L
.30154 .15371 L
Mistroke
.30099 .15299 L
.30012 .15211 L
.29915 .15132 L
.29633 .14957 L
.29318 .14808 L
.28924 .14656 L
.28092 .14403 L
.27067 .14169 L
.26543 .14074 L
.26036 .13998 L
.25543 .13937 L
.2508 .13893 L
.24818 .13874 L
.24681 .13866 L
.24536 .13859 L
.24415 .13853 L
.24283 .13849 L
.24163 .13845 L
.24051 .13843 L
.23984 .13842 L
.23912 .13842 L
.2378 .13842 L
.23653 .13842 L
.23536 .13844 L
.23411 .13847 L
.23273 .13852 L
.2313 .13858 L
.22997 .13865 L
.22771 .1388 L
.22536 .13899 L
.22121 .13943 L
.21682 .14003 L
.21209 .14083 L
.20247 .14289 L
.19376 .14522 L
.18357 .14848 L
.17416 .15203 L
.15688 .16058 L
.14869 .16617 L
.14117 .17296 L
.1358 .17966 L
.13336 .18371 L
.13152 .18756 L
.13015 .19125 L
.12955 .19325 L
.12902 .19541 L
.12861 .19751 L
.12831 .19943 L
.1282 .20036 L
.12809 .20138 L
Mistroke
.12802 .20233 L
.12796 .20323 L
.12791 .20435 L
.12788 .20537 L
.12788 .20647 L
.12788 .2071 L
.1279 .20768 L
.12793 .20864 L
.12799 .20971 L
.12807 .21082 L
.12817 .21187 L
.12845 .21429 L
.1288 .21651 L
.12986 .22152 L
.13114 .22607 L
.13285 .23102 L
.13723 .24084 L
.14242 .24984 L
.14788 .25757 L
.15505 .26593 L
.16233 .27288 L
.1778 .28403 L
.18661 .28879 L
.19631 .2931 L
.2323 .30439 L
.27092 .31467 L
.308 .32692 L
.327 .33357 L
.33712 .33676 L
.34798 .33968 L
.35771 .34177 L
.36294 .34267 L
.36863 .34349 L
.37387 .3441 L
.37872 .34457 L
.38832 .34523 L
.39325 .34548 L
.39855 .34568 L
.4081 .34598 L
.41935 .34631 L
.42965 .34669 L
.44013 .34718 L
.45138 .34784 L
.46187 .34851 L
.47171 .34912 L
.48093 .34958 L
.48616 .34978 L
.48849 .34985 L
.49098 .34991 L
.49242 .34994 L
Mistroke
.49373 .34996 L
.49505 .34998 L
.49629 .34999 L
.497 .34999 L
.49775 .35 L
.49909 .35 L
.49984 .35 L
.50054 .35 L
.50131 .34999 L
.50214 .34999 L
.50352 .34997 L
.50502 .34995 L
.50644 .34992 L
.50776 .34989 L
.51025 .34981 L
.51291 .34971 L
.51819 .34944 L
.52303 .34911 L
.53393 .34806 L
.54343 .34679 L
.55325 .3451 L
.57319 .34058 L
.59101 .33567 L
.62879 .32533 L
.6695 .31494 L
.68953 .30955 L
.7081 .30499 L
.7267 .3014 L
.74686 .29849 L
.76698 .29573 L
.78467 .29236 L
.79399 .28984 L
.8025 .2869 L
.81112 .28313 L
.82008 .27819 L
.83507 .26695 L
.84267 .25942 L
.84901 .25191 L
.85959 .23595 L
.8641 .22725 L
.86825 .21753 L
.8714 .20822 L
.87363 .1995 L
.87456 .19466 L
.87521 .19024 L
.87549 .18778 L
.87562 .18648 L
.87573 .18511 L
.87581 .1839 L
.87589 .18258 L
Mistroke
.87594 .18137 L
.87597 .18023 L
.87599 .17923 L
.876 .17817 L
.87599 .17716 L
.87598 .17624 L
.87595 .17516 L
.87591 .17399 L
.87585 .17289 L
.87579 .17187 L
.87565 .17002 L
.87544 .16799 L
.87487 .1639 L
.87416 .16018 L
.87319 .15623 L
.87087 .14926 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0DOl303So001lo`040?oo00So1@04o`80C_l50003o`0000@0<ol007Ko1002o`04
0?oo00So00@0ool01Ol00`3oo`1:o`<03?l00`3oo`0`o`00O?l0103oo`09o`030?l000Go00<0ool0
BOl00`3oo`0>o`80<?l007go0P0;o`801Ol00`3oo`17o`804ol00`3oo`0]o`00TOl00`3oo`16o`03
0?oo01Co00<0ool0;?l004Oo2012o`030?oo04Ko00<0ool05?l00`3oo`0/o`00@ol400So1`0ko`03
0?oo04Go00<0ool05_l00`3oo`0[o`00?_l501?o0`0ho`030?oo04Go00<0ool05ol00`3oo`0Zo`00
??l201_o0P0fo`030?oo04Co00<0ool06?l00`3oo`0Zo`00>Ol301oo0P0do`030?oo04Co00<0ool0
6?l00`3oo`0Zo`00=ol202?o00<0ool0<_l00`3oo`14o`030?oo01Wo00<0ool0:Ol003Go0P0To`80
=?l00`3oo`14o`030?oo01Wo00<0ool0:Ol003Co00<0ool09?l00`3oo`0co`030?oo04Co00<0ool0
6Ol00`3oo`0Yo`00<ol00`3oo`0To`030?oo03Co00<0ool0A?l00`3oo`0Io`030?oo02Wo000bo`03
0?oo02Co00<0ool0=Ol00`3oo`13o`030?oo01[o00<0ool0:Ol0037o00<0ool09?l00`3oo`0Ro`80
1?l200?o1005o`030?oo04?o00<0ool06_l202[o000`o`030?oo02Co00<0ool08_l0103oo`08o`03
0?oo00Ko00<0ool0@ol00`3oo`0Ko`030?oo02So000_o`030?oo02Co00<0ool08ol0103oo`09o`03
0?oo00Go0P14o`030?oo01[o0P0Zo`00;_l00`3oo`0To`030?oo01ko1002o`040?oo00[o00<0ool0
1?l00`3oo`13o`030?oo01[o00<0ool0:Ol002ko00<0ool08ol00`3oo`0Uo`040?oo00So00@0ool0
1Ol00`3oo`12o`030?oo01_o00<0ool0:Ol002ko00<0ool08ol00`3oo`0Vo`802_l200Ko00<0ool0
@_l00`3oo`0Ko`030?oo02Wo000^o`030?oo02?o00<0ool0>_l00`3oo`12o`030?oo01_o00<0ool0
:Ol002ko00<0ool08_l00`3oo`0ko`030?oo047o00<0ool07?l00`3oo`0Yo`00;_l00`3oo`0Ro`03
0?oo03_o00<0ool0@Ol00`3oo`0Lo`030?oo02Wo0007o`D00_l200Co0P0Ho`030?oo00So1@0Eo`03
0?oo00[o0P04o`801?l202?o00<0ool08_l200Co0P04o`804Ol00`3oo`0?o`D01ol00`3oo`0Ho`D0
0_l200Co0P03o`002Ol00`3oo`07o`040?oo01Oo00<0ool02_l00`3oo`0Eo`030?oo00Wo00@0ool0
2?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0?o`030?oo01;o00<0ool01ol00`3oo`0Jo`03
0?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Eo`030?oo00[o00<0ool05_l00`3oo`08o`04
0?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool03?l00`3oo`0Co`030?oo00Ko00<0ool0
6ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Ho`030?oo00;o1004o`030?oo01Ko
00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`803_l00`3oo`0Do`030?oo00Ko
00<0ool06ol00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Go`030?oo00[o00<0ool05ol00`3o
o`07o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02ol201Oo00<0ool01Ol00`3o
o`0Lo`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3oo`09o`806_l00`3oo`07o`802_l302;o
00<0ool08_l200[o0`0:o`030?oo01Ko0P07o`030?oo01_o0P0:o`<00_l002oo00<0ool09Ol00`3o
o`0go`030?oo03Wo0P0Ro`030?oo02co000_o`030?oo02Ko0P0go`030?oo03Oo0P0So`030?oo02go
000`o`030?oo02Oo0P0eo`030?oo03Go0P0To`030?oo02ko000`o`030?oo02Wo0P0co`030?oo03?o
0P0Vo`030?oo02ko000ao`030?oo02[o00<0ool0<?l00`3oo`0ao`809ol00`3oo`0_o`003?oo00`0
2Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool00ol20003o`3o00Oo00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00P04o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00D0oooo0004o`030?oo00Ko00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02So0P0/o`030?oo02Wo
0`0Yo`050?ooo`00:?l00`3oo`07o`00=?l00`3oo`0^o`80:_l00`3oo`0Wo`<0;_l00`3oo`0ao`00
=Ol00`3oo`0_o`<09ol00`3oo`0Uo`80<?l00`3oo`0bo`00=_l00`3oo`0ao`H08Ol00`3oo`0No`L0
<?l203Go000go`030?oo03Ko1@0Lo`030?oo01So1P0eo`80=ol003So00<0ool0>_l901?o00<0ool0
4ol503[o00<0ool0=ol003Wo0P13o`/02?l00`3oo`09o`X0?Ol203[o000ko`030?oo04_o5015o`80
??l003co0`1Bo`030?oo04co0P0no`00?ol304oo00<0ool0A_l6043o0012o`<0C?l00`3oo`11o`D0
A_l004Go0`19o`030?oo03co1@1;o`00B?l304Ko00<0ool0=ol5053o001;o`@0@_l00`3oo`0bo`D0
EOl004oo100Zo`801?l200?o1005o`030?oo02go1@1Jo`00Dol302Ko00@0ool02?l00`3oo`06o`03
0?oo02Wo101Oo`00E_l302?o00@0ool02Ol00`3oo`05o`809_l406?o001Io`<08?l0103oo`0:o`03
0?oo00Co00<0ool08Ol406Oo001Lo`<07Ol0103oo`08o`040?oo00Go00<0ool07_l306_o001Oo`80
7?l200[o0P06o`030?oo01[o101^o`00HOl302go00<0ool05ol307;o001To`<0:_l00`3oo`0Co`@0
MOl006Oo1@0Uo`030?oo00ko1@1io`00K?l;01[o00<0ool02_l407ko001goaD01Ol0103oool908;o
002<o`T0Rol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`030?oo08co
001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`040?oo00Wo
00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103o
o`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go
00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00oolQo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87312 .15188 m
.86926 .14386 L
.86393 .13584 L
.8608 .13219 L
.85712 .12864 L
.84968 .12322 L
.845 .12076 L
.8396 .11859 L
.83702 .11778 L
.83423 .11705 L
.83177 .11652 L
.82908 .11605 L
.82645 .1157 L
.82402 .11547 L
.82284 .11538 L
.82155 .11532 L
.82034 .11527 L
.8192 .11525 L
.81851 .11524 L
.81776 .11525 L
.8164 .11527 L
.81508 .11531 L
.81386 .11537 L
.81254 .11546 L
.8111 .11558 L
.8082 .11591 L
.80561 .1163 L
.80282 .11683 L
.79788 .11805 L
.79326 .11957 L
.78919 .12126 L
.78093 .12614 L
.77732 .12914 L
.77384 .13287 L
.77123 .13646 L
.7689 .14056 L
.76722 .1443 L
.7657 .14863 L
.76437 .15353 L
.76322 .15898 L
.76158 .16914 L
.75992 .17956 L
.75875 .18531 L
.75741 .19047 L
.75414 .19945 L
.75203 .2037 L
.74934 .20814 L
.74329 .21582 L
.73678 .22219 L
.71999 .23483 L
Mistroke
.70239 .24567 L
.6858 .25422 L
.66896 .2608 L
.64987 .26611 L
.60818 .27454 L
.58735 .27803 L
.56776 .28057 L
.54812 .2825 L
.53806 .28333 L
.52734 .28411 L
.52185 .28446 L
.51668 .28474 L
.51167 .28497 L
.50947 .28505 L
.50705 .28513 L
.50495 .28519 L
.50268 .28523 L
.50144 .28525 L
.5003 .28527 L
.49921 .28527 L
.49806 .28528 L
.49689 .28528 L
.49564 .28528 L
.49446 .28527 L
.49339 .28527 L
.49213 .28525 L
.49076 .28523 L
.48828 .28517 L
.48586 .2851 L
.48325 .285 L
.47853 .28477 L
.46777 .28402 L
.44607 .2819 L
.4262 .27964 L
.40523 .2768 L
.38512 .27297 L
.36631 .26808 L
.3297 .25555 L
.31263 .24816 L
.29698 .23985 L
.28116 .22874 L
.2744 .22208 L
.27171 .21859 L
.27053 .21672 L
.26945 .21468 L
.26863 .21275 L
.26834 .2119 L
.26806 .21096 L
.26786 .21013 L
.26769 .20924 L
Mistroke
.26762 .20875 L
.26756 .2083 L
.26748 .20743 L
.26745 .20658 L
.26746 .2058 L
.26751 .20495 L
.26762 .20404 L
.26776 .20317 L
.26794 .20236 L
.26838 .20084 L
.26907 .19905 L
.27 .19718 L
.27214 .19385 L
.27885 .18647 L
.29392 .17344 L
.29962 .16794 L
.30201 .16509 L
.30396 .16214 L
.30469 .16066 L
.30501 .15988 L
.30529 .15906 L
.30549 .15831 L
.30562 .15763 L
.30571 .15698 L
.30576 .15631 L
.30576 .15568 L
.30572 .15502 L
.30563 .1544 L
.30552 .15384 L
.30534 .1532 L
.30509 .15251 L
.30449 .15129 L
.30367 .15006 L
.30255 .14876 L
.30124 .14756 L
.29961 .14634 L
.29616 .14435 L
.29239 .14272 L
.2878 .14123 L
.28319 .14008 L
.27784 .1391 L
.27475 .13867 L
.27191 .13835 L
.26935 .13812 L
.26662 .13792 L
.26519 .13784 L
.26363 .13776 L
.26218 .13771 L
.26087 .13767 L
.25953 .13764 L
Mistroke
.25827 .13762 L
.25694 .13761 L
.25618 .1376 L
.25549 .1376 L
.25434 .13761 L
.25307 .13762 L
.25175 .13765 L
.25051 .13767 L
.24781 .13775 L
.24532 .13785 L
.23971 .13815 L
.22968 .13889 L
.21115 .14075 L
.20062 .14216 L
.19069 .14388 L
.18184 .1459 L
.17247 .14874 L
.16362 .15233 L
.15453 .15729 L
.146 .16365 L
.13916 .17065 L
.13627 .17445 L
.13351 .17882 L
.13144 .18286 L
.12962 .18733 L
.12834 .19147 L
.12776 .19387 L
.12732 .19616 L
.127 .19827 L
.12678 .20022 L
.12669 .20125 L
.12662 .20237 L
.12656 .20355 L
.12653 .20466 L
.12653 .20589 L
.12653 .20651 L
.12655 .20719 L
.1266 .2084 L
.12666 .20954 L
.12677 .21088 L
.12689 .2121 L
.12726 .21485 L
.12767 .21722 L
.12822 .21979 L
.12947 .22445 L
.13289 .23353 L
.137 .24162 L
.14283 .25064 L
.14909 .25849 L
.16235 .27131 L
Mistroke
.17861 .28254 L
.19507 .29081 L
.23253 .30432 L
.26816 .31585 L
.30666 .32791 L
.34417 .33806 L
.36346 .34243 L
.3738 .34442 L
.38492 .34623 L
.39491 .34757 L
.406 .34873 L
.41614 .34954 L
.42568 .35011 L
.43583 .35055 L
.44095 .35073 L
.44655 .35088 L
.45136 .35099 L
.45651 .35108 L
.45902 .35111 L
.46138 .35114 L
.46352 .35116 L
.46582 .35117 L
.46713 .35118 L
.46833 .35119 L
.46953 .35119 L
.47067 .35119 L
.472 .35119 L
.47323 .35119 L
.47458 .35119 L
.47527 .35118 L
.47602 .35118 L
.47742 .35117 L
.4787 .35116 L
.4816 .35113 L
.48415 .3511 L
.48684 .35105 L
.49164 .35094 L
.4961 .35081 L
.5062 .35039 L
.51145 .3501 L
.51713 .34973 L
.52738 .34888 L
.54641 .34669 L
.55601 .34525 L
.56652 .34337 L
.58752 .33863 L
.62628 .32655 L
.66302 .31448 L
.7036 .30483 L
.74334 .29758 L
Mistroke
.78115 .2896 L
.7999 .28436 L
.81898 .27708 L
.8279 .27256 L
.83711 .26679 L
.84463 .26104 L
.85227 .25392 L
.86452 .2386 L
.86978 .22951 L
.87371 .22075 L
.87528 .21636 L
.87673 .21153 L
.87778 .20723 L
.87866 .20262 L
.87925 .19843 L
.8795 .19603 L
.87959 .19493 L
.87968 .19376 L
.87974 .19277 L
.87979 .19168 L
.87983 .19068 L
.87986 .18975 L
.87988 .18865 L
.87989 .18765 L
.87988 .18649 L
.87987 .18539 L
.87984 .18439 L
.8798 .18332 L
.8797 .18137 L
.87953 .17905 L
.87932 .17692 L
.87871 .17228 L
.87792 .16786 L
.876 .16012 L
.87312 .15188 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0C_l803Ko001lo`040?oo00So1@04o`80C?l300Oo0`0do`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Wo0P0=o`<0<Ol007co00@0ool02Ol00`3o0005o`030?oo04Oo0P0Ao`03
0?oo02oo001mo`802ol200Go00<0ool0A_l00`3oo`0Bo`030?oo02ko002Ao`030?oo04Go00<0ool0
5?l00`3oo`0]o`00Aol@03[o00<0ool0AOl00`3oo`0Eo`030?oo02co0010o`L04?l303Oo00<0ool0
A?l00`3oo`0Go`030?oo02_o000ko`D06_l203Go00<0ool0A?l00`3oo`0Ho`030?oo02[o000io`80
8Ol00`3oo`0bo`030?oo04?o00<0ool06Ol00`3oo`0Zo`00=ol202?o0P0co`030?oo04?o00<0ool0
6_l00`3oo`0Yo`00=Ol202Ko00<0ool0<Ol00`3oo`13o`030?oo01[o00<0ool0:Ol003?o0P0Xo`03
0?oo037o00<0ool0@ol00`3oo`0Ko`030?oo02So000bo`030?oo02Oo00<0ool0<_l00`3oo`12o`03
0?oo01co00<0ool0:?l0037o00<0ool09ol00`3oo`0co`030?oo04;o00<0ool07?l00`3oo`0Xo`00
<Ol00`3oo`0Wo`030?oo03?o00<0ool0@_l00`3oo`0Lo`030?oo02So000`o`030?oo02Oo00<0ool0
8?l200Co0P03o`@01Ol00`3oo`12o`030?oo01co00<0ool0:?l002oo00<0ool09_l202;o00@0ool0
2?l00`3oo`06o`030?oo04;o00<0ool07?l00`3oo`0Xo`00;_l00`3oo`0Vo`030?oo02;o00@0ool0
2Ol00`3oo`05o`80@_l00`3oo`0No`030?oo02Oo000^o`030?oo02Go00<0ool07Ol400;o00@0ool0
2_l00`3oo`04o`030?oo047o00<0ool07_l00`3oo`0Wo`00;_l00`3oo`0To`030?oo02Co00@0ool0
2?l0103oo`05o`030?oo047o00<0ool07_l00`3oo`0Wo`00;Ol00`3oo`0To`030?oo02Ko0P0:o`80
1_l00`3oo`11o`030?oo01go0P0Yo`00;Ol00`3oo`0So`030?oo03_o00<0ool0@?l00`3oo`0No`03
0?oo02So000]o`030?oo02?o00<0ool0>ol00`3oo`10o`030?oo01ko00<0ool0:?l002go00<0ool0
8ol00`3oo`0ko`030?oo043o00<0ool07_l00`3oo`0Xo`001ol500;o0P04o`805ol00`3oo`09o`D0
5Ol00`3oo`0:o`801?l200Co0P0So`030?oo02;o0P04o`801?l200oo00<0ool04Ol500Wo00<0ool0
5_l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05Ol00`3oo`09o`04
0?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool03Ol00`3oo`0Do`030?oo00Wo00<0ool0
6?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool0
2Ol0103oo`0;o`030?oo023o00<0ool08Ol0103oo`0;o`030?oo00[o00<0ool05Ol00`3oo`08o`03
0?oo01Wo00<0ool02_l1007o0Ol00003o`0000801?l00`3oo`08o`806?l00`3oo`02o`@01?l00`3o
o`0Fo`030?oo00;o1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200co00<0ool05_l00`3o
o`08o`030?oo01Wo00<0ool02?l200?o0009o`030?oo00So00<0ool05ol00`3oo`0:o`030?oo01Ko
0P09o`040?oo00Wo00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool02_l00`3oo`0Go`030?oo00So
00<0ool06Ol00`3oo`08o`030?oo00;o0008o`802_l301Oo00<0ool02Ol201[o00<0ool01ol200[o
0`0Ro`030?oo02;o0P0:o`<02Ol00`3oo`0Go`802Ol00`3oo`0Io`802_l300;o000_o`030?oo02Go
0P0ho`030?oo03So0P0Uo`030?oo02[o000_o`030?oo02Oo00<0ool0=Ol00`3oo`0go`030?oo02Co
00<0ool0:ol0033o00<0ool09ol00`3oo`0do`030?oo03Go0P0Vo`030?oo02co000`o`030?oo02So
0P0do`030?oo03?o0P0Wo`030?oo02go000ao`030?oo02Wo0P0bo`030?oo03;o00<0ool09ol00`3o
o`0]o`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01?l300So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00@0oooo
0`02o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool00_l01@3oool000So00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l00`3oo`02o`030?oo02Wo0`0Zo`03
0?oo02Wo00@0o`00:?l01@3oool002So00<0ool01ol003Co00<0ool0<?l302Oo00<0ool09ol402co
0P0co`00=Ol00`3oo`0bo`809Ol00`3oo`0So`@0;ol00`3oo`0co`00=_l00`3oo`0co`@08Ol00`3o
o`0Oo`@0<_l00`3oo`0do`00=ol00`3oo`0fo`D07?l00`3oo`0Ko`@0=?l203Oo000ho`030?oo03[o
1P0Fo`030?oo01;o2@0eo`<0>Ol003Wo0P10o`/02ol00`3oo`0=o`D0??l203co000ko`030?oo04So
6`0no`<0?_l003co0P1Co`030?oo04Wo0P11o`00?_l3053o00<0ool0AOl404?o0011o`<0COl00`3o
o`10o`D0Aol004Co1019o`030?oo03_o1@1<o`00B?l304Ko00<0ool0=_l5057o001;o`80A?l00`3o
o`0ao`D0E_l004go0`0]o`801?l200?o1005o`030?oo02_o1P1Ko`00D?l202[o00@0ool02?l00`3o
o`06o`030?oo02Oo101Qo`00D_l302Oo00@0ool02Ol00`3oo`05o`809?l406Go001Eo`<09?l0103o
o`0:o`030?oo00Co00<0ool08?l306Wo001Ho`@08?l0103oo`08o`040?oo00Go00<0ool07Ol306co
001Lo`@07Ol200[o0P06o`030?oo01_o0P1_o`00H?l502co00<0ool06?l3077o001Uo`@0:?l00`3o
o`0Fo`80M?l006Wo0P0Vo`030?oo017o1@1fo`00Jol6023o00<0ool02_l707_o001ao`/05Ol00`3o
o`03o`L0P_l007co6`29o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87537 .1545 m
.87129 .14609 L
.8659 .13766 L
.85922 .13001 L
.85201 .12412 L
.84747 .12132 L
.84291 .11907 L
.83854 .11735 L
.83371 .11587 L
.83113 .11524 L
.82824 .11465 L
.82553 .11421 L
.82302 .1139 L
.82159 .11375 L
.82003 .11362 L
.81917 .11357 L
.81836 .11352 L
.81755 .11348 L
.81677 .11345 L
.81605 .11343 L
.81527 .11342 L
.81391 .11341 L
.81313 .11342 L
.81231 .11343 L
.81161 .11345 L
.81084 .11348 L
.80942 .11355 L
.80812 .11363 L
.80516 .1139 L
.80256 .11424 L
.79983 .11468 L
.79518 .11569 L
.79021 .11715 L
.78518 .11912 L
.78013 .12172 L
.77577 .12464 L
.77218 .1277 L
.76907 .13104 L
.76619 .13496 L
.76389 .13896 L
.76214 .14282 L
.76045 .1476 L
.75901 .153 L
.75795 .15825 L
.75714 .16317 L
.75578 .17243 L
.75406 .1822 L
.75274 .18763 L
.75124 .19245 L
.74733 .20149 L
Mistroke
.74215 .20993 L
.73631 .21733 L
.72079 .23271 L
.70437 .24557 L
.69703 .2502 L
.68909 .25431 L
.68037 .25782 L
.6719 .26045 L
.65139 .26524 L
.61348 .27411 L
.60314 .27654 L
.59334 .2785 L
.58435 .27994 L
.57427 .28117 L
.53311 .28474 L
.52251 .28568 L
.51277 .28643 L
.50792 .28673 L
.50543 .28686 L
.5027 .28697 L
.49987 .28705 L
.49833 .28709 L
.49759 .2871 L
.49689 .28711 L
.49624 .28711 L
.49553 .28712 L
.4943 .28713 L
.49359 .28713 L
.49285 .28713 L
.49152 .28712 L
.49027 .2871 L
.48956 .28709 L
.4889 .28708 L
.48641 .28702 L
.48414 .28695 L
.4817 .28685 L
.4766 .28658 L
.4709 .28619 L
.45103 .28441 L
.43074 .28213 L
.42089 .28076 L
.41054 .27892 L
.39184 .27423 L
.37128 .26772 L
.35016 .26115 L
.33232 .25592 L
.32409 .25324 L
.31543 .24985 L
.29947 .24143 L
.28231 .22929 L
Mistroke
.27457 .22245 L
.27093 .21857 L
.26815 .21494 L
.26719 .21339 L
.26634 .21175 L
.26594 .21082 L
.26564 .20999 L
.26539 .20914 L
.26521 .20838 L
.26509 .20768 L
.26502 .20695 L
.26499 .20622 L
.26502 .20543 L
.26509 .20476 L
.26522 .20403 L
.26539 .20336 L
.26558 .20274 L
.26626 .20118 L
.2671 .19975 L
.26977 .19654 L
.27289 .19372 L
.27687 .19068 L
.28532 .18482 L
.29451 .17809 L
.30139 .17183 L
.30451 .16813 L
.30676 .16462 L
.30771 .16266 L
.3084 .16081 L
.30863 .16001 L
.30884 .15916 L
.30901 .15824 L
.30912 .15739 L
.30919 .15649 L
.3092 .15554 L
.30915 .15464 L
.30907 .15384 L
.3089 .1529 L
.30866 .1519 L
.30836 .15097 L
.30803 .15012 L
.30709 .14829 L
.30598 .14665 L
.30334 .14383 L
.30157 .14239 L
.29977 .14118 L
.29747 .13988 L
.29521 .13882 L
.29091 .13726 L
.28841 .13656 L
Mistroke
.28597 .13602 L
.28382 .13563 L
.28151 .13529 L
.28029 .13515 L
.27894 .13502 L
.2778 .13493 L
.27655 .13484 L
.2758 .13481 L
.2751 .13478 L
.27371 .13473 L
.27245 .13472 L
.2711 .13471 L
.27034 .13472 L
.26952 .13474 L
.26874 .13476 L
.26803 .13478 L
.26667 .13484 L
.26524 .13492 L
.26265 .1351 L
.26024 .13531 L
.2548 .13591 L
.21523 .14026 L
.20512 .14082 L
.19952 .14115 L
.19426 .14154 L
.18863 .14209 L
.18248 .14293 L
.17737 .14387 L
.17199 .14516 L
.1673 .1466 L
.16237 .14849 L
.15777 .15068 L
.1531 .1534 L
.14545 .15925 L
.13923 .16577 L
.13346 .17412 L
.1308 .17927 L
.12877 .18425 L
.12735 .18876 L
.12618 .19385 L
.12576 .19645 L
.12558 .1979 L
.12543 .19927 L
.12533 .20055 L
.12528 .20123 L
.12524 .20195 L
.12519 .20323 L
.12517 .20441 L
.12517 .20569 L
.12519 .20642 L
Mistroke
.12521 .2071 L
.12524 .20782 L
.12527 .20847 L
.12537 .20994 L
.12549 .21119 L
.12564 .21254 L
.12599 .215 L
.12652 .21786 L
.12723 .22088 L
.12884 .2262 L
.13273 .23538 L
.13829 .24481 L
.15081 .25984 L
.15874 .26703 L
.16644 .27292 L
.20106 .29159 L
.23729 .30578 L
.27454 .32005 L
.31186 .33068 L
.35166 .33915 L
.3716 .34413 L
.3903 .34867 L
.39986 .3506 L
.40889 .35205 L
.41385 .35268 L
.4184 .35316 L
.42349 .35357 L
.4261 .35374 L
.42899 .35389 L
.43154 .354 L
.43395 .35407 L
.43519 .3541 L
.43651 .35413 L
.43795 .35415 L
.43927 .35417 L
.44042 .35418 L
.44165 .35418 L
.44282 .35418 L
.44388 .35418 L
.44512 .35417 L
.44648 .35416 L
.44777 .35414 L
.44896 .35412 L
.4516 .35407 L
.45408 .35401 L
.45975 .35382 L
.47129 .35329 L
.49297 .3519 L
.51203 .35039 L
.53237 .34855 L
Mistroke
.55122 .34663 L
.56993 .34413 L
.5805 .34221 L
.59001 .34003 L
.60886 .33413 L
.62799 .32611 L
.64556 .31839 L
.65514 .31474 L
.66577 .31139 L
.6862 .30672 L
.70566 .30345 L
.74669 .29515 L
.78579 .28473 L
.80372 .28 L
.81383 .27707 L
.82318 .27393 L
.8414 .26562 L
.85003 .26015 L
.85856 .25331 L
.86522 .24659 L
.87143 .23863 L
.87661 .2298 L
.88018 .22123 L
.88151 .21685 L
.8821 .21451 L
.88263 .21201 L
.88303 .20981 L
.88339 .20736 L
.88365 .20511 L
.88384 .20299 L
.88391 .20192 L
.88398 .20075 L
.88402 .19964 L
.88405 .19861 L
.88407 .19759 L
.88407 .19665 L
.88407 .19557 L
.88405 .19455 L
.88402 .19342 L
.88397 .19223 L
.88392 .19121 L
.88385 .19008 L
.88364 .18755 L
.88339 .18518 L
.88286 .18124 L
.88212 .17693 L
.88032 .16904 L
.878 .1614 L
.87537 .1545 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`03
0?oo00?o00<0ool0Bol<03Go001lo`040?oo00So1@04o`80BOl300co0P0co`00M_l400;o00@0ool0
2?l0103oo`05o`030?oo04Oo00<0ool03ol00`3oo`0`o`00O?l0103oo`09o`030?l000Go00<0ool0
AOl201?o0P0`o`00OOl200_o0P05o`030?oo04Co00<0ool05Ol00`3oo`0]o`00DOl603[o00<0ool0
@ol00`3oo`0Go`030?oo02co001;o`H01_l403Ko00<0ool0@ol00`3oo`0Ho`030?oo02_o000no`d0
4?l203Co00<0ool0@_l00`3oo`0Io`030?oo02_o000ho`H07_l00`3oo`0bo`030?oo04;o00<0ool0
6_l00`3oo`0Zo`00=ol00`3oo`0So`030?oo037o00<0ool0@_l00`3oo`0Jo`030?oo02[o000do`<0
9ol00`3oo`0`o`030?oo047o00<0ool07?l00`3oo`0Yo`00<ol00`3oo`0Xo`030?oo033o00<0ool0
@Ol00`3oo`0Lo`030?oo02Wo000bo`030?oo02Wo00<0ool0<?l00`3oo`11o`030?oo01go00<0ool0
:?l0037o00<0ool0:Ol00`3oo`0ao`030?oo047o00<0ool07Ol00`3oo`0Xo`00<Ol00`3oo`0Yo`03
0?oo037o00<0ool0@Ol00`3oo`0No`030?oo02Oo000`o`030?oo02Wo00<0ool0<_l00`3oo`11o`03
0?oo01ko00<0ool09ol002oo00<0ool0:Ol00`3oo`0Oo`801?l200?o1005o`030?oo047o00<0ool0
7_l00`3oo`0Wo`00;_l00`3oo`0Yo`030?oo01oo00@0ool02?l00`3oo`06o`030?oo043o00<0ool0
7ol00`3oo`0Wo`00;_l00`3oo`0Wo`808_l0103oo`09o`030?oo00Go0P11o`030?oo01oo00<0ool0
9ol002ko00<0ool09Ol201ko1002o`040?oo00[o00<0ool01?l00`3oo`10o`030?oo01oo0P0Xo`00
;Ol00`3oo`0Uo`030?oo02Co00@0ool02?l0103oo`05o`030?oo03oo00<0ool08Ol00`3oo`0Vo`00
;Ol00`3oo`0To`030?oo02Ko0P0:o`801_l00`3oo`0oo`030?oo027o00<0ool09_l002go00<0ool0
8ol00`3oo`0ko`030?oo03ko00<0ool08_l00`3oo`0Vo`00;Ol00`3oo`0Ro`80?Ol00`3oo`0no`03
0?oo02;o00<0ool09_l002go00<0ool08_l00`3oo`0lo`030?oo03go00<0ool08ol00`3oo`0Vo`00
1ol500;o0P04o`805ol00`3oo`09o`D05?l00`3oo`0;o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200go00<0ool04ol500[o00<0ool05Ol500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Fo`03
0?oo00_o00<0ool05?l200_o00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`0;o`03
0?oo01Ko00<0ool02_l00`3oo`0Go`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Do`03
0?oo00_o00<0ool05Ol00`3oo`09o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
2?l00`3oo`0Go`030?oo00[o00<0ool05ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Go`030?oo00?o1004o`030?oo01Ko00<0ool00_l400;o00@0ool02Ol202?o00<0ool08Ol0103o
o`09o`802_l00`3oo`0Ho`030?oo00[o00<0ool05ol00`3oo`08o`800ol000Wo00<0ool02?l00`3o
o`0Go`030?oo00[o00<0ool05ol200So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3o
o`08o`030?oo01Wo00<0ool02_l00`3oo`0Go`030?oo00So00<0ool00_l000So0P0:o`<05ol00`3o
o`09o`806ol00`3oo`06o`802_l302;o00<0ool08_l200[o0`07o`030?oo01Wo0P0;o`030?oo01Oo
0P0:o`<00_l002oo00<0ool09_l00`3oo`0fo`030?oo03Oo00<0ool09ol00`3oo`0Xo`00;ol00`3o
o`0Wo`030?oo03Go00<0ool0=_l00`3oo`0Wo`030?oo02Wo000_o`030?oo02So00<0ool0=?l00`3o
o`0eo`030?oo02Oo00<0ool0:_l0033o00<0ool0:?l203Co00<0ool0=?l00`3oo`0Xo`030?oo02[o
000`o`030?oo02[o0P0bo`030?oo03?o00<0ool0:?l00`3oo`0[o`003?oo00`02Ol000co00<0ool0
1Ol00`3oo`06o`030?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool00ol400So00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool00_l30003o`3o00Oo00<0ool01Ol00`3o
o`06o`030?oo00Ko00<0ool01?l00`3o0008o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Oo
000<o`030?oo02Co00<0ool00_l00`3oo`0Yo`<0:_l00`3oo`0Yo`030?oo00809ol00`3oo`03o`03
0?oo02Co00<0ool01ol003Co00<0ool0<?l402Ko00<0ool0:?l402go0P0ao`00=Ol00`3oo`0co`80
9?l00`3oo`0So`D0;_l303?o000fo`030?oo03Co1@0Oo`030?oo01ko1@0ao`80=_l003Oo0P0io`@0
6ol00`3oo`0Jo`@0<ol303So000io`80>ol201Wo00<0ool05ol303Go0P0ko`00>ol203_o1@0Do`03
0?oo00_o300do`@0?Ol003go0P0no`X02_l00`3oo`02o`T0?Ol3047o000oo`80A_l?04;o1014o`00
@Ol204ko00<0ool0@?l404So0013o`80C?l00`3oo`0lo`@0C?l004Go0`19o`030?oo03Ko1P1@o`00
B?l204Oo00<0ool0<Ol505Ko001:o`<0A?l00`3oo`0/o`D0Fol004go0P0^o`801?l200?o1005o`03
0?oo02So101Po`00Col302[o00@0ool02?l00`3oo`06o`030?oo02Go0`1To`00D_l202So00@0ool0
2Ol00`3oo`05o`808ol306Oo001Do`@09?l0103oo`0:o`030?oo00Co00<0ool08?l206[o001Ho`@0
8?l0103oo`08o`040?oo00Go00<0ool07_l206co001Lo`@07Ol200[o0P06o`030?oo01_o0`1^o`00
H?l402go00<0ool06?l3077o001To`@0:Ol00`3oo`0Fo`80M?l006So100Uo`030?oo01?o0`1fo`00
K?l5023o00<0ool03Ol607Wo001ao`@07?l00`3oo`03o`X0Ool007Go1`0<o`l0ROl007co3009o`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co001mo`80
1?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D0
1?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0
S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<0
1_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80
SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`80
2_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87761 .15712 m
.87333 .14833 L
.86787 .13948 L
.86131 .13138 L
.85435 .12502 L
.84998 .1219 L
.84558 .11932 L
.83663 .1154 L
.83122 .11373 L
.82599 .11253 L
.82317 .11203 L
.82002 .11159 L
.81841 .11141 L
.81667 .11125 L
.81501 .11113 L
.81428 .11108 L
.81348 .11105 L
.81202 .11099 L
.81064 .11096 L
.80929 .11096 L
.80804 .11097 L
.80669 .11101 L
.8052 .11107 L
.80445 .11111 L
.80364 .11116 L
.80219 .11128 L
.79938 .11157 L
.79672 .11195 L
.79181 .1129 L
.78905 .11359 L
.78617 .11446 L
.78119 .11633 L
.77618 .1188 L
.77113 .12208 L
.76723 .12538 L
.76363 .12926 L
.761 .13289 L
.75858 .13714 L
.75647 .14201 L
.75473 .14746 L
.7535 .15276 L
.75263 .15765 L
.75121 .16804 L
.75042 .17367 L
.74954 .17887 L
.7486 .18323 L
.74736 .18784 L
.74375 .19742 L
.74147 .20203 L
.73861 .207 L
Mistroke
.73258 .21589 L
.71949 .23175 L
.70478 .24608 L
.69641 .25212 L
.68848 .25622 L
.68438 .25779 L
.67988 .25919 L
.67134 .26119 L
.65028 .26497 L
.63007 .26963 L
.61263 .27488 L
.60316 .27765 L
.59416 .27977 L
.5858 .2812 L
.58091 .28183 L
.57626 .2823 L
.53433 .28568 L
.52437 .28677 L
.51422 .28785 L
.51008 .28823 L
.50568 .28857 L
.50313 .28872 L
.50082 .28883 L
.49969 .28888 L
.49847 .28892 L
.4974 .28895 L
.49625 .28898 L
.49502 .289 L
.49387 .28901 L
.49317 .28901 L
.4925 .28902 L
.49123 .28901 L
.4899 .289 L
.48866 .28899 L
.48728 .28896 L
.48581 .28892 L
.48339 .28884 L
.4811 .28874 L
.47577 .28845 L
.46451 .28761 L
.45217 .28648 L
.43032 .28405 L
.4203 .28248 L
.41006 .28021 L
.40088 .27741 L
.39247 .2743 L
.37227 .2663 L
.33295 .25532 L
.32552 .25355 L
.31794 .2512 L
Mistroke
.3097 .24758 L
.30166 .24293 L
.2839 .23014 L
.27446 .22228 L
.27074 .21885 L
.26738 .21535 L
.26603 .21375 L
.26479 .21208 L
.26391 .21066 L
.26321 .20922 L
.26291 .20842 L
.26279 .20801 L
.26268 .20757 L
.26261 .20717 L
.26256 .20681 L
.26253 .20645 L
.26252 .20612 L
.26256 .20545 L
.26265 .20485 L
.26272 .20454 L
.26283 .2042 L
.26309 .20352 L
.2634 .20292 L
.2638 .2023 L
.2647 .2012 L
.26591 .20003 L
.26754 .19875 L
.27519 .19418 L
.28627 .18815 L
.29707 .18107 L
.30132 .1775 L
.30526 .17345 L
.30704 .17123 L
.30874 .16872 L
.30996 .16657 L
.31103 .1642 L
.31151 .1629 L
.31187 .1617 L
.31218 .16046 L
.3124 .15934 L
.31255 .15823 L
.31265 .15719 L
.31269 .1561 L
.31267 .15493 L
.31258 .15375 L
.31249 .15309 L
.3124 .15247 L
.31215 .15125 L
.31179 .14995 L
.31093 .14767 L
Mistroke
.30992 .14568 L
.30735 .14207 L
.30558 .14022 L
.30376 .13864 L
.30004 .1361 L
.29563 .13394 L
.29314 .13302 L
.29069 .1323 L
.28845 .13177 L
.28596 .13132 L
.28454 .13112 L
.28324 .13098 L
.28252 .13091 L
.28174 .13086 L
.28032 .13078 L
.27899 .13074 L
.27758 .13073 L
.27637 .13075 L
.27506 .1308 L
.27374 .13088 L
.27249 .13098 L
.27017 .13121 L
.26769 .13155 L
.26501 .13199 L
.25599 .13393 L
.24676 .13622 L
.23686 .13834 L
.23407 .1388 L
.2311 .13921 L
.22831 .13953 L
.22568 .13977 L
.22318 .13995 L
.22047 .14008 L
.21891 .14014 L
.21743 .14018 L
.21609 .14021 L
.21533 .14022 L
.21463 .14023 L
.21334 .14024 L
.21215 .14024 L
.21078 .14024 L
.20946 .14023 L
.20825 .14022 L
.20712 .1402 L
.20455 .14017 L
.20175 .14013 L
.1991 .14009 L
.19762 .14008 L
.1968 .14007 L
.19604 .14006 L
Mistroke
.19469 .14006 L
.19327 .14006 L
.19185 .14007 L
.19056 .14009 L
.18985 .1401 L
.18907 .14012 L
.18767 .14016 L
.18637 .1402 L
.18499 .14026 L
.18253 .14041 L
.17971 .14064 L
.1771 .14092 L
.17429 .14131 L
.17132 .14183 L
.16635 .143 L
.16193 .14441 L
.15733 .14634 L
.1527 .14885 L
.14816 .15198 L
.14418 .1554 L
.13755 .1629 L
.13218 .17139 L
.12991 .17606 L
.12778 .18143 L
.12629 .18619 L
.12561 .18886 L
.12506 .19142 L
.12464 .19382 L
.12427 .19646 L
.12415 .19762 L
.12403 .19886 L
.12395 .20002 L
.12388 .2011 L
.12384 .20233 L
.12381 .20369 L
.12381 .20501 L
.12384 .20643 L
.12389 .20774 L
.12397 .20898 L
.12406 .2101 L
.12418 .2113 L
.12454 .21406 L
.12504 .21695 L
.12628 .22207 L
.12778 .22669 L
.12947 .23089 L
.13425 .23992 L
.14058 .24876 L
.14736 .25619 L
.16125 .26788 L
Mistroke
.17734 .27808 L
.19364 .2864 L
.23087 .30266 L
.24915 .31088 L
.26841 .3196 L
.27764 .3233 L
.28766 .32668 L
.29728 .32926 L
.30618 .33111 L
.31542 .33261 L
.32526 .33396 L
.34269 .33659 L
.35197 .33853 L
.36053 .34077 L
.37999 .34702 L
.38925 .35007 L
.39933 .35301 L
.40402 .35416 L
.40899 .35521 L
.41369 .35603 L
.41801 .35663 L
.42257 .3571 L
.42509 .35729 L
.4275 .35744 L
.42858 .35749 L
.42974 .35754 L
.43074 .35757 L
.43182 .3576 L
.43298 .35762 L
.43407 .35764 L
.43537 .35765 L
.43657 .35765 L
.43786 .35764 L
.43923 .35762 L
.44054 .3576 L
.44173 .35757 L
.44466 .35747 L
.44744 .35734 L
.4525 .35704 L
.45801 .35663 L
.48025 .35441 L
.50261 .35179 L
.53885 .34824 L
.5493 .34751 L
.55958 .34674 L
.56911 .34578 L
.57457 .34507 L
.57963 .34428 L
.58899 .34235 L
.59901 .33951 L
Mistroke
.60897 .33572 L
.61778 .33153 L
.63371 .32271 L
.65141 .3139 L
.66174 .31023 L
.67187 .30763 L
.6819 .30577 L
.69277 .30424 L
.71337 .30139 L
.72454 .29926 L
.73455 .29675 L
.77321 .28446 L
.79388 .27918 L
.81366 .27502 L
.83255 .27016 L
.84279 .26643 L
.85168 .26221 L
.8604 .25686 L
.86764 .25116 L
.87447 .2442 L
.8777 .24007 L
.88029 .23614 L
.88408 .22865 L
.88559 .22456 L
.88683 .22013 L
.88724 .21823 L
.88762 .21618 L
.88789 .21438 L
.88812 .21241 L
.88824 .2112 L
.88833 .21007 L
.88839 .20898 L
.88844 .20782 L
.88848 .20678 L
.88849 .2058 L
.8885 .20471 L
.88848 .20354 L
.88847 .20288 L
.88845 .20226 L
.8884 .2011 L
.88834 .19984 L
.88824 .19847 L
.88802 .19597 L
.88776 .19365 L
.88695 .18821 L
.88579 .18236 L
.88317 .17232 L
.87937 .16136 L
.87761 .15712 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`1;o`X0=ol007co00@0ool0
2ol00`3oo`03o`030?oo04Oo100:o`<0=?l007co00@0ool02?l500Co0P17o`030?oo00oo0P0bo`00
M_l400;o00@0ool02?l0103oo`05o`030?oo04Go00<0ool04_l00`3oo`0_o`00O?l0103oo`09o`03
0?l000Go00<0ool0A?l00`3oo`0Do`80;ol005?o1P0To`802ol200Go00<0ool0@ol00`3oo`0Go`03
0?oo02co001?o`@01Ol303Ko00<0ool0@_l00`3oo`0Ho`030?oo02co001:o`D03?l203Co00<0ool0
@Ol00`3oo`0Jo`030?oo02_o000ioa404ol00`3oo`0ao`030?oo047o00<0ool06_l00`3oo`0[o`00
=Ol402Go00<0ool0<?l00`3oo`10o`030?oo01co00<0ool0:_l003Co00<0ool09ol00`3oo`0`o`03
0?oo043o00<0ool07Ol00`3oo`0Yo`00<ol00`3oo`0Yo`030?oo02oo00<0ool0@?l00`3oo`0Mo`03
0?oo02Wo000bo`030?oo02[o00<0ool0;ol00`3oo`10o`030?oo01ko00<0ool0:?l0037o00<0ool0
:ol00`3oo`0_o`030?oo043o00<0ool07_l00`3oo`0Xo`00<?l00`3oo`0[o`030?oo033o00<0ool0
?ol00`3oo`0Oo`030?oo02So000_o`030?oo02co00<0ool0<?l00`3oo`0oo`030?oo023o00<0ool0
9ol002oo00<0ool0:ol00`3oo`0ao`030?oo03oo00<0ool08?l00`3oo`0Wo`00;_l00`3oo`0/o`03
0?oo01go0P04o`800ol400Go00<0ool0?ol00`3oo`0Po`030?oo02Oo000]o`030?oo02co00<0ool0
7Ol0103oo`08o`030?oo00Ko00<0ool0?ol00`3oo`0Qo`030?oo02Ko000]o`030?oo02_o00<0ool0
7_l0103oo`09o`030?oo00Go0P10o`030?oo027o00<0ool09_l002go00<0ool0:Ol201_o1002o`04
0?oo00[o00<0ool01?l00`3oo`0no`030?oo02;o00<0ool09_l002go00<0ool09ol202?o00@0ool0
2?l0103oo`05o`030?oo03ko00<0ool08_l00`3oo`0Vo`00;Ol00`3oo`0To`<09_l200[o0P06o`03
0?oo03go00<0ool09?l00`3oo`0Uo`00;?l202Go00<0ool0>ol00`3oo`0mo`030?oo02Co00<0ool0
9Ol002co00<0ool08ol00`3oo`0lo`030?oo03go00<0ool09?l00`3oo`0Uo`00;?l00`3oo`0Ro`03
0?oo03go00<0ool0??l00`3oo`0Uo`030?oo02Go0007o`D00_l200Co0P0Fo`802ol501?o0P0=o`80
1?l200Co0P0So`030?oo02;o0P04o`801?l200co00<0ool05?l500co00<0ool04ol500;o0P04o`80
0ol000Wo00<0ool01ol0103oo`0Fo`030?oo00_o00<0ool05?l00`3oo`0:o`040?oo00So00@0ool0
8_l00`3oo`0Qo`040?oo00So00@0ool02_l00`3oo`0Go`030?oo00co00<0ool05Ol00`3oo`07o`04
0?oo00;o0009o`030?oo00[o00<0ool05?l00`3oo`0;o`030?oo01Go00<0ool02Ol0103oo`0;o`03
0?oo023o00<0ool08Ol0103oo`0;o`030?oo00Oo00<0ool06?l00`3oo`0<o`030?oo01Go00<0ool0
2_l1007o0Ol00003o`0000801?l00`3oo`08o`805ol00`3oo`03o`@01?l00`3oo`0Fo`030?oo00;o
1002o`040?oo00Wo0P0So`030?oo027o00@0ool02Ol200Wo00<0ool06Ol00`3oo`0;o`805ol00`3o
o`08o`800ol000Wo00<0ool02?l00`3oo`0Fo`030?oo00_o00<0ool05ol00`3oo`07o`040?oo00Wo
00<0ool08_l00`3oo`0Qo`040?oo00Wo00<0ool01ol00`3oo`0Jo`030?oo00_o00<0ool05_l00`3o
o`08o`030?oo00;o0008o`802_l301Ko00<0ool02_l201[o00<0ool01ol200[o0`0Ro`030?oo02;o
0P0:o`<01ol00`3oo`0Io`803Ol00`3oo`0Eo`802_l300;o000^o`030?oo02Ko00<0ool0=ol00`3o
o`0go`030?oo02Wo00<0ool09_l002oo00<0ool09_l203Oo00<0ool0=_l00`3oo`0Yo`030?oo02Oo
000_o`030?oo02So0P0eo`030?oo03Go00<0ool0:Ol00`3oo`0Xo`00<?l00`3oo`0Yo`80<ol00`3o
o`0do`030?oo02Wo00<0ool0:Ol0033o00<0ool0:ol2037o00<0ool0<ol00`3oo`0Yo`030?oo02[o
000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`040?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`04o`@01ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`80
2?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`802?l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol00`3o000302Oo00<0ool0:Ol00`3o
o`0502Co00<0ool00ol202Go00<0ool01ol003Co00<0ool0<ol402?o00<0ool0:?l402go0P0ao`00
=Ol00`3oo`0fo`<08?l00`3oo`0Ro`H0;ol203?o000fo`80>Ol201ko00<0ool07_l403;o0`0eo`00
>?l203Wo0`0Ko`030?oo01co0P0do`80>?l003[o0P0jo`806Ol00`3oo`0Ho`@0<_l403[o000lo`03
0?oo03Wo0`0Fo`030?oo01;o1P0`o`H0?_l003go0P0lo`L03ol00`3oo`05o`d0<_l404Co000oo`80
@OlG03_o1018o`00@Ol204ko00<0ool0?Ol304co0013o`<0Bol00`3oo`0jo`<0Col004Ko0P19o`03
0?oo03Oo0`1Bo`00B?l304Ko00<0ool0;ol805Go001;o`80A?l00`3oo`0Yo`H0GOl004go0P0^o`80
1?l200?o1005o`030?oo02Oo0P1So`00Col202_o00@0ool02?l00`3oo`06o`030?oo02Co0`1Uo`00
DOl202Wo00@0ool02Ol00`3oo`05o`808ol206So001Co`@09Ol0103oo`0:o`030?oo00Co00<0ool0
8?l206[o001Go`<08_l0103oo`08o`040?oo00Go00<0ool07_l206co001Jo`808Ol200[o0P06o`03
0?oo01go00<0ool0K?l005co2@0/o`030?oo01_o0P1_o`00IOl302Wo00<0ool06?l3077o001Xo`D0
9?l00`3oo`0Fo`80M?l006go0`0Qo`030?oo00ko201fo`00L?l201oo00<0ool00_l<07ko001bo`D0
5ol808[o001go`H01ol:00?o00<0ool0S?l007go1`0=o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200Go0`04o`030?oo08co001lo`040?oo00_o00<0ool0
0ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0
S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool02?l300Ko00<0ool0S?l007co00@0ool0
2Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.87986 .15974 m
.87536 .15056 L
.86985 .1413 L
.8634 .13276 L
.85669 .12591 L
.84825 .11957 L
.83954 .11493 L
.83425 .11283 L
.82814 .11093 L
.82521 .11021 L
.82203 .10954 L
.81903 .10902 L
.81613 .10861 L
.81333 .10831 L
.81172 .10818 L
.81023 .10808 L
.80883 .10801 L
.80757 .10797 L
.8061 .10794 L
.80471 .10794 L
.80319 .10796 L
.80158 .10801 L
.80021 .10808 L
.79869 .10818 L
.79707 .10831 L
.79533 .1085 L
.79221 .10893 L
.78934 .10944 L
.78626 .11012 L
.78083 .1117 L
.77579 .11366 L
.77142 .11585 L
.76697 .11866 L
.76267 .12214 L
.75914 .1258 L
.75635 .12945 L
.75378 .13377 L
.75154 .13877 L
.74979 .1442 L
.74861 .14938 L
.7461 .1687 L
.74539 .17393 L
.7444 .17942 L
.74329 .18408 L
.74175 .1892 L
.73715 .20028 L
.73144 .21073 L
.71802 .2307 L
.71077 .23996 L
.70391 .24744 L
Mistroke
.69765 .25277 L
.69399 .25516 L
.69046 .257 L
.68641 .25861 L
.68178 .25995 L
.67716 .2609 L
.67268 .2616 L
.65361 .26397 L
.6432 .26579 L
.63374 .26811 L
.61418 .27484 L
.60634 .27772 L
.59811 .28027 L
.59371 .28132 L
.58872 .28225 L
.584 .2829 L
.5794 .28337 L
.57399 .28378 L
.56885 .28408 L
.55688 .28473 L
.54606 .28551 L
.5407 .28602 L
.53495 .28666 L
.52367 .28815 L
.5135 .28951 L
.50912 .29001 L
.50447 .29043 L
.50202 .2906 L
.49931 .29075 L
.4981 .2908 L
.49681 .29085 L
.49557 .29088 L
.49444 .2909 L
.49326 .29092 L
.49195 .29093 L
.49128 .29093 L
.49056 .29093 L
.48925 .29091 L
.48801 .2909 L
.48669 .29087 L
.48429 .2908 L
.48145 .29068 L
.47876 .29055 L
.47251 .29016 L
.44994 .28824 L
.4378 .28698 L
.42714 .28557 L
.41838 .28394 L
.41376 .28279 L
.40964 .28155 L
Mistroke
.3935 .27471 L
.37396 .26527 L
.3636 .2612 L
.35362 .25818 L
.34824 .25692 L
.34257 .25584 L
.33334 .25447 L
.32919 .25389 L
.32493 .25319 L
.32108 .25236 L
.31763 .25138 L
.31362 .24987 L
.3093 .24778 L
.3012 .24276 L
.28274 .2288 L
.26688 .21557 L
.26385 .2126 L
.26257 .21116 L
.26148 .20973 L
.26077 .20856 L
.26047 .20793 L
.26026 .20736 L
.26013 .20686 L
.26007 .20641 L
.26006 .20593 L
.26011 .20544 L
.26025 .20493 L
.26045 .20448 L
.26077 .20398 L
.26115 .20352 L
.26202 .20272 L
.26325 .20187 L
.26707 .19996 L
.27171 .19809 L
.28158 .19413 L
.29328 .18828 L
.29972 .184 L
.30501 .17951 L
.3095 .17448 L
.3128 .16932 L
.31413 .16644 L
.31467 .16501 L
.31509 .16369 L
.31541 .1625 L
.3157 .16119 L
.31593 .15987 L
.31609 .15848 L
.31615 .1577 L
.31619 .15695 L
.3162 .15557 L
Mistroke
.31613 .15409 L
.31605 .15325 L
.31596 .15249 L
.31575 .15123 L
.31545 .14985 L
.31462 .14713 L
.31358 .14466 L
.31242 .1425 L
.31095 .14028 L
.30912 .13802 L
.30536 .13443 L
.30127 .1316 L
.29869 .13022 L
.29578 .12897 L
.29295 .12802 L
.29029 .12734 L
.28787 .12689 L
.28648 .1267 L
.28519 .12656 L
.28382 .12646 L
.28257 .12641 L
.28114 .12639 L
.27978 .12641 L
.27844 .12648 L
.27768 .12653 L
.27697 .12659 L
.27554 .12674 L
.27401 .12694 L
.27128 .12742 L
.26882 .12796 L
.2633 .12952 L
.2577 .13145 L
.24852 .1349 L
.24354 .13661 L
.23877 .13797 L
.23423 .13896 L
.2316 .1394 L
.22914 .13972 L
.22703 .13992 L
.2258 .14002 L
.22467 .14009 L
.22347 .14015 L
.22216 .14019 L
.22094 .14022 L
.21978 .14023 L
.21839 .14023 L
.2176 .14022 L
.21688 .14021 L
.21553 .14017 L
.2141 .14012 L
Mistroke
.2116 .14001 L
.20882 .13984 L
.19665 .13891 L
.19115 .13851 L
.18809 .13834 L
.18665 .13827 L
.18532 .13822 L
.18408 .13818 L
.18278 .13816 L
.18147 .13814 L
.18004 .13814 L
.17882 .13816 L
.17751 .13819 L
.17629 .13824 L
.17518 .1383 L
.17377 .13839 L
.17246 .1385 L
.17004 .13876 L
.16749 .13914 L
.16481 .13966 L
.16019 .14091 L
.15617 .14242 L
.15203 .14449 L
.1478 .14726 L
.14371 .15069 L
.14019 .15437 L
.13389 .16312 L
.1312 .16798 L
.12898 .17276 L
.12558 .18226 L
.12424 .18751 L
.12369 .19027 L
.1232 .19321 L
.12286 .19589 L
.12271 .19742 L
.1226 .19884 L
.12252 .20016 L
.12248 .20135 L
.12244 .20273 L
.12244 .20404 L
.12246 .20545 L
.12251 .20695 L
.12258 .20822 L
.12269 .20962 L
.12284 .21112 L
.12303 .21272 L
.12348 .2156 L
.12401 .21822 L
.1247 .22104 L
.12628 .22605 L
Mistroke
.13023 .23492 L
.13319 .23994 L
.1363 .24436 L
.14273 .25184 L
.15877 .26523 L
.17512 .27517 L
.19131 .28339 L
.22899 .30088 L
.26286 .31847 L
.2802 .32642 L
.2854 .32827 L
.29043 .32979 L
.29497 .33093 L
.30004 .33198 L
.30491 .33277 L
.3095 .33335 L
.31434 .33383 L
.31954 .33422 L
.32877 .3348 L
.3332 .33512 L
.33719 .33547 L
.34152 .33597 L
.34611 .33666 L
.35106 .33762 L
.35571 .33876 L
.36493 .34167 L
.37493 .34561 L
.39285 .3532 L
.40254 .35665 L
.40796 .35819 L
.41319 .35937 L
.4182 .36021 L
.42108 .36057 L
.42378 .36083 L
.42508 .36092 L
.4263 .361 L
.42743 .36106 L
.42862 .36111 L
.42992 .36114 L
.43113 .36117 L
.43244 .36118 L
.43318 .36118 L
.43388 .36118 L
.43514 .36116 L
.43652 .36113 L
.43783 .36109 L
.43905 .36104 L
.44137 .36092 L
.44385 .36076 L
.44908 .36033 L
Mistroke
.45492 .35971 L
.47718 .35656 L
.4998 .35289 L
.50972 .35137 L
.51985 .35004 L
.52433 .34957 L
.52907 .34917 L
.53356 .34888 L
.53768 .34867 L
.54709 .34838 L
.55248 .34824 L
.55767 .34808 L
.56222 .34788 L
.5672 .34758 L
.57006 .34736 L
.5728 .3471 L
.57795 .34648 L
.58293 .3457 L
.58817 .34463 L
.59731 .34208 L
.60216 .3403 L
.60656 .3384 L
.61601 .33336 L
.63117 .3232 L
.63969 .3176 L
.64805 .31303 L
.65698 .30937 L
.66227 .30774 L
.66725 .30652 L
.67266 .30548 L
.67864 .30458 L
.68967 .30336 L
.70063 .30231 L
.71058 .30107 L
.71544 .30027 L
.72055 .29923 L
.73109 .29638 L
.74949 .28951 L
.75939 .28556 L
.7703 .28167 L
.78136 .27851 L
.79174 .27629 L
.80245 .27456 L
.81271 .27316 L
.82363 .2716 L
.83509 .26948 L
.84465 .26701 L
.85469 .26344 L
.8645 .25862 L
.87245 .25336 L
Mistroke
.87636 .25013 L
.88017 .2464 L
.8833 .2427 L
.88585 .23908 L
.88799 .23533 L
.88962 .23177 L
.89104 .22769 L
.89163 .2255 L
.89215 .22314 L
.89251 .22096 L
.89268 .21974 L
.8928 .21858 L
.8929 .21749 L
.89299 .2163 L
.89304 .21521 L
.89307 .21419 L
.89309 .21298 L
.89309 .21183 L
.89308 .21117 L
.89307 .21056 L
.89304 .20986 L
.89301 .20918 L
.89294 .20784 L
.89285 .20658 L
.89257 .20368 L
.89223 .20102 L
.89185 .19854 L
.89077 .19283 L
.88806 .18214 L
.88081 .16191 L
.87986 .15974 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`1:o`P0>_l007go0P04o`801Ol300Co00<0ool0Aol300So0`0go`00
O?l0103oo`0;o`030?oo00?o00<0ool0AOl200ko100co`00O?l0103oo`08o`D01?l204Go00<0ool0
4_l00`3oo`0`o`00M_l400;o00@0ool02?l0103oo`05o`030?oo04;o0P0Fo`030?oo02oo001Do`H0
8_l0103oo`09o`030?l000Go00<0ool0@Ol00`3oo`0Go`030?oo02ko001@o`@01Ol3027o0P0;o`80
1Ol00`3oo`11o`030?oo01So00<0ool0;Ol004ko0P0<o`80<ol00`3oo`10o`030?oo01[o00<0ool0
;?l003Wo2@08o`@04?l00`3oo`0`o`030?oo03oo00<0ool07?l00`3oo`0[o`00=Ol400Wo200Do`80
<Ol00`3oo`0oo`030?oo01go00<0ool0:_l003Co00<0ool0:?l00`3oo`0_o`030?oo03oo00<0ool0
7Ol00`3oo`0Zo`00<ol00`3oo`0Yo`030?oo02oo00<0ool0?ol00`3oo`0No`030?oo02Wo000bo`03
0?oo02[o0P0`o`030?oo03oo00<0ool07_l00`3oo`0Yo`00<Ol00`3oo`0/o`030?oo02ko00<0ool0
?ol00`3oo`0Oo`030?oo02So000`o`030?oo02co0P0`o`030?oo03ko00<0ool08Ol00`3oo`0Wo`00
;ol00`3oo`0]o`030?oo02oo00<0ool0?_l00`3oo`0Qo`030?oo02Oo000^o`030?oo02ko00<0ool0
;ol00`3oo`0no`030?oo027o00<0ool09ol002ko00<0ool0;_l00`3oo`0_o`030?oo03ko00<0ool0
8_l00`3oo`0Vo`00;_l00`3oo`0]o`030?oo01co0P04o`800ol400Go00<0ool0?_l00`3oo`0Ro`03
0?oo02Ko000]o`030?oo02co0P0No`040?oo00So00<0ool01_l00`3oo`0no`030?oo02?o00<0ool0
9Ol002go00<0ool0:ol00`3oo`0No`040?oo00Wo00<0ool01Ol203ko00<0ool09?l00`3oo`0Uo`00
;Ol00`3oo`0[o`030?oo01So1002o`040?oo00[o00<0ool01?l00`3oo`0mo`030?oo02Co00<0ool0
9Ol002co00<0ool0:Ol3027o00@0ool02?l0103oo`05o`030?oo03go00<0ool09?l00`3oo`0Uo`00
;?l00`3oo`0Vo`<09Ol200[o0P06o`030?oo03go00<0ool09?l00`3oo`0Uo`00;?l00`3oo`0Uo`03
0?oo03[o00<0ool0??l00`3oo`0Vo`030?oo02Co000/o`030?oo02;o0`0mo`030?oo03co00<0ool0
9_l00`3oo`0To`00;?l00`3oo`0Ro`030?oo03go00<0ool0>ol00`3oo`0Wo`030?oo02Co0007o`D0
0_l200Co0P0Fo`030?oo00[o1@0Co`030?oo00co0P04o`801?l202?o00<0ool08_l200Co0P04o`80
2ol00`3oo`0Eo`D03Ol00`3oo`0Bo`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01Go00<0ool0
3?l00`3oo`0Co`803?l0103oo`08o`040?oo02;o00<0ool08Ol0103oo`08o`040?oo00Wo00<0ool0
6?l00`3oo`0=o`030?oo01Co00<0ool01ol0103oo`02o`002Ol00`3oo`0:o`030?oo01?o00<0ool0
3?l00`3oo`0Do`030?oo00[o00@0ool02ol00`3oo`0Po`030?oo027o00@0ool02ol00`3oo`06o`03
0?oo01Wo00<0ool03Ol00`3oo`0Do`030?oo00[o0@01o`7o00000ol0000200Co00<0ool02?l201Ko
0P05o`@01?l00`3oo`0Eo`801?l400;o00@0ool02Ol202?o00<0ool08Ol0103oo`09o`802?l00`3o
o`0Jo`030?oo00go00<0ool05?l00`3oo`08o`800ol000Wo00<0ool02?l00`3oo`0Fo`030?oo00_o
00<0ool05ol200So00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`07o`030?oo01[o
00<0ool03Ol00`3oo`0Do`030?oo00So00<0ool00_l000So0P0:o`<05_l00`3oo`0:o`806ol00`3o
o`06o`802_l302;o00<0ool08_l200[o0`06o`030?oo01[o0P0?o`030?oo01?o0P0:o`<00_l002ko
00<0ool09ol00`3oo`0fo`030?oo03Ko00<0ool0:ol00`3oo`0Uo`00;_l00`3oo`0Xo`80=_l00`3o
o`0eo`030?oo02co00<0ool09Ol002oo00<0ool0:Ol00`3oo`0co`030?oo03Go00<0ool0:ol00`3o
o`0Vo`00<?l00`3oo`0Yo`80<ol00`3oo`0do`030?oo02_o00<0ool09ol0037o00<0ool0:_l00`3o
o`0`o`030?oo03?o00<0ool0;?l00`3oo`0Wo`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l0103oo`05o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
0ol500Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`05o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Ko00<0ool01?l300So00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1_l300Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01ol000co00<0ool09?l200?o00<0ool0
:Ol00`3o000502Go00<0ool0:Ol00`3oo`02o`809Ol00`3oo`05o`808ol00`3oo`07o`00=Ol00`3o
o`0do`808ol00`3oo`0Xo`H0;_l00`3oo`0]o`00=_l00`3oo`0eo`808Ol00`3oo`0Ro`H0<_l2033o
000go`030?oo03Ko0P0Oo`030?oo01oo0`0co`D0<_l003So0P0ho`807Ol00`3oo`0Mo`80<?l603Oo
000jo`030?oo03Oo0`0Jo`030?oo01Wo100/o`H0?Ol003_o0P0jo`<05ol00`3oo`0Eo`@0:_l604?o
000mo`80>ol501;o00<0ool02Ol<02co0P19o`00?ol203ko200:o`030?oo00;o1`0go`030?oo04Wo
0011o`80A?l?03_o0`1<o`00@ol204co00<0ool0>Ol404oo0015o`80B_l00`3oo`0fo`<0Dol004Oo
0`17o`030?oo02ko201Fo`00B_l204Go00<0ool09ol705ko001<o`80;ol200Co0P03o`@01Ol00`3o
o`0Uo`80IOl004ko0P0/o`040?oo00So00<0ool01_l00`3oo`0So`80Iol0053o0`0Yo`040?oo00Wo
00<0ool01Ol202;o0P1Yo`00Dol202Oo00@0ool02_l00`3oo`04o`030?oo01oo0P1[o`00EOl202Go
00@0ool02?l0103oo`05o`030?oo01go0P1]o`00Eol302?o0P0:o`801_l00`3oo`0Lo`030?oo06go
001Jo`P0;ol00`3oo`0Ko`030?oo06ko001Ro`P09ol00`3oo`0Jo`030?oo06oo001Zo`@08ol00`3o
o`0Eo`D0L_l006ko0`0Po`030?oo01;o0`1go`00LOl201ko00<0ool00ol?07[o001co`<05ol:08Wo
001fo`804?l500Co00<0ool0S?l007So1@05o`H02Ol00`3oo`2<o`00OOl500oo00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3o
o`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103o
o`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00
oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88211 .16237 m
.8774 .1528 L
.87182 .14313 L
.86542 .13403 L
.85903 .12681 L
.85102 .11989 L
.84645 .11679 L
.84107 .11374 L
.83568 .11125 L
.83032 .10923 L
.82487 .10758 L
.82179 .10682 L
.81877 .10618 L
.81584 .10566 L
.81258 .10519 L
.80957 .10486 L
.80821 .10474 L
.80673 .10464 L
.80582 .10459 L
.80498 .10455 L
.80333 .10449 L
.80182 .10447 L
.80022 .10447 L
.79849 .1045 L
.79688 .10456 L
.79511 .10466 L
.79322 .10481 L
.79008 .10516 L
.78669 .10567 L
.78353 .10629 L
.78066 .10698 L
.77539 .1086 L
.7701 .11076 L
.76479 .1136 L
.7604 .11663 L
.75615 .12038 L
.75237 .12475 L
.74965 .12885 L
.7476 .13287 L
.74583 .13743 L
.74444 .14244 L
.74389 .1451 L
.74341 .14794 L
.74276 .15304 L
.74172 .16456 L
.74122 .16934 L
.74049 .17448 L
.73936 .18015 L
.73795 .18551 L
.73402 .19639 L
Mistroke
.72824 .20867 L
.71555 .231 L
.70898 .24107 L
.70286 .24899 L
.69942 .25256 L
.69558 .25571 L
.69193 .25795 L
.69009 .25882 L
.68837 .25951 L
.68463 .26064 L
.68247 .26111 L
.68036 .26148 L
.67642 .26196 L
.6741 .26216 L
.67189 .26232 L
.66673 .26262 L
.66091 .26297 L
.65527 .26344 L
.65004 .26405 L
.64528 .2648 L
.64015 .26585 L
.63024 .26865 L
.62014 .27256 L
.61161 .27639 L
.60304 .28 L
.59882 .28144 L
.59485 .28253 L
.59111 .28333 L
.58699 .28398 L
.58477 .28424 L
.58265 .28445 L
.57766 .28482 L
.56821 .28523 L
.5628 .28543 L
.55762 .28566 L
.55202 .28599 L
.54586 .28648 L
.54072 .28701 L
.5353 .28769 L
.52399 .28944 L
.51416 .29105 L
.50932 .29172 L
.50479 .29223 L
.50247 .29243 L
.49997 .2926 L
.49852 .29268 L
.49714 .29274 L
.49588 .29279 L
.49518 .29281 L
.49451 .29282 L
Mistroke
.49324 .29284 L
.49206 .29285 L
.49078 .29286 L
.48937 .29285 L
.48813 .29283 L
.48679 .2928 L
.48552 .29277 L
.48435 .29273 L
.48144 .29261 L
.47823 .29245 L
.47223 .29207 L
.45986 .29109 L
.4462 .28985 L
.43482 .28863 L
.42897 .28785 L
.42376 .28697 L
.41822 .28578 L
.41349 .28445 L
.40488 .28104 L
.38968 .2724 L
.38006 .26676 L
.37014 .26182 L
.35989 .25792 L
.35409 .25628 L
.34887 .25514 L
.34383 .25434 L
.34108 .25402 L
.3386 .25378 L
.33381 .25346 L
.32907 .2532 L
.32699 .25307 L
.32485 .2529 L
.32308 .25271 L
.32118 .25243 L
.31918 .25204 L
.31706 .2515 L
.31324 .25013 L
.30929 .24816 L
.30532 .24573 L
.2971 .23976 L
.27706 .22388 L
.26864 .21727 L
.26153 .2114 L
.26013 .21008 L
.25896 .20883 L
.25824 .20787 L
.25794 .20737 L
.25774 .20693 L
.25762 .20651 L
.25758 .20613 L
Mistroke
.25761 .20581 L
.25771 .20548 L
.25789 .20514 L
.25818 .20479 L
.2585 .20451 L
.25893 .20422 L
.26017 .20361 L
.26164 .20307 L
.26608 .20184 L
.27732 .19885 L
.28353 .1968 L
.2904 .19397 L
.29628 .19093 L
.30233 .18699 L
.30866 .18151 L
.31118 .17872 L
.31353 .1756 L
.31536 .17265 L
.31702 .16929 L
.31819 .16615 L
.319 .16316 L
.31932 .1615 L
.31945 .16058 L
.31956 .15972 L
.31969 .15821 L
.31974 .15653 L
.31974 .15565 L
.3197 .15473 L
.31958 .15308 L
.31947 .15216 L
.31935 .15132 L
.31899 .14941 L
.31858 .14777 L
.31811 .14624 L
.31705 .14345 L
.31563 .14057 L
.3138 .13762 L
.30992 .13292 L
.30507 .12877 L
.30047 .12598 L
.29774 .12471 L
.29512 .12374 L
.29364 .1233 L
.29207 .12289 L
.2907 .12259 L
.28922 .12233 L
.28832 .12221 L
.28748 .12211 L
.2867 .12203 L
.28584 .12197 L
Mistroke
.28436 .1219 L
.28355 .12189 L
.2828 .1219 L
.2813 .12195 L
.27988 .12205 L
.27863 .12218 L
.27727 .12236 L
.27429 .12293 L
.27159 .12361 L
.26704 .12513 L
.26225 .12715 L
.25313 .13173 L
.24443 .13593 L
.23999 .13762 L
.23738 .13842 L
.23496 .13903 L
.23221 .13957 L
.23078 .13979 L
.2292 .13999 L
.22771 .14013 L
.22632 .14024 L
.22498 .1403 L
.22353 .14035 L
.22221 .14036 L
.22078 .14035 L
.2194 .14032 L
.21814 .14027 L
.21664 .14019 L
.21499 .14007 L
.21194 .13981 L
.20041 .13838 L
.18774 .13667 L
.18426 .13629 L
.18066 .13597 L
.17919 .13586 L
.17764 .13578 L
.17675 .13574 L
.17595 .13571 L
.17515 .13569 L
.1744 .13567 L
.17295 .13566 L
.17223 .13567 L
.17144 .13568 L
.17006 .13572 L
.16879 .13579 L
.16803 .13584 L
.16731 .13589 L
.16598 .13602 L
.16446 .13621 L
.16306 .13642 L
Mistroke
.16057 .1369 L
.1584 .13745 L
.15377 .13907 L
.15007 .1409 L
.14634 .14334 L
.14254 .14658 L
.13885 .15058 L
.13302 .1591 L
.12828 .16874 L
.12604 .17462 L
.12431 .18022 L
.12299 .18559 L
.12242 .18846 L
.12191 .19154 L
.12157 .19424 L
.12142 .19567 L
.12129 .19722 L
.12119 .19864 L
.12113 .19994 L
.12109 .20119 L
.12106 .20249 L
.12107 .20401 L
.12108 .20486 L
.12111 .20564 L
.12117 .20708 L
.12127 .2086 L
.12139 .20995 L
.12152 .21122 L
.12191 .21406 L
.12246 .2171 L
.12308 .21983 L
.12489 .22579 L
.12686 .23065 L
.12941 .23567 L
.13498 .244 L
.1425 .25225 L
.14987 .2585 L
.16702 .2694 L
.20256 .28679 L
.23945 .30545 L
.25645 .31567 L
.26521 .32096 L
.27478 .3261 L
.27967 .32833 L
.28442 .33019 L
.28915 .33173 L
.29361 .3329 L
.29789 .33378 L
.30249 .33449 L
.30489 .33476 L
Mistroke
.30714 .33497 L
.30962 .33514 L
.31101 .33521 L
.31229 .33527 L
.31347 .33531 L
.31472 .33534 L
.31537 .33536 L
.31608 .33537 L
.31734 .33538 L
.31852 .33539 L
.31979 .33539 L
.32098 .33539 L
.32207 .33538 L
.3234 .33537 L
.32485 .33536 L
.32612 .33535 L
.32746 .33533 L
.32866 .33532 L
.32934 .33532 L
.32998 .33532 L
.33109 .33532 L
.33229 .33532 L
.33343 .33533 L
.33448 .33535 L
.33564 .33537 L
.33687 .33541 L
.33815 .33546 L
.33937 .33552 L
.34162 .33567 L
.34372 .33585 L
.34565 .33607 L
.34997 .33674 L
.35407 .33763 L
.35788 .33868 L
.3665 .34186 L
.38389 .35061 L
.39193 .35481 L
.40061 .35881 L
.4057 .36072 L
.41039 .36216 L
.4127 .36274 L
.41521 .36329 L
.4176 .36372 L
.41986 .36405 L
.42242 .36434 L
.4238 .36446 L
.42528 .36457 L
.42603 .36461 L
.42685 .36465 L
.42758 .36468 L
Mistroke
.42835 .3647 L
.42905 .36472 L
.42982 .36473 L
.43119 .36473 L
.43259 .36471 L
.43389 .36468 L
.43531 .36463 L
.43683 .36455 L
.43941 .36438 L
.44228 .36414 L
.44801 .3635 L
.45426 .36261 L
.46571 .36064 L
.48879 .35598 L
.49989 .35368 L
.51055 .35163 L
.51926 .35028 L
.52156 .35 L
.524 .34974 L
.5261 .34955 L
.52839 .34939 L
.53062 .34927 L
.53171 .34922 L
.53271 .34919 L
.53367 .34916 L
.53472 .34914 L
.53583 .34912 L
.53689 .34911 L
.53785 .3491 L
.53892 .3491 L
.54004 .34911 L
.5411 .34912 L
.5421 .34913 L
.54302 .34914 L
.54511 .34918 L
.5497 .34928 L
.5523 .34933 L
.55359 .34936 L
.55478 .34938 L
.5558 .34939 L
.55693 .3494 L
.55813 .34941 L
.55927 .34941 L
.56054 .34941 L
.5619 .3494 L
.5632 .34939 L
.56438 .34937 L
.56578 .34933 L
.56726 .34928 L
.56995 .34916 L
Mistroke
.57285 .34897 L
.57606 .34868 L
.57883 .34837 L
.58179 .34795 L
.58708 .34698 L
.59269 .34557 L
.5977 .34394 L
.60825 .33907 L
.61661 .33364 L
.62476 .32718 L
.63205 .32109 L
.63997 .31517 L
.64727 .31089 L
.65601 .30726 L
.66082 .30585 L
.66616 .30467 L
.67137 .30382 L
.6762 .30324 L
.68188 .30274 L
.68805 .30232 L
.6938 .30197 L
.6991 .3016 L
.70394 .30116 L
.70902 .30056 L
.71449 .29966 L
.71935 .2986 L
.72948 .29548 L
.73881 .2916 L
.75736 .28291 L
.76592 .27943 L
.77561 .27631 L
.78126 .27492 L
.78657 .27387 L
.79176 .27306 L
.79734 .27239 L
.80333 .27185 L
.809 .27145 L
.81524 .27106 L
.82208 .27062 L
.82861 .27011 L
.83463 .26949 L
.84034 .26871 L
.84631 .26766 L
.85632 .26518 L
.86174 .26339 L
.8666 .26144 L
.87587 .25661 L
.88 .25382 L
.88406 .25053 L
.88713 .24751 L
Mistroke
.88999 .24412 L
.89211 .24101 L
.89399 .23759 L
.89486 .23562 L
.89568 .23342 L
.89632 .23134 L
.8968 .2294 L
.89724 .22706 L
.89743 .22575 L
.89757 .22449 L
.89768 .22332 L
.89775 .22222 L
.89777 .22164 L
.8978 .221 L
.89781 .22032 L
.89782 .21969 L
.89782 .2186 L
.89779 .21741 L
.89776 .2168 L
.89773 .21614 L
.89766 .21496 L
.89746 .21261 L
.8972 .21044 L
.89633 .20499 L
.89519 .19966 L
.88942 .18039 L
.88211 .16237 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`1:o`L0>ol0097o00<0ool0A_l400Oo0`0ho`00OOl200Co0P05o`<01?l00`3oo`14o`80
3_l303Go001lo`040?oo00_o00<0ool00ol00`3oo`13o`030?oo017o0P0co`00O?l0103oo`08o`D0
1?l204?o00<0ool05?l2037o001Do`L06ol400;o00@0ool02?l0103oo`05o`030?oo047o00<0ool0
5ol00`3oo`0^o`00DOl300Oo0P0Oo`040?oo00Wo00<0o`001Ol00`3oo`10o`030?oo01Wo00<0ool0
;Ol0053o00<0ool02_l00`3oo`0Mo`802ol200Go00<0ool0?ol00`3oo`0Ko`030?oo02co001?o`03
0?oo00co00<0ool0<?l00`3oo`0no`030?oo01go00<0ool0:ol003Co3`07o`D04?l00`3oo`0_o`03
0?oo03ko00<0ool07Ol00`3oo`0[o`00<ol00`3oo`0=o`L05Ol00`3oo`0_o`030?oo03ko00<0ool0
7_l00`3oo`0Zo`00<Ol202go00<0ool0;_l00`3oo`0mo`030?oo01oo00<0ool0:_l0033o00<0ool0
;Ol00`3oo`0^o`030?oo03go00<0ool08?l00`3oo`0Yo`00<?l00`3oo`0]o`80;ol00`3oo`0mo`03
0?oo027o00<0ool0:?l002oo00<0ool0;ol00`3oo`0]o`030?oo03go00<0ool08Ol00`3oo`0Xo`00
;ol00`3oo`0^o`80;ol00`3oo`0mo`030?oo02;o00<0ool09ol002ko00<0ool0;ol00`3oo`0^o`03
0?oo03go00<0ool08_l00`3oo`0Wo`00;_l00`3oo`0_o`030?oo02ko00<0ool0?Ol00`3oo`0Ro`03
0?oo02Oo000]o`030?oo02oo0P0`o`030?oo03go00<0ool08ol00`3oo`0Vo`00;Ol00`3oo`0_o`03
0?oo01_o0P04o`800ol400Go00<0ool0??l00`3oo`0To`030?oo02Ko000]o`030?oo02ko00<0ool0
6ol0103oo`08o`030?oo00Ko00<0ool0??l00`3oo`0Uo`030?oo02Go000/o`030?oo02ko0P0Mo`04
0?oo00Wo00<0ool01Ol203go00<0ool09Ol00`3oo`0Uo`00;?l00`3oo`0/o`806Ol400;o00@0ool0
2_l00`3oo`04o`030?oo03co00<0ool09_l00`3oo`0To`00;?l00`3oo`0[o`030?oo01oo00@0ool0
2?l0103oo`05o`030?oo03co00<0ool09_l00`3oo`0To`00;?l00`3oo`0Yo`808ol200[o0P06o`03
0?oo03_o00<0ool09ol00`3oo`0To`00;?l00`3oo`0Uo`@0>Ol00`3oo`0ko`030?oo02So00<0ool0
8ol002co00<0ool08ol203go00<0ool0>_l00`3oo`0Yo`030?oo02?o000/o`030?oo027o0P0oo`03
0?oo03[o00<0ool0:Ol00`3oo`0So`001ol500;o0P04o`805_l00`3oo`0:o`D04_l200ko0P04o`80
1?l202?o00<0ool08_l200Co0P04o`802Ol00`3oo`0Go`D03_l00`3oo`0Ao`D00_l200Co0P03o`00
2Ol00`3oo`07o`040?oo01Go00<0ool03?l00`3oo`0Co`803?l0103oo`08o`040?oo02;o00<0ool0
8Ol0103oo`08o`040?oo00Oo00<0ool06_l00`3oo`0>o`030?oo01?o00<0ool01ol0103oo`02o`00
2Ol00`3oo`0:o`030?oo01?o00<0ool03?l00`3oo`0Eo`030?oo00Wo00@0ool02ol00`3oo`0Po`03
0?oo027o00@0ool02ol00`3oo`05o`030?oo01[o00<0ool03_l00`3oo`0Co`030?oo00[o0@01o`7o
00000ol0000200Co00<0ool02?l201Ko00<0ool01?l400Co00<0ool05_l00`3oo`02o`@00_l0103o
o`09o`808ol00`3oo`0Qo`040?oo00Wo0P07o`030?oo01_o00<0ool03_l00`3oo`0Co`030?oo00So
0P03o`002Ol00`3oo`08o`030?oo01Ko00<0ool02ol00`3oo`0Go`030?oo00Oo00@0ool02Ol00`3o
o`0Ro`030?oo027o00@0ool02Ol00`3oo`06o`030?oo01_o00<0ool03_l00`3oo`0Co`030?oo00So
00<0ool00_l000So0P0:o`<05_l00`3oo`0:o`806_l200So0P0:o`<08_l00`3oo`0Ro`802_l300Go
00<0ool06ol2013o00<0ool04_l200[o0`02o`00;Ol00`3oo`0Xo`030?oo03Ko00<0ool0=_l00`3o
o`0]o`030?oo02?o000^o`030?oo02So0P0fo`030?oo03Go00<0ool0;_l00`3oo`0So`00;ol00`3o
o`0Yo`030?oo03?o00<0ool0=Ol00`3oo`0]o`030?oo02Co000_o`030?oo02[o00<0ool0<_l00`3o
o`0do`030?oo02go0P0Vo`00<?l00`3oo`0Zo`030?oo037o00<0ool0<ol00`3oo`0]o`030?oo02Ko
000<ool03009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`02o`X00ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`0200Co00<0ool01_l00`3o
o`06o`030?oo00Ko00<0ool01ol000co00<0ool08ol200Co00<0ool0:Ol00`3oo`03o`<09?l00`3o
o`0Yo`030?oo00?o0`0So`030?oo00Oo0P0Qo`030?oo00Oo000do`80=ol3027o00<0ool09ol902ko
00<0ool0:ol003Ko0P0ho`807ol00`3oo`0Qo`H0<ol402ko000ho`030?oo03Oo00<0ool07?l00`3o
o`0No`<0<?l903;o000io`80>?l201co00<0ool07?l202[o2@0ko`00>ol203So0P0Jo`030?oo01[o
0P0Yo`<0A?l003go0P0ho`806?l00`3oo`0Go`<0:?l304Oo000oo`80>?l201Ko00<0ool03?l;02Wo
0P1:o`00@Ol00`3oo`0go`H04?l00`3oo`02o`X0<_l204co0012o`80?OlE03Wo0`1>o`00A?l204_o
00<0ool0>Ol2057o0016o`80BOl00`3oo`0do`D0Dol004So0P17o`030?oo02[o2P1Ho`00B_l304Co
00<0ool09Ol506;o001=o`80;_l200Co0P03o`@01Ol00`3oo`0So`80Iol004oo0P0[o`040?oo00So
00<0ool01_l00`3oo`0Ro`030?oo06Oo001Ao`030?oo02So00@0ool02Ol00`3oo`05o`808_l00`3o
o`1Xo`00D_l202So00@0ool02_l00`3oo`04o`030?oo023o00<0ool0JOl005Co0P0Vo`040?oo00So
00@0ool01Ol00`3oo`0Oo`030?oo06[o001Fo`809Ol200[o0P06o`030?oo01go0P1]o`00F?l503Co
00<0ool07?l00`3oo`1]o`00GOl>02Ko00<0ool06_l2073o001[o`<08ol00`3oo`0Ho`80L_l006ko
0P0Qo`030?oo01Go0`1do`00L?l201oo00<0ool00olB07Oo001bo`030?oo01co1`28o`00Lol201Ko
1`2>o`00MOl300ko1@06o`030?oo08co001ho`H000?o00001@0;o`030?oo08co001mo`<04Ol00`3o
o`2<o`00TOl00`3oo`2<o`00OOl200Co0P05o`<01?l00`3oo`2<o`00O?l0103oo`0;o`030?oo00?o
00<0ool0S?l007co00@0ool02?l500Co0P2=o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co
00@0ool02Ol00`3o0005o`030?oo08co001mo`802ol200Go00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool02?l0103oo`05o`030?oo08co
001lo`040?oo00So00@0ool01Ol208go001lo`040?oo00So0`06o`030?oo08co001lo`040?oo00Wo
00<0ool01Ol00`3oo`2<o`00OOl200[o0`05o`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co
002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co003oob7o003oob7o003oob7o003oob7o
003oob7o0000\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88435 .16499 m
.87379 .14495 L
.86137 .12771 L
.85379 .12024 L
.84559 .11415 L
.83494 .10844 L
.82813 .10577 L
.82141 .10375 L
.81505 .10232 L
.81173 .10175 L
.80799 .10125 L
.80475 .10094 L
.80292 .10081 L
.80123 .10073 L
.79966 .10068 L
.79794 .10065 L
.79631 .10065 L
.79479 .10068 L
.7929 .10075 L
.79183 .10081 L
.79085 .10088 L
.78909 .10102 L
.78716 .10122 L
.78382 .10167 L
.78035 .1023 L
.77665 .10315 L
.77278 .10426 L
.76618 .10676 L
.76004 .10995 L
.75501 .11343 L
.75027 .11775 L
.74618 .12282 L
.74343 .12751 L
.74146 .13211 L
.74057 .13479 L
.73991 .13728 L
.73927 .14025 L
.73882 .14296 L
.73848 .1455 L
.73823 .1479 L
.73784 .15319 L
.73757 .15783 L
.73725 .16275 L
.73704 .16534 L
.73675 .1682 L
.73604 .17329 L
.73551 .1762 L
.73484 .17936 L
.73331 .18518 L
.72919 .19703 L
Mistroke
.72334 .21062 L
.71128 .23503 L
.70526 .24552 L
.69961 .25327 L
.69637 .25653 L
.69459 .25794 L
.69265 .25921 L
.69095 .2601 L
.68901 .26092 L
.68717 .26151 L
.68539 .26195 L
.68438 .26215 L
.68328 .26233 L
.68125 .26258 L
.68009 .26269 L
.67897 .26276 L
.67773 .26283 L
.67702 .26285 L
.67636 .26287 L
.67523 .2629 L
.67416 .26291 L
.67298 .26292 L
.67169 .26292 L
.67103 .26292 L
.67031 .26292 L
.66899 .26291 L
.6677 .2629 L
.66648 .2629 L
.66534 .2629 L
.6641 .2629 L
.66271 .26291 L
.66144 .26293 L
.6601 .26296 L
.65863 .263 L
.65724 .26306 L
.65595 .26312 L
.65365 .26327 L
.65111 .26349 L
.64846 .26379 L
.64597 .26413 L
.6408 .26508 L
.63531 .26647 L
.62983 .26826 L
.62501 .2702 L
.60941 .27812 L
.60555 .28006 L
.60139 .2819 L
.59764 .28323 L
.59553 .28385 L
.59351 .28434 L
Mistroke
.58959 .28508 L
.58742 .28538 L
.58501 .28564 L
.58275 .28582 L
.58054 .28595 L
.57851 .28605 L
.57624 .28613 L
.57109 .28625 L
.5683 .28631 L
.56531 .28637 L
.56265 .28644 L
.55971 .28653 L
.55702 .28664 L
.55451 .28677 L
.54848 .28719 L
.54514 .2875 L
.54214 .28784 L
.53153 .28941 L
.52157 .2913 L
.51636 .29232 L
.51152 .29317 L
.50705 .29382 L
.50287 .29428 L
.50056 .29447 L
.49808 .29462 L
.49677 .29468 L
.49554 .29473 L
.49416 .29476 L
.49339 .29478 L
.49268 .29478 L
.49146 .29479 L
.4901 .29479 L
.48866 .29477 L
.48729 .29475 L
.486 .29472 L
.48481 .29468 L
.48208 .29457 L
.47574 .29423 L
.46949 .2938 L
.44542 .29176 L
.43369 .2905 L
.42765 .28964 L
.42256 .28872 L
.41812 .28768 L
.41366 .28631 L
.40625 .2831 L
.39879 .27854 L
.39033 .27246 L
.38029 .26558 L
.36996 .25987 L
Mistroke
.3647 .25758 L
.35888 .25554 L
.35338 .25408 L
.3484 .25314 L
.34574 .25278 L
.34331 .25254 L
.34195 .25244 L
.34072 .25236 L
.34005 .25233 L
.33932 .25231 L
.33864 .25229 L
.33802 .25227 L
.33738 .25226 L
.33669 .25226 L
.33604 .25226 L
.33545 .25226 L
.33436 .25227 L
.33322 .25229 L
.33202 .25233 L
.33093 .25236 L
.32853 .25245 L
.32753 .25249 L
.32648 .25251 L
.32561 .25253 L
.32467 .25254 L
.32415 .25254 L
.32367 .25253 L
.32316 .25252 L
.32262 .25251 L
.32173 .25246 L
.32123 .25243 L
.32077 .25239 L
.31988 .2523 L
.31905 .25218 L
.31751 .2519 L
.31586 .25148 L
.31407 .25088 L
.31084 .24941 L
.30723 .24726 L
.29995 .24191 L
.26175 .2122 L
.25848 .20977 L
.25715 .20871 L
.25609 .20774 L
.2557 .20732 L
.25542 .20697 L
.25525 .20667 L
.25514 .20638 L
.25512 .20613 L
.25518 .2059 L
Mistroke
.2553 .20571 L
.25551 .20553 L
.25568 .20542 L
.25588 .20532 L
.25632 .20516 L
.25691 .205 L
.25768 .20484 L
.25942 .20458 L
.2614 .20434 L
.26607 .2038 L
.27226 .20288 L
.27915 .20146 L
.28593 .19952 L
.29255 .19701 L
.29937 .19359 L
.30638 .1889 L
.3121 .18369 L
.31721 .17713 L
.31944 .17314 L
.32104 .16937 L
.3221 .16597 L
.32255 .164 L
.32288 .16216 L
.32312 .1602 L
.32325 .15841 L
.32329 .15742 L
.3233 .15634 L
.32323 .15438 L
.3231 .15272 L
.32286 .1509 L
.32253 .14902 L
.32212 .14726 L
.32121 .14419 L
.31995 .14098 L
.31827 .13761 L
.31609 .13414 L
.31366 .13101 L
.31079 .12797 L
.30786 .12544 L
.30505 .12342 L
.29925 .12029 L
.29638 .11919 L
.29319 .11828 L
.29158 .11793 L
.29069 .11778 L
.28985 .11765 L
.28822 .11746 L
.28674 .11736 L
.28536 .11732 L
.28391 .11733 L
Mistroke
.28255 .1174 L
.28133 .1175 L
.2799 .11768 L
.27837 .11793 L
.27566 .11853 L
.27282 .11939 L
.2698 .12053 L
.26404 .12336 L
.25464 .12931 L
.25035 .13214 L
.24572 .13493 L
.24155 .13701 L
.23693 .13873 L
.23426 .13943 L
.23281 .13974 L
.23122 .14001 L
.22973 .14021 L
.22834 .14035 L
.22699 .14044 L
.22554 .14051 L
.2242 .14053 L
.22274 .14052 L
.22135 .14048 L
.22006 .14042 L
.21853 .14032 L
.21684 .14018 L
.21372 .13983 L
.1882 .13525 L
.18124 .13409 L
.17751 .13358 L
.17571 .13338 L
.17406 .13322 L
.17241 .13308 L
.17095 .13299 L
.17012 .13296 L
.16934 .13293 L
.16853 .13291 L
.16766 .1329 L
.16624 .13291 L
.16554 .13292 L
.16477 .13295 L
.16343 .13302 L
.1622 .13312 L
.16086 .13327 L
.15965 .13344 L
.15698 .13394 L
.15473 .13452 L
.15244 .13529 L
.14872 .13697 L
.14494 .13936 L
Mistroke
.14126 .14249 L
.13765 .14652 L
.1319 .15534 L
.12693 .16612 L
.12457 .17282 L
.12271 .17935 L
.12129 .18581 L
.12076 .1889 L
.12036 .19177 L
.12001 .1951 L
.11987 .19685 L
.11982 .19775 L
.11977 .1987 L
.11971 .20031 L
.11969 .20181 L
.11968 .20273 L
.11969 .20359 L
.11973 .20521 L
.1198 .20667 L
.1199 .20824 L
.12003 .20971 L
.12017 .21106 L
.1206 .21415 L
.12119 .21736 L
.12272 .22323 L
.12497 .22934 L
.12737 .2343 L
.13034 .23925 L
.13658 .2472 L
.1447 .2548 L
.15402 .26136 L
.17108 .27037 L
.20816 .28759 L
.24542 .30845 L
.26195 .31985 L
.27713 .32915 L
.28134 .33114 L
.28605 .33301 L
.29068 .33448 L
.29497 .33551 L
.29758 .336 L
.30006 .33637 L
.3027 .33666 L
.30419 .33679 L
.30557 .33688 L
.30676 .33694 L
.30788 .33698 L
.30911 .33701 L
.30979 .33702 L
.31042 .33702 L
Mistroke
.31168 .33702 L
.3124 .33701 L
.31306 .337 L
.31427 .33697 L
.31555 .33692 L
.31791 .33681 L
.32039 .33665 L
.32478 .33632 L
.32984 .33592 L
.33228 .33574 L
.33447 .33561 L
.33545 .33556 L
.33651 .33551 L
.33751 .33548 L
.33842 .33546 L
.33939 .33544 L
.34045 .33544 L
.34155 .33545 L
.34257 .33548 L
.3437 .33552 L
.34476 .33558 L
.34571 .33564 L
.34675 .33573 L
.34905 .336 L
.35116 .33633 L
.35566 .33732 L
.35817 .33806 L
.36046 .33885 L
.36907 .34273 L
.38506 .35252 L
.39296 .35749 L
.40158 .36215 L
.40665 .36432 L
.41138 .36591 L
.41371 .36653 L
.41628 .3671 L
.41875 .36754 L
.42106 .36785 L
.42232 .36798 L
.42352 .36808 L
.42461 .36816 L
.42581 .36822 L
.42712 .36827 L
.42786 .36828 L
.42853 .36829 L
.42985 .36829 L
.4311 .36826 L
.43229 .36823 L
.43356 .36817 L
.43586 .36802 L
Mistroke
.43837 .3678 L
.44116 .3675 L
.44714 .36666 L
.45372 .3655 L
.46585 .36293 L
.48931 .35718 L
.50037 .35439 L
.51078 .35197 L
.51484 .35117 L
.51902 .35047 L
.52135 .35016 L
.52346 .34992 L
.52547 .34975 L
.52658 .34967 L
.52758 .34962 L
.52845 .34958 L
.52939 .34955 L
.53038 .34952 L
.53133 .34951 L
.53222 .34951 L
.53318 .34951 L
.53399 .34952 L
.5349 .34954 L
.53598 .34957 L
.53698 .3496 L
.53926 .34971 L
.54354 .34997 L
.55235 .35058 L
.55505 .35074 L
.55765 .35088 L
.56003 .35098 L
.56138 .35102 L
.56264 .35105 L
.56405 .35108 L
.56538 .35109 L
.56684 .35109 L
.56766 .35108 L
.56842 .35107 L
.56921 .35106 L
.56994 .35104 L
.5716 .35098 L
.57306 .35091 L
.57462 .35081 L
.57739 .35058 L
.57997 .35029 L
.58279 .34989 L
.5858 .34936 L
.59119 .34808 L
.59602 .34653 L
.60107 .34442 L
Mistroke
.60613 .34173 L
.61545 .33502 L
.62297 .32802 L
.63021 .32081 L
.63822 .31385 L
.64581 .30899 L
.65052 .30679 L
.65521 .30512 L
.66028 .30378 L
.66286 .30326 L
.6653 .30284 L
.66995 .30224 L
.67244 .302 L
.67512 .3018 L
.67777 .30164 L
.68065 .30151 L
.68585 .30132 L
.69195 .30114 L
.69488 .30103 L
.69752 .3009 L
.70009 .30075 L
.70248 .30057 L
.70763 .30002 L
.71028 .29963 L
.71269 .2992 L
.71796 .29799 L
.72253 .2966 L
.72742 .29475 L
.73704 .29012 L
.74666 .28481 L
.75621 .2798 L
.76531 .27585 L
.77079 .27396 L
.77593 .27254 L
.78149 .27135 L
.78689 .27052 L
.78992 .27017 L
.79274 .26991 L
.79588 .26969 L
.79747 .26961 L
.79923 .26953 L
.80076 .26948 L
.80222 .26944 L
.80353 .26941 L
.80496 .26939 L
.80652 .26937 L
.80732 .26937 L
.80819 .26936 L
.80974 .26936 L
.8112 .26936 L
Mistroke
.81259 .26937 L
.81406 .26937 L
.8167 .26939 L
.81956 .26941 L
.82117 .26942 L
.82267 .26942 L
.82427 .26942 L
.82508 .26942 L
.82597 .26942 L
.82759 .26941 L
.8291 .26939 L
.83057 .26936 L
.83191 .26934 L
.83494 .26924 L
.83661 .26918 L
.83815 .2691 L
.84163 .26888 L
.84468 .26863 L
.8479 .2683 L
.85405 .26745 L
.85968 .26638 L
.86457 .26519 L
.86963 .26367 L
.87938 .25966 L
.88426 .25694 L
.88833 .25414 L
.89204 .25099 L
.89504 .24779 L
.89786 .24387 L
.89896 .24193 L
.89985 .24007 L
.90115 .23652 L
.90167 .23457 L
.90211 .23241 L
.90227 .23131 L
.90242 .23009 L
.90247 .22953 L
.90252 .22892 L
.90259 .22782 L
.90261 .22726 L
.90263 .22665 L
.90264 .22553 L
.90262 .22443 L
.90259 .2234 L
.90253 .22241 L
.90246 .22134 L
.90227 .21937 L
.90196 .21698 L
.9016 .21475 L
.89937 .20483 L
Mistroke
.89647 .19524 L
.89321 .18602 L
.88435 .16499 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`19o`H0
?Ol0097o00<0ool0AOl400Ko1@0ho`00TOl00`3oo`13o`803ol303Go001mo`801?l200Go0`04o`03
0?oo04;o00<0ool04_l00`3oo`0bo`00O?l0103oo`0;o`030?oo00?o00<0ool0@?l201Ko0P0bo`00
E?l901oo00@0ool02?l500Co0P10o`030?oo01So0P0`o`00D_l200Wo00<0ool05_l400;o00@0ool0
2?l0103oo`05o`030?oo03ko00<0ool06ol00`3oo`0]o`00D?l200_o00<0ool07?l0103oo`09o`03
0?l000Go00<0ool0?Ol00`3oo`0Mo`030?oo02co001?o`030?oo00co0P0Mo`802ol200Go00<0ool0
?Ol00`3oo`0No`030?oo02_o000do``03Ol2017o00<0ool0;_l00`3oo`0lo`808?l00`3oo`0[o`00
<ol00`3oo`0:o`<02?l201?o00<0ool0;_l00`3oo`0lo`030?oo023o00<0ool0:_l0037o0P0@o`P0
5_l00`3oo`0]o`030?oo03co00<0ool08?l00`3oo`0Zo`00<?l00`3oo`0^o`030?oo02go00<0ool0
??l00`3oo`0Qo`030?oo02Wo000`o`030?oo02ko00<0ool0;Ol00`3oo`0lo`030?oo02;o00<0ool0
:?l002oo00<0ool0;ol00`3oo`0]o`030?oo03co00<0ool08_l00`3oo`0Xo`00;ol00`3oo`0_o`80
;_l00`3oo`0lo`030?oo02?o00<0ool09ol002ko00<0ool0<?l00`3oo`0]o`030?oo03co00<0ool0
8ol00`3oo`0Wo`00;_l00`3oo`0`o`030?oo02go00<0ool0??l00`3oo`0To`030?oo02Ko000]o`03
0?oo037o00<0ool0;Ol00`3oo`0ko`030?oo02Go00<0ool09_l002go00<0ool0<Ol00`3oo`0]o`03
0?oo03_o00<0ool09_l00`3oo`0Uo`00;?l00`3oo`0ao`030?oo01[o0P04o`800ol400Go00<0ool0
>ol00`3oo`0Vo`030?oo02Go000/o`030?oo037o00<0ool06Ol0103oo`08o`030?oo00Ko00<0ool0
>ol00`3oo`0Vo`030?oo02Go000/o`030?oo033o00<0ool06_l0103oo`09o`030?oo00Go0P0lo`03
0?oo02Oo00<0ool09?l002co00<0ool0;ol00`3oo`0Eo`@00_l0103oo`0:o`030?oo00Co00<0ool0
>ol00`3oo`0Wo`030?oo02Co000/o`030?oo02go0P0No`040?oo00So00@0ool01Ol00`3oo`0jo`03
0?oo02So00<0ool09?l002_o0P0]o`808Ol200[o0P06o`030?oo03[o00<0ool0:Ol00`3oo`0So`00
:ol00`3oo`0[o`030?oo03Go00<0ool0>Ol00`3oo`0Zo`030?oo02?o000[o`030?oo02Go1P0ho`03
0?oo03Wo00<0ool0:ol00`3oo`0Ro`00:ol00`3oo`0Qo`@0?_l00`3oo`0ho`030?oo02co00<0ool0
8_l000Oo1@02o`801?l201Go00<0ool02ol5017o0P0?o`801?l200Co0P0So`030?oo02;o0P04o`80
1?l200So00<0ool06?l500oo00<0ool04?l500;o0P04o`800ol000Wo00<0ool01ol0103oo`0Do`80
3_l00`3oo`0Co`030?oo00_o00@0ool02?l0103oo`0Ro`030?oo027o00@0ool02?l0103oo`07o`03
0?oo01[o00<0ool03ol00`3oo`0Bo`030?oo00Oo00@0ool00_l000Wo00<0ool02_l00`3oo`0Co`03
0?oo00co00<0ool05?l00`3oo`0:o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool0
1?l00`3oo`0Ko`030?oo00oo00<0ool04_l00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So
0P0Fo`030?oo00Co1004o`030?oo01Go0P04o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo
0P07o`030?oo01_o00<0ool04?l00`3oo`0Ao`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Go
00<0ool03?l00`3oo`0Go`030?oo00Oo00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3o
o`05o`030?oo01co00<0ool04?l00`3oo`0Ao`030?oo00So00<0ool00_l000So0P0:o`<05_l00`3o
o`0:o`806_l00`3oo`07o`802_l302;o00<0ool08_l200[o0`05o`030?oo01_o0P0Bo`030?oo013o
0P0:o`<00_l002go00<0ool09ol00`3oo`0go`030?oo03Go00<0ool0;ol202?o000]o`030?oo02So
0P0go`030?oo03Go00<0ool0;ol00`3oo`0Ro`00;_l00`3oo`0Yo`030?oo03Co00<0ool0=?l00`3o
o`0`o`030?oo02;o000_o`030?oo02Wo0P0do`030?oo03Co00<0ool0;ol202Co000`o`030?oo02[o
00<0ool0<Ol00`3oo`0co`030?oo033o00<0ool08ol000coo`0<00Wo000<o`030?oo00Go00<0ool0
1_l00`3oo`06o`030?oo00Ko00@0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`03
0?oo00;o3P08o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`03o`040?oo00So00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02Co00<0ool0
0_l00`3oo`0Yo`030?oo00Ko00<0ool08Ol00`3oo`0Yo`030?oo00Co0P0So`030?oo00[o0P0No`03
0?oo00Oo000do`80>?l2027o00<0ool09_l:033o0P0Zo`00=_l203So0P0Oo`030?oo023o1P0fo`@0
;?l003So0P0ho`030?oo01co00<0ool07_l202So500`o`00>_l203Oo00<0ool06ol00`3oo`0Lo`80
9_l404Co000lo`80=_l201_o00<0ool06_l202Ko0P18o`00?_l203Ko0P0Io`030?oo01Oo0`0Vo`80
B_l0043o0P0fo`805ol00`3oo`0Co`@09ol204co0012o`80=_l301Co00<0ool01?l@02Wo00<0ool0
C?l004Co0P0go`T02ol00`3oo`0403Oo0P1?o`00A_l203ko3P0go`@0DOl004So00<0ool0A_l00`3o
o`0bo`D0EOl004Wo0P16o`030?oo02Oo2`1Jo`00Bol204Co00<0ool09?l306Go001=o`030?oo02go
0P04o`800ol400Go00<0ool08_l206So001>o`030?oo02_o00@0ool02?l00`3oo`06o`030?oo027o
00<0ool0J?l004oo0P0[o`040?oo00Wo00<0ool01Ol2027o00<0ool0JOl0057o00<0ool0:?l0103o
o`0:o`030?oo00Co00<0ool07ol00`3oo`1Zo`00D_l202So00@0ool02?l0103oo`05o`030?oo01ko
00<0ool0Jol005Co0P0Wo`802_l200Ko00<0ool07Ol00`3oo`1/o`00E_l403Oo00<0ool07?l00`3o
o`1]o`00F_lB02Go00<0ool06_l2073o001[o`809?l00`3oo`0Ho`80L_l006go0P0Ro`030?oo01Oo
00<0ool0L_l006oo0P0Po`030?oo00Co0`0<o`@0MOl0077o0P0No`P00_l=07So001co`030?oo01[o
0P2>o`00M?l00`3oo`0Do`D000?o0?l0SOl007Go100=o`D01_l00`3oo`2<o`00NOl=00_o00<0ool0
S?l0097o00<0ool0S?l007go0P04o`801Ol300Co00<0ool0S?l007co00@0ool02ol00`3oo`03o`03
0?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`030?oo08co001lo`04
0?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00O?l0103oo`09o`03
0?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00oolQo`00oolQo`00
oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.8866 .16761 m
.87577 .14677 L
.86371 .1286 L
.85645 .12051 L
.8485 .1137 L
.8379 .10708 L
.83098 .10385 L
.82406 .10131 L
.81742 .09943 L
.81362 .09858 L
.80996 .09791 L
.80652 .0974 L
.80333 .09704 L
.8015 .09688 L
.79974 .09676 L
.79789 .09666 L
.79585 .0966 L
.79479 .09658 L
.79364 .09657 L
.79256 .09658 L
.79156 .0966 L
.78967 .09666 L
.78768 .09677 L
.78554 .09694 L
.78355 .09714 L
.77908 .09775 L
.77527 .09847 L
.77187 .09928 L
.76459 .1016 L
.75837 .10437 L
.75247 .10788 L
.74808 .11129 L
.74395 .11544 L
.7404 .12022 L
.73888 .12285 L
.73753 .12568 L
.73654 .12821 L
.73566 .13102 L
.73501 .1336 L
.73454 .136 L
.73414 .13865 L
.73397 .14002 L
.73382 .14153 L
.73362 .14408 L
.73354 .14547 L
.73348 .14678 L
.73332 .15152 L
.73316 .15665 L
.73304 .15953 L
.73295 .16117 L
Mistroke
.73284 .16269 L
.73261 .16551 L
.73227 .16855 L
.73134 .17451 L
.73014 .18009 L
.7265 .1926 L
.72205 .20487 L
.71045 .23293 L
.70548 .24367 L
.70026 .25293 L
.69717 .25695 L
.69419 .25976 L
.69268 .26082 L
.69095 .26177 L
.68924 .26249 L
.68759 .263 L
.68672 .2632 L
.68578 .26339 L
.68495 .26352 L
.68405 .26363 L
.68304 .26372 L
.68209 .26378 L
.6815 .26381 L
.68093 .26383 L
.67985 .26384 L
.67887 .26384 L
.67782 .26382 L
.6769 .26379 L
.67588 .26374 L
.67126 .26347 L
.66586 .26313 L
.66425 .26305 L
.6627 .26298 L
.6613 .26293 L
.6605 .26291 L
.65976 .2629 L
.65843 .26288 L
.65716 .26288 L
.65577 .2629 L
.65427 .26293 L
.65277 .263 L
.65119 .26309 L
.64984 .26319 L
.64837 .26332 L
.64548 .26367 L
.64237 .26417 L
.63687 .2654 L
.63176 .26699 L
.6273 .26874 L
.61817 .27342 L
Mistroke
.60982 .27859 L
.60305 .28245 L
.59921 .28415 L
.59733 .28481 L
.59554 .28535 L
.59193 .28618 L
.58993 .28651 L
.58772 .28678 L
.58554 .28698 L
.58443 .28706 L
.5834 .28712 L
.58143 .28721 L
.58037 .28724 L
.57924 .28727 L
.57799 .2873 L
.57733 .28731 L
.57662 .28732 L
.5753 .28733 L
.57407 .28733 L
.57288 .28734 L
.57175 .28734 L
.57051 .28735 L
.56914 .28735 L
.56775 .28736 L
.56628 .28737 L
.5649 .28738 L
.56362 .2874 L
.56209 .28743 L
.56069 .28746 L
.55917 .2875 L
.55755 .28756 L
.55483 .28769 L
.55232 .28784 L
.54671 .28831 L
.54198 .28886 L
.53692 .28962 L
.52592 .29181 L
.51627 .294 L
.51217 .29483 L
.50777 .29559 L
.50519 .29594 L
.50281 .29621 L
.50035 .29642 L
.49914 .2965 L
.49799 .29657 L
.49677 .29663 L
.49565 .29667 L
.49438 .2967 L
.49301 .29673 L
.49166 .29674 L
Mistroke
.4909 .29674 L
.49018 .29673 L
.48885 .29672 L
.48743 .29669 L
.4862 .29666 L
.48486 .29662 L
.4824 .29652 L
.47659 .29621 L
.46272 .29521 L
.44966 .29412 L
.4377 .29296 L
.43157 .29223 L
.42619 .29141 L
.42032 .29022 L
.41549 .28885 L
.41134 .28726 L
.40718 .28518 L
.40005 .28031 L
.3921 .27362 L
.38254 .26587 L
.37785 .26259 L
.3724 .25929 L
.36669 .25643 L
.36124 .25426 L
.35598 .25266 L
.35333 .25203 L
.35044 .25149 L
.34897 .25127 L
.34744 .25107 L
.34613 .25094 L
.34475 .25083 L
.34398 .25078 L
.34328 .25075 L
.34194 .2507 L
.34119 .25069 L
.34039 .25068 L
.33965 .25069 L
.33897 .2507 L
.33778 .25074 L
.33669 .25079 L
.33551 .25087 L
.33428 .25097 L
.32976 .25146 L
.32591 .25196 L
.32495 .25207 L
.32394 .25218 L
.32308 .25225 L
.32218 .25232 L
.32167 .25234 L
.32112 .25236 L
Mistroke
.32063 .25237 L
.32017 .25236 L
.31931 .25234 L
.31884 .2523 L
.31841 .25227 L
.31757 .25217 L
.31667 .25201 L
.31582 .25183 L
.31504 .25161 L
.31351 .25108 L
.31205 .25044 L
.30855 .24845 L
.3003 .24215 L
.25917 .21076 L
.25563 .20855 L
.25422 .20765 L
.25329 .207 L
.253 .20676 L
.25278 .20655 L
.25271 .20645 L
.25266 .20636 L
.25265 .20622 L
.25274 .20612 L
.25292 .20604 L
.25319 .20599 L
.25351 .20596 L
.254 .20595 L
.25455 .20595 L
.25527 .20597 L
.25615 .206 L
.25792 .20607 L
.25899 .2061 L
.26022 .20613 L
.26089 .20614 L
.26165 .20614 L
.26306 .20614 L
.26459 .20612 L
.26608 .20607 L
.26704 .20603 L
.26797 .20599 L
.26977 .20587 L
.27178 .20569 L
.27405 .20544 L
.27763 .20492 L
.28162 .20416 L
.28908 .20218 L
.29366 .20054 L
.29797 .19867 L
.30569 .19429 L
.30933 .19163 L
Mistroke
.31294 .1885 L
.31855 .18212 L
.32134 .1778 L
.32348 .17349 L
.32496 .16946 L
.32561 .16713 L
.32609 .16497 L
.32645 .16285 L
.32668 .16082 L
.32681 .159 L
.32686 .15701 L
.32681 .15485 L
.32675 .15376 L
.32664 .15256 L
.32638 .15045 L
.32604 .14849 L
.32513 .14472 L
.32402 .14135 L
.32251 .13778 L
.32063 .13419 L
.31841 .13073 L
.31576 .12731 L
.31037 .12195 L
.30677 .11924 L
.30323 .1171 L
.29997 .11554 L
.2964 .11424 L
.29323 .11342 L
.29146 .11309 L
.28985 .11288 L
.28836 .11276 L
.28675 .11269 L
.28525 .11271 L
.28387 .11278 L
.28303 .11285 L
.28213 .11295 L
.28053 .11319 L
.27898 .11349 L
.27759 .11383 L
.2745 .1148 L
.27137 .11609 L
.26585 .11912 L
.26111 .12244 L
.25135 .13038 L
.24705 .13366 L
.24227 .13664 L
.23999 .13776 L
.23749 .13876 L
.23528 .13945 L
.234 .13978 L
Mistroke
.23275 .14004 L
.23155 .14025 L
.23021 .14043 L
.22897 .14056 L
.2278 .14065 L
.22638 .14071 L
.22507 .14072 L
.22354 .1407 L
.22206 .14063 L
.22074 .14055 L
.21933 .14042 L
.21674 .14012 L
.21068 .13914 L
.19671 .13599 L
.1833 .13278 L
.17766 .13162 L
.17188 .13067 L
.16899 .13032 L
.16738 .13017 L
.1659 .13007 L
.16457 .13 L
.16337 .12997 L
.16208 .12996 L
.16074 .12999 L
.15941 .13006 L
.15824 .13015 L
.15691 .1303 L
.15568 .13048 L
.15356 .13089 L
.15137 .13149 L
.14906 .13231 L
.14675 .13339 L
.14292 .13582 L
.13928 .13907 L
.13618 .14272 L
.13309 .1473 L
.13 .15298 L
.1251 .16479 L
.12317 .17075 L
.12139 .17736 L
.11997 .18411 L
.11903 .19033 L
.1187 .19335 L
.11856 .19506 L
.11845 .19662 L
.11838 .198 L
.11833 .1995 L
.11831 .20107 L
.11831 .20253 L
.11834 .20425 L
Mistroke
.11837 .20511 L
.11841 .20606 L
.11852 .20773 L
.11865 .20929 L
.11906 .21276 L
.11936 .21469 L
.11968 .21644 L
.12117 .22252 L
.12213 .22547 L
.12333 .2286 L
.12579 .23383 L
.12888 .23902 L
.13543 .24724 L
.14399 .25487 L
.15377 .26121 L
.17138 .26946 L
.20873 .28599 L
.24618 .30826 L
.2624 .32079 L
.26935 .32598 L
.27718 .33101 L
.28167 .33336 L
.28597 .33521 L
.29019 .33665 L
.29482 .33781 L
.29744 .33829 L
.29883 .33849 L
.30031 .33867 L
.30161 .33879 L
.30304 .3389 L
.30434 .33897 L
.30556 .33901 L
.30685 .33903 L
.30806 .33903 L
.30916 .33901 L
.31035 .33897 L
.31164 .33891 L
.31302 .33882 L
.31548 .33862 L
.3255 .33733 L
.33033 .33659 L
.33474 .33598 L
.33676 .33575 L
.33864 .33556 L
.33968 .33547 L
.34079 .3354 L
.3418 .33534 L
.34273 .33531 L
.34375 .33529 L
.3447 .33528 L
Mistroke
.34574 .33529 L
.34685 .33533 L
.348 .33539 L
.34909 .33548 L
.35007 .33558 L
.35112 .33572 L
.35314 .33605 L
.35536 .33653 L
.35979 .33789 L
.36386 .33959 L
.36765 .34156 L
.385 .35377 L
.39247 .35938 L
.40062 .36467 L
.40545 .36716 L
.40996 .36899 L
.41219 .36972 L
.41465 .37039 L
.417 .3709 L
.41924 .37128 L
.42165 .37157 L
.42295 .37168 L
.42436 .37177 L
.42508 .37181 L
.42586 .37183 L
.4266 .37185 L
.4273 .37185 L
.42862 .37185 L
.43004 .37181 L
.4314 .37175 L
.43289 .37166 L
.43562 .37142 L
.43834 .3711 L
.44088 .37075 L
.45328 .36837 L
.46686 .36497 L
.49082 .35806 L
.50886 .35286 L
.51293 .35185 L
.51709 .35097 L
.51892 .35064 L
.52083 .35036 L
.5226 .35015 L
.5242 .35 L
.52508 .34993 L
.52602 .34988 L
.52695 .34984 L
.52748 .34983 L
.52798 .34981 L
.52889 .34981 L
Mistroke
.52971 .34981 L
.53068 .34983 L
.53159 .34985 L
.53262 .3499 L
.53372 .34996 L
.53572 .35011 L
.54049 .35057 L
.54923 .35161 L
.55452 .35219 L
.55724 .35245 L
.55981 .35265 L
.56227 .35281 L
.56354 .35287 L
.56492 .35293 L
.5663 .35297 L
.56761 .353 L
.56837 .353 L
.56918 .35301 L
.57062 .353 L
.57182 .35297 L
.57311 .35293 L
.57433 .35288 L
.57544 .35281 L
.57815 .3526 L
.5807 .35232 L
.58341 .35192 L
.58593 .35146 L
.5915 .35006 L
.59668 .34822 L
.6019 .34573 L
.60639 .34295 L
.61036 .33996 L
.61708 .33357 L
.62416 .32543 L
.63071 .31791 L
.63706 .31191 L
.64099 .30902 L
.64486 .30673 L
.64906 .3048 L
.65397 .30313 L
.65657 .30246 L
.65943 .30186 L
.66218 .3014 L
.66477 .30107 L
.66781 .30078 L
.66943 .30066 L
.67116 .30056 L
.67257 .30049 L
.67409 .30043 L
.67497 .30041 L
Mistroke
.67577 .30039 L
.67653 .30037 L
.67735 .30036 L
.67875 .30034 L
.68006 .30033 L
.68086 .30032 L
.68161 .30032 L
.68302 .30032 L
.6845 .30032 L
.68587 .30032 L
.68736 .30032 L
.68811 .30032 L
.68894 .30032 L
.69042 .30031 L
.69178 .3003 L
.69326 .30028 L
.69409 .30027 L
.69483 .30025 L
.6962 .30021 L
.69765 .30016 L
.6989 .30011 L
.70026 .30003 L
.70277 .29985 L
.70537 .29959 L
.70773 .29928 L
.70989 .29893 L
.71504 .2978 L
.71953 .29642 L
.72452 .29442 L
.72911 .29215 L
.74495 .28245 L
.75454 .27682 L
.75953 .27436 L
.76428 .27236 L
.76913 .27069 L
.77463 .26921 L
.77774 .26857 L
.78068 .26807 L
.78355 .26769 L
.78666 .26738 L
.78839 .26725 L
.78999 .26715 L
.79177 .26707 L
.79267 .26704 L
.79366 .26702 L
.79454 .267 L
.79534 .26699 L
.79627 .26699 L
.79716 .26699 L
.79874 .267 L
Mistroke
.79964 .26701 L
.80047 .26703 L
.80201 .26707 L
.80346 .26711 L
.8068 .26726 L
.81365 .26766 L
.827 .26854 L
.83028 .26872 L
.83378 .26888 L
.8358 .26895 L
.83762 .269 L
.83851 .26902 L
.83946 .26904 L
.84119 .26906 L
.84276 .26906 L
.84423 .26905 L
.84583 .26903 L
.84755 .26899 L
.8484 .26897 L
.84934 .26893 L
.85103 .26886 L
.85417 .26866 L
.85711 .26842 L
.86019 .26808 L
.86561 .2673 L
.87213 .26596 L
.87778 .2644 L
.88353 .26232 L
.88903 .25975 L
.89301 .2574 L
.89675 .25464 L
.90022 .25133 L
.90284 .248 L
.90394 .24623 L
.90493 .24429 L
.90566 .24255 L
.9063 .24063 L
.9066 .23952 L
.90684 .2385 L
.90704 .23744 L
.90719 .23643 L
.90733 .23531 L
.90739 .23471 L
.90744 .23407 L
.90748 .23338 L
.9075 .23273 L
.90752 .23213 L
.90752 .23149 L
.90751 .23037 L
.90746 .22932 L
Mistroke
.90739 .22815 L
.90728 .22686 L
.90695 .22419 L
.90654 .22167 L
.90545 .21663 L
.90411 .21161 L
.89614 .18921 L
.8866 .16761 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`17o`P0?Ol0097o00<0ool0
AOl200So1P0go`00TOl00`3oo`11o`@04?l00`3oo`0do`00TOl00`3oo`10o`030?oo01?o00<0ool0
<ol005Oo0P0To`801?l200Go0`04o`030?oo03ko0P0Go`80<ol005Co10001?l0000Po`040?oo00_o
00<0ool00ol00`3oo`0mo`030?oo01Wo00<0ool0<?l005;o0P08o`807_l0103oo`08o`D01?l203go
00<0ool06ol00`3oo`0_o`00DOl00`3oo`0:o`030?oo01Go1002o`040?oo00So00@0ool01Ol00`3o
o`0lo`030?oo01co00<0ool0;_l004oo0P0>o`030?oo01[o00@0ool02Ol00`3o0005o`030?oo03co
00<0ool07Ol00`3oo`0]o`00=?l701?o00<0ool03ol00`3oo`0Jo`802ol200Go00<0ool0>ol00`3o
o`0Oo`030?oo02co000co`030?oo00Go100>o`030?oo017o00<0ool0;Ol00`3oo`0ko`030?oo023o
00<0ool0:ol0037o0P0<o`@02?l201Co00<0ool0;Ol00`3oo`0ko`030?oo023o00<0ool0:ol0033o
00<0ool04?l801Oo00<0ool0;?l00`3oo`0ko`030?oo027o00<0ool0:_l002oo00<0ool0<?l00`3o
o`0/o`030?oo03_o00<0ool08Ol00`3oo`0Zo`00;ol00`3oo`0`o`030?oo02co00<0ool0>ol00`3o
o`0Ro`030?oo02Wo000^o`030?oo037o00<0ool0;?l00`3oo`0ko`030?oo02?o00<0ool0:?l002ko
00<0ool0<Ol00`3oo`0/o`030?oo03_o00<0ool08ol00`3oo`0Xo`00;Ol00`3oo`0bo`030?oo02co
00<0ool0>_l00`3oo`0Uo`030?oo02Oo000]o`030?oo03;o00<0ool0;?l00`3oo`0jo`030?oo02Go
00<0ool09ol002co00<0ool0<ol00`3oo`0/o`030?oo03[o00<0ool09_l00`3oo`0Vo`00;?l00`3o
o`0co`030?oo02co00<0ool0>_l00`3oo`0Vo`030?oo02Ko000/o`030?oo03?o00<0ool06?l200Co
0P03o`@01Ol00`3oo`0jo`030?oo02Oo00<0ool09Ol002co00<0ool0<_l00`3oo`0Ho`040?oo00So
00<0ool01_l00`3oo`0jo`030?oo02Oo00<0ool09Ol002_o00<0ool0<_l00`3oo`0Io`040?oo00Wo
00<0ool01Ol203_o00<0ool0:?l00`3oo`0To`00:ol00`3oo`0bo`030?oo01?o1002o`040?oo00[o
00<0ool01?l00`3oo`0io`030?oo02Wo00<0ool09?l002_o00<0ool0<?l201co00@0ool02?l0103o
o`05o`030?oo03Wo00<0ool0:_l00`3oo`0So`00:ol00`3oo`0^o`807ol200[o0P06o`030?oo03Wo
00<0ool0:_l00`3oo`0So`00:ol00`3oo`0]o`030?oo03?o00<0ool0>Ol00`3oo`0[o`030?oo02;o
000[o`030?oo02Wo100fo`030?oo03So00<0ool0;?l00`3oo`0Ro`00:ol00`3oo`0Qo`P0>_l00`3o
o`0ho`030?oo02co00<0ool08_l000Oo1@02o`801?l201Go00<0ool02ol5017o0P0?o`801?l200Co
0P0So`030?oo02;o0P04o`801?l200Oo00<0ool06Ol5013o00<0ool03ol500;o0P04o`800ol000Wo
00<0ool01ol0103oo`0Do`030?oo00go00<0ool04_l00`3oo`0<o`040?oo00So00@0ool08_l00`3o
o`0Qo`040?oo00So00@0ool01_l00`3oo`0Ko`030?oo013o00<0ool04Ol00`3oo`07o`040?oo00;o
0009o`030?oo00[o00<0ool04_l00`3oo`0=o`030?oo01?o0P0<o`040?oo00_o00<0ool08?l00`3o
o`0Qo`040?oo00_o00<0ool00ol00`3oo`0Lo`030?oo013o00<0ool04Ol00`3oo`0:o`400Ol1o`00
00?o00000P04o`030?oo00So0P0Fo`030?oo00Co1004o`030?oo01Go00<0ool00ol400;o00@0ool0
2Ol202?o00<0ool08Ol0103oo`09o`801_l00`3oo`0Lo`030?oo017o00<0ool04?l00`3oo`08o`80
0ol000Wo00<0ool02?l00`3oo`0Eo`030?oo00co00<0ool05_l00`3oo`08o`040?oo00Wo00<0ool0
8_l00`3oo`0Qo`040?oo00Wo00<0ool01?l00`3oo`0Mo`030?oo017o00<0ool04?l00`3oo`08o`03
0?oo00;o0008o`802_l301Go00<0ool02ol201Wo0P09o`802_l302;o00<0ool08_l200[o0`04o`03
0?oo01co0P0Co`030?oo00oo0P0:o`<00_l002go00<0ool09ol00`3oo`0go`030?oo03Co00<0ool0
<_l00`3oo`0Po`00;Ol00`3oo`0Xo`030?oo03Ko00<0ool0=?l00`3oo`0bo`030?oo023o000^o`03
0?oo02So0P0fo`030?oo03?o00<0ool0<ol00`3oo`0Po`00;ol00`3oo`0Yo`030?oo03?o00<0ool0
<ol00`3oo`0bo`808_l0033o00<0ool0:Ol203?o00<0ool0<ol00`3oo`0bo`030?oo027o000<ool0
3009o`003?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`040?oo00Go00<0ool01_l00`3o
o`06o`030?oo00Go00<0ool01_l00`3oo`02o`<000?o0?l00ol500So00<0ool01_l00`3oo`06o`03
0?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool0
1_l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Ko00<0ool0
1_l00`3oo`07o`003?l00`3oo`0To`030?oo00;o00<0ool0:Ol00`3oo`06o`<08Ol00`3oo`0Yo`03
0?oo00Go00<0ool08Ol00`3oo`0=o`030?oo01[o00<0ool01ol003Co0P0jo`030?oo01ko00<0ool0
9Ol500Co0`0ao`<09ol003Ko0P0io`030?oo01go00<0ool08Ol400Go1@0`o`<0:_l003So0P0ho`80
7Ol00`3oo`0No`<09?lJ02go000jo`80>?l00`3oo`0Jo`030?oo01co0P0So`@0Aol003co0P0go`03
0?oo01Wo00<0ool06ol00`3oo`0Ro`030?oo04Wo000no`80=_l201Wo00<0ool06_l00`3oo`0Ro`03
0?oo04[o0010o`80=_l00`3oo`0Fo`030?oo01Ko100So`80COl004;o0P0eo`805_l00`3oo`08o`h0
9Ol204oo0014o`80=Ol700oo00<0ool00ol5037o0P1Ao`00A_l203[o5@0do`80Dol004So0P17o`03
0?oo02[o3@1Eo`00B_l204Go00<0ool09Ol506;o001<o`80@ol00`3oo`0So`80Iol004ko00<0ool0
;?l200Co0P03o`@01Ol00`3oo`0Qo`80JOl004oo00<0ool0:_l0103oo`08o`030?oo00Ko00<0ool0
8?l00`3oo`1Yo`00D?l00`3oo`0Yo`040?oo00Wo00<0ool01Ol2023o00<0ool0J_l0057o00<0ool0
:?l0103oo`0:o`030?oo00Co00<0ool07ol00`3oo`1Zo`00D_l202So00@0ool02?l0103oo`05o`03
0?oo01ko00<0ool0Jol005Co00<0ool09_l200[o0P06o`030?oo01go00<0ool0K?l005Go0P0jo`03
0?oo01co00<0ool0KOl005Oo1@05o`/09Ol00`3oo`0Ko`030?oo06ko001Ko`H02ol202?o00<0ool0
6_l00`3oo`1_o`00K_l00`3oo`0Po`030?oo01Oo0`1bo`00Kol2023o00<0ool05_l00`3oo`1co`00
LOl00`3oo`0MoaT0M_l007;o00<0ool06ol208ko001co`806?l30003o`3o08go001eo`804ol300Co
00<0ool0S?l007Oo00<0ool02Ol700Oo00<0ool0S?l007So2`0>o`030?oo08co001mo`801?l200Go
0`04o`030?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go
001lo`040?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go
0P0;o`801Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3o
o`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3o
o`2<o`00O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co
00@0ool02?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o
00<0ool0S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.88885 .17023 m
.87774 .14859 L
.86605 .1295 L
.85912 .12078 L
.85141 .11325 L
.84087 .10571 L
.83384 .10193 L
.8267 .09888 L
.81978 .09655 L
.81579 .09546 L
.81193 .09456 L
.80829 .09386 L
.8049 .09333 L
.80107 .09286 L
.79894 .09266 L
.79691 .09251 L
.79471 .09239 L
.79271 .09233 L
.7916 .09231 L
.7904 .0923 L
.78822 .09233 L
.78618 .0924 L
.78403 .09251 L
.78219 .09265 L
.7802 .09284 L
.77573 .09342 L
.77106 .09429 L
.76677 .09532 L
.763 .09645 L
.75618 .09909 L
.74969 .10252 L
.74484 .1059 L
.74029 .11002 L
.73637 .1148 L
.7346 .11762 L
.73322 .12027 L
.7313 .12527 L
.7306 .12782 L
.73001 .13063 L
.7296 .13329 L
.72943 .13479 L
.72931 .13618 L
.72921 .1375 L
.72914 .13874 L
.72909 .14003 L
.72907 .14071 L
.72905 .14145 L
.72902 .14272 L
.72901 .14411 L
.729 .14542 L
Mistroke
.729 .14664 L
.729 .14785 L
.72901 .14895 L
.72901 .15023 L
.729 .15143 L
.729 .15275 L
.72899 .15351 L
.72898 .15421 L
.72896 .15564 L
.72893 .15644 L
.72891 .15718 L
.72886 .1585 L
.72879 .15995 L
.72861 .16261 L
.72832 .16578 L
.72789 .16923 L
.72679 .17566 L
.72537 .18199 L
.72375 .18805 L
.71924 .20253 L
.70976 .22957 L
.70516 .24179 L
.70044 .25246 L
.69768 .25717 L
.69502 .26045 L
.69366 .26169 L
.69211 .2628 L
.69056 .26363 L
.68905 .26421 L
.6882 .26446 L
.68737 .26465 L
.6866 .26479 L
.68574 .26491 L
.68479 .265 L
.68429 .26503 L
.68374 .26505 L
.68274 .26507 L
.68177 .26506 L
.68083 .26502 L
.67982 .26496 L
.67794 .26481 L
.67343 .26429 L
.6685 .26367 L
.66565 .26335 L
.66288 .26307 L
.66129 .26294 L
.65954 .26283 L
.65876 .26278 L
.65794 .26275 L
.65645 .26269 L
Mistroke
.65495 .26267 L
.65354 .26266 L
.65199 .26269 L
.65112 .26272 L
.65033 .26275 L
.64881 .26284 L
.64739 .26295 L
.64474 .26324 L
.64192 .26367 L
.63893 .26427 L
.63372 .26569 L
.6292 .26737 L
.6199 .2722 L
.6117 .27788 L
.60524 .28243 L
.60174 .28446 L
.5983 .28599 L
.5949 .28706 L
.59287 .28751 L
.59091 .28784 L
.5892 .28806 L
.58724 .28824 L
.58623 .28831 L
.58511 .28837 L
.58404 .28841 L
.58304 .28844 L
.58179 .28846 L
.58059 .28848 L
.57949 .28848 L
.57829 .28848 L
.57695 .28847 L
.57626 .28847 L
.57551 .28846 L
.57287 .28842 L
.57034 .28839 L
.56893 .28837 L
.56758 .28836 L
.56632 .28836 L
.56493 .28836 L
.56366 .28836 L
.56248 .28837 L
.56096 .2884 L
.5601 .28842 L
.55931 .28844 L
.55788 .28848 L
.55631 .28854 L
.5536 .28869 L
.55075 .28889 L
.54767 .28917 L
.54443 .28955 L
Mistroke
.53875 .2904 L
.51924 .29497 L
.51436 .29617 L
.50978 .29714 L
.50485 .29791 L
.50214 .29822 L
.50058 .29835 L
.49911 .29846 L
.49767 .29854 L
.49634 .2986 L
.49563 .29862 L
.49484 .29865 L
.494 .29866 L
.49321 .29868 L
.4925 .29868 L
.49171 .29869 L
.49089 .29868 L
.49011 .29868 L
.48869 .29866 L
.48715 .29863 L
.48562 .29858 L
.48392 .29852 L
.48076 .29838 L
.4746 .29801 L
.4479 .29593 L
.43535 .2947 L
.42882 .29385 L
.42319 .29287 L
.41784 .29157 L
.41343 .29006 L
.40906 .288 L
.40486 .28531 L
.39035 .27177 L
.38283 .26492 L
.37332 .25805 L
.36751 .25486 L
.36192 .25246 L
.3565 .25073 L
.35377 .25008 L
.35079 .24954 L
.34919 .24932 L
.34768 .24915 L
.34634 .24904 L
.34491 .24896 L
.3441 .24893 L
.34337 .24892 L
.34254 .24891 L
.34177 .24892 L
.34041 .24896 L
.33965 .249 L
Mistroke
.33897 .24904 L
.33775 .24914 L
.33663 .24926 L
.33418 .24958 L
.32963 .25043 L
.32572 .25132 L
.32384 .25174 L
.32195 .2521 L
.32118 .25222 L
.32038 .25232 L
.31992 .25236 L
.31951 .25239 L
.31871 .25242 L
.31804 .25242 L
.31732 .25238 L
.31656 .2523 L
.31586 .25219 L
.31517 .25203 L
.31442 .25182 L
.31303 .25132 L
.31033 .24992 L
.30686 .24753 L
.30327 .24463 L
.28181 .22638 L
.25937 .21099 L
.25222 .20728 L
.25052 .20653 L
.25022 .20641 L
.25017 .20638 L
.25021 .20639 L
.25035 .20643 L
.2506 .2065 L
.25147 .20673 L
.25284 .20705 L
.2548 .20744 L
.25688 .20779 L
.25956 .20813 L
.26087 .20827 L
.26235 .20839 L
.26382 .20848 L
.26521 .20855 L
.26689 .20859 L
.26853 .2086 L
.27038 .20857 L
.27244 .20849 L
.2735 .20843 L
.27467 .20835 L
.27684 .20815 L
.279 .20789 L
.28104 .20759 L
Mistroke
.28515 .20683 L
.28893 .20591 L
.2974 .20306 L
.30207 .20092 L
.30622 .19862 L
.31046 .19578 L
.31477 .19225 L
.32106 .18536 L
.32389 .18116 L
.3261 .177 L
.32793 .17246 L
.32919 .16811 L
.32967 .1658 L
.33007 .16322 L
.33021 .16192 L
.33033 .1605 L
.3304 .15917 L
.33043 .15794 L
.33041 .15568 L
.3303 .15359 L
.33007 .15129 L
.32971 .14885 L
.32921 .14631 L
.32861 .14394 L
.32721 .1396 L
.32559 .1357 L
.32356 .13175 L
.31921 .12526 L
.31371 .11926 L
.30697 .11406 L
.30348 .11207 L
.29962 .11036 L
.29785 .10974 L
.29596 .10918 L
.29419 .10876 L
.29258 .10846 L
.29088 .10823 L
.28906 .10808 L
.28747 .10804 L
.28582 .10807 L
.28435 .10818 L
.283 .10833 L
.28154 .10857 L
.28 .1089 L
.27663 .10989 L
.27315 .11133 L
.26731 .11476 L
.26218 .1188 L
.25784 .12284 L
.2503 .13033 L
Mistroke
.24619 .13393 L
.24213 .13676 L
.23973 .13805 L
.23699 .13919 L
.23574 .13961 L
.23437 .13998 L
.23306 .14028 L
.23182 .1405 L
.23043 .14069 L
.22968 .14077 L
.22888 .14083 L
.22751 .1409 L
.22672 .14092 L
.22597 .14092 L
.22455 .14089 L
.22323 .14083 L
.22168 .14071 L
.22018 .14055 L
.21714 .14013 L
.2142 .13961 L
.20721 .13804 L
.19236 .13388 L
.17967 .13032 L
.17325 .12879 L
.16987 .12811 L
.16677 .12759 L
.16518 .12738 L
.16349 .12719 L
.16198 .12705 L
.16063 .12697 L
.15937 .12692 L
.15802 .12691 L
.15665 .12694 L
.15539 .12701 L
.15472 .12707 L
.154 .12714 L
.15272 .12731 L
.1515 .12752 L
.1504 .12776 L
.14801 .12845 L
.14562 .1294 L
.14366 .13042 L
.14166 .13172 L
.13832 .13458 L
.13537 .13796 L
.13287 .14157 L
.12774 .15153 L
.12362 .16252 L
.11994 .17592 L
.11909 .17995 L
Mistroke
.1184 .18382 L
.11785 .18759 L
.11743 .19115 L
.11716 .19434 L
.11705 .19613 L
.11698 .19777 L
.11693 .19947 L
.11692 .20132 L
.11692 .20225 L
.11694 .20324 L
.117 .20502 L
.11705 .20599 L
.11711 .20689 L
.11727 .20891 L
.11745 .21068 L
.11769 .21253 L
.1182 .2157 L
.11893 .21911 L
.12063 .22503 L
.12303 .23102 L
.12579 .23628 L
.12869 .24071 L
.13233 .24527 L
.13655 .2496 L
.1447 .25607 L
.15477 .26183 L
.16376 .26574 L
.18226 .27252 L
.19984 .27982 L
.23759 .30098 L
.25385 .31366 L
.27004 .32759 L
.27819 .33336 L
.28241 .33574 L
.28644 .33759 L
.29125 .33927 L
.29385 .33995 L
.29666 .34051 L
.2978 .34068 L
.29904 .34085 L
.30011 .34096 L
.30129 .34106 L
.302 .34111 L
.30267 .34115 L
.30331 .34117 L
.30399 .34119 L
.30529 .34121 L
.30648 .3412 L
.30768 .34117 L
.30881 .34111 L
Mistroke
.31004 .34103 L
.31137 .34092 L
.31408 .34062 L
.31655 .34026 L
.32573 .33849 L
.33082 .33735 L
.33611 .33625 L
.34051 .33552 L
.3417 .33537 L
.34297 .33523 L
.34416 .33513 L
.34525 .33506 L
.34623 .33502 L
.34728 .335 L
.34827 .33501 L
.34916 .33503 L
.3502 .33509 L
.35132 .33518 L
.35237 .33531 L
.35334 .33545 L
.35526 .33581 L
.35736 .33635 L
.36155 .33784 L
.36579 .33995 L
.3703 .34281 L
.37836 .34918 L
.39428 .36312 L
.39876 .36649 L
.40365 .36959 L
.40842 .37196 L
.41292 .37359 L
.41528 .37422 L
.41782 .37473 L
.42005 .37506 L
.42134 .3752 L
.42255 .3753 L
.42323 .37534 L
.42398 .37538 L
.42535 .37541 L
.4267 .37542 L
.42796 .37539 L
.42934 .37534 L
.43014 .37529 L
.43089 .37524 L
.43407 .37495 L
.4368 .37459 L
.43972 .37413 L
.44509 .37308 L
.45771 .36992 L
.50294 .35525 L
Mistroke
.51178 .35249 L
.51561 .35148 L
.51893 .35079 L
.52073 .3505 L
.5226 .35028 L
.52357 .35019 L
.52447 .35013 L
.52498 .35011 L
.52546 .35009 L
.526 .35007 L
.52652 .35007 L
.52735 .35007 L
.52824 .35008 L
.52901 .35011 L
.52986 .35016 L
.53079 .35022 L
.5318 .35031 L
.53369 .35051 L
.55111 .35331 L
.55683 .35411 L
.56011 .35448 L
.56175 .35463 L
.56329 .35475 L
.56472 .35485 L
.5663 .35493 L
.56707 .35497 L
.5679 .355 L
.56878 .35502 L
.56961 .35503 L
.57117 .35504 L
.57265 .35503 L
.57399 .35499 L
.57544 .35492 L
.57696 .35483 L
.5786 .35469 L
.58156 .35435 L
.5844 .35389 L
.58696 .35336 L
.59261 .35174 L
.59533 .3507 L
.5982 .34939 L
.60273 .34682 L
.60718 .34356 L
.61422 .3366 L
.62697 .31958 L
.63051 .31519 L
.63457 .31098 L
.63837 .30784 L
.64296 .30493 L
.64524 .30379 L
Mistroke
.64775 .30274 L
.65249 .30124 L
.65527 .30059 L
.65793 .30012 L
.66093 .29972 L
.66264 .29954 L
.66421 .29942 L
.6656 .29933 L
.66709 .29926 L
.66836 .29921 L
.66977 .29917 L
.6713 .29915 L
.67215 .29915 L
.67294 .29915 L
.67446 .29915 L
.67591 .29917 L
.67726 .29919 L
.67869 .29923 L
.68126 .2993 L
.68702 .29946 L
.6886 .2995 L
.69006 .29953 L
.69157 .29954 L
.69237 .29955 L
.69322 .29955 L
.69394 .29955 L
.69471 .29955 L
.69612 .29952 L
.69747 .29949 L
.6987 .29944 L
.70014 .29937 L
.70147 .29928 L
.70303 .29915 L
.70444 .299 L
.70726 .29861 L
.71021 .29807 L
.71491 .29683 L
.7198 .295 L
.72424 .29284 L
.72825 .29049 L
.7442 .27931 L
.75333 .27352 L
.7587 .27076 L
.76399 .26857 L
.76984 .26674 L
.77306 .26597 L
.77661 .26531 L
.77819 .26507 L
.77994 .26485 L
.78149 .26469 L
Mistroke
.78315 .26455 L
.78466 .26445 L
.78547 .26441 L
.78635 .26437 L
.7879 .26432 L
.78936 .2643 L
.791 .26429 L
.79255 .26431 L
.79425 .26436 L
.79521 .26439 L
.79609 .26443 L
.79982 .26464 L
.8034 .26493 L
.81697 .26641 L
.83104 .26809 L
.83819 .2688 L
.84231 .26912 L
.8441 .26924 L
.84605 .26935 L
.84782 .26943 L
.84946 .26949 L
.85102 .26954 L
.85267 .26957 L
.85359 .26958 L
.85458 .26959 L
.85636 .26958 L
.85808 .26955 L
.85966 .26951 L
.86115 .26946 L
.86253 .26939 L
.86564 .26919 L
.86861 .26892 L
.87173 .26855 L
.87503 .26806 L
.87843 .26744 L
.88422 .26606 L
.88996 .26421 L
.89562 .26177 L
.89987 .25934 L
.90377 .25643 L
.90678 .2534 L
.90897 .25039 L
.90991 .2487 L
.91063 .2471 L
.91131 .24519 L
.91157 .24429 L
.91181 .24332 L
.91198 .24247 L
.91214 .24153 L
.91225 .24065 L
Mistroke
.91234 .23982 L
.91241 .23883 L
.91244 .23791 L
.91245 .23683 L
.91243 .23579 L
.91238 .23475 L
.91229 .23357 L
.91216 .23232 L
.91201 .23112 L
.91118 .22651 L
.90984 .22117 L
.90794 .21506 L
.88885 .17023 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->0,
  AnimationCycleRepetitions->0,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`16o`X0??l0097o00<0ool0@_l400Wo100io`00
TOl00`3oo`11o`030?oo00oo0P0go`00TOl00`3oo`0oo`805?l203Go001Fo`D0=_l00`3oo`0no`03
0?oo01Ko00<0ool0<_l005Co0P04o`<08?l200Co0P05o`<01?l00`3oo`0lo`806_l203;o001Co`03
0?oo00Oo0P0Mo`040?oo00_o00<0ool00ol00`3oo`0ko`030?oo01co00<0ool0;ol0057o0P0<o`03
0?oo01[o00@0ool02?l500Co0P0lo`030?oo01go00<0ool0;_l0053o00<0ool03?l00`3oo`0Do`@0
0_l0103oo`08o`040?oo00Go00<0ool0>_l00`3oo`0Oo`030?oo02go000do`D05_l00`3oo`0>o`03
0?oo01Wo00@0ool02Ol00`3o0005o`030?oo03[o00<0ool08?l00`3oo`0/o`00<_l200Go1@0@o`03
0?oo013o00<0ool06Ol200_o0P05o`030?oo03[o00<0ool08Ol00`3oo`0[o`00<?l200co100;o`03
0?oo01;o00<0ool0;?l00`3oo`0jo`030?oo02;o00<0ool0:_l0033o00<0ool03ol200Oo0P0Eo`03
0?oo02co00<0ool0>Ol00`3oo`0So`030?oo02[o000_o`030?oo01;o200Go`030?oo02_o00<0ool0
>Ol00`3oo`0To`030?oo02Wo000^o`030?oo03;o00<0ool0:ol00`3oo`0io`030?oo02Co00<0ool0
:Ol002ko00<0ool0<_l00`3oo`0[o`030?oo03Wo00<0ool09Ol00`3oo`0Xo`00;Ol00`3oo`0co`03
0?oo02_o00<0ool0>Ol00`3oo`0Uo`030?oo02So000]o`030?oo03?o00<0ool0:ol00`3oo`0io`03
0?oo02Ko00<0ool09ol002co00<0ool0=?l00`3oo`0[o`030?oo03Wo00<0ool09_l00`3oo`0Wo`00
;?l00`3oo`0do`030?oo02_o00<0ool0>Ol00`3oo`0Wo`030?oo02Ko000/o`030?oo03Co00<0ool0
:ol00`3oo`0io`030?oo02Oo00<0ool09_l002_o00<0ool0=Ol00`3oo`0[o`030?oo03Wo00<0ool0
:?l00`3oo`0Uo`00:ol00`3oo`0eo`030?oo01Oo0P04o`800ol400Go00<0ool0>Ol00`3oo`0Xo`03
0?oo02Go000[o`030?oo03Co00<0ool05ol0103oo`08o`030?oo00Ko00<0ool0>?l00`3oo`0Zo`03
0?oo02Co000[o`030?oo03Co00<0ool05ol0103oo`09o`030?oo00Go0P0io`030?oo02[o00<0ool0
9?l002_o00<0ool0<ol00`3oo`0Bo`@00_l0103oo`0:o`030?oo00Co00<0ool0>?l00`3oo`0[o`03
0?oo02?o000[o`030?oo03;o00<0ool06Ol0103oo`08o`040?oo00Go00<0ool0>?l00`3oo`0[o`03
0?oo02?o000[o`030?oo037o00<0ool06ol200[o0P06o`030?oo03So00<0ool0:ol00`3oo`0So`00
:ol00`3oo`0_o`80<_l00`3oo`0go`030?oo02go00<0ool08_l002_o00<0ool0;_l00`3oo`0bo`03
0?oo03Oo00<0ool0;Ol00`3oo`0Ro`00:ol00`3oo`0Po`030?oo00Oo100eo`030?oo03Oo00<0ool0
;_l00`3oo`0Qo`001ol500;o0P04o`805Ol00`3oo`0;o`D04Ol900So0P04o`801?l202?o00<0ool0
8_l200Co0P04o`801_l00`3oo`0Jo`D04?l00`3oo`0?o`D00_l200Co0P03o`002Ol00`3oo`07o`04
0?oo01Co00<0ool03Ol00`3oo`0Bo`030?oo00co00@0ool02?l0103oo`0Ro`030?oo027o00@0ool0
2?l0103oo`05o`030?oo01co00<0ool04Ol00`3oo`0@o`030?oo00Oo00@0ool00_l000Wo00<0ool0
2_l00`3oo`0Bo`030?oo00go00<0ool04ol200co00@0ool02ol00`3oo`0Po`030?oo027o00@0ool0
2ol00`3oo`02o`030?oo01go00<0ool04Ol00`3oo`0@o`030?oo00[o0@01o`7o00000ol0000200Co
00<0ool02?l201Go00<0ool01Ol400Co00<0ool05Ol00`3oo`03o`@00_l0103oo`09o`808ol00`3o
o`0Qo`040?oo00Wo0P05o`030?oo01go00<0ool04_l00`3oo`0?o`030?oo00So0P03o`002Ol00`3o
o`08o`030?oo01Go00<0ool03?l00`3oo`0Fo`802Ol0103oo`09o`030?oo02;o00<0ool08Ol0103o
o`09o`030?oo00Co00<0ool07Ol00`3oo`0Bo`030?oo00oo00<0ool02?l00`3oo`02o`002?l200[o
0`0Eo`030?oo00_o0P0Jo`030?oo00Oo0P0:o`<08_l00`3oo`0Ro`802_l300?o00<0ool07Ol201Co
00<0ool03_l200[o0`02o`00;?l00`3oo`0Xo`030?oo03Oo00<0ool0=?l00`3oo`0co`030?oo01oo
000]o`030?oo02So0P0go`030?oo03Co00<0ool0<ol00`3oo`0Oo`00;_l00`3oo`0Yo`030?oo03Co
00<0ool0<ol00`3oo`0do`030?oo01oo000^o`030?oo02[o00<0ool0<ol00`3oo`0co`030?oo03Co
00<0ool07ol002oo00<0ool0:_l203?o00<0ool0<ol00`3oo`0do`030?oo01oo000<ool03009o`00
3?l00`3oo`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?l000Ko00<0ool01_l00`3oo`06o`03
0?oo00Go00<0ool01_l0103oool300;o00<0ool01Ol300Oo00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01Ol00`3oo`06o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01_l00`3o
o`05o`030?oo00Ko00<0ool01_l00`3oo`06o`030?oo00Ko00<0o`001_l00`3oo`06o`030?oo00Ko
00<0ool01ol000co00<0ool08ol200Co00<0ool0:Ol00`3oo`08o`030?oo01oo00<0ool0:Ol00`3o
o`05o`030?oo027o00<0ool03ol201Wo00<0ool01ol003Co0P0jo`030?oo01ko00<0ool09?l500Oo
00<0ool0<_l202Go000fo`<0>?l00`3oo`0Mo`030?oo023o1005o`L05?l?00oo0`0Wo`00>Ol203Oo
00<0ool07?l00`3oo`0No`808_l200oo3`0Zo`00>ol303Go00<0ool06ol00`3oo`0Lo`808_l204[o
000no`80=?l201_o00<0ool06_l202;o0P1<o`00@?l203Co00<0ool06?l00`3oo`0Io`030?oo027o
00<0ool0C?l004;o0P0co`030?oo01Oo00<0ool06?l00`3oo`0Po`80Col004Co0P0bo`030?oo01Ko
00<0ool02Ol?02;o00<0ool0Col004Ko00<0ool0<?l501?o00<0ool01Ol402oo0P1Bo`00Aol203Go
2@0:o`030?oo00D0<?l305Co0019o`80??l=02Ko3`1Go`00Bol00`3oo`13o`030?oo02?o0`1Vo`00
C?l00`3oo`12o`030?oo027o0`1Xo`00COl202ko0P04o`800ol400Go00<0ool08?l00`3oo`1Yo`00
Col00`3oo`0Zo`040?oo00So00<0ool01_l00`3oo`0Oo`030?oo06[o001@o`030?oo02Wo00@0ool0
2Ol00`3oo`05o`808?l00`3oo`1Zo`00DOl00`3oo`0Xo`040?oo00[o00<0ool01?l00`3oo`0No`03
0?oo06_o001Bo`030?oo02Oo00@0ool02?l0103oo`05o`030?oo01go00<0ool0K?l005?o0P0Xo`80
2_l200Ko00<0ool07Ol00`3oo`1/o`00EOl203[o00<0ool07?l00`3oo`1]o`00Eol200[o2P0To`03
0?oo01_o00<0ool0K_l005Wo2P0:o`030?oo027o00<0ool06_l00`3oo`1_o`00K_l00`3oo`0Po`03
0?oo01Wo00<0ool0L?l006oo0P0Po`030?oo01Oo0P1co`00LOl00`3oo`0Mo`030?l000/02Ol307Go
001bo`030?oo01_o0`0<o`T0N?l007?o00<0ool05ol30003o`3o08go001do`030?oo01?o0`04o`03
0?oo08co001eo`804?l300Oo00<0ool0S?l007Oo0P0;o`<02_l00`3oo`2<o`00NOl<00Go0`04o`03
0?oo08co001lo`040?oo00_o00<0ool00ol00`3oo`2<o`00O?l0103oo`08o`D01?l208go001lo`04
0?oo00So00@0ool01Ol00`3oo`2<o`00O?l0103oo`09o`030?l000Go00<0ool0S?l007go0P0;o`80
1Ol00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00OOl200Co0P03o`<01_l00`3oo`2<o`00
O?l0103oo`08o`040?oo00Go00<0ool0S?l007co00@0ool02?l0103oo`05o`80SOl007co00@0ool0
2?l300Ko00<0ool0S?l007co00@0ool02Ol00`3oo`05o`030?oo08co001mo`802_l300Go00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .5 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.333333 0.25 0.333333 [
[0 .2375 -12 -9 ]
[0 .2375 12 0 ]
[.16667 .2375 -6 -9 ]
[.16667 .2375 6 0 ]
[.33333 .2375 -12 -9 ]
[.33333 .2375 12 0 ]
[.66667 .2375 -9 -9 ]
[.66667 .2375 9 0 ]
[.83333 .2375 -3 -9 ]
[.83333 .2375 3 0 ]
[1 .2375 -9 -9 ]
[1 .2375 9 0 ]
[.4875 .05 -24 -4.5 ]
[.4875 .05 0 4.5 ]
[.4875 .11667 -24 -4.5 ]
[.4875 .11667 0 4.5 ]
[.4875 .18333 -24 -4.5 ]
[.4875 .18333 0 4.5 ]
[.4875 .31667 -18 -4.5 ]
[.4875 .31667 0 4.5 ]
[.4875 .38333 -18 -4.5 ]
[.4875 .38333 0 4.5 ]
[.4875 .45 -18 -4.5 ]
[.4875 .45 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .5 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 .25 m
0 .25625 L
s
[(-1.5)] 0 .2375 0 1 Mshowa
.16667 .25 m
.16667 .25625 L
s
[(-1)] .16667 .2375 0 1 Mshowa
.33333 .25 m
.33333 .25625 L
s
[(-0.5)] .33333 .2375 0 1 Mshowa
.66667 .25 m
.66667 .25625 L
s
[(0.5)] .66667 .2375 0 1 Mshowa
.83333 .25 m
.83333 .25625 L
s
[(1)] .83333 .2375 0 1 Mshowa
1 .25 m
1 .25625 L
s
[(1.5)] 1 .2375 0 1 Mshowa
.125 Mabswid
.03333 .25 m
.03333 .25375 L
s
.06667 .25 m
.06667 .25375 L
s
.1 .25 m
.1 .25375 L
s
.13333 .25 m
.13333 .25375 L
s
.2 .25 m
.2 .25375 L
s
.23333 .25 m
.23333 .25375 L
s
.26667 .25 m
.26667 .25375 L
s
.3 .25 m
.3 .25375 L
s
.36667 .25 m
.36667 .25375 L
s
.4 .25 m
.4 .25375 L
s
.43333 .25 m
.43333 .25375 L
s
.46667 .25 m
.46667 .25375 L
s
.53333 .25 m
.53333 .25375 L
s
.56667 .25 m
.56667 .25375 L
s
.6 .25 m
.6 .25375 L
s
.63333 .25 m
.63333 .25375 L
s
.7 .25 m
.7 .25375 L
s
.73333 .25 m
.73333 .25375 L
s
.76667 .25 m
.76667 .25375 L
s
.8 .25 m
.8 .25375 L
s
.86667 .25 m
.86667 .25375 L
s
.9 .25 m
.9 .25375 L
s
.93333 .25 m
.93333 .25375 L
s
.96667 .25 m
.96667 .25375 L
s
.25 Mabswid
0 .25 m
1 .25 L
s
.5 .05 m
.50625 .05 L
s
[(-0.6)] .4875 .05 1 0 Mshowa
.5 .11667 m
.50625 .11667 L
s
[(-0.4)] .4875 .11667 1 0 Mshowa
.5 .18333 m
.50625 .18333 L
s
[(-0.2)] .4875 .18333 1 0 Mshowa
.5 .31667 m
.50625 .31667 L
s
[(0.2)] .4875 .31667 1 0 Mshowa
.5 .38333 m
.50625 .38333 L
s
[(0.4)] .4875 .38333 1 0 Mshowa
.5 .45 m
.50625 .45 L
s
[(0.6)] .4875 .45 1 0 Mshowa
.125 Mabswid
.5 .06667 m
.50375 .06667 L
s
.5 .08333 m
.50375 .08333 L
s
.5 .1 m
.50375 .1 L
s
.5 .13333 m
.50375 .13333 L
s
.5 .15 m
.50375 .15 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .2 m
.50375 .2 L
s
.5 .21667 m
.50375 .21667 L
s
.5 .23333 m
.50375 .23333 L
s
.5 .26667 m
.50375 .26667 L
s
.5 .28333 m
.50375 .28333 L
s
.5 .3 m
.50375 .3 L
s
.5 .33333 m
.50375 .33333 L
s
.5 .35 m
.50375 .35 L
s
.5 .36667 m
.50375 .36667 L
s
.5 .4 m
.50375 .4 L
s
.5 .41667 m
.50375 .41667 L
s
.5 .43333 m
.50375 .43333 L
s
.5 .03333 m
.50375 .03333 L
s
.5 .01667 m
.50375 .01667 L
s
.5 .46667 m
.50375 .46667 L
s
.5 .48333 m
.50375 .48333 L
s
.25 Mabswid
.5 0 m
.5 .5 L
s
0 0 m
1 0 L
1 .5 L
0 .5 L
closepath
clip
newpath
.5 Mabswid
.89109 .17285 m
.87971 .15041 L
.86839 .1304 L
.86226 .12165 L
.85432 .1128 L
.84878 .10797 L
.84298 .10377 L
.83675 .10004 L
.82934 .09645 L
.82229 .09371 L
.81391 .09122 L
.81006 .09032 L
.80582 .0895 L
.80177 .08887 L
.79796 .08842 L
.79565 .08822 L
.79318 .08805 L
.79176 .08798 L
.79042 .08793 L
.78922 .0879 L
.78792 .08788 L
.78676 .08788 L
.7855 .0879 L
.78417 .08792 L
.78294 .08797 L
.78074 .08808 L
.77839 .08825 L
.77612 .08847 L
.77367 .08877 L
.76929 .08947 L
.76514 .09033 L
.76141 .0913 L
.75425 .09372 L
.74728 .09696 L
.74153 .10056 L
.73703 .10423 L
.73292 .10864 L
.7295 .11376 L
.72816 .11644 L
.72702 .11934 L
.72618 .12211 L
.72559 .12464 L
.72519 .12707 L
.72501 .12845 L
.72488 .12975 L
.72477 .13112 L
.7247 .13237 L
.72467 .13306 L
.72465 .13382 L
.72462 .1352 L
Mistroke
.72462 .13649 L
.72462 .13767 L
.72464 .13902 L
.72467 .14031 L
.72482 .14511 L
.72489 .1479 L
.72492 .14932 L
.72494 .15013 L
.72494 .15088 L
.72495 .15225 L
.72495 .15351 L
.72492 .15497 L
.72489 .15637 L
.72483 .15791 L
.72474 .15955 L
.72452 .16257 L
.72418 .1659 L
.72367 .1697 L
.72251 .17633 L
.72088 .18371 L
.71714 .19785 L
.70718 .23114 L
.70289 .24461 L
.70064 .25088 L
.69852 .25588 L
.69645 .25966 L
.69531 .26131 L
.69404 .26278 L
.69292 .26382 L
.69161 .26474 L
.69035 .2654 L
.6891 .26586 L
.68829 .26607 L
.68784 .26617 L
.6874 .26625 L
.68661 .26635 L
.68571 .26642 L
.68522 .26644 L
.6847 .26645 L
.68374 .26643 L
.68319 .26641 L
.68267 .26638 L
.68146 .26627 L
.68036 .26614 L
.67929 .26599 L
.67722 .26566 L
.67218 .26473 L
.66757 .26389 L
.6624 .26311 L
.6596 .26278 L
Mistroke
.6581 .26264 L
.65648 .26252 L
.65498 .26244 L
.6536 .26239 L
.65222 .26236 L
.65091 .26236 L
.6494 .2624 L
.64856 .26243 L
.64779 .26247 L
.64636 .26257 L
.64487 .26271 L
.6423 .26304 L
.63953 .26353 L
.63663 .2642 L
.63362 .26508 L
.62849 .26706 L
.62376 .26948 L
.61969 .27204 L
.61306 .27716 L
.60677 .28243 L
.60348 .28483 L
.60031 .28665 L
.59746 .28784 L
.59577 .28837 L
.59415 .28877 L
.59239 .2891 L
.59038 .28937 L
.58932 .28947 L
.58817 .28955 L
.58705 .28961 L
.58601 .28965 L
.5848 .28968 L
.58362 .28969 L
.58296 .28969 L
.58223 .28969 L
.58094 .28968 L
.57968 .28966 L
.57831 .28963 L
.5758 .28956 L
.5729 .28947 L
.5701 .28939 L
.56847 .28936 L
.56761 .28934 L
.5667 .28933 L
.56592 .28932 L
.56507 .28931 L
.56357 .28931 L
.56204 .28932 L
.56059 .28934 L
.55928 .28936 L
Mistroke
.55786 .28941 L
.55632 .28947 L
.55469 .28955 L
.55178 .28974 L
.54912 .28998 L
.54666 .29026 L
.54121 .29105 L
.53589 .2921 L
.53109 .29325 L
.52273 .29563 L
.51415 .29806 L
.50932 .29914 L
.50697 .29956 L
.50475 .29989 L
.50243 .30016 L
.49989 .30038 L
.49854 .30046 L
.49725 .30053 L
.4958 .30058 L
.49498 .3006 L
.49421 .30062 L
.49283 .30064 L
.49152 .30064 L
.49006 .30063 L
.48932 .30062 L
.48848 .3006 L
.48707 .30057 L
.48552 .30052 L
.48264 .3004 L
.47643 .30006 L
.46955 .2996 L
.45676 .29862 L
.44297 .29747 L
.43581 .29675 L
.42949 .29597 L
.42408 .29508 L
.41878 .29386 L
.41446 .29247 L
.41087 .29089 L
.40725 .28876 L
.40369 .286 L
.39763 .27981 L
.38996 .27098 L
.38134 .26239 L
.37074 .25468 L
.36455 .2515 L
.36129 .25019 L
.35774 .24902 L
.35433 .24815 L
.35285 .24785 L
Mistroke
.35123 .24758 L
.34981 .24739 L
.3483 .24723 L
.34748 .24716 L
.34673 .24712 L
.34529 .24706 L
.34391 .24704 L
.34266 .24707 L
.34131 .24713 L
.3399 .24723 L
.33871 .24736 L
.33762 .2475 L
.3352 .2479 L
.3309 .24892 L
.32674 .2502 L
.32319 .25139 L
.32143 .25193 L
.3206 .25215 L
.31985 .25232 L
.31918 .25244 L
.31845 .25254 L
.31774 .2526 L
.31735 .25261 L
.31698 .25261 L
.31634 .25258 L
.31576 .25252 L
.31508 .2524 L
.31444 .25224 L
.31318 .25181 L
.31197 .25125 L
.30905 .24942 L
.29071 .23336 L
.27827 .22323 L
.26635 .21498 L
.25639 .20939 L
.25232 .20757 L
.25056 .20693 L
.24926 .20654 L
.24873 .20643 L
.24848 .20639 L
.24828 .20636 L
.24799 .20635 L
.24779 .20639 L
.24771 .20647 L
.24771 .20657 L
.24782 .20672 L
.24805 .20691 L
.24841 .20715 L
.24896 .20744 L
.25035 .20805 L
Mistroke
.25431 .20933 L
.25711 .20999 L
.25876 .21031 L
.26063 .21062 L
.26237 .21086 L
.26438 .21108 L
.2663 .21125 L
.26816 .21136 L
.27018 .21142 L
.27215 .21144 L
.27396 .21141 L
.27597 .21133 L
.27821 .21118 L
.28062 .21095 L
.28271 .21069 L
.28505 .21032 L
.2903 .20923 L
.29512 .20786 L
.30035 .20593 L
.3057 .20341 L
.3141 .19801 L
.31839 .19434 L
.32199 .19054 L
.32506 .18657 L
.32786 .18206 L
.33036 .1768 L
.33126 .17438 L
.33209 .17172 L
.33274 .16917 L
.33322 .16682 L
.33358 .16443 L
.33383 .16214 L
.33399 .15946 L
.33401 .15698 L
.33391 .15425 L
.33379 .15272 L
.33365 .15134 L
.33332 .14893 L
.33286 .1464 L
.33176 .14194 L
.33019 .13726 L
.32806 .13234 L
.32327 .12419 L
.31789 .11759 L
.3109 .11139 L
.30671 .10865 L
.30277 .10662 L
.2993 .10525 L
.29731 .10463 L
.29549 .10417 L
Mistroke
.29359 .1038 L
.29255 .10365 L
.29157 .10353 L
.28984 .1034 L
.28797 .10336 L
.28618 .10344 L
.28451 .1036 L
.28304 .10381 L
.28148 .10413 L
.27981 .10456 L
.27809 .10511 L
.27511 .10632 L
.27014 .10915 L
.26521 .113 L
.25552 .12338 L
.25127 .12841 L
.24736 .13259 L
.24368 .13578 L
.2417 .13715 L
.23948 .13841 L
.23714 .13943 L
.23484 .14017 L
.23346 .14051 L
.23195 .14078 L
.23063 .14095 L
.22914 .14107 L
.22769 .14113 L
.22685 .14113 L
.22607 .14112 L
.2246 .14105 L
.22302 .14093 L
.22157 .14078 L
.22018 .14059 L
.21694 .14004 L
.2099 .13839 L
.20193 .13605 L
.18709 .13117 L
.17374 .12702 L
.16698 .12533 L
.16342 .12463 L
.16032 .12417 L
.15861 .12398 L
.15708 .12386 L
.1563 .12382 L
.15545 .12379 L
.15458 .12378 L
.15378 .12379 L
.15304 .12381 L
.15234 .12385 L
.15107 .12395 L
Mistroke
.15037 .12403 L
.14973 .12412 L
.14833 .12436 L
.14725 .12461 L
.14612 .12493 L
.14416 .12563 L
.14196 .12668 L
.14008 .12785 L
.13653 .13088 L
.13491 .1327 L
.13323 .13491 L
.13026 .13975 L
.1277 .14498 L
.1234 .15626 L
.11948 .17008 L
.11771 .17838 L
.11702 .18247 L
.11648 .18622 L
.11607 .18986 L
.11578 .19331 L
.11567 .1952 L
.11559 .19694 L
.11554 .19886 L
.11553 .19992 L
.11553 .20088 L
.11556 .20264 L
.11561 .20424 L
.11569 .20599 L
.11583 .20782 L
.11613 .21088 L
.1166 .21418 L
.11733 .21797 L
.11816 .22134 L
.1206 .22856 L
.12353 .23474 L
.12685 .24008 L
.13476 .2492 L
.13921 .25298 L
.14431 .25654 L
.15353 .26146 L
.16204 .26483 L
.17889 .27014 L
.19725 .27683 L
.21802 .28705 L
.23774 .29958 L
.25593 .31494 L
.27063 .32901 L
.27824 .33506 L
.28248 .33773 L
.28715 .34005 L
Mistroke
.29142 .34162 L
.29392 .34231 L
.29629 .34281 L
.29742 .34299 L
.29851 .34314 L
.29949 .34326 L
.30055 .34335 L
.3017 .34343 L
.30277 .34347 L
.30393 .3435 L
.30521 .34349 L
.30643 .34345 L
.30757 .34339 L
.30882 .34329 L
.31016 .34316 L
.3129 .34279 L
.31539 .34236 L
.32469 .34012 L
.33521 .33709 L
.33965 .33596 L
.34195 .33547 L
.34321 .33525 L
.34439 .33507 L
.34546 .33493 L
.34661 .3348 L
.34766 .33472 L
.34862 .33468 L
.3496 .33466 L
.35049 .33466 L
.35152 .3347 L
.35247 .33477 L
.35348 .33487 L
.35455 .33502 L
.35643 .33539 L
.35844 .33593 L
.36062 .3367 L
.3648 .33872 L
.3692 .34159 L
.37701 .3483 L
.39241 .36393 L
.39672 .36785 L
.40143 .37152 L
.40558 .37413 L
.40788 .37532 L
.41036 .37639 L
.41263 .37719 L
.41516 .3779 L
.4163 .37815 L
.41754 .37838 L
.41871 .37857 L
Mistroke
.4198 .37871 L
.4212 .37884 L
.42195 .3789 L
.42275 .37894 L
.4241 .37898 L
.42488 .37899 L
.42561 .37898 L
.427 .37895 L
.42828 .37889 L
.42899 .37884 L
.42978 .37878 L
.43122 .37864 L
.43387 .37831 L
.43687 .37783 L
.44318 .37653 L
.45512 .37328 L
.48252 .36385 L
.50362 .35579 L
.51187 .35283 L
.51537 .35177 L
.51833 .35105 L
.51988 .35076 L
.52131 .35055 L
.52209 .35046 L
.52281 .35039 L
.52359 .35034 L
.52443 .35031 L
.52521 .3503 L
.52606 .35031 L
.52686 .35034 L
.52761 .35038 L
.52849 .35045 L
.52946 .35055 L
.53129 .3508 L
.53982 .3525 L
.5496 .35464 L
.55442 .35553 L
.55979 .35634 L
.5627 .35667 L
.56426 .35682 L
.56595 .35695 L
.56751 .35704 L
.56894 .3571 L
.57038 .35714 L
.57174 .35715 L
.57314 .35714 L
.57446 .35711 L
.5759 .35705 L
.57744 .35695 L
.57897 .35682 L
Mistroke
.58057 .35664 L
.58339 .35622 L
.58597 .35571 L
.58832 .35512 L
.59345 .35341 L
.59619 .3522 L
.59868 .35089 L
.60303 .34806 L
.60727 .34446 L
.61137 .34002 L
.62334 .32191 L
.62966 .31283 L
.6337 .30846 L
.63784 .30509 L
.64218 .3025 L
.64484 .30131 L
.64741 .30037 L
.64995 .29963 L
.65243 .29907 L
.65538 .29856 L
.65668 .29839 L
.65812 .29823 L
.65941 .29811 L
.66081 .29801 L
.6623 .29793 L
.6637 .29788 L
.66511 .29785 L
.66645 .29784 L
.66765 .29784 L
.66897 .29785 L
.67041 .29788 L
.67201 .29793 L
.67524 .29806 L
.68103 .29838 L
.68402 .29855 L
.68724 .29871 L
.6887 .29877 L
.69024 .29883 L
.69168 .29886 L
.69297 .29889 L
.69433 .2989 L
.69556 .29889 L
.69689 .29887 L
.69829 .29882 L
.69949 .29876 L
.70059 .29869 L
.70306 .29847 L
.70439 .29831 L
.70582 .2981 L
.70835 .29762 L
Mistroke
.71075 .29704 L
.7129 .2964 L
.7176 .29455 L
.72234 .29202 L
.7385 .27982 L
.74605 .27405 L
.75062 .27102 L
.75512 .26846 L
.76027 .26608 L
.7652 .26431 L
.76784 .26357 L
.77076 .26288 L
.77394 .26231 L
.77701 .26191 L
.77864 .26175 L
.78019 .26163 L
.78106 .26158 L
.7819 .26154 L
.78287 .26151 L
.78376 .26148 L
.78464 .26147 L
.78559 .26146 L
.7873 .26148 L
.78836 .2615 L
.78936 .26154 L
.79124 .26163 L
.79292 .26173 L
.79471 .26187 L
.79794 .26218 L
.80542 .26312 L
.82069 .26562 L
.83559 .26809 L
.84212 .26901 L
.84903 .26981 L
.85261 .27013 L
.8545 .27027 L
.85652 .27041 L
.85837 .2705 L
.86005 .27058 L
.86164 .27063 L
.86331 .27067 L
.86512 .27069 L
.86675 .27068 L
.86861 .27066 L
.87035 .27061 L
.87195 .27055 L
.87341 .27047 L
.87664 .27023 L
.87983 .26991 L
.88323 .26946 L
Mistroke
.88668 .26886 L
.88971 .26823 L
.8956 .26661 L
.90121 .26447 L
.9057 .26213 L
.9093 .25961 L
.91204 .25703 L
.91441 .25389 L
.91533 .25221 L
.91613 .2503 L
.91643 .2494 L
.91671 .24842 L
.91692 .24749 L
.91708 .24663 L
.91721 .24575 L
.91731 .2449 L
.91738 .24399 L
.91742 .24299 L
.91742 .2421 L
.9174 .24113 L
.91736 .2403 L
.91729 .23936 L
.91717 .23825 L
.91704 .23721 L
.91664 .2348 L
.91558 .23023 L
.91214 .2196 L
.90778 .20871 L
.89554 .18198 L
.89109 .17285 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 144},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  AnimationCycleOffset->-1,
  AnimationCycleRepetitions->Infinity,
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000T0P10`40000100000800002008200000
P800P020P<30`<3L`:K:l:C8l3<006H009T00<`0000c03<c06Hc09Tc0<`c0?lc001V03=V06IV09UV
0<aV0?mV002I03>I06JI09VI0<bI0?nI003<03?<06K<09W<0<c<0?o<03?o06Ko09Wo0<co0000<c<0
<fH0<iT0<l`0<ol0<`0c<c<c<fHc<iTc<l`c<olc<`1V<c=V<fIV<iUV<laV<omV<`2I<c>I<fJI<iVI
<lbI<onI<`3<<c?<<fK<<iW<<lc<<oo<<`3o<c?o<fKo<iWo<lco<ooo<`00IS<0IVH0IYT0I/`0I_l0
IP0cIS<cIVHcIYTcI/`cI_lcIP1VIS=VIVIVIYUVI/aVI_mVIP2IIS>IIVJIIYVII/bII_nIIP3<IS?<
IVK<IYW<I/c<I_o<IP3oIS?oIVKoIYWoI/coI_ooIP00VC<0VFH0VIT0VL`0VOl0V@0cVC<cVFHcVITc
VL`cVOlcV@1VVC=VVFIVVIUVVLaVVOmVV@2IVC>IVFJIVIVIVLbIVOnIV@3<VC?<VFK<VIW<VLc<VOo<
V@3oVC?oVFKoVIWoVLcoVOooV@00c3<0c6H0c9T0c<`0c?l0c00cc3<cc6Hcc9Tcc<`cc?lcc01Vc3=V
c6IVc9UVc<aVc?mVc02Ic3>Ic6JIc9VIc<bIc?nIc03<c3?<c6K<c9W<c<c<c?o<c03oc3?oc6Koc9Wo
c<coc?ooc3<0ofH0oiT0ol`0o`0coc<cofHcoiTcol`coolco`1Voc=VofIVoiUVolaVoomVo`2Ioc>I
ofJIoiVIolbIoonIo`3<oc?<ofK<oiW<olc<ooo<oc?oofKooiWoolcoo`410@820P<30`@410D51@H6
1PL71`P820T92@X:2P/;2``<30h>3Pl?3a0@414A4A8B4Q<C4a@D5000000000000000000000000000
00000?okl:2PY820P?l0003o0?oo0000ool0o`3ooooooooo8Ol00?oo8Ol00?oo8Ol00?oo8Ol00?oo
8Ol0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0S?l0097o00<0ool0
S?l0097o00<0ool0S?l0097o00<0ool0S?l007go0P04o`800ol300Ko00<0ool0S?l007co00@0ool0
2?l0103oo`05o`030?oo08co001lo`040?oo00So00@0ool01Ol208go001fo`@00_l0103oo`08o`<0
1_l00`3oo`2<o`00O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`14o`/0?Ol0097o00<0ool0@?l400_o1@0ho`00TOl00`3oo`0no`80
5?l203Ko002Ao`030?oo03go00<0ool05_l203Co001Eo`L0=Ol00`3oo`0lo`030?oo01Wo00<0ool0
<Ol005Go00<0ool01?l203?o00<0ool0>_l201go00<0ool0<?l005?o0P09o`030?oo01co0P04o`80
1Ol300Co00<0ool0>_l00`3oo`0Mo`030?oo02oo001Bo`030?oo00[o00<0ool06_l0103oo`0;o`03
0?oo00?o00<0ool0>Ol2023o00<0ool0;_l0053o0P0>o`030?oo01Wo00@0ool02?l500Co0P0jo`03
0?oo023o00<0ool0;Ol003Co0`0Ho`030?oo00ko00<0ool04ol400;o00@0ool02?l0103oo`05o`03
0?oo03Wo00<0ool08Ol00`3oo`0/o`00<Ol300;o1P0Bo`030?oo013o00<0ool06?l0103oo`09o`03
0?l000Go00<0ool0>?l202?o00<0ool0;?l0033o0P0:o`<03ol00`3oo`0Ao`030?oo01So0P0;o`80
1Ol00`3oo`0ho`030?oo02?o00<0ool0:ol002oo00<0ool03Ol200co00<0ool04ol00`3oo`0[o`03
0?oo03So00<0ool09?l00`3oo`0Zo`00;_l00`3oo`0@o`@01_l201Ko00<0ool0:ol00`3oo`0ho`03
0?oo02Co00<0ool0:_l002ko00<0ool05?l701So00<0ool0:_l00`3oo`0ho`030?oo02Go00<0ool0
:Ol002go00<0ool0=?l00`3oo`0Zo`030?oo03So00<0ool09_l00`3oo`0Xo`00;Ol00`3oo`0do`03
0?oo02[o00<0ool0>?l00`3oo`0Vo`030?oo02So000/o`030?oo03Go00<0ool0:_l00`3oo`0ho`03
0?oo02Oo00<0ool09ol002co00<0ool0=Ol00`3oo`0Zo`030?oo03So00<0ool0:?l00`3oo`0Vo`00
;?l00`3oo`0eo`030?oo02[o00<0ool0>?l00`3oo`0Xo`030?oo02Ko000[o`030?oo03Ko00<0ool0
:_l00`3oo`0ho`030?oo02Wo00<0ool09Ol002_o00<0ool0=_l00`3oo`0Zo`030?oo03So00<0ool0
:Ol00`3oo`0Uo`00:ol00`3oo`0fo`030?oo02[o00<0ool0>?l00`3oo`0Zo`030?oo02Co000[o`03
0?oo03Ko00<0ool05_l200Co0P03o`@01Ol00`3oo`0ho`030?oo02[o00<0ool09?l002_o00<0ool0
=Ol00`3oo`0Fo`040?oo00So00<0ool01_l00`3oo`0go`030?oo02co00<0ool08ol002_o00<0ool0
=Ol00`3oo`0Fo`040?oo00Wo00<0ool01Ol203So00<0ool0;?l00`3oo`0So`00:_l00`3oo`0eo`03
0?oo017o1002o`040?oo00[o00<0ool01?l00`3oo`0go`030?oo02co00<0ool08ol002[o00<0ool0
=?l00`3oo`0Ho`040?oo00So00@0ool01Ol00`3oo`0fo`030?oo02ko00<0ool08_l002[o00<0ool0
<ol00`3oo`0Jo`802_l200Ko00<0ool0=_l00`3oo`0^o`030?oo02;o000Zo`030?oo03;o00<0ool0
;ol00`3oo`0fo`030?oo02oo00<0ool08Ol002[o00<0ool0<Ol00`3oo`0`o`030?oo03Ko00<0ool0
;ol00`3oo`0Qo`00:_l00`3oo`0Po`803?l303?o00<0ool0=_l00`3oo`0`o`030?oo023o0007o`D0
0_l200Co0P0Do`030?oo00co1@0?o`@01_l400Go0P04o`801?l202?o00<0ool08_l200Co0P04o`80
1Ol00`3oo`0Ko`D04Ol00`3oo`0>o`D00_l200Co0P03o`002Ol00`3oo`07o`040?oo01?o0P0?o`03
0?oo01?o1`07o`040?oo00So00@0ool08_l00`3oo`0Qo`040?oo00So00@0ool01?l00`3oo`0Mo`03
0?oo017o00<0ool04?l00`3oo`07o`040?oo00;o0009o`030?oo00[o00<0ool04_l00`3oo`0=o`03
0?oo01Co0P0;o`040?oo00_o00<0ool08?l00`3oo`0Qo`040?oo00_o00<0ool00_l00`3oo`0Mo`03
0?oo01;o00<0ool03ol00`3oo`0:o`400Ol1o`0000?o00000P04o`030?oo00So0P0Eo`030?oo00Go
1004o`030?oo01Ko0P03o`@00_l0103oo`09o`808ol00`3oo`0Qo`040?oo00Wo0P05o`030?oo01go
00<0ool04_l00`3oo`0?o`030?oo00So0P03o`002Ol00`3oo`08o`030?oo01Go00<0ool03?l00`3o
o`0Ho`030?oo00Ko00@0ool02Ol00`3oo`0Ro`030?oo027o00@0ool02Ol00`3oo`03o`030?oo01ko
00<0ool04_l00`3oo`0?o`030?oo00So00<0ool00_l000So0P0:o`<05Ol00`3oo`0;o`806ol00`3o
o`06o`802_l302;o00<0ool08_l200[o0`03o`030?oo01go0P0Eo`030?oo00go0P0:o`<00_l002co
00<0ool0:Ol00`3oo`0fo`030?oo03Co00<0ool0=?l00`3oo`0No`00;?l00`3oo`0Zo`030?oo03Go
00<0ool0<ol00`3oo`0eo`030?oo01ko000]o`030?oo02[o00<0ool0=?l00`3oo`0co`030?oo03Go
0P0Oo`00;_l00`3oo`0Zo`80=?l00`3oo`0bo`030?oo03Oo00<0ool07Ol002ko00<0ool0;?l00`3o
o`05o`D09ol00`3oo`0bo`030?oo03Ko0P0Oo`003?oo00`02Ol000co00<0ool01Ol00`3oo`06o`03
0?oo00Ko00<0ool01_l300Ko00<0ool01_l00`3oo`06o`030?oo00Go00<0ool01_l0103oool300;o
00<0ool01_l200Oo00<0ool01_l00`3oo`06o`030?oo00Ko00<0ool01Ol00`3oo`06o`030?oo00Ko
00<0ool01_l00`3oo`06o`030?oo00Go0P08o`030?oo00Go00<0ool01_l00`3oo`06o`030?oo00Ko
00<0ool01_l01@3oool000Co00<0ool01_l00`3oo`06o`030?oo00Oo000<o`030?oo02?o00<0ool0
0ol00`3oo`0Yo`030?oo00So0P0Po`030?oo02Wo00<0ool01Ol00`3oo`0Qo`030?oo01;o00<0ool0
5Ol00`3oo`07o`00<ol303_o00<0ool07Ol00`3oo`0Ro`L01ol201?o2`0Ho`808_l003Ko0`0io`03
0?oo01co00<0ool08?l300Ko200@o`@02ol4017o0`0To`00>Ol403Go00<0ool07?l00`3oo`0Mo`<0
8?l00`3oo`0Aoa409ol003go0P0do`030?oo01_o00<0ool07?l00`3oo`0Oo`807_l00`3oo`0[o`00
?ol203?o00<0ool06_l00`3oo`0Ko`030?oo01oo00<0ool0C?l0047o0P0bo`030?oo01Wo00<0ool0
6_l00`3oo`0Oo`030?oo04go0013o`80<Ol00`3oo`0Ho`030?oo01So0P0Po`80D?l004Go00<0ool0
;ol201So00<0ool02ol600?o100Qo`030?oo053o0016o`80<Ol201Ko00<0ool01_l600Go100Ro`80
Dol004So0P0ao`H04?l00`3oo`02o`@0;ol205Go001:o`80=OlE02;o4P1Fo`00C?l00`3oo`12o`03
0?oo02;o0P1Xo`00COl204;o00<0ool08Ol00`3oo`1Xo`00Col00`3oo`0[o`801?l200?o1005o`03
0?oo023o00<0ool0JOl0053o00<0ool0:Ol0103oo`08o`030?oo00Ko00<0ool07ol00`3oo`1Zo`00
DOl00`3oo`0Xo`040?oo00Wo00<0ool01Ol201oo00<0ool0Jol005;o00<0ool09ol0103oo`0:o`03
0?oo00Co00<0ool07Ol00`3oo`1/o`00D_l00`3oo`0Wo`040?oo00So00@0ool01Ol00`3oo`0Lo`03
0?oo06go001Co`030?oo02Oo0P0:o`801_l00`3oo`0Lo`030?oo06go001Do`80>ol00`3oo`0Ko`03
0?oo06ko001Fo`803?l802Go00<0ool06ol00`3oo`1^o`00F?l200Oo0`08o`<08_l00`3oo`0Jo`03
0?oo06oo001Jo`L03_l00`3oo`0Oo`030?oo01Wo00<0ool0L?l0073o00<0ool07_l00`3oo`0Ho`03
0?oo077o001ao`030?oo01go00<0o`002@0=o`80M?l007;o00<0ool06ol300[o3@1fo`00Lol00`3o
o`0Ho`8000?o0?l0SOl007Co00<0ool05?l300?o00<0ool0S?l007Go00<0ool04Ol200Ko00<0ool0
S?l007Ko00<0ool03_l200So00<0ool0S?l007Oo0P04o`801?l400?o0`04o`030?oo08co001io`X0
2?l00`3oo`03o`030?oo08co001lo`040?oo00So1@04o`80SOl007co00@0ool02?l0103oo`05o`03
0?oo08co001lo`040?oo00Wo00<0o`001Ol00`3oo`2<o`00OOl200_o0P05o`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`030?oo08co002Ao`03
0?oo08co002Ao`030?oo08co001mo`801?l200?o0`06o`030?oo08co001lo`040?oo00So00@0ool0
1Ol00`3oo`2<o`00O?l0103oo`08o`040?oo00Go0P2=o`00O?l0103oo`08o`<01_l00`3oo`2<o`00
O?l0103oo`09o`030?oo00Go00<0ool0S?l007go0P0:o`<01Ol00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00
TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00TOl00`3oo`2<o`00oolQo`00oolQo`00
oolQo`00oolQo`00oolQo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {143, 0}} -> {-1.63536, -0.8064, 0.0112783, \
0.0112783}}]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
WindowSize->{1016, 668},
WindowMargins->{{0, Automatic}, {Automatic, 0}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 140, 4, 240, "Title"],

Cell[CellGroupData[{
Cell[1904, 59, 32, 0, 53, "Section"],
Cell[1939, 61, 345, 9, 52, "Text"],
Cell[2287, 72, 320, 6, 52, "Text"],
Cell[2610, 80, 304, 6, 52, "Text"],
Cell[2917, 88, 196, 4, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[3150, 97, 157, 4, 33, "Section"],

Cell[CellGroupData[{
Cell[3332, 105, 48, 0, 45, "Subsection"],
Cell[3383, 107, 67, 1, 27, "Input"],
Cell[3453, 110, 103, 2, 27, "Input"],
Cell[3559, 114, 128, 2, 27, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[3724, 121, 46, 0, 29, "Subsection"],
Cell[3773, 123, 326, 7, 50, "Text"],
Cell[4102, 132, 410, 7, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[4549, 144, 103, 2, 29, "Subsection"],
Cell[4655, 148, 105, 3, 32, "Text"],
Cell[4763, 153, 244, 6, 42, "Input"],
Cell[5010, 161, 123, 2, 27, "Input"],
Cell[5136, 165, 258, 6, 42, "Input"],

Cell[CellGroupData[{
Cell[5419, 175, 78, 0, 42, "Subsubsection"],
Cell[5500, 177, 121, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[5670, 185, 60, 0, 29, "Subsection"],
Cell[5733, 187, 220, 3, 43, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[5990, 195, 130, 3, 46, "Subsection"],
Cell[6123, 200, 121, 2, 27, "Input"],
Cell[6247, 204, 132, 2, 43, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[6428, 212, 31, 0, 33, "Section"],
Cell[6462, 214, 101, 3, 33, "Text"],
Cell[6566, 219, 202, 3, 30, "Input"],
Cell[6771, 224, 154, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[6962, 231, 44, 0, 33, "Section"],
Cell[7009, 233, 66, 0, 33, "Text"],
Cell[7078, 235, 1951, 30, 290, "Input"],
Cell[9032, 267, 2032, 35, 290, "Input"],
Cell[11067, 304, 64, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11168, 310, 37, 0, 33, "Section"],

Cell[CellGroupData[{
Cell[11230, 314, 80, 1, 30, "Input"],
Cell[11313, 317, 14554, 640, 153, 7748, 551, "GraphicsData", "PostScript", \
"Graphics"],
Cell[25870, 959, 130, 3, 29, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[26049, 968, 35, 0, 53, "Section"],
Cell[26087, 970, 176, 4, 33, "Text"],
Cell[26266, 976, 461, 7, 71, "Text"],

Cell[CellGroupData[{
Cell[26752, 987, 142, 3, 30, "Input"],

Cell[CellGroupData[{
Cell[26919, 994, 20761, 1019, 153, 13602, 925, "GraphicsData", "PostScript", \
"Graphics"],
Cell[47683, 2015, 20222, 992, 152, 13216, 900, "GraphicsData", "PostScript", \
"Graphics"],
Cell[67908, 3009, 19912, 977, 152, 12974, 885, "GraphicsData", "PostScript", \
"Graphics"],
Cell[87823, 3988, 19492, 952, 152, 12599, 861, "GraphicsData", "PostScript", \
"Graphics"],
Cell[107318, 4942, 18666, 902, 152, 11842, 812, "GraphicsData", "PostScript", \
"Graphics"],
Cell[125987, 5846, 17015, 797, 152, 10187, 707, "GraphicsData", "PostScript", \
"Graphics"],
Cell[143005, 6645, 15669, 711, 152, 8849, 621, "GraphicsData", "PostScript", \
"Graphics"],
Cell[158677, 7358, 15089, 674, 152, 8236, 583, "GraphicsData", "PostScript", \
"Graphics"],
Cell[173769, 8034, 14613, 642, 152, 7748, 551, "GraphicsData", "PostScript", \
"Graphics"],
Cell[188385, 8678, 14894, 656, 152, 7977, 565, "GraphicsData", "PostScript", \
"Graphics"],
Cell[203282, 9336, 15721, 708, 152, 8771, 616, "GraphicsData", "PostScript", \
"Graphics"],
Cell[219006, 10046, 16307, 742, 152, 9276, 649, "GraphicsData", "PostScript", \
"Graphics"],
Cell[235316, 10790, 17926, 841, 152, 10818, 747, "GraphicsData", \
"PostScript", "Graphics"],
Cell[253245, 11633, 19241, 920, 152, 12056, 825, "GraphicsData", \
"PostScript", "Graphics"],
Cell[272489, 12555, 20294, 978, 152, 12899, 881, "GraphicsData", \
"PostScript", "Graphics"],
Cell[292786, 13535, 20527, 985, 152, 12978, 886, "GraphicsData", \
"PostScript", "Graphics"],
Cell[313316, 14522, 20967, 1004, 152, 13272, 903, "GraphicsData", \
"PostScript", "Graphics"]
}, Closed]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

