function dimaway = VecDimAway(mvec1,mvec2,inorm) 
% VecDimAway, VECtor DIMensions AWAY
%     Measures how distance (i.e. angles) between direction vectors
%     Finds the dimensions away either:
%           Between all pairs of the column vectors of mvec1
%                             or:
%           Between the column vectors of mvec1, 
%           and the corresponding columns of mvec2,
%
%     "Dimensions Away", between UNIT VECTORS u1 and u2,
%      is defined to be:
%           d * (1 - cosang) / 2 ;
%      where cosang is the cosine of the angle between, which
%      is just the inner product, u1' * u2.
%      This scale is interpreted in terms of the "Unit Hypercube"
%      whose vertices are vectors whose entries are all +-1.
%     "Dimensions Away" between two such vertices is the number 
%      of entries that are different between the two vertex vectors,
%      hence the name.  "Dimensions Away" = d/2 corresponds to 
%      orthogonality.  "Dimensions Away" = d corresponds to 
%      antipodal.  "Dimensions Away" = 0 corresponds to equal.
%   
%   Can use first 1, 2 or 3 arguments.
% Inputs:
%     mvec1  -  d x n (or 1) matrix of d-dim column vectors
%     mvec2  -  d x n (or 1) matrix of d-dim column vectors
%                     (when not specified, returns matrix of
%                      pairwise distances between cols of mvec1)
%     inorm  -  normalization type:
%                    0 - assume all vectors are unit vectors,
%                             don't need to normalize
%                    1 - need to normalize to get unit vectors
%
% Output:
%     dimaway     -  row vector (or matrix when only mvec1 is given)
%                        of dimensions away
%
% Assumes path can find personal function:
%    vec2mat

%    Copyright (c) J. S. Marron 1999





if nargin == 1 ;    

  calctype = 1 ;
          %  flag for type of calculation: 
          %      all pairwise dim's away of mvec1

  iinorm = 2 ;

end ;



if nargin == 2 ;

  if size(mvec2,1) == 1  &  size(mvec2,2) == 1 ;
                         %  then second entry is norm type

    calctype = 1 ;
          %  flag for type of calculation: 
          %      all pairwise dim's away of mvec1

    iinorm = mvec2 ;

  else ;

    calctype = 2 ;
          %  flag for type of calculation: 
          %      dim's away between mvec1 and mvec2

    iinorm = 2 ;

  end ;

end ;



if nargin == 3 ;

  calctype = 2 ;
          %  flag for type of calculation: 
          %      dim's away between mvec1 and mvec2

  iinorm = inorm ;

end ;




if calctype == 1 ;    %  then do all pairwise dist's of mvec1

  d = size(mvec1,1) ;
  n = size(mvec1,2) ;


  if n == 1 ;   %  then no work to do


    dimaway = 0 ;


  else ;    %  then do calculations


    if iinorm ~= 0 ;    %  then need to normalize

      lmvec1 = sqrt(sum(mvec1.^2,1)) ;
          %  length of each vector in mvec1 (row vector)

      if d == 1 ;
        munit1 = mvec1 / lmvec1 ;
      else ;
        munit1 = mvec1 ./ vec2mat(lmvec1,d) ;
      end ;

    else ;    %  assuming vectors already have length 1

      munit1 = mvec1 ;

    end ;



    %  do main dimensions away calculation
    %
    mcosang = munit1' * munit1 ;
          %  matrix of inner products,
          %  i.e. cosines of angles
    dimaway = d * (1 - mcosang) / 2 ;
          %  put on scale of number of dimensions away
    


  end ;  



elseif calctype == 2 ;    %  then do dim's away 
                          %          between mvec1 and mvec2

  d = size(mvec1,1) ;
  n1 = size(mvec1,2) ;
  n2 = size(mvec2,2) ;

  if size(mvec2,1) == d ;    %  then OK, go ahead

    if n1 == 1  & n2 > 1 ;    %  then expand mvec1 to n2
      mvec1 = vec2mat(mvec1,n2) ;
      n1 = n2 ;
    end ;

    if n2 == 1  & n1 > 1 ;    %  then expand mvec2 to n1
      mvec2 = vec2mat(mvec2,n1) ;
      n2 = n1 ;
    end ;


    if n1 == n2 ;     %  then OK, go ahead


      if iinorm ~= 0 ;    %  then need to normalize

        lmvec1 = sqrt(sum(mvec1.^2,1)) ;
          %  length of each vector in mvec1 (row vector)
        lmvec2 = sqrt(sum(mvec2.^2,1)) ;
          %  length of each vector in mvec2 (row vector)

        if d == 1 ;
          munit1 = mvec1 / lmvec1 ;
          munit2 = mvec2 / lmvec2 ;
        else ;
          munit1 = mvec1 ./ vec2mat(lmvec1,d) ;
          munit2 = mvec2 ./ vec2mat(lmvec2,d) ;
        end ;

      else ;    %  assuming vectors already have length 1

        munit1 = mvec1 ;
        munit2 = mvec2 ;

      end ;



      %  do main dimensions away calculation
      %
      mcosang = sum(munit1 .* munit2,1) ;
          %  vector of inner products,
          %  i.e. cosines of angles
      dimaway = d * (1 - mcosang) / 2 ;
          %  put on scale of number of dimensions away
    

    else ;   %  No good, give error message

      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp('!!!   Error from VecDist:              !!!') ;
      disp('!!!   Inputs need same number of cols  !!!') ;
      disp('!!!   Giving Empty Return              !!!') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

      dimaway = [] ;

    end ;


  else ;   %  then give error message

    disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
    disp('!!!   Error from VecDist:              !!!') ;
    disp('!!!   Inputs need same number of rows  !!!') ;
    disp('!!!   Giving Empty Return              !!!') ;
    disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

    dimaway = [] ;

  end ;

end ;
