function qqout = qq(data,paramstruct) 
% QQ, Quantile - Quantile plot for testing distributional form
%      against standard distributions, currently including
%      Gaussian (normal), log normal, Pareto, Pareto-log (still
%      Pareto, but viewed on scale of log data), Weibull, 
%      Weibull-log
% Inputs:
%   data        - n x 1 column vector of data
%   paramstruct - a Matlab structure of input parameters
%                    Use: "help struct" and "help datatypes" to
%                         learn about these.
%                    Create one, using commands of the form:
%
%       paramstruct = struct('field1',values1,...
%                            'field2',values2,...
%                                             ) ;
%
%                          where any of the following can be used,
%                          these are optional, misspecified values
%                          revert to defaults
%
%    fields            values
%
%    idist            1  (default)  Gaussian
%                     2  Pareto
%                     3  Weibull
%                     11 Lognormal (Classical distn: exp(N(mu,sig2)),
%                                      view on log-log scale)
%                     12 Logpareto (really Pareto, but log-log view)
%                     13 Logweibull (really Weibull, but log-log view)
%
%            Associated Parameters (will estimate when not specified)
%
%            1.  Gaussian:
%    mu               Mean 
%    sigma            Standard Deviation
%
%            2.  Pareto (moved to left, "to start at 0")
%                       (mean is (alpha * sigma) / (alpha - 1)  -  1):
%    alpha            Classical Tail Index
%    sigma            Scale Parameter
%
%            3.  Weibull (alpha is power of exponent)
%                        (mean is sigma * gamma(1/alpha + 1)):
%    alpha            Shape Parameter
%    sigma            Scale Parameter
%
%
%    vqalign          vector of percentages for 2 quantiles to align
%                     (default:  [.5; .75], except for Gaussian
%                             where sample mean and sd are used)
%
%    nsim             number of psuedo data sets to simulate,
%                              to display random variability
%                     (default = 100)
%                     0  no simulation, only straight QQ plot
%
%    simseed          seed for simulation
%
%    nsimplotval      number of values to use in display of
%                     simulated versions to assess variability
%                     (useful to speed graphics with large
%                      sample sizes, uses nsimplotval/3 values
%                      at each end, and the rest equally spaced
%                      in the middle, ignored when ndata is
%                      smaller or when nsim = 0)
%                     0  use default nsimplotval = 900
%                         (tests suggest:
%                              3000  "no visual difference"
%                              900   "small visual difference"
%                              300   "larger visual difference"
%
%    icolor           1  (default)  full color version (for talks)
%                     0  fully black and white version (for papers)
%
%    savestr          string controlling saving of output,
%                         either a full path, or a file prefix to
%                         save in matlab's current directory,
%                         will add .ps, and save as either
%                              color postscript (icolor = 1)
%                         or
%                              black&white postscript (when icolor = 0)
%                     unspecified:  results only appear on screen
%
%    xlabelstr        String for labeling x axes 
%                                  (default is Dist. Name + Q')
%
%    ylabelstr        String for labeling y axes 
%                                  (default is 'Data Q')
%
%    titlestr         Title (default is 'Q-Q Plot')
%
%    titlefontsize    Font Size for title (uses Matlab default)
%                                   (18 is "fairly large")
%
%    ishowpar         0  (default) don't show parameters
%                     1  show parameters
%
%    parfontsize      Font Size for parameter text (uses Matlab default)
%                                   (18 is "fairly large", consider 15)
%
%    left             left end of plot range (default is min of data)
%
%    right            right end of plot range (default is max of data)
% 
%    bottom           bottom of plot range (default is min of data)
%
%    top              top of plot range (default is max of data)
% 
%    ioverlay         0  no overlay line
%                     1  (default) overlay 45 degree line
%                     2  overlay least squares fit line
%
%    iscreenwrite     0  (default)  don't write progress to screen
%                     1  write messages to screen to show progress
%
%    maxstep          maximum number of steps for Pareto Estimation
%                     (default = 1000) 
%
%    relaccthreshold  Relative Accuracy Threshold for
%                     Pareto Estimation iterations
%                     (default = 10^(-6))
%
%
% Outputs:
%     For nargout = 1:  2 column vector, 
%                            theoretical quantiles in 1st column
%                            empirical quantiles in 2nd column
%     For nargout = 0:  graphics in current axes
%     When savestr exists,   Postscript file saved in 'savestr'.ps
%                        (color postscript for icolor = 1)
%                        (B & W postscript for icolor = 0)

%    Copyright (c) J. S. Marron 2000



%  First set parameters to defaults
%
idist = 1 ;
vqalign =[.5; .75] ;
nsim = 100 ;
simseed = 34568235 ;
nsimplotval = 900 ;
icolor = 1 ;
savestr = [] ;
xlabelstr = ', Q' ;
ylabelstr = 'Data Q' ;
titlestr = 'Q-Q Plot' ;
titlefontsize = [] ;
ishowpar = 0 ;
parfontsize = [] ;
left = min(data) ;
right = max(data) ;
bottom = min(data) ;
top = max(data) ;
ioverlay = 1 ;
iscreenwrite = 0 ;
maxstep = 1000 ;
relaccthreshold = 10^(-6) ;




%  Now update parameters as specified,
%  by parameter structure (if it is used)
%
if nargin > 1 ;   %  then paramstruct has been added

  if isfield(paramstruct,'idist') ;    %  then change to input value
    idist = getfield(paramstruct,'idist') ; 
  end ;

  if isfield(paramstruct,'vqalign') ;    %  then change to input value
    vqalign = getfield(paramstruct,'vqalign') ; 
  end ;

  if isfield(paramstruct,'nsim') ;    %  then change to input value
    nsim = getfield(paramstruct,'nsim') ; 
  end ;

  if isfield(paramstruct,'simseed') ;    %  then change to input value
    simseed = getfield(paramstruct,'simseed') ; 
  end ;

  if isfield(paramstruct,'nsimplotval') ;    %  then change to input value
    nsimplotval = getfield(paramstruct,'nsimplotval') ; 
  end ;

  if isfield(paramstruct,'icolor') ;    %  then change to input value
    icolor = getfield(paramstruct,'icolor') ; 
  end ;

  if isfield(paramstruct,'savestr') ;    %  then change to input value
    savestr = getfield(paramstruct,'savestr') ; 
    if ~ischar(savestr) ;    %  then invalid input, so give warning
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp('!!!   Warning from qq.m:         !!!') ;
      disp('!!!   Invalid savestr,           !!!') ;
      disp('!!!   using default of no save   !!!') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      savestr = [] ;
    end ;
  end ;

  if isfield(paramstruct,'xlabelstr') ;    %  then change to input value
    xlabelstr = getfield(paramstruct,'xlabelstr') ; 
  end ;

  if isfield(paramstruct,'ylabelstr') ;    %  then change to input value
    ylabelstr = getfield(paramstruct,'ylabelstr') ; 
  end ;

  if isfield(paramstruct,'titlestr') ;    %  then change to input value
    titlestr = getfield(paramstruct,'titlestr') ; 
  end ;

  if isfield(paramstruct,'titlefontsize') ;    %  then change to input value
    titlefontsize = getfield(paramstruct,'titlefontsize') ; 
  end ;

  if isfield(paramstruct,'ishowpar') ;    %  then change to input value
    ishowpar = getfield(paramstruct,'ishowpar') ; 
  end ;

  if isfield(paramstruct,'parfontsize') ;    %  then change to input value
    parfontsize = getfield(paramstruct,'parfontsize') ; 
  end ;

  if isfield(paramstruct,'left') ;    %  then change to input value
    left = getfield(paramstruct,'left') ; 
  end ;

  if isfield(paramstruct,'right') ;    %  then change to input value
    right = getfield(paramstruct,'right') ; 
  end ;

  if isfield(paramstruct,'bottom') ;    %  then change to input value
    bottom = getfield(paramstruct,'bottom') ; 
  end ;

  if isfield(paramstruct,'top') ;    %  then change to input value
    top = getfield(paramstruct,'top') ; 
  end ;

  if isfield(paramstruct,'ioverlay') ;    %  then change to input value
    ioverlay = getfield(paramstruct,'ioverlay') ; 
  end ;

  if isfield(paramstruct,'iscreenwrite') ;    %  then change to input value
    iscreenwrite = getfield(paramstruct,'iscreenwrite') ; 
  end ;

  if isfield(paramstruct,'maxstep') ;    %  then change to input value
    maxstep = getfield(paramstruct,'maxstep') ; 
  end ;

  if isfield(paramstruct,'relaccthreshold') ;    %  then change to input value
    relaccthreshold = getfield(paramstruct,'relaccthreshold') ; 
  end ;


else ;   %  create a dummy structure
  paramstruct = struct('nothing',[]) ;


end ;  %  of resetting of input parameters





if  size(data,2) > 1  |  size(data,1) < 2 ;

  disp(' ') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp('!!!   Error from QQ.m:                    !!!') ;  
  disp('!!!   Input data must be a column vector  !!!') ;
  disp('!!!   Terminating Execution               !!!') ;  
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp(' ') ;

  return ;

end ;




%  Set internal parameters
%
if icolor == 1 ;
  vcolorstr = ['r'; 'g'; 'b'] ;
          %  1st - data
          %  2nd - line
          %  3rd - simulated versions
  ltypestr = '-' ;
else ;
  vcolorstr = ['w'; 'w'; 'w'] ;
  ltypestr = '--' ;
  nsim = min(nsim,10) ;
         %  can't see rest with more than 10
end ;


%  Get data quantiles
n = length(data) ;
qdata = sort(data) ;
          %  put in increasing order




%  Get theoretical quantiles
%
pgrid = (1:n)' / (n + 1) ;

if  idist == 1  |  idist == 11  ;  
  if idist == 1 ;
    diststr = 'Gaussian' ;
  elseif idist == 11 ;
    diststr = 'LogNormal' ;

    if qdata(1) <= 0 ;
      disp(' ') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp('!!!   Error from QQ.m:             !!!') ;  
      disp('!!!   Input data must be positive  !!!') ;
      disp('!!!   Terminating Execution        !!!') ;  
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp(' ') ;

      return ;
    end ;

    qdata = log(qdata) ;
  end ;

  if isfield(paramstruct,'mu') ;    %  then change to input value
    mu = getfield(paramstruct,'mu') ; 
  else ;    %  estimate from data
    mu = mean(qdata) ;
  end ;

  if isfield(paramstruct,'sigma') ;    %  then change to input value
    sigma = getfield(paramstruct,'sigma') ; 
  else ;    %  use MLE from data
    sigma = std(qdata) ;
  end ;

  if isfield(paramstruct,'vqalign') ;    %  then do quantile matching
    p1 = vqalign(1) ;
    p2 = vqalign(2) ;
    vq = cquant(qdata,vqalign,0) ;
         %  0 to indicate presorted data
    q1hat = vq(1) ;
    q2hat = vq(2) ;

    sigma = (q1hat - q2hat) / (phiinv(p1) - phiinv(p2)) ;
    mu = q1hat - sigma * phiinv(p1) ;

  end ;


  qtheory = mu + sigma * phiinv(pgrid) ;



  par1str = '\mu' ;
  par1val = mu ;
  par2str = '\sigma' ;
  par2val = sigma ;


elseif  idist == 2  |  idist == 12  ;  
  if idist == 2 ;
    diststr = 'Pareto' ;
  elseif idist == 12 ;
    diststr = 'Pareto-log' ;
  end ;

  if  isfield(paramstruct,'alpha')  & ...
          isfield(paramstruct,'sigma')  ;    %  then change to input value
    alpha = getfield(paramstruct,'alpha') ; 
    sigma = getfield(paramstruct,'sigma') ; 
  else ;    %  estimate from data
    if qdata(1) <= 0 ;
      disp(' ') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp('!!!   Error from QQ.m:             !!!') ;  
      disp('!!!   Input data must be positive  !!!') ;
      disp('!!!   Terminating Execution        !!!') ;  
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp(' ') ;

      return ;
    end ;

    p1 = vqalign(1) ;
    p2 = vqalign(2) ;
    vq = cquant(qdata,vqalign,0) ;
         %  0 to indicate presorted data
    q1hat = vq(1) ;
    q2hat = vq(2) ;

    numera = log(1 - p1) - log(1 - p2);
    oldsigma = 0 ;
          %  common and starting values

    for istep = 1:maxstep ;

      q1z = q1hat + oldsigma ;
      q2z = q2hat + oldsigma ;

      alpha = numera / (log(q2z) - log(q1z)) ;
      sigma = q1z * (1 - p1)^(1/alpha) ;

      relacc = abs(sigma - oldsigma) / sigma ;

      if  iscreenwrite == 1  &  floor(istep/50) == istep/50 ;
        disp(['     For iteration step ' num2str(istep) ...
                    ', relacc = ' num2str(relacc) ...
                    ', alpha = ' num2str(alpha) ...
                    ', sigma = ' num2str(sigma)]) ;
       end;

      
      if relacc < relaccthreshold ;
        errflag = 0 ;
        if  iscreenwrite == 1  ;
          disp(['     For iteration step ' num2str(istep) ...
                      ', relacc = ' num2str(relacc) ...
                      ', alpha = ' num2str(alpha) ...
                      ', sigma = ' num2str(sigma)]) ;
        end;
        break ;
      else ;
        errflag = 1 ;
        oldsigma = sigma ;
      end ;

    end ;

    if errflag ~= 0 ;
      disp('!!!   Warning from QQ.m:  Pareto fit may be unstable   !!!') ;
    end ;

  end ;


  qtheory = ((1 - pgrid).^(-1/alpha) - 1) * sigma ;



  if idist == 12 ;    %  then need to fix scales
    qdata = log(qdata) ;
    qtheory = log(qtheory) ;
  end ;


  par1str = '\alpha' ;
  par1val = alpha ;
  par2str = '\sigma' ;
  par2val = sigma ;


elseif  idist == 3  |  idist == 13  ;
  if idist == 3 ;
    diststr = 'Weibull' ;
  elseif idist == 13 ;
    diststr = 'Weibull-log' ;
  end ;

  if  isfield(paramstruct,'alpha')  & ...
          isfield(paramstruct,'sigma')  ;    %  then change to input value
    alpha = getfield(paramstruct,'alpha') ; 
    sigma = getfield(paramstruct,'sigma') ; 
  else ;    %  estimate from data
    if qdata(1) <= 0 ;
      disp(' ') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp('!!!   Error from QQ.m:             !!!') ;  
      disp('!!!   Input data must be positive  !!!') ;
      disp('!!!   Terminating Execution        !!!') ;  
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp(' ') ;

      return ;
    end ;

    p1 = vqalign(1) ;
    p2 = vqalign(2) ;
    vq = cquant(qdata,vqalign,0) ;
         %  0 to indicate presorted data
    q1hat = vq(1) ;
    q2hat = vq(2) ;

    alpha = (log(-log(1 - p1)) - log(-log(1 - p2))) / ...
                      (log(q1hat) - log(q2hat)) ;
    sigma = exp(log(q1hat) - log(-log(1 - p1)) / alpha) ;

  end ;


  qtheory = sigma * (-log(1 - pgrid)).^(1/alpha) ;



  if idist == 13 ;    %  then need to fix scales
    qdata = log(qdata) ;
    qtheory = log(qtheory) ;
  end ;

  par1str = '\alpha' ;
  par1val = alpha ;
  par2str = '\sigma' ;
  par2val = sigma ;


end ;    %  of idist if block






if iscreenwrite == 1 ;
  disp(['   Comparing with ' diststr ' distribution']) ;
end;


if strcmp(xlabelstr,', Q') ;    %  default value
  xlabelstr = [diststr, xlabelstr] ;
end ;


if strcmp(ylabelstr,'Data Q') ;    %  default value
  if idist > 10 ;    %  working on log scale
    ylabelstr = ['log ' ylabelstr] ;
  end ;
end ;


if idist > 10 ;

  if bottom == min(data) ;
    bottom = log(bottom) ;
  end ;

  if top == max(data) ;
    top = log(top) ;
  end ;

end ;


if ~isfield(paramstruct,'left') ;    %  then change to theoretical left
  left = min(qtheory) ; 
end ;

if ~isfield(paramstruct,'right') ;    %  then change to theoretical right
  right = max(qtheory) ; 
end ;






%  Make main graphic in current axes
%
plot(qtheory,qdata, ...
    [vcolorstr(1) '-'], ...
    'LineWidth',3) ;
  axis equal ;
  axis([left,right,bottom,top]) ;

  th = title(titlestr) ;
  if ~isempty(titlefontsize) ;
    set(th,'FontSize',titlefontsize) ;
  end ;

  xlabel(xlabelstr) ;    
  ylabel(ylabelstr) ;    




%  Add lines (if needed)
%
if ioverlay == 1 ;    %  then overlay 45 degree line
  minmin = min(left,bottom) ;
  maxmax = max(right,top) ;
  hold on ;
    plot([minmin,maxmax],[minmin,maxmax], ...
         [vcolorstr(2) ltypestr], ...
         'LineWidth',2) ;
  hold off ;

elseif ioverlay == 2 ;    %  then overlay least squares fit line
  minmin = min(left,bottom) ;
  maxmax = max(right,top) ;
  lincoeffs = polyfit(qtheory,qdata,1) ;
          %  coefficients of simple least squares fit
  lineval = polyval(lincoeffs,[minmin;maxmax]) ;
  hold on ;
    plot([minmin;maxmax],lineval, ...
         [vcolorstr(2) ltypestr], ...
         'LineWidth',2) ;
  hold off ; 

end ;




%  Add simulated realizations (if needed)
%
if nsim > 0 ;

  if iscreenwrite == 1 ;
    disp('      generating simulated data') ;
  end ;
  if  idist == 1  |  idist == 11  ;    %  Gaussian

    randn('seed',simseed) ;
          %  set seed
    msimdat = mu + sigma * randn(n,nsim) ;


  elseif  idist == 2  |  idist == 12  ;    %  Pareto

    rand('seed',simseed) ;
          %  set seed
    msimdat = sigma * ((1 - rand(n,nsim)).^(-1/alpha) - 1) ;

    if idist == 12 ;    %  then need to fix scales
      msimdat = log(msimdat) ;
    end ;


  elseif  idist == 3  |  idist == 13  ;    %  Weibull

    rand('seed',simseed) ;
          %  set seed
    msimdat = sigma * (-log(1 - rand(n,nsim))).^(1 / alpha) ;

    if idist == 13 ;    %  then need to fix scales
      msimdat = log(msimdat) ;
    end ;


  end ;    %  of idist block



  if iscreenwrite == 1 ;
    disp('      sorting simulated data') ;
  end ;
  msimdat = sort(msimdat) ;
          %  sort each column


  if nsimplotval < n ;    %  then get reduced version for 
                          %  efficient plotting

    nspvo3 = floor(nsimplotval / 3) ;
          %  one third of total, to put at each end
    vindleft = (1:nspvo3)' ;
          %  left end  (include all)
    vindright = ((n-nspvo3+1):n)' ;
          %  right end  (include all)

    nspvlo = nsimplotval - length(vindleft) ...
                         - length(vindright) ;
          %  number of grid points left over (for use in middle)
    vindmid = round(linspace(nspvo3+1,n-nspvo3,nspvlo)') ;

    vind = [vindleft; vindmid; vindright] ;
          %  vector of indices, full

    qtheoryp = qtheory(vind,:) ;
    msimdatp = msimdat(vind,:) ;

  else ;  

    qtheoryp = qtheory ;
    msimdatp = msimdat ;

  end ;


  hold on  ;
    plot(qtheoryp,msimdatp,[vcolorstr(3) '-']) ;

    %  replot important stuff
    %
    plot(qtheory,qdata, ...
        [vcolorstr(1) '-'], ...
        'LineWidth',3) ;
    if ioverlay == 1 ;    %  then overlay 45 degree line
      plot([minmin,maxmax],[minmin,maxmax], ...
           [vcolorstr(2) ltypestr], ...
           'LineWidth',2) ;
    elseif ioverlay == 2 ;    %  then overlay least squares fit line
      plot([minmin;maxmax],lineval, ...
           [vcolorstr(2) ltypestr], ...
           'LineWidth',2) ;
    end ;


  hold off ;

end ;    %  of simulated data if block




%  Add text for parameters (if needed)
%
if ishowpar == 1 ;
    tx = left + 0.6 * (right - left) ;
    ty = bottom + 0.25 * (top - bottom) ;
  th = text(tx,ty,[par1str ' = ' num2str(par1val)]) ;
  if ~isempty(parfontsize) ;
    set(th,'FontSize',parfontsize) ;
  end ;

    ty = bottom + 0.1 * (top - bottom) ;
  th = text(tx,ty,[par2str ' = ' num2str(par2val)]) ;
  if ~isempty(parfontsize) ;
    set(th,'FontSize',parfontsize) ;
  end ;

end ;






%  Save results (if needed)
%
if ~isempty(savestr) ;     %  then save results

  if iscreenwrite == 1 ;
    disp('    QQ.m saving results') ;
  end ;



  orient landscape ;

  if icolor ~= 0 ;     %  then make color postscript
    eval(['print -dpsc ' savestr '.ps']) ;
  else ;                %  then make black and white
    eval(['print -dps ' savestr '.ps']) ;
  end ;



  if iscreenwrite == 1 ;
    disp('    QQ.m finished save') ;
    disp('  ') ;
  end ;

end ;



