function [vout1, vout2] = coamdf(indata, nring, nang, iendtype) ;
% COAMDF Missing Data Flag
%      Corneal Topography Analysis System
%   Creates a vector giving index of "last non-missing value"
%          and "first non-missing value" for each radial in 
%	   "spider web" of data matrix "indata".
%         Note: typically call this twice, once for "rdata",
%         and once for "cdata" or "hdata", depending on context,
%         then take minimum
%      Intended to be called by coazrec.m, coarrec.m
% Inputs:
%       indata - input data matrix, assumed to be 256 x 26 matrix
%                   (rows index angles, columns index radii)
%                   so the first two elements in Keratron files
%                   have already been removed,
%                   and so has an initial column of all zeros
%	nring -  number of radial rings in the dataset (typically 26).
%	nang  -  number of angles for each ring. (typically 256).
%       iendtype - 1 (or not given) return index before first non-missing
%                  2  return really last non-missing
% Output:
%	vout1 - 256x1 vector, which for each radial, gives the index of
%		   first non-missing index along that radial,
%		   in integer from 1,...,nring.
%       vout2 - 256x1 vector, which for each radial, gives index of
%                  last non-missing index along that radial,
%                  hence an integer from 1,...,26 (26 = no missings)
%                  (gives empty output when input size is wrong)

%    Copyright (c) J. S. Marron, N. Locantore 1998
%    Department of Statistics, University of North Carolina

if nargin == 3 ;    %  then iendtype not input, so use default
  iet = 1 ;
else ;              %  inendtype input, so use that
  iet = iendtype    
end ;

vout1 = nring * ones(nang, 1);
	% Initialize first output vector
mflag = (indata ~= 0) ;
	  %  ones in locations with data.
vflag = sum(mflag')' ;
          %  row sum of above version
vflag = (vflag > 0) ;
          %  one in rows where there is a missing
mflag = mflag(vflag,:) ;
          %  pluck off only rows with a missing
[temp,svout1] = max(mflag') ;
          %  first transpose, so radials are columns with nearest at top
          %  second find index of first one place with data
vout1(vflag) = svout1 ;
          %  replace values where there are some missings with index
          %  of last non-missing


%%%%%%%%%%%
%  Process data so that all missing data in the center rings is ignored.

vtemp = vout1 ;
nodata = find(vtemp == nring); 
vtemp(nodata) = (nring - 1) * ones(length(nodata), 1) ;
	%  In case there is no data on a row, kick a temporary marker
	%  back one for other missing value processing.
for iang = 1:nang ;
   
   indata(iang,1:vtemp(iang)) = ones(1, vtemp(iang));
	% change all inner zeros to ones so that next step 
	% will find only the outer missing values.
end;

%%%%%%%%%%%%

vout2 = nring * ones(nang, 1);
          %  Initialize output vector
mflag = (indata == 0) ;
          %  ones in locations with a missing value
vflag = sum(mflag')' ;
          %  row sum of above version
vflag = (vflag > 0) ;
          %  one in rows where there is a missing
mflag = mflag(vflag,:) ;
          %  pluck off only rows with a missing

if iet == 1 ;        %  get index before first non-missing
  [temp,svout2] = max(mflag') ;
          %  first transpose, so radials are columns with nearest at top
          %  second find index of first one
  svout2 = svout2' - 1 ;
          %  turn into column vector, and get index of 0 before first 1
elseif iet == 2 ;    %  get really last non-missing
  [temp,svout2] = min(rot90(mflag)) ;
          %  first rotate 90 degrees counterclockwise, so radials
          %    are columns with farthest at top
          %  second find index of first 0, coming down column
  svout2 = nring + 1 - svout2' ;
          %  turn into column vector, and adjust to right end
end ;

vout2(vflag) = svout2 ;
          %  replace values where there are some missings with index
          %  of last non-missing

