function coaspc2(iarg)
% COASPCA, Does Static Principal Component Analysis for 2 Populations
%      Corneal Topography Analysis System
%   Creates a large size window with one page PCA
%      Intended to be called by coa2pm.m
%      Reads list of files from LISTFILESTR1 and LISTFILESTR2
% Input:
%     iarg - 1   for standard PCA 
%                  (eigen analysis of usual covariance matrix)
%     iarg - 2   for Intra Class Spread normalized PCA
%                  (eigen analysis of cov matrix, modulo intra class spread)
% Outputs:
%     Only graphics, in current Figure

%    Copyright (c) J. S. Marron 1998
%    Department of Statistics, University of North Carolina




%  Set local parameters
%
ngrid = 64 ;     %  Construct surfaces as ngrid x ngrid array
    


%  Set up global variables
%
%  First some general globals
global R0 NMAX COMAP ;
%  R0 - radius of analysis
          %    this gets reloaded when a feature vec is reloaded from file
%  NMAX - maximum number of Zernike columns
          %    this gets reloaded when a feature vec is reloaded from file
%  COMAP - Color Map
global SPCAHANDPOS SPC2HAND FEATMPATH LISTR ;
        %  window positioning vector,
        %  handle for Static PCA Window
        %  path for storing Zernike feature matrices
        %  string for inputting list file
global LISTFILESTR1 LISTFILESTR2 SETLIST1PATH SETLIST2PATH ;
        %  list file strings
        %  strings to set paths to lists
global PC2STR PC2LOAD1 PC2LOAD2 ;
        %  PC2STR - String with title info for 2 population analyses
        %  PC2LOAD1 - 1st Population Loadings for 2 population analyses
        %  PC2LOAD2 - 2nd Population Loadings for 2 population analyses



%  Read in matrix of Zernike feature coefficients, Population 1
%
disp('      Reading Files for Population 1') ;
eval(SETLIST1PATH) ;
          %  sets LISTR to Full Path List, for Population 1
mfv1 = coaflr ;
d1 = size(mfv1,1) ;
n1 = size(mfv1,2) ;



%  Read in matrix of Zernike feature coefficients, Population 2
%
disp('      Reading Files for Population 2') ;
eval(SETLIST2PATH) ;
          %  sets LISTR to Full Path List, for Population 2
mfv2 = coaflr ;
d2 = size(mfv2,1) ;
n2 = size(mfv2,2) ;




%  Calculate grid quantities
%
xgrid = linspace(-R0, R0, ngrid) ;
ygrid = xgrid ;	        %  Vectors of image coordinate values 
[mxgrid, mygrid] = meshgrid(xgrid,ygrid) ;
              		%  Matrices of image coordinate values 





%  Combine into one big data set
if d1 == d2 ;
  d = d1 ;
else ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp('!!!   Error from coaspc2.m:          !!!') ;
  disp('!!!   Different Number of Zernikes   !!!') ;
  disp('!!!      for these 2 lists           !!!') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
end ;
mfv = [mfv1, mfv2] ;
n = n1 + n2 ;




%  Do Principal Components (combined data set)
%
sst = sum(sum(mfv.^2)) ;
%  get mean feature vector, and resids from that
vmean = mean(mfv')' ;
vsd = std(mfv')' ;
ssm = n * sum(vmean.^2) ;
ssmr = sst - ssm ;
mresid = mfv - vec2mat(vmean,n) ;

%  Get cov's and eigen values
mcov = cov(mresid') ;
          %  Get covariance matrix, transpose, since want "coordinates
          %                      as variables"


if iarg == 1 ;
  PC2STR = 'PCA' ;
elseif iarg == 2 ;    % then normalize by "within class cov"
  PC2STR = 'ICS' ;

  vmean1 = mean(mfv1')' ;
  mresid1 = mfv1 - vec2mat(vmean1,n1) ;
  mcov1 = cov(mresid1') ;

  vmean2 = mean(mfv2')' ;
  mresid2 = mfv2 - vec2mat(vmean2,n2) ;
  mcov2 = cov(mresid2') ;
  
  mcovw = (n1 / n) * mcov1 + (n2 / n) * mcov2 ;
            %  Average of "within class covariances"

wfact = max(max(mcovw)) / n ;
mcovw = mcovw + wfact * eye(size(mcovw)) ;
          %  kludge to avoid numerical problems

  mcovw = sqrtm(mcovw) ;

  mcov = (mcovw \ mcov) / mcovw ;
          %  Multiply by inverse square root on each side
          %  to preserve nonegative definiteness for eigenvalues

          %  Caution: this will usually produce warning messages which 
          %  "should" be no problem
%  disp('        (hopefully above warning messages can be ignored)') ;

end ;

[meigvec,meigval] = eig(mcov) ;
veigval = (diag(meigval)) ;
          %  Get eigenvalues and eigenvectors


[veigval,vind] = sort(veigval) ;
meigvec = meigvec(:,vind) ;
          %  Sort to avoid possible ordering problems
  
ev1 = meigvec(:,d) ;
          %  eigenvector with largest eigenvalue
pc1 = mresid' * ev1 ;
          %  vector of projections of data onto ev1
pc1rep = max(abs(pc1 - mean(pc1))) ;
          %  "most representative pc", defined as MAD
mpc1 = ev1 * pc1rep ;
          %  "average vector" for display
mresid1 = (mresid - ev1 * pc1') ;
ss1r = sum(sum(mresid1.^2)) ;
ss1 = ssmr - ss1r ;

ev2 = meigvec(:,d-1) ;
          %  eigenvector with largest eigenvalue
pc2 = mresid1' * ev2 ;
          %  vector of projections of data onto ev2
pc2rep = max(abs(pc2 - mean(pc2))) ;
          %  "most representative pc", defined as MAD
mpc2 = ev2 * pc2rep ;
          %  "average vector" for display
mresid2 = (mresid1 - ev2 * pc2') ;
ss2r = sum(sum(mresid2.^2)) ;
ss2 = ss1r - ss2r ;

ev3 = meigvec(:,d-2) ;
          %  eigenvector with largest eigenvalue
pc3 = mresid2' * ev3 ;
          %  vector of projections of data onto ev3
pc3rep = max(abs(pc3 - mean(pc3))) ;
          %  "most representative pc", defined as MAD
mpc3 = ev3 * pc3rep ;
          %  "average vector" for display
mresid3 = (mresid2 - ev3 * pc3') ;
ss3r = sum(sum(mresid3.^2)) ;
ss3 = ss2r - ss3r ;

ev4 = meigvec(:,d-3) ;
          %  eigenvector with largest eigenvalue
pc4 = mresid3' * ev4 ;
          %  vector of projections of data onto ev4
pc4rep = max(abs(pc4 - mean(pc4))) ;
          %  "most representative pc", defined as MAD
mpc4 = ev4 * pc4rep ;
          %  "average vector" for display
mresid4 = (mresid3 - ev4 * pc4') ;
ss4r = sum(sum(mresid4.^2)) ;
ss4 = ss3r - ss4r ;



%  Form mean +- representative eigenvectors
mpc1p = vmean + mpc1 ;
mpc1m = vmean - mpc1 ;
mpc2p = vmean + mpc2 ;
mpc2m = vmean - mpc2 ;
mpc3p = vmean + mpc3 ;
mpc3m = vmean - mpc3 ;
mpc4p = vmean + mpc4 ;
mpc4m = vmean - mpc4 ;




%  Create images
%
imm = coazr(vmean) ;
impc1p = coazr(mpc1p) ;
impc1m = coazr(mpc1m) ;
impc2p = coazr(mpc2p) ;
impc2m = coazr(mpc2m) ;
impc3p = coazr(mpc3p) ;
impc3m = coazr(mpc3m) ;
impc4p = coazr(mpc4p) ;
impc4m = coazr(mpc4m) ;




%  Set up window page
%
SPC2HAND = figure ;
          %  create a new window
set(SPC2HAND, 'NumberTitle', 'off');
set(SPC2HAND,'Position', SPCAHANDPOS) ;
set(SPC2HAND,'Name',['Static ' PC2STR]) ;


%  Put down plots in 4 x 4 array
%
%  Means first
%
for i = 1:4 ;
  subplot(4,4,4+i) ;
  pcolor(mxgrid,mygrid,imm) ;
    shading flat ;
    axis('square') ;
    title(['mean, ' LISTFILESTR1 ', ' LISTFILESTR2]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;
  hold on ;
    tx = -.95 * R0 ;
    ty = .7 * R0 ;
    eval(['pcss = 100 * ss' num2str(i) ' / ssmr ;']) ;
    ssstr = num2str(pcss) ;
    tstr = [ssstr '% of MR'] ;
    text(tx,ty,tstr) ;
  hold off ;
end ;


%  Now mean +- Pc's
%
for i = 1:4 ;
  subplot(4,4,i) ;
  eval(['pcolor(mxgrid,mygrid,impc' num2str(i) 'p)']) ;
    shading flat ;
    axis('square') ;
    title(['mean + ' PC2STR num2str(i)]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;

  subplot(4,4,8+i) ;
  eval(['pcolor(mxgrid,mygrid,impc' num2str(i) 'm)']) ;
    shading flat ;
    axis('square') ;
    title(['mean - ' PC2STR num2str(i)]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;
end ;


%  Now do kde's
%
for i = 1:4 ;
  subplot(4,4,12+i) ;
  eval(['pccur = pc' num2str(i) ';']) ;
          %  Matrix of current PC loadings
  pccur1 = pccur(1:n1) ;
          %  Current PC Loadings, for Population 1
  pccur2 = pccur(n1+1:n) ;
          %  Current PC Loadings, for Population 2
  [kde1, xgrid1] = gpkde(pccur1) ;
  [kde2, xgrid2] = gpkde(pccur2) ;
  plot(xgrid1,kde1,'r',xgrid2,kde2,'c') ;
    title('Projections') ;
    vlim = axis ;
          %  Get axis values for drawing vertical lines
    eval(['left = -pc' num2str(i) 'rep ;']) ;
    eval(['right = pc' num2str(i) 'rep ;']) ;
    range = right - left ;
    left = left - .05 * range ;
    right = right + .05 * range ;
    bottom = vlim(3) ;
    top = vlim(4) ;
    axis([left right bottom top]) ;
    hold on ;
      plot([pccur2 pccur2],[bottom top], 'c') ;
      plot([pccur1 pccur1],[bottom top], 'r') ;
          %  Show locations of projections 
          %  (pop'n 2 first, since likely more spread)
      plot(xgrid1,kde1,'r',xgrid2,kde2,'c') ;
          %  Do a replot to ensure seeing kde
    hold off ;
end ;




%  Add a title
%
tx = 0.05 ;
ty = 0.94 ;
tw = 0.8 ;
th = 0.05 ;
          %  set position for new axes, containing only title
axes('Position',[tx,ty,tw,th]) ;
          %  create new axes
set(gca,'XTickLabelMode','manual') ;
set(gca,'XTickLabel',[]) ;
set(gca,'XTickMode','manual') ;
set(gca,'XTick',[]) ;
set(gca,'XColor',[0,0,0]) ;
set(gca,'YTickLabelMode','manual') ;
set(gca,'YTickLabel',[]) ;
set(gca,'YTickMode','manual') ;
set(gca,'YTick',[]) ;
set(gca,'YColor',[0,0,0]) ;
          %  Switch off axis stuff
set(gca,'visible','off') ;
          %  Turn off surrounding box
titstr = ['2 Population Static ' PC2STR ' for file lists: ' ...
               LISTFILESTR1 LISTFILESTR2] ;
titstr = [titstr ',  ' date] ;
titstr = [titstr '  NMAX = ' num2str(NMAX) '  r0 = ' num2str(R0)] ;
text(0,0.5,titstr) ;
          %  put down actual title



%  Save 2 population analysis Loadings
%
mload = [pc1 pc2 pc3 pc4] ;
PC2LOAD1 = mload(1:n1,:) ;
PC2LOAD2 = mload(n1+1:n,:) ;

