function mzc = coaflr
% COAFLR, Reads Zernike coefficient vectors from File List
%      Corneal Topography Analysis System
%   For each Zernike feature vector in the current file list
%      (in the file LISTFILESTR, whose full path is in the 
%      string LISTR), this reads the coefficients from the
%      stored file, and combines them into a large matrix.
%      Intended to be called by coaspca.m and coadpca.m
% Output:
%     mzc   -   matrix of Zernike Coefficients

%    Copyright (c) J. S. Marron 1998
%    Department of Statistics, University of North Carolina




%  Set up global variables
%
global R0 NMAX IBASIS IRECENT LISTR FEATMPATH SUBDIRSTR ;
%  R0 - radius of analysis
          %    this gets reloaded when a feature vec is reloaded from file
%  NMAX - maximum number of Zernike columns
          %    this gets reloaded when a feature vec is reloaded from file
%  IBASIS - 1 for full (triangle), 2 for radial (3 row)
%  IRECENT - To recenter (or not) decompositions
%  LISTR - string for inputting list file
%  FEATMPATH - path for storing Zernike feature matrices
%  SUBDIRSTR - current subdirectory (of both indata and outputs)




%  Initialize variables
%
r0last = R0 ;          %  Original radius of analysis
nmaxlast = [] ;        %  Original maximum number of Zernike columns
                       %       set to empty, so can tell when read
                       %       is first
ibasislast = IBASIS ;  %  Original basis type (full or radial)
irecentlast = IRECENT ;%  Original recentering (or not)
%mzc = [] ;           %  output matrix (now intialized further down)




%  Read in data
%
% LISTR, (read-in file string, set in advance in coapam.m)
fid = fopen(LISTR) ;
          %  assigns a file handle, (read only assumed)
if (fid == -1) ;      %  then file not found, so report

  disp('   ') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp('!!!   Can''t read Zernike Coeffs, because can''t find file:') ;
  disp(['!!!       ' LISTR]) ;
  disp('!!!') ;
  disp('!!!   Terminating Population Analysis') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

else ;


  %  Read in list of filenames
  %
  minfilestr = fscanf(fid,'%s',[8,inf]) ;
            %  formatted read, assumes single column
  fclose(fid) ;

  minfilestr = minfilestr' ;
          %  each row corresponds to one filename



  %  loop through filenames
  %
  nfile = size(minfilestr,1) ;
          %  number of files = rows of input matrix
  for ifile = 1:nfile ;
  
    infilestr = minfilestr(ifile,:) ;
          %  combine indiviual entries into file string
    zmistr = [FEATMPATH SUBDIRSTR infilestr '.mat'] ;
          %  full path and filename 

    fid = fopen(zmistr) ;
          %  assigns a file handle, (read only assumed)
    if (fid == -1) ;      %  then file not found, so report

      disp('   ') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
      disp(['!!!   Can''t find file:  ' zmistr]) ;
      disp('!!!   So will skip this feature vector') ;
      disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

    else ;      %  then proceed with read
      ifc = fclose(fid) ;    %  only needed to know file was there,
                             %  so now close

      eval(['load ' zmistr]) ;
          %  loads this Zernike feature matrix and parameter values


      if isempty(nmaxlast) ;  
                      %  then are at first read, so only check R0

        if (R0 == r0last) & (IBASIS == ibasislast) & ...
                                (IRECENT == irecentlast)  ;
                      %  then have same R0, IBASIS & IRECENT
                      %  values, so proceed

          mzc = betahat ;     %  first column of this matrix

        else ;      %  write appropriate warning message
                    %  and query about proceeding

          if (R0 ~= r0last) ;
            disp('   ') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
            disp('!!!   Warning from coaflr.m:  R0 has changed,') ;
            disp(['!!!      Old R0 = ' num2str(r0last) ',   New R0 = ' num2str(R0)]) ;
            disp('!!!   Hit Any Key to continue with new R0,') ;
            disp('!!!   Or hit (ctrl + C) (twice) to quit') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

            pause ;          
            mzc = betahat ;     %  first column of this matrix
          end ;

          if (IBASIS ~= ibasislast) ;
            disp('   ') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
            disp('!!!   Warning from coalwzr.m:  IBASIS has changed,') ;
            disp(['!!!      Old IBASIS = ' num2str(ibasislast) ...
                                 ',   New IBASIS = ' num2str(IBASIS)]) ;
            disp('!!!   Hit Any Key to continue with new IBASIS,') ;
            disp('!!!   Or hit (ctrl + C) (twice) to quit') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

            pause ;          
            mzc = betahat ;     %  first column of this matrix
          end ;

          if (IRECENT ~= irecentlast) ;
            disp('   ') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
            disp('!!!   Warning from coalwzr.m:  IRECENT has changed,') ;
            disp(['!!!      Old IRECENT = ' num2str(irecentlast) ...
                                 ',   New IRECENT = ' num2str(IRECENT)]) ;
            disp('!!!   Hit Any Key to continue with new IRECENT,') ;
            disp('!!!   Or hit (ctrl + C) (twice) to quit') ;
            disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

            pause ;          
            mzc = betahat ;     %  first column of this matrix
          end ;

        end ;


      else ;          %  then are at later reads, so check params

        if (R0 == r0last) & (NMAX == nmaxlast) & ...
                (IBASIS == ibasislast) & (IRECENT == irecentlast) ;
                        %  then have common parameter values, so proceed

          mzc = [mzc, betahat] ;

        else ;       %  parameter values have changed, so quit

          %  Write latest values to screen
          NMAX
          nmaxlast
          R0
          r0last
          IBASIS
          ibasislast
          IRECENT
          irecentlast

          disp('   ') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
          disp('!!!   Input Zernike Parameters have changed,') ;
          disp(['!!!           (differing file is ' infilestr ')']) ;
          disp('!!!') ;
          disp('!!!   Terminating Population Analysis') ;
          disp('!!!') ;
          disp('!!!           (Need to redo at least some of the') ;
          disp('!!!            Zernike Decompositions to fix this) ') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

          mzc = [] ;
            %  reset this to empty, to abort population analysis      
          break ;
            %  jump out of loop

        end ;

      end ;

      r0last = R0 ;
      nmaxlast = NMAX ;


    end ;

  end ;

end ;

