function coaspca(rmp,rcp) ;
% COASPCA, Does Static Principal Component Analysis
%      Corneal Topography Analysis System
%   Creates a large size window with one page PCA
%      Intended to be called by coapam.m
%      Reads list of files from LISTFILESTR
% Inputs:
%     rmp - Robust Mean Parameter:
%             0 - (or not specified) use the ordinary sample mean
%             1 - Use the Spherical L1 location estimate
%             2 - Use the Elliptical (rescaled by MADs) L1 location estimate
%     rcp - Robust Covariance Parameter:
%             0 - (or not specified) use the ordinary covariance matrix
%             1 - Use the Spherical covariance matrix
%             2 - Use the Elliptical (rescaled by MADs) covariance matrix
% Outputs:
%     Only graphics, in current Figure

%    Copyright (c) J. S. Marron 1998
%    Department of Statistics, University of North Carolina



%  Set parameters and defaults according to number of input arguments
%
if nargin == 0 ;    %  no arguments input, use default
  irmp = 0 ;      %  default mean robustness parameter
else ;              %  was specified, use that
  irmp = rmp ;
end ;

if nargin <= 1 ;    %  at most 1 argument input, use default
  ircp = 0 ;      %  default covarince robustness parameter
else ;              %  was specified, use that
  ircp = rcp ;
end ;




%  Set local parameters
%
ngrid = 64 ;     %  Construct surfaces as ngrid x ngrid array
    


%  Set up global variables
%
%  First some general globals
global R0 NMAX IBASIS IRECENT COMAP ;
%  R0 - radius of analysis
          %    this gets reloaded when a feature vec is reloaded from file
%  NMAX - maximum number of Zernike columns
          %    this gets reloaded when a feature vec is reloaded from file
%  IBASIS - 1 for full (triangle), 2 for radial (3 row)
%  IRECENT - To recenter (or not) decompositions
%  COMAP - Color Map

global SPCAHANDPOS SPCAHAND ;
        %  window positioning vector,
        %  handle for Static PCA Window

global SETLISTPATH LISTR FEATMPATH SUBDIRSTR LISTFILESTR ;
        %  SETLISTPATH - string to set path to lists
        %  LISTR - string for inputting list file
        %  FEATMPATH - path for storing Zernike feature matrices
        %  SUBDIRSTR - current subdirectory (of both indata and outputs)
        %  list file string


%  Read in matrix of Zernike feature coefficients
%
eval(SETLISTPATH) ;
% sets LISTR (read-in file string)
mfv = coaflr ;
d = size(mfv,1) ;
n = size(mfv,2) ;



%  Calculate grid quantities
%
xgrid = linspace(-R0, R0, ngrid) ;
ygrid = xgrid ;	        %  Vectors of image coordinate values 
[mxgrid, mygrid] = meshgrid(xgrid,ygrid) ;
              		%  Matrices of image coordinate values 



%  Main Principal Component Block
%
sst = sum(sum(mfv.^2)) ;

%  Get mean feature vector
%
if irmp == 0 ;        %  then do original non-robust sample mean
  vmean = mean(mfv')' ;
  meanstr = 'mean' ;
elseif irmp == 1 ;    %  then do spherical robust sample mean
  vmean = rmean(mfv')' ;
  meanstr = 'S. L1' ;
elseif irmp == 2 ;    %  then do elliptical robust sample mean
  vmad = gpmad(mfv',0)' ;
          %  column vector of MADs of each row
  smfv = mfv .* vec2mat(1 ./ vmad, n) ;
          %  make each coordinate have MAD 1
  vmean = rmean(smfv')' ;
  vmean = vmean .* vmad ;
  meanstr = 'E. L1' ;
end ;

%  Get residuals from mean
%
vsd = std(mfv')' ;
ssm = n * sum(vmean.^2) ;
ssmr = sst - ssm ;
mresid = mfv - vec2mat(vmean,n) ;

%  Get covariance matrix
%
if ircp == 0 ;        %  then do original non-robust covariance matrix
  mcov = cov(mresid') ;
          %  Get covariance matrix, transpose, since want "coordinates
          %                      as variables"
  covstr = 'PC' ;
elseif ircp == 1 ;    %  then do spherical robust sample mean
  vrad = sqrt(sum(mresid .^2)) ;
          %  Transpose, since want "coordinates as variables"
          %  vector of radii of each curve
  sphereresid = mresid' ./ vec2mat(vrad',d) ;
          %  make each curve have "length" one
  mcov = cov(sphereresid) ;
          %  Get covariance matrix of "sphered" data
  covstr = 'SPC' ;
elseif ircp == 2 ;    %  then do elliptical robust sample mean
  vmad = gpmad(mresid',0)' ;
          %  column vector of MADs of each row
  smresid = mresid .* vec2mat(1 ./ vmad, n) ;
          %  make each coordinate have MAD 1
  vrad = sqrt(sum(smresid .^2)) ;
  sphereresid = smresid' ./ vec2mat(vrad',d) ;
          %  Transpose, since want "coordinates as variables"
          %  vector of radii of each curve
          %  make each curve have "length" one
  ellipsresid = sphereresid .* vec2mat(vmad', n) ;
          %  return each coordinate to original scale
  mcov = cov(ellipsresid) ;
          %  Get covariance matrix of "ellipsed" data
  covstr = 'EPC' ;
end ;

%  Do eigen analysis
%
[meigvec,meigval] = eig(mcov) ;
veigval = (diag(meigval)) ;
          %  Get eigenvalues and eigenvectors

[veigval,vind] = sort(veigval) ;
meigvec = meigvec(:,vind) ;
          %  Sort to avoid possible ordering problems
  
ev1 = meigvec(:,d) ;
          %  eigenvector with largest eigenvalue
pc1 = mresid' * ev1 ;
          %  vector of projections of data onto ev1
pc1rep = max(abs(pc1 - mean(pc1))) ;
          %  "most representative pc", defined as MAD
mpc1 = ev1 * pc1rep ;
          %  "average vector" for display
mresid1 = (mresid - ev1 * pc1') ;
ss1r = sum(sum(mresid1.^2)) ;
ss1 = ssmr - ss1r ;

ev2 = meigvec(:,d-1) ;
          %  eigenvector with largest eigenvalue
pc2 = mresid1' * ev2 ;
          %  vector of projections of data onto ev2
pc2rep = max(abs(pc2 - mean(pc2))) ;
          %  "most representative pc", defined as MAD
mpc2 = ev2 * pc2rep ;
          %  "average vector" for display
mresid2 = (mresid1 - ev2 * pc2') ;
ss2r = sum(sum(mresid2.^2)) ;
ss2 = ss1r - ss2r ;

ev3 = meigvec(:,d-2) ;
          %  eigenvector with largest eigenvalue
pc3 = mresid2' * ev3 ;
          %  vector of projections of data onto ev3
pc3rep = max(abs(pc3 - mean(pc3))) ;
          %  "most representative pc", defined as MAD
mpc3 = ev3 * pc3rep ;
          %  "average vector" for display
mresid3 = (mresid2 - ev3 * pc3') ;
ss3r = sum(sum(mresid3.^2)) ;
ss3 = ss2r - ss3r ;

ev4 = meigvec(:,d-3) ;
          %  eigenvector with largest eigenvalue
pc4 = mresid3' * ev4 ;
          %  vector of projections of data onto ev4
pc4rep = max(abs(pc4 - mean(pc4))) ;
          %  "most representative pc", defined as MAD
mpc4 = ev4 * pc4rep ;
          %  "average vector" for display
mresid4 = (mresid3 - ev4 * pc4') ;
ss4r = sum(sum(mresid4.^2)) ;
ss4 = ss3r - ss4r ;


%  Form mean +- representative eigenvectors
mpc1p = vmean + mpc1 ;
mpc1m = vmean - mpc1 ;
mpc2p = vmean + mpc2 ;
mpc2m = vmean - mpc2 ;
mpc3p = vmean + mpc3 ;
mpc3m = vmean - mpc3 ;
mpc4p = vmean + mpc4 ;
mpc4m = vmean - mpc4 ;




%  Create images
%
imm = coazr(vmean) ;
impc1p = coazr(mpc1p) ;
impc1m = coazr(mpc1m) ;
impc2p = coazr(mpc2p) ;
impc2m = coazr(mpc2m) ;
impc3p = coazr(mpc3p) ;
impc3m = coazr(mpc3m) ;
impc4p = coazr(mpc4p) ;
impc4m = coazr(mpc4m) ;




%  Set up window page
%
SPCAHAND = figure ;
          %  create a new window
set(SPCAHAND, 'NumberTitle', 'off');
set(SPCAHAND,'Position', SPCAHANDPOS) ;
set(SPCAHAND,'Name','Static PCA') ;



%  Put down plots in 4 x 4 array
%
%  Means first
%
for i = 1:4 ;
  subplot(4,4,4+i) ;
  pcolor(mxgrid,mygrid,imm) ;
    shading flat ;
    axis('square') ;
    title([meanstr ', ' LISTFILESTR]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;
  hold on ;
    if IRECENT == 0 ;  %  then everything is centered at apex normals
      plot(0,0,'+k');
    elseif IRECENT == 1 ;  %  then everything is recentered
      plot(0,0,'ok');
    end ;
    tx = -.95 * R0 ;
    ty = .7 * R0 ;
    eval(['pcss = 100 * ss' num2str(i) ' / ssmr ;']) ;
    ssstr = num2str(pcss) ;
    tstr = [ssstr '% of MR'] ;
    text(tx,ty,tstr) ;
  hold off ;
end ;


%  Now mean +- Pc's
%
for i = 1:4 ;
  subplot(4,4,i) ;
  eval(['pcolor(mxgrid,mygrid,impc' num2str(i) 'p)']) ;
    shading flat ;
    axis('square') ;
    title([meanstr ' + ' covstr num2str(i)]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;

  subplot(4,4,8+i) ;
  eval(['pcolor(mxgrid,mygrid,impc' num2str(i) 'm)']) ;
    shading flat ;
    axis('square') ;
    title([meanstr ' - ' covstr num2str(i)]) ;
      caxis([0 1]) ;
    colormap(COMAP) ;
end ;


%  Now do kde's
%
for i = 1:4 ;
  subplot(4,4,12+i) ;
  eval(['[kde, xgrid] = gpkde(pc' num2str(i)  ') ;']) ;
  plot(xgrid,kde,'w') ;
    title('Projections') ;
    vlim = axis ;
          %  Get axis values for drawing vertical lines
    eval(['left = -pc' num2str(i) 'rep ;']) ;
    eval(['right = pc' num2str(i) 'rep ;']) ;
    range = right - left ;
    left = left - .05 * range ;
    right = right + .05 * range ;
    bottom = vlim(3) ;
    top = vlim(4) ;
    axis([left right bottom top]) ;
    hold on ;
      evstr = ['plot([pc' num2str(i) ' pc' num2str(i) ...
                    '], [bottom top], ''c'') ;'] ;
      eval(evstr) ;
          %  Show locations of projections
      plot(xgrid,kde,'w') ;
          %  Do a replot to ensure seeing kde
    hold off ;
end ;




%  Add a title
%
tx = 0.05 ;
ty = 0.94 ;
tw = 0.8 ;
th = 0.05 ;
          %  set position for new axes, containing only title
axes('Position',[tx,ty,tw,th]) ;
          %  create new axes
set(gca,'XTickLabelMode','manual') ;
set(gca,'XTickLabel',[]) ;
set(gca,'XTickMode','manual') ;
set(gca,'XTick',[]) ;
set(gca,'XColor',[0,0,0]) ;
set(gca,'YTickLabelMode','manual') ;
set(gca,'YTickLabel',[]) ;
set(gca,'YTickMode','manual') ;
set(gca,'YTick',[]) ;
set(gca,'YColor',[0,0,0]) ;
          %  Switch off axis stuff
set(gca,'visible','off') ;
          %  Turn off surrounding box
titstr = ['Static PCA, using ' meanstr ' and ' covstr ...
                         'A, for file list: ' LISTFILESTR] ;
titstr = [titstr ',  ' date] ;
titstr = [titstr '  NMAX = ' num2str(NMAX) '  r0 = ' num2str(R0)] ;
text(0,0.5,titstr) ;
          %  put down actual title



