function coalwzr
% COALWZR, List-Wise Zernike Reconstruction
%      Corneal Topography Analysis System, ver. 0.1
%   Creates plots of multiple Zernike Reconstructions (several on a page)
%      from the current list of Zernike Feature matrix files
%      in a new Figure Window (creates several windows, if needed)
%   Intended to be called by coapam.m
%      Reads list of files from LISTFILESTR
% Output:
%     Matlab data file, with Zernike decomposition in vector form
%            in the directory in the global variable FEATMPATH,


%    Copyright (c) J. S. Marron, N. Locantore 1998
%    Department of Statistics, University of North Carolina




%  Set up global variables
%
%  First some general globals
global R0 NMAX CRAD CANG COMAP ;
%  R0 - radius of analysis
          %    this gets reloaded when a feature vec is reloaded from file
%  NMAX - maximum number of Zernike columns
          %    this gets reloaded when a feature vec is reloaded from file
%  CRAD, radius of recentering (read in as editable text)
          %    this gets reloaded when a feature vec is reloaded from file
%  CANG, angle of recentering (read in as editable text)
          %    this gets reloaded when a feature vec is reloaded from file
%  COMAP - Color Map

%  Next some globals for file names, etc.
global LISTFILESTR LISTR ;
%  LISTFILESTR - string with input name of file with list of filenames
%  LISTR - string for inputting list file

%  Next the paths for file control
global FEATMPATH ;
%  FEATMPATH - path for storing Zernike feature matrices

%  Next some file handles, etc.
global VLWZRHAND LWZRHANDPOS ;
%  VLWZRHAND - vector of handles for List-Wise Zernike Recon's
%  LWZRHANDPOS - Position coordinates for List-Wise Zernike recons


%  set local variables
%
ngrid = 64 ;     %  Construct surfaces as ngrid x ngrid array
picrows = 4 ;
          %  number of rows for images for output figures
piccols = 6 ;
          %  number of cols for images for output figures
r0last = R0 ;          %  Original radius of analysis
nmaxlast = [] ;        %  Original maximum number of Zernike columns
                       %       set to empty, so can tell when read
                       %       is first



%  Read in filename
%
% LISTR, (read-in file string, set in advance in coapam.m)
fid = fopen(LISTR) ;
          %  assigns a file handle, (read only assumed)
if (fid == -1) ;      %  then file not found, so report

  disp('   ') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
  disp('!!!   Can''t do List Wise Recon., because can''t find file:') ;
  disp(['!!!       ' LISTR]) ;
  disp('!!!') ;
  disp('!!!   Terminating List Wise Recon.') ;
  disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

else ;


  %  Read in list of filenames
  %
  minfilestr = fscanf(fid,'%s',[8,inf]) ;
            %  formatted read, assumes single column
  fclose(fid) ;

  minfilestr = minfilestr' ;
          %  each row corresponds to one filename


  %  set page stuff ;
  %
  nfile = size(minfilestr,1) ;
          %  number of files = rows of input matrix
  npage = ceil(nfile / (picrows * piccols)) ;
  ifilestart = 1 ;
  VLWZRHAND = [] ;


  %  Loop through pages
  %
  for ipage = 1:npage ;


    %  Create New Window for This page
    %
    lwzrhand = figure ;
    set(lwzrhand,'Position', LWZRHANDPOS) ;
    VLWZRHAND = [VLWZRHAND; lwzrhand] ;
    orient landscape;
    %  loop through filenames on this page
    %
    ifileend = min((ifilestart + (picrows * piccols) - 1),nfile) ;
    for ifile = ifilestart:ifileend ;

      %  get current feature vector
      %
      infilestr = minfilestr(ifile,:) ;
          %  combine indiviual entries into file string
      zmistr = [FEATMPATH infilestr '.mat'] ;
          %  full path and filename 
      eval(['load ' zmistr]) ;
          %  loads this Zernike feature matrix and parameter values

      if isempty(nmaxlast) ;  
                      %  then are at first read, so only check R0

        if (R0 ~= r0last) ;
          disp('   ') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
          disp('!!!   Warning from coalwzr.m:  R0 has changed,') ;
          disp(['!!!      Old R0 = ' num2str(r0last) ',   New R0 = ' num2str(R0)]) ;
          disp('!!!   Hit Any Key to continue with new R0,') ;
          disp('!!!   Or hit (ctrl + C) (twice) to quit') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

          pause ;          
        end ;

      else ;          %  then are at later reads, so check params

        if (R0 ~= r0last) | (NMAX ~= nmaxlast) ;
                  %  parameter values have changed, so quit

          %  Write latest values to screen
          NMAX
          nmaxlast
          R0
          r0last

          disp('   ') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;
          disp('!!!   Input Zernike Parameters have changed,') ;
          disp(['!!!           (differing file is ' infilestr ')']) ;
          disp('!!!') ;
          disp('!!!   Terminating Listwise Reconstruction') ;
          disp('!!!') ;
          disp('!!!           (Need to redo at least some of the') ;
          disp('!!!            Zernike Decompositions to fix this) ') ;
          disp('!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!') ;

          break ;
            %  jump out of loop

        end ;

      end ;

      r0last = R0 ;
      nmaxlast = NMAX ;


      %  Calculate grid quantities
      %
      xgrid = linspace(-R0, R0, ngrid) ;
      ygrid = xgrid ;	        %  Vectors of image coordinate values 



      disp(['  Page ' num2str(ipage) ',  file ' infilestr]) ;

      %  setup current axis for plotting image
      %
      iax = ifilestart - (ipage - 1) * (picrows * piccols) ;
      subplot(picrows,piccols,iax) ;

      %  Create image with Keratron Color scheme, and put in current axes
      %
      mout = coazr(betahat) ;
      pcolor(xgrid,ygrid,mout) ;
        shading flat ;
        axis('square') ;
          titstr = infilestr ;
        title(titstr) ;
          caxis([0 1]) ;
        colormap(COMAP) ;

        %  Show where missing data is:
        hold on ;
          adata = linspace(0, (1 - 1/256) * 2 * pi, 256)' ;
% 	  polar(adata, inmissr,'-k') ;
%  comment out inner missing data ring, may be used for future analyses
          polar(adata,nonmissr,'-k') ;    
        hold off ;

      ifilestart = ifilestart + 1 ;

    end ;

    %  Add a title for this page
    %
    tx = 0.05 ;
    ty = 0.94 ;
    tw = 0.8 ;
    th = 0.05 ;
          %  set position for new axes, containing only title
    axes('Position',[tx,ty,tw,th]) ;
          %  create new axes
    set(gca,'XTickLabelMode','manual') ;
    set(gca,'XTickLabel',[]) ;
    set(gca,'XTickMode','manual') ;
    set(gca,'XTick',[]) ;
    set(gca,'XColor',[0,0,0]) ;
    set(gca,'YTickLabelMode','manual') ;
    set(gca,'YTickLabel',[]) ;
    set(gca,'YTickMode','manual') ;
    set(gca,'YTick',[]) ;
    set(gca,'YColor',[0,0,0]) ;
          %  Switch off axis stuff
set(gca,'visible','off') ;
          %  Turn off surrounding box
    titstr = ['Recon''s for file list: ' LISTFILESTR] ;
    titstr = [titstr ',  page ' num2str(ipage) ',  ' date] ;
    titstr = [titstr '  NMAX = ' num2str(NMAX) '  r0 = ' num2str(R0)] ;
    text(0,0.5,titstr) ;
          %  put down actual title

  end ;

end ;










